// This file is part of the FXT library.
// Copyright (C) 2014, 2015, 2019, 2023 Joerg Arndt
// License: GNU General Public License version 3 or later,
// see the file COPYING.txt in the main directory.

#include "comb/tree-lev-seq-aux.h"

#include "fxtio.h"
#include "fxttypes.h"


void
tree_lev_seq_aux::print_aa(const char *bla, const ulong *a, ulong n)  const
// Print tree as ASCII art.
// Example:
// [ 0 1 1 1 1 2 2 3 3 1 2 2 ] gives
// O--o
// .--o
// .--o
// .--o--o
//    .--o--o
//       .--o
// .--o--o
//    .--o
// The dots should be read as a connection upwards as follows.
// O--o
// |
// .--o
// |
// .--o
// |
// .--o--o
// |  |
// |  .--o--o
// |     |
// |     .--o
// |
// .--o--o
//    |
//    .--o
{
    cout << bla;
    cout << "O";  // root
    ulong t = 0;  // prior level
    for (ulong j=1; j<n; ++j)
    {
        const ulong d = a[j];
        if ( d > t )
        {
            cout << "--o";
        }
        else
        {
            cout << endl;
            for (ulong i=1; i<d; ++i)  cout << "   ";
            cout << ".--o";
        }
        t = d;
    }
}
// -------------------------


void
tree_lev_seq_aux::print_branching_numbers(const char *bla,
                                          const ulong *a, ulong n,
                                          bool dfz/*=true*/)  const
// Print sequence of branching numbers.
// Example:
// [ 0 1 1 1 1 2 2 3 3 1 2 2 ] gives
// [ 5 . . . 2 . 2 . . 2 . . ] (dots for zeros)
{
    // note: use all_branching_numbers() to make cost linear
    cout << bla;
    cout << "[ ";
    for (ulong j=0; j<n; ++j)
    {
        const ulong b = branching_number(a, n, j);
        if ( b==0 )
            cout << ( dfz ? '.' : '0' ) << " ";
        else
            cout << b << " ";
    }
    cout << "]";
}
// -------------------------


void
tree_lev_seq_aux::print_base_seq(const char *bla,
                                 const ulong *a, ulong n,
                                 bool dfz/*=true*/)  const
// Print sequence of indices where the non-nodes are attached.
// Example:
// [ 0 1 2 3 2 1 2 1 2 1 1 1 ] gives
//   [ . 1 2 1 . 5 . 7 . . . ] (dots for zeros)
// The tree for comparison:
// O--o--o--o
//    .--o
// .--o--o
// .--o--o
// .--o
// .--o
// .--o
{
    cout << bla;
    cout << "[ ";
    for (ulong j=1; j<n; ++j)
    {
        ulong b = j;
        if ( b != 0 )
        {
            const ulong bv = a[j] - 1;
            do  { --b; }  while ( a[b] != bv );
        }
//        b = j - b;  // distance
        if ( b==0 )
            cout << ( dfz ? '.' : '0' ) << " ";
        else
            cout << (long)b << " ";
    }
    cout << "]";
}
// -------------------------


void
tree_lev_seq_aux::print_paren_word(const char *bla,
                                   const ulong *a, ulong n,
                                   const char * s/*="()"*/)  const
// Example:
// [ 0 1 1 1 1 2 2 3 3 1 2 2 ]  gives
//   (()()()(()(()()))(()()))
{
    cout << bla;
    if ( n == 0 )  return;

    cout << s[0];  // opening paren
    for (ulong j=1; j<n; ++j)
    {
        if ( a[j] <= a[j-1] )
        {
            ulong d = a[j-1] - a[j];  // >= 0
            do  { cout << s[1]; }  while ( d-- );  // closing parens
        }
        cout << s[0];  // opening paren
    }
    ulong d = a[n-1];
    do  { cout << s[1]; }  while ( d-- );  // closing parens
}
// -------------------------


#include "jjassert.h"

void
tree_lev_seq_aux::print_balanced_composition(const char *bla,
                                             const ulong *a, ulong n)  const
// For balanced trees only!
// Example (all balanced trees with 6 non-root nodes):
//        level sequence        composition
//   1:  [ 0 1 1 1 1 1 1 ]    [ 1 1 1 1 1 1 ]
//   2:  [ 0 1 2 1 2 1 2 ]    [ 2 2 2 ]
//   3:  [ 0 1 2 1 2 2 2 ]    [ 2 2 1 1 ]
//   4:  [ 0 1 2 2 1 2 2 ]    [ 2 1 2 1 ]
//   5:  [ 0 1 2 2 2 1 2 ]    [ 2 1 1 2 ]
//   6:  [ 0 1 2 2 2 2 2 ]    [ 2 1 1 1 1 ]
//   7:  [ 0 1 2 3 1 2 3 ]    [ 3 3 ]
//   8:  [ 0 1 2 3 2 3 3 ]    [ 3 2 1 ]
//   9:  [ 0 1 2 3 3 2 3 ]    [ 3 1 2 ]
//  10:  [ 0 1 2 3 3 3 3 ]    [ 3 1 1 1 ]
//  11:  [ 0 1 2 3 4 3 4 ]    [ 4 2 ]
//  12:  [ 0 1 2 3 4 4 4 ]    [ 4 1 1 ]
//  13:  [ 0 1 2 3 4 5 5 ]    [ 5 1 ]
//  14:  [ 0 1 2 3 4 5 6 ]    [ 6 ]
{
    ulong s = 0;  // sum
    cout << bla;
    cout << "[ ";
    if ( n <= 1 )
    {
        if ( n==1 )  cout << "1 ";
    }
    else
    {
        for (ulong j=1; j<n-1; ++j)
        {
            ulong h = a[j];
            ulong p = 1;  // part of the composition
            while ( a[j+1] == h + 1 )
            {
                p += 1;
                if ( j + 1 == n )  break;
                h = a[j+1];
                ++j;
            }
            cout << p << " ";

            s += p;
        }

        if ( a[n-2] + 1 != a[n-1] )
        {
            cout << 1 << " ";
            s += 1;
        }
    }
    cout << "]";

    jjassert( s == n - 1 );
}
// -------------------------
