#if !defined  HAVE_WALSHGRAY_H__
#define       HAVE_WALSHGRAY_H__
// This file is part of the FXT library.
// Copyright (C) 2010, 2011, 2012 Joerg Arndt
// License: GNU General Public License version 3 or later,
// see the file COPYING.txt in the main directory.

//#include "walsh/grsnegate.h"
#include "fxttypes.h"


template <typename Type>
void walsh_gray(Type *f, ulong ldn)
// Gray variant of the Walsh transform.
// Radix-2 decimation in frequency (DIF) algorithm
//.
// The basis (sequency at end of lines, '*':=1, ' ':=-1):
//  0: [* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *] ( 0)
//  1: [*     * *     * *     * *     * *     * *     * *     * *     *] (16)
//  2: [* *         * * * *         * * * *         * * * *         * *] ( 8)
//  3: [  *   * *   *     *   * *   *     *   * *   *     *   * *   *  ] (24)
//  4: [* * * *                 * * * * * * * *                 * * * *] ( 4)
//  5: [*     *   * *     * *   *     * *     *   * *     * *   *     *] (20)
//  6: [    * *     * * * *     * *         * *     * * * *     * *    ] (12)
//  7: [*   *   *   *     *   *   *   * *   *   *   *     *   *   *   *] (28)
//  8: [* * * * * * * *                                 * * * * * * * *] ( 2)
//  9: [*     * *     *   * *     * *     * *     * *   *     * *     *] (18)
// 10: [* *         * *     * * * *         * * * *     * *         * *] (10)
// 11: [  *   * *   *   *   *     *   * *   *     *   *   *   * *   *  ] (26)
// 12: [        * * * *         * * * * * * * *         * * * *        ] ( 6)
// 13: [  * *   *     *   * *   *     * *     *   * *   *     *   * *  ] (22)
// 14: [* *     * *     * *     * *         * *     * *     * *     * *] (14)
// 15: [  *   *   *   *   *   *   *   * *   *   *   *   *   *   *   *  ] (30)
// 16: [* * * * * * * * * * * * * * * *                                ] ( 1)
// 17: [*     * *     * *     * *     *   * *     * *     * *     * *  ] (17)
// 18: [* *         * * * *         * *     * * * *         * * * *    ] ( 9)
// 19: [  *   * *   *     *   * *   *   *   *     *   * *   *     *   *] (25)
// 20: [* * * *                 * * * *         * * * * * * * *        ] ( 5)
// 21: [*     *   * *     * *   *     *   * *   *     * *     *   * *  ] (21)
// 22: [    * *     * * * *     * *     * *     * *         * *     * *] (13)
// 23: [*   *   *   *     *   *   *   *   *   *   *   * *   *   *   *  ] (29)
// 24: [                * * * * * * * *                 * * * * * * * *] ( 3)
// 25: [  * *     * *   *     * *     *   * *     * *   *     * *     *] (19)
// 26: [    * * * *     * *         * *     * * * *     * *         * *] (11)
// 27: [*   *     *   *   *   * *   *   *   *     *   *   *   * *   *  ] (27)
// 28: [* * * *         * * * *         * * * *         * * * *        ] ( 7)
// 29: [*     *   * *   *     *   * *   *     *   * *   *     *   * *  ] (23)
// 30: [    * *     * *     * *     * *     * *     * *     * *     * *] (15)
// 31: [*   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *  ] (31)
//
//
// Fast on some machines, due to cache fiendly memory access.
//
// walsh_wak() =^=
//   inverse_gray_permute(f, n);
//   walsh_gray(f, ldn);
//   grs_negate(f, n);
//
// walsh_wal() =^=
//   walsh_gray(f, ldn);
//   grs_negate(f, n);
//   revbin_permute(f, n);
//
{
    const ulong n = (1UL<<ldn);

    for (ulong ldm=ldn; ldm>0; --ldm)  // DIF
    {
        const ulong m = (1UL<<ldm);
        for (ulong r=0; r<n; r+=m)
        {
            ulong t1 = r;
            ulong t2 = r + m - 1;
            for (  ;  t1<t2;  ++t1, --t2)
            {
                Type u = f[t1];
                Type v = f[t2];
                f[t1] = u + v;
                f[t2] = u - v;
            }
        }
    }
}
// -------------------------


template <typename Type>
void inverse_walsh_gray(Type *f, ulong ldn)
// Inverse of walsh_gray().
// Radix-2 decimation in time (DIT) algorithm.
//.
// The basis (sequency at end of lines, '*':=1, ' ':=-1):
//  0: [* * *   * *   * * * *       *   * * *   * *   *       * * *   *] (16)
//  1: [*   * * *       *   * *   * * * *   * * *         *     *      ] (15)
//  2: [*       *   * * *         *     *       *   * *   * * * *   * *] (16)
//  3: [* *   * * * *   * *   *       * * *   * * * *       *   * * *  ] (15)
//  4: [* *   *       * * *   * * * *   * *   *       *     *         *] (16)
//  5: [*         *     *       *   * * *         *       * * *   *    ] (15)
//  6: [*   * *   * * * *   * * *       *   * *   * * *   *       * * *] (16)
//  7: [* * *       *   * * *   * *   * * * *       *         *     *  ] (15)
//  8: [* * *       *         *     *   * * *       *   * * *   * *   *] (16)
//  9: [*   * *   * * *   *       * * * *   * *   * * * *   * * *      ] (15)
// 10: [*         *       * * *   *     *         *     *       *   * *] (16)
// 11: [* *   *       *     *         * * *   *       * * *   * * * *  ] (15)
// 12: [* *   * * * *       *   * * *   * *   * * * *   * *   *       *] (16)
// 13: [*       *   * *   * * * *   * * *       *   * * *         *    ] (15)
// 14: [*   * * *         *     *       *   * * *       *   * *   * * *] (16)
// 15: [* * *   * *   *       * * *   * * * *   * *   * * * *       *  ] (15)
// 16: [* * *   * *   *       * * *   *       *     *         * * *   *] (16)
// 17: [*   * * *         *     *         *       * * *   *     *      ] (15)
// 18: [*       *   * *   * * * *   * *   * * *   *       * * * *   * *] (16)
// 19: [* *   * * * *       *   * * *       *         *     *   * * *  ] (15)
// 20: [* *   *       *     *         *     *   * * *       *         *] (16)
// 21: [*         *       * * *   *       * * * *   * *   * * *   *    ] (15)
// 22: [*   * *   * * *   *       * * *   *     *         *       * * *] (16)
// 23: [* * *       *         *     *         * * *   *       *     *  ] (15)
// 24: [* * *       *   * * *   * *   *       * * *   * * * *   * *   *] (16)
// 25: [*   * *   * * * *   * * *         *     *       *   * * *      ] (15)
// 26: [*         *     *       *   * *   * * * *   * * *       *   * *] (16)
// 27: [* *   *       * * *   * * * *       *   * * *   * *   * * * *  ] (15)
// 28: [* *   * * * *   * *   *       *     *         * * *   *       *] (16)
// 29: [*       *   * * *         *       * * *   *     *         *    ] (15)
// 30: [*   * * *       *   * *   * * *   *       * * * *   * *   * * *] (16)
// 31: [* * *   * *   * * * *       *         *     *   * * *       *  ] (15)
{
    const ulong n = (1UL<<ldn);

    for (ulong ldm=1; ldm<=ldn; ++ldm)  // DIT
    {
        const ulong m = (1UL<<ldm);
        for (ulong r=0; r<n; r+=m)
        {
            ulong t1 = r;
            ulong t2 = r + m - 1;
            for (  ;  t1<t2;  ++t1, --t2)
            {
                Type u = f[t1];
                Type v = f[t2];
                f[t1] = u + v;
                f[t2] = u - v;
            }
        }
    }
}
// -------------------------



#endif  // !defined HAVE_WALSHGRAY_H__
