// This file is part of the FXT library.
// Copyright (C) 2010, 2012, 2014, 2018, 2023, 2024 Joerg Arndt
// License: GNU General Public License version 3 or later,
// see the file COPYING.txt in the main directory.

#include "perm/permq.h"
#include "ds/bitarray.h"
#include "ds/left-right-array.h"

#include "fxttypes.h"


ulong
perm_count_inversions(const ulong *f, ulong n)
// Return number of inversions in f[],
// that is, number of pairs k,j where k<j and f[k]>f[j]
// Algorithm is O(n^2).
{
    ulong ct = 0;
    for (ulong k=1; k<n; ++k)
    {
        ulong fk = f[k];
        for (ulong j=0; j<k; ++j)  ct += ( fk < f[j] );
    }
    return ct;
}
// -------------------------

ulong
perm_count_inversions(const ulong *f, ulong n, left_right_array *tLR)
// Return number of inversions in f[],
// that is, number of pairs k,j where k<j and f[k]>f[j]
// Version for large arrays, algorithm is O(n*log_2(n))
{
    left_right_array *LR = tLR;
    if ( tLR==nullptr )  LR = new left_right_array(n);

    ulong ct = 0;
    LR->set_all();
    for (ulong k=0; k<n-1; ++k)
    {
        ulong i = LR->num_SLE( f[k] );
        LR->get_set_idx_chg( i );
        ct += i;
    }

    if ( tLR==nullptr )  delete LR;
    return ct;
}
// -------------------------

bool
perm_is_valid(const ulong *f, ulong n, bitarray *bp/*=nullptr*/)
// Return whether all values 0...n-1 appear exactly once,
// that is, whether f represents a permutation of [0,1,...,n-1].
{
    // check whether any element is out of range:
    for (ulong k=0; k<n; ++k)  if ( f[k] >= n )  return false;

    // check whether values are unique:
    bitarray *tp = bp;
    if ( nullptr == bp )  tp = new bitarray(n);  // tags
    tp->clear_all();

    ulong k;
    for (k=0; k<n; ++k)  if ( tp->test_set(f[k]) )  break;

    if ( nullptr == bp )  delete tp;

    return  ( k == n );
}
// -------------------------


ulong
perm_count_transpositions(const ulong *f, ulong n, bitarray *bp/*=nullptr*/)
// Return minimal number of transpositions required for the permutation.
// The lowest bit of the return value is the parity of the permutation.
// Algorithm is O(n).
{
    bitarray *tp = bp;
    if ( nullptr == bp )  tp = new bitarray(n);  // tags
    tp->clear_all();

    ulong nt = 0;  // minimal number of transpositions
    for (ulong k=0; k<n; ++k)
    {
        if ( tp->test(k) )  continue;  // already processed

        // Follow a cycle:
        ulong len = 0;
        ulong i = k;
        while ( 0==(tp->test_set(i)) )
        {
            i = f[i];
            ++len;
        }
        nt += (len - 1);  // length-len cycle needs len-1 transpositions
    }

    if ( nullptr == bp )  delete tp;
    return nt;
}
// -------------------------

ulong
perm_get_parity(const ulong *f, ulong n, bitarray *bp/*=nullptr*/)
// Return parity (sign) of the permutation.
{
    return perm_count_transpositions( f, n, bp ) & 1;
}
// -------------------------


ulong
perm_count_cycles(const ulong *f, ulong n, bitarray *bp/*=nullptr*/)
{
    return  n - perm_count_transpositions(f, n, bp);
}
// -------------------------

//ulong
//perm_get_periods(const ulong *f, ulong n, ulong *p, bitarray *bp/*=nullptr*/)
//// Write to p[k] the period of element f[k]
//// (period := length of cycle containing f[k])
//// Return minimal number of transpositions required for the permutation.
////
//// Example (n=11):
////   k:   f[k]  p[k]
////   0:    3     2
////   1:    7     3
////   2:    9     5
////   3:    0     2
////   4:    2     5
////   5:    1     3
////   6:   10     5
////   7:    5     3
////   8:    8     1
////   9:    6     5
////  10:    4     5
//// return nt=7
//{
//    bitarray *tp = bp;
//    if ( 0==bp )  tp = new bitarray(n);  // tags
//    tp->clear_all();
//
//    ulong nt = 0;  // minimal number of transpositions
//    for (ulong k=0; k<n; ++k)
//    {
//        if ( tp->test_clear(k) )  continue;  // already processed
//
//        // Follow a cycle:
//        ulong len = 0;
//        ulong i = k;
//        while ( 0==(tp->test_set(i)) )
//        {
//            i = f[i];
//            ++len;
//        }
//        nt += (len - 1);  // length-len cycle needs len-1 transpositions
//
//        i = k;
//        do
//        {
//            p[i] = len;
//            i = f[i];
//        }
//        while ( i!=k );
//    }
//
//    if ( 0==bp )  delete tp;
//    return nt;
//}
//// -------------------------



bool
perm_is_simple(const ulong *f, ulong n)
// Return whether permutation is simple.
// A permutation is simple if the only intervals that are mapped to an
//  interval of the same length is the full permutation and the singletons.
// Cf. OEIS sequence A111111.
// Complexity is O(n^2).
{
    if ( n <= 2 )  return true;

    for (ulong j=0; j < n-1; ++j)  // start of range
    {
        ulong mi = f[j];  // min of images in range
        ulong ma = f[j];  // max of images in range
        for (ulong k=j+1;  k < n - (j==0);  ++k)  // end of range
        {
            const ulong fk = f[k];
            if ( fk < mi )  mi = fk;
            if ( fk > ma )  ma = fk;

            if ( ma - mi == k - j )  return false;
        }
    }

    return true;
}
// -------------------------
