// This file is part of the FXT library.
// Copyright (C) 2010, 2012 Joerg Arndt
// License: GNU General Public License version 3 or later,
// see the file COPYING.txt in the main directory.

#include "dctdst/dctdst.h"
#include "fht/fht-default.h"

#include "fxttypes.h"  // ulong

#include <cmath>  // sqrt()

#define  VV   (1.0)  // norm preserving
//#define  VV   (2.0)  // preserves the norm of the signal _without_ the DC part

void
dct4(double *f, ulong ldn)
//  Cosine transform wrt. the basis
//      cos( 2*Pi * 0.25 * (4*k + 1) * i / n )
//    == cos( Pi * (2*k + 0.5) * i / n )
//   where k is the index of the basis
//
// the basis:
//  0: [* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *] ( 0)
//  1: [* * * * * * *                           * * * * * * * * * * * *] ( 2)
//  2: [* * * *               * * * * * * *               * * * * * * *] ( 4)
//  3: [* * *           * * * * *           * * * * *           * * * *] ( 6)
//  4: [* *         * * * *         * * *         * * * *         * * *] ( 8)
//  5: [* *       * * *       * * *       * * *       * * *       * * *] (10)
//  6: [* *     * * *     * * *       * *       * *       * *       * *] (12)
//  7: [* *     * *     * *       * *     * *     * * *     * *     * *] (14)
//  8: [*     * *     * *     * *     * *     * *     * *     * *     *] (16)
//  9: [*     * *     *     * *   * *     * *   * *     * *   * *     *] (18)
// 10: [*     *     * *   * *   * *   * *   * *     *     *     *     *] (20)
// 11: [*     *   * *   * *   * *   *     *     *     *   * *   * *   *] (22)
// 12: [*   * *   *     *   * *   *     *   *     *   * *   *     *   *] (24)
// 13: [*   * *   *   *     *   *   * *   *   *     *   *   * *   *   *] (26)
// 14: [*   *   * *   *   *   *   * *   *   *   *   * *   *   *   *   *] (28)
// 15: [*   *   *   *   *   * *   *   *   *   *   *   *   *   *   *   *] (30)
// 16: [*   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *  ] (31)
// 17: [*   *   *   * *   *   *   *   *   *   * *   *   *   *   *   *  ] (29)
// 18: [*   *     *   *   *     *   *   *     *   *   *     *   *   *  ] (27)
// 19: [*   * *   *   * *   *   * *   *   * *   *   * *   *   * *   *  ] (25)
// 20: [*     *   * *   *     *   * *   * *   *     *   * *   *     *  ] (23)
// 21: [*     *     *     *     *     *     *     *     *     *     *  ] (21)
// 22: [*     * *   * *     *     *     * *   * *     *     * *   * *  ] (19)
// 23: [*     * *     * *     *     * *     * *     *     * *     * *  ] (17)
// 24: [* *     * *     * *     * *     * *     * *     * *     * *    ] (15)
// 25: [* *     * *       * *       * *     * * *     * *       * *    ] (13)
// 26: [* *       * *       * * *       * * *     * * *       * * *    ] (11)
// 27: [* *         * * *       * * * *       * * *         * * *      ] ( 9)
// 28: [* * *         * * * *         * * * * *         * * * *        ] ( 7)
// 29: [* * *             * * * * * *             * * * * * *          ] ( 5)
// 30: [* * * * *                   * * * * * * * * *                  ] ( 3)
// 31: [* * * * * * * * * * *                                          ] ( 1)
//
//  The mean value of all basis functions is nonzero
//  therefore the DC part has to be treated separately
{
    ulong n = (1UL<<ldn);
    double mn = f[0] * sqrt(1.0/VV);
    f[0] = 0;  // subtract DC part
    cos_rot(f, f, n);
    fht(f, ldn);
    for (ulong i=0; i<n; ++i)  f[i] += mn;  // add back DC part
}
// -------------------------


void
idct4(double *f, ulong ldn)
// Inverse of dct4()
//
//  ... which is a transform wrt. the basis
//  cos(  Pi * k * (2*i+0.5) / n )
//   where k is the index of the basis
{
    ulong n = (1UL<<ldn);
    double mn = 0;
    for (ulong i=0; i<n; ++i)  mn += f[i];  // find DC part
    for (ulong i=0; i<n; ++i)  f[i] -= mn;  // subtract DC part
    fht(f, ldn);
    cos_rot(f, f, n);
    // here:  f[0] == 0
    f[0] = mn * sqrt(VV);  // add back DC part
}
// -------------------------

// idsth(f, ldn); =^=
//   reverse(f, n);
//   dct4(f, ldn);  // with VV==1
//   zip_rev(f, n);

void
dct4_basis(double *f, ulong n, ulong k)
{
//    double vv = 1.0;
    const double ph = M_PI * 0.5 * (4.0*(double)k+1) / (double)n;
    for (ulong i=0; i<n; ++i)
    {
        // f[i] = vv * cos( M_PI * 0.5 * (4*k+1) * i / n );
        f[i] = cos( (double)i * ph );
    }
}
// -------------------------

// this would give the functions sequency-ordered:
//
// forward:=
//   dct4(a, ldn);
//   zip_rev(a, n);
//
// inverse:=
//   unzip_rev(a, n);
//   idct4(a, ldn);
//
// then:
//void
//basis(double *f, ulong n, ulong k)
//{
//    k *= 2;
//    if ( k>=n )  k = 2*n - 1 - k;
//    for (ulong i=0; i<n; ++i)  f[i] = cos( 2*M_PI * 0.25 * (4*k + 1) * i / n );
//}



// basis functions produced by the inverse transform of delta peaks:
//
// VV == 2.0  (therefore the norm is not preserved)
// with VV==1 all zero components have to be multiplied by 1/sqrt(2)
//
//
//   ++++++++++++++++++++++++++++++++++++   k = 0
//
//   norm = 1.03078
//                             |---------------------------*         0.353553   0
//                             |---------------------------*         0.351851   1
//                             |--------------------------*           0.34676   2
//                             |--------------------------*           0.33833   3
//                             |-------------------------*           0.326641   4
//                             |------------------------*            0.311806   5
//                             |----------------------*              0.293969   6
//                             |---------------------*                 0.2733   7
//                             |-------------------*                     0.25   8
//                             |-----------------*                   0.224292   9
//                             |---------------*                     0.196424   10
//                             |------------*                        0.166664   11
//                             |----------*                          0.135299   12
//                             |-------*                             0.102631   13
//                             |-----*                              0.0689748   14
//                             |--*                                 0.0346543   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 1
//
//   norm = 1.03078
//                             |---------------------------*         0.353553   0
//                             |------------------------*            0.311806   1
//                             |---------------*                     0.196424   2
//                             |--*                                 0.0346543   3
//                   *---------|                                    -0.135299   4
//        *--------------------|                                      -0.2733   5
//   *-------------------------|                                     -0.34676   6
//   *-------------------------|                                     -0.33833   7
//          *------------------|                                        -0.25   8
//                      *------|                                    -0.102631   9
//                             |-----*                              0.0689748   10
//                             |-----------------*                   0.224292   11
//                             |-------------------------*           0.326641   12
//                             |---------------------------*         0.351851   13
//                             |----------------------*              0.293969   14
//                             |------------*                        0.166664   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 2
//
//   norm = 1.03078
//                             |---------------------------*         0.353553   0
//                             |-----------------*                   0.224292   1
//                        *----|                                   -0.0689748   2
//     *-----------------------|                                    -0.311806   3
//    *------------------------|                                    -0.326641   4
//                      *------|                                    -0.102631   5
//                             |---------------*                     0.196424   6
//                             |---------------------------*         0.351851   7
//                             |-------------------*                     0.25   8
//                           *-|                                   -0.0346543   9
//       *---------------------|                                    -0.293969   10
//   *-------------------------|                                     -0.33833   11
//                   *---------|                                    -0.135299   12
//                             |------------*                        0.166664   13
//                             |--------------------------*           0.34676   14
//                             |---------------------*                 0.2733   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 3
//
//   norm = 1.03078
//                             |---------------------------*         0.353553   0
//                             |-------*                             0.102631   1
//       *---------------------|                                    -0.293969   2
//        *--------------------|                                      -0.2733   3
//                             |----------*                          0.135299   4
//                             |---------------------------*         0.351851   5
//                             |-----*                              0.0689748   6
//     *-----------------------|                                    -0.311806   7
//          *------------------|                                        -0.25   8
//                             |------------*                        0.166664   9
//                             |--------------------------*           0.34676   10
//                             |--*                                 0.0346543   11
//    *------------------------|                                    -0.326641   12
//            *----------------|                                    -0.224292   13
//                             |---------------*                     0.196424   14
//                             |--------------------------*           0.33833   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 4
//
//   norm = 1.03078
//                             |---------------------------*         0.353553   0
//                           *-|                                   -0.0346543   1
//   *-------------------------|                                     -0.34676   2
//                             |-------*                             0.102631   3
//                             |-------------------------*           0.326641   4
//                 *-----------|                                    -0.166664   5
//       *---------------------|                                    -0.293969   6
//                             |-----------------*                   0.224292   7
//                             |-------------------*                     0.25   8
//        *--------------------|                                      -0.2733   9
//              *--------------|                                    -0.196424   10
//                             |------------------------*            0.311806   11
//                             |----------*                          0.135299   12
//   *-------------------------|                                     -0.33833   13
//                        *----|                                   -0.0689748   14
//                             |---------------------------*         0.351851   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 5
//
//   norm = 1.03078
//                             |---------------------------*         0.353553   0
//                 *-----------|                                    -0.166664   1
//              *--------------|                                    -0.196424   2
//                             |---------------------------*         0.351851   3
//                   *---------|                                    -0.135299   4
//            *----------------|                                    -0.224292   5
//                             |--------------------------*           0.34676   6
//                      *------|                                    -0.102631   7
//          *------------------|                                        -0.25   8
//                             |--------------------------*           0.33833   9
//                        *----|                                   -0.0689748   10
//        *--------------------|                                      -0.2733   11
//                             |-------------------------*           0.326641   12
//                           *-|                                   -0.0346543   13
//       *---------------------|                                    -0.293969   14
//                             |------------------------*            0.311806   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 6
//
//   norm = 1.03078
//                             |---------------------------*         0.353553   0
//        *--------------------|                                      -0.2733   1
//                             |-----*                              0.0689748   2
//                             |------------*                        0.166664   3
//    *------------------------|                                    -0.326641   4
//                             |--------------------------*           0.33833   5
//              *--------------|                                    -0.196424   6
//                           *-|                                   -0.0346543   7
//                             |-------------------*                     0.25   8
//  *--------------------------|                                    -0.351851   9
//                             |----------------------*              0.293969   10
//                      *------|                                    -0.102631   11
//                   *---------|                                    -0.135299   12
//                             |------------------------*            0.311806   13
//   *-------------------------|                                     -0.34676   14
//                             |-----------------*                   0.224292   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 7
//
//   norm = 1.03078
//                             |---------------------------*         0.353553   0
//   *-------------------------|                                     -0.33833   1
//                             |----------------------*              0.293969   2
//            *----------------|                                    -0.224292   3
//                             |----------*                          0.135299   4
//                           *-|                                   -0.0346543   5
//                        *----|                                   -0.0689748   6
//                             |------------*                        0.166664   7
//          *------------------|                                        -0.25   8
//                             |------------------------*            0.311806   9
//   *-------------------------|                                     -0.34676   10
//                             |---------------------------*         0.351851   11
//    *------------------------|                                    -0.326641   12
//                             |---------------------*                 0.2733   13
//              *--------------|                                    -0.196424   14
//                             |-------*                             0.102631   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 8
//
//   norm = 1.03078
//                             |---------------------------*         0.353553   0
//  *--------------------------|                                    -0.351851   1
//                             |--------------------------*           0.34676   2
//   *-------------------------|                                     -0.33833   3
//                             |-------------------------*           0.326641   4
//     *-----------------------|                                    -0.311806   5
//                             |----------------------*              0.293969   6
//        *--------------------|                                      -0.2733   7
//                             |-------------------*                     0.25   8
//            *----------------|                                    -0.224292   9
//                             |---------------*                     0.196424   10
//                 *-----------|                                    -0.166664   11
//                             |----------*                          0.135299   12
//                      *------|                                    -0.102631   13
//                             |-----*                              0.0689748   14
//                           *-|                                   -0.0346543   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 9
//
//   norm = 1.03078
//                             |---------------------------*         0.353553   0
//     *-----------------------|                                    -0.311806   1
//                             |---------------*                     0.196424   2
//                           *-|                                   -0.0346543   3
//                   *---------|                                    -0.135299   4
//                             |---------------------*                 0.2733   5
//   *-------------------------|                                     -0.34676   6
//                             |--------------------------*           0.33833   7
//          *------------------|                                        -0.25   8
//                             |-------*                             0.102631   9
//                             |-----*                              0.0689748   10
//            *----------------|                                    -0.224292   11
//                             |-------------------------*           0.326641   12
//  *--------------------------|                                    -0.351851   13
//                             |----------------------*              0.293969   14
//                 *-----------|                                    -0.166664   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 10
//
//   norm = 1.03078
//                             |---------------------------*         0.353553   0
//            *----------------|                                    -0.224292   1
//                        *----|                                   -0.0689748   2
//                             |------------------------*            0.311806   3
//    *------------------------|                                    -0.326641   4
//                             |-------*                             0.102631   5
//                             |---------------*                     0.196424   6
//  *--------------------------|                                    -0.351851   7
//                             |-------------------*                     0.25   8
//                             |--*                                 0.0346543   9
//       *---------------------|                                    -0.293969   10
//                             |--------------------------*           0.33833   11
//                   *---------|                                    -0.135299   12
//                 *-----------|                                    -0.166664   13
//                             |--------------------------*           0.34676   14
//        *--------------------|                                      -0.2733   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 11
//
//   norm = 1.03078
//                             |---------------------------*         0.353553   0
//                      *------|                                    -0.102631   1
//       *---------------------|                                    -0.293969   2
//                             |---------------------*                 0.2733   3
//                             |----------*                          0.135299   4
//  *--------------------------|                                    -0.351851   5
//                             |-----*                              0.0689748   6
//                             |------------------------*            0.311806   7
//          *------------------|                                        -0.25   8
//                 *-----------|                                    -0.166664   9
//                             |--------------------------*           0.34676   10
//                           *-|                                   -0.0346543   11
//    *------------------------|                                    -0.326641   12
//                             |-----------------*                   0.224292   13
//                             |---------------*                     0.196424   14
//   *-------------------------|                                     -0.33833   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 12
//
//   norm = 1.03078
//                             |---------------------------*         0.353553   0
//                             |--*                                 0.0346543   1
//   *-------------------------|                                     -0.34676   2
//                      *------|                                    -0.102631   3
//                             |-------------------------*           0.326641   4
//                             |------------*                        0.166664   5
//       *---------------------|                                    -0.293969   6
//            *----------------|                                    -0.224292   7
//                             |-------------------*                     0.25   8
//                             |---------------------*                 0.2733   9
//              *--------------|                                    -0.196424   10
//     *-----------------------|                                    -0.311806   11
//                             |----------*                          0.135299   12
//                             |--------------------------*           0.33833   13
//                        *----|                                   -0.0689748   14
//  *--------------------------|                                    -0.351851   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 13
//
//   norm = 1.03078
//                             |---------------------------*         0.353553   0
//                             |------------*                        0.166664   1
//              *--------------|                                    -0.196424   2
//  *--------------------------|                                    -0.351851   3
//                   *---------|                                    -0.135299   4
//                             |-----------------*                   0.224292   5
//                             |--------------------------*           0.34676   6
//                             |-------*                             0.102631   7
//          *------------------|                                        -0.25   8
//   *-------------------------|                                     -0.33833   9
//                        *----|                                   -0.0689748   10
//                             |---------------------*                 0.2733   11
//                             |-------------------------*           0.326641   12
//                             |--*                                 0.0346543   13
//       *---------------------|                                    -0.293969   14
//     *-----------------------|                                    -0.311806   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 14
//
//   norm = 1.03078
//                             |---------------------------*         0.353553   0
//                             |---------------------*                 0.2733   1
//                             |-----*                              0.0689748   2
//                 *-----------|                                    -0.166664   3
//    *------------------------|                                    -0.326641   4
//   *-------------------------|                                     -0.33833   5
//              *--------------|                                    -0.196424   6
//                             |--*                                 0.0346543   7
//                             |-------------------*                     0.25   8
//                             |---------------------------*         0.351851   9
//                             |----------------------*              0.293969   10
//                             |-------*                             0.102631   11
//                   *---------|                                    -0.135299   12
//     *-----------------------|                                    -0.311806   13
//   *-------------------------|                                     -0.34676   14
//            *----------------|                                    -0.224292   15
//
//   ++++++++++++++++++++++++++++++++++++   k = 15
//
//   norm = 1.03078
//                             |---------------------------*         0.353553   0
//                             |--------------------------*           0.33833   1
//                             |----------------------*              0.293969   2
//                             |-----------------*                   0.224292   3
//                             |----------*                          0.135299   4
//                             |--*                                 0.0346543   5
//                        *----|                                   -0.0689748   6
//                 *-----------|                                    -0.166664   7
//          *------------------|                                        -0.25   8
//     *-----------------------|                                    -0.311806   9
//   *-------------------------|                                     -0.34676   10
//  *--------------------------|                                    -0.351851   11
//    *------------------------|                                    -0.326641   12
//        *--------------------|                                      -0.2733   13
//              *--------------|                                    -0.196424   14
//                      *------|                                    -0.102631   15
//
//
