(*
   Copyright 2008-2018 Microsoft Research

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*)
module Splice
open FStar.List.Tot
open FStar.Tactics

let make_42 (nm:string) : Tac decls =
    (* The let binds are needed due to the Tac effect *)
    (* TODO: make the cur_module call unneeded? it doesn't make sense to use another module *)
    let lbv = {lb_fv = (pack_fv (cur_module () @ [nm]));
               lb_us = [];
               lb_typ = (`nat);
               lb_def = (`42)} in
    let lb = pack_lb lbv in
    let sv : sigelt_view = Sg_Let false [lb] in
    let ses : list sigelt = [pack_sigelt sv] in
    ses

%splice[x] (make_42 "x")
%splice[]  (make_42 "y")

let _ = assert (x == 42)

(* This fails at desugaring time, since `y` cannot be found. *)
(* let _ = assert (y == 42) *)


// Tests crafting [recursive] a total recursive top-level with a
// (non-trivial) decreases clause, and that returns a [Type0].
// The function generated by the splice below is the following one:
//
//   let rec recursive (n: nat): Tot Type0 (decreases (10 - n))
//     = if n>=10 then int else int * recursive (n + 1)
//
%splice[recursive] (
  let n = fresh_binder (`nat) in
  let recursive_fv = pack_fv (cur_module () @ ["recursive"]) in
  let recursive = pack (Tv_FVar recursive_fv) in
  [pack_sigelt (
    Sg_Let true [pack_lb ({
      lb_fv = recursive_fv
    ; lb_us = []
    ; lb_typ = mk_arr [n] (pack_comp (C_Total (`Type0) (pack_universe (Uv_Succ (pack_universe Uv_Zero))) [`(10 - (`#(binder_to_term n)))]))
    ; lb_def = `(fun n -> if n>=10 then int else int * (`#recursive) (n + 1))
    })]
    )
  ]
)

let _ = assert_norm (recursive 10 ==          int );
        assert_norm (recursive  9 ==      int * int );
        assert_norm (recursive  8 == int * (int * int))
