(* (c) Microsoft Corporation 2005-2007.  *)

(*-------------------------------------------------------------------------
 * Define Initial Environment.  A bunch of types and values are hard-wired 
 * into the compiler.  This lets the compiler perform particular optimizations
 * for these types and values, for example emitting optimized calls for
 * comparison and hashing functions.  The compiler generates the compiled code 
 * for these types and values when the the --compiling-fslib switch is 
 * provided when linking the FSharp.Core.dll assembly.
 *------------------------------------------------------------------------- *)

(*F# 
module Microsoft.FSharp.Compiler.Env 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 
module Il = Microsoft.Research.AbstractIL.IL 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
module Ilx    = Microsoft.Research.AbstractIL.Extensions.ILX 
F#*) 

open Il
open Tast
open Range
open Ast
open List
open Ildiag
open Ilx
open Lib

let env_range = rangeN "builtin" 0

let vara = mk_rigid_typar "a" env_range
let varb = mk_rigid_typar "b" env_range
let varc = mk_rigid_typar "c" env_range
let vard = mk_rigid_typar "d" env_range
let vare = mk_rigid_typar "e" env_range
let varf = mk_rigid_typar "f" env_range
let varg = mk_rigid_typar "g" env_range

let vara_ty = TType_var vara 
let varb_ty = TType_var varb 
let varc_ty = TType_var varc
let vard_ty = TType_var vard
let vare_ty = TType_var vare
let varf_ty = TType_var varf
let varg_ty = TType_var varg

type val_info = Intrinsic of nonlocal_path * string * Tast.typ
let vref_for_val_info (Intrinsic(mvr,nm,ty)) = (mk_nonlocal_ref mvr nm  : val_ref)
let vref_for_val_sinfo (mvr,nm) = (mk_nonlocal_ref mvr nm  : val_ref)

(*-------------------------------------------------------------------------
 * Access the initial environment: names
 *------------------------------------------------------------------------- *)

let lib_MF_name                     = "Microsoft.FSharp"
let lib_MFCore_name                 = "Microsoft.FSharp.Core"
let lib_MFCompatibility_name        = "Microsoft.FSharp.Compatibility"
let lib_MFOperators_name            = "Microsoft.FSharp.Core.Operators"
let lib_MFControl_name              = "Microsoft.FSharp.Control"
let lib_MFControlLazyStatus_name    = "Microsoft.FSharp.Control.LazyStatus"
let lib_MFColl_name                 = "Microsoft.FSharp.Collections"
let lib_MFLanguagePrimitives_name   = "Microsoft.FSharp.Core.LanguagePrimitives"
let lib_MFLanguagePrimitivesIntrinsicFunctions_name   = "Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicFunctions"
let lib_MFText_name                 = "Microsoft.FSharp.Text"
let lib_MFExperimental_name         = "Microsoft.FSharp.Experimental"
let lib_MFIEnumerable_name          = "Microsoft.FSharp.Collections.Seq"

let lib_MLLib_OCaml_name            = "Microsoft.FSharp.Compatibility.OCaml"
let lib_MLLib_FSharp_name           = "Microsoft.FSharp.Compatibility.FSharp"
let lib_FSLib_Pervasives_name       = "Microsoft.FSharp.Core.Pervasives" 
let lib_MLLib_Pervasives_name       = "Microsoft.FSharp.Compatibility.OCaml.Pervasives" 
let lib_Quotations_name             = "Microsoft.FSharp.Quotations"
let lib_Quotations_Typed_name       = "Microsoft.FSharp.Quotations.Typed"
let lib_MFLanguagePrimitivesHashCompare_name = "Microsoft.FSharp.Core.LanguagePrimitives.HashCompare"
let lib_MFLanguagePrimitivesIntrinsicOperators_name = "Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators"

let lib_MF_path                              = Il.split_namespace lib_MF_name 
let lib_MFCore_path                          = Il.split_namespace lib_MFCore_name 
let lib_MFCompatibility_path                          = Il.split_namespace lib_MFCompatibility_name
let lib_MFOperators_path                          = Il.split_namespace lib_MFOperators_name
let lib_MFControlLazyStatus_path             = Il.split_namespace lib_MFControlLazyStatus_name
let lib_MFControl_path                       = Il.split_namespace lib_MFControl_name
let lib_MFColl_path                          = Il.split_namespace lib_MFColl_name
let lib_MFText_path                          = Il.split_namespace lib_MFText_name
let lib_MFLanguagePrimitives_path            = Il.split_namespace lib_MFLanguagePrimitives_name
let lib_MFLanguagePrimitivesIntrinsicFunctions_path            = Il.split_namespace lib_MFLanguagePrimitivesIntrinsicFunctions_name
let lib_MFLanguagePrimitivesIntrinsicOperators_path            = Il.split_namespace lib_MFLanguagePrimitivesIntrinsicOperators_name
let lib_MFLanguagePrimitivesHashCompare_path = Il.split_namespace lib_MFLanguagePrimitivesHashCompare_name
let lib_MFExperimental_path                  = Il.split_namespace lib_MFExperimental_name
let lib_MFIEnumerable_path                   = Il.split_namespace lib_MFIEnumerable_name

let lazy_path = lib_MFControl_path
let lazystatus_path = lib_MFControlLazyStatus_path

(*-------------------------------------------------------------------------
 * Access the initial environment: helpers to build references
 *------------------------------------------------------------------------- *)

let mk_top_nlpath ccu p = NLPath(ccu,p)
let mk_mono_typ tcref = TType_app(tcref,[])
let mk_nonlocal_tcref nlpath n : tycon_ref = mk_nonlocal_ref nlpath n

let mk_MF_tcref ccu n = mk_nonlocal_tcref (mk_top_nlpath ccu lib_MF_path) n 
let mk_MFCompatibility_tcref ccu n = mk_nonlocal_tcref (mk_top_nlpath ccu lib_MFCompatibility_path) n 
let mk_MFCore_tcref ccu n = mk_nonlocal_tcref (mk_top_nlpath ccu lib_MFCore_path) n 
let mk_MFControlLazyStatus_tcref ccu n = mk_nonlocal_tcref (mk_top_nlpath ccu lib_MFControlLazyStatus_path) n 
let mk_MFColl_tcref ccu n = mk_nonlocal_tcref (mk_top_nlpath ccu lib_MFColl_path) n 
let mk_MFControl_nlpath ccu = mk_top_nlpath ccu lib_MFControl_path
let mk_MFControl_tcref ccu n = mk_nonlocal_tcref (mk_MFControl_nlpath ccu) n 
let mk_MFControl_ecref ccu n = mk_nonlocal_tcref (mk_MFControl_nlpath ccu) n 

let mk_MFLanguagePrimitives_nlpath  ccu = NLPath(ccu,lib_MFLanguagePrimitives_path)
let mk_MFLanguagePrimitivesIntrinsicOperators_nlpath  ccu = NLPath(ccu,lib_MFLanguagePrimitivesIntrinsicOperators_path)
let mk_MFLanguagePrimitivesIntrinsicFunctions_nlpath  ccu = NLPath(ccu,lib_MFLanguagePrimitivesIntrinsicFunctions_path)
let mk_MFLanguagePrimitivesHashCompare_nlpath  ccu = NLPath(ccu,lib_MFLanguagePrimitivesHashCompare_path)
let mk_MFText_nlpath ccu = NLPath(ccu,lib_MFText_path)
let mk_MFText_tcref ccu n = (mk_nonlocal_tcref (mk_MFText_nlpath ccu) n : tycon_ref)
let mk_MFExperimental_nlpath ccu = NLPath(ccu,lib_MFExperimental_path)
let mk_MFExperimental_tcref ccu n = (mk_nonlocal_tcref (mk_MFExperimental_nlpath ccu) n : tycon_ref)
let mk_MFIEnumerable_nlpath ccu = NLPath(ccu,lib_MFIEnumerable_path)
let mk_MFOperators_nlpath ccu = NLPath(ccu,lib_MFOperators_path)

let system_nlpath ccu = NLPath(ccu,["System"])
let mk_system_tcref ccu n = (mk_nonlocal_tcref (system_nlpath ccu) n : tycon_ref)
let system_tcref ccu p = let a,b = split_type_name p in mk_nonlocal_tcref (NLPath(ccu,a)) b

let mk_system_mono_typ ccu n = mk_mono_typ(mk_system_tcref ccu n)

let system_collections_generic_nlpath ccu = NLPath(ccu,["System";"Collections";"Generic"])
let systemcollections_generic_tcr ccu n = (mk_nonlocal_tcref (system_collections_generic_nlpath ccu) n : tycon_ref)

type attrib_info = AttribInfo of Il.type_ref * tycon_ref

(*-------------------------------------------------------------------------
 * Table of all these "globals"
 *------------------------------------------------------------------------- *)

(*F#
[<StructuralEquality(false); StructuralComparison(false)>]
F#*)
type tcGlobals = 
    { ilg : Il.mscorlib_refs;
      ilxPubCloEnv : Pubclo.cenv;
      compilingFslib: bool;
      fslibCcu: ccu; 
      sysCcu: ccu; 
      typeCheckerConfiguredToAssumeV20Library: bool;
      typeCheckerConfiguredToAssumeErasureOfGenerics : bool;
      better_tcref_map: tycon_ref -> tinst -> Tast.typ option;
      refcell_tcr: tycon_ref;
      option_tcr : tycon_ref;
      choice2_tcr : tycon_ref;
      choice3_tcr : tycon_ref;
      choice4_tcr : tycon_ref;
      choice5_tcr : tycon_ref;
      choice6_tcr : tycon_ref;
      choice7_tcr : tycon_ref;
      list_tcr_canon   : tycon_ref;
      lazy_tcr_canon   : tycon_ref; 
      
      (* These have a slightly different behaviour when compiling fslib *)
      (* hence they are 'methods' on the tcGlobals structure. *)

      (* modref_eq : modul_ref -> modul_ref -> bool; *)
      tcref_eq : tycon_ref       -> tycon_ref       -> bool;
      ecref_eq : tycon_ref       -> tycon_ref       -> bool;
      ucref_eq : unionconstr_ref -> unionconstr_ref -> bool;
      vref_eq  : val_ref         -> val_ref         -> bool;

      refcell_tcr_nice: tycon_ref;
      option_tcr_nice : tycon_ref;
      list_tcr_nice   : tycon_ref;
      lazy_tcr_nice   : tycon_ref; 

      format_tcr      : tycon_ref;
      lazystatus_tcr  : tycon_ref; 
      int_tcr         : tycon_ref; 
      nativeint_tcr   : tycon_ref; 
      unativeint_tcr  : tycon_ref;
      int32_tcr       : tycon_ref;
      int16_tcr       : tycon_ref;
      int64_tcr       : tycon_ref;
      uint16_tcr      : tycon_ref;
      uint32_tcr      : tycon_ref;
      uint64_tcr      : tycon_ref;
      sbyte_tcr       : tycon_ref;
      byte_tcr        : tycon_ref;
      bool_tcr        : tycon_ref;
      string_tcr      : tycon_ref;
      obj_tcr         : tycon_ref;
      unit_tcr_canon  : tycon_ref;
      unit_tcr_nice   : tycon_ref;
      exn_tcr         : tycon_ref;
      char_tcr        : tycon_ref;
      float_tcr       : tycon_ref;
      float32_tcr     : tycon_ref;
      byref_tcr       : tycon_ref;
      nativeptr_tcr   : tycon_ref;
      ilsigptr_tcr    : tycon_ref;
      array_tcr       : tycon_ref;
      seq_tcr         : tycon_ref;
      il_arr1_tcr     : tycon_ref;
      il_arr2_tcr     : tycon_ref;
      il_arr3_tcr     : tycon_ref;
      il_arr4_tcr     : tycon_ref;
      tuple2_tcr      : tycon_ref;
      tuple3_tcr      : tycon_ref;
      tuple4_tcr      : tycon_ref;
      tuple5_tcr      : tycon_ref;
      tuple6_tcr      : tycon_ref;
      tuple7_tcr      : tycon_ref;
      fslib_IEvent_tcr: tycon_ref;
      fslib_IDelegateEvent_tcr         : tycon_ref;
      fslib_IPrimitiveDelegateEvent_tcr: tycon_ref;
      system_Nullable_tcref            : tycon_ref; 
      system_IndexOutOfRangeException_tcref : tycon_ref;
      int_ty         : Tast.typ;
      nativeint_ty   : Tast.typ; 
      unativeint_ty  : Tast.typ; 
      int32_ty       : Tast.typ; 
      int16_ty       : Tast.typ; 
      int64_ty       : Tast.typ; 
      uint16_ty      : Tast.typ; 
      uint32_ty      : Tast.typ; 
      uint64_ty      : Tast.typ; 
      sbyte_ty       : Tast.typ; 
      byte_ty        : Tast.typ; 
      bool_ty        : Tast.typ; 
      string_ty      : Tast.typ; 
      obj_ty         : Tast.typ; 
      unit_ty        : Tast.typ; 
      exn_ty         : Tast.typ; 
      char_ty        : Tast.typ; 
      decimal_ty                   : Tast.typ; 
      bigint_ty                    : Tast.typ; 
      bignum_ty                    : Tast.typ; 
      float_ty                     : Tast.typ; 
      float32_ty                   : Tast.typ; 
      system_Array_typ             : Tast.typ; 
      system_Object_typ            : Tast.typ; 
      system_IDisposable_typ       : Tast.typ; 
      system_Value_typ             : Tast.typ; 
      system_Enum_typ              : Tast.typ; 
      system_Exception_typ         : Tast.typ; 
      system_Int32_typ             : Tast.typ; 
      system_String_typ            : Tast.typ; 
      system_Type_typ              : Tast.typ; 
      system_RuntimeTypeHandle_typ : Tast.typ; 
      mk_IComparable_ty            : Tast.typ;
      mk_IStructuralHash_ty        : Tast.typ;
      attrib_AttributeUsageAttribute   : attrib_info;
      attrib_SystemObsolete            : attrib_info;
      attrib_DllImportAttribute        : attrib_info;
      attrib_NonSerializedAttribute    : attrib_info;
      attrib_AutoSerializableAttribute    : attrib_info;
      attrib_StructLayoutAttribute     : attrib_info;
      attrib_FieldOffsetAttribute      : attrib_info;
      attrib_MarshalAsAttribute        : attrib_info;
      attrib_InAttribute               : attrib_info;
      attrib_OutAttribute              : attrib_info;
      attrib_OptionalAttribute         : attrib_info;
      attrib_ThreadStaticAttribute     : attrib_info;
      attrib_ContextStaticAttribute    : attrib_info;
      attrib_FlagsAttribute            : attrib_info;
      attrib_DefaultMemberAttribute    : attrib_info;
      tcref_System_Collections_Generic_IList       : tycon_ref;
      tcref_System_Collections_Generic_ICollection : tycon_ref;
      tcref_System_Collections_Generic_IEnumerable : tycon_ref;
      tcref_System_Collections_Generic_IEnumerator : tycon_ref;

      attrib_OverloadIDAttribute                    : attrib_info; 
      attrib_EntryPointAttribute                    : attrib_info; 
      attrib_DefaultAugmentationAttribute           : attrib_info; 
      attrib_OCamlCompatibilityAttribute            : attrib_info; 
      attrib_ExperimentalAttribute                  : attrib_info; 
      attrib_LiteralAttribute                       : attrib_info; 
      attrib_ConditionalAttribute                   : attrib_info; 
      attrib_OptionalArgumentAttribute              : attrib_info; 
      attrib_RequiresExplicitTypeArgumentsAttribute : attrib_info; 
      attrib_DefaultValueAttribute                  : attrib_info; 
      attrib_ClassAttribute                         : attrib_info; 
      attrib_InterfaceAttribute                     : attrib_info; 
      attrib_StructAttribute                        : attrib_info; 
      attrib_ReflectedDefinitionAttribute           : attrib_info; 
      attrib_CompilationRepresentationAttribute     : attrib_info; 
      attrib_ReferenceEqualityAttribute             : attrib_info; 
      attrib_StructuralEqualityAttribute            : attrib_info; 
      attrib_StructuralComparisonAttribute          : attrib_info; 
      attrib_SealedAttribute                        : attrib_info; 
      attrib_AbstractClassAttribute                 : attrib_info; 
      attrib_GeneralizableValueAttribute           : attrib_info;

      
      cons_ucref : unionconstr_ref;
      nil_ucref : unionconstr_ref;
      (* These are the library values the compiler needs to know about *)
      (* These 'forward references' from the compiler into the library *)
      (* are a right pain. *)
      unary_neg_vref            : val_ref;
      and_vref                  : val_ref;
      and2_vref                 : val_ref;
      addrof_vref               : val_ref;
      addrof2_vref              : val_ref;
      or_vref                   : val_ref;
      or2_vref                  : val_ref;
      poly_equals_inner_vref    : val_ref;
      poly_compare_inner_vref   : val_ref;
      poly_hash_param_inner_vref: val_ref;
      poly_eq_inner_vref        : val_ref;
      bitwise_or_vref           : val_ref;
      typeof_info               : val_info;
      typeof_vref               : val_ref;
      typedefof_vref            : val_ref;
      enum_vref                 : val_ref;
      string_to_decimal_info    : val_info;
      string_to_bigint_info     : val_info;
      string_to_bignum_info     : val_info;
      poly_compare_outer_info   : val_info;
      poly_equals_outer_info    : val_info;
      poly_hash_param_outer_info: val_info;
      create_instance_info      : val_info;
      unbox_vref                : val_ref;
      unbox_fast_vref           : val_ref;
      istype_vref               : val_ref;
      istype_fast_vref          : val_ref;
      unbox_info                : val_info;
      unbox_fast_info           : val_info;
      istype_info               : val_info;
      istype_fast_info          : val_info;
      range_op_vref             : val_ref;
      idx_lookup_op_vref        : val_ref;
      array_lookup_op_vref      : val_ref;
      seq_map_concat_info       : val_info;
      seq_using_info            : val_info;
      seq_delay_info            : val_info;
      seq_append_info           : val_info;
      seq_generated_info        : val_info;
      seq_finally_info        : val_info;
      seq_of_functions_info     : val_info;
      seq_to_array_info         : val_info;
      seq_to_list_info          : val_info;
      seq_map_info              : val_info;
      seq_singleton_info        : val_info;
      seq_empty_info            : val_info;
      new_format_info            : val_info;
      
 } 


let mk_tcGlobals (compilingFslib,sysCcu,ilg,fslibCcu,generics) = 
  let int_tcr        = mk_MFCore_tcref fslibCcu "int" in
  let nativeint_tcr  = mk_MFCore_tcref fslibCcu "nativeint" in
  let unativeint_tcr = mk_MFCore_tcref fslibCcu "unativeint" in
  let int32_tcr      = mk_MFCore_tcref fslibCcu "int32" in
  let int16_tcr      = mk_MFCore_tcref fslibCcu "int16" in
  let int64_tcr      = mk_MFCore_tcref fslibCcu "int64" in
  let uint16_tcr     = mk_MFCore_tcref fslibCcu "uint16" in
  let uint32_tcr     = mk_MFCore_tcref fslibCcu "uint32" in
  let uint64_tcr     = mk_MFCore_tcref fslibCcu "uint64" in
  let sbyte_tcr      = mk_MFCore_tcref fslibCcu "sbyte" in
  let byte_tcr       = mk_MFCore_tcref fslibCcu "byte" in
  let bool_tcr       = mk_MFCore_tcref fslibCcu "bool" in
  let string_tcr     = mk_MFCore_tcref fslibCcu "string" in
  let obj_tcr        = mk_MFCore_tcref fslibCcu "obj" in
  let unit_tcr_canon       = mk_MFCore_tcref fslibCcu "Unit" in
  let unit_tcr_nice       = mk_MFCore_tcref fslibCcu "unit" in
  let exn_tcr        = mk_MFCore_tcref fslibCcu "exn" in
  let char_tcr       = mk_MFCore_tcref fslibCcu "char" in
  let float_tcr      = mk_MFCore_tcref fslibCcu "float" in
  let float32_tcr    = mk_MFCore_tcref fslibCcu "float32" in
  let byref_tcr      = mk_MFCore_tcref fslibCcu "byref`1" in
  let nativeptr_tcr  = mk_MFCore_tcref fslibCcu "nativeptr`1" in
  let ilsigptr_tcr  = mk_MFCore_tcref fslibCcu "ilsigptr`1" in
  let array_tcr      = mk_MFCore_tcref fslibCcu "array`1" in
  let fastFunc_tcr   = mk_MFCore_tcref fslibCcu "FastFunc`2" in
  let fslib_IEvent_tcr = mk_MFControl_tcref fslibCcu "IEvent`1" in
  let fslib_IDelegateEvent_tcr = mk_MFControl_tcref fslibCcu "IDelegateEvent`2" in
  let fslib_IPrimitiveDelegateEvent_tcr = mk_MFControl_tcref fslibCcu "IPrimitiveDelegateEvent`1" in
  let option_tcr_nice     = mk_MFCore_tcref fslibCcu "option`1" in
  let list_tcr_not_nice   = mk_MFColl_tcref fslibCcu "List`1" in
  let list_tcr_nice       = mk_MFColl_tcref fslibCcu "list`1" in
  let seq_tcr       = mk_MFColl_tcref fslibCcu "seq`1" in
  let format_tcr = mk_MFText_tcref     fslibCcu "Format`5" in 
  let tcr_IEnumerable = 
      if generics then systemcollections_generic_tcr sysCcu "IEnumerable`1" 
      else mk_MFCompatibility_tcref fslibCcu "IEnumerable`1"  in
  
  let bigint_tcr     = mk_MFCore_tcref fslibCcu "bigint" in
  let bignum_tcr     = mk_MFCore_tcref fslibCcu "bignum"  in
  let bool_ty        = mk_mono_typ bool_tcr   in 
  let int_ty         = mk_mono_typ int_tcr    in 
  let obj_ty         = mk_mono_typ obj_tcr    in 
  let string_ty      = mk_mono_typ string_tcr in
  let bigint_ty      = mk_mono_typ bigint_tcr in
  let bignum_ty      = mk_mono_typ bignum_tcr in
  let decimal_ty     = mk_system_mono_typ sysCcu "Decimal" in
  let unit_ty = mk_mono_typ unit_tcr_nice in 
  let systemType_ty =mk_system_mono_typ sysCcu "Type" in 
  
  (* local helpers to build value infos *)
  let mk_byref_typ ty = TType_app(byref_tcr, [ty]) in 
  let mk_nativeptr_ty ty = TType_app(nativeptr_tcr, [ty]) in 
  let mk_fun_ty d r = TType_fun (d,r) in 
  let (-->) d r = mk_fun_ty d r in
  let mk_forall_ty d r = TType_forall (d,r) in
  let try_mk_forall_ty d r = if d = [] then r else mk_forall_ty d r in 
  let (+->) d r = try_mk_forall_ty d r in 
  let mk_tuple_ty l = TType_tuple l in 
  let mk_binop_ty ty   = ty --> (ty --> ty) in 
  let mk_rel_ty ty   = ty --> (ty --> bool_ty) in 
  let mk_compare_ty ty = ty --> (ty --> int_ty) in
  let mk_hash_param_ty ty = ty --> (mk_byref_typ int_ty  --> int_ty) in 
  let mk_option_ty ty = TType_app(option_tcr_nice,[ty]) in
  let mk_list_ty ty = TType_app(list_tcr_nice,[ty]) in
  let mk_seq_ty ty1 = TType_app(seq_tcr,[ty1]) in
  let mk_array_ty ty = TType_app (array_tcr, [ty]) in
  
  let mk_format_ty aty bty cty dty ety = TType_app(format_tcr, [aty;bty;cty;dty; ety]) in
  let cons_ucref = mk_ucref list_tcr_not_nice "op_ColonColon" in
  let nil_ucref = mk_ucref list_tcr_not_nice "op_Nil" in

  (* value infos *)
  let fslib_MFLanguagePrimitives_nlpath = mk_MFLanguagePrimitives_nlpath fslibCcu in 
  let fslib_MFIEnumerable_nlpath = mk_MFIEnumerable_nlpath fslibCcu in 
  let fslib_MFText_nlpath = mk_MFText_nlpath fslibCcu in 
  let fslib_MFOperators_nlpath = mk_MFOperators_nlpath fslibCcu in 
  let fslib_MFLanguagePrimitivesIntrinsicOperators_nlpath = mk_MFLanguagePrimitivesIntrinsicOperators_nlpath fslibCcu in 
  let fslib_MFLanguagePrimitivesIntrinsicFunctions_nlpath = mk_MFLanguagePrimitivesIntrinsicFunctions_nlpath fslibCcu in 
  let fslib_MFLanguagePrimitivesHashCompare_nlpath = mk_MFLanguagePrimitivesHashCompare_nlpath fslibCcu in 
  let mfmN = mk_top_nlpath fslibCcu (lib_MF_path @ ["Math";"BigNum"]) in 
  let mfmI = mk_top_nlpath fslibCcu (lib_MF_path @ ["Math";"BigInt"]) in 

  let unary_neg_sinfo =            fslib_MFLanguagePrimitivesIntrinsicOperators_nlpath, (compileOpName "~-")         in
  let and_info =                   Intrinsic(fslib_MFLanguagePrimitivesIntrinsicOperators_nlpath, (compileOpName "&")        ,(mk_rel_ty bool_ty))  in
  let addrof_info =                Intrinsic(fslib_MFLanguagePrimitivesIntrinsicOperators_nlpath, (compileOpName "~&")        ,([vara] +-> (vara_ty --> mk_byref_typ vara_ty)))   in 
  let addrof2_info =               Intrinsic(fslib_MFLanguagePrimitivesIntrinsicOperators_nlpath, (compileOpName "~&&")        ,([vara] +-> (vara_ty --> mk_nativeptr_ty vara_ty)))   in 
  let and2_info =                  Intrinsic(fslib_MFLanguagePrimitivesIntrinsicOperators_nlpath, (compileOpName "&&")         ,(mk_rel_ty bool_ty) )  in
  let or_info =                    Intrinsic(fslib_MFLanguagePrimitivesIntrinsicOperators_nlpath, "or"                         ,(mk_rel_ty bool_ty))  in
  let or2_info =                   Intrinsic(fslib_MFLanguagePrimitivesIntrinsicOperators_nlpath, (compileOpName "||")         ,(mk_rel_ty bool_ty))  in
  let poly_compare_outer_info =    Intrinsic(fslib_MFLanguagePrimitives_nlpath, "GenericComparison"          ,([vara] +-> mk_compare_ty vara_ty))  in
  let poly_equals_outer_info =     Intrinsic(fslib_MFLanguagePrimitives_nlpath, "GenericEquality"          ,([vara] +-> mk_rel_ty vara_ty))  in
  let poly_hash_param_outer_info = Intrinsic(fslib_MFLanguagePrimitives_nlpath, "GenericHashParam"    ,([vara] +-> mk_hash_param_ty vara_ty))  in
  let create_instance_info       = Intrinsic(fslib_MFLanguagePrimitivesIntrinsicFunctions_nlpath, "CreateInstance"    ,([vara] +-> (unit_ty --> vara_ty)))  in
  let unbox_info                 = Intrinsic(fslib_MFLanguagePrimitivesIntrinsicFunctions_nlpath, "UnboxGeneric"    ,([vara] +-> (obj_ty --> vara_ty)))  in
  let unbox_fast_info            = Intrinsic(fslib_MFLanguagePrimitivesIntrinsicFunctions_nlpath, "UnboxFast"    ,([vara] +-> (obj_ty --> vara_ty)))  in
  let istype_info                = Intrinsic(fslib_MFLanguagePrimitivesIntrinsicFunctions_nlpath, "TypeTestGeneric"    ,([vara] +-> (obj_ty --> bool_ty)))  in
  let istype_fast_info           = Intrinsic(fslib_MFLanguagePrimitivesIntrinsicFunctions_nlpath, "TypeTestFast"    ,([vara] +-> (obj_ty --> bool_ty)))  in

  let poly_equals_inner_info     = Intrinsic(fslib_MFLanguagePrimitivesHashCompare_nlpath, "GenericEqualityIntrinsic"  ,([vara]                +-> mk_rel_ty vara_ty))  in
  let poly_compare_inner_info    = Intrinsic(fslib_MFLanguagePrimitivesHashCompare_nlpath, "GenericComparisonIntrinsic"  ,([vara]                +-> mk_compare_ty vara_ty))  in
  let poly_hash_param_inner_info = Intrinsic(fslib_MFLanguagePrimitivesHashCompare_nlpath, "GenericHashParamIntrinsic"   ,([vara]                +-> mk_hash_param_ty vara_ty))  in
  let poly_eq_inner_info         = Intrinsic(fslib_MFLanguagePrimitivesHashCompare_nlpath, "PhysicalEqualityIntrinsic"      ,([vara]                +-> mk_rel_ty vara_ty))  in
  let bitwise_or_info            = Intrinsic(fslib_MFOperators_nlpath, "op_BitwiseOr"      ,([vara]                +-> mk_binop_ty vara_ty))  in
  let typeof_info                = Intrinsic(fslib_MFOperators_nlpath, "typeof"            ,([vara]                +-> systemType_ty))  in
  let typedefof_info             = Intrinsic(fslib_MFOperators_nlpath, "typedefof"         ,([vara]                +-> systemType_ty))  in
  let enum_info                  = Intrinsic(fslib_MFOperators_nlpath, "enum"              ,([vara]                +-> (int_ty --> vara_ty)))  in
  let range_op_info              = Intrinsic(fslib_MFOperators_nlpath, "op_Range"      ,([vara]                +-> (vara_ty --> (vara_ty --> mk_seq_ty vara_ty))))  in
  let idx_lookup_op_info         = Intrinsic(fslib_MFOperators_nlpath, "op_IndexedLookup",([vara;varb;varc] +-> (vara_ty --> (varb_ty --> varc_ty))))  in
  let array_lookup_op_info       = Intrinsic(fslib_MFOperators_nlpath, "op_ArrayLookup",([vara;] +-> (mk_array_ty vara_ty --> (int_ty --> vara_ty))))  in
  let array_lookup_op_info       = Intrinsic(fslib_MFOperators_nlpath, "op_ArrayLookup",([vara;] +-> (mk_array_ty vara_ty --> (int_ty --> vara_ty))))  in
  let seq_map_concat_info        = Intrinsic(fslib_MFIEnumerable_nlpath,                   "map_concat"                   ,([vara;varb;varc;vard] +-> ((vara_ty --> varb_ty) --> (vard_ty --> mk_seq_ty varc_ty))))  in
  let seq_using_info             = Intrinsic(fslib_MFIEnumerable_nlpath,                   "using"                   ,([vara;varb;varc] +-> (vara_ty --> ((vara_ty --> varb_ty) --> mk_seq_ty varc_ty))))  in
  let seq_delay_info             = Intrinsic(fslib_MFIEnumerable_nlpath,                   "delay"                   ,([vara;varb] +-> ((unit_ty --> vara_ty) --> mk_seq_ty varb_ty)))  in
  let seq_append_info            = Intrinsic(fslib_MFIEnumerable_nlpath,                   "append"                   ,([vara;varb;varc] +-> (vara_ty --> (varc_ty --> mk_seq_ty varb_ty))))  in
  let seq_generated_info         = Intrinsic(fslib_MFIEnumerable_nlpath,                   "generated"                   ,([vara;varb] +-> ((unit_ty --> bool_ty) --> (vara_ty --> mk_seq_ty varb_ty))))  in
  let seq_finally_info           = Intrinsic(fslib_MFIEnumerable_nlpath,                   "do_finally"                   ,([vara;varb] +-> (vara_ty --> ((unit_ty --> unit_ty) --> mk_seq_ty varb_ty))))  in
  let seq_of_functions_info      = Intrinsic(fslib_MFIEnumerable_nlpath,                   "of_functions"                 ,([vara;varb]           +-> ((unit_ty --> vara_ty)  --> ((vara_ty --> bool_ty) --> ((vara_ty --> varb_ty) --> mk_seq_ty varb_ty)))))  in
  let seq_to_array_info          = Intrinsic(fslib_MFIEnumerable_nlpath,                   "to_array"                      ,([vara;varb]                +-> (vara_ty  --> mk_array_ty varb_ty)))  in
  let seq_to_list_info           = Intrinsic(fslib_MFIEnumerable_nlpath,                   "to_list"                      ,([vara;varb]                +-> (vara_ty  --> mk_list_ty varb_ty)))  in
  let seq_map_info               = Intrinsic(fslib_MFIEnumerable_nlpath,                   "map"                          ,([vara;varb;varc]      +-> ((vara_ty --> varb_ty) --> (varc_ty --> mk_seq_ty varb_ty))))  in
  let seq_singleton_info         = Intrinsic(fslib_MFIEnumerable_nlpath,                   "singleton"                    ,([vara]                +-> (vara_ty --> (mk_seq_ty vara_ty))))  in
  let seq_empty_info             = Intrinsic(fslib_MFIEnumerable_nlpath,                   "empty"                        ,([vara]                +-> (mk_seq_ty vara_ty)))  in
  let new_format_info            = Intrinsic(fslib_MFText_nlpath,                   "Format`5.ctor.1"               ,([vara;varb;varc;vard;vare]                +-> (string_ty --> mk_format_ty vara_ty varb_ty varc_ty vard_ty vare_ty)))  in
  let string_to_bignum_info      = Intrinsic(mfmN, "of_string"                ,(string_ty --> bignum_ty))  in
  let string_to_bigint_info      = Intrinsic(mfmI, "of_string"                ,(string_ty --> bigint_ty))  in
  let string_to_decimal_info     = Intrinsic(fslib_MFLanguagePrimitivesIntrinsicFunctions_nlpath, "ParseDecimal"                ,(string_ty --> decimal_ty))  in
  let tuple2_tcr      = mk_MFCore_tcref fslibCcu "Tuple`2" in
  let tuple3_tcr      = mk_MFCore_tcref fslibCcu "Tuple`3" in
  let tuple4_tcr      = mk_MFCore_tcref fslibCcu "Tuple`4" in
  let tuple5_tcr      = mk_MFCore_tcref fslibCcu "Tuple`5" in
  let tuple6_tcr      = mk_MFCore_tcref fslibCcu "Tuple`6" in
  let tuple7_tcr      = mk_MFCore_tcref fslibCcu "Tuple`7" in
  let choice2_tcr     = mk_MFCore_tcref fslibCcu "Choice`2" in
  let choice3_tcr     = mk_MFCore_tcref fslibCcu "Choice`3" in
  let choice4_tcr     = mk_MFCore_tcref fslibCcu "Choice`4" in
  let choice5_tcr     = mk_MFCore_tcref fslibCcu "Choice`5" in
  let choice6_tcr     = mk_MFCore_tcref fslibCcu "Choice`6" in
  let choice7_tcr     = mk_MFCore_tcref fslibCcu "Choice`7" in
  let tcref_eq = prim_tcref_eq compilingFslib fslibCcu in 
  let vref_eq  = prim_vref_eq compilingFslib fslibCcu in 
  let ecref_eq = prim_tcref_eq compilingFslib fslibCcu in 
  let ucref_eq = prim_ucref_eq compilingFslib fslibCcu in 
  (* let modref_eq = prim_modref_eq compilingFslib fslibCcu in  *)

  let mk_MFCore_attrib nm : attrib_info = 
      let tname = nm ^ "." ^ nm in
      AttribInfo(mk_tref(Msilxlib.scoref (), nm),mk_MFCore_tcref fslibCcu nm)  in

  let mk_mscorlib_attrib nm : attrib_info = 
      AttribInfo(mk_tref (ilg.mscorlib_scoref,nm), system_tcref sysCcu nm) in
    
  { ilg=ilg;
    ilxPubCloEnv=Pubclo.new_cenv(ilg,None);
    compilingFslib=compilingFslib;
    (* modref_eq = modref_eq; *)
    tcref_eq = tcref_eq;
    ecref_eq = ecref_eq;
    ucref_eq = ucref_eq;
    vref_eq = vref_eq;
    fslibCcu       = fslibCcu;
    sysCcu         = sysCcu;
    typeCheckerConfiguredToAssumeV20Library = generics;
    typeCheckerConfiguredToAssumeErasureOfGenerics = not generics;
    refcell_tcr       = mk_MFCore_tcref     fslibCcu "Ref`1";
    option_tcr        = mk_MFCore_tcref     fslibCcu "Option`1";
    list_tcr_canon    = mk_MFColl_tcref     fslibCcu "List`1";
    lazy_tcr_canon    = mk_MFControl_tcref  fslibCcu "Lazy`1";
    refcell_tcr_nice  = mk_MFCore_tcref     fslibCcu "ref`1";
    option_tcr_nice   = mk_MFCore_tcref     fslibCcu "option`1";
    list_tcr_nice     = mk_MFColl_tcref     fslibCcu "list`1";
    lazy_tcr_nice     = mk_MFControl_tcref  fslibCcu "lazy`1";
    format_tcr       = format_tcr;
    lazystatus_tcr    = mk_MFControlLazyStatus_tcref  fslibCcu "LazyStatus`1";
    int_tcr        = int_tcr;
    nativeint_tcr  = nativeint_tcr;
    unativeint_tcr = unativeint_tcr;
    int32_tcr      = int32_tcr;
    int16_tcr      = int16_tcr;
    int64_tcr      = int64_tcr;
    uint16_tcr     = uint16_tcr;
    uint32_tcr     = uint32_tcr;
    uint64_tcr     = uint64_tcr;
    sbyte_tcr      = sbyte_tcr;
    byte_tcr       = byte_tcr;
    bool_tcr       = bool_tcr;
    string_tcr     = string_tcr;
    obj_tcr        = obj_tcr;
    unit_tcr_canon       = unit_tcr_canon;
    unit_tcr_nice       = unit_tcr_nice;
    exn_tcr        = exn_tcr;
    char_tcr       = char_tcr;
    float_tcr      = float_tcr;
    float32_tcr    = float32_tcr;
    byref_tcr      = byref_tcr;
    nativeptr_tcr      = nativeptr_tcr;
    ilsigptr_tcr      = ilsigptr_tcr;
    fslib_IEvent_tcr      = fslib_IEvent_tcr;
    fslib_IDelegateEvent_tcr      = fslib_IDelegateEvent_tcr;
    fslib_IPrimitiveDelegateEvent_tcr      = fslib_IPrimitiveDelegateEvent_tcr;
    array_tcr      = array_tcr;
    seq_tcr        = seq_tcr;
    (* iPattern_tcr     = iPattern_tcr; *)
    (* iPartialPattern_tcr     = iPartialPattern_tcr; *)
    il_arr1_tcr    = mk_MFCore_tcref fslibCcu "[]`1";
    il_arr2_tcr    = mk_MFCore_tcref fslibCcu "[,]`1";
    il_arr3_tcr    = mk_MFCore_tcref fslibCcu "[,,]`1";
    il_arr4_tcr    = mk_MFCore_tcref fslibCcu "[,,,]`1";
    tuple2_tcr     = tuple2_tcr;
    tuple3_tcr     = tuple3_tcr;
    tuple4_tcr     = tuple4_tcr;
    tuple5_tcr     = tuple5_tcr;
    tuple6_tcr     = tuple6_tcr;
    tuple7_tcr     = tuple7_tcr;
    choice2_tcr    = choice2_tcr;
    choice3_tcr    = choice3_tcr;
    choice4_tcr    = choice4_tcr;
    choice5_tcr    = choice5_tcr;
    choice6_tcr    = choice6_tcr;
    choice7_tcr    = choice7_tcr;
    int_ty        = int_ty;
    nativeint_ty  = mk_mono_typ nativeint_tcr;
    unativeint_ty = mk_mono_typ unativeint_tcr;
    int32_ty      = mk_mono_typ int32_tcr;
    int16_ty      = mk_mono_typ int16_tcr;
    int64_ty      = mk_mono_typ int64_tcr;
    uint16_ty     = mk_mono_typ uint16_tcr;
    uint32_ty     = mk_mono_typ uint32_tcr;
    uint64_ty     = mk_mono_typ uint64_tcr;
    sbyte_ty      = mk_mono_typ sbyte_tcr;
    byte_ty       = mk_mono_typ byte_tcr;
    bigint_ty     = bigint_ty;
    bignum_ty     = bignum_ty;
    bool_ty       = bool_ty;
    string_ty     = string_ty;
    obj_ty        = mk_mono_typ obj_tcr;
    unit_ty       = unit_ty;
    exn_ty        = mk_mono_typ exn_tcr;
    char_ty       = mk_mono_typ char_tcr;
    decimal_ty    = decimal_ty;
    float_ty      = mk_mono_typ float_tcr;
    float32_ty    = mk_mono_typ float32_tcr;

    mk_IStructuralHash_ty =
       (let tcref_IStructuralHash = mk_MFCore_tcref fslibCcu "IStructuralHash" in 
       TType_app(tcref_IStructuralHash,[]));

    system_Array_typ     = mk_system_mono_typ sysCcu "Array";
    system_Object_typ    = mk_system_mono_typ sysCcu "Object";
    system_IDisposable_typ    = mk_system_mono_typ sysCcu "IDisposable";
    system_Value_typ     = mk_system_mono_typ sysCcu "ValueType";
    system_Enum_typ      = mk_system_mono_typ sysCcu "Enum";
    system_Exception_typ = mk_system_mono_typ sysCcu "Exception";
    system_String_typ    = mk_system_mono_typ sysCcu "String";
    system_Int32_typ     = mk_system_mono_typ sysCcu "Int32";
    system_Type_typ      = systemType_ty;
    system_RuntimeTypeHandle_typ = mk_system_mono_typ sysCcu "RuntimeTypeHandle";
    system_IndexOutOfRangeException_tcref    = mk_system_tcref sysCcu "IndexOutOfRangeException";
    system_Nullable_tcref = mk_system_tcref sysCcu "Nullable`1";
    mk_IComparable_ty    = mk_system_mono_typ sysCcu "IComparable";

    tcref_System_Collections_Generic_IList       = systemcollections_generic_tcr sysCcu "IList`1";
    tcref_System_Collections_Generic_ICollection = systemcollections_generic_tcr sysCcu "ICollection`1";
    tcref_System_Collections_Generic_IEnumerable = tcr_IEnumerable;
    tcref_System_Collections_Generic_IEnumerator = systemcollections_generic_tcr sysCcu "IEnumerator`1";

    attrib_AttributeUsageAttribute = mk_mscorlib_attrib "System.AttributeUsageAttribute";
    attrib_SystemObsolete          = mk_mscorlib_attrib "System.ObsoleteAttribute";
    attrib_DllImportAttribute      = mk_mscorlib_attrib "System.Runtime.InteropServices.DllImportAttribute";
    attrib_StructLayoutAttribute   = mk_mscorlib_attrib "System.Runtime.InteropServices.StructLayoutAttribute";
    attrib_FieldOffsetAttribute    = mk_mscorlib_attrib "System.Runtime.InteropServices.FieldOffsetAttribute" ;
    attrib_MarshalAsAttribute      = mk_mscorlib_attrib "System.Runtime.InteropServices.MarshalAsAttribute";
    attrib_InAttribute             = mk_mscorlib_attrib "System.Runtime.InteropServices.InAttribute" ;
    attrib_OutAttribute            = mk_mscorlib_attrib "System.Runtime.InteropServices.OutAttribute" ;
    attrib_OptionalAttribute       = mk_mscorlib_attrib "System.Runtime.InteropServices.OptionalAttribute" ;
    attrib_ThreadStaticAttribute   = mk_mscorlib_attrib "System.ThreadStaticAttribute";
    attrib_ContextStaticAttribute  = mk_mscorlib_attrib "System.ContextStaticAttribute";
    attrib_FlagsAttribute          = mk_mscorlib_attrib "System.FlagsAttribute";
    attrib_DefaultMemberAttribute  = mk_mscorlib_attrib "System.Reflection.DefaultMemberAttribute";

    attrib_NonSerializedAttribute  = mk_mscorlib_attrib "System.NonSerializedAttribute";
    attrib_AutoSerializableAttribute  = mk_MFCore_attrib "AutoSerializableAttribute";
    attrib_OverloadIDAttribute                    = mk_MFCore_attrib "OverloadIDAttribute";
    attrib_EntryPointAttribute                    = mk_MFCore_attrib "EntryPointAttribute";
    attrib_DefaultAugmentationAttribute           = mk_MFCore_attrib "DefaultAugmentationAttribute";
    attrib_OCamlCompatibilityAttribute            = mk_MFCore_attrib "OCamlCompatibilityAttribute";
    attrib_ExperimentalAttribute                  = mk_MFCore_attrib "ExperimentalAttribute";
    attrib_LiteralAttribute                       = mk_MFCore_attrib "LiteralAttribute";
    attrib_ConditionalAttribute                       = mk_mscorlib_attrib "System.Diagnostics.ConditionalAttribute";
    attrib_OptionalArgumentAttribute              = mk_MFCore_attrib "OptionalArgumentAttribute";
    attrib_RequiresExplicitTypeArgumentsAttribute = mk_MFCore_attrib "RequiresExplicitTypeArgumentsAttribute";
    attrib_DefaultValueAttribute                  = mk_MFCore_attrib "DefaultValueAttribute";
    attrib_ClassAttribute                         = mk_MFCore_attrib "ClassAttribute";
    attrib_InterfaceAttribute                     = mk_MFCore_attrib "InterfaceAttribute";
    attrib_StructAttribute                        = mk_MFCore_attrib "StructAttribute";
    attrib_ReflectedDefinitionAttribute           = mk_MFCore_attrib "ReflectedDefinitionAttribute";
    attrib_CompilationRepresentationAttribute     = mk_MFCore_attrib "CompilationRepresentationAttribute";
    attrib_ReferenceEqualityAttribute             = mk_MFCore_attrib "ReferenceEqualityAttribute";
    attrib_StructuralEqualityAttribute            = mk_MFCore_attrib "StructuralEqualityAttribute";
    attrib_StructuralComparisonAttribute          = mk_MFCore_attrib "StructuralComparisonAttribute";
    attrib_SealedAttribute                        = mk_MFCore_attrib "SealedAttribute";
    attrib_AbstractClassAttribute                        = mk_MFCore_attrib "AbstractClassAttribute";
    attrib_GeneralizableValueAttribute           = mk_MFCore_attrib "GeneralizableValueAttribute";


    better_tcref_map = 
       begin 
        let entries1 = 
         [ "Int32", int_tcr; 
           "IntPtr", nativeint_tcr; 
           "UIntPtr", unativeint_tcr;
           "Int16",int16_tcr; 
           "Int64",int64_tcr; 
           "UInt16",uint16_tcr;
           "UInt32",uint32_tcr;
           "UInt64",uint64_tcr;
           "SByte",sbyte_tcr;
           "Byte",byte_tcr;
           "Boolean",bool_tcr;
           "String",string_tcr;
           "Object",obj_tcr;
           "Exception",exn_tcr;
           "Char",char_tcr;
           "Double",float_tcr;
           "Single",float32_tcr; ] 
             |> List.map (fun (nm,tcr) -> 
                  let ty = mk_mono_typ tcr in 
                  let sys_tcr = mk_system_tcref sysCcu nm in 
                  nm,(fun x _ -> if tcref_eq x sys_tcr then Some(ty) else None)) in
        let entries2 =
          if generics then 
            [ "IEnumerable`2", (fun x tinst -> if tcref_eq tcr_IEnumerable x then Some(mk_seq_ty (List.nth tinst 0)) else None);
              "FastFunc`2", (fun x tinst -> if tcref_eq fastFunc_tcr x then Some(mk_fun_ty (List.nth tinst 0) (List.nth tinst 1)) else None);
              "Tuple`2", (fun x tinst -> if tcref_eq x tuple2_tcr then Some(mk_tuple_ty tinst) else None);
              "Tuple`3", (fun x tinst -> if tcref_eq x tuple3_tcr then Some(mk_tuple_ty tinst) else None);
              "Tuple`4", (fun x tinst -> if tcref_eq x tuple4_tcr then Some(mk_tuple_ty tinst) else None);
              "Tuple`5", (fun x tinst -> if tcref_eq x tuple5_tcr then Some(mk_tuple_ty tinst) else None);
              "Tuple`6", (fun x tinst -> if tcref_eq x tuple6_tcr then Some(mk_tuple_ty tinst) else None); ]
          else [] in 
            
        let map = Namemap.of_list (entries1 @ entries2) in 
        (fun tcref tinst -> 
           let nm = (name_of_tcref tcref) in 
           if Map.mem nm map then (Map.find nm map) tcref tinst else None)
       end;
           
      string_to_decimal_info = string_to_decimal_info;
      string_to_bigint_info = string_to_bigint_info;
      string_to_bignum_info = string_to_bignum_info;
      unary_neg_vref = (vref_for_val_sinfo unary_neg_sinfo) ;
      and_vref    = (vref_for_val_info and_info) ;
      and2_vref   = (vref_for_val_info and2_info);
      addrof_vref = (vref_for_val_info addrof_info);
      addrof2_vref = (vref_for_val_info addrof2_info);
      or_vref     = (vref_for_val_info or_info);
      or2_vref    = (vref_for_val_info or2_info); 
      poly_equals_inner_vref     = vref_for_val_info poly_equals_inner_info;
      poly_compare_inner_vref    = vref_for_val_info poly_compare_inner_info;
      poly_hash_param_inner_vref = vref_for_val_info poly_hash_param_inner_info;
      poly_eq_inner_vref         = vref_for_val_info poly_eq_inner_info;
      bitwise_or_vref            = vref_for_val_info bitwise_or_info;
      typeof_info                = typeof_info;
      typeof_vref                = vref_for_val_info typeof_info;
      typedefof_vref             = vref_for_val_info typedefof_info;
      enum_vref                  = vref_for_val_info enum_info;
      range_op_vref              = vref_for_val_info range_op_info;
      idx_lookup_op_vref         = vref_for_val_info idx_lookup_op_info;
      array_lookup_op_vref       = vref_for_val_info array_lookup_op_info;
      poly_compare_outer_info    = poly_compare_outer_info;
      poly_equals_outer_info     = poly_equals_outer_info;
      poly_hash_param_outer_info = poly_hash_param_outer_info;
      seq_map_concat_info        = seq_map_concat_info;
      seq_using_info             = seq_using_info;
      seq_delay_info             = seq_delay_info;
      seq_append_info            = seq_append_info;
      seq_generated_info         = seq_generated_info;
      seq_finally_info           = seq_finally_info;
      seq_of_functions_info      = seq_of_functions_info;
      seq_map_info               = seq_map_info;
      seq_singleton_info         = seq_singleton_info;
      seq_empty_info             = seq_empty_info;
      new_format_info            = new_format_info;
      unbox_vref                 = vref_for_val_info unbox_info;
      unbox_fast_vref            = vref_for_val_info unbox_fast_info;
      istype_vref                = vref_for_val_info istype_info;
      istype_fast_vref           = vref_for_val_info istype_fast_info;
      unbox_info                 = unbox_info;
      unbox_fast_info            = unbox_fast_info;
      istype_info                = istype_info;
      istype_fast_info           = istype_fast_info;
      create_instance_info       = create_instance_info;
      seq_to_list_info           = seq_to_list_info;
      seq_to_array_info          = seq_to_array_info;
      
      cons_ucref = cons_ucref;
      nil_ucref = nil_ucref;
   }
     
