(* (c) Microsoft Corporation. All rights reserved *)

(*F# 
module Microsoft.FSharp.Compiler.Detuple 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
F#*) 

open Ildiag
open List
open Ast
open Tast
open Tastops
open Env
open Layout
open Lib

(* REVIEW: review this function and its uses.  It should either not exist or should be in tastops.ml{i} *)
let combineTyappWithApp (f,fty,tys,args,m) =
   (* combine apps if possible. recursive ids are inside reclinks and maybe be type instanced with a TExpr_app *)
   match strip_expr f with
   | TExpr_app(f2,fty2,tys2,[]     ,m2) -> (f2,fty2,tys2 @ tys,args,m)
   | TExpr_app(f2,fty2,tys2,argtys2,m2) -> (f,fty,tys,args,m) (* has args, so not combine ty args *)
   | f                                  -> (f,fty,tys,args,m)

(*-------------------------------------------------------------------------
 *INDEX: vals_bound_in_expr
 *-------------------------------------------------------------------------*)

module GlobalUsageAnalysis = struct
    let bindAccBounds vals (isInDTree,v) =  Zset.add v vals

    let vals_bound_in_expr expr =
       let folder = {exprFolder0 with valDefAcc = bindAccBounds} in
       let z0 = Zset.empty val_spec_order in
       let z  = foldExpr folder z0 expr in
       z


    (*-------------------------------------------------------------------------
     *INDEX: xinfo - state and ops
     *-------------------------------------------------------------------------*)

    type accessor = PTup of int * typ list

    type xinfo =
      (* Expr information.
       * For each v,
       *   (a) log it's usage site context = accessors // APP type-inst args
       *       where first accessor in list applies first to the v/app.
       *   (b) log it's binding site representation.
       *------
       * Future, could generalise to be graph representation of expr. (partly there).
       * This type used to be called "usage".
       *)
       { xinfo_uses     : (val_spec,(accessor list * typ list * expr list) list)Zmap.map; (* v -> context / APP inst args *)
         xinfo_eqns     : (val_spec,expr)Zmap.map;                                        (* v -> binding repr *)
         xinfo_dtree    : val_spec Zset.set;                                              (* bound in a decision tree? *)
         xinfo_mubinds  : (val_spec,bool * val_spec list)Zmap.map;                        (* v -> v list * recursive? -- the others in the mutual binding *)
         xinfo_toplevel : val_spec Zset.set;
         xinfo_top      : bool
       }

    let z0 =
       { xinfo_uses     = Zmap.empty (val_spec_order );
         xinfo_eqns     = Zmap.empty (val_spec_order );
         xinfo_mubinds    = Zmap.empty (val_spec_order );
         xinfo_dtree    = Zset.empty (val_spec_order );
         xinfo_toplevel = Zset.empty (val_spec_order );
         xinfo_top      = true
       }

    let logUse (f:val_spec) tup z =
       {z with xinfo_uses = 
           zmap_map f (function None       -> Some [tup]
                              | Some sites -> Some (tup::sites)) z.xinfo_uses }

    let logBinding z (isInDTree,v) =
        let z = if isInDTree then {z with xinfo_dtree = Zset.add v z.xinfo_dtree} else z in
        let z = if z.xinfo_top then {z with xinfo_toplevel = Zset.add v z.xinfo_toplevel} else z in
        z
        

    let logVeqns z (recursive,binds) =
        (* log mubind v -> vs *)
        let vs = map var_of_bind binds in
        let z = 
            let addMu mubinds v = Zmap.add v (recursive,vs) mubinds in
            {z with xinfo_mubinds = fold_left addMu z.xinfo_mubinds vs} in
        let z = 
            let addEqn map eqn =  Zmap.add (var_of_bind eqn) (rhs_of_bind eqn) map  in
            {z with xinfo_eqns = fold_left addEqn z.xinfo_eqns binds }  in
        z

    let foldUnderLambda f z x =
        let saved = z.xinfo_top in
        let z = {z with xinfo_top=false} in
        let z = f z x in
        let z = {z with xinfo_top=saved} in
        z

    let dumpXInfo z =
        let soAccessor (PTup (n,ts)) = "#" ^ string_of_int n in
        let dumpSite v (accessors,inst,args) =
            dprintf4 "- use %s%s %s %s\n"
              (showL (vspecL v))
              (match inst with
	        [] -> ""
              | _  -> "@[" ^ showL (commaListL (map typeL inst)) ^ "]")
              (showL (spaceListL (map exprL args)))
              (match accessors with
	        [] -> ""
              | _  -> "|> " ^ String.concat " " (map soAccessor accessors)) in
        let dumpUse v sites = List.iter (dumpSite v) sites in
        let dumpTop v = dprintf1 "- toplevel: %s\n" (name_of_val v) in
        if false then
         ( dprintf0 "usage:\n";
           Zmap.iter dumpUse z.xinfo_uses;
           Zset.iter dumpTop z.xinfo_toplevel
          )
        else
         ()


    (*-------------------------------------------------------------------------
     *INDEX: xinfo - foldExpr, foldBind collectors
     *-------------------------------------------------------------------------*)

    let usageFolders g =
      (* Fold expr, intercepts selected exprs.
       *   "val v"        - count []     callpattern of v
       *   "app (f,args)" - count <args> callpattern of f
       *---
       * On intercepted nodes, must continue exprF fold over any subexpressions, e.g. args.
       *------
       * Also, noting top-level bindings,
       * so must cancel top-level "foldUnderLambda" whenever step under loop/lambda:
       *   - lambdas
       *   - try_catch + try_finally???
       *   - for body
       *   - match targets
       *   - tmethods
       *)  
      let foldLocalVal f z vref = 
          if item_ref_in_this_assembly g.compilingFslib vref then f z (deref_val vref)
          else z in                   
      let exprUsageIntercept exprF z expr =
        let exprsF z xs = fold_left exprF z xs in
        let rec recognise context expr = 
          match expr with
           | TExpr_val (v,_,m)                  -> (* YES: count free occurance *)
                                                   let z = foldLocalVal (fun z v -> logUse v (context,[],[]) z) z v in
                                                   Some z
           | TExpr_app (f,fty,tys,args,m)       -> let f,fty,tys,args,m = combineTyappWithApp (f,fty,tys,args,m) in
                                                   (match f with
                                                      TExpr_val (f,_,_) ->
                                                      (* app where function is val *)
                                                      (* YES: count instance/app (app when have term args), and then
                                                       *      collect from args (have intercepted this node) *)
                                                      let collect z f = logUse f (context,tys,args) z in
                                                      let z = foldLocalVal collect z f in
                                                      let z = fold_left exprF z args in
                                                        Some z
                                                   | _ ->
                                                       (* NO: app but function is not val *)
                                                       None)
           | TExpr_op(TOp_tuple_field_get (n),ts,[x],m)   -> 
                                                   let context = PTup (n,ts) :: context in
                                                   recognise context x
           (* lambdas end top-level status *)
           | TExpr_lambda(id,basevopt,vs,body,m,rty,_)   -> let z = foldUnderLambda exprF z body in
                                                   Some z
           | TExpr_tlambda(id,tps,body,m,rty,_) -> let z = foldUnderLambda exprF z body in     
                                                   Some z
           | _                                  -> None (* NO: no intercept *)
         in
         let context = [] in
         recognise context expr in

      let targetIntercept exprF z = function TTarget (argvs,body) -> Some (foldUnderLambda exprF z body) in
      let tmethodIntercept exprF z = function TMethod(_,_,_,e,m) -> Some (foldUnderLambda exprF z e) in
      
      {exprFolder0 with
         exprIntercept    = exprUsageIntercept; 
         valBindAcc         = logVeqns;
         valDefAcc          = logBinding;
         targetIntercept  = targetIntercept;
         tmethodIntercept = tmethodIntercept;
      }


    (*-------------------------------------------------------------------------
     *INDEX: xinfo - entry point
     *-------------------------------------------------------------------------*)

    let xinfo_of_assembly g expr =
       let folder = usageFolders g in
       let z = foldAssembly folder z0 expr in
       dumpXInfo z;
       z
end


open GlobalUsageAnalysis

(*-------------------------------------------------------------------------
 *INDEX: misc
 *-------------------------------------------------------------------------*)
  
let internalError str = dprintf1 "Error: %s\n" str;raise (Failure str)  


let mkLocalVal m n ty arity_info =
    let compgen    = false in (* REVIEW: review: should this be true? *)
    new_vspec(mksyn_id m n,ty,Immutable,compgen,arity_info,None,taccessPublic,ValNotInRecScope,None,NormalVal,[],OptionalInline,emptyXMLDoc,false,false,false,false,None,ParentNone) 

let dprintTerm header expr =
  if false then
    let str = Layout.showL (Layout.squashTo 192 (mimplsL expr)) in  (* improve cxty! *)
    dprintf2 "\n\n\n%s:\n%s\n" header str
  else
    ()


(*-------------------------------------------------------------------------
 *INDEX: OVERVIEW:
 *-----------------------------------------------------------------------*)

(*
 * This pass has one aim.
 * - to eliminate tuples allocated at call sites (due to uncurried style)
 *
 * After PASS, 
 *   Functions f which had explicit tuples at all callsites,
 *   have been replaced by fC taking the individual tuple fields,
 *   subject to the type of the f formal permitting the split.
 * 
 * For now, the decisions are based on call site analysis only.
 * Information about function requirements could be used (later possibility).
 * [No longer true, using projection info in defn].
 *
 *----------
 * TUPLE COLLAPSE SIMPLIFIED.
 *
 * The aim is to eliminate (redundant) tuple allocs arising due to calls.
 * These typically arise from code written in uncurried form.
 *
 * Q: When is a tuple allocation at callsite redundant?
 * A1: If the function called only wants the fields of the tuple.
 * A2: If all call sites allocate a tuple argument,
 *     then can factor that tuple creation into the function,
 *     and hope the optimiser will eliminate it if possible.
 *     e.g. if only the fields are required.
 *
 * The COLLAPSE transform is based on answer A2...
 *
 *   [[ let rec f p = ... f (a,b) ... in
 *      f (x,y) ]]
 *   ->
 *      let rec fC p1 p2 = let p = p1,p2 in
 *                        ... (fC a b) ...
 *      in
 *      fC x y
 *
 * Q: What about cases where some calls to f provide just a tuple?
 * A: If f requires the original tuple argument, then this transform
 *    would insert a tuple allocation inside f, where none was before...
 *
 *----------
 * IMPLEMENTATION OVERVIEW.
 *
 * 1. Require call-pattern info about callsites of each function, e.g.
 *
 *      [ (_,_) ; (_,(_,_,_)) ; _ ]
 *      [ (_,_) ; (_,_)       ]
 *      [ (_,_) ]
 *
 *    Detailing the number of arguments applied and their explicit tuple structure.
 *
 *    ASIDE: Efficiency note.
 *           The rw pass does not change the call-pattern info,
 *           so call-pattern info can be collected for all ids in pre-pass.
 *
 * 2. Given the above, can *CHOOSE* a call-pattern for the transformed function.
 *    Informally,
 *      Collapse any tuple structure if it is known at ALL call sites.
 *    Formally,
 *      - n = max length of call-pattern args.
 *      - extend call patterns to length n with _ (no tuple info known)
 *      - component-wise intersect argument tuple-structures over call patterns.
 *      - gives least known call-pattern of length n.
 *      - can trim to minimum non-trivual length.
 *
 *    [Used to] have INVARIANT on this chosen call pattern:
 *
 *      Have: For each argi with non-trivial tuple-structure,
 *            at every call have an explicit tuple argument,
 *            with (at least) that structure.
 *            ----
 *            Note, missing args in partial application will always
 *            have trivial tuple structure in chosen call-pattern.
 *
 *    [PS: now defn arg projection info can override call site info]
 *
 * 2b.Choosing callPattern also needs to check type of formals for the function.
 *    If function is not expecting a tuple (accoring to types) do not split them.
 *
 * 3. Given callPattern for selected f,
 *    (a) Can choose replacement formals, ybi where needed. (b, bar, means vector of formals).
 *
 *     cpi                | xi    | ybi
 *    --------------------|-------|----------
 *     UnknownTS          | xi    | SameArg xi
 *     TupleTS []         | []    | SameArg []     // unit case, special case for now.
 *     TupleTS ts1...tsN  | xi    | NewArgs (mapConcat createFringeFormals [ts1..tsN])
 *
 *    (b) Can define fC replacement function id.
 *
 * 4. Fixup defn bindings.
 *
 *    [[DEFN: f  = LAM tps. lam x1 ...xp xq...xN. body ]]
 *    ->
 *           fC = LAM tps. lam [[FORMALS: yb1...ybp]] xq...xN. [[REBINDS x1,yb1 ... xp,ybp]] [[FIX: body]]
 *
 *    [[FORMAL: SameArg xi]] -> xi
 *    [[FORMAL: NewArgs vs]] -> [ [v1] ... [vN] ]                // list up individual args for TExpr_lambda
 *
 *    [[REBIND: xi , SameArg xi]] -> // no binding needed
 *    [[REBIND: [u], NewArgs vs]] -> u = "rebuildTuple(cpi,vs)"
 *    [[REBIND: us , NewArgs vs]] -> "rebuildTuple(cpi,vs)" then bind us to projections. // for TExpr_lambda
 *
 *    rebuildTuple - create tuple based on vs fringe according to cpi tuple structure.
 *
 *    Note, fixup body...
 *
 * 5. Fixup callsites.
 *
 *    [[FIXCALL: APP f tps args]] -> when f is transformed, APP f tps [[collapse args wrt cpf]]
 *                                   otherwise, unchanged,  APP f tps args.
 *
 * 6. Overview.
 *    - pre-pass to find callPatterns.
 *    - choose callPattern (tuple allocs on all callsites)
 *    - create replacement formals and fC where needed.
 *    - rw pass over expr - fixing defns and applications as required.
 *    - sanity checks and done.
 *)

(* PS:  ids can occur in several ways in expr at this point in compiler.
 *      val id                                        - freely
 *      app (val id) tys args                         - applied to tys/args (if no args, then free occurance)
 *      app (reclink (val id)) tys args               - applied (recursive case)
 *      app (reclink (app (val id) tys' []) tys args  - applied (recursive type instanced case)
 * So, taking care counting callpatterns.
 *
 * PS:  now considering defn projection requirements in decision.
 *      no longer can assume that all call sites have explicit tuples if collapsing.
 *      in these new cases, take care to have let binding sequence (eval order...)
 *)


(*-------------------------------------------------------------------------
 *INDEX: tupleTS = tuple structure
 *-------------------------------------------------------------------------*)

type tupleTS = (* tuple structure *)
  | UnknownTS
  | TupleTS   of tupleTS list

let rec arityTS = function
  | UnknownTS  -> [TopValData.unnamedTopArg]
  | TupleTS ts -> concat (map arityTS ts)

let rec andTS ts tsB =
   match ts,tsB with
   |         _   ,UnknownTS    -> UnknownTS
   | UnknownTS   ,_            -> UnknownTS
   | TupleTS ss  ,TupleTS ssB  -> if length ss <> length ssB then UnknownTS (* different tuple instances *)
	                                                     else TupleTS (map2 andTS ss ssB)

let checkTS = function
    TupleTS []   -> internalError "exprTS: Tuple[]  not expected. (units not done that way)."
  | TupleTS [ts] -> internalError "exprTS: Tuple[x] not expected. (singleton tuples should not exist."
  | ts           -> ts	 
	  
let rec uncheckedExprTS = function (* explicit tuple-structure in expr *)
  | TExpr_op(TOp_tuple,tys,args,m) -> TupleTS (list_map uncheckedExprTS args)
  | _                        -> UnknownTS

let rec uncheckedTypeTS ty =
   let ty = strip_tpeqns_and_tcabbrevs ty in
   match ty with
   | TType_tuple tys -> TupleTS (map uncheckedTypeTS tys)
   | _               -> UnknownTS

let exprTS = uncheckedExprTS >> checkTS
let typeTS = uncheckedTypeTS >> checkTS

let rebuildTS g m ts vs =
  let rec rebuild vs ts = 
    match vs,ts with
      []   ,UnknownTS   -> internalError "rebuildTS: not enough fringe to build tuple"
    | v::vs,UnknownTS   -> vs,(expr_for_val m v,type_of_val v)
    | vs   ,TupleTS tss -> let vs,xtys = fmap rebuild vs tss in
                           let xs,tys  = split xtys in
                           let x  = mk_tupled g m xs tys in
                           let ty = mk_tupled_ty g tys in
                           vs,(x,ty)
  in
  let vs,(x,ty) = rebuild vs ts in
  if length vs<>0 then internalError "rebuildTS: had move fringe vars than fringe. REPORT BUG" else ();
  x

(* naive string concats, just for testing *)
let rec soTS = function (UnknownTS) -> "_" | TupleTS ss -> "(" ^ String.concat "," (map soTS ss) ^ ")"


(*-------------------------------------------------------------------------
 *INDEX: callPattern
 *-------------------------------------------------------------------------*)

type callPattern =
    (* callPattern is tuple-structure for each argument position.
     * - callsites have a callPattern (possibly instancing f at tuple types...).
     * - the definition lambdas may imply a one-level callPattern
     * - the definition formal projection info suggests a callPattern
     *)
    tupleTS list (* equality/ordering ok on this type *)
      
let soCP tss = String.concat ";" (map soTS tss)
let callPatternOrder = (Pervasives.compare : callPattern -> callPattern -> int)
let argsCP exprs = map exprTS exprs
let noArgsCP = []
let isTrivialCP xs = (xs=[])

let minimalCP cp =
   let rec mcp = function
     []                -> []
   | UnknownTS::tss    -> (match mcp tss with
                            []  -> []              (* drop trailing UnknownTS *)
                          | tss -> UnknownTS::tss) (* non triv tss tail *)
   | (TupleTS ts)::tss -> TupleTS ts :: mcp tss
   in
   mcp cp

let commonCP f cps =
   (* INTERSECTION.
    * Combines a list of callpatterns into one common callpattern.
    *)
   let rec andCPs cpA cpB =
     match cpA,cpB with
       []       ,[]        -> []
     | tsA::tsAs,tsB::tsBs -> andTS tsA tsB :: andCPs tsAs tsBs
     | tsA::tsAs,[]        -> [] (* now trim to shortest - UnknownTS     :: andCPs tsAs []   *)
     | []       ,tsB::tsBs -> [] (* now trim to shortest - UnknownTS     :: andCPs []   tsBs *)
   in
   let andCPs cpA cpB =
     try andCPs cpA cpB
     with e -> (Printf.printf "commonCP: failed on\ncpA = %s\ncpB = %s\nf = %s" (soCP cpA) (soCP cpB) (name_of_val f); raise e)
   in
   fold1 andCPs cps
   

let siteCP (accessors,inst,args) = argsCP args
let sitesCPs sites = map siteCP sites


(*-------------------------------------------------------------------------
 *INDEX: transform
 *-------------------------------------------------------------------------*)

type newFormal =
    SameArg                          (* arg position unchanged, keep original formal *)
  | NewArgs of val_spec list * expr  (* collapsing, use these formals. expr is tuple of them *)

type transform =
   (* Info needed to convert f to curried form.
    * - yb1..ybp - replacement formal choices for x1...xp.
    * - fC       - replaces f.
    *)
   { tranformCP : callPattern;
     yb1ps      : newFormal list; (* REVIEW: could push these to fixup binding site *)
     f          : val_spec;
     fC         : val_spec;
     fCty       : typ;
   }


(*-------------------------------------------------------------------------
 *INDEX: transform - mkTransform - decided, create necessary stuff
 *-------------------------------------------------------------------------*)
      
let mkFormal m i ty = let name = "y" ^ string_of_int i in
                      let name = nng.nngApply name m in
                      mk_compgen_local m name ty |> fst
    
let mkTransform g f m tps x1Ntys rty (cp,tyfringes) =
  (* Create formal choices for x1...xp under cp (callPattern) *)
  let chooseFormals cpi tyfringe =
    match cpi with
    | UnknownTS  -> SameArg
    | TupleTS [] -> SameArg  (* unit: Q: what is xi? [u]? []? ..? *)
    | TupleTS ts -> let vs = list_mapi (mkFormal m) tyfringe in
                    NewArgs (vs,rebuildTS g m cpi vs)
  in
  let yb1ps = map2 chooseFormals cp tyfringes in
  (* Create fC replacement for f *)
  (* Mark the arity of the value *)
  let arity_info = match (arity_of_val f) with None -> None | _ -> Some(TopValInfo (length tps,mapConcat arityTS cp,TopValData.unnamedRetVal)) in  
  (* type(fC) tyfringes types replace initial arg types of f *)
  let r = length tyfringes in
  let tys1r = concat tyfringes in  (* types for collapsed initial r args *)
  let tysrN = drop r x1Ntys  in  (* types for remaining args *)
  let argtys = tys1r @ tysrN in
  let fCty  = mk_lambda_ty tps argtys rty                   in  
  let fC  = mkLocalVal m (name_of_val f ^ "$detupled") fCty arity_info in
(*dprintf1 "mkTransform: f=%s\n"         (showL (vspecL f));
  dprintf1 "mkTransform: tps=%s\n"       (showL (commaListL (map typarL tps)));
  dprintf1 "mkTransform: cp=%s\n"        (soCP cp);
  dprintf1 "mkTransform: tyfringes=%s\n" (showL (commaListL (map (fun fr -> tupleL (map typeL fr)) tyfringes)));
  dprintf1 "mkTransform: tys1r=%s\n"     (showL (commaListL (map typeL tys1r)));
  dprintf1 "mkTransform: tysrN=%s\n"     (showL (commaListL (map typeL tysrN)));
  dprintf1 "mkTransform: rty  =%s\n"     ((DebugPrint.showType rty));
*)   
  { tranformCP = cp;
    yb1ps      = yb1ps;
    f          = f;
    fC         = fC;
    fCty       = fCty
  }

open Layout    
let dumpTransform trans =
  let argTS = function
      1 -> UnknownTS
    | n -> TupleTS (repeat n UnknownTS)
  in
  dprintf3 " - cp   : %s\n - fC   : %s\n - fCty : %s\n"
    (soCP trans.tranformCP)
    (showL (vspecL trans.fC))
    ((DebugPrint.showType  trans.fCty))


(*-------------------------------------------------------------------------
 *INDEX: transform - vTransforms - support
 *-------------------------------------------------------------------------*)

let zipCallPatternArgTys (cp : tupleTS list) (tys : typ list) =
  let rec zipTSTyp ts typ =
    (* match a tuple-structure and type, yields:
     *  (a) (restricted) tuple-structure, and
     *  (b) type fringe for each arg position.
     *)
    let typ = strip_tpeqns_and_tcabbrevs typ in
    match ts,typ with
      UnknownTS ,ty              -> UnknownTS,[ty]
    | TupleTS ts,TType_tuple tys -> let ts,tyfringe = zipTSListTypList ts tys in
                                    TupleTS ts,tyfringe
    | TupleTS _ ,ty              -> UnknownTS,[ty] (* trim back callPattern, function more general *)
   and zipTSListTypList tss tys =
     let tstys = map2 zipTSTyp tss tys in (* assumes tss tys same length *)
     let ts  = map fst tstys          in 
     let tys = mapConcat snd tstys      in (* link fringes *)
     ts,tys
   in
   let tys = front (length cp) tys in    (* drop excessive tys if cp shorter *)
   let tstys = map2 zipTSTyp cp tys in
   let cp        = map fst tstys in  (* cp        = ts1 ...tsp  -- the collapsed call pattern *)
   let tyfringes = map snd tstys in  (* tyfringes = tys1...tysp -- the type fringes for each arg position *)
   cp,tyfringes

let callPatternTyRestrict tys cp =
  let cp,tyfringes = zipCallPatternArgTys cp tys in
  cp


(*-------------------------------------------------------------------------
 *INDEX: transform - vTransforms - defnSuggestedCP
 *-------------------------------------------------------------------------*)

let rec trimTsByAccess accessors ts =
   match ts,accessors with
     UnknownTS ,_                       -> UnknownTS
   | TupleTS tss,[]                     -> UnknownTS (* trim it, require the val at this point *)
   | TupleTS tss,PTup (i,ty)::accessors -> let tss = mapNth i (trimTsByAccess accessors) tss in
                                           TupleTS tss

let trimTsByVal z ts v =
   match Zmap.tryfind v z.xinfo_uses with
     None       -> UnknownTS (* formal has no usage info, it is unused *)
   | Some sites -> let trim ts (accessors,inst,args) = trimTsByAccess accessors ts in
                   fold_left trim ts sites

let trimTsByFormal z ts = function
     [v]  -> trimTsByVal z ts v
   | vs   -> let tss = match ts with TupleTS tss -> tss | _ -> internalError "trimByFormal: ts must be tuple?? PLEASE REPORT\n" in
             let tss = map2 (trimTsByVal z) tss vs in
             TupleTS tss

let formalSuggestedCP z vss tys =
   (* v = LAM tps. lam vs1:ty1 ... vsN:tyN. body.
    * The types suggest a tuple structure callPattern.
    * The projections of the vsi trim this down,
    * since do not want to take as components any tuple that is required (projected to).
    *------
    *)
   let tss = map typeTS tys in (* most general TS according to type *)
   let tss = map2 (trimTsByFormal z) tss vss in
   tss


(*-------------------------------------------------------------------------
 *INDEX: transform - decideTransform
 *-------------------------------------------------------------------------*)

let decideTransform g z v cps (m,tps,vss,tys,rty) (* tys are types of outer args *) =
  (* let dprintf1 _ _ = () in      *)
  (* NOTE: 'a in arg types may have been instanced at different tuples... *)
  (*       commonCP has to handle those cases. *)
  let cp           = commonCP v cps in                  (* common callPattern *)
  let cp           = front (length tys) cp in           (* restricted to max nArgs *)
  (* NOW: get formal cp by defn usage of formals *)
  let fcp          = formalSuggestedCP z vss tys in
  let cp           = front (length cp) fcp in
  let cp,tyfringes = zipCallPatternArgTys cp tys in     (* zip with types of known args *)
  let cp           = minimalCP cp in                    (* drop trivial tail AND *)
  let tyfringes    = take (length cp) tyfringes in      (* shorten tyfringes (zippable) *)
(*dprintf1 "decideTransform: for v=%s\n" (showL (vspecL v));
  List.iter (fun cp -> dprintf1 "- site cp    = %s\n" (soCP cp)) cps;
  dprintf1 "- common  cp = %s\n" (soCP cp);     
  dprintf1 "- front   cp = %s\n" (soCP cp);
  dprintf1 "- arg tys    = %s\n" (showL (commaListL (map typeL tys)));  
  dprintf1 "- fcp        = %s\n" (soCP fcp);  
  dprintf1 "- front fcp  = %s\n" (soCP cp);
  dprintf1 "- zipped  cp = %s\n" (soCP cp);
  dprintf1 "- tyfringes  = %s\n" (showL (commaListL (map (length >> intL) tyfringes)));  
  dprintf1 "- minimal cp = %s\n\n" (soCP cp);
*)   
  if isTrivialCP cp then
    None (* no transform *)
  else
    (
     Some (v,mkTransform g v m tps tys rty (cp,tyfringes))
    )


(*-------------------------------------------------------------------------
 *INDEX: transform - determineVTransforms
 *-------------------------------------------------------------------------*)
      
(* Public f could be used beyond assembly.
 * For now, suppressing any transforms on these.
 * Later, could transform f and fix up local calls and provide an f wrapper for beyond. *)
let eligibleVal g v =
  let dllImportStubOrOtherNeverInline = (inlineFlag_of_val v = NeverInline) in 
  let mutableVal = mutability_of_val v <> Immutable in
  let byrefVal = is_byref_ty g (type_of_val v) in
  not dllImportStubOrOtherNeverInline &&
  not byrefVal &&
  not mutableVal &&
  isNone(arity_of_val v) &&
  isNone(member_info_of_val v)

let determineVTransforms g (z : GlobalUsageAnalysis.xinfo) =
   let selectTransform f sites =
     if not (eligibleVal g f) then None else
     (* consider f, if it has top-level lambda (meaning has term args) *)
     match Zmap.tryfind f z.xinfo_eqns with
     | None   -> None (* no binding site, so no transform *)
     | Some e -> 
        let tps,vss,b,rty = dest_top_lambda (e,type_of_val f) in
        match concat vss with
        | []      -> None (* defn has no term args *)
        | arg1::_ -> (* consider f *)
          let m   = range_of_val arg1          in      (* mark of first arg *)
          let tys = map type_of_lambda_arg vss in      (* arg types *)
          let cps = sitesCPs sites             in      (* cps from sites *)
          decideTransform g z f cps (m,tps,vss,tys,rty) (* make transform (if required) *)
   in
   let vtransforms = Zmap.chooseL selectTransform z.xinfo_uses in
   let vtransforms = Zmap.of_list (Zmap.empty val_spec_order) vtransforms in
   vtransforms

let dumpVTransform v tr =
   dprintf1 "Transform for %s\n" (showL (vspecL v));
   dumpTransform tr;
   flush stdout


(*-------------------------------------------------------------------------
 *INDEX: pass - penv - env of pass
 *-------------------------------------------------------------------------*)

type penv =
   { transforms : (val_spec,transform) Zmap.map; (* planned transforms *)
     ccu        : ccu;
     g          : Env.tcGlobals;
   }

let hasTransform penv f = Zmap.tryfind f penv.transforms

(*-------------------------------------------------------------------------
 *INDEX: pass - app fixup - collapseArgs
 *-------------------------------------------------------------------------*)

(* collapseArgs:
   - the args may not be tuples (decision made on defn projection).
   - need to factor any side-effecting args out into a let binding sequence.
   - also factor projections, so they share common tmps.
*)

type env = {eg : tcGlobals;
            prefix : string;
            m      : Range.range; }
let suffixE env s = {env with prefix = env.prefix ^ s}
let rangeE  env m = {env with m = m}

let push  b  bs = b::bs
let pushL xs bs = xs@bs

let newLocal  env   ty = mk_compgen_local env.m env.prefix ty
let newLocalN env i ty = mk_compgen_local env.m (env.prefix ^ string_of_int i) ty

let noEffectExpr env bindings x =
   match x with
   | TExpr_val (v,_,m) -> bindings,x
   | x                 -> let tmp,xtmp = newLocal env (type_of_expr env.eg x) in
                          let bind = TBind (tmp,x) in
                          push bind bindings,xtmp

let projections env bindings x xtys =
   let build i xty = let vi,vix = newLocalN env i xty in
                     TBind(vi,mk_tuple_field_get (x,xtys,i,env.m)),vix
   in
   let bindxs = list_mapi build xtys in
   pushL (rev (map fst bindxs)) bindings,map snd bindxs

let rec collapseArg env bindings ts x =
  let m = range_of_expr x in
  let env = rangeE env m in
  match ts,x with
  | UnknownTS  ,x                      -> let bindings,vx = noEffectExpr env bindings x in
                                          bindings,[vx]
  | TupleTS tss,TExpr_op(TOp_tuple,xtys,xs,m) -> 
                                          let env = suffixE env "'" in
                                          collapseArgs env bindings 1 tss xs
  | TupleTS tss,x                      -> (* NOTE: no longer expect explicit tuples, so project components *)
                                          let bindings,x = noEffectExpr env bindings x in
                                          let env  = suffixE env "_p"  in
                                          let xty = type_of_expr env.eg x in 
                                          let xtys = dest_tuple_typ xty in
                                          let bindings,xs = projections env bindings x xtys in
                                          collapseArg env bindings (TupleTS tss) (mk_tupled env.eg m xs xtys)

and collapseArgs env bindings n (cp as tss) args =
  match cp,args with
    []     ,args        -> bindings,args
  | ts::tss,arg::args -> let env1 = suffixE env (string_of_int n) in
                         let bindings,xty  = collapseArg  env1 bindings ts    arg      in
                         let bindings,xtys = collapseArgs env  bindings (n+1) tss args in
                         bindings,xty @ xtys
  | ts::tss,[]            -> internalError "collapseArgs: callPattern longer than callsite args. REPORT BUG"


(*-------------------------------------------------------------------------
 *INDEX: pass - app fixup
 *-------------------------------------------------------------------------*)

(* REVIEW: use mk_let etc. *)
let nestedLet = list_fold_right (fun b acc -> mk_let_bind (range_of_expr acc) b acc) 

let fixupApp (penv:penv) (fx,fty,tys,args,m) =
    let fixit trans (tys,args,m) =
       let cp       = trans.tranformCP  in
       let fC       = trans.fC          in
       let fCty     = trans.fCty        in
       let fCx      = expr_for_val m fC in   
       try
           (* [[f tps args ]] -> fC tps [[COLLAPSED: args]] *)
             let env      = {prefix = "arg";m = m;eg=penv.g} in
             let bindings = [] in
             let bindings,args = collapseArgs env bindings 0 cp args in
             let bindings = rev bindings in
             nestedLet bindings (TExpr_app (fCx,fCty,tys,args,m))
       with
             e -> errorR(e);
                  error(Error(Printf.sprintf "internalError error: detuple optimization: fixit: failed = %s\nfixit: args   = %s\n"
                     (showL (vspecL trans.fC))
                     (showL (commaListL (map exprL args))),m)) in
    (* Is it a val app, where the val has a transform? *)
    match fx with
    | TExpr_val (vref,_,m) -> 
                           let f = deref_val vref in
                           (match hasTransform penv f with
                              Some trans -> ((* XXX - dprintf1 "fixupApp: f=%s\n" (showL (vspecL f)); *)
                                             fixit trans (tys,args,m))       (* fix it *)
                            | None       -> TExpr_app (fx,fty,tys,args,m)) (* no change, f untransformed val *)
    | _               -> TExpr_app (fx,fty,tys,args,m)                      (* no change, f is expr *)


(*-------------------------------------------------------------------------
 *INDEX: pass - mubinds - translation support
 *-------------------------------------------------------------------------*)

let transFormal ybi xi =
  match ybi with
    SameArg        -> [xi]                  (* one arg   - where arg=vpsecs *)
  | NewArgs (vs,x) -> map singletonList vs  (* many args *)

let transRebind ybi xi =
  let forceTuple = function
    | TExpr_op(TOp_tuple,ty,xs,m) -> xs
    | _ -> internalError "transRebind: impossible! REPORT BUG" in
  match xi,ybi with
  | xi ,SameArg        -> []                    (* no rebinding, reused original formal *)
  | [u],NewArgs (vs,x) -> [mk_bind u x]
  | us ,NewArgs (vs,x) -> let xs = forceTuple x in
                          map2 mk_bind us xs


(*-------------------------------------------------------------------------
 *INDEX: pass - mubinds
 *-------------------------------------------------------------------------*)

   (* Foreach (f,repr) where
    *   If f has trans, then
    *   repr = LAM tps. lam x1...xN . body
    *
    *   fC, yb1...ybp in trans.
    *
    * New binding:
    *
    *   fC = LAM tps. lam [[FORMALS: yb1 ... ybp]] xq...xN = let [[REBINDS: x1,yb1 ...]] in
    *                                                        body
    *
    * Does not fix calls/defns in binding rhs, that is done by caller.
    *)

let pass_bind penv (TBind(f,repr) as bind) =
     let m = range_of_val f in
     match hasTransform penv f with
     | None ->
         (* f no transform *)
         bind
     | Some trans ->
         (* f has transform *)
         let tps,vss,body,rty = dest_top_lambda (repr,type_of_val f) in (* expectation *)
         (* fC is curried version of f *)
         let fC    = trans.fC in
         (* fCBody - parts - formals *)
	 let yb1ps = trans.yb1ps  in
	 let p     = length yb1ps in
	 if (length vss < p) then internalError "pass_binds: |vss|<p - detuple pass" else (); (* ASSERTION *)
	 let xqNs  = drop p vss   in
	 let x1ps  = take p vss   in
	 let y1Ps  = concat (map2 transFormal yb1ps x1ps) in
	 let formals = y1Ps @ xqNs in
	 (* fCBody - parts *)
	 let rebinds = concat (map2 transRebind yb1ps x1ps) in
         (* fCBody - rebuild *)
         (* fCBody = TLambda tps. Lam formals. let rebinds in body *)
         let rbody,rt  = mk_lets_bind            m rebinds body,rty    in
         let bind      = mk_multi_lambda_bind fC m tps formals (rbody,rt) in
         (* result *)
	 bind
   (* end transBind *)

let pass_binds penv binds = map (pass_bind penv) binds

(*-------------------------------------------------------------------------
 *INDEX: pass - pass_bind_rhs
 *
 * At bindings (letrec/let),
 *   0. run pass of bodies first.
 *   1. transform bindings (as required),
 *      yields new bindings and fixup data for callsites.
 *   2. required to fixup any recursive calls in the bodies (beware O(n^2) cost)
 *   3. run pass over following code.
 *-------------------------------------------------------------------------*)

let pass_bind_rhs penv conv (TBind (f,repr)) = TBind(f,conv repr)
let pre_intercept_expr (penv:penv) conv expr =
  match expr with
  | TExpr_letrec (binds,e,m,_) ->
     let binds = map (pass_bind_rhs penv conv) binds in 
     let binds = pass_binds penv binds in
     Some (mk_letrec_binds m binds (conv e))
  | TExpr_let (bind,e,m,_) ->  
     let bind = pass_bind_rhs penv conv bind in 
     let bind = pass_bind penv bind in
     Some (mk_let_bind m bind (conv e))
  | TExpr_app (f,fty,tys,args,m) ->
     (* match app, and fixup if needed *)
     let args = map conv args in 
     let f = conv f in 
     let f,fty,tys,args,m = combineTyappWithApp (f,fty,tys,args,m) in
     Some (fixupApp penv (f,fty,tys,args,m) )
  | _ -> None
  

let post_transform_expr (penv:penv) expr =
  match expr with
  | TExpr_letrec (binds,e,m,_) ->
     let binds = pass_binds penv binds in
     Some (mk_letrec_binds m binds e)
  | TExpr_let (bind,e,m,_) ->  
     let bind = pass_bind penv bind in
     Some (mk_let_bind m bind e)
  | TExpr_app (f,fty,tys,args,m) ->
     (* match app, and fixup if needed *)
     let f,fty,tys,args,m = combineTyappWithApp (f,fty,tys,args,m) in
     Some (fixupApp penv (f,fty,tys,args,m) )
  | _ -> None
  

let pass_assembly penv ass = 
    rewrite_assembly {pre_intercept =None (* Some (pre_intercept_expr penv) *);
                      post_transform= post_transform_expr penv (* (fun _ -> None)  *);
                      underQuotations=false } ass


(*-------------------------------------------------------------------------
 *INDEX: entry point
 *-------------------------------------------------------------------------*)

let transformApps ccu g expr =
   (* collect expr info - wanting usage contexts and bindings *)
   let (z : xinfo) = xinfo_of_assembly g expr in
   (* For each val_spec, decide Some "transform", or None if not changing *)
   let vtrans = determineVTransforms g z in
   (* Diagnostics - summary of planned transforms *)
   if verbose then dprintf1 "note: detuple - %d functions transformed\n" (length (Zmap.keys vtrans));
   (if verbose then Zmap.iter dumpVTransform vtrans);
   (* Pass over term, rewriting bindings and fixing up call sites, under penv *)
   let penv = {g=g; transforms = vtrans; ccu = ccu} in
   (if verbose then dprintTerm "transformApps before:" expr);
   (if verbose then dprintf0   "transformApps: pass\n");
   let z = () in (* z=state, relic, to be removed *)
   let expr = pass_assembly penv expr in
   (if verbose then dprintTerm "transformApps after:" expr);
   (if verbose then dprintf0   "transformApps: done\n");
   expr
