
(* (c) Microsoft Corporation. All rights reserved *)
(*----------------------------------------------------------------------------
 * Loading initial context, reporting errors etc.
 *--------------------------------------------------------------------------*)

(*F# 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.Research.AbstractIL.Extensions.ILX
open Microsoft.FSharp.Compiler 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
module Ilsupp = Microsoft.Research.AbstractIL.Internal.Support 
module Ilread = Microsoft.Research.AbstractIL.BinaryReader 
module Il = Microsoft.Research.AbstractIL.IL 
module Ccuthunk = Microsoft.FSharp.Compiler.CcuThunk 
F#*) 

open Ildiag
open Il
open Range
open Ast
open Tc
open Tast
open Tastops
open List
open Env
open Printf
open Lexhelp
open Lib
open Ccuthunk
open Infos
open Csolve
open Typrelns
open Nameres

(*----------------------------------------------------------------------------
 * Latest CLR on machine
 *
 * Autoconfigue a default maximum_cli_library_version_to_depend_upon based upone
 * the latest CLR on the machine.
 *--------------------------------------------------------------------------*)

let compilerProcessRunsMono10() = 
  (*F#
    begin 
      let sysDir = System.Runtime.InteropServices.RuntimeEnvironment.GetRuntimeDirectory() in 
      sysDir.EndsWith("1.0") 
    end or
   F#*)
   false

let latestCLRVersionOnMachine_or_PretendNET20ForMono10_or_CLRVersionAlreadyLoaded() = 
    try 
      if verbose then dprintf0 ("Calling absilWriteGetMetadataVersion... \n");   
      let v = Ilsupp.absilWriteGetMetadataVersion () in 
      if verbose then dprintf1 "absilWriteGetMetadataVersion returned '%s'\n" v;   
      Il.parse_version (if String.get v 0 = 'v' then String.sub v 1 (String.length v - 1) else v)
    with Failure s -> 
      dprintf1 "warning, could not determine CLR version, using 1.0.3705: '%s'\n" s; flush stderr;
      Il.parse_version "1.0.3705" 


let configureIlxDefaultsForCLRVersion(desiredCLIMetadataVersion) = 
  begin 
    if Il.version_compare desiredCLIMetadataVersion (Il.parse_version "1.1.4322")  <= 0 then 
      Msilxlib.maximum_cli_library_version_to_depend_upon := Ilxsettings.CLI_Library_v1_0
    else if Il.version_compare desiredCLIMetadataVersion (Il.parse_version "2.0.0") >= 0 then 
      Msilxlib.maximum_cli_library_version_to_depend_upon := Ilxsettings.CLI_Library_v2_0rc
    else ()
  end;
  if Il.version_compare desiredCLIMetadataVersion (Msilxlib.minCLIMetadataVersion()) < 0 then begin
    Ilxerase.set_no_generics()
  end;
  (* The versions of the compiler and fsi.exe compiled with --no-tailcalls bake in this setting into *)
  (* the binary since it is not inferrable from the version of the CLR being used. *)
(*F#
#if NOTAILCALLS
   Ilxerase.set_notailcalls();
#else
#endif
F#*)
   ()


(*----------------------------------------------------------------------------
 * Parsing
 *--------------------------------------------------------------------------*)


let canon_of_filename filename = 
  let basic = Filename.basename filename in 
  String.capitalize (try Filename.chop_extension basic with _ -> basic)

let qname_of_modname m modname = QualifiedNameOfFile(mksyn_id m (text_of_lid modname))
let qname_of_filename m filename = QualifiedNameOfFile(mksyn_id m (canon_of_filename filename))
let qname_of_unique_path m p = QualifiedNameOfFile(mksyn_id m (String.concat "_" p))
let qualFileName_of_specs filename specs = 
    match specs with 
    | [ModuleSpec(modname,true,_,_,_,_,m)] -> qname_of_modname m modname
    | _ -> qname_of_filename (rangeN filename 1) filename
let qualFileName_of_impls filename specs = 
    match specs with 
    | [ModuleImpl(modname,true,_,_,_,_,m)] -> qname_of_modname m modname
    | _ -> qname_of_filename (rangeN filename 1) filename

let prepend_path_to_qname x (QualifiedNameOfFile(q)) = qname_of_unique_path q.idRange (path_of_lid x@[q.idText])
let prepend_path_to_impl x (ModuleImpl(p,c,d,e,f,g,h)) = ModuleImpl(x@p,c,d,e,f,g,h)
let prepend_path_to_spec x (ModuleSpec(p,c,d,e,f,g,h)) = ModuleSpec(x@p,c,d,e,f,g,h)

let prepend_path_to_input x inp = 
  match inp with 
  | ImplFileInput (ImplFile(b,q,impls)) -> ImplFileInput (ImplFile(b,prepend_path_to_qname x q,List.map (prepend_path_to_impl x) impls))
  | SigFileInput (SigFile(b,q,specs)) -> SigFileInput(SigFile(b,prepend_path_to_qname x q,List.map (prepend_path_to_spec x) specs))

let computeModuleName defaultNamespace filename m = 
  let modname = canon_of_filename filename in 
  let combined = 
    match defaultNamespace with 
    | None -> modname
    | Some ns -> text_of_path [ns;modname] in
  path_to_lid m (split_namespace combined)

let postParseModuleImpl i defaultNamespace filename impl = 
  match impl with 
  | NamedTopModuleImpl(x) -> x
  | AnonTopModuleImpl (defs,m)-> 
    let modname = computeModuleName defaultNamespace filename m in 
    ModuleImpl(modname,true,defs,emptyXMLDoc,[],None,m)
  | AnonNamespaceFragmentImpl (nsname,b,c,d,e,m)-> 
    ModuleImpl(nsname,b,c,d,e,None,m)

let postParseModuleSpec i  defaultNamespace filename intf = 
  match intf with 
  | NamedTopModuleSpec(x) -> x 
  | AnonTopModuleSpec (defs,m) -> 
    let modname = computeModuleName defaultNamespace filename m in 
    ModuleSpec(modname,true,defs,emptyXMLDoc,[],None,m)
  | AnonNamespaceFragmentSpec (nsname,b,c,d,e,m)-> 
    ModuleSpec(nsname,b,c,d,e,None,m)

let postParseModuleImpls defaultNamespace filename impls = 
  let impls = list_mapi (fun i x -> postParseModuleImpl i defaultNamespace filename x) impls in 
  let qualName = qualFileName_of_impls filename impls in 
  ImplFileInput(ImplFile(filename,qualName,impls))
  
let postParseModuleSpecs defaultNamespace filename specs = 
  let specs = list_mapi (fun i x -> postParseModuleSpec i defaultNamespace filename x) specs in 
  let qualName = qualFileName_of_specs filename specs in 
  SigFileInput(SigFile(filename,qualName,specs))

let sigSuffixes = [".mli";".fsi"]
let implSuffixes = [".ml";".fs";".fsscript";".fsx"]
let parseInput lexer (lexbuf:Lexing.lexbuf) defaultNamespace filename = 
  let lower = String.lowercase filename in 
  if List.exists (Filename.check_suffix lower) implSuffixes  then  begin
    let impl = Pars.implementationFile lexer lexbuf in
    postParseModuleImpls defaultNamespace filename impl 
  end else if List.exists (Filename.check_suffix lower) sigSuffixes  then  begin
    let intfs = Pars.signatureFile lexer lexbuf in
    postParseModuleSpecs defaultNamespace filename intfs
  end else failwith "parseInput: unknown file suffix"


(*----------------------------------------------------------------------------
 * ERROR REPORTING
 *--------------------------------------------------------------------------*)

let showFullPaths = ref false
let emacsStyle = ref false

let exnMsg e = Printexc.to_string e

let rec mark_of_err err = 
  match err with 
  | ErrorFromAddingConstraint(_,err2,_) -> mark_of_err err2 
  | ReservedKeyword(_,m)
  | IndentationProblem(_,m)
  | ErrorFromAddingTypeEquation(_,_,_,_,_,m) 
  | ErrorFromApplyingDefault(_,_,_,_,_,m) 
  | ErrorsFromAddingSubsumptionConstraint(_,_,_,_,_,m) 
  | FunctionExpected(_,_,m)
  | IndexOutOfRangeExceptionWarning(m)
  | FieldNotMutable (_,_,m) 
  | Recursion (_,_,_,_,m) 
  | InvalidRuntimeCoercion(_,_,_,m) 
  | IndeterminateRuntimeCoercion(_,_,_,m)
  | IndeterminateStaticCoercion (_,_,_,m)
  | StaticCoercionShouldUseBox (_,_,_,m)
  | CoercionTargetSealed(_,_,m)
  | UpcastUnnecessary(m)
  | Creflect.IgnoringPartOfQuotedTermWarning (_,m) 
  
  | TypeTestUnnecessary(m)
  | RuntimeCoercionSourceSealed(_,_,m)
  | OverrideDoesntOverride(_,_,_,_,_,m)
  | UnionPatternsBindDifferentNames m 
  | UnionConstrWrongArguments (_,_,_,m) 
  | AbstractType m 
  | RequiredButNotSpecified (_,_,_,_,m) 
  | FunctionValueUnexpected (_,_,m)
  | UnitTypeExpected (_,_,_,m )
  | UseOfAddressOfOperator m 
  | ThreadStaticWarning(m) 
  | NonUniqueInferredAbstractSlot (_,_,_,_,_,m) 
  | PotentialMutationWarning (_,m)
  | DeprecatedClassFieldInference(m)
  | LetRecCheckedAtRuntime m 
  | UpperCaseIdentifierInPattern m
  | LowerCaseConstructor m
  | RecursiveUseCheckedAtRuntime (_,_,m) 
  | LetRecEvaluatedOutOfOrder (_,_,_,m) 
  | Error (_,m)
  | InternalError (_,m)
  | FullAbstraction(_,m)
  | InterfaceNotRevealed(_,_,m) 
  | WrappedError (_,m)
  | Patcompile.MatchIncomplete (_,_,m)
  | Patcompile.RuleNeverMatched m 
  | ValNotMutable(_,_,m)
  | ValNotLocal(_,_,m) 
  | MissingFields(_,m) 
  | OverrideInAugmentation(m)
  | ValueRestriction(_,_,_,_,m) 
  | LetRecUnsound (_,_,m) 
  | Obsolete (_,m) 
  | Experimental (_,m) 
  | OCamlCompatibility (_,m) 
  | Deprecated(_,m) 
  | LibraryUseOnly(m) 
  | FieldsFromDifferentTypes (_,_,_,m) 
  | IndeterminateType(m)
  | TyconBadArgs(_,_,_,m) -> Some m
  | FieldNotContained(_,arf,frf,_) -> Some (id_of_rfield arf).idRange
  | ValueNotContained(_,_,aval,_,_) -> Some (range_of_val aval)
  | ConstrNotContained(_,aval,_,_) -> Some aval.uconstr_id.idRange
  | ExnconstrNotContained(_,aexnc,_,_) -> Some (range_of_tycon aexnc)
  | Duplicate(_,id) 
  | VarBoundTwice(id) 
  | UndefinedName(_,_,id,_) -> Some id.idRange 
  | UnresolvedOverloading(_,_,_,_,m) 
  | VirtualAugmentationOnNullValuedType(m)
  | NonVirtualAugmentationOnNullValuedType(m)
  | NonRigidTypar(_,m)
  | ConstraintSolverTupleDiffLengths(_,_,_,m,_) 
  | ConstraintSolverArrayKindMismatch(_,_,_,m,_) 
  | ConstraintSolverInfiniteTypes(_,_,_,m,_) 
  | ConstraintSolverMissingConstraint(_,_,_,m,_) 
  | ConstraintSolverTypesNotInEqualityRelation(_,_,_,m,_) 
  | ConstraintSolverError(_,m,_) 
  | ConstraintSolverTypesNotInSubsumptionRelation(_,_,_,m,_) 
  | ConstraintSolverRelatedInformation(_,m,_) 
  | SelfRefObjCtor(_,m) -> Some m
  | NotAFunction(_,_,mapp,marg) -> Some mapp
  | IntfImplInAugmentation(m) -> Some m
  | _ -> None

let rec errorNumber err = 
  match err with 
  (* DO NOT CHANGE THESE NUMBERS *)
  | ErrorFromAddingTypeEquation _ -> 1
  | FunctionExpected _ -> 2
  | NotAFunction _  -> 3
  | IndexOutOfRangeExceptionWarning _ -> 4
  | FieldNotMutable  _ -> 5
  | Recursion _ -> 6
  | InvalidRuntimeCoercion _ -> 7
  | IndeterminateRuntimeCoercion _ -> 8
  | IndeterminateStaticCoercion  _ -> 13
  | StaticCoercionShouldUseBox  _ -> 14
  | RuntimeCoercionSourceSealed _ -> 16 
  | OverrideDoesntOverride _ -> 17
  | UnionPatternsBindDifferentNames _  -> 18
  | UnionConstrWrongArguments  _ -> 19
  | UnitTypeExpected _  -> 20
  | RecursiveUseCheckedAtRuntime  _ -> 21
  | LetRecEvaluatedOutOfOrder  _ -> 22
  | Patcompile.MatchIncomplete _ -> 25
  | Patcompile.RuleNeverMatched _ -> 26
  | ValNotMutable _ -> 27
  | ValNotLocal _ -> 28
  | MissingFields _ -> 29
  | ValueRestriction _ -> 30
  | LetRecUnsound  _ -> 31
  | FieldsFromDifferentTypes  _ -> 32
  | TyconBadArgs _ -> 33
  | ValueNotContained _ -> 34
  | Deprecated  _ -> 35
  | ConstrNotContained _ -> 36
  | Duplicate _ -> 37
  | VarBoundTwice _  -> 38
  | UndefinedName _ -> 39
  | LetRecCheckedAtRuntime _ -> 40
  | UnresolvedOverloading _ -> 41
  | LibraryUseOnly _ -> 42
  | ErrorFromAddingConstraint _ -> 43
  | Obsolete _ -> 44
  | FullAbstraction _ -> 45
  | ReservedKeyword _ -> 46
  | SelfRefObjCtor _ -> 47
  | VirtualAugmentationOnNullValuedType _ -> 48
  | UpperCaseIdentifierInPattern _ -> 49
  | InterfaceNotRevealed _ -> 50
  | UseOfAddressOfOperator _ -> 51
  | PotentialMutationWarning _ -> 52
  | LowerCaseConstructor _ -> 53
  | AbstractType _ -> 54
  | DeprecatedClassFieldInference _ -> 55
  | ThreadStaticWarning _ -> 56
  | Experimental _ -> 57
  | IndentationProblem _ -> 58
  | CoercionTargetSealed _ -> 59 
  | OverrideInAugmentation _ -> 60
  | NonVirtualAugmentationOnNullValuedType _ -> 61
  | OCamlCompatibility _ -> 62
  | ExnconstrNotContained _ -> 63
  | NonRigidTypar _ -> 64
  | UpcastUnnecessary _ -> 66
  | TypeTestUnnecessary _ -> 67
  | Creflect.IgnoringPartOfQuotedTermWarning _ -> 68
  | IntfImplInAugmentation _ -> 69
  | NonUniqueInferredAbstractSlot _ -> 70
  | ErrorFromApplyingDefault _ -> 71
  | IndeterminateType _ -> 72
  | InternalError _ -> 73
   (* DO NOT CHANGE THE NUMBERS *)

  | WrappedError(e,_) -> errorNumber e   
 (* These do not have good error numbers yet *)
  | Error  _ -> 191
  | Failure _ -> 192
  | _ -> 193

let warningOnByDefault err = 
  match err with 
  | RecursiveUseCheckedAtRuntime _
  | LetRecEvaluatedOutOfOrder  _
  | PotentialMutationWarning _
  | FullAbstraction _ -> false
  | _ -> true

let rec splitRelatedErrors err = 
   match err with 
   | UnresolvedOverloading(a,overloads,errors,b,c) -> 
       UnresolvedOverloading(a,[],[],b,c), (overloads@errors)
  | ConstraintSolverRelatedInformation(fopt,m2,e) -> 
      let e,related = splitRelatedErrors e in 
      ConstraintSolverRelatedInformation(fopt,m2,e), related
  | ErrorFromAddingTypeEquation(g,denv,t1,t2,e,m) ->
      let e,related = splitRelatedErrors e in 
      ErrorFromAddingTypeEquation(g,denv,t1,t2,e,m) , related
  | ErrorFromApplyingDefault(g,denv,tp,defaultType,e,m) ->  
      let e,related = splitRelatedErrors e in 
      ErrorFromApplyingDefault(g,denv,tp,defaultType,e,m) , related
  | ErrorsFromAddingSubsumptionConstraint(g,denv,t1,t2,e,m) ->  
      let e,related = splitRelatedErrors e in 
      ErrorsFromAddingSubsumptionConstraint(g,denv,t1,t2,e,m), related
  | ErrorFromAddingConstraint(x,e,m) ->  
      let e,related = splitRelatedErrors e in 
      ErrorFromAddingConstraint(x,e,m), related
  | WrappedError (e,m) -> 
      let e,related = splitRelatedErrors e in 
      WrappedError(e,m), related
   | _ -> 
       err, []
   


let stringsOfTypes denv ts = 
    let _,ts,tpcs = PrettyTypes.prettifyN ts in 
    let denvMin = { denv with showImperativeTyparAnnotations=false; showConstraintTyparAnnotations=false  } in 
    List.map (NicePrint.string_of_typ denvMin) ts 

(* If the output text is different without showing constraints and/or imperative type variable *)
(* annotations then don't show them! *)
let minimalStringsOfTwoTypes denv t1 t2= 
    let _,(t1,t2),tpcs = PrettyTypes.prettify2 (t1,t2) in 
    let denvMin = { denv with showImperativeTyparAnnotations=false; showConstraintTyparAnnotations=false  } in 
    let min1 = NicePrint.string_of_typ denvMin t1 in 
    let min2 = NicePrint.string_of_typ denvMin t2 in 
    if min1 <> min2 then (min1,min2,"") else
    (NicePrint.string_of_typ denv t1, NicePrint.string_of_typ denv t2,  "\n" ^ NicePrint.string_of_typar_constraints denv tpcs)

(* Note: Always show imperative annotations when comparing value signatures *)
let minimalStringsOfTwoValues denv v1 v2= 
    let denvMin = { denv with showImperativeTyparAnnotations=true; showConstraintTyparAnnotations=false  } in 
    let min1 = bufs (fun buf -> NicePrint.output_qualified_val_spec denvMin buf v1) in
    let min2 = bufs (fun buf -> NicePrint.output_qualified_val_spec denvMin buf v2)  in
    if min1 <> min2 then (min1,min2) else
    let denvMax = { denv with showImperativeTyparAnnotations=true; showConstraintTyparAnnotations=true  } in 
    let max1 = bufs (fun buf -> NicePrint.output_qualified_val_spec denvMax buf v1) in
    let max2 = bufs (fun buf -> NicePrint.output_qualified_val_spec denvMax buf v2)  in
    max1,max2

let rec output_exn os exn = 
  match exn with 
  | ConstraintSolverTupleDiffLengths(denv,tl1,tl2,m,m2) -> 
      Printf.bprintf os "The tuples have different lengths.";
      (if start_line_of_range m <> start_line_of_range m2 then 
         Printf.bprintf os " See also %a" boutput_range m);
  | ConstraintSolverInfiniteTypes(denv,t1,t2,m,m2) ->
      let t1,t2,tpcs = minimalStringsOfTwoTypes denv t1 t2 in 
      Printf.bprintf os "The resulting type would be infinite when unifying '%s' and '%s'." t1 t2 ;
      (if start_line_of_range m <> start_line_of_range m2 then 
         Printf.bprintf os " See also %a" boutput_range m);
  | ConstraintSolverArrayKindMismatch(denv,t1,t2,m,m2) ->
      let t1,t2,tpcs = minimalStringsOfTwoTypes denv t1 t2 in 
      Printf.bprintf os "When compiling for .NET v1.x, the type '%s' is a .NET compatible array, and the type '%s' is an F#-specific array type. You must generate and manipulate .NET compatible array values using the functions in Compatibility.CompatArray." t1 t2;
      (if start_line_of_range m <> start_line_of_range m2 then 
         Printf.bprintf os " See also %a" boutput_range m);
  | ConstraintSolverMissingConstraint(denv,tpr,tpc,m,m2) -> 
      Printf.bprintf os "A type parameter is missing a constraint '%a'."  (NicePrint.output_typar_constraint denv) (tpr,tpc);
      (if start_line_of_range m <> start_line_of_range m2 then 
         Printf.bprintf os " See also %a" boutput_range m);
  | ConstraintSolverTypesNotInEqualityRelation(denv,t1,t2,m,m2) -> 
      let t1,t2,tpcs = minimalStringsOfTwoTypes denv t1 t2 in 
      Printf.bprintf os "The type '%s' does not match the type '%s'." t1 t2;
      (if start_line_of_range m <> start_line_of_range m2 then 
         Printf.bprintf os " See also %a" boutput_range m);
  | ConstraintSolverTypesNotInSubsumptionRelation(denv,t1,t2,m,m2) -> 
      let t1,t2,tpcs = minimalStringsOfTwoTypes denv t1 t2 in 
      Printf.bprintf os "The type '%s' is not compatible with the type '%s'%s." t2 t1 tpcs;
      (if start_line_of_range m <> start_line_of_range m2 then 
         Printf.bprintf os " See also %a" boutput_range m2);
  | ConstraintSolverError(msg,m,m2) -> 
      Printf.bprintf os "%s" msg;
      (if start_line_of_range m <> start_line_of_range m2 then 
         Printf.bprintf os " See also %a" boutput_range m2);
  | ConstraintSolverRelatedInformation(fopt,m2,e) -> 
      begin match e with 
      | ConstraintSolverError _ -> Printf.bprintf os "%a." output_exn e;
      | _ -> ()
      end;
      fopt |> Option.iter (Printf.bprintf os " %s.")

  | ErrorFromAddingTypeEquation(g,denv,t1,t2,ConstraintSolverTypesNotInEqualityRelation(_,t1',t2',m,_),_) 
     when type_equiv g t1 t1'
     &&   type_equiv g t2 t2' ->  
      let t1,t2,tpcs = minimalStringsOfTwoTypes denv t1 t2 in 
      Printf.bprintf os "This expression has type\n\t%s\nbut is here used with type\n\t%s%s" t2 t1 tpcs
  | ErrorFromAddingTypeEquation(g,denv,_,_,((ConstraintSolverTypesNotInSubsumptionRelation _ | ConstraintSolverError _) as e),m)  ->  
      output_exn os e
  | ErrorFromAddingTypeEquation(g,denv,t1,t2,e,m) ->
      if not (type_equiv g t1 t2) then (
          let t1,t2,tpcs = minimalStringsOfTwoTypes denv t1 t2 in 
          if t1<>t2 ^ tpcs then Printf.bprintf os "Type mismatch. Expecting a\n\t%s\nbut given a\n\t%s%s.\n" t1 t2 tpcs;
      );
      output_exn os e
  | ErrorFromApplyingDefault(g,denv,tp,defaultType,e,m) ->  
      let defaultType = List.hd (stringsOfTypes denv [defaultType]) in 
      Printf.bprintf os "Type constraint mismatch when applying the default type '%s' for a type inference variable. " defaultType;
      output_exn os e;
      Printf.bprintf os " Consider adding further type constraints." 
  | ErrorsFromAddingSubsumptionConstraint(g,denv,t1,t2,e,m) ->  
      if not (type_equiv g t1 t2) then (
          let t1,t2,tpcs = minimalStringsOfTwoTypes denv t1 t2 in 
          if t1<>t2 ^ tpcs then Printf.bprintf os "Type constraint mismatch. The type \n\t%s\nis not compatibile with type\n\t%s%s.\n" t2 t1 tpcs;
      );
      output_exn os e;
  | UpperCaseIdentifierInPattern(m) -> 
      Printf.bprintf os "Uppercase variable identifiers should not generally be used in patterns, and may indicate a misspelt constructor name." 
  | LowerCaseConstructor(m) -> 
      Printf.bprintf os "Datatype constructors may not be lowercase." 
  | ErrorFromAddingConstraint(_,e,_) ->  
      output_exn os e;
  | UnresolvedOverloading(_,overloads,sampleErrors,mtext,m) -> 
      Printf.bprintf os "%s." mtext
  | PossibleOverload(_,minfo,m) -> 
      Printf.bprintf os "\n\nPossible overload: '%s'." minfo
  | FunctionExpected(denv,t,m) ->
      Printf.bprintf os "This function takes too many arguments, or is used in a context where a function is not expected.\n"
  | IndexOutOfRangeExceptionWarning(m) ->
      Printf.bprintf os "Note, the OCaml-compatibility exception type Not_found is now mapped to System.Collections.Generic.KeyNotFoundException, rather than System.IndexOutOfRangeException. This means a number of F# functions now raise KeyNotFoundException instead of IndexOutOfRangeException. If you explicitly catch IndexOutOfRangeException in your code you may need to adjust your exception handling accordingly. This change was made because IndexOutOfRangeException is a CLR-reserved exception which should not be raised in user or library code"
  | InterfaceNotRevealed(denv,ity,m) ->
      Printf.bprintf os "The type implements the interface %s but this is not revealed by the signature. You should list the interface in the signature, as the interface will be discoverable via dynamic type casts and/or reflection" (NicePrint.pretty_string_of_typ denv ity)
  | NotAFunction(denv,t,mapp,marg) ->
      if start_col_of_range marg = 0 then 
        Printf.bprintf os "This value is not a function and cannot be applied. Did you forget a ';;' to terminate a declaration before a top-level expression?\n"
      else
        Printf.bprintf os "This value is not a function and cannot be applied"
      
  | TyconBadArgs(denv,tycon,d,m) -> 
      let exp = List.length (typars_of_tycon (deref_tycon tycon)) in 
      Printf.bprintf os "The type '%s' expects %d type argument(s) but is given %d" (full_name_of_tcref tycon) exp d
  | IndeterminateType(m) -> 
      Printf.bprintf os "Lookup on object of indeterminate type. A type annotation may be needed prior to this program point to constrain the type of the object. This may allow the lookup to be resolved"

  | Duplicate(k,id)  -> 
      Printf.bprintf os "Duplicate definition of %s '%s'" k (decompileOpName id.idText)
  | UndefinedName(_,k,id,avail) -> 
      Printf.bprintf os "The %s '%s' is not defined." k (decompileOpName id.idText)
  | InternalUndefinedTyconItem(k,tcref,s) ->
      Printf.bprintf os "internal error: the type %s did not contain the %s '%s'" (full_name_of_tcref tcref) k s
  | InternalUndefinedItemRef(k,smr,s) ->
      Printf.bprintf os "internal error: the module '%s' from compilation unit '%s' did not contain the %s '%s'" (full_name_of_nlpath smr) (name_of_ccu (ccu_of_nlpath smr)) k s
  | FieldNotMutable (denv,fref,m) -> 
      Printf.bprintf os "This field is not mutable"
  | FieldsFromDifferentTypes (denv,fref1,fref2,m) -> 
      Printf.bprintf os "The fields '%s' and '%s' are from different types" (name_of_rfref fref1) (name_of_rfref fref2)
  | VarBoundTwice(id) ->  
      Printf.bprintf os "'%s' is bound twice in this pattern." (decompileOpName id.idText)
  | Recursion (denv,id,ty1,ty2,m) -> 
      let t1,t2,tpcs = minimalStringsOfTwoTypes denv ty1 ty2 in 
      Printf.bprintf os "A use of the function '%s' does not match a type inferred elsewhere. The inferred type of the function is\n\t%s.\nThe type of the function required at this point of use is\n\t%s%s\nThis error may be due to limitations associated with generic recursion, i.e. type inference requires that generic functions and members are instantiated in only one way within their own recursive scope, e.g. within a 'let rec' collection or within a group of classes. Consider splitting these recursive bindings into two groups" (decompileOpName id.idText) t1 t2 tpcs
  | InvalidRuntimeCoercion(denv,ty1,ty2,m) -> 
      let t1,t2,tpcs = minimalStringsOfTwoTypes denv ty1 ty2 in 
      Printf.bprintf os "Invalid runtime coercion or type test from type %s to %s\n%s" t1 t2 tpcs
  | IndeterminateRuntimeCoercion(denv,ty1,ty2,m) -> 
      let t1,t2,tpcs = minimalStringsOfTwoTypes denv ty1 ty2 in 
      Printf.bprintf os "This runtime coercion or type test from type\n\t%s\n to \n\t%s\ninvolves an indeterminate type. Runtime type tests are not allowed on some types. Further type annotations are needed." t1 t2
  | IndeterminateStaticCoercion(denv,ty1,ty2,m) -> 
      let t1,t2,tpcs = minimalStringsOfTwoTypes denv ty1 ty2 in 
      Printf.bprintf os "The static coercion from type\n\t%s\nto \n\t%s\n involves an indeterminate type. Static coercions are not allowed on some types. Further type annotations are needed." t1 t2
  | StaticCoercionShouldUseBox(denv,ty1,ty2,m) ->
      let t1,t2,tpcs = minimalStringsOfTwoTypes denv ty1 ty2 in 
      Printf.bprintf os "A coercion from the value type \n\t%s\nto the type \n\t%s\nwill involve boxing. Consider using 'box' instead." t1 t2
  | AbstractType(m) -> 
      Printf.bprintf os "This type is 'abstract' since some abstract members have not been given an implementation. If this is intentional then add the '[<AbstractClass>]' attribute to your type"
  | NonRigidTypar(msg,m) -> 
      Printf.bprintf os "%s." msg

  | RuntimeCoercionSourceSealed(denv,ty,m) -> 
      let _,ty,tpcs = PrettyTypes.prettify1 ty in 
      if is_typar_ty ty 
      then Printf.bprintf os "The type '%a' cannot be used as the source of a type test or runtime coercion." (NicePrint.output_typ denv) ty 
      else Printf.bprintf os "The type '%a' does not have any proper subtypes and cannot be used as the source of a type test or runtime coercion." (NicePrint.output_typ denv) ty 
  | CoercionTargetSealed(denv,ty,m) -> 
      let _,ty,tpcs = PrettyTypes.prettify1 ty in 
      Printf.bprintf os "The type '%a' does not have any proper subtypes and need not be used as the target of a static coercion." (NicePrint.output_typ denv) ty 
  | UpcastUnnecessary(m) -> 
      Printf.bprintf os "This upcast is unnecessary - the types are identical"
  | TypeTestUnnecessary(m) -> 
      Printf.bprintf os "This type test or downcast will always hold"
  | Creflect.IgnoringPartOfQuotedTermWarning (msg,_) -> 
      Printf.bprintf os "%s" msg
  | OverrideDoesntOverride(denv,impl,minfoVirtOpt,g,amap,m) ->
      let sig1 = (string_of_override denv impl) in 
      begin match minfoVirtOpt with 
      | None -> 
          Printf.bprintf os "The member '%s' does not have the correct type to override any given virtual method." sig1
      | Some minfoVirt -> 
          Printf.bprintf os "The member '%s' does not have the correct type to override the corresponding abstract method." sig1;
          let sig2 = string_of_minfo_sig g amap m denv minfoVirt in
          if sig1 <> sig2 then 
              Printf.bprintf os " The required signature is '%s'."  sig2
      end
  | UnionConstrWrongArguments (denv,n1,n2,m) ->
      Printf.bprintf os "This constructor is applied to %d argument(s) but expects %d" n2 n1; 
  | UnionPatternsBindDifferentNames m -> 
      Printf.bprintf os "The two sides of this 'or' pattern bind different sets of variables";
  | ValueNotContained (denv,mref,v1,v2,s) ->
      let text1,text2 = minimalStringsOfTwoValues denv v1 v2 in
      Printf.bprintf os "Module '%s' contains\n\t%s\nbut its signature specifies\n\t%s\n%s." 
         (full_name_of_modref mref) 
         text1 
         text2 
         s
  | ConstrNotContained (denv,v1,v2,msg) ->
      Printf.bprintf os "The module contains the constructor\n\t%a\nbut its signature specifies\n\t%a\n%s." (NicePrint.output_uconstr denv) v1 (NicePrint.output_uconstr denv) v2 msg
  | ExnconstrNotContained (denv,v1,v2,s) ->
      Printf.bprintf os "The exception definitions are not compatible because %s. The module contains the exception definition\n\t%a\nbut its signature specifies\n\t%a" s (NicePrint.output_exnc denv) v1 (NicePrint.output_exnc denv) v2
  | FieldNotContained (denv,v1,v2,msg) ->
      Printf.bprintf os "The module contains the field\n\t%a\nbut its signature specifies\n\t%a\n%s." (NicePrint.output_rfield denv) v1 (NicePrint.output_rfield denv) v2 msg;
  | RequiredButNotSpecified (denv,mref,k,name,m) ->
      Printf.bprintf os "Module '%s' requires a %s '%t'." (full_name_of_modref mref) k name
  | UseOfAddressOfOperator _ -> 
      Printf.bprintf os "The address-of operator may result in non-verifiable code. Its use is restricted to passing byrefs to functions that require them.";
  | PotentialMutationWarning(s,m) -> Printf.bprintf os "%s." s
  | ThreadStaticWarning(m) -> 
       Printf.bprintf os "Thread static variables must be explicitly re-initialized on each thread where they are used. The initial binding will only be executed once, for the first thread that triggers the initialization bindings of the containing module. Other threads may see a zero or null value for this field. Consider using None or 0 as the initial value for such variables."
  | DeprecatedClassFieldInference(m) -> Printf.bprintf os "This lookup uses a deprecated feature, where a class type is inferred from the use of a class field label. Consider using a type annotation to make it clear which class the field comes from."

  | FunctionValueUnexpected (denv,ty,m) ->
      let _,ty,tpcs = PrettyTypes.prettify1 ty in 
      Printf.bprintf os "This expression is a function value, i.e. is missing arguments. Its type is %a" (NicePrint.output_typ denv) ty 
  | UnitTypeExpected (denv,ty,perhapsProp,m) ->
      let _,ty,tpcs = PrettyTypes.prettify1 ty in 
      Printf.bprintf os "This expression should have type 'unit', but has type '%a'." (NicePrint.output_typ denv) ty;
      if perhapsProp then Printf.bprintf os " If assigning to a property use the syntax 'obj.Prop <- expr'."; 
  | RecursiveUseCheckedAtRuntime (denv,v,m) -> 
      Printf.bprintf os "This recursive use will be checked for initialization-soundness at runtime. This warning is usually harmless, and may be suppressed by using #nowarn \"40\" or --no-warn 40";
  | LetRecUnsound (denv,[v],m) ->  
      Printf.bprintf os "The value '%s' will be evaluated as part of its own definition." (display_name_of_val (deref_val v))
  | LetRecUnsound (denv,path,m) ->  
      Printf.bprintf os "This value will be eventually evaluated as part of its own definition. You may need to make the value lazy or a function. Value '%s'%a" (name_of_val (deref_val (hd path))) (fun os -> iter (fun v -> Printf.bprintf os " will evaluate '%s'" (display_name_of_val (deref_val v)))) (tl path @ [hd path])
  | LetRecEvaluatedOutOfOrder (denv,v1,v2,m) -> 
      Printf.bprintf os "Bindings may be executed out-of-order because of this forward reference.";
  | LetRecCheckedAtRuntime _ -> 
      Printf.bprintf os "This and other recursive references will be checked for initialization-soundness at runtime because you are defining a recursive object or function value, rather than a simple recursive function. This warning is often harmless, and may be suppressed by using #nowarn \"40\" or --no-warn 40";
  | SelfRefObjCtor(false,m) -> 
      Printf.bprintf os "Self-referential uses within object constructors will be checked for initialization soundness at runtime. Consider placing self-references in members or within a trailing expression of the form '<ctor-expr> then <expr>'"
  | SelfRefObjCtor(true,m) -> 
      Printf.bprintf os "Self-referential uses within object constructors will be checked for initialization soundness at runtime. Consider placing self-references within 'do' statements after the last 'let' binding in a class"
  | VirtualAugmentationOnNullValuedType(m) ->
      Printf.bprintf os "The containing type can use 'null' as a representation value for its nullary union case. Invoking an abstract or virtual member or an interface implementation on a null value will lead to an exception. If necessary add a dummy data value to the nullary constructor to avoid 'null' being used as a representation for this type."
  | NonVirtualAugmentationOnNullValuedType(m) ->
      Printf.bprintf os "The containing type can use 'null' as a representation value for its nullary union case. This member will be compiled as a static member."

  | NonUniqueInferredAbstractSlot(g,denv,bindnm,bvirt1,bvirt2,m) ->
      Printf.bprintf os "The member '%s' doesn't correspond to a unique abstract slot based on name and argument count alone" bindnm;
      let ty1 = (Infos.typ_of_minfo bvirt1) in
      let ty2 = (Infos.typ_of_minfo bvirt2) in 
      let t1,t2,tpcs = minimalStringsOfTwoTypes denv ty1 ty2 in 
      Printf.bprintf os ". Multiple implemented interfaces have a member with this name and argument count";
      if t1 <> t2 then 
          Printf.bprintf os ". Consider implementing interfaces '%s' and '%s' explicitly" t1 t2;
      Printf.bprintf os ". Additional type annotations may be required to indicate the relevant override. This warning can be disabled using #nowarn \"70\" or --no-warn 70";

  | Error (s,m) -> Printf.bprintf os "%s." s
  | InternalError (s,m) -> Printf.bprintf os "internal error: %s. Please build a small example that reproduces this problem and report it to fsbugs@microsoft.com" s
  | FullAbstraction(s,m) -> Printf.bprintf os "%s." s
  | WrappedError (exn,m) -> output_exn os exn
  | Failure s -> Printf.bprintf os "%s" s
(*IF-OCAML*)  | Sys_error s -> Printf.bprintf os "%s" s (*ENDIF-OCAML*)
  | Invalid_argument s -> Printf.bprintf os "internal error (invalid argument): %s" s
  | Patcompile.MatchIncomplete (isComp,cexOpt,m) -> 
      Printf.bprintf os "Incomplete pattern matches on this expression.";
      begin match cexOpt with 
      | None -> ()
      | Some cex ->  Printf.bprintf os " The value '%s' will not be matched" cex
      end;
      if isComp then 
          Printf.bprintf os " Sequence expressions involving incomplete matches on 'for' and 'let' constructs are deprecated and will raise an IncompleteMatchFailure exception in a future revision of the F# language. Please use a separate match expression instead, adding 'yield! []' for the failing branch."
  | Patcompile.RuleNeverMatched m -> Printf.bprintf os "This rule will never be matched."
  | ValNotMutable(denv,vr,m) -> Printf.bprintf os "This value is not mutable."
  | ValNotLocal(denv,vr,m) -> Printf.bprintf os "This value is not local."
  | Obsolete (s, _) -> 
        Printf.bprintf os "This construct is deprecated";
        if s <> "" then Printf.bprintf os ". %s" s
  | Experimental (s, _) -> Printf.bprintf os "%s. This warning can be disabled using '--no-warn 57' or '#nowarn \"57\"'" s
  | OCamlCompatibility (s, _) -> Printf.bprintf os "This construct is for compatibility with OCaml. %sThis warning can be disabled using '--ml-compatibility', '--no-warn 62' or '#nowarn \"62\"'" (if s = "" then "" else s^". ")
  | Deprecated(s, _) -> Printf.bprintf os "This construct is deprecated: %s" s
  | LibraryUseOnly(_) -> Printf.bprintf os "This construct is deprecated: it is only for use in the F# library"
  | MissingFields(sl,m) -> Printf.bprintf os "The following fields require values: %s" (String.concat "," sl ^".")
  | ValueRestriction(denv,hassig,v,tp,m) -> 
      let denv = { denv with showImperativeTyparAnnotations=true; } in          
      let tps,tau = try_dest_forall_typ (type_of_val v) in
      if hassig then 
        if is_fun_ty tau && TopValData.hasNoArgs (arity2_of_val v) then 
          Printf.bprintf os "Value restriction. The signature for '%s' is generic\n\t%a\nEither define '%s' as a syntactic function, or instantiate the type parameters in the signature." 
            (display_name_of_val v) 
            (NicePrint.output_qualified_val_spec denv) v
            (display_name_of_val v) 
        else
          Printf.bprintf os "Value restriction. The signature for '%s' is generic\n\t%a\nYou may need to make '%s' into a function, or else instantiate the type parameters in the signature." 
            (display_name_of_val v) 
            (NicePrint.output_qualified_val_spec denv) v
            (display_name_of_val v) 
      else
        begin match member_info_of_val v with 
        | Some(membInfo) when 
            begin match membInfo.vspr_flags.memFlagsKind with 
            | MemberKindPropertyGet 
            | MemberKindPropertySet 
            | MemberKindConstructor -> true (* can't infer extra polymorphism *)
            | _ -> false                     (* can infer extra polymorphism *)
            end -> 
              Printf.bprintf os "Value restriction. Type inference has inferred the polymorphic signature\n\t%a\nbut this member is a constructor or property getter/setter, which cannot be more generic than the enclosing type. Add type constraints to indicate the exact types involved." 
                (NicePrint.output_qualified_val_spec denv) v 
        | _ -> 
          if is_fun_ty tau && TopValData.hasNoArgs (arity2_of_val v) then 
            Printf.bprintf os "Value restriction. Type inference has inferred the signature\n\t%a\nEither define '%s' as a syntactic function, or add a type constraint to instantiate the type parameters." 
              (NicePrint.output_qualified_val_spec denv) v 
              (display_name_of_val v) 
          else
            Printf.bprintf os "Value restriction. Type inference has inferred the signature\n\t%a\nEither define '%s' as a simple data term, make it a function, or add a type constraint to instantiate the type parameters." 
              (NicePrint.output_qualified_val_spec denv) v 
              (display_name_of_val v) 
        end
            
  | Parsing.Parse_error -> Printf.bprintf os "syntax error"
  | ReservedKeyword (s,m) -> Printf.bprintf os "%s" s
  | IndentationProblem (s,m) -> Printf.bprintf os "%s" s
  | OverrideInAugmentation(m) -> Printf.bprintf os "Override implementations should be given as part of the initial declaration of a type. While the current language specification allows overrides implementations in augmentations, this is likely to be deprecated in a future revision of the language"
  | IntfImplInAugmentation(m) -> Printf.bprintf os "Interface implementations should be given on the initial declaration of a type. While the current language specification allows interface implementations in augmentations, this is likely to be deprecated in a future revision of the language"

(*F#
  | :? System.IO.FileNotFoundException as e -> Printf.bprintf os "%s" e.Message
  | :? System.ArgumentException as e -> Printf.bprintf os "%s" e.Message
F#*)

  | e -> Printf.bprintf os "%s" (exnMsg e); ()

and output_plural os n = if n <> 1 then Printf.bprintf os "s"

let output_where warn exn os m = 
  let file = file_of_range m in 
  let file = if !showFullPaths then fullpath file else file in 
  let errkind = if warn then "warning " else "error "  in 
  if !emacsStyle then 
    Printf.bprintf os "File \"%s\", line %d, characters %d-%d: %s" file (start_line_of_range m) (start_col_of_range m) (end_col_of_range m) errkind
  else
    Printf.bprintf os "%s(%d,%d): %s" file (start_line_of_range m) (start_col_of_range m) errkind

(* used by fsc.exe but not vs *)
let rec output_err warn os err = 
  match err with 
  | ReportedError -> 
      dprintf0 "unexpected ReportedError"  (* this should actually never happen *)
  | _ -> 
      Printf.bprintf os "\n";
      begin match mark_of_err err with 
      | Some m -> output_where warn err os m 
      | None -> ()
      end;
      let num = errorNumber err in 
      Printf.bprintf os "FS%04d: " num;
      let mainError,relatedErrors = splitRelatedErrors err in
      output_exn os mainError;
      List.iter (output_exn os) relatedErrors
      
let output_err_context prefix fileLineFn os err =
  match mark_of_err err with
    | None   -> ()      
    | Some m -> let filename = file_of_range m in
                let lineA = start_line_of_range m in    
                let lineB = end_line_of_range m in
                let iA    = start_col_of_range m in
                let iB    = end_col_of_range m in
                let line  = fileLineFn filename lineA in
                if line="" then () else
                let iLen  = if lineA = lineB then (iB - iA) + 1 else 1 in
                Printf.bprintf os "%s%s\n"   prefix line;
                Printf.bprintf os "%s%s%s\n" prefix (String.make iA '-') (String.make iLen '^')

(*----------------------------------------------------------------------------
 * ARGUMENT PARSING and FILE PARSING
 * Search paths etc.
 *--------------------------------------------------------------------------*)

type dllinfo = 
    { dllinfo_filename: string;
      dllinfo_isFsharp: bool; 
      dllinfo_il_modul: Il.modul; 
      dllinfo_il_scoref: scope_ref }

type ccuinfo = 
    { ccuinfo_il_scoref: scope_ref; 
      ccuinfo_ccu: ccu;
      ccuinfo_optdata : Opt.modul_info option Lazy.t}

let core_framework = 
  [ "System";
    "System.Xml" ]

let extended_framework = 
  [ "System.Runtime.Remoting";
    "System.Runtime.Serialization.Formatters.Soap";
    "System.Data";
    "System.Drawing";
    "System.Web";
    "System.Web.Services";
    "System.Windows.Forms"; ]

let all_framework = [ "mscorlib" ] @ core_framework @ extended_framework

let (++) x s = x @ [s]

let _ = Msilxlib.namespace_ref := lib_MFCore_name

(*----------------------------------------------------------------------------
 * Configuration
 *--------------------------------------------------------------------------*)

type warningOptions = 
    { mutable globalWarnDisplay: bool option;
      mutable globalWarnAsError: bool;
      mutable specificWarnOn: int list;
      mutable specificWarnOff: int list; 
      mutable specificWarnAsError: int list }

let newWarningOptions () = 
    { globalWarnDisplay=None;
      globalWarnAsError=false;
      specificWarnOn=[];
      specificWarnOff=[]; 
      specificWarnAsError=[] }

type tcConfig =
    { mutable mscorlib_assembly_name : string;
      mutable sscli: bool;
      mutable autoResolveOpenDirectivesToDlls: bool;
      mutable noFeedback: bool;
      mutable implicitIncludeDir: string; (* normally "." *)
      mutable openBinariesInMemory: bool; (* false for command line, true for VS *)
      mutable openDebugInformationForLaterStaticLinking: bool; (* only for --standalone *)
      mutable fsharpBinariesDir: string;
      mutable compilingFslib: bool;
      mutable compilingMllib: bool;
      mutable includes: string list;
      mutable implicitOpens: string list;
      mutable useMLLib: bool;
      mutable useFsiAuxLib: bool;
      mutable framework: bool;
      mutable light: bool;
      mutable ifdefs: string list;
      mutable fullGenericsSupported: bool;
      mutable retargetable: string list;
      mutable desiredCLILibraryVersionOpt: Il.version_info option;
      mutable defaultNamespace: string option;
      mutable referencedDLLs: (range * string) list;
      mutable copyLocalFiles: (range * string) list;
      mutable clrRoot: string option;
      mutable manager: Il.manager option;
      mutable optimizeForMemory: bool;
      mutable inputCodePage: int option;
      warningOptions: warningOptions }

let newTcConfig (fsharpBinariesDir,warningOptions,optimizeForMemory,implicitIncludeDir) =
  { mscorlib_assembly_name = "mscorlib";
    light = false;
    noFeedback=false;
    ifdefs=[];
    implicitIncludeDir = implicitIncludeDir;
    autoResolveOpenDirectivesToDlls = false;
    sscli = false;
    openBinariesInMemory = false;
    openDebugInformationForLaterStaticLinking=false;
    fsharpBinariesDir=fsharpBinariesDir;
    compilingFslib=false;
    compilingMllib=false;
    useMLLib=true;
    useFsiAuxLib=false;
    implicitOpens=[];
    includes=[];
    framework=true;
    fullGenericsSupported=true;
    retargetable=[];
    desiredCLILibraryVersionOpt=None;
    defaultNamespace = None;
    referencedDLLs = [];
    copyLocalFiles = [];
    warningOptions = warningOptions;
    manager = None;
    clrRoot  = None;
    inputCodePage=None;
    optimizeForMemory=optimizeForMemory } 

type tcImports = 
    { tciBase: tcImports option;
      mutable dllinfos: dllinfo list;
      mutable dlltable: dllinfo namemap;
      mutable ccuinfos: ccuinfo list;
      mutable ccu_table: ccuinfo namemap;
      mutable disposeActions : (unit -> unit) list }

let newTcImports base =
    { tciBase=base;
      dllinfos=[];
      ccuinfos=[];
      dlltable = Namemap.empty;
      ccu_table = Namemap.empty;
      disposeActions=[] } 

let rec dllinfosOfTcImports tcImports = 
  (match tcImports.tciBase with None -> [] | Some x -> dllinfosOfTcImports x) @ tcImports.dllinfos
let rec ccuinfosOfTcImports tcImports = 
  (match tcImports.tciBase with None -> [] | Some x -> ccuinfosOfTcImports x) @ tcImports.ccuinfos

(* closing deliberately only closes this tcImports, not the ones up the chain *)
let disposeTcImports tcImports = 
    dprintf1 "disposing of TcImports, %d binaries\n" (List.length tcImports.disposeActions);
    List.iter (fun f -> f()) tcImports.disposeActions

(* This call can fail if no CLR is found *)
let clrRootInclude tcConfig = 
  match tcConfig.clrRoot with 
  | Some x -> [x]
  | None -> 
      (* When running fscp10.exe on Mono we lead everyone to believe we're doing .NET 2.0 compilation *)
      (* by default. *)
      if compilerProcessRunsMono10() then begin 
        (*F# 
          let mono10SysDir = System.Runtime.InteropServices.RuntimeEnvironment.GetRuntimeDirectory() in 
          assert(mono10SysDir.EndsWith("1.0"));
          let mono20SysDir = System.IO.Path.Combine(System.IO.Path.GetDirectoryName mono10SysDir, "2.0") in 
          if System.IO.Directory.Exists(mono20SysDir) then [mono20SysDir]
          else [mono10SysDir]
        F#*)
        (*IF-OCAML*)
          assert false
        (*ENDIF-OCAML*)
      end else begin 
          try [Ilsupp.clrInstallationDirectory ()] with e -> warning (e); [] 
      end


(*----------------------------------------------------------------------------
 * Typecheck and optimization environments on disk
 *--------------------------------------------------------------------------*)
open Pickle

let rname_OptData = "FSharpOptimizationData"
let rname_IntfData = "FSharpInterfaceData"
let is_IntfDataResource r = has_prefix r.resourceName rname_IntfData
let is_OptDataResource r = has_prefix r.resourceName rname_OptData
let get_IntfDataResourceCcuName r = drop_prefix (drop_prefix r.resourceName rname_IntfData) "."
let get_OptDataResourceCcuName r = drop_prefix (drop_prefix r.resourceName rname_OptData) "."

let is_PickledDefinitionsResource r = has_prefix r.resourceName Sreflect.Raw.pickledDefinitionsResourceNameBase

let unpickle_from_resource m u sref r = 
  match r.resourceWhere with 
  | Resource_local b -> unpickle_obj_with_dangling_ccus sref u (b())
  | _-> error(InternalError("unpickle_from_resource",m))
let mk_resource rname bytes = 
  { resourceName = rname;
    resourceWhere = Resource_local (fun () -> bytes);
    resourceAccess = Resource_public;
    resourceCustomAttrs = mk_custom_attrs [] }
let pickle_to_resource  scope rname p x = 
  { resourceName = rname;
    resourceWhere = (let bytes = pickle_obj_with_dangling_ccus scope p x in Resource_local (fun () -> bytes));
    resourceAccess = Resource_public;
    resourceCustomAttrs = mk_custom_attrs [] }

type pickled_modul_info =
  { mspec: modul_spec;
    compile_time_working_dir: string }
    
let pickle_modul_info minfo st = p_tup2 pickle_modul_spec p_string (minfo.mspec, minfo.compile_time_working_dir) st
let unpickle_modul_info st = let a,b = u_tup2 unpickle_modul_spec u_string st in { mspec=a; compile_time_working_dir=b }

let get_IntfData m sref r : (pickled_modul_info,_) pickled_data_with_dangling_references = 
    unpickle_from_resource m unpickle_modul_info sref r
let write_IntfData tcConfig tcGlobals exportRemapping ccu : Il.resource = 
    let mspec = (top_modul_of_ccu ccu) in 
    if !DebugPrint.layout_stamps then dprintf1 "Interface data beforeremap:\n%s\n" (Layout.showL (Layout.squashTo 192 (mspecL mspec)));
    let mspec = apply_export_remapping_to_mspec tcGlobals exportRemapping mspec in 
    if !DebugPrint.layout_stamps then dprintf1 "Interface data after remap:\n%s\n" (Layout.showL (Layout.squashTo 192 (mspecL mspec)));
    pickle_to_resource ccu (rname_IntfData^"."^name_of_ccu ccu) pickle_modul_info 
        { mspec=mspec; 
          compile_time_working_dir=tcConfig.implicitIncludeDir }

let get_OptData m sref ca = unpickle_from_resource m Opt.umodul_info sref ca
let write_OptData (ccu,modulInfo) = 
    if verbose then  dprintf1 "Optimization data after remap:\n%s\n" (Layout.showL (Layout.squashTo 192 (Opt.modul_infoL modulInfo)));
    pickle_to_resource ccu (rname_OptData^"."^name_of_ccu ccu) Opt.pmodul_info modulInfo

(*----------------------------------------------------------------------------
 * Find .NET DLLs and load them in AbsIL format
 *--------------------------------------------------------------------------*)

let makeAbsolute tcConfig path = 
   (*F# try  F#*)
            if Filename.is_relative path then
              Filename.concat tcConfig.implicitIncludeDir path
            else path 
   (*F# with :? System.ArgumentException -> path  F#*)


let searchPathsForLibraryFiles(tcConfig) = 
  clrRootInclude(tcConfig) @ 
  List.map (makeAbsolute tcConfig) tcConfig.includes ++
  tcConfig.implicitIncludeDir ++
  tcConfig.fsharpBinariesDir

let searchPathsForSourceFiles(tcConfig) = 
  List.map (makeAbsolute tcConfig) tcConfig.includes ++
  tcConfig.implicitIncludeDir ++
  Filename.concat tcConfig.fsharpBinariesDir ".." ++
  Filename.concat (Filename.concat tcConfig.fsharpBinariesDir "..") "src"

(* A bad bug in OCaml 3.08.1 prevents us from using Sys.file_exists or Sys.readdir. *)
let fileExists f = 
  (*F# System.IO.File.Exists(f) F#*)
  (*IF-OCAML*) try let is = open_in f in close_in is; true with _ -> false (*ENDIF-OCAML*)

exception FileNameNotResolved of exn 

let resolveFileUsingPaths paths m name =
  if not (Filename.is_relative name) && fileExists name then name else
  let res = paths |> choose (fun path ->  let n = Filename.concat path name in if fileExists n then  Some n else None) in
  match res with 
  | Some f -> f
  | None -> raise (FileNameNotResolved(Error("Unable to find the file "^name^" in any of\n "^(String.concat "\n " paths),m)))

let tryResolveLibFile tcConfig m nm = resolveFileUsingPaths (searchPathsForLibraryFiles tcConfig) m nm  
  
let resolveLibFile tcConfig m nm = try tryResolveLibFile tcConfig m nm  with FileNameNotResolved err -> raise err
let resolveSourceFile tcConfig m nm = try resolveFileUsingPaths (searchPathsForSourceFiles tcConfig) m nm  with FileNameNotResolved err -> raise err


(* Note: the returned binary reader is associated with the tcImports, i.e. when the tcImports are closed *)
(* then the reader is closed. *)
let openIlBinaryModule tcConfig tcImports mscorlibRefsOpt syslib filename = 
    let pdbPathOption = 
        (* We open the pdb file if one exists parallel to the binary we *)
        (* are reading, so that --standalone will preserve debug information. *)
        if tcConfig.openDebugInformationForLaterStaticLinking then 
            let pdbDir = (try Filename.dirname filename with _ -> ".") in 
            let pdbFile = (try Filename.chop_extension filename with _ -> filename)^".pdb" in 
            if fileExists pdbFile then 
              begin
                if verbose then dprintf2 "reading PDB file %s from directory %s\n" pdbFile pdbDir;
                Some pdbDir
              end 
            else None 
        else None in 

    let ilBinaryReader = 
        let opts = { Ilread.defaults with 
                        Ilread.mscorlib=(match mscorlibRefsOpt with
                                         | Some ilg -> ilg
                                         | None     -> let mscorlib_assembly_name_option = Some "mscorlib" in
                                                       mk_MscorlibRefs ScopeRef_local mscorlib_assembly_name_option);
                        Ilread.optimizeForMemory=tcConfig.optimizeForMemory;
                        Ilread.pdbPath = pdbPathOption; 
                        Ilread.manager=tcConfig.manager } in 
        if tcConfig.openBinariesInMemory && not syslib then 
          Ilread.open_binary_reader_in_memory filename opts
        else 
          Ilread.open_binary_reader filename opts in 
    tcImports.disposeActions <- (fun () -> Ilread.close_binary_reader ilBinaryReader) :: tcImports.disposeActions;
    Ilread.modul_of_binary_reader ilBinaryReader

(*----------------------------------------------------------------------------
 * Names to match up refs and defs for assemblies and modules
 *--------------------------------------------------------------------------*)

let name_of_scoref sref = 
    match sref with 
    | ScopeRef_local -> "<local>"
    | ScopeRef_module mref -> mref.modulRefName
    | ScopeRef_assembly aref -> aref.assemRefName
  
let name_of_module m = if module_is_mainmod m then assname_of_mainmod m else m.modulName

let ccusOfTcImportsInDeclOrder tcImports = 
    map (fun x -> x.ccuinfo_ccu) (ccuinfosOfTcImports tcImports)  

let mkScopeRefForIlModule tcConfig ilModule = 
    if module_is_mainmod ilModule then 
      let assref = assref_for_mainmod  ilModule in 
      let assref =
        if List.mem assref.assemRefName tcConfig.retargetable then 
          {assref with assemRefRetargetable=true; 
                       assemRefPublicKeyInfo=Some (ecma_public_token) }
        else assref  in 
    (* Fixup references to .NET-versioned assemblies if given a --cli-version switch *)
    (* REVIEW: this should not be based only on a string comparison *)
      let assref =
        try 
          if List.mem assref.assemRefName all_framework then 
            match tcConfig.desiredCLILibraryVersionOpt with 
            | None -> assref
            | Some v -> {assref with assemRefVersion = Some v; } 
          else assref  
       with _ -> assref in 
       ScopeRef_assembly assref
    else
      ScopeRef_module (modref_for_modul ilModule)


(*----------------------------------------------------------------------------
 * Relink blobs of saved data by fixing up ccus.
 *--------------------------------------------------------------------------*)

let fslib () = "FSharp.Core"^Msilxlib.suffix()
let mllib () = "FSharp.Compatibility"^Msilxlib.suffix()

let fsiaux () = "FSharp.Compiler.Interactive.Settings"  ^Msilxlib.suffix()
let fsiAuxSettingsModulePath = "Microsoft.FSharp.Compiler.Interactive.Settings"

(* NOTE! This does NOT do auto resolution. Consider using findCcuInfo instead! *)
let findDllInfo tcImports m n =
    let rec look t = 
        if Namemap.mem n t.dlltable then Namemap.find n t.dlltable
        else 
            match t.tciBase with 
            | Some t2 -> look t2 
            | None -> raise(Error ("Assembly "^n^" is required by an imported module but was not referenced. Available assemblies are: "^String.concat ";" (map (fun d -> name_of_scoref d.dllinfo_il_scoref) (dllinfosOfTcImports tcImports)),m)) in 
    look tcImports
      
(* This is the main "assembly reference --> assembly" resolution routine. *)
(* We parameterize by a fallback resolution function that will go and look for DLLs matching the assembly name *)
(* in the include search paths. *)
let findCcuInfoAux ccuResolveFallback tcConfig tcImports m n = 
    let rec look t = 
      if Namemap.mem n t.ccu_table then (
        (* dprintf1 "findCcuInfoAux: found %s\n" n; REMOVE DIAGNOSTICS *)
        (Namemap.find n t.ccu_table)
      ) else 
        match t.tciBase with 
        | Some t2 -> look t2 
        | None -> 
            match ccuResolveFallback (m,n) with
            | None -> raise(Error("Compilation unit "^n^" is required by an imported module but was not referenced. Available assemblies are: "^String.concat ";" (map (fun d -> name_of_ccu d.ccuinfo_ccu) (ccuinfosOfTcImports tcImports)),m)) 
            | Some res -> res in 
    look tcImports

let findCcuAux ccuResolveFallback tcConfig tcImports m n = (findCcuInfoAux ccuResolveFallback tcConfig tcImports m n).ccuinfo_ccu

(* This function is used to eagerly fix up references in F#-specific metadata during the initial unpickling of that metadata. *)
(* This includes references to .NET dlls. *)
let relinkCcu ccuResolveFallback tcConfig tcImports m data = Ccuthunk.fixup_ccu_thunks_in_pickled_data (findCcuAux ccuResolveFallback tcConfig tcImports m) data

(*----------------------------------------------------------------------------
 * Add an referenced assembly
 *
 * mscorlib is very special, and must be assigned to the ccu_thunk 
 * mscorilb_modul before any serious compilation is done. This enables 
 * us to equate many F# types with the types we read from mscorlib.
 *
 * Retargetable assembly refs are required for binaries that must run 
 * against DLLs supported by multiple publishers. For example
 * Compact Framework binaries must use this. However it is not
 * clear when else it is required, e.g. for Mono.
 *--------------------------------------------------------------------------*)

(* aux_mod_loader is used for multi-module assemblies *)
let mkAuxModuleLoaderForMultiModuleIlAssemblies ccuResolveFallback tcConfig tcImports mscorlibRefsOpt m syslib =
    let aux_mod_tab = Hashtbl.create 10 in 
    fun viewingScopeRef viewedScopeRef ->
        match viewedScopeRef with
        | ScopeRef_module modref -> 
            let nm = modref.modulRefName in
            if Hashtbl.mem aux_mod_tab nm then 
              viewingScopeRef,Hashtbl.find aux_mod_tab nm 
            else 
              let res = openIlBinaryModule tcConfig tcImports mscorlibRefsOpt syslib (resolveLibFile tcConfig m nm) in 
              Hashtbl.add aux_mod_tab nm res; 
              viewingScopeRef,res 
        (* Type Forwarders: dereference the forwarder to its IL *)
        | ScopeRef_assembly assref ->  
            let dllinfo = (findDllInfo tcImports m assref.assemRefName) in 
            dllinfo.dllinfo_il_scoref,dllinfo.dllinfo_il_modul
            
        | ScopeRef_local -> 
            error(InternalError("unexpected ScopeRef_local in exported type",m))

let registerDll tcImports dllinfo =
    tcImports.dllinfos <- tcImports.dllinfos ++ dllinfo;
    tcImports.dlltable <- Namemap.add (name_of_scoref dllinfo.dllinfo_il_scoref) dllinfo tcImports.dlltable

let registerCcu tcImports ccuinfo =
    tcImports.ccuinfos <- tcImports.ccuinfos ++ ccuinfo;
    tcImports.ccu_table <- Namemap.add (name_of_ccu ccuinfo.ccuinfo_ccu) ccuinfo tcImports.ccu_table

let checkAlreadyRegistered tcImports nm =
    let isAssemblyCalled nm dll = match dll.dllinfo_il_scoref with ScopeRef_assembly a -> a.assemRefName =  nm | _ -> false in
    List.exists (isAssemblyCalled nm) (dllinfosOfTcImports tcImports)

let new_ccu_mspec sref m nm mty =
    new_mspec (Some(CompPath(sref,[]))) taccessPublic (mksyn_id m nm) emptyXMLDoc [] (notlazy mty)
  
let prepareToImportReferencedIlDll ccuResolveFallback tcConfig tcImports mscorlibRefsOpt syslib m filename dllinfo =
    let ilModule = dllinfo.dllinfo_il_modul in 
    let sref = dllinfo.dllinfo_il_scoref in 
    let aref = (match sref with ScopeRef_assembly aref -> aref | _ -> failwith "prepareToImportReferencedIlDll: cannot reference .NET netmodules directly, reference the containing assembly instead") in

    if verbose then dprint_endline ("Converting IL assembly to F# data structures "^(name_of_scoref sref));
    let nm = (name_of_scoref sref) in 
    let mty = Import.mtyp_of_il_mainmod m false (mkAuxModuleLoaderForMultiModuleIlAssemblies ccuResolveFallback tcConfig tcImports mscorlibRefsOpt m syslib) aref ilModule in 
    let ccu_spec = 
      { ccu_fsharp=false;
        ccu_qname= Some(qualified_name_of_scoref sref);
        ccu_contents = new_ccu_mspec sref m nm mty ;
        ccu_scoref = sref;
        ccu_stamp = new_stamp();
        ccu_code_dir = tcConfig.implicitIncludeDir;  (* note: not an accurate value, but IL assemblies don't give us this information in any attributes. *)
        ccu_filename = Some filename } in 
    let ccu =  new_ccu nm ccu_spec in
    let ccuinfo = 
      { ccuinfo_ccu=ccu; 
        ccuinfo_il_scoref = sref; 
        ccuinfo_optdata = notlazy None } in 
    registerCcu tcImports ccuinfo;
    let phase2 () = [ccuinfo] in 
    phase2

let customAttributesOfIlModule ilModule = 
    dest_custom_attrs (if module_is_mainmod ilModule then (manifest_of_mainmod ilModule).manifestCustomAttrs else ilModule.modulCustomAttrs) 

let prepareToImportReferencedFsDll ccuResolveFallback tcConfig tcImports mscorlibRefsOpt syslib m filename dllinfo =
    let ilModule = dllinfo.dllinfo_il_modul in 
    let sref = dllinfo.dllinfo_il_scoref in 
    let modname = name_of_scoref sref in 
    if verbose then dprint_endline ("Converting F# assembly to F# data structures "^(name_of_scoref sref));
    let attrs = customAttributesOfIlModule ilModule in 
    assert (List.exists is_IntfDataVersionAttr attrs);
    if verbose then dprint_endline ("Relinking interface info from F# assembly "^modname);
    let resources = dest_resources ilModule.modulResources in 
    assert (List.exists is_IntfDataResource resources);
    let optDatas = 
        resources 
        |> chooseList (fun r -> if is_OptDataResource r then Some(get_OptDataResourceCcuName r,r) else None) in 
    let ccuRawDataAndInfos = 
        resources 
        |> List.filter is_IntfDataResource 
        |> List.map (fun iresource -> 
            let ccu_name = get_IntfDataResourceCcuName iresource in 
            let data = get_IntfData m sref iresource in

            (* Look for optimization data in a file *)
            let optDataFromFile = 
                let optDataFileName = (Filename.chop_extension filename)^".optdata" in
                if fileExists optDataFileName then 
                   try Some(ccu_name,mk_resource optDataFileName (readBinaryFile optDataFileName))
                   with _ -> None 
                else None in 
            let optDatas = Option.to_list optDataFromFile @ optDatas in 
            if isNil optDatas then warning(Error(Printf.sprintf "No optimization information found for compilation unit '%s'" ccu_name,m));
            let optDatas = pmap_of_list (Option.to_list optDataFromFile @ optDatas) in
                 
            let minfo : pickled_modul_info = data.ie_raw in 
            let mspec = minfo.mspec in

            (* F# DLLs are allowed to include types not covered by the F# type system, arising from static linking of C# components *)
            (* Fold these into the data for last CCU *)
            let mspec =
                if module_is_mainmod ilModule then  
                   let aref = (match sref with ScopeRef_assembly aref -> aref | _ -> failwith "importReferencedFsDll") in
                   if verbose then dprintf1 "converting IL fragments for CCU %s\n" ccu_name; 
                   let mty2 = Import.mtyp_of_il_mainmod m true (mkAuxModuleLoaderForMultiModuleIlAssemblies ccuResolveFallback tcConfig tcImports mscorlibRefsOpt m syslib) aref ilModule in 
                   let mspec2 = new_ccu_mspec sref m modname mty2 in 
                   if verbose then dprintf1 "combining IL fragments with F# fragments for CCU %s\n" ccu_name; 
                   combine_mspecs mspec mspec2
                else 
                   mspec in 

            (* Take a stab at working out where the code for this DLL is likely to live. Used by Visual Studio etc. *)
            let code_dir = 
                let norm s = 
                    let s = String.lowercase s in 
                    let buf = Buffer.create(String.length s) in 
                    for i = 0 to String.length s - 1 do 
                        Buffer.add_char buf (if s.[i] = '\\' then '/' else s.[i]) 
                    done; 
                    let s = Buffer.contents buf in
                    if has_suffix s "/" then drop_suffix s "/" else s in 
                let equiv_filenames_approx f1 f2 =
                  try 
                    let f1 = norm (if Filename.is_relative f1 then Filename.concat tcConfig.implicitIncludeDir f1 else f1)  in 
                    let f2 = norm (if Filename.is_relative f2 then Filename.concat tcConfig.implicitIncludeDir f2 else f2) in 
                    let rec check f1 f2 = 
                        let b1 = Filename.basename f1 in 
                        let b2 = Filename.basename f2 in 
                        let d1 = Filename.dirname f1 |> norm in 
                        let d2 = Filename.dirname f2 |> norm in 
                        (b1 = b2) && (if (d1 = "") || (d2 = "") || d1="." || d2 = "." then d1 = d2 else  (d1 = d2) || check d1 d2) in 
                    check f1 f2 
                  with _ -> false in
                      
                (* dprintf3 "loading, filename = %s, Filename.dirname filename = %s, tcConfig.fsharpBinariesDir = %s\n" filename (Filename.dirname filename) tcConfig.fsharpBinariesDir ; *)
                if equiv_filenames_approx (Filename.dirname filename) tcConfig.fsharpBinariesDir  then (
                    Filename.concat tcConfig.fsharpBinariesDir ".."
                ) else  (
                     minfo.compile_time_working_dir 
                ) in 
            let ccu = 
               new_ccu ccu_name
                { ccu_scoref=sref;
                  ccu_stamp = new_stamp();
                  ccu_filename = Some filename; 
                  ccu_qname= Some(qualified_name_of_scoref sref);
                  ccu_code_dir = code_dir;  (* note: in some cases we fix up this information later *)
                  ccu_fsharp=true;
                  ccu_contents = mspec; } in 

             let optdata = 
                let report res= 
                  begin match res with 
                  | Some _ -> if verbose then dprintf1 "found optimization data for CCU %s\n" ccu_name; 
                  | None -> if verbose then dprintf1 "*** no optimization data for CCU %s, was DLL compiled with --no-optimization-data??\n" ccu_name; 
                  end;
                  res in 
                lazy ((Map.tryfind ccu_name optDatas |> Option.map (get_OptData m sref >> relinkCcu ccuResolveFallback tcConfig tcImports m)) |> report) in 
            data,{ ccuinfo_ccu=ccu; 
                   ccuinfo_optdata=optdata; 
                   ccuinfo_il_scoref = sref }  ) in 
                 
    (* Register all before relinking to cope with mutually-referential ccus *)
    ccuRawDataAndInfos |> List.iter (snd >> registerCcu tcImports);
    let phase2 () = 
        (* Relink *)
        (* dprintf1 "Phase2: %s\n" filename; REMOVE DIAGNOSTICS *)
        ccuRawDataAndInfos |> List.iter (fun (data,_) -> relinkCcu ccuResolveFallback tcConfig tcImports m data |> ignore);
        List.map snd ccuRawDataAndInfos in
    phase2

let registerAndPrepareToImportReferencedDll ccuResolveFallback warnIfAlreadyLoaded tcConfig tcImports mscorlibRefsOpt syslib (m,nm) =
    if verbose then dprintf1 "Resolving file name %s...\n" nm;
    let filename = resolveLibFile tcConfig m nm in
    if verbose then dprintf1 "Attempting to open scope on binary %s...\n" nm; 
    let ilModule = openIlBinaryModule tcConfig tcImports mscorlibRefsOpt syslib filename in 
    if verbose then dprintf1 "Opened scope on binary %s...\n" nm;
    let modname = name_of_module ilModule  in 
    if checkAlreadyRegistered tcImports modname then begin
        
        (* dprintf1 "ALREADY REGISTERED: %s\n" filename; REMOVE DIAGNOSTICS *)
        (*
        if warnIfAlreadyLoaded then 
          warning(Error("Ignoring multiple references to assembly "^modname,m));
        *)
        let dllinfo = findDllInfo tcImports m modname in
        (* was: dllinfo,(fun () -> []) *)
        let phase2() = [(findCcuInfoAux (fun _ -> None (* no auto resolve *)) tcConfig tcImports m modname)] in
        dllinfo,phase2
    end else begin
        let sref = mkScopeRefForIlModule tcConfig ilModule in
        let dllinfo = {dllinfo_il_modul=ilModule; 
                       dllinfo_filename=filename;
                       dllinfo_isFsharp=true; 
                       dllinfo_il_scoref = sref } in 
        registerDll tcImports dllinfo;
        let attrs = customAttributesOfIlModule ilModule in 
        let phase2 = 
            if (List.exists is_IntfDataVersionAttr attrs) then 
                if not (List.exists (is_matching_IntfDataVersionAttr (Il.parse_version Ilxconfig.version)) attrs) then begin
                  errorR(Error("The F#-compiled DLL '"^filename^"' needs to be recompiled to be used with this version of F#",m));
                  prepareToImportReferencedIlDll ccuResolveFallback tcConfig tcImports mscorlibRefsOpt syslib m filename dllinfo
                end else begin                
                  (* dprintf1 "Importing F# DLL '%s'...\n" filename; REMOVE DIAGNOSTICS *)
                  prepareToImportReferencedFsDll ccuResolveFallback tcConfig tcImports mscorlibRefsOpt syslib m filename dllinfo
                end
            else (
                (* dprintf1 "Importing DLL '%s'...\n" filename; REMOVE DIAGNOSTICS *)
                prepareToImportReferencedIlDll ccuResolveFallback tcConfig tcImports mscorlibRefsOpt syslib m filename dllinfo
            ) in
        dllinfo,phase2
    end

let registerAndImportReferencedDllsAux ccuResolveFallback  tcConfig tcImports mscorlibRefsOpt warnIfAlreadyLoaded syslib nms =
    let dllinfos,phase2s = List.split (List.map (registerAndPrepareToImportReferencedDll  ccuResolveFallback warnIfAlreadyLoaded tcConfig tcImports mscorlibRefsOpt syslib) nms) in
    let ccuinfos = List.concat (List.map (fun phase2 -> phase2()) phase2s)  in 
    dllinfos,ccuinfos
  
let doRegisterAndImportReferencedDllsAux ccuResolveFallback tcConfig tcImports mscorlibRefs syslib nms = 
    try ignore(registerAndImportReferencedDllsAux ccuResolveFallback tcConfig tcImports (Some mscorlibRefs) true syslib nms)
    with e -> errorRecoveryPoint e; ()

let rec findCcuAutoResolveFallback (tcConfig,tcImports,mscorlibRefs) (m,ccuName) = 
    let speculativeDllName = ccuName^".dll" in 
    let foundFileName = 
        try Some(tryResolveLibFile tcConfig m speculativeDllName)
        with FileNameNotResolved _ -> 
           (* use version number with FSI to do Load *)
           None in 
    match foundFileName with 
    | None -> None 
    | Some fileName -> 
         if not tcConfig.noFeedback then dprintf1 "Implicitly referencing '%s'...\n" fileName;
         doRegisterAndImportReferencedDllsAux (findCcuAutoResolveFallback(tcConfig,tcImports,mscorlibRefs))  tcConfig tcImports mscorlibRefs false [(m,speculativeDllName)];
         Some(findCcuInfoAux (fun _ -> None) tcConfig tcImports m ccuName)

(* entry points with knots tied *)

let findCcu (tcConfig,tcImports,tcGlobals) m name = 
    findCcuAux (findCcuAutoResolveFallback (tcConfig,tcImports,tcGlobals.ilg))  tcConfig tcImports m name

(*
let findCcuByAssemblyRef (tcConfig,tcImports,tcGlobals) m assref = 
    findCcuAux (findCcuAutoResolveFallback (tcConfig,tcImports,tcGlobals.ilg))  tcConfig tcImports m assref.assemRefName
*)

let doRegisterAndImportReferencedDlls (tcConfig,tcImports,mscorlibRefs) syslib nms =
    doRegisterAndImportReferencedDllsAux (findCcuAutoResolveFallback (tcConfig,tcImports,mscorlibRefs))  tcConfig tcImports mscorlibRefs syslib nms

let registerAndImportReferencedDlls (tcConfig,tcImports,mscorlibRefsOpt) warnIfAlreadyLoaded syslib nms =
    registerAndImportReferencedDllsAux 
        (match mscorlibRefsOpt with 
         | None -> (fun _ -> None)
         | Some mscorlibRefs -> (findCcuAutoResolveFallback (tcConfig,tcImports,mscorlibRefs)))
        tcConfig tcImports mscorlibRefsOpt warnIfAlreadyLoaded syslib nms

(*----------------------------------------------------------------------------
 * Add "#r" and "#I" declarations to the tcConfig
 *--------------------------------------------------------------------------*)

let hasSuffixCaseInsensitive filename suffix = (* case-insensitive *)
  Filename.check_suffix (String.lowercase filename) (String.lowercase suffix)

let isDll file = hasSuffixCaseInsensitive file ".dll" 
let isExe file = hasSuffixCaseInsensitive file ".exe" 

(* Add the reference and add the ccu to the type checking environment *)
let requireDLL tcConfig tcImports tcGlobals tcEnv m file = 
  if not (isDll file || isExe file) then 
    error(Error("unknown suffix for referenced .NET assembly '"^file,m));
  let dllinfos,ccuinfos = registerAndImportReferencedDlls (tcConfig,tcImports,Some tcGlobals.ilg) false false [(m,file)] in
  let tcEnv = List.fold_left (fun tcEnv ccuinfo -> Tc.add_nonlocal_ccu m tcEnv ccuinfo.ccuinfo_ccu) tcEnv ccuinfos in 
  tcEnv,(dllinfos,ccuinfos)

let requireIncludePath tcConfig m path = 
  let absolutePath = makeAbsolute tcConfig path in 
  let ok = 
(*F#  begin 
        let existsOpt = 
            try Some(System.IO.Directory.Exists(absolutePath)) 
            with e -> warning(Error("The search directory '"^path^"' is invalid",m)); None in
        match existsOpt with 
        | Some(exists) -> 
            if not exists then warning(Error("The search directory '"^absolutePath^"' could not be found",m));         
            exists
        | None -> false
      end; F#*)
(*IF-OCAML*)
      true
(*ENDIF-OCAML*) 
  in
  if ok && not (List.mem absolutePath tcConfig.includes) then 
     tcConfig.includes <- tcConfig.includes ++ absolutePath

let requireDLLReference tcConfig m path  = 
  if not (List.mem path (List.map snd tcConfig.referencedDLLs)) then 
       tcConfig.referencedDLLs <- tcConfig.referencedDLLs ++ (m,path)

let addCopyLocalFile tcConfig m nm  = 
    if not (List.mem nm (List.map snd tcConfig.copyLocalFiles)) then 
       tcConfig.copyLocalFiles <- tcConfig.copyLocalFiles ++ (m,nm)

let turnWarnOff tcConfig m s = 
    try 
        let n = int_of_string s in 
        tcConfig.warningOptions.specificWarnOff <- gen_insert (=) n tcConfig.warningOptions.specificWarnOff
    with err -> 
        errorR(Error("invalid warning number: '"^s^"'",m))

let processMetaCommand (dllRequireF,copyLocalF) tcConfig hash m state =
    try 
        match hash with 
        | IHash ("I",[path],m) ->
          requireIncludePath tcConfig m path; state
        | IHash ("nowarn",[d],m) ->
          turnWarnOff tcConfig m d; state
        | IHash (("reference" | "r"),[path],m) -> 
          dllRequireF state m path
        | IHash (("Reference" | "R"),[path],m) -> 
          copyLocalF state m path; dllRequireF state m path
        | _ -> (* warning(Error("This meta-command has been ignored",m)); *) state
    with e -> errorRecoveryPoint e; state

let rec warnOnIgnoredSpecDecls decls = 
    decls |> List.iter (fun d -> 
      match d with 
      | Spec_hash (h,m) -> warning(Error("Meta-commands inside modules are currently ignored",m)); 
      | Spec_module (_,subDecls,_) -> warnOnIgnoredSpecDecls subDecls
      | _ -> ())

let rec warnOnIgnoredImplDecls decls = 
    decls |> List.iter (fun d -> 
      match d with 
      | Def_hash (h,m) -> warning(Error("Meta-commands inside modules are currently ignored",m)); 
      | Def_module (_,subDecls,_,_) -> warnOnIgnoredImplDecls subDecls
      | _ -> ())

let processMetaCommandsFromModuleSpec dllRequireF tcConfig state (ModuleSpec(_,_,decls,_,_,_,_)) =
    List.fold_left (fun s d -> 
      match d with 
      | Spec_hash (h,m) -> processMetaCommand dllRequireF tcConfig h m s
      | Spec_module (_,subDecls,_) -> warnOnIgnoredSpecDecls subDecls; s
      | _ -> s)
     state
     decls 

let processMetaCommandsFromModuleImpl dllRequireF tcConfig state (ModuleImpl(_,_,decls,_,_,_,_)) =
    List.fold_left (fun s d -> 
      match d with 
      | Def_hash (h,m) -> processMetaCommand dllRequireF tcConfig h m s
      | Def_module (_,subDecls,_,_) -> warnOnIgnoredImplDecls subDecls; s
      | _ -> s)
     state
     decls
       
let processMetaCommandsFromInput dllRequireF  tcConfig inp state =
    match inp with 
    | SigFileInput(SigFile(_,_,specs)) -> List.fold_left (processMetaCommandsFromModuleSpec dllRequireF tcConfig) state specs
    | ImplFileInput(ImplFile(_,_,impls)) -> List.fold_left (processMetaCommandsFromModuleImpl dllRequireF tcConfig) state impls

let processMetaCommandsFromInputAsCommandLineFlags tcConfig inp = 
    processMetaCommandsFromInput ((fun () -> requireDLLReference tcConfig),(fun () -> addCopyLocalFile tcConfig)) tcConfig inp ()

(*----------------------------------------------------------------------------
 * 
 *--------------------------------------------------------------------------*)

let buildFoundationalTcImports tcConfig =
    let tcImports = newTcImports None in 
    let _,[sysCcu] = registerAndImportReferencedDlls (tcConfig,tcImports,None) true true [(rangeStartup,"mscorlib.dll")] in 
    sysCcu.ccuinfo_ccu,tcImports

let mkFslibAssemRef tcConfig tcImports = 
    let scoref = (findCcuInfoAux (fun _ -> None) tcConfig tcImports Range.rangeStartup (fslib())).ccuinfo_il_scoref in 
    match scoref with
    | ScopeRef_assembly aref             -> aref
    | ScopeRef_local | ScopeRef_module _ -> failwith "fslib_assembly_ref: not ScopeRef_assembly"

let buildFrameworkTcImports tcConfig baseTcImports sysCcu = 
    if verbose then dprint_endline ("building a new framework TcImports... ");   
    let mscorlibRefs   = Il.mk_MscorlibRefs (scoref_of_ccu sysCcu) None in 

    let tcImports = newTcImports (Some baseTcImports) in 
    let m = rangeStartup in 
    if tcConfig.framework then  doRegisterAndImportReferencedDlls (tcConfig,tcImports,mscorlibRefs) true (List.map (fun s -> m,s^".dll") core_framework);
    if tcConfig.framework && not tcConfig.sscli then  doRegisterAndImportReferencedDlls (tcConfig,tcImports,mscorlibRefs) true (List.map (fun s -> (m,s^".dll")) extended_framework);
    if verbose then dprintf1 "adding %s...\n" (fslib()^".dll"); 
    let fslibCcu =
      if tcConfig.compilingFslib then begin
        Ccuthunk.new_delayed_ccu_thunk (fslib()) 
      end else begin
        let _,[fslibCcuInfo] = registerAndImportReferencedDlls (tcConfig,tcImports,Some mscorlibRefs) true true [(m,Filename.concat tcConfig.fsharpBinariesDir (fslib()^".dll"))] in 
        Msilxlib.ilxLibraryAssemRef := Some (mkFslibAssemRef tcConfig tcImports);
        fslibCcuInfo.ccuinfo_ccu 
      end in 
    let tcGlobals = mk_tcGlobals(tcConfig.compilingFslib,sysCcu,mscorlibRefs,fslibCcu,tcConfig.fullGenericsSupported) in 
    tcGlobals,tcImports

let buildReferencedDllTcImports tcConfig tcGlobals baseTcImports = 
    let tcImports = newTcImports (Some baseTcImports) in 
    let m = rangeStartup in 
    if tcConfig.useMLLib && not tcConfig.compilingMllib && not tcConfig.compilingFslib then 
        doRegisterAndImportReferencedDlls (tcConfig,tcImports,tcGlobals.ilg) true [(m,Filename.concat tcConfig.fsharpBinariesDir (mllib()^".dll"))];
    if tcConfig.useFsiAuxLib then 
        doRegisterAndImportReferencedDlls (tcConfig,tcImports,tcGlobals.ilg) true [(m,Filename.concat tcConfig.fsharpBinariesDir (fsiaux()^".dll"))];
    if verbose then dprintf0 "adding referenced DLLS... \n";   
    doRegisterAndImportReferencedDlls (tcConfig,tcImports,tcGlobals.ilg) false tcConfig.referencedDLLs;
    if verbose then dprintf0 "added referenced DLLS...\n"; 
    tcImports

(*----------------------------------------------------------------------------
 * Build the whole lot
 *--------------------------------------------------------------------------*)

let buildTcImports(tcConfig) = 
    let sysCcu,foundationalTcImports = buildFoundationalTcImports tcConfig in 
    let tcGlobals,frameworkTcImports = buildFrameworkTcImports tcConfig foundationalTcImports sysCcu in 
    let tcImports = buildReferencedDllTcImports tcConfig tcGlobals frameworkTcImports in 
    tcGlobals,tcImports

(*----------------------------------------------------------------------------
 * Build the initial type checking environment
 *--------------------------------------------------------------------------*)

let initialImportedCcus tcImports = 
    map (fun x -> x.ccuinfo_ccu) (ccuinfosOfTcImports tcImports) 

let implicitOpenFSLib tcConfig = 
    if tcConfig.compilingFslib then []
    else [ lib_MF_name ; 
           lib_MFCore_name ; 
           lib_MFText_name ; 
           lib_MFColl_name ;
           lib_MFControl_name ;
           (* We'd prefer not to open this one but it is needed to access the canonical lazy 'or' and '&' pseudo-values *)
           lib_MFLanguagePrimitivesIntrinsicOperators_name;
           lib_MFOperators_name;
           lib_FSLib_Pervasives_name ]

let implicitOpenMLLib tcConfig = 
    if tcConfig.useMLLib && not tcConfig.compilingFslib  && not tcConfig.compilingMllib then 
      [ lib_MLLib_OCaml_name; 
        lib_MLLib_FSharp_name; 
        lib_MLLib_Pervasives_name ] 
    else []

let implicitOpenFsiAuxLib tcConfig = 
    if tcConfig.useFsiAuxLib then  [ fsiAuxSettingsModulePath ] else []

(* This is experimental to give OCaml-like file-by-file compilation *)
let autoModuleResolver tcConfig tcGlobals tcImports (id:ident) : modul_ref option = 
    let nm = id.idText in 
    let m = id.idRange in 
    let speculative_ccu_name = String.lowercase (String.sub nm 0 1)^String.sub nm 1 (String.length nm - 1) in 
    let speculativeDllName = speculative_ccu_name^".netmodule" in 
    try 
        let ccu = findCcu (tcConfig,tcImports,tcGlobals) id.idRange speculative_ccu_name in
        Some (mk_nonlocal_modref (nlpath_of_ccu ccu) nm)
    with _ -> 
        let found = 
            try ignore(tryResolveLibFile tcConfig m speculativeDllName); true
            with FileNameNotResolved _ -> false in 
        if not found then None 
        else begin 
            doRegisterAndImportReferencedDlls (tcConfig,tcImports,tcGlobals.ilg) false [(m,speculativeDllName)];
            let ccu = findCcu (tcConfig,tcImports,tcGlobals) id.idRange speculative_ccu_name in
            Some(mk_nonlocal_modref (nlpath_of_ccu ccu) nm)
        end 
  
let implicitOpen resolver m tcEnv p = 
    if verbose then dprintf1 "opening %s\n" p; 
    Tc.tc_open_namespace resolver m m tcEnv (path_to_lid m (split_namespace p)) 

let initTcEnv initm tcConfig tcImports tcGlobals =
    let resolver = autoModuleResolver tcConfig tcGlobals tcImports in
    let initm = start_range_of_range initm in 
    if verbose then dprintf0 "--- building initial tcEnv\n"; 
    let tcEnv = Tc.init_tenv tcGlobals initm (initialImportedCcus tcImports) in 
    if verbose then dprintf0 "--- opening implicit paths\n"; 
    if verbose then dprintf1 "--- initTcEnv, top modules = %s\n" (String.concat ";" (Namemap.domainL (nenv_of_tenv tcEnv).eModulesAndNamespaces)); 
    let tcEnv = fold_left (implicitOpen resolver initm) tcEnv (implicitOpenFSLib tcConfig @ implicitOpenMLLib tcConfig @ implicitOpenFsiAuxLib tcConfig @ tcConfig.implicitOpens )  in
    if verbose then dprintf1 "--- initTcEnv, top modules = %s\n" (String.concat ";" (Namemap.domainL (nenv_of_tenv tcEnv).eModulesAndNamespaces)); 
    if verbose then dprintf0 "<-- initTcEnv\n"; 
    tcEnv

(*----------------------------------------------------------------------------
 * TYPECHECK
 *--------------------------------------------------------------------------*)

(* The incremental state of type checking files *)
(* REVIEW: clean this up  *)

type topRootedSigs = (qualifiedNameOfFile, modul_typ) Zmap.t
type topRootedImpls = qualifiedNameOfFile Zset.t
type tcSigsAndImpls = TopSigsAndImpls of topRootedSigs * topRootedImpls * modul_typ list * modul_typ list

let qname_ord = (orderOn text_of_qualNameOfFile string_ord)

type tcState = 
  { tcsCcu: ccu;
    tcsCcuType: modul_spec;
    tcsNiceNameGen: niceNameGenerator;
    tcsTcSigEnv: Tc.tcEnv;
    tcsTcImplEnv: Tc.tcEnv;
    (* The accumulated results of type checking for this assembly *)
    tcsTopSigsAndImpls : tcSigsAndImpls }
  
let typecheckInitialState m ccuName tcConfig tcGlobals niceNameGen tcEnv0 =
  if verbose then dprintf0 "Typecheck (constructing initial state)....\n";
  (* Create a ccu to hold all the results of compilation *)
  let ccuType = new_ccu_mspec ScopeRef_local m ccuName (empty_mtype Namespace) in
  let ccu = 
    new_ccu ccuName 
      {ccu_fsharp=true;
       ccu_filename=None; 
       ccu_stamp = new_stamp();
       ccu_qname= None;
       ccu_code_dir = tcConfig.implicitIncludeDir; 
       ccu_scoref=ScopeRef_local;
       ccu_contents=ccuType } in 

  (* OK, is this is the F# library CCU then fix it up. *)
  if tcConfig.compilingFslib then 
    Ccuthunk.fixup_ccu_thunk tcGlobals.fslibCcu ccu;

  { tcsCcu= ccu;
    tcsCcuType=ccuType;
    tcsNiceNameGen=niceNameGen;
    tcsTcSigEnv=tcEnv0;
    tcsTcImplEnv=tcEnv0;
    tcsTopSigsAndImpls = TopSigsAndImpls (Zmap.empty qname_ord,Zset.empty qname_ord, [], []) }

let tcEnvOfTcState    s = s.tcsTcSigEnv
let nngOfTcState      s = s.tcsNiceNameGen
let topRootedSigsOfTcState s = let (TopSigsAndImpls(a,_,_,_)) = s.tcsTopSigsAndImpls     in a


(* Typecheck a single file or interactive entry into F# Interactive *)
let typecheckOneInput 
      checkForNoErrors 
      tcConfig 
      tcImports 
      tcGlobals 
      prefixPathOpt  
      (tcState:tcState)
      inp =
  try 
      let (TopSigsAndImpls(topRootedSigs,topRootedImpls,allSigTypes, allImplementedSigTypes)) = tcState.tcsTopSigsAndImpls in 
      let m = range_of_input inp in 
      let assemMap = findCcu (tcConfig,tcImports,tcGlobals) in 
      let resolver = autoModuleResolver tcConfig tcGlobals tcImports in
      let topAttrs, mimpls,tcEnvAtEnd,tcSigEnv,tcImplEnv,topSigsAndImpls,ccuType = 
        match inp with 
        | SigFileInput (SigFile(filename,qualNameOfFile, mspecs) as file) ->
            (* Check if we've seen this top module signature before. *)
            if Zmap.mem qualNameOfFile topRootedSigs then 
                errorR(Error("A signature for the file or module "^text_of_qualNameOfFile qualNameOfFile^" has already been specified",start_range_of_range m));

            (* Check if the implementation came first in compilation order *)
            if Zset.mem qualNameOfFile topRootedImpls then 
                errorR(Error("An implementation of file or module "^text_of_qualNameOfFile qualNameOfFile^" has already been given. Compilation order is significant in F# because of type inference. You may need to adjust the order of your files to place the signature file before the implementation. In Visual Studio files are type-checked in the order they appear in the project file, and to edit a project file right-click on the project node, unload the project, right-click again, edit the project file manually then the reload the project",m));

            (* Typecheck the signature file *)
            let (tcEnvAtEnd,tcEnv,smodulTypeTopRooted) = 
                Tc.typecheckOneSigFile (tcGlobals,tcState.tcsNiceNameGen,assemMap,tcState.tcsCcu,resolver,tcConfig.ifdefs) tcState.tcsTcSigEnv file in

            let topRootedSigs = Zmap.add qualNameOfFile  smodulTypeTopRooted topRootedSigs in  

            (* Hack open the prefixPath for fsi.exe *)
            let tcEnv = 
                match prefixPathOpt with 
                | None -> tcEnv 
                | Some prefixPath -> 
                    let resolver = autoModuleResolver tcConfig tcGlobals tcImports in
                    let m = range_of_qualNameOfFile qualNameOfFile in 
                    tc_open_namespace resolver m m tcEnv prefixPath in 

            (* Build the incremental results *)
            let allSigTypes = smodulTypeTopRooted :: allSigTypes in

            let ccuType = 
                let mtyp = combine_mtyps m  allSigTypes in 
                new_ccu_mspec ScopeRef_local m (name_of_ccu tcState.tcsCcu) mtyp in

            if verbose then dprintf2 "SigFileInput, nm = %s, qualNameOfFile = %s\n" (name_of_modul (top_modul_of_ccu tcState.tcsCcu)) (text_of_qualNameOfFile qualNameOfFile);
            emptyTopAttrs, 
            [],
            tcEnvAtEnd,
            tcEnv,
            tcState.tcsTcImplEnv,
            TopSigsAndImpls(topRootedSigs,topRootedImpls,allSigTypes, allImplementedSigTypes),
            tcState.tcsCcuType

        | ImplFileInput (ImplFile(filename,qualNameOfFile,impls) as file) ->
        
            (* Check if we've got an interface for this fragment *)
            let topRootedSigOpt = Zmap.tryfind qualNameOfFile topRootedSigs in 

            if verbose then dprintf3 "ImplFileInput, nm = %s, qualNameOfFile = %s, ?topRootedSigOpt = %b\n" filename (text_of_qualNameOfFile qualNameOfFile) (isSome topRootedSigOpt);

            (* Check if we've already seen an implementation for this fragment *)
            if Zset.mem qualNameOfFile topRootedImpls then 
              errorR(Error("An implementation of the file or module "^text_of_qualNameOfFile qualNameOfFile^" has already been given",m));

            let tcImplEnv = tcState.tcsTcImplEnv in 

            (* Typecheck the implementation file *)
            let topAttrs,implFile,tcEnvAtEnd = 
              Tc.typecheckOneImplFile  (tcGlobals,tcState.tcsNiceNameGen,assemMap,tcState.tcsCcu,resolver,checkForNoErrors,tcConfig.ifdefs) tcImplEnv topRootedSigOpt file in
            let hadSig = isSome topRootedSigOpt in
            let implFileSigType = sigTypeOfImplFile implFile in 

            if verbose then  dprintf0 "done typecheckOneImplFile...\n";
            let topRootedImpls = Zset.add qualNameOfFile topRootedImpls in  
    
            (* Only add it to the environment if it didn't have a signature *)
            let m = range_of_qualNameOfFile qualNameOfFile in 
            let tcImplEnv = Tc.add_local_top_rooted_mtyp m tcImplEnv implFileSigType in
            let tcSigEnv = if hadSig then tcState.tcsTcSigEnv else Tc.add_local_top_rooted_mtyp m tcState.tcsTcSigEnv implFileSigType in
            
            (* Hack open the prefixPath for fsi.exe *)
            let tcImplEnv = 
                match prefixPathOpt with 
                | None -> tcImplEnv 
                | Some prefixPath -> 
                    let resolver = autoModuleResolver tcConfig tcGlobals tcImports in
                    tc_open_namespace resolver m m tcImplEnv prefixPath in 

            let allImplementedSigTypes = implFileSigType :: allImplementedSigTypes in

            (* Add it to the CCU *)
            let ccuType = 
                (* the signature must be reestablished *)
                let mtyp = combine_mtyps m  allImplementedSigTypes in 
                new_ccu_mspec ScopeRef_local m (name_of_ccu tcState.tcsCcu) mtyp in

            if verbose then  dprintf0 "done typecheckOneInput...\n";

              (* Register the fully-qualified path of this top-fragment as having an implementation *)
            topAttrs,[implFile], 
            tcEnvAtEnd, 
            tcSigEnv, 
            tcImplEnv,
            TopSigsAndImpls(topRootedSigs,topRootedImpls,allSigTypes,allImplementedSigTypes),
            ccuType 
    in 
    (tcEnvAtEnd,topAttrs,mimpls),
    { tcState with 
        tcsCcuType=ccuType;
        tcsTcSigEnv=tcSigEnv;
        tcsTcImplEnv=tcImplEnv;
        tcsTopSigsAndImpls = topSigsAndImpls }
  with e -> errorRecoveryPoint e; (tcEnvOfTcState tcState,emptyTopAttrs,[]),tcState

let typecheckMultipleInputs checkForNoErrors tcConfig tcImports tcGlobals prefixPathOpt tcState inputs =
    let results,tcState =  map_acc_list (typecheckOneInput checkForNoErrors tcConfig tcImports tcGlobals prefixPathOpt) tcState inputs in
    let tcEnvsAtEndFile,topAttrs,mimpls = split3 results in 
    let topAttrs = fold_right combineTopAttrs topAttrs emptyTopAttrs in 
    let mimpls = List.concat mimpls in 
    (* This is the environment required by fsi.exe when incrementally adding definitions *)
    let tcEnvAtEndOfLastFile = (match tcEnvsAtEndFile with h :: _ -> h | _ -> tcEnvOfTcState tcState) in 
    if verbose then  dprintf0 "done typecheckMultipleInputs...\n";
    (tcEnvAtEndOfLastFile,topAttrs,mimpls),tcState


let typecheckClosedInputSet checkForNoErrors tcConfig tcImports tcGlobals prefixPathOpt tcState inputs =
    (* tcEnvAtEndOfLastFile is the environment required by fsi.exe when incrementally adding definitions *)
    let (tcEnvAtEndOfLastFile,topAttrs,mimpls),tcState = typecheckMultipleInputs checkForNoErrors tcConfig tcImports tcGlobals prefixPathOpt tcState inputs in 
    (* Publish the latest ovcontents to the CCU *)
    (deref_ccu_thunk tcState.tcsCcu).ccu_contents <- tcState.tcsCcuType;

    (* Check all interfaces have implementations *)
    let (TopSigsAndImpls(topRootedSigs,topRootedImpls,_,_)) = tcState.tcsTopSigsAndImpls in 
    topRootedSigs |> Zmap.iter (fun qualNameOfFile y ->  
      if not (Zset.mem qualNameOfFile topRootedImpls) then 
        errorR(Error("The module signature or signature file "^text_of_qualNameOfFile qualNameOfFile^" does not have a corresponding module implementation. If an implementation file appears to exist then check the 'module' and 'namespace' declarations in the signature and implementation files match", range_of_qualNameOfFile qualNameOfFile)));
    if verbose then  dprintf0 "done typecheckClosedInputSet...\n";
    let tassembly = TAssembly(mimpls) in
    tcState, topAttrs, tassembly,tcEnvAtEndOfLastFile

(*----------------------------------------------------------------------------
 * The argument parser is used by both the VS plug-in and the fsc.exe to
 * parse the include file path and other front-end arguments.
 *--------------------------------------------------------------------------*)

let parseArgs other (specs: (string * Arg.spec * string) list) args =
  let rec process_arg args = 
    match args with 
    | [] -> ()
    | arg :: t -> 
        if verbose then dprint_endline ("one arg... "^arg);
        let rec attempt l = 
          match l, t with 
          | ((s, Arg.Unit f, _) :: _) , _ when s = arg -> f (); t
          | ((s, Arg.Set f, _) :: _) , _ when s = arg -> f := true; t
          | ((s, Arg.Clear f, _) :: _) , _ when s = arg -> f := false; t
          | ((s, Arg.String f, _) :: _) , h2::t2 when s = arg -> f h2; t2
          | ((s, Arg.Int f, _) :: _) , h2::t2 when s = arg -> f (try int_of_string h2 with _ -> error(Failure (h2^" is not a valid integer argument"))); t2
          | ((s, Arg.Float f, _) :: _) , h2::t2 when s = arg -> f (try float_of_string h2 with _ -> error(Failure (h2^" is not a valid floating point argument"))); t2
          | ((s, Arg.Rest f, _) :: _) , _ when s = arg -> List.iter f t; []
          | ((s, _, _) :: _) , _  when s = arg -> error(Failure ("argument requires parameter: "^ arg)) 
          | (_ :: more) , _ -> attempt more 
          | [], _ -> if String.get arg 0 = '-' then error(Failure ("unrecognized argument: "^ arg)) else (other arg; t) in 
        let rest = attempt specs  in
        process_arg rest in
  process_arg args
    

let warningMem n l = n <> 0 && List.mem n l

let reportWarning info err = 
    let n = errorNumber err in 
    match info.globalWarnDisplay with
    | None -> (warningOnByDefault err || warningMem n info.specificWarnOn) && not (warningMem n info.specificWarnOff)
    | Some true -> not (warningMem n info.specificWarnOff)
    | Some false -> warningMem n info.specificWarnOn

let reportWarningAsError info err =
    info.globalWarnAsError or
    warningMem (errorNumber err) info.specificWarnAsError


let frontEndArgSpecs tcConfig = 
  [ "-I", Arg.String (fun s -> requireIncludePath tcConfig rangeStartup s), "Specify a directory for the include path."; 
    "-r", Arg.String (fun s -> requireDLLReference tcConfig rangeStartup s), "Reference an F# or .NET assembly.";
    "-R", Arg.String (fun s -> requireDLLReference tcConfig rangeStartup s; addCopyLocalFile tcConfig rangeStartup s), "Reference an F# or .NET assembly and copy it locally."; ],

  [ "--no-mllib",               Arg.Unit(fun ()   -> (tcConfig.useMLLib <- false)), "\n\tDo not implicitly reference FSharp.Compatibility.dll or open\n\tMicrosoft.FSharp.Compatibility.OCaml.Pervasives,\n\tuseful if you wish to avoid dependencies on modules."; 
    "--all-warnings",           Arg.Unit(fun ()   -> tcConfig.warningOptions.globalWarnDisplay <- Some true), "Print all warnings."; 
    "--no-warnings",            Arg.Unit(fun ()   -> tcConfig.warningOptions.globalWarnDisplay <- Some false), "Do not print any warnings."; 
    "--warn",                   Arg.Int (fun n    -> tcConfig.warningOptions.specificWarnOn <- gen_insert (=) n tcConfig.warningOptions.specificWarnOn), "Report the given specific warning."; 
    "--no-warn",                Arg.String (fun n -> turnWarnOff tcConfig rangeStartup n), "Do not report the given specific warning."; 
    "--all-warnings-as-errors", Arg.Unit(fun ()   -> tcConfig.warningOptions.globalWarnAsError <- true), "Treat all warnings as errors."; 
    "--warn-as-error",          Arg.Int (fun n    -> tcConfig.warningOptions.specificWarnAsError <- gen_insert (=) n tcConfig.warningOptions.specificWarnAsError), "Treat the given specific warning as an error."; 
    "--gnu-style-errors",       Arg.Unit (fun ()  -> emacsStyle := true), "Print line number errors in GNU style."; 
(*F#
    "--codepage", Arg.Int (fun n -> 
                     let encoding = System.Text.Encoding.GetEncoding(n) in 
                     tcConfig.inputCodePage <- Some(n)), "Specify the codepage to use when opening source files"; 
#if CLI_AT_LEAST_2_0
    "--utf8-output", Arg.Unit (fun () -> 
                     System.Console.OutputEncoding <- System.Text.Encoding.UTF8), "Output compiler messages in UTF-8 encoding"; 
#endif
F#*)

    "--namespace",        Arg.String (fun s  -> tcConfig.defaultNamespace <- Some s),                   "\n\tDefine the default root namespace which will contain subsequent\n\tF# modules.";
    "--open",             Arg.String (fun s  -> tcConfig.implicitOpens <- tcConfig.implicitOpens ++ s), "\n\tOpen the given module as an initial definition for each source file."; 
    "--fullpaths",        Arg.Set showFullPaths,                                                        "Compiler generates fully qualified paths.";
    "--clr-root",         Arg.String (fun s  -> tcConfig.clrRoot <- Some s),                            "\n\tUse to override where the compiler looks for mscorlib.dll and framework\n\tcomponents. Often needed when using --cli-version, compiling for\n\tSilverlight or tthe .NET Compact Framework"; 
    "--ml-compatibility", Arg.Unit   (fun () -> turnWarnOff tcConfig rangeStartup "62"),                "Ignore OCaml-compatibility warnings.";
    "--ml-keywords",      Arg.Unit   (fun () -> Lexhelp.permitFsharpKeywords := false),                 "\n\tTreat F# keywords which are not OCaml keywords as identifiers.";
    "--copy-local",       Arg.String (fun s  -> addCopyLocalFile tcConfig rangeStartup s),              "\n\tLocate the given file in the search paths and copy it\n\tlocally (e.g. a PDB)."; 
    "--compiling-fslib",  Arg.Unit   (fun () -> tcConfig.compilingFslib <- true; turnWarnOff tcConfig rangeStartup "42"; Msilxlib.compiling_msilxlib_ref := true), "<internal use only>"; 
    "--compiling-mllib",  Arg.Unit   (fun () -> tcConfig.compilingMllib <- true), "<internal use only>"; ]


