

// Copyright (c) Microsoft Corporation 2005-2006.
// This sample code is provided "as is" without warranty of any kind. 
// We disclaim all warranties, either express or implied, including the 
// warranties of merchantability and fitness for a particular purpose. 
//

#light
#nowarn "57"
//---------------------------------------------------------------------------
// Part I. Quotations.

open Microsoft.FSharp.Quotations
open Microsoft.FSharp.Quotations.Typed
open Microsoft.FSharp.Quotations.Raw
open StructuredFormat.LayoutOps

// This is a quotation
let e1 = <@ 1 @>

let e1a = <@ 1 @>

let e2 = <@ 1.0  @>

let e3 = <@ 1.0f  @>

let e4 = <@ "hello"  @>

e1
(e1 = e1)
(e1 = e1a)
// These are not well typed, as the type annotations are
// different
// (e1 = e2)
// (e1 < e2)
// (e2 < e1)


// Quotations must be closed up to the use of fully public values
// Not ok:
let e6 = let x = 5 in <@ x  @>

// Also not ok:
let f7 x = <@ x @>

// splicing constants: gives <@ 1 + 10 @>
let e7 = let x = 5 + 5 in <@ 1 + x  @>



// This is a quotation template
let template = <@ 1 + _ @>

// This is using a quotation template ("filling in the holes")
template <@ 2  @>
template (lift (2+4))

// splicing constants = holes + lift, 
let e7b = let x = 5 + 5 in <@ 1 + _  @> (lift x)

// This is a quotation template with two holes
let template2 = <@ 1 + _  + _ @>
 
// This is using a quotation template ("filling in the holes, left-to-right")
template2 <@ 2  @>  <@ 5  @>


/// Statements are just expressions in F#, so you can quote statements too...
<@ print_string "Hello"; print_string "World";   @>

/// Local declarations are also expressions in F#, so likewise...
<@ let x = 3 + 4 in x + 4  @>

/// So are recursive definitions...
<@ let rec f x = if x <= 2 then 1 else f x + f (x-1) in f @>

/// So are while loops...
<@ while true do print_string "hello" done  @>

/// So are for loops...
<@ for i = 1 to 10 do printf "i = %d\n" i done  @>

/// So are data constructions...
<@ Some([1;2;3]) @>
<@ { contents = 6 } @>

/// So are pattern matching and data projections (the pattern matching gets compiled away)...
<@ match Some(1) with Some y -> y | None -> 2 @>
<@ { contents = 6 }.contents @>

/// So are tuples...
<@ (1,2,3) @>
<@ match (1,2,3) with (a,b,c) -> b @>



// This shows the quotation of a use of a top-level definition
let sinExpr = <@ sin @>


//----------------------------------------------------
// Raw quotations (for quotation compilers)
//
// Typed quotations are just shallow-type-annotated
// expression trees.  
// You can access an underlying runtime-typed representation.

let re1 = <@@ 1  @@>
let re1a = <@@ 1  @@>
let re2 = <@@ 4.0  @@>
re1
(re1 = re2)
(re1 = re1)
(re1 = re1a)
(re1 < re2)
(re2 < re1)


// You can also quote raw expressions directly:
let re5 = <@@ "hello"  @@>

// You can also construct raw expressions dynamically/programatically.
// Here 'ef' stands for 'expression family'.
MkInt32 1
MkString "1"


// Quotations must be closed up to the use of fully public values
// Not ok:
let re6 = let x = 5 in <@@ x  @@>

// Also not ok:
let rf7 x = <@@ x @@>


// OK:
let x = 5
let re7 = <@@ x @@>

//type 'a option = Some of 'a | None

// You can query the structure of raw terms at several
// levels.  At the bottom-most level we have the 
// higher-order-abstract-syntax encoding 
// where everything is a Const/App/Lambda/Var/Quote (the
// latter used for nested quotations).
//
// Each of these correspond to non-overlapping families of
// expressions efConst, efApp, efLambda, efVar, efQuote.
//
// Here's an example:
let f e = 
    match e with
    | App (f,x) -> printf "APP: f = %A, x = %A\n" f x
    | Lambda(v,body) -> printf "LAMBDA: v = %A, body = %A\n" v body
    | Var(v) -> printf "VAR: v = %A\n" v 
    | Quote(x) -> printf "NESTED QUOTE: v = %A\n" x 
    | Double(x) -> printf "FLOAT, x = %A\n" x
    | Int32(x) -> printf "Int32, x = %A\n" x
    | _ -> printf "not known!\n"

// Here are some more raw expressions:
f e1.Raw
f e2.Raw
f <@@ 1 @@>
f <@@ 2 @@>
f <@@ 1.0 @@>
//f <@@ 1.3 @@>
f <@@ 1.2 @@>
f <@@ 1.4 @@>
f <@@ sin 1.4 @@>
f <@@ sin (sin 1.4) @@>
f <@@ max 1.2 1.4 @@>
f <@@ (+) 1 2 @@>
f <@@ fun x -> x + 1 @@>
f <@@ print_endline "hello"; 1+2 @@>

// Nested quotations are rarely used, but here are some
// examples:
f <@@ <@@ 1 @@> @@>
f <@@ f <@@ 1 @@> @@>



// This is a quotation-generating template
let templ1 = <@@ 1 + _  @@>

let templ2 = <@@ 2 + _  @@>

// This is using a quotation-generating template ("filling in the holes")
templ1 <@@ 2  @@>

// You can reuses the same generating template again ...
templ1 (templ2 <@@ 3 @@>)


// This is a quotation-generating template with two holes
let templ3 = <@ 1 + _  + _ @>
 
// This is using a quotation template ("filling in the holes, left-to-right")
templ3 <@ 2  @>  <@ 5  @>

/// Statements are just expressions in F#, so you can quote statements too...
<@ print_string "Hello"; print_string "World";   @>

/// Local declarations are also expressions in F#, so likewise...
<@ let x = 3 + 4 in x + 4  @>

/// So are recursive definitions...
<@ let rec f x = if x <= 2 then 1 else f x + f (x-1) in f @>

/// So are while loops...
<@ while true do print_string "hello" done  @>

/// So are for loops...
<@ for i = 1 to 10 do printf "i = %d\n" i done  @>

/// So are data constructions...
<@ Some([1;2;3]) @>
<@ { contents = 6 } @>

/// So are pattern matching and data projections (the pattern matching gets compiled away)...
<@ match Some(1) with Some y -> y | None -> 2 @>
<@ { contents = 6 }.contents @>

/// So are tuples...
<@ (1,2,3) @>
<@ match (1,2,3) with (a,b,c) -> b @>




// You can also use a (currently very limited) form of pattern matching...
begin match <@| sin _ |@> <@ sin 1.0 @> with 
 | Some(res) -> res
 | None -> failwith "no match"
end

// Overloading brings some subtleties. Overloading
// has been resolved to the extent that the types
// involved have been confirmed as valid for use with the
// overload.  However, the overload has not been rewritten
// away to the implementation for each overload.
//
(efTemplate <@@. (_:int) + (_:int) .@@>).Query(<@@ 1.0 + 1.0 @@>)
(efTemplate <@@. _ + _ .@@>).Query(<@@ 1 + 1 @@>)

let rec evalIntArith (inp : Expr) = 
  match inp with 
  | Template <@@. _ + _ .@@> (a,b) -> evalIntArith a + evalIntArith b
  | Template <@@. _ * _ .@@> (a,b) -> evalIntArith a * evalIntArith b
  | _-> 
  match (efTemplate <@@. _ / _ .@@>).Query inp with 
  | Some(a,b) -> evalIntArith a / evalIntArith b
  | None -> 
  match (efTemplate <@@. _ - _ .@@>).Query inp with 
  | Some(a,b) -> evalIntArith a - evalIntArith b
  | None -> 
  match (efTemplate <@@. _ % _ .@@>).Query inp with 
  | Some(a,b) -> evalIntArith a - evalIntArith b
  | None -> 
  match efInt32.Query(inp) with 
  | Some n -> n
  | None -> 
  failwith "evalIntArith"
  
// We can use templates as pseudo-patterns.  We expect this one to fail with "no match"
let rtemplate3 = <@@| sin (sin _) |@@> 
match  rtemplate3 <@@ sin 1.0 @@> with 
 | Some(res) -> failwith "whoa - that's not a match!"
 | None -> printf "no match - good!\n"


// This one will match:
match rtemplate3 <@@ sin (sin 1.0) @@> with 
 | Some(res) -> res
 | None -> failwith "no match"


// Programs that use imperative features can also be quoted
// (it's up to you to give a meaning to the expressions!)
let r = ref 4
r := 6
!r
r := 7
!r
f <@@ r := 8 @@>
f <@@ (:=) r 8 @@>
f <@@ r.contents <- 8 @@>

// Quoting a while loop:
<@@ while true do incr r done  @@>

// Decomposing a while loop:
let f2 e = 
  match efWhileLoop.Query(e) with
  | Some (e1,e2) -> printf "WHILE: e1 = %a, e2 = %a\n" output_any e1 output_any e2
  | None -> 
  printf "not known!\n"

f2 <@@ while true do incr r done  @@>


// This shows the quotation of a use of a top-level definition
let rsinExpr = <@@ sin @@>


// We can also show that it's a use of a top-level definition:
let sinTopDefData,sinTypeArgs = 
 match Raw.efAnyTopDefn.Query rsinExpr with 
 | Some(res) -> res
 | None -> failwith "no match"
 
// And we can resolve the top level definition to its body, which if you look
// carefully reveals the fact that F#'s sin function just calls System.Math.Sin(x)
// The 'sinTypeArgs' are empty, since 'sin' is not generic.
Raw.resolveTopDef(sinTopDefData,sinTypeArgs)

// Not all top-definitions have data.  These should be regarded as the
// 'primitives' of the language of quotations, in addition 
// to the operations covered by expression families
// like efApp, efRecdGet, efTupleGet etc.  Uses of these 
// constructs typically require special cases in quotation 
// compilers.
Raw.resolveTopDef(Option.get (efAnyTopDefn.Query <@@ (+) @@>))

// deepMacroExpandUntil is an example high-level utility
// that repeatedly reduces applications of functions to 
// arguments, replacing formals by actuals until a cutoff is reached.
// Here we don't use any cutoff ('false').  Normally the cutoff 
// would define a "target language" that is accepted by a compiler.
//
// deepMacroExpandUntil doesn't do many things, and it may not
// terminate when used with recursive expressions.  It's only an
// example.

let deepMacroExpand e = deepMacroExpandUntil (fun _ -> false) e
deepMacroExpand rsinExpr
deepMacroExpand <@@ (fun x -> x) 1  @@>
deepMacroExpand <@@ (fun f -> f) (fun (x:int) -> x) @@>
deepMacroExpand <@@ 1 + 1 @@>
deepMacroExpand <@@ let f x = x + x in f 1 @@>

// Quotations of a polymorphic library 
// currently need type annotations.
// <@ List.map @>

// So we can type-restrict it in this case:
let mapTerm = <@ List.map : (int -> string) -> int list -> string list @>
mapTerm

//---------------------------------------------------------------------------
// Part III. Database connectivity

#r @"C:\Program Files\LINQ Preview\Bin\System.Query.dll"
#r @"C:\Program Files\LINQ Preview\Bin\System.Data.DLinq.dll"
#r @"C:\Program Files\LINQ Preview\Bin\System.Xml.XLinq.dll"
#r @"Northwind.dll"
#r @"FLinq.dll"
#r @"System.Transactions.dll"

//t3 p3a
open System
open System.Collections
open System.Collections.Generic
open System.Query
open System.Data.DLinq
open nwind
open System.IO
open System.Windows.Forms
open System.Data.SqlClient
open System.Reflection
open System.Transactions

open Microsoft.FSharp.Bindings.DLinq
open Microsoft.FSharp.Bindings.DLinq.Query
open Microsoft.FSharp.Bindings.Linq
open Microsoft.FSharp.Bindings.XLinq

let dbPath = System.IO.Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.ProgramFiles), @"LINQ Preview\Data\Northwnd.mdf")
let sqlServerInstance = @".\SQLEXPRESS"
let connString = "AttachDBFileName='" + dbPath + "';Server='" + sqlServerInstance + "';Integrated Security=SSPI;enlist=false"

let db = new Northwind(connString)

let (|>) x f = f x

db.Customers
db.Customers |> select <@ fun c -> c @>

let dump = IEnumerable.to_array
db.Customers |> select <@ fun c -> c.City @> |> dump
db.Customers |> where <@ fun c -> c.City = "London" @> |> dump
db.Employees |> select <@ fun e -> (e.FirstName + " " + e.LastName), e.HomePhone @> |> dump

<@ fun (db : Northwind) -> db @>
                   

<@ fun (db : Northwind) -> db.Customers @>

<@ fun (db : Northwind) -> { for x in db.Customers -> x.City } @>

<@ fun (db : Northwind) -> { for x in db.Customers 
                                for y in db.Contacts -> x.City, y.City } @>


open System.IO
open System.Windows.Forms
//let dbFolder = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.ProgramFiles), @"LINQ Preview\Samples\VB\Sample Queries\SampleQueries")
let dbFolder = @"c:\misc"
if not (Directory.Exists(dbFolder)) then 
    MessageBox.Show("The May 2006 LINQ CTP available from msdn.microsoft.com must be installed to run this sample") |> ignore

  
let dbPath = Path.Combine(dbFolder,"Northwind.mdf")
let sqlServerInstance = @".\SQLEXPRESS"
let connString = "Database=northwind;AttachDBFileName='" + dbPath + "';Server='" + sqlServerInstance + "';Integrated Security=SSPI;User Instance=True"

#r @"Northwind.dll";;

let mutable db = new nwind.Northwind(connString)

db.Log <- Console.Out

let (|>) x f = f x

SQL <@ (db).Customers |> Seq.map (fun c -> c.Address) @>

db.Customers
SQL <@ { for c in (db).Customers -> c} @>

SQL <@ { for c in (db).Customers -> c.Address} @>
SQL <@ { for c in (db).Customers 
         for o in c.Orders 
         when c.City = "London" 
         -> o } @>

SQL <@ { for c in (db).Customers 
         for o in c.Orders 
         -> c.Address, o.ShipName } @>

SQL <@ { for c in (db).Customers 
         for d in (db).Contacts ->
         d.Address} @>

SQL <@ { for c in (db).Customers 
         when c.City = "London" 
         -> c } @>


SQL <@ { for c in (db).Customers 
         for d in db.Contacts 
         when d.City = "London" 
         -> c.Address, d.Address} @>

