// (c) Microsoft Corporation 2005-2007. 

#light

namespace Microsoft.FSharp.Text.StructuredFormat

    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Collections
    open Microsoft.FSharp.Primitives.Basics

    /// Data representing structured layouts of terms.  The representation
    /// of this data type is only for the consumption of formatting engines.
    type Layout =
     | Leaf of bool * obj * bool
     | Node of bool * layout * bool * layout * bool * joint
     | Attr of string * (string * string) list * layout
    /// Data representing joints in structured layouts of terms.  The representation
    /// of this data type is only for the consumption of formatting engines.
    and Joint =
     | Unbreakable
     | Breakable of int
     | Broken of int
    /// Data representing structured layouts of terms.  The representation
    /// of this data type is only for the consumption of formatting engines.
    and layout = Layout
    /// Data representing joints in structured layouts of terms.  The representation
    /// of this data type is only for the consumption of formatting engines.
    and joint = Joint


    type IEnvironment = 
      interface
        /// Return to the layout-generation 
        /// environment to layout any otherwise uninterpreted object
        abstract GetLayout : obj -> layout
        /// The maximum number of elements for which to generate layout for 
        /// list-like structures, or columns in table-like 
        /// structures.  -1 if no maximum.
        abstract MaxColumns : int
        /// The maximum number of rows for which to generate layout for table-like 
        /// structures.  -1 if no maximum.
        abstract MaxRows : int
      end
      
    type IFormattable = 
      interface
        /// Generate an object representing a structured formatting of the
        /// term.
        abstract GetLayout : IEnvironment -> layout
      end
    

    /// A layout is a sequence of strings which have been joined together.
    /// The strings are classified as words, separators and left and right parenthesis.
    /// This classification determines where spaces are inserted.
    /// A joint is either unbreakable, breakable or broken.
    /// If a joint is broken the RHS layout occurs on the next line with optional indentation.
    /// A layout can be squashed to for given width which forces breaks as required.
    module LayoutOps =

        /// The empty layout
        val emptyL     : layout
        /// Is it the empty layout?
        val isEmptyL   : layout -> bool
        
        /// An uninterpreted leaf, to be interpreted into a string
        /// by the layout engine. This allows leaf layouts for numbers, strings and
        /// other atoms to be customized according to culture.
        val objL       : obj -> layout

        /// An string leaf 
        val wordL      : string -> layout
        /// An string which requires no spaces either side.
        val sepL       : string -> layout
        /// An string which is right parenthesis (no space on it's left).
        val rightL     : string -> layout
        /// An string which is left  parenthesis (no space on the right).
        val leftL      : string -> layout

        /// Join, unbreakable. 
        val ( $$ )     : layout -> layout -> layout   
        /// Join, possible break with indent=0
        val ( ++ )     : layout -> layout -> layout   
        /// Join, possible break with indent=1
        val ( -- )     : layout -> layout -> layout   
        /// Join, possible break with indent=2 
        val ( --- )    : layout -> layout -> layout   
        /// Join broken with ident=0
        val ( @@ )     : layout -> layout -> layout   
        /// Join broken with ident=1 
        val ( @@- )    : layout -> layout -> layout   
        /// Join broken with ident=2 
        val ( @@-- )   : layout -> layout -> layout   

        /// Join layouts into a comma separated list.
        val commaListL : layout list -> layout
          
        /// Join layouts into a space separated list.    
        val spaceListL : layout list -> layout
          
        /// Join layouts into a semi-colon separated list.
        val semiListL  : layout list -> layout

        /// Join layouts into a list separated using the given layout.
        val sepListL   : layout -> layout list -> layout

        /// Wrap round brackets around layout.
        val bracketL   : layout -> layout
        /// Wrap square brackets around layout.    
        val squareBracketL   : layout -> layout
        /// Wrap braces around layout.        
        val braceL     : layout -> layout
        /// Form tuple of layouts.            
        val tupleL     : layout list -> layout
        /// Layout two vertically.
        val aboveL     : layout -> layout -> layout
        /// Layout list vertically.    
        val aboveListL : layout list -> layout

        /// Layout like an F# option.
        val optionL    : ('a -> layout) -> 'a option -> layout
        /// Layout like an F# list.    
        val listL      : ('a -> layout) -> 'a list   -> layout

        /// An tag is a name which can be wrapped around a layout.
        /// Tag names convey structure to renderers, e.g. for syntax coloring
        /// or hyperlinks.  Standard tag names are in the process
        /// of being finalized, though 'html:a' may be used for hyperlinks.
        /// tagAttrL provides settings for attributes associated with the
        /// tag, e.g. 'href' for 'html:a'.
        val tagL  : string -> layout -> layout
        
        /// See tagL
        val tagAttrL : string -> (string * string) list -> layout -> layout

        /// For limitting layout of list-like sequences (lists,arrays,etc).
        /// unfold a list of items using (project and z) making layout list via itemL.
        /// If reach maxLength (before exhausting) then truncate.
        val unfoldL : ('a -> layout) -> ('z -> ('a * 'z) option) -> 'z -> int -> layout list
