// (c) Microsoft Corporation 2005-2007. 
#light

namespace Microsoft.FSharp.Collections

open System

open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Collections
open Microsoft.FSharp.Primitives.Basics
open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators

type Permutation(arr: int array) =
    let arr = Array.copy arr
    do 
      (let arr2 = Array.zero_create arr.Length
       for i = 0 to arr.Length - 1 do 
           let x = arr.[i] 
           if x < 0 or x >= arr.Length then failwith "invalid permutation" 
           arr2.[x] <- 1
       for i = 0 to arr.Length - 1 do 
           if arr2.[i] <> 1 then failwith "invalid permutation")

    new (size:int,mappings:seq<int*int>) = 
      let arr2 = Array.init size (fun i -> -1) 
      mappings |> Seq.iter (fun (src,dest) -> 
                                if src < 0 or src >= size or dest < 0 or dest > size then failwith (System.String.Format("invalid mapping in permutation of size {0}, src = {1}, dest = {2}",size,src,dest))
                                if arr2.[src] > 0 then failwith(System.String.Format("duplicate mapping in permutation of size {0}, src = {1}, dest = {2}",size,src,dest))
                                arr2.[src] <- dest);
      for i = 0 to arr2.Length - 1 do
          let x = arr2.[i]
          if x < 0 then arr2.[i] <- i
      Permutation(arr2)
            
    member p.Length = arr.Length
    member p.Inverse = 
        let arr2 = Array.zero_create arr.Length
        for i = 0 to arr.Length - 1 do
             arr2.[arr.[i]] <- i
        Permutation(arr2)

    override p.Equals(p2:obj) = 
        let p2 = (p2 :?> Permutation)
        (p.Length = p2.Length) &&
        let rec check i = (i >= arr.Length) || (arr.[i] = p2.[i]) && check (i+1) 
        check 0

    override p.GetHashCode() = hash arr

    interface IComparable with 
        override p.CompareTo(p2:obj) = 
            let p2 = (p2 :?> Permutation)
            let c = compare p.Length p2.Length 
            if c <> 0 then c else
            let rec check i = 
                if (i >= arr.Length) then 0 else 
                let c = compare arr.[i] p2.[i] 
                if c <> 0 then c else
                check (i+1) 
            check 0

    static member Compose(p1:Permutation,p2:Permutation) = 
        if p1.Length <> p2.Length then failwith(System.String.Format("permutations have different lengths, one is {0}, other is {1}",p1.Length,p2.Length))
        Permutation(Array.init p1.Length (fun i -> p1.[p2.[i]]))
       
    member p.Item 
       with get(n) = 
        if n < 0 or n >= arr.Length then 
            failwith(System.String.Format("index {0} out of range in permutation of length {1}",n,arr.Length))
        arr.[n]

    static member Identity(size) = Permutation(Array.init size (fun i -> i))
    static member Swap(size,n,m) = Permutation(Array.init size (fun i -> if i = n then m else if i = m then n else i))
    static member Reverse(size) = Permutation(Array.init size (fun i -> size - i - 1))
    static member Rotate(size:int,?dist) = 
        let dist = match dist with Some v -> v | None -> 1
        let dist = if dist < 0 then size - (-dist % size) else dist
        Permutation(Array.init size (fun i -> (i-dist+size)%size))
