#
# FNC Common Build
#

# CONFIGURATION
CC ?=		cc
PREFIX ?=	/usr/local
MANDIR ?=	/share/man
VERSION ?=	0.18
HASH !=		cut -f 1 manifest.uuid
DATE !=		sed '2q;d' manifest | cut -d ' ' -f 2 | tr T ' '

CFLAGS +=	-Wall -Werror -Wsign-compare -pedantic -std=c99

# FLAGS NEEDED TO BUILD SQLITE3
SQLITE_CFLAGS =	${CFLAGS} \
		-DNDEBUG=1 \
		-DSQLITE_DQS=0 \
		-DSQLITE_DEFAULT_MEMSTATUS=0 \
		-DSQLITE_DEFAULT_WAL_SYNCHRONOUS=1 \
		-DSQLITE_LIKE_DOESNT_MATCH_BLOBS \
		-DSQLITE_OMIT_DECLTYPE \
		-DSQLITE_OMIT_PROGRESS_CALLBACK \
		-DSQLITE_OMIT_SHARED_CACHE \
		-DSQLITE_OMIT_LOAD_EXTENSION \
		-DSQLITE_MAX_EXPR_DEPTH=0 \
		-DSQLITE_ENABLE_LOCKING_STYLE=0 \
		-DSQLITE_DEFAULT_FILE_FORMAT=4 \
		-DSQLITE_ENABLE_EXPLAIN_COMMENTS \
		-DSQLITE_ENABLE_FTS4 \
		-DSQLITE_ENABLE_DBSTAT_VTAB \
		-DSQLITE_ENABLE_JSON1 \
		-DSQLITE_ENABLE_FTS5 \
		-DSQLITE_ENABLE_STMTVTAB \
		-DSQLITE_HAVE_ZLIB \
		-DSQLITE_INTROSPECTION_PRAGMAS \
		-DSQLITE_ENABLE_DBPAGE_VTAB \
		-DSQLITE_TRUSTED_SCHEMA=0

# FLAGS NEEDED TO BUILD LIBFOSSIL
FOSSIL_CFLAGS =	${CFLAGS}

# On SOME Linux (e.g., Ubuntu 18.04.6), we have to include wchar curses from
# I/.../ncursesw, but linking to -lncursesw (w/ no special -L path) works fine.
# FLAGS NEEDED TO BUILD FNC
FNC_CFLAGS =	${CFLAGS} -Wstrict-prototypes -Wmissing-prototypes -fPIC \
		-Wunused-variable -I./lib -I./include -I/usr/include/ncursesw \
		-DFNC_VERSION=${VERSION} -DFNC_HASH=${HASH} -DFNC_DATE="${DATE}"

FNC_LDFLAGS =	${LDFLAGS} -lm -lutil -lz -lpthread

# Compile-time checks and runtime protection mechanisms from the compiler
# hardening document: https://best.openssf.org/Compiler-Hardening-Guides
HARDEN =	-Wstrict-overflow -Wformat -Werror=format-security \
		-Werror=implicit -Werror=incompatible-pointer-types \
		-Werror=int-conversion -U_FORTIFY_SOURCE -D_FORTIFY_SOURCE=3 \
		-fstack-protector-strong -fno-strict-aliasing \
		-fno-delete-null-pointer-checks -fno-strict-overflow

UBSAN_FLAGS ?=	-fsanitize=undefined \
		-fsanitize=unsigned-integer-overflow,implicit-conversion \
		-fsanitize=local-bounds,nullability,float-divide-by-zero

all: bin

bin: lib/sqlite3.o lib/libfossil.o src/fnc.o src/fnc

lib/sqlite3.o: lib/sqlite3.c lib/sqlite3.h
	${CC} ${SQLITE_CFLAGS} -c $< -o $@

lib/libfossil.o: lib/libfossil.c lib/libfossil.h
	${CC} ${FOSSIL_CFLAGS} -c $< -o $@

src/diff.o: src/diff.c include/diff.h include/error.h include/fnc_compat.h
	${CC} ${FNC_CFLAGS} -c $< -o $@

src/error.o: src/error.c include/error.h
	${CC} ${FNC_CFLAGS} -c $< -o $@

compat/reallocarray.o: compat/reallocarray.c include/fnc_compat.h
	${CC} ${FNC_CFLAGS} -c $< -o $@

src/fnc.o: src/fnc.c include/diff.h include/enum.h include/error.h \
	include/fnc.h include/fnc_compat.h include/opt.h fnc.bld.mk
	${CC} ${FNC_CFLAGS} -c $< -o $@

src/fnc: src/fnc.o src/diff.o src/error.o compat/reallocarray.o \
	lib/libfossil.o lib/sqlite3.o fnc.bld.mk
	${CC} -o $@ src/fnc.o src/diff.o src/error.o compat/reallocarray.o \
	lib/libfossil.o lib/sqlite3.o ${FNC_LDFLAGS}

dev:
	${MAKE} GOHARD=1

ubsan:
	@test $(CLANG) -eq 1 || (echo "gcc ubsan not supported" && exit 1)
	${MAKE} UBSAN=1 GOHARD=1
	@echo "ubsan runtime diagnostics output to stderr"
	@echo "log to <path> with UBSAN_OPTIONS=log_path=<path>"

ccdb: clean
	@test $(CLANG) -eq 1 || \
		(echo "compilation database requires clang" && exit 1)
	@-rm -rf $@
	@mkdir $@
	${MAKE} CCDB=1
	@-rm -f compile_commands.json
	@{ \
		echo [; \
		cat $@/*.o.json | tr '\n' ' ' | sed -e 's/, $$//'; \
		echo ]; \
	} > compile_commands.json
	@-rm -rf $@
	@echo "generated compile_commands.json"

install:
	mkdir -p -m 0755 ${PREFIX}/bin
	mkdir -p -m 0755 ${PREFIX}${MANDIR}/man1
	install -s -m 0755 src/fnc ${PREFIX}/bin/fnc
	install -m 0644 src/fnc.1 ${PREFIX}${MANDIR}/man1/fnc.1

uninstall:
	rm -f ${PREFIX}/bin/fnc ${PREFIX}${MANDIR}/man1/fnc.1

clean:
	rm -f compat/*.o lib/*.o src/*.o src/fnc

release: clean
	mkdir /tmp/fnc-${VERSION}
	pax -rw * /tmp/fnc-${VERSION}
	tar czvf ../fnc-${VERSION}.tgz -C /tmp fnc-${VERSION}
	rm -rf /tmp/fnc-${VERSION}

.PHONY: clean release
