/*
 * Copyright (c) 2023 Mark Jamsek <mark@jamsek.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/* Error result codes cannot exceed 100 (which is where libf codes begin). */
#define FNC_RC_OK		0
#define FNC_RC_ERROR		1
#define FNC_RC_BREAK		2
#define FNC_RC_CANCELLED	3
#define FNC_RC_NO_CKOUT		4
#define FNC_RC_NO_REPO		5
#define FNC_RC_NO_MATCH		6
#define FNC_RC_ERRNO		7
#define FNC_RC_BAD_PATH		8
#define FNC_RC_NO_REPO_PATH	9
#define FNC_RC_NO_CKOUT_PATH	10
#define FNC_RC_NO_TREE_PATH	11
#define FNC_RC_NO_BRANCH	12
#define FNC_RC_NO_USER		13
#define FNC_RC_NO_TAG		14
#define FNC_RC_NO_RID		15
#define FNC_RC_NO_COMMIT	16
#define FNC_RC_NO_REF		17
#define FNC_RC_BAD_HASH		18
#define FNC_RC_BAD_ARTIFACT	19
#define FNC_RC_AMBIGUOUS_ID	20
#define FNC_RC_EMPTY_TREE	21
#define FNC_RC_IO		22
#define FNC_RC_EOF		23
#define FNC_RC_NO_SPACE		24
#define FNC_RC_RANGE		25
#define FNC_RC_BAD_OPTION	26
#define FNC_RC_BAD_CMD		27
#define FNC_RC_AMBIGUOUS_CMD	28
#define FNC_RC_CKOUT_BUSY	29
#define FNC_RC_NYI		30
#define FNC_RC_BAD_KEYWORD	31
#define FNC_RC_DIFF_BINARY	32
#define FNC_RC_BLAME_BINARY	33
#define FNC_RC_AMBIGUOUS_DATE	34
#define FNC_RC_BAD_DATE		35
#define FNC_RC_REGEX		36
#define FNC_RC_CURSES		37
#define FNC_RC_FATAL		38
#define FNC_RC_SITREP		39
#define FNC_RC_BAD_NUMBER	40

struct fnc_error {
	int		 rc;
	const char	*msg;
};

/* Private implementation. */
#ifdef __OpenBSD__
#define FNC_ERRNO_STR_SZ	NL_TEXTMAX
#else
#define FNC_ERRNO_STR_SZ	1024
#endif

#define FNC_ERR_PREFIX_SZ	1024
#define FNC_ERR_MSG_BUFSZ	(FNC_ERR_PREFIX_SZ + FNC_ERRNO_STR_SZ)

#if DEBUG
#define RC_SET(_r, _fmt, ...)	fsl_error_set(&fnc__error, _r, "%s::%s " _fmt,\
				    __func__, FILE_POSITION, __VA_ARGS__)
#else
#define RC_SET(_r, _fmt, ...)	fsl_error_set(&fnc__error, _r, _fmt,	\
				    __VA_ARGS__)
#endif /* DEBUG */

#define RCX(_r, ...)	_r == FNC_RC_ERRNO ? RC_ERRNO(__VA_ARGS__) :		\
			    _r == FNC_RC_BREAK ? RC_BREAK(__VA_ARGS__) :	\
			    _r >= FSL_RC_ERROR ? RC_LIBF(_r, __VA_ARGS__) :	\
			    RC_SET(_r, "%s", fnc_error(_r, __VA_ARGS__))

/* Public API */

/*
 * Set the error state to result code rc with a message built from the
 * optional provided format string and variable length argument list,
 * which will be prefixed with rc's corresponding error message mapped
 * from the below list. FNC_RC_ERRNO, FNC_RC_BREAK, and valid fsl_rc_e
 * FSL_RC_* enum result codes are special cases with slightly different
 * semantics per the below documented RC_ERRNO, RC_BREAK, and RC_LIBF API.
 *	RC(rc)
 *	RC(rc, fmt, ...)
 */
#define RC(...)	RCX(__VA_ARGS__, "", 0)

/*
 * Set the error state to FNC_RC_ERRNO, with a message built from the
 * provided format string and variable length argument list, which will
 * be suffixed with an errno error string obtained from strerror(3).
 */
#define RC_ERRNO(...)	RC_SET(FNC_RC_ERRNO, "%s",			\
			    fnc_error_from_errno(__VA_ARGS__))

/*
 * Set errno to _e and the error state to FNC_RC_ERRNO, with a message built
 * from the provided format string and variable length argument list, which
 * will be suffixed with an errno error string obtained from strerror(3).
 */
#define RC_ERRNO_SET(_e, ...)	RC((errno =_e) == FNC_RC_OK ?		\
				    FNC_RC_ERROR : FNC_RC_ERRNO,	\
				    "%s", __VA_ARGS__)

/*
 * Set the error state based on the error indicator for file stream _f
 * as returned by ferror(3): either the RC_ERRNO or RC macro will be used
 * if the indicator is set or unset, respectively, with the resulting
 * message built per the corresponding API as documented above.
 */
#define RC_FERROR(_f, ...)	ferror(_f) ?				\
				    RC_ERRNO(__VA_ARGS__) :		\
				    RC_SET(FNC_RC_EOF, __VA_ARGS__, "")

/*
 * Set the error state to FNC_RC_BREAK, a special case used to report
 * that the requested operation cannot proceed due to unmet preconditions
 * (e.g., different artifacts selected from the timeline view to diff),
 * but will not fatally error. This is often used to report such cases
 * before resuming the view loop, and to return zero on exit with a message
 * built from the provided format string and variable length argument list.
 */
#define RC_BREAK(...)	RC_SET(FNC_RC_BREAK, __VA_ARGS__, "")

/*
 * Set the error state to _e, where _e is a valid fsl_rc_e FSL_RC_* enum
 * error code, with a message built from the provided format string and
 * variable length argument list, prefixed by a libfossil error obtained
 * from fsl_error_get() or fsl_rc_cstr() via fnc_error_from_libf().
 */
#define RC_LIBF(_e, ...) RC_SET(_e, "%s", fnc_error_from_libf(_e, __VA_ARGS__))

/* Reset the error state. */
#define RC_RESET()	fnc_error_reset()

/* Error strings cannot exceed FNC_ERRNO_STR_SZ (OpenBSD: 255, Linux: 1024). */
#define GENERATE_ERROR_MAP \
	ERR_(FNC_RC_OK,			NULL),					\
	ERR_(FNC_RC_ERROR,		NULL),					\
	ERR_(FNC_RC_BREAK,		NULL),					\
	ERR_(FNC_RC_CANCELLED,		"operation in progress cancelled"),	\
	ERR_(FNC_RC_NO_CKOUT,		"no work tree found"),			\
	ERR_(FNC_RC_NO_REPO,		"no fossil repository found"),		\
	ERR_(FNC_RC_NO_MATCH,		"no matches found"),			\
	ERR_(FNC_RC_ERRNO,		NULL),					\
	ERR_(FNC_RC_BAD_PATH,		"invalid path"),			\
	ERR_(FNC_RC_NO_REPO_PATH,	"path not found in the repository"),	\
	ERR_(FNC_RC_NO_CKOUT_PATH,	"path not found in the checkout"),	\
	ERR_(FNC_RC_NO_TREE_PATH,	"path not found in tree"),		\
	ERR_(FNC_RC_NO_BRANCH,		"branch not found"),			\
	ERR_(FNC_RC_NO_USER,		"user not found"),			\
	ERR_(FNC_RC_NO_TAG,		"tag not found"),			\
	ERR_(FNC_RC_NO_RID,		"rid not found in the database"),	\
	ERR_(FNC_RC_NO_COMMIT,		"commit not found"),			\
	ERR_(FNC_RC_NO_REF,		"no such reference found"),		\
	ERR_(FNC_RC_BAD_HASH,		"invalid SHA hash"),			\
	ERR_(FNC_RC_BAD_ARTIFACT,	"bad artifact type"),			\
	ERR_(FNC_RC_AMBIGUOUS_ID,	"ambiguous artifact id"),		\
	ERR_(FNC_RC_EMPTY_TREE,		"tree is empty"),			\
	ERR_(FNC_RC_IO,			"input/output error"),			\
	ERR_(FNC_RC_EOF,		"unexpected end of file"),		\
	ERR_(FNC_RC_NO_SPACE,		"buffer too small"),			\
	ERR_(FNC_RC_RANGE,		"value out of range"),			\
	ERR_(FNC_RC_BAD_OPTION,		"invalid option"),			\
	ERR_(FNC_RC_BAD_CMD,		"invalid command"),			\
	ERR_(FNC_RC_AMBIGUOUS_CMD,	"ambiguous command"),			\
	ERR_(FNC_RC_CKOUT_BUSY,		"checkout database is locked"),		\
	ERR_(FNC_RC_NYI,		"feature is not implemented"),		\
	ERR_(FNC_RC_BAD_KEYWORD,	"invalid keyword"),			\
	ERR_(FNC_RC_DIFF_BINARY,	"cannot diff binary file"),		\
	ERR_(FNC_RC_BLAME_BINARY,	"cannot blame binary file"),		\
	ERR_(FNC_RC_AMBIGUOUS_DATE,	"ambiguous date"),			\
	ERR_(FNC_RC_BAD_DATE,		"invalid date"),			\
	ERR_(FNC_RC_REGEX,		"regular expression error"),		\
	ERR_(FNC_RC_CURSES,		"fatal curses error"),			\
	ERR_(FNC_RC_FATAL,		"unexpected fatality"),			\
	ERR_(FNC_RC_SITREP,		NULL),					\
	ERR_(FNC_RC_BAD_NUMBER,		"invalid number")
