#include "libfossil.h"
/* start of file ./src/xdirent.h */
/**
  Origin: https://gist.github.com/isakbosman/758eb668938806aabb04830736f4ac41

  Modified only very slightly for use in the libfossil project: a
  couple of #if's were added to allow us to include this file without
  having to check which platform we're building on. In non-Windows builds
  it uses the corresponding POSIX APIs.
*/
/*
 * dirent.h - dirent API for Microsoft Visual Studio
 *
 * Copyright (C) 2006-2012 Toni Ronkko
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * ``Software''), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED ``AS IS'', WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL TONI RONKKO BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * $Id: dirent.h,v 1.20 2014/03/19 17:52:23 tronkko Exp $
 */
#ifndef DIRENT_H
#define DIRENT_H

#if FSL_PLATFORM_IS_WINDOWS
/*
 * Define architecture flags so we don't need to include windows.h.
 * Avoiding windows.h makes it simpler to use windows sockets in conjunction
 * with dirent.h.
 */
#if !defined(_68K_) && !defined(_MPPC_) && !defined(_X86_) && !defined(_IA64_) && !defined(_AMD64_) && defined(_M_IX86)
#   define _X86_
#endif
#if !defined(_68K_) && !defined(_MPPC_) && !defined(_X86_) && !defined(_IA64_) && !defined(_AMD64_) && defined(_M_AMD64)
#define _AMD64_
#endif

#include <stdio.h>
#include <stdarg.h>
#include <windef.h>
#include <winbase.h>
#include <wchar.h>
#include <string.h>
#include <stdlib.h>
#include <malloc.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

/* Indicates that d_type field is available in dirent structure */
#define _DIRENT_HAVE_D_TYPE

/* Indicates that d_namlen field is available in dirent structure */
#define _DIRENT_HAVE_D_NAMLEN

/* Entries missing from MSVC 6.0 */
#if !defined(FILE_ATTRIBUTE_DEVICE)
#   define FILE_ATTRIBUTE_DEVICE 0x40
#endif

/* File type and permission flags for stat() */
#if !defined(S_IFMT)
#   define S_IFMT   _S_IFMT                     /* File type mask */
#endif
#if !defined(S_IFDIR)
#   define S_IFDIR  _S_IFDIR                    /* Directory */
#endif
#if !defined(S_IFCHR)
#   define S_IFCHR  _S_IFCHR                    /* Character device */
#endif
#if !defined(S_IFFIFO)
#   define S_IFFIFO _S_IFFIFO                   /* Pipe */
#endif
#if !defined(S_IFREG)
#   define S_IFREG  _S_IFREG                    /* Regular file */
#endif
#if !defined(S_IREAD)
#   define S_IREAD  _S_IREAD                    /* Read permission */
#endif
#if !defined(S_IWRITE)
#   define S_IWRITE _S_IWRITE                   /* Write permission */
#endif
#if !defined(S_IEXEC)
#   define S_IEXEC  _S_IEXEC                    /* Execute permission */
#endif
#if !defined(S_IFIFO)
#   define S_IFIFO _S_IFIFO                     /* Pipe */
#endif
#if !defined(S_IFBLK)
#   define S_IFBLK   0                          /* Block device */
#endif
#if !defined(S_IFLNK)
#   define S_IFLNK   0                          /* Link */
#endif
#if !defined(S_IFSOCK)
#   define S_IFSOCK  0                          /* Socket */
#endif

#if defined(_MSC_VER)
#   define S_IRUSR  S_IREAD                     /* Read user */
#   define S_IWUSR  S_IWRITE                    /* Write user */
#   define S_IXUSR  0                           /* Execute user */
#   define S_IRGRP  0                           /* Read group */
#   define S_IWGRP  0                           /* Write group */
#   define S_IXGRP  0                           /* Execute group */
#   define S_IROTH  0                           /* Read others */
#   define S_IWOTH  0                           /* Write others */
#   define S_IXOTH  0                           /* Execute others */
#endif

/* Maximum length of file name */
#if !defined(PATH_MAX)
#   define PATH_MAX MAX_PATH
#endif
#if !defined(FILENAME_MAX)
#   define FILENAME_MAX MAX_PATH
#endif
#if !defined(NAME_MAX)
#   define NAME_MAX FILENAME_MAX
#endif

/* File type flags for d_type */
#define DT_UNKNOWN  0
#define DT_REG      S_IFREG
#define DT_DIR      S_IFDIR
#define DT_FIFO     S_IFIFO
#define DT_SOCK     S_IFSOCK
#define DT_CHR      S_IFCHR
#define DT_BLK      S_IFBLK
#define DT_LNK      S_IFLNK

/* Macros for converting between st_mode and d_type */
#define IFTODT(mode) ((mode) & S_IFMT)
#define DTTOIF(type) (type)

/*
 * File type macros.  Note that block devices, sockets and links cannot be
 * distinguished on Windows and the macros S_ISBLK, S_ISSOCK and S_ISLNK are
 * only defined for compatibility.  These macros should always return false
 * on Windows.
 */
#define	S_ISFIFO(mode) (((mode) & S_IFMT) == S_IFIFO)
#define	S_ISDIR(mode)  (((mode) & S_IFMT) == S_IFDIR)
#define	S_ISREG(mode)  (((mode) & S_IFMT) == S_IFREG)
#define	S_ISLNK(mode)  (((mode) & S_IFMT) == S_IFLNK)
#define	S_ISSOCK(mode) (((mode) & S_IFMT) == S_IFSOCK)
#define	S_ISCHR(mode)  (((mode) & S_IFMT) == S_IFCHR)
#define	S_ISBLK(mode)  (((mode) & S_IFMT) == S_IFBLK)

/* Return the exact length of d_namlen without zero terminator */
#define _D_EXACT_NAMLEN(p) ((p)->d_namlen)

/* Return number of bytes needed to store d_namlen */
#define _D_ALLOC_NAMLEN(p) (PATH_MAX)


#ifdef __cplusplus
extern "C" {
#endif


/* Wide-character version */
struct _wdirent {
    long d_ino;                                 /* Always zero */
    unsigned short d_reclen;                    /* Structure size */
    size_t d_namlen;                            /* Length of name without \0 */
    int d_type;                                 /* File type */
    wchar_t d_name[PATH_MAX];                   /* File name */
};
typedef struct _wdirent _wdirent;

struct _WDIR {
    struct _wdirent ent;                        /* Current directory entry */
    WIN32_FIND_DATAW data;                      /* Private file data */
    int cached;                                 /* True if data is valid */
    HANDLE handle;                              /* Win32 search handle */
    wchar_t *patt;                              /* Initial directory name */
};
typedef struct _WDIR _WDIR;

static _WDIR *_wopendir (const wchar_t *dirname);
static struct _wdirent *_wreaddir (_WDIR *dirp);
static int _wclosedir (_WDIR *dirp);
static void _wrewinddir (_WDIR* dirp);


/* For compatibility with Symbian */
#define wdirent _wdirent
#define WDIR _WDIR
#define wopendir _wopendir
#define wreaddir _wreaddir
#define wclosedir _wclosedir
#define wrewinddir _wrewinddir


/* Multi-byte character versions */
struct dirent {
    long d_ino;                                 /* Always zero */
    unsigned short d_reclen;                    /* Structure size */
    size_t d_namlen;                            /* Length of name without \0 */
    int d_type;                                 /* File type */
    char d_name[PATH_MAX];                      /* File name */
};
typedef struct dirent dirent;

struct DIR {
    struct dirent ent;
    struct _WDIR *wdirp;
};
typedef struct DIR DIR;

static DIR *opendir (const char *dirname);
static struct dirent *readdir (DIR *dirp);
static int closedir (DIR *dirp);
static void rewinddir (DIR* dirp);


/* Internal utility functions */
static WIN32_FIND_DATAW *dirent_first (_WDIR *dirp);
static WIN32_FIND_DATAW *dirent_next (_WDIR *dirp);

static int dirent_mbstowcs_s(
    size_t *pReturnValue,
    wchar_t *wcstr,
    size_t sizeInWords,
    const char *mbstr,
    size_t count);

static int dirent_wcstombs_s(
    size_t *pReturnValue,
    char *mbstr,
    size_t sizeInBytes,
    const wchar_t *wcstr,
    size_t count);

static void dirent_set_errno (int error);

/*
 * Open directory stream DIRNAME for read and return a pointer to the
 * internal working area that is used to retrieve individual directory
 * entries.
 */
static _WDIR*
_wopendir(
    const wchar_t *dirname)
{
    _WDIR *dirp = NULL;
    int error;

    /* Must have directory name */
    if (dirname == NULL  ||  dirname[0] == '\0') {
        dirent_set_errno (ENOENT);
        return NULL;
    }

    /* Allocate new _WDIR structure */
    dirp = (_WDIR*) malloc (sizeof (struct _WDIR));
    if (dirp != NULL) {
        DWORD n;

        /* Reset _WDIR structure */
        dirp->handle = INVALID_HANDLE_VALUE;
        dirp->patt = NULL;
        dirp->cached = 0;

        /* Compute the length of full path plus zero terminator */
        n = GetFullPathNameW (dirname, 0, NULL, NULL);

        /* Allocate room for absolute directory name and search pattern */
        dirp->patt = (wchar_t*) malloc (sizeof (wchar_t) * n + 16);
        if (dirp->patt) {

            /*
             * Convert relative directory name to an absolute one.  This
             * allows rewinddir() to function correctly even when current
             * working directory is changed between opendir() and rewinddir().
             */
            n = GetFullPathNameW (dirname, n, dirp->patt, NULL);
            if (n > 0) {
                wchar_t *p;

                /* Append search pattern \* to the directory name */
                p = dirp->patt + n;
                if (dirp->patt < p) {
                    switch (p[-1]) {
                    case '\\':
                    case '/':
                    case ':':
                        /* Directory ends in path separator, e.g. c:\temp\ */
                        /*NOP*/;
                        break;

                    default:
                        /* Directory name doesn't end in path separator */
                        *p++ = '\\';
                    }
                }
                *p++ = '*';
                *p = '\0';

                /* Open directory stream and retrieve the first entry */
                if (dirent_first (dirp)) {
                    /* Directory stream opened successfully */
                    error = 0;
                } else {
                    /* Cannot retrieve first entry */
                    error = 1;
                    dirent_set_errno (ENOENT);
                }

            } else {
                /* Cannot retrieve full path name */
                dirent_set_errno (ENOENT);
                error = 1;
            }

        } else {
            /* Cannot allocate memory for search pattern */
            error = 1;
        }

    } else {
        /* Cannot allocate _WDIR structure */
        error = 1;
    }

    /* Clean up in case of error */
    if (error  &&  dirp) {
        _wclosedir (dirp);
        dirp = NULL;
    }

    return dirp;
}

/*
 * Read next directory entry.  The directory entry is returned in dirent
 * structure in the d_name field.  Individual directory entries returned by
 * this function include regular files, sub-directories, pseudo-directories
 * "." and ".." as well as volume labels, hidden files and system files.
 */
static struct _wdirent*
_wreaddir(
    _WDIR *dirp)
{
    WIN32_FIND_DATAW *datap;
    struct _wdirent *entp;

    /* Read next directory entry */
    datap = dirent_next (dirp);
    if (datap) {
        size_t n;
        DWORD attr;
        
        /* Pointer to directory entry to return */
        entp = &dirp->ent;

        /* 
         * Copy file name as wide-character string.  If the file name is too
         * long to fit in to the destination buffer, then truncate file name
         * to PATH_MAX characters and zero-terminate the buffer.
         */
        n = 0;
        while (n + 1 < PATH_MAX  &&  datap->cFileName[n] != 0) {
            entp->d_name[n] = datap->cFileName[n];
            n++;
        }
        dirp->ent.d_name[n] = 0;

        /* Length of file name excluding zero terminator */
        entp->d_namlen = n;

        /* File type */
        attr = datap->dwFileAttributes;
        if ((attr & FILE_ATTRIBUTE_DEVICE) != 0) {
            entp->d_type = DT_CHR;
        } else if ((attr & FILE_ATTRIBUTE_DIRECTORY) != 0) {
            entp->d_type = DT_DIR;
        } else {
            entp->d_type = DT_REG;
        }

        /* Reset dummy fields */
        entp->d_ino = 0;
        entp->d_reclen = sizeof (struct _wdirent);

    } else {

        /* Last directory entry read */
        entp = NULL;

    }

    return entp;
}

/*
 * Close directory stream opened by opendir() function.  This invalidates the
 * DIR structure as well as any directory entry read previously by
 * _wreaddir().
 */
static int
_wclosedir(
    _WDIR *dirp)
{
    int ok;
    if (dirp) {

        /* Release search handle */
        if (dirp->handle != INVALID_HANDLE_VALUE) {
            FindClose (dirp->handle);
            dirp->handle = INVALID_HANDLE_VALUE;
        }

        /* Release search pattern */
        if (dirp->patt) {
            free (dirp->patt);
            dirp->patt = NULL;
        }

        /* Release directory structure */
        free (dirp);
        ok = /*success*/0;

    } else {
        /* Invalid directory stream */
        dirent_set_errno (EBADF);
        ok = /*failure*/-1;
    }
    return ok;
}

/*
 * Rewind directory stream such that _wreaddir() returns the very first
 * file name again.
 */
static void
_wrewinddir(
    _WDIR* dirp)
{
    if (dirp) {
        /* Release existing search handle */
        if (dirp->handle != INVALID_HANDLE_VALUE) {
            FindClose (dirp->handle);
        }

        /* Open new search handle */
        dirent_first (dirp);
    }
}

/* Get first directory entry (internal) */
static WIN32_FIND_DATAW*
dirent_first(
    _WDIR *dirp)
{
    WIN32_FIND_DATAW *datap;

    /* Open directory and retrieve the first entry */
    dirp->handle = FindFirstFileW (dirp->patt, &dirp->data);
    if (dirp->handle != INVALID_HANDLE_VALUE) {

        /* a directory entry is now waiting in memory */
        datap = &dirp->data;
        dirp->cached = 1;

    } else {

        /* Failed to re-open directory: no directory entry in memory */
        dirp->cached = 0;
        datap = NULL;

    }
    return datap;
}

/* Get next directory entry (internal) */
static WIN32_FIND_DATAW*
dirent_next(
    _WDIR *dirp)
{
    WIN32_FIND_DATAW *p;

    /* Get next directory entry */
    if (dirp->cached != 0) {

        /* A valid directory entry already in memory */
        p = &dirp->data;
        dirp->cached = 0;

    } else if (dirp->handle != INVALID_HANDLE_VALUE) {

        /* Get the next directory entry from stream */
        if (FindNextFileW (dirp->handle, &dirp->data) != FALSE) {
            /* Got a file */
            p = &dirp->data;
        } else {
            /* The very last entry has been processed or an error occured */
            FindClose (dirp->handle);
            dirp->handle = INVALID_HANDLE_VALUE;
            p = NULL;
        }

    } else {

        /* End of directory stream reached */
        p = NULL;

    }

    return p;
}

/* 
 * Open directory stream using plain old C-string.
 */
static DIR*
opendir(
    const char *dirname) 
{
    struct DIR *dirp;
    int error;

    /* Must have directory name */
    if (dirname == NULL  ||  dirname[0] == '\0') {
        dirent_set_errno (ENOENT);
        return NULL;
    }

    /* Allocate memory for DIR structure */
    dirp = (DIR*) malloc (sizeof (struct DIR));
    if (dirp) {
        wchar_t wname[PATH_MAX];
        size_t n;

        /* Convert directory name to wide-character string */
        error = dirent_mbstowcs_s (&n, wname, PATH_MAX, dirname, PATH_MAX);
        if (!error) {

            /* Open directory stream using wide-character name */
            dirp->wdirp = _wopendir (wname);
            if (dirp->wdirp) {
                /* Directory stream opened */
                error = 0;
            } else {
                /* Failed to open directory stream */
                error = 1;
            }

        } else {
            /* 
             * Cannot convert file name to wide-character string.  This
             * occurs if the string contains invalid multi-byte sequences or
             * the output buffer is too small to contain the resulting
             * string.
             */
            error = 1;
        }

    } else {
        /* Cannot allocate DIR structure */
        error = 1;
    }

    /* Clean up in case of error */
    if (error  &&  dirp) {
        free (dirp);
        dirp = NULL;
    }

    return dirp;
}

/*
 * Read next directory entry.
 *
 * When working with text consoles, please note that file names returned by
 * readdir() are represented in the default ANSI code page while any output to
 * console is typically formatted on another code page.  Thus, non-ASCII
 * characters in file names will not usually display correctly on console.  The
 * problem can be fixed in two ways: (1) change the character set of console
 * to 1252 using chcp utility and use Lucida Console font, or (2) use
 * _cprintf function when writing to console.  The _cprinf() will re-encode
 * ANSI strings to the console code page so many non-ASCII characters will
 * display correcly.
 */
static struct dirent*
readdir(
    DIR *dirp) 
{
    WIN32_FIND_DATAW *datap;
    struct dirent *entp;

    /* Read next directory entry */
    datap = dirent_next (dirp->wdirp);
    if (datap) {
        size_t n;
        int error;

        /* Attempt to convert file name to multi-byte string */
        error = dirent_wcstombs_s(
            &n, dirp->ent.d_name, PATH_MAX, datap->cFileName, PATH_MAX);

        /* 
         * If the file name cannot be represented by a multi-byte string,
         * then attempt to use old 8+3 file name.  This allows traditional
         * Unix-code to access some file names despite of unicode
         * characters, although file names may seem unfamiliar to the user.
         *
         * Be ware that the code below cannot come up with a short file
         * name unless the file system provides one.  At least
         * VirtualBox shared folders fail to do this.
         */
        if (error  &&  datap->cAlternateFileName[0] != '\0') {
            error = dirent_wcstombs_s(
                &n, dirp->ent.d_name, PATH_MAX, 
                datap->cAlternateFileName, PATH_MAX);
        }

        if (!error) {
            DWORD attr;

            /* Initialize directory entry for return */
            entp = &dirp->ent;

            /* Length of file name excluding zero terminator */
            entp->d_namlen = n - 1;

            /* File attributes */
            attr = datap->dwFileAttributes;
            if ((attr & FILE_ATTRIBUTE_DEVICE) != 0) {
                entp->d_type = DT_CHR;
            } else if ((attr & FILE_ATTRIBUTE_DIRECTORY) != 0) {
                entp->d_type = DT_DIR;
            } else {
                entp->d_type = DT_REG;
            }

            /* Reset dummy fields */
            entp->d_ino = 0;
            entp->d_reclen = sizeof (struct dirent);

        } else {
            /* 
             * Cannot convert file name to multi-byte string so construct
             * an errornous directory entry and return that.  Note that
             * we cannot return NULL as that would stop the processing
             * of directory entries completely.
             */
            entp = &dirp->ent;
            entp->d_name[0] = '?';
            entp->d_name[1] = '\0';
            entp->d_namlen = 1;
            entp->d_type = DT_UNKNOWN;
            entp->d_ino = 0;
            entp->d_reclen = 0;
        }

    } else {
        /* No more directory entries */
        entp = NULL;
    }

    return entp;
}

/*
 * Close directory stream.
 */
static int
closedir(
    DIR *dirp) 
{
    int ok;
    if (dirp) {

        /* Close wide-character directory stream */
        ok = _wclosedir (dirp->wdirp);
        dirp->wdirp = NULL;

        /* Release multi-byte character version */
        free (dirp);

    } else {

        /* Invalid directory stream */
        dirent_set_errno (EBADF);
        ok = /*failure*/-1;

    }
    return ok;
}

/*
 * Rewind directory stream to beginning.
 */
static void
rewinddir(
    DIR* dirp) 
{
    /* Rewind wide-character string directory stream */
    _wrewinddir (dirp->wdirp);
}

/* Convert multi-byte string to wide character string */
static int
dirent_mbstowcs_s(
    size_t *pReturnValue,
    wchar_t *wcstr,
    size_t sizeInWords,
    const char *mbstr,
    size_t count)
{
    int error;

#if defined(_MSC_VER)  &&  _MSC_VER >= 1400

    /* Microsoft Visual Studio 2005 or later */
    error = mbstowcs_s (pReturnValue, wcstr, sizeInWords, mbstr, count);

#else

    /* Older Visual Studio or non-Microsoft compiler */
    size_t n;

    /* Convert to wide-character string (or count characters) */
    n = mbstowcs (wcstr, mbstr, sizeInWords);
    if (!wcstr  ||  n < count) {

        /* Zero-terminate output buffer */
        if (wcstr  &&  sizeInWords) {
            if (n >= sizeInWords) {
                n = sizeInWords - 1;
            }
            wcstr[n] = 0;
        }

        /* Length of resuting multi-byte string WITH zero terminator */
        if (pReturnValue) {
            *pReturnValue = n + 1;
        }

        /* Success */
        error = 0;

    } else {

        /* Could not convert string */
        error = 1;

    }

#endif

    return error;
}

/* Convert wide-character string to multi-byte string */
static int
dirent_wcstombs_s(
    size_t *pReturnValue,
    char *mbstr,
    size_t sizeInBytes, /* max size of mbstr */
    const wchar_t *wcstr,
    size_t count)
{
    int error;

#if defined(_MSC_VER)  &&  _MSC_VER >= 1400

    /* Microsoft Visual Studio 2005 or later */
    error = wcstombs_s (pReturnValue, mbstr, sizeInBytes, wcstr, count);

#else

    /* Older Visual Studio or non-Microsoft compiler */
    size_t n;

    /* Convert to multi-byte string (or count the number of bytes needed) */
    n = wcstombs (mbstr, wcstr, sizeInBytes);
    if (!mbstr  ||  n < count) {

        /* Zero-terminate output buffer */
        if (mbstr  &&  sizeInBytes) {
            if (n >= sizeInBytes) {
                n = sizeInBytes - 1;
            }
            mbstr[n] = '\0';
        }

        /* Lenght of resulting multi-bytes string WITH zero-terminator */
        if (pReturnValue) {
            *pReturnValue = n + 1;
        }

        /* Success */
        error = 0;

    } else {

        /* Cannot convert string */
        error = 1;

    }

#endif

    return error;
}

/* Set errno variable */
static void
dirent_set_errno(
    int error)
{
#if defined(_MSC_VER)  &&  _MSC_VER >= 1400

    /* Microsoft Visual Studio 2005 and later */
    _set_errno (error);

#else

    /* Non-Microsoft compiler or older Microsoft compiler */
    errno = error;

#endif
}


#ifdef __cplusplus
}
#endif

#else /* !FSL_PLATFORM_IS_WINDOWS */
#include <sys/types.h>
#include <dirent.h>
#endif /* !FSL_PLATFORM_IS_WINDOWS */

#endif /*DIRENT_H*/
/* end of file ./src/xdirent.h */
/* start of file ./src/fsl.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/
/*****************************************************************************
  This file houses some context-independent API routines as well as
  some of the generic helper functions and types.
*/

#include <assert.h>
#include <stdlib.h> /* malloc() and friends, qsort() */
#include <memory.h> /* memset() */
#include <time.h> /* strftime() and gmtime() */

#if defined(_WIN32) || defined(WIN32)
# include <io.h>
#define isatty(h) _isatty(h)
#else
# include <unistd.h> /* isatty() */
#endif
/* extern int isatty(int); */

#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)

/*
  Please keep all fsl_XXX_empty initializers in one place (here)
  and lexically sorted.
*/
const fsl__bccache fsl__bccache_empty = fsl__bccache_empty_m;
const fsl_branch_opt fsl_branch_opt_empty = fsl_branch_opt_empty_m;
const fsl_buffer fsl_buffer_empty = fsl_buffer_empty_m;
const fsl_card_F fsl_card_F_empty = fsl_card_F_empty_m;
const fsl_card_F_list fsl_card_F_list_empty = fsl_card_F_list_empty_m;
const fsl_card_J fsl_card_J_empty = fsl_card_J_empty_m;
const fsl_card_Q fsl_card_Q_empty = fsl_card_Q_empty_m;
const fsl_card_T fsl_card_T_empty = fsl_card_T_empty_m;
const fsl_checkin_opt fsl_checkin_opt_empty = fsl_checkin_opt_empty_m;
const fsl_cidiff_opt fsl_cidiff_opt_empty = fsl_cidiff_opt_empty_m;
const fsl_cidiff_state fsl_cidiff_state_empty = fsl_cidiff_state_empty_m;
const fsl_ckout_manage_opt fsl_ckout_manage_opt_empty =
  fsl_ckout_manage_opt_empty_m;
const fsl_ckout_rename_opt fsl_ckout_rename_opt_empty =
  fsl_ckout_rename_opt_empty_m;
const fsl_ckout_unmanage_opt fsl_ckout_unmanage_opt_empty =
  fsl_ckout_unmanage_opt_empty_m;
const fsl_ckup_opt fsl_ckup_opt_empty = fsl_ckup_opt_m;
const fsl_confirmer fsl_confirmer_empty = fsl_confirmer_empty_m;
const fsl_cx fsl_cx_empty = fsl_cx_empty_m;
const fsl_cx_config fsl_cx_config_empty = fsl_cx_config_empty_m;
const fsl_cx_init_opt fsl_cx_init_opt_default = fsl_cx_init_opt_default_m;
const fsl_cx_init_opt fsl_cx_init_opt_empty = fsl_cx_init_opt_empty_m;
const fsl_db fsl_db_empty = fsl_db_empty_m;
const fsl_deck fsl_deck_empty = fsl_deck_empty_m;
const fsl_confirm_detail fsl_confirm_detail_empty =
  fsl_confirm_detail_empty_m;
const fsl_confirm_response fsl_confirm_response_empty =
  fsl_confirm_response_empty_m;
const fsl_error fsl_error_empty = fsl_error_empty_m;
const fsl_checkin_queue_opt fsl_checkin_queue_opt_empty =
  fsl_checkin_queue_opt_empty_m;
const fsl_fstat fsl_fstat_empty = fsl_fstat_empty_m;
const fsl_list fsl_list_empty = fsl_list_empty_m;
const fsl__mcache fsl__mcache_empty = fsl__mcache_empty_m;
const fsl_merge_opt fsl_merge_opt_empty = fsl_merge_opt_empty_m;
const fsl_outputer fsl_outputer_FILE = fsl_outputer_FILE_m;
const fsl_outputer fsl_outputer_empty = fsl_outputer_empty_m;
const fsl_pathfinder fsl_pathfinder_empty = fsl_pathfinder_empty_m;
const fsl__pq fsl__pq_empty = fsl__pq_empty_m;
const fsl_rebuild_step fsl_rebuild_step_empty = fsl_rebuild_step_empty_m;
const fsl_repo_create_opt fsl_repo_create_opt_empty =
  fsl_repo_create_opt_empty_m;
const fsl_repo_extract_opt fsl_repo_extract_opt_empty =
  fsl_repo_extract_opt_empty_m;
const fsl_repo_extract_state fsl_repo_extract_state_empty =
  fsl_repo_extract_state_empty_m;
const fsl_repo_open_ckout_opt fsl_repo_open_ckout_opt_empty =
  fsl_repo_open_ckout_opt_m;
const fsl_ckout_revert_opt fsl_ckout_revert_opt_empty =
  fsl_ckout_revert_opt_empty_m;
const fsl_sha1_cx fsl_sha1_cx_empty = fsl_sha1_cx_empty_m;
const fsl_state fsl_state_empty = fsl_state_empty_m;
const fsl_stmt fsl_stmt_empty = fsl_stmt_empty_m;
const fsl_timer_state fsl_timer_state_empty = fsl_timer_state_empty_m;
const fsl_xlinker fsl_xlinker_empty = fsl_xlinker_empty_m;
const fsl_xlinker_list fsl_xlinker_list_empty = fsl_xlinker_list_empty_m;
const fsl_zip_writer fsl_zip_writer_empty = fsl_zip_writer_empty_m;

const fsl_allocator fsl_allocator_stdalloc = {
fsl_realloc_f_stdalloc,
NULL
};

fsl_lib_configurable_t fsl_lib_configurable = {
  {/*allocator*/ fsl_realloc_f_stdalloc, NULL}
};

void * fsl_malloc( fsl_size_t n ){
  return n
    ? fsl_realloc(NULL, n)
    : NULL;
}

void fsl_free( void * mem ){
  if(mem) fsl_realloc(mem, 0);
}

void * fsl_realloc( void * mem, fsl_size_t n ){
#define FLCA fsl_lib_configurable.allocator
  if(!mem){
    /* malloc() */
    return n
      ? FLCA.f(FLCA.state, NULL, n)
      : NULL;
  }else if(!n){
    /* free() */
    FLCA.f(FLCA.state, mem, 0);
    return NULL;
  }else{
    /* realloc() */
    return FLCA.f(FLCA.state, mem, n);
  }
#undef FLCA
}

void * fsl_realloc_f_stdalloc(void * state __unused, void * mem, fsl_size_t n){
  if(!mem){
    return malloc(n);
  }else if(!n){
    free(mem);
    return NULL;
  }else{
    return realloc(mem, n);
  }
}

int fsl_is_uuid(char const * str){
  fsl_size_t const len = fsl_strlen(str);
  if(FSL_STRLEN_SHA1==len){
    return fsl_validate16(str, FSL_STRLEN_SHA1) ? FSL_STRLEN_SHA1 : 0;
  }else if(FSL_STRLEN_K256==len){
    return fsl_validate16(str, FSL_STRLEN_K256) ? FSL_STRLEN_K256 : 0;
  }else{
    return 0;
  }
}
int fsl_is_uuid_len(int x){
  switch(x){
    case FSL_STRLEN_SHA1:
    case FSL_STRLEN_K256:
      return x;
    default:
      return 0;
  }
}
void fsl_error_clear( fsl_error * const err ){
  fsl_buffer_clear(&err->msg);
  *err = fsl_error_empty;
}

void fsl_error_reset( fsl_error * const err ){
  err->code = 0;
  fsl_buffer_reuse(&err->msg);
}

int fsl_error_copy( fsl_error const * const src, fsl_error * const dest ){
  if(src==dest) return FSL_RC_MISUSE;
  else {
    int rc = 0;
    fsl_buffer_reuse(&dest->msg);
    dest->code = src->code;
    if(FSL_RC_OOM!=src->code){
      rc = fsl_buffer_append( &dest->msg, src->msg.mem, src->msg.used );
    }
    return rc;
  }
}

void fsl_error_move( fsl_error * const lower, fsl_error * const higher ){
  fsl_error const err = *lower;
  *lower = *higher;
  lower->code = 0;
  lower->msg.used = lower->msg.cursor = 0;
  *higher = err;
}

int fsl_error_setv( fsl_error * const err, int code, char const * fmt,
                    va_list args ){
  fsl_buffer_reuse(&err->msg);
  if(code){
    int rc = 0;
    err->code = code;
    if(FSL_RC_OOM!=code){
      if(fmt) rc = fsl_buffer_appendfv(&err->msg, fmt, args);
      else rc = fsl_buffer_appendf(&err->msg, "fsl_rc_e #%d: %s",
                                   code, fsl_rc_cstr(code));
      if(rc) err->code = rc;
    }
    return rc ? rc : code;
  }else{ /* clear error state */
    err->code = 0;
    return 0;
  }

}

int fsl_error_set( fsl_error * const err, int code, char const * fmt,
                   ... ){
  int rc;
  va_list args;
  va_start(args,fmt);
  rc = fsl_error_setv(err, code, fmt, args);
  va_end(args);
  return rc;
}


int fsl_error_get( fsl_error const * const err, char const ** str,
                   fsl_size_t * const len ){
  if(str) *str = err->msg.used
            ? (char const *)err->msg.mem
            : NULL;
  if(len) *len = err->msg.used;
  return err->code;
}


char const * fsl_rc_cstr(int rc){
  switch((fsl_rc_e)rc){
    /* we cast ^^^^ so that gcc will warn if the switch() below is
       missing any fsl_rc_e entries. */;
#define STR(T) case FSL_RC_##T: return "FSL_RC_" #T
    STR(ACCESS);
    STR(ALREADY_EXISTS);
    STR(AMBIGUOUS);
    STR(BREAK);
    STR(SYNTAX);
    STR(CANNOT_HAPPEN);
    STR(CHECKSUM_MISMATCH);
    STR(CONFLICT);
    STR(CONSISTENCY);
    STR(DB);
    STR(DELTA_INVALID_OPERATOR);
    STR(DELTA_INVALID_SEPARATOR);
    STR(DELTA_INVALID_SIZE);
    STR(DELTA_INVALID_TERMINATOR);
    STR(DIFF_BINARY);
    STR(DIFF_WS_ONLY);
    STR(end);
    STR(ERROR);
    STR(INTERRUPTED);
    STR(IO);
    STR(LOCKED);
    STR(MISSING_INFO);
    STR(MISUSE);
    STR(NOOP);
    STR(NOT_A_CKOUT);
    STR(NOT_A_REPO);
    STR(NOT_FOUND);
    STR(NYI);
    STR(OK);
    STR(OOM);
    STR(PHANTOM);
    STR(RANGE);
    STR(REPO_MISMATCH);
    STR(REPO_NEEDS_REBUILD);
    STR(REPO_VERSION);
    STR(SIZE_MISMATCH);
    STR(STEP_DONE);
    STR(STEP_ERROR);
    STR(STEP_ROW);
    STR(TYPE);
    STR(UNKNOWN_RESOURCE);
    STR(UNSUPPORTED);
    STR(WOULD_FORK);
#undef STR
  }
  return NULL;
}

char const * fsl_library_version(){
  return FSL_LIBRARY_VERSION;
}

bool fsl_library_version_matches(char const * yourLibVersion){
  return 0 == fsl_strcmp(FSL_LIBRARY_VERSION, yourLibVersion);
}

double fsl_unix_to_julian( fsl_time_t unix_ ){
  return (unix_ * 1.0 / 86400.0 ) + 2440587.5;
}

double fsl_julian_now(){
  return fsl_unix_to_julian( time(0) );
}


int fsl_strcmp(const char *zA, const char *zB){
  if( zA==0 ) return zB ? -1 : 0;
  else if( zB==0 ) return 1;
  else{
    int a, b;
    do{
      a = *zA++;
      b = *zB++;
    }while( a==b && a!=0 );
    return ((unsigned char)a) - (unsigned char)b;
  }
}


int fsl_strcmp_cmp( void const * lhs, void const * rhs ){
  return fsl_strcmp((char const *)lhs, (char const *)rhs);
}

int fsl_strncmp(const char *zA, const char *zB, fsl_size_t nByte){
  if( !zA ) return zB ? -1 : 0;
  else if( !zB ) return +1;
  else if(!nByte) return 0;
  else{
    int a, b;
    do{
      a = *zA++;
      b = *zB++;
    }while( a==b && a!=0 && (--nByte)>0 );
    return (nByte>0) ? (((unsigned char)a) - (unsigned char)b) : 0;
  }
}

  
int fsl_uuidcmp( fsl_uuid_cstr lhs, fsl_uuid_cstr rhs ){
  if(!lhs) return rhs ? -1 : 0;
  else if(!rhs) return 1;
  else if(lhs[FSL_STRLEN_SHA1] && rhs[FSL_STRLEN_SHA1]){
    return fsl_strncmp( lhs, rhs, FSL_STRLEN_K256);
  }else if(!lhs[FSL_STRLEN_SHA1] && !rhs[FSL_STRLEN_SHA1]){
    return fsl_strncmp( lhs, rhs, FSL_STRLEN_SHA1 );
  }else{
    return fsl_strcmp(lhs, rhs);
  }      
}

int fsl_strnicmp(const char *zA, const char *zB, fsl_int_t nByte){
  if( zA==0 ){
    if( zB==0 ) return 0;
    return -1;
  }else if( zB==0 ){
    return +1;
  }
  if( nByte<0 ) nByte = (fsl_int_t)fsl_strlen(zB);
  return sqlite3_strnicmp(zA, zB, nByte);
}

int fsl_stricmp(const char *zA, const char *zB){
  if( zA==0 ) return zB ? -1 : 0;
  else if( zB==0 ) return 1;
  else{
    fsl_int_t nByte;
    int rc;
    nByte = (fsl_int_t)fsl_strlen(zB);
    rc = sqlite3_strnicmp(zA, zB, nByte);
    return ( rc==0 && zA[nByte] ) ? 1 : rc;
  }
}

int fsl_stricmp_cmp( void const * lhs, void const * rhs ){
  return fsl_stricmp((char const *)lhs, (char const *)rhs);
}

fsl_size_t fsl_strlen( char const * src ){
  char const * const b = src;
  if(src) while( *src ) ++src;
  return (fsl_size_t)(src - b);
}

char * fsl_strndup( char const * src, fsl_int_t len ){
  if(!src) return NULL;
  else{
    fsl_buffer b = fsl_buffer_empty;
    if(len<0) len = (fsl_int_t)fsl_strlen(src);
    fsl_buffer_append( &b, src, len );
    return (char*)b.mem;
  }
}

char * fsl_strdup( char const * src ){
  return fsl_strndup(src, -1);
}

fsl_size_t fsl_strlcpy(char * dst, const char * src, fsl_size_t dstsz){
  fsl_size_t offset = 0;

  if(dstsz){
    while((*(dst+offset) = *(src+offset))!='\0'){
      if(++offset == dstsz){
        --offset;
        break;
      }
    }
  }
  *(dst+offset) = '\0';
  while(*(src+offset)!='\0'){
    ++offset;	/* Return src length. */
  }
  return offset;
}

fsl_size_t fsl_strlcat(char *dst, const char *src, fsl_size_t dstsz){
  fsl_size_t offset;
  int dstlen, srclen, idx = 0;

  offset = dstlen = fsl_strlen(dst);
  srclen = fsl_strlen(src);
  if( offset>=dstsz-1 )
    return dstlen+srclen;

  while((*(dst+offset++) = *(src+idx++))!='\0'){
    if(offset==dstsz-1){
      break;
    }
  }
  *(dst+offset)='\0';
  return dstlen+srclen;
}

/*
   Return TRUE if the string begins with something that looks roughly
   like an ISO date/time string.  The SQLite date/time functions will
   have the final say-so about whether or not the date/time string is
   well-formed.
*/
char fsl_str_is_date(const char *z){
  if(!z || !*z) return 0;
  else if( !fsl_isdigit(z[0]) ) return 0;
  else if( !fsl_isdigit(z[1]) ) return 0;
  else if( !fsl_isdigit(z[2]) ) return 0;
  else if( !fsl_isdigit(z[3]) ) return 0;
  else if( z[4]!='-') return 0;
  else if( !fsl_isdigit(z[5]) ) return 0;
  else if( !fsl_isdigit(z[6]) ) return 0;
  else if( z[7]!='-') return 0;
  else if( !fsl_isdigit(z[8]) ) return 0;
  else if( !fsl_isdigit(z[9]) ) return 0;
  else return 1;
}

int fsl_str_is_date2(const char *z){
  int rc = -1;
  int pos = 0;
  if(!z || !*z) return 0;
  else if( !fsl_isdigit(z[pos++]) ) return 0;
  else if( !fsl_isdigit(z[pos++]) ) return 0;
  else if( !fsl_isdigit(z[pos++]) ) return 0;
  else if( !fsl_isdigit(z[pos++]) ) return 0;
  else if( z[pos]=='-') ++pos;
  else{
    if(fsl_isdigit(z[pos++]) && '-'==z[pos++]){
      rc = 1;
    }else{
      return 0;
    }
  }
  if( !fsl_isdigit(z[pos++]) ) return 0;
  else if( !fsl_isdigit(z[pos++]) ) return 0;
  else if( z[pos++]!='-') return 0;
  else if( !fsl_isdigit(z[pos++]) ) return 0;
  else if( !fsl_isdigit(z[pos++]) ) return 0;
  assert(10==pos || 11==pos);
  return rc;
}

bool fsl_str_bool( char const * s ){
  switch(s ? *s : 0){
    case 0: case '0':
    case 'f': case 'F': // "false"
    case 'n': case 'N': // "no"
      return false;
    case '1':
    case 't': case 'T': // "true"
    case 'y': case 'Y': // "yes"
      return true;
    default: {
      char buf[5] = {0,0,0,0,0};
      int i;
      for( i = 0; (i<5) && *s; ++i, ++s ){
        buf[i] = fsl_tolower(*s);
      }
      if(0==fsl_strncmp(buf, "off", 3)) return false;
      return true;
    }
  }
}

char * fsl_user_name_guess(){
  char const ** e;
  static char const * list[] = {
  "FOSSIL_USER",
#if defined(_WIN32)
  "USERNAME",
#else
  "USER",
  "LOGNAME",
#endif
  NULL /* sentinel */
  };
  char * rv = NULL;
  for( e = list; *e; ++e ){
    rv = fsl_getenv(*e);
    if(rv){
      /*
        Because fsl_getenv() has the odd requirement of needing
        fsl_filename_free(), and we want strings returned from this
        function to be safe for passing to fsl_free(), we have to dupe
        the string. We "could" block this off to happen only on the
        platforms for which fsl_getenv() requires an extra encoding
        step, but that would likely eventually lead to a bug.
      */
      char * kludge = fsl_strdup(rv);
      fsl_filename_free(rv);
      rv = kludge;
      break;
    }
  }
  return rv;
}

void fsl__fatal( int code, char const * fmt, ... ){
  static bool inFatal = false;
  if(inFatal){
    /* This can only happen if the fsl_appendv() bits
       call this AND trigger it via fsl_fprintf() below,
       neither of which is currently the case.
    */
    assert(!"fsl__fatal() called recursively.");
    abort();
  }else{
    va_list args;
    inFatal = true;
    fsl_fprintf(stderr, "FATAL ERROR: code=%d (%s)\n",
                code, fsl_rc_cstr(code));
    if(fmt){
      va_start(args,fmt);
      fsl_fprintfv(stderr, fmt, args);
      va_end(args);
      fwrite("\n", 1, 1, stderr);
    }
    exit(EXIT_FAILURE);
  }
}

#if 0
char * fsl_unix_to_iso8601( fsl_time_t u ){
  enum { BufSize = 20 };
  char buf[BufSize]= {0,};
  time_t const tt = (time_t)u;
  fsl_strftime( buf, BufSize, "%Y-%m-%dT%H:%M:%S", gmtime(&tt) );
  return fsl_strdup(buf);
}
#endif


bool fsl_iso8601_to_julian( char const * zDate, double * out ){
  /* Adapted from this article:

     https://quasar.as.utexas.edu/BillInfo/JulianDatesG.html
  */
  char const * p = zDate;
  int y = 0, m = 0, d = 0;
  int h = 0, mi = 0, s = 0, f = 0;
  double j = 0;
  if(!zDate || !*zDate){
    return 0;
  }
#define DIG(NUM) if(!fsl_isdigit(*p)) return 0; \
  NUM=(NUM*10)+(*(p++)-'0')

  DIG(y);DIG(y);DIG(y);DIG(y);
  if('-'!=*p++) return 0;
  DIG(m);DIG(m);
  if('-'!=*p++) return 0;
  DIG(d);DIG(d);
  if('T' != *p++) return 0;
  DIG(h);DIG(h);
  if(':'!=*p++) return 0;
  DIG(mi);DIG(mi);
  if(':'!=*p++) return 0;
  DIG(s);DIG(s);
  if('.'==*p++){
    DIG(f);DIG(f);DIG(f);
  }
  if(out){
    typedef int64_t TI;
    TI A, B, C, E, F;
    if(m<3){
      --y;
      m += 12;
    }
    A = y/100;
    B = A/4;
    C = 2-A+B;
    E = (TI)(365.25*(y+4716));
    F = (TI)(30.6001*(m+1));
    j = C + d + E + F - 1524.5;
    j += ((1.0*h)/24) + ((1.0*mi)/1440) + ((1.0*s)/86400);
    if(0 != f){
      j += (1.0*f)/86400000;
    }
    *out = j;
  }
  return 1;
#undef DIG
}

fsl_time_t fsl_julian_to_unix( double JD ){
  return (fsl_time_t) ((JD - 2440587.5) * 86400);
}

bool fsl_julian_to_iso8601( double J, char * out, bool addMs ){
  /* Adapted from this article:

     https://quasar.as.utexas.edu/BillInfo/JulianDatesG.html
  */
  typedef int64_t TI;
  int Y, M, D, H, MI, S, F;
  TI ms;
  char * z = out;
  if(!out || (J<=0)) return 0;
  else{
    double Z;
    TI W, X;
    TI A, B;
    TI C, DD, E, F;

    Z = J + 0.5;
    W = (TI)((Z-1867216.25)/36524.25);
    X = W/4;
    A = (TI)(Z+1+W-X);
    B = A+1524;
    C = (TI)((B-122.1)/365.25);
    DD = (TI)(365.25 * C);
    E = (TI)((B-DD)/30.6001);
    F = (TI)(30.6001 * E);
    D = (int)(B - DD - F);
    M = (E<=13) ? (E-1) : (E-13);
    Y = (M<3) ? (C-4715) : (C-4716);
  }

  if(Y<0 || Y>9999) return 0;
  else if(M<1 || M>12) return 0;
  else if(D<1 || D>31) return 0;

  ms = (TI)((J-(TI)J) * 86400001.0)
    /* number of milliseconds in the fraction part of the JDay. The
       non-0 at the end works around a problem where SS.000 converts
       to (SS-1).999. This will only hide the bug for the cases i've
       seen it, and might introduce other inaccuracies
       elsewhere. Testing it against the current libfossil event table
       produces good results - at most a 1ms round-trip fidelity loss
       for the (currently ~1157) records being checked. The suffix of
       1.0 was found to be a decent value via much testing with the
       libfossil and fossil(1) source repos.
    */;

  if( (H = ms / 3600000) ){
    ms -= H * 3600000;
    H = (H + 12) % 24;
  }else{
    H = 12 /* astronomers start their day at noon. */;
  }
  if( (MI = ms / 60000) ) ms -= MI * 60000;
  if( (S = ms / 1000) ) ms -= S * 1000;
  assert(ms<1000);
  F = (int)(ms);

  assert(H>=0 && H<24);
  assert(MI>=0 && MI<60);
  assert(S>=0 && S<60);
  assert(F>=0 && F<1000);

  if(H<0 || H>23) return 0;
  else if(MI<0 || MI>59) return 0;
  else if(S<0 || S>59) return 0;
  else if(F<0 || F>999) return 0;
#define UGLY_999_KLUDGE 1
  /* The fossil(1) repo has 27 of 10041 records which exhibit the
     SS.999 behaviour commented on above. With this kludge, that
     number drops to 0. But it's still an ugly, ugly kludge.
     OTOH, the chance of the .999 being correct is 1 in 1000,
     whereas we see "correct" behaviour more often (2.7 in 1000)
     with this workaround.
   */
#if UGLY_999_KLUDGE
  if(999==F){
    char oflow = 0;
    int s2 = S, mi2 = MI, h2 = H;
    if(++s2 == 60){ /* Overflow minute */
      s2 = 0;
      if(++mi2 == 60){ /* Overflow hour */
        mi2 = 0;
        if(++h2 == 24){ /* Overflow day */
          /* leave this corner-corner case in place */
          oflow = 1;
        }
      }
    }
    /* MARKER(("UGLY 999 KLUDGE (A): H=%d MI=%d S=%d F=%d\n", H, MI, S, F)); */
    if(!oflow){
      F = 0;
      S = s2;
      MI = mi2;
      H = h2;
      /* MARKER(("UGLY 999 KLUDGE (B): H=%d MI=%d S=%d F=%d\n", H, MI, S, F)); */
    }
  }
#endif
#undef UGLY_999_KLUDGE
  *(z++) = '0'+(Y/1000);
  *(z++) = '0'+(Y%1000/100);
  *(z++) = '0'+(Y%100/10);
  *(z++) = '0'+(Y%10);
  *(z++) = '-';
  *(z++) = '0'+(M/10);
  *(z++) = '0'+(M%10);
  *(z++) = '-';
  *(z++) = '0'+(D/10);
  *(z++) = '0'+(D%10);
  *(z++) = 'T';
  *(z++) = '0'+(H/10);
  *(z++) = '0'+(H%10);
  *(z++) = ':';
  *(z++) = '0'+(MI/10);
  *(z++) = '0'+(MI%10);
  *(z++) = ':';
  *(z++) = '0'+(S/10);
  *(z++) = '0'+(S%10);
  if(addMs){
    *(z++) = '.';
    *(z++) = '0'+(F%1000/100);
    *(z++) = '0'+(F%100/10);
    *(z++) = '0'+(F%10);
  }
  *z = 0;
  return 1;
}

#if FSL_CONFIG_ENABLE_TIMER
/**
   For the fsl_timer_xxx() family of functions...
*/
#ifdef _WIN32
# include <windows.h>
#else
# include <sys/time.h>
# include <sys/resource.h>
# include <unistd.h>
# include <fcntl.h>
# include <errno.h>
#endif
#endif
/* FSL_CONFIG_ENABLE_TIMER */

/**
   Get user and kernel times in microseconds.
*/
static void fsl_cpu_times(uint64_t *piUser, uint64_t *piKernel){
#if !FSL_CONFIG_ENABLE_TIMER
  if(piUser) *piUser = 0U;
  if(piKernel) *piKernel = 0U;
#else
#ifdef _WIN32
  FILETIME not_used;
  FILETIME kernel_time;
  FILETIME user_time;
  GetProcessTimes(GetCurrentProcess(), &not_used, &not_used,
                  &kernel_time, &user_time);
  if( piUser ){
     *piUser = ((((uint64_t)user_time.dwHighDateTime)<<32) +
                         (uint64_t)user_time.dwLowDateTime + 5)/10;
  }
  if( piKernel ){
     *piKernel = ((((uint64_t)kernel_time.dwHighDateTime)<<32) +
                         (uint64_t)kernel_time.dwLowDateTime + 5)/10;
  }
#else
  struct rusage s;
  getrusage(RUSAGE_SELF, &s);
  if( piUser ){
    *piUser = ((uint64_t)s.ru_utime.tv_sec)*1000000 + s.ru_utime.tv_usec;
  }
  if( piKernel ){
    *piKernel =
              ((uint64_t)s.ru_stime.tv_sec)*1000000 + s.ru_stime.tv_usec;
  }
#endif
#endif
/* FSL_CONFIG_ENABLE_TIMER */
}


void fsl_timer_start(fsl_timer_state * const ft){
  fsl_cpu_times( &ft->user, &ft->system );
}

uint64_t fsl_timer_fetch(fsl_timer_state const * const t){
  uint64_t eu = 0, es = 0;
  fsl_cpu_times( &eu, &es );
  return (eu - t->user) + (es - t->system);
}

uint64_t fsl_timer_reset(fsl_timer_state * const t){
  uint64_t const rc = fsl_timer_fetch(t);
  fsl_cpu_times( &t->user, &t->system );
  return rc;
}

uint64_t fsl_timer_stop(fsl_timer_state * const t){
  uint64_t const rc = fsl_timer_fetch(t);
  *t = fsl_timer_state_empty;
  return rc;
}

unsigned int fsl_rgb_encode( int r, int g, int b ){
  return (unsigned int)(((r&0xFF)<<16) + ((g&0xFF)<<8) + (b&0xFF));
}

void fsl_rgb_decode( unsigned int src, int *r, int *g, int *b ){
  if(r) *r = (src&0xFF0000)>>16;
  if(g) *g = (src&0xFF00)>>8;
  if(b) *b = src&0xFF;
}

unsigned fsl_gradient_color(unsigned c1, unsigned c2, unsigned int n, unsigned int i){
  unsigned c;   /* Result color */
  unsigned x1, x2;
  if( i==0 || n==0 ) return c1;
  else if(i>=n) return c2;
  x1 = (c1>>16)&0xff;
  x2 = (c2>>16)&0xff;
  c = (x1*(n-i) + x2*i)/n<<16 & 0xff0000;
  x1 = (c1>>8)&0xff;
  x2 = (c2>>8)&0xff;
  c |= (x1*(n-i) + x2*i)/n<<8 & 0xff00;
  x1 = c1&0xff;
  x2 = c2&0xff;
  c |= (x1*(n-i) + x2*i)/n & 0xff;
  return c;
}


fsl_size_t fsl_simplify_sql( char * sql, fsl_int_t len ){
  char * wat = sql /* write pos */;
  char * rat = sql /* read pos */;
  char const * end /* one-past-the-end */;
  char inStr = 0 /* in an SQL string? */;
  char prev = 0 /* previous character. Sometimes. */;
  if(!sql || !*sql) return 0;
  else if(len < 0) len = fsl_strlen(sql);
  if(!len) return 0;
  end = sql + len;
  while( *rat && (rat < end) ){
    switch(*rat){
      case 0: break;
      case '\r':
      case '\n':
        /* Bug: we don't handle \r\n pairs. Because nobody
           should never have to :/. */
        if(inStr || (prev!=*rat)){
          /* Keep them as-is */
          prev = *wat++ = *rat++;
        }else{
          /* Collapse multiples into one. */
          ++rat;
        }
        continue;
      case ' ':
      case '\t':
      case '\v':
      case '\f':
        if(inStr){
          /* Keep them as-is */
          prev = *wat++ = *rat++;
        }else{
          /* Reduce to a single space. */
          /* f_out("prev=[%c] rat=[%c]\n", prev, *rat); */
          if(prev != *rat){
            *wat++ = ' ';
            prev = *rat;
          }
          ++rat;
        }
        continue;
      case '\'': /* SQL strings */
        prev = *wat++ = *rat++;
        if(!inStr){
          inStr = 1;
        }else if('\'' == *rat){
          /* Escaped quote */
          *wat++ = *rat++;
        }else{
          /* End of '...' string. */
          inStr = 0;
        }
        continue;
      default:
        prev = *wat++ = *rat++;
        continue;
    }
  }
  *wat = 0;
  return (fsl_size_t)(wat - sql);
}

/**
   Convenience form of fsl_simplify_sql() which assumes b holds an SQL
   string. It gets processed by fsl_simplify_sql() and its 'used'
   length potentially gets adjusted to match the adjusted SQL string.
*/
fsl_size_t fsl_simplify_sql_buffer( fsl_buffer * const b ){
  return b->used = fsl_simplify_sql( (char *)b->mem, (fsl_int_t)b->used );
}

char const *fsl_preferred_ckout_db_name(){
#if FSL_PLATFORM_IS_WINDOWS
  return "_FOSSIL_";
#else
  return ".fslckout";
#endif
}

bool fsl_isatty(int fd){
  return isatty(fd) ? true : false;
}

bool fsl__is_reserved_fn_windows(const char *zPath, fsl_int_t nameLen){
  static const char *const azRes[] = {
    "CON", "PRN", "AUX", "NUL", "COM", "LPT"
  };
  unsigned int i;
  char const * zEnd;
  if(nameLen<0) nameLen = (fsl_int_t)fsl_strlen(zPath);
  zEnd = zPath + nameLen;
  while( zPath < zEnd ){
    for(i=0; i<sizeof(azRes)/sizeof(azRes[0]); ++i){
      if( fsl_strnicmp(zPath, azRes[i], 3)==0
       && ((i>=4 && fsl_isdigit(zPath[3])
                 && (zPath[4]=='/' || zPath[4]=='.' || zPath[4]==0))
          || (i<4 && (zPath[3]=='/' || zPath[3]=='.' || zPath[3]==0)))
      ){
        return true;
      }
    }
    while( zPath<zEnd && zPath[0]!='/' ) ++zPath;
    while( zPath<zEnd && zPath[0]=='/' ) ++zPath;
  }
  return false;
}

bool fsl_is_reserved_fn(const char *zFilename, fsl_int_t nameLen){
  fsl_size_t nFilename = nameLen>=0
    ? (fsl_size_t)nameLen : fsl_strlen(zFilename);
  char const * zEnd;
  int gotSuffix = 0;
  assert( zFilename && "API misuse" );
#if FSL_PLATFORM_IS_WINDOWS // || 1
  if(nFilename>2 && fsl__is_reserved_fn_windows(zFilename, nameLen)){
    return true;
  }
#endif
  if( nFilename<8 ) return false; /* strlen("_FOSSIL_") */
  zEnd = zFilename + nFilename;
  if( nFilename>=12 ){ /* strlen("_FOSSIL_-(shm|wal)") */
    /* Check for (-wal, -shm, -journal) suffixes, with an eye towards
    ** runtime speed. */
    if( zEnd[-4]=='-' ){
      if( fsl_strnicmp("wal", &zEnd[-3], 3)
       && fsl_strnicmp("shm", &zEnd[-3], 3) ){
        return false;
      }
      gotSuffix = 4;
    }else if( nFilename>=16 && zEnd[-8]=='-' ){ /*strlen(_FOSSIL_-journal) */
      if( fsl_strnicmp("journal", &zEnd[-7], 7) ) return false;
      gotSuffix = 8;
    }
    if( gotSuffix ){
      assert( 4==gotSuffix || 8==gotSuffix );
      zEnd -= gotSuffix;
      nFilename -= gotSuffix;
      gotSuffix = 1;
    }
    assert( nFilename>=8 && "strlen(_FOSSIL_)" );
    assert( gotSuffix==0 || gotSuffix==1 );
  }
  switch( zEnd[-1] ){
    case '_':{
      if( fsl_strnicmp("_FOSSIL_", &zEnd[-8], 8) ) return false;
      if( 8==nFilename ) return true;
      return zEnd[-9]=='/' ? true : !!gotSuffix;
    }
    case 'T':
    case 't':{
      if( nFilename<9 || zEnd[-9]!='.'
       || fsl_strnicmp(".fslckout", &zEnd[-9], 9) ){
        return false;
      }
      if( 9==nFilename ) return true;
      return zEnd[-10]=='/' ? true : !!gotSuffix;
    }
    default:{
      return false;
    }
  }
}

void fsl_randomness(unsigned int n, void *tgt){
  sqlite3_randomness((int)n, tgt);
}

int fsl_system(const char *zOrigCmd){
  int rc;
  /* The following was ported over from fossil(1). As of this writing,
     the Windows version is completely untested even for
     compilability. */
#if defined(_WIN32)
  /* On windows, we have to put double-quotes around the entire command.
  ** Who knows why - this is just the way windows works.
  */
  char *zNewCmd = fsl_mprintf("\"%s\"", zOrigCmd);
  if(!zNewCmd){FSL__WARN_OOM; return FSL_RC_OOM;}
  wchar_t *zUnicode = (wchar_t *)fsl_utf8_to_unicode(zNewCmd);
  if(!zUnicode){
    fsl_free(zNewCmd);
    FSL__WARN_OOM;
    return FSL_RC_OOM;
  }
  //fossil_assert_safe_command_string(zOrigCmd);
  rc = _wsystem(zUnicode);
  fsl_unicode_free(zUnicode);
  free(zNewCmd);
#else
  /* On unix, evaluate the command directly.
  */
  //fossil_assert_safe_command_string(zOrigCmd);
  /* The regular system() call works to get a shell on unix */
  rc = system(zOrigCmd);
  if(rc) {
    if(-1==rc) rc = errno;
    else if(rc>0){
      rc = FSL_RC_ERROR;
    }
  }
#endif
  return rc ? fsl_errno_to_rc(rc, FSL_RC_ERROR) : 0;
}


#undef MARKER
#if defined(_WIN32) || defined(WIN32)
#undef isatty
#endif
/* end of file ./src/fsl.c */
/* start of file ./src/annotate.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/
/************************************************************************
  This file implements the annoate/blame/praise-related APIs.
*/
#include <assert.h>
#include <string.h>/*memset()*/

/* Only for debugging */
#include <stdio.h>
#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)

const fsl_annotate_opt fsl_annotate_opt_empty = fsl_annotate_opt_empty_m;

/*
** The status of an annotation operation is recorded by an instance
** of the following structure.
*/
typedef struct Annotator Annotator;
struct Annotator {
  fsl__diff_cx c;       /* The diff-engine context */
  fsl_buffer headVersion;/*starting version of the content*/
  struct AnnLine {  /* Lines of the original files... */
    const char *z;       /* The text of the line. Points into
                            this->headVersion. */
    short int n;         /* Number of bytes (omitting trailing \n) */
    short int iVers;     /* Level at which tag was set */
  } *aOrig;
  unsigned int nOrig;/* Number of elements in aOrig[] */
  unsigned int nVers;/* Number of versions analyzed */
  bool bMoreToDo;    /* True if the limit was reached */
  fsl_id_t origId;       /* RID for the zOrigin version */
  fsl_id_t showId;       /* RID for the version being analyzed */
  struct AnnVers {
    char *zFUuid;   /* File being analyzed */
    char *zMUuid;   /* Check-in containing the file */
    char *zUser;    /* Name of user who did the check-in */
    double mtime;   /* [event].[mtime] db entry */
  } *aVers;         /* For each check-in analyzed */
  unsigned int naVers; /* # of entries allocated in this->aVers */
  fsl_timer_state timer;
};

static const Annotator Annotator_empty = {
fsl__diff_cx_empty_m,
fsl_buffer_empty_m/*headVersion*/,
NULL/*aOrig*/,
0U/*nOrig*/, 0U/*nVers*/,
false/*bMoreToDo*/,
0/*origId*/,
0/*showId*/,
NULL/*aVers*/,
0U/*naVerse*/,
fsl_timer_state_empty_m
};

static void fsl__annotator_clean(Annotator * const a){
  unsigned i;
  fsl__diff_cx_clean(&a->c);
  for(i = 0; i < a->nVers; ++i){
    fsl_free(a->aVers[i].zFUuid);
    fsl_free(a->aVers[i].zMUuid);
    fsl_free(a->aVers[i].zUser);
  }
  fsl_free(a->aVers);
  fsl_free(a->aOrig);
  fsl_buffer_clear(&a->headVersion);
}

static uint64_t fsl__annotate_opt_difflags(fsl_annotate_opt const * const opt){
  uint64_t diffFlags = FSL_DIFF2_STRIP_EOLCR;
  if(opt->spacePolicy>0) diffFlags |= FSL_DIFF2_IGNORE_ALLWS;
  else if(opt->spacePolicy<0) diffFlags |= FSL_DIFF2_IGNORE_EOLWS;
  return diffFlags;
}

/**
   Initializes the annocation process by populating `a` from
   a->toAnnote, which must have been previously populated.  `a` must
   have already been cleanly initialized via copying from
   Annotator_empty and a->headVersion populated.  Returns 0 on success,
   else:

   - FSL_RC_RANGE if pInput is empty.
   - FSL_RC_OOM on OOM.

   Regardless of success or failure, `a` must eventually be passed
   to fsl__annotator_clean() to free up any resources.
*/
static int fsl__annotation_start(Annotator * const a,
                                 fsl_annotate_opt const * const opt){
  int rc;
  uint64_t const diffFlags = fsl__annotate_opt_difflags(opt);
  if(opt->spacePolicy>0){
    a->c.cmpLine = fsl_dline_cmp_ignore_ws;
  }else{
    assert(fsl_dline_cmp == a->c.cmpLine);
  }
  rc = fsl_break_into_dlines(fsl_buffer_cstr(&a->headVersion),
                             (fsl_int_t)a->headVersion.used,
                             (uint32_t*)&a->c.nTo, &a->c.aTo, diffFlags);
  if(rc) goto end;
  if(!a->c.nTo){
    rc = FSL_RC_RANGE;
    goto end;
  }
  a->aOrig = fsl_malloc( (fsl_size_t)(sizeof(a->aOrig[0]) * a->c.nTo) );
  if(!a->aOrig){
    rc = FSL_RC_OOM;
    goto end;
  }
  for(int i = 0; i < a->c.nTo; ++i){
    a->aOrig[i].z = a->c.aTo[i].z;
    a->aOrig[i].n = a->c.aTo[i].n;
    a->aOrig[i].iVers = -1;
  }
  a->nOrig = (unsigned)a->c.nTo;
  end:
  return rc;
}

/**
   The input pParent is the next most recent ancestor of the file
   being annotated.  Do another step of the annotation. On success
   return 0 and, if additional annotation is required, assign *doMore
   (if not NULL) to true.
*/
static int fsl__annotation_step(
  Annotator * const a,
  fsl_buffer const *pParent,
  int iVers,
  fsl_annotate_opt const * const opt
){
  int i, j, rc;
  int lnTo;
  uint64_t const diffFlags = fsl__annotate_opt_difflags(opt);

  /* Prepare the parent file to be diffed */
  rc = fsl_break_into_dlines(fsl_buffer_cstr(pParent),
                             (fsl_int_t)pParent->used,
                             (uint32_t*)&a->c.nFrom, &a->c.aFrom,
                             diffFlags);
  if(rc) goto end;
  else if( a->c.aFrom==0 ){
    return 0;
  }
  //MARKER(("Line #1: %.*s\n", (int)a->c.aFrom[0].n, a->c.aFrom[0].z));
  /* Compute the differences going from pParent to the file being
  ** annotated. */
  rc = fsl__diff_all(&a->c);
  if(rc) goto end;

  /* Where new lines are inserted on this difference, record the
  ** iVers as the source of the new line.
  */
  for(i=lnTo=0; i<a->c.nEdit; i+=3){
    int const nCopy = a->c.aEdit[i];
    int const nIns = a->c.aEdit[i+2];
    lnTo += nCopy;
    for(j=0; j<nIns; ++j, ++lnTo){
      if( a->aOrig[lnTo].iVers<0 ){
        a->aOrig[lnTo].iVers = iVers;
      }
    }
  }

  /* Clear out the diff results except for c.aTo, as that's pointed to
     by a->aOrig.*/
  fsl_free(a->c.aEdit);
  a->c.aEdit = 0;
  a->c.nEdit = 0;
  a->c.nEditAlloc = 0;

  /* Clear out the from file */
  fsl_free(a->c.aFrom);
  a->c.aFrom = 0;
  a->c.nFrom = 0;
  end:
  return rc;
}

/* MISSING(?) fossil(1) converts the diff inputs into utf8 with no
   BOM. Whether we really want to do that here or rely on the caller
   to is up for debate. If we do it here, we have to make the inputs
   non-const, which seems "wrong" for a library API. */
#define blob_to_utf8_no_bom(A,B) (void)0

static int fsl__annotate_file(fsl_cx * const f,
                              Annotator * const a,
                              fsl_annotate_opt const * const opt){
  int rc = FSL_RC_NYI;
  fsl_buffer step = fsl_buffer_empty /*previous revision*/;
  fsl_id_t cid = 0, fnid = 0; // , rid = 0;
  fsl_stmt q = fsl_stmt_empty;
  bool openedTransaction = false;
  fsl_db * const db = fsl_needs_repo(f);
  if(!db) return FSL_RC_NOT_A_REPO;
  rc = fsl_cx_transaction_begin(f);
  if(rc) goto dberr;
  openedTransaction = true;

  fnid = fsl_db_g_id(db, 0,
                     "SELECT fnid FROM filename WHERE name=%Q %s",
                     opt->filename, fsl_cx_filename_collation(f));
  if(0==fnid){
    rc = fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                        "File not found in repository: %s",
                        opt->filename);
    goto end;
  }
  if(opt->versionRid>0){
    cid = opt->versionRid;
  }else{
    fsl_ckout_version_info(f, &cid, NULL);
    if(cid<=0){
      rc = fsl_cx_err_set(f, FSL_RC_NOT_A_CKOUT,
                          "Cannot determine version RID to "
                          "annotate from.");
      goto end;
    }
  }
  if(opt->originRid>0){
    rc = fsl_vpath_shortest_store_in_ancestor(f, cid, opt->originRid, NULL);
  }else{
    rc = fsl_compute_direct_ancestors(f, cid);
  }
  if(rc) goto end;
  
  rc = fsl_db_prepare(db, &q,
    "SELECT DISTINCT"
    "   (SELECT uuid FROM blob WHERE rid=mlink.fid),"
    "   (SELECT uuid FROM blob WHERE rid=mlink.mid),"
    "   coalesce(event.euser,event.user),"
    "   mlink.fid, event.mtime"
    "  FROM mlink, event, ancestor"
    " WHERE mlink.fnid=%" FSL_ID_T_PFMT
    "   AND ancestor.rid=mlink.mid"
    "   AND event.objid=mlink.mid"
    "   AND mlink.mid!=mlink.pid"
    " ORDER BY ancestor.generation;",
    fnid
  );
  if(rc) goto dberr;
  
  while(FSL_RC_STEP_ROW==fsl_stmt_step(&q)){
    if(a->nVers>=3){
      /* Process at least 3 rows before imposing any limit. That is
         historical behaviour inherited from fossil(1). */
      if(opt->limitMs>0 &&
         fsl_timer_fetch(&a->timer)/1000 >= opt->limitMs){
        a->bMoreToDo = true;
        break;
      }else if(opt->limitVersions>0 && a->nVers>=opt->limitVersions){
        a->bMoreToDo = true;
        break;
      }
    }
    char * zTmp = 0;
    char const * zCol = 0;
    fsl_size_t nCol = 0;
    fsl_id_t const rid = fsl_stmt_g_id(&q, 3);
    double const mtime = fsl_stmt_g_double(&q, 4);
    if(0==a->nVers){
      rc = fsl_content_get(f, rid, &a->headVersion);
      if(rc) goto end;
      blob_to_utf8_no_bom(&a->headVersion,0);
      rc = fsl__annotation_start(a, opt);
      if(rc) goto end;
      a->bMoreToDo = opt->originRid>0;
      a->origId = opt->originRid;
      a->showId = cid;
      assert(0==a->nVers);
      assert(NULL==a->aVers);
    }
    if(a->naVers==a->nVers){
      unsigned int const n = a->naVers ? a->naVers*3/2 : 10;
      void * const x = fsl_realloc(a->aVers, n*sizeof(a->aVers[0]));
      if(NULL==x){
        rc = FSL_RC_OOM;
        goto end;
      }
      a->aVers = x;
      a->naVers = n;
    }
#define AnnStr(COL,FLD) zCol = NULL; \
    rc = fsl_stmt_get_text(&q, COL, &zCol, &nCol);  \
    if(rc) goto end;                                \
    else if(!zCol){ goto end;                                           \
      /*zCol=""; nCol=0; //causes downstream 'RID 0 is invalid' error*/}  \
    zTmp = fsl_strndup(zCol, (fsl_int_t)nCol);  \
    if(!zTmp){ rc = FSL_RC_OOM; goto end; } \
    a->aVers[a->nVers].FLD = zTmp
    AnnStr(0,zFUuid);
    AnnStr(1,zMUuid);
    AnnStr(2,zUser);
#undef AnnStr
    a->aVers[a->nVers].mtime = mtime;
    if( a->nVers>0 ){
      rc = fsl_content_get(f, rid, &step);
      if(!rc){
        rc = fsl__annotation_step(a, &step, a->nVers-1, opt);
      }
      fsl_buffer_reuse(&step);
      if(rc) goto end;
    }
    ++a->nVers;
  }

  assert(0==rc);
  if(0==a->nVers){
    if(opt->versionRid>0){
      rc = fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                          "File [%s] does not exist "
                          "in checkin RID %" FSL_ID_T_PFMT,
                          opt->filename, opt->versionRid);
    }else{
      rc = fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                          "No history found for file: %s",
                          opt->filename);
    }
  }
  
  end:
  fsl_buffer_clear(&step);
  fsl_stmt_finalize(&q);
  if(openedTransaction) fsl_cx_transaction_end(f, rc!=0);
  return rc;
  dberr:
  assert(openedTransaction);
  assert(rc!=0);
  fsl_stmt_finalize(&q);
  fsl_buffer_clear(&step);
  rc = fsl_cx_uplift_db_error2(f, db, rc);
  if(openedTransaction) fsl_cx_transaction_end(f, rc!=0);
  return rc;
}

int fsl_annotate_step_f_fossilesque(void * state,
                                    fsl_annotate_opt const * const opt,
                                    fsl_annotate_step const * const step){
  static const int szHash = 10;
  fsl_outputer const * fout = (fsl_outputer*)state;
  int rc = 0;
  char ymd[24];
  if(step->mtime>0){
    fsl_julian_to_iso8601(step->mtime, &ymd[0], false);
    ymd[10] = 0;
  }
  switch(step->stepType){
    case FSL_ANNOTATE_STEP_VERSION:
      rc = fsl_appendf(fout->out, fout->state,
                       "version %3d: %s %.*s file %.*s\n",
                       step->stepNumber+1, ymd, szHash,
                       step->versionHash, szHash, step->fileHash);
      break;
    case FSL_ANNOTATE_STEP_FULL:
      if(opt->praise){
        rc = fsl_appendf(fout->out, fout->state,
                         "%.*s %s %13.13s: %.*s\n",
                         szHash,
                         opt->fileVersions ? step->fileHash : step->versionHash,
                         ymd, step->username,
                         (int)step->lineLength, step->line);
      }else{
        rc = fsl_appendf(fout->out, fout->state,
                         "%.*s %s %5d: %.*s\n",
                         szHash, opt->fileVersions ? step->fileHash : step->versionHash,
                         ymd, step->lineNumber,
                         (int)step->lineLength, step->line);
      }
      break;
    case FSL_ANNOTATE_STEP_LIMITED:
      if(opt->praise){
        rc = fsl_appendf(fout->out, fout->state,
                         "%*s %.*s\n", szHash+26, "",
                         (int)step->lineLength, step->line);
      }else{
        rc = fsl_appendf(fout->out, fout->state,
                         "%*s %5" PRIu32 ": %.*s\n",
                         szHash+11, "", step->lineNumber,
                         (int)step->lineLength, step->line);
      }
      break;
  }
  return rc;
}


int fsl_annotate( fsl_cx * const f, fsl_annotate_opt const * const opt ){
  int rc;
  Annotator ann = Annotator_empty;
  unsigned int i;
  fsl_buffer * const scratch = fsl__cx_scratchpad(f);
  fsl_annotate_step aStep;

  if(!opt->out){
    return fsl_cx_err_set(f, FSL_RC_MISUSE,
                          "fsl_annotate_opt is missing its output function.");
  }
  if(opt->limitMs>0) fsl_timer_start(&ann.timer);
  rc = fsl__annotate_file(f, &ann, opt);
  if(rc) goto end;
  memset(&aStep,0,sizeof(fsl_annotate_step));

  if(opt->dumpVersions){
    struct AnnVers *av;
    for(av = ann.aVers, i = 0;
        0==rc && i < ann.nVers; ++i, ++av){
      aStep.fileHash = av->zFUuid;
      aStep.versionHash = av->zMUuid;
      aStep.mtime = av->mtime;
      aStep.stepNumber = i;
      aStep.stepType = FSL_ANNOTATE_STEP_VERSION;
      rc = opt->out(opt->outState, opt, &aStep);
    }
    if(rc) goto end;
  }

  for(i = 0; 0==rc && i<ann.nOrig; ++i){
    short iVers = ann.aOrig[i].iVers;
    char const * z = ann.aOrig[i].z;
    int const n = ann.aOrig[i].n;
    if(iVers<0 && !ann.bMoreToDo){
      iVers = ann.nVers-1;
    }
    fsl_buffer_reuse(scratch);
    rc = fsl_buffer_append(scratch, z, n);
    if(rc) break;
    aStep.stepNumber = iVers;
    ++aStep.lineNumber;
    aStep.line = fsl_buffer_cstr(scratch);
    aStep.lineLength = (uint32_t)scratch->used;

    if(iVers>=0){
      struct AnnVers * const av = &ann.aVers[iVers];
      aStep.fileHash = av->zFUuid;
      aStep.versionHash = av->zMUuid;
      aStep.mtime = av->mtime;
      aStep.username = av->zUser;
      aStep.stepType = FSL_ANNOTATE_STEP_FULL;
    }else{
      aStep.fileHash = aStep.versionHash =
        aStep.username = NULL;
      aStep.stepType = FSL_ANNOTATE_STEP_LIMITED;
      aStep.mtime = 0.0;
    }
    rc = opt->out(opt->outState, opt, &aStep);
  }
  
  end:
  fsl__cx_scratchpad_yield(f, scratch);
  fsl__annotator_clean(&ann);
  return rc;
}

#undef MARKER
#undef blob_to_utf8_no_bom
/* end of file ./src/annotate.c */
/* start of file ./src/appendf.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/************************************************************************
The printf-like implementation in this file is based on the one found
in the sqlite3 distribution is in the Public Domain.

This copy was forked for use with the clob API in Feb 2008 by Stephan
Beal (https://wanderinghorse.net/home/stephan/) and modified to send
its output to arbitrary targets via a callback mechanism. Also
refactored the %X specifier handlers a bit to make adding/removing
specific handlers easier.

All code in this file is released into the Public Domain.

The printf implementation (fsl_appendfv()) is pretty easy to extend
(e.g. adding or removing %-specifiers for fsl_appendfv()) if you're
willing to poke around a bit and see how the specifiers are declared
and dispatched. For an example, grep for 'etSTRING' and follow it
through the process of declaration to implementation.

See below for several FSLPRINTF_OMIT_xxx macros which can be set to
remove certain features/extensions.

LICENSE:

  This program is free software; you can redistribute it and/or
  modify it under the terms of the Simplified BSD License (also
  known as the "2-Clause License" or "FreeBSD License".)

  This program is distributed in the hope that it will be useful,
  but without any warranty; without even the implied warranty of
  merchantability or fitness for a particular purpose.
**********************************************************************/

#include <string.h> /* strlen() */
#include <ctype.h>
#include <assert.h>

#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)

/* FIXME: determine this type at compile time via configuration
   options. OTOH, it compiles everywhere as-is so far.
 */
typedef long double LONGDOUBLE_TYPE;

/*
  If FSLPRINTF_OMIT_FLOATING_POINT is defined to a true value, then
  floating point conversions are disabled.
*/
#ifndef FSLPRINTF_OMIT_FLOATING_POINT
#  define FSLPRINTF_OMIT_FLOATING_POINT 0
#endif

/*
  If FSLPRINTF_OMIT_SQL is defined to a true value, then
  the %q, %Q, and %B specifiers are disabled.
*/
#ifndef FSLPRINTF_OMIT_SQL
#  define FSLPRINTF_OMIT_SQL 0
#endif

/*
  If FSLPRINTF_OMIT_HTML is defined to a true value then the %h (HTML
  escape), %t (URL escape), and %T (URL unescape) specifiers are
  disabled.
*/
#ifndef FSLPRINTF_OMIT_HTML
#  define FSLPRINTF_OMIT_HTML 0
#endif

/**
   If true, the %j (JSON string) format is enabled.
*/
#define FSLPRINTF_ENABLE_JSON 1

/*
  Most C compilers handle variable-sized arrays, so we enable
  that by default. Some (e.g. tcc) do not, so we provide a way
  to disable it: set FSLPRINTF_HAVE_VARARRAY to 0.

  One approach would be to look at:

  defined(__STDC_VERSION__) && (__STDC_VERSION__ >= 199901L)

  but some compilers support variable-sized arrays even when not
  explicitly running in c99 mode.
*/
/*
  2022-05-17: apparently VLAs were made OPTIONAL in C11 and MSVC
  decided not to support them. So we'll go ahead and remove the VLA
  usage altogether.
*/
#define FSLPRINTF_HAVE_VARARRAY 0
#if 0
#if !defined(FSLPRINTF_HAVE_VARARRAY)
#  if defined(__TINYC__)
#    define FSLPRINTF_HAVE_VARARRAY 0
#  else
#    if defined(__STDC_VERSION__) && (__STDC_VERSION__ >= 199901L)
#        define FSLPRINTF_HAVE_VARARRAY 1 /*use 1 in C99 mode */
#    else
#        define FSLPRINTF_HAVE_VARARRAY 0
#    endif
#  endif
#endif
#endif

/*
  Conversion types fall into various categories as defined by the
  following enumeration.
*/
enum PrintfCategory {etRADIX = 1, /* Integer types.  %d, %x, %o, and so forth */
                     etFLOAT = 2, /* Floating point.  %f */
                     etEXP = 3, /* Exponentional notation. %e and %E */
                     etGENERIC = 4, /* Floating or exponential, depending on exponent. %g */
                     /* 5 can be reused. Formerly etSIZE (%n) */
                     etSTRING = 6, /* Strings. %s */
                     etDYNSTRING = 7, /* Dynamically allocated strings. %z */
                     etPERCENT = 8, /* Percent symbol. %% */
                     etCHARX = 9, /* Characters. %c */
                     /* The rest are extensions, not normally found in printf() */
                     etCHARLIT = 10, /* Literal characters.  %' */
#if !FSLPRINTF_OMIT_SQL
                     etSQLESCAPE = 11, /* Strings with '\'' doubled.  %q */
                     etSQLESCAPE2 = 12, /* Strings with '\'' doubled and enclosed in '',
                                           NULL pointers replaced by SQL NULL.  %Q */
                     etSQLESCAPE3 = 14, /* %!Q -> identifiers wrapped in \"
                                            with inner  '\"' doubled */
                     etBLOBSQL = 13, /* %B -> Works like %Q,
                                        but requires a (fsl_buffer*) argument. */
#endif /* !FSLPRINTF_OMIT_SQL */
                     etPOINTER = 15, /* The %p conversion */
                     etORDINAL = 17, /* %r -> 1st, 2nd, 3rd, 4th, etc.  English only */
#if ! FSLPRINTF_OMIT_HTML
                     etHTML = 18, /* %h -> basic HTML escaping. */
                     etURLENCODE = 19, /* %t -> URL encoding. */
                     etURLDECODE = 20, /* %T -> URL decoding. */
#endif
                     etPATH = 21, /* %/ -> replace '\\' with '/' in path-like strings. */
                     etBLOB = 22, /* Works like %s, but requires a (fsl_buffer*) argument. */
                     etFOSSILIZE = 23, /* %F => like %s, but fossilizes it. */
                     etSTRINGID = 24, /* String with length limit for a UUID prefix: %S */
#if FSLPRINTF_ENABLE_JSON
                     etJSONSTR = 25,
#endif
                     etPLACEHOLDER = 100
                     };

/*
  An "etByte" is an 8-bit unsigned value.
*/
typedef unsigned char etByte;

/*
  Each builtin conversion character (ex: the 'd' in "%d") is described
  by an instance of the following structure
*/
typedef struct et_info {   /* Information about each format field */
  char fmttype;            /* The format field code letter */
  etByte base;             /* The base for radix conversion */
  etByte flags;            /* One or more of FLAG_ constants below */
  etByte type;             /* Conversion paradigm */
  etByte charset;          /* Offset into aDigits[] of the digits string */
  etByte prefix;           /* Offset into aPrefix[] of the prefix string */
} et_info;

/*
  Allowed values for et_info.flags
*/
enum et_info_flags { FLAG_SIGNED = 1,    /* True if the value to convert is signed */
                     FLAG_EXTENDED = 2,  /* True if for internal/extended use only. */
                     FLAG_STRING = 4     /* Allow infinity precision */
};

/*
  Historically, the following table was searched linearly, so the most
  common conversions were kept at the front.

  Change 2008 Oct 31 by Stephan Beal: we reserve an array of ordered
  entries for all chars in the range [32..126]. Format character
  checks can now be done in constant time by addressing that array
  directly.  This takes more static memory, but reduces the time and
  per-call overhead costs of fsl_appendfv().
*/
static const char aDigits[] = "0123456789ABCDEF0123456789abcdef";
static const char aPrefix[] = "-x0\000X0";
static const et_info fmtinfo[] = {
/*
  These entries MUST stay in ASCII order, sorted
  on their fmttype member! They MUST start with
  fmttype==32 and end at fmttype==126.
*/
{' '/*32*/, 0, 0, 0, 0, 0 },
{'!'/*33*/, 0, 0, 0, 0, 0 },
{'"'/*34*/, 0, 0, 0, 0, 0 },
{'#'/*35*/, 0, 0, 0, 0, 0 },
{'$'/*36*/, 0, 0, 0, 0, 0 },
{'%'/*37*/, 0, 0, etPERCENT, 0, 0 },
{'&'/*38*/, 0, 0, 0, 0, 0 },
{'\''/*39*/, 0, 0, 0, 0, 0 },
{'('/*40*/, 0, 0, 0, 0, 0 },
{')'/*41*/, 0, 0, 0, 0, 0 },
{'*'/*42*/, 0, 0, 0, 0, 0 },
{'+'/*43*/, 0, 0, 0, 0, 0 },
{','/*44*/, 0, 0, 0, 0, 0 },
{'-'/*45*/, 0, 0, 0, 0, 0 },
{'.'/*46*/, 0, 0, 0, 0, 0 },
{'/'/*47*/, 0, 0, etPATH, 0, 0 },
{'0'/*48*/, 0, 0, 0, 0, 0 },
{'1'/*49*/, 0, 0, 0, 0, 0 },
{'2'/*50*/, 0, 0, 0, 0, 0 },
{'3'/*51*/, 0, 0, 0, 0, 0 },
{'4'/*52*/, 0, 0, 0, 0, 0 },
{'5'/*53*/, 0, 0, 0, 0, 0 },
{'6'/*54*/, 0, 0, 0, 0, 0 },
{'7'/*55*/, 0, 0, 0, 0, 0 },
{'8'/*56*/, 0, 0, 0, 0, 0 },
{'9'/*57*/, 0, 0, 0, 0, 0 },
{':'/*58*/, 0, 0, 0, 0, 0 },
{';'/*59*/, 0, 0, 0, 0, 0 },
{'<'/*60*/, 0, 0, 0, 0, 0 },
{'='/*61*/, 0, 0, 0, 0, 0 },
{'>'/*62*/, 0, 0, 0, 0, 0 },
{'?'/*63*/, 0, 0, 0, 0, 0 },
{'@'/*64*/, 0, 0, 0, 0, 0 },
{'A'/*65*/, 0, 0, 0, 0, 0 },
#if FSLPRINTF_OMIT_SQL
{'B'/*66*/, 0, 0, 0, 0, 0 },
#else
{'B'/*66*/, 0, 2, etBLOBSQL, 0, 0 },
#endif
{'C'/*67*/, 0, 0, 0, 0, 0 },
{'D'/*68*/, 0, 0, 0, 0, 0 },
{'E'/*69*/, 0, FLAG_SIGNED, etEXP, 14, 0 },
{'F'/*70*/, 0, 4, etFOSSILIZE, 0, 0 },
{'G'/*71*/, 0, FLAG_SIGNED, etGENERIC, 14, 0 },
{'H'/*72*/, 0, 0, 0, 0, 0 },
{'I'/*73*/, 0, 0, 0, 0, 0 },
{'J'/*74*/, 0, 0, 0, 0, 0 },
{'K'/*75*/, 0, 0, 0, 0, 0 },
{'L'/*76*/, 0, 0, 0, 0, 0 },
{'M'/*77*/, 0, 0, 0, 0, 0 },
{'N'/*78*/, 0, 0, 0, 0, 0 },
{'O'/*79*/, 0, 0, 0, 0, 0 },
{'P'/*80*/, 0, 0, 0, 0, 0 },
#if FSLPRINTF_OMIT_SQL
{'Q'/*81*/, 0, 0, 0, 0, 0 },
#else
{'Q'/*81*/, 0, FLAG_STRING, etSQLESCAPE2, 0, 0 },
#endif
{'R'/*82*/, 0, 0, 0, 0, 0 },
{'S'/*83*/, 0, FLAG_STRING, etSTRINGID, 0, 0 },
{'T'/*84*/, 0, FLAG_STRING, etURLDECODE, 0, 0 },
{'U'/*85*/, 0, 0, 0, 0, 0 },
{'V'/*86*/, 0, 0, 0, 0, 0 },
{'W'/*87*/, 0, 0, 0, 0, 0 },
{'X'/*88*/, 16, 0, etRADIX,      0,  4 },
{'Y'/*89*/, 0, 0, 0, 0, 0 },
{'Z'/*90*/, 0, 0, 0, 0, 0 },
{'['/*91*/, 0, 0, 0, 0, 0 },
{'\\'/*92*/, 0, 0, 0, 0, 0 },
{']'/*93*/, 0, 0, 0, 0, 0 },
{'^'/*94*/, 0, 0, 0, 0, 0 },
{'_'/*95*/, 0, 0, 0, 0, 0 },
{'`'/*96*/, 0, 0, 0, 0, 0 },
{'a'/*97*/, 0, 0, 0, 0, 0 },
{'b'/*98*/, 0, 2, etBLOB, 0, 0 },
{'c'/*99*/, 0, 0, etCHARX,      0,  0 },
{'d'/*100*/, 10, FLAG_SIGNED, etRADIX,      0,  0 },
{'e'/*101*/, 0, FLAG_SIGNED, etEXP,        30, 0 },
{'f'/*102*/, 0, FLAG_SIGNED, etFLOAT,      0,  0},
{'g'/*103*/, 0, FLAG_SIGNED, etGENERIC,    30, 0 },
{'h'/*104*/, 0, FLAG_STRING, etHTML, 0, 0 },
{'i'/*105*/, 10, FLAG_SIGNED, etRADIX,      0,  0},
#if FSLPRINTF_ENABLE_JSON
{'j'/*106*/, 0, 0, etJSONSTR, 0, 0 },
#else
{'j'/*106*/, 0, 0, 0, 0, 0 },
#endif
{'k'/*107*/, 0, 0, 0, 0, 0 },
{'l'/*108*/, 0, 0, 0, 0, 0 },
{'m'/*109*/, 0, 0, 0, 0, 0 },
{'n'/*110*/, 0, 0, 0, 0, 0 },
{'o'/*111*/, 8, 0, etRADIX,      0,  2 },
{'p'/*112*/, 16, 0, etPOINTER, 0, 1 },
#if FSLPRINTF_OMIT_SQL
{'q'/*113*/, 0, 0, 0, 0, 0 },
#else
{'q'/*113*/, 0, FLAG_STRING, etSQLESCAPE,  0, 0 },
#endif
{'r'/*114*/, 10, (FLAG_EXTENDED|FLAG_SIGNED), etORDINAL,    0,  0},
{'s'/*115*/, 0, FLAG_STRING, etSTRING,     0,  0 },
{'t'/*116*/,  0, FLAG_STRING, etURLENCODE, 0, 0 },
{'u'/*117*/, 10, 0, etRADIX,      0,  0 },
{'v'/*118*/, 0, 0, 0, 0, 0 },
#if 1 || FSLPRINTF_OMIT_SQL
{'w'/*119*/, 0, 0, 0, 0, 0 },
#else
/* This role is filled by %!Q. %w is not currently used/documented. */
{'w'/*119*/, 0, FLAG_STRING, etSQLESCAPE3, 0, 0 },
#endif
{'x'/*120*/, 16, 0, etRADIX,      16, 1  },
{'y'/*121*/, 0, 0, 0, 0, 0 },
{'z'/*122*/, 0, FLAG_STRING, etDYNSTRING,  0,  0},
{'{'/*123*/, 0, 0, 0, 0, 0 },
{'|'/*124*/, 0, 0, 0, 0, 0 },
{'}'/*125*/, 0, 0, 0, 0, 0 },
{'~'/*126*/, 0, 0, 0, 0, 0 }
};
#define etNINFO  (sizeof(fmtinfo)/sizeof(fmtinfo[0]))

#if ! FSLPRINTF_OMIT_FLOATING_POINT
/*
  "*val" is a double such that 0.1 <= *val < 10.0
  Return the ascii code for the leading digit of *val, then
  multiply "*val" by 10.0 to renormalize.
    
  Example:
  input:     *val = 3.14159
  output:    *val = 1.4159    function return = '3'
    
  The counter *cnt is incremented each time.  After counter exceeds
  16 (the number of significant digits in a 64-bit float) '0' is
  always returned.
*/
static int et_getdigit(LONGDOUBLE_TYPE *val, int *cnt){
  int digit;
  LONGDOUBLE_TYPE d;
  if( (*cnt)++ >= 16 ) return '0';
  digit = (int)*val;
  d = digit;
  digit += '0';
  *val = (*val - d)*10.0;
  return digit;
}
#endif /* !FSLPRINTF_OMIT_FLOATING_POINT */

/*
  On machines with a small(?) stack size, you can redefine the
  FSLPRINTF_BUF_SIZE to be less than 350.  But beware - for smaller
  values some %f conversions may go into an infinite loop.
*/
#ifndef FSLPRINTF_BUF_SIZE
#  define FSLPRINTF_BUF_SIZE 350  /* Size of the output buffer for numeric conversions */
#endif

#if defined(FSL_INT_T_PFMT)
/* int64_t is already defined. */
#else
#if ! defined(__STDC__) && !defined(__TINYC__)
#ifdef FSLPRINTF_INT64_TYPE
typedef FSLPRINTF_INT64_TYPE int64_t;
typedef unsigned FSLPRINTF_INT64_TYPE uint64_t;
#elif defined(_MSC_VER) || defined(__BORLANDC__)
typedef __int64 int64_t;
typedef unsigned __int64 uint64_t;
#else
typedef long long int int64_t;
typedef unsigned long long int uint64_t;
#endif
#endif
#endif
/* Set up of int64 type */

#if 0
/   Not yet used. */
enum PrintfArgTypes {
TypeInt = 0,
TypeIntP = 1,
TypeFloat = 2,
TypeFloatP = 3,
TypeCString = 4
};
#endif


#if 0
/   Not yet used. */
typedef struct fsl_appendf_spec_handler_def
{
  char letter; /   e.g. %s */
  int xtype; /* reference to the etXXXX values, or fmtinfo[*].type. */
  int ntype; /* reference to PrintfArgTypes enum. */
} spec_handler;
#endif

/**
   fsl_appendf_spec_handler is an almost-generic interface for farming
   work out of fsl_appendfv()'s code into external functions.  It doesn't
   actually save much (if any) overall code, but it makes the fsl_appendfv()
   code more manageable.


   REQUIREMENTS of implementations:

   - Expects an implementation-specific vargp pointer.
   fsl_appendfv() passes a pointer to the converted value of
   an entry from the format va_list. If it passes a type
   other than the expected one, undefined results.

   - If it calls pf it must do: pf( pfArg, D, N ), where D is
   the data to export and N is the number of bytes to export.
   It may call pf() an arbitrary number of times

   - If pf() successfully is called, the return value must be the
   accumulated totals of its return value(s), plus (possibly, but
   unlikely) an implementation-specific amount.

   - If it does not call pf() then it must return 0 (success)
   or a negative number (an error) or do all of the export
   processing itself and return the number of bytes exported.

   SIGNIFICANT LIMITATIONS:

   - Has no way of iterating over the format string, so handling
   precisions and such here can't work too well. (Nevermind:
   precision/justification is handled in fsl_appendfv().)
*/
typedef int (*fsl_appendf_spec_handler)( fsl_output_f pf,
                                               void * pfArg,
                                               unsigned int pfLen,
                                               void * vargp );


/**
   fsl_appendf_spec_handler for etSTRING types. It assumes that varg
   is a NUL-terminated (char [const] *)
*/
static int spech_string( fsl_output_f pf, void * pfArg,
                         unsigned int pfLen, void * varg ){
  char const * ch = (char const *) varg;
  return ch ? pf( pfArg, ch, pfLen ) : 0;
}

/**
   fsl_appendf_spec_handler for etDYNSTRING types.  It assumes that
   varg is a non-const (char *). It behaves identically to
   spec_string() and then calls fsl_free() on that (char *).
*/
static int spech_dynstring( fsl_output_f pf, void * pfArg,
                            unsigned int pfLen, void * varg ){
  int const rc = spech_string( pf, pfArg, pfLen, varg );
  fsl_free( varg );
  return rc;
}

#if !FSLPRINTF_OMIT_HTML
static int spech_string_to_html( fsl_output_f pf, void * pfArg,
                                 unsigned int pfLen, void * varg ){
  char const * ch = (char const *) varg;
  unsigned int i;
  int rc = 0;
  if( ! ch ) return 0;
  rc = 0;
  for( i = 0; 0==rc && (i<pfLen) && *ch; ++ch, ++i )
  {
    switch( *ch )
    {
      case '<': rc = pf( pfArg, "&lt;", 4 );
        break;
      case '&': rc = pf( pfArg, "&amp;", 5 );
        break;
      default:
        rc = pf( pfArg, ch, 1 );
        break;
    };
  }
  return rc;
}

static int httpurl_needs_escape( int c ){
  /*
    Definition of "safe" and "unsafe" chars
    was taken from:

    https://www.codeguru.com/cpp/cpp/cpp_mfc/article.php/c4029/
  */
  return ( (c >= 32 && c <=47)
           || ( c>=58 && c<=64)
           || ( c>=91 && c<=96)
           || ( c>=123 && c<=126)
           || ( c<32 || c>=127)
           );
}

/**
   The handler for the etURLENCODE specifier.

   It expects varg to be a string value, which it will preceed to
   encode using an URL encoding algothrim (certain characters are
   converted to %XX, where XX is their hex value) and passes the
   encoded string to pf(). It returns the total length of the output
   string.
*/
static int spech_urlencode( fsl_output_f pf, void * pfArg,
                            unsigned int pfLen __unused, void * varg ){
  char const * str = (char const *) varg;
  int rc = 0;
  char ch = 0;
  char const * hex = "0123456789ABCDEF";
#define xbufsz 10
  char xbuf[xbufsz];
  int slen = 0;
  if( ! str ) return 0;
  memset( xbuf, 0, xbufsz );
  ch = *str;
#define xbufsz 10
  slen = 0;
  for( ; 0==rc && ch; ch = *(++str) ){
    if( ! httpurl_needs_escape( ch ) ){
      rc = pf( pfArg, str, 1 );
      continue;
    }else{
      xbuf[0] = '%';
      xbuf[1] = hex[((ch>>4)&0xf)];
      xbuf[2] = hex[(ch&0xf)];
      xbuf[3] = 0;
      slen = 3;
      rc = pf( pfArg, xbuf, slen );
    }
  }
#undef xbufsz
  return rc;
}

/* 
   hexchar_to_int():

   For 'a'-'f', 'A'-'F' and '0'-'9', returns the appropriate decimal
   number.  For any other character it returns -1.
*/
static int hexchar_to_int( int ch ){
  if( (ch>='0' && ch<='9') ) return ch-'0';
  else if( (ch>='a' && ch<='f') ) return ch-'a'+10;
  else if( (ch>='A' && ch<='F') ) return ch-'A'+10;
  else return -1;
}

/**
   The handler for the etURLDECODE specifier.

   It expects varg to be a ([const] char *), possibly encoded
   with URL encoding. It decodes the string using a URL decode
   algorithm and passes the decoded string to
   pf(). It returns the total length of the output string.
   If the input string contains malformed %XX codes then this
   function will return prematurely.
*/
static int spech_urldecode( fsl_output_f pf, void * pfArg,
                            unsigned int pfLen, void * varg ){
  char const * str = (char const *) varg;
  int rc = 0;
  char ch = 0;
  char ch2 = 0;
  char xbuf[4];
  int decoded;
  char const * end = str + pfLen;
  if( !str || !pfLen ) return 0;
  ch = *str;
  while(0==rc && ch && str<end){
    if( ch == '%' ){
      if(str+2>=end) goto outro/*invalid partial encoding - simply skip it*/;
      ch = *(++str);
      ch2 = *(++str);
      if( isxdigit((int)ch) &&
          isxdigit((int)ch2) )
      {
        decoded = (hexchar_to_int( ch ) * 16)
          + hexchar_to_int( ch2 );
        xbuf[0] = (char)decoded;
        xbuf[1] = 0;
        rc = pf( pfArg, xbuf, 1 );
        ch = *(++str);
        continue;
      }else{
        xbuf[0] = '%';
        xbuf[1] = ch;
        xbuf[2] = ch2;
        xbuf[3] = 0;
        rc = pf( pfArg, xbuf, 3 );
        ch = *(++str);
        continue;
      }
    }else if( ch == '+' ){
      xbuf[0] = ' ';
      xbuf[1] = 0;
      rc = pf( pfArg, xbuf, 1 );
      ch = *(++str);
      continue;
    }
    outro:
    xbuf[0] = ch;
    xbuf[1] = 0;
    rc = pf( pfArg, xbuf, 1 );
    ch = *(++str);
  }
  return rc;
}

#endif /* !FSLPRINTF_OMIT_HTML */


#if !FSLPRINTF_OMIT_SQL
/**
   Quotes the (char *) varg as an SQL string 'should' be quoted. The
   exact type of the conversion is specified by xtype, which must be
   one of etSQLESCAPE, etSQLESCAPE2, etSQLESCAPE3.

   Search this file for those constants to find the associated
   documentation.
*/
static int spech_sqlstring( int xtype, fsl_output_f pf,
                            void * pfArg, unsigned int pfLen,
                            void * varg ){
  enum { BufLen = 512 };
  char buf[BufLen];
  unsigned int i = 0, j = 0;
  int ch;
  char const q = xtype==etSQLESCAPE3 ?'"':'\''; /* Quote character */
  char const * escarg = (char const *) varg;
  bool const isnull = escarg==0;
  bool const needQuote =
    !isnull && (xtype==etSQLESCAPE2
                || xtype==etBLOBSQL
                || xtype==etSQLESCAPE3);
  if( isnull ){
    if(xtype==etSQLESCAPE2||xtype==etSQLESCAPE3){
      escarg = "NULL";
      pfLen = 4;
    }else{
      escarg = "(NULL)";;
      pfLen = 6;
    }
  }
  if( needQuote ) buf[j++] = q;
  for(i=0; (ch=escarg[i])!=0 && i<pfLen; ++i){
    buf[j++] = ch;
    if( ch==q ) buf[j++] = ch;
    if(j+2>=BufLen){
      int const rc = pf( pfArg, &buf[0], j );
      if(rc) return rc;
      j = 0;
    }
  }
  if( needQuote ) buf[j++] = q;
  buf[j] = 0;
  return j>0 ? pf( pfArg, &buf[0], j ) : 0;
}

#endif /* !FSLPRINTF_OMIT_SQL */

#if FSLPRINTF_ENABLE_JSON
/* TODO? Move these UTF8 bits into the public API? */
/*
** This lookup table is used to help decode the first byte of
** a multi-byte UTF8 character.
**
** Taken from sqlite3:
** https://www.sqlite.org/src/artifact?ln=48-61&name=810fbfebe12359f1
*/
static const unsigned char fsl_utfTrans1[] = {
  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
  0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
  0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
  0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
  0x00, 0x01, 0x02, 0x03, 0x00, 0x01, 0x00, 0x00
};
unsigned int fsl_utf8_read_char(
  const unsigned char *zIn,       /* First byte of UTF-8 character */
  const unsigned char *zTerm,     /* Pretend this byte is 0x00 */
  const unsigned char **pzNext    /* Write first byte past UTF-8 char here */
){
  /*
    Adapted from sqlite3:
    https://www.sqlite.org/src/artifact?ln=155-165&name=810fbfebe12359f1
  */
  unsigned c;
  if(zIn>=zTerm){
    *pzNext = zTerm;
    c = 0;
  }else{
    c = (unsigned int)*(zIn++);
    if( c>=0xc0 ){
      c = fsl_utfTrans1[c-0xc0];
      while( zIn!=zTerm && (*zIn & 0xc0)==0x80 )
        c = (c<<6) + (0x3f & *(zIn++));
      if( c<0x80
          || (c&0xFFFFF800)==0xD800
          || (c&0xFFFFFFFE)==0xFFFE ) c = 0xFFFD;
    }
    *pzNext = zIn;
  }
  return c;
}

static int fsl_utf8_char_to_cstr(unsigned int c, unsigned char *output, unsigned char length){
    /* Stolen from the internet, adapted from several variations which
      all _seem_ to have derived from librdf. */
    unsigned char size=0;

    /* check for illegal code positions:
     * U+D800 to U+DFFF (UTF-16 surrogates)
     * U+FFFE and U+FFFF
     */
    if((c > 0xD7FF && c < 0xE000)
       || c == 0xFFFE || c == 0xFFFF) return -1;

    /* Unicode 3.2 only defines U+0000 to U+10FFFF and UTF-8 encodings of it */
    if(c > 0x10ffff) return -1;
    
    if (c < 0x00000080) size = 1;
    else if (c < 0x00000800) size = 2;
    else if (c < 0x00010000) size = 3;
    else size = 4;
    if(!output) return (int)size;
    else if(size > length) return -1;
    else switch(size) {
      case 0:
          assert(!"can't happen anymore");
          output[0] = 0;
          return 0;
      case 4:
          output[3] = 0x80 | (c & 0x3F);
          c = c >> 6;
          c |= 0x10000;
          /* Fall through */
      case 3:
          output[2] = 0x80 | (c & 0x3F);
          c = c >> 6;
          c |= 0x800;
          /* Fall through */
      case 2:
          output[1] = 0x80 | (c & 0x3F);
          c = c >> 6;
          c |= 0xc0; 
          /* Fall through */
      case 1:
        output[0] = (unsigned char)c;
          /* Fall through */
      default:
        return (int)size;
    }
}

struct SpechJson {
  char const * z;
  bool addQuotes;
  bool escapeSmallUtf8;
};

/**
   fsl_appendf_spec_handler for etJSONSTR. It assumes that varg is a
   SpechJson struct instance.
*/
static int spech_json( fsl_output_f pf, void * pfArg,
                       unsigned int pfLen, void * varg ){
  struct SpechJson const * state = (struct SpechJson *)varg;
  int pfRc = 0;
  const unsigned char *z = (const unsigned char *)state->z;
  const unsigned char *zEnd = z + pfLen;
  const unsigned char * zNext = 0;
  unsigned int c;
  unsigned char c1;

#define out(X,N) pfRc=pf(pfArg, (char const *)(X), N); \
  if(0!=pfRc) return pfRc
#define outc c1 = (unsigned char)c; out(&c1,1)
  if(!z){
    out("null",4);
    return pfRc;
  }    
  if(state->addQuotes){
    out("\"", 1);
  }
  for( ; 0==pfRc && (z < zEnd)
         && (c=fsl_utf8_read_char(z, zEnd, &zNext));
       z = zNext ){
    if( c=='\\' || c=='"' ){
      out("\\", 1);
      outc;
    }else if( c<' ' ){
      out("\\",1);
      switch(c){
        case '\b': out("b",1); break;
        case '\f': out("f",1); break;
        case '\n': out("n",1); break;
        case '\t': out("t",1); break;
        case '\r': out("r",1); break;
        default:{
          unsigned char ubuf[5] = {'u',0,0,0,0};
          int i;
          for(i = 4; i>0; --i){
            ubuf[i] = "0123456789abcdef"[c&0xf];
            c >>= 4;
          }
          out(ubuf,5);
          break;
        }
      }
    }else if(c<128){
      outc;
    }/* At this point we know that c is part of a multi-byte
        character. We're assuming legal UTF8 input, which means
        emitting a surrogate pair if the value is > 0xffff. */
    else if(c<0xFFFF){
      unsigned char ubuf[6];
      if(state->escapeSmallUtf8){
        /* Output char in \u#### form. */
        fsl_snprintf((char *)ubuf, 6, "\\u%04x", c);
        out(ubuf, 6);
      }else{
        /* Output character literal. */
        int const n = fsl_utf8_char_to_cstr(c, ubuf, 4);
        if(n<0){
          out("?",1);
        }else{
          assert(n>0);
          out(ubuf, n);
        }
      }
    }else{
      /* Surrogate pair. */
      unsigned char ubuf[12];
      c -= 0x10000;
      fsl_snprintf((char *)ubuf, 12, "\\u%04x\\u%04x",
                   (0xd800 | (c>>10)),
                   (0xdc00 | (c & 0x3ff)));
      out(ubuf, 12);
    }
  }
  if(state->addQuotes){
    out("\"",1);
  }
  return pfRc;
#undef out
#undef outc
}
#endif /* FSLPRINTF_ENABLE_JSON */

/*
   Find the length of a string as long as that length does not
   exceed N bytes.  If no zero terminator is seen in the first
   N bytes then return N.  If N is negative, then this routine
   is an alias for strlen().
*/
static int StrNLen32(const char *z, int N){
  int n = 0;
  while( (N-- != 0) && *(z++)!=0 ){ n++; }
  return n;
}

#if 0
/**
   Given the first byte of an assumed-to-be well-formed UTF8
   character, returns the length of that character. Returns 0 if the
   character appears to be an invalid UTF8 character, else returns its
   length, in bytes (1-4). Note that a NUL byte is a valid length-1
   character.
*/
static int utf8__char_length( unsigned char const * const c ){
  switch(0xF0 & *c) {
    case 0xF0: return (c[1]&0x80 && c[2]&0x80 && c[3]&0x80) ? 4 : 0;
    case 0xE0: return (c[1]&0x80 && c[2]&0x80) ? 3 : 0;
    case 0xC0: return (c[1]&0x80) ? 2 : 0;
    case 0x80: return 0;
    default: return 1;
      /* See also: https://stackoverflow.com/questions/4884656/utf-8-encoding-size */
  }
}
#endif

/**
   Internal helper for %#W.Ps format.
*/
static void appendf__utf8_altform(char const * z, int * pLength,
                                  int * pPrecision, int * pWidth){
  /* Treat %#W.Ps as a width/precision limit of W resp. P UTF8
     characters instead of bytes. */
  int pC = 0/*precision, chars*/, pB = 0/*precision, bytes*/,
    wC = 0/*width, chars*/, wB = 0/*width, bytes*/;
  char const * const zEnd = z + *pLength;
  int lc;
  while( z < zEnd ){
    switch(0xF0 & *z) {
      case 0xF0: lc = (z[1]&0x80 && z[2]&0x80 && z[3]&0x80) ? 4 : 0; break;
      case 0xE0: lc = (z[1]&0x80 && z[2]&0x80) ? 3 : 0; break;
      case 0xC0: lc = (z[1]&0x80) ? 2 : 0; break;
      case 0x80: lc = 0; break;
      default: lc = 1; break;
    }
    if(!lc) break;
    else if(wC<*pWidth && (*pPrecision<=0 || pC<*pPrecision)){ ++wC; wB+=lc;}
    if(pC<*pPrecision){ ++pC; pB+=lc;}
    z+=lc;
  }
  if(*pPrecision>0) *pLength = pB;
  if(*pWidth>0) *pWidth = *pWidth - wC + wB;
}

/*
  The root printf program.  All variations call this core.  It
  implements most of the common printf behaviours plus (optionally)
  some extended ones.

  INPUTS:

  pfAppend : The is a fsl_output_f function which is responsible for
  accumulating the output. If pfAppend returns non-0 then processing
  stops immediately.

  pfAppendArg : is ignored by this function but passed as the first
  argument to pfAppend. pfAppend will presumably use it as a data
  store for accumulating its string.

  fmt : This is the format string, as in the usual printf().

  ap : This is a pointer to a list of arguments.  Same as in
  vprintf() and friends.

  OUTPUTS:

  The return value is 0 on success, non-0 on error. Historically it
  returned the total number bytes reported appended by pfAppend, but
  those semantics (A) are only very, very rarely useful and (B) they
  make sensibly reporting errors via the generic callback interface
  next to impossible. e.g. the callback may encounter I/O or allocation
  errors.

  Much of this code dates back to the early 1980's, supposedly.

  Known change history (most historic info has been lost):

  10 Feb 2008 by Stephan Beal: refactored to remove the 'useExtended'
  flag (which is now always on). Added the fsl_output_f typedef to
  make this function generic enough to drop into other source trees
  without much work.

  31 Oct 2008 by Stephan Beal: refactored the et_info lookup to be
  constant-time instead of linear.
*/
int fsl_appendfv(fsl_output_f pfAppend, /* Accumulate results here */
                 void * pfAppendArg,     /* Passed as first arg to pfAppend. */
                 const char *fmt,        /* Format string */
                 va_list ap              /* arguments */
                 ){
  /**
     HISTORIC NOTE (author and year unknown):

     Note that the order in which automatic variables are declared below
     seems to make a big difference in determining how fast this beast
     will run.
  */
  int pfrc = 0;              /* result from calling pfAppend */
  int c;                     /* Next character in the format string */
  char *bufpt = 0;           /* Pointer to the conversion buffer */
  int precision = 0;         /* Precision of the current field */
  int length;                /* Length of the field */
  int idx;                   /* A general purpose loop counter */
  int width;                 /* Width of the current field */
  etByte flag_leftjustify;   /* True if "-" flag is present */
  etByte flag_plussign;      /* True if "+" flag is present */
  etByte flag_blanksign;     /* True if " " flag is present */
  etByte flag_alternateform; /* True if "#" flag is present */
  etByte flag_altform2;      /* True if "!" flag is present */
  etByte flag_zeropad;       /* True if field width constant starts with zero */
  etByte flag_long;          /* True if "l" flag is present */
  etByte flag_longlong;      /* True if the "ll" flag is present */
  etByte done;               /* Loop termination flag */
  etByte cThousand           /* Thousands separator for %d and %u */
    /* ported in from https://fossil-scm.org/home/info/2cdbdbb1c9b7ad2b */;
  uint64_t longvalue;        /* Value for integer types */
  LONGDOUBLE_TYPE realvalue; /* Value for real types */
  const et_info *infop = 0;      /* Pointer to the appropriate info structure */
  char buf[FSLPRINTF_BUF_SIZE];       /* Conversion buffer */
  char prefix;               /* Prefix character.  "+" or "-" or " " or '\0'. */
  etByte xtype = 0;              /* Conversion paradigm */
  char * zExtra = 0;              /* Extra memory used for etTCLESCAPE conversions */
#if ! FSLPRINTF_OMIT_FLOATING_POINT
  int  exp, e2;              /* exponent of real numbers */
  double rounder;            /* Used for rounding floating point values */
  etByte flag_dp;            /* True if decimal point should be shown */
  etByte flag_rtz;           /* True if trailing zeros should be removed */
  etByte flag_exp;           /* True to force display of the exponent */
  int nsd;                   /* Number of significant digits returned */
#endif


/**
   FSLPRINTF_CHARARRAY is a helper to allocate variable-sized arrays.
   This exists mainly so this code can compile with the tcc compiler.
*/
#if FSLPRINTF_HAVE_VARARRAY
#  define FSLPRINTF_CHARARRAY(V,N) char V[N+1]; memset(V,0,N+1)
#  define FSLPRINTF_CHARARRAY_FREE(V)
#else
#  define FSLPRINTF_CHARARRAY_STACK(V) 
#  define FSLPRINTF_CHARARRAY(V,N) char V##2[256]; \
  char * V;                                                      \
  if((int)(N)<((int)sizeof(V##2))){                              \
    V = V##2;                                   \
  }else{                                        \
    V = (char *)fsl_malloc(N+1);       \
    if(!V) {FSLPRINTF_RETURN(FSL_RC_OOM);} \
  }
#  define FSLPRINTF_CHARARRAY_FREE(V) if(V!=V##2) fsl_free(V)
#endif

  /* FSLPRINTF_RETURN, FSLPRINTF_CHECKERR, and FSLPRINTF_SPACES
     are internal helpers.
  */
#define FSLPRINTF_RETURN(RC) if( zExtra ) fsl_free(zExtra); return RC
#define FSLPRINTF_CHECKERR if( 0!=pfrc ) { FSLPRINTF_RETURN(pfrc); } (void)0
#define FSLPRINTF_SPACES(N)                     \
  {                                             \
    FSLPRINTF_CHARARRAY(zSpaces,N);             \
    memset( zSpaces,' ',N);                     \
    pfrc = pfAppend(pfAppendArg, zSpaces, N);   \
    FSLPRINTF_CHARARRAY_FREE(zSpaces);          \
    FSLPRINTF_CHECKERR;                \
  } (void)0

  length = 0;
  bufpt = 0;
  for(; (c=(*fmt))!=0; ++fmt){
    assert(0==pfrc);
    if( c!='%' ){
      int amt;
      bufpt = (char *)fmt;
      amt = 1;
      while( (c=(*++fmt))!='%' && c!=0 ) amt++;
      pfrc = pfAppend( pfAppendArg, bufpt, amt);
      FSLPRINTF_CHECKERR;
      if( c==0 ) break;
    }
    if( (c=(*++fmt))==0 ){
      pfrc = pfAppend( pfAppendArg, "%", 1);
      FSLPRINTF_CHECKERR;
      break;
    }
    /* Find out what flags are present */
    flag_leftjustify = flag_plussign = flag_blanksign = cThousand =
      flag_alternateform = flag_altform2 = flag_zeropad = 0;
    done = 0;
    do{
      switch( c ){
        case '-':   flag_leftjustify = 1;     break;
        case '+':   flag_plussign = 1;        break;
        case ' ':   flag_blanksign = 1;       break;
        case '#':   flag_alternateform = 1;   break;
        case '!':   flag_altform2 = 1;        break;
        case '0':   flag_zeropad = 1;         break;
        case ',':   cThousand = ',';          break;
        default:    done = 1;                 break;
      }
    }while( !done && (c=(*++fmt))!=0 );
    /* Get the field width */
    width = 0;
    if( c=='*' ){
      width = va_arg(ap,int);
      if( width<0 ){
        flag_leftjustify = 1;
        width = width >= -2147483647 ? -width : 0;
      }
      c = *++fmt;
    }else{
      unsigned wx = 0;
      while( c>='0' && c<='9' ){
        wx = wx * 10 + c - '0';
        width = width*10 + c - '0';
        c = *++fmt;
      }
      width = wx & 0x7fffffff;
    }
    if( width > FSLPRINTF_BUF_SIZE-10 ){
      width = FSLPRINTF_BUF_SIZE-10;
    }
    /* Get the precision */
    if( c=='.' ){
      precision = 0;
      c = *++fmt;
      if( c=='*' ){
        precision = va_arg(ap,int);
        c = *++fmt;
        if( precision<0 ){
          precision = precision >= -2147483647 ? -precision : -1;
        }
      }else{
        unsigned px = 0;
        while( c>='0' && c<='9' ){
          px = px*10 + c - '0';
          c = *++fmt;
        }
        precision = px & 0x7fffffff;
      }
    }else{
      precision = -1;
    }
    /* Get the conversion type modifier */
    if( c=='l' ){
      flag_long = 1;
      c = *++fmt;
      if( c=='l' ){
        flag_longlong = 1;
        c = *++fmt;
      }else{
        flag_longlong = 0;
      }
    }else{
      flag_long = flag_longlong = 0;
    }
    /* Fetch the info entry for the field */
    infop = 0;
#define FMTNDX(N) (N - fmtinfo[0].fmttype)
#define FMTINFO(N) (fmtinfo[ FMTNDX(N) ])
    infop = ((c>=(fmtinfo[0].fmttype)) && (c<fmtinfo[etNINFO-1].fmttype))
      ? &FMTINFO(c)
      : 0;
    /*fprintf(stderr,"char '%c'/%d @ %d,  type=%c/%d\n",c,c,FMTNDX(c),infop->fmttype,infop->type);*/
    if( infop ) xtype = infop->type;
#undef FMTINFO
#undef FMTNDX
    zExtra = 0;
    if( (!infop) || (!infop->type) ){
      FSLPRINTF_RETURN(FSL_RC_RANGE);
    }

    /* Limit the precision to prevent overflowing buf[] during conversion */
    if( precision>FSLPRINTF_BUF_SIZE-40 && (infop->flags & FLAG_STRING)==0 ){
      precision = FSLPRINTF_BUF_SIZE-40;
    }

    /*
      At this point, variables are initialized as follows:
        
      flag_alternateform          TRUE if a '#' is present.
      flag_altform2               TRUE if a '!' is present.
      flag_plussign               TRUE if a '+' is present.
      flag_leftjustify            TRUE if a '-' is present or if the
                                  field width was negative.
      flag_zeropad                TRUE if the width began with 0.
      flag_long                   TRUE if the letter 'l' (ell) prefixed
                                  the conversion character.
      flag_longlong               TRUE if the letter 'll' (ell ell) prefixed
                                  the conversion character.
      flag_blanksign              TRUE if a ' ' is present.
      width                       The specified field width.  This is
                                  always non-negative.  Default is 0.
      precision                   The specified precision.  The default
                                  is -1.
      xtype                       The class of the conversion.
      infop                       Pointer to the appropriate info struct.
    */
    switch( xtype ){
      case etPOINTER:
        flag_longlong = sizeof(char*)==sizeof(int64_t);
        flag_long = sizeof(char*)==sizeof(long int);
        FSL_SWITCH_FALL_THROUGH;
      case etORDINAL:
      case etRADIX:
        if( infop->flags & FLAG_SIGNED ){
          int64_t v;
          if( flag_longlong )   v = va_arg(ap,int64_t);
          else if( flag_long )  v = va_arg(ap,long int);
          else                  v = va_arg(ap,int);
          if( v<0 ){
            longvalue = -v;
            prefix = '-';
          }else{
            longvalue = v;
            if( flag_plussign )        prefix = '+';
            else if( flag_blanksign )  prefix = ' ';
            else                       prefix = 0;
          }
        }else{
          if( flag_longlong )   longvalue = va_arg(ap,uint64_t);
          else if( flag_long )  longvalue = va_arg(ap,unsigned long int);
          else                  longvalue = va_arg(ap,unsigned int);
          prefix = 0;
        }
        if( longvalue==0 ) flag_alternateform = 0;
        if( flag_zeropad && precision<width-(prefix!=0) ){
          precision = width-(prefix!=0);
        }
        bufpt = &buf[FSLPRINTF_BUF_SIZE-1];
        if( xtype==etORDINAL ){
          /** i sure would like to shake the hand of whoever figured this out: */
          static const char zOrd[] = "thstndrd";
          int x = longvalue % 10;
          if( x>=4 || (longvalue/10)%10==1 ){
            x = 0;
          }
          buf[FSLPRINTF_BUF_SIZE-3] = zOrd[x*2];
          buf[FSLPRINTF_BUF_SIZE-2] = zOrd[x*2+1];
          bufpt -= 2;
        }
        {
          int const base = infop->base;
          const char *cset = &aDigits[infop->charset];
          do{                                           /* Convert to ascii */
            *(--bufpt) = cset[longvalue%base];
            longvalue = longvalue/base;
          }while( longvalue>0 );
        }
        length = &buf[FSLPRINTF_BUF_SIZE-1]-bufpt;
        while( precision>length ){
          *(--bufpt) = '0';                             /* Zero pad */
          ++length;
        }
        if( cThousand ){
          int nn = (length - 1)/3;  /* Number of "," to insert */
          int ix = (length - 1)%3 + 1;
          bufpt -= nn;
          for(idx=0; nn>0; idx++){
            bufpt[idx] = bufpt[idx+nn];
            ix--;
            if( ix==0 ){
              bufpt[++idx] = cThousand;
              nn--;
              ix = 3;
            }
          }
        }
        if( prefix ) *(--bufpt) = prefix;               /* Add sign */
        if( flag_alternateform && infop->prefix ){      /* Add "0" or "0x" */
          const char *pre;
          char x;
          pre = &aPrefix[infop->prefix];
          if( *bufpt!=pre[0] ){
            for(; (x=(*pre))!=0; pre++) *(--bufpt) = x;
          }
        }
        length = &buf[FSLPRINTF_BUF_SIZE-1]-bufpt;
        break;
      case etFLOAT:
      case etEXP:
      case etGENERIC:
        realvalue = va_arg(ap,double);
#if ! FSLPRINTF_OMIT_FLOATING_POINT
        if( precision<0 ) precision = 6;         /* Set default precision */
        if( precision>FSLPRINTF_BUF_SIZE/2-10 ) precision = FSLPRINTF_BUF_SIZE/2-10;
        if( realvalue<0.0 ){
          realvalue = -realvalue;
          prefix = '-';
        }else{
          if( flag_plussign )          prefix = '+';
          else if( flag_blanksign )    prefix = ' ';
          else                         prefix = 0;
        }
        if( xtype==etGENERIC && precision>0 ) precision--;
#if 0
        /* Rounding works like BSD when the constant 0.4999 is used.  Wierd! */
        for(idx=precision & 0xfff, rounder=0.4999; idx>0; idx--, rounder*=0.1);
#else
        /* It makes more sense to use 0.5 */
        for(idx=precision & 0xfff, rounder=0.5; idx>0; idx--, rounder*=0.1){}
#endif
        if( xtype==etFLOAT ) realvalue += rounder;
        /* Normalize realvalue to within 10.0 > realvalue >= 1.0 */
        exp = 0;
#if 1
        if( (realvalue)!=(realvalue) ){
          /* from sqlite3: #define sqlite3_isnan(X)  ((X)!=(X)) */
          /* This weird array thing is to avoid constness violations
             when assinging, e.g. "NaN" to bufpt.
          */
          static char NaN[4] = {'N','a','N','\0'};
          bufpt = NaN;
          length = 3;
          break;
        }
#endif
        if( realvalue>0.0 ){
          while( realvalue>=1e32 && exp<=350 ){ realvalue *= 1e-32; exp+=32; }
          while( realvalue>=1e8 && exp<=350 ){ realvalue *= 1e-8; exp+=8; }
          while( realvalue>=10.0 && exp<=350 ){ realvalue *= 0.1; exp++; }
          while( realvalue<1e-8 && exp>=-350 ){ realvalue *= 1e8; exp-=8; }
          while( realvalue<1.0 && exp>=-350 ){ realvalue *= 10.0; exp--; }
          if( exp>350 || exp<-350 ){
            if( prefix=='-' ){
              static char Inf[5] = {'-','I','n','f','\0'};
              bufpt = Inf;
            }else if( prefix=='+' ){
              static char Inf[5] = {'+','I','n','f','\0'};
              bufpt = Inf;
            }else{
              static char Inf[4] = {'I','n','f','\0'};
              bufpt = Inf;
            }
            length = (int)strlen(bufpt);
            break;
          }
        }
        bufpt = buf;
        /*
          If the field type is etGENERIC, then convert to either etEXP
          or etFLOAT, as appropriate.
        */
        flag_exp = xtype==etEXP;
        if( xtype!=etFLOAT ){
          realvalue += rounder;
          if( realvalue>=10.0 ){ realvalue *= 0.1; exp++; }
        }
        if( xtype==etGENERIC ){
          flag_rtz = !flag_alternateform;
          if( exp<-4 || exp>precision ){
            xtype = etEXP;
          }else{
            precision = precision - exp;
            xtype = etFLOAT;
          }
        }else{
          flag_rtz = 0;
        }
        if( xtype==etEXP ){
          e2 = 0;
        }else{
          e2 = exp;
        }
        nsd = 0;
        flag_dp = (precision>0) | flag_alternateform | flag_altform2;
        /* The sign in front of the number */
        if( prefix ){
          *(bufpt++) = prefix;
        }
        /* Digits prior to the decimal point */
        if( e2<0 ){
          *(bufpt++) = '0';
        }else{
          for(; e2>=0; e2--){
            *(bufpt++) = et_getdigit(&realvalue,&nsd);
          }
        }
        /* The decimal point */
        if( flag_dp ){
          *(bufpt++) = '.';
        }
        /* "0" digits after the decimal point but before the first
           significant digit of the number */
        for(e2++; e2<0 && precision>0; precision--, e2++){
          *(bufpt++) = '0';
        }
        /* Significant digits after the decimal point */
        while( (precision--)>0 ){
          *(bufpt++) = et_getdigit(&realvalue,&nsd);
        }
        /* Remove trailing zeros and the "." if no digits follow the "." */
        if( flag_rtz && flag_dp ){
          while( bufpt[-1]=='0' ) *(--bufpt) = 0;
          /* assert( bufpt>buf ); */
          if( bufpt[-1]=='.' ){
            if( flag_altform2 ){
              *(bufpt++) = '0';
            }else{
              *(--bufpt) = 0;
            }
          }
        }
        /* Add the "eNNN" suffix */
        if( flag_exp || (xtype==etEXP && exp) ){
          *(bufpt++) = aDigits[infop->charset];
          if( exp<0 ){
            *(bufpt++) = '-'; exp = -exp;
          }else{
            *(bufpt++) = '+';
          }
          if( exp>=100 ){
            *(bufpt++) = (exp/100)+'0';                /* 100's digit */
            exp %= 100;
          }
          *(bufpt++) = exp/10+'0';                     /* 10's digit */
          *(bufpt++) = exp%10+'0';                     /* 1's digit */
        }
        *bufpt = 0;

        /* The converted number is in buf[] and zero terminated. Output it.
           Note that the number is in the usual order, not reversed as with
           integer conversions. */
        length = bufpt-buf;
        bufpt = buf;

        /* Special case:  Add leading zeros if the flag_zeropad flag is
           set and we are not left justified */
        if( flag_zeropad && !flag_leftjustify && length < width){
          int i;
          int nPad = width - length;
          for(i=width; i>=nPad; i--){
            bufpt[i] = bufpt[i-nPad];
          }
          i = prefix!=0;
          while( nPad-- ) bufpt[i++] = '0';
          length = width;
        }
#endif /* !FSLPRINTF_OMIT_FLOATING_POINT */
        break;
      case etPERCENT:
        buf[0] = '%';
        bufpt = buf;
        length = 1;
        break;
      case etCHARLIT:
      case etCHARX:
        c = buf[0] = (xtype==etCHARX ? va_arg(ap,int) : *++fmt);
        if( precision>=0 ){
          for(idx=1; idx<precision; idx++) buf[idx] = c;
          length = precision;
        }else{
          length =1;
        }
        bufpt = buf;
        break;
      case etPATH: {
        /* Sanitize path-like inputs, replacing \\ with /. */
        int i;
        int limit = flag_alternateform ? va_arg(ap,int) : -1;
        char const *e = va_arg(ap,char const*);
        if( e && *e ){
          length = StrNLen32(e, limit);
          zExtra = bufpt = fsl_malloc(length+1);
          if(!zExtra) return FSL_RC_OOM;
          for( i=0; i<length; i++ ){
            if( e[i]=='\\' ){
              bufpt[i]='/';
            }else{
              bufpt[i]=e[i];
            }
          }
          bufpt[length]='\0';
        }
        break;
      }
      case etSTRINGID: {
        precision = flag_altform2 ? -1 : 16
          /* In fossil(1) this is configurable, but in this lib we
             don't have access to that state from here. Fossil also
             has the '!' flag_altform2, which indicates that it
             should be for a URL, and thus longer than the default.
             We are only roughly approximating that behaviour here. */;
        FSL_SWITCH_FALL_THROUGH;
      }
      case etSTRING: {
        bufpt = va_arg(ap,char*);
        length = bufpt
          ? StrNLen32(bufpt,
                      (precision>0 && flag_alternateform)
                      ? precision*4/*max bytes per char*/
                      : (precision>=0 ? precision : -1))
          : (int)0;
        if(flag_alternateform && length && (precision>0 || width>0)){
          appendf__utf8_altform(bufpt, &length, &precision, &width);
        }else if( length && precision>=0 && precision<length ){
          length = precision;
        }
        break;
      }
      case etDYNSTRING: {
        /* etDYNSTRING needs to be handled separately because it
           free()s its argument (which isn't available outside this
           block). This means, though, that %-#z does not work. */
        bufpt = va_arg(ap,char*);
        length = bufpt
          ? StrNLen32(bufpt,
                      (precision>0 && flag_alternateform)
                      ? precision*4/*max bytes per char*/
                      : (precision>=0 ? precision : -1))
          : (int)0;
        if(flag_alternateform && length && (precision>0 || width>0)){
          appendf__utf8_altform(bufpt, &length, &precision, &width);
        }else if( length && precision>=0 && precision<length ){
          length = precision;
        }
        pfrc = spech_dynstring( pfAppend, pfAppendArg,
                                length, bufpt );
        bufpt = NULL;
        FSLPRINTF_CHECKERR;
        length = 0;
        break;
      }
      case etBLOB: {
        /* int const limit = flag_alternateform ? va_arg(ap, int) : -1; */
        fsl_buffer *pBlob = va_arg(ap, fsl_buffer*);
        bufpt = fsl_buffer_str(pBlob);
        length = (int)fsl_buffer_size(pBlob);
        if( precision>=0 && precision<length ) length = precision;
        /* if( limit>=0 && limit<length ) length = limit; */
        break;
      }
      case etFOSSILIZE:{
        int const limit = -1; /*flag_alternateform ? va_arg(ap,int) : -1;*/
        fsl_buffer fb = fsl_buffer_empty;
        int check;
        bufpt = va_arg(ap,char*);
        length = bufpt ? (int)fsl_strlen(bufpt) : 0;
        if((limit>=0) && (length>limit)) length = limit;
        check = fsl_bytes_fossilize((unsigned char const *)bufpt, length, &fb);
        if(check){
          fsl_buffer_reserve(&fb,0);
          FSLPRINTF_RETURN(check);
        }
        zExtra = bufpt = (char*)fb.mem
          /*transfer ownership*/;
        length = (int)fb.used;
        if( precision>=0 && precision<length ) length = precision;
        break;
      }
#if FSLPRINTF_ENABLE_JSON
      case etJSONSTR: {
        struct SpechJson state;
        bufpt = va_arg(ap,char *);
        length = bufpt
          ? (precision>=0 ? precision : (int)fsl_strlen(bufpt))
          : 0;
        state.z = bufpt;
        state.addQuotes = flag_altform2 ? true : false;
        state.escapeSmallUtf8 = flag_alternateform ? true : false;
        pfrc = spech_json( pfAppend, pfAppendArg, (unsigned)length, &state );
        bufpt = NULL;
        FSLPRINTF_CHECKERR;
        length = 0;
        break;
      }
#endif
#if ! FSLPRINTF_OMIT_HTML
      case etHTML:{
        bufpt = va_arg(ap,char*);
        length = bufpt ? (int)fsl_strlen(bufpt) : 0;
        pfrc = spech_string_to_html( pfAppend, pfAppendArg,
                                     (precision>=0 && precision<length) ? precision : length,
                                     bufpt );
        bufpt = NULL;
        FSLPRINTF_CHECKERR;
        length = 0;
        break;
      }
      case etURLENCODE:{
        bufpt = va_arg(ap,char*);
        length = bufpt ? (int)fsl_strlen(bufpt) : 0;
        pfrc = spech_urlencode( pfAppend, pfAppendArg,
                                (precision>=0 && precision<length) ? precision : length,
                                bufpt );
        bufpt = NULL;
        FSLPRINTF_CHECKERR;
        length = 0;
        break;
      }
      case etURLDECODE:{

        bufpt = va_arg(ap,char*);
        length = bufpt ? (int)fsl_strlen(bufpt) : 0;
        pfrc = spech_urldecode( pfAppend, pfAppendArg,
                                (precision>=0 && precision<length) ? precision : length,
                                bufpt );
        bufpt = NULL;
        FSLPRINTF_CHECKERR;
        length = 0;
        break;
      }
#endif /* FSLPRINTF_OMIT_HTML */
#if ! FSLPRINTF_OMIT_SQL
      case etBLOBSQL:
      case etSQLESCAPE:
      case etSQLESCAPE2:
      case etSQLESCAPE3: {
        if(flag_altform2 && etSQLESCAPE2==xtype){
          xtype = etSQLESCAPE3;
        }
        if(etBLOBSQL==xtype){
          fsl_buffer * const b = va_arg(ap,fsl_buffer*);
          bufpt = b ? fsl_buffer_str(b) : NULL;
          length = b ? (int)fsl_buffer_size(b) : 0;
          if(flag_altform2) xtype = etSQLESCAPE3;
        }else{
          bufpt = va_arg(ap,char*);
          length = bufpt ? (int)strlen(bufpt) : 0;
        }
        pfrc = spech_sqlstring( xtype, pfAppend, pfAppendArg,
                                (precision>=0 && precision<length) ? precision : length,
                                bufpt );
        FSLPRINTF_CHECKERR;
        length = 0;
      }
#endif /* !FSLPRINTF_OMIT_SQL */
    }/* End switch over the format type */
    /*
      The text of the conversion is pointed to by "bufpt" and is
      "length" characters long.  The field width is "width".  Do
      the output.
    */
    if( !flag_leftjustify ){
      int nspace;
      nspace = width-length;
      if( nspace>0 ){
        FSLPRINTF_SPACES(nspace);
      }
    }
    if( length>0 ){
      pfrc = pfAppend( pfAppendArg, bufpt, length);
      FSLPRINTF_CHECKERR;
    }
    if( flag_leftjustify ){
      int nspace;
      nspace = width-length;
      if( nspace>0 ){
        FSLPRINTF_SPACES(nspace);
      }
    }
    if( zExtra ){
      fsl_free(zExtra);
      zExtra = 0;
    }
  }/* End for loop over the format string */
  FSLPRINTF_RETURN(0);
} /* End of function */


#undef FSLPRINTF_CHARARRAY_STACK
#undef FSLPRINTF_CHARARRAY
#undef FSLPRINTF_CHARARRAY_FREE
#undef FSLPRINTF_SPACES
#undef FSLPRINTF_CHECKERR
#undef FSLPRINTF_RETURN
#undef FSLPRINTF_OMIT_FLOATING_POINT
#undef FSLPRINTF_OMIT_SIZE
#undef FSLPRINTF_OMIT_SQL
#undef FSLPRINTF_BUF_SIZE
#undef FSLPRINTF_OMIT_HTML

int fsl_appendf(fsl_output_f pfAppend, void * pfAppendArg,
                const char *fmt, ... ){
  int ret;
  va_list vargs;
  va_start( vargs, fmt );
  ret = fsl_appendfv( pfAppend, pfAppendArg, fmt, vargs );
  va_end(vargs);
  return ret;
}

int fsl_fprintfv( FILE * fp, char const * fmt, va_list args ){
  return (fp && fmt)
    ? fsl_appendfv( fsl_output_f_FILE, fp, fmt, args )
    :  FSL_RC_MISUSE;
}

int fsl_fprintf( FILE * fp, char const * fmt, ... ){
  int ret;
  va_list vargs;
  va_start( vargs, fmt );
  ret = fsl_appendfv( fsl_output_f_FILE, fp, fmt, vargs );
  va_end(vargs);
  return ret;
}

char * fsl_mprintfv( char const * fmt, va_list vargs ){
  if( !fmt ) return 0;
  else if(!*fmt) return fsl_strndup("",0);
  else{
    fsl_buffer buf = fsl_buffer_empty;
    int const rc = fsl_buffer_appendfv( &buf, fmt, vargs );
    if(rc){
      fsl_buffer_reserve(&buf, 0);
      assert(0==buf.mem);
    }
    return (char*)buf.mem /*transfer ownership*/;
  }
}

char * fsl_mprintf( char const * fmt, ... ){
  char * ret;
  va_list vargs;
  va_start( vargs, fmt );
  ret = fsl_mprintfv( fmt, vargs );
  va_end( vargs );
  return ret;
}


/**
    Internal state for fsl_snprintfv().
 */
struct fsl_snp_state {
  /** Destination memory */
  char * dest;
  /** Current output position in this->dest. */
  fsl_size_t pos;
  /** Length of this->dest. */
  fsl_size_t len;
};
typedef struct fsl_snp_state fsl_snp_state;

static int fsl_output_f_snprintf( void * arg,
                                  void const * data_,
                                  fsl_size_t n ){
  char const * data = (char const *)data_;
  fsl_snp_state * st = (fsl_snp_state*) arg;
  if(n==0 || (st->pos >= st->len)) return 0;
  else if((n + st->pos) > st->len){
    n = st->len - st->pos;
  }
  memcpy(st->dest + st->pos, data, n);
  st->pos += n;
  assert(st->pos <= st->len);
  return 0;
}

int fsl_snprintfv( char * dest, fsl_size_t n,
                   char const * fmt, va_list args){
  fsl_snp_state st = {NULL,0,0};
  int rc = 0;
  if(!dest || !fmt) return FSL_RC_MISUSE;
  else if(!n || !*fmt){
    if(dest) *dest = 0;
    return 0;
  }
  st.len = n;
  st.dest = dest;
  rc = fsl_appendfv( fsl_output_f_snprintf, &st, fmt, args );
  if(st.pos < st.len){
    dest[st.pos] = 0;
  }
  return rc;
}

int fsl_snprintf( char * dest, fsl_size_t n, char const * fmt, ... ){
  int rc = 0;
  va_list vargs;
  va_start( vargs, fmt );
  rc = fsl_snprintfv( dest, n, fmt, vargs );
  va_end( vargs );
  return rc;
}

#undef MARKER
/* end of file ./src/appendf.c */
/* start of file ./src/auth.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/

/***************************************************************************
  This file contains routines related to working with user authentication.
*/


FSL_EXPORT char * fsl_sha1_shared_secret( fsl_cx * const f,
                                          char const * zLoginName,
                                          char const * zPw ){
    if(!f || !zPw || !zLoginName) return 0;
    else{
        fsl_sha1_cx hash = fsl_sha1_cx_empty;
        unsigned char zResult[20];
        char zDigest[41];
        if(!f->cache.projectCode){
            f->cache.projectCode = fsl_config_get_text(f, FSL_CONFDB_REPO,
                                                       "project-code", 0);
            /*
              fossil(1) returns a copy of zPw here if !f->cache.projectCode,
              with the following comment:
            */
            /* On the first xfer request of a clone, the project-code is not yet
            ** known.  Use the cleartext password, since that is all we have.
            */
            if(!f->cache.projectCode) return 0;
        }
        fsl_sha1_update(&hash, f->cache.projectCode,
                        fsl_strlen(f->cache.projectCode));
        fsl_sha1_update(&hash, "/", 1);
        fsl_sha1_update(&hash, zLoginName, fsl_strlen(zLoginName));
        fsl_sha1_update(&hash, "/", 1);
        fsl_sha1_update(&hash, zPw, fsl_strlen(zPw));
        fsl_sha1_final(&hash, zResult);
        fsl_sha1_digest_to_base16(zResult, zDigest);
        return fsl_strndup( zDigest, FSL_STRLEN_SHA1 );
    }
}

FSL_EXPORT char * fsl_repo_login_group_name(fsl_cx * const f){
  return f
    ? fsl_config_get_text(f, FSL_CONFDB_REPO,
                          "login-group-name", 0)
    : 0;
}

FSL_EXPORT char * fsl_repo_login_cookie_name(fsl_cx * const f){
  fsl_db * db;
  if(!f || !(db = fsl_cx_db_repo(f))) return 0;
  else{
    char const * sql =
      "SELECT 'fossil-' || substr(value,1,16)"
      "  FROM config"
      " WHERE name IN ('project-code','login-group-code')"
      " ORDER BY name /*sort*/";
    return fsl_db_g_text(db, 0, sql);
  }
}

FSL_EXPORT int fsl_repo_login_search_uid(fsl_cx * const f, char const * zUsername,
                                         char const * zPasswd,
                                         fsl_id_t * userId){
  int rc;
  char * zSecret;
  fsl_db * db;
  if(!f || !userId
     || !zUsername || !*zUsername
     || !zPasswd /*??? || !*zPasswd*/){
    return FSL_RC_MISUSE;
  }
  else if(!(db = fsl_needs_repo(f))){
    return FSL_RC_NOT_A_REPO;
  }
  *userId = 0;
  zSecret = fsl_sha1_shared_secret(f, zUsername, zPasswd );
  if(!zSecret) return FSL_RC_OOM;
  rc = fsl_db_get_id(db, userId,
                     "SELECT uid FROM user"
                     " WHERE login=%Q"
                     "   AND length(cap)>0 AND length(pw)>0"
                     "   AND login NOT IN ('anonymous','nobody','developer','reader')"
                     "   AND (pw=%Q OR (length(pw)<>40 AND pw=%Q))",
                     zUsername, zSecret, zPasswd);
  fsl_free(zSecret);
  return rc;
}

FSL_EXPORT int fsl_repo_login_clear( fsl_cx * const f, fsl_id_t userId ){
  fsl_db * db;
  if(!f) return FSL_RC_MISUSE;
  else if(!(db = fsl_needs_repo(f))) return FSL_RC_NOT_A_REPO;
  else{
    int const rc = fsl_db_exec(db,
                       "UPDATE user SET cookie=NULL, ipaddr=NULL, "
                       " cexpire=0 WHERE "
                       " CASE WHEN %"FSL_ID_T_PFMT">=0 THEN uid=%"FSL_ID_T_PFMT""
                       " ELSE uid>0 END"
                       " AND login NOT IN('anonymous','nobody',"
                       " 'developer','reader')",
                       (fsl_id_t)userId, (fsl_id_t)userId);
    if(rc){
      fsl_cx_uplift_db_error(f, db);
    }
    return rc;
  }
}

#undef MARKER
/* end of file ./src/auth.c */
/* start of file ./src/bag.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/
/***************************************************************************
  This file houses the code for the fsl_id_bag class.
*/
#include <assert.h>
#include <string.h> /* memset() */

const fsl_id_bag fsl_id_bag_empty = fsl_id_bag_empty_m;

void fsl_id_bag_clear(fsl_id_bag * const p){
  fsl_free(p->list);
  *p = fsl_id_bag_empty;
}

/*
   The hash function
*/
#define fsl_id_bag_hash(i)  (i*101)

/*
   Change the size of the hash table on a bag so that
   it contains N slots
  
   Completely reconstruct the hash table from scratch.  Deleted
   entries (indicated by a -1) are removed. When finished,
   p->entryCount==p->used and p->capacity==newSize.

   Returns on on success, FSL_RC_OOM on allocation error.
*/
static int fsl_id_bag_resize(fsl_id_bag * const p, fsl_size_t newSize){
  fsl_size_t i;
  fsl_id_bag old;
  fsl_size_t nDel = 0;   /* Number of deleted entries */
  fsl_size_t nLive = 0;  /* Number of live entries */
  fsl_id_t * newList;
  assert( newSize > p->entryCount );
  newList = (fsl_id_t*)fsl_malloc( sizeof(p->list[0])*newSize );
  if(!newList) return FSL_RC_OOM;
  old = *p;
  p->list = newList;
  p->capacity = newSize;
  memset(p->list, 0, sizeof(p->list[0])*newSize );
  for(i=0; i<old.capacity; i++){
    fsl_id_t const e = old.list[i];
    if( e>0 ){
      unsigned h = fsl_id_bag_hash(e)%newSize;
      while( p->list[h] ){
        h++;
        if( h==newSize ) h = 0;
      }
      p->list[h] = e;
      nLive++;
    }else if( e<0 ){
      nDel++;
    }
  }
  assert( p->entryCount == nLive );
  assert( p->used == nLive+nDel );
  p->used = p->entryCount;
  fsl_id_bag_clear(&old);
  return 0;
}

void fsl_id_bag_reset(fsl_id_bag * const p){
  p->entryCount = p->used = 0;
  if(p->list){
    memset(p->list, 0, sizeof(p->list[0])*p->capacity);
  }
}


int fsl_id_bag_insert(fsl_id_bag * const p, fsl_id_t e){
  fsl_size_t h;
  int rc = 0;
  assert( e>0 );
  if( p->used+1 >= p->capacity/2 ){
    fsl_size_t const n = p->capacity ? p->capacity*2 : 30;
    rc = fsl_id_bag_resize(p,  n );
    if(rc) return rc;
  }
  h = fsl_id_bag_hash(e)%p->capacity;
  while( p->list[h]>0 && p->list[h]!=e ){
    h++;
    if( h>=p->capacity ) h = 0;
  }
  if( p->list[h]<=0 ){
    if( p->list[h]==0 ) ++p->used;
    p->list[h] = e;
    ++p->entryCount;
    rc = 0;
  }
  return rc;
}

bool fsl_id_bag_contains(fsl_id_bag const * const p, fsl_id_t e){
  fsl_size_t h;
  assert( e>0 );
  if( p->capacity==0 || 0==p->used ){
    return false;
  }
  assert(p->list);
  h = fsl_id_bag_hash(e)%p->capacity;
  while( p->list[h] && p->list[h]!=e ){
    h++;
    if( h>=p->capacity ) h = 0
      /*loop around to the start*/
      ;
  }
  return p->list[h]==e;
}

bool fsl_id_bag_remove(fsl_id_bag * const p, fsl_id_t e){
  fsl_size_t h;
  bool rv = false;
  assert( e>0 );
  if( !p->capacity || !p->used ) return rv;
  assert(p->list);
  h = fsl_id_bag_hash(e)%p->capacity;
  while( p->list[h] && p->list[h]!=e ){
    ++h;
    if( h>=p->capacity ) h = 0;
  }
  rv = p->list[h]==e;
  if( p->list[h] ){
    fsl_size_t nx = h+1;
    if( nx>=p->capacity ) nx = 0;
    if( p->list[nx]==0 ){
      p->list[h] = 0;
      --p->used;
    }else{
      p->list[h] = -1;
    }
    --p->entryCount;
    if( p->entryCount==0 ){
      memset(p->list, 0, p->capacity*sizeof(p->list[0]));
      p->used = 0;
    }else if( p->capacity>40 && p->entryCount<p->capacity/8 ){
      fsl_id_bag_resize(p, p->capacity/2)
        /* ignore realloc error and keep the old size. */;
    }
  }
  return rv;
}

fsl_id_t fsl_id_bag_first(fsl_id_bag const * const p){
  if( p->capacity==0 || 0==p->used ){
    return 0;
  }else{
    fsl_size_t i;
    for(i=0; i<p->capacity && p->list[i]<=0; ++i){}
    if( i<p->capacity ){
      return p->list[i];
    }else{
      return 0;
    }
  }
}

fsl_id_t fsl_id_bag_next(fsl_id_bag const * const p, fsl_id_t e){
  fsl_size_t h;
  assert( p->capacity>0 );
  assert( e>0 );
  assert(p->list);
  h = fsl_id_bag_hash(e)%p->capacity;
  while( p->list[h] && p->list[h]!=e ){
    ++h;
    if( h>=p->capacity ) h = 0;
  }
  assert( p->list[h] );
  h++;
  while( h<p->capacity && p->list[h]<=0 ){
    h++;
  }
  return h<p->capacity ? p->list[h] : 0;
}

fsl_size_t fsl_id_bag_count(fsl_id_bag const * const p){
  return p->entryCount;
}

void fsl_id_bag_swap(fsl_id_bag * const lhs, fsl_id_bag * const rhs){
  fsl_id_bag x = *lhs;
  *lhs = *rhs;
  *rhs = x;
}

#undef fsl_id_bag_hash
/* end of file ./src/bag.c */
/* start of file ./src/buffer.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/

#include <assert.h>
#include <string.h> /* strlen() */
#include <stddef.h> /* NULL on linux */
#include <errno.h>

#include <zlib.h>


#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)

#define buf__is_external(b) (b->mem && 0==b->capacity)

#define buff__errcheck(B) if((B)->errCode) return (B)->errCode
/**
   Materializes external buffer b by allocating b->used+extra+1
   bytes, copying b->used bytes from b->mem to the new block,
   NUL-terminating the block, and replacing b->mem with the new
   block. Returns 0 on success, else FSL_RC_OOM.

   Asserts that b is an external buffer.
*/
static int fsl__buffer_materialize( fsl_buffer * const b, fsl_size_t extra ){
  assert(buf__is_external(b));
  buff__errcheck(b);
  fsl_size_t const n = b->used + extra + 1;
  unsigned char * x = (unsigned char *)fsl_malloc(n);
  if(!x) return b->errCode = FSL_RC_OOM;
  memcpy(x, b->mem, b->used);
  b->capacity = n;
  x[b->used] = 0;
  b->mem = x;
  return 0;
}

int fsl_buffer_err( fsl_buffer const * b ){
  return b->errCode;
}

void fsl_buffer_err_clear(fsl_buffer * const b){
  b->errCode = 0;
}

int fsl_buffer_materialize( fsl_buffer * const b ){
  buff__errcheck(b);
  return buf__is_external(b) ? fsl__buffer_materialize(b, 0) : 0;
}

#define buf__materialize(B,N) (buf__is_external(B) ? fsl__buffer_materialize((B),(N)) : 0)

void fsl_buffer_external( fsl_buffer * const b, void const * mem, fsl_int_t n ){
  if(b->mem) fsl_buffer_clear(b);
  if(n<0) n =(fsl_int_t)fsl_strlen((char const *)mem);
  b->used = n;
  b->cursor = 0;
  b->errCode = 0;
  b->mem = (unsigned char *)mem;
  b->capacity = 0;
}

fsl_buffer * fsl_buffer_reuse( fsl_buffer * const b ){
  if(buf__is_external(b)){
    *b = fsl_buffer_empty;
  }else{
    if(b->capacity){
      assert(b->mem);
      b->mem[0] = 0;
      b->used = 0;
    }
    b->cursor = 0;
    b->errCode = 0;
  }
  return b;
}

void fsl_buffer_clear( fsl_buffer * const buf ){
  if(buf->capacity) fsl_free(buf->mem);
  *buf = fsl_buffer_empty;
}

int fsl_buffer_reserve( fsl_buffer * const buf, fsl_size_t n ){
  if( 0 == n ){
    if(!buf__is_external(buf)){
      fsl_free(buf->mem);
    }/* else if it has memory, it's owned elsewhere */
    *buf = fsl_buffer_empty;
    return 0;
  }
  else buff__errcheck(buf);
  else if( !buf__is_external(buf) && buf->capacity >= n ){
    assert(buf->mem);
    return 0;
  }else{
    unsigned char * x;
    bool const isExt = buf__is_external(buf);
    assert((buf->used < n) && "Buffer in-use greater than capacity!");
    if(isExt && n<=buf->used){
      /*For external buffers, always keep at least the initially-pointed-to
        size. */
      n = buf->used + 1;
    }
    x = (unsigned char *)fsl_realloc( isExt ? NULL : buf->mem, n );
    if( !x ) return buf->errCode = FSL_RC_OOM;
    else if(isExt){
      memcpy( x, buf->mem, buf->used );
      x[buf->used] = 0;
    }else{
      memset( x + buf->used, 0, n - buf->used );
    }
    buf->mem = x;
    buf->capacity = n;
    return 0;
  }
}

int fsl_buffer_resize( fsl_buffer * const b, fsl_size_t n ){
  buff__errcheck(b);
  else if(buf__is_external(b)){
    if(n==b->used) return 0;
    else if(n==0){
      b->capacity = 0;
      fsl_buffer_external(b, "", 0);
      return 0;
    }
    unsigned char * x = (unsigned char *)fsl_malloc( n+1/*NUL*/ );
    if( !x ) return b->errCode = FSL_RC_OOM;
    memcpy(x, b->mem, n < b->used ? n : b->used);
    x[n] = 0;
    b->mem = x;
    b->capacity = n+1;
    b->used = n;
    return 0;
  }else if(n && (b->capacity == n+1)){
    b->used = n;
    b->mem[n] = 0;
    return 0;
  }else{
    unsigned char * x = (unsigned char *)fsl_realloc( b->mem,
                                                      n+1/*NUL*/ );
    if( ! x ) return b->errCode = FSL_RC_OOM;
    if(n > b->capacity){
      /* zero-fill new parts */
      memset( x + b->capacity, 0, n - b->capacity +1/*NUL*/ );
    }
    b->capacity = n + 1 /*NUL*/;
    b->used = n;
    b->mem = x;
    b->mem[b->used] = 0;
    return 0;
  }
}

int fsl_buffer_compare(fsl_buffer const * const lhs, fsl_buffer const * const rhs){
  fsl_size_t const szL = lhs->used;
  fsl_size_t const szR = rhs->used;
  fsl_size_t const sz = (szL<szR) ? szL : szR;
  int rc = memcmp(lhs->mem, rhs->mem, sz);
  if(0 == rc){
    rc = (szL==szR)
      ? 0
      : ((szL<szR) ? -1 : 1);
  }
  return rc;
}

bool fsl_buffer_eq(fsl_buffer const * const b, char const * str,
                   fsl_int_t nStr){
  if(nStr<0) nStr = (fsl_int_t)fsl_strlen(str);
  fsl_buffer rhs = fsl_buffer_empty;
  fsl_buffer_external(&rhs, str, nStr);
  return 0==fsl_buffer_compare(b, &rhs);
}

/*
   Compare two blobs in constant time and return zero if they are equal.
   Constant time comparison only applies for blobs of the same length.
   If lengths are different, immediately returns 1.
*/
int fsl_buffer_compare_O1(fsl_buffer const * const lhs, fsl_buffer const * const rhs){
  fsl_size_t const szL = lhs->used;
  fsl_size_t const szR = rhs->used;
  fsl_size_t i;
  unsigned char const *buf1;
  unsigned char const *buf2;
  unsigned char rc = 0;
  if( szL!=szR || szL==0 ) return 1;
  buf1 = lhs->mem;
  buf2 = rhs->mem;
  for( i=0; i<szL; i++ ){
    rc = rc | (buf1[i] ^ buf2[i]);
  }
  return rc;
}


int fsl_buffer_append( fsl_buffer * const b,
                       void const * const data,
                       fsl_int_t len ){
  if(0==b->errCode){
    fsl_size_t sz = b->used;
    if(len<0) len = (fsl_int_t)fsl_strlen((char const *)data);
    if(buf__materialize(b, (fsl_size_t)len + 1)) return b->errCode;
    assert(b->capacity ? !!b->mem : !b->mem);
    assert(b->used <= b->capacity);
    sz += len + 1/*NUL*/;
    if(b->capacity<sz) fsl_buffer_reserve( b, sz );
    if(!b->errCode){
      assert(b->capacity >= sz);
      if(len>0) memcpy(b->mem + b->used, data, (size_t)len);
      b->used += len;
      b->mem[b->used] = 0;
    }
  }
  return b->errCode;
}

int fsl_buffer_appendfv( fsl_buffer * const b, char const * fmt,
                         va_list args){
  return fsl_appendfv( fsl_output_f_buffer, b, fmt, args );
}


int fsl_buffer_appendf( fsl_buffer * const b,
                        char const * fmt, ... ){
  buff__errcheck(b);
  else{
    va_list args;
    va_start(args,fmt);
    fsl_buffer_appendfv( b, fmt, args );
    va_end(args);
    return b->errCode;
  }
}

char const * fsl_buffer_cstr(fsl_buffer const * const b){
  return b->errCode ? NULL : (char const *)b->mem;
}

char const * fsl_buffer_cstr2(fsl_buffer const * const b, fsl_size_t * const len){
  char const * rc = NULL;
  if(0==b->errCode){
    rc = (char const *)b->mem;
    if(len) *len = b->used;
  }
  return rc;
}

char * fsl_buffer_str(fsl_buffer const * const b){
  return b->errCode ? NULL : (char *)b->mem;
}


#if 0
fsl_size_t fsl_buffer_size(fsl_buffer const * const b){
  return b->used;
}

fsl_size_t fsl_buffer_capacity(fsl_buffer const * const b){
  return b->capacity;
}
#endif

bool fsl_data_is_compressed(unsigned char const * const mem, fsl_size_t len){
  if(!mem || (len<6)) return 0;
#if 0
  else return ('x'==mem[4])
    && (0234==mem[5]);
  /*
    This check fails for one particular artifact in the tcl core.
    Notes gathered while debugging...

    https://core.tcl.tk/tcl/

    Delta manifest #5f37dcc3 while processing file #687
    (1-based):

    FSL_RC_RANGE: "Delta: copy extends past end of input"

    To reproduce from tcl repo:

    f-acat 5f37dcc3 | f-mfparse -r

    More details:

    Filename: library/encoding/gb2312-raw.enc
    Content: dba09c670f24d47b95d12d4bb9704391b81dda9a

    That artifact is a delta of bccc899015b688d5c426bc791c2fcde3a03a3eb5,
    which is actually two files:

    library/encoding/euc-cn.enc
    library/encoding/gb2312.enc

    When we go to apply the delta, the contents of bccc8 appear to
    be badly compressed data. They have the 'x' at byte offset
    4 but not the 0234 at byte offset 5.

    Turns out it is the fsl_buffer_is_compressed() impl which fails
    for that one.
  */
#else
  else{
    /**
       Adapted from:
       
       https://blog.2of1.org/2011/03/03/decompressing-zlib-images/

       Remember that fossil-compressed data has a 4-byte big-endian
       header holding the uncompressed size of the data, so we skip
       those first 4 bytes.

       See also:

       https://tools.ietf.org/html/rfc6713

       search for "magic number".
    */
    int16_t const head = (((int16_t)mem[4]) << 8) | mem[5];
    /* MARKER(("isCompressed header=%04x\n", head)); */
    switch(head){
      case 0x083c: case 0x087a: case 0x08b8: case 0x08f6:
      case 0x1838: case 0x1876: case 0x18b4: case 0x1872:
      case 0x2834: case 0x2872: case 0x28b0: case 0x28ee:
      case 0x3830: case 0x386e: case 0x38ac: case 0x38ea:
      case 0x482c: case 0x486a: case 0x48a8: case 0x48e6:
      case 0x5828: case 0x5866: case 0x58a4: case 0x58e2:
      case 0x6824: case 0x6862: case 0x68bf: case 0x68fd:
      case 0x7801: case 0x785e: case 0x789c: case 0x78da:
        return true;
      default:
        return false;
    }
  }
#endif
}

bool fsl_buffer_is_compressed(fsl_buffer const *buf){
  return fsl_data_is_compressed( buf->mem, buf->used );
}

fsl_int_t fsl_data_uncompressed_size(unsigned char const *mem,
                                     fsl_size_t len){
  return fsl_data_is_compressed(mem,len)
    ? ((mem[0]<<24) + (mem[1]<<16) + (mem[2]<<8) + mem[3])
    : -1;
}

fsl_int_t fsl_buffer_uncompressed_size(fsl_buffer const * b){
  return fsl_data_uncompressed_size(b->mem, b->used);
}

int fsl_buffer_compress(fsl_buffer const *pIn, fsl_buffer * const pOut){
  buff__errcheck(pIn);
  else buff__errcheck(pOut);
  unsigned int nIn = pIn->used;
  unsigned int nOut = 13 + nIn + (nIn+999)/1000;
  fsl_buffer temp = fsl_buffer_empty;
  int rc = fsl_buffer_resize(&temp, nOut+4);
  if(rc) return rc;
  else{
    unsigned long int nOut2;
    unsigned char *outBuf;
    unsigned long int outSize;
    outBuf = temp.mem;
    outBuf[0] = nIn>>24 & 0xff;
    outBuf[1] = nIn>>16 & 0xff;
    outBuf[2] = nIn>>8 & 0xff;
    outBuf[3] = nIn & 0xff;
    nOut2 = (long int)nOut;
    rc = compress(&outBuf[4], &nOut2,
                  pIn->mem, pIn->used);
    if(rc){
      fsl_buffer_clear(&temp);
      return FSL_RC_ERROR;
    }
    outSize = nOut2+4;
    rc = fsl_buffer_resize(&temp, outSize);
    if(rc){
      fsl_buffer_clear(&temp);
    }else{
      fsl_buffer_swap_free(&temp, pOut, -1);
      assert(0==temp.used);
      assert(outSize==pOut->used);
    }
    return rc;
  }
}

int fsl_buffer_compress2(fsl_buffer const *pIn1,
                         fsl_buffer const *pIn2, fsl_buffer * const pOut){
  buff__errcheck(pIn1);
  else buff__errcheck(pIn2);
  else buff__errcheck(pOut);
  unsigned int nIn = pIn1->used + pIn2->used;
  unsigned int nOut = 13 + nIn + (nIn+999)/1000;
  fsl_buffer temp = fsl_buffer_empty;
  int rc;
  rc = fsl_buffer_resize(&temp, nOut+4);
  if(rc) return rc;
  else{
    unsigned char *outBuf;
    z_stream stream;
    outBuf = temp.mem;
    outBuf[0] = nIn>>24 & 0xff;
    outBuf[1] = nIn>>16 & 0xff;
    outBuf[2] = nIn>>8 & 0xff;
    outBuf[3] = nIn & 0xff;
    stream.zalloc = (alloc_func)0;
    stream.zfree = (free_func)0;
    stream.opaque = 0;
    stream.avail_out = nOut;
    stream.next_out = &outBuf[4];
    deflateInit(&stream, 9);
    stream.avail_in = pIn1->used;
    stream.next_in = pIn1->mem;
    deflate(&stream, 0);
    stream.avail_in = pIn2->used;
    stream.next_in = pIn2->mem;
    deflate(&stream, 0);
    deflate(&stream, Z_FINISH);
    rc = fsl_buffer_resize(&temp, stream.total_out + 4);
    deflateEnd(&stream);
    if(!rc){
      temp.used = stream.total_out + 4;
      if( pOut==pIn1 ) fsl_buffer_reserve(pOut, 0);
      else if( pOut==pIn2 ) fsl_buffer_reserve(pOut, 0);
      assert(!pOut->mem);
      *pOut = temp;
    }else{
      fsl_buffer_reserve(&temp, 0);
    }
    return rc;
  }
}

int fsl_buffer_uncompress(fsl_buffer const * const pIn, fsl_buffer * const pOut){
  buff__errcheck(pIn);
  else buff__errcheck(pOut);
  unsigned int nOut;
  unsigned char *inBuf;
  unsigned int const nIn = pIn->used;
  fsl_buffer temp = fsl_buffer_empty;
  int rc;
  unsigned long int nOut2;
  if(nIn<=4 || !fsl_data_is_compressed(pIn->mem, pIn->used)){
    if(pIn==pOut || !pIn->mem) rc = 0;
    else{
      fsl_buffer_reuse(pOut);
      rc = fsl_buffer_append(pOut, pIn->mem, pIn->used);
    }
    assert(pOut->errCode == rc);
    return rc;
  }
  inBuf = pIn->mem;
  nOut = (inBuf[0]<<24) + (inBuf[1]<<16) + (inBuf[2]<<8) + inBuf[3];
  /* MARKER(("decompress size: %u\n", nOut)); */
  if(pIn!=pOut && pOut->capacity>=nOut+1){
    assert(pIn->mem != pOut->mem);
#if 0
    /* why does this cause corruption (in the form of overwriting a
       buffer somewhere in the fsl_content_get() constellation)?
       fsl_repo_rebuild() works but fsl_repo_extract() can trigger
       it:

       (FSL_RC_RANGE): Delta: copy extends past end of input
    */
    fsl_buffer_external(&temp, pOut->mem, pOut->capacity);
#else
    fsl_buffer_swap(&temp, pOut);
#endif
  }else{
    rc = fsl_buffer_reserve(&temp, nOut+1);
    if(rc) return pOut->errCode = rc;
    temp.mem[nOut] = 0;
  }
  
  nOut2 = (long int)nOut;
  rc = uncompress(temp.mem, &nOut2, &inBuf[4], nIn - 4)
    /* In some libz versions (<1.2.4, apparently), valgrind says
       there's an uninitialized memory access somewhere under
       uncompress(), _presumably_ for one of these arguments, but i
       can't find it. fsl_buffer_reserve() always memsets() new bytes
       to 0.

       Turns out it's a known problem:

       https://www.zlib.net/zlib_faq.html#faq36
    */;
  switch(rc){
    case 0:
      /* this is always true but having this assert
         here makes me nervous: assert(nOut2 == nOut); */
      assert(nOut2<=nOut);
      temp.mem[nOut2] = 0;
      temp.used = (fsl_size_t)nOut2;
#if 1
      fsl_buffer_swap(&temp, pOut);
#else
      if(temp.mem!=pOut->mem){
        if(pOut->capacity>=temp.capacity){
          pOut->used = 0;
          MARKER(("uncompress() re-using target buffer.\n"));
          fsl_buffer_append(pOut, temp.mem, temp.capacity);
        }else{
          fsl_buffer_swap(pOut, &temp);
        }
      }
#endif
      break;
    case Z_DATA_ERROR: rc = FSL_RC_CONSISTENCY; break;
    case Z_MEM_ERROR: rc = FSL_RC_OOM; break;
    case Z_BUF_ERROR:
      assert(!"Cannot happen!");
      rc = FSL_RC_RANGE; break;
    default: rc = FSL_RC_ERROR; break;
  }
  if(temp.mem!=pOut->mem) fsl_buffer_clear(&temp);
  return pOut->errCode = rc;
}


int fsl_buffer_fill_from( fsl_buffer * const dest, fsl_input_f src,
                          void * const state ){
  buff__errcheck(dest);
  int rc;
  enum { BufSize = 512 * 8 };
  char rbuf[BufSize];
  fsl_size_t total = 0;
  fsl_size_t rlen = 0;
  fsl_buffer_reuse(dest);
  while(1){
    rlen = BufSize;
    rc = src( state, rbuf, &rlen );
    if( rc ) break;
    total += rlen;
    if(total<rlen){
      /* Overflow! */
      rc = FSL_RC_RANGE;
      break;
    }
    if( dest->capacity < (total+1) ){
      rc = fsl_buffer_reserve( dest,
                               total + ((rlen<BufSize) ? 1 : BufSize)
                               );
      if( 0 != rc ) break;
    }
    memcpy( dest->mem + dest->used, rbuf, rlen );
    dest->used += rlen;
    if( rlen < BufSize ) break;
  }
  if( !rc && dest->used ){
    assert( dest->used < dest->capacity );
    dest->mem[dest->used] = 0;
  }
  return rc;
}

int fsl_buffer_fill_from_FILE( fsl_buffer * const dest,
                               FILE * const src ){
  return fsl_buffer_fill_from( dest, fsl_input_f_FILE, src );
}          


int fsl_buffer_fill_from_filename( fsl_buffer * const dest,
                                   char const * filename ){
  buff__errcheck(dest);
  int rc;
  FILE * src;
  fsl_fstat st = fsl_fstat_empty;
  /* This stat() is only an optimization to reserve all needed
     memory up front.
  */
  rc = fsl_stat( filename, &st, 1 );
  if(!rc && st.size>0){
    rc = fsl_buffer_reserve(dest, st.size +1/*NUL terminator*/);
    if(rc) return rc;
  } /* Else it might not be a real file, e.g. "-", so we'll try anyway... */
  src = fsl_fopen(filename,"rb");
  if(!src) rc = fsl_errno_to_rc(errno, FSL_RC_IO);
  else {
    rc = fsl_buffer_fill_from( dest, fsl_input_f_FILE, src );
    fsl_fclose(src);
  }
  return rc;
}

void fsl_buffer_swap( fsl_buffer * const left, fsl_buffer * const right ){
  fsl_buffer const tmp = *left;
  *left = *right;
  *right = tmp;
}

void fsl_buffer_swap_free( fsl_buffer * const left, fsl_buffer * const right, int clearWhich ){
  fsl_buffer_swap(left, right);
  if(0 != clearWhich) fsl_buffer_reserve((clearWhich<0) ? left : right, 0);
}

int fsl_buffer_copy( fsl_buffer * const dest,
                     fsl_buffer const * const src ){
  fsl_buffer_reuse(dest);
  return src->used
    ? fsl_buffer_append( dest, src->mem, src->used )
    : 0;
}

int fsl_buffer_delta_apply2( fsl_buffer const * const orig,
                             fsl_buffer const * const pDelta,
                             fsl_buffer * const pTarget,
                             fsl_error * const pErr){
  buff__errcheck(orig);
  else buff__errcheck(pDelta);
  else buff__errcheck(pTarget);
  int rc;
  fsl_size_t n = 0;
  fsl_buffer out = fsl_buffer_empty;
  rc = fsl_delta_applied_size( pDelta->mem, pDelta->used, &n);
  if(rc){
    if(pErr){
      fsl_error_set(pErr, rc, "fsl_delta_applied_size() failed.");
    }
    return rc;
  }
  assert(n>0);
  rc = fsl_buffer_resize( &out, n );
  if(0==rc){
    rc = fsl_delta_apply2( orig->mem, orig->used,
                          pDelta->mem, pDelta->used,
                          out.mem, pErr);
    if(0==rc) fsl_buffer_swap(&out, pTarget);
  }
  fsl_buffer_clear(&out);
  return rc;
}

int fsl_buffer_delta_apply( fsl_buffer const * const orig,
                            fsl_buffer const * const pDelta,
                            fsl_buffer * const pTarget){
  return fsl_buffer_delta_apply2(orig, pDelta, pTarget, NULL);
}

void fsl_buffer_defossilize( fsl_buffer * const b ){
  fsl_bytes_defossilize( b->mem, &b->used );
}

int fsl_buffer_to_filename( fsl_buffer const * const b, char const * fname ){
  buff__errcheck(b);
  FILE * f;
  int rc = 0;
  if(!b || !fname) return FSL_RC_MISUSE;
  f = fsl_fopen(fname, "wb");
  if(!f) rc = fsl_errno_to_rc(errno, FSL_RC_IO);
  else{
    if(b->used) {
      size_t const frc = fwrite(b->mem, b->used, 1, f);
      rc = (1==frc) ? 0 : FSL_RC_IO;
    }
    fsl_fclose(f);
  }
  return rc;
}

int fsl_buffer_delta_create( fsl_buffer const * const src,
                             fsl_buffer const * const newVers,
                             fsl_buffer * const delta){
  if((src == newVers)
     || (src==delta)
     || (newVers==delta)) return FSL_RC_MISUSE;
  int rc = fsl_buffer_reserve( delta, newVers->used + 60 );
  if(!rc){
    delta->used = 0;
    rc = fsl_delta_create( src->mem, src->used,
                           newVers->mem, newVers->used,
                           delta->mem, &delta->used );
  }
  return rc;
}


int fsl_output_f_buffer( void * state, void const * src, fsl_size_t n ){
  return fsl_buffer_append((fsl_buffer*)state, src, n);
}

int fsl_finalizer_f_buffer( void * state __unused, void * mem ){
  fsl_buffer * b = (fsl_buffer*)mem;
  fsl_buffer_reserve(b, 0);
  *b = fsl_buffer_empty;
  return 0;
}

int fsl_buffer_strftime(fsl_buffer * const b, char const * format,
                        const struct tm *timeptr){
  if(!b || !format || !*format || !timeptr) return FSL_RC_MISUSE;
  else{
    enum {BufSize = 128};
    char buf[BufSize];
    fsl_size_t const len = fsl_strftime(buf, BufSize, format, timeptr);
    return len ? fsl_buffer_append(b, buf, (fsl_int_t)len) : FSL_RC_RANGE;
  }
}

int fsl_buffer_stream_lines(fsl_output_f fTo, void * const toState,
                            fsl_buffer * const pFrom, fsl_size_t N){
  buff__errcheck(pFrom);
  char *z = (char *)pFrom->mem;
  fsl_size_t i = pFrom->cursor;
  fsl_size_t n = pFrom->used;
  fsl_size_t cnt = 0;
  int rc = 0;
  if( N==0 ) return 0;
  while( i<n ){
    if( z[i]=='\n' ){
      cnt++;
      if( cnt==N ){
        i++;
        break;
      }
    }
    i++;
  }
  if( fTo ){
    rc = fTo(toState, &pFrom->mem[pFrom->cursor], i - pFrom->cursor);
  }
  if(!rc){
    pFrom->cursor = i;
  }
  return rc;
}


int fsl_buffer_copy_lines(fsl_buffer * const pTo,
                          fsl_buffer * const pFrom,
                          fsl_size_t N){
#if 1
  if(pTo) buff__errcheck(pTo);
  return fsl_buffer_stream_lines( pTo ? fsl_output_f_buffer : NULL, pTo,
                                  pFrom, N );
#else
  char *z = (char *)pFrom->mem;
  fsl_size_t i = pFrom->cursor;
  fsl_size_t n = pFrom->used;
  fsl_size_t cnt = 0;
  int rc = 0;
  if( N==0 ) return 0;
  while( i<n ){
    if( z[i]=='\n' ){
      ++cnt;
      if( cnt==N ){
        ++i;
        break;
      }
    }
    ++i;
  }
  if( pTo ){
    rc = fsl_buffer_append(pTo, &pFrom->mem[pFrom->cursor], i - pFrom->cursor);
  }
  if(!rc){
    pFrom->cursor = i;
  }
  return rc;
#endif
}

int fsl_input_f_buffer( void * state, void * dest, fsl_size_t * n ){
  fsl_buffer * const b = (fsl_buffer*)state;
  buff__errcheck(b);
  fsl_size_t const from = b->cursor;
  fsl_size_t to;
  fsl_size_t c;
  if(from >= b->used){
    *n = 0;
    return 0;
  }
  to = from + *n;
  if(to>b->used) to = b->used;
  c = to - from;
  if(c){
    memcpy(dest, b->mem+from, c);
    b->cursor += c;
  }
  *n = c;
  return 0;
}

int fsl_buffer_compare_file( fsl_buffer const * b, char const * zFile ){
  int rc;
  fsl_fstat fst = fsl_fstat_empty;
  rc = fsl_stat(zFile, &fst, 1);
  if(rc || (FSL_FSTAT_TYPE_FILE != fst.type)) return -1;
  else if(b->used < fst.size) return -1;
  else if(b->used > fst.size) return 1;
  else{
#if 1
    FILE * f;
    f = fsl_fopen(zFile,"r");
    if(!f) rc = -1;
    else{
      fsl_buffer fc = *b /* so fsl_input_f_buffer() can manipulate its
                            cursor */;
      rc = fsl_stream_compare(fsl_input_f_buffer, &fc,
                              fsl_input_f_FILE, f);
      assert(fc.mem==b->mem);
      fsl_fclose(f);
    }

#else
    fsl_buffer fc = fsl_buffer_empty;
    rc = fsl_buffer_fill_from_filename(&fc, zFile);
    if(rc){
      rc = -1;
    }else{
      rc = fsl_buffer_compare(b, &fc);
    }
    fsl_buffer_clear(&fc);
#endif
    return rc;
  }
}

char * fsl_buffer_take(fsl_buffer * const b){
  char * z = NULL;
  if(0==buf__materialize(b,0)){
    z = (char *)b->mem;
    *b = fsl_buffer_empty;
  }
  return z;
}

fsl_size_t fsl_buffer_seek(fsl_buffer * const b, fsl_int_t offset,
                           fsl_buffer_seek_e  whence){
  int64_t c = (int64_t)b->cursor;
  switch(whence){
    case FSL_BUFFER_SEEK_SET: c = offset;
    __attribute__ ((fallthrough));
    case FSL_BUFFER_SEEK_CUR: c = (int64_t)b->cursor + offset; break;
    case FSL_BUFFER_SEEK_END:
      c = (int64_t)b->used + offset;
      /* ^^^^^ fossil(1) uses (used + offset - 1) but

         That seems somewhat arguable because (used + 0 - 1) is at the
         last-written byte (or 1 before the begining), not the
         one-past-the-end point (which corresponds to the
         "end-of-file" described by the fseek() man page). It then
         goes on, in other algos, to operate on that final byte using
         that position, e.g.  blob_read() after a seek-to-end would
         read that last byte, rather than treating the buffer as being
         at the end.

         So... i'm going to naively remove that -1 bit.
      */
      break;
  }
  if(!b->used || c<0) b->cursor = 0;
  else if((fsl_size_t)c > b->used) b->cursor = b->used;
  else b->cursor = (fsl_size_t)c;
  return b->cursor;
}

fsl_size_t fsl_buffer_tell(fsl_buffer const * const b){
  return b->cursor;
}

void fsl_buffer_rewind(fsl_buffer * const b){
  b->cursor = 0;
}

int fsl_id_bag_to_buffer(fsl_id_bag const * bag, fsl_buffer * const b,
                         char const * separator){
  int i = 0;
  fsl_int_t const sepLen = (fsl_id_t)fsl_strlen(separator);
  fsl_buffer_reserve(b, b->used + (bag->entryCount * 7)
                     + (bag->entryCount * sepLen));
  for(fsl_id_t e = fsl_id_bag_first(bag);
      !b->errCode && e; e = fsl_id_bag_next(bag, e)){
    if(i++) fsl_buffer_append(b, separator, sepLen);
    fsl_buffer_appendf(b, "%" FSL_ID_T_PFMT, e);
  }
  return b->errCode;
}

int fsl_buffer_append_tcl_literal(fsl_buffer * const b,
                                  bool escapeSquigglies,
                                  char const * z, fsl_int_t n){
  buff__errcheck(b);
  int rc;
  if(n<0) n = fsl_strlen(z);
  fsl_buffer_append(b, "\"", 1);
  for(fsl_int_t i=0; 0==b->errCode && i<n; ++i){
    char c = z[i];
    bool skipSlash = false;
    switch( c ){
      case '\r':  c = 'r'; goto slash;
      case '}': case '{': skipSlash = !escapeSquigglies;
        /* fall through */
      case '[':
      case ']':
      case '$':
      case '"':
      case '\\':
      slash:
        if(!skipSlash && (rc = fsl_buffer_append(b, "\\", 1))) break;
        /* fall through */
      default:
        fsl_buffer_append(b, &c, 1);
    }
  }
  fsl_buffer_append(b, "\"", 1);
  return b->errCode;
}

#undef MARKER
#undef buf__is_external
#undef buf__errcheck
#undef buf__materialize
/* end of file ./src/buffer.c */
/* start of file ./src/cache.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/
/*****************************************************************************
  This file some of the caching-related APIs.
*/
#include <assert.h>

/* Only for debugging */
#include <stdio.h>
#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)

bool fsl__bccache_expire_oldest(fsl__bccache * const c){
  static uint16_t const sentinel = 0xFFFF;
  uint16_t i;
  fsl_uint_t mnAge = c->nextAge;
  uint16_t mn = sentinel;
  for(i=0; i<c->used; i++){
    if( c->list[i].age<mnAge ){
      mnAge = c->list[i].age;
      mn = i;
    }
  }
  if( mn<sentinel ){
    fsl_id_bag_remove(&c->inCache, c->list[mn].rid);
    c->szTotal -= (unsigned)c->list[mn].content.capacity;
    fsl_buffer_clear(&c->list[mn].content);
    --c->used;
    c->list[mn] = c->list[c->used];
  }
  return sentinel!=mn;
}

int fsl__bccache_insert(fsl__bccache * const c, fsl_id_t rid, fsl_buffer * const pBlob){
  fsl__bccache_line *p;
  if( c->used>c->usedLimit || c->szTotal>c->szLimit ){
    fsl_size_t szBefore;
    do{
      szBefore = c->szTotal;
      fsl__bccache_expire_oldest(c);
    }while( c->szTotal>c->szLimit && c->szTotal<szBefore );
  }
  if((!c->usedLimit || !c->szLimit)
     || (c->used+1 >= c->usedLimit)){
    fsl_buffer_clear(pBlob);
    return 0;
  }
  if( c->used>=c->capacity ){
    uint16_t const cap = c->capacity ? (c->capacity*2) : 10;
    void * remem = c->list
      ? fsl_realloc(c->list, cap*sizeof(c->list[0]))
      : fsl_malloc( cap*sizeof(c->list[0]) );
    assert((c->capacity && cap<c->capacity) ? !"Numeric overflow" : 1);
    if(c->capacity && cap<c->capacity){
      fsl__fatal(FSL_RC_RANGE,"Numeric overflow. Bump "
                 "fsl__bccache::capacity to a larger int type.");
    }
    if(!remem){
      fsl_buffer_clear(pBlob) /* for consistency */;
      return FSL_RC_OOM;
    }
    c->capacity = cap;
    c->list = (fsl__bccache_line*)remem;
  }
  int const rc = fsl_id_bag_insert(&c->inCache, rid);
  if(0==rc){
    p = &c->list[c->used++];
    p->rid = rid;
    p->age = c->nextAge++;
    c->szTotal += pBlob->capacity;
    p->content = *pBlob /* Transfer ownership */;
    *pBlob = fsl_buffer_empty;
  }else{
    fsl_buffer_clear(pBlob);
  }
  return rc;
}


void fsl__bccache_clear(fsl__bccache * const c){
#if 0
  while(fsl__bccache_expire_oldest(c)){}
#else
  fsl_size_t i;
  for(i=0; i<c->used; ++i){
    fsl_buffer_clear(&c->list[i].content);
  }
#endif
  fsl_free(c->list);
  fsl_id_bag_clear(&c->missing);
  fsl_id_bag_clear(&c->available);
  fsl_id_bag_clear(&c->inCache);
  *c = fsl__bccache_empty;
}

void fsl__bccache_reset(fsl__bccache * const c){
  static const fsl__bccache_line line_empty = fsl__bccache_line_empty_m;
  fsl_size_t i;
  for(i=0; i<c->used; ++i){
    fsl_buffer_clear(&c->list[i].content);
    c->list[i] = line_empty;
  }
  c->used = 0;
  c->szTotal = 0;
  c->nextAge = 0;
  fsl_id_bag_reset(&c->missing);
  fsl_id_bag_reset(&c->available);
  fsl_id_bag_reset(&c->inCache);
}


int fsl__bccache_check_available(fsl_cx * const f, fsl_id_t rid){
  fsl_id_t srcid;
  int depth = 0;  /* Limit to recursion depth */
  static const int limit = 10000000 /* historical value */;
  int rc;
  fsl__bccache * const c = &f->cache.blobContent;
  assert(f);
  assert(c);
  assert(rid>0);
  assert(fsl_cx_db_repo(f));
  while( depth++ < limit ){  
    fsl_int_t cSize = -1;
    if( fsl_id_bag_contains(&c->missing, rid) ){
      return FSL_RC_NOT_FOUND;
    }
    else if( fsl_id_bag_contains(&c->available, rid) ){
      return 0;
    }
    else if( (cSize=fsl_content_size(f, rid)) <0){
      rc = fsl_id_bag_insert(&c->missing, rid);
      return rc ? rc : FSL_RC_NOT_FOUND;
    }
    srcid = 0;
    rc = fsl_delta_src_id(f, rid, &srcid);
    if(rc) return rc;
    else if( srcid==0 ){
      rc = fsl_id_bag_insert(&c->available, rid);
      return rc ? rc : 0;
    }
    rid = srcid;
  }
  assert(!"delta-loop in repository");
  return fsl_cx_err_set(f, FSL_RC_CONSISTENCY,
                        "Serious problem: delta-loop in repository");
}

#undef MARKER
/* end of file ./src/cache.c */
/* start of file ./src/checkin.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/

/*****************************************************************************
  This file houses the code for checkin-level APIS.
*/
#include <assert.h>


/* Only for debugging */
#include <stdio.h>
#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)


/**
   Expects f to have an opened checkout. Assumes zRelName is a
   checkout-relative simple path. It loads the file's contents and
   stores them into the blob table. If rid is not NULL, *rid is
   assigned the blob.rid (possibly new, possilbly re-used!). If uuid
   is not NULL then *uuid is assigned to the content's UUID. The *uuid
   bytes are owned by the caller, who must eventually fsl_free()
   them. If content with the same UUID already exists, it does not get
   re-imported but rid/uuid will (if not NULL) contain the old values.

   If parentRid is >0 then it must refer to the previous version of
   zRelName's content. The parent version gets deltified vs the new
   one. Note that deltification is a suggestion which the library will
   ignore if (e.g.) the parent content is already a delta of something
   else.

   The wise caller will have a transaction in place when calling this.

   Returns 0 on success. On error rid and uuid are not modified.
*/
static int fsl_checkin_import_file( fsl_cx * const f,
                                    char const * const zRelName,
                                    fsl_id_t parentRid,
                                    bool allowMergeConflict,
                                    fsl_id_t * const rid,
                                    fsl_uuid_str * const uuid){
  fsl_buffer * nbuf = fsl__cx_scratchpad(f);
  fsl_size_t const oldSize = nbuf->used;
  fsl_buffer * const fbuf = &f->cache.fileContent;
  char const * fn;
  int rc;
  fsl_id_t fnid = 0;
  fsl_id_t rcRid = 0;
  assert(!fbuf->used && "Misuse of f->fileContent");
  assert(f->ckout.dir);
  rc = fsl__repo_filename_fnid2(f, zRelName, &fnid, 1);
  if(rc) goto end;
  assert(fnid>0);

  rc = fsl_buffer_appendf(nbuf, "%s%s", f->ckout.dir, zRelName);
  nbuf->used = oldSize;
  if(rc) goto end;
  fn = fsl_buffer_cstr(nbuf) + oldSize;
  rc = fsl_buffer_fill_from_filename( fbuf, fn );
  if(rc){
    fsl_cx_err_set(f, rc, "Error %s importing file: %s",
                   fsl_rc_cstr(rc), fn);
    goto end;
  }else if(!allowMergeConflict &&
           fsl_buffer_contains_merge_marker(fbuf)){
    rc = fsl_cx_err_set(f, FSL_RC_CONFLICT,
                        "File contains a merge conflict marker: %s",
                        zRelName);
    goto end;
  }

  rc = fsl__content_put( f, fbuf, &rcRid );
  if(!rc){
    assert(rcRid > 0);
    if(parentRid>0){
      rc = fsl__content_deltify(f, parentRid, rcRid, 0);
    }
    if(!rc){
      if(rid) *rid = rcRid;
      if(uuid){
        *uuid = fsl_rid_to_uuid(f, rcRid);
        if(!*uuid) rc = (f->error.code ? f->error.code : FSL_RC_OOM);
      }
    }
  }
  end:
  fsl__cx_scratchpad_yield(f, nbuf);
  fsl__cx_content_buffer_yield(f);
  assert(0==fbuf->used);
  return rc;
}

int fsl_filename_to_vfile_ids( fsl_cx * f, fsl_id_t vid,
                               fsl_id_bag * dest, char const * zName,
                               bool changedOnly){
  fsl_stmt st = fsl_stmt_empty;
  fsl_db * const db = fsl_needs_ckout(f);
  int rc;
  fsl_buffer * sql = 0;
  if(!db) return FSL_RC_NOT_A_CKOUT;
  sql = fsl__cx_scratchpad(f);
  if(0>=vid) vid = f->ckout.rid;
  if(zName && *zName
     && !('.'==*zName && !zName[1])){
    fsl_buffer_appendf(sql,
                       "SELECT id FROM vfile WHERE vid=%"
                       FSL_ID_T_PFMT
                       " AND fsl_match_vfile_or_dir(pathname,%Q)",
                       vid, zName);
  }else{
    fsl_buffer_appendf(sql,
                       "SELECT id FROM vfile WHERE vid=%" FSL_ID_T_PFMT,
                       vid);
  }
  if(changedOnly){
    fsl_buffer_append(sql, " AND (chnged OR deleted OR rid=0 "
                      "OR (origname IS NOT NULL AND "
                      "    origname<>pathname))", -1);
  }
  rc = fsl_buffer_appendf(sql, " /* %s() */", __func__);
  if(0==rc) rc = fsl_db_prepare(db, &st, "%b", sql);
  while(!rc && (FSL_RC_STEP_ROW == (rc=fsl_stmt_step(&st)))){
    rc = fsl_id_bag_insert( dest, fsl_stmt_g_id(&st, 0) );
  }
  if(FSL_RC_STEP_DONE==rc) rc = 0;
  fsl__cx_scratchpad_yield(f, sql);
  fsl_stmt_finalize(&st);
  if(rc && !f->error.code && db->error.code){
    fsl_cx_uplift_db_error(f, db);
  }
  return rc;
}

int fsl_filename_to_vfile_id( fsl_cx * const f, fsl_id_t vid,
                              char const * zName,
                              fsl_id_t * const vfid ){
  fsl_db * db = fsl_needs_ckout(f);
  int rc;
  fsl_stmt st = fsl_stmt_empty;
  assert(db);
  if(!db) return FSL_RC_NOT_A_CKOUT;
  else if(!zName || !fsl_is_simple_pathname(zName, true)){
    return fsl_cx_err_set(f, FSL_RC_RANGE,
                          "Filename is not a \"simple\" path: %s",
                          zName);
  }
  if(0>=vid) vid = f->ckout.rid;
  rc = fsl_db_prepare(db, &st,
                      "SELECT id FROM vfile WHERE vid=%" FSL_ID_T_PFMT
                      " AND pathname=%Q %s /*%s()*/",
                      vid, zName, 
                      fsl_cx_filename_collation(f),
                      __func__);
  if(!rc){
    rc = fsl_stmt_step(&st);
    switch(rc){
      case FSL_RC_STEP_ROW:
        rc = 0;
        *vfid = fsl_stmt_g_id(&st, 0);
        break;
      case FSL_RC_STEP_DONE:
        rc = 0;
        /* fall through */
      default:
        *vfid = 0;
    }
    fsl_stmt_finalize(&st);
  }
  if(rc){
    rc = fsl_cx_uplift_db_error2(f, db, rc);
  }
  return rc;
}

/**
   Internal helper for fsl_checkin_enqueue() and
   fsl_checkin_dequeue(). Prepares, if needed, st with a query to
   fetch a vfile entry where vfile.id=vfid, then passes that name on
   to opt->callback(). Returns 0 on success.
*/
static int fsl_xqueue_callback(fsl_cx * f, fsl_db * db, fsl_stmt * st,
                               fsl_id_t vfid,
                               fsl_checkin_queue_opt const * opt){

  int rc;
  assert(opt->callback);
  if(!st->stmt){
    rc = fsl_db_prepare(db, st,
                        "SELECT pathname FROM vfile "
                        "WHERE id=?1");
    if(rc) return fsl_cx_uplift_db_error2(f, db, rc);
  }
  fsl_stmt_bind_id(st, 1, vfid);
  rc = fsl_stmt_step(st);
  switch(rc){
    case FSL_RC_STEP_ROW:{
      char const * zName = fsl_stmt_g_text(st, 0, NULL);
      rc = opt->callback(zName, opt->callbackState);
      break;
    }
    case FSL_RC_STEP_DONE:
      rc = fsl_cx_err_set(f, rc, "Very unexpectedly did not find "
                          "vfile.id which we just found.");
      break;
    default:
      rc = fsl_cx_uplift_db_error2(f, db, rc);
      break;
  }
  fsl_stmt_reset(st);
  return rc;
}

int fsl_checkin_enqueue(fsl_cx * f, fsl_checkin_queue_opt const * opt){
  fsl_db * const db = fsl_needs_ckout(f);
  if(!db) return FSL_RC_NOT_A_CKOUT;
  fsl_buffer * const canon = opt->vfileIds ? 0 : fsl__cx_scratchpad(f);
  fsl_stmt qName = fsl_stmt_empty;
  fsl_id_bag _vfileIds = fsl_id_bag_empty;
  fsl_id_bag const * const vfileIds =
    opt->vfileIds ? opt->vfileIds : &_vfileIds;
  int rc = fsl_db_transaction_begin(db);
  if(rc) return fsl_cx_uplift_db_error2(f, db, rc);
  if(opt->vfileIds){
    if(!fsl_id_bag_count(opt->vfileIds)){
      rc = fsl_cx_err_set(f, FSL_RC_MISUSE,
                          "fsl_checkin_queue_opt::vfileIds "
                          "may not be empty.");
      goto end;
    }
  }else{
    rc = fsl_ckout_filename_check(f, opt->relativeToCwd,
                                  opt->filename, canon);
    if(rc) goto end;
    fsl_buffer_strip_slashes(canon);
  }
  if(opt->scanForChanges){
    rc = fsl_vfile_changes_scan(f, -1, 0);
    if(rc) goto end;
  }
  if(opt->vfileIds){
    assert(vfileIds == opt->vfileIds);
  }else{
    assert(vfileIds == &_vfileIds);
    rc = fsl_filename_to_vfile_ids(f, 0, &_vfileIds,
                                   fsl_buffer_cstr(canon),
                                   opt->onlyModifiedFiles);
  }
  if(rc) goto end;
  /* Walk through each found ID and queue up any which are not already
     enqueued. */
  for(fsl_id_t vfid = fsl_id_bag_first(vfileIds);
      !rc && vfid; vfid = fsl_id_bag_next(vfileIds, vfid)){
    fsl_size_t const entryCount = f->ckin.selectedIds.entryCount;
    rc = fsl_id_bag_insert(&f->ckin.selectedIds, vfid);
    if(!rc
       && entryCount < f->ckin.selectedIds.entryCount
       /* Was enqueued */
       && opt->callback){
      rc = fsl_xqueue_callback(f, db, &qName, vfid, opt);
    }
  }
  end:
  if(opt->vfileIds){
    assert(!canon);
    assert(!_vfileIds.list);
  }else{
    assert(canon);
    fsl__cx_scratchpad_yield(f, canon);
    fsl_id_bag_clear(&_vfileIds);
  }
  fsl_stmt_finalize(&qName);
  if(rc) fsl_db_transaction_rollback(db);
  else{
    rc = fsl_cx_uplift_db_error2(f, db, fsl_db_transaction_commit(db));
  }
  return rc;
}

int fsl_checkin_dequeue(fsl_cx * const f, fsl_checkin_queue_opt const * opt){
  fsl_db * const db = fsl_needs_ckout(f);
  if(!db) return FSL_RC_NOT_A_CKOUT;
  int rc = fsl_db_transaction_begin(db);
  if(rc) return fsl_cx_uplift_db_error2(f, db, rc);
  fsl_id_bag list = fsl_id_bag_empty;
  fsl_buffer * canon = 0;
  char const * fn;
  fsl_stmt qName = fsl_stmt_empty;
  if(opt->filename && *opt->filename){
    canon = fsl__cx_scratchpad(f);
    rc = fsl_ckout_filename_check(f, opt->relativeToCwd,
                                  opt->filename, canon);
    if(rc) goto end;
    else fsl_buffer_strip_slashes(canon);
  }
  fn = canon ? fsl_buffer_cstr(canon) : opt->filename;
  rc = fsl_filename_to_vfile_ids(f, 0, &list, fn, false);
  if(!rc && list.entryCount){
    /* Walk through each found ID and dequeue up any which are
       enqueued. */
    for( fsl_id_t nid = fsl_id_bag_first(&list);
         !rc && nid;
         nid = fsl_id_bag_next(&list, nid)){
      if(fsl_id_bag_remove(&f->ckin.selectedIds, nid)
         && opt->callback){
        rc = fsl_xqueue_callback(f, db, &qName, nid, opt);
      }
    }
  }
  end:
  if(canon) fsl__cx_scratchpad_yield(f, canon);
  fsl_stmt_finalize(&qName);
  fsl_id_bag_clear(&list);
  if(rc) fsl_db_transaction_rollback(db);
  else{
    rc = fsl_cx_uplift_db_error2(f, db, fsl_db_transaction_commit(db));
  }
  return rc;
}

bool fsl_checkin_is_enqueued(fsl_cx * const f, char const * zName,
                             bool relativeToCwd){
  if(!zName || !*zName) return false;
  else if(!fsl_cx_db_ckout(f)) return false;
  else if(!f->ckin.selectedIds.entryCount){
    /* Behave like fsl_is_enqueued() SQL function. */
    return true;
  }
  else {
    bool rv = false;
    fsl_buffer * const canon = fsl__cx_scratchpad(f);
    int rc = fsl_ckout_filename_check(f, relativeToCwd, zName, canon);
    if(!rc){
      fsl_id_t vfid = 0;
      rc = fsl_filename_to_vfile_id(f, 0, fsl_buffer_cstr(canon),
                                    &vfid);
      rv = (rc && (vfid>0))
        ? false
        : ((vfid>0)
           ? fsl_id_bag_contains(&f->ckin.selectedIds, vfid)
           /* ^^^^ asserts that arg2!=0*/
           : false);
    }
    fsl__cx_scratchpad_yield(f, canon);
    return rv;
  }
}

void fsl_checkin_discard(fsl_cx * const f){
  if(f){
    fsl_id_bag_clear(&f->ckin.selectedIds);
    fsl_deck_finalize(&f->ckin.mf);
  }
}

/**
   Adds the given rid to the "unsent" db list, Returns 0 on success,
   updates f's error state on error.
*/
static int fsl_checkin_add_unsent(fsl_cx * f, fsl_id_t rid){
  fsl_db * const r = fsl_cx_db_repo(f);
  int rc;
  assert(r);
  rc = fsl_db_exec(r,"INSERT OR IGNORE INTO unsent "
                   "VALUES(%" FSL_ID_T_PFMT ")", rid);
  if(rc){
    fsl_cx_uplift_db_error(f, r);
  }
  return rc;
}

/**
   Calculates the F-cards for deck d based on the commit file
   selection list and the contents of the vfile table (where vid==the
   vid parameter). vid is the version to check against, and this code
   assumes that the vfile table has been populated with that version
   and its state represents a recent scan (with no filesystem-level
   changes made since the scan).

   If pBaseline is not NULL then d is calculated as being a delta
   from pBaseline, but d->B is not modified by this routine.

   On success, d->F.list will contain "all the F-cards it needs."

   If changeCount is not NULL, then on success it is set to the number
   of F-cards added to d due to changes queued via the checkin process
   (as opposed to those added solely for delta inheritance reasons).
*/
static
int fsl_checkin_calc_F_cards2( fsl_cx * f, fsl_deck * d,
                               fsl_deck * pBaseline, fsl_id_t vid,
                               fsl_size_t * changeCount,
                               fsl_checkin_opt const * ciOpt){
  int rc = 0;
  fsl_db * dbR = fsl_needs_repo(f);
  fsl_db * dbC = fsl_needs_ckout(f);
  fsl_stmt stUpdateFileRid = fsl_stmt_empty;
  fsl_stmt stmt = fsl_stmt_empty;
  fsl_stmt * q = &stmt;
  char * fUuid = NULL;
  fsl_card_F const * pFile = NULL;
  fsl_size_t changeCounter = 0;
  if(!f) return FSL_RC_MISUSE;
  else if(!dbR) return FSL_RC_NOT_A_REPO;
  else if(!dbC) return FSL_RC_NOT_A_CKOUT;
  assert( (!pBaseline || !pBaseline->B.uuid) && "Baselines must not have a baseline." );
  assert( d->B.baseline ? (!pBaseline || pBaseline==d->B.baseline) : 1 );
  assert(vid>=0);
#define RC if(rc) goto end

  if(pBaseline){
    assert(!d->B.baseline);
    assert(0!=vid);
    rc = fsl_deck_F_rewind(pBaseline);
    RC;
    fsl_deck_F_next( pBaseline, &pFile );
  }

  rc = fsl_db_prepare(dbC, &stUpdateFileRid,
                      "UPDATE vfile SET mrid=?1, rid=?1, "
                      "mhash=NULL WHERE id=?2");
  RC;

  rc = fsl_db_prepare( dbC, q,
                       "SELECT "
                       /*0*/"fsl_is_enqueued(vf.id) as isSel, "
                       /*1*/"vf.id,"
                       /*2*/"vf.vid,"
                       /*3*/"vf.chnged,"
                       /*4*/"vf.deleted,"
                       /*5*/"vf.isexe,"
                       /*6*/"vf.islink,"
                       /*7*/"vf.rid,"
                       /*8*/"mrid,"
                       /*9*/"pathname,"
                       /*10*/"origname, "
                       /*11*/"b.rid, "
                       /*12*/"b.uuid "
                       "FROM vfile vf LEFT JOIN blob b ON vf.mrid=b.rid "
                       "WHERE"
                       " vf.vid=%"FSL_ID_T_PFMT" AND"
#if 0
                       /* Historical (fossil(1)). This introduces an interesting
                          corner case which i would like to avoid here because
                          it causes a "no files changed" error in the checkin
                          op. The behaviour is actually correct (and the deletion
                          is picked up) but fsl_checkin_commit() has no mechanism
                          for catching this particular case. So we'll try a
                          slightly different approach...
                       */
                       " (NOT deleted OR NOT isSel)"
#else
                       " ((NOT deleted OR NOT isSel)"
                       "  OR (deleted AND isSel))" /* workaround to allow
                                                     us to count deletions via
                                                     changeCounter. */
#endif
                       " ORDER BY fsl_if_enqueued(vf.id, pathname, origname)",
                       (fsl_id_t)vid);
  RC;
  /* MARKER(("SQL:\n%s\n", (char const *)q->sql.mem)); */
  while( FSL_RC_STEP_ROW==fsl_stmt_step(q) ){
    int const isSel = fsl_stmt_g_int32(q,0);
    fsl_id_t const id = fsl_stmt_g_id(q,1);
#if 0
    fsl_id_t const vid = fsl_stmt_g_id(q,2);
#endif
    int const changed = fsl_stmt_g_int32(q,3);
    int const deleted = fsl_stmt_g_int32(q,4);
    int const isExe = fsl_stmt_g_int32(q,5);
    int const isLink = fsl_stmt_g_int32(q,6);
    fsl_id_t const rid = fsl_stmt_g_id(q,7);
    fsl_id_t const mergeRid = fsl_stmt_g_id(q,8);
    char const * zName = fsl_stmt_g_text(q, 9, NULL);
    char const * zOrig = fsl_stmt_g_text(q, 10, NULL);
    fsl_id_t const frid = fsl_stmt_g_id(q,11);
    char const * zUuid = fsl_stmt_g_text(q, 12, NULL);
    fsl_fileperm_e perm = FSL_FILE_PERM_REGULAR;
    int cmp;
    fsl_id_t fileBlobRid = rid;
    int const renamed = (zOrig && *zOrig) ? fsl_strcmp(zName,zOrig) : 0
      /* For some as-yet-unknown reason, some fossil(1) code
         sets (origname=pathname WHERE origname=NULL). e.g.
         the 'mv' command does that.
      */;
    if(zOrig && !renamed) zOrig = NULL;
    fUuid = NULL;
    if(!isSel && !zUuid){
      assert(!rid);
      assert(!mergeRid);
      /* An unselected ADDed file. Skip it. */
      continue;
    }

    if(isExe) perm = FSL_FILE_PERM_EXE;
    else if(isLink){
      fsl__fatal(FSL_RC_NYI, "This code does not yet deal "
                "with symlinks. file: %s", zName)
        /* does not return */;
      perm = FSL_FILE_PERM_LINK;
    }
    /*
      TODO: symlinks
    */

    if(!f->cache.markPrivate){
      rc = fsl_content_make_public(f, frid);
      if(rc) break;
    }

#if 0
    if(mergeRid && (mergeRid != rid)){
      fsl__fatal(FSL_RC_NYI, "This code does not yet deal "
                "with merges. file: %s", zName)
        /* does not return */;
    }
#endif
    while(pFile && fsl_strcmp(pFile->name, zName)<0){
      /* Baseline has files with lexically smaller names.
         Interesting corner case:

         f-rm th1ish/makefile.gnu
         f-checkin ... th1ish/makefile.gnu

         makefile.gnu does not get picked up by the historical query
         but gets picked up here. We really need to ++changeCounter in
         that case, but we don't know we're in that case because we're
         now traversing a filename which is not in the result set.
         The end result (because we don't increment changeCounter) is
         that fsl_checkin_commit() thinks we have no made any changes
         and errors out. If we ++changeCounter for all deletions we
         have a different corner case, where a no-change commit is not
         seen as such because we've counted deletions from (other)
         versions between the baseline and the checkout.
      */
      rc = fsl_deck_F_add(d, pFile->name, NULL, pFile->perm, NULL);
      if(rc) break;
      fsl_deck_F_next(pBaseline, &pFile);
    }
    if(rc) goto end;
    else if(isSel && (changed || deleted || renamed)){
      /* MARKER(("isSel && (changed||deleted||renamed): %s\n", zName)); */
      ++changeCounter;
      if(deleted){
        zOrig = NULL;
      }else if(changed){
        rc = fsl_checkin_import_file(f, zName, rid,
                                     ciOpt->allowMergeConflict,
                                     &fileBlobRid, &fUuid);
        if(!rc) rc = fsl_checkin_add_unsent(f, fileBlobRid);
        RC;
        /* MARKER(("New content: %d / %s / %s\n", (int)fileBlobRid, fUuid, zName)); */
        if(0 != fsl_uuidcmp(zUuid, fUuid)){
          zUuid = fUuid;
        }
        fsl_stmt_reset(&stUpdateFileRid);
        fsl_stmt_bind_id(&stUpdateFileRid, 1, fileBlobRid);
        fsl_stmt_bind_id(&stUpdateFileRid, 2, id);
        if(FSL_RC_STEP_DONE!=fsl_stmt_step(&stUpdateFileRid)){
          rc = fsl_cx_uplift_db_error(f, stUpdateFileRid.db);
          assert(rc);
          goto end;
        }
      }else{
        assert(renamed);
        assert(zOrig);
      }
    }
    assert(!rc);
    cmp = 1;
    if(!pFile
       || (cmp = fsl_strcmp(pFile->name,zName))!=0
       /* ^^^^ the cmp assignment must come right after (!pFile)! */
       || deleted
       || (perm != pFile->perm)/* permissions change */
       || fsl_strcmp(pFile->uuid, zUuid)!=0
       /* ^^^^^ file changed somewhere between baseline and delta */
       ){
      if(isSel && deleted){
        if(pBaseline /* d is-a delta */){
          /* Deltas mark deletions with F-cards having only 
             a file name (no UUID or permission).
          */
          rc = fsl_deck_F_add(d, zName, NULL, perm, NULL);
        }/*else elide F-card to mark a deletion in a baseline.*/
      }else{
        if(zOrig && !isSel){
          /* File is renamed in vfile but is not being committed, so
             make sure we use the original name for the F-card.
          */
          zName = zOrig;
          zOrig = NULL;
        }
        assert(zUuid);
        assert(fileBlobRid);
        if( !zOrig || !renamed ){
          rc = fsl_deck_F_add(d, zName, zUuid, perm, NULL);
        }else{
          /* Rename this file */
          rc = fsl_deck_F_add(d, zName, zUuid, perm, zOrig);
        }
      }
    }
    fsl_free(fUuid);
    fUuid = NULL;
    RC;
    if( 0 == cmp ){
      fsl_deck_F_next(pBaseline, &pFile);
    }
  }/*while step()*/

  while( !rc && pFile ){
    /* Baseline has remaining files with lexically larger names. Let's import them. */
    rc = fsl_deck_F_add(d, pFile->name, NULL, pFile->perm, NULL);
    if(!rc) fsl_deck_F_next(pBaseline, &pFile);
  }

  end:
#undef RC
  fsl_free(fUuid);
  fsl_stmt_finalize(q);
  fsl_stmt_finalize(&stUpdateFileRid);
  if(!rc && changeCount) *changeCount = changeCounter;
  return rc;
}

/**
   Cancels all symbolic tags (branches) on the given version by
   adding one T-card to d for each active branch tag set on vid.
   When creating a branch, d would represent the branch and vid
   would be the version being branched from.

   Returns 0 on success.
*/
static int fsl_cancel_sym_tags( fsl_deck * d, fsl_id_t vid ){
  int rc;
  fsl_stmt q = fsl_stmt_empty;
  fsl_db * db = fsl_needs_repo(d->f);
  assert(db);
  rc = fsl_db_prepare(db, &q,
                      "SELECT tagname FROM tagxref, tag"
                      " WHERE tagxref.rid=%"FSL_ID_T_PFMT
                      " AND tagxref.tagid=tag.tagid"
                      "   AND tagtype>0 AND tagname GLOB 'sym-*'"
                      " ORDER BY tagname",
                      (fsl_id_t)vid);
  while( !rc && (FSL_RC_STEP_ROW==fsl_stmt_step(&q)) ){
    const char *zTag = fsl_stmt_g_text(&q, 0, NULL);
    rc = fsl_deck_T_add(d, FSL_TAGTYPE_CANCEL,
                        NULL, zTag, "Cancelled by branch.");
  }
  fsl_stmt_finalize(&q);
  return rc;
}

#if 0
static int fsl_leaf_set( fsl_cx * f, fsl_id_t rid, char isLeaf ){
  int rc;
  fsl_stmt * st = NULL;
  fsl_db * db = fsl_needs_repo(f);
  assert(db);
  rc = fsl_db_prepare_cached(db, &st, isLeaf
                             ? "INSERT OR IGNORE INTO leaf(rid) VALUES(?)"
                             : "DELETE FROM leaf WHERE rid=?");
  if(!rc){
    fsl_stmt_bind_id(st, 1, rid);
    fsl_stmt_step(st);
    fsl_stmt_cached_yield(st);
  }
  if(rc){
    fsl_cx_uplift_db_error(f, db);
  }
  return rc;
}
#endif

/**
   Checks vfile for any files (where chnged in (2,3,4,5)), i.e.
   having something to do with a merge. If either all of those
   changes are enqueued for checkin, or none of them are, then
   this function returns 0, otherwise it sets f's error
   state and returns non-0.
*/
static int fsl_check_for_partial_merge(fsl_cx * f){
  if(!f->ckin.selectedIds.entryCount){
    /* All files are considered enqueued. */
    return 0;
  }else{
    fsl_db * db = fsl_cx_db_ckout(f);
    int32_t counter = 0;
    int rc =
      fsl_db_get_int32(db, &counter,
                       "SELECT COUNT(*) FROM ("
#if 1
                       "SELECT DISTINCT fsl_is_enqueued(id)"
                       " FROM vfile WHERE chnged IN (2,3,4,5)"
#else
                       "SELECT fsl_is_enqueued(id) isSel "
                       "FROM vfile WHERE chnged IN (2,3,4,5) "
                       "GROUP BY isSel"
#endif
                       ")"
                       );
    /**
       Result is 0 if no merged files are in vfile, 1 row if isSel is
       the same for all merge-modified files, and 2 if there is a mix
       of selected/unselected merge-modified files.
     */
    if(!rc && (counter>1)){
      assert(2==counter);
      rc = fsl_cx_err_set(f, FSL_RC_MISUSE,
                          "Only Chuck Norris can commit "
                          "a partial merge. Commit either all "
                          "or none of it.");
    }
    return rc;
  }
}

/**
   Populates d with the contents for a FSL_SATYPE_CHECKIN manifest
   based on repository version basedOnVid.

   d is the deck to populate.

   basedOnVid must currently be f->ckout.rid OR the vfile table must
   be current for basedOnVid (see fsl_vfile_changes_scan() and
   fsl_vfile_load()). It "should" work with basedOnVid==0 but
   that's untested so far.

   opt is the options object passed to fsl_checkin_commit().
*/
static int fsl_checkin_calc_manifest( fsl_cx * f, fsl_deck * d,
                                      fsl_id_t basedOnVid,
                                      fsl_checkin_opt const * opt ){
  int rc;
  fsl_db * dbR = fsl_cx_db_repo(f);
  fsl_db * dbC = fsl_cx_db_ckout(f);
  fsl_stmt q = fsl_stmt_empty;
  fsl_deck dBase = fsl_deck_empty;
  char const * zColor;
  int deltaPolicy = opt->deltaPolicy;
  assert(d->f == f);
  assert(FSL_SATYPE_CHECKIN==d->type);
#define RC if(rc) goto end
  /* assert(basedOnVid>0); */
  rc = (opt->message && *opt->message)
    ? fsl_deck_C_set( d, opt->message, -1 )
    : fsl_cx_err_set(f, FSL_RC_MISSING_INFO,
                     "Cowardly refusing to commit with "
                     "empty checkin comment.");
  RC;

  if(deltaPolicy!=0 && fsl_repo_forbids_delta_manifests(f)){
    deltaPolicy = 0;
  }else if(deltaPolicy<0 && f->cache.seenDeltaManifest<=0){
    deltaPolicy = 0;
  }
  {
    char const * zUser = opt->user ? opt->user : fsl_cx_user_get(f);
    rc = (zUser && *zUser)
      ? fsl_deck_U_set( d, zUser )
      : fsl_cx_err_set(f, FSL_RC_MISSING_INFO,
                       "Cowardly refusing to commit without "
                       "a user name.");
    RC;
  }

  rc = fsl_check_for_partial_merge(f);
  RC;

  rc = fsl_deck_D_set( d, (opt->julianTime>0)
                       ? opt->julianTime
                       : fsl_db_julian_now(dbR) );
  RC;

  if(opt->messageMimeType && *opt->messageMimeType){
    rc = fsl_deck_N_set( d, opt->messageMimeType, -1 );
    RC;
  }


  { /* F-cards */
    static char const * errNoFilesMsg =
      "No files have changed. Cowardly refusing to commit.";
    static int const errNoFilesRc = FSL_RC_NOOP;
    fsl_deck * pBase = NULL /* baseline for delta generation purposes */;
    fsl_size_t szD = 0, szB = 0 /* see commentary below */;
    if(basedOnVid && deltaPolicy!=0){
      /* Figure out a baseline for a delta manifest... */
      fsl_uuid_str bUuid = NULL /* UUID for d's B-card */;
      rc = fsl_deck_load_rid(f, &dBase, basedOnVid, FSL_SATYPE_CHECKIN);
      RC;
      if(dBase.B.uuid){
        /* dBase is a delta. Let's use its baseline for manifest
           generation. */
        fsl_id_t const baseRid = fsl_uuid_to_rid(f, dBase.B.uuid);
        fsl_deck_finalize(&dBase);
        if(baseRid>0){
          rc = fsl_deck_load_rid(f, &dBase, baseRid,
                                 FSL_SATYPE_CHECKIN);
        }else{
          rc = fsl_cx_err_get(f, NULL, NULL);
          assert(0!=rc);
        }
        RC;
      }else{
        /* dBase version is a suitable baseline. */
        bUuid = fsl_rid_to_uuid(f, basedOnVid);
        if(!bUuid){
          assert(f->error.code);
          rc = f->error.code;
          RC;
        }
      }
      /* MARKER(("Baseline = %d / %s\n", (int)pBase->rid, pBase->uuid)); */
      assert(dBase.B.uuid || bUuid);
      rc = fsl_deck_B_set(d, dBase.B.uuid ? dBase.B.uuid : bUuid);
      fsl_free(bUuid);
      RC;
      pBase = &dBase;
    }
    rc = fsl_checkin_calc_F_cards2(f, d, pBase, basedOnVid,
                                   &szD, opt);
    /*MARKER(("szD=%d\n", (int)szD));*/
    RC;
    if(basedOnVid && !szD){
      rc = fsl_cx_err_set(f, errNoFilesRc, errNoFilesMsg);
      goto end;
    }
    szB = pBase ? pBase->F.used : 0;
    /* The following text was copied verbatim from fossil(1). It does
       not apply 100% here (because we use a slightly different
       manifest generation approach) but it clearly describes what's
       going on after the comment block....
    */
    /*
    ** At this point, two manifests have been constructed, either of
    ** which would work for this checkin.  The first manifest (held
    ** in the "manifest" variable) is a baseline manifest and the second
    ** (held in variable named "delta") is a delta manifest.  The
    ** question now is: which manifest should we use?
    **
    ** Let B be the number of F-cards in the baseline manifest and
    ** let D be the number of F-cards in the delta manifest, plus one for
    ** the B-card.  (B is held in the szB variable and D is held in the
    ** szD variable.)  Assume that all delta manifests adds X new F-cards.
    ** Then to minimize the total number of F- and B-cards in the repository,
    ** we should use the delta manifest if and only if:
    **
    **      D*D < B*X - X*X
    **
    ** X is an unknown here, but for most repositories, we will not be
    ** far wrong if we assume X=3.
    */
    ++szD /* account for the d->B card */;
    if(pBase){
      /* For this calculation, i believe the correct approach is to
         simply count the F-cards, including those changed between the
         baseline and the delta, as opposed to only those changed in
         the delta itself.
      */
      szD = 1 + d->F.used;
    }
    /* MARKER(("szB=%d szD=%d\n", (int)szB, (int)szD)); */
    if(pBase && (deltaPolicy<0/*non-force-mode*/
                 && !(((int)(szD*szD)) < (((int)szB*3)-9))
                 /* ^^^ see comments above */
                 )
       ){
      /* Too small of a delta to be worth it. Re-calculate
         F-cards with no baseline.

         Maintenance reminder: i initially wanted to update vfile's
         status incrementally as F-cards are calculated, but this
         discard/retry breaks on the retry because vfile's state has
         been modified. Thus instead of updating vfile incrementally,
         we re-scan it after the checkin completes.
      */
      fsl_deck tmp = fsl_deck_empty;
      /* Free up d->F using a kludge... */
      tmp.F = d->F;
      d->F = fsl_deck_empty.F;
      fsl_deck_finalize(&tmp);
      fsl_deck_B_set(d, NULL);
      /* MARKER(("Delta is too big - re-calculating F-cards for a baseline.\n")); */
      szD = 0;
      rc = fsl_checkin_calc_F_cards2(f, d, NULL, basedOnVid,
                                     &szD, opt);
      RC;
      if(basedOnVid && !szD){
        rc = fsl_cx_err_set(f, errNoFilesRc, errNoFilesMsg);
        goto end;
      }
    }
  }/* F-cards */

  /* parents... */
  if( basedOnVid ){
    char * zParentUuid = fsl_rid_to_artifact_uuid(f, basedOnVid, FSL_SATYPE_CHECKIN);
    if(!zParentUuid){
      assert(f->error.code);
      rc = f->error.code
        ? f->error.code
        : fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                         "Could not find checkin UUID "
                         "for RID %"FSL_ID_T_PFMT".",
                         basedOnVid);
      goto end;
    }
    rc = fsl_deck_P_add(d, zParentUuid)
      /* pedantic side-note: we could alternately transfer ownership
         of zParentUuid by fsl_list_append()ing it to d->P, but that
         would bypass e.g. any checking that routine chooses to apply.
      */;
    fsl_free(zParentUuid);
    /* if(!rc) rc = fsl_leaf_set(f, basedOnVid, 0); */
    /* TODO:
       if( p->verifyDate ) checkin_verify_younger(vid, zParentUuid, zDate); */
    RC;
    rc = fsl_db_prepare(dbC, &q, "SELECT merge FROM vmerge WHERE id=0 OR id<-2");
    RC;
    while( FSL_RC_STEP_ROW == fsl_stmt_step(&q) ){
      char *zMergeUuid;
      fsl_id_t const mid = fsl_stmt_g_id(&q, 0);
      //MARKER(("merging? %d\n", (int)mid));
      if( (mid == basedOnVid)
          || (!f->cache.markPrivate && fsl_content_is_private(f,mid))){
        continue;
      }
      zMergeUuid = fsl_rid_to_uuid(f, mid)
        /* FIXME? Adjust the query to join on blob and return the UUID? */
        ;
      //MARKER(("merging %d %s\n", (int)mid, zMergeUuid));
      if(zMergeUuid){
        rc = fsl_deck_P_add(d, zMergeUuid);
        fsl_free(zMergeUuid);
      }
      RC;
      /* TODO:
         if( p->verifyDate ) checkin_verify_younger(mid, zMergeUuid, zDate); */
    }
    fsl_stmt_finalize(&q);
  }

  { /* Q-cards... */
    rc = fsl_db_prepare(dbR, &q,
                        "SELECT "
                        "CASE vmerge.id WHEN -1 THEN '+' ELSE '-' END || mhash,"
                        "  merge"
                        "  FROM vmerge"
                        " WHERE (vmerge.id=-1 OR vmerge.id=-2)"
                        " ORDER BY 1");
    while( !rc && (FSL_RC_STEP_ROW==fsl_stmt_step(&q)) ){
      fsl_id_t const mid = fsl_stmt_g_id(&q, 1);
      if( mid != basedOnVid ){
        const char *zCherrypickUuid = fsl_stmt_g_text(&q, 0, NULL);
        int const qType = '+'==*(zCherrypickUuid++) ? 1 : -1;
        rc = fsl_deck_Q_add( d, qType, zCherrypickUuid, NULL );
      }
    }
    fsl_stmt_finalize(&q);
    RC;
  }

  zColor = opt->bgColor;
  if(opt->branch && *opt->branch){
    char * sym = fsl_mprintf("sym-%s", opt->branch);
    if(!sym){
      rc = FSL_RC_OOM;
      goto end;
    }
    rc = fsl_deck_T_add( d, FSL_TAGTYPE_PROPAGATING,
                         NULL, sym, NULL );
    fsl_free(sym);
    RC;
    if(opt->bgColor && *opt->bgColor){
      zColor = NULL;
      rc = fsl_deck_T_add( d, FSL_TAGTYPE_PROPAGATING,
                           NULL, "bgcolor", opt->bgColor);
      RC;
    }
    rc = fsl_deck_T_add( d, FSL_TAGTYPE_PROPAGATING,
                         NULL, "branch", opt->branch );
    RC;
    if(basedOnVid){
      rc = fsl_cancel_sym_tags(d, basedOnVid);
    }
  }
  if(zColor && *zColor){
    /* One-shot background color */
    rc = fsl_deck_T_add( d, FSL_TAGTYPE_ADD,
                         NULL, "bgcolor", opt->bgColor);
    RC;
  }

  if(opt->closeBranch){
    rc = fsl_deck_T_add( d, FSL_TAGTYPE_ADD,
                         NULL, "closed",
                         *opt->closeBranch
                         ? opt->closeBranch
                         : NULL);
    RC;
  }

  {
    /*
      Close any INTEGRATE merges if !op->integrate, or type-0 and
      integrate merges if opt->integrate.
    */
    rc = fsl_db_prepare(dbC, &q,
                        "SELECT mhash, merge FROM vmerge "
                        " WHERE id %s ORDER BY 1",
                        opt->integrate ? "IN(0,-4)" : "=(-4)");
    while( !rc && (FSL_RC_STEP_ROW==fsl_stmt_step(&q)) ){
      fsl_id_t const rid = fsl_stmt_g_id(&q, 1);
      //MARKER(("Integrating %d? opt->integrate=%d\n",(int)rid, opt->integrate));
      if( fsl_rid_is_leaf(f, rid)
          && !fsl_db_exists(dbR, /* Is not closed already... */
                            "SELECT 1 FROM tagxref "
                            "WHERE tagid=%d AND rid=%"FSL_ID_T_PFMT
                            " AND tagtype>0",
                            FSL_TAGID_CLOSED, rid)){
        const char *zIntegrateUuid = fsl_stmt_g_text(&q, 0, NULL);
        //MARKER(("Integrating %d %s\n",(int)rid, zIntegrateUuid));
        rc = fsl_deck_T_add( d, FSL_TAGTYPE_ADD, zIntegrateUuid,
                             "closed", "Closed by integrate-merge." );
      }
    }
    fsl_stmt_finalize(&q);
    RC;
  }

  end:
#undef RC
  fsl_stmt_finalize(&q);
  fsl_deck_finalize(&dBase);
  assert(NULL==d->B.baseline || &dBase==d->B.baseline);
  d->B.baseline = NULL /* if it was set, it was &dBase */;
  if(rc && !f->error.code){
    if(dbR->error.code) fsl_cx_uplift_db_error(f, dbR);
    else if(dbC->error.code) fsl_cx_uplift_db_error(f, dbC);
    else if(f->dbMain->error.code) fsl_cx_uplift_db_error(f, f->dbMain);
  }
  return rc;
}

int fsl_checkin_T_add2( fsl_cx * f, fsl_card_T * t){
  return fsl_deck_T_add2( &f->ckin.mf, t );
}

int fsl_checkin_T_add( fsl_cx * f, fsl_tagtype_e tagType,
                       fsl_uuid_cstr uuid, char const * name,
                       char const * value){
  return fsl_deck_T_add( &f->ckin.mf, tagType, uuid, name, value );
}

/**
   Returns true if the given blob RID is has a "closed" tag. This is
   generally intended only to be passed the RID of the current
   checkout, before attempting to perform a commit against it.
*/
static bool fsl_leaf_is_closed(fsl_cx * f, fsl_id_t rid){
  fsl_db * const dbR = fsl_needs_repo(f);
  return dbR
    ? fsl_db_exists(dbR, "SELECT 1 FROM tagxref"
                    " WHERE tagid=%d "
                    " AND rid=%"FSL_ID_T_PFMT" AND tagtype>0",
                    FSL_TAGID_CLOSED, rid)
    : false;
}

/**
   Returns true if the given name is the current branch
   for the given checkin version.
 */
static bool fsl_is_current_branch(fsl_db * dbR, fsl_id_t vid,
                                  char const * name){
  return fsl_db_exists(dbR,
                       "SELECT 1 FROM tagxref"
                       " WHERE tagid=%d AND rid=%"FSL_ID_T_PFMT
                       " AND tagtype>0"
                       " AND value=%Q",
                       FSL_TAGID_BRANCH, vid, name);
}

int fsl_checkin_commit(fsl_cx * f, fsl_checkin_opt const * opt,
                       fsl_id_t * newRid, fsl_uuid_str * newUuid ){
  int rc;
  fsl_deck deck = fsl_deck_empty;
  fsl_deck *d = &deck;
  fsl_db * dbC;
  fsl_db * dbR;
  char inTrans = 0;
  char oldPrivate;
  int const oldFlags = f ? f->flags : 0;
  fsl_id_t const vid = f ? f->ckout.rid : 0;
  if(!f || !opt) return FSL_RC_MISUSE;
  else if(!(dbC = fsl_needs_ckout(f))) return FSL_RC_NOT_A_CKOUT;
  else if(!(dbR = fsl_needs_repo(f))) return FSL_RC_NOT_A_REPO;
  assert(vid>=0);
  /**
     Do not permit a checkin to a closed leaf unless opt->branch would
     switch us to a new branch.
  */
  if( fsl_leaf_is_closed(f, vid)
      && (!opt->branch || !*opt->branch
          || fsl_is_current_branch(dbR, vid, opt->branch))){
    return fsl_cx_err_set(f, FSL_RC_ACCESS,
                          "Only Chuck Norris can commit to "
                          "a closed leaf.");
  }

  if(vid && opt->scanForChanges){
    /* We need to ensure this state is current in order to determine
       whether a given file is locally modified vis-a-vis the
       commit-time vfile state. */
    rc = fsl_vfile_changes_scan(f, vid, 0);
    if(rc) return rc;
  }

  fsl_cx_err_reset(f) /* avoid propagating an older error by accident.
                         Did that in test code. */;

  oldPrivate = f->cache.markPrivate;
  if(opt->isPrivate || fsl_content_is_private(f, vid)){
    f->cache.markPrivate = 1;
  }

#define RC if(rc) goto end
  fsl_deck_init(f, d, FSL_SATYPE_CHECKIN);

  rc = fsl_db_transaction_begin(dbR);
  RC;
  inTrans = 1;
  if(f->ckin.mf.T.used){
    /* Transfer accumulated tags. */
    assert(!f->ckin.mf.content.used);
    d->T = f->ckin.mf.T;
    f->ckin.mf.T = fsl_deck_empty.T;
  }
  rc = fsl_checkin_calc_manifest(f, d, vid, opt);
  RC;
  if(!d->F.used){
    rc = fsl_cx_err_set(f, FSL_RC_NOOP,
                        "Cowardly refusing to generate an empty commit.");
    RC;
  }

  if(opt->calcRCard) f->flags |= FSL_CX_F_CALC_R_CARD;
  else f->flags &= ~FSL_CX_F_CALC_R_CARD;
  rc = fsl_deck_save( d, opt->isPrivate );
  RC;
  assert(d->rid>0);
  /* Now get vfile back into shape. We do not do a vfile scan
     because that loses state like add/rm-queued files. */
  rc = fsl_db_exec_multi(dbC,
                         "DELETE FROM vfile WHERE vid<>"
                         "%" FSL_ID_T_PFMT ";"
                         "UPDATE vfile SET vid=%" FSL_ID_T_PFMT ";"
                         "DELETE FROM vfile WHERE deleted AND "
                         "fsl_is_enqueued(id); "
                         "UPDATE vfile SET rid=mrid, mhash=NULL, "
                         "chnged=0, deleted=0, origname=NULL "
                         "WHERE fsl_is_enqueued(id)",
                         vid, d->rid);
  if(!rc) rc = fsl__ckout_version_write(f, d->rid, NULL);
  RC;
  assert(d->f == f);
  rc = fsl_checkin_add_unsent(f, d->rid);
  RC;
  rc = fsl__ckout_clear_merge_state(f, true);
  RC;
  /*
    todo(?) from fossil(1) follows. Most of this seems to be what the
    vfile handling does (above).

    db_multi_exec("PRAGMA %s.application_id=252006673;", db_name("repository"));
    db_multi_exec("PRAGMA %s.application_id=252006674;", db_name("localdb"));

    // Update the vfile and vmerge tables
    db_multi_exec(
      "DELETE FROM vfile WHERE (vid!=%d OR deleted) AND is_selected(id);"
      "DELETE FROM vmerge;"
      "UPDATE vfile SET vid=%d;"
      "UPDATE vfile SET rid=mrid, chnged=0, deleted=0, origname=NULL"
      " WHERE is_selected(id);"
      , vid, nvid
    );
    db_lset_int("checkout", nvid);


    // Update the isexe and islink columns of the vfile table
    db_prepare(&q,
      "UPDATE vfile SET isexe=:exec, islink=:link"
      " WHERE vid=:vid AND pathname=:path AND (isexe!=:exec OR islink!=:link)"
    );
    db_bind_int(&q, ":vid", nvid);
    pManifest = manifest_get(nvid, CFTYPE_MANIFEST, 0);
    manifest_file_rewind(pManifest);
    while( (pFile = manifest_file_next(pManifest, 0)) ){
      db_bind_int(&q, ":exec", pFile->zPerm && strstr(pFile->zPerm, "x"));
      db_bind_int(&q, ":link", pFile->zPerm && strstr(pFile->zPerm, "l"));
      db_bind_text(&q, ":path", pFile->zName);
      db_step(&q);
      db_reset(&q);
    }
    db_finalize(&q);
  */

  if(opt->dumpManifestFile){
    FILE * out;
    /* MARKER(("Dumping generated manifest to file [%s]:\n", opt->dumpManifestFile)); */
    out = fsl_fopen(opt->dumpManifestFile, "w");
    if(out){
      rc = fsl_deck_output( d, fsl_output_f_FILE, out );
      fsl_fclose(out);
    }else{
      rc = fsl_cx_err_set(f, FSL_RC_IO, "Could not open output "
                          "file for writing: %s", opt->dumpManifestFile);
    }
    RC;
  }

  if(d->P.used){
    /* deltify the parent manifest */
    char const * p0 = (char const *)d->P.list[0];
    fsl_id_t const prid = fsl_uuid_to_rid(f, p0);
    /* MARKER(("Deltifying parent manifest #%d...\n", (int)prid)); */
    assert(p0);
    assert(prid>0);
    rc = fsl__content_deltify(f, prid, d->rid, 0);
    RC;
  }

  end:
  f->flags = oldFlags;
#undef RC
  f->cache.markPrivate = oldPrivate;
  /* fsl_buffer_reuse(&f->fileContent); */
  if(inTrans){
    if(rc) fsl_db_transaction_rollback(dbR);
    else{
      rc = fsl_db_transaction_commit(dbR);
      if(!rc){
        if(newRid) *newRid = d->rid;
        if(newUuid){
          if(NULL==(*newUuid = fsl_rid_to_uuid(f, d->rid))){
            rc = FSL_RC_OOM;
          }
        }
      }
    }
  }
  if(rc && !f->error.code){
    if(f->dbMain->error.code) fsl_cx_uplift_db_error(f, f->dbMain);
    else f->error.code = rc;
  }
  fsl_checkin_discard(f);
  fsl_deck_finalize(d);
  return rc;
}


#undef MARKER
/* end of file ./src/checkin.c */
/* start of file ./src/checkout.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/
/*  
  *****************************************************************************
  This file houses the code for checkout-level APIS.
*/
#include <assert.h>

#include <string.h> /* memcmp() */

/* Only for debugging */
#include <stdio.h>
#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)


/**
    Kludge for type-safe strncmp/strnicmp inconsistency.
*/
static int fsl_strnicmp_int(char const *zA, char const * zB, fsl_size_t nByte){
  return fsl_strnicmp( zA, zB, (fsl_int_t)nByte);
}

int fsl_ckout_filename_check( fsl_cx * const f, bool relativeToCwd,
                              char const * zOrigName, fsl_buffer * const pOut ){
  int rc;
  if(!zOrigName || !*zOrigName) return FSL_RC_MISUSE;
  else if(!fsl_needs_ckout(f)/* will update f's error state*/){
    return FSL_RC_NOT_A_CKOUT;
  }
#if 0
  /* Is this sane? */
  else if(fsl_is_simple_pathname(zOrigName,1)){
    rc = 0;
    if(pOut){
      rc = fsl_buffer_append(pOut, zOrigName, fsl_strlen(zOrigName));
    }
  }
#endif
  else{
    char const * zLocalRoot;
    char const * zFull;
    fsl_size_t nLocalRoot;
    fsl_size_t nFull = 0;
    fsl_buffer * const full = fsl__cx_scratchpad(f);
    int (*xCmp)(char const *, char const *,fsl_size_t);
    bool endsWithSlash;
    assert(f->ckout.dir);
    zLocalRoot = f->ckout.dir;
    assert(zLocalRoot);
    assert(*zLocalRoot);
    nLocalRoot = f->ckout.dirLen;
    assert(nLocalRoot);
    assert('/' == zLocalRoot[nLocalRoot-1]);
    rc = fsl_file_canonical_name2(relativeToCwd ? NULL : zLocalRoot,
                                  zOrigName, full, 1);
#if 0
    MARKER(("canon2: %p (%s) %s ==> %s\n", (void const *)full->mem,
            relativeToCwd ? "cwd" : "ckout", zOrigName, fsl_buffer_cstr(full)));
#endif
    if(rc){
      if(FSL_RC_OOM != rc){
        rc = fsl_cx_err_set(f, rc, "Error #%d (%s) canonicalizing "
                            "file name: %s\n",
                            rc, fsl_rc_cstr(rc),
                            zOrigName);
      }
      goto end;
    }
    zFull = fsl_buffer_cstr2(full, &nFull);
    xCmp = fsl_cx_is_case_sensitive(f,false)
      ? fsl_strncmp
      : fsl_strnicmp_int;
    assert(zFull);
    assert(nFull>0);
    endsWithSlash = '/' == zFull[nFull-1];
    if( ((nFull==nLocalRoot-1 || (nFull==nLocalRoot && endsWithSlash))
         && xCmp(zLocalRoot, zFull, nFull)==0)
        || (nFull==1 && zFull[0]=='/' && nLocalRoot==1 && zLocalRoot[0]=='/') ){
      /* Special case.  zOrigName refers to zLocalRoot directory.

         Outputing "." instead of nothing is a historical decision
         which may be worth re-evaluating. Currently fsl_cx_stat() relies
         on it.
      */
      if(pOut){
        char const * zOut;
        fsl_size_t nOut;
        if(endsWithSlash){ /* retain trailing slash */
          zOut = "./";
          nOut = 2;
        }else{
          zOut = ".";
          nOut = 1;
        };
        rc = fsl_buffer_append(pOut, zOut, nOut);
      }else{
        rc = 0;
      }
      goto end;
    }

    if( nFull<=nLocalRoot || xCmp(zLocalRoot, zFull, nLocalRoot) ){
      rc = fsl_cx_err_set(f, FSL_RC_RANGE,
                          "File is outside of checkout tree: %s",
                          zOrigName);
      goto end;
    }

    if(pOut){
      rc = fsl_buffer_append(pOut, zFull + nLocalRoot, nFull - nLocalRoot);
    }

    end:
    fsl__cx_scratchpad_yield(f, full);
  }
  return rc;
}


/**
    Returns a fsl_ckout_change_e value for the given
    fsl_vfile_change_e value.

    Why are these not consolidated into one enum?  2021-03-13: because
    there are more checkout-level change codes than vfile-level
    changes. We could still consolidate them, giving the vfile changes
    their hard-coded values and leaving room in the enum for upward
    growth of that set.
*/
static fsl_ckout_change_e fsl__vfile_to_ckout_change(int vChange){
  switch((fsl_vfile_change_e)vChange){
#define EE(X) case FSL_VFILE_CHANGE_##X: return FSL_CKOUT_CHANGE_##X
    EE(NONE);
    EE(MOD);
    EE(MERGE_MOD);
    EE(MERGE_ADD);
    EE(INTEGRATE_MOD);
    EE(INTEGRATE_ADD);
    EE(IS_EXEC);
    EE(BECAME_SYMLINK);
    EE(NOT_EXEC);
    EE(NOT_SYMLINK);
#undef EE
    default:
       assert(!"Unhandled fsl_vfile_change_e value!");
      return FSL_CKOUT_CHANGE_NONE;
  }
}

int fsl_ckout_changes_visit( fsl_cx * f, fsl_id_t vid,
                             bool doScan,
                             fsl_ckout_changes_f visitor,
                             void * state ){
  int rc;
  fsl_db * db;
  fsl_stmt st = fsl_stmt_empty;
  int count = 0;
  fsl_ckout_change_e coChange;
  fsl_fstat fstat;
  if(!f || !visitor) return FSL_RC_MISUSE;
  db = fsl_needs_ckout(f);
  if(!db) return FSL_RC_NOT_A_CKOUT;
  if(vid<0){
    vid = f->ckout.rid;
    assert(vid>=0);
  }
  if(doScan){
    rc = fsl_vfile_changes_scan(f, vid, 0);
    if(rc) goto end;
  }
  rc = fsl_db_prepare(db, &st,
                      "SELECT chnged, deleted, rid, "
                      "pathname, origname "
                      "FROM vfile WHERE vid=%" FSL_ID_T_PFMT
                      " /*%s()*/",
                      vid,__func__);
  assert(!rc);
  while( FSL_RC_STEP_ROW == fsl_stmt_step(&st) ){
    int const changed = fsl_stmt_g_int32(&st, 0);
    int const deleted = fsl_stmt_g_int32(&st,1);
    fsl_id_t const vrid = fsl_stmt_g_id(&st,2);
    char const * name;
    char const * oname = NULL;
    name = fsl_stmt_g_text(&st, 3, NULL);
    oname = fsl_stmt_g_text(&st,4,NULL);
    if(oname && (0==fsl_strcmp(name, oname))){
      /* Work around a fossil oddity which sets origname=pathname
         during a 'mv' operation.
      */
      oname = NULL;
    }
    coChange = FSL_CKOUT_CHANGE_NONE;
    if(deleted){
      coChange = FSL_CKOUT_CHANGE_REMOVED;
    }else if(0==vrid){
      coChange = FSL_CKOUT_CHANGE_ADDED;
    }else if(!changed && NULL != oname){
      /* In fossil ^^, the "changed" state trumps the "renamed" state
       for status view purposes, so we'll do that here. */
      coChange = FSL_CKOUT_CHANGE_RENAMED;
    }else{
      fstat = fsl_fstat_empty;
      if( fsl_cx_stat(f, false, name, &fstat ) ){
        coChange = FSL_CKOUT_CHANGE_MISSING;
        fsl_cx_err_reset(f) /* keep FSL_RC_NOT_FOUND from bubbling
                               up to the client! */;
      }else if(!changed){
        continue;
      }else{
        coChange = fsl__vfile_to_ckout_change(changed);
      }
    }
    if(!coChange){
      MARKER(("INTERNAL ERROR: unhandled vfile.chnged "
              "value %d for file [%s]\n",
              changed, name));
      continue;
    }
    ++count;
    rc = visitor(state, coChange, name, oname);
    if(rc){
      if(FSL_RC_BREAK==rc){
        rc = 0;
        break;
      }else if(!f->error.code && (FSL_RC_OOM!=rc)){
        fsl_cx_err_set(f, rc, "Error %s returned from changes callback.",
                       fsl_rc_cstr(rc));
      }
      break;
    }
  }
  end:
  fsl_stmt_finalize(&st);
  if(rc && db->error.code && !f->error.code){
    fsl_cx_uplift_db_error(f, db);
  }

  return rc;
}

static bool fsl_co_is_in_vfile(fsl_cx *f,
                               char const *zFilename){
  return fsl_db_exists(fsl_cx_db_ckout(f),
                       "SELECT 1 FROM vfile"
                       " WHERE vid=%"FSL_ID_T_PFMT
                       " AND pathname=%Q %s",
                       f->ckout.rid, zFilename,
                       fsl_cx_filename_collation(f));
}


/** Initialized-with-defaults fsl_ckout_manage_state structure, intended for
    const-copy initialization. */
#define fsl_ckout_manage_state_empty_m {NULL,NULL,NULL}
/** Initialized-with-defaults fsl_ckout_manage_state structure, intended for
    non-const copy initialization. */
static const fsl_ckout_manage_state fsl_ckout_manage_state_empty
= fsl_ckout_manage_state_empty_m;

/**
   Internal machinery for fsl_ckout_manage(). zFilename MUST
   be a checkout-relative file which is known to exist. fst MUST
   be an object populated by fsl_stat()'ing zFilename. isInVFile
   MUST be the result of having passed zFilename to fsl_co_is_in_vfile().
 */
static int fsl_ckout_manage_impl(fsl_cx * const f, char const *zFilename,
                                 fsl_fstat const *fst,
                                 bool isInVFile){
  int rc = 0;
  fsl_db * const db = fsl_needs_ckout(f);
  assert(fsl_is_simple_pathname(zFilename, true));
  if( isInVFile ){
    rc = fsl_db_exec(db, "UPDATE vfile SET deleted=0,"
                     " mtime=%"PRIi64
                     " WHERE vid=%"FSL_ID_T_PFMT
                     " AND pathname=%Q %s",
                     (int64_t)fst->mtime,
                     f->ckout.rid, zFilename,
                     fsl_cx_filename_collation(f));
  }else{
    int const chnged = FSL_VFILE_CHANGE_MOD
      /* fossil(1) sets chnged=0 on 'add'ed vfile records, but then the 'status'
         command updates the field to 1. To avoid down-stream inconsistencies
         (such as the ones which lead me here), we'll go ahead and set it to
         1 here.
      */;
    rc = fsl_db_exec(db,
                     "INSERT INTO "
                     "vfile(vid,chnged,deleted,rid,mrid,pathname,isexe,islink,mtime)"
                     "VALUES(%"FSL_ID_T_PFMT",%d,0,0,0,%Q,%d,%d,%"PRIi64")",
                     f->ckout.rid, chnged, zFilename,
                     (FSL_FSTAT_PERM_EXE==fst->perm) ? 1 : 0,
                     (FSL_FSTAT_TYPE_LINK==fst->type) ? 1 : 0,
                     (int64_t)fst->mtime
                     );
  }
  if(rc) rc = fsl_cx_uplift_db_error2(f, db, rc);
  return rc;
}

/**
   Internal state for the recursive file-add process.
*/
struct CoAddState {
  fsl_cx * f;
  fsl_ckout_manage_opt * opt;
  fsl_buffer * absBuf; // absolute path of file to check
  fsl_buffer * coRelBuf; // checkout-relative path of absBuf
  fsl_fstat fst; // fsl_stat() state of absBuf's file
};
typedef struct CoAddState CoAddState;
static const CoAddState CoAddState_empty =
  {NULL, NULL, NULL, NULL, fsl_fstat_empty_m};

/**
   fsl_dircrawl_f() impl for recursively adding files to a
   repo. state must be a (CoAddState*)/
*/
static int fsl_dircrawl_f_add(fsl_dircrawl_state const *);

/**
   Attempts to add file or directory (recursively) cas->absBuf to the
   current repository. isCrawling must be true if this is a
   fsl_dircrawl()-invoked call, else false.
*/
static int co_add_one(CoAddState * const cas, bool isCrawling){
  int rc = 0;
  fsl_buffer_reuse(cas->coRelBuf);
  rc = fsl_cx_stat2(cas->f, cas->opt->relativeToCwd,
                    fsl_buffer_cstr(cas->absBuf), &cas->fst,
                    fsl_buffer_reuse(cas->coRelBuf), false)
    /* Reminder: will fail if file is outside of the checkout tree */;
  if(rc) return rc;
  switch(cas->fst.type){
    case FSL_FSTAT_TYPE_FILE:{
      bool skipped = false;
      char const * zCoRel = fsl_buffer_cstr(cas->coRelBuf);
      bool const isInVFile = fsl_co_is_in_vfile(cas->f, zCoRel);
      if(!isInVFile){
        if(fsl_reserved_fn_check(cas->f, zCoRel,-1,false)){
          /* ^^^ we need to use fsl_reserved_fn_check(), instead of
             fsl_is_reserved_fn(), so that we will inherit any
             new checks which require a context object. If that
             check fails, though, it updates cas->f with an error
             message which we need to suppress here to avoid it
             accidentally propagating and causing downstream
             confusion. */
          fsl_cx_err_reset(cas->f);
          skipped = true;
        }else if(cas->opt->checkIgnoreGlobs){
          char const * m =
            fsl_cx_glob_matches(cas->f, FSL_GLOBS_IGNORE, zCoRel);
          if(m) skipped = true;
        }
        if(!skipped && cas->opt->callback){
          bool yes = false;
          fsl_ckout_manage_state mst = fsl_ckout_manage_state_empty;
          mst.opt = cas->opt;
          mst.filename = zCoRel;
          mst.f = cas->f;
          rc = cas->opt->callback( &mst, &yes );
          if(rc) goto end;
          else if(!yes) skipped = true;
        }
      }
      if(skipped){
        ++cas->opt->counts.skipped;
      }else{
        rc = fsl_ckout_manage_impl(cas->f, zCoRel, &cas->fst,
                                        isInVFile);
        if(!rc){
          if(isInVFile) ++cas->opt->counts.updated;
          else ++cas->opt->counts.added;
        }
      }
      break;
    }
    case FSL_FSTAT_TYPE_DIR:
      if(!isCrawling){
        /* Reminder to self: fsl_dircrawl() copies its first argument
           for canonicalizing it, so this is safe even though
           cas->absBuf may be reallocated during the recursive
           call. We're done with these particular contents of
           cas->absBuf at this point. */
        rc = fsl_dircrawl(fsl_buffer_cstr(cas->absBuf),
                          fsl_dircrawl_f_add, cas);
        if(rc && !cas->f->error.code){
          rc = fsl_cx_err_set(cas->f, rc, "fsl_dircrawl() returned %s.",
                              fsl_rc_cstr(rc));
        }
      }else{
        assert(!"Cannot happen - caught higher up");
        fsl__fatal(FSL_RC_ERROR, "Internal API misuse in/around %s().",
                  __func__);
      }
      break;
    default:
      rc = fsl_cx_err_set(cas->f, FSL_RC_TYPE,
                          "Unhandled filesystem entry type: "
                          "fsl_fstat_type_e #%d", cas->fst.type);
      break;
  }
  end:
  return rc;
}

/**
   fsl_dircrawl_f() impl for fsl_ckout_manage().
*/
static int fsl_dircrawl_f_add(fsl_dircrawl_state const *dst){
  switch(dst->entryType){
    case FSL_FSTAT_TYPE_DIR:
    case FSL_FSTAT_TYPE_FILE:{
      CoAddState * const cas = (CoAddState*)dst->callbackState;
      int const rc = fsl_buffer_appendf(fsl_buffer_reuse(cas->absBuf),
                                        "%s/%s", dst->absoluteDir, dst->entryName);
      if(rc) return rc;
      switch(dst->entryType){
        case FSL_FSTAT_TYPE_DIR:
          return fsl_is_top_of_ckout(fsl_buffer_cstr(cas->absBuf))
            /* Never recurse into nested checkouts */
            ? FSL_RC_NOOP : 0;
        case FSL_FSTAT_TYPE_FILE:
          return co_add_one(cas, true);
        default:
          fsl__fatal(FSL_RC_ERROR,"Not possible: caught above.");
          return 0;
      }
    }
    default:
      return 0;
  }
}

/**
   Returns true if the absolute path zAbsName is f->ckout.dir, disregarding
   an optional trailing slash on zAbsName.
*/
static bool fsl__is_ckout_dir(fsl_cx * const f, char const * const zAbsName){
  /* Keeping in mind that f->ckout.dir is always slash-terminated...*/
  assert(f->ckout.dirLen>0);
  return (0==fsl_strncmp(zAbsName, f->ckout.dir, f->ckout.dirLen-1)
          && 0==zAbsName[f->ckout.dirLen-1]
          /* ==> matches except that zAbsName is NUL-terminated where
             ckout.dir has a trailing slash. */)
    || 0==fsl_strcmp(zAbsName, f->ckout.dir);
}

int fsl_ckout_manage( fsl_cx * const f, fsl_ckout_manage_opt * const opt_ ){
  int rc = 0;
  CoAddState cas = CoAddState_empty;
  fsl_ckout_manage_opt opt;
  if(!f) return FSL_RC_MISUSE;
  else if(!fsl_needs_ckout(f)) return FSL_RC_NOT_A_CKOUT;
  assert(f->ckout.rid>=0);
  opt = *opt_
    /*use a copy in case the user manages to modify
      opt_ from a callback. */;
  cas.absBuf = fsl__cx_scratchpad(f);
  cas.coRelBuf = fsl__cx_scratchpad(f);
  rc = fsl_file_canonical_name(opt.filename, cas.absBuf, false);
  if(!rc){
    char const * const zAbs = fsl_buffer_cstr(cas.absBuf);
    if(!fsl_is_top_of_ckout(zAbs) || fsl__is_ckout_dir(f, zAbs)){
      cas.f = f;
      cas.opt = &opt;
      rc = co_add_one(&cas, false);
      opt_->counts = opt.counts;
    }
  }
  fsl__cx_scratchpad_yield(f, cas.absBuf);
  fsl__cx_scratchpad_yield(f, cas.coRelBuf);
  return rc;
}

/**
   Creates, if needed, a TEMP TABLE named [tableName] with a single
   [id] field and populates it with all ids from the given bag.

   Returns 0 on success, any number of non-0 codes on error.
*/
static int fsl_ckout_bag_to_ids(fsl_cx * const f, fsl_db * const db,
                                char const * tableName,
                                fsl_id_bag const * bag){
  fsl_stmt insId = fsl_stmt_empty;
  int rc = fsl_db_exec_multi(db,
                             "CREATE TEMP TABLE IF NOT EXISTS "
                             "[%s](id); "
                             "DELETE FROM [%s] /* %s() */;",
                             tableName, tableName, __func__);
  if(rc) goto dberr;
  rc = fsl_db_prepare(db, &insId,
                      "INSERT INTO [%s](id) values(?1) "
                      "/* %s() */", tableName, __func__);
  if(rc) goto dberr;
  for(fsl_id_t e = fsl_id_bag_first(bag);
      e; e = fsl_id_bag_next(bag, e)){
    fsl_stmt_bind_id(&insId, 1, e);
    rc = fsl_stmt_step(&insId);
    switch(rc){
      case FSL_RC_STEP_DONE:
        rc = 0;
        break;
      default:
        fsl_stmt_finalize(&insId);
        goto dberr;
    }
    fsl_stmt_reset(&insId);
  }
  assert(!rc);
  end:
  fsl_stmt_finalize(&insId);
  return rc;
  dberr:
  assert(rc);
  rc = fsl_cx_uplift_db_error2(f, db, rc);
  goto end;
}

/** Initialized-with-defaults fsl_ckout_unmanage_state structure, intended for
    const-copy initialization. */
#define fsl_ckout_unmanage_state_empty_m {NULL,NULL,NULL}
/** Initialized-with-defaults fsl_ckout_unmanage_state structure, intended for
    non-const copy initialization. */
static const fsl_ckout_unmanage_state fsl_ckout_unmanage_state_empty
= fsl_ckout_unmanage_state_empty_m;

int fsl_ckout_unmanage(fsl_cx * const f, fsl_ckout_unmanage_opt const * opt){
  int rc;
  fsl_db * const db = fsl_needs_ckout(f);
  fsl_buffer * fname = 0;
  fsl_id_t const vid = f->ckout.rid;
  fsl_stmt q = fsl_stmt_empty;
  bool inTrans = false;
  if(!db) return FSL_RC_NOT_A_CKOUT;
  else if((!opt->filename || !*opt->filename)
          && !opt->vfileIds){
    return fsl_cx_err_set(f, FSL_RC_MISUSE,
                          "Empty file set is not legal for %s()",
                          __func__);
  }
  assert(vid>=0);
  rc = fsl_db_transaction_begin(db);
  if(rc) goto dberr;
  inTrans = true;
  if(opt->vfileIds){
    rc = fsl_ckout_bag_to_ids(f, db, "fx_unmanage_id", opt->vfileIds);
    if(rc) goto end;
    rc = fsl_db_exec(db,
                     "UPDATE vfile SET deleted=1 "
                     "WHERE vid=%" FSL_ID_T_PFMT " "
                     "AND NOT deleted "
                     "AND id IN fx_unmanage_id /* %s() */",
                     vid, __func__);
    if(rc) goto dberr;
    if(opt->callback){
      rc = fsl_db_prepare(db,&q,
                          "SELECT pathname FROM vfile "
                          "WHERE vid=%" FSL_ID_T_PFMT " "
                          "AND deleted "
                          "AND id IN fx_unmanage_id "
                          "/* %s() */",
                          vid, __func__);
      if(rc) goto dberr;
    }
  }else{// Process opt->filename
    fname = fsl__cx_scratchpad(f);
    rc = fsl_ckout_filename_check(f, opt->relativeToCwd,
                                  opt->filename, fname);
    if(rc) goto end;
    char const * zNorm = fsl_buffer_cstr(fname);
    /* MARKER(("fsl_ckout_unmanage(%d, %s) ==> %s\n", relativeToCwd, zFilename, zNorm)); */
    assert(zNorm);
    if(fname->used){
      fsl_buffer_strip_slashes(fname);
      if(1==fname->used && '.'==*zNorm){
        /* Special case: handle "." from ckout root intuitively */
        fsl_buffer_reuse(fname);
        assert(0==*zNorm);
      }
    }
    rc = fsl_db_exec(db,
                     "UPDATE vfile SET deleted=1 "
                     "WHERE vid=%" FSL_ID_T_PFMT " "
                     "AND NOT deleted "
                     "AND CASE WHEN %Q='' THEN 1 "
                     "ELSE fsl_match_vfile_or_dir(pathname,%Q) "
                     "END /*%s()*/",
                     vid, zNorm, zNorm, __func__);
    if(rc) goto dberr;
    if(opt->callback){
      rc = fsl_db_prepare(db,&q,
                          "SELECT pathname FROM vfile "
                          "WHERE vid=%" FSL_ID_T_PFMT " "
                          "AND deleted "
                          "AND CASE WHEN %Q='' THEN 1 "
                          "ELSE fsl_match_vfile_or_dir(pathname,%Q) "
                          "END "
                          "UNION "
                          "SELECT pathname FROM vfile "
                          "WHERE vid=%" FSL_ID_T_PFMT " "
                          "AND rid=0 AND deleted "
                          "/*%s()*/",
                          vid, zNorm, zNorm, vid, __func__);
      if(rc) goto dberr;
    }
  }/*opt->filename*/

  if(q.stmt){
    fsl_ckout_unmanage_state ust = fsl_ckout_unmanage_state_empty;
    ust.opt = opt;
    ust.f = f;
    while(FSL_RC_STEP_ROW==fsl_stmt_step(&q)){
      rc = fsl_stmt_get_text(&q, 0, &ust.filename, NULL);
      if(rc){
        rc = fsl_cx_uplift_db_error2(f, db, rc);
        goto end;
      }
      rc = opt->callback(&ust);
      if(rc) goto end;
    }
    fsl_stmt_finalize(&q);
  }
  /* Remove rm'd ADDed-but-not-yet-committed entries... */
  rc = fsl_db_exec(db,
                   "DELETE FROM vfile WHERE vid=%" FSL_ID_T_PFMT
                   " AND rid=0 AND deleted",
                   vid);
  if(rc) goto dberr;
  end:
  if(fname) fsl__cx_scratchpad_yield(f, fname);
  fsl_stmt_finalize(&q);
  if(opt->vfileIds){
    fsl_db_exec(db, "DROP TABLE IF EXISTS fx_unmanage_id /* %s() */",
                __func__)
      /* Ignoring result code */;
  }
  if(inTrans){
    int const rc2 = fsl_db_transaction_end(db, !!rc);
    if(!rc) rc = rc2;
  }
  return rc;
  dberr:
  assert(rc);
  rc = fsl_cx_uplift_db_error2(f, db, rc);
  goto end;

}

int fsl_ckout_changes_scan(fsl_cx * const f){
  return fsl_vfile_changes_scan(f, -1, 0);
}

int fsl__ckout_install_schema(fsl_cx * const f, bool dropIfExists){
  char const * tNames[] = {
  "vvar", "vfile", "vmerge", 0
  };
  int rc;
  fsl_db * const db = fsl_needs_ckout(f);
  if(!db) return f->error.code;
  if(dropIfExists){
    char const * t;
    int i;
    char const * dbName = fsl_db_role_name(FSL_DBROLE_CKOUT);
    for(i=0; 0!=(t = tNames[i]); ++i){
      rc = fsl_db_exec(db, "DROP TABLE IF EXISTS %s.%s /*%s()*/",
                       dbName, t, __func__);
      if(rc) break;
    }
    if(!rc){
      rc = fsl_db_exec(db, "DROP TRIGGER IF EXISTS "
                       "%s.vmerge_ck1 /*%s()*/",
                       dbName, __func__);
    }
  }else{
    if(fsl_db_table_exists(db, FSL_DBROLE_CKOUT,
                           tNames[0])){
      return 0;
    }
  }
  rc = fsl_db_exec_multi(db, "%s", fsl_schema_ckout());
  return fsl_cx_uplift_db_error2(f, db, rc);
}

bool fsl_ckout_has_changes(fsl_cx *f){
  fsl_db * const db = fsl_cx_db_ckout(f);
  if(!db) return false;
  return fsl_db_exists(db,
                       "SELECT 1 FROM vfile WHERE chnged "
                       "OR coalesce(origname != pathname, 0) "
                       "/*%s()*/", __func__)
    || fsl_db_exists(db,"SELECT 1 FROM vmerge /*%s()*/", __func__);
}

int fsl__ckout_clear_merge_state( fsl_cx * const f, bool fullWipe ){
  int rc;
  if(fullWipe){
    rc = fsl_cx_exec(f,"DELETE FROM vmerge /*%s()*/", __func__);
  }else{
    rc = fsl_cx_exec_multi(f,
                     "DELETE FROM vmerge WHERE id IN("
                     "SELECT vm.id FROM vmerge vm, vfile vf "
                     "WHERE vm.id=vf.id AND vf.chnged=0"
                     ");"
                     "DELETE FROM vmerge WHERE NOT EXISTS("
                     "SELECT 1 FROM vmerge WHERE id>0"
                     ") AND NOT EXISTS ("
                     "SELECT 1 FROM vfile WHERE chnged>1"
                     ");"
                     "/*%s()*/", __func__ );

  }
  return rc;
}

int fsl_ckout_clear_db(fsl_cx *f){
  fsl_db * const db = fsl_needs_ckout(f);
  if(!db) return f->error.code;
  return fsl_db_exec_multi(db,
                           "DELETE FROM vfile;"
                           "DELETE FROM vmerge;"
                           "DELETE FROM vvar WHERE name IN"
                           "('checkout','checkout-hash') "
                           "/*%s()*/", __func__);
}

/**
   Updates f->ckout.dir and dirLen based on the current state of
   f->ckout.db. Returns 0 on success, FSL_RC_OOM on allocation error,
   some other code if canonicalization of the name fails
   (e.g. filesystem error or cwd cannot be resolved).
*/
static int fsl_update_ckout_dir(fsl_cx *f){
  int rc;
  fsl_buffer ckDir = fsl_buffer_empty;
  fsl_db * dbC = fsl__cx_db_for_role(f, FSL_DBROLE_CKOUT);
  assert(dbC->filename);
  assert(*dbC->filename);
  rc = fsl_file_canonical_name(dbC->filename, &ckDir, false);
  if(rc) return rc;
  char * zCanon = fsl_buffer_take(&ckDir);
  //MARKER(("dbC->filename=%s\n", dbC->filename));
  //MARKER(("zCanon=%s\n", zCanon));
  rc = fsl_file_dirpart(zCanon, -1, &ckDir, true);
  fsl_free(zCanon);
  if(rc){
    fsl_buffer_clear(&ckDir);
  }else{
    fsl_free(f->ckout.dir);
    f->ckout.dirLen = ckDir.used;
    f->ckout.dir = fsl_buffer_take(&ckDir);
    assert('/'==f->ckout.dir[f->ckout.dirLen-1]);
    /*MARKER(("Updated ckout.dir: %d %s\n",
      (int)f->ckout.dirLen, f->ckout.dir));*/
  }
  return rc;
}


int fsl_repo_open_ckout(fsl_cx * const f, const fsl_repo_open_ckout_opt *opt){
  fsl_db *dbC = 0;
  fsl_buffer *cwd = 0;
  int rc = 0;
  bool didChdir = false;
  if(!opt) return FSL_RC_MISUSE;
  else if(!fsl_needs_repo(f)){
    return f->error.code;
  }else if(fsl_cx_db_ckout(f)){
    return fsl_cx_err_set(f, FSL_RC_MISUSE,
                          "A checkout is already attached.");
  }
  cwd = fsl__cx_scratchpad(f);
  assert(!cwd->used);
  if((rc = fsl_cx_getcwd(f, cwd))){
    assert(!cwd->used);
    fsl__cx_scratchpad_yield(f, cwd);
    return fsl_cx_err_set(f, rc, "Error %d [%s]: unable to "
                          "determine current directory.",
                          rc, fsl_rc_cstr(rc));
  }
  if(opt->targetDir && *opt->targetDir){
    if(fsl_chdir(opt->targetDir)){
      fsl__cx_scratchpad_yield(f, cwd);
      return fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                            "Directory not found or inaccessible: %s",
                            opt->targetDir);
    }
    didChdir = true;
  }
  /**
     AS OF HERE: do not use 'return'. Use goto end so that we can
     chdir() back to our original cwd!
  */
  if(!fsl_dir_is_empty("."/*we've already chdir'd if
                            we were going to*/)) {
    switch(opt->fileOverwritePolicy){
      case FSL_OVERWRITE_ALWAYS:
      case FSL_OVERWRITE_NEVER: break;
      default:
        assert(FSL_OVERWRITE_ERROR==opt->fileOverwritePolicy);
        rc = fsl_cx_err_set(f, FSL_RC_ACCESS,
                            "Directory is not empty and "
                            "fileOverwritePolicy is "
                            "FSL_OVERWRITE_ERROR: "
                            "%b", cwd);
        goto end;
    }
  }
  if(opt->checkForOpenedCkout){
    /* Check target and parent dirs for a checkout and bail out if we
       find one. If opt->checkForOpenedCkout is false then we will use
       the dbOverwritePolicy to determine what to do if we find a
       checkout db in cwd (as opposed to a parent). */
    fsl_buffer * const foundAt = fsl__cx_scratchpad(f);
    if (!fsl_ckout_db_search(fsl_buffer_cstr(cwd), true, foundAt)) {
      rc = fsl_cx_err_set(f, FSL_RC_ALREADY_EXISTS,
                          "There is already a checkout db at %b",
                          foundAt);
    }
    fsl__cx_scratchpad_yield(f, foundAt);
    if(rc) goto end;
  }

  /**
     Create and attach ckout db...
  */
  assert(!fsl_cx_db_ckout(f));
  const char * dbName = opt->ckoutDbFile
    ? opt->ckoutDbFile : fsl_preferred_ckout_db_name();
  fsl_cx_err_reset(f);
  rc = fsl__cx_attach_role(f, dbName, FSL_DBROLE_CKOUT, true);
  if(rc) goto end;
  fsl_db * const theDbC = fsl_cx_db_ckout(f);
  dbC = fsl__cx_db_for_role(f, FSL_DBROLE_CKOUT);
  assert(dbC->name);
  assert(dbC->filename);
  rc = fsl__ckout_install_schema(f, opt->dbOverwritePolicy);
  if(!rc){
    rc = fsl_db_exec(theDbC,"INSERT OR IGNORE INTO "
                     "%s.vvar (name,value) "
                     "VALUES('checkout',0),"
                     "('checkout-hash',null)",
                     dbC->name);
  }
  if(rc) rc = fsl_cx_uplift_db_error(f, theDbC);
  end:
  if(didChdir){
    assert(opt->targetDir && *opt->targetDir);
    assert(cwd->used /* is this true in the root dir? */);
    fsl_chdir(fsl_buffer_cstr(cwd))
      /* Ignoring error because we have no recovery strategy! */;
  }
  fsl__cx_scratchpad_yield(f, cwd);
  if(!rc){
    fsl_db * const dbR = fsl__cx_db_for_role(f, FSL_DBROLE_REPO);
    assert(dbR);
    assert(dbR->filename && *dbR->filename);
    rc = fsl_config_set_text(f, FSL_CONFDB_CKOUT, "repository",
                             dbR->filename);
  }
  if(!rc) rc = fsl_update_ckout_dir(f);
  return rc;
}

int fsl__is_locally_modified(fsl_cx * const f,
                            const char * zFilename,
                            fsl_size_t origSize,
                            const char * zOrigHash,
                            fsl_int_t zOrigHashLen,
                            fsl_fileperm_e origPerm,
                            int * isModified){
  int rc = 0;
  int const hashLen = zOrigHashLen>=0
    ? zOrigHashLen : fsl_is_uuid(zOrigHash);
  fsl_buffer * hash = 0;
  fsl_buffer * fname = fsl__cx_scratchpad(f);
  fsl_fstat * const fst = &f->cache.fstat;
  int mod = 0;
  if(!fsl_is_uuid_len(hashLen)){
    return fsl_cx_err_set(f, FSL_RC_RANGE, "%s(): invalid hash length "
                          "%d for file: %s", __func__, hashLen, zFilename);
  }else if(!f->ckout.dir){
    return fsl_cx_err_set(f, FSL_RC_NOT_A_CKOUT,
                          "%s() requires a checkout.", __func__);
  }
  if(!fsl_is_absolute_path(zFilename)){
    rc = fsl_file_canonical_name2(f->ckout.dir, zFilename, fname, false);
    if(rc) goto end;
    zFilename = fsl_buffer_cstr(fname);
  }
  rc = fsl_stat(zFilename, fst, false);
  if(0==rc){
    if(origSize!=fst->size){
      mod |= 0x02;
    }
    if((FSL_FILE_PERM_EXE==origPerm &&
        FSL_FSTAT_PERM_EXE!=fst->perm)
       || (FSL_FILE_PERM_EXE!=origPerm &&
           FSL_FSTAT_PERM_EXE==fst->perm)){
      mod |= 0x01;
    }else if((FSL_FILE_PERM_LINK==origPerm &&
              FSL_FSTAT_TYPE_LINK!=fst->type)
             || (FSL_FILE_PERM_LINK!=origPerm &&
                 FSL_FSTAT_TYPE_LINK==fst->type)){
      mod |= 0x04;
    }
    if(mod & 0x06) goto end;
    /* ^^^^^^^^^^ else we unfortunately need, for behavioral
       consistency, to fall through and determine whether the file
       contents differ. */
  }else{
    if(FSL_RC_NOT_FOUND==rc){
      rc = 0;
      mod = 0x10;
    }else{
      rc = fsl_cx_err_set(f, rc, "%s(): stat() failed for file: %s",
                          __func__, zFilename);
    }
    goto end;
  }
  hash = fsl__cx_scratchpad(f);
  switch(hashLen){
    case FSL_STRLEN_SHA1:
      rc = fsl_sha1sum_filename(zFilename, hash);
      break;
    case FSL_STRLEN_K256:
      rc = fsl_sha3sum_filename(zFilename, hash);
      break;
    default:
      fsl__fatal(FSL_RC_UNSUPPORTED, "This cannot happen. %s()",
                __func__);
  }
  if(rc){
    rc = fsl_cx_err_set(f, rc, "%s: error hashing file: %s",
                        __func__, zFilename);
  }else{
    assert(hashLen==(int)hash->used);
    mod |= memcmp(hash->mem, zOrigHash, (size_t)hashLen)
      ? 0x02 : 0;
    /*MARKER(("%d: %s %s %s\n", *isModified, zOrigHash,
      (char const *)hash.mem, zFilename));*/
  }
  end:
  if(!rc && isModified) *isModified = mod;
  fsl__cx_scratchpad_yield(f, fname);
  if(hash) fsl__cx_scratchpad_yield(f, hash);
  return rc;
}

/**
   Infrastructure for fsl_repo_ckout(),
   fsl_ckout_update(), and fsl_ckout_merge().
*/
typedef struct {
  /** The pre-checkout vfile.vid. 0 if no version was
      checked out. */
  fsl_id_t originRid;
  fsl_repo_extract_opt const * eOpt;
  fsl_ckup_opt const * cOpt;
  /* Checkout root. We re-use this when internally converting to
     absolute paths. */
  fsl_buffer * tgtDir;
  /* Initial length of this->tgtDir, including trailing slash */
  fsl_size_t tgtDirLen;
  /* Number of files we've written out so far. Used for adapting
     some error reporting. */
  fsl_size_t fileWriteCount;
  /* Stores the most recent fsl_cx_confirm() answer for questions
     about overwriting/removing modified files. (Exactly which answer
     it represents depends on the current phase of processing.)
  */
  fsl_confirm_response confirmAnswer;
  /* Is-changed vis-a-vis vfile query. */
  fsl_stmt stChanged;
  /* Is-same-filename-and-rid-in-vfile query. */
  fsl_stmt stIsInVfile;
  /* blob.size for vfile.rid query. */
  fsl_stmt stRidSize;
} RepoExtractCkup;

static const RepoExtractCkup RepoExtractCkup_empty = {
0/*originRid*/,NULL/*eOpt*/, NULL/*cOpt*/,
NULL/*tgtDir*/, 0/*tgtDirLen*/,
0/*fileWriteCount*/,
fsl_confirm_response_empty_m/*confirmAnswer*/,
fsl_stmt_empty_m/*stChanged*/,
fsl_stmt_empty_m/*stIsInVfile*/,
fsl_stmt_empty_m/*stRidSize*/
};

static const fsl_ckup_state fsl_ckup_state_empty = {
NULL/*xState*/, NULL/*callbackState*/,
FSL_CKUP_FCHANGE_INVALID/*fileChangeType*/,
FSL_CKUP_RM_NOT/*fileRmInfo*/,
0/*mtime*/,0/*size*/,
false/*dryRun*/
};

/**
   File modification types reported by
   fsl_reco_is_file_modified().
 */
typedef enum {
// Sentinel value
FSL_RECO_MOD_UNKNOWN,
// Not modified
FSL_RECO_MOD_NO,
// Modified
FSL_RECO_MOD_YES,
// "Unmanaged replaced by managed"
FSL_RECO_MOD_UnReMa
} fsl_ckup_localmod_e;

/**
   Determines whether the file referred to by the given
   checkout-root-relative file name, which is assumed to be known to
   exist, has been modified. It simply looks to the vfile state,
   rather than doing its own filesystem-level comparison. Returns 0 on
   success and stores its answer in *modType. Errors must be
   considered unrecoverable.
*/
static int fsl_reco_is_file_modified(fsl_cx *f, fsl_stmt * st,
                                     char const *zName,
                                     fsl_ckup_localmod_e * modType){
  int rc = 0;
  if(!st->stmt){ // no prior version
    *modType = FSL_RECO_MOD_NO;
    return 0;
  }
  fsl_stmt_reset(st);
  rc = fsl_stmt_bind_text(st, 1, zName, -1, false);
  if(rc){
    return fsl_cx_uplift_db_error2(f, st->db, rc);
  }
  rc = fsl_stmt_step(st);
  switch(rc){
    case FSL_RC_STEP_DONE:
      /* This can happen when navigating from a version in which a
         file was SCM-removed/unmanaged, but on disk, to a version
         where that file was in SCM. For now we'll mark these as
         modified but we need a better way of handling this case, and
         maybe a new FSL_CEVENT_xxx ID. */
      *modType = FSL_RECO_MOD_UnReMa;
      rc = 0;
      break;
    case FSL_RC_STEP_ROW:
      *modType = fsl_stmt_g_int32(st,0)>0
        ? FSL_RECO_MOD_YES : FSL_RECO_MOD_NO;
      rc = 0;
      break;
    default:
      rc = fsl_cx_uplift_db_error2(f, st->db, rc);
      break;
  }
  return rc;
}

/**
   Sets *isInVfile to true if the given combination of filename and
   file content RID are in the vfile table, as per
   RepoExtractCkup::stIsInVfile, else false. Returns non-0 on
   catastrophic failure.
*/
static int fsl_repo_co_is_in_vfile(fsl_stmt * st,
                                   char const *zFilename,
                                   fsl_id_t fileRid,
                                   bool *isInVfile){
  int rc = 0;
  if(st->stmt){
    fsl_stmt_reset(st);
    rc = fsl_stmt_bind_text(st, 1, zFilename, -1, false);
    if(!rc) rc = fsl_stmt_bind_id(st, 2, fileRid);
    if(!rc) *isInVfile = (FSL_RC_STEP_ROW==fsl_stmt_step(st));
  }else{ // no prior version
    *isInVfile = false;
  }
  return rc;
}

/**
   Infrastructure for fsl_repo_ckout(). This is the fsl_repo_extract_f
   impl which fsl_repo_extract() calls to give us the pieces we want to
   check out.

   When this is run (once for each row of the new checkout version),
   the vfile table still holds the state for the previous version, and
   we use that to determine whether a file is changed or new.
*/
static int fsl_repo_extract_f_ckout( fsl_repo_extract_state const * xs ){
  int rc = 0;
  fsl_cx * const f = xs->f;
  RepoExtractCkup * const rec = (RepoExtractCkup *)xs->callbackState;
  const char * zFilename;
  fsl_ckup_state coState = fsl_ckup_state_empty;
  fsl_time_t mtime = 0;
  fsl_fstat fst = fsl_fstat_empty;
  fsl_ckup_localmod_e modType = FSL_RECO_MOD_UNKNOWN;
  bool loadedContent = false;
  fsl_buffer * const content = &f->cache.fileContent;
  assert(0==content->used
         && "Internal Misuse of fsl_cx::fileContent buffer.");
  //assert(xs->content);
  assert(xs->fCard->uuid && "We shouldn't be getting deletions "
         "via delta manifests.");
  rc = fsl_buffer_append(rec->tgtDir, xs->fCard->name, -1);
  if(rc) return rc;
  fsl_buffer_reuse(content);
  coState.dryRun = rec->cOpt->dryRun;
  coState.fileRmInfo = FSL_CKUP_RM_NOT;
  coState.fileChangeType = FSL_CKUP_FCHANGE_INVALID;
  zFilename = fsl_buffer_cstr(rec->tgtDir);
  rc = fsl_stat(zFilename, &fst, 0);
  switch(rc){
    case 0:
      /* File exists. If it is modified, as reported by vfile, get
         confirmation before overwriting it, otherwise just overwrite
         it (or keep it - that's much more efficient). */
      mtime = fst.mtime;
      if(rec->confirmAnswer.response!=FSL_CRESPONSE_ALWAYS){
        rc = fsl_reco_is_file_modified(f, &rec->stChanged,
                                       xs->fCard->name, &modType);
        if(rc) goto end;
        switch(modType){
          case FSL_RECO_MOD_YES:
          case FSL_RECO_MOD_UnReMa:
            if(rec->confirmAnswer.response!=FSL_CRESPONSE_NEVER){
              fsl_confirm_detail detail = fsl_confirm_detail_empty;
              detail.eventId = FSL_RECO_MOD_YES==modType
                ? FSL_CEVENT_OVERWRITE_MOD_FILE
                : FSL_CEVENT_OVERWRITE_UNMGD_FILE;
              detail.filename = xs->fCard->name;
              rec->confirmAnswer.response = FSL_CRESPONSE_INVALID;
              rc = fsl_cx_confirm(f, &detail, &rec->confirmAnswer);
              if(rc) goto end;
            }
            break;
          case FSL_RECO_MOD_NO:{
            /** If vfile says that the content of this exact
                combination of filename and file RID is unchanged, we
                already have this content. If so, skip rewriting
                it. */
            bool isSameFile = false;
            rc = fsl_repo_co_is_in_vfile(&rec->stIsInVfile, xs->fCard->name,
                                         xs->fileRid, &isSameFile);
            if(rc) goto end;
            rec->confirmAnswer.response = isSameFile
              ? FSL_CRESPONSE_NO // We already have this content
              : FSL_CRESPONSE_YES; // Overwrite it
            coState.fileChangeType = isSameFile
              ? FSL_CKUP_FCHANGE_NONE
              : FSL_CKUP_FCHANGE_UPDATED;
            break;
          }
          default:
            fsl__fatal(FSL_RC_UNSUPPORTED,"Internal error: invalid "
                      "fsl_reco_is_file_modified() response.");
        }
      }
      switch(rec->confirmAnswer.response){
        case FSL_CRESPONSE_NO:
        case FSL_CRESPONSE_NEVER:
          // Keep existing.
          coState.fileChangeType = FSL_CKUP_FCHANGE_NONE;
          goto do_callback;
        case FSL_CRESPONSE_YES:
        case FSL_CRESPONSE_ALWAYS:
          // Overwrite it.
          coState.fileChangeType = FSL_CKUP_FCHANGE_UPDATED;
          break;
        case FSL_CRESPONSE_CANCEL:
          rc = fsl_cx_err_set(f, FSL_RC_BREAK,
                              "Checkout operation cancelled by "
                              "confirmation callback.%s",
                              rec->fileWriteCount
                              ? " Filesystem contents may now be "
                                "in an inconsistent state!"
                              : "");
          goto end;
        default:
          rc = fsl_cx_err_set(f, FSL_RC_MISUSE,
                              "Invalid response from confirmation "
                              "callback.");
          goto end;
      }
      break;
    case FSL_RC_NOT_FOUND:
      rc = 0;
      coState.fileChangeType = FSL_CKUP_FCHANGE_UPDATED;
      // Write it
      break;
    default:
      rc = fsl_cx_err_set(f, rc, "Error %s stat()'ing file: %s",
                          fsl_rc_cstr(rc), zFilename);
      goto end;
  }
  assert(FSL_CKUP_FCHANGE_INVALID != coState.fileChangeType);
  if(coState.dryRun){
    mtime = time(0);
  }else{
    if((rc=fsl_mkdir_for_file(zFilename, true))){
      rc = fsl_cx_err_set(f, rc, "mkdir() failed for file: %s", zFilename);
      goto end;
    }
    assert(!xs->content);
    rc = fsl_card_F_content(f, xs->fCard, content);
    if(rc) goto end;
    else if((rc=fsl_buffer_to_filename(content, zFilename))){
      rc = fsl_cx_err_set(f, rc, "Error %s writing to file: %s",
                          fsl_rc_cstr(rc), zFilename);
      goto end;
    }else{
      loadedContent = true;
      ++rec->fileWriteCount;
      mtime = time(0);
    }
    rc = fsl_file_exec_set(zFilename,
                           FSL_FILE_PERM_EXE == xs->fCard->perm);
    if(rc){
      rc = fsl_cx_err_set(f, rc, "Error %s changing file permissions: %s",
                          fsl_rc_cstr(rc), xs->fCard->name);
      goto end;
    }
  }
  if(rec->cOpt->setMtime){
    rc = fsl_mtime_of_manifest_file(xs->f, xs->checkinRid,
                                    xs->fileRid, &mtime);
    if(rc) goto end;
    if(!coState.dryRun){
      rc = fsl_file_mtime_set(zFilename, mtime);
      if(rc){
        rc = fsl_cx_err_set(f, rc, "Error %s setting mtime of file: %s",
                            fsl_rc_cstr(rc), zFilename);
        goto end;
      }
    }
  }
  do_callback:
  assert(0==rc);
  if(rec->cOpt->callback){
    assert(mtime);
    coState.mtime = mtime;
    coState.extractState = xs;
    coState.callbackState = rec->cOpt->callbackState;
    if(loadedContent){
      coState.size = content->used;
    }else{
      fsl_stmt_reset(&rec->stRidSize);
      fsl_stmt_bind_id(&rec->stRidSize, 1, xs->fileRid);
      coState.size =
        (FSL_RC_STEP_ROW==fsl_stmt_step(&rec->stRidSize))
        ? (fsl_int_t)fsl_stmt_g_int64(&rec->stRidSize, 0)
        : -1;
    }
    rc = rec->cOpt->callback( &coState );
  }
  end:
  fsl_buffer_reuse(content);
  rec->tgtDir->used = rec->tgtDirLen;
  rec->tgtDir->mem[rec->tgtDirLen] = 0;
  return rc;
}

/**
   For each file in vfile(vid=rec->originRid) which is not in the
   current vfile(vid=rec->cOpt->checkinRid), remove it from disk (or
   not, depending on confirmer response). Afterwards, try to remove
   any dangling directories left by that removal.

   Returns 0 on success. Ignores any filesystem-level errors during
   removal because, frankly, we have no recovery strategy for that
   case.

   TODO: do not remove dirs from the 'empty-dirs' config setting.
*/
static int fsl_repo_ckout_rm_list_fini(fsl_cx * f,
                                       RepoExtractCkup * rec){
  int rc;
  fsl_db * db = fsl_cx_db_ckout(f);
  fsl_stmt q = fsl_stmt_empty;
  fsl_buffer * absPath = fsl__cx_scratchpad(f);
  fsl_size_t const ckdirLen = f->ckout.dirLen;
  char const *zAbs;
  int rmCounter = 0;
  fsl_ckup_opt const * cOpt = rec->cOpt;
  fsl_ckup_state cuState = fsl_ckup_state_empty;
  fsl_repo_extract_state rxState = fsl_repo_extract_state_empty;
  fsl_card_F fCard = fsl_card_F_empty;
  
  assert(db);
  rc = fsl_buffer_append(absPath, f->ckout.dir,
                         (fsl_int_t)f->ckout.dirLen);
  if(rc) goto end;
  /* Select files which were in the previous version
     (rec->originRid) but are not in the newly co'd version
     (cOpt->checkinRid). */
  rc = fsl_db_prepare(db, &q,
                      "SELECT "
                      /*0*/"v.rid frid,"
                      /*1*/"v.pathname fn,"
                      /*2*/"b.uuid,"
                      /*3*/"v.isexe,"
                      /*4*/"v.islink,"
                      /*5*/"v.chnged, "
                      /*6*/"b.size "
                      "FROM vfile v, blob b "
                      "WHERE v.vid=%" FSL_ID_T_PFMT " "
                      "AND v.rid=b.rid "
                      "AND fn NOT IN "
                      "(SELECT pathname FROM vfile "
                      " WHERE vid=%" FSL_ID_T_PFMT
                      ") "
                      "ORDER BY fn %s /*%s()*/",
                      rec->originRid,
                      cOpt->checkinRid
                      /*new checkout version resp. update target
                        version*/,
                      fsl_cx_filename_collation(f),
                      __func__);
  if(rc) goto end;

  rec->confirmAnswer.response = FSL_CRESPONSE_INVALID;
  cuState.mtime = 0;
  cuState.size = -1;
  cuState.callbackState = cOpt->callbackState;
  cuState.extractState = &rxState;
  cuState.dryRun = cOpt->dryRun;
  cuState.fileChangeType = FSL_CKUP_FCHANGE_RM;
  rxState.f = f;
  rxState.fCard = &fCard;
  rxState.checkinRid = cOpt->checkinRid;
  while(FSL_RC_STEP_ROW==(rc = fsl_stmt_step(&q))){
    /**
       Each row is one file listed in vfile (the old checkout
       version) which is not in vfile (the new checkout).
    */
    fsl_size_t nFn = 0;
    fsl_size_t hashLen = 0;
    char const * fn = fsl_stmt_g_text(&q, 1, &nFn);
    char const * hash = fsl_stmt_g_text(&q, 2, &hashLen);
    bool const isChanged = fsl_stmt_g_int32(&q, 5)!=0;
    int64_t const fSize = fsl_stmt_g_int64(&q, 6);
    if(FSL_CRESPONSE_ALWAYS!=rec->confirmAnswer.response){
      /**
         If the user has previously responded to
         FSL_CEVENT_RM_MOD_UNMGD_FILE, keep that response, else
         ask again if the file was flagged as changed in the
         vfile table before all of this started.
      */
      if(isChanged){
        // Modified: ask user unless they've already answered NEVER.
        if(FSL_CRESPONSE_NEVER!=rec->confirmAnswer.response){
          fsl_confirm_detail detail = fsl_confirm_detail_empty;
          detail.eventId = FSL_CEVENT_RM_MOD_UNMGD_FILE;
          detail.filename = fn;
          rec->confirmAnswer.response = FSL_CRESPONSE_INVALID;
          rc = fsl_cx_confirm(f, &detail, &rec->confirmAnswer);
          if(rc) goto end;
        }
      }else{
        // Not modified. Nuke it.
        rec->confirmAnswer.response = FSL_CRESPONSE_YES;
      }
    }
    absPath->used = ckdirLen;
    rc = fsl_buffer_append(absPath, fn, nFn);
    if(rc) break;
    zAbs = fsl_buffer_cstr(absPath);
    /* Ignore deletion errors. We cannot roll back previous deletions,
       so failing here, which would roll back the transaction, could
       leave the checkout in a weird state, potentially with some
       files missing and others not. */
    switch(rec->confirmAnswer.response){
      case FSL_CRESPONSE_YES:
      case FSL_CRESPONSE_ALWAYS:
        //MARKER(("Unlinking: %s\n",zAbs));
        if(!cOpt->dryRun && 0==fsl_file_unlink(zAbs)){
          ++rmCounter;
        }
        cuState.fileRmInfo = FSL_CKUP_RM;
        break;
      case FSL_CRESPONSE_NO:
      case FSL_CRESPONSE_NEVER:
        //assert(FSL_RECO_MOD_YES==modType);
        //MARKER(("NOT removing locally-modified file: %s\n", zN));
        cuState.fileRmInfo = FSL_CKUP_RM_KEPT;
        break;
      case FSL_CRESPONSE_CANCEL:
        rc = fsl_cx_err_set(f, FSL_RC_BREAK,
                            "Checkout operation cancelled by "
                            "confirmation callback. "
                            "Filesystem contents may now be "
                            "in an inconsistent state!");
        goto end;
      default:
        fsl__fatal(FSL_RC_UNSUPPORTED,"Internal error: invalid "
                  "fsl_cx_confirm() response #%d.",
                  rec->confirmAnswer.response);
        break;
    }
    if(!cOpt->callback) continue;
    /* Now report the deletion to the callback... */
    fsl_id_t const frid = fsl_stmt_g_id(&q, 0);
    const bool isExe = 0!=fsl_stmt_g_int32(&q, 3);
    const bool isLink = 0!=fsl_stmt_g_int32(&q, 4);
    cuState.size = (FSL_CKUP_RM==cuState.fileRmInfo) ? -1 : fSize;
    rxState.fileRid = frid;
    fCard = fsl_card_F_empty;
    fCard.name = (char *)fn;
    fCard.uuid = (char *)hash;
    fCard.perm = isExe ? FSL_FILE_PERM_EXE :
      (isLink ? FSL_FILE_PERM_LINK : FSL_FILE_PERM_REGULAR);
    rc = cOpt->callback( &cuState );
    if(rc) goto end;
  }
  if(FSL_RC_STEP_DONE==rc) rc = 0;
  else goto end;
  if(rmCounter>0){
    /* Clean up any empty directories left over by removal of
       files... */
    assert(!cOpt->dryRun);
    fsl_stmt_finalize(&q);
    /* Select dirs which were in the previous version
       (rec->originRid) but are not in the newly co'd version
       (cOpt->checkinRid). Any of these may _potentially_
       be empty now. This query could be improved to filter 
       out more in advance. */
    rc = fsl_db_prepare(db, &q,
                        "SELECT DISTINCT(fsl_dirpart(pathname,0)) dir "
                        "FROM vfile "
                        "WHERE vid=%" FSL_ID_T_PFMT " "
                        "AND pathname NOT IN "
                        "(SELECT pathname FROM vfile "
                        "WHERE vid=%" FSL_ID_T_PFMT ") "
                        "AND dir IS NOT NULL "
                        "ORDER BY length(dir) DESC /*%s()*/",
                        /*get deepest dirs first*/
                        rec->originRid, cOpt->checkinRid,
                        __func__);
    if(rc) goto end;
    while(FSL_RC_STEP_ROW==(rc = fsl_stmt_step(&q))){
      fsl_size_t nFn = 0;
      char const * fn = fsl_stmt_g_text(&q, 0, &nFn);
      absPath->used = ckdirLen;
      rc = fsl_buffer_append(absPath, fn, nFn);
      if(rc) break;
      fsl__ckout_rm_empty_dirs(f, absPath)
        /* To see this in action, use (f-co tip) to check out the tip of
           a repo, then use (f-co rid:1) to back up to the initial empty
           checkin. It "should" leave you with a directory devoid of
           anything but .fslckout and any non-SCM'd content.
        */;
    }
    if(FSL_RC_STEP_DONE==rc) rc = 0;
  }
  end:
  fsl_stmt_finalize(&q);
  fsl__cx_scratchpad_yield(f, absPath);
  return fsl_cx_uplift_db_error2(f, db, rc);
}

int fsl_repo_ckout(fsl_cx * f, fsl_ckup_opt const * cOpt){
  int rc = 0;
  fsl_id_t const prevRid = f->ckout.rid;
  fsl_db * const dbR = fsl_needs_repo(f);
  RepoExtractCkup rec = RepoExtractCkup_empty;
  fsl_confirmer oldConfirm = fsl_confirmer_empty;
  if(!dbR) return f->error.code;
  else if(!fsl_needs_ckout(f)) return f->error.code;
  rc = fsl_cx_transaction_begin(f);
  if(rc) return rc;
  rec.tgtDir = fsl__cx_scratchpad(f);
  if(cOpt->confirmer.callback){
    fsl_cx_confirmer(f, &cOpt->confirmer, &oldConfirm);
  }
  //MARKER(("ckout.rid=%d\n",(int)prevRid));
  if(prevRid>=0 && cOpt->scanForChanges){
    /* We need to ensure this state is current in order to determine
       whether a given file is locally modified vis-a-vis the
       pre-extract checkout state. */
    rc = fsl_vfile_changes_scan(f, prevRid, 0);
    if(rc) goto end;
  }
  if(0){
    fsl_db_each(dbR,fsl_stmt_each_f_dump, NULL,
                "SELECT * FROM vfile ORDER BY pathname");
  }
  assert(f->ckout.dirLen);
  fsl_repo_extract_opt eOpt = fsl_repo_extract_opt_empty;
  rc = fsl_buffer_append(rec.tgtDir, f->ckout.dir,
                         (fsl_int_t)f->ckout.dirLen);
  if(rc) goto end;
  if(prevRid){
    rc = fsl_db_prepare(dbR, &rec.stChanged,
                        "SELECT chnged FROM vfile "
                        "WHERE vid=%" FSL_ID_T_PFMT
                        " AND pathname=? %s",
                        prevRid,
                        fsl_cx_filename_collation(f));
  }
  if(!rc && prevRid){
    /* Optimization: before we load content for a blob and write it to
       a file, check this query for whether we already have the same
       name/rid combination in vfile, and skip loading/writing the
       content if we do. */
    rc = fsl_db_prepare(dbR, &rec.stIsInVfile,
                        "SELECT 1 FROM vfile "
                        "WHERE vid=%" FSL_ID_T_PFMT
                        " AND pathname=? AND rid=? %s",
                        prevRid, fsl_cx_filename_collation(f));
  }
  if(!rc){
    /* Files for which we don't load content (see rec.stIsInVfile)
       still have a size we need to report via fsl_ckup_state,
       and we fetch that with this query. */
    rc = fsl_db_prepare(dbR, &rec.stRidSize,
                        "SELECT size FROM blob WHERE rid=?");
  }
  if(rc){
    rc = fsl_cx_uplift_db_error2(f, dbR, rc);
    goto end;
  }
  rec.originRid = prevRid;
  rec.tgtDirLen = f->ckout.dirLen;
  eOpt.checkinRid = cOpt->checkinRid;
  eOpt.extractContent = false;
  eOpt.callbackState = &rec;
  eOpt.callback = fsl_repo_extract_f_ckout;
  rec.eOpt = &eOpt;
  rec.cOpt = cOpt;
  rc = fsl_repo_extract(f, &eOpt);
  if(!rc){
    /*
      We need to call fsl_vfile_load(f, cOpt->vid) to
      populate vfile but we also need to call
      fsl_vfile_changes_scan(f, cOpt->vid, 0) to set the vfile.mtime
      fields. The latter calls the former, so...
    */
    rc = fsl_vfile_changes_scan(f, cOpt->checkinRid,
                                FSL_VFILE_CKSIG_WRITE_CKOUT_VERSION
                                |
                                (prevRid==0
                                 ? 0 : FSL_VFILE_CKSIG_KEEP_OTHERS)
                                |
                                (cOpt->setMtime
                                 ? 0 : FSL_VFILE_CKSIG_SETMTIME)
                                /* Note that mtimes were set during
                                   extraction if cOpt->setMtime is
                                   true. */);
    if(rc) goto end;
    assert(f->ckout.rid==cOpt->checkinRid);
    assert(f->ckout.rid ? !!f->ckout.uuid : 1);
  }
  if(!rc && prevRid!=0){
    rc = fsl_repo_ckout_rm_list_fini(f, &rec);
    if(rc) goto end;
  }
  rc = fsl_ckout_manifest_write(f, -1, -1, -1, NULL);

  end:
  if(!rc){
    rc = fsl_vfile_unload_except(f, cOpt->checkinRid);
    if(!rc) rc = fsl__ckout_clear_merge_state(f, true);
  }
  /*
    TODO: if "repo-cksum" config db setting is set, confirm R-card of
    cOpt->checkinRid against on-disk contents.
  */
  if(cOpt->confirmer.callback){
    fsl_cx_confirmer(f, &oldConfirm, NULL);
  }
  fsl_stmt_finalize(&rec.stChanged);
  fsl_stmt_finalize(&rec.stIsInVfile);
  fsl_stmt_finalize(&rec.stRidSize);
  fsl__cx_scratchpad_yield(f, rec.tgtDir);
  int const rc2 = fsl_cx_transaction_end(f, rc || cOpt->dryRun);
  return rc ? rc : rc2;
}

int fsl_ckout_update(fsl_cx * f, fsl_ckup_opt const *cuOpt){
  fsl_db * const dbR = fsl_needs_repo(f);
  fsl_db * const dbC = dbR ? fsl_needs_ckout(f) : 0;
  if(!dbR) return FSL_RC_NOT_A_REPO;
  else if(!dbC) return FSL_RC_NOT_A_CKOUT;
  int rc = 0, rc2 = 0;
  char const * collation = fsl_cx_filename_collation(f);
  fsl_id_t const ckRid = f->ckout.rid /* current version */;
  fsl_id_t const tid = cuOpt->checkinRid /* target version */;
  fsl_stmt q = fsl_stmt_empty;
  fsl_stmt mtimeXfer = fsl_stmt_empty;
  fsl_stmt mtimeGet = fsl_stmt_empty;
  fsl_stmt mtimeSet = fsl_stmt_empty;
  fsl_buffer * bFullPath = 0;
  fsl_buffer * bFullNewPath = 0;
  fsl_buffer * bFileUuid = 0;
  fsl_repo_extract_opt eOpt = fsl_repo_extract_opt_empty
    /* We won't actually use fsl_repo_extract() here because it's a
       poor fit for the update selection algorithm, but in order to
       consolidate some code between the ckout/update cases we need to
       behave as if we were using it. */;
  fsl_repo_extract_state xState = fsl_repo_extract_state_empty;
  fsl_card_F fCard = fsl_card_F_empty;
  fsl_ckup_state uState = fsl_ckup_state_empty;
  RepoExtractCkup rec = RepoExtractCkup_empty;
  enum { MergeBufCount = 4 };
  fsl_buffer bufMerge[MergeBufCount] = {
    fsl_buffer_empty_m/* pivot: ridv */,
    fsl_buffer_empty_m/* local file to merge into */,
    fsl_buffer_empty_m/* update-to: ridt */,
    fsl_buffer_empty_m/* merged copy */
  };

  rc = fsl_db_transaction_begin(dbC);
  if(rc) return fsl_cx_uplift_db_error2(f, dbC, rc);
  if(cuOpt->scanForChanges){
    rc = fsl_vfile_changes_scan(f, ckRid, FSL_VFILE_CKSIG_ENOTFILE);
    if(rc) goto end;
  }
  if(tid != ckRid){
    uint32_t missingCount = 0;
    rc = fsl_vfile_load(f, tid, false,
                                 &missingCount);
    if(rc) goto end;
    else if(missingCount/* && !forceMissing*/){
      rc = fsl_cx_err_set(f, FSL_RC_PHANTOM,
                          "Unable to update due to missing content in "
                          "%"PRIu32" blob(s).", missingCount);
      goto end;
    }
  }
  /*
  ** The record.fn field is used to match files against each other.  The
  ** FV table contains one row for each each unique filename in
  ** in the current checkout, the pivot, and the version being merged.
  */
  rc = fsl_db_exec_multi(dbC,
    "CREATE TEMP TABLE IF NOT EXISTS fv("
    "  fn TEXT %s PRIMARY KEY,"   /* The filename relative to root */
    "  idv INTEGER,"              /* VFILE entry for current version */
    "  idt INTEGER,"              /* VFILE entry for target version */
    "  chnged BOOLEAN,"           /* True if current version has been edited */
    "  islinkv BOOLEAN,"          /* True if current file is a link */
    "  islinkt BOOLEAN,"          /* True if target file is a link */
    "  ridv INTEGER,"             /* Record ID for current version */
    "  ridt INTEGER,"             /* Record ID for target */
    "  isexe BOOLEAN,"            /* Does target have execute permission? */
    "  deleted BOOLEAN DEFAULT 0,"/* File marked by "rm" to become unmanaged */
    "  fnt TEXT %s"               /* Filename of same file on target version */
    ") /*%s()*/; "
    "DELETE FROM fv;",
    collation, collation, __func__ );
  if(rc) goto dberr;
  /* Add files found in the current version
  */
  rc = fsl_db_exec_multi(dbC,
    "INSERT OR IGNORE INTO fv("
            "fn,fnt,idv,idt,ridv,"
            "ridt,isexe,chnged,deleted"
    ") SELECT pathname, pathname, id, 0, rid, 0, "
       "isexe, chnged, deleted "
       "FROM vfile WHERE vid=%" FSL_ID_T_PFMT
       "/*%s()*/",
    ckRid, __func__
  );
  if(rc) goto dberr;

  /* Compute file name changes on V->T.  Record name changes in files that
  ** have changed locally.
  */
  if( ckRid ){
    uint32_t nChng = 0;
    fsl_id_t * aChng = 0;
    rc = fsl__find_filename_changes(f, ckRid, tid,
                                    true, &nChng, &aChng);
    if(rc){
      assert(!aChng);
      assert(!nChng);
      goto end;
    }
    if( nChng ){
      for(uint32_t i=0; i<nChng; ++i){
        rc = fsl_db_exec_multi(dbC,
          "UPDATE fv"
          "   SET fnt=(SELECT name FROM filename WHERE fnid=%"
              FSL_ID_T_PFMT ")"
          " WHERE fn=(SELECT name FROM filename WHERE fnid=%"
            FSL_ID_T_PFMT ") AND chnged /*%s()*/",
          aChng[i*2+1], aChng[i*2], __func__
        );
        if(rc) goto dberr;
      }
      fsl_free(aChng);
    }else{
      assert(!aChng);
    }
  }/*ckRid!=0*/

  /* Add files found in the target version T but missing from the current
  ** version V.
  */
  rc = fsl_db_exec_multi(dbC,
    "INSERT OR IGNORE INTO fv(fn,fnt,idv,idt,ridv,ridt,isexe,chnged)"
    " SELECT pathname, pathname, 0, 0, 0, 0, isexe, 0 FROM vfile"
    "  WHERE vid=%" FSL_ID_T_PFMT
    "    AND pathname %s NOT IN (SELECT fnt FROM fv) /*%s()*/",
    tid, collation, __func__
  );
  if(rc) goto dberr;

  /*
  ** Compute the file version ids for T
  */
  rc = fsl_db_exec_multi(dbC,
    "UPDATE fv SET"
    " idt=coalesce((SELECT id FROM vfile WHERE vid=%"
                   FSL_ID_T_PFMT " AND fnt=pathname),0),"
    " ridt=coalesce((SELECT rid FROM vfile WHERE vid=%"
                    FSL_ID_T_PFMT " AND fnt=pathname),0) /*%s()*/",
    tid, tid, __func__
  );
  if(rc) goto dberr;

  /*
  ** Add islink information
  */
  rc = fsl_db_exec_multi(dbC,
    "UPDATE fv SET"
    " islinkv=coalesce((SELECT islink FROM vfile"
                       " WHERE vid=%" FSL_ID_T_PFMT
                         " AND fnt=pathname),0),"
    " islinkt=coalesce((SELECT islink FROM vfile"
                       " WHERE vid=%" FSL_ID_T_PFMT
                         " AND fnt=pathname),0) /*%s()*/",
    ckRid, tid, __func__
  );
  if(rc) goto dberr;

  /**
     Right here, fossil(1) permits passing on a subset of
     filenames/dirs to update, but it's apparently a little-used
     feature and we're going to skip it for the time being:

     https://fossil-scm.org/forum/forumpost/1da828facf
   */

  /*
  ** Alter the content of the checkout so that it conforms with the
  ** target
  */
  rc = fsl_db_prepare(dbC, &q,
                      "SELECT fn, idv, ridv, "/* 0..2  */
                      "idt, ridt, chnged, "   /* 3..5  */
                      "fnt, isexe, islinkv, " /* 6..8  */
                      "islinkt, deleted "     /* 9..10 */
                      "FROM fv ORDER BY 1 /*%s()*/",
                      __func__);
  if(rc) goto dberr;
  rc = fsl_db_prepare(dbC, &mtimeXfer,
                      "UPDATE vfile SET mtime=(SELECT mtime FROM vfile "
                      "WHERE id=?1/*idv*/) "
                      "WHERE id=?2/*idt*/ /*%s()*/",
                      __func__);
  if(rc) goto dberr;
  rc = fsl_db_prepare(dbR, &rec.stChanged,
                      "SELECT chnged FROM vfile "
                      "WHERE vid=%" FSL_ID_T_PFMT
                      " AND pathname=? %s /*%s()*/",
                      ckRid, collation, __func__);
  if(rc) goto dberr;
  if(cuOpt->callback){
    /* Queries we need only if we need to collect info for a
       callback... */
    rc = fsl_db_prepare(dbC, &mtimeGet,
                        "SELECT mtime FROM vfile WHERE id=?1"/*idt*/);
    if(rc) goto dberr;
    rc = fsl_db_prepare(dbC, &mtimeSet,
                        "UPDATE vfile SET mtime=?2 WHERE id=?1"/*idt*/);
    if(rc) goto dberr;
    /* Files for which we don't load content still have a size we need
       to report via fsl_ckup_state, and we fetch that with this
       query. */
    rc = fsl_db_prepare(dbR, &rec.stRidSize,
                        "SELECT size FROM blob WHERE rid=?");
    if(rc) goto dberr;
  }

  xState.f = f;
  xState.fCard = &fCard;
  xState.checkinRid = eOpt.checkinRid = tid;
  xState.count.fileCount =
    (uint32_t)fsl_db_g_int32(dbC, 0, "SELECT COUNT(*) FROM vfile "
                             "WHERE vid=%" FSL_ID_T_PFMT,
                             tid);
  uState.extractState = &xState;
  uState.callbackState = cuOpt->callbackState;
  uState.dryRun = cuOpt->dryRun;
  uState.fileRmInfo = FSL_CKUP_RM_NOT;
  rec.originRid = ckRid;
  rec.eOpt = &eOpt;
  rec.cOpt = cuOpt;
  rec.tgtDir = fsl__cx_scratchpad(f);
  rec.tgtDirLen = f->ckout.dirLen;
  rc = fsl_buffer_append(rec.tgtDir, f->ckout.dir,
                         (fsl_int_t)f->ckout.dirLen);
  if(rc) goto end;

  /**
     Missing features from fossil we still need for this include,
     but are not limited to:

     - file_unsafe_in_tree_path() (done, untested)
     - file_nondir_objects_on_path() (done, untested)
     - symlink_create() (done, untested)
     - ...
  */
  bFullPath = fsl__cx_scratchpad(f);
  bFullNewPath = fsl__cx_scratchpad(f);
  bFileUuid = fsl__cx_scratchpad(f);
  rc = fsl_buffer_append(bFullPath, f->ckout.dir,
                         (fsl_int_t)f->ckout.dirLen);
  if(rc) goto end;
  rc = fsl_buffer_append(bFullNewPath, f->ckout.dir,
                         (fsl_int_t)f->ckout.dirLen);
  if(rc) goto end;
  unsigned int nConflict = 0;
  while( FSL_RC_STEP_ROW==fsl_stmt_step(&q) ){
    const char *zName = fsl_stmt_g_text(&q, 0, NULL)
      /* The filename from root */;
    fsl_id_t const idv = fsl_stmt_g_id(&q, 1)
      /* VFILE entry for current */;
    fsl_id_t const ridv = fsl_stmt_g_id(&q, 2)
      /* RecordID for current */;
    fsl_id_t const idt = fsl_stmt_g_id(&q, 3)
      /* VFILE entry for target */;
    fsl_id_t const ridt = fsl_stmt_g_id(&q, 4)
      /* RecordID for target */;
    int const chnged = fsl_stmt_g_int32(&q, 5)
      /* Current is edited */;
    const char *zNewName = fsl_stmt_g_text(&q,6, NULL)
      /* New filename */;
    int const isexe = fsl_stmt_g_int32(&q, 7)
      /* EXE perm for new file */;
    int const islinkv = fsl_stmt_g_int32(&q, 8)
      /* Is current file is a link */;
    int const islinkt = fsl_stmt_g_int32(&q, 9)
      /* Is target file is a link */;
    int const deleted = fsl_stmt_g_int32(&q, 10)
      /* Marked for deletion */;
    char const *zFullPath /* Full pathname of the file */;
    char const *zFullNewPath /* Full pathname of dest */;
    bool const nameChng = !!fsl_strcmp(zName, zNewName)
      /* True if the name changed */;
    int wasWritten = 0
      /* 1=perms written to disk, 2=content written */;
    fsl_fstat fst = fsl_fstat_empty;
    if(chnged || isexe || islinkv || islinkt){/*unused*/}
    
    bFullPath->used = bFullNewPath->used = f->ckout.dirLen;
    rc = fsl_buffer_appendf(bFullPath, zName, -1);
    if(!rc) rc = fsl_buffer_appendf(bFullNewPath, zNewName, -1);
    if(rc) goto end;
    zFullPath = fsl_buffer_cstr(bFullPath);
    zFullNewPath = fsl_buffer_cstr(bFullNewPath);
    uState.mtime = 0;
    uState.fileChangeType = FSL_CKUP_FCHANGE_INVALID;
    uState.fileRmInfo = FSL_CKUP_RM_NOT;
    ++xState.count.fileNumber;
    //MARKER(("#%03u/%03d %s\n", xState.count.fileNumber, xState.count.fileCount, zName));
    if( deleted ){
      /* Carry over pending file deletions from the current version
         into the target version. If the file was already deleted in
         the target version, that will be picked up by the file-deletion
         loop later on. */
      uState.fileChangeType = FSL_CKUP_FCHANGE_RM_PROPAGATED;
      rc = fsl_db_exec(dbC, "UPDATE vfile SET deleted=1 "
                       "WHERE id=%" FSL_ID_T_PFMT" /*%s()*/",
                       idt, __func__);
      if(rc) goto dberr;
    }
    if( idv>0 && ridv==0 && idt>0 && ridt>0 ){
      /* Conflict.  This file has been added to the current checkout
      ** but also exists in the target checkout.  Use the current version.
      */
      uState.fileChangeType = FSL_CKUP_FCHANGE_CONFLICT_ADDED;
      //fossil_print("CONFLICT %s\n", zName);
      nConflict++;
    }else if( idt>0 && idv==0 ){
      /* File added in the target. */
      if( fsl_is_file_or_link(zFullPath) ){
        //fossil_print("ADD %s - overwrites an unmanaged file\n", zName);
        uState.fileChangeType =
          FSL_CKUP_FCHANGE_CONFLICT_ADDED_UNMANAGED;
        //nOverwrite++;
        /* TODO/FIXME: if the files have the same content, treat this
           as FSL_CKUP_FCHANGE_ADDED. If they don't, use confirmer to
           ask the user what to do. */
      }else{
        //fsl_outputf(f, "ADD %s\n", zName);
        uState.fileChangeType = FSL_CKUP_FCHANGE_ADDED;
      }
      //if( !dryRunFlag && !internalUpdate ) undo_save(zName);
      if( !cuOpt->dryRun ){
        rc = fsl__vfile_to_ckout(f, idt, &wasWritten);
        if(rc) goto end;
      }
    }else if( idt>0 && idv>0 && ridt!=ridv && (chnged==0 || deleted) ){
      /* The file is unedited.  Change it to the target version */
      if( deleted ){
        //fossil_print("UPDATE %s - change to unmanaged file\n", zName);
        uState.fileChangeType = FSL_CKUP_FCHANGE_RM;
      }else{
        //fossil_print("UPDATE %s\n", zName);
        uState.fileChangeType = FSL_CKUP_FCHANGE_UPDATED;
      }
      if( !cuOpt->dryRun ){
        rc = fsl__vfile_to_ckout(f, idt, &wasWritten);
        if(rc) goto end;
      }
    }else if( idt>0 && idv>0 && !deleted &&
              0!=fsl_stat(zFullPath, NULL, false) ){
      /* The file is missing from the local check-out. Restore it to
      ** the version that appears in the target. */
      uState.fileChangeType = FSL_CKUP_FCHANGE_UPDATED;
      if( !cuOpt->dryRun ){
        rc = fsl__vfile_to_ckout(f, idt, &wasWritten);
        if(rc) goto end;
      }
    }else if( idt==0 && idv>0 ){
      /* Is in the current version but not in the target. */
      if( ridv==0 ){
        /* Added in current checkout.  Continue to hold the file as
        ** as an addition */
        uState.fileChangeType = FSL_CKUP_FCHANGE_ADD_PROPAGATED;
        rc = fsl_db_exec(dbC, "UPDATE vfile SET vid=%" FSL_ID_T_PFMT
                         " WHERE id=%" FSL_ID_T_PFMT " /*%s()*/",
                         tid, idv, __func__);
        if(rc) goto dberr;
      }else if( chnged ){
        /* Edited locally but deleted from the target.  Do not track the
        ** file but keep the edited version around. */
        uState.fileChangeType = FSL_CKUP_FCHANGE_CONFLICT_RM;
        ++nConflict;
        uState.fileRmInfo = FSL_CKUP_RM_KEPT;
        /* Delete idv from vfile so that the post-processing rm
           loop will not delete this file. */
        rc = fsl_db_exec(dbC, "DELETE FROM vfile WHERE id=%"
                         FSL_ID_T_PFMT " /*%s()*/",
                         idv, __func__);
        if(rc) goto dberr;

      }else{
        uState.fileChangeType = FSL_CKUP_FCHANGE_RM;
        if( !cuOpt->dryRun ){
          fsl_file_unlink(zFullPath)/*ignore errors*/;
          /* At this point fossil(1) adds each directory to the
             dir_to_delete table. We can probably use the same
             infrastructure which ckout uses, though. One
             hiccup there is that our infrastructure does not
             handle the locally-modified-removed case from the
             block above this one. */
        }
      }
    }else if( idt>0 && idv>0 && ridt!=ridv && chnged ){
      /* Merge the changes in the current tree into the target version */
      if( islinkv || islinkt ){
        uState.fileChangeType = FSL_CKUP_FCHANGE_CONFLICT_SYMLINK;
        ++nConflict;
      }else{
        unsigned int conflictCount = 0;
        for(int i = 0; i < MergeBufCount; ++i){
          fsl_buffer_reuse(&bufMerge[i]);
        }
        rc = fsl_content_get(f, ridv, &bufMerge[0]);
        if(!rc) rc = fsl_content_get(f, ridt, &bufMerge[2]);
        if(!rc){
          rc = fsl_buffer_fill_from_filename(&bufMerge[1], zFullPath);
        }
        if(rc) goto end;
        rc = fsl_buffer_merge3(&bufMerge[0], &bufMerge[1],
                               &bufMerge[2], &bufMerge[3],
                               &conflictCount);
        if(FSL_RC_TYPE==rc){
          /* Binary content: we can't merge this, so use target
             version. */
          rc = 0;
          uState.fileChangeType = FSL_CKUP_FCHANGE_UPDATED_BINARY;
          if( !cuOpt->dryRun ){
            rc = fsl_buffer_to_filename(&bufMerge[2], zFullNewPath);
            if(!rc) fsl_file_exec_set(zFullNewPath, !!isexe);
          }
        }else if(!rc){
          if( !cuOpt->dryRun ){
            rc = fsl_buffer_to_filename(&bufMerge[3], zFullNewPath);
            if(!rc) fsl_file_exec_set(zFullNewPath, !!isexe);
          }
          uState.fileChangeType = conflictCount
            ? FSL_CKUP_FCHANGE_CONFLICT_MERGED
            : FSL_CKUP_FCHANGE_MERGED;
          if(conflictCount) ++nConflict;
        }
        if(rc) goto end;
      }
      if( nameChng && !cuOpt->dryRun ){
        fsl_file_unlink(zFullPath);
      }
    }else{
      if( chnged ){
        if( !deleted ){
          uState.fileChangeType = FSL_CKUP_FCHANGE_EDITED;
        }else{
          assert(FSL_CKUP_FCHANGE_RM_PROPAGATED==uState.fileChangeType);
        }
      }else{
        uState.fileChangeType = FSL_CKUP_FCHANGE_NONE;
        rc = fsl_stmt_bind_step(&mtimeXfer, "RR", idv, idt);
        if(rc) goto dberr;
      }
    }
    if(wasWritten && cuOpt->setMtime){
      if(0==fsl_mtime_of_manifest_file(f, tid, ridt, &uState.mtime)){
        fsl_file_mtime_set(zFullNewPath, uState.mtime);
        rc = fsl_stmt_bind_step(&mtimeSet, "RI", idt, uState.mtime);
        if(rc) goto dberr;
      }
    }
    assert(FSL_CKUP_FCHANGE_INVALID != uState.fileChangeType);
    assert(!rc);
    if(cuOpt->callback
       && (FSL_CKUP_FCHANGE_RM != uState.fileChangeType)
       /* removals are reported separately in the file
          deletion phase */){
      if(FSL_CKUP_FCHANGE_ADD_PROPAGATED==uState.fileChangeType){
        /* This file is not yet in SCM, so its size is not in
           the db. */
        if(0==fsl_stat(zFullNewPath, &fst, false)){
          uState.size = (fsl_int_t)fst.size;
          uState.mtime = fst.mtime;
        }else{
          uState.size = -1;
        }
      }else{
        /* If we have the record's size in the db, use that. */
        fsl_stmt_bind_id(&rec.stRidSize, 1, ridt);
        if(FSL_RC_STEP_ROW==fsl_stmt_step(&rec.stRidSize)){
          uState.size = fsl_stmt_g_int32(&rec.stRidSize, 0);
        }else{
          uState.size = -1;
        }
        fsl_stmt_reset(&rec.stRidSize);
      }
      if(!uState.mtime){
        fsl_stmt_bind_id(&mtimeGet, 1, idt);
        if(FSL_RC_STEP_ROW==fsl_stmt_step(&mtimeGet)){
          uState.mtime = fsl_stmt_g_id(&mtimeGet, 0);
        }
        if(0==uState.mtime && 0==fsl_stat(zFullNewPath, &fst, false)){
          uState.mtime = fst.mtime;
        }
        fsl_stmt_reset(&mtimeGet);
      }
      xState.fileRid = ridt;
      fCard.name = (char *)zNewName;
      fCard.priorName = (char *)(nameChng ? zName : NULL);
      fCard.perm = islinkt ? FSL_FILE_PERM_LINK
        : (isexe ? FSL_FILE_PERM_EXE : FSL_FILE_PERM_REGULAR);
      if(ridt){
        rc = fsl_rid_to_uuid2(f, ridt, bFileUuid);
        if(rc) goto end;
        fCard.uuid = fsl_buffer_str(bFileUuid);
      }else{
        //MARKER(("ridt=%d uState.fileChangeType=%d name=%s\n",
        //        ridt, uState.fileChangeType, fCard.name));
        assert(FSL_CKUP_FCHANGE_CONFLICT_RM==uState.fileChangeType
               || FSL_CKUP_FCHANGE_ADD_PROPAGATED==uState.fileChangeType
               || FSL_CKUP_FCHANGE_EDITED==uState.fileChangeType
               );
        fCard.uuid = 0;
      }
      rc = cuOpt->callback( &uState );
      if(rc) goto end;
      uState.mtime = 0;
    }
  }/*fsl_stmt_step(&q)*/
  fsl_stmt_finalize(&q);
  if(nConflict){/*unused*/}
  /*
    At this point, fossil(1) does:

    ensure_empty_dirs_created(1);
    checkout_set_all_exe();
  */
  assert(!rc);
  rc = fsl_repo_ckout_rm_list_fini(f, &rec);
  if(!rc){
    rc = fsl_vfile_unload_except(f, tid);
  }
  if(!rc){
    rc = fsl__ckout_version_write(f, tid, 0);
  }
  
  end:
  /* clang bug? If we declare rc2 here, it says "expression expected".
     Moving the decl to the top resolves it. Wha? */
  if(rec.tgtDir) fsl__cx_scratchpad_yield(f, rec.tgtDir);
  if(bFullPath) fsl__cx_scratchpad_yield(f, bFullPath);
  if(bFullNewPath) fsl__cx_scratchpad_yield(f, bFullNewPath);
  if(bFileUuid) fsl__cx_scratchpad_yield(f, bFileUuid);
  for(int i = 0; i < MergeBufCount; ++i){
    fsl_buffer_clear(&bufMerge[i]);
  }
  fsl_stmt_finalize(&rec.stRidSize);
  fsl_stmt_finalize(&rec.stChanged);
  fsl_stmt_finalize(&mtimeGet);
  fsl_stmt_finalize(&mtimeSet);
  fsl_stmt_finalize(&q);
  fsl_stmt_finalize(&mtimeXfer);
  fsl_db_exec(dbC, "DROP TABLE fv /*%s()*/", __func__);
  rc2 = fsl_db_transaction_end(dbC, !!rc);
  return rc ? rc : rc2;
  dberr:
  assert(rc);
  rc = fsl_cx_uplift_db_error2(f, dbC, rc);
  goto end;
}


/** Helper for generating a list of ambiguous leaf UUIDs. */
struct AmbiguousLeavesOutput {
  int count;
  int rc;
  fsl_buffer * buffer;
};
typedef struct AmbiguousLeavesOutput AmbiguousLeavesOutput;
static const AmbiguousLeavesOutput AmbiguousLeavesOutput_empty =
  {0, 0, NULL};

static int fsl_stmt_each_f_ambiguous_leaves( fsl_stmt * stmt, void * state ){
  AmbiguousLeavesOutput * alo = (AmbiguousLeavesOutput*)state;
  if(alo->count++){
    alo->rc = fsl_buffer_append(alo->buffer, ", ", 2);
  }
  if(!alo->rc){
    fsl_size_t n = 0;
    char const * uuid = fsl_stmt_g_text(stmt, 0, &n);
    assert(n==FSL_STRLEN_SHA1 || n==FSL_STRLEN_K256);
    alo->rc = fsl_buffer_append(alo->buffer, uuid, 16);
  }
  return alo->rc;
}

int fsl_ckout_calc_update_version(fsl_cx * f, fsl_id_t * outRid){
  fsl_db * const dbRepo = fsl_needs_repo(f);
  if(!dbRepo) return FSL_RC_NOT_A_REPO;
  else if(!fsl_needs_ckout(f)) return FSL_RC_NOT_A_CKOUT;
  int rc = 0;
  fsl_id_t tgtRid = 0;
  fsl_leaves_compute_e leafMode = FSL_LEAVES_COMPUTE_OPEN;
  fsl_id_t const ckRid = f->ckout.rid;
  rc = fsl_leaves_compute(f, ckRid, leafMode);
  if(rc) goto end;
  if( !fsl_leaves_computed_has(f) ){
    leafMode = FSL_LEAVES_COMPUTE_ALL;
    rc = fsl_leaves_compute(f, ckRid, leafMode);
    if(rc) goto end;
  }
  /* Delete [leaves] entries from any branches other than
     ckRid's... */
  rc = fsl_db_exec_multi(dbRepo,
        "DELETE FROM leaves WHERE rid NOT IN"
        "   (SELECT leaves.rid FROM leaves, tagxref"
        "     WHERE leaves.rid=tagxref.rid AND tagxref.tagid=%d"
        "       AND tagxref.value==(SELECT value FROM tagxref"
                                   " WHERE tagid=%d AND rid=%"
                             FSL_ID_T_PFMT "))",
        FSL_TAGID_BRANCH, FSL_TAGID_BRANCH, ckRid
  );
  if(rc) goto end;
  else if( fsl_leaves_computed_count(f)>1 ){
    AmbiguousLeavesOutput alo = AmbiguousLeavesOutput_empty;
    alo.buffer = fsl__cx_scratchpad(f);
    rc = fsl_buffer_append(alo.buffer,
                           "Multiple viable descendants found: ", -1);
    if(!rc){
      fsl_stmt q = fsl_stmt_empty;
      rc = fsl_db_prepare(dbRepo, &q, "SELECT uuid FROM blob "
                          "WHERE rid IN leaves ORDER BY uuid");
      if(!rc){
        rc = fsl_stmt_each(&q, fsl_stmt_each_f_ambiguous_leaves, &alo);
      }
      fsl_stmt_finalize(&q);
    }
    if(!rc){
      rc = fsl_cx_err_set(f, FSL_RC_AMBIGUOUS, "%b", alo.buffer);
    }
    fsl__cx_scratchpad_yield(f, alo.buffer);
  }
  end:
  if(!rc){
    tgtRid = fsl_leaves_computed_latest(f);
    *outRid = tgtRid;
    fsl_leaves_computed_cleanup(f)
      /* We might want to keep [leaves] around for the case where we
         return FSL_RC_AMBIGUOUS, to give the client a way to access
         that list in its raw form. Higher-level code could join that
         with the event table to give the user more context. */;
  }
  return rc;
}

void fsl_ckout_manifest_setting(fsl_cx *f, int *m){
  if(!m){
    f->cache.manifestSetting = -1;
    return;
  }else if(f->cache.manifestSetting>=0){
    *m = f->cache.manifestSetting;
    return;
  }
  char * str = fsl_config_get_text(f, FSL_CONFDB_VERSIONABLE,
                                   "manifest", NULL);
  if(!str){
    str = fsl_config_get_text(f, FSL_CONFDB_REPO,
                              "manifest", NULL);
  }
  *m = 0;
  if(str){
    char const * z = str;
    if('1'==*z || 0==fsl_strncmp(z,"on",2)
       || 0==fsl_strncmp(z,"true",4)){
      z = "ru"/*historical default*/;
    }else if(!fsl_str_bool(z)){
      z = "";
    }
    for(;*z;++z){
      switch(*z){
        case 'r': *m |= FSL_MANIFEST_MAIN; break;
        case 'u': *m |= FSL_MANIFEST_UUID; break;
        case 't': *m |= FSL_MANIFEST_TAGS; break;
        default: break;
      }
    }
    fsl_free(str);
  }
  f->cache.manifestSetting = (short)*m;
}

int fsl_ckout_manifest_write(fsl_cx * const f, int manifest, int manifestUuid,
                             int manifestTags,
                             int * const wrote){
  fsl_db * const db = fsl_needs_ckout(f);
  if(!db) return FSL_RC_NOT_A_CKOUT;
  else if(!f->ckout.rid){
    return fsl_cx_err_set(f, FSL_RC_RANGE,
                          "Checkout RID is 0, so it has no manifest.");
  }
  int W = 0;
  int rc = 0;
  fsl_buffer * const b = fsl__cx_scratchpad(f);
  fsl_buffer * const content = &f->cache.fileContent;
  char * str = 0;
  fsl_time_t const mtime = f->ckout.mtime>0
    ? fsl_julian_to_unix(f->ckout.mtime)
    : 0;
  fsl_buffer_reuse(content);
  if(manifest<0 || manifestUuid<0 || manifestTags<0){
    int setting = 0;
    fsl_ckout_manifest_setting(f, &setting);
    if(manifest<0 && setting & FSL_MANIFEST_MAIN) manifest=1;
    if(manifestUuid<0 && setting & FSL_MANIFEST_UUID) manifestUuid=1;
    if(manifestTags<0 && setting & FSL_MANIFEST_TAGS) manifestTags=1;
  }
  if(manifest || manifestUuid || manifestTags){
    rc = fsl_buffer_append(b, f->ckout.dir, (fsl_int_t)f->ckout.dirLen);
    if(rc) goto end;
  }
  if(manifest>0){
    rc = fsl_buffer_append(b, "manifest", 8);
    if(rc) goto end;
    rc = fsl_content_get(f, f->ckout.rid, content);
    if(rc) goto end;
    rc = fsl_buffer_to_filename(content, fsl_buffer_cstr(b));
    if(rc){
      rc = fsl_cx_err_set(f, rc, "Error writing file: %b", b);
      goto end;
    }
    if(mtime) fsl_file_mtime_set(fsl_buffer_cstr(b), mtime);
    W |= FSL_MANIFEST_MAIN;
  }else if(!fsl_db_exists(db,
                          "SELECT 1 FROM vfile WHERE "
                          "pathname='manifest' /*%s()*/",
                          __func__)){
    b->used = f->ckout.dirLen;
    rc = fsl_buffer_append(b, "manifest", 8);
    if(rc) goto end;
    fsl_file_unlink(fsl_buffer_cstr(b));
  }

  if(manifestUuid>0){
    b->used = f->ckout.dirLen;
    fsl_buffer_reuse(content);
    rc = fsl_buffer_append(b, "manifest.uuid", 13);
    if(rc) goto end;
    assert(f->ckout.uuid);
    rc = fsl_buffer_append(content, f->ckout.uuid, -1);
    if(!rc) rc = fsl_buffer_append(content, "\n", 1);
    if(rc) goto end;
    rc = fsl_buffer_to_filename(content, fsl_buffer_cstr(b));
    if(rc){
      rc = fsl_cx_err_set(f, rc, "Error writing file: %b", b);
      goto end;
    }
    if(mtime) fsl_file_mtime_set(fsl_buffer_cstr(b), mtime);
    W |= FSL_MANIFEST_UUID;
  }else if(!fsl_db_exists(db,
                          "SELECT 1 FROM vfile WHERE "
                          "pathname='manifest.uuid' /*%s()*/",
                          __func__)){
    b->used = f->ckout.dirLen;
    rc = fsl_buffer_append(b, "manifest.uuid", 13);
    if(rc) goto end;
    fsl_file_unlink(fsl_buffer_cstr(b));
  }

  if(manifestTags>0){
    fsl_stmt q = fsl_stmt_empty;
    fsl_db * const db = fsl_cx_db_repo(f);
    assert(db && "We can't have a checkout w/o a repo.");
    b->used = f->ckout.dirLen;
    fsl_buffer_reuse(content);
    rc = fsl_buffer_append(b, "manifest.tags", 13);
    if(rc) goto end;
    str = fsl_db_g_text(db, NULL, "SELECT VALUE FROM tagxref "
                        "WHERE rid=%" FSL_ID_T_PFMT
                        " AND tagid=%d /*%s()*/",
                        f->ckout.rid, FSL_TAGID_BRANCH, __func__);
    rc = fsl_buffer_appendf(content, "branch %z\n", str);
    str = 0;
    if(rc) goto end;
    rc = fsl_db_prepare(db, &q,
                        "SELECT substr(tagname, 5)"
                        "  FROM tagxref, tag"
                        " WHERE tagxref.rid=%" FSL_ID_T_PFMT
                        "   AND tagxref.tagtype>0"
                        "   AND tag.tagid=tagxref.tagid"
                        "   AND tag.tagname GLOB 'sym-*'"
                        " /*%s()*/",
                        f->ckout.rid, __func__);
    if(rc) goto end;
    while( FSL_RC_STEP_ROW==fsl_stmt_step(&q) ){
      const char *zName = fsl_stmt_g_text(&q, 0, NULL);
      rc = fsl_buffer_appendf(content, "tag %s\n", zName);
      if(rc) break;
    }
    fsl_stmt_finalize(&q);
    if(!rc){
      rc = fsl_buffer_to_filename(content, fsl_buffer_cstr(b));
      if(rc){
        rc = fsl_cx_err_set(f, rc, "Error writing file: %b", b);
      }
    }
    if(mtime) fsl_file_mtime_set(fsl_buffer_cstr(b), mtime);
    W |= FSL_MANIFEST_TAGS;
  }else if(!fsl_db_exists(db,
                          "SELECT 1 FROM vfile WHERE "
                          "pathname='manifest.tags' /*%s()*/",
                          __func__)){
    b->used = f->ckout.dirLen;
    rc = fsl_buffer_append(b, "manifest.tags", 13);
    if(rc) goto end;
    fsl_file_unlink(fsl_buffer_cstr(b));
  }

  end:
  if(wrote) *wrote = W;
  fsl__cx_scratchpad_yield(f, b);
  fsl_buffer_reuse(content);
  return rc;
}

/**
   Check every sub-directory of f's current checkout dir along the
   path to zFilename. If any sub-directory part is really an ordinary file
   or a symbolic link, set *errLen to the length of the prefix of zFilename
   which is the name of that object.

   Returns 0 except on allocation error, in which case it returned FSL_RC_OOM.
   If it finds nothing untowards about the path, *errLen will be set to 0.
   
   Example:  Given inputs
   
   ckout     = /home/alice/project1
   zFilename = /home/alice/project1/main/src/js/fileA.js
   
   Look for objects in the following order:
   
   /home/alice/project/main
   /home/alice/project/main/src
   /home/alice/project/main/src/js
   
   If any of those objects exist and are something other than a
   directory then *errLen will be the length of the name of the first
   non-directory object seen.

   If a given element of the path does not exist in the filesystem,
   traversal stops without an error.
*/
static int fsl_ckout_nondir_file_check(fsl_cx *f, char const * zFilename,
                                       fsl_size_t * errLen);

int fsl_ckout_nondir_file_check(fsl_cx *f, char const * zFilename,
                                fsl_size_t * errLen){
  if(!fsl_needs_ckout(f)) return FSL_RC_NOT_A_CKOUT;
  int rc = 0;
  int frc;
  fsl_buffer * const fn = fsl__cx_scratchpad(f);
  if(!fsl_is_rooted_in_ckout(f, zFilename)){
    assert(!"Misuse of this API. This condition should never fail.");
    rc = fsl_cx_err_set(f, FSL_RC_MISUSE, "Path is not rooted at the "
                        "current checkout directory: %s", zFilename);
    goto end;
  }
  rc = fsl_buffer_append(fn, zFilename, -1);
  if(rc) goto end;
  char * z = fsl_buffer_str(fn);
  fsl_size_t i = f->ckout.dirLen;
  fsl_size_t j;
  fsl_fstat fst = fsl_fstat_empty;
  char const * const zRoot = f->ckout.dir;
  if(i && '/'==zRoot[i-1]) --i;
  *errLen = 0;
  while( z[i]=='/' ){
    for(j=i+1; z[j] && z[j]!='/'; ++j){}
    if( z[j]!='/' ) break;
    z[j] = 0;
    frc = fsl_stat(z, &fst, false);
    if(frc){
      /* A not[-yet]-existing path element is okay */
      break;
    }
    if(FSL_FSTAT_TYPE_DIR!=fst.type){
      *errLen = j;
      break;
    }
    z[j] = '/';
    i = j;
  }
  end:
  fsl__cx_scratchpad_yield(f, fn);
  return rc;
}

int fsl__ckout_safe_file_check(fsl_cx * const f, char const * zFilename){
  if(!fsl_needs_ckout(f)) return FSL_RC_NOT_A_CKOUT;
  int rc = 0;
  fsl_buffer * const fn = fsl__cx_scratchpad(f);
  if(!fsl_is_absolute_path(zFilename)){
    rc = fsl_file_canonical_name2(f->ckout.dir, zFilename, fn, false);
    if(rc) goto end;
    zFilename = fsl_buffer_cstr(fn);
  }else if(!fsl_is_rooted_in_ckout(f, zFilename)){
    rc = fsl_cx_err_set(f, FSL_RC_MISUSE, "Path is not rooted at the "
                        "current checkout directory: %s", zFilename);
    goto end;
  }

  fsl_size_t errLen = 0;
  rc = fsl_ckout_nondir_file_check(f, zFilename, &errLen);
  if(rc) goto end /* OOM */;
  else if(errLen){
    rc = fsl_cx_err_set(f, FSL_RC_TYPE, "Directory part of path refers "
                        "to a non-directory: %.*s",
                        (int)errLen, zFilename);
  }
  end:
  fsl__cx_scratchpad_yield(f, fn);
  return rc;
}

bool fsl_is_rooted_in_ckout(fsl_cx * const f, char const * const zAbsPath){
  return f->ckout.dir
    ? 0==fsl_strncmp(zAbsPath, f->ckout.dir, f->ckout.dirLen)
    /* ^^^ fossil(1) uses stricmp() there, but that's a bug. However,
       NOT using stricmp() on case-insensitive filesystems is arguably
       also a bug. */
    : false;
}

int fsl_is_rooted_in_ckout2(fsl_cx * const f, char const * const zAbsPath){
  int rc = 0;
  if(!fsl_is_rooted_in_ckout(f, zAbsPath)){
    rc = fsl_cx_err_set(f, FSL_RC_RANGE, "Path is not rooted "
                        "in the current checkout: %s",
                        zAbsPath);
  }
  return rc;
}

int fsl__ckout_symlink_create(fsl_cx * const f, char const *zTgtFile,
                             char const * zLinkFile){
  if(!fsl_needs_ckout(f)) return FSL_RC_NOT_A_CKOUT;
  int rc = 0;
  fsl_buffer * const fn = fsl__cx_scratchpad(f);
  if(!fsl_is_absolute_path(zLinkFile)){
    rc = fsl_file_canonical_name2(f->ckout.dir, zLinkFile, fn, false);
    if(rc) goto end;
    zLinkFile = fsl_buffer_cstr(fn);
  }else if(0!=(rc = fsl_is_rooted_in_ckout2(f, zLinkFile))){
    goto end;
  }
  fsl_buffer * const b = fsl__cx_scratchpad(f);
  rc = fsl_buffer_append(b, zTgtFile, -1);
  if(!rc){
    rc = fsl_buffer_to_filename(b, fsl_buffer_cstr(fn));
  }
  fsl__cx_scratchpad_yield(f, b);
  end:
  fsl__cx_scratchpad_yield(f, fn);
  return rc;
}

/**
   Queues the directory part of the given filename into temp table
   fx_revert_rmdir for an eventual rmdir() attempt on it in
   fsl_revert_rmdir_fini().
*/
static int fsl_revert_rmdir_queue(fsl_cx * const f, fsl_db * const db,
                                  fsl_stmt * const st,
                                  char const * zFilename){
  int rc = 0;
  if( !st->stmt ){
    rc = fsl_cx_exec(f, "CREATE TEMP TABLE IF NOT EXISTS "
                     "fx_revert_rmdir(n TEXT PRIMARY KEY) "
                     "WITHOUT ROWID /* %s() */", __func__);
    if(0==rc) rc = fsl_cx_prepare(f, st, "INSERT OR IGNORE INTO "
                                  "fx_revert_rmdir(n) "
                                  "VALUES(fsl_dirpart(?,0)) /* %s() */",
                                  __func__);
  }
  if(0==rc){
    rc = fsl_stmt_bind_step(st, "s", zFilename);
    if(rc) rc = fsl_cx_uplift_db_error2(f, db, rc);
  }
  return rc;
}

/**
   Attempts to rmdir all dirs queued by fsl_revert_rmdir_queue(). Silently
   ignores rmdir failure but will return non-0 for db errors.
*/
static int fsl_revert_rmdir_fini(fsl_cx * const f){
  int rc;
  fsl_stmt st = fsl_stmt_empty;
  fsl_buffer * const b = fsl__cx_scratchpad(f);
  rc = fsl_cx_prepare(f, &st,
                      "SELECT fsl_ckout_dir()||n "
                      "FROM fx_revert_rmdir "
                      "ORDER BY length(n) DESC /* %s() */",
                      __func__);
  while(0==rc && FSL_RC_STEP_ROW == fsl_stmt_step(&st)){
    fsl_size_t nDir = 0;
    char const * zDir = fsl_stmt_g_text(&st, 0, &nDir);
    fsl_buffer_reuse(b);
    rc = fsl_buffer_append(b, zDir, (fsl_int_t)nDir);
    if(0==rc) fsl__ckout_rm_empty_dirs(f, b);
  }
  fsl__cx_scratchpad_yield(f, b);
  fsl_stmt_finalize(&st);
  return rc;
}

int fsl_ckout_revert( fsl_cx * const f,
                      fsl_ckout_revert_opt const * opt ){
  /**
     Reminder to whoever works on this code: the initial
     implementation was done almost entirely without the benefit of
     looking at fossil's implementation, thus this code is notably
     different from fossil's. If any significant misbehaviors are
     found here, vis a vis fossil, it might be worth reverting (as it
     were) to that implementation.
  */
  int rc;
  fsl_db * const db = fsl_needs_ckout(f);
  fsl_buffer * fname = 0;
  char const * zNorm = 0;
  fsl_id_t const vid = f->ckout.rid;
  bool inTrans = false;
  fsl_stmt q = fsl_stmt_empty;
  fsl_stmt vfUpdate = fsl_stmt_empty;
  fsl_stmt qRmdir = fsl_stmt_empty;
  fsl_buffer * sql = 0;
  if(!db) return FSL_RC_NOT_A_CKOUT;
  assert(vid>=0);
  if(!opt->vfileIds && opt->filename && *opt->filename){
    fname = fsl__cx_scratchpad(f);
    rc = fsl_ckout_filename_check(f, opt->relativeToCwd,
                                  opt->filename, fname);
    if(rc){
      fsl__cx_scratchpad_yield(f, fname);
      return rc;
    }
    zNorm = fsl_buffer_cstr(fname);
    /* MARKER(("fsl_ckout_unmanage(%d, %s) ==> %s\n", opt->relativeToCwd, opt->filename, zNorm)); */
    assert(zNorm);
    if(fname->used) fsl_buffer_strip_slashes(fname);
    if(1==fname->used && '.'==*zNorm){
      /* Special case: handle "." from ckout root intuitively */
      fsl_buffer_reuse(fname);
      assert(0==*zNorm);
    }
  }
  rc = fsl_cx_transaction_begin(f);
  if(rc) goto end;
  inTrans = true;
  if(opt->scanForChanges){
    rc = fsl_vfile_changes_scan(f, 0, 0);
    if(rc) goto end;
  }
  sql = fsl__cx_scratchpad(f);
  rc = fsl_buffer_appendf(sql, 
                          "SELECT id, rid, deleted, "
                          "fsl_ckout_dir()||pathname, "
                          "fsl_ckout_dir()||origname "
                          "FROM vfile WHERE vid=%" FSL_ID_T_PFMT " ",
                          vid);
  if(rc) goto end;
  if(zNorm && *zNorm){
    rc = fsl_buffer_appendf(sql,
                            "AND CASE WHEN %Q='' THEN 1 "
                            "ELSE ("
                            "     fsl_match_vfile_or_dir(pathname,%Q) "
                            "  OR fsl_match_vfile_or_dir(origname,%Q)"
                            ") END",
                            zNorm, zNorm, zNorm);
    if(rc) goto end;
  }else if(opt->vfileIds){
    rc = fsl_ckout_bag_to_ids(f, db, "fx_revert_id", opt->vfileIds);
    if(rc) goto end;
    rc = fsl_buffer_append(sql, "AND id IN fx_revert_id", -1);
    if(rc) goto end;
  }else{
    rc = fsl_buffer_append(sql,
                           "AND ("
                           " chnged<>0"
                           " OR deleted<>0"
                           " OR rid=0"
                           " OR coalesce(origname,pathname)"
                           "    <>pathname"
                           ")", -1);
  }
  assert(!rc);
  rc = fsl_cx_prepare(f, &q, "%b /* %s() */", sql, __func__);
  fsl__cx_scratchpad_yield(f, sql);
  sql = 0;
  if(rc) goto end;
  while((FSL_RC_STEP_ROW==fsl_stmt_step(&q))){
    fsl_id_t const id = fsl_stmt_g_id(&q, 0);
    fsl_id_t const rid = fsl_stmt_g_id(&q, 1);
    int32_t const deleted = fsl_stmt_g_int32(&q, 2);
    char const * const zName = fsl_stmt_g_text(&q, 3, NULL);
    char const * const zNameOrig = fsl_stmt_g_text(&q, 4, NULL);
    bool const renamed =
      zNameOrig ? !!fsl_strcmp(zName, zNameOrig) : false;
    fsl_ckout_revert_e changeType = FSL_REVERT_NONE;
    if(!rid){ // Added but not yet checked in.
      rc = fsl_cx_exec(f, "DELETE FROM vfile WHERE id=%" FSL_ID_T_PFMT,
                       id);
      if(rc) goto end;
      changeType = FSL_REVERT_UNMANAGE;
    }else{
      int wasWritten = 0;
      if(renamed){
        if((rc=fsl_mkdir_for_file(zNameOrig, true))){
          rc = fsl_cx_err_set(f, rc, "mkdir() failed for file: %s",
                              zNameOrig);
          break;
        }
        /* Move, if possible, the new name back over the original
           name. This will possibly allow fsl__vfile_to_ckout() to
           avoid having to load that file's contents and overwrite
           it. */
        int mvCheck = fsl_stat(zName, NULL, false);
        if(0==mvCheck || FSL_RC_NOT_FOUND==mvCheck){
          mvCheck = fsl_file_unlink(zNameOrig);
          if(0==mvCheck || FSL_RC_NOT_FOUND==mvCheck){
            if(0==fsl_file_rename(zName, zNameOrig)){
              rc = fsl_revert_rmdir_queue(f, db, &qRmdir, zName);
              if(rc) break;
            }
          }
        }
        /* Ignore any errors: this operation is an optimization,
           not a requirement. Worse case, the entry with the old
           name is left in the filesystem. */
      }
      if(!vfUpdate.stmt){
        rc = fsl_cx_prepare(f, &vfUpdate,
                            "UPDATE vfile SET chnged=0, deleted=0, "
                            "pathname=coalesce(origname,pathname), "
                            "origname=NULL "
                            "WHERE id=?1 /*%s()*/", __func__);
        if(rc) goto end;
      }
      rc = fsl_stmt_bind_step(&vfUpdate, "R", id)
        /* Has to be done before fsl__vfile_to_ckout() because that
           function writes to vfile.pathname. */;
      if(rc) goto dberr;
      rc = fsl__vfile_to_ckout(f, id, &wasWritten);
      if(rc) break;
      if(opt->callback){
        if(renamed){
          changeType = FSL_REVERT_RENAME;
        }else if(wasWritten){
          changeType = (2==wasWritten)
            ? FSL_REVERT_CONTENTS
            : FSL_REVERT_PERMISSIONS;
        }else if(deleted){
          changeType = FSL_REVERT_REMOVE;
        }
      }
    }
    if(opt->callback && FSL_REVERT_NONE!=changeType){
      char const * name = renamed ? zNameOrig : zName;
      rc = opt->callback(&name[f->ckout.dirLen],
                         changeType, opt->callbackState);
      if(rc) break;
    }
  }/*step() loop*/
  end:
  if(fname) fsl__cx_scratchpad_yield(f, fname);
  if(sql) fsl__cx_scratchpad_yield(f, sql);
  fsl_stmt_finalize(&q);
  fsl_stmt_finalize(&vfUpdate);
  if(qRmdir.stmt){
    fsl_stmt_finalize(&qRmdir);
    if(!rc) rc = fsl_revert_rmdir_fini(f);
    fsl_db_exec(db, "DROP TABLE IF EXISTS fx_revert_rmdir /* %s() */",
                __func__);
  }
  if(opt->vfileIds){
    fsl_db_exec_multi(db, "DROP TABLE IF EXISTS fx_revert_id "
                      "/* %s() */", __func__)
      /* Ignoring result code */;
  }
  if(0==rc){
    rc = fsl__ckout_clear_merge_state(f, false);
  }
  if(inTrans){
    int const rc2 = fsl_db_transaction_end(db, !!rc);
    if(!rc) rc = rc2;
  }
  return rc;
  dberr:
  assert(rc);
  rc = fsl_cx_uplift_db_error2(f, db, rc);
  goto end;
}

int fsl_ckout_vfile_ids( fsl_cx * const f, fsl_id_t vid,
                         fsl_id_bag * const dest, char const * zName,
                         bool relativeToCwd, bool changedOnly ) {
  if(!fsl_needs_ckout(f)) return FSL_RC_NOT_A_CKOUT;
  fsl_buffer * const canon = fsl__cx_scratchpad(f);
  int rc = fsl_ckout_filename_check(f, relativeToCwd, zName, canon);
  if(!rc){
    fsl_buffer_strip_slashes(canon);
    rc = fsl_filename_to_vfile_ids(f, vid, dest,
                                   fsl_buffer_cstr(canon),
                                   changedOnly);
  }
  fsl__cx_scratchpad_yield(f, canon);
  return rc;
}

int fsl_ckout_file_content(fsl_cx * const f, bool relativeToCwd, char const * zName,
                           fsl_buffer * const dest ){
  int rc;
  fsl_buffer * fname;
  if(!fsl_needs_ckout(f)) return FSL_RC_NOT_A_CKOUT;
  fname = fsl__cx_scratchpad(f);
  rc = fsl_file_canonical_name2( relativeToCwd
                                 ? NULL
                                 : fsl_cx_ckout_dir_name(f, NULL),
                                 zName, fname, 1 );
  if(!rc){
    assert(fname->used);
    if('/'==fname->mem[fname->used-1]){
      rc = fsl_cx_err_set(f, FSL_RC_MISUSE,"Filename may not have a trailing slash.");
      /* If we don't do this, we might end up reading a directory entry in raw form.
         Well, we still might. */
    }else{
      fsl_fstat fstat = fsl_fstat_empty;
      const char * zCanon = fsl_buffer_cstr(fname);
      rc = fsl_stat(zCanon, &fstat, true);
      if(rc){
        rc = fsl_cx_err_set(f, rc, "Cannot stat file; %b", fname);
      }else if(FSL_FSTAT_TYPE_FILE!=fstat.type){
        rc = fsl_cx_err_set(f, FSL_RC_TYPE,
                            "Not a regular file file; %b", fname);
      }else{
        dest->used =0;
        rc = fsl_buffer_fill_from_filename(dest, fsl_buffer_cstr(fname));
        if(rc){
          rc = fsl_cx_err_set(f, rc, "%s error reading file; %b",
                              fsl_rc_cstr(rc), fname);
        }
      }
    }
  }
  fsl__cx_scratchpad_yield(f, fname);
  return rc;
}

int fsl_card_F_ckout_mtime(fsl_cx * const f,
                           fsl_id_t vid,
                           fsl_card_F const * fc,
                           fsl_time_t * const repoMtime,
                           fsl_time_t * const localMtime){

  int rc = 0;
  fsl_id_t fid = 0;
  fsl_fstat fst = fsl_fstat_empty;
  if(!fsl_needs_ckout(f)) return FSL_RC_NOT_A_CKOUT;
  if(0>=vid){
    fsl_ckout_version_info(f, &vid, NULL);
  }
  fid = fsl_repo_filename_fnid(f, fc->name);
  if(fid<=0){
    rc = fsl_cx_err_get(f, NULL, NULL);
    return rc ? rc : fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                                    "Could not resolve filename: %s",
                                    fc->name);
  }
  else if(!fid){
    return fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                          "Could not resolve filename: %s",
                          fc->name);
  }
  if(localMtime){
    rc = fsl_cx_stat(f, 0, fc->name, &fst);
    if(rc){
      return fsl_cx_err_set(f, rc, "Could not stat() file: %s",
                            fc->name);
    }
    *localMtime = fst.mtime;
  }
  if(repoMtime){
    rc = fsl_mtime_of_manifest_file(f, vid, fid, repoMtime);
  }
  return rc;
}


char const ** fsl_ckout_dbnames(void){
  static char const *dbNames[] = {".fslckout", "_FOSSIL_", NULL};
  return dbNames;
}

char const * fsl_is_top_of_ckout(char const *zDirName){
  // counterpart of fossil(1)'s vfile_top_of_checkout()
  enum {BufLen = 2048};
  char nameBuf[BufLen];
  char * z = &nameBuf[0];
  fsl_size_t sz = fsl_strlcpy(z, zDirName, BufLen);
  if(sz>=(fsl_size_t)BufLen - 11/*_FOSSIL_/.fslckout suffix*/) return NULL;
  char const **dbNames = fsl_ckout_dbnames();
  char const * dbName;
  z[sz++] = '/';
  z[sz] = 0;
  for( int i = 0; NULL!=(dbName=dbNames[i]); ++i){
    fsl_strlcpy(z + sz , dbName, (fsl_size_t)BufLen - sz);
    if(fsl_file_size(z)>=1024) return dbName;
  }
  return NULL;
}

/**
   Internal helper for fsl_ckout_rename(). Performs the vfile update
   for renaming zFrom to zTo, taking into account certain
   vfile-semantics error conditions.
*/
static int fsl__mv_one_file(fsl_cx * const f, char const * zFrom,
                            char const * zTo, bool dryRun){
  fsl_db * const db = fsl_cx_db_ckout(f);
  int deleted = fsl_db_g_int32(db, -1,
                               "SELECT deleted FROM vfile WHERE vid=%"FSL_ID_T_PFMT
                               " AND pathname=%Q %s",
                               f->ckout.rid, zTo, fsl_cx_filename_collation(f));
  if(deleted>=0){
    if(0==deleted){
      if( !fsl_cx_is_case_sensitive(f,false) &&
          0==fsl_stricmp(zFrom, zTo) ){
        /* Case change only */
      }else{
        return fsl_cx_err_set(f, FSL_RC_ALREADY_EXISTS,
                              "Cannot rename '%s' to '%s' because "
                              "another file named '%s' is already "
                              "under management.", zFrom, zTo, zTo);
      }
    }else{
      return fsl_cx_err_set(f, FSL_RC_CONSISTENCY,
                            "Cannot rename '%s' to '%s' because "
                            "a pending deletion of '%s' has not "
                            "yet been checked in.", zFrom, zTo, zTo);
    }
  }
  int rc = 0;
  if( !dryRun ){
    rc = fsl_cx_exec(f, "UPDATE vfile SET pathname=%Q WHERE "
                     "pathname=%Q %s AND vid=%"FSL_ID_T_PFMT,
                     zTo, zFrom, fsl_cx_filename_collation(f),
                     f->ckout.rid);
  }
  return rc;
}

/**
   Internal helper for fsl_ckout_rename(). Performs the filesystem-level
   moving of all files in the TEMP.ckout_mv table. All fs-level errors
   _are ignored_.
*/
static int fsl__rename_process_fmove(fsl_cx * const f){
  int rc = 0;
  fsl_stmt q = fsl_stmt_empty;
  bool const allowSymlinks = fsl_cx_allows_symlinks(f, false);
  rc = fsl_cx_prepare(f, &q, "SELECT fsl_ckout_dir()||f, "
                      "fsl_ckout_dir()||t "
                      "FROM ckout_mv ORDER BY 1");
  while(0==rc && FSL_RC_STEP_ROW==fsl_stmt_step(&q)){
    char const * zFrom = fsl_stmt_g_text(&q, 0, NULL);
    char const * zTo = fsl_stmt_g_text(&q, 1, NULL);
    if(!zFrom || !zTo){FSL__WARN_OOM; rc = FSL_RC_OOM; break;}
    //MARKER(("MOVING: %s ==> %s\n", zFrom, zTo));
    int const fromDirCheck = fsl_dir_check(zFrom);
    int fsrc;
    if(fromDirCheck>0){
      /* This case is "impossible." Unless... perhaps... a user
         somehow moves things around in the filesystem during the
         fsl_ckout_rename(), such that a to-be-renamed entry which was
         formerly a file is not a directory. */
#if 0
      assert(!"This case cannot possibly happen.");
      fsl__fatal(FSL_RC_CANNOT_HAPPEN,
                "Input name for a file-rename is a directory: %s",
                zFrom)/*does not return*/;
#endif
      int const toDirCheck = fsl_dir_check(zTo);
      if(0==toDirCheck){
        fsl_file_rename(zFrom, zTo);
      }
    }else if(fromDirCheck<0){
      if(fsl_is_symlink(zFrom)){
        fsrc = fsl_symlink_copy(zFrom, zTo, allowSymlinks);
      }else{
        fsrc = fsl_file_copy(zFrom, zTo);
      }
      if(0==fsrc){
        /* fossil(1) unconditionally unlinks zFrom if zFrom is not a
           directory. Maybe we should too? */
        fsl_file_unlink(zFrom);
      }
    }
  }
  fsl_stmt_finalize(&q);
  return rc;
}

int fsl_ckout_rename(fsl_cx * const f, fsl_ckout_rename_opt const * opt){
  int rc = 0;
  bool inTrans = false;
  fsl_buffer * const bDest = fsl__cx_scratchpad(f)
    /* Destination directory */;
  fsl_buffer * const bSrc = fsl__cx_scratchpad(f)
    /* One source file/dir */;
  fsl_stmt qName = fsl_stmt_empty;
  fsl_stmt qMv = fsl_stmt_empty;
  int origType = 0
    /* -1 == multiple input files, 0 == one file, 1 == directory */;
  int destType = 0
    /* >0==directory, 0==does not exist, <0==non-dir */;
  uint32_t srcCount = 0;

  if(!opt->src->used){
    rc = fsl_cx_err_set(f, FSL_RC_RANGE,
                        "Expecting 1 or more source files/directories.");
    goto end;
  }
  rc = fsl_cx_transaction_begin(f);
  if(rc) goto end;
  inTrans = true;
  rc = fsl_ckout_filename_check(f, opt->relativeToCwd, opt->dest,
                                bDest);
  if(rc) goto end;
  fsl_buffer_strip_slashes(bDest);

  rc = fsl_cx_exec_multi(f, "DROP TABLE IF EXISTS TEMP.ckout_mv; "
                         "CREATE TEMP TABLE ckout_mv("
                         "f TEXT UNIQUE ON CONFLICT IGNORE, t TEXT)");
  if(rc) goto end;
  rc = fsl_cx_exec(f, "UPDATE vfile SET origname=pathname "
                   "WHERE origname IS NULL");
  if(rc) goto end;

  if(opt->src->used > 1){
    origType = -1;
  }else{
    /* Make opt->src->list[0] absolute and see if it resolves to an
       existing dir. */
    char const * zSrc= (char const *)opt->src->list[0];
    rc = fsl_ckout_filename_check(f, opt->relativeToCwd, zSrc, bSrc);
    if(rc) goto end;
    fsl_buffer * const bCheck = fsl__cx_scratchpad(f);
    int oCheck = 0;
    rc = fsl_buffer_append(bCheck, f->ckout.dir, f->ckout.dirLen);
    if(0==rc) rc = fsl_buffer_append(bCheck, bSrc->mem, bSrc->used);
    if(0==rc) oCheck = fsl_dir_check(fsl_buffer_cstr(bCheck));
    fsl__cx_scratchpad_yield(f, bCheck);
    if(rc){FSL__WARN_OOM; goto end;}
    if(oCheck>0) origType = 1;
    else if(oCheck<0) origType = 0;
  }
  {
    /* Make bDest absolute and see if it resolves to an existing dir. */
    fsl_buffer * const bCheck = fsl__cx_scratchpad(f);
    rc = fsl_buffer_append(bCheck, f->ckout.dir, f->ckout.dirLen);
    if(0==rc) rc = fsl_buffer_append(bCheck, bDest->mem, bDest->used);
    if(0==rc) destType = fsl_dir_check(fsl_buffer_cstr(bCheck));
    fsl__cx_scratchpad_yield(f, bCheck);
    if(rc){FSL__WARN_OOM; goto end;}
  }
  if(-1==origType && destType<=0){
    rc = fsl_cx_err_set(f, FSL_RC_MISUSE,
                        "Multiple source files provided, so "
                        "destination must be an existing directory.");
    goto end;
  }else if(1==origType && destType<0){
    rc = fsl_cx_err_set(f, FSL_RC_TYPE,
                        "Cannot rename '%s' to '%s' "
                        "because a non-directory named '%s' already exists.",
                        (char const *)opt->src->list[0],
                        opt->dest, opt->dest);
    goto end;
  }else if( 0==origType && destType<=0 ){
    /* Move single file to dest. */
    fsl_id_t vfidCheck = 0;
    rc = fsl_filename_to_vfile_id(f, 0, fsl_buffer_cstr(bSrc),
                                  &vfidCheck);
    if(rc) goto end;
    else if(!vfidCheck){
      rc = fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                          "File not under SCM management: %B",
                          bSrc);
      goto end;
    }
    rc = fsl_cx_exec(f, "INSERT INTO ckout_mv(f,t) VALUES(%B,%B)",
                     bSrc, bDest);
    if(rc) goto end;
    else ++srcCount;
  } else {
    if(fsl_buffer_eq(bDest, ".", 1)){
      fsl_buffer_reuse(bDest);
    }else{
      rc = fsl_buffer_append(bDest, "/", 1);
      if(rc){FSL__WARN_OOM; goto end;}
    }
    rc = fsl_cx_prepare(f, &qName, "SELECT pathname FROM vfile"
                        " WHERE vid=%"FSL_ID_T_PFMT
                        " AND fsl_match_vfile_or_dir(pathname,?1)"
                        " ORDER BY 1", f->ckout.rid);
    if(rc) goto end;
    rc = fsl_cx_prepare(f, &qMv, "INSERT INTO ckout_mv(f,t) VALUES("
                        "?1, ?2||?3)");
    if(rc) goto end;
    for(fsl_size_t i = 0; i < opt->src->used; ++i){
      uint32_t nFound = 0;
      char const * zSrc = (char const *)opt->src->list[i];
      fsl_buffer_reuse(bSrc);
      rc = fsl_ckout_filename_check(f, opt->relativeToCwd,
                                    zSrc, bSrc);
      if(rc) goto end;
      fsl_size_t nOrig = 0;
      char const * const zOrig = fsl_buffer_cstr2(bSrc, &nOrig);
      rc = fsl_stmt_bind_text(&qName, 1, zOrig, (fsl_int_t)nOrig, false);
      if(rc) goto end;
      while(FSL_RC_STEP_ROW==fsl_stmt_step(&qName)){
        fsl_size_t nPath = 0;
        char const * zPath = NULL;
        ++nFound;
        rc = fsl_stmt_get_text(&qName, 0, &zPath, &nPath);
        if(rc){fsl_cx_uplift_db_error(f, qName.db); goto end;}
        else if(!zPath){FSL__WARN_OOM; rc = FSL_RC_OOM; goto end;}
        char const * zTail;
        if(nPath==nOrig){
          zTail = fsl_file_tail(zPath);
        }else if(origType!=0 && destType>0 ){
          zTail = &zPath[nOrig-fsl_strlen(fsl_file_tail(zOrig))];
        }else{
          zTail = &zPath[nOrig+1];
        }
        rc = fsl_stmt_bind_step(&qMv, "sbs", zPath, bDest, zTail);
        if(0!=rc){
          fsl_cx_uplift_db_error(f, qMv.db);
          goto end;
        }
      }
      srcCount += nFound;
      fsl_stmt_reset(&qName);
      if(!nFound){
        rc = fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                            "Name does not resolve to any "
                            "SCM-managed files: %B",
                            bSrc);
        goto end;
      }
    }/*for each opt->src*/
  }
  assert(0==rc);
  fsl_stmt_finalize(&qName);
  fsl_stmt_finalize(&qMv);
  if(0==srcCount){
    rc = fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                        "Source name(s) do not resolve to "
                        "any managed files.");
    goto end;
  }
  rc = fsl_cx_prepare(f, &qName, "SELECT f, t FROM ckout_mv ORDER BY f");
  if(rc) goto end;
  //rc = fsl_cx_prepare(f, &qMv, "INSERT
  while(FSL_RC_STEP_ROW==fsl_stmt_step(&qName)){
    char const * zFrom = fsl_stmt_g_text(&qName, 0, NULL);
    char const * zTo = fsl_stmt_g_text(&qName, 1, NULL);
    rc = fsl__mv_one_file(f, zFrom, zTo, opt->dryRun);
    if(rc) goto end;
    if(opt->callback){
      rc = opt->callback(f, opt, zFrom, zTo);
      if(rc) goto end;
    }
  }
  end:
  fsl_stmt_finalize(&qName);
  fsl_stmt_finalize(&qMv);
  fsl__cx_scratchpad_yield(f, bDest);
  fsl__cx_scratchpad_yield(f, bSrc);
  if(0==rc){
    assert(inTrans);
    if(!opt->dryRun && opt->doFsMv){
      rc = fsl__rename_process_fmove(f);
    }
    fsl_cx_exec(f, "DROP TABLE TEMP.ckout_mv");
  }
  if(inTrans){
    if(rc) fsl_cx_transaction_end(f, true);
    else rc = fsl_cx_transaction_end(f, false);
  }
  return rc;
}


int fsl_ckout_rename_revert(fsl_cx * const f, char const *zNewName,
                            bool relativeToCwd, bool doFsMv,
                            bool *didSomething){
  fsl_buffer * bufFName = fsl__cx_scratchpad(f);
  int rc = 0;
  bool inTrans = false;
  fsl_db * const dbC = fsl_needs_ckout(f);
  fsl_stmt q = fsl_stmt_empty;
  if(!dbC) return FSL_RC_NOT_A_CKOUT;
  rc = fsl_ckout_filename_check(f, relativeToCwd, zNewName, bufFName);
  if(rc) goto end;
  rc = fsl_cx_transaction_begin(f);
  if(rc) goto end;
  inTrans = true;
  rc = fsl_cx_prepare(f, &q,
                      "SELECT id FROM vfile "
                      "WHERE pathname=%Q AND origname<>pathname "
                      "and origname IS NOT NULL %s",
                      fsl_buffer_cstr(bufFName),
                      fsl_cx_filename_collation(f));
  if(rc) goto end;
  switch(fsl_stmt_step(&q)){
    case FSL_RC_STEP_ROW: {
      char const * zNameP = NULL;
      char const * zNameO = NULL;
      fsl_id_t const vfid = fsl_stmt_g_id(&q, 0);
      assert(vfid>0);
      fsl_stmt_finalize(&q);
      if(doFsMv){
        rc = fsl_cx_prepare(f, &q, "SELECT fsl_ckout_dir()||pathname, "
                            "fsl_ckout_dir()||origname FROM vfile "
                            "WHERE id=%"FSL_ID_T_PFMT, vfid);
        if(rc) goto end;
        rc = fsl_stmt_step(&q);
        assert(FSL_RC_STEP_ROW==rc && "We _just_ confirmed that these are there.");
        zNameP = fsl_stmt_g_text(&q, 0, NULL);
        zNameO = fsl_stmt_g_text(&q, 1, NULL);
        if(!zNameO || !zNameO) {FSL__WARN_OOM; rc = FSL_RC_OOM; goto end;}
      }
      rc = fsl_cx_exec(f, "UPDATE vfile SET pathname=origname, origname=NULL "
                       "WHERE id=%"FSL_ID_T_PFMT, vfid);
      if(rc) goto end;
      if(didSomething) *didSomething = true;
      if(doFsMv && fsl_is_file(zNameP)){
        assert(zNameO && zNameP);
        fsl_file_unlink(zNameO);
        rc = fsl_file_rename(zNameP, zNameO);
        if(rc){
          rc = fsl_cx_err_set(f, rc, "File rename failed with code %s: "
                              "'%s' => '%s'", fsl_rc_cstr(rc),
                              zNameO + f->ckout.dirLen,
                              zNameP + f->ckout.dirLen);
        }
      }
      break;
    }
    case FSL_RC_STEP_DONE:
      if(didSomething) *didSomething = false;
      goto end;
    default:
      rc = fsl_cx_uplift_db_error(f, dbC);
      goto end;
  }
  
  end:
  fsl_stmt_finalize(&q);
  fsl__cx_scratchpad_yield(f, bufFName);
  if(inTrans){
    if(0==rc) rc = fsl_cx_transaction_end(f, false);
    else fsl_cx_transaction_end(f, true);
  }
  return rc;

}

#undef MARKER
/* end of file ./src/checkout.c */
/* start of file ./src/cli.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/
#include <string.h> /* for strchr() */
#include <errno.h>
#if !defined(FSL_AMALGAMATION_BUILD)
/* When not in the amalgamation build, force assert() to always work... */
#  if defined(NDEBUG)
#    undef NDEBUG
#    undef DEBUG
#    define DEBUG 1
#  endif
#endif
#include <assert.h> /* for the benefit of test apps */

/* Only for debugging */
#include <stdio.h>
#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)

/** Convenience form of FCLI_VN for level-3 verbosity. */
#define FCLI_V3(pfexp) FCLI_VN(3,pfexp)
#define fcli_empty_m {    \
  NULL/*appHelp*/,  \
  NULL/*cliFlags*/,      \
  NULL/*f*/,      \
  NULL/*argv*/,      \
  0/*argc*/,      \
  NULL/*appName*/,      \
  {/*clientFlags*/ \
    "."/*checkoutDir*/,      \
    0/*verbose*/            \
  }, \
  {/*transient*/      \
    NULL/*repoDb*/,      \
    NULL/*userArg*/, \
    0/*helpRequested*/, \
    false/*versionRequested*/\
  },                    \
  {/*config*/      \
    -1/*traceSql*/,      \
    fsl_outputer_empty_m      \
  },                          \
  {/*paths*/fsl_pathfinder_empty_m/*bins*/},    \
  fsl_error_empty_m/*err*/  \
}

const fcli_t fcli_empty = fcli_empty_m;
fcli_t fcli = fcli_empty_m;
const fcli_cliflag fcli_cliflag_empty = fcli_cliflag_empty_m;
static fsl_timer_state fcliTimer = fsl_timer_state_empty_m;

void fcli_printf(char const * fmt, ...){
  va_list args;
  va_start(args,fmt);
  if(fcli.f){
    fsl_outputfv(fcli.f, fmt, args);
  }else{
    fsl_fprintfv(stdout, fmt, args);
  }
  va_end(args);
}

/**
   Outputs app-level help. How it does this depends on the state of
   the fcli object, namely fcli.cliFlags and the verbosity
   level. Normally this is triggered automatically by the CLI flag
   handling in fcli_setup().
*/
static void fcli_help(void);

unsigned short fcli_is_verbose(void){
  return fcli.clientFlags.verbose;
}

fsl_cx * fcli_cx(void){
  return fcli.f;
}

static int fcli_open(void){
  int rc = 0;
  fsl_cx * f = fcli.f;
  assert(f);
  if(fcli.transient.repoDbArg){
    FCLI_V3(("Trying to open repo db file [%s]...\n", fcli.transient.repoDbArg));
    rc = fsl_repo_open( f, fcli.transient.repoDbArg );
  }
  else if(fcli.clientFlags.checkoutDir){
    char const * dirName = fcli.clientFlags.checkoutDir;
    FCLI_V3(("Trying to open checkout from [%s]...\n",
             dirName));
    rc = fsl_ckout_open_dir(f, dirName, true);
    FCLI_V3(("checkout open rc=%s\n", fsl_rc_cstr(rc)));

    /* if(FSL_RC_NOT_FOUND==rc) rc = FSL_RC_NOT_A_CKOUT; */
    if(rc){
      if(!fsl_cx_err_get(f,NULL,NULL)){
        rc = fsl_cx_err_set(f, rc, "Opening of checkout under "
                            "[%s] failed with code %d (%s).",
                            dirName, rc, fsl_rc_cstr(rc));
      }
    }
    if(rc) return rc;
  }
  if(!rc){
    if(fcli.clientFlags.verbose>1){
      fsl_db * dbC = fsl_cx_db_ckout(f);
      fsl_db * dbR = fsl_cx_db_repo(f);
      if(dbC){
        FCLI_V3(("Checkout DB name: %s\n", f->ckout.db.filename));
      }
      if(dbR){
        FCLI_V3(("Opened repo db: %s\n", f->repo.db.filename));
        FCLI_V3(("Repo user name: %s\n", f->repo.user));
      }
    }
#if 0
    /*
      Only(?) here for testing purposes.

       We don't really need/want to update the repo db on each
       open of the checkout db, do we? Or do we?
     */
    fsl__repo_record_filename(f) /* ignore rc - not critical */;
#endif
  }
  return rc;
}


#define fcli__error (fcli.f ? &fcli.f->error : &fcli.err)
fsl_error * fcli_error(void){
  return fcli__error;
}

void fcli_err_reset(void){
  fsl_error_reset(fcli__error);
}


static struct TempFlags {
  bool traceSql;
  bool doTimer;
} TempFlags = {
false,
false
};

static struct {
  fsl_list list;
}  FCliFree = {
fsl_list_empty_m
};

static void fcli_shutdown(void){
  fsl_cx * const f = fcli.f;
  int rc = 0;
 
  fsl_error_clear(&fcli.err);
  fsl_free(fcli.argv)/*contents are in the FCliFree list*/;
  fsl_pathfinder_clear(&fcli.paths.bins);

  if(f){
    while(fsl_cx_transaction_level(f)){
      MARKER(("WARNING: open db transaction at shutdown-time. "
              "Rolling back.\n"));
      fsl_cx_transaction_end(f, true);
    }
    if(1 &&
       fsl_cx_db_ckout(f)){
      /* For testing/demo only: this is implicit
         when we call fsl_cx_finalize().
      */
      rc = fsl_close_scm_dbs(f);
      FCLI_V3(("Closed checkout/repo db(s). rc=%s\n", fsl_rc_cstr(rc)));
      //assert(0==rc);
    }
  }
  fsl_list_clear(&FCliFree.list, fsl_list_v_fsl_free, 0);
  fsl_list_reserve(&FCliFree.list, 0);
  if(f){
    FCLI_V3(("Finalizing fsl_cx @%p\n", (void const *)f));
    fsl_cx_finalize( f );
  }
  fcli = fcli_empty;
  if(TempFlags.doTimer){
    double const runTime =
      ((int64_t)fsl_timer_stop(&fcliTimer)) / 1000.0;
    f_out("Total fcli run time: %f seconds of CPU time\n",
          runTime/1000);
  }
}

static struct {
  fcli_cliflag const * flags;
} FCliHelpState = {
NULL
};


static int fcli_flag_f_nocheckoutDir(fcli_cliflag const *f){
  if(f){/*unused*/}
  fcli.clientFlags.checkoutDir = 0;
  return 0;
}
static int fcli_flag_f_verbose(fcli_cliflag const *f){
  if(f){/*unused*/}
  ++fcli.clientFlags.verbose;
  return FCLI_RC_FLAG_AGAIN;
}
static int fcli_flag_f_help(fcli_cliflag const *f){
  if(f){/*unused*/}
  ++fcli.transient.helpRequested;
  return FCLI_RC_FLAG_AGAIN;
}

static const fcli_cliflag FCliFlagsGlobal[] = {
  FCLI_FLAG_BOOL_X("?","help",NULL,
                   fcli_flag_f_help,
                   "Show app help. Also triggered if the first non-flag is \"help\"."),
  FCLI_FLAG_BOOL(0,"lib-version", &fcli.transient.versionRequested,
                 "Show libfossil version number."),
  FCLI_FLAG("R","repo","REPO-FILE",&fcli.transient.repoDbArg,
            "Selects a specific repository database, ignoring the one "
            "used by the current directory's checkout (if any)."),
  FCLI_FLAG(NULL,"user","username",&fcli.transient.userArg,
            "Sets the name of the fossil user name for this session."),
  FCLI_FLAG_BOOL_X(NULL, "no-checkout",NULL,fcli_flag_f_nocheckoutDir,
                   "Disable automatic attempt to open checkout."),
  FCLI_FLAG(NULL,"checkout-dir","DIRECTORY", &fcli.clientFlags.checkoutDir,
            "Open the given directory as a checkout, instead of the current dir."),
  FCLI_FLAG_BOOL_X("V","verbose",NULL,fcli_flag_f_verbose,
              "Increases the verbosity level by 1. May be used multiple times."),
  FCLI_FLAG_BOOL(NULL,"trace-sql",&TempFlags.traceSql,
                 "Enable SQL tracing."),
  FCLI_FLAG_BOOL(NULL,"timer",&TempFlags.doTimer,
                 "At the end of successful app execution, output how long it took "
                 "from the call to fcli_setup() until the end of main()."),
  fcli_cliflag_empty_m
};

void fcli_cliflag_help(fcli_cliflag const *defs){
  fcli_cliflag const * f;
  const char * tab = "  ";
  for( f = defs; f->flagShort || f->flagLong; ++f ){
    const char * s = f->flagShort;
    const char * l = f->flagLong;
    const char * fvl = f->flagValueLabel;
    const char * valLbl = 0;
    switch(f->flagType){
      case FCLI_FLAG_TYPE_BOOL:
      case FCLI_FLAG_TYPE_BOOL_INVERT: break;
      case FCLI_FLAG_TYPE_INT32: valLbl = fvl ? fvl : "int32"; break;
      case FCLI_FLAG_TYPE_INT64: valLbl = fvl ? fvl : "int64"; break;
      case FCLI_FLAG_TYPE_ID: valLbl = fvl ? fvl : "db-record-id"; break;
      case FCLI_FLAG_TYPE_DOUBLE: valLbl = fvl ? fvl : "double"; break;
      case FCLI_FLAG_TYPE_CSTR: valLbl = fvl ? fvl : "string"; break;
      default:
        break;
    }
    f_out("%s%s%s%s%s%s%s%s",
          tab,
          s?"-":"", s?s:"", (s&&l)?"|":"",
          l?"--":"",l?l:"",
          valLbl ? "=" : "", valLbl);
    if(f->helpText){
      f_out("\n%s%s%s", tab, tab, f->helpText);
    }
    f_out("\n\n");
  }
}

void fcli_help(void){
  if(fcli.appHelp){
    if(fcli.appHelp->briefUsage){
      f_out("Usage: %s [options] %s\n", fcli.appName, fcli.appHelp->briefUsage);
    }
    if(fcli.appHelp->briefDescription){
      f_out("\n%s\n", fcli.appHelp->briefDescription);
    }
  }else{
    f_out("Help for %s:\n", fcli.appName);
  }
  const int helpCount = fcli.transient.helpRequested
    + fcli.clientFlags.verbose;
  bool const showGlobal = helpCount>1;
  bool const showApp = (2!=helpCount);
  if(showGlobal){
    f_out("\nFCli global flags:\n\n");
    fcli_cliflag_help(FCliFlagsGlobal);
  }else{
    f_out("\n");
  }
  if(showApp){
    if(FCliHelpState.flags
       && (FCliHelpState.flags[0].flagShort || FCliHelpState.flags[0].flagLong)){
      f_out("App-specific flags:\n\n");
      fcli_cliflag_help(FCliHelpState.flags);
      //f_out("\n");
    }
    if(fcli.appHelp && fcli.appHelp->callback){
      fcli.appHelp->callback();
      f_out("\n");
    }
  }
  if(showGlobal){
    if(!showApp){
      f_out("Invoke --help three times (or --help -V -V) to list "
            "both the framework- and app-level options.\n");
    }else{
      f_out("Invoke --help once to list only the "
            "app-level flags.\n");
    }
  }else{
    f_out("Invoke --help twice (or --help -V) to list the "
          "framework-level options. Use --help three times "
          "to list both framework- and app-level options.\n");
  }
  f_out("\nFlags which require values may be passed as "
        "--flag=value or --flag value.\n\n");
}

int fcli_process_flags( fcli_cliflag const * defs ) {
  fcli_cliflag const * f;
  int rc = 0;
  /**
     TODO/FIXME/NICE-TO-HAVE: we "really should" process the CLI flags
     in the order they are provided on the CLI, as opposed to the
     order they're defined in the defs array. The current approach is
     much simpler to process but keeps us from being able to support
     certain useful flag-handling options, e.g.:

     f-tag -a artifact-id-1 --tag x=y --tag y=z -a artifact-id-2 --tag a=b...

     The current approach consumes the -a flags first, leaving us
     unable to match the --tag flags to their corresponding
     (left-hand) -a flag.

     Processing them the other way around, however, requires that we
     keep track of which flags we've already seen so that we can
     reject, where appropriate, duplicate invocations.

     We could, instead of looping on the defs array, loop over the
     head of fcli.argv. If it's a non-flag, move it out of the way
     temporarily (into a new list), else look over the defs array
     looking for a flag match. We don't know, until finding such a
     match, whether the current flag requires a value. If it does, we
     then have to check the current fcli.argv entry to see if it has a
     value (--x=y) or whether the next argv entry is its value (--x
     y). If the current tip has no matching defs entry, we have no
     choice but to skip over it in the hopes that the user can use
     fcli_flag() and friends to consume it, but we cannot know, from
     here, whether such a stray flag requires a value, which means we
     cannot know, for sure, how to process the _next_ argument. The
     best we could do is have a heuristic like "if it starts with a
     dash, assume it's a flag, otherwise assume it's a value for the
     previous flag and skip over it," but whether or not that's sane
     enough for daily use is as yet undetermined.

     If we change the CLI interface to require --flag=value for all
     flags, as opposed to optionally allowing (--flag value), the
     above becomes simpler, but CLI usage suffers. Hmmm. e.g.:

     f-ci -m="message" ...

     simply doesn't fit the age-old muscle memory of:

     svn ci -m ...
     cvs ci -m ...
     fossil ci -m ...
     f-ci -m ...
  */
  for( f = defs; f->flagShort || f->flagLong; ++f ){
    if(!f->flagValue && !f->callback){
      /* We accept these for purposes of generating the --help text,
         but we can't otherwise do anything sensible with them and
         assume the app will handle such flags downstream or ignore
         them altogether.*/
      continue;
    }
    char const * v = NULL;
    const char ** passV = f->flagValue ? &v : NULL;
    switch(f->flagType){
      case FCLI_FLAG_TYPE_BOOL:
      case FCLI_FLAG_TYPE_BOOL_INVERT:
        passV = NULL;
        break;
      default: break;
    };
    bool const gotIt = fcli_flag2(f->flagShort, f->flagLong, passV);
    if(fcli__error->code){
      /**
         Corner case. Consider:

         FCLI_FLAG("x","y","xy", &foo, "blah");

         And: my-app -x

         That will cause fcli_flag2() to return false, but it will
         also populate fcli__error for us.
      */
      rc = fcli__error->code;
      break;
    }
    //MARKER(("Got?=%d flag: %s/%s %s\n",gotIt, f->flagShort, f->flagLong, v ? v : ""));
    if(!gotIt){
      continue;
    }
    if(f->flagValue) switch(f->flagType){
      case FCLI_FLAG_TYPE_BOOL:
        *((bool*)f->flagValue) = true;
        break;
      case FCLI_FLAG_TYPE_BOOL_INVERT:
        *((bool*)f->flagValue) = false;
        break;
      case FCLI_FLAG_TYPE_CSTR:
        if(!v) goto missing_val;
        *((char const **)f->flagValue) = v;
        break;
      case FCLI_FLAG_TYPE_INT32:
        if(!v) goto missing_val;
        *((int32_t*)f->flagValue) = atoi(v);
        break;
      case FCLI_FLAG_TYPE_INT64:
        if(!v) goto missing_val;
        *((int64_t*)f->flagValue) = atoll(v);
        break;
      case FCLI_FLAG_TYPE_ID:
        if(!v) goto missing_val;
        if(sizeof(fsl_id_t)>32){
          *((fsl_id_t*)f->flagValue) = (fsl_id_t)atoll(v);
        }else{
          *((fsl_id_t*)f->flagValue) = (fsl_id_t)atol(v);
        }
        break;
      case FCLI_FLAG_TYPE_DOUBLE:
        if(!v) goto missing_val;
        *((double*)f->flagValue) = strtod(v, NULL);
        break;
      default:
        MARKER(("As-yet-unhandled flag type for flag %s%s%s.",
                f->flagShort ? f->flagShort : "",
                (f->flagShort && f->flagLong) ? "|" : "",
                f->flagLong ? f->flagLong : ""));
        rc = FSL_RC_MISUSE;
        break;
    }
    if(rc) break;
    else if(f->callback){
      rc = f->callback(f);
      if(rc==FCLI_RC_FLAG_AGAIN){
        rc = 0;
        --f;
      }else if(rc){
        break;
      }
    }
  }
  //MARKER(("fcli__error->code==%s\n", fsl_rc_cstr(fcli__error->code)));
  return rc;
  missing_val:
  rc = fcli_err_set(FSL_RC_MISUSE,"Missing value for flag %s%s%s.",
                    f->flagShort ? f->flagShort : "",
                    (f->flagShort && f->flagLong) ? "|" : "",
                    f->flagLong ? f->flagLong : "");
  return rc;
}

/**
   oldMode must be true if fcli.cliFlags is NULL, else false.
*/
static int fcli_process_argv( bool oldMode, int argc, char const * const * argv ){
  int i;
  int rc = 0;
  char * cp;
  fcli.appName = argv[0];
  fcli.argc = 0;
  fcli.argv = (char **)fsl_malloc( (argc + 1) * sizeof(char*));
  fcli.argv[argc] = NULL;
  for( i = 1; i < argc; ++i ){
    char const * arg = argv[i];
    if('-'==*arg){
      char const * flag = arg+1;
      while('-'==*flag) ++flag;
#define FLAG(F) if(0==fsl_strcmp(F,flag))
      if(oldMode){
        FLAG("help") {
          ++fcli.transient.helpRequested;
          continue;
        }
        FLAG("?") {
          ++fcli.transient.helpRequested;
          continue;
        }
        FLAG("V") {
          fcli.clientFlags.verbose += 1;
          continue;
        }
        FLAG("verbose") {
          fcli.clientFlags.verbose += 1;
          continue;
        }
      }
#undef FLAG
      /* else fall through */
    }
    cp = fsl_strdup(arg);
    if(!cp) return FSL_RC_OOM;
    fcli.argv[fcli.argc++] = cp;
    fcli_fax(cp);
  }
  if(!rc && !oldMode){
    rc = fcli_process_flags(FCliFlagsGlobal);
  }
  return rc;
}

bool fcli_flag(char const * opt, const char ** value){
  int i = 0;
  int remove = 0 /* number of items to remove from argv */;
  bool rc = false /* true if found, else 0 */;
  fsl_size_t optLen = fsl_strlen(opt);
  for( ; i < fcli.argc; ++i ){
    char const * arg = fcli.argv[i];
    char const * x;
    char const * vp = NULL;
    if(!arg || ('-' != *arg)) continue;
    rc = false;
    x = arg+1;
    if('-' == *x) { ++x;}
    if(0 != fsl_strncmp(x, opt, optLen)) continue;
    if(!value){
      if(x[optLen]) continue /* not exact match */;
      /* Treat this as a boolean. */
      rc = true;
      ++remove;
      break;
    }else{
      /* -FLAG VALUE or -FLAG=VALUE */
      if(x[optLen] == '='){
        rc = true;
        vp = x+optLen+1;
        ++remove;
      }
      else if(x[optLen]) continue /* not an exact match */;
      else if(i<(fcli.argc-1)){ /* -FLAG VALUE */
        vp = fcli.argv[i+1];
        if('-'==*vp && vp[1]/*allow "-" by itself!*/){
          // VALUE looks like a flag.
          fcli_err_set(FSL_RC_MISUSE, "Missing value for flag [%s].",
                       opt);
          rc = false;
          assert(!remove);
          break;
        }
        rc = true;
        remove += 2;
      }
      else{
        /*
          --FLAG is expecting VALUE but we're at end of argv.  Leave
          --FLAG in the args and report this as "not found."
        */
        rc = false;
        assert(!remove);
        fcli_err_set(FSL_RC_MISUSE,
                     "Missing value for flag [%s].",
                     opt);
        assert(fcli__error->code);
        //MARKER(("Missing flag value for [%s]\n",opt));
        break;
      }
      if(rc){
        *value = vp;
      }
      break;
    }
  }
  if(remove>0){
    int x;
    for( x = 0; x < remove; ++x ){
      fcli.argv[i+x] = NULL/*memory ownership==>FCliFree*/;
    }
    for( ; i < fcli.argc; ++i ){
      fcli.argv[i] = fcli.argv[i+remove];
    }
    fcli.argc -= remove;
    fcli.argv[i] = NULL;
  }
  //MARKER(("flag %s check rc=%s\n",opt,fsl_rc_cstr(fcli__error->code)));
  return rc;
}

bool fcli_flag2(char const * shortOpt,
                char const * longOpt,
                const char ** value){
  bool rc = 0;
  if(shortOpt) rc = fcli_flag(shortOpt, value);
  if(!rc && longOpt && !fcli__error->code) rc = fcli_flag(longOpt, value);
  //MARKER(("flag %s check rc=%s\n",shortOpt,fsl_rc_cstr(fcli__error->code)));
  return rc;
}

bool fcli_flag_or_arg(char const * shortOpt,
                      char const * longOpt,
                      const char ** value){
  bool rc = fcli_flag(shortOpt, value);
  if(!rc && !fcli__error->code){
    rc = fcli_flag(longOpt, value);
    if(!rc && value){
      const char * arg = fcli_next_arg(1);
      if(arg){
        rc = true;
        *value = arg;
      }
    }
  }
  return rc;
}


/**
    We copy fsl_lib_configurable.allocator as a base allocator.
 */
static fsl_allocator fslAllocOrig;

/**
    Proxies fslAllocOrig.f() and abort()s on OOM conditions.
*/
static void * fsl_realloc_f_failing(void * state __unused, void * mem, fsl_size_t n){
  void * rv = fslAllocOrig.f(fslAllocOrig.state, mem, n);
  if(n && !rv){
    fsl__fatal(FSL_RC_OOM, NULL)/*does not return*/;
  }
  return rv;
}

/**
    Replacement for fsl_memory_allocator() which abort()s on OOM.
    Why? Because fossil(1) has shown how much that can simplify error
    checking in an allocates-often API.
 */
static const fsl_allocator fcli_allocator = {
fsl_realloc_f_failing,
NULL/*state*/
};

#if !defined(FCLI_USE_SIGACTION)
#  if (defined(_POSIX_C_SOURCE) || defined(sa_sigaction/*BSD*/)) \
  && defined(HAVE_SIGACTION)
/* ^^^ on Linux, sigaction() is only available in <signal.h>
   if _POSIX_C_SOURCE is set */
#    define FCLI_USE_SIGACTION HAVE_SIGACTION
#  else
#    define FCLI_USE_SIGACTION 0
#  endif
#endif

#if FCLI_USE_SIGACTION
#include <signal.h> /* sigaction(), if our feature macros are set right */
/**
   SIGINT handler which calls fsl_cx_interrupt().
*/
static void fcli__sigc_handler(int s){
  static fsl_cx * f = 0;
  if(f) return/*disable concurrent interruption*/;
  f = fcli_cx();
  if(f && !fsl_cx_interrupted(f)){
    //f_out("^C\n"); // no - this would interfere with curses apps
    fsl_cx_interrupt(f, FSL_RC_INTERRUPTED,
                     "Interrupted by signal #%d.", s);
    f = NULL;
  }
}
#endif
/* ^^^ FCLI_USE_SIGACTION */

void fcli_pre_setup(void){
  static int run = 0;
  if(run++) return;
  fslAllocOrig = fsl_lib_configurable.allocator;
  fsl_lib_configurable.allocator = fcli_allocator
    /* This MUST be done BEFORE the fsl API allocates
       ANY memory! */;
  atexit(fcli_shutdown);
#if FCLI_USE_SIGACTION
  struct sigaction sigIntHandler;
  sigIntHandler.sa_handler = fcli__sigc_handler;
  sigemptyset(&sigIntHandler.sa_mask);
  sigIntHandler.sa_flags = 0;
  sigaction(SIGINT, &sigIntHandler, NULL);
#endif
}
/**
   oldMode must be true if fcli.cliFlags is NULL, else false.
*/
static int fcli_setup_common1(bool oldMode, int argc, char const * const *argv){
  static char once = 0;
  int rc = 0;
  if(once++){
    fprintf(stderr,"MISUSE: fcli_setup() must "
            "not be called more than once.");
    return FSL_RC_MISUSE;
  }
  fsl_timer_start(&fcliTimer);
  fcli_pre_setup();
  rc = fcli_process_argv(oldMode, argc, argv);
  if(!rc && fcli.argc && 0==fsl_strcmp("help",fcli.argv[0])){
    fcli_next_arg(1) /* strip argument */;
    ++fcli.transient.helpRequested;
  }
  return rc;
}

static int fcli_setup_common2(void){
  int rc = 0;
  fsl_cx_init_opt init = fsl_cx_init_opt_empty;
  fsl_cx * f = 0;

  init.config.sqlPrint = 1;
  if(fcli.config.outputer.out){
    init.output = fcli.config.outputer;
    fcli.config.outputer = fsl_outputer_empty
      /* To avoid any confusion about ownership */;
  }else{
    init.output = fsl_outputer_FILE;
    init.output.state = stdout;
  }
  if(fcli.config.traceSql>0 || TempFlags.traceSql){
    init.config.traceSql = fcli.config.traceSql;
  }
    
  rc = fsl_cx_init( &f, &init );
  fcli.f = f;
#if 0
  /* Just for testing cache size effects... */
  f->cache.arty.szLimit = 1024 * 1024 * 20;
  f->cache.arty.usedLimit = 300;
#endif
  fsl_error_clear(&fcli.err);
  FCLI_V3(("Initialized fsl_cx @0x%p. rc=%s\n",
           (void const *)f, fsl_rc_cstr(rc)));
  if(!rc){
#if 0
    if(fcli.transient.gmtTime){
      fsl_cx_flag_set(f, FSL_CX_F_LOCALTIME_GMT, 1);
    }
#endif
    if(fcli.clientFlags.checkoutDir || fcli.transient.repoDbArg){
      rc = fcli_open();
      FCLI_V3(("fcli_open() rc=%s\n", fsl_rc_cstr(rc)));
      if(!fcli.transient.repoDbArg && fcli.clientFlags.checkoutDir
         && (FSL_RC_NOT_FOUND == rc)){
        /* If [it looks like] we tried an implicit checkout-open but
           didn't find one, suppress the error. */
        rc = 0;
        fcli_err_reset();
      }
    }
  }
  if(!rc){
    char const * userName = fcli.transient.userArg;
    if(userName){
      fsl_cx_user_set(f, userName);
    }else if(!fsl_cx_user_get(f)){
      char * u = fsl_user_name_guess();
      fsl_cx_user_set(f, u);
      fsl_free(u);
    }
  }
  return rc;
}

static int check_help_invoked(void){
  int rc = 0;
  if(fcli.transient.helpRequested){
    /* Do this last so that we can get the default user name and such
       for display in the help text. */
    fcli_help();
    rc = FCLI_RC_HELP;
  }else if(fcli.transient.versionRequested){
    f_out("libfossil version: %s\nCheckin: %s\nCheckin timestamp: %s\n",
          fsl_library_version(),
          FSL_LIB_VERSION_HASH,
          FSL_LIB_VERSION_TIMESTAMP);
    rc = FCLI_RC_HELP;
  }
  return rc;
}

static int fcli_setup2(int argc, char const * const * argv,
                       const fcli_cliflag * flags){
  int rc;
  FCliHelpState.flags = flags;
  rc = fcli_setup_common1(false, argc, argv);
  if(rc) return rc;
  assert(!fcli__error->code);
  rc = check_help_invoked();
  if(!rc){
    rc = fcli_process_flags(flags);
    if(rc) assert(fcli__error->msg.used);
    if(!rc){
      rc = fcli_setup_common2();
    }
  }
  return rc;
}

int fcli_setup_v2(int argc, char const * const * argv,
                  fcli_cliflag const * const cliFlags,
                  fcli_help_info const * const helpInfo ){
  if(NULL!=cliFlags) fcli.cliFlags = cliFlags;
  if(NULL!=helpInfo) fcli.appHelp = helpInfo;
  if(cliFlags || fcli.cliFlags){
    return fcli_setup2(argc, argv, cliFlags ? cliFlags : fcli.cliFlags);
  }
  int rc = fcli_setup_common1(true, argc, argv);
  if(!rc){
    rc = check_help_invoked();
    if(!rc){
      if( fcli_flag2(NULL, "no-checkout", NULL) ){
        fcli.clientFlags.checkoutDir = NULL;
      }
      fcli_flag2(NULL,"user", &fcli.transient.userArg);
      fcli.config.traceSql =  fcli_flag2(NULL,"trace-sql", NULL);
      fcli_flag2("R", "repo", &fcli.transient.repoDbArg);
      rc = fcli_setup_common2();
    }
  }
  return rc;
}

int fcli_setup(int argc, char const * const * argv ){
  return fcli_setup_v2(argc, argv, fcli.cliFlags, fcli.appHelp);
}

int fcli_err_report2(bool clear, char const * file, int line){
  int errRc = 0;
  char const * msg = NULL;
  errRc = fsl_error_get( fcli__error, &msg, NULL );
  if(!errRc && fcli.f && fcli.f->interrupted){
    errRc = fcli.f->interrupted;
    msg = "Interrupted.";
  }
  if(FCLI_RC_HELP==errRc){
    errRc = 0;
  }else if(errRc || msg){
    if(fcli.clientFlags.verbose>0){
      fcli_printf("%s %s:%d: ERROR #%d (%s): %s\n",
                  fcli.appName,
                  file, line, errRc, fsl_rc_cstr(errRc), msg);
    }else{
      fcli_printf("%s: ERROR #%d (%s): %s\n",
                  fcli.appName, errRc, fsl_rc_cstr(errRc), msg);
    }
  }
  if(clear){
    fcli_err_reset();
    if(fcli.f) fsl_cx_interrupt(fcli.f, 0, NULL);
  }
  return errRc;
}


const char * fcli_next_arg(bool remove){
  const char * rc = (fcli.argc>0) ? fcli.argv[0] : NULL;
  if(rc && remove){
    int i;
    --fcli.argc;
    for(i = 0; i < fcli.argc; ++i){
      fcli.argv[i] = fcli.argv[i+1];
    }
    fcli.argv[fcli.argc] = NULL/*owned by FCliFree*/;
  }
  return rc;
}

int fcli_has_unused_args(bool outputError){
  int rc = 0;
  if(fcli.argc){
    rc = fsl_cx_err_set(fcli.f, FSL_RC_MISUSE,
                        "Unhandled extra argument: %s",
                        fcli.argv[0]);
    if(outputError){
      fcli_err_report(false);
    }
  }
  return rc;
}
int fcli_has_unused_flags(bool outputError){
  int i;
  for( i = 0; i < fcli.argc; ++i ){
    char const * arg = fcli.argv[i];
    if('-'==*arg){
      int rc = fsl_cx_err_set(fcli.f, FSL_RC_MISUSE,
                              "Unhandled/unknown flag or missing value: %s",
                              arg);
      if(outputError){
        fcli_err_report(false);
      }
      return rc;
    }
  }
  return 0;
}

int fcli_err_set(int code, char const * fmt, ...){
  int rc;
  va_list va;
  va_start(va, fmt);
  rc = fsl_error_setv(fcli__error, code, fmt, va);
  va_end(va);
  return rc;
}

int fcli_end_of_main(int mainRc){
  if(FCLI_RC_HELP==mainRc){
    mainRc = 0;
  }
  if(fcli_err_report(true)){
    return EXIT_FAILURE;
  }else if(mainRc){
    fcli_err_set(mainRc,"Ending with unadorned end-of-app "
                 "error code %d/%s.",
                 mainRc, fsl_rc_cstr(mainRc));
    fcli_err_report(true);
    return EXIT_FAILURE;
  }
  return EXIT_SUCCESS;
}

int fcli_dispatch_commands( fcli_command const * cmd,
                            bool reportErrors){
  int rc = 0;
  const char * arg = fcli_next_arg(0);
  fcli_command const * orig = cmd;
  fcli_command const * helpPos = 0;
  int helpState = 0;
  if(!arg){
    return fcli_err_set(FSL_RC_MISUSE,
                        "Missing command argument. Try --help.");
  }
  assert(fcli.f);
  for(; arg && cmd->name; ++cmd ){
    if(cmd==orig && 0==fsl_strcmp(arg,"help")){
      /* Accept either (help command) or (command help) as help. */
      /* Except that it turns out that fcli_setup() will trump the
         former and doesn't have the fcli_command state, so can't do
         this. Maybe we can change that somehow. */
      helpState = 1;
      helpPos = orig;
      arg = fcli_next_arg(1); // consume it
    }else if(0==fsl_strcmp(arg,cmd->name) || 0==fcli_cmd_aliascmp(cmd,arg)){
      if(!cmd->f){
        rc = fcli_err_set(FSL_RC_NYI,
                               "Command [%s] has no "
                               "callback function.");
      }else{
        fcli_next_arg(1)/*consume it*/;
        if(helpState){
          assert(1==helpState);
          helpState = 2;
          helpPos = cmd;
          break;
        }
        const char * helpCheck = fcli_next_arg(false);
        if(helpCheck && 0==fsl_strcmp("help",helpCheck)){
          helpState = 3;
          helpPos = cmd;
          break;
        }else{
          rc = cmd->f(cmd);
        }
      }
      break;
    }
  }
  if(helpState){
    f_out("\n");
    fcli_command_help(helpPos, true, helpState>1);
    fcli.transient.helpRequested++;
  }else if(!cmd->name){
    fsl_buffer msg = fsl_buffer_empty;
    int rc2;
    if(!arg){
      rc2 = FSL_RC_MISUSE;
      fsl_buffer_appendf(&msg, "No command provided.");
    }else{
      rc2 = FCLI_RC_NO_CMD;
      fsl_buffer_appendf(&msg, "Command not found: %s.",arg);
    }
    fsl_buffer_appendf(&msg, " Available commands: ");
    cmd = orig;
    for( ; cmd && cmd->name; ++cmd ){
      fsl_buffer_appendf( &msg, "%s%s",
                          (cmd==orig) ? "" : ", ",
                          cmd->name);
    }
    rc = fcli_err_set(rc2, "%b", &msg);
    fsl_buffer_clear(&msg);
  }
  if(rc && reportErrors){
    fcli_err_report(0);
  }
  return rc;
}

int fcli_cmd_aliascmp(fcli_command const * cmd, char const * arg){
  char const * alias = cmd->aliases;
  while ( alias && *alias!=0 ){
    if( 0==fsl_strcmp(alias, arg) ){
      return 0;
    }
    alias = strchr(alias, 0) + 1;
  }
  return 1;
}

void fcli_command_help(fcli_command const * cmd, bool showUsage, bool onlyOne){
  fcli_command const * c = cmd;
  for( ; c->name; ++c ){
    f_out("[%s] command:\n\n", c->name);
    if(c->briefDescription){
      f_out("  %s\n", c->briefDescription);
    }
    if(c->aliases){
      fcli_help_show_aliases(c->aliases);
    }else{
      f_out("\n");
    }
    if(c->flags){
      f_out("\n");
      fcli_cliflag_help(c->flags);
    }
    if(showUsage && c->usage){
      c->usage();
    }
    if(onlyOne) break;
  }
}

void fcli_help_show_aliases(char const * aliases){
  char const * alias = aliases;
  f_out("  (aliases: ");
  while ( *alias!=0 ){
    f_out("%s%s", alias, *(strchr(alias, 0) + 1) ? ", " : ")\n");
    alias = strchr(alias, 0) + 1;
  }
}

void * fcli_fax(void * mem){
  if(mem){
    fsl_list_append( &FCliFree.list, mem );
  }
  return mem;
}

int fcli_ckout_show_info(bool useUtc){
  fsl_cx * const f = fcli_cx();
  int rc = 0;
  fsl_stmt st = fsl_stmt_empty;
  fsl_db * const dbR = fsl_cx_db_repo(f);
  fsl_db * const dbC = fsl_cx_db_ckout(f);
  int lblWidth = -20;
  if(!fsl_needs_ckout(f)){
    return FSL_RC_NOT_A_CKOUT;
  }
  assert(dbR);
  assert(dbC);

  fsl_id_t rid = 0;
  fsl_uuid_cstr uuid = NULL;
  fsl_ckout_version_info(f, &rid, &uuid);
  assert((uuid && (rid>0)) || (!uuid && (0==rid)));

  f_out("%*s %s\n", lblWidth, "repository-db:",
        fsl_cx_db_file_repo(f, NULL));
  f_out("%*s %s\n", lblWidth, "checkout-root:",
        fsl_cx_ckout_dir_name(f, NULL));

  rc = fsl_cx_prepare(f, &st, "SELECT "
                      /*0*/"datetime(event.mtime%s) AS timestampString, "
                      /*1*/"coalesce(euser, user) AS user, "
                      /*2*/"(SELECT group_concat(substr(tagname,5), ', ') FROM tag, tagxref "
                      "WHERE tagname GLOB 'sym-*' AND tag.tagid=tagxref.tagid "
                      "AND tagxref.rid=blob.rid AND tagxref.tagtype>0) as tags, "
                      /*3*/"coalesce(ecomment, comment) AS comment, "
                      /*4*/"uuid AS uuid "
                      "FROM event JOIN blob "
                      "WHERE "
                      "event.type='ci' "
                      "AND blob.rid=%"FSL_ID_T_PFMT" "
                      "AND blob.rid=event.objid "
                      "ORDER BY event.mtime DESC",
                      useUtc ? "" : ", 'localtime'",
                      rid);
  if(rc) goto end;
  if( FSL_RC_STEP_ROW != fsl_stmt_step(&st)){
    /* fcli_err_set(FSL_RC_ERROR, "Event data for checkout not found."); */
    f_out("\nNo 'event' data found. This is only normal for an empty repo.\n");
    goto end;
  }

  f_out("%*s %s %s %s (RID %"FSL_ID_T_PFMT")\n",
        lblWidth, "checkout-version:",
        fsl_stmt_g_text(&st, 4, NULL),
        fsl_stmt_g_text(&st, 0, NULL),
        useUtc ? "UTC" : "local",
        rid );

  {
    /* list parent(s) */
    fsl_stmt stP = fsl_stmt_empty;
    rc = fsl_cx_prepare(f, &stP, "SELECT "
                        "uuid, pid, isprim "
                        "FROM plink JOIN blob ON pid=rid "
                        "WHERE cid=%"FSL_ID_T_PFMT" "
                        "ORDER BY isprim DESC, mtime DESC /*sort*/",
                        rid);
    if(rc) goto end;
    while( FSL_RC_STEP_ROW == fsl_stmt_step(&stP) ){
      char const * zLabel = fsl_stmt_g_int32(&stP,2)
        ? "parent:" : "merged-from:";
      f_out("%*s %s\n", lblWidth, zLabel,
            fsl_stmt_g_text(&stP, 0, NULL));
      
    }
    fsl_stmt_finalize(&stP);
  }
  {
    /* list merge parent(s) */
    fsl_stmt stP = fsl_stmt_empty;
    rc = fsl_cx_prepare(f, &stP, "SELECT "
                        "mhash, id FROm vmerge WHERE id<=0");
    if(rc) goto end;
    while( FSL_RC_STEP_ROW == fsl_stmt_step(&stP) ){
      char const * zClass;
      int32_t const id = fsl_stmt_g_int32(&stP,1);
      switch(id){
        case FSL_MERGE_TYPE_INTEGRATE: zClass = "integrate-merge:"; break;
        case FSL_MERGE_TYPE_BACKOUT: zClass = "backout-merge:"; break;
        case FSL_MERGE_TYPE_CHERRYPICK: zClass = "cherrypick-merge:"; break;
        case FSL_MERGE_TYPE_NORMAL: zClass = "merged-with:"; break;
        default:
          fsl__fatal(FSL_RC_RANGE,
                     "Unexpected value %"PRIi32" in vmerge.id",id);
          break;
      }
      f_out("%*s %s\n", lblWidth, zClass,
            fsl_stmt_g_text(&stP, 0, NULL));
    }
    fsl_stmt_finalize(&stP);
  }
  {
    /* list children */
    fsl_stmt stC = fsl_stmt_empty;
    rc = fsl_cx_prepare(f, &stC, "SELECT "
                        "uuid, cid, isprim "
                        "FROM plink JOIN blob ON cid=rid "
                        "WHERE pid=%"FSL_ID_T_PFMT" "
                        "ORDER BY isprim DESC, mtime DESC /*sort*/",
                        rid);
    if(rc) goto end;
    while( FSL_RC_STEP_ROW == fsl_stmt_step(&stC) ){
      char const * zLabel = fsl_stmt_g_int32(&stC,2)
        ? "child:" : "merged-into:";
      f_out("%*s %s\n", lblWidth, zLabel,
            fsl_stmt_g_text(&stC, 0, NULL));
      
    }
    fsl_stmt_finalize(&stC);
  }

  f_out("%*s %s\n", lblWidth, "user:",
        fsl_stmt_g_text(&st, 1, NULL));

  f_out("%*s %s\n", lblWidth, "tags:",
        fsl_stmt_g_text(&st, 2, NULL));

  f_out("%*s %s\n", lblWidth, "comment:",
        fsl_stmt_g_text(&st, 3, NULL));

  end:
  fsl_stmt_finalize(&st);

  return rc;
}

static int fsl_stmt_each_f_ambiguous( fsl_stmt * stmt, void * state ){
  int rc;
  if(1==stmt->rowCount) stmt->rowCount=0
                          /* HORRIBLE KLUDGE to elide header. */;
  rc = fsl_stmt_each_f_dump(stmt, state);
  if(0==stmt->rowCount) stmt->rowCount = 1;
  return rc;
}

void fcli_list_ambiguous_artifacts(char const * label,
                                   char const *prefix){
  fsl_db * const db = fsl_cx_db_repo(fcli.f);
  assert(db);
  if(!label){
    f_out("Artifacts matching ambiguous prefix: %s\n",prefix);
  }else if(*label){
    f_out("%s\n", label);
  }  
  /* Possible fixme? Do we only want to list checkins
     here? */
  int rc = fsl_db_each(db, fsl_stmt_each_f_ambiguous, 0,
              "SELECT uuid, CASE "
              "WHEN type='ci' THEN 'Checkin' "
              "WHEN type='w'  THEN 'Wiki' "
              "WHEN type='g'  THEN 'Control' "
              "WHEN type='e'  THEN 'Technote' "
              "WHEN type='t'  THEN 'Ticket' "
              "WHEN type='f'  THEN 'Forum' "
              "ELSE '?'||'?'||'?' END " /* '???' ==> trigraph! */
              "FROM blob b, event e WHERE uuid LIKE %Q||'%%' "
              "AND b.rid=e.objid "
              "ORDER BY uuid",
              prefix);
  if(rc){
    fsl_cx_uplift_db_error(fcli.f, db);
    fcli_err_report(false);
  }
}

fsl_db * fcli_db_ckout(void){
  return fcli.f ? fsl_cx_db_ckout(fcli.f) : NULL;
}

fsl_db * fcli_db_repo(void){
  return fcli.f ? fsl_cx_db_repo(fcli.f) : NULL;
}

fsl_db * fcli_needs_ckout(void){
  if(fcli.f) return fsl_needs_ckout(fcli.f);
  fcli_err_set(FSL_RC_NOT_A_CKOUT,
               "No checkout db is opened.");
  return NULL;
}

fsl_db * fcli_needs_repo(void){
  if(fcli.f) return fsl_needs_repo(fcli.f);
  fcli_err_set(FSL_RC_NOT_A_REPO,
               "No repository db is opened.");
  return NULL;
}

int fcli_args_to_vfile_ids(fsl_id_bag * const tgt,
                           fsl_id_t vid,
                           bool relativeToCwd,
                           bool changedFilesOnly){
  if(!fcli.argc){
    return fcli_err_set(FSL_RC_MISUSE,
                        "No file/dir name arguments provided.");
  }
  int rc = 0;
  char const * zName;
  while( !rc && (zName = fcli_next_arg(true))){
    FCLI_V3(("Collecting vfile ID(s) for: %s\n", zName));
    rc = fsl_ckout_vfile_ids(fcli.f, vid, tgt, zName,
                             relativeToCwd, changedFilesOnly);
  }
  return rc;
}

int fcli_fingerprint_check(bool reportImmediately){
  int rc = fsl_ckout_fingerprint_check(fcli.f);
  if(rc && reportImmediately){
    f_out("ERROR: repo/checkout fingerprint mismatch detected. "
          "To recover from this, (fossil close) the current checkout, "
          "then re-open it. Be sure to store any modified files somewhere "
          "safe and restore them after re-opening the repository.\n");
  }
  return rc;
}

char const * fcli_progname(){
  if(!fcli.appName || !*fcli.appName) return NULL;
  char const * z = fcli.appName;
  char const * zEnd = z + fsl_strlen(z) - 1;
  for( ; zEnd > z; --zEnd ){
    switch((int)*zEnd){
      case (int)'/':
      case (int)'\\':
        return zEnd+1;
      default: break;
    }
  }
  return zEnd;
}

void fcli_diff_colors(fsl_dibu_opt * const tgt, fcli_diff_colors_e theme){
  char const * zIns = 0;
  char const * zEdit = 0;
  char const * zDel = 0;
  char const * zReset = 0;
  switch(theme){
    case FCLI_DIFF_COLORS_RG:
          zIns = "\x1b[32m";
          zEdit = "\x1b[36m";
          zDel = "\x1b[31m";
          zReset = "\x1b[0m";
          break;
    case FCLI_DIFF_COLORS_NONE:
    default: break;
  }
  tgt->ansiColor.insertion = zIns;
  tgt->ansiColor.edit = zEdit;
  tgt->ansiColor.deletion = zDel;
  tgt->ansiColor.reset = zReset;
}

void fcli_dump_stmt_cache(bool forceVerbose){
  int i = 0;
  fsl_stmt * st;
  fsl_db * const db = fsl_cx_db(fcli_cx());
  assert(db);
  for( st = db->cacheHead; st; st = st->next ) ++i;
  f_out("%s(): Cached fsl_stmt count: %d\n", __func__, i);
  if(i>0 && (forceVerbose || fcli_is_verbose()>1)){
    for( i = 1, st = db->cacheHead; st; ++i, st = st->next ){
      f_out("CACHED fsl_stmt #%d (%d hit(s)): %b\n", i,
            (int)st->cachedHits, &st->sql);
    }
  }
}

void fcli_dump_cache_metrics(void){
  fsl_cx * const f = fcli.f;
  if(!f) return;
  f_out("fsl_cx::cache::mcache hits = %u misses = %u\n",
        f->cache.mcache.hits,
        f->cache.mcache.misses);
  f_out("fsl_cx::cache::blobContent hits = %u misses = %u. "
        "Entry count=%u totaling %u byte(s).\n",
        f->cache.blobContent.metrics.hits,
        f->cache.blobContent.metrics.misses,
        f->cache.blobContent.used,
        f->cache.blobContent.szTotal);
}

char const * fcli_fossil_binary(bool errIfNotFound, int reportPolicy){
  static bool once = false;
  if(!once){
    int rc = 0;
    char const * path = getenv("PATH");
    if(path && *path){
       fsl_path_splitter pt = fsl_path_splitter_empty;
       fsl_size_t tLen = 0;
       char const * t = 0;
       fsl_path_splitter_init(&pt, path, -1);
       while(0==rc && 0==fsl_path_splitter_next(&pt, &t, &tLen)){
         rc = fsl_pathfinder_dir_add2(&fcli.paths.bins,
                                      t, (fsl_int_t)tLen);
       }
    }
    if(0==rc){
      fsl_pathfinder_ext_add2(&fcli.paths.bins,".exe", 4);
    }
    once = true;
  }
  char const * z = NULL;
  fsl_pathfinder_search(&fcli.paths.bins, "fossil", &z, NULL);
  if(!z && errIfNotFound){
    fcli_err_set(FSL_RC_NOT_FOUND,
                 "Fossil binary not found in $PATH.");
    if(reportPolicy){
      fcli_err_report(reportPolicy>0);
    }
  } 
  return z;
}

static int fcli__transaction_check(void){
  if(fsl_cx_transaction_level(fcli.f)){
    return fcli_err_set(FSL_RC_LOCKED,
                        "Sync cannot succeed if a transaction "
                        "is opened. Close all transactions before "
                        "calling %s().", __func__);
  }
  return 0;
}

static bool fcli__autosync_setting(void){
  return fsl_configs_get_bool(fcli.f, "crg",
                              fsl_configs_get_bool(fcli.f, "crg",
                                                   false, "autosync"),
                              "fcli.autosync");
}

int fcli_sync( int ops ){
  int rc = 0;
  if((rc = fcli__transaction_check())) return rc;

  int doPush = -1;
  int doPull = -1;
  char const * zSuppressOut = "";
  fsl_db * const dbR = fsl_needs_repo(fcli.f);
  if(!dbR){
    return FSL_RC_NOT_A_REPO;
  }else if(!fsl_db_exists(dbR, "select 1 from config "
                          "where name like 'syncwith:%%'")){
    /* No remote, so nothing to do (and any attempt would fail). */
    return 0;
  }
  if(FCLI_SYNC_PULL & ops){
    doPull = 1;
  }
  if(FCLI_SYNC_PUSH & ops){
    doPush = 1;
  }
#if !FSL_PLATFORM_IS_WINDOWS
  if(FCLI_SYNC_NO_OUTPUT & ops){
    zSuppressOut = " >/dev/null 2>&1";
  }else if(FCLI_SYNC_NO_STDOUT & ops){
    zSuppressOut = " >/dev/null";
  }
#endif
  bool const autosync = fcli__autosync_setting();
  if(!autosync && (FCLI_SYNC_AUTO & ops)){
    return 0;
  }
  if(doPull<=0 && doPush<=0){
    return 0;
  }
  char const * zCmd;
  char const * fslBin;
  if(doPull>0 && doPush>0) zCmd = "sync";
  else if(doPull>0) zCmd = "pull";
  else{
    assert(doPush>0);
    zCmd = "push";
  }
  fslBin = fcli_fossil_binary(true, 0);
  if(!fslBin){
    assert(fcli__error->code);
    return fcli__error->code;
  }
  ;
  char * cmd = fsl_mprintf("%s %s%s", fslBin, zCmd, zSuppressOut);
  rc = fsl_system(cmd);
  if(rc){
    fsl_cx_caches_reset(fcli.f);
    rc = fcli_err_set(rc, "Command exited with non-0 result: %s", cmd);
  }
  fsl_free(cmd);
  return rc;
}

#undef FCLI_V3
#undef fcli_empty_m
#undef fcli__error
#undef MARKER
#undef FCLI_USE_SIGACTION
/* end of file ./src/cli.c */
/* start of file ./src/content.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/
/**************************************************************************
  This file houses the code for the fsl_content_xxx() APIS.
*/
#include <assert.h>
#include <memory.h> /* memcmp() */

/* Only for debugging */
#include <stdio.h>
#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)


fsl_int_t fsl_content_size( fsl_cx * const f, fsl_id_t blobRid ){
  if(blobRid<=0) return -3;
  else if(!fsl_needs_repo(f)) return -4;
  else{
    int rc = 0;
    fsl_int_t rv = -2;
    fsl_stmt * const q = &f->cache.stmt.contentSize;
    if(!q->stmt){
      rc = fsl_cx_prepare(f, q,
                          "SELECT size FROM blob WHERE rid=?1 "
                          "/*%s()*/",__func__);
      if(rc) return -6;
    }
    rc = fsl_stmt_bind_step(q, "R", blobRid);
    if(FSL_RC_STEP_ROW==rc){
      rv = (fsl_int_t)fsl_stmt_g_int64(q, 0);
    }
    fsl_stmt_reset(q);
    return rv;
  }
}

static bool fsl_content_is_available(fsl_cx * const f, fsl_id_t rid){
  fsl_id_t srcid = 0;
  int rc = 0, depth = 0 /* Limit delta recursion depth */;
  while( depth++ < 100000 ){
    if( fsl_id_bag_contains(&f->cache.blobContent.missing, rid) ){
      return false;
    }else if( fsl_id_bag_contains(&f->cache.blobContent.available, rid) ){
      return true;
    }else if( fsl_content_size(f, rid)<0 ){
      fsl_id_bag_insert(&f->cache.blobContent.missing, rid)
        /* ignore possible OOM error - not fatal */;
      return false;
    }
    rc = fsl_delta_src_id(f, rid, &srcid);
    if(rc) break;
    else if( 0==srcid ){
      fsl_id_bag_insert(&f->cache.blobContent.available, rid)
        /* ignore possible OOM error - not fatal */;
      return true;
    }
    rid = srcid;
  }
  if(0==rc){
    /* This "cannot happen" (never has historically, and would be
       indicative of what amounts to corruption in the repo). */
    fsl__fatal(FSL_RC_RANGE,"delta-loop in repository");
  }
  return false;
}


int fsl_content_blob( fsl_cx * const f, fsl_id_t blobRid, fsl_buffer * const tgt ){
  fsl_db * const dbR = fsl_needs_repo(f);
  if(!dbR) return FSL_RC_NOT_A_REPO;
  else if(blobRid<=0){
    return fsl_cx_err_set(f, FSL_RC_RANGE,
                          "Invalid RID for %s().", __func__);
  }
  int rc = 0;
  fsl_stmt * const q = &f->cache.stmt.contentBlob;
  if(!q->stmt){
    rc = fsl_db_prepare( dbR, q,
                         "SELECT content, size FROM blob "
                         "WHERE rid=?1"
                         "/*%s()*/",__func__);
    if(rc) goto end;
  }
  rc = fsl_stmt_bind_id(q, 1, blobRid);
  if(!rc && (FSL_RC_STEP_ROW==(rc=fsl_stmt_step(q)))){
    void const * mem = NULL;
    fsl_size_t memLen = 0;
    int64_t const sz = fsl_stmt_g_int64(q, 1);
    if(sz<0){
      rc = fsl_cx_err_set(f, FSL_RC_PHANTOM,
                          "Cannot fetch content for phantom "
                          "blob #%"FSL_ID_T_PFMT".",
                          blobRid);
    }
    else if(sz){
      rc = fsl_stmt_get_blob(q, 0, &mem, &memLen);
      if(rc){
        rc = fsl_cx_err_set(f, rc,
                            "Error fetching blob content for "
                            "blob #%"FSL_ID_T_PFMT".", blobRid);
      }else{
        fsl_buffer bb = fsl_buffer_empty;
        assert(memLen>0);
        fsl_buffer_external(&bb, mem, memLen);
        rc = fsl_buffer_uncompress(&bb, tgt);
      }
    }else{
      rc = 0;
      fsl_buffer_reuse(tgt);
    }
  }
  else if(FSL_RC_STEP_DONE==rc){
    rc = fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                        "No blob found for rid %"FSL_ID_T_PFMT".",
                        blobRid);
  }
  end:
  fsl_stmt_reset(q);
  if(rc && !f->error.code && dbR->error.code){
    rc = fsl_cx_uplift_db_error2(f, dbR, rc);
  }
  return rc;
}


bool fsl_content_is_private(fsl_cx * const f, fsl_id_t rid){
  fsl_stmt * s1 = NULL;
  fsl_db * db = fsl_cx_db_repo(f);
  int rc = db
    ? fsl_db_prepare_cached(db, &s1,
                            "SELECT 1 FROM private "
                            "WHERE rid=?"
                            "/*%s()*/",__func__)
    : FSL_RC_MISUSE;
  if(!rc){
    rc = fsl_stmt_bind_id(s1, 1, rid);
    if(!rc) rc = fsl_stmt_step(s1);
    fsl_stmt_cached_yield(s1);
  }
  return rc==FSL_RC_STEP_ROW ? true : false;
}


int fsl_content_get( fsl_cx * const f, fsl_id_t rid,
                     fsl_buffer * const tgt ){
  fsl_db * const db = fsl_cx_db_repo(f);
  if(!tgt) return FSL_RC_MISUSE;
  else if(rid<=0){
    return fsl_cx_err_set(f, FSL_RC_RANGE,
                          "RID %"FSL_ID_T_PFMT" is out of range.",
                          rid);
  }
  else if(!db){
    return fsl_cx_err_set(f, FSL_RC_NOT_A_REPO,
                          "Fossil has no repo opened.");
  }
  else{
    int rc;
    bool gotIt = 0;
    fsl_id_t nextRid;
    fsl__bccache * const ac = &f->cache.blobContent;
    if(fsl_id_bag_contains(&ac->missing, rid)){
      /* Early out if we know the content is not available */
      return FSL_RC_NOT_FOUND;
    }
    /* Look for the artifact in the cache first */
    if(0!=(FSL_CX_F_BLOB_CACHE & f->flags)
       && fsl_id_bag_contains(&ac->inCache, rid) ){
      fsl_size_t i;
      fsl__bccache_line * line;
      for(i=0; i<ac->used; ++i){
        line = &ac->list[i];
        if( line->rid==rid ){
          ++ac->metrics.hits;
          rc = fsl_buffer_copy(tgt, &line->content);
          line->age = ac->nextAge++;
          return rc;
        }
      }
    }
    fsl_buffer_reuse(tgt);
    ++ac->metrics.misses;
    nextRid = 0;
    rc = fsl_delta_src_id(f, rid, &nextRid);
    /* MARKER(("rc=%d, nextRid=%"FSL_ID_T_PFMT"\n", rc, nextRid)); */
    if(rc) return rc;
    if( nextRid == 0 ){
      /* This is not a delta, so get its raw content. */
      rc = fsl_content_blob(f, rid, tgt);
      gotIt = 0==rc;
    }else{
      /* Looks like a delta, so let's expand it... */
      fsl_int_t n           /* number of used entries in 'a' */;
      fsl_int_t mx;
      fsl_id_t * a = NULL;
      //fsl_buffer D = fsl_buffer_empty;
      fsl_buffer * const delta = &f->cache.deltaContent;
      fsl_buffer next = fsl_buffer_empty  /* delta-applied content */ ;
      assert(nextRid>0);
      unsigned int nAlloc = 20;
      a = (fsl_id_t*)fsl_malloc(sizeof(fsl_id_t) * nAlloc);
      if(!a){
        rc = FSL_RC_OOM;
        goto end_delta;
      }
      a[0] = rid;
      a[1] = nextRid;
      n = 1;
      while( !fsl_id_bag_contains(&ac->inCache, nextRid)
             && 0==(rc=fsl_delta_src_id(f, nextRid, &nextRid))
             && (nextRid>0)){
        /* Figure out how big n needs to be... */
        ++n;
        if( n >= (fsl_int_t)nAlloc ){
          /* Expand 'a' */
          void * remem;
          if( n > fsl_db_g_int64(db, 0,
                                "SELECT max(rid) FROM blob")){
            rc = fsl_cx_err_set(f, FSL_RC_RANGE,
                                "Infinite loop in delta table.");
            goto end_delta;
          }
          unsigned int const nAlloc2 = nAlloc * 2;
          remem = fsl_realloc(a, nAlloc2 * sizeof(fsl_id_t));
          if(!remem){
            rc = FSL_RC_OOM;
            goto end_delta;
          }
          a = (fsl_id_t*)remem;
          nAlloc = nAlloc2;
          /*MARKER(("deltaIds allocated = %u\n", nAlloc));*/
        }
        a[n] = nextRid;
      }
      /**
         Recursively expand deltas to get the content...
      */
      mx = n;
      rc = fsl_content_get( f, a[n], tgt );
      /* MARKER(("Getting content for rid #%"FSL_ID_T_PFMT", rc=%d\n", a[n], rc)); */
      --n;
      for( ; !rc && (n>=0); --n){
        rc = fsl_content_blob(f, a[n], delta);
        /* MARKER(("Getting/applying delta rid #%"FSL_ID_T_PFMT", rc=%d\n", a[n], rc)); */
        if(rc) goto end_delta;
        if(!delta->used){
          assert(!"Is this possible? The fossil tree has a similar "
                 "condition but i naively don't believe it's necessary.");
          continue;
        }
        next = fsl_buffer_empty;
        rc = fsl_buffer_delta_apply2(tgt, delta, &next, &f->error);
        //assert(FSL_RC_RANGE!=rc);
        if(rc) goto end_delta;
#if 1
        /*
           2021-03-24: in a debug build, running:

           f-parseparty -t c -c -q

           (i.e.: parse and crosslink all checkin artifacts)

           on the libfossil repo with 2003 checkins takes:

           10.5s without this cache
           5.2s with this cache

           We shave another 0.5s if we always cache instead of using
           this mysterious (mx-n)%8 heuristic.

           Later testing with f-rebuild gives much different results:
           the (mx-n)%8 heuristic provides the best results of the
           variations tested, including always caching.
        */
        //MARKER(("mx=%d, n=%d, (mx-n)%%8=%d\n",
        //(int)mx, (int)n, (int)(mx-n)%8));
        //MARKER(("nAlloc=%d\n", (int)nAlloc));
        if( (mx-n)%8==0 ){
          //MARKER(("Caching artifact %d\n", (int)a[n+1]));
          fsl__bccache_insert( ac, a[n+1], tgt )
            /*Ignoring error (OOM) - it's not (yet) fatal. */;
          assert(!tgt->mem && "Passed to artifact cache (even on failure).");
        }else{
          fsl_buffer_clear(tgt);
        }
#else
        if(mx){/*unused var*/}
        fsl_buffer_clear(tgt);
#endif
        *tgt = next;
      }
      end_delta:
      fsl_free(a);
      fsl_buffer_reuse(delta);
      gotIt = 0==rc;
    }

    if(!rc){
      rc = fsl_id_bag_insert(gotIt
                             ? &f->cache.blobContent.available
                             : &f->cache.blobContent.missing,
                             rid);
    }
    return rc;
  }
}

int fsl_content_get_sym( fsl_cx * const f, char const * sym,
                         fsl_buffer * const tgt ){
  int rc;
  fsl_db * db = f ? fsl_needs_repo(f) : NULL;
  fsl_id_t rid = 0;
  if(!f || !sym || !tgt) return FSL_RC_MISUSE;
  else if(!db) return FSL_RC_NOT_A_REPO;
  rc = fsl_sym_to_rid(f, sym, FSL_SATYPE_ANY, &rid);
  return rc ? rc : fsl_content_get(f, rid, tgt);
}

/**
    Mark artifact rid as being available now. Update f's cache to show
    that everything that was formerly unavailable because rid was
    missing is now available. Returns 0 on success. f must have
    an opened repo and rid must be valid.
 */
static int fsl__content_mark_available(fsl_cx * const f, fsl_id_t rid){
  fsl_id_bag pending = fsl_id_bag_empty;
  int rc;
  fsl_stmt * st = NULL;
  fsl_db * db = fsl_cx_db_repo(f);
  assert(f);
  assert(db);
  assert(rid>0);
  if( fsl_id_bag_contains(&f->cache.blobContent.available, rid) ) return 0;
  rc = fsl_id_bag_insert(&pending, rid);
  while( 0==rc && (rid = fsl_id_bag_first(&pending))!=0 ){
    fsl_id_bag_remove(&pending, rid);
    rc = fsl_id_bag_insert(&f->cache.blobContent.available, rid);
    if(rc) break;
    fsl_id_bag_remove(&f->cache.blobContent.missing, rid);
    if(!st){
      rc = fsl_db_prepare_cached(db, &st,
                                 "SELECT rid FROM delta "
                                 "WHERE srcid=?"
                                 "/*%s()*/",__func__);
      if(rc) break;
    }
    rc = fsl_stmt_bind_id(st, 1, rid);
    while( !rc && (FSL_RC_STEP_ROW==fsl_stmt_step(st)) ){
      fsl_id_t const nx = fsl_stmt_g_id(st,0);
      assert(nx>0);
      rc = fsl_id_bag_insert(&pending, nx);
    }
  }
  if(st) fsl_stmt_cached_yield(st);
  fsl_id_bag_clear(&pending);
  return rc;
}

/**
   When a record is converted from a phantom to a real record, if that
   record has other records that are derived by delta, then call
   fsl__deck_crosslink() on those other records.

   If the formerly phantom record or any of the other records derived
   by delta from the former phantom are a baseline manifest, then also
   invoke fsl__deck_crosslink() on the delta-manifests associated with
   that baseline.

   Tail recursion is used to minimize stack depth.

   Returns 0 on success, any number of non-0 results on error.

   The 3rd argument must always be false except in recursive calls to
   this function.
*/
static int fsl_after_dephantomize(fsl_cx * f, fsl_id_t rid, bool doCrosslink){
  int rc = 0;
  unsigned nChildAlloc = 0;
  fsl_id_t * aChild = 0;
  fsl_buffer bufChild = fsl_buffer_empty;
  fsl_db * const db = fsl_cx_db_repo(f);
  fsl_stmt q = fsl_stmt_empty;

  MARKER(("WARNING: fsl_after_dephantomization() is UNTESTED.\n"));
  if(f->cache.ignoreDephantomizations) return 0;
  while(rid){
    unsigned nChildUsed = 0;
    unsigned i = 0;

    /* Parse the object rid itself */
    if(doCrosslink){
      fsl_deck deck = fsl_deck_empty;
      rc = fsl_deck_load_rid(f, &deck, rid, FSL_SATYPE_ANY);
      if(!rc){
        assert(aChild[i]==deck.rid);
        rc = fsl__deck_crosslink(&deck);
      }
      fsl_deck_finalize(&deck);
      if(rc) break;
    }
    /* Parse all delta-manifests that depend on baseline-manifest rid */
    rc = fsl_db_prepare(db, &q,
                        "SELECT rid FROM orphan WHERE baseline=%"FSL_ID_T_PFMT,
                        rid);
    if(rc) break;
    while(FSL_RC_STEP_ROW==fsl_stmt_step(&q)){
      fsl_id_t const child = fsl_stmt_g_id(&q, 0);
      if(nChildUsed>=nChildAlloc){
        nChildAlloc = nChildAlloc ? nChildAlloc*2 : 10;
        rc = fsl_buffer_reserve(&bufChild, sizeof(fsl_id_t)*nChildAlloc);
        if(rc) goto end;
        aChild = (fsl_id_t*)bufChild.mem;
      }
      aChild[nChildUsed++] = child;
    }
    fsl_stmt_finalize(&q);
    for(i=0; i<nChildUsed; ++i){
      fsl_deck deck = fsl_deck_empty;
      rc = fsl_deck_load_rid(f, &deck, aChild[i], FSL_SATYPE_ANY);
      if(!rc){
        assert(aChild[i]==deck.rid);
        rc = fsl__deck_crosslink(&deck);
      }
      fsl_deck_finalize(&deck);
      if(rc) goto end;
    }
    if( nChildUsed ){
      rc = fsl_db_exec_multi(db,
                             "DELETE FROM orphan WHERE baseline=%"FSL_ID_T_PFMT,
                             rid);
      if(rc){
        rc = fsl_cx_uplift_db_error(f, db);
      }
      break;
    }
    /* Recursively dephantomize all artifacts that are derived by
    ** delta from artifact rid and which have not already been
    ** cross-linked.  */
    nChildUsed = 0;
    rc = fsl_db_prepare(db, &q,
                        "SELECT rid FROM delta WHERE srcid=%"FSL_ID_T_PFMT
                        " AND NOT EXISTS(SELECT 1 FROM mlink WHERE mid=delta.rid)",
                        rid);
    if(rc){
      rc = fsl_cx_uplift_db_error(f, db);
      break;
    }
    while( FSL_RC_STEP_ROW==fsl_stmt_step(&q) ){
      fsl_id_t const child = fsl_stmt_g_id(&q, 0);
      if(nChildUsed>=nChildAlloc){
        nChildAlloc = nChildAlloc ? nChildAlloc*2 : 10;
        rc = fsl_buffer_reserve(&bufChild, sizeof(fsl_id_t)*nChildAlloc);
        if(rc) goto end;
        aChild = (fsl_id_t*)bufChild.mem;
      }
      aChild[nChildUsed++] = child;
    }
    fsl_stmt_finalize(&q);
    for(i=1; i<nChildUsed; ++i){
      rc = fsl_after_dephantomize(f, aChild[i], true);
      if(rc) break;
    }
    /* Tail recursion for the common case where only a single artifact
    ** is derived by delta from rid...
    ** (2021-06-06: this libfossil impl is not tail-recursive due to
    ** necessary cleanup) */
    rid = nChildUsed>0 ? aChild[0] : 0;
    doCrosslink = true;
  }
  end:
  fsl_stmt_finalize(&q);
  fsl_buffer_clear(&bufChild);
  return rc;
}

int fsl__content_put_ex( fsl_cx * const f,
                        fsl_buffer const * pBlob,
                        fsl_uuid_cstr zUuid,
                        fsl_id_t srcId,
                        fsl_size_t uncompSize,
                        bool isPrivate,
                        fsl_id_t * outRid){
  fsl_size_t size;
  fsl_id_t rid;
  fsl_stmt * s1 = NULL;
  fsl_buffer cmpr = fsl_buffer_empty;
  fsl_buffer hash = fsl_buffer_empty;
  bool markAsUnclustered = false;
  bool markAsUnsent = true;
  bool isDephantomize = false;
  fsl_db * dbR = fsl_cx_db_repo(f);
  int const zUuidLen = zUuid ? fsl_is_uuid(zUuid) : 0;
  int rc = 0;
  bool inTrans = false;
  assert(f);
  assert(dbR);
  assert(pBlob);
  assert(srcId==0 || zUuid!=NULL);
  assert(!zUuid || zUuidLen);
  if(!dbR) return FSL_RC_NOT_A_REPO;
  static const fsl_size_t MaxSize = 0x70000000;
  if(pBlob->used>=MaxSize || uncompSize>=MaxSize){
    /* fossil(1) uses int for all blob sizes, and therefore has a
       hard-coded limit of 2GB max size per blob. That property of the
       API is well-entrenched, and correcting it properly, including
       all algorithms which access blobs using integer indexes, would
       require a large coding effort with a non-trivial risk of
       lingering, difficult-to-trace bugs.

       For compatibility, we limit ourselves to 2GB, but to ensure a
       bit of leeway, we set our limit slightly less than 2GB.
    */
    return fsl_cx_err_set(f, FSL_RC_RANGE,
                          "For compatibility with fossil(1), "
                          "blobs may not exceed %d bytes in size.",
                          (int)MaxSize);
  }
  if(!zUuid){
    assert(0==uncompSize);
    /* "auxiliary hash" bits from:
       https://fossil-scm.org/fossil/file?ci=c965636958eb58aa&name=src%2Fcontent.c&ln=527-537
    */
    /* First check the auxiliary hash to see if there is already an artifact
    ** that uses the auxiliary hash name */
    /* 2021-04-13: we can now use fsl_repo_blob_lookup() to do this,
       but the following code is known to work, so touching it is a
       low priority. */
    rc = fsl_cx_hash_buffer(f, true, pBlob, &hash);
    if(FSL_RC_UNSUPPORTED==rc) rc = 0;
    else if(rc) goto end;
    assert(hash.used==0 || hash.used>=FSL_STRLEN_SHA1);
    rid = hash.used ? fsl_uuid_to_rid(f, fsl_buffer_cstr(&hash)) : 0;
    assert(rid>=0 && "Cannot have malformed/ambiguous UUID at this point.");
    if(!rid){
      /* No existing artifact with the auxiliary hash name.  Therefore, use
      ** the primary hash name. */
      hash.used = 0;
      rc = fsl_cx_hash_buffer(f, false, pBlob, &hash);
      if(rc) goto end;
      assert(hash.used>=FSL_STRLEN_SHA1);
    }
  }else{
    rc = fsl_buffer_append(&hash, zUuid, zUuidLen);
    if(rc) goto end;
  }
  assert(!rc);
  if(uncompSize){
    /* pBlob is assumed to be compressed. */
    assert(fsl_buffer_is_compressed(pBlob));
    size = uncompSize;
  }else{
    size = pBlob->used;
    if(srcId>0){
      rc = fsl_delta_applied_size(pBlob->mem, pBlob->used, &size);
      if(rc) goto end;
    }
  }
  rc = fsl_db_transaction_begin(dbR);
  if(rc) goto end;
  inTrans = true;
  if( f->cxConfig.hashPolicy==FSL_HPOLICY_AUTO && hash.used>FSL_STRLEN_SHA1 ){
    fsl_cx_err_reset(f);
    fsl_cx_hash_policy_set(f, FSL_HPOLICY_SHA3);
    if((rc = f->error.code)){
      goto end;
    }
  }
  /* Check to see if the entry already exists and if it does whether
     or not the entry is a phantom. */
  rc = fsl_db_prepare_cached(dbR, &s1,
                             "SELECT rid, size FROM blob "
                             "WHERE uuid=?"
                             "/*%s()*/",__func__);
  if(rc) goto end;
  rc = fsl_stmt_bind_step( s1, "b", &hash);
  switch(rc){
    case FSL_RC_STEP_ROW:
      rc = 0;
      rid = fsl_stmt_g_id(s1, 0);
      if( fsl_stmt_g_int64(s1, 1)>=0 ){
        /* The entry is not a phantom. There is nothing for us to do
           other than return the RID.
        */
        /*
          Reminder: the do-nothing-for-empty-phantom behaviour is
          arguable (but historical). There is a corner case there
          involving an empty file. So far, so good, though. After
          all...  all empty files have the same hash.
        */
        fsl_stmt_cached_yield(s1);
        assert(inTrans);
        fsl_db_transaction_end(dbR,0);
        if(outRid) *outRid = rid;
        fsl_buffer_clear(&hash);
        return 0;
      }
      break;
    case 0:
      /* No entry with the same UUID currently exists */
      rid = 0;
      markAsUnclustered = true;
      break;
    default:
      goto end;
  }
  if(s1){
    fsl_stmt_cached_yield(s1);
    s1 = NULL;
  }
  if(rc) goto end;

#if 0
  /* Requires app-level data. We might need a client hook mechanism or
     other metadata here.
  */
  /* Construct a received-from ID if we do not already have one */
  if( f->cache.rcvid <= 0 ){
    /* FIXME: use cached statement. */
    rc = fsl_db_exec(dbR, 
       "INSERT INTO rcvfrom(uid, mtime, nonce, ipaddr)"
       "VALUES(%d, julianday('now'), %Q, %Q)",
       g.userUid, g.zNonce, g.zIpAddr
    );
    f->cache.rcvid = fsl_db_last_insert_id(dbR);
  }
#endif

  if( uncompSize ){
    cmpr = *pBlob;
  }else{
    rc = fsl_buffer_compress(pBlob, &cmpr);
    if(rc) goto end;
  }

  if( rid>0 ){
#if 0
    assert(!"NYI: adding data to phantom. Requires some missing pieces.");
    rc = fsl_cx_err_set(f, FSL_RC_NYI,
                        "NYI: adding data to phantom. "
                        "Requires missing rcvId pieces.");
    goto end;
#else
    /* We are just adding data to a phantom */
    rc = fsl_db_prepare_cached(dbR, &s1,
                               "UPDATE blob SET "
                               "rcvid=?, size=?, content=? "
                               "WHERE rid=?"
                               "/*%s()*/",__func__);
    if(rc) goto end;
    rc = fsl_stmt_bind_step(s1, "RIBR", f->cache.rcvId, (int64_t)size,
                            &cmpr, rid);
    if(!rc){
      rc = fsl_db_exec(dbR, "DELETE FROM phantom "
                       "WHERE rid=%"FSL_ID_T_PFMT, rid
                       /* FIXME? use cached statement? */);
      if( !rc && (srcId==0 ||
                  0==fsl__bccache_check_available(f, srcId)) ){
        isDephantomize = true;
        rc = fsl__content_mark_available(f, rid);
      }
    }
    fsl_stmt_cached_yield(s1);
    s1 = NULL;
    if(rc) goto end;
#endif
  }else{
    /* We are creating a new entry */
    rc = fsl_db_prepare_cached(dbR, &s1,
                               "INSERT INTO blob "
                               "(rcvid,size,uuid,content) "
                               "VALUES(?,?,?,?)"
                               "/*%s()*/",__func__);
    if(rc) goto end;
    rc = fsl_stmt_bind_step(s1, "RIbB", f->cache.rcvId, (int64_t)size,
                            &hash, &cmpr);
    if(!rc){
      rid = fsl_db_last_insert_id(dbR);
      if(!pBlob ){
        rc = fsl_db_exec_multi(dbR,/* FIXME? use cached statement? */
                               "INSERT OR IGNORE INTO phantom "
                               "VALUES(%"FSL_ID_T_PFMT")",
                               rid);
        markAsUnsent = false;
      }
      if( !rc && (f->cache.markPrivate || isPrivate) ){
        rc = fsl_db_exec_multi(dbR,/* FIXME? use cached statement? */
                               "INSERT INTO private "
                               "VALUES(%"FSL_ID_T_PFMT")",
                               rid);
        markAsUnclustered = false;
        markAsUnsent = false;
      }
    }
    if(rc) rc = fsl_cx_uplift_db_error2(f, dbR, rc);
    fsl_stmt_cached_yield(s1);
    s1 = NULL;
    if(rc) goto end;
  }

  /* If the srcId is specified, then the data we just added is
     really a delta. Record this fact in the delta table.
  */
  if( srcId ){
    rc = fsl_db_prepare_cached(dbR, &s1,
                               "REPLACE INTO delta(rid,srcid) "
                               "VALUES(?,?)"
                               "/*%s()*/",__func__);
    if(!rc){
      rc = fsl_stmt_bind_step(s1, "RR", rid, srcId);
      if(rc) rc = fsl_cx_uplift_db_error2(f, dbR, rc);
      fsl_stmt_cached_yield(s1);
      s1 = NULL;
    }
    if(rc) goto end;
  }
  if( !isDephantomize
      && fsl_id_bag_contains(&f->cache.blobContent.missing, rid) && 
      (srcId==0 || (0==fsl__bccache_check_available(f,srcId)))){
    /*
      TODO: document what this is for.
      TODO: figure out what that is.
    */
    rc = fsl__content_mark_available(f, rid);
    if(rc) goto end;
  }
  if( isDephantomize ){
    rc = fsl_after_dephantomize(f, rid, false);
    if(rc) goto end;
  }

  /* Add the element to the unclustered table if has never been
     previously seen.
  */
  if( markAsUnclustered ){
    /* FIXME: use a cached statement. */
    rc = fsl_db_exec_multi(dbR,
                           "INSERT OR IGNORE INTO unclustered VALUES"
                           "(%"FSL_ID_T_PFMT")", rid);
    if(rc) goto end;
  }

  if( markAsUnsent ){
    /* FIXME: use a cached statement. */
    rc = fsl_db_exec(dbR, "INSERT OR IGNORE INTO unsent "
                     "VALUES(%"FSL_ID_T_PFMT")", rid);
    if(rc) goto end;
  }
  
  rc = fsl__repo_verify_before_commit(f, rid);
  if(rc) goto end /* FSL_RC_OOM is basically the "only possible" failure
                     after this point. */;
  /* Code after end: relies on the following 2 lines: */
  rc = fsl_db_transaction_end(dbR, false);
  inTrans = false;
  if(!rc){
    if(outRid) *outRid = rid;
  }
  end:
  if(inTrans){
    assert(0!=rc);
    fsl_db_transaction_end(dbR,true);
  }
  fsl_buffer_clear(&hash);
  if(!uncompSize){
    fsl_buffer_clear(&cmpr);
  }/* else cmpr.mem (if any) belongs to pBlob */
  return rc;
}

int fsl__content_put( fsl_cx * const f, fsl_buffer const * pBlob, fsl_id_t * newRid){
  return fsl__content_put_ex(f, pBlob, NULL, 0, 0, 0, newRid);
}

int fsl_uuid_is_shunned(fsl_cx * const f, fsl_uuid_cstr zUuid){
  fsl_db * db = fsl_cx_db_repo(f);
  if( !db || zUuid==0 || zUuid[0]==0 ) return 0;
  else if(FSL_HPOLICY_SHUN_SHA1==f->cxConfig.hashPolicy
          && FSL_STRLEN_SHA1==fsl_is_uuid(zUuid)){
    return 1;
  }
  /* TODO? cached query */
  return 1==fsl_db_g_int32( db, 0,
                            "SELECT 1 FROM shun WHERE uuid=%Q",
                            zUuid);
}

int fsl__content_new( fsl_cx * const f, fsl_uuid_cstr uuid,
                      bool isPrivate, fsl_id_t * const newId ){
  fsl_id_t rid = 0;
  int rc;
  fsl_db * db = fsl_cx_db_repo(f);
  fsl_stmt * s1 = NULL, * s2 = NULL;
  int const uuidLen = uuid ? fsl_is_uuid(uuid) : 0;
  if(!f || !uuid) return FSL_RC_MISUSE;
  else if(!uuidLen) return FSL_RC_RANGE;
  if(!db) return FSL_RC_NOT_A_REPO;
  if( fsl_uuid_is_shunned(f, uuid) ){
    return fsl_cx_err_set(f, FSL_RC_ACCESS,
                          "UUID is shunned: %s", uuid)
      /* need new error code? */;
  }
  rc = fsl_db_transaction_begin(db);
  if(rc) return rc;

  rc = fsl_db_prepare_cached(db, &s1,
                             "INSERT INTO blob(rcvid,size,uuid,content)"
                             "VALUES(0,-1,?,NULL)"
                             "/*%s()*/",__func__);
  if(rc) goto end;
  rc = fsl_stmt_bind_text(s1, 1, uuid, uuidLen, 0);
  if(!rc) rc = fsl_stmt_step(s1);
  fsl_stmt_cached_yield(s1);
  if(FSL_RC_STEP_DONE!=rc) goto end;
  else rc = 0;
  rid = fsl_db_last_insert_id(db);
  assert(rid>0);
  rc = fsl_db_prepare_cached(db, &s2,
                             "INSERT INTO phantom VALUES (?)"
                             "/*%s()*/",__func__);
  if(rc) goto end;
  rc = fsl_stmt_bind_id(s2, 1, rid);
  if(!rc) rc = fsl_stmt_step(s2);
  fsl_stmt_cached_yield(s2);
  if(FSL_RC_STEP_DONE!=rc) goto end;
  else rc = 0;

  if( f->cache.markPrivate || isPrivate ){
    /* Should be seldom enough that we don't need to cache
       this statement. */
    rc = fsl_db_exec(db,
                     "INSERT INTO private VALUES(%"FSL_ID_T_PFMT")",
                     (fsl_id_t)rid);
  }else{
    fsl_stmt * s3 = NULL;
    rc = fsl_db_prepare_cached(db, &s3,
                               "INSERT INTO unclustered VALUES(?)");
    if(!rc){
      rc = fsl_stmt_bind_id(s3, 1, rid);
      if(!rc) rc = fsl_stmt_step(s3);
      fsl_stmt_cached_yield(s3);
      if(FSL_RC_STEP_DONE!=rc) goto end;
      else rc = 0;
    }
  }

  if(!rc) rc = fsl_id_bag_insert(&f->cache.blobContent.missing, rid);
  
  end:
  if(rc){
    if(db->error.code && !f->error.code){
      fsl_cx_uplift_db_error(f, db);
    }
    fsl_db_transaction_rollback(db);
  }
  else{
    rc = fsl_db_transaction_commit(db);
    if(!rc && newId) *newId = rid;
    else if(rc && !f->error.code){
      fsl_cx_uplift_db_error(f, db);
    }
  }
  return rc;
}

int fsl__content_undeltify(fsl_cx * const f, fsl_id_t rid){
  int rc;
  fsl_db * const db = fsl_needs_repo(f);
  fsl_id_t srcid = 0;
  fsl_buffer x = fsl_buffer_empty;
  fsl_stmt s = fsl_stmt_empty;
  if(!db) return FSL_RC_NOT_A_REPO;
  else if(rid<=0) return FSL_RC_RANGE;
  rc = fsl_db_transaction_begin(db);
  if(rc) return fsl_cx_uplift_db_error2(f, db, rc);
  /* Reminder: the original impl does not do this in a
     transaction, _possibly_ because it's only done from places
     where a transaction is active (that's unconfirmed).
     Nested transactions are very cheap, though. */
  rc = fsl_delta_src_id( f, rid, &srcid );
  if(rc || srcid<=0) goto end;
  rc = fsl_content_get(f, rid, &x);
  if( rc || !x.used ) goto end;
  rc = fsl_db_prepare(db, &s,
                      "UPDATE blob SET content=?,"
                      " size=%" FSL_SIZE_T_PFMT
                      " WHERE rid=%" FSL_ID_T_PFMT,
                      x.used, rid);
  if(rc) goto dberr;
  rc = fsl_buffer_compress(&x, &x);
  if(rc) goto end;
  rc = fsl_stmt_bind_step(&s, "B", &x);
  if(rc) goto dberr;
  rc = fsl_db_exec(db, "DELETE FROM delta WHERE rid=%"FSL_ID_T_PFMT,
                   rid);
  if(rc) goto dberr;
#if 0
  /*
    fossil does not do this, but that seems like an inconsistency.

    On that topic Richard says:

    "When you undelta an artifact, however, it is then stored as
    plain text.  (Actually, as zlib compressed plain text.)  There
    is no possibility of delta loops or bugs in the delta encoder or
    missing source artifacts.  And so there is much less of a chance
    of losing content.  Hence, I didn't see the need to verify the
    content of artifacts that are undelta-ed."

    Potential TODO: f->flags FSL_CX_F_PEDANTIC_VERIFICATION, which
    enables the R-card and this check, and any similarly superfluous
    ones.
  */
  if(!rc) fsl__repo_verify_before_commit(f, rid);
#endif
  end:
  fsl_buffer_clear(&x);
  fsl_stmt_finalize(&s);
  if(rc) fsl_db_transaction_rollback(db);
  else rc = fsl_db_transaction_commit(db);
  return rc;
  dberr:
  assert(rc);
  rc = fsl_cx_uplift_db_error2(f, db, rc);
  goto end;
}

int fsl__content_deltify(fsl_cx * f, fsl_id_t rid,
                        fsl_id_t srcid, bool force){
  fsl_id_t s;
  fsl_buffer data = fsl_buffer_empty;
  fsl_buffer src = fsl_buffer_empty;
  fsl_buffer * const delta = &f->cache.deltaContent;
  fsl_db * const db = fsl_needs_repo(f);
  int rc = 0;
  enum { MinSizeThreshold = 50 };
  if(rid<=0 || srcid<=0) return FSL_RC_RANGE;
  else if(!db) return FSL_RC_NOT_A_REPO;
  else if( srcid==rid ) return 0;
  else if(!fsl_content_is_available(f, rid)){
    return 0;
  }
  if(!force){
    fsl_id_t tmpRid = 0;
    rc = fsl_delta_src_id(f, rid, &tmpRid);
    if(tmpRid>0){
      /*
        We already have a delta, it seems. Nothing left to do
        :-D. Should we return FSL_RC_ALREADY_EXISTS here?
      */
      return 0;
    }
    else if(rc) return rc;
  }

  if( fsl_content_is_private(f, srcid)
      && !fsl_content_is_private(f, rid) ){
    /*
      See API doc comments about crossing the private/public
      boundaries. Do we want to report okay here or
      FSL_RC_ACCESS? Not yet sure how this routine is used.

      Since delitifying is an internal optimization/implementation
      detail, it seems best to return 0 for this case.
    */
    return 0;
  }
  /**
     Undeltify srcid if needed...
  */
  s = srcid;
  while( (0==(rc=fsl_delta_src_id(f, s, &s)))
         && (s>0) ){
    if( s==rid ){
      rc = fsl__content_undeltify(f, srcid);
      break;
    }
  }
  if(rc) return rc;
  /* As of here, don't return on error. Use (goto end) instead, or be
     really careful, b/c buffers might need cleaning. */
  rc = fsl_content_get(f, srcid, &src);
  if(rc || (src.used < MinSizeThreshold)
     /* See API doc comments about minimum size to delta/undelta. */
     ) goto end;
  rc = fsl_content_get(f, rid, &data);
  if(rc || (data.used < MinSizeThreshold)) goto end;
  rc = fsl_buffer_delta_create(&src, &data, delta);
  if( !rc && (delta->used <= (data.used * 3 / 4 /* 75% */))){
    fsl_stmt * s1 = NULL;
    fsl_stmt * s2 = NULL;
    rc = fsl_buffer_compress(delta, &data);
    if(rc) goto end;
    rc = fsl_db_prepare_cached(db, &s1,
                               "UPDATE blob SET content=? "
                               "WHERE rid=?/*%s()*/",__func__);
    if(!rc){
      fsl_stmt_bind_id(s1, 2, rid);
      rc = fsl_stmt_bind_blob(s1, 1, data.mem, data.used, 0);
      if(!rc){
        rc = fsl_db_prepare_cached(db, &s2,
                                   "REPLACE INTO delta(rid,srcid) "
                                   "VALUES(?,?)/*%s()*/",__func__);
        if(!rc){
          fsl_stmt_bind_id(s2, 1, rid);
          fsl_stmt_bind_id(s2, 2, srcid);
          rc = fsl_db_transaction_begin(db);
          if(!rc){
            rc = fsl_stmt_step(s1);
            if(FSL_RC_STEP_DONE==rc){
              rc = fsl_stmt_step(s2);
              if(FSL_RC_STEP_DONE==rc) rc = 0;
            }
            if(!rc) rc = fsl_db_transaction_end(db, 0);
            else fsl_db_transaction_end(db, 1) /* keep rc intact */;
          }
        }
      }
    }
    fsl_stmt_cached_yield(s1);
    fsl_stmt_cached_yield(s2);
    if(!rc) fsl__repo_verify_before_commit(f, rid);
  }
  end:
  if(rc && db->error.code && !f->error.code){
    fsl_cx_uplift_db_error(f,db);
  }
  fsl_buffer_clear(&src);
  fsl_buffer_clear(&data);
  fsl_buffer_reuse(delta);
  return rc;
}

/**
    Removes all entries from the repo's blob table which are listed in
    the shun table.
 */
int fsl__repo_shun_artifacts(fsl_cx * const f){
  fsl_stmt q = fsl_stmt_empty;
  int rc;
  fsl_db * db = f ? fsl_cx_db_repo(f) : NULL;
  if(!f) return FSL_RC_MISUSE;
  else if(!db) return FSL_RC_NOT_A_REPO;
  rc = fsl_db_transaction_begin(db);
  if(rc) return rc;
  rc = fsl_db_exec_multi(db,
                         "CREATE TEMP TABLE IF NOT EXISTS "
                         "toshun(rid INTEGER PRIMARY KEY); "
                         "DELETE FROM toshun; "
                         "INSERT INTO toshun SELECT rid FROM blob, shun "
                         "WHERE blob.uuid=shun.uuid;"
  );
  if(rc) goto end;
  /* Ensure that deltas generated from the to-be-shunned data
     are unpacked into non-delta form...
  */
  rc = fsl_db_prepare(db, &q,
                      "SELECT rid FROM delta WHERE srcid IN toshun"
                      );
  if(rc) goto end;
  while( !rc && (FSL_RC_STEP_ROW==fsl_stmt_step(&q)) ){
    fsl_id_t const srcid = fsl_stmt_g_id(&q, 0);
    rc = fsl__content_undeltify(f, srcid);
  }
  fsl_stmt_finalize(&q);
  if(!rc){
    rc = fsl_db_exec_multi(db,
            "DELETE FROM delta WHERE rid IN toshun;"
            "DELETE FROM blob WHERE rid IN toshun;"
            "DELETE FROM toshun;"
            "DELETE FROM private "
            "WHERE NOT EXISTS "
            "(SELECT 1 FROM blob WHERE rid=private.rid);"
    );
  }
  end:
  if(!rc) rc = fsl_db_transaction_commit(db);
  else fsl_db_transaction_rollback(db);
  if(rc && db->error.code && !f->error.code){
    rc = fsl_cx_uplift_db_error(f, db);
  }
  return rc;
}

int fsl_content_make_public(fsl_cx * const f, fsl_id_t rid){
  int rc;
  fsl_db * db = f ? fsl_cx_db_repo(f) : NULL;
  if(!f) return FSL_RC_MISUSE;
  else if(!db) return FSL_RC_NOT_A_REPO;
  rc = fsl_db_exec(db, "DELETE FROM private "
                   "WHERE rid=%" FSL_ID_T_PFMT, rid);
  return rc ? fsl_cx_uplift_db_error(f, db) : 0;
}

/**
    Load the record ID rid and up to N-1 closest ancestors into
    the "fsl_computed_ancestors" table.
 */
static int fsl__compute_ancestors( fsl_db * const db, fsl_id_t rid,
                                   int N, bool directOnly ){
  fsl_stmt st = fsl_stmt_empty;
  int rc = fsl_db_prepare(db, &st,
    "WITH RECURSIVE "
    "  ancestor(rid, mtime) AS ("
    "    SELECT ?, mtime "
    "      FROM event WHERE objid=? "
    "    UNION "
    "    SELECT plink.pid, event.mtime"
    "      FROM ancestor, plink, event"
    "     WHERE plink.cid=ancestor.rid"
    "       AND event.objid=plink.pid %s"
    "     ORDER BY mtime DESC LIMIT ?"
    "  )"
    "INSERT INTO fsl_computed_ancestors"
    "  SELECT rid FROM ancestor;",
    directOnly ? "AND plink.isPrim" : ""
  );
  if(!rc){
    rc = fsl_stmt_bind_step(&st, "RRi", rid, rid, (int32_t)N);
  }
  fsl_stmt_finalize(&st);
  return rc;
}

int fsl_mtime_of_F_card(fsl_cx * const f, fsl_id_t vid,
                        fsl_card_F const * const fc,
                        fsl_time_t * const pMTime){
  if(!f || !fc) return FSL_RC_MISUSE;
  else if(vid<=0) return FSL_RC_RANGE;
  else if(!fc->uuid){
    if(pMTime) *pMTime = 0;
    return 0;
  }else{
    fsl_id_t fid = fsl_uuid_to_rid(f, fc->uuid);
    if(fid<=0){
      assert(f->error.code);
      return f->error.code;
    }else{
      return fsl_mtime_of_manifest_file(f, vid, fid, pMTime);
    }
  }
}

int fsl_mtime_of_manifest_file(fsl_cx * const f, fsl_id_t vid, fsl_id_t fid,
                               fsl_time_t * const pMTime){
  fsl_db * const db = fsl_needs_repo(f);
  fsl_stmt * q = NULL;
  int rc = 0;
  if(!db) return FSL_RC_NOT_A_REPO;

  if(fid<=0){
    /* Only fetch the checkin time... */
    int64_t i = -1;
    rc = fsl_db_get_int64(db, &i, 
                          "SELECT (mtime-2440587.5)*86400 "
                          "FROM event WHERE objid=%"FSL_ID_T_PFMT
                          " AND type='ci'",
                          (fsl_id_t)vid);
    if(!rc){
      if(i<0) rc = FSL_RC_NOT_FOUND;
      else if(pMTime) *pMTime = (fsl_time_t)i;
    }
    return rc;
  }

  if( f->cache.mtimeManifest != vid ){
    /*
      Computing (and keeping) ancestors is relatively costly, so we
      keep only the copy associated with f->cache.mtimeManifest
      around. For the general case, we will be feeding this function
      files from the same manifest.
    */
    f->cache.mtimeManifest = vid;
    rc = fsl_db_exec_multi(db, "CREATE TEMP TABLE IF NOT EXISTS "
                           "fsl_computed_ancestors"
                           "(x INTEGER PRIMARY KEY); "
                           "DELETE FROM fsl_computed_ancestors;");
    if(!rc){
      rc = fsl__compute_ancestors(db, vid, 1000000, 1);
    }
    if(rc){
      fsl_cx_uplift_db_error(f, db);
      return rc;
    }
  }
  rc = fsl_db_prepare_cached(db, &q,
    "SELECT (max(event.mtime)-2440587.5)*86400 FROM mlink, event"
    " WHERE mlink.mid=event.objid"
    "   AND mlink.fid=?"
    "   AND +mlink.mid IN fsl_computed_ancestors"
  );
  if(!rc){
    fsl_stmt_bind_id(q, 1, fid);
    rc = fsl_stmt_step(q);
    if( FSL_RC_STEP_ROW==rc ){
      rc = 0;
      if(pMTime) *pMTime = (fsl_time_t)fsl_stmt_g_int64(q, 0);
    }else{
      assert(rc);
      if(FSL_RC_STEP_DONE==rc) rc = FSL_RC_NOT_FOUND;
    }
    fsl_stmt_cached_yield(q);
    /* Reminder: DO NOT clean up fsl_computed ancestors here. Doing so
       is not only costly later on but also breaks test code. */
  }
  return rc;
}

int fsl_card_F_content( fsl_cx * f, fsl_card_F const * fc,
                        fsl_buffer * const dest ){
  if(!f || !fc || !dest) return FSL_RC_MISUSE;
  else if(!fc->uuid){
    return fsl_cx_err_set(f, FSL_RC_RANGE,
                          "Cannot fetch content of a deleted file "
                          "because it has no UUID.");
  }
  else if(!fsl_needs_repo(f)) return FSL_RC_NOT_A_REPO;
  else{
    fsl_id_t const rid = fsl_uuid_to_rid(f, fc->uuid);
    if(!rid) return fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                                   "UUID not found: %s",
                                   fc->uuid);
    else if(rid<0){
      assert(f->error.code);
      return f->error.code;
    }else{
      return fsl_content_get(f, rid, dest);
    }
  }
}


/**
   UNTESTED (but closely derived from known-working code).

   Expects f to have an opened checkout. Assumes zName is resolvable
   (via fsl_ckout_filename_check() - see that function for the
   meaning of the relativeToCwd argument) to a path under the current
   checkout root. It loads the file's contents and stores them into
   the blob table. If rid is not NULL, *rid is assigned the blob.rid
   (possibly new, possilbly re-used!). If uuid is not NULL then *uuid
   is assigned to the content's UUID. The *uuid bytes are owned by the
   caller, who must eventually fsl_free() them. If content with the
   same UUID already exists, it does not get re-imported but rid/uuid
   will (if not NULL) contain the values of any previous content
   with the same hash.

   ACHTUNG: this function DOES NOT CARE whether or not the file is
   actually part of a checkout or not, nor whether it is actually
   referenced by any checkins, or such, other than that it must
   resolve to something under the checkout root (to avoid breaking any
   internal assumptions in fossil about filenames). It will add new
   repo.filename entries as needed for this function. Thus is can be
   used to import "shadow files" either not known about by fossil or
   not _yet_ known about by fossil.

   If parentRid is >0 then it must refer to the previous version of
   zName's content. The parent version gets deltified vs the new one,
   but deltification is a suggestion which the library will ignore if
   (e.g.) the parent content is already a delta of something else.

   This function does its DB-side work in a transaction, so, e.g.  if
   saving succeeds but deltification of the parent version fails for
   some reason, the whole save operation is rolled back.

   Returns 0 on success. On error rid and uuid are not modified.
*/
int fsl_import_file( fsl_cx * f, char relativeToCwd,
                     char const * zName,
                     fsl_id_t parentRid,
                     fsl_id_t *rid, fsl_uuid_str * uuid ){
  fsl_buffer * canon = 0; // canonicalized filename
  fsl_buffer * nbuf = 0; // filename buffer
  char const * fn;
  int rc;
  fsl_id_t fnid = 0;
  fsl_id_t rcRid = 0;
  fsl_db * db = f ? fsl_needs_repo(f) : NULL;
  char inTrans = 0;
  if(!zName || !*zName) return FSL_RC_MISUSE;
  else if(!f->ckout.dir) return FSL_RC_NOT_A_CKOUT;
  else if(!db) return FSL_RC_NOT_A_REPO;
  fsl_buffer * const fbuf = fsl__cx_content_buffer(f);
  canon = fsl__cx_scratchpad(f);
  nbuf = fsl__cx_scratchpad(f);

  assert(f->ckout.dir);

  /* Normalize the name... i often regret having
     fsl_ckout_filename_check() return checkout-relative paths.
  */
  rc = fsl_ckout_filename_check(f, relativeToCwd, zName, canon);
  if(rc) goto end;

  /* Find or create a repo.filename entry... */
  fn = fsl_buffer_cstr(canon);

  rc = fsl_db_transaction_begin(db);
  if(rc) goto end;
  inTrans = 1;

  rc = fsl__repo_filename_fnid2(f, fn, &fnid, 1);
  if(rc) goto end;

  /* Import the file... */
  assert(fnid>0);
  rc = fsl_buffer_appendf(nbuf, "%s%s", f->ckout.dir, fn);
  if(rc) goto end;
  fn = fsl_buffer_cstr(nbuf);
  rc = fsl_buffer_fill_from_filename( fbuf, fn );
  if(rc){
    fsl_cx_err_set(f, rc, "Error %s importing file: %s",
                   fsl_rc_cstr(rc), fn);
    goto end;
  }
  fn = NULL;
  rc = fsl__content_put( f, fbuf, &rcRid );
  if(!rc){
    assert(rcRid > 0);
    if(parentRid>0){
      /* Make parent version a delta of this one, if possible... */
      rc = fsl__content_deltify(f, parentRid, rcRid, 0);
    }
    if(!rc){
      if(rid) *rid = rcRid;
      if(uuid){
        fsl_cx_err_reset(f);
        *uuid = fsl_rid_to_uuid(f, rcRid);
        if(!*uuid) rc = (f->error.code ? f->error.code : FSL_RC_OOM);
      }
    }
  }

  if(!rc){
    assert(inTrans);
    inTrans = 0;
    rc = fsl_db_transaction_commit(db);
  }

  end:
  fsl__cx_content_buffer_yield(f);
  assert(0==fbuf->used);
  fsl__cx_scratchpad_yield(f, canon);
  fsl__cx_scratchpad_yield(f, nbuf);
  if(inTrans) fsl_db_transaction_rollback(db);
  return rc;
}

fsl_hash_types_e fsl_validate_hash(const char *zHash, int nHash){
  /* fossil(1) counterpart: hname_validate() */
  fsl_hash_types_e rc;
  switch(nHash){
    case FSL_STRLEN_SHA1: rc = FSL_HTYPE_SHA1; break;
    case FSL_STRLEN_K256: rc = FSL_HTYPE_K256; break;
    default: return FSL_HTYPE_ERROR;
  }
  return fsl_validate16(zHash, (fsl_size_t)nHash) ? rc : FSL_HTYPE_ERROR;
}

const char * fsl_hash_type_name(fsl_hash_types_e h, const char *zUnknown){
  /* fossil(1) counterpart: hname_alg() */
  switch(h){
    case FSL_HTYPE_SHA1: return "SHA1";
    case FSL_HTYPE_K256: return "SHA3-256";
    default: return zUnknown;
  }
}

fsl_hash_types_e fsl_verify_blob_hash(fsl_buffer const * pIn,
                                      const char *zHash, int nHash){
  fsl_hash_types_e id = FSL_HTYPE_ERROR;
  switch(nHash){
    case FSL_STRLEN_SHA1:{
      fsl_sha1_cx cx;
      char hex[FSL_STRLEN_SHA1+1] = {0};
      fsl_sha1_init(&cx);
      fsl_sha1_update(&cx, pIn->mem, (unsigned)pIn->used);
      fsl_sha1_final_hex(&cx, hex);
      if(0==memcmp(hex, zHash, FSL_STRLEN_SHA1)){
        id = FSL_HTYPE_SHA1;
      }
      break;
    }
    case FSL_STRLEN_K256:{
      fsl_sha3_cx cx;
      unsigned char const * hex;
      fsl_sha3_init(&cx);
      fsl_sha3_update(&cx, pIn->mem, (unsigned)pIn->used);
      hex = fsl_sha3_end(&cx);
      if(0==memcmp(hex, zHash, FSL_STRLEN_K256)){
        id = FSL_HTYPE_K256;
      }
      break;
    }
    default:
      break;
  }
  return id;
}


int fsl__shunned_remove(fsl_cx * const f){
  fsl_stmt q = fsl_stmt_empty;
  int rc;
  assert(fsl_cx_db_repo(f));
  rc = fsl_cx_exec_multi(f,
     "CREATE TEMP TABLE toshun(rid INTEGER PRIMARY KEY);"
     "INSERT INTO toshun SELECT rid FROM blob, shun WHERE blob.uuid=shun.uuid;"
  );
  if(rc) goto end;
  rc = fsl_cx_prepare(f, &q,
     "SELECT rid FROM delta WHERE srcid IN toshun"
  );
  while( 0==rc && FSL_RC_STEP_ROW==fsl_stmt_step(&q) ){
    rc = fsl__content_undeltify(f, fsl_stmt_g_id(&q, 0));
  }
  fsl_stmt_finalize(&q);
  if(rc) goto end;
  rc = fsl_cx_exec_multi(f,
     "DELETE FROM delta WHERE rid IN toshun;"
     "DELETE FROM blob WHERE rid IN toshun;"
     "DROP TABLE toshun;"
     "DELETE FROM private "
     " WHERE NOT EXISTS (SELECT 1 FROM blob WHERE rid=private.rid);"
  );
  end:
  fsl_stmt_finalize(&q);
  return rc;
}

#undef MARKER
/* end of file ./src/content.c */
/* start of file ./src/config.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/
/**************************************************************************
  This file implements (most of) the fsl_xxx APIs related to handling
  configuration data from the db(s).
*/
#include <assert.h>
#include <stdlib.h> /* bsearch() */
#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)

/**
   File-local macro used to ensure that all cached statements used by
   the fsl_config_get_xxx() APIs use an equivalent string (so that
   they use the same underlying cache fsl_stmt handle).  The first %s
   represents one of the config table names (config, vvfar,
   global_config). Normally we wouldn't use %s in a cached statement,
   but we're only expecting 3 values for table here and each one will
   only be cached once. The 2nd %s must be __FILE__.
*/
#define SELECT_FROM_CONFIG "SELECT value FROM %s WHERE name=?/*%s*/"

static int fsl__confdb_to_role(fsl_confdb_e m){
  switch(m){
    case FSL_CONFDB_REPO: return FSL_DBROLE_REPO;
    case FSL_CONFDB_CKOUT: return FSL_DBROLE_CKOUT;
    case FSL_CONFDB_GLOBAL: return FSL_DBROLE_CONFIG;
    default: return FSL_DBROLE_NONE;
  }
}

char const * fsl_config_table_for_role(fsl_confdb_e mode){
  switch(mode){
    case FSL_CONFDB_REPO: return "config";
    case FSL_CONFDB_CKOUT: return "vvar";
    case FSL_CONFDB_GLOBAL: return "global_config";
    case FSL_CONFDB_VERSIONABLE: return NULL;
    default:
      assert(!"Invalid fsl_confdb_e value");
      return NULL;
  }
}

fsl_db * fsl_config_for_role(fsl_cx * const f, fsl_confdb_e mode){
  switch(mode){
    case FSL_CONFDB_REPO: return fsl_cx_db_repo(f);
    case FSL_CONFDB_CKOUT: return fsl_cx_db_ckout(f);
    case FSL_CONFDB_GLOBAL: return fsl_cx_db_config(f);
    case FSL_CONFDB_VERSIONABLE: return fsl_cx_db(f);
    default:
      assert(!"Invalid fsl_confdb_e value");
      return NULL;
  }
}

int fsl_config_versionable_filename(fsl_cx *f, char const * key,
                                    fsl_buffer *b){
  if(!fsl_needs_ckout(f)) return FSL_RC_NOT_A_CKOUT;
  else if(!key || !*key || !fsl_is_simple_pathname(key, true)){
    return FSL_RC_MISUSE;
  }
  fsl_buffer_reuse(b);
  return fsl_buffer_appendf(b, "%s.fossil-settings/%s",
                            f->ckout.dir, key);
}


int fsl_config_unset( fsl_cx * const f, fsl_confdb_e mode, char const * key ){
  fsl_db * const db = fsl_config_for_role(f, mode);
  if(!db || !key || !*key) return FSL_RC_MISUSE;
  else if(mode==FSL_CONFDB_VERSIONABLE) return FSL_RC_UNSUPPORTED;
  else{
    char const * table = fsl_config_table_for_role(mode);
    assert(table);
    return fsl_db_exec(db, "DELETE FROM %s WHERE name=%Q", table, key);
  }
}

int32_t fsl_config_get_int32( fsl_cx * const f, fsl_confdb_e mode,
                              int32_t dflt, char const * key ){
  int32_t rv = dflt;
  switch(mode){
    case FSL_CONFDB_VERSIONABLE:{
      char * const val = fsl_config_get_text(f, mode, key, NULL);
      if(val){
        rv = (int32_t)atoi(val);
        fsl_free(val);
      }
      break;
    }
    default: {
      fsl_db * const db = fsl_config_for_role(f, mode);
      char const * const table = fsl_config_table_for_role(mode);
      assert(table);
      if(db){
        fsl_stmt * st = NULL;
        fsl_db_prepare_cached(db, &st, SELECT_FROM_CONFIG,
                              table, __FILE__);
        if(st){
          st->role = fsl__confdb_to_role(mode);
          fsl_stmt_bind_text(st, 1, key, -1, 0);
          if(FSL_RC_STEP_ROW==fsl_stmt_step(st)){
            rv = fsl_stmt_g_int32(st, 0);
          }
          fsl_stmt_cached_yield(st);
        }
      }
      break;
    }
  }
  return rv;
}

int64_t fsl_config_get_int64( fsl_cx * const f, fsl_confdb_e mode,
                              int64_t dflt, char const * key ){
  int64_t rv = dflt;
  switch(mode){
    case FSL_CONFDB_VERSIONABLE:{
      char * const val = fsl_config_get_text(f, mode, key, NULL);
      if(val){
        rv = (int64_t)strtoll(val, NULL, 10);
        fsl_free(val);
      }
      break;
    }
    default: {
      fsl_db * const db = fsl_config_for_role(f, mode);
      char const * const table = fsl_config_table_for_role(mode);
      assert(table);
      if(db){
        fsl_stmt * st = NULL;
        fsl_db_prepare_cached(db, &st, SELECT_FROM_CONFIG,
                              table, __FILE__);
        if(st){
          st->role = fsl__confdb_to_role(mode);
          fsl_stmt_bind_text(st, 1, key, -1, 0);
          if(FSL_RC_STEP_ROW==fsl_stmt_step(st)){
            rv = fsl_stmt_g_int64(st, 0);
          }
          fsl_stmt_cached_yield(st);
        }
      }
      break;
    }
  }
  return rv;
}

fsl_id_t fsl_config_get_id( fsl_cx * const f, fsl_confdb_e mode,
                            fsl_id_t dflt, char const * key ){
  return (sizeof(fsl_id_t)==sizeof(int32_t))
    ? (fsl_id_t)fsl_config_get_int32(f, mode, dflt, key)
    : (fsl_id_t)fsl_config_get_int64(f, mode, dflt, key);
}

double fsl_config_get_double( fsl_cx * const f, fsl_confdb_e mode,
                              double dflt, char const * key ){
  double rv = dflt;
  switch(mode){
    case FSL_CONFDB_VERSIONABLE:{
      char * const val = fsl_config_get_text(f, mode, key, NULL);
      if(val){
        rv = strtod(val, NULL);
        fsl_free(val);
      }
      break;
    }
    default: {
      fsl_db * const db = fsl_config_for_role(f, mode);
      if(!db) break/*e.g. global config is not opened*/;
      fsl_stmt * st = NULL;
      char const * const table = fsl_config_table_for_role(mode);
      assert(table);
      fsl_db_prepare_cached(db, &st, SELECT_FROM_CONFIG,
                            table, __FILE__);
      if(st){
        st->role = fsl__confdb_to_role(mode);
        fsl_stmt_bind_text(st, 1, key, -1, 0);
        if(FSL_RC_STEP_ROW==fsl_stmt_step(st)){
          rv = fsl_stmt_g_double(st, 0);
        }
        fsl_stmt_cached_yield(st);
      }
      break;
    }
  }
  return rv;
}

char * fsl_config_get_text( fsl_cx * const f, fsl_confdb_e mode,
                            char const * key, fsl_size_t * len ){
  char * rv = NULL;
  fsl_buffer val = fsl_buffer_empty;
  if(fsl_config_get_buffer(f, mode, key, &val)){
    fsl_cx_err_reset(f);
    if(len) *len = 0;
    fsl_buffer_clear(&val)/*in case of partial read failure*/;
  }else{
    if(len) *len = val.used;
    rv = fsl_buffer_take(&val);
  }
  return rv;
}

int fsl_config_get_buffer( fsl_cx * const f, fsl_confdb_e mode,
                           char const * key, fsl_buffer * const b ){
  int rc = FSL_RC_NOT_FOUND;
  fsl_buffer_reuse(b);
  switch(mode){
    case FSL_CONFDB_VERSIONABLE:{
      if(!fsl_needs_ckout(f)){
        rc = FSL_RC_NOT_A_CKOUT;
        break;
      }
      fsl_buffer * const fname = fsl__cx_scratchpad(f);
      rc = fsl_config_versionable_filename(f, key, fname);
      if(!rc){
        char const * const zFile = fsl_buffer_cstr(fname);
        rc = fsl_stat(zFile, 0, false);
        if(rc){
          rc = fsl_cx_err_set(f, rc, "Could not stat file: %s",
                              zFile);
        }else{
          rc = fsl_buffer_fill_from_filename(b, zFile);
        }
      }
      fsl__cx_scratchpad_yield(f,fname);
      break;
    }
    default: {
      char const * const table = fsl_config_table_for_role(mode);
      assert(table);
      fsl_db * const db = fsl_config_for_role(f, mode);
      if(!db) break;
      fsl_stmt * st = NULL;
      rc = fsl_db_prepare_cached(db, &st, SELECT_FROM_CONFIG,
                                 table, __FILE__);
      if(rc){
        rc = fsl_cx_uplift_db_error2(f, db, rc);
        break;
      }
      st->role = fsl__confdb_to_role(mode);
      fsl_stmt_bind_text(st, 1, key, -1, 0);
      if(FSL_RC_STEP_ROW==fsl_stmt_step(st)){
        fsl_size_t len = 0;
        char const * const s = fsl_stmt_g_text(st, 0, &len);
        rc = s ? fsl_buffer_append(b, s, len) : 0;
      }else{
        rc = FSL_RC_NOT_FOUND;
      }
      fsl_stmt_cached_yield(st);
      break;
    }
  }
  return rc;
}

bool fsl_config_get_bool( fsl_cx * const f, fsl_confdb_e mode,
                          bool dflt, char const * key ){
  bool rv = dflt;
  switch((key && *key) ? mode : 999){
    case 999: break;
    case FSL_CONFDB_VERSIONABLE:{
      char * const val = fsl_config_get_text(f, mode, key, NULL);
      if(val){
        rv = fsl_str_bool(val);
        fsl_free(val);
      }
      break;
    }
    default:{
      int rc;
      fsl_stmt * st = NULL;
      char const * const table = fsl_config_table_for_role(mode);
      fsl_db * db;
      db = fsl_config_for_role(f, mode);
      if(!db) break;
      assert(table);
      rc = fsl_db_prepare_cached(db, &st, SELECT_FROM_CONFIG,
                                 table, __FILE__);
      if(!rc){
        st->role = fsl__confdb_to_role(mode);
        fsl_stmt_bind_text(st, 1, key, -1, 0);
        if(FSL_RC_STEP_ROW==fsl_stmt_step(st)){
          char const * const col = fsl_stmt_g_text(st, 0, NULL);
          rv = col ? fsl_str_bool(col) : dflt /* 0? */;
        }
        fsl_stmt_cached_yield(st);
      }
      break;
    }
  }
  return rv;
}

/**
    Sets up a REPLACE statement for the given config db and key. On
    success 0 is returned and *st holds the cached statement. The caller
    must bind() parameter #2 and step() the statement, then
    fsl_stmt_cached_yield() it.
   
    Returns non-0 on error.
*/
static int fsl_config_set_prepare( fsl_cx * const f, fsl_stmt **st,
                                   fsl_confdb_e mode, char const * key ){
  char const * table = fsl_config_table_for_role(mode);
  fsl_db * const db = fsl_config_for_role(f,mode);
  assert(table);
  if(!db || !key) return FSL_RC_MISUSE;
  else if(!*key) return FSL_RC_RANGE;
  else{
    const char * sql = FSL_CONFDB_REPO==mode
      ? "REPLACE INTO %!Q(name,value,mtime) VALUES(?,?,now())/*%s()*/"
      : "REPLACE INTO %!Q(name,value) VALUES(?,?)/*%s()*/";
    int rc = fsl_db_prepare_cached(db, st, sql,  table, __func__);
    if(!rc){
      (*st)->role = fsl__confdb_to_role(mode);
      rc = fsl_stmt_bind_text(*st, 1, key, -1, 1);
    }
    if(rc && !f->error.code){
      fsl_cx_uplift_db_error(f, db);
    }
    return rc;
  }
}

/*
  TODO/FIXME: the fsl_config_set_xxx() routines all use the same basic
  structure, differing only in the concrete bind() op they call. They
  should be consolidated somehow.
*/

/**
   Writes valLen bytes of val to a versioned-setting file. Returns 0
   on success. Requires a checkout db.
*/
static int fsl_config_set_versionable( fsl_cx * const f, char const * key,
                                       char const * val,
                                       fsl_size_t valLen){
  assert(key && *key);
  if(!fsl_needs_ckout(f)){
    return FSL_RC_NOT_A_CKOUT;
  }
  fsl_buffer * fName = fsl__cx_scratchpad(f);
  int rc = fsl_config_versionable_filename(f, key, fName);
  if(!rc){
    fsl_buffer fake = fsl_buffer_empty;
    fake.mem = (void*)val;
    fake.capacity = fake.used = valLen;
    rc = fsl_buffer_to_filename(&fake, fsl_buffer_cstr(fName));
  }
  fsl__cx_scratchpad_yield(f, fName);
  return rc;
}

  
int fsl_config_set_text( fsl_cx * const f, fsl_confdb_e mode,
                         char const * key, char const * val ){
  if(!key) return FSL_RC_MISUSE;
  else if(!*key) return FSL_RC_RANGE;
  else if(FSL_CONFDB_VERSIONABLE==mode){
    return fsl_config_set_versionable(f, key, val,
                                      val ? fsl_strlen(val) : 0);
  }
  fsl_db * db = fsl_config_for_role(f,mode);
  if(!db) return FSL_RC_MISUSE;
  fsl_stmt * st = NULL;
  int rc = fsl_config_set_prepare(f, &st, mode, key);
  //MARKER(("config-set mode=%d k=%s v=%s\n",
  //        mode, key, val));
  if(!rc){
    if(val){
      rc = fsl_stmt_bind_text(st, 2, val, -1, 0);
    }else{
      rc = fsl_stmt_bind_null(st, 2);
    }
    if(!rc){
      rc = fsl_stmt_step(st);
    }
    fsl_stmt_cached_yield(st);
    if(FSL_RC_STEP_DONE==rc) rc = 0;
  }
  if(rc && !f->error.code) fsl_cx_uplift_db_error(f, db);
  return rc;
}

int fsl_config_set_blob( fsl_cx * const f, fsl_confdb_e mode, char const * key,
                         void const * val, fsl_int_t len ){
  if(!key) return FSL_RC_MISUSE;
  else if(!*key) return FSL_RC_RANGE;
  else if(FSL_CONFDB_VERSIONABLE==mode){
    return fsl_config_set_versionable(f, key, val,
                                      (val && len<0)
                                      ? fsl_strlen((char const *)val)
                                      : (fsl_size_t)len);
  }
  fsl_db * db = fsl_config_for_role(f,mode);
  if(!db) return FSL_RC_MISUSE;
  fsl_stmt * st = NULL;
  int rc = fsl_config_set_prepare(f, &st, mode, key);
  if(!rc){
    if(val){
      if(len<0) len = fsl_strlen((char const *)val);
      rc = fsl_stmt_bind_blob(st, 2, val, len, 0);
    }else{
      rc = fsl_stmt_bind_null(st, 2);
    }
    if(!rc){
      rc = fsl_stmt_step(st);
    }
    fsl_stmt_cached_yield(st);
    if(FSL_RC_STEP_DONE==rc) rc = 0;
  }
  if(rc && !f->error.code) fsl_cx_uplift_db_error(f, db);
  return rc;
}

int fsl_config_set_int32( fsl_cx * const f, fsl_confdb_e mode,
                          char const * key, int32_t val ){
  if(!key) return FSL_RC_MISUSE;
  else if(!*key) return FSL_RC_RANGE;
  else if(FSL_CONFDB_VERSIONABLE==mode){
    char buf[64] = {0};
    fsl_snprintf(buf, sizeof(buf), "%" PRIi32 "\n", val);
    return fsl_config_set_versionable(f, key, buf,
                                      fsl_strlen(buf));
  }
  fsl_db * db = fsl_config_for_role(f,mode);
  if(!db) return FSL_RC_MISUSE;
  fsl_stmt * st = NULL;
  int rc = fsl_config_set_prepare(f, &st, mode, key);
  if(!rc){
    rc = fsl_stmt_bind_int32(st, 2, val);
    if(!rc){
      rc = fsl_stmt_step(st);
    }
    fsl_stmt_cached_yield(st);
    if(FSL_RC_STEP_DONE==rc) rc = 0;
  }
  if(rc && !f->error.code) fsl_cx_uplift_db_error(f, db);
  return rc;
}

int fsl_config_set_int64( fsl_cx * const f, fsl_confdb_e mode,
                          char const * key, int64_t val ){
  if(!key) return FSL_RC_MISUSE;
  else if(!*key) return FSL_RC_RANGE;
  else if(FSL_CONFDB_VERSIONABLE==mode){
    char buf[64] = {0};
    fsl_snprintf(buf, sizeof(buf), "%" PRIi64 "\n", val);
    return fsl_config_set_versionable(f, key, buf,
                                      fsl_strlen(buf));
  }
  fsl_db * db = fsl_config_for_role(f,mode);
  if(!db) return FSL_RC_MISUSE;
  fsl_stmt * st = NULL;
  int rc = fsl_config_set_prepare(f, &st, mode, key);
  if(!rc){
    rc = fsl_stmt_bind_int64(st, 2, val);
    if(!rc){
      rc = fsl_stmt_step(st);
    }
    fsl_stmt_cached_yield(st);
    if(FSL_RC_STEP_DONE==rc) rc = 0;
  }
  if(rc && !f->error.code) fsl_cx_uplift_db_error(f, db);
  return rc;
}

int fsl_config_set_id( fsl_cx * const f, fsl_confdb_e mode,
                          char const * key, fsl_id_t val ){
  if(!key) return FSL_RC_MISUSE;
  else if(!*key) return FSL_RC_RANGE;
  else if(FSL_CONFDB_VERSIONABLE==mode){
    char buf[64] = {0};
    fsl_snprintf(buf, sizeof(buf), "%" FSL_ID_T_PFMT "\n", val);
    return fsl_config_set_versionable(f, key, buf,
                                      fsl_strlen(buf));
  }
  fsl_db * db = fsl_config_for_role(f,mode);
  if(!db) return FSL_RC_MISUSE;
  fsl_stmt * st = NULL;
  int rc = fsl_config_set_prepare(f, &st, mode, key);
  if(!rc){
    rc = fsl_stmt_bind_id(st, 2, val);
    if(!rc){
      rc = fsl_stmt_step(st);
    }
    fsl_stmt_cached_yield(st);
    if(FSL_RC_STEP_DONE==rc) rc = 0;
  }
  if(rc && !f->error.code) fsl_cx_uplift_db_error(f, db);
  return rc;
}

int fsl_config_set_double( fsl_cx * const f, fsl_confdb_e mode,
                           char const * key, double val ){
  if(!key) return FSL_RC_MISUSE;
  else if(!*key) return FSL_RC_RANGE;
  else if(FSL_CONFDB_VERSIONABLE==mode){
    char buf[128] = {0};
    fsl_snprintf(buf, sizeof(buf), "%f\n", val);
    return fsl_config_set_versionable(f, key, buf,
                                      fsl_strlen(buf));
  }
  fsl_db * db = fsl_config_for_role(f,mode);
  if(!db) return FSL_RC_MISUSE;
  fsl_stmt * st = NULL;
  int rc = fsl_config_set_prepare(f, &st, mode, key);
  if(!rc){
    rc = fsl_stmt_bind_double(st, 2, val);
    if(!rc){
        rc = fsl_stmt_step(st);
    }
    fsl_stmt_cached_yield(st);
    if(FSL_RC_STEP_DONE==rc) rc = 0;
  }
  if(rc && !f->error.code) fsl_cx_uplift_db_error(f, db);
  return rc;
}

int fsl_config_set_bool( fsl_cx * const f, fsl_confdb_e mode,
                         char const * key, bool val ){
  if(!key) return FSL_RC_MISUSE;
  else if(!*key) return FSL_RC_RANGE;
  char buf[4] = {'o','n','\n','\n'};
  if(!val){
    buf[1] = buf[2] = 'f';
  }
  if(FSL_CONFDB_VERSIONABLE==mode){
    return fsl_config_set_versionable(f, key, buf,
                                      val ? 3 : 4);
  }
  fsl_db * db = fsl_config_for_role(f,mode);
  if(!db) return FSL_RC_MISUSE;
  fsl_stmt * st = NULL;
  int rc = fsl_config_set_prepare(f, &st, mode, key);
  if(!rc){
    rc = fsl_stmt_bind_text(st, 2, buf, val ? 2 : 3, false);
    if(!rc){
      rc = fsl_stmt_step(st);
    }
    fsl_stmt_cached_yield(st);
    if(FSL_RC_STEP_DONE==rc) rc = 0;
  }
  if(rc && !f->error.code) fsl_cx_uplift_db_error(f, db);
  return rc;
}

int fsl_config_transaction_begin(fsl_cx * const f, fsl_confdb_e mode){
  fsl_db * db = fsl_config_for_role(f,mode);
  if(!db) return FSL_RC_MISUSE;
  else{
    int const rc = fsl_db_transaction_begin(db);
    if(rc) fsl_cx_uplift_db_error(f, db);
    return rc;
  }
}

int fsl_config_transaction_end(fsl_cx * const f, fsl_confdb_e mode, bool rollback){
  fsl_db * db = fsl_config_for_role(f,mode);
  if(!db) return FSL_RC_MISUSE;
  else{
    int const rc = fsl_db_transaction_end(db, rollback);
    if(rc) fsl_cx_uplift_db_error(f, db);
    return rc;
  }
}

int fsl_config_globs_load(fsl_cx * const f, fsl_list * const li, char const * key){
  int rc = 0;
  char * val = NULL;
  if(!f || !li || !key || !*key) return FSL_RC_MISUSE;
  else if(fsl_cx_db_ckout(f)){
    /* Try versionable settings... */
    fsl_buffer buf = fsl_buffer_empty;
    rc = fsl_config_get_buffer(f, FSL_CONFDB_VERSIONABLE, key, &buf);
    if(rc){
      switch(rc){
        case FSL_RC_NOT_FOUND:
          fsl_cx_err_reset(f);
          rc = 0;
          break;
        default:
          fsl_buffer_clear(&buf);
          return rc;
      }
      /* Fall through and try the next option. */
    }else{
      if(buf.mem){
        val = fsl_buffer_take(&buf);
      }else{
        /* Empty but existing list, so it trumps the
           repo/global settings. */;
        fsl_buffer_clear(&buf);
      }
      goto gotone;
    }
  }
  if(fsl_cx_db_repo(f)){
    /* See if the repo can serve us... */
    val = fsl_config_get_text(f, FSL_CONFDB_REPO, key, NULL);
    if(val) goto gotone;
    /* Else fall through and try global config... */
  }
  if(fsl_cx_db_config(f)){
    /*FIXME?: we arguably should open the global config for this if it
      is not already opened.*/
    val = fsl_config_get_text(f, FSL_CONFDB_GLOBAL, key, NULL);
    if(val) goto gotone;
  }
  gotone:
  if(val){
      rc = fsl_glob_list_parse( li, val );
      fsl_free(val);
      val = 0;
      return rc;
  }
  return rc;
}

/*
  TODO???: the infrastructure from fossil's configure.c and db.c which
  deals with the config db and the list of known/allowed settings.

  ==================

static const struct {
  const char *zName;   / * Name of the configuration set * /
  int groupMask;       / * Mask for that configuration set * /
  const char *zHelp;   / * What it does * /

} fslConfigGroups[] = {
  { "/email",        FSL_CONFIGSET_ADDR,  "Concealed email addresses in tickets" },
  { "/project",      FSL_CONFIGSET_PROJ,  "Project name and description"         },
  { "/skin",         FSL_CONFIGSET_SKIN | FSL_CONFIGSET_CSS,
                                      "Web interface appearance settings"    },
  { "/css",          FSL_CONFIGSET_CSS,   "Style sheet"                          },
  { "/shun",         FSL_CONFIGSET_SHUN,  "List of shunned artifacts"            },
  { "/ticket",       FSL_CONFIGSET_TKT,   "Ticket setup",                        },
  { "/user",         FSL_CONFIGSET_USER,  "Users and privilege settings"         },
  { "/xfer",         FSL_CONFIGSET_XFER,  "Transfer setup",                      },
  { "/all",          FSL_CONFIGSET_ALL,   "All of the above"                     },
  {NULL, 0, NULL}
};
 */

/*
   The following is a list of settings that we are willing to
   transfer.
  
   Setting names that begin with an alphabetic characters refer to
   single entries in the CONFIG table.  Setting names that begin with
   "@" are for special processing.
*/
static struct {
  const char *zName;   /* Name of the configuration parameter */
  int groupMask;       /* Which config groups is it part of */
} fslConfigXfer[] = {
  { "css",                    FSL_CONFIGSET_CSS  },

  { "header",                 FSL_CONFIGSET_SKIN },
  { "footer",                 FSL_CONFIGSET_SKIN },
  { "logo-mimetype",          FSL_CONFIGSET_SKIN },
  { "logo-image",             FSL_CONFIGSET_SKIN },
  { "background-mimetype",    FSL_CONFIGSET_SKIN },
  { "background-image",       FSL_CONFIGSET_SKIN },
  { "index-page",             FSL_CONFIGSET_SKIN },
  { "timeline-block-markup",  FSL_CONFIGSET_SKIN },
  { "timeline-max-comment",   FSL_CONFIGSET_SKIN },
  { "timeline-plaintext",     FSL_CONFIGSET_SKIN },
  { "adunit",                 FSL_CONFIGSET_SKIN },
  { "adunit-omit-if-admin",   FSL_CONFIGSET_SKIN },
  { "adunit-omit-if-user",    FSL_CONFIGSET_SKIN },

  { "th1-setup",              FSL_CONFIGSET_ALL },

  { "tcl",                    FSL_CONFIGSET_SKIN|FSL_CONFIGSET_TKT|FSL_CONFIGSET_XFER },
  { "tcl-setup",              FSL_CONFIGSET_SKIN|FSL_CONFIGSET_TKT|FSL_CONFIGSET_XFER },

  { "project-name",           FSL_CONFIGSET_PROJ },
  { "project-description",    FSL_CONFIGSET_PROJ },
  { "manifest",               FSL_CONFIGSET_PROJ },
  { "binary-glob",            FSL_CONFIGSET_PROJ },
  { "clean-glob",             FSL_CONFIGSET_PROJ },
  { "ignore-glob",            FSL_CONFIGSET_PROJ },
  { "keep-glob",              FSL_CONFIGSET_PROJ },
  { "crnl-glob",              FSL_CONFIGSET_PROJ },
  { "encoding-glob",          FSL_CONFIGSET_PROJ },
  { "empty-dirs",             FSL_CONFIGSET_PROJ },
  { "allow-symlinks",         FSL_CONFIGSET_PROJ },

  { "ticket-table",           FSL_CONFIGSET_TKT  },
  { "ticket-common",          FSL_CONFIGSET_TKT  },
  { "ticket-change",          FSL_CONFIGSET_TKT  },
  { "ticket-newpage",         FSL_CONFIGSET_TKT  },
  { "ticket-viewpage",        FSL_CONFIGSET_TKT  },
  { "ticket-editpage",        FSL_CONFIGSET_TKT  },
  { "ticket-reportlist",      FSL_CONFIGSET_TKT  },
  { "ticket-report-template", FSL_CONFIGSET_TKT  },
  { "ticket-key-template",    FSL_CONFIGSET_TKT  },
  { "ticket-title-expr",      FSL_CONFIGSET_TKT  },
  { "ticket-closed-expr",     FSL_CONFIGSET_TKT  },
  { "@reportfmt",             FSL_CONFIGSET_TKT  },

  { "@user",                  FSL_CONFIGSET_USER },

  { "@concealed",             FSL_CONFIGSET_ADDR },

  { "@shun",                  FSL_CONFIGSET_SHUN },

  { "xfer-common-script",     FSL_CONFIGSET_XFER },
  { "xfer-push-script",       FSL_CONFIGSET_XFER },
};

#define ARRAYLEN(X) (sizeof(X)/sizeof(X[0]))
/*
   Return a pointer to a string that contains the RHS of an IN
   operator that will select CONFIG table names that are part of the
   configuration that matches iMatch. The returned string must
   eventually be fsl_free()'d.
*/
char *fsl__config_inop_rhs(int iMask){
  fsl_buffer x = fsl_buffer_empty;
  const char *zSep = "";
  const int n = (int)ARRAYLEN(fslConfigXfer);
  int i;
  int rc = fsl_buffer_append(&x, "(", 1);
  for(i=0; !rc && (i<n); i++){
    if( (fslConfigXfer[i].groupMask & iMask)==0 ) continue;
    if( fslConfigXfer[i].zName[0]=='@' ) continue;
    rc = fsl_buffer_appendf(&x, "%s%Q", zSep, fslConfigXfer[i].zName);
    zSep = ",";
  }
  if(!rc){
    rc = fsl_buffer_append(&x, ")", 1);
  }
  if(rc){
    fsl_buffer_clear(&x);
    assert(!x.mem);
  }else{
    fsl_buffer_resize(&x, x.used);
  }
  return (char *)x.mem;
}


/**
   Holds metadata for fossil-defined configuration settings.

   As of 2021, this library does NOT intend to maintain 100%
   config-entry parity with fossil, as the vast majority of its config
   settings are application-level preferences. The library will
   maintain compatibility with versionable settings and a handful of
   settings which make sense for a library-level API
   (e.g. 'ignore-glob' and 'manifest'). The majority of fossil's
   settings, however, are specific to that application and will not be
   treated specially by the library.

   @see fsl_config_ctrl_get()
   @see fsl_config_has_versionable()
   @see fsl_config_key_is_versionable()
   @see fsl_config_key_default_value()
*/  
struct fsl_config_ctrl {
  /** Name of the setting */
  char const *name;
  /**
     Historical (fossil(1)) internal variable name used by
     db_set(). Not currently used by this impl.
  */
  char const *var;
  /**
     Historical (HTML UI). Width of display.  0 for boolean values.
  */
  int width;
  /**
     Is this setting versionable?
  */
  bool versionable;
  /**
     Default value
  */
  char const *defaultValue;
};
typedef struct fsl_config_ctrl fsl_config_ctrl;

/**
   If key is the name of a fossil-defined config key, this returns
   the fsl_config_ctrl value describing that configuration property,
   else it returns NULL.
*/
FSL_EXPORT fsl_config_ctrl const * fsl_config_ctrl_get(char const * key);

/**
   Returns true if key is the name of a config property
   as defined by fossil(1).
*/
FSL_EXPORT bool fsl_config_key_is_fossil(char const * key);


/**
   Returns true if key is the name of a versionable property, as
   defined by fossil(1). Returning false is NOT a guaranty that fossil
   does NOT define that setting (this library does not track _all_
   settings), but returning true is a a guarantee that it is a
   fossil-defined versionable setting.
*/
FSL_EXPORT bool fsl_config_key_is_versionable(char const * key);

/**
   If key refers to a fossil-defined configuration setting, this
   returns its default value as a NUL-terminated string. Its bytes are
   static and immutable. Returns NULL if key is not a known
   configuration key.
*/
FSL_EXPORT char const * fsl_config_key_default_value(char const * key);

/**
   Returns true if f's current checkout contains the given
   versionable configuration setting, else false.

   @see fsl_config_ctrl
*/
FSL_EXPORT bool fsl_config_has_versionable( fsl_cx * const f, char const * key );

static fsl_config_ctrl const fslConfigCtrl[] = {
/*
  These MUST stay sorted by name and the .defaultValue field MUST have
  a non-NULL value so that some API guarantees can be made.

  FIXME: bring this up to date wrt post-2014 fossil. Or abandon it
  altogether: it has since been decided that this library will not
  attempt to enforce application-level config constraints for the vast
  majority of fossil's config settings. The main benefit for us in
  keeping this is to be able to quickly look up versionable settings,
  as we really need to keep compatibility with fossil for those.
*/
  { "access-log",    0,                0, 0, "off"                 },
  { "allow-symlinks",0,                0, 0/*as of late 2020*/,
                                             "off"                 },
  { "auto-captcha",  "autocaptcha",    0, 0, "on"                  },
  { "auto-hyperlink",0,                0, 0, "on"                  },
  { "auto-shun",     0,                0, 0, "on"                  },
  { "autosync",      0,                0, 0, "on"                  },
  { "binary-glob",   0,               40, 1, ""                    },
  { "clearsign",     0,                0, 0, "off"                 },
#if defined(_WIN32) || defined(__CYGWIN__) || defined(__DARWIN__) || defined(__APPLE__)
  { "case-sensitive",0,                0, 0, "off"                 },
#else
  { "case-sensitive",0,                0, 0, "on"                  },
#endif
  { "clean-glob",    0,               40, 1, ""                    },
  { "crnl-glob",     0,               40, 1, ""                    },
  { "default-perms", 0,               16, 0, "u"                   },
  { "diff-binary",   0,                0, 0, "on"                  },
  { "diff-command",  0,               40, 0, ""                    },
  { "dont-push",     0,                0, 0, "off"                 },
  { "dotfiles",      0,                0, 1, "off"                 },
  { "editor",        0,               32, 0, ""                    },
  { "empty-dirs",    0,               40, 1, ""                    },
  { "encoding-glob",  0,              40, 1, ""                    },
  { "gdiff-command", 0,               40, 0, "gdiff"               },
  { "gmerge-command",0,               40, 0, ""                    },
  { "http-port",     0,               16, 0, "8080"                },
  { "https-login",   0,                0, 0, "off"                 },
  { "ignore-glob",   0,               40, 1, ""                    },
  { "keep-glob",     0,               40, 1, ""                    },
  { "localauth",     0,                0, 0, "off"                 },
  { "main-branch",   0,               40, 0, "trunk"               },
  { "manifest",      0,                0, 1, "off"                 },
  { "max-upload",    0,               25, 0, "250000"              },
  { "mtime-changes", 0,                0, 0, "on"                  },
  { "pgp-command",   0,               40, 0, "gpg --clearsign -o " },
  { "proxy",         0,               32, 0, "off"                 },
  { "relative-paths",0,                0, 0, "on"                  },
  { "repo-cksum",    0,                0, 0, "on"                  },
  { "self-register", 0,                0, 0, "off"                 },
  { "ssh-command",   0,               40, 0, ""                    },
  { "ssl-ca-location",0,              40, 0, ""                    },
  { "ssl-identity",  0,               40, 0, ""                    },
  { "tcl",           0,                0, 0, "off"                 },
  { "tcl-setup",     0,               40, 0, ""                    },
  { "th1-setup",     0,               40, 0, ""                    },
  { "web-browser",   0,               32, 0, ""                    },
  { "white-foreground", 0,             0, 0, "off"                 },
  { 0,0,0,0,0 }
};

char *fsl_db_setting_inop_rhs(){
  fsl_buffer x = fsl_buffer_empty;
  const char *zSep = "";
  fsl_config_ctrl const * ct = &fslConfigCtrl[0];
  int rc = fsl_buffer_append(&x, "(", 1);
  for( ; !rc && ct && ct->name; ++ct){
    rc = fsl_buffer_appendf(&x, "%s%Q", zSep, ct->name);
    zSep = ",";
  }
  if(!rc){
    rc = fsl_buffer_append(&x, ")", 1);
  }
  if(rc){
    fsl_buffer_clear(&x);
    assert(!x.mem);
  }else{
    fsl_buffer_resize(&x, x.used);
  }
  return (char *)x.mem;
}

static int fsl_config_ctrl_cmp(const void *lhs, const void *rhs){
  fsl_config_ctrl const * l = (fsl_config_ctrl const *)lhs;
  fsl_config_ctrl const * r = (fsl_config_ctrl const *)rhs;
  if(!l) return r ? -1 : 0;
  else if(!r) return 1;
  else return fsl_strcmp(l->name, r->name);
}

fsl_config_ctrl const * fsl_config_ctrl_get(char const * key){
  fsl_config_ctrl const * fcc;
  fsl_config_ctrl bogo = {0,0,0,0,0};
  bogo.name = key;
  fcc = (fsl_config_ctrl const *)
    bsearch( &bogo, fslConfigCtrl,
             ARRAYLEN(fslConfigCtrl) -1 /* for empty tail entry */,
             sizeof(fsl_config_ctrl),
             fsl_config_ctrl_cmp );
  return (fcc && fcc->name) ? fcc : NULL;
}

bool fsl_config_key_is_fossil(char const * key){
  fsl_config_ctrl const * fcc = fsl_config_ctrl_get(key);
  return (fcc && fcc->name) ? 1 : 0;
}

bool fsl_config_key_is_versionable(char const * key){
  fsl_config_ctrl const * fcc = fsl_config_ctrl_get(key);
  return (fcc && fcc->versionable) ? 1 : 0;
}

char const * fsl_config_key_default_value(char const * key){
  fsl_config_ctrl const * fcc = fsl_config_ctrl_get(key);
  return (fcc && fcc->name) ? fcc->defaultValue : NULL;
}

bool fsl_config_has_versionable( fsl_cx * const f, char const * key ){
  if(!f || !key || !*key || !f->ckout.dir) return 0;
  else if(!fsl_config_key_is_fossil(key)) return 0;
  else{
    fsl_buffer * fn = fsl__cx_scratchpad(f);
    int rc = fsl_config_versionable_filename(f, key, fn);
    if(!rc) rc = fsl_stat(fsl_buffer_cstr(fn), NULL, 0);
    fsl__cx_scratchpad_yield(f, fn);
    return 0==rc;
  }
}

static fsl_confdb_e fsl__char_to_confdb(char ch){
  fsl_confdb_e rc = FSL_CONFDB_NONE;
  switch(ch){
    case 'c': rc = FSL_CONFDB_CKOUT; break;
    case 'r': rc = FSL_CONFDB_REPO; break;
    case 'g': rc = FSL_CONFDB_GLOBAL; break;
    case 'v': rc = FSL_CONFDB_VERSIONABLE; break;
    default: break;
  }
  return rc;
}

#define fsl__configs_get_v(CONV) {                                      \
  char * val = fsl_config_get_text(f, FSL_CONFDB_VERSIONABLE, key, NULL); \
  fsl_cx_err_reset(f); \
  if(val){ rv = CONV; fsl_free(val); goto end; } \
  break; }

#define fsl__configs_get_x1 \
  fsl_db * const db = fsl_config_for_role(f, mode); \
  char const * table = fsl_config_table_for_role(mode); \
  assert(table); \
  if(db){ \
    fsl_stmt * st = NULL; \
    fsl_db_prepare_cached(db, &st, SELECT_FROM_CONFIG, table, __FILE__); \
    if(st){                                                             \
      fsl_stmt_bind_text(st, 1, key, -1, 0); \
      if(FSL_RC_STEP_ROW==fsl_stmt_step(st)){ (void)0
#define fsl__configs_get_x2 \
        fsl_stmt_cached_yield(st);            \
        goto end;                             \
      }                                       \
      fsl_stmt_cached_yield(st);              \
    }                                         \
  } \
  break

int32_t fsl_configs_get_int32(fsl_cx * const f, char const * zCfg, int32_t dflt, char const * key){
  int32_t rv = dflt;
  for( char const * z = zCfg; *z; ++z ){
    fsl_confdb_e const mode = fsl__char_to_confdb(*z);
    switch(mode){
      case FSL_CONFDB_VERSIONABLE: fsl__configs_get_v((int32_t)atoi(val));
      case FSL_CONFDB_CKOUT:
      case FSL_CONFDB_REPO:
      case FSL_CONFDB_GLOBAL: {
        fsl__configs_get_x1;
        rv = fsl_stmt_g_int32(st, 0);
        fsl__configs_get_x2;
      }
      default: continue;
    }
  }
  end:
  return rv;  
}

int64_t fsl_configs_get_int64(fsl_cx * const f, char const * zCfg, int64_t dflt, char const * key){
  int64_t rv = dflt;
  for( char const * z = zCfg; *z; ++z ){
    fsl_confdb_e const mode = fsl__char_to_confdb(*z);
    switch(mode){
      case FSL_CONFDB_VERSIONABLE: fsl__configs_get_v((int64_t)strtoll(val, NULL, 10));
      case FSL_CONFDB_CKOUT:
      case FSL_CONFDB_REPO:
      case FSL_CONFDB_GLOBAL: {
        fsl__configs_get_x1;
        rv = fsl_stmt_g_int64(st, 0);
        fsl__configs_get_x2;
      }
      default: continue;
    }
  }
  end:
  return rv;  
}

fsl_id_t fsl_configs_get_id(fsl_cx * const f, char const * zCfg, fsl_id_t dflt, char const * key){
  return (sizeof(fsl_id_t)==sizeof(int32_t))
    ? (fsl_id_t)fsl_configs_get_int32(f, zCfg, dflt, key)
    : (fsl_id_t)fsl_configs_get_int64(f, zCfg, dflt, key);
}

bool fsl_configs_get_bool(fsl_cx * const f, char const * zCfg, bool dflt, char const * key){
  bool rv = dflt;
  for( char const * z = zCfg; *z; ++z ){
    fsl_confdb_e const mode = fsl__char_to_confdb(*z);
    switch(mode){
      case FSL_CONFDB_VERSIONABLE: fsl__configs_get_v(fsl_str_bool(val));
      case FSL_CONFDB_CKOUT:
      case FSL_CONFDB_REPO:
      case FSL_CONFDB_GLOBAL: {
        fsl__configs_get_x1;
        char const * col = fsl_stmt_g_text(st, 0, NULL);
        rv = col ? fsl_str_bool(col) : dflt;
        fsl__configs_get_x2;
      }
      default: continue;
    }
  }
  end:
  return rv;  
}

double fsl_configs_get_double(fsl_cx * const f, char const * zCfg, double dflt, char const * key){
  double rv = dflt;
  for( char const * z = zCfg; *z; ++z ){
    fsl_confdb_e const mode = fsl__char_to_confdb(*z);
    switch(mode){
      case FSL_CONFDB_VERSIONABLE: fsl__configs_get_v(strtod(val,NULL));
      case FSL_CONFDB_CKOUT:
      case FSL_CONFDB_REPO:
      case FSL_CONFDB_GLOBAL: {
        fsl__configs_get_x1;
        rv = fsl_stmt_g_double(st, 0);
        fsl__configs_get_x2;
      }
      default: continue;
    }
  }
  end:
  return rv;  
}

char * fsl_configs_get_text(fsl_cx * const f, char const * zCfg, char const * key,
                            fsl_size_t * len){
  char * rv = NULL;
  fsl_buffer val = fsl_buffer_empty;
  if(fsl_configs_get_buffer(f, zCfg, key, &val)){
    fsl_cx_err_reset(f);
    if(len) *len = 0;
    fsl_buffer_clear(&val)/*in case of partial read failure*/;
  }else{
    if(len) *len = val.used;
    rv = fsl_buffer_take(&val);
  }
  return rv;
}

int fsl_configs_get_buffer(fsl_cx * const f, char const * zCfg, char const * key,
                           fsl_buffer * const b){
  int rc = FSL_RC_NOT_FOUND;
  fsl_buffer_reuse(b);
  for( char const * z = zCfg;
       (rc && FSL_RC_OOM!=rc) && *z; ++z ){
    fsl_confdb_e const mode = fsl__char_to_confdb(*z);
    switch(mode){
      case FSL_CONFDB_VERSIONABLE:
        rc = fsl_config_get_buffer(f, mode, key, b);
        if(rc){
          if(FSL_RC_OOM!=rc) rc = FSL_RC_NOT_FOUND;
          fsl_cx_err_reset(f);
        }
        break;
      case FSL_CONFDB_CKOUT:
      case FSL_CONFDB_REPO:
      case FSL_CONFDB_GLOBAL: {
        fsl__configs_get_x1;
        fsl_size_t len = 0;
        char const * s = fsl_stmt_g_text(st, 0, &len);
        rc = s ? fsl_buffer_append(b, s, len) : 0;
        fsl__configs_get_x2;
      }
      default: break;
    }
  }
  end:
  return rc; 
}


#undef fsl__configs_get_v
#undef fsl__configs_get_x1
#undef fsl__configs_get_x2
#undef SELECT_FROM_CONFIG
#undef MARKER
#undef ARRAYLEN
/* end of file ./src/config.c */
/* start of file ./src/cx.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/
/*************************************************************************
  This file houses most of the context-related APIs.
*/
#if !defined(FSL_ENABLE_SQLITE_REGEXP)
#  define FSL_ENABLE_SQLITE_REGEXP 0
#endif
#if FSL_ENABLE_SQLITE_REGEXP
#endif
#include <assert.h>

#if defined(_WIN32)
# include <windows.h>
# define F_OK 0
# define W_OK 2
#else
# include <unistd.h> /* F_OK */
#endif

#include <stdlib.h>
#include <string.h>
#include <stdio.h> /* FILE class */
#include <errno.h>

/* Only for debugging */
#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)

/** Number of fsl_cx::scratchpads buffers. */
#define FSL_CX_NSCRATCH \
  ((int)(sizeof(fsl_cx_empty.scratchpads.buf) \
         /sizeof(fsl_cx_empty.scratchpads.buf[0])))
const int StaticAssert_scratchpadsCounts[
     (FSL_CX_NSCRATCH==
      ((int)(sizeof(fsl_cx_empty.scratchpads.used)
             /sizeof(fsl_cx_empty.scratchpads.used[0]))))
     ? 1 : -1
];

#if FSL_ENABLE_SQLITE_REGEXP
/**
   Used for setup and teardown of sqlite3_auto_extension().
*/
static volatile long sg_autoregctr = 0;
#endif


/**
   Clears (most) dynamic state in f, but does not free f and does
   not free "static" state (that set up by the init process). If
   closeDatabases is true then any databases managed by f are
   closed, else they are kept open.

   Client code will not normally need this - it is intended for a
   particular potential memory optimization case. If (and only if)
   closeDatabases is true then after calling this, f may be legally
   re-used as a target for fsl_cx_init().

   This function does not trigger any finializers set for f's client
   state or output channel.

   Results are undefined if !f or f's memory has not been properly
   initialized.
*/
static void fsl__cx_reset( fsl_cx * const f, bool closeDatabases );

int fsl_cx_init( fsl_cx ** tgt, fsl_cx_init_opt const * param ){
  static fsl_cx_init_opt paramDefaults = fsl_cx_init_opt_default_m;
  int rc = 0;
  fsl_cx * f;
  extern int fsl__cx_install_timeline_crosslinkers(fsl_cx * const f)
    /*in deck.c*/;
  if(!tgt) return FSL_RC_MISUSE;
  else if(!param){
    if(!paramDefaults.output.state){
      paramDefaults.output.state = stdout;
    }
    param = &paramDefaults;
  }
  if(*tgt){
    void const * allocStamp = (*tgt)->allocStamp;
    fsl__cx_reset(*tgt, true) /* just to be safe */;
    f = *tgt;
    *f = fsl_cx_empty;
    f->allocStamp = allocStamp;
  }else{
    f = fsl_cx_malloc();
    if(!f) return FSL_RC_OOM;

    *tgt = f;
  }
  memset(&f->cache.mcache, 0, sizeof(f->cache.mcache));
  f->output = param->output;
  f->cxConfig = param->config;

  enum {
    /* Because testing shows a lot of re-allocs via some of the
       lower-level stat()-related bits, we pre-allocate this many
       bytes into f->scratchpads.buf[].  Curiously, there is almost no
       difference in (re)allocation behaviour until this size goes
       above about 200.

       We ignore allocation errors here, as they're not critical (but
       upcoming ops will fail when _they_ run out of memory).
    */
    InitialScratchCapacity = 256
  };
  assert(FSL_CX_NSCRATCH
         == (sizeof(f->scratchpads.used)/sizeof(f->scratchpads.used[0])));
  for(int i = 0; i < FSL_CX_NSCRATCH; ++i){
    f->scratchpads.buf[i] = fsl_buffer_empty;
    f->scratchpads.used[i] = false;
    fsl_buffer_reserve(&f->scratchpads.buf[i], InitialScratchCapacity);
  }
  /* We update f->error.msg often, so go ahead and pre-allocate that, too,
     also ignoring any OOM error at this point. */
  fsl_buffer_reserve(&f->error.msg, InitialScratchCapacity);

#if defined(SQLITE_THREADSAFE) && SQLITE_THREADSAFE>0
  sqlite3_initialize(); /*the SQLITE_MUTEX_STATIC_MASTER will not cause autoinit of sqlite for some reason*/
  sqlite3_mutex_enter(sqlite3_mutex_alloc(SQLITE_MUTEX_STATIC_MASTER));
#endif
#if FSL_ENABLE_SQLITE_REGEXP
  if ( 1 == ++sg_autoregctr ){
    /*register our statically linked extensions to be auto-init'ed at the appropriate time*/
    sqlite3_auto_extension((void(*)(void))(sqlite3_regexp_init));     /*sqlite regexp extension*/
    atexit(sqlite3_reset_auto_extension)
      /* Clean up pseudo-leak valgrind complains about:
         https://www.sqlite.org/c3ref/auto_extension.html */;
  }
#endif
#if defined(SQLITE_THREADSAFE) && SQLITE_THREADSAFE>0
  sqlite3_mutex_leave(sqlite3_mutex_alloc(SQLITE_MUTEX_STATIC_MASTER));
#endif
  if(!rc) rc = fsl__cx_install_timeline_crosslinkers(f);
  if(!rc){
    f->cache.tempDirs = fsl_temp_dirs_get();
    if(!f->cache.tempDirs) rc = FSL_RC_OOM;
  }
  return rc;
}

void fsl__cx_mcache_clear(fsl_cx * const f){
  const unsigned cacheLen =
    (unsigned)(sizeof(fsl__mcache_empty.aAge)
               /sizeof(fsl__mcache_empty.aAge[0]));
  for(unsigned i = 0; i < cacheLen; ++i){
    fsl_deck_finalize(&f->cache.mcache.decks[i]);
  }
  f->cache.mcache = fsl__mcache_empty;
}

void fsl_cx_caches_reset(fsl_cx * const f){
  fsl__bccache_reset(&f->cache.blobContent);
  fsl__cx_mcache_clear(f);
  fsl__cx_clear_mf_seen(f, false);
  f->cache.allowSymlinks =
    f->cache.caseInsensitive =
    f->cache.seenDeltaManifest =
    f->cache.manifestSetting = -1;
  if(fsl_cx_db_ckout(f)){
    fsl__ckout_version_fetch(f)
      /* FIXME: this "really should" be fsl__cx_ckout_clear(), but that data is not fetched
         on demand (maybe it should be?). */;
  }else{
    fsl__cx_ckout_clear(f);
  }
}

static void fsl__cx_finalize_cached_stmt(fsl_cx * const f){
#define STMT(X) fsl_stmt_finalize(&f->cache.stmt.X)
  STMT(deltaSrcId);
  STMT(uuidToRid);
  STMT(uuidToRidGlob);
  STMT(contentSize);
  STMT(contentBlob);
  STMT(nextEntry);
#undef STMT
}

static void fsl__cx_reset(fsl_cx * const f, bool closeDatabases){
  fsl_checkin_discard(f);
  fsl__cx_finalize_cached_stmt(f);
#define SFREE(X) fsl_free(X); X = NULL
  if(closeDatabases){
    fsl_cx_close_dbs(f);
    /*
      Reminder: f->dbMem is NOT closed here: it's an internal detail,
      not public state. We could arguably close and reopen it here,
      but then we introduce a potenital error case (OOM) where we
      currently have none (thus the void return).

      2021-11-09: it turns out we've had an error case all along
      here: if any cached statements are opened for one of the dbs,
      that can prohibit its detachment.
    */
    SFREE(f->ckout.dir);
    f->ckout.dirLen = 0;
    /* assert(NULL==f->dbMain); */
  }
  SFREE(f->repo.user);
  fsl__cx_ckout_clear(f);
  SFREE(f->cache.projectCode);
  SFREE(f->ticket.titleColumn);
  SFREE(f->ticket.statusColumn);
#undef SFREE
  fsl_error_clear(&f->error);
  f->interrupted = 0;
  fsl__card_J_list_free(&f->ticket.customFields, true);
  fsl_buffer_clear(&f->cache.fileContent);
  fsl_buffer_clear(&f->cache.deltaContent);
  for(int i = 0; i < FSL_CX_NSCRATCH; ++i){
    fsl_buffer_clear(&f->scratchpads.buf[i]);
    f->scratchpads.used[i] = false;
  }
  fsl_cx_caches_reset(f);
  fsl__bccache_clear(&f->cache.blobContent);
  fsl__cx_clear_mf_seen(f, true);
  fsl_id_bag_clear(&f->cache.leafCheck);
  fsl_id_bag_clear(&f->cache.toVerify);
  assert(NULL==f->cache.mfSeen.list);
  if(f->xlinkers.list){
    fsl_free(f->xlinkers.list);
    f->xlinkers = fsl_xlinker_list_empty;
  }
#define SLIST(L) fsl_list_visit_free(L, 1)
#define GLOBL(X) SLIST(&f->cache.globs.X)
  GLOBL(ignore);
  GLOBL(binary);
  GLOBL(crnl);
#undef GLOBL
#undef SLIST
  f->cache = fsl_cx_empty.cache;
}

void fsl__cx_clear_mf_seen(fsl_cx * const f, bool freeMemory){
  if(freeMemory) fsl_id_bag_clear(&f->cache.mfSeen);
  else fsl_id_bag_reset(&f->cache.mfSeen);
}

void fsl_cx_finalize( fsl_cx * const f ){
  void const * const allocStamp = f ? f->allocStamp : NULL;
  if(!f) return;
  if(f->clientState.finalize.f){
    f->clientState.finalize.f( f->clientState.finalize.state,
                               f->clientState.state );
  }
  f->clientState = fsl_state_empty;
  f->output = fsl_outputer_empty;
  fsl_temp_dirs_free(f->cache.tempDirs);
  fsl__cx_reset(f, true);
  *f = fsl_cx_empty;
  if(&fsl_cx_empty == allocStamp){
    fsl_free(f);
  }else{
    f->allocStamp = allocStamp;
  }

  /* clean up the auto extension; not strictly necessary, but pleases debug malloc's */
#if defined(SQLITE_THREADSAFE) && SQLITE_THREADSAFE>0
  sqlite3_mutex_enter(sqlite3_mutex_alloc(SQLITE_MUTEX_STATIC_MASTER));
#endif
#if FSL_ENABLE_SQLITE_REGEXP
  if ( 0 == --sg_autoregctr ){
    /*register our statically linked extensions to be auto-init'ed at the appropriate time*/
    sqlite3_cancel_auto_extension((void(*)(void))(sqlite3_regexp_init));     /*sqlite regexp extension*/
  }
  assert(sg_autoregctr>=0);
#endif
#if defined(SQLITE_THREADSAFE) && SQLITE_THREADSAFE>0
  sqlite3_mutex_leave(sqlite3_mutex_alloc(SQLITE_MUTEX_STATIC_MASTER));
#endif
}

void fsl_cx_err_reset(fsl_cx * const f){
  //f->interrupted = 0; // No! ONLY modify this via fsl_cx_interrupt()
  fsl_error_reset(&f->error);
  fsl_db_err_reset(&f->repo.db);
  fsl_db_err_reset(&f->config.db);
  fsl_db_err_reset(&f->ckout.db);
}

int fsl_cx_err_set_e( fsl_cx * const f, fsl_error * const err ){
  if(!err){
    return fsl_cx_err_set(f, 0, NULL);
  }else{
    fsl_error_move(err, &f->error);
    fsl_error_clear(err);
    return f->error.code;
  }
}

int fsl_cx_err_setv( fsl_cx * const f, int code, char const * fmt,
                     va_list args ){
  return fsl_error_setv( &f->error, code, fmt, args );
}

int fsl_cx_err_set( fsl_cx * const f, int code, char const * fmt,
                    ... ){
  int rc;
  va_list args;
  va_start(args,fmt);
  rc = fsl_error_setv( &f->error, code, fmt, args );
  va_end(args);
  return rc;
}

int fsl_cx_err_get( fsl_cx * const f, char const ** str, fsl_size_t * len ){
#if 1
  return fsl_error_get( &f->error, str, len );
#else
  /* For the docs: 
   If fsl_cx_interrupted() has been called with an error code and the
   context has no other pending error state, that code is returned.
  */
  int const rc = fsl_error_get( &f->error, str, len );
  return rc ? rc : f->interrupted;
#endif
}

fsl_id_t fsl_cx_last_insert_id(fsl_cx * const f){
  return (f && f->dbMain && f->dbMain->dbh)
    ? fsl_db_last_insert_id(f->dbMain)
    : -1;
}

fsl_cx * fsl_cx_malloc(){
  fsl_cx * rc = (fsl_cx *)fsl_malloc(sizeof(fsl_cx));
  if(rc) {
    *rc = fsl_cx_empty;
    rc->allocStamp = &fsl_cx_empty;
  }
  return rc;
}

int fsl_cx_err_report( fsl_cx * const f, bool addNewline ){
  if(!f) return FSL_RC_MISUSE;
  else if(f->error.code){
    char const * msg = f->error.msg.used
      ? (char const *)f->error.msg.mem
      : fsl_rc_cstr(f->error.code)
      ;
    return fsl_outputf(f, "Error #%d: %s%s",
                       f->error.code, msg,
                       addNewline ? "\n" : "");
  }
  else return 0;
}

int fsl_cx_uplift_db_error( fsl_cx * const f, fsl_db * db ){
  assert(f);
  if(!db){
    db = f->dbMain;
    assert(db && "misuse: no DB handle to uplift error from!");
    if(!db) return FSL_RC_MISUSE;
  }
  fsl_error_move( &db->error, &f->error );
  return f->error.code;
}

int fsl_cx_uplift_db_error2(fsl_cx * const f, fsl_db * db, int rc){
  assert(f);
  if(!f->error.code && rc && rc!=FSL_RC_OOM){
    if(!db) db = f->dbMain;
    assert(db && "misuse: no DB handle to uplift error from!");
    if(db->error.code) rc = fsl_cx_uplift_db_error(f, db);
  }
  return rc;
}

fsl_db * fsl_cx_db_config( fsl_cx * const f ){
  return f->config.db.dbh ? &f->config.db : NULL;
}

fsl_db * fsl_cx_db_repo( fsl_cx * const f ){
  if(f->dbMain && (FSL_DBROLE_REPO & f->dbMain->role)) return f->dbMain;
  else if(f->repo.db.dbh) return &f->repo.db;
  else return NULL;
}

fsl_db * fsl_needs_repo(fsl_cx * const f){
  fsl_db * const db = fsl_cx_db_repo(f);
  if(!db){
    fsl_cx_err_set(f, FSL_RC_NOT_A_REPO,
                   "Fossil context has no opened repository db.");
  }
  return db;
}

fsl_db * fsl_needs_ckout(fsl_cx * const f){
  fsl_db * const db = fsl_cx_db_ckout(f);
  if(!db){
    fsl_cx_err_set(f, FSL_RC_NOT_A_CKOUT,
                   "Fossil context has no opened checkout db.");
  }
  return db;
}

fsl_db * fsl_cx_db_ckout( fsl_cx * const f ){
  if(f->dbMain && (FSL_DBROLE_CKOUT & f->dbMain->role)) return f->dbMain;
  else if(f->ckout.db.dbh) return &f->ckout.db;
  else return NULL;
}

fsl_db * fsl_cx_db( fsl_cx * const f ){
  return f->dbMain;
}

fsl_db * fsl__cx_db_for_role(fsl_cx * const f, fsl_dbrole_e r){
  switch(r){
    case FSL_DBROLE_CONFIG:
      return &f->config.db;
    case FSL_DBROLE_REPO:
      return &f->repo.db;
    case FSL_DBROLE_CKOUT:
      return &f->ckout.db;
    case FSL_DBROLE_MAIN:
      return f->dbMain;
    case FSL_DBROLE_NONE:
    default:
      return NULL;
  }
}

/**
   Returns the "counterpart" role for the given db role.
*/
static fsl_dbrole_e fsl__dbrole_counterpart(fsl_dbrole_e r){
  switch(r){
    case FSL_DBROLE_REPO: return FSL_DBROLE_CKOUT;
    case FSL_DBROLE_CKOUT: return FSL_DBROLE_REPO;
    default:
      fsl__fatal(FSL_RC_MISUSE,
                 "Serious internal API misuse uncovered by %s().",
                 __func__);
      return FSL_DBROLE_NONE;
  }
}

/**
    Detaches/closes the given db role from f->dbMain and removes the
    role from f->dbMain->role. If r reflects the current primary db
    and a secondary db is attached, the secondary gets detached.  If r
    corresponds to the secondary db, only that db is detached. If
    f->dbMain is the given role then f->dbMain is set to NULL.
*/
static int fsl__cx_detach_role(fsl_cx * const f, fsl_dbrole_e r){
  assert(FSL_DBROLE_CONFIG!=r && "Config db now has its own handle.");
  assert(FSL_DBROLE_REPO==r || FSL_DBROLE_CKOUT==r);
  if(NULL==f->dbMain){
    assert(!"Internal API misuse: don't try to detach when dbMain is NULL.");
    return fsl_cx_err_set(f, FSL_RC_MISUSE,
                          "Cannot close/detach db: none opened.");
  }else if(!(r & f->dbMain->role)){
    assert(!"Misuse: cannot detach unattached role.");
    return fsl_cx_err_set(f, FSL_DBROLE_CKOUT==r
                          ? FSL_RC_NOT_A_CKOUT
                          : FSL_RC_NOT_A_REPO,
                          "Cannot close/detach unattached role: %s",
                          fsl_db_role_name(r));
  }else{
    fsl_db * const db = fsl__cx_db_for_role(f,r);
    int rc = 0;
    switch(r){
      case FSL_DBROLE_REPO:
      case FSL_DBROLE_CKOUT:
        fsl_cx_caches_reset(f);
        break;
      default:
        fsl__fatal(FSL_RC_ERROR, "Cannot happen. Really.");
    }
    fsl__cx_finalize_cached_stmt(f);
    fsl__db_cached_clear_role(f->dbMain, r)
      /* Make sure that we destroy any cached statements which are
         known to be tied to this db role. */;    
    if(db->dbh){
      /* This is our MAIN db. CLOSE it. If we still have a
         secondary/counterpart db open, we'll detach it first. */
      fsl_dbrole_e const counterpart = fsl__dbrole_counterpart(r);
      assert(f->dbMain == db);
      if(db->role & counterpart){
        /* When closing the main db, detach the counterpart db first
           (if it's attached). We'll ignore any result code here, for
           sanity's sake. */
        assert(fsl__cx_db_for_role(f,counterpart)->filename &&
               "Inconsistent internal db handle state.");
        fsl__cx_detach_role(f, counterpart);
      }
      fsl_db_close(db);
      f->dbMain = NULL;
    }else{
      /* This is our secondary db. DETACH it. */
      assert(f->dbMain != db);
      rc = fsl_db_detach( f->dbMain, fsl_db_role_name(r) );
      //MARKER(("rc=%s %s %s\n", fsl_rc_cstr(rc), fsl_db_role_name(r),
      //        fsl_buffer_cstr(&f->dbMain->error.msg)));
      if(rc){
        fsl_cx_uplift_db_error(f, f->dbMain);
      }else{
        f->dbMain->role &= ~r;
        fsl__db_clear_strings(db, true);
      }
    }
    return rc;
  }
}

int fsl__cx_attach_role(fsl_cx * const f, const char *zDbName,
                         fsl_dbrole_e r, bool createIfNotExists){
  char const * dbName = fsl_db_role_name(r);
  fsl_db * const db = fsl__cx_db_for_role(f, r);
  int rc;
  assert(db);
  assert(!db->dbh && "Internal API misuse: don't call this when db is connected.");
  assert(!db->filename && "Don't re-attach!");
  assert(!db->name && "Don't re-attach!");
  assert(dbName);
  assert(f->dbMain != db && "Don't re-attach the main db!");
  switch(r){
    case FSL_DBROLE_REPO:
    case FSL_DBROLE_CKOUT:
      break;
    case FSL_DBROLE_CONFIG:
    case FSL_DBROLE_MAIN:
    case FSL_DBROLE_NONE:
    default:
      assert(!"cannot happen/not legal");
      fsl__fatal(FSL_RC_RANGE, "Serious internal API misuse via %s().",
                 __func__);
      return FSL_RC_RANGE;
  }
  db->f = f;
  db->name = fsl_strdup(dbName);
  if(!db->name){
    rc = FSL_RC_OOM;
    goto end;
  }
  if(!f->dbMain){
    // This is our first/main db. OPEN it.
    rc = fsl_db_open( db, zDbName, createIfNotExists
                      ? FSL_OPEN_F_RWC
                      : FSL_OPEN_F_RW );
    if(rc){
      rc = fsl_cx_uplift_db_error2(f, db, rc);
      fsl_db_close(db);
      goto end;
    }
    int const sqrc = sqlite3_db_config(db->dbh,
                                       SQLITE_DBCONFIG_MAINDBNAME,
                                       dbName);
    if(sqrc){
      rc = fsl__db_errcode(&f->config.db, sqrc);
      fsl_cx_uplift_db_error2(f, db, rc);
      fsl_db_close(db);
    }else{
      rc = fsl__cx_init_db(f, db);
      db->role |= r;
      assert(db == f->dbMain)/*even on failure*/;
      /* Should we fsl__cx_detach_role() here? */
    }
  }else{
    // This is our secondary db. ATTACH it.
    assert(db != f->dbMain);
    db->filename = fsl_strdup(zDbName);
    if(!db->filename){
      rc = FSL_RC_OOM;
    }else{
      bool createdIt = false;
      if(createIfNotExists
         && 0!=fsl_file_access( zDbName, F_OK )){
        FILE * const cf = fsl_fopen(zDbName, "w");
        if(!cf){
          rc = fsl_cx_err_set(f, fsl_errno_to_rc(errno, FSL_RC_IO),
                              "Error creating new db file [%s].",
                              zDbName);
          goto end;
        }
        fsl_fclose(cf);
        createdIt = true;
      }
      rc = fsl_db_attach(f->dbMain, zDbName, dbName);
      if(rc){
        fsl_cx_uplift_db_error(f, f->dbMain);
        fsl_db_close(db)/*cleans up strings*/;
        db->f = NULL;
        if(createdIt) fsl_file_unlink(zDbName);
      }else{
        /*MARKER(("Attached %p role %d %s %s\n",
          (void const *)db, r, db->name, db->filename));*/
        f->dbMain->role |= r;
      }
    }
  }
  end:
  return rc;
}

int fsl_config_close( fsl_cx * const f ){
  if(fsl_db_transaction_level(&f->config.db)){
    return fsl_cx_err_set(f, FSL_RC_MISUSE,
                          "Cannot close config db with an "
                          "opened transaction.");
  }
  fsl_db_close(&f->config.db);
  return 0;
}

int fsl_close_scm_dbs(fsl_cx * const f){
  if(fsl_cx_transaction_level(f)){
    /* TODO???: force a full rollback and close it */
    //if(f->repo.db.dbh) fsl_db_rollback_force(&f->repo.db);
    //if(f->ckout.db.dbh) fsl_db_rollback_force(&f->ckout.db);
    return fsl_cx_err_set(f, FSL_RC_MISUSE,
                          "Cannot close repo or checkout with an "
                          "opened transaction.");
  }else if(!f->dbMain){
    // Make sure that all string resources are cleaned up...
    fsl_db_close(&f->repo.db);
    fsl_db_close(&f->ckout.db);
    return 0;
  }else{
    fsl_db * const dbR = &f->repo.db;
    return fsl__cx_detach_role(f, f->dbMain == dbR
                               ? FSL_DBROLE_REPO
                               : FSL_DBROLE_CKOUT)
      /* Will also close the counterpart db. */;
  }    
}

int fsl_repo_close( fsl_cx * const f ){
  return fsl_close_scm_dbs(f);
}

int fsl_ckout_close( fsl_cx * const f ){
  return fsl_close_scm_dbs(f);
}

/**
   If zDbName is a valid checkout database file, open it and return 0.
   If it is not a valid local database file, return a non-0 code.
*/
static int fsl__cx_ckout_open_db(fsl_cx * f, const char *zDbName){
  /* char *zVFileDef; */
  int rc;
  fsl_int_t const lsize = fsl_file_size(zDbName);
  if( -1 == lsize  ){
    return FSL_RC_NOT_FOUND /* might be FSL_RC_ACCESS? */;
  }
  if( lsize%1024!=0 || lsize<4096 ){
    return fsl_cx_err_set(f, FSL_RC_RANGE,
                          "File's size is not correct for a "
                          "checkout db: %s",
                          zDbName);
  }
  rc = fsl__cx_attach_role(f, zDbName, FSL_DBROLE_CKOUT, false);
  return rc;
}


int fsl_cx_execv( fsl_cx * const f, char const * sql, va_list args ){
  int const rc = (f->dbMain && sql)
    ? fsl_db_execv(f->dbMain, sql, args)
    : FSL_RC_MISUSE;
  return rc ? fsl_cx_uplift_db_error2(f, f->dbMain, rc) : 0;
}

int fsl_cx_exec( fsl_cx * const f, char const * sql, ... ){
  int rc;
  va_list args;
  va_start(args,sql);
  rc = fsl_cx_execv( f, sql, args );
  va_end(args);
  return rc;
}

int fsl_cx_exec_multiv( fsl_cx * const f, char const * sql, va_list args ){
  int const rc = (f->dbMain && sql)
    ? fsl_db_exec_multiv(f->dbMain, sql, args)
    : FSL_RC_MISUSE;
  return rc ? fsl_cx_uplift_db_error2(f, f->dbMain, rc) : 0;
}

int fsl_cx_exec_multi( fsl_cx * const f, char const * sql, ... ){
  int rc;
  va_list args;
  va_start(args,sql);
  rc = fsl_cx_exec_multiv( f, sql, args );
  va_end(args);
  return rc;
}

int fsl_cx_preparev( fsl_cx * const f, fsl_stmt * const tgt, char const * sql,
                     va_list args ){
  int const rc = (f->dbMain && tgt)
    ? fsl_db_preparev(f->dbMain, tgt, sql, args)
    : FSL_RC_MISUSE;
  return rc ? fsl_cx_uplift_db_error2(f, f->dbMain, rc) : 0;
}

int fsl_cx_prepare( fsl_cx * const f, fsl_stmt * const tgt, char const * sql,
                      ... ){
  int rc;
  va_list args;
  va_start(args,sql);
  rc = fsl_cx_preparev( f, tgt, sql, args );
  va_end(args);
  return rc;
}

int fsl_cx_preparev_cached( fsl_cx * const f, fsl_stmt ** tgt, char const * sql,
                            va_list args ){
  int const rc = (f->dbMain && tgt)
    ? fsl_db_preparev_cached(f->dbMain, tgt, sql, args)
    : FSL_RC_MISUSE;
  return rc ? fsl_cx_uplift_db_error2(f, f->dbMain, rc) : 0;
}

int fsl_cx_prepare_cached( fsl_cx * const f, fsl_stmt ** tgt, char const * sql,
                           ... ){
  int rc;
  va_list args;
  va_start(args,sql);
  rc = fsl_cx_preparev_cached( f, tgt, sql, args );
  va_end(args);
  return rc;
}


/**
    Passes the fsl_schema_config() SQL code through a new/truncated
    file named dbName. If the file exists before this call, it is
    unlink()ed and fails if that operation fails.

    FIXME: this was broken by the addition of "cfg." prefix on the
    schema's tables.
 */
static int fsl_config_file_reset(fsl_cx * const f, char const * dbName){
  fsl_db DB = fsl_db_empty;
  fsl_db * db = &DB;
  int rc = 0;
  bool isAttached = false;
  const char * zPrefix = fsl_db_role_name(FSL_DBROLE_CONFIG);
  if(-1 != fsl_file_size(dbName)){
    rc = fsl_file_unlink(dbName);
    if(rc){
      return fsl_cx_err_set(f, rc,
                            "Error %s while removing old config file (%s)",
                            fsl_rc_cstr(rc), dbName);
    }
  }
  /**
     Hoop-jumping: because the schema file has a cfg. prefix for the
     table(s), and we cannot assign an arbitrary name to an open()'d
     db, we first open the db (making the the "main" db), then
     ATTACH it to itself to provide the fsl_db_role_name() alias.
  */
  rc = fsl_db_open(db, dbName, FSL_OPEN_F_RWC);
  if(rc) goto end;
  rc = fsl_db_attach(db, dbName, zPrefix);
  if(rc) goto end;
  isAttached = true;
  rc = fsl_db_exec_multi(db, "%s", fsl_schema_config());
  end:
  rc = fsl_cx_uplift_db_error2(f, db, rc);
  if(isAttached) fsl_db_detach(db, zPrefix);
  fsl_db_close(db);
  return rc;
}

int fsl_config_global_preferred_name(char ** zOut){
  char * zEnv = 0 /* from fsl_getenv(). Note the special-case free()
                     semantics!!! */;
  char * zRc = 0 /* `*zOut` result, from fsl_mprintf() */;
  int rc = 0;

#if FSL_PLATFORM_IS_WINDOWS
  zEnv = fsl_getenv("FOSSIL_HOME");
  if( zEnv==0 ){
    zEnv = fsl_getenv("LOCALAPPDATA");
    if( zEnv==0 ){
      zEnv = fsl_getenv("APPDATA");
      if( zEnv==0 ){
        zEnv = fsl_getenv("USERPROFILE");
        if( zEnv==0 ){
          char * const zDrive = fsl_getenv("HOMEDRIVE");
          char * const zPath = fsl_getenv("HOMEPATH");
          if( zDrive && zPath ){
            zRc = fsl_mprintf("%s%//_fossil", zDrive, zPath);
          }
          if(zDrive) fsl_filename_free(zDrive);
          if(zPath) fsl_filename_free(zPath);
        }
      }
    }
  }
  if(!zRc){
    if(!zEnv) rc = FSL_RC_NOT_FOUND;
    else{
      zRc = fsl_mprintf("%//_fossil", zEnv);
      if(!zRc) rc = FSL_RC_OOM;
    }
  }
#else
  fsl_buffer buf = fsl_buffer_empty;
  /* Option 1: $FOSSIL_HOME/.fossil */
  zEnv = fsl_getenv("FOSSIL_HOME");
  if(zEnv){
    zRc = fsl_mprintf("%s/.fossil", zEnv);
    if(!zRc) rc = FSL_RC_OOM;
    goto end;
  }
  /* Option 2: if $HOME/.fossil exists, use that */  
  rc = fsl_find_home_dir(&buf, 0);
  if(rc) goto end;
  rc = fsl_buffer_append(&buf, "/.fossil", 8);
  if(rc) goto end;
  if(fsl_file_size(fsl_buffer_cstr(&buf))>1024*3){
    zRc = fsl_buffer_take(&buf);
    goto end;
  }
  /* Option 3: $XDG_CONFIG_HOME/fossil.db */
  fsl_filename_free(zEnv);
  zEnv = fsl_getenv("XDG_CONFIG_HOME");
  if(zEnv){
    zRc = fsl_mprintf("%s/fossil.db", zEnv);
    if(!zRc) rc = FSL_RC_OOM;
    goto end;
  }
  /* Option 4: If $HOME/.config is a directory,
     use $HOME/.config/fossil.db */
  buf.used -= 8 /* "/.fossil" */;
  buf.mem[buf.used] = 0;
  rc = fsl_buffer_append(&buf, "/.config", 8);
  if(rc) goto end;
  if(fsl_dir_check(fsl_buffer_cstr(&buf))>0){
    zRc = fsl_mprintf("%b/fossil.db", &buf);
    if(!zRc) rc = FSL_RC_OOM;
    goto end;
  }
  /* Option 5: fall back to $HOME/.fossil */
  buf.used -= 8 /* "/.config" */;
  buf.mem[buf.used] = 0;
  rc = fsl_buffer_append(&buf, "/.fossil", 8);
  if(!rc) zRc = fsl_buffer_take(&buf);
  end:
  fsl_buffer_clear(&buf);
#endif
  if(zEnv) fsl_filename_free(zEnv);
  if(!rc){
    if(zRc) *zOut = zRc;
    else rc = FSL_RC_OOM;
  }
  return rc;
}

int fsl_config_open( fsl_cx * const f, char const * openDbName ){
  int rc = 0;
  const char * zDbName = 0;
  char * zPrefName = 0;
  if(fsl_cx_db_config(f)){
    if(NULL==openDbName || 0==*openDbName) return 0/*nothing to do*/;
    fsl_config_close(f);
    assert(!f->config.db.dbh);
  }
  if(openDbName && *openDbName){
    zDbName = openDbName;
  }else{
    rc = fsl_config_global_preferred_name(&zPrefName);
    if(rc) goto end;
    zDbName = zPrefName;
  }
  {
    fsl_int_t const fsize = fsl_file_size(zDbName);
    if( -1==fsize || (fsize<1024*3) ){
      rc = fsl_config_file_reset(f, zDbName);
      if(rc) goto end;
    }
  }
#if defined(_WIN32) || defined(__CYGWIN__)
  /* TODO: Jan made some changes in this area in fossil(1) in
     January(?) 2014, such that only the config file needs to be
     writable, not the directory. Port that in.
  */
  if( fsl_file_access(zDbName, W_OK) ){
    rc = fsl_cx_err_set(f, FSL_RC_ACCESS,
                        "Configuration database [%s] "
                        "must be writeable.", zDbName);
    goto end;
  }
#endif
  assert(NULL==fsl_cx_db_config(f));
  rc = fsl_db_open(&f->config.db, zDbName,
                   FSL_OPEN_F_RW | (f->cxConfig.traceSql
                                    ? FSL_OPEN_F_TRACE_SQL
                                    : 0));
  if(0==rc){
    int const sqrc = sqlite3_db_config(f->config.db.dbh,
                                       SQLITE_DBCONFIG_MAINDBNAME,
                                       fsl_db_role_name(FSL_DBROLE_CONFIG));
    if(sqrc) rc = fsl__db_errcode(&f->config.db, sqrc);
  }
  if(rc){
    rc = fsl_cx_uplift_db_error2(f, &f->config.db, rc);
    fsl_db_close(&f->config.db);
  }
  end:
  fsl_free(zPrefName);
  return rc;
}

static void fsl_cx_username_from_repo(fsl_cx * f){
  fsl_db * dbR = fsl_cx_db_repo(f);
  char * u;
  assert(dbR);
  u = fsl_db_g_text(fsl_cx_db_repo(f), NULL,
                    "SELECT login FROM user WHERE uid=1");
  if(u){
    fsl_free(f->repo.user);
    f->repo.user = u;
  }
}

static int fsl_cx_load_glob_lists(fsl_cx * f){
  int rc;
  rc = fsl_config_globs_load(f, &f->cache.globs.ignore, "ignore-glob");
  if(!rc) rc = fsl_config_globs_load(f, &f->cache.globs.binary, "binary-glob");
  if(!rc) rc = fsl_config_globs_load(f, &f->cache.globs.crnl, "crnl-glob");
  return rc;
}

int fsl_cx_glob_list( fsl_cx * const f,
                      fsl_glob_category_e gtype,
                      fsl_list **tgt,
                      bool reload ){
  fsl_list * li = NULL;
  char const * reloadKey = NULL;
  switch(gtype){
    case FSL_GLOBS_IGNORE: li = &f->cache.globs.ignore;
      reloadKey = "ignore-glob"; break;
    case FSL_GLOBS_CRNL: li = &f->cache.globs.crnl;
      reloadKey = "crnl-glob"; break;
    case FSL_GLOBS_BINARY: li = &f->cache.globs.binary;
      reloadKey = "binary-glob"; break;
    default:
      return FSL_RC_RANGE;
  }
  int rc = 0;
  if(reload){
    assert(reloadKey);
    fsl_glob_list_clear(li);
    rc = fsl_config_globs_load(f, li, reloadKey);
  }
  if(0==rc) *tgt = li;
  return rc;
}

fsl_glob_category_e fsl_glob_name_to_category(char const * str){
  if(str){
#define CHECK(PRE,E) \
    if(*str==PRE[0] &&                          \
       (0==fsl_strcmp(PRE "-glob",str)          \
        || 0==fsl_strcmp(PRE,str))) return E;
    CHECK("ignore", FSL_GLOBS_IGNORE);
    CHECK("binary", FSL_GLOBS_BINARY);
    CHECK("crnl", FSL_GLOBS_CRNL);
#undef CHECK
  }
  return FSL_GLOBS_INVALID;
}


/**
   To be called after a repo or checkout/repo combination has been
   opened. This updates some internal cached info based on the
   checkout and/or repo.
*/
static int fsl_cx_after_open(fsl_cx * f){
  int rc = fsl__ckout_version_fetch(f);
  if(!rc) rc = fsl_cx_load_glob_lists(f);
  return rc;
}


static void fsl_cx_fetch_hash_policy(fsl_cx * f){
  int const iPol =
    fsl_config_get_int32( f, FSL_CONFDB_REPO,
                          FSL_HPOLICY_AUTO, "hash-policy");
  fsl_hashpolicy_e p;
  switch(iPol){
    case FSL_HPOLICY_SHA3: p = FSL_HPOLICY_SHA3; break;
    case FSL_HPOLICY_SHA3_ONLY: p = FSL_HPOLICY_SHA3_ONLY; break;
    case FSL_HPOLICY_SHA1: p = FSL_HPOLICY_SHA1; break;
    case FSL_HPOLICY_SHUN_SHA1: p = FSL_HPOLICY_SHUN_SHA1; break;
    default: p = FSL_HPOLICY_AUTO; break;
  }
  f->cxConfig.hashPolicy = p;
}

#if 0
/**
    Return true if the schema is out-of-date. db must be an opened
    repo db.
 */
static bool fsl__db_repo_schema_is_outofdate(fsl_db *db){
  return fsl_db_exists(db, "SELECT 1 FROM config "
                       "WHERE name='aux-schema' "
                       "AND value<>'%s'",
                       FSL_AUX_SCHEMA);
}

/*
   Returns 0 if db appears to have a current repository schema, 1 if
   it appears to have an out of date schema, and -1 if it appears to
   not be a repository.
*/
int fsl__db_repo_verify_schema(fsl_db * const db){
  if(fsl__db_repo_schema_is_outofdate(db)) return 1;
  else return fsl_db_exists(db,
                            "SELECT 1 FROM config "
                            "WHERE name='project-code'")
    ? 0 : -1;
}
int fsl_repo_schema_validate(fsl_cx * const f, fsl_db * const db){
  int rc = 0;
  int const check = fsl__db_repo_verify_schema(db);
  if(0 != check){
    rc = (check<0)
      ? fsl_cx_err_set(f, FSL_RC_NOT_A_REPO,
                      "DB file [%s] does not appear to be "
                      "a repository.", db->filename)
      : fsl_cx_err_set(f, FSL_RC_REPO_NEEDS_REBUILD,
                      "DB file [%s] appears to be a fossil "
                      "repsitory, but is out-of-date and needs "
                      "a rebuild.",
                      db->filename);
  }
  return rc;
}
#endif

int fsl_repo_open( fsl_cx * const f, char const * repoDbFile
                   /* , bool readOnlyCurrentlyIgnored */ ){
  if(fsl_cx_db_repo(f)){
    return fsl_cx_err_set(f, FSL_RC_ACCESS,
                          "Context already has an opened repository.");
  }else{
    int rc;
    if(0!=fsl_file_access( repoDbFile, F_OK )){
      rc = fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                          "Repository db [%s] not found or cannot be read.",
                          repoDbFile);
    }else{
      rc = fsl__cx_attach_role(f, repoDbFile, FSL_DBROLE_REPO, false);
      if(!rc && !(FSL_CX_F_IS_OPENING_CKOUT & f->flags)){
        rc = fsl_cx_after_open(f);
      }
      if(!rc){
        fsl_db * const db = fsl_cx_db_repo(f);
        fsl_cx_username_from_repo(f);
        fsl_cx_allows_symlinks(f, true);
        fsl_cx_is_case_sensitive(f, true);
        f->cache.seenDeltaManifest =
          fsl_config_get_int32(f, FSL_CONFDB_REPO, -1,
                               "seen-delta-manifest");
        fsl_cx_fetch_hash_policy(f);
        if(f->cxConfig.hashPolicy==FSL_HPOLICY_AUTO){
          if(fsl_db_exists(db, "SELECT 1 FROM blob WHERE length(uuid)>40")
             || !fsl_db_exists(db, "SELECT 1 FROM blob WHERE length(uuid)==40")){
            f->cxConfig.hashPolicy = FSL_HPOLICY_SHA3;
          }
        }
      }
    }
    return rc;
  }
}

/**
    Tries to open the repository from which the current checkout
    derives. Returns 0 on success.
*/
static int fsl_repo_open_for_ckout(fsl_cx * f){
  char * repoDb = NULL;
  int rc;
  fsl_buffer nameBuf = fsl_buffer_empty;
  fsl_db * db = fsl_cx_db_ckout(f);
  assert(f);
  assert(f->ckout.dir);
  assert(db);
  rc = fsl_db_get_text(db, &repoDb, NULL,
                       "SELECT value FROM vvar "
                       "WHERE name='repository'");
  if(rc) fsl_cx_uplift_db_error( f, db );
  else if(repoDb){
    if(!fsl_is_absolute_path(repoDb)){
      /* Make it relative to the checkout db dir */
      rc = fsl_buffer_appendf(&nameBuf, "%s/%s", f->ckout.dir, repoDb);
      fsl_free(repoDb);
      if(rc) {
        fsl_buffer_clear(&nameBuf);
        return rc;
      }
      repoDb = (char*)nameBuf.mem /* transfer ownership */;
      nameBuf = fsl_buffer_empty;
    }
    rc = fsl_file_canonical_name(repoDb, &nameBuf, 0);
    fsl_free(repoDb);
    if(!rc){
      repoDb = fsl_buffer_str(&nameBuf);
      assert(repoDb);
      rc = fsl_repo_open(f, repoDb);
    }
    fsl_buffer_reserve(&nameBuf, 0);
  }else{
    /* This can only happen if we are not using a proper
       checkout db or someone has removed the repo link.
    */
    rc = fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                        "Could not determine this checkout's "
                        "repository db file.");
  }
  return rc;
}

static void fsl_ckout_mtime_set(fsl_cx * const f){
  f->ckout.mtime = f->ckout.rid>0
    ? fsl_db_g_double(fsl_cx_db_repo(f), 0.0,
                      "SELECT mtime FROM event "
                      "WHERE objid=%" FSL_ID_T_PFMT,
                      f->ckout.rid)
    : 0.0;
}

void fsl__cx_ckout_clear( fsl_cx * const f ){
  fsl_free(f->ckout.uuid);
  f->ckout.rid = -1;
  f->ckout.uuid = NULL;
  f->ckout.mtime = 0.0;
}

int fsl__ckout_version_fetch( fsl_cx * const f ){
  fsl_id_t rid = 0;
  int rc = 0;
  fsl_db * dbC = fsl_cx_db_ckout(f);
  fsl_db * dbR = dbC ? fsl_needs_repo(f) : NULL;
  fsl__cx_ckout_clear(f);
  if(!dbC) return 0;
  else if(!dbR) return FSL_RC_NOT_A_REPO;
  rid = fsl_config_get_id(f, FSL_CONFDB_CKOUT, -1, "checkout");
  //MARKER(("rc=%s rid=%d\n",fsl_rc_cstr(f->error.code), (int)rid));
  if(rid>0){
    f->ckout.uuid = fsl_rid_to_uuid(f, rid);
    if(!f->ckout.uuid){
      assert(f->error.code);
      if(!f->error.code){
        rc = fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                            "Could not load UUID for RID %"FSL_ID_T_PFMT,
                            (fsl_id_t)rid);
      }
    }else{
      assert(fsl_is_uuid(f->ckout.uuid));
    }
    f->ckout.rid = rid;
    fsl_ckout_mtime_set(f);
  }else if(rid==0){
    /* This is a legal case not possible before libfossil (and only
       afterwards possible in fossil(1)) - an empty repo without an
       active checkin. [Much later:] that capability has since been
       removed from fossil.
    */
    f->ckout.rid = 0;
  }else{
    rc = fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                        "Cannot determine checkout version.");
  }
  return rc;
}

/** @internal

    Sets f->ckout.rid to the given rid (which must be 0 or a valid
    RID) and f->ckout.uuid to a copy of the given uuid. If uuid is
    NULL and rid is not 0 then the uuid is fetched using
    fsl_rid_to_uuid(), else if uuid is not NULL then it is assumed to
    be the UUID for the given RID and is copies to f->ckout.uuid.

    Returns 0 on success, FSL_RC_OOM if copying uuid fails, or some
    error from fsl_rid_to_uuid() if that fails.

    Does not write the changes to disk. Use fsl__ckout_version_write()
    for that. That routine also calls this one, so there's no need to
    call both.
*/
static int fsl_cx_ckout_version_set(fsl_cx *f, fsl_id_t rid,
                                    fsl_uuid_cstr uuid){
  char * u = 0;
  assert(rid>=0);
  u = uuid
    ? fsl_strdup(uuid)
    : (rid ? fsl_rid_to_uuid(f, rid) : NULL);
  if(rid && !u) return FSL_RC_OOM;
  f->ckout.rid = rid;
  fsl_free(f->ckout.uuid);
  f->ckout.uuid = u;
  fsl_ckout_mtime_set(f);
  return 0;
}

int fsl__ckout_version_write( fsl_cx * const f, fsl_id_t vid,
                             fsl_uuid_cstr hash ){
  int rc = 0;
  if(!fsl_needs_ckout(f)) return FSL_RC_NOT_A_CKOUT;
  else if(vid<0){
    return fsl_cx_err_set(f, FSL_RC_MISUSE,
                          "Invalid vid for fsl__ckout_version_write()");
  }
  if(f->ckout.rid!=vid){
    rc = fsl_cx_ckout_version_set(f, vid, hash);
  }
  if(!rc){
    rc = fsl_config_set_id(f, FSL_CONFDB_CKOUT,
                           "checkout", f->ckout.rid);
    if(!rc){
      rc = fsl_config_set_text(f, FSL_CONFDB_CKOUT,
                               "checkout-hash", f->ckout.uuid);
    }
  }
  if(!rc){
    char * zFingerprint = 0;
    rc = fsl__repo_fingerprint_search(f, 0, &zFingerprint);
    if(!rc){
      rc = fsl_config_set_text(f, FSL_CONFDB_CKOUT,
                               "fingerprint", zFingerprint);
      fsl_free(zFingerprint);
    }
  }
  if(!rc){
    int const mode = vid ? -1 : 0;
    rc = fsl_ckout_manifest_write(f, mode, mode, mode, 0);
  }
  return rc;
}

void fsl_ckout_version_info(fsl_cx * const f, fsl_id_t * const rid,
                            fsl_uuid_cstr * const uuid ){
  if(uuid) *uuid = f->ckout.uuid;
  if(rid) *rid = f->ckout.rid>=0 ? f->ckout.rid : 0;
}

int fsl_ckout_db_search( char const * dirName, bool checkParentDirs,
                         fsl_buffer * const pOut ){
  int rc;
  fsl_int_t dLen = 0, i;
  enum { DbCount = 2 };
  const char aDbName[DbCount][10] = { "_FOSSIL_", ".fslckout" };
  fsl_buffer Buf = fsl_buffer_empty;
  fsl_buffer * buf = &Buf;
  if(dirName){
    dLen = fsl_strlen(dirName);
    if(0==dLen) return FSL_RC_RANGE;
    rc = fsl_buffer_reserve( buf, (fsl_size_t)(dLen + 10) );
    if(!rc) rc = fsl_buffer_append( buf, dirName, dLen );
    if(rc){
      fsl_buffer_clear(buf);
      return rc;
    }
  }else{
    char zPwd[4000];
    fsl_size_t pwdLen = 0;
    rc = fsl_getcwd( zPwd, sizeof(zPwd)/sizeof(zPwd[0]), &pwdLen );
    if(rc){
      fsl_buffer_clear(buf);
#if 0
      return fsl_cx_err_set(f, rc,
                            "Could not determine current directory. "
                            "Error code %d (%s).",
                            rc, fsl_rc_cstr(rc));
#else
      return rc;
#endif
    }
    if(1 == pwdLen && '/'==*zPwd) *zPwd = '.'
      /* When in the root directory (or chroot) then change dir name
         name to something we can use.
      */;
    rc = fsl_buffer_append(buf, zPwd, pwdLen);
    if(rc){
      fsl_buffer_clear(buf);
      return rc;
    }
    dLen = (fsl_int_t)pwdLen;
  }
  if(rc){
    fsl_buffer_clear(buf);
    return rc;
  }
  assert(buf->capacity>=buf->used);
  assert((buf->used == (fsl_size_t)dLen) || (1==buf->used && (int)'.'==(int)buf->mem[0]));
  assert(0==buf->mem[buf->used]);

  while(dLen>0){
    /*
      Loop over the list in aDbName, appending each one to
      the dir name in the search for something we can use.
    */
    fsl_int_t lenMarker = dLen /* position to re-set to on each
                                  sub-iteration. */ ;
    /* trim trailing slashes on this part, so that we don't end up
       with multiples between the dir and file in the final output. */
    while( dLen && ((int)'/'==(int)buf->mem[dLen-1])) --dLen;
    for( i = 0; i < DbCount; ++i ){
      char const * zName;
      buf->used = (fsl_size_t)lenMarker;
      dLen = lenMarker;
      rc = fsl_buffer_appendf( buf, "/%s", aDbName[i]);
      if(rc){
        fsl_buffer_clear(buf);
        return rc;
      }
      zName = fsl_buffer_cstr(buf);
      if(0==fsl_file_access(zName, 0)){
        if(pOut) rc = fsl_buffer_append( pOut, buf->mem, buf->used );
        fsl_buffer_clear(buf);
        return rc;
      }
      if(!checkParentDirs){
        dLen = 0;
        break;
      }else{
        /* Traverse up one dir and try again. */
        --dLen;
        while( dLen>0 && (int)buf->mem[dLen]!=(int)'/' ){ --dLen; }
        while( dLen>0 && (int)buf->mem[dLen-1]==(int)'/' ){ --dLen; }
        if(dLen>lenMarker){
          buf->mem[dLen] = 0;
        }
      }
    }
  }
  fsl_buffer_clear(buf);
  return FSL_RC_NOT_FOUND;
}

int fsl_cx_getcwd(fsl_cx * const f, fsl_buffer * const pOut){
  char cwd[FILENAME_MAX] = {0};
  fsl_size_t cwdLen = 0;
  int rc = fsl_getcwd(cwd, (fsl_size_t)sizeof(cwd), &cwdLen);
  if(rc){
    return fsl_cx_err_set(f, rc,
                          "Could not get current working directory!");
  }
  rc = fsl_buffer_append(pOut, cwd, cwdLen);
  return rc
    ? fsl_cx_err_set(f, rc/*must be an OOM*/, NULL)
    : 0;
}

int fsl_ckout_open_dir( fsl_cx * const f, char const * dirName,
                        bool checkParentDirs ){
  int rc;
  fsl_buffer * const buf = fsl__cx_scratchpad(f);
  fsl_buffer * const bufD = fsl__cx_scratchpad(f);
  char const * zName;
  if(fsl_cx_db_ckout(f)){
    rc = fsl_cx_err_set( f, FSL_RC_ACCESS,
                         "A checkout is already opened. "
                         "Close it before opening another.");
    goto end; 
  }else if(!dirName){
    dirName = ".";
  }
  rc = fsl_file_canonical_name( dirName, bufD, false );
  if(rc) goto end;
  dirName = fsl_buffer_cstr(bufD);
  rc = fsl_ckout_db_search(dirName, checkParentDirs, buf);
  if(rc){
    if(FSL_RC_NOT_FOUND==rc){
      rc = fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                          "Could not find checkout under [%s].",
                          dirName ? dirName : ".");
    }
    goto end;
  }
  assert(buf->used>1 /* "/<FILENAME>" */);
  zName = fsl_buffer_cstr(buf);
  rc = fsl__cx_ckout_open_db(f, zName);
  if(0==rc){
    /* Checkout db is now opened. Fiddle some internal
       bits...
    */
    unsigned char * end = buf->mem+buf->used-1;
    /* Find dir part */
    while(end>buf->mem && (unsigned char)'/'!=*end) --end;
    assert('/' == (char)*end && "fsl_ckout_db_search() appends '/<DBNAME>'");
    fsl_free(f->ckout.dir);
    f->ckout.dirLen = end - buf->mem +1 /* for trailing '/' */ ;
    *(end+1) = 0; /* Rather than strdup'ing, we'll just lop off the
                     filename part. Keep the '/' for historical
                     conventions purposes - it simplifies path
                     manipulation later on. */
    f->ckout.dir = fsl_buffer_take(buf);
    assert(!f->ckout.dir[f->ckout.dirLen]);
    assert('/' == f->ckout.dir[f->ckout.dirLen-1]);
    f->flags |= FSL_CX_F_IS_OPENING_CKOUT;
    rc = fsl_repo_open_for_ckout(f);
    f->flags &= ~FSL_CX_F_IS_OPENING_CKOUT;
    if(!rc) rc = fsl_cx_after_open(f);
    if(rc){
      /* Is this sane? Is not doing it sane? */
      fsl_close_scm_dbs(f);
    }
  }
  end:
  fsl__cx_scratchpad_yield(f, buf);
  fsl__cx_scratchpad_yield(f, bufD);
  return rc;
}


char const * fsl_cx_db_file_for_role(fsl_cx const * f,
                                     fsl_dbrole_e r,
                                     fsl_size_t * len){
  fsl_db const * db = fsl__cx_db_for_role((fsl_cx*)f, r);
  char const * rc = db ? db->filename : NULL;
  if(len) *len = fsl_strlen(rc);
  return rc;
}

char const * fsl_cx_db_name_for_role(fsl_cx const * f,
                                     fsl_dbrole_e r,
                                     fsl_size_t * len){
  if(FSL_DBROLE_MAIN == r){
    if(f->dbMain){
      if(len) *len=4;
      return "main";
    }else{
      return NULL;
    }
  }else{
    fsl_db const * db = fsl__cx_db_for_role((fsl_cx*)f, r);
    char const * rc = db ? db->name : NULL;
    if(len) *len = rc ? fsl_strlen(rc) : 0;
    return rc;
  }
}

char const * fsl_cx_db_file_config(fsl_cx const * f,
                                   fsl_size_t * len){
  char const * rc = NULL;
  if(f && f->config.db.filename){
    rc = f->config.db.filename;
    if(len) *len = fsl_strlen(rc);
  }
  return rc;
}

char const * fsl_cx_db_file_repo(fsl_cx const * f,
                                 fsl_size_t * len){
  char const * rc = NULL;
  if(f && f->repo.db.filename){
    rc = f->repo.db.filename;
    if(len) *len = fsl_strlen(rc);
  }
  return rc;
}

char const * fsl_cx_db_file_ckout(fsl_cx const * f,
                                     fsl_size_t * len){
  char const * rc = NULL;
  if(f && f->ckout.db.filename){
    rc = f->ckout.db.filename;
    if(len) *len = fsl_strlen(rc);
  }
  return rc;
}

char const * fsl_cx_ckout_dir_name(fsl_cx const * f,
                                      fsl_size_t * len){
  char const * rc = NULL;
  if(f && f->ckout.dir){
    rc = f->ckout.dir;
    if(len) *len = f->ckout.dirLen;
  }
  return rc;
}

int fsl_cx_flags_get( fsl_cx const * const f ){
  return f->flags;
}

int fsl_cx_flag_set( fsl_cx * const f, int flags, bool enable ){
  int const oldFlags = f->flags;
  if(enable) f->flags |= flags;
  else f->flags &= ~flags;
  return oldFlags;
}


fsl_xlinker * fsl_xlinker_by_name( fsl_cx * f, char const * name ){

  fsl_xlinker * rv = NULL;
  fsl_size_t i;
  for( i = 0; i < f->xlinkers.used; ++i ){
    rv = f->xlinkers.list + i;
    if(0==fsl_strcmp(rv->name, name)) return rv;
  }
  return NULL;
}

int fsl_xlink_listener( fsl_cx * const f, char const * name,
                        fsl_deck_xlink_f cb, void * cbState ){
  fsl_xlinker * x;
  if(!*name) return FSL_RC_MISUSE;
  x = fsl_xlinker_by_name(f, name);
  if(x){
    /* Replace existing entry */
    x->f = cb;
    x->state = cbState;
    return 0;
  }else if(f->xlinkers.used <= f->xlinkers.capacity){
    /* Expand the array */
    fsl_size_t const n = f->xlinkers.used ? f->xlinkers.used * 2 : 5;
    fsl_xlinker * re =
      (fsl_xlinker *)fsl_realloc(f->xlinkers.list,
                                 n * sizeof(fsl_xlinker));
    if(!re) return FSL_RC_OOM;
    f->xlinkers.list = re;
    f->xlinkers.capacity = n;
  }
  x = f->xlinkers.list + f->xlinkers.used++;
  *x = fsl_xlinker_empty;
  x->f = cb;
  x->state = cbState;
  x->name = name;
  return 0;
}

int fsl_cx_user_set( fsl_cx * const f, char const * userName ){
  if(!f) return FSL_RC_MISUSE;
  else if(!userName || !*userName){
    fsl_free(f->repo.user);
    f->repo.user = NULL;
    return 0;
  }else{
    char * u = fsl_strdup(userName);
    if(!u) return FSL_RC_OOM;
    else{
      fsl_free(f->repo.user);
      f->repo.user = u;
      return 0;
    }    
  }
}

char const * fsl_cx_user_guess(fsl_cx * const f){
  if(!f->repo.user){
    char * u = fsl_user_name_guess();
    if(u){
      fsl_free(f->repo.user);
      f->repo.user = u;
      // don't use fsl_cx_user_set(f, u), to avoid another strdup()
    }
  }
  return f->repo.user;
}
  

char const * fsl_cx_user_get( fsl_cx const * const f ){
  return f->repo.user;
}

int fsl_cx_schema_ticket(fsl_cx * f, fsl_buffer * pOut){
  fsl_db * db = f ? fsl_needs_repo(f) : NULL;
  if(!f || !pOut) return FSL_RC_MISUSE;
  else if(!db) return FSL_RC_NOT_A_REPO;
  else{
    fsl_size_t const oldUsed = pOut->used;
    int rc = fsl_config_get_buffer(f, FSL_CONFDB_REPO,
                                   "ticket-table", pOut);
    if((FSL_RC_NOT_FOUND==rc)
       || (oldUsed == pOut->used/*found but it was empty*/)
       ){
      rc = fsl_buffer_append(pOut, fsl_schema_ticket(), -1);
    }
    return rc;
  }
}


int fsl_cx_stat2( fsl_cx * const f, bool relativeToCwd,
                  char const * zName, fsl_fstat * const tgt,
                  fsl_buffer * const nameOut, bool fullPath){
  int rc;
  fsl_buffer * b = 0;
  fsl_buffer * bufRel = 0;
  fsl_size_t n = 0;
  assert(f);
  if(!zName || !*zName) return FSL_RC_MISUSE;
  else if(!fsl_needs_ckout(f)) return FSL_RC_NOT_A_CKOUT;
  b = fsl__cx_scratchpad(f);
  bufRel = fsl__cx_scratchpad(f);
#if 1
  rc = fsl_ckout_filename_check(f, relativeToCwd, zName, bufRel);
  if(rc) goto end;
  zName = fsl_buffer_cstr2( bufRel, &n );
#else
  if(!fsl_is_simple_pathname(zName, 1)){
    rc = fsl_ckout_filename_check(f, relativeToCwd, zName, bufRel);
    if(rc) goto end;
    zName = fsl_buffer_cstr2( bufRel, &n );
    /* MARKER(("bufRel=%s\n",zName)); */
  }else{
    n = fsl_strlen(zName);
  }
#endif
  assert(n>0 &&
         "Will fail if fsl_ckout_filename_check() changes "
         "to return nothing if zName==checkout root");
  if(!n
     /* i don't like the "." resp "./" result when zName==checkout root */
     || (1==n && '.'==bufRel->mem[0])
     || (2==n && '.'==bufRel->mem[0] && '/'==bufRel->mem[1])){
    rc = fsl_buffer_appendf(b, "%s%s", f->ckout.dir,
                            (2==n) ? "/" : "");
  }else{
    rc = fsl_buffer_appendf(b, "%s%s", f->ckout.dir, zName);
  }
  if(!rc){
    rc = fsl_stat( fsl_buffer_cstr(b), tgt, false );
    if(rc){
      fsl_cx_err_set(f, rc, "Error %s from fsl_stat(\"%b\")",
                     fsl_rc_cstr(rc), b);
    }else if(nameOut){
      rc = fullPath
        ? fsl_buffer_append(nameOut, b->mem, b->used)
        : fsl_buffer_append(nameOut, zName, n);
    }
  }
  end:
  fsl__cx_scratchpad_yield(f, b);
  fsl__cx_scratchpad_yield(f, bufRel);
  return rc;
}

int fsl_cx_stat(fsl_cx * const f, bool relativeToCwd,
                char const * zName, fsl_fstat * const tgt){
  return fsl_cx_stat2(f, relativeToCwd, zName, tgt, NULL, false);
}

bool fsl_cx_allows_symlinks(fsl_cx * const f, bool forceRecheck){
  if(forceRecheck || f->cache.allowSymlinks<0){
    f->cache.allowSymlinks = fsl_config_get_bool(f, FSL_CONFDB_REPO,
                                                 false, "allow-symlinks");
  }
  return f->cache.allowSymlinks>0;
}

void fsl_cx_case_sensitive_set(fsl_cx * const f, bool caseSensitive){
  f->cache.caseInsensitive = caseSensitive ? 0 : 1;
}

bool fsl_cx_is_case_sensitive(fsl_cx * const f, bool forceRecheck){
  if(forceRecheck || f->cache.caseInsensitive<0){
    f->cache.caseInsensitive =
      fsl_config_get_bool(f, FSL_CONFDB_REPO,
                          true, "case-sensitive") ? 0 : 1;
  }
  return f->cache.caseInsensitive <= 0;
}

char const * fsl_cx_filename_collation(fsl_cx const * f){
  return f->cache.caseInsensitive>0 ? "COLLATE nocase" : "";
}

fsl_buffer * fsl__cx_content_buffer(fsl_cx * const f){
  if(f->cache.fileContent.used){
    fsl__fatal(FSL_RC_MISUSE,
               "Called %s() while the content buffer has bytes in use.");
  }
  return &f->cache.fileContent;
}

void fsl__cx_content_buffer_yield(fsl_cx * const f){
  enum { MaxSize = 1024 * 1024 * 10 };
  assert(f);
  if(f->cache.fileContent.capacity>MaxSize){
    fsl_buffer_resize(&f->cache.fileContent, MaxSize);
    assert(f->cache.fileContent.capacity<=MaxSize+1);
  }
  fsl_buffer_reuse(&f->cache.fileContent);
}

fsl_error const * fsl_cx_err_get_e(fsl_cx const * f){
  return &f->error;
}

int fsl_cx_close_dbs( fsl_cx * const f ){
  if(fsl_cx_transaction_level(f)
     || (f->config.db.dbh && fsl_db_transaction_level(&f->config.db))){
    /* Is this really necessary? Should we instead
       force rollback(s) and close the dbs? */
    return fsl_cx_err_set(f, FSL_RC_MISUSE,
                          "Cannot close the databases when a "
                          "transaction is pending.");
  }
  fsl_config_close(f);
  return fsl_close_scm_dbs(f);
}

char const * fsl_cx_glob_matches( fsl_cx * const f, int gtype,
                                  char const * str ){
  int i, count = 0;
  char const * rv = NULL;
  fsl_list const * lists[] = {0,0,0};
  if(!f || !str || !*str) return NULL;
  if(gtype & FSL_GLOBS_IGNORE) lists[count++] = &f->cache.globs.ignore;
  if(gtype & FSL_GLOBS_CRNL) lists[count++] = &f->cache.globs.crnl;
  /*CRNL/BINARY together makes little sense, but why strictly prohibit
    it?*/
  if(gtype & FSL_GLOBS_BINARY) lists[count++] = &f->cache.globs.binary;
  for( i = 0; i < count; ++i ){
    if( (rv = fsl_glob_list_matches( lists[i], str )) ) break;
  }
  return rv;
}

int fsl_output_f_fsl_cx(void * state, void const * src, fsl_size_t n ){
  return (state && src && n)
    ? fsl_output((fsl_cx*)state, src, n)
    : (n ? FSL_RC_MISUSE : 0);
}

int fsl_cx_hash_buffer( fsl_cx const * f, bool useAlternate,
                        fsl_buffer const * pIn, fsl_buffer * pOut){
  /* fossil(1) counterpart: hname_hash() */
  if(useAlternate){
    switch(f->cxConfig.hashPolicy){
      case FSL_HPOLICY_AUTO:
      case FSL_HPOLICY_SHA1:
        return fsl_sha3sum_buffer(pIn, pOut);
      case FSL_HPOLICY_SHA3:
        return fsl_sha1sum_buffer(pIn, pOut);
      default: return FSL_RC_UNSUPPORTED;
    }
  }else{
    switch(f->cxConfig.hashPolicy){
      case FSL_HPOLICY_SHA1:
      case FSL_HPOLICY_AUTO:
        return fsl_sha1sum_buffer(pIn, pOut);
      case FSL_HPOLICY_SHA3:
      case FSL_HPOLICY_SHA3_ONLY:
      case FSL_HPOLICY_SHUN_SHA1:
        return fsl_sha3sum_buffer(pIn, pOut);
    }
  }
  assert(!"not reached");
  return FSL_RC_RANGE;
}

int fsl_cx_hash_filename( fsl_cx * f, bool useAlternate,
                          const char * zFilename, fsl_buffer * pOut){
  /* FIXME: reimplement this to stream the content in bite-sized
     chunks. That requires duplicating most of fsl_buffer_fill_from()
     and fsl_cx_hash_buffer(). */
  fsl_buffer * const content = &f->cache.fileContent;
  int rc;
  assert(!content->used && "Internal recursive misuse of fsl_cx::fileContent");
  fsl_buffer_reuse(content);
  rc = fsl_buffer_fill_from_filename(content, zFilename);
  if(!rc){
    rc = fsl_cx_hash_buffer(f, useAlternate, content, pOut);
  }
  fsl_buffer_reuse(content);
  return rc;
}

char const * fsl_hash_policy_name(fsl_hashpolicy_e p){
  switch(p){
    case FSL_HPOLICY_SHUN_SHA1: return "shun-sha1";
    case FSL_HPOLICY_SHA3: return "sha3";
    case FSL_HPOLICY_SHA3_ONLY: return "sha3-only";
    case FSL_HPOLICY_SHA1: return "sha1";
    case FSL_HPOLICY_AUTO: return "auto";
    default: return NULL;
  }
}

fsl_hashpolicy_e fsl_cx_hash_policy_set(fsl_cx *f, fsl_hashpolicy_e p){
  fsl_hashpolicy_e const old = f->cxConfig.hashPolicy;
  fsl_db * const dbR = fsl_cx_db_repo(f);
  if(dbR){
    /* Write it regardless of whether it's the same as the old policy
       so that we're sure the db knows the policy. */
    if(FSL_HPOLICY_AUTO==p &&
       fsl_db_exists(dbR,"SELECT 1 FROM blob WHERE length(uuid)>40")){
      p = FSL_HPOLICY_SHA3;
    }
    fsl_config_set_int32(f, FSL_CONFDB_REPO, "hash-policy", p);
  }
  f->cxConfig.hashPolicy = p;
  return old;
}

fsl_hashpolicy_e fsl_cx_hash_policy_get(fsl_cx const*f){
  return f->cxConfig.hashPolicy;
}

int fsl_cx_transaction_level(fsl_cx * const f){
  return f->dbMain
    ? fsl_db_transaction_level(f->dbMain)
    : 0;
}

int fsl_cx_transaction_begin(fsl_cx * const f){
  int const rc = fsl_db_transaction_begin(f->dbMain);
  return rc ? fsl_cx_uplift_db_error2(f, f->dbMain, rc) : 0;
}

int fsl_cx_transaction_end(fsl_cx * const f, bool doRollback){
  int const rc = fsl_db_transaction_end(f->dbMain, doRollback);
  return rc ? fsl_cx_uplift_db_error2(f, f->dbMain, rc) : 0;
}

void fsl_cx_confirmer(fsl_cx * f,
                      fsl_confirmer const * newConfirmer,
                      fsl_confirmer * prevConfirmer){
  if(prevConfirmer) *prevConfirmer = f->confirmer;
  f->confirmer = newConfirmer ? *newConfirmer : fsl_confirmer_empty;
}

void fsl_cx_confirmer_get(fsl_cx const * f, fsl_confirmer * dest){
  *dest = f->confirmer;
}

int fsl_cx_confirm(fsl_cx * const f, fsl_confirm_detail const * detail,
                   fsl_confirm_response *outAnswer){
  if(f->confirmer.callback){
    return f->confirmer.callback(detail, outAnswer,
                                 f->confirmer.callbackState);
  }
  /* Default answers... */
  switch(detail->eventId){
    case FSL_CEVENT_OVERWRITE_MOD_FILE:
    case FSL_CEVENT_OVERWRITE_UNMGD_FILE:
      outAnswer->response =  FSL_CRESPONSE_NEVER;
      break;
    case FSL_CEVENT_RM_MOD_UNMGD_FILE:
      outAnswer->response = FSL_CRESPONSE_NEVER;
      break;
    case FSL_CEVENT_MULTIPLE_VERSIONS:
      outAnswer->response = FSL_CRESPONSE_CANCEL;
      break;
    default:
      assert(!"Unhandled fsl_confirm_event_e value");
      fsl__fatal(FSL_RC_UNSUPPORTED,
                "Unhandled fsl_confirm_event_e value: %d",
                detail->eventId)/*does not return*/;
  }
  return 0;
}

int fsl__cx_update_seen_delta_deck(fsl_cx * const f){
  int rc = 0;
  fsl_db * const d = fsl_cx_db_repo(f);
  if(d && f->cache.seenDeltaManifest <= 0){
    f->cache.seenDeltaManifest = 1;
    rc = fsl_config_set_bool(f, FSL_CONFDB_REPO,
                             "seen-delta-manifest", 1);
  }
  return rc;
}

int fsl_reserved_fn_check(fsl_cx * const f, const char *zPath,
                          fsl_int_t nPath, bool relativeToCwd){
  static const int errRc = FSL_RC_RANGE;
  int rc = 0;
  char const * z1 = 0;
  if(nPath<0) nPath = (fsl_int_t)fsl_strlen(zPath);
  if(fsl_is_reserved_fn(zPath, nPath)){
    return fsl_cx_err_set(f, errRc,
                        "Filename is reserved, not legal "
                        "for adding to a repository: %.*s",
                        (int)nPath, zPath);
  }
  if(!(f->flags & FSL_CX_F_ALLOW_WINDOWS_RESERVED_NAMES)
     && fsl__is_reserved_fn_windows(zPath, nPath)){
    return fsl_cx_err_set(f, errRc,
                          "Filename is a Windows reserved name: %.*s",
                          (int)nPath, zPath);
  }
  if((z1 = fsl_cx_db_file_for_role(f, FSL_DBROLE_REPO, NULL))){
    fsl_buffer * const c1 = fsl__cx_scratchpad(f);
    fsl_buffer * const c2 = fsl__cx_scratchpad(f);
    rc = fsl_file_canonical_name2(relativeToCwd ? NULL : f->ckout.dir/*NULL is okay*/,
                                  z1, c1, false);
    if(!rc) rc = fsl_file_canonical_name2(relativeToCwd ? NULL : f->ckout.dir,
                                          zPath, c2, false);
    //MARKER(("\nzPath=%s\nc1=%s\nc2=%s\n", zPath,
    //fsl_buffer_cstr(c1), fsl_buffer_cstr(c2)));
    if(!rc && c1->used == c2->used &&
       0==fsl_stricmp(fsl_buffer_cstr(c1), fsl_buffer_cstr(c2))){
      rc = fsl_cx_err_set(f, errRc, "File is the repository database: %.*s",
                          (int)nPath, zPath);
    }
    fsl__cx_scratchpad_yield(f, c1);
    fsl__cx_scratchpad_yield(f, c2);
    if(rc) return rc;
  }
  assert(!rc);
  while(true){
    /* Check the name against the repo's "manifest" setting and reject
       any filenames which that setting implies. */
    int manifestSetting = 0;
    fsl_ckout_manifest_setting(f, &manifestSetting);
    if(!manifestSetting) break;
    typedef struct {
      short flag;
      char const * fn;
    } MSetting;
    const MSetting M[] = {
    {FSL_MANIFEST_MAIN, "manifest"},
    {FSL_MANIFEST_UUID, "manifest.uuid"},
    {FSL_MANIFEST_TAGS, "manifest.tags"},
    {0,0}
    };
    fsl_buffer * const c1 = fsl__cx_scratchpad(f);
    if(f->ckout.dir){
      rc = fsl_ckout_filename_check(f, relativeToCwd, zPath, c1);
    }else{
      rc = fsl_file_canonical_name2("", zPath, c1, false);
    }
    if(rc) goto yield;
    char const * const z = fsl_buffer_cstr(c1);
    //MARKER(("Checking file against manifest setting 0x%03x: %s\n",
    //manifestSetting, z));
    for( MSetting const * m = &M[0]; m->fn; ++m ){
      if((m->flag & manifestSetting)
         && 0==fsl_strcmp(z, m->fn)){
        rc = fsl_cx_err_set(f, errRc,
                            "Filename is reserved due to the "
                            "'manifest' setting: %s",
                            m->fn);
        break;
      }
    }
    yield:
    fsl__cx_scratchpad_yield(f, c1);
    break;
  }
  return rc;
}

fsl_buffer * fsl__cx_scratchpad(fsl_cx * const f){
  fsl_buffer * rc = 0;
  int i = (f->scratchpads.next<FSL_CX_NSCRATCH)
    ? f->scratchpads.next : 0;
  for(; i < FSL_CX_NSCRATCH; ++i){
    if(!f->scratchpads.used[i]){
      rc = &f->scratchpads.buf[i];
      f->scratchpads.used[i] = true;
      ++f->scratchpads.next;
      //MARKER(("Doling out scratchpad[%d] w/ capacity=%d next=%d\n",
      //        i, (int)rc->capacity, f->scratchpads.next));
      break;
    }
  }
  if(!rc){
    assert(!"Fatal fsl_cx::scratchpads misuse.");
    fsl__fatal(FSL_RC_MISUSE,
              "Fatal internal fsl_cx::scratchpads misuse: "
              "too many unyielded buffer requests.");
  }else if(0!=rc->used){
    assert(!"Fatal fsl_cx::scratchpads misuse.");
    fsl__fatal(FSL_RC_MISUSE,
              "Fatal internal fsl_cx::scratchpads misuse: "
              "used buffer after yielding it.");
  }
  return rc;
}

void fsl__cx_scratchpad_yield(fsl_cx * const f, fsl_buffer * const b){
  int i;
  assert(b);
  for(i = 0; i < FSL_CX_NSCRATCH; ++i){
    if(b == &f->scratchpads.buf[i]){
      assert(f->scratchpads.next != i);
      assert(f->scratchpads.used[i] && "Scratchpad misuse.");
      f->scratchpads.used[i] = false;
      fsl_buffer_reuse(b);
      if(f->scratchpads.next>i) f->scratchpads.next = i;
      //MARKER(("Yielded scratchpad[%d] w/ capacity=%d, next=%d\n",
      //        i, (int)b->capacity, f->scratchpads.next));
      return;
    }
  }
  fsl__fatal(FSL_RC_MISUSE,
            "Fatal internal fsl_cx::scratchpads misuse: "
            "passed a non-scratchpad buffer.");
}


/** @internal

   Don't use this. Use fsl__ckout_rm_empty_dirs() instead.

   Attempts to remove empty directories from under a checkout,
   starting with tgtDir and working upwards until it either cannot
   remove one or it reaches the top of the checkout dir.

   The first argument must be the canonicalized absolute path to the
   checkout root. The second is the length of coRoot - if it's
   negative then fsl_strlen() is used to calculate it. The third must
   be the canonicalized absolute path to some directory under the
   checkout root. The contents of the buffer may, for efficiency's
   sake, be modified by this routine as it traverses the directory
   tree. It will never grow the buffer but may mutate its memory's
   contents.

   Returns the number of directories it is able to remove.

   Results are undefined if tgtDir is not an absolute path or does not
   have coRoot as its initial prefix.

   There are any number of valid reasons removal of a directory might
   fail, and this routine stops at the first one which does.
*/
static unsigned fsl__rm_empty_dirs(char const * const coRoot,
                                   fsl_int_t rootLen,
                                  fsl_buffer const * const tgtDir){
  if(rootLen<0) rootLen = (fsl_int_t)fsl_strlen(coRoot);
  char const * zAbs = fsl_buffer_cstr(tgtDir);
  char const * zCoDirPart = zAbs + rootLen;
  char * zEnd = fsl_buffer_str(tgtDir) + tgtDir->used - 1;
  unsigned rc = 0;
  assert(coRoot);
  if(0!=memcmp(coRoot, zAbs, (size_t)rootLen)){
    assert(!"Misuse of fsl__rm_empty_dirs()");
    return 0;
  }
  if(fsl_rmdir(zAbs)) return rc;
  ++rc;
  /** Now walk up each dir in the path and try to remove each,
      stopping when removal of one fails or we reach coRoot. */
  while(zEnd>zCoDirPart){
    for( ; zEnd>zCoDirPart && '/'!=*zEnd; --zEnd ){}
    if(zEnd==zCoDirPart) break;
    else if('/'==*zEnd){
      *zEnd = 0;
      assert(zEnd>zCoDirPart);
      if(fsl_rmdir(zAbs)) break;
      ++rc;
    }
  }
  return rc;
}

unsigned int fsl__ckout_rm_empty_dirs(fsl_cx * const f,
                                      fsl_buffer const * const tgtDir){
  int rc = f->ckout.dir ? 0 : FSL_RC_NOT_A_CKOUT;
  if(!rc){
    rc = fsl__rm_empty_dirs(f->ckout.dir, f->ckout.dirLen, tgtDir);
  }
  return rc;
}

int fsl__ckout_rm_empty_dirs_for_file(fsl_cx * const f, char const *zAbsPath){
  if(!fsl_is_rooted_in_ckout(f, zAbsPath)){
    assert(!"Internal API misuse!");
    return FSL_RC_MISUSE;
  }else{
    fsl_buffer * const p = fsl__cx_scratchpad(f);
    fsl_int_t const nAbs = (fsl_int_t)fsl_strlen(zAbsPath);
    int const rc = fsl_file_dirpart(zAbsPath, nAbs, p, false);
    if(!rc) fsl__rm_empty_dirs(f->ckout.dir, f->ckout.dirLen, p);
    fsl__cx_scratchpad_yield(f,p);
    return rc;
  }
}

int fsl_ckout_fingerprint_check(fsl_cx * const f){
  fsl_db * const db = fsl_cx_db_ckout(f);
  if(!db) return 0;
  int rc = 0;
  char const * zCkout = 0;
  char * zRepo = 0;
  fsl_id_t rcvCkout = 0;
  fsl_buffer * const buf = fsl__cx_scratchpad(f);
  rc = fsl_config_get_buffer(f, FSL_CONFDB_CKOUT, "fingerprint", buf);
  if(FSL_RC_NOT_FOUND==rc){
    /* Older checkout with no fingerprint. Assume it's okay. */
    rc = 0;
    goto end;
  }else if(rc){
    goto end;
  }
  zCkout = fsl_buffer_cstr(buf);
#if 0
  /* Inject a bogus byte for testing purposes */
  buf->mem[6] = 'x';
#endif
  rcvCkout = (fsl_id_t)atoi(zCkout);
  rc = fsl__repo_fingerprint_search(f, rcvCkout, &zRepo);
  switch(rc){
    case FSL_RC_NOT_FOUND: goto mismatch;
    case 0:
      assert(zRepo);
      if(fsl_strcmp(zRepo,zCkout)){
        goto mismatch;
      }
      break;
    default:
      break;
  }
  end:
  fsl__cx_scratchpad_yield(f, buf);
  fsl_free(zRepo);
  return rc;
  mismatch:
  rc = fsl_cx_err_set(f, FSL_RC_REPO_MISMATCH,
                      "Mismatch found between repo/checkout "
                      "fingerprints.");
  goto end;
}

bool fsl_cx_has_ckout(fsl_cx const * const f ){
  return f->ckout.dir ? true : false;
}

int fsl_cx_interruptv(fsl_cx * const f, int code, char const * fmt, va_list args){
  f->interrupted = code;
  if(code && NULL!=fmt){
    code = fsl_cx_err_setv(f, code, fmt, args);
  }
  return code;
}

int fsl_cx_interrupt(fsl_cx * const f, int code, char const * fmt, ...){
  int rc;
  va_list args;
  va_start(args,fmt);
  rc = fsl_cx_interruptv(f, code, fmt, args);
  va_end(args);
  return rc;
}

int fsl_cx_interrupted(fsl_cx const * const f){
  return f->interrupted;
}

#if 0
struct tm * fsl_cx_localtime( fsl_cx const * f, const time_t * clock ){
  if(!clock) return NULL;
  else if(!f) return localtime(clock);
  else return (f->flags & FSL_CX_F_LOCALTIME_GMT)
         ? gmtime(clock)
         : localtime(clock)
         ;
}

struct tm * fsl_localtime( const time_t * clock ){
  return fsl_cx_localtime(NULL, clock);
}

time_t fsl_cx_time_adj(fsl_cx const * f, time_t clock){
  struct tm * tm = fsl_cx_localtime(f, &clock);
  return tm ? mktime(tm) : 0;
}
#endif

#undef MARKER
#undef FSL_CX_NSCRATCH
/* end of file ./src/cx.c */
/* start of file ./src/db.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).

  *****************************************************************************
  This file contains the fsl_db_xxx() and fsl_stmt_xxx() parts of the
  API.
  
  Maintenance reminders:
  
  When returning dynamically allocated memory to the client, it needs
  to come from fsl_malloc(), as opposed to sqlite3_malloc(), so that
  it is legal to pass to fsl_free().
*/
#include <assert.h>
#include <stddef.h> /* NULL on linux */
#include <time.h> /* time() and friends */

/* Only for debugging */
#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)

#if 0
/**
    fsl_list_visitor_f() impl which requires that obj be NULL or
    a (fsl_stmt*), which it passed to fsl_stmt_finalize().
 */
static int fsl_list_v_fsl_stmt_finalize(void * obj, void * visitorState ){
  if(obj) fsl_stmt_finalize( (fsl_stmt*)obj );
  return 0;
}
#endif


int fsl__db_errcode(fsl_db * const db, int sqliteCode){
  int rc = 0;
  if(!sqliteCode) sqliteCode = sqlite3_errcode(db->dbh);
  switch(sqliteCode & 0xff){
    case SQLITE_ROW:
    case SQLITE_DONE:
    case SQLITE_OK: rc = 0; break;
    case SQLITE_NOMEM: rc = FSL_RC_OOM; break;
    case SQLITE_INTERRUPT: rc = FSL_RC_INTERRUPTED; break;
    case SQLITE_TOOBIG:
    case SQLITE_FULL:
    case SQLITE_NOLFS:
    case SQLITE_RANGE: rc = FSL_RC_RANGE; break;
    case SQLITE_NOTFOUND: rc = FSL_RC_NOT_FOUND; break;
    case SQLITE_PERM:
    case SQLITE_AUTH:
    case SQLITE_LOCKED: rc = FSL_RC_LOCKED; break;
    case SQLITE_READONLY: rc = FSL_RC_ACCESS; break;
    case SQLITE_CORRUPT: rc = FSL_RC_CONSISTENCY; break;
    case SQLITE_CANTOPEN:
    case SQLITE_IOERR:
      rc = FSL_RC_IO; break;
    default:
      //MARKER(("sqlite3_errcode()=0x%04x\n", rc));
      rc = FSL_RC_DB; break;
  }
  return rc
    ? fsl_error_set(&db->error, rc,
                    "sqlite3 error #%d: %s",
                    sqliteCode, sqlite3_errmsg(db->dbh))
    : (fsl_error_reset(&db->error), 0);
}

void fsl__db_clear_strings(fsl_db * const db, bool alsoErrorState ){
  fsl_free(db->filename);
  db->filename = NULL;
  fsl_free(db->name);
  db->name = NULL;
  if(alsoErrorState) fsl_error_clear(&db->error);
}

int fsl_db_err_get( fsl_db const * const db, char const ** msg, fsl_size_t * len ){
  return fsl_error_get(&db->error, msg, len);
}

fsl_db * fsl_stmt_db( fsl_stmt * const stmt ){
  return stmt->db;
}

char const * fsl_stmt_sql( fsl_stmt * const stmt, fsl_size_t * const len ){
  return fsl_buffer_cstr2(&stmt->sql, len);
}

char const * fsl_db_filename(fsl_db const * db, fsl_size_t * len){
  if(len && db->filename) *len = fsl_strlen(db->filename);
  return db->filename;
}

fsl_id_t fsl_db_last_insert_id(fsl_db * const db){
  return (db && db->dbh)
    ? (fsl_id_t)sqlite3_last_insert_rowid(db->dbh)
    : -1;
}

/**
    Cleans up db->beforeCommit and its contents.
 */
static void fsl_db_cleanup_beforeCommit( fsl_db * const db ){
  fsl_list_visit( &db->beforeCommit, -1, fsl_list_v_fsl_free, NULL );
  fsl_list_reserve(&db->beforeCommit, 0);
}


/**
   Immediately cleans up all cached statements (if any) and returns
   the number of statements cleaned up. It is illegal to call this
   while any of the cached statements are actively being used (have
   not been fsl_stmt_cached_yield()ed), and doing so will lead to
   undefined results if the statement(s) in question are used after
   this function completes.

   @see fsl_db_prepare_cached()
   @see fsl_stmt_cached_yield()
*/
static fsl_size_t fsl_db_stmt_cache_clear(fsl_db * const db){
  fsl_size_t rc = 0;
  if(db && db->cacheHead){
    fsl_stmt * st;
    fsl_stmt * next = 0;
    for( st = db->cacheHead; st; st = next, ++rc ){
      next = st->next;
      st->next = 0;
      fsl_stmt_finalize( st );
    }
    db->cacheHead = 0;
  }
  return rc;
}

void fsl_db_close( fsl_db * const db ){
  void const * const allocStamp = db->allocStamp;
  fsl_cx * const f = db->f;
  if(!db->dbh) return;
  fsl_db_stmt_cache_clear(db);
  if(db->f && db->f->dbMain==db){
    /*
      Horrible, horrible dependency, and only necessary if the
      fsl_cx API gets sloppy or repo/checkout/config DBs are
      otherwised closed improperly (i.e. not via the fsl_cx API).
    */
    assert(0 != db->role);
    f->dbMain = NULL;
  }
  while(db->beginCount>0){
    fsl_db_transaction_end(db, 1);
  }
  if(0!=db->openStatementCount){
    MARKER(("WARNING: %d open statement(s) left on db [%s].\n",
            (int)db->openStatementCount, db->filename));
  }
  if(db->dbh){
    sqlite3_close_v2(db->dbh);
    /* Ignore results in the style of "destructors
       may not throw.". */
  }
  fsl__db_clear_strings(db, true);
  fsl_db_cleanup_beforeCommit(db);
  fsl_buffer_clear(&db->cachePrepBuf);
  *db = fsl_db_empty;
  if(&fsl_db_empty == allocStamp){
    fsl_free( db );
  }else{
    db->allocStamp = allocStamp;
    db->f = f;
  }
  return;
}

void fsl_db_err_reset( fsl_db * const db ){
  if(db && (db->error.code||db->error.msg.used)){
    fsl_error_reset(&db->error);
  }
}


int fsl_db_attach(fsl_db * const db, const char *zDbName, const char *zLabel){
  return (db && db->dbh && zDbName && *zDbName && zLabel && *zLabel)
    ? fsl_db_exec(db, "ATTACH DATABASE %Q AS %s", zDbName, zLabel)
    : FSL_RC_MISUSE;
}
int fsl_db_detach(fsl_db * const db, const char *zLabel){
  return (db && db->dbh && zLabel && *zLabel)
    ? fsl_db_exec(db, "DETACH DATABASE %s /*%s()*/", zLabel, __func__)
    : FSL_RC_MISUSE;
}

char const * fsl_db_name(fsl_db const * const db){
  return db ? db->name : NULL;
}

/**
    Returns the db name for the given role.
 */
const char * fsl_db_role_name(fsl_dbrole_e r){
  switch(r){
    case FSL_DBROLE_CONFIG:
      return "cfg";
    case FSL_DBROLE_REPO:
      return "repo";
    case FSL_DBROLE_CKOUT:
      return "ckout";
    case FSL_DBROLE_MAIN:
      return "main";
    case FSL_DBROLE_TEMP:
      return "temp";
    case FSL_DBROLE_NONE:
    default:
      return NULL;
  }
}

char * fsl_db_julian_to_iso8601( fsl_db * const db, double j,
                                 bool msPrecision,
                                 bool localTime){
  char * s = NULL;
  fsl_stmt * st = NULL;
  if(db && db->dbh && (j>=0.0)){
    char const * sql;
    if(msPrecision){
      sql = localTime
        ? "SELECT strftime('%%Y-%%m-%%dT%%H:%%M:%%f',?, 'localtime')"
        : "SELECT strftime('%%Y-%%m-%%dT%%H:%%M:%%f',?)";
    }else{
      sql = localTime
        ? "SELECT strftime('%%Y-%%m-%%dT%%H:%%M:%%S',?, 'localtime')"
        : "SELECT strftime('%%Y-%%m-%%dT%%H:%%M:%%S',?)";
    }
    fsl_db_prepare_cached(db, &st, sql);
    if(st){
      fsl_stmt_bind_double( st, 1, j );
      if( FSL_RC_STEP_ROW==fsl_stmt_step(st) ){
        s = fsl_strdup(fsl_stmt_g_text(st, 0, NULL));
      }
      fsl_stmt_cached_yield(st);
    }
  }
  return s;
}

char * fsl_db_unix_to_iso8601( fsl_db * const db, fsl_time_t t, bool localTime ){
  char * s = NULL;
  fsl_stmt st = fsl_stmt_empty;
  if(db && db->dbh && (t>=0)){
    char const * sql = localTime
      ? "SELECT datetime(?, 'unixepoch', 'localtime')/*%s()*/"
      : "SELECT datetime(?, 'unixepoch')/*%s()*/"
      ;
    int const rc = fsl_db_prepare(db, &st, sql,__func__);
    if(!rc){
      fsl_stmt_bind_int64( &st, 1, t );
      if( FSL_RC_STEP_ROW==fsl_stmt_step(&st) ){
        fsl_size_t n = 0;
        char const * v = fsl_stmt_g_text(&st, 0, &n);
        s = (v&&n) ? fsl_strndup(v, (fsl_int_t)n) : NULL;
      }
      fsl_stmt_finalize(&st);
    }
  }
  return s;
}

enum fsl_stmt_flags_e {
/**
    fsl_stmt::flags bit indicating that fsl_db_preparev_cached() has
    doled out this statement, effectively locking it until
    fsl_stmt_cached_yield() is called to release it.
 */
FSL_STMT_F_CACHE_HELD = 0x01,

/**
   Propagates our intent to "statically" prepare a given statement
   through various internal API calls.
*/
FSL_STMT_F_PREP_CACHE = 0x10
};

int fsl_db_preparev( fsl_db  * const db, fsl_stmt * const tgt, char const * sql, va_list args ){
  if(!db || !tgt || !sql) return FSL_RC_MISUSE;
  else if(!db->dbh){
    return fsl_error_set(&db->error, FSL_RC_NOT_FOUND, "Db is not opened.");
  }else if(!*sql){
    return fsl_error_set(&db->error, FSL_RC_RANGE, "SQL is empty.");
  }else if(tgt->stmt){
    return fsl_error_set(&db->error, FSL_RC_ALREADY_EXISTS,
                         "Error: attempt to re-prepare "
                         "active statement.");
  }
  else{
    int rc;
    fsl_buffer buf = fsl_buffer_empty;
    fsl_stmt_t * liteStmt = NULL;
    rc = fsl_buffer_appendfv( &buf, sql, args );
    if(!rc){
#if 0
      /* Arguably improves readability of some queries.
         And breaks some caching uses. */
      fsl_simplify_sql_buffer(&buf);
#endif
      sql = fsl_buffer_cstr(&buf);
      if(!sql || !*sql){
        rc = fsl_error_set(&db->error, FSL_RC_RANGE,
                           "Input SQL is empty.");
      }else{
        /*
          Achtung: if sql==NULL here, or evaluates to a no-op
          (e.g. only comments or spaces), prepare_v2 succeeds but has
          a NULL liteStmt, which is why we handle the empty-SQL case
          specially. We don't want that specific behaviour leaking up
          through the API. Though doing so would arguably more correct
          in a generic API, for this particular API we have no reason
          to be able to handle empty SQL. Were we do let through
          through we'd have to add a flag to fsl_stmt to tell us
          whether it's really prepared or not, since checking of
          st->stmt would no longer be useful.
        */
        rc = sqlite3_prepare_v3(db->dbh, sql, (int)buf.used,
                                (FSL_STMT_F_PREP_CACHE & tgt->flags)
                                ? SQLITE_PREPARE_PERSISTENT
                                : 0,
                                &liteStmt, NULL);
        if(rc){
          rc = fsl_error_set(&db->error, FSL_RC_DB,
                             "Db statement preparation failed. "
                             "Error #%d: %s. SQL: %.*s",
                             rc, sqlite3_errmsg(db->dbh),
                             (int)buf.used, (char const *)buf.mem);
        }else if(!liteStmt){
          /* SQL was empty. In sqlite this is allowed, but this API will
             disallow this because it leads to headaches downstream.
          */
          rc = fsl_error_set(&db->error, FSL_RC_RANGE,
                             "Input SQL is empty.");
        }
      }
    }
    if(!rc){
      assert(liteStmt);
      ++db->openStatementCount;
      tgt->stmt = liteStmt;
      tgt->db = db;
      tgt->sql = buf /*transfer ownership*/;
      tgt->colCount = sqlite3_column_count(tgt->stmt);
      tgt->paramCount = sqlite3_bind_parameter_count(tgt->stmt);
    }else{
      assert(!liteStmt);
      fsl_buffer_clear(&buf);
      /* TODO: consider _copying_ the error state to db->f->error if
         db->f is not NULL. OTOH, we don't _always_ want to propagate
         a db error to the parent fossil context, and doing so here
         could lead to a "stale" error laying around downstream and
         getting evaluated later on (incorrectly) in a success
         context.
      */
    }
    return rc;
  }
}

int fsl_db_prepare( fsl_db * const db, fsl_stmt * const tgt, char const * sql, ... ){
  int rc;
  va_list args;
  va_start(args,sql);
  rc = fsl_db_preparev( db, tgt, sql, args );
  va_end(args);
  return rc;
}


int fsl_db_preparev_cached( fsl_db * const db, fsl_stmt ** rv,
                            char const * sql, va_list args ){
  int rc = 0;
  fsl_buffer * const buf = &db->cachePrepBuf;
  fsl_stmt * st = NULL;
  fsl_stmt * cs = NULL;
  if(!db || !rv || !sql) return FSL_RC_MISUSE;
  else if(!*sql) return FSL_RC_RANGE;
  if(!buf->capacity && fsl_buffer_reserve(buf, 1024*2)){
    return FSL_RC_OOM;
  }
  fsl_buffer_reuse(buf);
  rc = fsl_buffer_appendfv(buf, sql, args);
  if(rc) goto end;
  /**
     Hash buf's contents using a very primitive algo and stores the
     hash in buf->cursor. This is a blatant abuse of that member but
     we're otherwise not using it on these buffer instances. We use
     this to slightly speed up lookup of previously-cached entries and
     reduce the otherwise tremendous number of calls to
     fsl_buffer_compare() libfossil makes.
  */
  for(fsl_size_t i = 0; i < buf->used; ++i){
    //buf->cursor = (buf->cursor<<3) ^ buf->cursor ^ buf->mem[i];
    buf->cursor = 31 * buf->cursor + (buf->mem[i] * 307);
  }
  for( cs = db->cacheHead; cs; cs = cs->next ){
    if(cs->sql.cursor==buf->cursor/*hash value!*/
       && buf->used==cs->sql.used
       && 0==fsl_buffer_compare(buf, &cs->sql)){
      if(cs->flags & FSL_STMT_F_CACHE_HELD){
        rc = fsl_error_set(&db->error, FSL_RC_ACCESS,
                           "Cached statement is already in use. "
                           "Do not use cached statements if recursion "
                           "involving the statement is possible, and use "
                           "fsl_stmt_cached_yield() to release them "
                           "for further (re)use. SQL: %b",
                           &cs->sql);
        goto end;
      }
      cs->flags |= FSL_STMT_F_CACHE_HELD;
      ++cs->cachedHits;
      *rv = cs;
      goto end;
    }
  }
  st = fsl_stmt_malloc();
  if(!st){
    rc = FSL_RC_OOM;
    goto end;
  }
  st->flags |= FSL_STMT_F_PREP_CACHE;
  rc = fsl_db_prepare( db, st, "%b", buf );
  if(rc){
    fsl_free(st);
    st = 0;
  }else{
    st->sql.cursor = buf->cursor/*hash value!*/;
    st->next = db->cacheHead;
    st->role = db->role
      /* Pessimistic assumption for purposes of invalidating
         fsl__db_cached_clear_role(). */;
    db->cacheHead = st;
    st->flags = FSL_STMT_F_CACHE_HELD;
    *rv = st;
  }
  end:
  return rc;
}

int fsl_db_prepare_cached( fsl_db * const db, fsl_stmt ** st, char const * sql, ... ){
  int rc;
  va_list args;
  va_start(args,sql);
  rc = fsl_db_preparev_cached( db, st, sql, args );
  va_end(args);
  return rc;
}

int fsl_stmt_cached_yield( fsl_stmt * const st ){
  if(!st || !st->db || !st->stmt) return FSL_RC_MISUSE;
  else if(!(st->flags & FSL_STMT_F_CACHE_HELD)) {
    return fsl_error_set(&st->db->error, FSL_RC_MISUSE,
                         "fsl_stmt_cached_yield() was passed a "
                         "statement which is not marked as cached. "
                         "SQL: %b",
                         &st->sql);
  }else{
    fsl_stmt_reset(st);
    st->flags &= ~FSL_STMT_F_CACHE_HELD;
    return 0;
  }
}

int fsl_db_before_commitv( fsl_db * const db, char const * const sql,
                           va_list args ){
  int rc = 0;
  char * cp = NULL;
  if(!db || !sql) return FSL_RC_MISUSE;
  else if(!*sql) return FSL_RC_RANGE;
  cp = fsl_mprintfv(sql, args);
  if(cp){
    rc = fsl_list_append(&db->beforeCommit, cp);
    if(rc) fsl_free(cp);
  }else{
    rc = FSL_RC_OOM;
  }
  return rc;
}

int fsl_db_before_commit( fsl_db * const db, char const * const sql, ... ){
  int rc;
  va_list args;
  va_start(args,sql);
  rc = fsl_db_before_commitv( db, sql, args );
  va_end(args);
  return rc;
}

int fsl_stmt_finalize( fsl_stmt * const stmt ){
  if(!stmt) return FSL_RC_MISUSE;
  else{
    void const * allocStamp = stmt->allocStamp;
    fsl_db * const db = stmt->db;
    if(db){
      if(stmt->sql.mem){
        /* ^^^ b/c that buffer is set at the same time
           that openStatementCount is incremented.
        */
        --stmt->db->openStatementCount;
      }
      if(allocStamp && db->cacheHead){
        /* It _might_ be cached - let's remove it.
           We use allocStamp as a check here only
           because most statements allocated on the
           heap currently come from caching.
        */
        fsl_stmt * s;
        fsl_stmt * prev = 0;
        for( s = db->cacheHead; s; prev = s, s = s->next ){
          if(s == stmt){
            if(prev){
              assert(prev->next == s);
              prev->next = s->next;
            }else{
              assert(s == db->cacheHead);
              db->cacheHead = s->next;
            }
            s->next = 0;
            break;
          }
        }
      }
    }
    fsl_buffer_clear(&stmt->sql);
    if(stmt->stmt){
      sqlite3_finalize( stmt->stmt );
    }
    *stmt = fsl_stmt_empty;
    if(&fsl_stmt_empty==allocStamp){
      fsl_free(stmt);
    }else{
      stmt->allocStamp = allocStamp;
    }
    return 0;
  }
}

int fsl__db_cached_clear_role(fsl_db * const db, int role){
  int rc = 0;
  fsl_stmt * s;
  fsl_stmt * prev = 0;
  fsl_stmt * next = 0;
  for( s = db->cacheHead; s; s = next ){
    next = s->next;
    if(0!=role && 0==(s->role & role)){
      prev = s;
      continue;
    }
    else if(FSL_STMT_F_CACHE_HELD & s->flags){
      rc = fsl_error_set(&db->error, FSL_RC_MISUSE,
                         "Cannot clear cached SQL statement "
                         "for role #%d because it is currently "
                         "being held by a call to "
                         "fsl_db_preparev_cached(). SQL=%B",
                         &s->sql);
      break;
    }
    //MARKER(("Closing cached stmt: %s\n", fsl_buffer_cstr(&s->sql)));
    if(prev){
      prev->next = next;
    }else if(s==db->cacheHead){
      db->cacheHead = next;
    }
    s->next = 0;
    s->flags = 0;
    s->role = FSL_DBROLE_NONE;
    fsl_stmt_finalize(s);
    break;
  }
  return rc;
}

int fsl_stmt_step( fsl_stmt * const stmt ){
  if(!stmt->stmt) return FSL_RC_MISUSE;
  else{
    int const rc = sqlite3_step(stmt->stmt);
    assert(stmt->db);
    switch( rc ){
      case SQLITE_ROW:
        ++stmt->rowCount;
        return FSL_RC_STEP_ROW;
      case SQLITE_DONE:
        return FSL_RC_STEP_DONE;
      default:
        return fsl__db_errcode(stmt->db, rc);
    }
  }
}

int fsl_db_eachv( fsl_db * const db, fsl_stmt_each_f callback,
                  void * callbackState, char const * sql, va_list args ){
  if(!db->dbh || !callback || !sql) return FSL_RC_MISUSE;
  else if(!*sql) return FSL_RC_RANGE;
  else{
    fsl_stmt st = fsl_stmt_empty;
    int rc;
    rc = fsl_db_preparev( db, &st, sql, args );
    if(!rc){
      rc = fsl_stmt_each( &st, callback, callbackState );
      fsl_stmt_finalize( &st );
    }
    return rc;
  }
}

int fsl_db_each( fsl_db * const db, fsl_stmt_each_f callback,
                 void * callbackState, char const * sql, ... ){
  int rc;
  va_list args;
  va_start(args,sql);
  rc = fsl_db_eachv( db, callback, callbackState, sql, args );
  va_end(args);
  return rc;
}

int fsl_stmt_each( fsl_stmt * const stmt, fsl_stmt_each_f callback,
                   void * callbackState ){
  if(!callback) return FSL_RC_MISUSE;
  else{
    int strc;
    int rc = 0;
    bool doBreak = false;
    while( !doBreak && (FSL_RC_STEP_ROW == (strc=fsl_stmt_step(stmt)))){
      rc = callback( stmt, callbackState );
      switch(rc){
        case 0: continue;
        case FSL_RC_BREAK:
          rc = 0;
          /* fall through */
        default:
          doBreak = true;
          break;
      }
    }
    return rc
      ? rc
      : ((FSL_RC_STEP_ERROR==strc)
         ? FSL_RC_DB
         : 0);
  }
}

int fsl_stmt_reset2( fsl_stmt * const stmt, bool resetRowCounter ){
  if(!stmt->stmt || !stmt->db) return FSL_RC_MISUSE;
  else{
    int const rc = sqlite3_reset(stmt->stmt);
    if(resetRowCounter) stmt->rowCount = 0;
    assert(stmt->db);
    return rc
      ? fsl__db_errcode(stmt->db, rc)
      : 0;
  }
}

int fsl_stmt_reset( fsl_stmt * const stmt ){
  return fsl_stmt_reset2(stmt, 0);
}

int fsl_stmt_col_count( fsl_stmt const * const stmt ){
  return (!stmt || !stmt->stmt)
    ? -1
    : stmt->colCount
    ;
}

char const * fsl_stmt_col_name(fsl_stmt * const stmt, int index){
  return (stmt && stmt->stmt && (index>=0 && index<stmt->colCount))
    ? sqlite3_column_name(stmt->stmt, index)
    : NULL;
}

int fsl_stmt_param_count( fsl_stmt const * const stmt ){
  return (!stmt || !stmt->stmt)
    ? -1
    : stmt->paramCount;
}

int fsl_stmt_bind_fmtv( fsl_stmt * st, char const * fmt, va_list args ){
  int rc = 0, ndx;
  char const * pos = fmt;
  if(!fmt ||
     !(st && st->stmt && st->db && st->db->dbh)) return FSL_RC_MISUSE;
  else if(!*fmt) return FSL_RC_RANGE;
  for( ndx = 1; !rc && *pos; ++pos, ++ndx ){
    if(' '==*pos){
      --ndx;
      continue;
    }
    if(ndx > st->paramCount){
      rc = fsl_error_set(&st->db->error, FSL_RC_RANGE,
                         "Column index %d is out of bounds.", ndx);
      break;
    }
    switch(*pos){
      case '-':
        va_arg(args,void const *) /* skip arg */;
        rc = fsl_stmt_bind_null(st, ndx);
        break;
      case 'i':
        rc = fsl_stmt_bind_int32(st, ndx, va_arg(args,int32_t));
        break;
      case 'I':
        rc = fsl_stmt_bind_int64(st, ndx, va_arg(args,int64_t));
        break;
      case 'R':
        rc = fsl_stmt_bind_id(st, ndx, va_arg(args,fsl_id_t));
        break;
      case 'f':
        rc = fsl_stmt_bind_double(st, ndx, va_arg(args,double));
        break;
      case 's':{/* C-string as TEXT or NULL */
        char const * s = va_arg(args,char const *);
        rc = s
          ? fsl_stmt_bind_text(st, ndx, s, -1, false)
          : fsl_stmt_bind_null(st, ndx);
        break;
      }
      case 'S':{ /* C-string as BLOB or NULL */
        char const * s = va_arg(args,char const *);
        rc = s
          ? fsl_stmt_bind_blob(st, ndx, s, fsl_strlen(s), false)
          : fsl_stmt_bind_null(st, ndx);
        break;
      }
      case 'b':{ /* fsl_buffer as TEXT or NULL */
        fsl_buffer const * b = va_arg(args,fsl_buffer const *);
        rc = (b && b->mem)
          ? fsl_stmt_bind_text(st, ndx, (char const *)b->mem,
                               (fsl_int_t)b->used, false)
          : fsl_stmt_bind_null(st, ndx);
        break;
      }
      case 'B':{ /* fsl_buffer as BLOB or NULL */
        fsl_buffer const * b = va_arg(args,fsl_buffer const *);
        rc = (b && b->mem)
          ? fsl_stmt_bind_blob(st, ndx, b->mem, b->used, false)
          : fsl_stmt_bind_null(st, ndx);
        break;
      }
      default:
        rc = fsl_error_set(&st->db->error, FSL_RC_RANGE,
                           "Invalid format character: '%c'", *pos);
        break;
    }
  }
  return rc;
}

/**
   The elipsis counterpart of fsl_stmt_bind_fmtv().
*/
int fsl_stmt_bind_fmt( fsl_stmt * const st, char const * fmt, ... ){
  int rc;
  va_list args;
  va_start(args,fmt);
  rc = fsl_stmt_bind_fmtv(st, fmt, args);
  va_end(args);
  return rc;
}

int fsl_stmt_bind_stepv( fsl_stmt * const st, char const * fmt,
                         va_list args ){
  int rc;
  fsl_stmt_reset(st);
  rc = fsl_stmt_bind_fmtv(st, fmt, args);
  if(!rc){
    rc = fsl_stmt_step(st);
    switch(rc){
      case FSL_RC_STEP_DONE:
        rc = 0;
        fsl_stmt_reset(st);
        break;
      case FSL_RC_STEP_ROW:
        /* Don't reset() for ROW b/c that clears the column
           data! */
        break;
      default:
        rc = fsl_error_set(&st->db->error, rc,
                           "Error stepping statement: %s",
                           sqlite3_errmsg(st->db->dbh));
        break;
    }
  }
  return rc;
}

int fsl_stmt_bind_step( fsl_stmt * st, char const * fmt, ... ){
  int rc;
  va_list args;
  va_start(args,fmt);
  rc = fsl_stmt_bind_stepv(st, fmt, args);
  va_end(args);
  return rc;
}


#define BIND_PARAM_CHECK \
  if(!(stmt && stmt->stmt && stmt->db && stmt->db->dbh)) return FSL_RC_MISUSE; else
#define BIND_PARAM_CHECK2 BIND_PARAM_CHECK \
  if(ndx<1 || ndx>stmt->paramCount) return FSL_RC_RANGE; else
int fsl_stmt_bind_null( fsl_stmt * const stmt, int ndx ){
  BIND_PARAM_CHECK2 {
    int const rc = sqlite3_bind_null( stmt->stmt, ndx );
    return rc ? fsl__db_errcode(stmt->db, rc) : 0;
  }
}

int fsl_stmt_bind_int32( fsl_stmt * const stmt, int ndx, int32_t v ){
  BIND_PARAM_CHECK2 {
    int const rc = sqlite3_bind_int( stmt->stmt, ndx, (int)v );
    return rc ? fsl__db_errcode(stmt->db, rc) : 0;
  }
}

int fsl_stmt_bind_int64( fsl_stmt * const stmt, int ndx, int64_t v ){
  BIND_PARAM_CHECK2 {
    int const rc = sqlite3_bind_int64( stmt->stmt, ndx, (sqlite3_int64)v );
    return rc ? fsl__db_errcode(stmt->db, rc) : 0;
  }
}

int fsl_stmt_bind_id( fsl_stmt * const stmt, int ndx, fsl_id_t v ){
  BIND_PARAM_CHECK2 {
    int const rc = sqlite3_bind_int64( stmt->stmt, ndx, (sqlite3_int64)v );
    return rc ? fsl__db_errcode(stmt->db, rc) : 0;
  }
}

int fsl_stmt_bind_double( fsl_stmt * const stmt, int ndx, double v ){
  BIND_PARAM_CHECK2 {
    int const rc = sqlite3_bind_double( stmt->stmt, ndx, (double)v );
    return rc ? fsl__db_errcode(stmt->db, rc) : 0;
  }
}

int fsl_stmt_bind_blob( fsl_stmt * const stmt, int ndx, void const * src,
                        fsl_size_t len, bool makeCopy ){
  BIND_PARAM_CHECK2 {
    int rc;
    rc = sqlite3_bind_blob( stmt->stmt, ndx, src, (int)len,
                            makeCopy ? SQLITE_TRANSIENT : SQLITE_STATIC );
    return rc ? fsl__db_errcode(stmt->db, rc) : 0;
  }
}

int fsl_stmt_bind_text( fsl_stmt * const stmt, int ndx, char const * src,
                        fsl_int_t len, bool makeCopy ){
  BIND_PARAM_CHECK {
    int rc;
    if(len<0) len = fsl_strlen((char const *)src);
    rc = sqlite3_bind_text( stmt->stmt, ndx, src, len,
                            makeCopy ? SQLITE_TRANSIENT : SQLITE_STATIC );
    return rc ? fsl__db_errcode(stmt->db, rc) : 0;
  }
}

int fsl_stmt_bind_null_name( fsl_stmt * const stmt, char const * param ){
  BIND_PARAM_CHECK{
    return fsl_stmt_bind_null( stmt,
                               sqlite3_bind_parameter_index( stmt->stmt,
                                                             param) );
  }
}

int fsl_stmt_bind_int32_name( fsl_stmt * const stmt, char const * param, int32_t v ){
  BIND_PARAM_CHECK {
    return fsl_stmt_bind_int32( stmt,
                                sqlite3_bind_parameter_index( stmt->stmt,
                                                              param),
                                v);
  }
}

int fsl_stmt_bind_int64_name( fsl_stmt * const stmt, char const * param, int64_t v ){
  BIND_PARAM_CHECK {
    return fsl_stmt_bind_int64( stmt,
                                sqlite3_bind_parameter_index( stmt->stmt,
                                                              param),
                                v);
  }
}

int fsl_stmt_bind_id_name( fsl_stmt * const stmt, char const * param, fsl_id_t v ){
  BIND_PARAM_CHECK {
    return fsl_stmt_bind_id( stmt,
                             sqlite3_bind_parameter_index( stmt->stmt,
                                                           param),
                             v);
  }
}

int fsl_stmt_bind_double_name( fsl_stmt * const stmt, char const * param, double v ){
  BIND_PARAM_CHECK {
    return fsl_stmt_bind_double( stmt,
                                 sqlite3_bind_parameter_index( stmt->stmt,
                                                               param),
                                 v);
  }
}

int fsl_stmt_bind_text_name( fsl_stmt * const stmt, char const * param,
                             char const * v, fsl_int_t n,
                             bool makeCopy ){
  BIND_PARAM_CHECK {
    return fsl_stmt_bind_text(stmt,
                              sqlite3_bind_parameter_index( stmt->stmt,
                                                            param),
                              v, n, makeCopy);
  }
}

int fsl_stmt_bind_blob_name( fsl_stmt * const stmt, char const * param,
                             void const * v, fsl_int_t len,
                             bool makeCopy ){
  BIND_PARAM_CHECK {
    return fsl_stmt_bind_blob(stmt,
                         sqlite3_bind_parameter_index( stmt->stmt,
                                                       param),
                              v, len, makeCopy);
  }
}

int fsl_stmt_param_index( fsl_stmt * const stmt, char const * const param){
  return (stmt && stmt->stmt)
    ? sqlite3_bind_parameter_index( stmt->stmt, param)
    : -1;
}

#undef BIND_PARAM_CHECK
#undef BIND_PARAM_CHECK2

#define GET_CHECK if(!stmt->colCount) return FSL_RC_MISUSE; \
  else if((ndx<0) || (ndx>=stmt->colCount)) return FSL_RC_RANGE; else

int fsl_stmt_get_int32( fsl_stmt * const stmt, int ndx, int32_t * v ){
  GET_CHECK {
    if(v) *v = (int32_t)sqlite3_column_int(stmt->stmt, ndx);
    return 0;
  }
}
int fsl_stmt_get_int64( fsl_stmt * const stmt, int ndx, int64_t * v ){
  GET_CHECK {
    if(v) *v = (int64_t)sqlite3_column_int64(stmt->stmt, ndx);
    return 0;
  }
}

int fsl_stmt_get_double( fsl_stmt * const stmt, int ndx, double * v ){
  GET_CHECK {
    if(v) *v = (double)sqlite3_column_double(stmt->stmt, ndx);
    return 0;
  }
}

int fsl_stmt_get_id( fsl_stmt * const stmt, int ndx, fsl_id_t * v ){
  GET_CHECK {
    if(v) *v = (4==sizeof(fsl_id_t))
      ? (fsl_id_t)sqlite3_column_int(stmt->stmt, ndx)
      : (fsl_id_t)sqlite3_column_int64(stmt->stmt, ndx);
    return 0;
  }
}

int fsl_stmt_get_text( fsl_stmt * const stmt, int ndx, char const **out,
                       fsl_size_t * outLen ){
  GET_CHECK {
    unsigned char const * t = (out || outLen)
      ? sqlite3_column_text(stmt->stmt, ndx)
      : NULL;
    if(out) *out = (char const *)t;
    if(outLen){
      int const x = sqlite3_column_bytes(stmt->stmt, ndx);
      *outLen = (x>0) ? (fsl_size_t)x : 0;
    }
    return t ? 0 : fsl__db_errcode(stmt->db, 0);
  }
}

int fsl_stmt_get_blob( fsl_stmt * const stmt, int ndx, void const **out,
                       fsl_size_t * outLen ){
  GET_CHECK {
    void const * t = (out || outLen)
      ? sqlite3_column_blob(stmt->stmt, ndx)
      : NULL;
    if(out) *out = t;
    if(outLen){
      if(!t) *outLen = 0;
      else{
        int sz = sqlite3_column_bytes(stmt->stmt, ndx);
        *outLen = (sz>=0) ? (fsl_size_t)sz : 0;
      }
    }
    return t ? 0 : fsl__db_errcode(stmt->db, 0);
  }
}

#undef GET_CHECK

fsl_id_t fsl_stmt_g_id( fsl_stmt * const stmt, int index ){
  fsl_id_t rv = -1;
  fsl_stmt_get_id(stmt, index, &rv);
  return rv;
}
int32_t fsl_stmt_g_int32( fsl_stmt * const stmt, int index ){
  int32_t rv = 0;
  fsl_stmt_get_int32(stmt, index, &rv);
  return rv;
}
int64_t fsl_stmt_g_int64( fsl_stmt * const stmt, int index ){
  int64_t rv = 0;
  fsl_stmt_get_int64(stmt, index, &rv);
  return rv;
}
double fsl_stmt_g_double( fsl_stmt * const stmt, int index ){
  double rv = 0;
  fsl_stmt_get_double(stmt, index, &rv);
  return rv;
}

char const * fsl_stmt_g_text( fsl_stmt * const stmt, int index,
                              fsl_size_t * outLen ){
  char const * rv = NULL;
  fsl_stmt_get_text(stmt, index, &rv, outLen);
  return rv;
}


/**
   This sqlite3_trace_v2() callback outputs tracing info using
   fsl_fprintf((FILE*)c,...).  Defaults to stdout if c is NULL.
*/
static int fsl__db_sq3TraceV2(unsigned t,void*c,void*p,void*x){
  static unsigned int counter = 0;
  if(c ||p){/*unused*/}
  switch(t){
    case SQLITE_TRACE_STMT:{
      char const * zSql = (char const *)x;
      char * zExp = zSql
        ? sqlite3_expanded_sql((sqlite3_stmt*)p)
        : NULL;
      fsl_fprintf(c ? (FILE*)c : stdout,
                  "SQL TRACE #%u: %s\n",
                  ++counter,
                  zExp ? zExp :
                  (zSql ? zSql : "(NO SQL?)"));
      sqlite3_free(zExp);
      break;
    }
    default:
      break;
  }
  return 0;
}

fsl_db * fsl_db_malloc(){
  fsl_db * rc = (fsl_db *)fsl_malloc(sizeof(fsl_db));
  if(rc){
    *rc = fsl_db_empty;
    rc->allocStamp = &fsl_db_empty;
  }
  return rc;
}

fsl_stmt * fsl_stmt_malloc(){
  fsl_stmt * rc = (fsl_stmt *)fsl_malloc(sizeof(fsl_stmt));
  if(rc){
    *rc = fsl_stmt_empty;
    rc->allocStamp = &fsl_stmt_empty;
  }
  return rc;
}

/**
   Callback for use with sqlite3_commit_hook(). The argument must be a
   (fsl_db*). This function returns 0 only if it surmises that
   fsl_db_transaction_end() triggered the COMMIT. On error it might
   assert() or abort() the application, so this really is just a
   sanity check for something which "must not happen."
*/
static int fsl_db_verify_begin_was_not_called(void * db_fsl){
  fsl_db * const db = (fsl_db *)db_fsl;
  assert(db && "What else could it be?");
  assert(db->dbh && "Else we can't have been called by sqlite3, could we have?");
  if(db->beginCount>0){
    fsl__fatal(FSL_RC_MISUSE,"SQL: COMMIT was called from "
              "outside of fsl_db_transaction_end() while a "
              "fsl_db_transaction_begin()-started transaction "
              "is pending.");
    return 2;
  }
  return 0;
}

int fsl_db_open( fsl_db * const db, char const * dbFile,
                 int openFlags ){
  int rc;
  fsl_dbh_t * dbh = NULL;
  int isMem = 0;
  if(!db || !dbFile) return FSL_RC_MISUSE;
  else if(db->dbh) return FSL_RC_MISUSE;
  else if(!(isMem = (!*dbFile || 0==fsl_strcmp(":memory:", dbFile)))
          && !(FSL_OPEN_F_CREATE & openFlags)
          && fsl_file_access(dbFile, 0)){
    return fsl_error_set(&db->error, FSL_RC_NOT_FOUND,
                         "DB file not found: %s", dbFile);
  }
  else{
    int sOpenFlags = 0;
    if(FSL_OPEN_F_RO & openFlags){
      sOpenFlags |= SQLITE_OPEN_READONLY;
    }else{
      if(FSL_OPEN_F_RW & openFlags){
        sOpenFlags |= SQLITE_OPEN_READWRITE;
      }
      if(FSL_OPEN_F_CREATE & openFlags){
        sOpenFlags |= SQLITE_OPEN_READWRITE | SQLITE_OPEN_CREATE;
      }
      if(!sOpenFlags) sOpenFlags = SQLITE_OPEN_READONLY;
    }
    rc = sqlite3_open_v2( dbFile, &dbh, sOpenFlags, NULL );
    if(rc){
      if(dbh){
        /* By some complete coincidence, FSL_RC_DB==SQLITE_CANTOPEN. */
        rc = fsl_error_set(&db->error, FSL_RC_DB,
                           "Opening db file [%s] failed with "
                           "sqlite code #%d: %s",
                           dbFile, rc, sqlite3_errmsg(dbh));
      }else{
        rc = fsl_error_set(&db->error, FSL_RC_DB,
                           "Opening db file [%s] failed with "
                           "sqlite code #%d",
                           dbFile, rc);
      }
      /* MARKER(("Error msg: %s\n", (char const *)db->error.msg.mem)); */
      goto end;
    }else{
      assert(!db->filename);
      if(!*dbFile || ':'==*dbFile){
        /* assume "" or ":memory:" or some such: don't canonicalize it,
           but copy it nonetheless for consistency. */
        db->filename = fsl_strdup(dbFile);
      }else{
        fsl_buffer tmp = fsl_buffer_empty;
        rc = fsl_file_canonical_name(dbFile, &tmp, 0);
        if(!rc){
          db->filename = (char *)tmp.mem
            /* transfering ownership */;
        }else if(tmp.mem){
          fsl_buffer_clear(&tmp);
        }
      }
      if(rc){
        goto end;
      }else if(!db->filename){
        rc = FSL_RC_OOM;
        goto end;
      }
    }
    db->dbh = dbh;
    sqlite3_extended_result_codes(dbh, 1);
    sqlite3_commit_hook(dbh, fsl_db_verify_begin_was_not_called, db);
    if(FSL_OPEN_F_TRACE_SQL & openFlags){
      fsl_db_sqltrace_enable(db, stdout);
    }
  }
  end:
  if(rc){
#if 1
    /* This is arguable... */
    if(db->f && db->error.code && !db->f->error.code){
      /* COPY db's error state as f's. */
      fsl_error_copy( &db->error, &db->f->error );
    }
#endif
    if(dbh){
      sqlite3_close(dbh);
      db->dbh = NULL;
    }
  }else{
    assert(db->dbh);
  }
  return rc;
}


static int fsl__db_err_not_opened(fsl_db * const db){
  return fsl_error_set(&db->error, FSL_RC_MISUSE,
                       "DB is not opened.");
}
static int fsl__db_err_sql_empty(fsl_db * const db){
  return fsl_error_set(&db->error, FSL_RC_MISUSE,
                       "Empty SQL is not permitted.");
 }

int fsl_db_exec_multiv( fsl_db * const db, const char * sql, va_list args){
  if(!db->dbh) return fsl__db_err_not_opened(db);
  else if(!sql || !*sql) return fsl__db_err_sql_empty(db);
  else{
    fsl_buffer buf = fsl_buffer_empty;
    int rc = 0;
    char const * z;
    char const * zEnd = NULL;
    rc = fsl_buffer_appendfv( &buf, sql, args );
    if(rc){
      fsl_buffer_clear(&buf);
      return rc;
    }
    z = fsl_buffer_cstr(&buf);
    while( (SQLITE_OK==rc) && *z ){
      fsl_stmt_t * pStmt = NULL;
      rc = sqlite3_prepare_v2(db->dbh, z, buf.used, &pStmt, &zEnd);
      if( SQLITE_OK != rc ){
        rc = fsl__db_errcode(db, rc);
        break;
      }
      if(pStmt){
        while( SQLITE_ROW == sqlite3_step(pStmt) ){}
        rc = sqlite3_finalize(pStmt);
        if(rc) rc = fsl__db_errcode(db, rc);
      }
      buf.used -= (zEnd-z);
      z = zEnd;
    }
    fsl_buffer_reserve(&buf, 0);
    return rc;
  }
}

int fsl_db_exec_multi( fsl_db * const db, const char * sql, ...){
  if(!db->dbh) return fsl__db_err_not_opened(db);
  else if(!sql || !*sql) return fsl__db_err_sql_empty(db);
  else{
    int rc;
    va_list args;
    va_start(args,sql);
    rc = fsl_db_exec_multiv( db, sql, args );
    va_end(args);
    return rc;
  }
}

int fsl_db_execv( fsl_db * const db, const char * sql, va_list args){
  if(!db->dbh) return fsl__db_err_not_opened(db);
  else if(!sql || !*sql) return fsl__db_err_sql_empty(db);
  else{
    fsl_stmt st = fsl_stmt_empty;
    int rc = 0;
    rc = fsl_db_preparev( db, &st, sql, args );
    if(0==rc){
      //while(FSL_RC_STEP_ROW == (rc=fsl_stmt_step(&st))){}
      //^^^ why did we historically do this instead of:
      rc = fsl_stmt_step( &st );
      fsl_stmt_finalize(&st);
    }
    switch(rc){
      case FSL_RC_STEP_DONE:
      case FSL_RC_STEP_ROW: rc = 0; break;
      default: break;
    }
    return rc;
  }
}

int fsl_db_exec( fsl_db * const db, const char * sql, ...){
  if(!db->dbh) return fsl__db_err_not_opened(db);
  else if(!sql || !*sql) return fsl__db_err_sql_empty(db);
  else{
    int rc;
    va_list args;
    va_start(args,sql);
    rc = fsl_db_execv( db, sql, args );
    va_end(args);
    return rc;
  }
}

int fsl_db_changes_recent(fsl_db * const db){
  return db->dbh
    ? sqlite3_changes(db->dbh)
    : 0;
}

int fsl_db_changes_total(fsl_db * const db){
  return db->dbh
    ? sqlite3_total_changes(db->dbh)
    : 0;
}

/**
    Sets db->priorChanges to sqlite3_total_changes(db->dbh).
*/
static void fsl_db_reset_change_count(fsl_db * const db){
  db->priorChanges = sqlite3_total_changes(db->dbh);
}

int fsl_db_transaction_begin(fsl_db * const db){
  if(!db || !db->dbh) return FSL_RC_MISUSE;
  else {
    int rc = (0==db->beginCount)
      ? fsl_db_exec(db,"BEGIN TRANSACTION")
      : 0;
    if(!rc){
      if(1 == ++db->beginCount){
        fsl_db_reset_change_count(db);
      }
    }
    return rc;
  }
}

int fsl_db_transaction_level(fsl_db * const db){
  return db->doRollback ? -db->beginCount : db->beginCount;
}

int fsl_db_transaction_commit(fsl_db * const db){
  return db->dbh
    ? fsl_db_transaction_end(db, 0)
    : FSL_RC_MISUSE;
}

int fsl_db_transaction_rollback(fsl_db * const db){
  return db->dbh
    ? fsl_db_transaction_end(db, 1)
    : FSL_RC_MISUSE;
}

int fsl_db_rollback_force( fsl_db * const db ){
  if(!db->dbh){
    return fsl__db_err_not_opened(db);
  }else{
    int rc;
    db->beginCount = 0;
    fsl_db_cleanup_beforeCommit(db);
    rc = fsl_db_exec(db, "ROLLBACK");
    fsl_db_reset_change_count(db);
    return rc;
  }
}

int fsl_db_transaction_end(fsl_db * const db, bool doRollback){
  int rc = 0;
  if(!db->dbh){
    return fsl__db_err_not_opened(db);
  }else if (db->beginCount<=0){
    return fsl_error_set(&db->error, FSL_RC_RANGE,
                         "No transaction is active.");
  }
  if(doRollback) ++db->doRollback
    /* ACHTUNG: note that db->dbRollback is set before
       continuing so that if we return due to a non-0 beginCount
       that the rollback flag propagates through the
       transaction's stack.
    */;
  if(--db->beginCount > 0) return 0;
  assert(0==db->beginCount && "The commit-hook check relies on this.");
  assert(db->doRollback>=0);
  int const changeCount =
    sqlite3_total_changes(db->dbh) - db->priorChanges;
  if(0==db->doRollback){
    if(changeCount>0){
      /* Execute before-commit hooks and leaf checks */
      fsl_size_t x = 0;
      for( ; !rc && (x < db->beforeCommit.used); ++x ){
        char const * sql = (char const *)db->beforeCommit.list[x];
        /* MARKER(("Running before-commit code: [%s]\n", sql)); */
        if(sql) rc = fsl_db_exec_multi( db, "%s", sql );
      }
      if(!rc && db->f && (FSL_DBROLE_REPO & db->role)){
        /*
          i don't like this one bit - this is low-level SCM
          functionality in an otherwise generic routine. Maybe we need
          fsl_cx_transaction_begin/end() instead.

          Much later: we have that routine now but will need to replace
          all relevant calls to fsl_db_transaction_begin()/end() with
          those routines before we can consider moving this there.
        */
        rc = fsl__repo_leafdo_pending_checks(db->f);
        if(!rc && db->f->cache.toVerify.used){
          rc = fsl__repo_verify_at_commit(db->f);
        }else{
          fsl_repo_verify_cancel(db->f);
        }
      }
      db->doRollback = rc ? 1 : 0;
    }
  }
  if(db->doRollback && db->f && changeCount>0){
    /**
       If a rollback is underway (from a transaction in which data was
       written), certain fsl_cx caches might be referring to record
       IDs which were injected as part of the being-rolled-back
       transaction. The only(?) reasonably sane way to deal with that
       is to flush all relevant caches. It is unfortunate that this
       bit is in the db class, as opposed to the fsl_cx class, but we
       currently have no hook which would allow us to trigger this
       from that class.
    */
    fsl_cx_caches_reset(db->f);
  }
  fsl_db_cleanup_beforeCommit(db);
  fsl_db_reset_change_count(db);
  rc = fsl_db_exec(db, db->doRollback ? "ROLLBACK" : "COMMIT");
  if(db->doRollback && db->f && changeCount>0 && db->f->ckout.rid>0){
    int const rc2 = fsl__ckout_version_fetch(db->f)
      /*Else it might be out of sync, leading to chaos.*/;
    if(0==rc && rc2!=0) rc = rc2;
  }
  db->doRollback = 0;
  return rc;
}

int fsl_db_get_int32v( fsl_db * const db, int32_t * rv,
                       char const * sql, va_list args){
  /* Potential fixme: the fsl_db_get_XXX() funcs are 95%
     code duplicates. We "could" replace these with a macro
     or supermacro, though the latter would be problematic
     in the context of an amalgamation build.
  */
  if(!db || !db->dbh || !rv || !sql || !*sql) return FSL_RC_MISUSE;
  else{
    fsl_stmt st = fsl_stmt_empty;
    int rc = 0;
    rc = fsl_db_preparev( db, &st, sql, args );
    if(rc) return rc;
    rc = fsl_stmt_step( &st );
    switch(rc){
      case FSL_RC_STEP_ROW:
        *rv = sqlite3_column_int(st.stmt, 0);
        /* Fall through */
      case FSL_RC_STEP_DONE:
        rc = 0;
        break;
      default:
        assert(FSL_RC_STEP_ERROR==rc);
        break;
    }
    fsl_stmt_finalize(&st);
    return rc;
  }
}

int fsl_db_get_int32( fsl_db * const db, int32_t * rv,
                      char const * sql, ... ){
  int rc;
  va_list args;
  va_start(args,sql);
  rc = fsl_db_get_int32v(db, rv, sql, args);
  va_end(args);
  return rc;
}

int fsl_db_get_int64v( fsl_db * const db, int64_t * rv,
                       char const * sql, va_list args){
  if(!db || !db->dbh || !rv || !sql || !*sql) return FSL_RC_MISUSE;
  else{
    fsl_stmt st = fsl_stmt_empty;
    int rc = 0;
    rc = fsl_db_preparev( db, &st, sql, args );
    if(rc) return rc;
    rc = fsl_stmt_step( &st );
    switch(rc){
      case FSL_RC_STEP_ROW:
        *rv = sqlite3_column_int64(st.stmt, 0);
        /* Fall through */
      case FSL_RC_STEP_DONE:
        rc = 0;
        break;
      default:
        assert(FSL_RC_STEP_ERROR==rc);
        break;
    }
    fsl_stmt_finalize(&st);
    return rc;
  }
}

int fsl_db_get_int64( fsl_db * const db, int64_t * rv,
                      char const * sql, ... ){
  int rc;
  va_list args;
  va_start(args,sql);
  rc = fsl_db_get_int64v(db, rv, sql, args);
  va_end(args);
  return rc;
}


int fsl_db_get_idv( fsl_db * const db, fsl_id_t * rv,
                       char const * sql, va_list args){
  if(!db || !db->dbh || !rv || !sql || !*sql) return FSL_RC_MISUSE;
  else{
    fsl_stmt st = fsl_stmt_empty;
    int rc = 0;
    rc = fsl_db_preparev( db, &st, sql, args );
    if(rc) return rc;
    rc = fsl_stmt_step( &st );
    switch(rc){
      case FSL_RC_STEP_ROW:
        *rv = (fsl_id_t)sqlite3_column_int64(st.stmt, 0);
        /* Fall through */
      case FSL_RC_STEP_DONE:
        rc = 0;
        break;
      default:
        break;
    }
    fsl_stmt_finalize(&st);
    return rc;
  }
}

int fsl_db_get_id( fsl_db * const db, fsl_id_t * rv,
                      char const * sql, ... ){
  int rc;
  va_list args;
  va_start(args,sql);
  rc = fsl_db_get_idv(db, rv, sql, args);
  va_end(args);
  return rc;
}


int fsl_db_get_sizev( fsl_db * const db, fsl_size_t * rv,
                      char const * sql, va_list args){
  if(!db || !db->dbh || !rv || !sql || !*sql) return FSL_RC_MISUSE;
  else{
    fsl_stmt st = fsl_stmt_empty;
    int rc = 0;
    rc = fsl_db_preparev( db, &st, sql, args );
    if(rc) return rc;
    rc = fsl_stmt_step( &st );
    switch(rc){
      case FSL_RC_STEP_ROW:{
        sqlite3_int64 const i = sqlite3_column_int64(st.stmt, 0);
        if(i<0){
          rc = FSL_RC_RANGE;
          break;
        }
        *rv = (fsl_size_t)i;
        rc = 0;
        break;
      }
      case FSL_RC_STEP_DONE:
        rc = 0;
        break;
      default:
        assert(FSL_RC_STEP_ERROR==rc);
        break;
    }
    fsl_stmt_finalize(&st);
    return rc;
  }
}

int fsl_db_get_size( fsl_db * const db, fsl_size_t * rv,
                      char const * sql, ... ){
  int rc;
  va_list args;
  va_start(args,sql);
  rc = fsl_db_get_sizev(db, rv, sql, args);
  va_end(args);
  return rc;
}


int fsl_db_get_doublev( fsl_db * const db, double * rv,
                       char const * sql, va_list args){
  if(!db || !db->dbh || !rv || !sql || !*sql) return FSL_RC_MISUSE;
  else{
    fsl_stmt st = fsl_stmt_empty;
    int rc = 0;
    rc = fsl_db_preparev( db, &st, sql, args );
    if(rc) return rc;
    rc = fsl_stmt_step( &st );
    switch(rc){
      case FSL_RC_STEP_ROW:
        *rv = sqlite3_column_double(st.stmt, 0);
        /* Fall through */
      case FSL_RC_STEP_DONE:
        rc = 0;
        break;
      default:
        assert(FSL_RC_STEP_ERROR==rc);
        break;
    }
    fsl_stmt_finalize(&st);
    return rc;
  }
}

int fsl_db_get_double( fsl_db * const db, double * rv,
                      char const * sql,
                      ... ){
  int rc;
  va_list args;
  va_start(args,sql);
  rc = fsl_db_get_doublev(db, rv, sql, args);
  va_end(args);
  return rc;
}


int fsl_db_get_textv( fsl_db * const db, char ** rv,
                      fsl_size_t *rvLen,
                      char const * sql, va_list args){
  if(!db || !db->dbh || !rv || !sql || !*sql) return FSL_RC_MISUSE;
  else{
    fsl_stmt st = fsl_stmt_empty;
    int rc = 0;
    rc = fsl_db_preparev( db, &st, sql, args );
    if(rc) return rc;
    rc = fsl_stmt_step( &st );
    switch(rc){
      case FSL_RC_STEP_ROW:{
        char const * str = (char const *)sqlite3_column_text(st.stmt, 0);
        int const len = sqlite3_column_bytes(st.stmt,0);
        if(!str){
          *rv = NULL;
          if(rvLen) *rvLen = 0;
        }else{
          char * x = fsl_strndup(str, len);
          if(!x){
            rc = FSL_RC_OOM;
          }else{
            *rv = x;
            if(rvLen) *rvLen = (fsl_size_t)len;
            rc = 0;
          }
        }
        break;
      }
      case FSL_RC_STEP_DONE:
        *rv = NULL;
        if(rvLen) *rvLen = 0;
        rc = 0;
        break;
      default:
        assert(FSL_RC_STEP_ERROR==rc);
        break;
    }
    fsl_stmt_finalize(&st);
    return rc;
  }
}

int fsl_db_get_text( fsl_db * const db, char ** rv,
                     fsl_size_t * rvLen,
                     char const * sql, ... ){
  int rc;
  va_list args;
  va_start(args,sql);
  rc = fsl_db_get_textv(db, rv, rvLen, sql, args);
  va_end(args);
  return rc;
}

int fsl_db_get_blobv( fsl_db * const db, void ** rv,
                      fsl_size_t *rvLen,
                      char const * sql, va_list args){
  if(!db || !db->dbh || !rv || !sql || !*sql) return FSL_RC_MISUSE;
  else{
    fsl_stmt st = fsl_stmt_empty;
    int rc = 0;
    rc = fsl_db_preparev( db, &st, sql, args );
    if(rc) return rc;
    rc = fsl_stmt_step( &st );
    switch(rc){
      case FSL_RC_STEP_ROW:{
        fsl_buffer buf = fsl_buffer_empty;
        void const * str = sqlite3_column_blob(st.stmt, 0);
        int const len = sqlite3_column_bytes(st.stmt,0);
        if(!str){
          *rv = NULL;
          if(rvLen) *rvLen = 0;
        }else{
          rc = fsl_buffer_append(&buf, str, len);
          if(!rc){
            *rv = buf.mem;
            if(rvLen) *rvLen = buf.used;
          }
        }
        break;
      }
      case FSL_RC_STEP_DONE:
        *rv = NULL;
        if(rvLen) *rvLen = 0;
        rc = 0;
        break;
      default:
        assert(FSL_RC_STEP_ERROR==rc);
        break;
    }
    fsl_stmt_finalize(&st);
    return rc;
  }
}

int fsl_db_get_blob( fsl_db * const db, void ** rv,
                     fsl_size_t * rvLen,
                     char const * sql, ... ){
  int rc;
  va_list args;
  va_start(args,sql);
  rc = fsl_db_get_blobv(db, rv, rvLen, sql, args);
  va_end(args);
  return rc;
}

int fsl_db_get_bufferv( fsl_db * const db, fsl_buffer * const b,
                        bool asBlob, char const * sql,
                        va_list args){
  if(!sql || !*sql) return FSL_RC_MISUSE;
  else{
    fsl_stmt st = fsl_stmt_empty;
    int rc = 0;
    rc = fsl_db_preparev( db, &st, sql, args );
    if(rc) return rc;
    rc = fsl_stmt_step( &st );
    switch(rc){
      case FSL_RC_STEP_ROW:{
        void const * str = asBlob
          ? sqlite3_column_blob(st.stmt, 0)
          : (void const *)sqlite3_column_text(st.stmt, 0);
        int const len = sqlite3_column_bytes(st.stmt,0);
        if(len && !str){
          rc = FSL_RC_OOM;
        }else{
          rc = 0;
          rc = fsl_buffer_append( b, str, len );
        }
        break;
      }
      case FSL_RC_STEP_DONE:
        rc = 0;
        break;
      default:
        break;
    }
    fsl_stmt_finalize(&st);
    return rc;
  }
}

int fsl_db_get_buffer( fsl_db * const db, fsl_buffer * const b,
                       bool asBlob,
                       char const * sql, ... ){
  int rc;
  va_list args;
  va_start(args,sql);
  rc = fsl_db_get_bufferv(db, b, asBlob, sql, args);
  va_end(args);
  return rc;
}

int32_t fsl_db_g_int32( fsl_db * const db, int32_t dflt,
                        char const * sql, ... ){
  int32_t rv = dflt;
  va_list args;
  va_start(args,sql);
  fsl_db_get_int32v(db, &rv, sql, args);
  va_end(args);
  return rv;
}

int64_t fsl_db_g_int64( fsl_db * const db, int64_t dflt,
                            char const * sql,
                            ... ){
  int64_t rv = dflt;
  va_list args;
  va_start(args,sql);
  fsl_db_get_int64v(db, &rv, sql, args);
  va_end(args);
  return rv;
}

fsl_id_t fsl_db_g_id( fsl_db * const db, fsl_id_t dflt,
                            char const * sql,
                            ... ){
  fsl_id_t rv = dflt;
  va_list args;
  va_start(args,sql);
  fsl_db_get_idv(db, &rv, sql, args);
  va_end(args);
  return rv;
}

fsl_size_t fsl_db_g_size( fsl_db * const db, fsl_size_t dflt,
                        char const * sql,
                        ... ){
  fsl_size_t rv = dflt;
  va_list args;
  va_start(args,sql);
  fsl_db_get_sizev(db, &rv, sql, args);
  va_end(args);
  return rv;
}

double fsl_db_g_double( fsl_db * const db, double dflt,
                              char const * sql,
                              ... ){
  double rv = dflt;
  va_list args;
  va_start(args,sql);
  fsl_db_get_doublev(db, &rv, sql, args);
  va_end(args);
  return rv;
}

char * fsl_db_g_text( fsl_db * const db, fsl_size_t * len,
                      char const * sql,
                      ... ){
  char * rv = NULL;
  va_list args;
  va_start(args,sql);
  fsl_db_get_textv(db, &rv, len, sql, args);
  va_end(args);
  return rv;
}

void * fsl_db_g_blob( fsl_db * const db, fsl_size_t * len,
                      char const * sql,
                      ... ){
  void * rv = NULL;
  va_list args;
  va_start(args,sql);
  fsl_db_get_blob(db, &rv, len, sql, args);
  va_end(args);
  return rv;
}

double fsl_db_julian_now(fsl_db * const db){
  double rc = -1.0;
  if(db && db->dbh){
    /* TODO? use cached statement? So far not used often enough to
       justify it. */
    fsl_db_get_double( db, &rc, "SELECT julianday('now')");
  }
  return rc;
}

double fsl_db_string_to_julian(fsl_db * const db, char const * str){
  double rc = -1.0;
  if(db && db->dbh){
    /* TODO? use cached statement? So far not used often enough to
       justify it. */
    fsl_db_get_double( db, &rc, "SELECT julianday(%Q)",str);
  }
  return rc;
}

bool fsl_db_existsv(fsl_db * const db, char const * sql, va_list args ){
  if(!db || !db->dbh || !sql) return 0;
  else if(!*sql) return 0;
  else{
    fsl_stmt st = fsl_stmt_empty;
    bool rv = false;
    if(0==fsl_db_preparev(db, &st, sql, args)){
      rv = FSL_RC_STEP_ROW==fsl_stmt_step(&st) ? true : false;
    }
    fsl_stmt_finalize(&st);
    return rv;
  }

}

bool fsl_db_exists(fsl_db * const db, char const * sql, ... ){
  bool rc;
  va_list args;
  va_start(args,sql);
  rc = fsl_db_existsv(db, sql, args);
  va_end(args);
  return rc;
}

bool fsl_db_table_exists(fsl_db * const db,
                        fsl_dbrole_e whichDb,
                        const char *zTable
){
  const char *zDb = fsl_db_role_name( whichDb );
  int rc = db->dbh
    ? sqlite3_table_column_metadata(db->dbh, zDb, zTable, 0,
                                    0, 0, 0, 0, 0)
    : !SQLITE_OK;
  return rc==SQLITE_OK ? true : false;
}

bool fsl_db_table_has_column( fsl_db * const db, char const *zTableName, char const *zColName ){
  fsl_stmt q = fsl_stmt_empty;
  int rc = 0;
  bool rv = 0;
  if(!zTableName || !*zTableName || !zColName || !*zColName) return false;
  rc = fsl_db_prepare(db, &q, "PRAGMA table_info(%Q)", zTableName );
  if(!rc) while(FSL_RC_STEP_ROW==fsl_stmt_step(&q)){
    /* Columns: (cid, name, type, notnull, dflt_value, pk) */
    fsl_size_t colLen = 0;
    char const * zCol = fsl_stmt_g_text(&q, 1, &colLen);
    if(0==fsl_strncmp(zColName, zCol, colLen)){
      rv = true;
      break;
    }
  }
  fsl_stmt_finalize(&q);
  return rv;
}

char * fsl_db_random_hex(fsl_db * const db, fsl_size_t n){
  if(!db->dbh || !n) return NULL;
  else{
    fsl_size_t rvLen = 0;
    char * rv = fsl_db_g_text(db, &rvLen,
                              "SELECT lower(hex("
                              "randomblob(%"FSL_SIZE_T_PFMT")))",
                              (fsl_size_t)(n/2+1));
    if(rv){
      assert(rvLen>=n);
      rv[n]=0;
    }
    return rv;
  }
}


int fsl_db_select_slistv( fsl_db * const db, fsl_list * const tgt,
                          char const * fmt, va_list args ){
  if(!db->dbh) return fsl__db_err_not_opened(db);
  else if(!fmt || !*fmt) return fsl__db_err_sql_empty(db);
  else if(!*fmt) return FSL_RC_RANGE;
  else{
    int rc;
    fsl_stmt st = fsl_stmt_empty;
    fsl_size_t nlen;
    char const * n;
    char * cp;
    rc = fsl_db_preparev(db, &st, fmt, args);
    while( !rc && (FSL_RC_STEP_ROW==fsl_stmt_step(&st)) ){
      nlen = 0;
      n = fsl_stmt_g_text(&st, 0, &nlen);
      cp = n ? fsl_strndup(n, (fsl_int_t)nlen) : NULL;
      if(n && !cp) rc = FSL_RC_OOM;
      else{
        rc = fsl_list_append(tgt, cp);
        if(rc && cp) fsl_free(cp);
      }
    }
    fsl_stmt_finalize(&st);
    return rc;
  }
}

int fsl_db_select_slist( fsl_db * const db, fsl_list * const tgt,
                         char const * fmt, ... ){
  int rc;
  va_list va;
  va_start (va,fmt);
  rc = fsl_db_select_slistv(db, tgt, fmt, va);
  va_end(va);
  return rc;
}

void fsl_db_sqltrace_enable( fsl_db * const db, FILE * outStream ){
  if(db->dbh){
    sqlite3_trace_v2(db->dbh, SQLITE_TRACE_STMT,
                     fsl__db_sq3TraceV2, outStream);
  }
}

int fsl_db_init( fsl_error * err,
                 char const * zFilename,
                 char const * zSchema,
                 ... ){
  fsl_db DB = fsl_db_empty;
  fsl_db * db = &DB;
  char const * zSql;
  int rc;
  char inTrans = 0;
  va_list ap;
  rc = fsl_db_open(db, zFilename, 0);
  if(rc) goto end;
  rc = fsl_db_exec(db, "BEGIN EXCLUSIVE");
  if(rc) goto end;
  inTrans = 1;
  rc = fsl_db_exec_multi(db, "%s", zSchema);
  if(rc) goto end;
  va_start(ap, zSchema);
  while( !rc && (zSql = va_arg(ap, const char*))!=NULL ){
    rc = fsl_db_exec_multi(db, "%s", zSql);
  }
  va_end(ap);
  end:
  if(rc){
    if(inTrans) fsl_db_exec(db, "ROLLBACK");
  }else{
    rc = fsl_db_exec(db, "COMMIT");
  }
  if(err){
    if(db->error.code){
      fsl_error_move(&db->error, err);
    }else if(rc){
      err->code = rc;
      err->msg.used = 0;
    }
  }
  fsl_db_close(db);
  return rc;
}

int fsl_stmt_each_f_dump( fsl_stmt * const stmt, void * state ){
  int i;
  fsl_cx * f = (stmt && stmt->db) ? stmt->db->f : NULL;
  char const * sep = "\t";
  if(!f) return FSL_RC_MISUSE;
  if(state){/*unused arg*/}
  if(1==stmt->rowCount){
    for( i = 0; i < stmt->colCount; ++i ){
      fsl_outputf(f, "%s%s", fsl_stmt_col_name(stmt, i),
            (i==stmt->colCount-1) ? "" : sep);
    }
    fsl_output(f, "\n", 1);
  }
  for( i = 0; i < stmt->colCount; ++i ){
    char const * val = fsl_stmt_g_text(stmt, i, NULL);
    fsl_outputf(f, "%s%s", val ? val : "NULL",
          (i==stmt->colCount-1) ? "" : sep);
  }
  fsl_output(f, "\n", 1);
  return 0;
}


#undef MARKER
/* end of file ./src/db.c */
/* start of file ./src/deck.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/
/*
  This file houses the manifest/control-artifact-related APIs.
*/
#include <assert.h>
#include <stdlib.h> /* qsort() */
#include <memory.h> /* memcmp() */

/* Only for debugging */
#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)

typedef int StaticAssertMCacheArraySizes[
 ((sizeof(fsl__mcache_empty.aAge)
  /sizeof(fsl__mcache_empty.aAge[0]))
 == (sizeof(fsl__mcache_empty.decks)
     /sizeof(fsl__mcache_empty.decks[0])))
 ? 1 : -1
];

enum fsl_card_F_list_flags_e {
FSL_CARD_F_LIST_NEEDS_SORT = 0x01
};

/**
   Transfers the contents of d into f->cache.mcache. If d is
   dynamically allocated then it is also freed. In any case, after
   calling this the caller must behave as if the deck had been passed
   to fsl_deck_finalize() and (if it is unknown whether d is stack
   allocated) also freed.

   If manifest caching is disabled for f, d is immediately finalized.
*/
static void fsl__cx_mcache_insert(fsl_cx * const f, fsl_deck * d){
  if(!(f->flags & FSL_CX_F_MANIFEST_CACHE)){
    fsl_deck_finalize(d);
    return;
  }
  static const unsigned cacheLen =
    (unsigned)(sizeof(fsl__mcache_empty.aAge)
               /sizeof(fsl__mcache_empty.aAge[0]));
  fsl__mcache * const mc = &f->cache.mcache;
  while( d ){
    unsigned i;
    fsl_deck *pBaseline = d->B.baseline;
    d->B.baseline = 0;
    for(i=0; i<cacheLen; ++i){
      if( !mc->decks[i].rid ) break;
    }
    if( i>=cacheLen ){
      unsigned oldest = 0;
      unsigned oldestAge = mc->aAge[0];
      for(i=1; i<cacheLen; ++i){
        if( mc->aAge[i]<oldestAge ){
          oldest = i;
          oldestAge = mc->aAge[i];
        }
      }
      fsl_deck_finalize(&mc->decks[oldest]);
      i = oldest;
    }
    mc->aAge[i] = ++mc->nextAge;
    mc->decks[i] = *d;
    *d = fsl_deck_empty;
    if(&fsl_deck_empty == mc->decks[i].allocStamp){
      /* d was fsl_deck_malloc()'d so we need to free it, but cannot
         send it through fsl_deck_finalize() because that would try to
         clean up the memory we just transferred ownership of to
         mc->decks[i]. So... */
      mc->decks[i].allocStamp = NULL;
      fsl_free(d);
    }
    d = pBaseline;
  }
}

/**
   Searches f->cache.mcache for a deck with the given RID. If found,
   it is bitwise copied over tgt, that entry is removed from the
   cache, and true is returned. If no match is found, tgt is not
   modified and false is returned.

   If manifest caching is disabled for f, false is immediately
   returned without causing side effects.
*/
static bool fsl__cx_mcache_search(fsl_cx * const f, fsl_id_t rid,
                                 fsl_deck * const tgt){
  if(!(f->flags & FSL_CX_F_MANIFEST_CACHE)){
    ++f->cache.mcache.misses;
    return false;
  }
  static const unsigned cacheLen =
    (int)(sizeof(fsl__mcache_empty.aAge)
          /sizeof(fsl__mcache_empty.aAge[0]));
  unsigned i;
  assert(cacheLen ==
         (unsigned)(sizeof(fsl__mcache_empty.decks)
                    /sizeof(fsl__mcache_empty.decks[0])));
  for(i=0; i<cacheLen; ++i){
    if( f->cache.mcache.decks[i].rid==rid ){
      *tgt = f->cache.mcache.decks[i];
      f->cache.mcache.decks[i] = fsl_deck_empty;
      ++f->cache.mcache.hits;
      return true;
    }
  }
  ++f->cache.mcache.misses;
  return false;
}

/**
   Code duplication reducer for fsl_deck_parse2() and
   fsl_deck_load_rid(). Checks fsl__cx_mcache_search() for rid.  If
   found, overwrites tgt with its contents and returns true.  If not
   found, returns false. If an entry is found and type!=FSL_SATYPE_ANY
   and the found deck->type differs from type then false is returned,
   FSL_RC_TYPE is returned via *rc, and f's error state is updated
   with a description of the problem. In all other case *rc is set to
   0.
 */
static bool fsl__cx_mcache_search2(fsl_cx * const f, fsl_id_t rid,
                                  fsl_deck * const tgt,
                                  fsl_satype_e type,
                                  int * const rc){
  *rc = 0;
  if(fsl__cx_mcache_search(f, rid, tgt)){
    assert(f == tgt->f);
    if(type!=FSL_SATYPE_ANY && type!=tgt->type){
      *rc = fsl_cx_err_set(f, FSL_RC_TYPE,
                           "Unexpected match of RID #%" FSL_ID_T_PFMT " "
                           "to a different artifact type (%d) "
                           "than requested (%d).",
                           tgt->type, type);
      fsl__cx_mcache_insert(f, tgt);
      assert(!tgt->f);
      return false;
    }else{
      //MARKER(("Got cached deck: rid=%d\n", (int)d->rid));
      return true;
    }
  }
  return false;
}


/**
   If mem is NULL or inside d->content.mem then this function does
   nothing, else it passes mem to fsl_free(). Intended to be used to
   clean up d->XXX string members (or sub-members) which have been
   optimized away via d->content.
*/
static void fsl_deck_free_string(fsl_deck * d, char * mem){
  assert(d);
  if(mem
     && (!d->content.used
         || !(((unsigned char const *)mem >=d->content.mem)
              &&
              ((unsigned char const *)mem < (d->content.mem+d->content.capacity)))
         )){
    fsl_free(mem);
  }/* else do nothing - the memory is NULL or owned by d->content. */
}

/**
   fsl_list_visitor_f() impl which frees fsl_list-of-(char*) card entries
   in ((fsl_deck*)visitorState).
*/
static int fsl_list_v_card_string_free(void * mCard, void * visitorState ){
  fsl_deck_free_string( (fsl_deck*)visitorState, mCard );
  return 0;
}

/** Evals to a pointer to the F-card at the given index
    in the given fsl_card_F_list pointer. Each arg is
    evaluated only once. */
#define F_at(LISTP,NDX) (&(LISTP)->list[NDX])

static int fsl_card_F_list_reserve2( fsl_card_F_list * li ){
  return (li->used<li->capacity)
    ? 0
    : fsl_card_F_list_reserve(li, li->capacity
                               ? li->capacity*4/3+1
                               : 50);
}

static void fsl_card_F_clean( fsl_card_F * f ){
  if(!f->deckOwnsStrings){
    fsl_free(f->name);
    fsl_free(f->uuid);
    fsl_free(f->priorName);
  }
  *f = fsl_card_F_empty;
}

/**
   Cleans up the F-card at li->list[ndx] and shifts all F-cards to its
   right one entry to the left.
*/
static void fsl_card_F_list_remove(fsl_card_F_list * li,
                                   uint32_t ndx){
  uint32_t i;
  assert(li->used);
  assert(ndx<li->used);
  fsl_card_F_clean(F_at(li,ndx));
  for( i = ndx; i < li->used - 1; ++i ){
    li->list[i] = li->list[i+1];
  }
  li->list[li->used] = fsl_card_F_empty;
  --li->used;
}

void fsl_card_F_list_finalize( fsl_card_F_list * li ){
  uint32_t i;
  for(i=0; i < li->used; ++i){
    fsl_card_F_clean(F_at(li,i));  
  }
  li->used = li->capacity = 0;
  fsl_free(li->list);
  *li = fsl_card_F_list_empty;
}

int fsl_card_F_list_reserve( fsl_card_F_list * li, uint32_t n ){
  if(li->capacity>=n) return 0;
  else if(n==0){
    fsl_card_F_list_finalize(li);
    return 0;
  }else{
    fsl_card_F * re = fsl_realloc(li->list, n * sizeof(fsl_card_F));
    if(re){
      li->list = re;
      li->capacity = n;
    }
    return re ? 0 : FSL_RC_OOM;
  }
}

/**
   Adjusts the end of the give list by +1, reserving more space if
   needed, and returns the next available F-card in a cleanly-wiped
   state. Returns NULL on alloc error.
*/
static fsl_card_F * fsl_card_F_list_push( fsl_card_F_list * li ){
  if(li->used==li->capacity && fsl_card_F_list_reserve2(li)) return NULL;
  li->list[li->used] = fsl_card_F_empty;
  if(li->used){
    li->flags |= FSL_CARD_F_LIST_NEEDS_SORT/*pessimistic assumption*/;
  }
  return &li->list[li->used++];
}
/**
   Chops the last entry off of the given list, freeing any resources
   owned by that entry. Decrements li->used. Asserts that li->used is
   positive.
*/
static void fsl_card_F_list_pop( fsl_card_F_list * li ){
  assert(li->used);
  if(li->used) fsl_card_F_clean(F_at(li, --li->used));
}

fsl_card_Q * fsl_card_Q_malloc(fsl_cherrypick_type_e type,
                               fsl_uuid_cstr target,
                               fsl_uuid_cstr baseline){
  int const targetLen = target ? fsl_is_uuid(target) : 0;
  int const baselineLen = baseline ? fsl_is_uuid(baseline) : 0;
  if((type!=FSL_CHERRYPICK_ADD && type!=FSL_CHERRYPICK_BACKOUT)
     || !target || !targetLen
     || (baseline && !baselineLen)) return NULL;
  else{
    fsl_card_Q * c =
      (fsl_card_Q*)fsl_malloc(sizeof(fsl_card_Q));
    if(c){
      int rc = 0;
      *c = fsl_card_Q_empty;
      c->type = type;
      c->target = fsl_strndup(target, targetLen);
      if(!c->target) rc = FSL_RC_OOM;
      else if(baseline){
        c->baseline = fsl_strndup(baseline, baselineLen);
        if(!c->baseline) rc = FSL_RC_OOM;
      }
      if(rc){
        fsl_card_Q_free(c);
        c = NULL;
      }
    }
    return c;
  }
}

void fsl_card_Q_free( fsl_card_Q * cp ){
  if(cp){
    fsl_free(cp->target);
    fsl_free(cp->baseline);
    *cp = fsl_card_Q_empty;
    fsl_free(cp);
  }
}

fsl_card_J * fsl_card_J_malloc(bool isAppend,
                               char const * field,
                               char const * value){
  if(!field || !*field) return NULL;
  else{
    fsl_card_J * c =
      (fsl_card_J*)fsl_malloc(sizeof(fsl_card_J));
    if(c){
      int rc = 0;
      fsl_size_t const lF = fsl_strlen(field);
      fsl_size_t const lV = value ? fsl_strlen(value) : 0;
      *c = fsl_card_J_empty;
      c->append = isAppend ? 1 : 0;
      c->field = fsl_strndup(field, (fsl_int_t)lF);
      if(!c->field) rc = FSL_RC_OOM;
      else if(value && *value){
        c->value = fsl_strndup(value, (fsl_int_t)lV);
        if(!c->value) rc = FSL_RC_OOM;
      }
      if(rc){
        fsl_card_J_free(c);
        c = NULL;
      }
    }
    return c;
  }
}

void fsl_card_J_free( fsl_card_J * cp ){
  if(cp){
    fsl_free(cp->field);
    fsl_free(cp->value);
    *cp = fsl_card_J_empty;
    fsl_free(cp);
  }
}

/**
    fsl_list_visitor_f() impl which requires that obj be-a (fsl_card_T*),
    which this function passes to fsl_card_T_free().
*/
static int fsl_list_v_card_T_free(void * obj, void * visitorState __unused){
  if(obj) fsl_card_T_free( (fsl_card_T*)obj );
  return 0;
}

static int fsl_list_v_card_Q_free(void * obj, void * visitorState __unused ){
  if(obj) fsl_card_Q_free( (fsl_card_Q*)obj );
  return 0;
}

static int fsl_list_v_card_J_free(void * obj, void * visitorState __unused){
  if(obj) fsl_card_J_free( (fsl_card_J*)obj );
  return 0;
}

fsl_deck * fsl_deck_malloc(){
  fsl_deck * rc = (fsl_deck *)fsl_malloc(sizeof(fsl_deck));
  if(rc){
    *rc = fsl_deck_empty;
    rc->allocStamp = &fsl_deck_empty;
  }
  return rc;
}

void fsl_deck_init( fsl_cx * const f, fsl_deck * const cards, fsl_satype_e type ){
  void const * allocStamp = cards->allocStamp;
  *cards = fsl_deck_empty;
  cards->allocStamp = allocStamp;
  cards->f = f;
  cards->type = type;
}

void fsl__card_J_list_free(fsl_list * li, bool alsoListMem){
  if(li->used) fsl_list_visit(li, 0, fsl_list_v_card_J_free, NULL);
  if(alsoListMem) fsl_list_reserve(li, 0);
  else li->used = 0;
}

/* fsl_deck cleanup helpers... */
#define SFREE(X) fsl_deck_clean_string(m, &m->X)
#define SLIST(X) fsl_list_clear(&m->X, fsl_list_v_card_string_free, m)
#define CBUF(X) fsl_buffer_clear(&m->X)
static void fsl_deck_clean_string(fsl_deck *m, char **member){
  fsl_deck_free_string(m, *member);
  *member = 0;
}
static void fsl_deck_clean_version(fsl_deck *const m){
  m->rid = 0;
}
static void fsl_deck_clean_A(fsl_deck *const m){
  SFREE(A.name);
  SFREE(A.tgt);
  SFREE(A.src);
}
static void fsl_deck_clean_B(fsl_deck * const m){
  if(m->B.baseline){
    assert(!m->B.baseline->B.uuid && "Baselines cannot have a B-card. API misuse?");
    fsl_deck_finalize(m->B.baseline);
    m->B.baseline = NULL;
  }
  SFREE(B.uuid);
}
static void fsl_deck_clean_C(fsl_deck * const m){
  fsl_deck_clean_string(m, &m->C);
}
static void fsl_deck_clean_E(fsl_deck * const m){
  fsl_deck_clean_string(m, &m->E.uuid);
  m->E = fsl_deck_empty.E;
}
static void fsl_deck_clean_F(fsl_deck * const m){
  if(m->F.list){
    fsl_card_F_list_finalize(&m->F);
    m->F = fsl_deck_empty.F;
  }
}
static void fsl_deck_clean_G(fsl_deck * const m){
  fsl_deck_clean_string(m, &m->G);
}
static void fsl_deck_clean_H(fsl_deck * const m){
  fsl_deck_clean_string(m, &m->H);
}
static void fsl_deck_clean_I(fsl_deck * const m){
  fsl_deck_clean_string(m, &m->I);
}
static void fsl_deck_clean_J(fsl_deck * const m, bool alsoListMem){
  fsl__card_J_list_free(&m->J, alsoListMem);
}
static void fsl_deck_clean_K(fsl_deck * const m){
  fsl_deck_clean_string(m, &m->K);
}
static void fsl_deck_clean_L(fsl_deck * const m){
  fsl_deck_clean_string(m, &m->L);
}
static void fsl_deck_clean_M(fsl_deck * const m){
  SLIST(M);
}
static void fsl_deck_clean_N(fsl_deck * const m){
  fsl_deck_clean_string(m, &m->N);
}
static void fsl_deck_clean_P(fsl_deck * const m){
  fsl_list_clear(&m->P, fsl_list_v_card_string_free, m);
}
static void fsl_deck_clean_Q(fsl_deck * const m){
  fsl_list_clear(&m->Q, fsl_list_v_card_Q_free, NULL);
}
static void fsl_deck_clean_R(fsl_deck * const m){
  fsl_deck_clean_string(m, &m->R);
}
static void fsl_deck_clean_T(fsl_deck * const m){
  fsl_list_clear(&m->T, fsl_list_v_card_T_free, NULL);
}
static void fsl_deck_clean_U(fsl_deck * const m){
  fsl_deck_clean_string(m, &m->U);
}
static void fsl_deck_clean_W(fsl_deck * const m){
  if(m->W.capacity/*dynamically-allocated buffer*/){
    CBUF(W);
  }else{/*empty or external buffer pointing into to m->content.mem*/
    m->W = fsl_buffer_empty;
  }
}

void fsl_deck_clean2(fsl_deck * const m, fsl_buffer * const xferBuf){
  if(!m) return;
  fsl_deck_clean_version(m);  
  fsl_deck_clean_A(m);
  fsl_deck_clean_B(m);
  fsl_deck_clean_C(m);
  m->D = 0.0;
  fsl_deck_clean_E(m);
  fsl_deck_clean_F(m);
  fsl_deck_clean_G(m);
  fsl_deck_clean_H(m);
  fsl_deck_clean_I(m);
  fsl_deck_clean_J(m,true);
  fsl_deck_clean_K(m);
  fsl_deck_clean_L(m);
  fsl_deck_clean_M(m);
  fsl_deck_clean_N(m);
  fsl_deck_clean_P(m);
  fsl_deck_clean_Q(m);
  fsl_deck_clean_R(m);
  fsl_deck_clean_T(m);
  fsl_deck_clean_U(m);
  fsl_deck_clean_W(m);
  if(xferBuf){
    fsl_buffer_swap(&m->content, xferBuf);
    fsl_buffer_reuse(xferBuf);
  }
  CBUF(content) /* content must be after all cards because some point
                   back into it and we need this memory intact in
                   order to know that!
                */;
  {
    void const * const allocStampKludge = m->allocStamp;
    fsl_cx * const f = m->f;
    *m = fsl_deck_empty;
    m->allocStamp = allocStampKludge;
    m->f = f;
  }
}
#undef CBUF
#undef SFREE
#undef SLIST

void fsl_deck_clean(fsl_deck * const m){
  fsl_deck_clean2(m, NULL);
}

void fsl_deck_finalize(fsl_deck * const m){
  void const * allocStamp;
  if(!m) return;
  allocStamp = m->allocStamp;
  fsl_deck_clean(m);
  if(allocStamp == &fsl_deck_empty){
    fsl_free(m);
  }else{
    m->allocStamp = allocStamp;
  }
}

int fsl_card_is_legal( fsl_satype_e t, char card ){
  /*
    Implements this table:
    
    https://fossil-scm.org/index.html/doc/trunk/www/fileformat.wiki#summary
  */
  if('Z'==card) return 1;
  else switch(t){
    case FSL_SATYPE_ANY:
      switch(card){
        case 'A': case 'B': case 'C': case 'D':
        case 'E': case 'F': case 'J': case 'K':
        case 'L': case 'M': case 'N': case 'P':
        case 'Q': case 'R': case 'T': case 'U':
        case 'W':
          return -1;
        default:
          return 0;
      }
    case FSL_SATYPE_ATTACHMENT:
      switch(card){
        case 'A': case 'D':
          return 1;
        case 'C': case 'N': case 'U':
          return -1;
        default:
          return 0;
      };
    case FSL_SATYPE_CLUSTER:
      return 'M'==card ? 1 : 0;
    case FSL_SATYPE_CONTROL:
      switch(card){
        case 'D': case 'T': case 'U':
          return 1;
        default:
          return 0;
      };
    case FSL_SATYPE_EVENT:
      switch(card){
        case 'D': case 'E':
        case 'W':
          return 1;
        case 'C': case 'N':
        case 'P': case 'T':
        case 'U':
          return -1;
        default:
          return 0;
      };
    case FSL_SATYPE_CHECKIN:
      switch(card){
        case 'C': case 'D':
        case 'U':
          return 1;
        case 'B': case 'F': 
        case 'N': case 'P':
        case 'Q': case 'R':
        case 'T': 
          return -1;
        default:
          return 0;
      };
    case FSL_SATYPE_TICKET:
      switch(card){
        case 'D': case 'J':
        case 'K': case 'U':
          return 1;
        default:
          return 0;
      };
    case FSL_SATYPE_WIKI:
      switch(card){
        case 'D': case 'L':
        case 'U': case 'W':
          return 1;
        case 'C':
        case 'N': case 'P':
          return -1;
        default:
          return 0;
      };
    case FSL_SATYPE_FORUMPOST:
      switch(card){
        case 'D': case 'U': case 'W':
          return 1;
        case 'G': case 'H': case 'I':
        case 'N': case 'P':
          return -1;
        default:
          return 0;
      };
    default:
      MARKER(("invalid fsl_satype_e value: %d, card=%c\n", t, card));
      assert(!"Invalid fsl_satype_e.");
      return 0;
  };
}

bool fsl_deck_has_required_cards( fsl_deck const * d ){
  if(!d) return 0;
  switch(d->type){
    case FSL_SATYPE_ANY:
      return 0;
#define NEED(CARD,COND) \
      if(!(COND)) {                                         \
        fsl_cx_err_set(d->f, FSL_RC_SYNTAX,                 \
                       "Required %c-card is missing or invalid.", \
                       *#CARD);                                   \
        return false;                                             \
      } (void)0
    case FSL_SATYPE_ATTACHMENT:
      NEED(A,d->A.name);
      NEED(A,d->A.tgt);
      NEED(D,d->D > 0);
      return 1;
    case FSL_SATYPE_CLUSTER:
      NEED(M,d->M.used);
      return 1;
    case FSL_SATYPE_CONTROL:
      NEED(D,d->D > 0);
      NEED(U,d->U);
      NEED(T,d->T.used>0);
      return 1;
    case FSL_SATYPE_EVENT:
      NEED(D,d->D > 0);
      NEED(E,d->E.julian>0);
      NEED(E,d->E.uuid);
      NEED(W,d->W.used);
      return 1;
    case FSL_SATYPE_CHECKIN:
      /*
        Historically we need both or neither of F- and R-cards, but
        the R-card has become optional because it's so expensive to
        calculate and verify.

        Manifest #1 has an empty file list and an R-card with a
        constant (repo/manifest-independent) hash
        (d41d8cd98f00b204e9800998ecf8427e, the initial MD5 hash
        state).

        R-card calculation is runtime-configurable option.
      */
      NEED(D,d->D > 0);
      NEED(C,d->C);
      NEED(U,d->U);
#if 0
      /* It turns out that because the R-card is optional,
         we can have a legal manifest with no F-cards. */
      NEED(F,d->F.used || d->R/*with initial-state md5 hash!*/);
#endif
      if(!d->R
         && (FSL_CX_F_CALC_R_CARD & d->f->flags)){
        fsl_cx_err_set(d->f, FSL_RC_SYNTAX,
                       "%s deck is missing an R-card, "
                       "yet R-card calculation is enabled.",
                       fsl_satype_cstr(d->type));
        return 0;
      }else if(d->R
               && !d->F.used
               && 0!=fsl_strcmp(d->R, FSL_MD5_INITIAL_HASH)
               ){
        fsl_cx_err_set(d->f, FSL_RC_SYNTAX,
                       "Deck has no F-cards, so we expect its "
                       "R-card is to have the initial-state MD5 "
                       "hash (%.12s...). Instead we got: %s",
                       FSL_MD5_INITIAL_HASH, d->R);
        return 0;
      }
      return 1;
    case FSL_SATYPE_TICKET:
      NEED(D,d->D > 0);
      NEED(K,d->K);
      NEED(U,d->U);
      NEED(J,d->J.used)
        /* Is a J strictly required?  Spec is not clear but DRH
           confirms the current fossil(1) code expects a J card. */;
      return 1;
    case FSL_SATYPE_WIKI:
      NEED(D,d->D > 0);
      NEED(L,d->L);
      NEED(U,d->U);
      /*NEED(W,d->W.used);*/
      return 1;
    case FSL_SATYPE_FORUMPOST:
      NEED(D,d->D > 0);
      NEED(U,d->U);
      /*NEED(W,d->W.used);*/
      return 1;
    case FSL_SATYPE_INVALID:
    default:
      assert(!"Invalid fsl_satype_e.");
      return 0;
  }
#undef NEED
}

char const * fsl_satype_cstr(fsl_satype_e t){
  switch(t){
#define C(X) case FSL_SATYPE_##X: return #X
    C(ANY);
    C(CHECKIN);
    C(CLUSTER);
    C(CONTROL);
    C(WIKI);
    C(TICKET);
    C(ATTACHMENT);
    C(EVENT);
    C(FORUMPOST);
    C(INVALID);
    C(BRANCH_START);
    default:
      assert(!"UNHANDLED fsl_satype_e");
      return "!UNKNOWN!";
  }
}

char const * fsl_satype_event_cstr(fsl_satype_e t){
  switch(t){
    case FSL_SATYPE_ANY: return "*";
    case FSL_SATYPE_BRANCH_START:
    case FSL_SATYPE_CHECKIN: return "ci";
    case FSL_SATYPE_EVENT: return "e";
    case FSL_SATYPE_CONTROL: return "g";
    case FSL_SATYPE_TICKET: return "t";
    case FSL_SATYPE_WIKI: return "w";
    case FSL_SATYPE_FORUMPOST: return "f";
    default:
      return NULL;
  }
}


/**
    If fsl_card_is_legal(d->type, card), returns true, else updates
    d->f->error with a description of the constraint violation and
    returns 0.
 */
static bool fsl_deck_check_type( fsl_deck * const d, char card ){
  if(fsl_card_is_legal(d->type, card)) return true;
  else{
    fsl_cx_err_set(d->f, FSL_RC_TYPE,
                   "Card type '%c' is not allowed "
                   "in artifacts of type %s.",
                   card, fsl_satype_cstr(d->type));
    return false;
  }
}

/**
   If the first n bytes of the given string contain any values <=32,
   returns FSL_RC_SYNTAX, else returns 0. mf->f's error state is
   updated no error. n<0 means to use fsl_strlen() to count the
   length.
*/
static int fsl_deck_strcheck_ctrl_chars(fsl_deck * const mf, char cardName, char const * v, fsl_int_t n){
  const char * z = v;
  int rc = 0;
  if(v && n<0) n = fsl_strlen(v);
  for( ; v && z < v+n; ++z ){
    if(*z <= 32){
      rc = fsl_cx_err_set(mf->f, FSL_RC_SYNTAX,
                         "Invalid character in %c-card.", cardName);
      break;
    }
  }
  return rc;
}

/*
  Implements fsl_deck_LETTER_set() for certain letters: those
  implemented as a fsl_uuid_str or an md5, holding a single hex string
  value.
  
  The function returns FSL_RC_SYNTAX if
  (valLen!=ASSERTLEN). ASSERTLEN is assumed to be either an SHA1,
  SHA3, or MD5 hash value and it is validated against
  fsl_validate16(value,valLen), returning FSL_RC_SYNTAX if that
  check fails. In debug builds, the expected ranges are assert()ed.

  If value is NULL then it is removed from the card instead
  (semantically freed), *mfMember is set to NULL, and 0 is returned.
*/
static int fsl_deck_sethex_impl( fsl_deck * const mf, fsl_uuid_cstr value,
                                 char letter,
                                 fsl_size_t assertLen,
                                 char ** mfMember ){
  assert(mf);
  assert( value ? (assertLen==FSL_STRLEN_SHA1
                   || assertLen==FSL_STRLEN_K256
                   || assertLen==FSL_STRLEN_MD5)
          : 0==assertLen );
  if(value && !fsl_deck_check_type(mf,letter)) return mf->f->error.code;
  else if(!value){
    fsl_deck_free_string(mf, *mfMember);
    *mfMember = NULL;
    return 0;
  }else if(fsl_strlen(value) != assertLen){
    return fsl_cx_err_set(mf->f, FSL_RC_SYNTAX,
                          "Invalid length for %c-card: expecting %d.",
                          letter, (int)assertLen);
  }else if(!fsl_validate16(value, assertLen)) {
    return fsl_cx_err_set(mf->f, FSL_RC_SYNTAX,
                          "Invalid hexadecimal value for %c-card.", letter);
  }else{
    fsl_deck_free_string(mf, *mfMember);
    *mfMember = fsl_strndup(value, assertLen);
    return *mfMember ? 0 : FSL_RC_OOM;
  }
}

/**
    Implements fsl_set_set_XXX() where XXX is a fsl_buffer member of fsl_deck.
 */
static int fsl_deck_b_setuffer_impl( fsl_deck * const mf, char const * value,
                                     fsl_int_t valLen,
                                     char letter, fsl_buffer * buf){
  assert(mf);  
  if(!fsl_deck_check_type(mf,letter)) return mf->f->error.code;
  else if(valLen<0) valLen = (fsl_int_t)fsl_strlen(value);
  buf->used = 0;
  if(value && (valLen>0)){
    return fsl_buffer_append( buf, value, valLen );
  }else{
    if(buf->mem) buf->mem[0] = 0;
    return 0;
  }
}

int fsl_deck_B_set( fsl_deck * const mf, fsl_uuid_cstr uuidBaseline){
  if(!mf) return FSL_RC_MISUSE;
  else{
    int const bLen = uuidBaseline ? fsl_is_uuid(uuidBaseline) : 0;
    if(uuidBaseline && !bLen){
      return fsl_cx_err_set(mf->f, FSL_RC_SYNTAX,
                            "Invalid B-card value: %s", uuidBaseline);
    }
    if(mf->B.baseline){
      fsl_deck_finalize(mf->B.baseline);
      mf->B.baseline = NULL;
    }
    return fsl_deck_sethex_impl(mf, uuidBaseline, 'B',
                                bLen, &mf->B.uuid);
  }
}

/**
    Internal impl for card setters which consist of a simple (char *)
    member. Replaces and frees any prior value. Passing NULL for the
    4th argument unsets the given card (assigns NULL to it).
 */
static int fsl_deck_set_string( fsl_deck * const mf, char letter, char ** member, char const * v, fsl_int_t n ){
  if(!fsl_deck_check_type(mf, letter)) return mf->f->error.code;
  fsl_deck_free_string(mf, *member);
  *member = v ? fsl_strndup(v, n) : NULL;
  if(v && !*member) return FSL_RC_OOM;
  else return 0;
}

int fsl_deck_C_set( fsl_deck * const mf, char const * v, fsl_int_t n){
  return fsl_deck_set_string( mf, 'C', &mf->C, v, n );
}

int fsl_deck_G_set( fsl_deck * const mf, fsl_uuid_cstr uuid){
  int const uLen = fsl_is_uuid(uuid);
  return uLen
    ? fsl_deck_sethex_impl(mf, uuid, 'G', uLen, &mf->G)
    : FSL_RC_SYNTAX;
}

int fsl_deck_H_set( fsl_deck * const mf, char const * v, fsl_int_t n){
  if(v && mf->I) return FSL_RC_SYNTAX;
  return fsl_deck_set_string( mf, 'H', &mf->H, v, n );
}

int fsl_deck_I_set( fsl_deck * const mf, fsl_uuid_cstr uuid){
  if(uuid && mf->H) return FSL_RC_SYNTAX;
  int const uLen = uuid ? fsl_is_uuid(uuid) : 0;
  return fsl_deck_sethex_impl(mf, uuid, 'I', uLen, &mf->I);
}

int fsl_deck_J_add( fsl_deck * const mf, bool isAppend,
                    char const * field, char const * value){
  if(!field) return FSL_RC_MISUSE;
  else if(!*field) return FSL_RC_SYNTAX;
  else if(!fsl_deck_check_type(mf,'J')) return mf->f->error.code;
  else{
    int rc;
    fsl_card_J * cp = fsl_card_J_malloc(isAppend, field, value);
    if(!cp) rc = FSL_RC_OOM;
    else if( 0 != (rc = fsl_list_append(&mf->J, cp))){
      fsl_card_J_free(cp);
    }
    return rc;
  }
}


int fsl_deck_K_set( fsl_deck * const mf, fsl_uuid_cstr uuid){
  if(uuid){
    int const uLen = fsl_is_uuid(uuid);
    return uLen
      ? fsl_deck_sethex_impl(mf, uuid, 'K', uLen, &mf->K)
      : FSL_RC_SYNTAX;
  }else{
    char buf[FSL_STRLEN_SHA1+1];
    unsigned char rnd[FSL_STRLEN_SHA1/20];
    fsl_randomness(FSL_STRLEN_SHA1/2, rnd);
    fsl_sha1_digest_to_base16(rnd, buf);
    return fsl_deck_sethex_impl(mf, buf, 'K', FSL_STRLEN_SHA1, &mf->K);
  }
}

int fsl_deck_L_set( fsl_deck * const mf, char const * v, fsl_int_t n){
  return mf
    ? fsl_deck_set_string(mf, 'L', &mf->L, v, n)
    : FSL_RC_SYNTAX;
}

int fsl_deck_M_add( fsl_deck * const mf, char const *uuid){
  int rc;
  char * dupe;
  int const uLen = uuid ? fsl_is_uuid(uuid) : 0;
  if(!uuid) return FSL_RC_MISUSE;
  else if(!fsl_deck_check_type(mf, 'M')) return mf->f->error.code;
  else if(!uLen) return FSL_RC_SYNTAX;
  dupe = fsl_strndup(uuid, uLen);
  if(!dupe) rc = FSL_RC_OOM;
  else{
    rc = fsl_list_append( &mf->M, dupe );
    if(rc){
      fsl_free(dupe);
    }
  }
  return rc;
}

int fsl_deck_N_set( fsl_deck * const mf, char const * v, fsl_int_t n){
  int rc = 0;
  if(v && n!=0){
    if(n<0) n = fsl_strlen(v);
    rc = fsl_deck_strcheck_ctrl_chars(mf, 'N', v, n);
  }
  return rc ? rc : fsl_deck_set_string( mf, 'N', &mf->N, v, n );
}


/**
   Adds either parentUuid or takeParentUuid to mf->P. ONE
   of those must e non-NULL and the other must be NULL. If
   takeParentUuid is not NULL then ownership of it is transfered
   to this function regardless of success or failure.
*/
static int fsl__deck_P_add_impl( fsl_deck * const mf,
                                 fsl_uuid_cstr parentUuid,
                                 fsl_uuid_str takeParentUuid){
  if(!fsl_deck_check_type(mf, 'P')){
    fsl_free(takeParentUuid);
    return mf->f->error.code;
  }
  int rc;
  char * dupe;
  fsl_uuid_cstr z = parentUuid ? parentUuid : takeParentUuid;
  assert(parentUuid ? !takeParentUuid : !!takeParentUuid);
  int const uLen = fsl_is_uuid(z);
  if(!uLen){
    fsl_free(takeParentUuid);
    return fsl_cx_err_set(mf->f, FSL_RC_SYNTAX,
                         "Invalid UUID for P-card.");
  }
  dupe = takeParentUuid
    ? takeParentUuid
    : fsl_strndup(parentUuid, uLen);
  if(!dupe) rc = FSL_RC_OOM;
  else{
    rc = fsl_list_append( &mf->P, dupe );
    if(rc){
      fsl_free(dupe);
    }
  }
  return rc;
}


int fsl_deck_P_add(fsl_deck * const mf, char const *parentUuid){
  return fsl__deck_P_add_impl(mf, parentUuid, NULL);
}

int fsl_deck_P_add_rid( fsl_deck * const mf, fsl_id_t rid ){
  fsl_uuid_str pU = fsl_rid_to_uuid(mf->f, rid);
  return pU ? fsl__deck_P_add_impl(mf, NULL, pU) : mf->f->error.code;
}


fsl_id_t fsl_deck_P_get_id(fsl_deck * const d, int index){
  if(!d->f) return -1;
  else if(index>(int)d->P.used) return 0;
  else return fsl_uuid_to_rid(d->f, (char const *)d->P.list[index]);
}


int fsl_deck_Q_add( fsl_deck * const mf, int type,
                    fsl_uuid_cstr target,
                    fsl_uuid_cstr baseline ){
  if(!target) return FSL_RC_MISUSE;
  else if(!fsl_deck_check_type(mf,'Q')) return mf->f->error.code;
  else if(!type || !fsl_is_uuid(target)
          || (baseline && !fsl_is_uuid(baseline))) return FSL_RC_SYNTAX;
  else{
    int rc;
    fsl_card_Q * cp = fsl_card_Q_malloc(type, target, baseline);
    if(!cp) rc = FSL_RC_OOM;
    else if( 0 != (rc = fsl_list_append(&mf->Q, cp))){
      fsl_card_Q_free(cp);
    }
    return rc;
  }
}

/**
    A comparison routine for qsort(3) which compares fsl_card_F
    instances in a lexical manner based on their names.  The order is
    important for card ordering in generated manifests.

    It expects that each argument is a (fsl_card_F const *).
*/
static int fsl_card_F_cmp( void const * lhs, void const * rhs ){
  fsl_card_F const * const l = (fsl_card_F const *)lhs;
  fsl_card_F const * const r = (fsl_card_F const *)rhs;
  /* Compare NULL as larger so that NULLs move to the right. That said,
     we aren't expecting any NULLs. */
  assert(l);
  assert(r);
  if(!l) return r ? 1 : 0;
  else if(!r) return -1;
  else return fsl_strcmp(l->name, r->name);
}

static void fsl_card_F_list_sort(fsl_card_F_list * li){
  if(FSL_CARD_F_LIST_NEEDS_SORT & li->flags){
    qsort(li->list, li->used, sizeof(fsl_card_F),
          fsl_card_F_cmp );
    li->flags &= ~FSL_CARD_F_LIST_NEEDS_SORT;
  }
}

static void fsl_deck_F_sort(fsl_deck * const mf){
  fsl_card_F_list_sort(&mf->F);
}

int fsl_card_F_compare_name( fsl_card_F const * const lhs,
                             fsl_card_F const * const rhs){
  return (lhs == rhs) ? 0 : fsl_card_F_cmp( lhs, rhs );
}

int fsl_deck_R_set( fsl_deck * const mf, fsl_uuid_cstr md5){
  return mf
    ? fsl_deck_sethex_impl(mf, md5, 'R', md5 ? FSL_STRLEN_MD5 : 0, &mf->R)
    : FSL_RC_MISUSE;
}

int fsl_deck_R_calc2(fsl_deck * const mf, char ** tgt){
  fsl_cx * const f = mf->f;
  char const * theHash = 0;
  char hex[FSL_STRLEN_MD5+1];
  if(!f) return FSL_RC_MISUSE;
  else if(!fsl_needs_repo(f)){
    return FSL_RC_NOT_A_REPO;
  }else if(!fsl_deck_check_type(mf,'R')) {
    assert(mf->f->error.code);
    return mf->f->error.code;
  }else if(!mf->F.used){
    theHash = FSL_MD5_INITIAL_HASH;
    /* fall through and set hash */
  }else{
    int rc = 0;
    fsl_card_F const * fc;
    fsl_id_t fileRid;
    fsl_buffer * const buf = &f->cache.fileContent;
    unsigned char digest[16];
    fsl_md5_cx md5 = fsl_md5_cx_empty;
    enum { NumBufSize = 40 };
    char numBuf[NumBufSize] = {0};
    assert(!buf->used && "Misuse of f->fileContent buffer.");
    rc = fsl_deck_F_rewind(mf);
    if(rc) goto end;
    fsl_deck_F_sort(mf);
    /*
      TODO:

      Missing functionality:

      - The "wd" (working directory) family of functions, needed for
      symlink handling.
    */
    while(1){
      rc = fsl_deck_F_next(mf, &fc);
      /* MARKER(("R rc=%s file #%d: %s %s\n", fsl_rc_cstr(rc), ++i, fc ? fc->name : "<END>", fc ? fc->uuid : NULL)); */
      if(rc || !fc) break;
      assert(fc->uuid && "We no longer iterate over deleted entries.");
      if(FSL_FILE_PERM_LINK==fc->perm){
        rc = fsl_cx_err_set(f, FSL_RC_UNSUPPORTED,
                            "This code does not yet properly handle "
                            "F-cards of symlinks.");
        goto end;
      }
      fileRid = fsl_uuid_to_rid( f, fc->uuid );
      if(0==fileRid){
        rc = fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                            "Cannot resolve RID for F-card UUID [%s].",
                            fc->uuid);
        goto end;
      }else if(fileRid<0){
        assert(f->error.code);
        rc = f->error.code
          ? f->error.code
          : fsl_cx_err_set(f, FSL_RC_ERROR,
                           "Error resolving RID for F-card UUID [%s].",
                           fc->uuid);
        goto end;
      }
      fsl_md5_update_cstr(&md5, fc->name, -1);
      rc = fsl_content_get(f, fileRid, buf);
      if(rc){
        goto end;
      }
      numBuf[0] = 0;
      fsl_snprintf(numBuf, NumBufSize,
                   " %"FSL_SIZE_T_PFMT"\n",
                   buf->used);
      fsl_md5_update_cstr(&md5, numBuf, -1);
      fsl_md5_update_buffer(&md5, buf);
    }
    if(!rc){
      fsl_md5_final(&md5, digest);
      fsl_md5_digest_to_base16(digest, hex);
    }
    end:
    fsl__cx_content_buffer_yield(f);
    assert(0==buf->used);
    if(rc) return rc;
    fsl_deck_F_rewind(mf);
    theHash = hex;
  }
  assert(theHash);
  if(*tgt){
    memcpy(*tgt, theHash, FSL_STRLEN_MD5);
    (*tgt)[FSL_STRLEN_MD5] = 0;
    return 0;
  }else{
    char * x = fsl_strdup(theHash);
    if(x) *tgt = x;
    return x ? 0 : FSL_RC_OOM;
  }
}

int fsl_deck_R_calc(fsl_deck * const mf){
  char R[FSL_STRLEN_MD5+1] = {0};
  char * r = R;
  const int rc = fsl_deck_R_calc2(mf, &r);
  return rc ? rc : fsl_deck_R_set(mf, r);
}

int fsl_deck_T_add2( fsl_deck * const mf, fsl_card_T * t){
  if(!t) return FSL_RC_MISUSE;
  else if(!fsl_deck_check_type(mf, 'T')){
    return mf->f->error.code;
  }else if(FSL_SATYPE_CONTROL==mf->type && NULL==t->uuid){
    return fsl_cx_err_set(mf->f, FSL_RC_SYNTAX,
                            "CONTROL artifacts may not have "
                            "self-referential tags.");
  }else if(FSL_SATYPE_TECHNOTE==mf->type){
    if(NULL!=t->uuid){
      return fsl_cx_err_set(mf->f, FSL_RC_SYNTAX,
                              "TECHNOTE artifacts may not have "
                              "tags which refer to other objects.");
    }else if(FSL_TAGTYPE_ADD!=t->type){
      return fsl_cx_err_set(mf->f, FSL_RC_SYNTAX,
                            "TECHNOTE artifacts may only have "
                            "ADD-type tags.");
    }
  }
  if(!t->name || !*t->name){
    return fsl_cx_err_set(mf->f, FSL_RC_SYNTAX,
                         "Tag name may not be empty.");
  }else if(fsl_validate16(t->name, fsl_strlen(t->name))){
    return fsl_cx_err_set(mf->f, FSL_RC_SYNTAX,
                         "Tag name may not be hexadecimal.");
  }else if(t->uuid && !fsl_is_uuid(t->uuid)){
    return fsl_cx_err_set(mf->f, FSL_RC_SYNTAX,
                         "Invalid UUID in tag.");
  }
  return fsl_list_append(&mf->T, t);
}

int fsl_deck_T_add( fsl_deck * const mf, fsl_tagtype_e tagType,
                    char const * uuid, char const * name,
                    char const * value){
  if(!name) return FSL_RC_MISUSE;
  else if(!fsl_deck_check_type(mf, 'T')) return mf->f->error.code;
  else if(!*name || (uuid &&!fsl_is_uuid(uuid))) return FSL_RC_SYNTAX;
  else switch(tagType){
    case FSL_TAGTYPE_CANCEL:
    case FSL_TAGTYPE_ADD:
    case FSL_TAGTYPE_PROPAGATING:{
      int rc;
      fsl_card_T * t;
      t = fsl_card_T_malloc(tagType, uuid, name, value);
      if(!t) return FSL_RC_OOM;
      rc = fsl_deck_T_add2( mf, t );
      if(rc) fsl_card_T_free(t);
      return rc;
    }
    default:
      assert(!"Invalid tagType value");
      return fsl_cx_err_set(mf->f, FSL_RC_SYNTAX,
                            "Invalid tag-type value: %d",
                            (int)tagType);
  }
}

/**
   Returns true if the NUL-terminated string contains only
   "reasonable" branch name character, with the native assumption that
   anything <=32d is "unreasonable" and anything >=128 is part of a
   multibyte UTF8 character.
*/
static bool fsl_is_valid_branchname(char const * z_){
  unsigned char const * z = (unsigned char const*)z_;
  unsigned len = 0;
  for(; z[len]; ++len){
    if(z[len] <= 32) return false;
  }
  return len>0;
}

int fsl_deck_branch_set( fsl_deck * const d, char const * branchName ){
  if(!fsl_is_valid_branchname(branchName)){
    return fsl_cx_err_set(d->f, FSL_RC_RANGE, "Branch name contains "
                          "invalid characters.");
  }
  int rc= fsl_deck_T_add(d, FSL_TAGTYPE_PROPAGATING, NULL,
                         "branch", branchName);
  if(!rc){
    char * sym = fsl_mprintf("sym-%s", branchName);
    if(sym){
      rc = fsl_deck_T_add(d, FSL_TAGTYPE_PROPAGATING, NULL,
                          sym, NULL);
      fsl_free(sym);
    }else{
      rc = FSL_RC_OOM;
    }
  }
  return rc;
}

int fsl_deck_U_set( fsl_deck * const mf, char const * v){
  return fsl_deck_set_string( mf, 'U', &mf->U, v, -1 );
}

int fsl_deck_W_set( fsl_deck * const mf, char const * v, fsl_int_t n){
  return fsl_deck_b_setuffer_impl(mf, v, n, 'W', &mf->W);
}

int fsl_deck_A_set( fsl_deck * const mf, char const * name,
                    char const * tgt,
                    char const * uuidSrc ){
  int const uLen = (uuidSrc && *uuidSrc) ? fsl_is_uuid(uuidSrc) : 0;
  if(!name || !tgt) return FSL_RC_MISUSE;
  else if(!fsl_deck_check_type(mf, 'A')) return mf->f->error.code;
  else if(!*tgt){
    return fsl_cx_err_set(mf->f, FSL_RC_SYNTAX,
                          "Invalid target name in A card.");
  }
  /* TODO: validate tgt based on mf->type and require UUID
     for types EVENT/TICKET.
  */
  else if(uuidSrc && *uuidSrc && !uLen){
    return fsl_cx_err_set(mf->f, FSL_RC_SYNTAX,
                          "Invalid source UUID in A card.");
  }
  else{
    int rc = 0;
    fsl_deck_free_string(mf, mf->A.tgt);
    fsl_deck_free_string(mf, mf->A.src);
    fsl_deck_free_string(mf, mf->A.name);
    mf->A.name = mf->A.src = NULL;
    if(! (mf->A.tgt = fsl_strdup(tgt))) rc = FSL_RC_OOM;
    else if( !(mf->A.name = fsl_strdup(name))) rc = FSL_RC_OOM;
    else if(uLen){
      mf->A.src = fsl_strndup(uuidSrc,uLen);
      if(!mf->A.src) rc = FSL_RC_OOM
        /* Leave mf->A.tgt/name for downstream cleanup. */;
    }
    return rc;
  }
}


int fsl_deck_D_set( fsl_deck * const mf, double date){
  if(date<0) return FSL_RC_RANGE;
  else if(date>0 && !fsl_deck_check_type(mf, 'D')){
    return mf->f->error.code;
  }else{
    mf->D = date;
    return 0;
  }
}

int fsl_deck_E_set( fsl_deck * const mf, double date, char const * uuid){
  int const uLen = uuid ? fsl_is_uuid(uuid) : 0;
  if(!mf || !uLen) return FSL_RC_MISUSE;
  else if(date<=0){
    return fsl_cx_err_set(mf->f, FSL_RC_RANGE,
                          "Invalid date value for E card.");
  }else if(!uLen){
    return fsl_cx_err_set(mf->f, FSL_RC_RANGE,
                          "Invalid UUID for E card.");
  }
  else{
    mf->E.julian = date;
    fsl_deck_free_string(mf, mf->E.uuid);
    mf->E.uuid = fsl_strndup(uuid, uLen);
    return mf->E.uuid ? 0 : FSL_RC_OOM;
  }
}

int fsl_deck_F_add( fsl_deck * const mf, char const * name,
                    char const * uuid,
                    fsl_fileperm_e perms, 
                    char const * oldName){
  int const uLen = uuid ? fsl_is_uuid(uuid) : 0;
  if(!mf || !name) return FSL_RC_MISUSE;
  else if(!uuid && !mf->B.uuid){
    return fsl_cx_err_set(mf->f, FSL_RC_MISUSE,
                         "NULL UUID is not valid for baseline "
                          "manifests.");
  }
  else if(!fsl_deck_check_type(mf, 'F')) return mf->f->error.code;
  else if(!*name){
    return fsl_cx_err_set(mf->f, FSL_RC_RANGE,
                         "F-card name may not be empty.");
  }
  else if(!fsl_is_simple_pathname(name, 1)
          || (oldName && !fsl_is_simple_pathname(oldName, 1))){
    return fsl_cx_err_set(mf->f, FSL_RC_RANGE,
                          "Invalid filename for F-card (simple form required): "
                          "name=[%s], oldName=[%s].", name, oldName);
  }
  else if(uuid && !uLen){
    return fsl_cx_err_set(mf->f, FSL_RC_RANGE,
                          "Invalid UUID for F-card.");
  }
  else {
    int rc = 0;
    fsl_card_F * t;
    switch(perms){
      case FSL_FILE_PERM_EXE:
      case FSL_FILE_PERM_LINK:
      case FSL_FILE_PERM_REGULAR:
        break;
      default:
        assert(!"Invalid fsl_fileperm_e value");
        return fsl_cx_err_set(mf->f, FSL_RC_RANGE,
                              "Invalid fsl_fileperm_e value "
                              "(%d) for file [%s].",
                              perms, name);
    }
    t = fsl_card_F_list_push(&mf->F);
    if(!t) return FSL_RC_OOM;
    assert(mf->F.used>1
           ? (FSL_CARD_F_LIST_NEEDS_SORT & mf->F.flags)
           : 1);
    assert(!t->name);
    assert(!t->uuid);
    assert(!t->priorName);
    assert(!t->deckOwnsStrings);
    t->perm = perms;
    if(0==(t->name = fsl_strdup(name))){
      rc = FSL_RC_OOM;
    }else if(uuid && 0==(t->uuid = fsl_strdup(uuid))){
      rc = FSL_RC_OOM;
    }else if(oldName && 0==(t->priorName = fsl_strdup(oldName))){
      rc = FSL_RC_OOM;
    }
    if(rc){
      fsl_card_F_list_pop(&mf->F);
    }
    return rc;
  }
}

int fsl_deck_F_foreach( fsl_deck * const d, fsl_card_F_visitor_f cb, void * const visitorState ){
  if(!cb) return FSL_RC_MISUSE;
  else{
    fsl_card_F const * fc;
    int rc = fsl_deck_F_rewind(d);
    while( !rc && !(rc=fsl_deck_F_next(d, &fc)) && fc) {
      rc = cb( fc, visitorState );
    }
    return (FSL_RC_BREAK==rc) ? 0 : rc;
  }
}

  
/**
    Output state for fsl_output_f_mf() and friends. Used for managing
    the output of a fsl_deck.
 */
struct fsl_deck_out_state {
  /**
      The set of cards being output. We use this to delegate certain
      output bits.
   */
  fsl_deck const * d;
  /**
      Output routine to send manifest to.
   */
  fsl_output_f out;
  /**
      State to pass as the first arg of this->out().
   */
  void * outState;

  /**
     The previously-visited card, for confirming that all cards are in
     proper lexical order.
   */
  fsl_card_F const * prevCard;
  /**
      f() result code, so that we can feed the code back through the
      fsl_appendf() layer. If this is non-0, processing must stop.  We
      "could" use this->error.code instead, but this is simple.
   */
  int rc;
  /**
      Counter for list-visiting routines. Must be re-set before each
      visit loop if the visitor makes use of this (most do not).
   */
  fsl_int_t counter;

  /**
      Incrementally-calculated MD5 sum of all output sent via
      fsl_output_f_mf().
   */
  fsl_md5_cx md5;

  /* Holds error state for propagating back to the client. */
  fsl_error error;

  /**
      Scratch buffer for fossilizing bytes and other temporary work.
      This value comes from fsl__cx_scratchpad().
  */
  fsl_buffer * scratch;
};
typedef struct fsl_deck_out_state fsl_deck_out_state;
static const fsl_deck_out_state fsl_deck_out_state_empty = {
NULL/*d*/,
NULL/*out*/,
NULL/*outState*/,
NULL/*prevCard*/,
0/*rc*/,
0/*counter*/,
fsl_md5_cx_empty_m/*md5*/,
fsl_error_empty_m/*error*/,
NULL/*scratch*/
};

/**
    fsl_output_f() impl which forwards its data to arg->out(). arg
    must be a (fsl_deck_out_state *). Updates arg->rc to the result of
    calling arg->out(fp->fState, data, n). If arg->out() succeeds then
    arg->md5 is updated to reflect the given data. i.e. this is where
    the Z-card gets calculated incrementally during output of a deck.
*/
static int fsl_output_f_mf( void * arg, void const * data,
                            fsl_size_t n ){
  fsl_deck_out_state * const os = (fsl_deck_out_state *)arg;
  if((n>0)
     && !(os->rc = os->out(os->outState, data, (fsl_size_t)n))
     && (os->md5.isInit)){
    fsl_md5_update( &os->md5, data, (fsl_size_t)n );
  }
  return os->rc;
}

/**
    Internal helper for fsl_deck_output(). Appends formatted output to
    os->out() via fsl_output_f_mf(). Returns os->rc (0 on success).
 */
static int fsl_deck_append( fsl_deck_out_state * const os,
                            char const * fmt, ... ){
  fsl_int_t rc;
  va_list args;
  assert(os);
  assert(fmt && *fmt);
  va_start(args,fmt);
  rc = fsl_appendfv( fsl_output_f_mf, os, fmt, args);
  va_end(args);
  if(rc<0 && !os->rc) os->rc = FSL_RC_IO;
  return os->rc;
}

/**
    Fossilizes (inp, inp+len] bytes to os->scratch,
    overwriting any existing contents.
    Updates and returns os->rc.
 */
static int fsl_deck_fossilize( fsl_deck_out_state * const os,
                               unsigned char const * inp,
                               fsl_int_t len){
  fsl_buffer_reuse(os->scratch);
  return os->rc = len
    ? fsl_bytes_fossilize(inp, len, os->scratch)
    : 0;
}

/** Confirms that the given card letter is valid for od->d->type, and
    updates os->rc and os->error if it's not. Returns true if it's
    valid.
*/
static bool fsl_deck_out_tcheck(fsl_deck_out_state * const os, char letter){
  if(!fsl_card_is_legal(os->d->type, letter)){
    os->rc = fsl_error_set(&os->error, FSL_RC_TYPE,
                           "%c-card is not valid for deck type %s.",
                           letter, fsl_satype_cstr(os->d->type));
  }
  return os->rc ? false : true;
}

/* Appends a UUID-valued card to os from os->d->{{card}} if the given
   UUID is not NULL, else this is a no-op. */
static int fsl_deck_out_uuid( fsl_deck_out_state * const os, char card, fsl_uuid_str uuid ){
  if(uuid && fsl_deck_out_tcheck(os, card)){
    if(!fsl_is_uuid(uuid)){
      os->rc = fsl_error_set(&os->error, FSL_RC_RANGE,
                             "Malformed UUID in %c card.", card);
    }else{
      fsl_deck_append(os, "%c %s\n", card, uuid);
    }
  }
  return os->rc;
}

/* Appends the B card to os from os->d->B. */
static int fsl_deck_out_B( fsl_deck_out_state * const os ){
  return fsl_deck_out_uuid(os, 'B', os->d->B.uuid);
}


/* Appends the A card to os from os->d->A. */
static int fsl_deck_out_A( fsl_deck_out_state * const os ){
  if(os->d->A.name && fsl_deck_out_tcheck(os, 'A')){
    if(!os->d->A.name || !*os->d->A.name){
      os->rc = fsl_error_set(&os->error, FSL_RC_SYNTAX,
                             "A-card is missing its name property");

    }else if(!os->d->A.tgt || !*os->d->A.tgt){
      os->rc = fsl_error_set(&os->error, FSL_RC_SYNTAX,
                             "A-card is missing its tgt property: %s",
                             os->d->A.name);

    }else if(os->d->A.src && !fsl_is_uuid(os->d->A.src)){
      os->rc = fsl_error_set(&os->error, FSL_RC_TYPE,
                             "Invalid src UUID in A-card: name=%s, "
                             "invalid uuid=%s",
                             os->d->A.name, os->d->A.src);
    }else{
      fsl_deck_append(os, "A %F %F",
                      os->d->A.name, os->d->A.tgt);
      if(!os->rc){
        if(os->d->A.src){
          fsl_deck_append(os, " %s", os->d->A.src);
        }
        if(!os->rc) fsl_deck_append(os, "\n");
      }
    }
  }
  return os->rc;
}

/**
    Internal helper for outputing cards which are simple strings.
    str is the card to output (NULL values are ignored), letter is
    the card letter being output. If doFossilize is true then
    the output gets fossilize-formatted.
 */
static int fsl_deck_out_letter_str( fsl_deck_out_state * const os, char letter,
                                    char const * str, char doFossilize ){
  if(str && fsl_deck_out_tcheck(os, letter)){
    if(doFossilize){
      fsl_deck_fossilize(os, (unsigned char const *)str, -1);
      if(!os->rc){
        fsl_deck_append(os, "%c %b\n", letter, os->scratch);
      }
    }else{
      fsl_deck_append(os, "%c %s\n", letter, str);
    }
  }
  return os->rc;
}

/* Appends the C card to os from os->d->C. */
static int fsl_deck_out_C( fsl_deck_out_state * const os ){
  return fsl_deck_out_letter_str( os, 'C', os->d->C, 1 );
}


/* Appends the D card to os from os->d->D. */
static int fsl_deck_out_D( fsl_deck_out_state * const os ){
  if((os->d->D > 0.0) && fsl_deck_out_tcheck(os, 'D')){
    char ds[24];
    if(!fsl_julian_to_iso8601(os->d->D, ds, 1)){
      os->rc = fsl_error_set(&os->error, FSL_RC_RANGE,
                             "D-card contains invalid "
                             "Julian Day value.");
    }else{
      fsl_deck_append(os, "D %s\n", ds);
    }
  }
  return os->rc;
}

/* Appends the E card to os from os->d->E. */
static int fsl_deck_out_E( fsl_deck_out_state * const os ){
  if(os->d->E.uuid && fsl_deck_out_tcheck(os, 'E')){
    char ds[24];
    char msPrecision = FSL_SATYPE_EVENT!=os->d->type
      /* The timestamps on Events historically have seconds precision,
         not ms.
      */;
    if(!fsl_is_uuid(os->d->E.uuid)){
      os->rc = fsl_error_set(&os->error, FSL_RC_TYPE,
                             "Invalid UUID in E-card: %s",
                             os->d->E.uuid);
    }
    else if(!fsl_julian_to_iso8601(os->d->E.julian, ds, msPrecision)){
      os->rc = fsl_error_set(&os->error, FSL_RC_TYPE,
                             "Invalid Julian Day value in E-card.");
    }
    else{
      fsl_deck_append(os, "E %s %s\n", ds, os->d->E.uuid);
    }
  }
  return os->rc;
}

/* Appends the G card to os from os->d->G. */
static int fsl_deck_out_G( fsl_deck_out_state * const os ){
  return fsl_deck_out_uuid(os, 'G', os->d->G);
}

/* Appends the H card to os from os->d->H. */
static int fsl_deck_out_H( fsl_deck_out_state * const os ){
  if(os->d->H && os->d->I){
    return os->rc = fsl_error_set(&os->error, FSL_RC_SYNTAX,
                                  "Forum post may not have both H- and I-cards.");
  }
  return fsl_deck_out_letter_str( os, 'H', os->d->H, 1 );
}

/* Appends the I card to os from os->d->I. */
static int fsl_deck_out_I( fsl_deck_out_state * const os ){
  if(os->d->I && os->d->H){
    return os->rc = fsl_error_set(&os->error, FSL_RC_SYNTAX,
                                  "Forum post may not have both H- and I-cards.");
  }
  return fsl_deck_out_uuid(os, 'I', os->d->I);
}


static int fsl_deck_out_F_one(fsl_deck_out_state *os,
                              fsl_card_F const * f){
  int rc;
  char hasOldName;
  char const * zPerm;
  assert(f);
  if(os->prevCard){
    int const cmp = fsl_strcmp(os->prevCard->name, f->name);
    if(0==cmp){
      return fsl_error_set(&os->error, FSL_RC_RANGE,
                           "Duplicate F-card name: %s",
                           f->name);
    }else if(cmp>0){
      return fsl_error_set(&os->error, FSL_RC_RANGE,
                           "Out-of-order F-card names: %s before %s",
                           os->prevCard->name, f->name);
    }
  }
  if(!fsl_is_simple_pathname(f->name, true)){
    return fsl_error_set(&os->error, FSL_RC_RANGE,
                         "Filename is invalid as F-card: %s",
                         f->name);
  }
  if(!f->uuid && !os->d->B.uuid){
    return fsl_error_set(&os->error, FSL_RC_MISUSE,
                         "Baseline manifests may not have F-cards "
                         "without UUIDs (file deletion entries). To "
                         "delete files, simply do not inject an F-card "
                         "for them. Delta manifests, however, require "
                         "NULL UUIDs for deletion entries! File: %s",
                         f->name);
  }

  rc = fsl_deck_fossilize(os, (unsigned char const *)f->name, -1);
  if(!rc) rc = fsl_deck_append(os, "F %b", os->scratch);
  if(!rc && f->uuid){
    assert(fsl_is_uuid(f->uuid));
    rc = fsl_deck_append( os, " %s", f->uuid);
    if(rc) return rc;
  }
  if(f->uuid){
    hasOldName = f->priorName && (0!=fsl_strcmp(f->name,f->priorName));
    switch(f->perm){
      case FSL_FILE_PERM_EXE: zPerm = " x"; break;
      case FSL_FILE_PERM_LINK: zPerm = " l"; break;
      default:
        /* When hasOldName, we have to inject an otherwise optional
           'w' to avoid an ambiguity. Or at least that's what the
           fossil F-card-generating code does.
        */
        zPerm = hasOldName ? " w" : ""; break;
    }
    if(*zPerm) rc = fsl_deck_append( os, "%s", zPerm);
    if(!rc && hasOldName){
      assert(*zPerm);
      rc = fsl_deck_fossilize(os, (unsigned char const *)f->priorName, -1);
      if(!rc) rc = fsl_deck_append( os, " %b", os->scratch);
    }
  }
  if(!rc) fsl_output_f_mf(os, "\n", 1);
  return os->rc;
}

static int fsl_deck_out_list_obj( fsl_deck_out_state * const os,
                                  char letter,
                                  fsl_list const * li,
                                  fsl_list_visitor_f visitor){
  if(li->used && fsl_deck_out_tcheck(os, letter)){
    os->rc = fsl_list_visit( li, 0, visitor, os );
  }
  return os->rc;
}

static int fsl_deck_out_F( fsl_deck_out_state * const os ){
  if(os->d->F.used && fsl_deck_out_tcheck(os, 'F')){
    uint32_t i;
    for(i=0; !os->rc && i <os->d->F.used; ++i){
      os->rc = fsl_deck_out_F_one(os, F_at(&os->d->F, i));
    }
  }
  return os->rc;
}


/**
    A comparison routine for qsort(3) which compares fsl_card_J
    instances in a lexical manner based on their names.  The order is
    important for card ordering in generated manifests.
 */
int fsl__qsort_cmp_J_cards( void const * lhs, void const * rhs ){
  fsl_card_J const * l = *((fsl_card_J const **)lhs);
  fsl_card_J const * r = *((fsl_card_J const **)rhs);
  /* Compare NULL as larger so that NULLs move to the right. That said,
     we aren't expecting any NULLs. */
  assert(l);
  assert(r);
  if(!l) return r ? 1 : 0;
  else if(!r) return -1;
  else{
    /* The '+' sorts before any legal field name bits (letters). */
    if(l->append != r->append) return r->append - l->append
      /* Footnote: that will break if, e.g. l->isAppend==2 and
         r->isAppend=1, or some such. Shame C89 doesn't have a true
         boolean.
      */;
    else return fsl_strcmp(l->field, r->field);
  }
}

/**
    fsl_list_visitor_f() impl for outputing J cards. obj must
    be a (fsl_card_J *).
 */
static int fsl_list_v_mf_output_card_J(void * obj, void * visitorState ){
  fsl_deck_out_state * const os = (fsl_deck_out_state *)visitorState;
  fsl_card_J const * c = (fsl_card_J const *)obj;
  fsl_deck_fossilize( os, (unsigned char const *)c->field, -1 );
  if(!os->rc){
    fsl_deck_append(os, "J %s%b", c->append ? "+" : "", os->scratch);
    if(!os->rc){
      if(c->value && *c->value){
        fsl_deck_fossilize( os, (unsigned char const *)c->value, -1 );
        if(!os->rc){
          fsl_deck_append(os, " %b\n", os->scratch);
        }
      }else{
        fsl_deck_append(os, "\n");
      }
    }
  }
  return os->rc;
}

static int fsl_deck_out_J( fsl_deck_out_state * const os ){
  return fsl_deck_out_list_obj(os, 'J', &os->d->J,
                               fsl_list_v_mf_output_card_J);
}

/* Appends the K card to os from os->d->K. */
static int fsl_deck_out_K( fsl_deck_out_state * const os ){
  if(os->d->K && fsl_deck_out_tcheck(os, 'K')){
    if(!fsl_is_uuid(os->d->K)){
      os->rc = fsl_error_set(&os->error, FSL_RC_RANGE,
                             "Invalid UUID in K card.");
    }
    else{
      fsl_deck_append(os, "K %s\n", os->d->K);
    }
  }
  return os->rc;
}


/* Appends the L card to os from os->d->L. */
static int fsl_deck_out_L( fsl_deck_out_state * const os ){
  return fsl_deck_out_letter_str(os, 'L', os->d->L, 1);
}

/* Appends the N card to os from os->d->N. */
static int fsl_deck_out_N( fsl_deck_out_state * const os ){
  return fsl_deck_out_letter_str( os, 'N', os->d->N, 1 );
}

/**
    fsl_list_visitor_f() impl for outputing P cards. obj must
    be a (fsl_deck_out_state *) and obj->counter must be
    set to 0 before running the visit iteration.
 */
static int fsl_list_v_mf_output_card_P(void * obj, void * visitorState ){
  fsl_deck_out_state * const os = (fsl_deck_out_state *)visitorState;
  char const * uuid = (char const *)obj;
  int const uLen = uuid ? fsl_is_uuid(uuid) : 0;
  if(!uLen){
    os->rc = fsl_error_set(&os->error, FSL_RC_RANGE,
                           "Invalid UUID in P card.");
  }
  else if(!os->counter++) fsl_output_f_mf(os, "P ", 2);
  else fsl_output_f_mf(os, " ", 1);
  /* Reminder: fsl_appendf_f_mf() updates os->rc. */
  if(!os->rc){
    fsl_output_f_mf(os, uuid, (fsl_size_t)uLen);
  }
  return os->rc;
}


static int fsl_deck_out_P( fsl_deck_out_state * const os ){
  if(!fsl_deck_out_tcheck(os, 'P')) return os->rc;
  else if(os->d->P.used){
    os->counter = 0;
    os->rc = fsl_list_visit( &os->d->P, 0, fsl_list_v_mf_output_card_P, os );
    assert(os->counter);
    if(!os->rc) fsl_output_f_mf(os, "\n", 1);
  }
#if 1
  /* Arguable: empty P-cards are harmless but cosmetically unsightly. */
  else if(FSL_SATYPE_CHECKIN==os->d->type){
    /*
      Evil ugly hack, primarily for round-trip compatibility with
      manifest #1, which has an empty P card.

      fossil(1) ignores empty P-cards in all cases, and must continue
      to do so for backwards compatibility with rid #1 in all repos.

      Pedantic note: there must be no space between the 'P' and the
      newline.
    */
    fsl_deck_append(os, "P\n");
  }
#endif
  return os->rc;
}

/**
    A comparison routine for qsort(3) which compares fsl_card_Q
    instances in a lexical manner. The order is important for card
    ordering in generated manifests.
*/
static int qsort_cmp_Q_cards( void const * lhs, void const * rhs ){
  fsl_card_Q const * l = *((fsl_card_Q const **)lhs);
  fsl_card_Q const * r = *((fsl_card_Q const **)rhs);
  /* Compare NULL as larger so that NULLs move to the right. That said,
     we aren't expecting any NULLs. */
  assert(l);
  assert(r);
  if(!l) return r ? 1 : 0;
  else if(!r) return -1;
  else{
    /* Lexical sorting must account for the +/- characters, and a '+'
       sorts before '-', which is why this next part may seem
       backwards at first. */
    assert(l->type);
    assert(r->type);
    if(l->type<0 && r->type>0) return 1;
    else if(l->type>0 && r->type<0) return -1;
    else return fsl_strcmp(l->target, r->target);
  }
}

/**
    fsl_list_visitor_f() impl for outputing Q cards. obj must
    be a (fsl_deck_out_state *).
*/
static int fsl_list_v_mf_output_card_Q(void * obj, void * visitorState ){
  fsl_deck_out_state * const os = (fsl_deck_out_state *)visitorState;
  fsl_card_Q const * cp = (fsl_card_Q const *)obj;
  char const prefix = (cp->type==FSL_CHERRYPICK_ADD)
    ? '+' : '-';
  assert(cp->type);
  assert(cp->target);
  if(cp->type != FSL_CHERRYPICK_ADD &&
     cp->type != FSL_CHERRYPICK_BACKOUT){
    os->rc = fsl_error_set(&os->error, FSL_RC_RANGE,
                           "Invalid type value in Q-card.");
  }else if(!fsl_card_is_legal(os->d->type, 'Q')){
    os->rc = fsl_error_set(&os->error, FSL_RC_TYPE,
                           "Q-card is not valid for deck type %s",
                           fsl_satype_cstr(os->d->type));
  }else if(!fsl_is_uuid(cp->target)){
    os->rc = fsl_error_set(&os->error, FSL_RC_RANGE,
                           "Invalid target UUID in Q-card: %s",
                           cp->target);
  }else if(cp->baseline){
    if(!fsl_is_uuid(cp->baseline)){
      os->rc = fsl_error_set(&os->error, FSL_RC_RANGE,
                             "Invalid baseline UUID in Q-card: %s",
                             cp->baseline);
    }else{
      fsl_deck_append(os, "Q %c%s %s\n", prefix, cp->target, cp->baseline);
    }
  }else{
    fsl_deck_append(os, "Q %c%s\n", prefix, cp->target);
  }
  return os->rc;
}

static int fsl_deck_out_Q( fsl_deck_out_state * const os ){
  return fsl_deck_out_list_obj(os, 'Q', &os->d->Q,
                               fsl_list_v_mf_output_card_Q);
}

/**
    Appends the R card from os->d->R to os.
 */
static int fsl_deck_out_R( fsl_deck_out_state * const os ){
  if(os->d->R && fsl_deck_out_tcheck(os, 'R')){
    if((FSL_STRLEN_MD5!=fsl_strlen(os->d->R))
            || !fsl_validate16(os->d->R, FSL_STRLEN_MD5)){
      os->rc = fsl_error_set(&os->error, FSL_RC_RANGE,
                             "Malformed MD5 in R-card.");
    }
    else{
      fsl_deck_append(os, "R %s\n", os->d->R);
    }
  }
  return os->rc;
}

/**
    fsl_list_visitor_f() impl for outputing T cards. obj must
    be a (fsl_deck_out_state *).
 */
static int fsl_list_v_mf_output_card_T(void * obj, void * visitorState ){
  fsl_deck_out_state * const os = (fsl_deck_out_state *)visitorState;
  fsl_card_T * t = (fsl_card_T *)obj;
  char prefix = 0;
  switch(os->d->type){
    case FSL_SATYPE_TECHNOTE:
      if( t->uuid ){
        return os->rc = fsl_error_set(&os->error, FSL_RC_SYNTAX,
                                      "Non-self-referential T-card is not "
                                      "permitted in a technote.");
      }else if(FSL_TAGTYPE_ADD!=t->type){
        return os->rc = fsl_error_set(&os->error, FSL_RC_SYNTAX,
                                      "Non-ADD T-card is not permitted "
                                      "in a technote.");
      }
      break;
    case FSL_SATYPE_CONTROL:
      if( !t->uuid ){
        return os->rc = fsl_error_set(&os->error, FSL_RC_SYNTAX,
                                      "Self-referential T-card is not "
                                      "permitted in a control artifact.");
      }
      break;
    default:
      break;
  }
  /* Determine the prefix character... */
  switch(t->type){
    case FSL_TAGTYPE_CANCEL: prefix = '-'; break;
    case FSL_TAGTYPE_ADD: prefix = '+'; break;
    case FSL_TAGTYPE_PROPAGATING: prefix = '*'; break;
    default:
      return os->rc = fsl_error_set(&os->error, FSL_RC_TYPE,
                                    "Invalid tag type #%d in T-card.",
                                    t->type);
  }
  if(!t->name || !*t->name){
    return os->rc = fsl_error_set(&os->error, FSL_RC_SYNTAX,
                                  "T-card name may not be empty.");
  }else if(fsl_validate16(t->name, fsl_strlen(t->name))){
    return os->rc = fsl_error_set(&os->error, FSL_RC_SYNTAX,
                                  "T-card name may not be hexadecimal.");
  }else if(t->uuid && !fsl_is_uuid(t->uuid)){
      return os->rc = fsl_error_set(&os->error, FSL_RC_SYNTAX,
                                    "Malformed UUID in T-card: %s",
                                    t->uuid);
  }
  /*
     Fossilize and output the prefix, name, and uuid, or a '*' if no
     uuid is set (which is only legal when tagging the current
     artifact, as '*' is a placeholder for the current artifact's
     UUID, which is not yet known).
  */
  fsl_buffer_reuse(os->scratch);
  fsl_deck_fossilize(os, (unsigned const char *)t->name, -1);
  if(os->rc) return os->rc;
  os->rc = fsl_deck_append(os, "T %c%s %s", prefix,
                           (char const*)os->scratch->mem,
                           t->uuid ? t->uuid : "*");
  if(os->rc) return os->rc;
  if(/*(t->type != FSL_TAGTYPE_CANCEL) &&*/t->value && *t->value){
    /* CANCEL tags historically don't store a value but
       the spec doesn't disallow it and they are harmless
       for (aren't used by) fossil(1). */
    fsl_deck_fossilize(os, (unsigned char const *)t->value, -1);
    if(!os->rc) fsl_output_f_mf(os, " ", 1);
    if(!os->rc) fsl_output_f_mf(os, (char const*)os->scratch->mem,
                                (fsl_int_t)os->scratch->used);
  }
  if(!os->rc){
    fsl_output_f_mf(os, "\n", 1);
  }
  return os->rc;
}


char fsl_tag_prefix_char( fsl_tagtype_e t ){
  switch(t){
    case FSL_TAGTYPE_CANCEL: return '-';
    case FSL_TAGTYPE_ADD: return '+';
    case FSL_TAGTYPE_PROPAGATING: return '*';
    default:
      return 0;
  }
}

/**
    A comparison routine for qsort(3) which compares fsl_card_T
    instances in a lexical manner based on (type, name, uuid, value).
    The order of those is important for card ordering in generated
    manifests. Interestingly, CANCEL tags (with a '-' prefix) sort
    last, meaning it is possible to cancel a tag set in the same
    manifest because crosslinking processes them in the order given
    (which will be lexical order for all legal manifests).

    Reminder: lhs and rhs must be (fsl_card_T**), as we use this to
    qsort() such lists. When using it to compare two tags, make sure
    to pass ptr-to-ptr.
*/
static int fsl_card_T_cmp( void const * lhs, void const * rhs ){
  fsl_card_T const * l = *((fsl_card_T const **)lhs);
  fsl_card_T const * r = *((fsl_card_T const **)rhs);
  /* Compare NULL as larger so that NULLs move to the right. That said,
     we aren't expecting any NULLs. */
  assert(l);
  assert(r);
  if(!l) return r ? 1 : 0;
  else if(!r) return -1;
  else if(l->type != r->type){
    char const lc = fsl_tag_prefix_char(l->type);
    char const rc = fsl_tag_prefix_char(r->type);
    return (lc<rc) ? -1 : 1;
  }else{
    int rc = fsl_strcmp(l->name, r->name);
    if(rc) return rc;
    else {
      rc = fsl_uuidcmp(l->uuid, r->uuid);
      return rc
        ? rc
        : fsl_strcmp(l->value, r->value);
    }
  }
}

/**
   Confirms that any T-cards in d are properly sorted. If not,
   returns non-0. If err is not NULL, it is updated with a
   description of the problem.

   Possibly fixme one day: this code permits that the same tag/target
   combination may be added or removed, or added as a normal and
   propagating tag, in the same deck. Though that's not technically
   disallowed, we "should" disallow it. That requires a more thorough
   scan of the cards, though.
*/
static int fsl_deck_T_verify_order( fsl_deck const * d, fsl_error * err ){
  if(d->T.used<2) return 0;
  else{
    fsl_size_t i = 0, j;
    int rc = 0;
    fsl_card_T const * tag;
    fsl_card_T const * prev = NULL;
    for( i = 0; i < d->T.used; ++i, prev = tag, rc = 0){
      tag = (fsl_card_T const *)d->T.list[i];
      if(prev){
        if( (rc = fsl_card_T_cmp(&prev, &tag)) >= 0 ){
          if(!err) rc = FSL_RC_SYNTAX;
          else{
            rc = rc
              ? fsl_error_set(err, FSL_RC_SYNTAX,
                              "Invalid T-card order: "
                              "[%c%s] must precede [%c%s]",
                              fsl_tag_prefix_char(prev->type),
                              prev->name,
                              fsl_tag_prefix_char(tag->type),
                              tag->name)
              : fsl_error_set(err, FSL_RC_SYNTAX,
                              "Duplicate T-card: %c%s",
                              fsl_tag_prefix_char(prev->type),
                              prev->name)
              ;
          }
          break;
        }
      }
    }
    /**
       And now, for bonus points: disallow the same tag name/artifact
       combination appearing twice in the deck. Though that's not
       explicitly disallowed by the fossil specs, we "should" disallow
       it. That requires a more thorough scan of the cards, though.

       This logic is NOT in fossil, and though we don't have any such
       tags in the fossil repo, we may have to disable this for
       compatibility's sake. OTOH, we only check this when outputing
       manifests, and we never (aside from testing) have to output
       manifests which were generated by fossil. Thus... this only
       triggers (except for some tests) on manifests generated by
       libfossil, so we can justify having it.
    */
    for( i=0; !rc && i < d->T.used; ++i ){
      fsl_card_T const * t1 = (fsl_card_T const *)d->T.list[i];
      for( j = 0; j < d->T.used; ++j ){
        if(i==j) continue;
        fsl_card_T const * t2 = (fsl_card_T const *)d->T.list[j];
        if(0==fsl_strcmp(t1->name, t2->name)
           && ((!t1->uuid && !t2->uuid)
               || 0==fsl_strcmp(t1->uuid, t2->uuid))){
              rc = fsl_error_set(err, FSL_RC_SYNTAX,
                                 "An artifact may not contain the same "
                                 "T-card name and target artifact "
                                 "multiple times: "
                                 "name=%s target=%s",
                                 t1->name, t1->uuid ? t1->uuid : "*");
              break;
        }
      }
    }
    return rc;
  }
}

/* Appends the T cards to os from os->d->T. */
static int fsl_deck_out_T( fsl_deck_out_state * const os ){
  os->rc = fsl_deck_T_verify_order( os->d, &os->error);
  return os->rc
    ? os->rc
    : fsl_deck_out_list_obj(os, 'T', &os->d->T,
                            fsl_list_v_mf_output_card_T);
}

/* Appends the U card to os from os->d->U. */
static int fsl_deck_out_U( fsl_deck_out_state * const os ){
  return fsl_deck_out_letter_str(os, 'U', os->d->U, 1);
}

/* Appends the W card to os from os->d->W. */
static int fsl_deck_out_W( fsl_deck_out_state * const os ){
  if(os->d->W.used && fsl_deck_out_tcheck(os, 'W')){
    fsl_deck_append(os, "W %"FSL_SIZE_T_PFMT"\n%b\n",
                    (fsl_size_t)os->d->W.used,
                    &os->d->W );
  }
  return os->rc;
}


/* Appends the Z card to os from os' accummulated md5 hash. */
static int fsl_deck_out_Z( fsl_deck_out_state * const os ){
  unsigned char digest[16];
  char md5[FSL_STRLEN_MD5+1];
  fsl_md5_final(&os->md5, digest);
  fsl_md5_digest_to_base16(digest, md5);
  assert(!md5[32]);
  os->md5.isInit = 0 /* Keep further output from updating the MD5 */;
  return fsl_deck_append(os, "Z %.*s\n", FSL_STRLEN_MD5, md5);
}

static int qsort_cmp_strings( void const * lhs, void const * rhs ){
  char const * l = *((char const **)lhs);
  char const * r = *((char const **)rhs);
  return fsl_strcmp(l,r);
}

static int fsl_list_v_mf_output_card_M(void * obj, void * visitorState ){
  fsl_deck_out_state * const os = (fsl_deck_out_state *)visitorState;
  char const * m = (char const *)obj;
  return fsl_deck_append(os, "M %s\n", m);
}

static int fsl_deck_output_cluster( fsl_deck_out_state * const os ){
  if(!os->d->M.used){
    os->rc = fsl_error_set(&os->error, FSL_RC_RANGE,
                           "M-card list may not be empty.");
  }else{
    fsl_deck_out_list_obj(os, 'M', &os->d->M,
                          fsl_list_v_mf_output_card_M);
  }
  return os->rc;
}


/* Helper for fsl_deck_output_CATYPE() */
#define DOUT(LETTER) rc = fsl_deck_out_##LETTER(os); \
  if(rc || os->rc) return os->rc ? os->rc : rc

static int fsl_deck_output_control( fsl_deck_out_state * const os ){
  int rc;
  /* Reminder: cards must be output in strict lexical order. */
  DOUT(D);
  DOUT(T);
  DOUT(U);
  return os->rc;
}

static int fsl_deck_output_event( fsl_deck_out_state * const os ){
  int rc = 0;
  /* Reminder: cards must be output in strict lexical order. */
  DOUT(C);
  DOUT(D);
  DOUT(E);
  DOUT(N);
  DOUT(P);
  DOUT(T);
  DOUT(U);
  DOUT(W);
  return os->rc;
}

static int fsl_deck_output_mf( fsl_deck_out_state * const os ){
  int rc = 0;
  /* Reminder: cards must be output in strict lexical order. */
  DOUT(B);
  DOUT(C);
  DOUT(D);
  DOUT(F);
  DOUT(K);
  DOUT(L);
  DOUT(N);
  DOUT(P);
  DOUT(Q);
  DOUT(R);
  DOUT(T);
  DOUT(U);
  DOUT(W);
  return os->rc;
}


static int fsl_deck_output_ticket( fsl_deck_out_state * const os ){
  int rc;
  /* Reminder: cards must be output in strict lexical order. */
  DOUT(D);
  DOUT(J);
  DOUT(K);
  DOUT(U);
  return os->rc;
}

static int fsl_deck_output_wiki( fsl_deck_out_state * const os ){
  int rc;
  /* Reminder: cards must be output in strict lexical order. */
  DOUT(C);
  DOUT(D);
  DOUT(L);
  DOUT(N);
  DOUT(P);
  DOUT(U);
  DOUT(W);
  return os->rc;
}

static int fsl_deck_output_attachment( fsl_deck_out_state * const os ){
  int rc = 0;
  /* Reminder: cards must be output in strict lexical order. */
  DOUT(A);
  DOUT(C);
  DOUT(D);
  DOUT(N);
  DOUT(U);
  return os->rc;
}

static int fsl_deck_output_forumpost( fsl_deck_out_state * const os ){
  int rc;
  /* Reminder: cards must be output in strict lexical order. */
  DOUT(D);
  DOUT(G);
  DOUT(H);
  DOUT(I);
  DOUT(N);
  DOUT(P);
  DOUT(U);
  DOUT(W);
  return os->rc;
}

/**
    Only for testing/debugging purposes, as it allows constructs which
    are not semantically legal and are CERTAINLY not legal to stuff in
    the database.
 */
static int fsl_deck_output_any( fsl_deck_out_state * const os ){
  int rc = 0;
  /* Reminder: cards must be output in strict lexical order. */
  DOUT(B);
  DOUT(C);
  DOUT(D);
  DOUT(E);
  DOUT(F);
  DOUT(J);
  DOUT(K);
  DOUT(L);
  DOUT(N);
  DOUT(P);
  DOUT(Q);
  DOUT(R);
  DOUT(T);
  DOUT(U);
  DOUT(W);
  return os->rc;
}

#undef DOUT


int fsl_deck_unshuffle( fsl_deck * const d, bool calculateRCard ){
  fsl_list * li;
  int rc = 0;
  if(!d || !d->f) return FSL_RC_MISUSE;
  fsl_cx_err_reset(d->f);
#define SORT(CARD,CMP) li = &d->CARD; fsl_list_sort(li, CMP)
  SORT(J,fsl__qsort_cmp_J_cards);
  SORT(M,qsort_cmp_strings);
  SORT(Q,qsort_cmp_Q_cards);
  SORT(T,fsl_card_T_cmp);
#undef SORT
  if(FSL_SATYPE_CHECKIN!=d->type){
    assert(!fsl_card_is_legal(d->type,'R'));
    assert(!fsl_card_is_legal(d->type,'F'));
  }else{
    assert(fsl_card_is_legal(d->type, 'R') && "in-lib unit testing");
    if(calculateRCard){
      rc = fsl_deck_R_calc(d) /* F-card list is sorted there */;
    }else{
      fsl_deck_F_sort(d);
      if(!d->R){
        rc = fsl_deck_R_set(d,
                            (d->F.used || d->B.uuid || d->P.used)
                            ? NULL
                            : FSL_MD5_INITIAL_HASH)
          /* Special case: for manifests with no (B,F,P)-cards we inject
             the initial-state R-card, analog to the initial checkin
             (RID 1). We need one of (B,F,P,R) to unambiguously identify
             a MANIFEST from a CONTROL, but RID 1 has an empty P-card,
             no F-cards, and no B-card, so it _needs_ an R-card in order
             to be unambiguously a Manifest. That said, that ambiguity
             is/would be harmless in practice because CONTROLs go
             through most of the same crosslinking processes as
             MANIFESTs (the ones which are important for this purpose,
             anyway).
          */;
      }
    }
  }
  return rc;
}

int fsl_deck_output( fsl_deck * const d, fsl_output_f out,
                     void * outputState ){
  static const bool allowTypeAny = false
    /* Only enable for debugging/testing. Allows outputing decks of
       type FSL_SATYPE_ANY, which bypasses some validation checks and
       may trigger other validation assertions. And may allow you to
       inject garbage into the repo. So be careful.
    */;

  fsl_deck_out_state OS = fsl_deck_out_state_empty;
  fsl_deck_out_state * const os = &OS;
  fsl_cx * const f = d->f;
  int rc = 0;
  if(NULL==out && NULL==outputState && f){
    out = f->output.out;
    outputState = f->output.state;
  }
  if(!f || !out) return FSL_RC_MISUSE;
  else if(FSL_SATYPE_ANY==d->type){
    if(!allowTypeAny){
      return fsl_cx_err_set(d->f, FSL_RC_TYPE,
                            "Artifact type ANY cannot be"
                            "output unless it is enabled in this "
                            "code (it's dangerous).");
    }
    /* fall through ... */
  }
  rc = fsl_deck_unshuffle(d,
                          (FSL_CX_F_CALC_R_CARD & f->flags)
                          ? ((d->F.used && !d->R) ? 1 : 0)
                          : 0);
  /* ^^^^ unshuffling might install an R-card, so we have to
     do that before checking whether all required cards are
     set... */  
  if(rc) return rc;
  else if(!fsl_deck_has_required_cards(d)){
    return FSL_RC_SYNTAX;
  }

  os->d = d;
  os->out = out;
  os->outState = outputState;
  os->scratch = fsl__cx_scratchpad(f);
  switch(d->type){
    case FSL_SATYPE_CLUSTER:
      rc = fsl_deck_output_cluster(os);
      break;
    case FSL_SATYPE_CONTROL:
      rc = fsl_deck_output_control(os);
      break;
    case FSL_SATYPE_EVENT:
      rc = fsl_deck_output_event(os);
      break;
    case FSL_SATYPE_CHECKIN:
      rc = fsl_deck_output_mf(os);
      break;
    case FSL_SATYPE_TICKET:
      rc = fsl_deck_output_ticket(os);
      break;
    case FSL_SATYPE_WIKI:
      rc = fsl_deck_output_wiki(os);
      break;
    case FSL_SATYPE_ANY:
      assert(allowTypeAny);
      rc = fsl_deck_output_any(os);
      break;
    case FSL_SATYPE_ATTACHMENT:
      rc = fsl_deck_output_attachment(os);
      break;
    case FSL_SATYPE_FORUMPOST:
      rc = fsl_deck_output_forumpost(os);
      break;
    default:
      rc = fsl_cx_err_set(f, FSL_RC_TYPE,
                          "Invalid/unhandled deck type (#%d).",
                          d->type);
      goto end;
  }
  if(!rc){
    rc = fsl_deck_out_Z( os );
  }
  end:
  fsl__cx_scratchpad_yield(f, os->scratch);
  if(os->rc && os->error.code){
    fsl_error_move(&os->error, &f->error);
  }
  fsl_error_clear(&os->error);
  return os->rc ? os->rc : rc;
}

/* Timestamps might be adjusted slightly to ensure that checkins appear
   on the timeline in chronological order.  This is the maximum amount
   of the adjustment window, in days.
*/
#define AGE_FUDGE_WINDOW      (2.0/86400.0)       /* 2 seconds */

/* This is increment (in days) by which timestamps are adjusted for
   use on the timeline.
*/
#define AGE_ADJUST_INCREMENT  (25.0/86400000.0)   /* 25 milliseconds */

/**
   Adds a record in the pending_xlink temp table, to be processed
   when crosslinking is completed. Returns 0 on success, non-0 for
   db error.
*/
static int fsl__deck_crosslink_add_pending(fsl_cx * f, char cType, fsl_uuid_cstr uuid){
  assert(f->cache.isCrosslinking);
  return fsl_cx_exec(f,
                   "INSERT OR IGNORE INTO pending_xlink VALUES('%c%q')",
                   cType, uuid);
}

/** @internal
   
    Add a single entry to the mlink table.  Also add the filename to
    the filename table if it is not there already.
   
    Parameters:
   
    pmid: Record for parent manifest. Use 0 to indicate no parent.

    zFromUuid: UUID for the content in parent (the new ==mlink.pid). 0
    or "" to add file.

    mid: The record ID of the manifest
   
    zToUuid:UUID for the mlink.fid. "" to delete
   
    zFilename: Filename
   
    zPrior: Previous filename. NULL if unchanged 
   
    isPublic:True if mid is not a private manifest
   
    isPrimary: true if pmid is the primary parent of mid.

    mperm: permissions
 */
static
int fsl_mlink_add_one( fsl_cx * f,
                       fsl_id_t pmid, fsl_uuid_cstr zFromUuid,
                       fsl_id_t mid, fsl_uuid_cstr zToUuid,
                       char const * zFilename,
                       char const * zPrior,
                       bool isPublic,
                       bool isPrimary,
                       fsl_fileperm_e mperm){
  fsl_id_t fnid, pfnid, pid, fid;
  fsl_db * db = fsl_cx_db_repo(f);
  fsl_stmt * s1 = NULL;
  int rc;
  bool doInsert = false;
  assert(f);
  assert(db);
  assert(db->beginCount>0);
  //MARKER(("%s() pmid=%d mid=%d\n", __func__, (int)pmid, (int)mid));

  rc = fsl__repo_filename_fnid2(f, zFilename, &fnid, 1);
  if(rc) return rc;
  if( zPrior && *zPrior ){
    rc = fsl__repo_filename_fnid2(f, zPrior, &pfnid, 1);
    if(rc) return rc;
  }else{
    pfnid = 0;
  }
  if( zFromUuid && *zFromUuid ){
    pid = fsl__uuid_to_rid2(f, zFromUuid, FSL_PHANTOM_PUBLIC);
    if(pid<0){
      assert(f->error.code);
      return f->error.code;
    }
    assert(pid>0);
  }else{
    pid = 0;
  }

  if( zToUuid && *zToUuid ){
    fid = fsl__uuid_to_rid2(f, zToUuid, FSL_PHANTOM_PUBLIC);
    if(fid<0){
      assert(f->error.code);
      return f->error.code;
    }else if( isPublic ){
      rc = fsl_content_make_public(f, fid);
      if(rc) return rc;
    }
  }else{
    fid = 0;
  }

  if(isPrimary){
    doInsert = true;
  }else{
    fsl_stmt * sInsCheck = 0;
    rc = fsl_db_prepare_cached(db, &sInsCheck,
                               "SELECT 1 FROM mlink WHERE "
                               "mid=? AND fnid=? AND NOT isaux"
                               "/*%s()*/",__func__);
    if(rc){
      rc = fsl_cx_uplift_db_error(f, db);
      goto end;
    }
    fsl_stmt_bind_id(sInsCheck, 1, mid);
    fsl_stmt_bind_id(sInsCheck, 2, fnid);
    rc = fsl_stmt_step(sInsCheck);
    fsl_stmt_cached_yield(sInsCheck);
    doInsert = (FSL_RC_STEP_ROW==rc) ? true : false;
    rc = 0;
  }
  if(doInsert){
    rc = fsl_db_prepare_cached(db, &s1,
                               "INSERT INTO mlink("
                               "mid,fid,pmid,pid,"
                               "fnid,pfnid,mperm,isaux"
                               ")VALUES("
                               ":m,:f,:pm,:p,:n,:pfn,:mp,:isaux"
                               ")"
                               "/*%s()*/",__func__);
    if(!rc){
      fsl_stmt_bind_id_name(s1, ":m", mid);
      fsl_stmt_bind_id_name(s1, ":f", fid);
      fsl_stmt_bind_id_name(s1, ":pm", pmid);
      fsl_stmt_bind_id_name(s1, ":p", pid);
      fsl_stmt_bind_id_name(s1, ":n", fnid);
      fsl_stmt_bind_id_name(s1, ":pfn", pfnid);
      fsl_stmt_bind_id_name(s1, ":mp", mperm);    
      fsl_stmt_bind_int32_name(s1, ":isaux", isPrimary ? 0 : 1);    
      rc = fsl_stmt_step(s1);
      fsl_stmt_cached_yield(s1);
      if(FSL_RC_STEP_DONE==rc){
        rc = 0;
      }else{
        fsl_cx_uplift_db_error(f, db);
      }
    }
  }
  if(!rc && pid>0 && fid){
    /* Reminder to self: this costs almost 1ms per checkin in very
       basic tests with 2003 checkins on my NUC unit. */
    rc = fsl__content_deltify(f, pid, fid, 0);
  }
  end:
  return rc;  
}

/**
    Do a binary search to find a file in d->F.list.  
   
    As an optimization, guess that the file we seek is at index
    d->F.cursor.  That will usually be the case.  If it is not found
    there, then do the actual binary search.

    Update d->F.cursor to be the index of the file that is found.

    If d->f is NULL then this perform a case-sensitive search,
    otherwise it uses case-sensitive or case-insensitive,
    depending on f->cache.caseInsensitive.    

    If the 3rd argument is not NULL and non-NULL is returned then
    *atNdx gets set to the d->F.list index of the resulting object.
    If NULL is returned, *atNdx is not modified.

    Reminder to self: if this requires a non-const deck (and it does
    right now) then the whole downstream chain will require a
    non-const instance or they'll have to make local copies to make
    the manipulation of d->F.cursor legal (but that would break
    following of baselines without yet more trickery).

    Reminder to self:

    Fossil(1) added another parameter to this since it was ported,
    indicating whether only an exact match or the "closest match" is
    acceptable, but currently (2021-03-10) only the fusefs module uses
    the closest-match option. It's a trivial code change but currently
    looks like YAGNI.
*/
static fsl_card_F * fsl__deck_F_seek_base(fsl_deck * d,
                                         char const * zName,
                                         uint32_t * atNdx ){
  /* Maintenance reminder: this algo relies on the various
     counters being signed. */
  fsl_int_t lwr, upr;
  int c;
  fsl_int_t i;
  assert(d);
  assert(zName && *zName);
  if(!d->F.used) return NULL;
  else if(FSL_CARD_F_LIST_NEEDS_SORT & d->F.flags){
    fsl_card_F_list_sort(&d->F);
  }
  bool const caseSensitive = fsl_cx_is_case_sensitive(d->f, false);
#define FCARD(NDX) F_at(&d->F, (NDX))
  lwr = 0;
  upr = d->F.used-1;
  if( d->F.cursor>=lwr && d->F.cursor<upr ){
    c = (d->f && caseSensitive)
      ? fsl_strcmp(FCARD(d->F.cursor+1)->name, zName)
      : fsl_stricmp(FCARD(d->F.cursor+1)->name, zName);
    if( c==0 ){
      if(atNdx) *atNdx = (uint32_t)d->F.cursor+1;
      return FCARD(++d->F.cursor);
    }else if( c>0 ){
      upr = d->F.cursor;
    }else{
      lwr = d->F.cursor+1;
    }
  }
  while( lwr<=upr ){
    i = (lwr+upr)/2;
    c = (d->f && caseSensitive)
      ? fsl_strcmp(FCARD(i)->name, zName)
      : fsl_stricmp(FCARD(i)->name, zName);
    if( c<0 ){
      lwr = i+1;
    }else if( c>0 ){
      upr = i-1;
    }else{
      d->F.cursor = i;
      if(atNdx) *atNdx = (uint32_t)i;
      return FCARD(i);
    }
  }
  return NULL;
#undef FCARD
}

fsl_card_F * fsl__deck_F_seek(fsl_deck * const d, const char *zName){
  fsl_card_F *pFile;
  assert(d);
  assert(zName && *zName);
  if(!d || (FSL_SATYPE_CHECKIN!=d->type) || !zName || !*zName
     || !d->F.used) return NULL;
  pFile = fsl__deck_F_seek_base(d, zName, NULL);
  if( !pFile &&
      (d->B.baseline /* we have a baseline or... */
       || (d->f && d->B.uuid) /* we can load the baseline */
       )){
        /* Check baseline manifest...

           Sidebar: while the delta manifest model outwardly appears
           to support recursive delta manifests, fossil(1) does not
           use them and there would seem to be little practical use
           for them (no notable size benefit for the majority of
           cases), so we're not recursing here.
         */
    int const rc = d->B.baseline ? 0 : fsl_deck_baseline_fetch(d);
    if(rc){
      assert(d->f->error.code);
    }else if( d->B.baseline ){
      assert(d->B.baseline->f && "How can this happen?");
      assert((d->B.baseline->f == d->f) &&
             "Universal laws are out of balance.");
      pFile = fsl__deck_F_seek_base(d->B.baseline, zName, NULL);
      if(pFile){
        assert(pFile->uuid &&
               "Per fossil-dev thread with DRH on 20140422, "
               "baselines never have removed files.");
      }
    }
  }
  return pFile;
}

fsl_card_F const * fsl_deck_F_search(fsl_deck *d, const char *zName){
  assert(d);
  return fsl__deck_F_seek(d, zName);
}

int fsl_deck_F_set( fsl_deck * d, char const * zName,
                    char const * uuid,
                    fsl_fileperm_e perms, 
                    char const * priorName){
  uint32_t fcNdx = 0;
  fsl_card_F * fc = 0;
  if(d->rid>0){
    return fsl_cx_err_set(d->f, FSL_RC_MISUSE,
                          "%s() cannot be applied to a saved deck.",
                          __func__);
  }else if(!fsl_deck_check_type(d, 'F')){
    return d->f->error.code;
  }
  fc = fsl__deck_F_seek_base(d, zName, &fcNdx);
  if(!uuid){
    if(fc){
      fsl_card_F_list_remove(&d->F, fcNdx);
      return 0;
    }else{
      return FSL_RC_NOT_FOUND;
    }
  }else if(!fsl_is_uuid(uuid)){
    return fsl_cx_err_set(d->f, FSL_RC_RANGE,
                          "Invalid UUID for F-card.");
  }
  if(fc){
    /* Got a match. Replace its contents. */
    char * n = 0;
    if(!fc->deckOwnsStrings){
      /* We can keep fc->name but need a tiny bit of hoop-jumping
         to do so. */
      n = fc->name;
      fc->name = 0;
    }
    fsl_card_F_clean(fc);
    assert(!fc->deckOwnsStrings);
    if(!(fc->name = n ? n : fsl_strdup(zName))) return FSL_RC_OOM;
    if(!(fc->uuid = fsl_strdup(uuid))) return FSL_RC_OOM;
    if(priorName && *priorName){
      if(!fsl_is_simple_pathname(priorName, 1)){
        return fsl_cx_err_set(d->f, FSL_RC_RANGE,
                              "Invalid priorName for F-card "
                              "(simple form required): %s", priorName);
      }else if(!(fc->priorName = fsl_strdup(priorName))){
        return FSL_RC_OOM;
      }
    }
    fc->perm = perms;
    return 0;
  }else{
    return fsl_deck_F_add(d, zName, uuid, perms, priorName);
  }
}

int fsl_deck_F_set_content( fsl_deck * const d, char const * zName,
                            fsl_buffer const * const src,
                            fsl_fileperm_e perm, 
                            char const * priorName){
  fsl_uuid_str zHash = 0;
  fsl_id_t rid = 0;
  fsl_id_t prevRid = 0;
  int rc = 0;
  assert(d->f);
  if(d->rid>0){
    return fsl_cx_err_set(d->f, FSL_RC_MISUSE,
                          "%s() cannot be applied to a saved deck.",
                          __func__);
  }else if(!fsl_cx_transaction_level(d->f)){
    return fsl_cx_err_set(d->f, FSL_RC_MISUSE,
                          "%s() requires that a transaction is active.",
                          __func__);
  }else if(!fsl_is_simple_pathname(zName, true)){
    return fsl_cx_err_set(d->f, FSL_RC_RANGE,
                          "Filename is not valid for use as a repository "
                          "entry: %s", zName);
  }
  rc = fsl_repo_blob_lookup(d->f, src, &rid, &zHash);
  if(rc && FSL_RC_NOT_FOUND!=rc) goto end;
  assert(zHash);
  if(!rid){
    fsl_card_F const * fc;
    /* This is new content. Save it, then see if we have a previous version
       to delta against this one. */
    rc = fsl__content_put_ex(d->f, src, zHash, 0, 0, false, &rid);
    if(rc) goto end;
    fc = fsl__deck_F_seek(d, zName);
    if(fc){
      prevRid = fsl_uuid_to_rid(d->f, fc->uuid);
      if(prevRid<0) goto end;
      else if(!prevRid){
        assert(!"cannot happen");
        rc = fsl_cx_err_set(d->f, FSL_RC_NOT_FOUND,
                            "Cannot find RID of file content %s [%s]\n",
                            fc->name, fc->uuid);
        goto end;
      }
      rc = fsl__content_deltify(d->f, prevRid, rid, false);
      if(rc) goto end;
    }
  }
  rc = fsl_deck_F_set(d, zName, zHash, perm, priorName);
  end:
  fsl_free(zHash);
  return rc;
}

void fsl__deck_clean_cards(fsl_deck * const d, char const * letters){
  char const * c = letters
    ? letters
    : "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
  for( ; *c; ++c ){
    switch(*c){
      case 'A': fsl_deck_clean_A(d); break;
      case 'B': fsl_deck_clean_B(d); break;
      case 'C': fsl_deck_clean_C(d); break;
      case 'D': d->D = 0.0; break;
      case 'E': fsl_deck_clean_E(d); break;
      case 'F': fsl_deck_clean_F(d); break;
      case 'G': fsl_deck_clean_G(d); break;
      case 'H': fsl_deck_clean_H(d); break;
      case 'I': fsl_deck_clean_I(d); break;
      case 'J': fsl_deck_clean_J(d,true); break;
      case 'K': fsl_deck_clean_K(d); break;
      case 'L': fsl_deck_clean_L(d); break;
      case 'M': fsl_deck_clean_M(d); break;
      case 'N': fsl_deck_clean_N(d); break;
      case 'P': fsl_deck_clean_P(d); break;
      case 'Q': fsl_deck_clean_Q(d); break;
      case 'R': fsl_deck_clean_R(d); break;
      case 'T': fsl_deck_clean_T(d); break;
      case 'U': fsl_deck_clean_U(d); break;
      case 'W': fsl_deck_clean_W(d); break;
      default: break;
    }
  }
}

int fsl_deck_derive(fsl_deck * const d){
  int rc = 0;
  if(d->rid<=0) return FSL_RC_MISUSE;
  assert(d->f);
  if(FSL_SATYPE_CHECKIN!=d->type) return FSL_RC_TYPE;
  fsl_deck_clean_P(d);
  {
    fsl_uuid_str pUuid = fsl_rid_to_uuid(d->f, d->rid);
    if(pUuid){
      rc = fsl_list_append(&d->P, pUuid);
      if(rc){
        assert(NULL==d->P.list);
        fsl_free(pUuid);
      }
    }else{
      assert(d->f->error.code);
      rc = d->f->error.code;
    }
    if(rc) return rc;
  }
  d->rid = 0;
  fsl__deck_clean_cards(d, "ACDEGHIJKLMNQRTUW");
  while(d->B.uuid){
    /* This is a delta manifest. Convert this deck into a baseline by
       build a new, complete F-card list. */
    fsl_card_F const * fc;
    fsl_card_F_list flist = fsl_card_F_list_empty;
    uint32_t fCount = 0;
    rc = fsl_deck_F_rewind(d);
    if(rc) return rc;
    while( 0==(rc=fsl_deck_F_next(d, &fc)) && fc ){
      ++fCount;
    }
    rc = fsl_deck_F_rewind(d);
    assert(0==rc
           && "fsl_deck_F_rewind() cannot fail after initial call.");
    assert(0==d->F.cursor);
    assert(0==d->B.baseline->F.cursor);
    rc = fsl_card_F_list_reserve(&flist, fCount);
    if(rc) break;
    while( 1 ){
      rc = fsl_deck_F_next(d, &fc);
      if(rc || !fc) break;
      fsl_card_F * const fNew = fsl_card_F_list_push(&flist);
      assert(fc->uuid);
      assert(fc->name);
      /* We must copy these strings because their ownership is
         otherwise unmanageable. e.g. they might live in d->content
         or d->B.baseline->content. */
      if(!(fNew->name = fsl_strdup(fc->name))
         || !(fNew->uuid = fsl_strdup(fc->uuid))){
        /* Reminder: we do not want/need to copy fc->priorName. Those
           renames were already applied in the parent checkin. */
        rc = FSL_RC_OOM;
        break;
      }
      fNew->perm = fc->perm;
    }
    fsl_deck_clean_B(d);
    fsl_deck_clean_F(d);
    if(rc) fsl_card_F_list_finalize(&flist);
    else d->F = flist/*transfer ownership*/;
    break;
  }
  return rc;
}

/**
    Returns true if repo contains an mlink entry where mid=rid, else
    false.
*/
static bool fsl_repo_has_mlink_mid( fsl_db * repo, fsl_id_t rid ){
#if 0
  return fsl_db_exists(repo,
                       "SELECT 1 FROM mlink WHERE mid=%"FSL_ID_T_PFMT,
                       rid);
#else
  fsl_stmt * st = NULL;
  bool gotone = false;
  int rc = fsl_db_prepare_cached(repo, &st,
                                 "SELECT 1 FROM mlink WHERE mid=?"
                                 "/*%s()*/",__func__);
  
  if(!rc){
    fsl_stmt_bind_id(st, 1, rid);
    rc = fsl_stmt_step(st);
    fsl_stmt_cached_yield(st);
    gotone = rc==FSL_RC_STEP_ROW;
  }
  return gotone;
#endif
}

static bool fsl_repo_has_mlink_pmid_mid( fsl_db * repo, fsl_id_t pmid, fsl_id_t mid ){
  fsl_stmt * st = NULL;
  int rc = fsl_db_prepare_cached(repo, &st,
                                 "SELECT 1 FROM mlink WHERE mid=? "
                                 "AND pmid=?"
                                 "/*%s()*/",__func__);
  if(!rc){
    fsl_stmt_bind_id(st, 1, mid);
    fsl_stmt_bind_id(st, 2, pmid);
    rc = fsl_stmt_step(st);
    fsl_stmt_cached_yield(st);
    if( rc==FSL_RC_STEP_ROW ) rc = 0;
  }
  /* MARKER(("fsl_repo_has_mlink_mid(%d) rc=%d\n", (int)rid, rc)); */
  return rc ? false : true;
}

/**
    Add mlink table entries associated with manifest cid, pChild.  The
    parent manifest is pid, pParent.  One of either pChild or pParent
    will be NULL and it will be computed based on cid/pid.
   
    A single mlink entry is added for every file that changed content,
    name, and/or permissions going from pid to cid.
   
    Deleted files have mlink.fid=0.
   
    Added files have mlink.pid=0.
   
    File added by merge have mlink.pid=-1.

    Edited files have both mlink.pid!=0 and mlink.fid!=0

    Comments from the original implementation:

    Many mlink entries for merge parents will only be added if another
    mlink entry already exists for the same file from the primary
    parent.  Therefore, to ensure that all merge-parent mlink entries
    are properly created:

    (1) Make this routine a no-op if pParent is a merge parent and the
        primary parent is a phantom.

    (2) Invoke this routine recursively for merge-parents if pParent
        is the primary parent.
*/
static int fsl_mlink_add( fsl_cx * const f,
                          fsl_id_t pmid, fsl_deck /*const*/ * pParent,
                          fsl_id_t cid, fsl_deck /*const*/ * pChild,
                          bool isPrimary){
  fsl_buffer otherContent = fsl_buffer_empty;
  fsl_id_t otherRid;
  fsl_size_t i = 0;
  int rc = 0;
  fsl_card_F const * pChildFile = NULL;
  fsl_card_F const * pParentFile = NULL;
  fsl_deck dOther = fsl_deck_empty;
  fsl_db * const db = fsl_cx_db_repo(f);
  bool isPublic;
  assert(db);
  assert(db->beginCount>0);
  /* If mlink table entires are already set for pmid/cid, then abort
     early doing no work.
  */
  //MARKER(("%s() pmid=%d cid=%d\n", __func__, (int)pmid, (int)cid));
  if(fsl_repo_has_mlink_pmid_mid(db, pmid, cid)) return 0;
  /* Compute the value of the missing pParent or pChild parameter.
     Fetch the baseline checkins for both.
  */
  assert( pParent==0 || pChild==0 );
  if( pParent ){
    assert(!pChild);
    pChild = &dOther;
    otherRid = cid;
  }else{
    pParent = &dOther;
    otherRid = pmid;
  }

  if(otherRid && !fsl__cx_mcache_search(f, otherRid, &dOther)){
    rc = fsl_content_get(f, otherRid, &otherContent);
    if(rc){
      /* fossil(1) simply ignores errors here and returns. We'll ignore
         the phantom case because (1) erroring out here would be bad and
         (2) fossil does so. The exact implications of doing so are
         unclear, though. */
      if(FSL_RC_PHANTOM==rc){
        rc = 0;
      }else if(!f->error.msg.used && FSL_RC_OOM!=rc){
        rc = fsl_cx_err_set(f, rc,
                            "Fetching content of rid %"FSL_ID_T_PFMT" failed: %s",
                            otherRid, fsl_rc_cstr(rc));
      }
      goto end;
    }
    if( !otherContent.used ){
      /* ??? fossil(1) ignores this case and returns. */
      fsl_buffer_clear(&otherContent)/*for empty file case*/;
      rc = 0;
      goto end;
    }
    dOther.f = f;
    rc = fsl_deck_parse2(&dOther, &otherContent, otherRid);
    assert(dOther.f);
    if(rc) goto end;
  }
  if( (pParent->f && (rc=fsl_deck_baseline_fetch(pParent)))
      || (pChild->f && (rc=fsl_deck_baseline_fetch(pChild)))){
    goto end;
  }
  isPublic = !fsl_content_is_private(f, cid);

  /* If pParent is not the primary parent of pChild, and the primary
  ** parent of pChild is a phantom, then abort this routine without
  ** doing any work.  The mlink entries will be computed when the
  ** primary parent dephantomizes.
  */
  if( !isPrimary && otherRid==cid ){
    assert(pChild->P.used);
    if(!fsl_db_exists(db,"SELECT 1 FROM blob WHERE uuid=%Q AND size>0",
                      (char const *)pChild->P.list[0])){
      rc = 0;
      fsl__cx_mcache_insert(f, &dOther);
      goto end;
    }
  }

  if(pmid>0){
    /* Try to make the parent manifest a delta from the child, if that
       is an appropriate thing to do.  For a new baseline, make the 
       previous baseline a delta from the current baseline.
    */
    if( (pParent->B.uuid==0)==(pChild->B.uuid==0) ){
      rc = fsl__content_deltify(f, pmid, cid, 0);
    }else if( pChild->B.uuid==NULL && pParent->B.uuid!=NULL ){
      rc = fsl__content_deltify(f, pParent->B.baseline->rid, cid, 0);
    }
    if(rc) goto end;
  }

  /* Remember all children less than a few seconds younger than their parent,
     as we might want to fudge the times for those children.
  */
  if( f->cache.isCrosslinking &&
      (pChild->D < pParent->D+AGE_FUDGE_WINDOW)
  ){
    rc = fsl_db_exec(db, "INSERT OR REPLACE INTO time_fudge VALUES"
                     "(%"FSL_ID_T_PFMT", %"FSL_JULIAN_T_PFMT
                     ", %"FSL_ID_T_PFMT", %"FSL_JULIAN_T_PFMT");",
                     pParent->rid, pParent->D,
                     pChild->rid, pChild->D);
    if(rc) goto end;
  }

  /* First look at all files in pChild, ignoring its baseline.  This
     is where most of the changes will be found.
  */  
#define FCARD(DECK,NDX) \
  ((((NDX)<(DECK)->F.used)) \
   ? F_at(&(DECK)->F,NDX)  \
   : NULL)
  for(i=0, pChildFile=FCARD(pChild,0);
      i<pChild->F.used;
      ++i, pChildFile=FCARD(pChild,i)){
    fsl_fileperm_e const mperm = pChildFile->perm;
    if( pChildFile->priorName ){
      pParentFile = pmid
        ? fsl__deck_F_seek(pParent, pChildFile->priorName)
        : 0;
      if( pParentFile ){
        /* File with name change */
        /*
          libfossil checkin 8625a31eff708dea93b16582e4ec5d583794d1af
          contains these two interesting F-cards:

F src/net/wanderinghorse/libfossil/FossilCheckout.java
F src/org/fossil_scm/libfossil/Checkout.java 6e58a47089d3f4911c9386c25bac36c8e98d4d21 w src/net/wanderinghorse/libfossil/FossilCheckout.java

          Note the placement of FossilCheckout.java (twice).

          Up until then, i thought a delete/rename combination was not possible.
        */
        rc = fsl_mlink_add_one(f, pmid, pParentFile->uuid,
                               cid, pChildFile->uuid, pChildFile->name,
                               pChildFile->priorName, isPublic,
                               isPrimary, mperm);
      }else{
         /* File name changed, but the old name is not found in the parent!
            Treat this like a new file. */
        rc = fsl_mlink_add_one(f, pmid, 0, cid, pChildFile->uuid,
                               pChildFile->name, 0,
                               isPublic, isPrimary, mperm);
      }
    }else if(pmid){
      pParentFile = fsl__deck_F_seek(pParent, pChildFile->name);
      if(!pParentFile || !pParentFile->uuid){
        /* Parent does not have it or it was removed in parent. */
        if( pChildFile->uuid ){
          /* A new or re-added file */
          rc = fsl_mlink_add_one(f, pmid, 0, cid, pChildFile->uuid,
                                 pChildFile->name, 0,
                                 isPublic, isPrimary, mperm);
        }
      }
      else if( fsl_strcmp(pChildFile->uuid, pParentFile->uuid)!=0
                || (pParentFile->perm!=mperm) ){
         /* Changes in file content or permissions */
        rc = fsl_mlink_add_one(f, pmid, pParentFile->uuid,
                               cid, pChildFile->uuid,
                               pChildFile->name, 0,
                               isPublic, isPrimary, mperm);
      }
    }
  } /* end pChild card list loop */
  if(rc) goto end;
  else if( pParent->B.uuid && pChild->B.uuid ){
    /* Both parent and child are delta manifests.  Look for files that
       are deleted or modified in the parent but which reappear or revert
       to baseline in the child and show such files as being added or changed
       in the child. */
    for(i=0, pParentFile=FCARD(pParent,0);
        i<pParent->F.used;
        ++i, pParentFile = FCARD(pParent,i)){
      if( pParentFile->uuid ){
        pChildFile = fsl__deck_F_seek_base(pChild, pParentFile->name, NULL);
        if( !pChildFile || !pChildFile->uuid){
          /* The child file reverts to baseline or is deleted.
             Show this as a change. */
          if(!pChildFile){
            pChildFile = fsl__deck_F_seek(pChild, pParentFile->name);
          }
          if( pChildFile && pChildFile->uuid ){
            rc = fsl_mlink_add_one(f, pmid, pParentFile->uuid, cid,
                                   pChildFile->uuid, pChildFile->name,
                                   0, isPublic, isPrimary,
                                   pChildFile->perm);
          }
        }
      }else{
        /* Was deleted in the parent. */
        pChildFile = fsl__deck_F_seek(pChild, pParentFile->name);
        if( pChildFile && pChildFile->uuid ){
          /* File resurrected in the child after having been deleted in
             the parent.  Show this as an added file. */
          rc = fsl_mlink_add_one(f, pmid, 0, cid, pChildFile->uuid,
                                 pChildFile->name, 0, isPublic,
                                 isPrimary, pChildFile->perm);
        }
      }
      if(rc) goto end;
    }
    assert(0==rc);
  }else if( pmid && !pChild->B.uuid ){
    /* pChild is a baseline with a parent.  Look for files that are
       present in pParent but are missing from pChild and mark them as
       having been deleted. */
    fsl_card_F const * cfc = NULL;
    fsl_deck_F_rewind(pParent);
    while( (0==(rc=fsl_deck_F_next(pParent,&cfc))) && cfc){
      pParentFile = cfc;
      pChildFile = fsl__deck_F_seek(pChild, pParentFile->name);
      if( (!pChildFile || !pChildFile->uuid) && pParentFile->uuid ){
        rc = fsl_mlink_add_one(f, pmid, pParentFile->uuid, cid, 0,
                               pParentFile->name, 0, isPublic,
                               isPrimary, pParentFile->perm);
      }
    }
    if(rc) goto end;
  }

  fsl__cx_mcache_insert(f, &dOther);
  
  /* If pParent is the primary parent of pChild, also run this analysis
  ** for all merge parents of pChild */
  if( pmid && isPrimary ){
    for(i=1; i<pChild->P.used; i++){
      pmid = fsl_uuid_to_rid(f, (char const*)pChild->P.list[i]);
      if( pmid<=0 ) continue;
      rc = fsl_mlink_add(f, pmid, 0, cid, pChild, false);
      if(rc) goto end;
    }
    for(i=0; i<pChild->Q.used; i++){
      fsl_card_Q const * q = (fsl_card_Q const *)pChild->Q.list[i];
      if( q->type>0 && (pmid = fsl_uuid_to_rid(f, q->target))>0 ){
        rc = fsl_mlink_add(f, pmid, 0, cid, pChild, false);
        if(rc) goto end;
      }
    }
  }

  end:
  fsl_deck_finalize(&dOther);
  fsl_buffer_clear(&otherContent);
  if(rc && !f->error.code && db->error.code){
    rc = fsl_cx_uplift_db_error(f, db);
  }
  return rc;
#undef FCARD
}

/**
   Apply all tags defined in deck d. If parentId is >0 then any
   propagating tags from that parent are well and duly propagated.
   Returns 0 on success. Potential TODO: if parentId<=0 and
   d->P.used>0 then use d->P.list[0] in place of parentId.
*/
static int fsl__deck_crosslink_apply_tags(fsl_cx * f, fsl_deck *d,
                                         fsl_db * db, fsl_id_t rid,
                                         fsl_id_t parentId){
  int rc = 0;
  fsl_size_t i;
  fsl_list const * li = &d->T;
  double tagTime = d->D;
  if(li->used && tagTime<=0){
    tagTime = fsl_db_julian_now(db);
    if(tagTime<=0){
      rc = FSL_RC_DB;
      goto end;
    }
  }      
  for( i = 0; !rc && (i < li->used); ++i){
    fsl_id_t tid;
    fsl_card_T const * tag = (fsl_card_T const *)li->list[i];
    assert(tag);
    if(!tag->uuid){
      tid = rid;
    }else{
      tid = fsl_uuid_to_rid( f, tag->uuid);
    }
    if(tid<0){
      assert(f->error.code);
      rc = f->error.code;
      break;
    }else if(0==tid){
      rc = fsl_cx_err_set(f, FSL_RC_RANGE,
                          "Could not get RID for [%.12s].",
                          tag->uuid);
      break;
    }
    rc = fsl__tag_insert(f, tag->type,
                        tag->name, tag->value,
                        rid, tagTime, tid, NULL);
  }
  if( !rc && (parentId>0) ){
    rc = fsl__tag_propagate_all(f, parentId);
  }
  end:
  return rc;
}

/**
   Part of the checkin crosslink machinery: create all appropriate
   plink and mlink table entries for d->P.

   If parentId is not NULL, *parentId gets assigned to the rid of the
   first parent, or 0 if d->P is empty.
*/
static int fsl_deck_add_checkin_linkages(fsl_deck *d, fsl_id_t * parentId){
  int rc = 0;
  fsl_size_t nLink = 0;
  char zBaseId[30] = {0}/*RID of baseline or "NULL" if no baseline */;
  fsl_size_t i;
  fsl_stmt q = fsl_stmt_empty;
  fsl_id_t _parentId = 0;
  fsl_cx * const f = d->f;
  fsl_db * const db = fsl_cx_db_repo(f);
  assert(f && db);
  if(!parentId) parentId = &_parentId;
  if(d->B.uuid){
    fsl_id_t const baseid = d->B.baseline
      ? d->B.baseline->rid
      : fsl_uuid_to_rid(d->f, d->B.uuid);
    if(baseid<0){
      rc = d->f->error.code;
      assert(0 != rc);
      goto end;
    }
    assert(baseid>0);
    fsl_snprintf( zBaseId, sizeof(zBaseId),
                  "%"FSL_ID_T_PFMT,
                  baseid );
        
  }else{
    fsl_snprintf( zBaseId, sizeof(zBaseId), "NULL" );
  }
  *parentId = 0;
  for(i=0; i<d->P.used; ++i){
    char const * parentUuid = (char const *)d->P.list[i];
    fsl_id_t const pid = fsl__uuid_to_rid2(f, parentUuid, FSL_PHANTOM_PUBLIC);
    if(pid<0){
      assert(f->error.code);
      rc = f->error.code;
      goto end;
    }
    rc = fsl_db_exec(db, "INSERT OR IGNORE "
                     "INTO plink(pid, cid, isprim, mtime, baseid) "
                     "VALUES(%"FSL_ID_T_PFMT", %"FSL_ID_T_PFMT
                     ", %d, %"FSL_JULIAN_T_PFMT", %s)",
                     pid, d->rid,
                     ((i==0) ? 1 : 0), d->D, zBaseId);
    if(rc) goto end;
    if(0==i) *parentId = pid;
  }
  rc = fsl_mlink_add(f, *parentId, NULL, d->rid, d, true);
  if(rc) goto end;
  nLink = d->P.used;
  for(i=0; i<d->Q.used; ++i){
    fsl_card_Q const * q = (fsl_card_Q const *)d->Q.list[i];
    if(q->type>0) ++nLink;
  }
  if(nLink>1){
    /* https://www.fossil-scm.org/index.html/info/8e44cf6f4df4f9f0 */
    /* Change MLINK.PID from 0 to -1 for files that are added by merge. */
    rc = fsl_db_exec(db,
                     "UPDATE mlink SET pid=-1"
                     " WHERE mid=%"FSL_ID_T_PFMT
                     "   AND pid=0"
                     "   AND fnid IN "
                     "  (SELECT fnid FROM mlink WHERE mid=%"FSL_ID_T_PFMT
                     " GROUP BY fnid"
                     "    HAVING count(*)<%d)",
                     d->rid, d->rid, (int)nLink
                     );
    if(rc) goto end;
  }
  rc = fsl_db_prepare(db, &q,
                      "SELECT cid, isprim FROM plink "
                      "WHERE pid=%"FSL_ID_T_PFMT,
                      d->rid);
  while( !rc && (FSL_RC_STEP_ROW==(rc=fsl_stmt_step(&q))) ){
    fsl_id_t const cid = fsl_stmt_g_id(&q, 0);
    int const isPrim = fsl_stmt_g_int32(&q, 1);
    /* This block is only hit a couple of times during a fresh rebuild (empty mlink/plink
       tables), but many times on a rebuilds if those tables are not emptied in advance? */
    assert(cid>0);
    rc = fsl_mlink_add(f, d->rid, d, cid, NULL, isPrim ? true : false);
  }
  if(FSL_RC_STEP_DONE==rc) rc = 0;
  fsl_stmt_finalize(&q);
  if(rc) goto end;
  if( !d->P.used ){
    /* For root files (files without parents) add mlink entries
       showing all content as new.

       Historically, fossil has been unable to create such checkins
       because the initial checkin has no files.
    */
    int const isPublic = !fsl_content_is_private(f, d->rid);
    for(i=0; !rc && (i<d->F.used); ++i){
      fsl_card_F const * fc = F_at(&d->F, i);
      rc = fsl_mlink_add_one(f, 0, 0, d->rid, fc->uuid, fc->name, 0,
                             isPublic, 1, fc->perm);
    }
  }
  end:
  return rc;
}

/**
   Applies the value of a "parent" tag (reparent) to the given
   artifact id. zTagVal must be the value of a parent tag (a list of
   full UUIDs). This is only to be run as part of fsl__crosslink_end().

   Returns 0 on success.

   POTENTIAL fixme: perhaps return without side effects if rid is not
   found (like fossil(1) does). That said, this step is only run after
   crosslinking and would only result in a not-found if the tagxref
   table contents is out of date.

   POTENTIAL fixme: fail without error if the tag value is malformed,
   under the assumption that the tag was intended for some purpose
   other than reparenting.
*/
static int fsl_crosslink_reparent(fsl_cx * f, fsl_id_t rid, char const *zTagVal){
  int rc = 0;
  char * zDup = 0;
  char * zPos;
  fsl_size_t maxP, nP = 0;
  fsl_deck d = fsl_deck_empty;
  fsl_list fakeP = fsl_list_empty
    /* fake P-card for purposes of passing the reparented deck through
       fsl_deck_add_checkin_linkages() */;
  maxP = (fsl_strlen(zTagVal)+1) / (FSL_STRLEN_SHA1+1);
  if(!maxP) return FSL_RC_RANGE;
  rc = fsl_list_reserve(&fakeP, maxP);
  if(rc) return rc;
  zDup = fsl_strdup(zTagVal);
  if(!zDup){
    rc = FSL_RC_OOM;
    goto end;
  }
  /* Split zTagVal into list of parent IDs... */
  for( nP = 0, zPos = zDup; *zPos; ){
    char const * zBegin = zPos;
    for( ; *zPos && ' '!=*zPos; ++zPos){}
    if(' '==*zPos){
      *zPos = 0;
      ++zPos;
    }
    if(!fsl_is_uuid(zBegin)){
      rc = fsl_cx_err_set(f, FSL_RC_RANGE,
                          "Invalid value [%s] in reparent tag value "
                          "[%s] for rid %"FSL_ID_T_PFMT".",
                          zBegin, zTagVal, rid);
      goto end;
    }
    fakeP.list[nP++] = (void *)zBegin;
  }
  assert(!rc);
  fakeP.used = nP;
  rc = fsl_deck_load_rid(f, &d, rid, FSL_SATYPE_ANY);
  if(rc) goto end;
  switch(d.type){
    case FSL_SATYPE_CHECKIN:
    case FSL_SATYPE_TECHNOTE:
    case FSL_SATYPE_WIKI:
    case FSL_SATYPE_FORUMPOST:
      break;
    default:
      rc = fsl_cx_err_set(f, FSL_RC_TYPE, "Invalid deck type (%s) "
                          "for use with the 'parent' tag.",
                          fsl_satype_cstr(d.type));
      goto end;
  }
  assert(d.rid==rid);
  assert(d.f);
  fsl_db * const db = fsl_cx_db_repo(f);
  rc = fsl_db_exec_multi(db,
                         "DELETE FROM plink WHERE cid=%"FSL_ID_T_PFMT";"
                         "DELETE FROM mlink WHERE mid=%"FSL_ID_T_PFMT";",
                         rid, rid);
  if(rc) goto end;
  fsl_list const origP = d.P;
  d.P = fakeP;
  rc = fsl_deck_add_checkin_linkages(&d, NULL);
  d.P = origP;
  fsl_deck_finalize(&d);

  end:
  fsl_list_reserve(&fakeP, 0);
  fsl_free(zDup);
  return rc;
}

/**
   Inserts plink entries for FORUM, WIKI, and TECHNOTE manifests. May
   assert for other manifest types. If a parent entry exists, it also
   propagates any tags for that parent. This is a no-op if
   the deck has no parents.
*/
static int fsl__deck_crosslink_fwt_plink(fsl_deck * d){
  int i;
  fsl_id_t parentId = 0;
  fsl_db * db;
  int rc = 0;
  assert(d->type==FSL_SATYPE_WIKI ||
         d->type==FSL_SATYPE_FORUMPOST ||
         d->type==FSL_SATYPE_TECHNOTE);
  assert(d->f);
  assert(d->rid>0);
  if(!d->P.used) return rc;
  db = fsl_cx_db_repo(d->f);
  fsl__phantom_e const fantomMode = fsl_content_is_private(d->f, d->rid)
    ? FSL_PHANTOM_PRIVATE : FSL_PHANTOM_PUBLIC;
  for(i=0; 0==rc && i<(int)d->P.used; ++i){
    fsl_id_t const pid = fsl__uuid_to_rid2(d->f, (char const *)d->P.list[i],
                                          fantomMode);
    if(0==i) parentId = pid;
    rc = fsl_db_exec_multi(db,
                           "INSERT OR IGNORE INTO plink"
                           "(pid, cid, isprim, mtime, baseid)"
                           "VALUES(%"FSL_ID_T_PFMT", %"FSL_ID_T_PFMT", "
                           "%d, %"FSL_JULIAN_T_PFMT", NULL)",
                           pid, d->rid, i==0, d->D);
  }
  if(!rc && parentId){
    rc = fsl__tag_propagate_all(d->f, parentId);
  }
  return rc;
}

int fsl__call_xlink_listeners(fsl_deck * const d){
  int rc = 0;
  fsl_xlinker * xl = NULL;
  fsl_cx_err_reset(d->f);
  for( fsl_size_t i = 0; !rc && (i < d->f->xlinkers.used); ++i ){
    xl = d->f->xlinkers.list+i;
    rc = xl->f( d, xl->state );
  }
  if(rc && !d->f->error.code){
    assert(xl);
    rc = fsl_cx_err_set(d->f, rc, "Crosslink callback handler "
                        "'%s' failed with code %d (%s) for "
                        "artifact RID #%" FSL_ID_T_PFMT ".",
                        xl->name, rc, fsl_rc_cstr(rc),
                        d->rid);
  }
  return rc;
}

/**
   Overrideable crosslink listener which updates the timeline for
   attachment records.
*/
static int fsl_deck_xlink_f_attachment(fsl_deck * const d, void * state __unused){
  if(FSL_SATYPE_ATTACHMENT!=d->type) return 0;
  int rc;
  fsl_db * const db = fsl_cx_db_repo(d->f);
  fsl_buffer * const comment = fsl__cx_scratchpad(d->f);
  const bool isAdd = (d->A.src && *d->A.src) ? 1 : 0;
  char attachToType = 'w'
    /* Assume wiki until we know otherwise, keeping in mind that the
       d->A.tgt might not yet be in the blob table, in which case
       we are unable to know, for certain, what the target is.
       That only affects the timeline (event table), though, not
       the crosslinking of the attachment itself. */;
  assert(db);
  if(fsl_is_uuid(d->A.tgt)){
    if( fsl_db_exists(db, "SELECT 1 FROM tag WHERE tagname='tkt-%q'",
                      d->A.tgt)){
      attachToType = 't' /* attach to a known ticket */;
    }else if( fsl_db_exists(db, "SELECT 1 FROM tag WHERE tagname='event-%q'",
                            d->A.tgt)){
      attachToType = 'e' /* attach to a known technote (event) */;
    }
  }
  if('w'==attachToType){
    /* Attachment applies to a wiki page */
    if(isAdd){
      rc = fsl_buffer_appendf(comment,
                              "Add attachment \"%h\" "
                              "to wiki page [%h]",
                              d->A.name, d->A.tgt);
    }else{
      rc = fsl_buffer_appendf(comment,
                              "Delete attachment \"%h\" "
                              "from wiki page [%h]",
                              d->A.name, d->A.tgt);
    }
  }else if('e' == attachToType){/*technote*/
    if(isAdd){
      rc = fsl_buffer_appendf(comment,
                              "Add attachment [/artifact/%!S|%h] to "
                              "tech note [/technote/%!S|%S]",
                              d->A.src, d->A.name, d->A.tgt, d->A.tgt);
    }else{
      rc = fsl_buffer_appendf(comment,
                              "Delete attachment \"/artifact/%!S|%h\" "
                              "from tech note [/technote/%!S|%S]",
                              d->A.name, d->A.name, d->A.tgt,
                              d->A.tgt);
    }
  }else{
    /* Attachment applies to a ticket */
    if(isAdd){
      rc = fsl_buffer_appendf(comment,
                              "Add attachment [/artifact/%!S|%h] "
                              "to ticket [%!S|%S]",
                              d->A.src, d->A.name, d->A.tgt, d->A.tgt);
    }else{
      rc = fsl_buffer_appendf(comment,
                              "Delete attachment \"%h\" "
                              "from ticket [%!S|%S]",
                              d->A.name, d->A.tgt, d->A.tgt);
    }
  }
  if(!rc){
    rc = fsl_db_exec(db,
                     "REPLACE INTO event(type,mtime,objid,user,comment)"
                     "VALUES("
                     "'%c',%"FSL_JULIAN_T_PFMT",%"FSL_ID_T_PFMT","
                     "%Q,%B)",
                     attachToType, d->D, d->rid, d->U, comment);
  }
  fsl__cx_scratchpad_yield(d->f, comment);
  return rc;
}

/**
   Overrideable crosslink listener which updates the timeline for
   checkin records.
*/
static int fsl_deck_xlink_f_checkin(fsl_deck * const d, void * state __unused){
  if(FSL_SATYPE_CHECKIN!=d->type) return 0;
  int rc;
  fsl_db * db;
  db = fsl_cx_db_repo(d->f);
  assert(db);
  rc = fsl_db_exec(db,
       "REPLACE INTO event(type,mtime,objid,user,comment,"
       "bgcolor,euser,ecomment,omtime)"
       "VALUES('ci',"
       "  coalesce(" /*mtime*/
       "    (SELECT julianday(value) FROM tagxref "
       "      WHERE tagid=%d AND rid=%"FSL_ID_T_PFMT
       "    ),"
       "    %"FSL_JULIAN_T_PFMT""
       "  ),"
       "  %"FSL_ID_T_PFMT","/*objid*/
       "  %Q," /*user*/
#if 1
       "  %Q," /*comment. No, the comment _field_. */
#else
       /* just for testing... */
       "  'xlink: %q'," /*comment. No, the comment _field_. */
#endif
       "  (SELECT value FROM tagxref " /*bgcolor*/
       "    WHERE tagid=%d AND rid=%"FSL_ID_T_PFMT
       "    AND tagtype>0"
       "  ),"
       "  (SELECT value FROM tagxref " /*euser*/
       "    WHERE tagid=%d AND rid=%"FSL_ID_T_PFMT
       "  ),"
       "  (SELECT value FROM tagxref " /*ecomment*/
       "    WHERE tagid=%d AND rid=%"FSL_ID_T_PFMT
       "  ),"
       "  %"FSL_JULIAN_T_PFMT/*omtime*/
       /* RETURNING coalesce(ecomment,comment)
          see comments below about zCom */
       ")",
       /* The casts here are to please the va_list. */
       (int)FSL_TAGID_DATE, d->rid, d->D,
       d->rid, d->U, d->C,
       (int)FSL_TAGID_BGCOLOR, d->rid,
       (int)FSL_TAGID_USER, d->rid,
       (int)FSL_TAGID_COMMENT, d->rid, d->D
  );
  return fsl_cx_uplift_db_error2(d->f, db, rc);
}

static int fsl_deck_xlink_f_control(fsl_deck * const d, void * state __unused){
  if(FSL_SATYPE_CONTROL!=d->type) return 0;
  /*
    Create timeline event entry for all tags in this control
    construct. Note that we are using a lot of historical code which
    hard-codes english-lanuage text and links which only work in
    fossil(1). i would prefer to farm this out to a crosslink
    callback, and provide a default implementation which more or
    less mimics fossil(1).
  */
  int rc = 0;
  fsl_buffer * const comment = fsl__cx_scratchpad(d->f);
  fsl_size_t i;
  const char *zName;
  const char *zValue;
  const char *zUuid;
  int branchMove = 0;
  int const uuidLen = 8;
  fsl_card_T const * tag = NULL;
  fsl_card_T const * prevTag = NULL;
  fsl_list const * li = &d->T;
  fsl_db * const db = fsl_cx_db_repo(d->f);
  double mtime = (d->D>0)
    ? d->D
    : fsl_db_julian_now(db);
  assert(db);
  /**
     Reminder to self: fossil(1) has a comment here:

     // Next loop expects tags to be sorted on UUID, so sort it.
     qsort(p->aTag, p->nTag, sizeof(p->aTag[0]), tag_compare);

     That sort plays a role in hook code execution and is needed to
     avoid duplicate hook execution in some cases. libfossil
     outsources that type of thing to crosslink callbacks, though,
     so we won't concern ourselves with it here. We also don't
     really want to modify the deck during crosslinking. The only
     reason the deck is not const in this routine is because of the
     fsl_deck::F::cursor bits inherited from fossil(1), largely
     worth its cost except that many routines can no longer be
     const. Shame C doesn't have C++'s "mutable" keyword.

     That said, sorting by UUID would have a nice side-effect on the
     output of grouping tags by the UUID they tag. So far
     (201404) such groups of tags have not appeared in the wild
     because fossil(1) has no mechanism for creating them.
  */
  for( i = 0; !rc && (i < li->used); ++i, prevTag = tag){
    bool isProp = 0, isAdd = 0, isCancel = 0;
    tag = (fsl_card_T const *)li->list[i];
    zUuid = tag->uuid;
    if(!zUuid /*tag on self*/) continue;
    if( i==0 || 0!=fsl_uuidcmp(tag->uuid, prevTag->uuid)){
      rc = fsl_buffer_appendf(comment,
                              " Edit [%.*s]:", uuidLen, zUuid);
      branchMove = 0;
    }
    if(rc) goto end;
    isProp = FSL_TAGTYPE_PROPAGATING==tag->type;
    isAdd = FSL_TAGTYPE_ADD==tag->type;
    isCancel = FSL_TAGTYPE_CANCEL==tag->type;
    assert(isProp || isAdd || isCancel);
    zName = tag->name;
    zValue = tag->value;
    if( isProp && 0==fsl_strcmp(zName, "branch")){
      rc = fsl_buffer_appendf(comment,
                              " Move to branch %s"
                              "[/timeline?r=%h&nd&dp=%.*s | %h].",
                              zValue, zValue, uuidLen, zUuid, zValue);
      branchMove = 1;
    }else if( isProp && fsl_strcmp(zName, "bgcolor")==0 ){
      rc = fsl_buffer_appendf(comment,
                              " Change branch background color to \"%h\".", zValue);
    }else if( isAdd && fsl_strcmp(zName, "bgcolor")==0 ){
      rc = fsl_buffer_appendf(comment,
                              " Change background color to \"%h\".", zValue);
    }else if( isCancel && fsl_strcmp(zName, "bgcolor")==0 ){
      rc = fsl_buffer_appendf(comment, " Cancel background color.");
    }else if( isAdd && fsl_strcmp(zName, "comment")==0 ){
      rc = fsl_buffer_appendf(comment, " Edit check-in comment.");
    }else if( isAdd && fsl_strcmp(zName, "user")==0 ){
      rc = fsl_buffer_appendf(comment, " Change user to \"%h\".", zValue);
    }else if( isAdd && fsl_strcmp(zName, "date")==0 ){
      rc = fsl_buffer_appendf(comment, " Timestamp %h.", zValue);
    }else if( isCancel && memcmp(zName, "sym-",4)==0 ){
      if( !branchMove ){
        rc = fsl_buffer_appendf(comment, " Cancel tag %h.", zName+4);
      }
    }else if( isProp && memcmp(zName, "sym-",4)==0 ){
      if( !branchMove ){
        rc = fsl_buffer_appendf(comment, " Add propagating tag \"%h\".", zName+4);
      }
    }else if( isAdd && memcmp(zName, "sym-",4)==0 ){
      rc = fsl_buffer_appendf(comment, " Add tag \"%h\".", zName+4);
    }else if( isCancel && memcmp(zName, "sym-",4)==0 ){
      rc = fsl_buffer_appendf(comment, " Cancel tag \"%h\".", zName+4);
    }else if( isAdd && fsl_strcmp(zName, "closed")==0 ){
      rc = fsl_buffer_append(comment, " Marked \"Closed\"", -1);
      if( !rc && zValue && *zValue ){
        rc = fsl_buffer_appendf(comment, " with note \"%h\"", zValue);
      }
      if(!rc) rc = fsl_buffer_append(comment, ".", 1);
    }else if( isCancel && fsl_strcmp(zName, "closed")==0 ){
      rc = fsl_buffer_append(comment, " Removed the \"Closed\" mark", -1);
      if( !rc && zValue && *zValue ){
        rc = fsl_buffer_appendf(comment, " with note \"%h\"", zValue);
      }
      if(!rc) rc = fsl_buffer_append(comment, ".", 1);
    }else {
      if( isCancel ){
        rc = fsl_buffer_appendf(comment, " Cancel \"%h\"", zName);
      }else if( isAdd ){
        rc = fsl_buffer_appendf(comment, " Add \"%h\"", zName);
      }else{
        assert(isProp);
        rc = fsl_buffer_appendf(comment, " Add propagating \"%h\"", zName);
      }
      if(rc) goto end;
      if( zValue && zValue[0] ){
        rc = fsl_buffer_appendf(comment, " with value \"%h\".", zValue);
      }else{
        rc = fsl_buffer_append(comment, ".", 1);
      }
    }
  } /* foreach tag loop */
  if(!rc){
    /* TODO: cached statement */
    rc = fsl_db_exec(db,
                     "REPLACE INTO event"
                     "(type,mtime,objid,user,comment) "
                     "VALUES('g',"
                     "%"FSL_JULIAN_T_PFMT","
                     "%"FSL_ID_T_PFMT","
                     "%Q,%Q)",
                     mtime, d->rid, d->U,
                     (comment->used>1)
                     ? (fsl_buffer_cstr(comment)
                        +1/*leading space on all entries*/)
                     : NULL);
  }

  end:
  fsl__cx_scratchpad_yield(d->f, comment);
  return rc;

}

static int fsl_deck_xlink_f_forum(fsl_deck * const d, void * state __unused){
  if(FSL_SATYPE_FORUMPOST!=d->type) return 0;
  int rc = 0;
  fsl_db * const db = fsl_cx_db_repo(d->f);
  assert(db);
  fsl_cx * const f = d->f;
  fsl_id_t const froot = d->G ? fsl_uuid_to_rid(f, d->G) : d->rid;
  fsl_id_t const fprev = d->P.used ? fsl_uuid_to_rid(f, (char const *)d->P.list[0]): 0;
  fsl_id_t const firt = d->I ? fsl_uuid_to_rid(f, d->I) : 0;
  if( 0==firt ){
    /* This is the start of a new thread, either the initial entry
    ** or an edit of the initial entry. */
    const char * zTitle = d->H;
    const char * zFType;
    if(!zTitle || !*zTitle){
      zTitle = "(Deleted)";
    }
    zFType = fprev ? "Edit" : "Post";
    /* FSL-MISSING:
       assert( manifest_event_triggers_are_enabled ); */
    rc = fsl_db_exec_multi(db,
        "REPLACE INTO event(type,mtime,objid,user,comment)"
        "VALUES('f',%"FSL_JULIAN_T_PFMT",%" FSL_ID_T_PFMT
        ",%Q,'%q: %q')",
        d->D, d->rid, d->U, zFType, zTitle);
    if(rc) goto dberr;
      /*
      ** If this edit is the most recent, then make it the title for
      ** all other entries for the same thread
      */
    if( !fsl_db_exists(db,"SELECT 1 FROM forumpost "
                       "WHERE froot=%" FSL_ID_T_PFMT " AND firt=0"
                       " AND fpid!=%" FSL_ID_T_PFMT
                       " AND fmtime>%"FSL_JULIAN_T_PFMT,
                       froot, d->rid, d->D)){
        /* This entry establishes a new title for all entries on the thread */
      rc = fsl_db_exec_multi(db,
          "UPDATE event"
          " SET comment=substr(comment,1,instr(comment,':')) || ' %q'"
          " WHERE objid IN (SELECT fpid FROM forumpost WHERE froot=% " FSL_ID_T_PFMT ")",
          zTitle, froot);
      if(rc) goto dberr;
    }
  }else{
      /* This is a reply to a prior post.  Take the title from the root. */
    char const * zFType = 0;
    char * zTitle = fsl_db_g_text(
           db, 0, "SELECT substr(comment,instr(comment,':')+2)"
           "  FROM event WHERE objid=%"FSL_ID_T_PFMT, froot);
    if( zTitle==0 ){
      zTitle = fsl_strdup("<i>Unknown</i>");
      if(!zTitle){
        rc = FSL_RC_OOM;
        goto end;
      }
    }
    if( !d->W.used ){
      zFType = "Delete reply";
    }else if( fprev ){
      zFType = "Edit reply";
    }else{
      zFType = "Reply";
    }
    /* FSL-MISSING:
       assert( manifest_event_triggers_are_enabled ); */
    rc = fsl_db_exec_multi(db,
        "REPLACE INTO event(type,mtime,objid,user,comment)"
        "VALUES('f',%"FSL_JULIAN_T_PFMT
        ",%"FSL_ID_T_PFMT",%Q,'%q: %q')",
        d->D, d->rid, d->U, zFType, zTitle);
    fsl_free(zTitle);
    if(rc) goto end;
    if( d->W.used ){
      /* FSL-MISSING:
         backlink_extract(&d->W, d->N, d->rid, BKLNK_FORUM, d->D, 1); */
    }
  }
  end:
  return rc;
  dberr:
  assert(rc);
  assert(db->error.code);
  return fsl_cx_uplift_db_error(f, db);
}


static int fsl_deck_xlink_f_technote(fsl_deck * const d, void * state __unused){
  if(FSL_SATYPE_TECHNOTE!=d->type) return 0;
  char buf[FSL_STRLEN_K256 + 7 /* event-UUID\0 */] = {0};
  fsl_id_t tagid;
  char const * zTag;
  int rc = 0;
  fsl_cx * const f = d->f;
  fsl_db * const db = fsl_cx_db_repo(d->f);
  fsl_snprintf(buf, sizeof(buf), "event-%s", d->E.uuid);
  zTag = buf;
  tagid = fsl_tag_id( f, zTag, 1 );
  if(tagid<=0){
    return f->error.code ? f->error.code :
      fsl_cx_err_set(f, FSL_RC_RANGE,
                     "Got unexpected RID (%"FSL_ID_T_PFMT") "
                     "for tag [%s].",
                     tagid, zTag);
  }
  fsl_id_t const subsequent
    = fsl_db_g_id(db, 0,
                  "SELECT rid FROM tagxref"
                  " WHERE tagid=%"FSL_ID_T_PFMT
                  " AND mtime>=%"FSL_JULIAN_T_PFMT
                  " AND rid!=%"FSL_ID_T_PFMT
                  " ORDER BY mtime",
                  tagid, d->D, d->rid);
  if(subsequent<0){
    rc = fsl_cx_uplift_db_error(d->f, db);
  }else{
    rc = fsl_db_exec(db,
                     "REPLACE INTO event("
                     "type,mtime,"
                     "objid,tagid,"
                     "user,comment,bgcolor"
                     ")VALUES("
                     "'e',%"FSL_JULIAN_T_PFMT","
                     "%"FSL_ID_T_PFMT",%"FSL_ID_T_PFMT","
                     "%Q,%Q,"
                     "  (SELECT value FROM tagxref WHERE "
                     "   tagid=%d"
                     "   AND rid=%"FSL_ID_T_PFMT")"
                     ");",
                     d->E.julian, d->rid, tagid,
                     d->U, d->C, 
                     (int)FSL_TAGID_BGCOLOR, d->rid);
  }
  return rc;
}

static int fsl_deck_xlink_f_wiki(fsl_deck * const d, void * state __unused){
  if(FSL_SATYPE_WIKI!=d->type) return 0;
  int rc;
  char const * zWiki;
  fsl_size_t nWiki = 0;
  char cPrefix = 0;
  char * zTag = fsl_mprintf("wiki-%s", d->L);
  if(!zTag) return FSL_RC_OOM;
  fsl_id_t const tagid = fsl_tag_id( d->f, zTag, 1 );
  if(tagid<=0){
    rc = fsl_cx_err_set(d->f, FSL_RC_ERROR,
                        "Tag [%s] must have been added by main wiki crosslink step.",
                        zTag);
    goto end;
  }
  /* Some of this is duplicated in the main wiki crosslinking code :/. */
  zWiki = d->W.used ? fsl_buffer_cstr(&d->W) : "";
  while( *zWiki && fsl_isspace(*zWiki) ){
    ++zWiki;
    /* Historical behaviour: strip leading spaces. */
  }
  /* As of late 2020, fossil changed the conventions for how wiki
     entries are to be added to the timeline. They requrie a prefix
     character which tells the timeline display and email notification
     generator code what type of change this is: create/update/delete */
  nWiki = fsl_strlen(zWiki);
  if(!nWiki) cPrefix = '-';
  else if( !d->P.used ) cPrefix = '+';
  else cPrefix = ':';
  fsl_db * const db = fsl_cx_db_repo(d->f);
  rc = fsl_db_exec(db,
                   "REPLACE INTO event(type,mtime,objid,user,comment) "
                   "VALUES('w',%"FSL_JULIAN_T_PFMT
                   ",%"FSL_ID_T_PFMT",%Q,'%c%q%q%q');",
                   d->D, d->rid, d->U, cPrefix, d->L,
                   ((d->C && *d->C) ? ": " : ""),
                   ((d->C && *d->C) ? d->C : ""));
  /* Note that wiki pages optionally support d->C (change comment),
     but it's historically unused because it was a late addition to
     the artifact format and is not supported by older fossil
     versions. */
  rc = fsl_cx_uplift_db_error2(d->f, db, rc);
  end:
  fsl_free(zTag);
  return rc;
}


/** @internal

    Installs the core overridable crosslink listeners. "The plan" is
    to do all updates to the event (timeline) table via these
    crosslinkers and perform the core, UI-agnostic, crosslinking bits
    in the internal fsl__deck_crosslink_XXX() functions. That should
    allow clients to override how the timeline is updated without
    requiring them to understand the rest of the required schema
    updates.
*/
int fsl__cx_install_timeline_crosslinkers(fsl_cx * const f){
  int rc;
  assert(!f->xlinkers.used);
  assert(!f->xlinkers.list);
  rc = fsl_xlink_listener(f, "fsl/attachment/timeline",
                          fsl_deck_xlink_f_attachment, 0);
  if(!rc) rc = fsl_xlink_listener(f, "fsl/checkin/timeline",
                          fsl_deck_xlink_f_checkin, 0);
  if(!rc) rc = fsl_xlink_listener(f, "fsl/control/timeline",
                          fsl_deck_xlink_f_control, 0);
  if(!rc) rc = fsl_xlink_listener(f, "fsl/forumpost/timeline",
                          fsl_deck_xlink_f_forum, 0);
  if(!rc) rc = fsl_xlink_listener(f, "fsl/technote/timeline",
                          fsl_deck_xlink_f_technote, 0);
  if(!rc) rc = fsl_xlink_listener(f, "fsl/wiki/timeline",
                          fsl_deck_xlink_f_wiki, 0);
  return rc;
}


static int fsl__deck_crosslink_checkin(fsl_deck * const d,
                                      fsl_id_t *parentid ){
  int rc = 0;
  fsl_cx * const f = d->f;
  fsl_db * const db = fsl_cx_db_repo(f);

  /* TODO: convert these queries to cached statements, for
     the sake of rebuild and friends. And bind() doubles
     instead of %FSL_JULIAN_T_PFMT'ing them.
  */
  if(d->Q.used && fsl_db_table_exists(db, FSL_DBROLE_REPO,
                                      "cherrypick")){
    fsl_size_t i;
    for(i=0; i < d->Q.used; ++i){
      fsl_card_Q const * q = (fsl_card_Q const *)d->Q.list[i];
      rc = fsl_db_exec(db,
          "REPLACE INTO cherrypick(parentid,childid,isExclude)"
          " SELECT rid, %"FSL_ID_T_PFMT", %d"
          " FROM blob WHERE uuid=%Q",
          d->rid, q->type<0 ? 1 : 0, q->target
      );
      if(rc) goto end;
    }
  }
  if(!fsl_repo_has_mlink_mid(db, d->rid)){
    rc = fsl_deck_add_checkin_linkages(d, parentid);
    if(rc) goto end;
    /* FSL-MISSING:
       assert( manifest_event_triggers_are_enabled ); */
    rc = fsl__search_doc_touch(f, d->type, d->rid, 0);
    if(rc) goto end;
    /* If this is a delta-manifest, record the fact that this repository
       contains delta manifests, to free the "commit" logic to generate
       new delta manifests. */
    if(d->B.uuid){
      rc = fsl__cx_update_seen_delta_deck(f);
      if(rc) goto end;
    }
    assert(!rc);
  }/*!exists mlink*/
  end:
  if(rc && !f->error.code && db->error.code){
    fsl_cx_uplift_db_error(f, db);
  }
  return rc;
}

static int fsl__deck_crosslink_wiki(fsl_deck *d){
  char zLength[40] = {0};
  fsl_id_t prior = 0;
  char const * zWiki;
  fsl_size_t nWiki = 0;
  int rc;
  char * zTag = fsl_mprintf("wiki-%s", d->L);
  fsl_cx * const f = d->f;
  fsl_db * const db = fsl_cx_db_repo(f);
  if(!zTag){
    return FSL_RC_OOM;
  }
  assert(f && db);
  zWiki = d->W.used ? fsl_buffer_cstr(&d->W) : "";
  while( *zWiki && fsl_isspace(*zWiki) ){
    ++zWiki;
    /* Historical behaviour: strip leading spaces. */
  }
  nWiki = fsl_strlen(zWiki)
    /* Reminder: use strlen instead of d->W.used just in case that
       one contains embedded NULs in the content. "Shouldn't
       happen," but the API doesn't explicitly prohibit it.
    */;
  fsl_snprintf(zLength, sizeof(zLength), "%"FSL_SIZE_T_PFMT,
               (fsl_size_t)nWiki);
  rc = fsl__tag_insert(f, FSL_TAGTYPE_ADD, zTag, zLength,
                      d->rid, d->D, d->rid, NULL );
  if(rc) goto end;
  if(d->P.used){
    prior = fsl_uuid_to_rid(f, (const char *)d->P.list[0]);
  }
  if(prior>0){
    rc = fsl__content_deltify(f, prior, d->rid, 0);
    if(rc) goto end;
  }
  rc = fsl__search_doc_touch(f, d->type, d->rid, d->L);
  if(rc) goto end;
  if( f->cache.isCrosslinking ){
    rc = fsl__deck_crosslink_add_pending(f, 'w',d->L);
    if(rc) goto end;
  }else{
    /* FSL-MISSING:
       backlink_wiki_refresh(d->L); */
  }
  assert(0==rc);
  rc = fsl__deck_crosslink_fwt_plink(d);
  end:
  fsl_free(zTag);
  return rc;
}

static int fsl__deck_crosslink_attachment(fsl_deck * const d){
  int rc;
  fsl_cx * const f = d->f;
  fsl_db * const db = fsl_cx_db_repo(f);

  rc = fsl_db_exec(db,
                   /* REMINDER: fossil(1) uses INSERT here, but that
                      breaks libfossil crosslinking tests due to a
                      unique constraint violation on attachid. */
                   "REPLACE INTO attachment(attachid, mtime, src, target,"
                   "filename, comment, user) VALUES("
                   "%"FSL_ID_T_PFMT",%"FSL_JULIAN_T_PFMT","
                   "%Q,%Q,%Q,"
                   "%Q,%Q);",
                   d->rid, d->D,
                   d->A.src, d->A.tgt, d->A.name,
                   (d->C ? d->C : ""), d->U);
  if(!rc){
    rc = fsl_db_exec(db,
                   "UPDATE attachment SET isLatest = (mtime=="
                   "(SELECT max(mtime) FROM attachment"
                   "  WHERE target=%Q AND filename=%Q))"
                   " WHERE target=%Q AND filename=%Q",
                   d->A.tgt, d->A.name,
                   d->A.tgt, d->A.name);
  }
  return rc;  
}

static int fsl__deck_crosslink_cluster(fsl_deck * const d){
  /* Clean up the unclustered table... */
  fsl_size_t i;
  fsl_stmt * st = NULL;
  int rc;
  fsl_cx * const f = d->f;
  fsl_db * const db = fsl_cx_db_repo(f);

  assert(d->rid>0);
  rc = fsl__tag_insert(f, FSL_TAGTYPE_ADD, "cluster", NULL,
                       d->rid, d->D, d->rid, NULL);
  if(rc) return rc;
  
  rc = fsl_db_prepare_cached(db, &st,
                             "DELETE FROM unclustered WHERE rid=?"
                             "/*%s()*/",__func__);
  if(rc) return fsl_cx_uplift_db_error(f, db);
  assert(st);
  for( i = 0; i < d->M.used; ++i ){
    fsl_id_t mid;
    char const * uuid = (char const *)d->M.list[i];
    mid = fsl_uuid_to_rid(f, uuid);
    if(mid>0){
      fsl_stmt_bind_id(st, 1, mid);
      if(FSL_RC_STEP_DONE!=fsl_stmt_step(st)){
        rc = fsl_cx_uplift_db_error(f, db);
        break;
      }
      fsl_stmt_reset(st);
    }
  }
  fsl_stmt_cached_yield(st);
  return rc;
}

#if 0
static int fsl__deck_crosslink_control(fsl_deck *const d){
}
#endif

static int fsl__deck_crosslink_forum(fsl_deck * const d){
  int rc = 0;
  fsl_cx * const f = d->f;
  rc = fsl_repo_install_schema_forum(f);
  if(rc) return rc;
  fsl_db * const db = fsl_cx_db_repo(f);
  fsl_id_t const froot = d->G ? fsl_uuid_to_rid(f, d->G) : d->rid;
  fsl_id_t const fprev = d->P.used ? fsl_uuid_to_rid(f, (char const *)d->P.list[0]): 0;
  fsl_id_t const firt = d->I ? fsl_uuid_to_rid(f, d->I) : 0;
  assert(f && db);
  rc = fsl_db_exec_multi(db,
      "REPLACE INTO forumpost(fpid,froot,fprev,firt,fmtime)"
      "VALUES(%" FSL_ID_T_PFMT ",%" FSL_ID_T_PFMT ","
      "nullif(%" FSL_ID_T_PFMT ",0),"
      "nullif(%" FSL_ID_T_PFMT ",0),%"FSL_JULIAN_T_PFMT")",
      d->rid, froot, fprev, firt, d->D
  );
  rc = fsl_cx_uplift_db_error2(f, db, rc);
  if(!rc){
    rc = fsl__search_doc_touch(f, d->type, d->rid, 0);
  }
  if(!rc){
    rc = fsl__deck_crosslink_fwt_plink(d);
  }
  return rc;
}

static int fsl__deck_crosslink_technote(fsl_deck * const d){
  char buf[FSL_STRLEN_K256 + 7 /* event-UUID\0 */] = {0};
  char zLength[40] = {0};
  fsl_id_t tagid;
  fsl_id_t prior = 0, subsequent;
  char const * zWiki;
  char const * zTag;
  fsl_size_t nWiki = 0;
  int rc;
  fsl_cx * const f = d->f;
  fsl_db * const db = fsl_cx_db_repo(f);
  fsl_snprintf(buf, sizeof(buf), "event-%s", d->E.uuid);
  zTag = buf;
  tagid = fsl_tag_id( f, zTag, 1 );
  if(tagid<=0){
    rc = f->error.code ? f->error.code :
      fsl_cx_err_set(f, FSL_RC_RANGE,
                     "Got unexpected RID (%"FSL_ID_T_PFMT") "
                     "for tag [%s].",
                     tagid, zTag);
    goto end;
  }
  zWiki = d->W.used ? fsl_buffer_cstr(&d->W) : "";
  while( *zWiki && fsl_isspace(*zWiki) ){
    ++zWiki;
    /* Historical behaviour: strip leading spaces. */
  }
  nWiki = fsl_strlen(zWiki);
  fsl_snprintf( zLength, sizeof(zLength), "%"FSL_SIZE_T_PFMT,
                (fsl_size_t)nWiki);
  rc = fsl__tag_insert(f, FSL_TAGTYPE_ADD, zTag, zLength,
                      d->rid, d->D, d->rid, NULL );
  if(rc) goto end;
  if(d->P.used){
    prior = fsl_uuid_to_rid(f, (const char *)d->P.list[0]);
    if(prior<0){
      assert(f->error.code);
      rc = f->error.code;
      goto end;
    }
  }
  subsequent = fsl_db_g_id(db, 0,
  /* BUG: see:
     https://fossil-scm.org/forum/forumpost/c58fd8de53 */
                           "SELECT rid FROM tagxref"
                           " WHERE tagid=%"FSL_ID_T_PFMT
                           " AND mtime>=%"FSL_JULIAN_T_PFMT
                           " AND rid!=%"FSL_ID_T_PFMT
                           " ORDER BY mtime",
                           tagid, d->D, d->rid);
  if(subsequent<0){
    assert(db->error.code);
    rc = fsl_cx_uplift_db_error(f, db);
    goto end;
  }
  else if( prior > 0 ){
    rc = fsl__content_deltify(f, prior, d->rid, 0);
    if( !rc && !subsequent ){
      rc = fsl_db_exec(db,
                       "DELETE FROM event"
                       " WHERE type='e'"
                       "   AND tagid=%"FSL_ID_T_PFMT
                       "   AND objid IN"
                       " (SELECT rid FROM tagxref "
                       " WHERE tagid=%"FSL_ID_T_PFMT")",
                       tagid, tagid);
    }
  }
  if(rc) goto end;
  if( subsequent>0 ){
    rc = fsl__content_deltify(f, d->rid, subsequent, 0);
  }else{
    /* timeline update is deferred to another crosslink
       handler */
    rc = fsl__search_doc_touch(f, d->type, d->rid, 0);
    /* FSL-MISSING:
       assert( manifest_event_triggers_are_enabled ); */
  }
  if(!rc){
    rc = fsl__deck_crosslink_fwt_plink(d);
  }
  end:
  return rc;
}

static int fsl__deck_crosslink_ticket(fsl_deck * const d){
  int rc;
  fsl_cx * const f = d->f;
  char * zTag;
  fsl_stmt qAtt = fsl_stmt_empty;
  assert(f->cache.isCrosslinking
         && "This only works if fsl__crosslink_begin() is active.");
  zTag = fsl_mprintf("tkt-%s", d->K);
  if(!zTag) return FSL_RC_OOM;
  rc = fsl__tag_insert(f, FSL_TAGTYPE_ADD, zTag, NULL,
                       d->rid, d->D, d->rid, NULL);
  fsl_free(zTag);
  if(rc) goto end;
  assert(d->K);
  rc = fsl__deck_crosslink_add_pending(f, 't', d->K);
  if(rc) goto end;;
  /* Locate and update comment for any attachments */
  rc = fsl_cx_prepare(f, &qAtt,
                      "SELECT attachid, src, target, filename FROM attachment"
                      " WHERE target=%Q",
                      d->K);
  while(0==rc && FSL_RC_STEP_ROW==fsl_stmt_step(&qAtt)){
    const char *zAttachId = fsl_stmt_g_text(&qAtt, 0, NULL);
    const char *zSrc = fsl_stmt_g_text(&qAtt, 1, NULL);
    const char *zTarget = fsl_stmt_g_text(&qAtt, 2, NULL);
    const char *zName = fsl_stmt_g_text(&qAtt, 3, NULL);
    const bool isAdd = (zSrc && zSrc[0]) ? true : false;
    char *zComment;
    if( isAdd ){
      zComment =
        fsl_mprintf(
                    "Add attachment [/artifact/%!S|%h] to ticket [%!S|%S]",
                    zSrc, zName, zTarget, zTarget);
    }else{
      zComment =
        fsl_mprintf("Delete attachment \"%h\" from ticket [%!S|%S]",
                    zName, zTarget, zTarget);
    }
    if(!zComment){
      rc = FSL_RC_OOM;
      break;
    }
    rc = fsl_cx_exec_multi(f, "UPDATE event SET comment=%Q, type='t'"
                           " WHERE objid=%Q",
                           zComment, zAttachId);
    fsl_free(zComment);
  }
  end:
  fsl_stmt_finalize(&qAtt);
  return rc;
}

int fsl__deck_crosslink_one( fsl_deck * const d ){
  int rc;
  assert(d->f && "API misuse:fsl_deck::f == NULL");
  rc = fsl__crosslink_begin(d->f);
  if(rc) return rc;
  rc = fsl__deck_crosslink(d);
  assert(0!=fsl_db_transaction_level(fsl_cx_db_repo(d->f))
         && "Expecting transaction level from fsl__crosslink_begin()");
  rc = fsl__crosslink_end(d->f, rc);
  return rc;
}

int fsl__deck_crosslink( fsl_deck /* const */ * const d ){
  int rc = 0;
  fsl_cx * f = d->f;
  fsl_db * db = f ? fsl_needs_repo(f) : NULL;
  fsl_id_t parentid = 0;
  fsl_int_t const rid = d->rid;
  if(!f) return FSL_RC_MISUSE;
  else if(rid<=0){
    return fsl_cx_err_set(f, FSL_RC_RANGE,
                          "Invalid RID for crosslink: %"FSL_ID_T_PFMT,
                          rid);
  }
  else if(!db) return FSL_RC_NOT_A_REPO;
  else if(!fsl_deck_has_required_cards(d)){
    assert(d->f->error.code);
    return d->f->error.code;
  }else if(f->cache.xlinkClustersOnly && (FSL_SATYPE_CLUSTER!=d->type)){
    /* is it okay to bypass the registered xlink listeners here?  The
       use case called for by this is not yet implemented in
       libfossil. */
    return 0;
  }
  rc = fsl_db_transaction_begin(db);
  if(rc) goto end;
  if(FSL_SATYPE_CHECKIN==d->type
     && d->B.uuid && !d->B.baseline){
    rc = fsl_deck_baseline_fetch(d);
    if(rc) goto end;
    assert(d->B.baseline);
  }
  switch(d->type){
    case FSL_SATYPE_CHECKIN:
      rc = fsl__deck_crosslink_checkin(d, &parentid);
      break;
    case FSL_SATYPE_CLUSTER:
      rc = fsl__deck_crosslink_cluster(d);
      break;
    default:
      break;
  }
  if(rc) goto end;
  switch(d->type){
    case FSL_SATYPE_CONTROL:
    case FSL_SATYPE_CHECKIN:
    case FSL_SATYPE_TECHNOTE:
      rc = fsl__deck_crosslink_apply_tags(f, d, db, rid, parentid);
      break;
    default:
      break;
  }
  if(rc) goto end;
  switch(d->type){
    case FSL_SATYPE_WIKI:
      rc = fsl__deck_crosslink_wiki(d);
      break;
    case FSL_SATYPE_FORUMPOST:
      rc = fsl__deck_crosslink_forum(d);
      break;
    case FSL_SATYPE_TECHNOTE:
      rc = fsl__deck_crosslink_technote(d);
      break;
    case FSL_SATYPE_TICKET:
      rc = fsl__deck_crosslink_ticket(d);
      break;
    case FSL_SATYPE_ATTACHMENT:
      rc = fsl__deck_crosslink_attachment(d);
      break;
    /* FSL_SATYPE_CONTROL is handled above except for the timeline
       update, which is handled by a callback below */
    default:
      break;
  }
  /* Call any crosslink callbacks... */
  if(0==rc && FSL_SATYPE_TICKET!=d->type){
    /* ^^^ the real work of ticket crosslinking is delated until
       fsl__crosslink_end(), for reasons lost to history, so we'll skip
       calling the xlink listeners for those until that step. */
    rc = fsl__call_xlink_listeners(d);
  }
  end:
  if(!rc){
    rc = fsl_db_transaction_end(db, false);
  }else{
    if(db->error.code && !f->error.code){
      fsl_cx_uplift_db_error(f,db);
    }
    fsl_db_transaction_end(db, true);
  }
  return rc ? rc : f->interrupted;
}/*end fsl__deck_crosslink()*/


/**
    Return true if z points to the first character after a blank line.
    Tolerate either \r\n or \n line endings. As this looks backwards
    in z, z must point to at least 3 characters past the beginning of
    a legal string.
 */
static bool fsl_after_blank_line(const char *z){
  if( z[-1]!='\n' ) return false;
  if( z[-2]=='\n' ) return true;
  if( z[-2]=='\r' && z[-3]=='\n' ) return true;
  return false;
}

/**
    Verifies that ca points to at least 35 bytes of memory
    which hold (at the end) a Z card and its hash value.
   
    Returns 0 if the string does not contain a Z card,
    a positive value if it can validate the Z card's hash,
    and a negative value on hash mismatch.
*/
static int fsl_deck_verify_Z_card(unsigned char const * ca, fsl_size_t n){
  if( n<35 ) return 0;
  if( ca[n-35]!='Z' || ca[n-34]!=' ' ) return 0;
  else{
    unsigned char digest[16];
    char hex[FSL_STRLEN_MD5+1];
    unsigned char const * zHash = ca+n-FSL_STRLEN_MD5-1;
    fsl_md5_cx md5 = fsl_md5_cx_empty;
    unsigned char const * zHashEnd =
      ca + n -
      2 /* 'Z ' */
      - FSL_STRLEN_MD5
      - 1 /* \n */;
    assert( 'Z' == (char)*zHashEnd );
    fsl_md5_update(&md5, ca, zHashEnd-ca);
    fsl_md5_final(&md5, digest);
    fsl_md5_digest_to_base16(digest, hex);
    return (0==memcmp(zHash, hex, FSL_STRLEN_MD5))
      ? 1
      : -1;
  }
}


/** @internal

    Remove the PGP signature from a raw artifact, if there is one.

    Expects *pz to point to *pn bytes of string memory which might
    or might not be prefixed by a PGP signature.  If the string is
    enveloped in a signature, then upon returning *pz will point to
    the first byte after the end of the PGP header and *pn will
    contain the length of the content up to, but not including, the
    PGP footer.

    If *pz does not look like a PGP header then this is a no-op.

    Neither pointer may be NULL and *pz must point to *pn bytes of
    valid memory. If *pn is initially less than 59, this is a no-op.
*/
static void fsl__remove_pgp_signature(unsigned char const **pz, fsl_size_t *pn){
  unsigned char const *z = *pz;
  fsl_int_t n = (fsl_int_t)*pn;
  fsl_int_t i;
  if( n<59 || memcmp(z, "-----BEGIN PGP SIGNED MESSAGE-----", 34)!=0 ) return;
  for(i=34; i<n && !fsl_after_blank_line((char const *)(z+i)); i++){}
  if( i>=n ) return;
  z += i;
  *pz = z;
#if 1
  unsigned char const * bps =
    (unsigned char const *)strstr((char const *)z, "\n-----BEGIN PGP SIGNATURE-");
  if(bps){
    n = (fsl_int_t)(bps - z) + 1 /*newline*/;
  }
#else
  n -= i;
  for(i=n-1; i>=0; i--){
    if( z[i]=='\n' && memcmp(&z[i],"\n-----BEGIN PGP SIGNATURE-", 25)==0 ){
      /** valgrind warns on ^^^^ this ^^^^ line:
          Conditional jump or move depends on uninitialised value(s)

          It affects at least 2 artifacts in the libfossil repo:

          240deb757b12bf953b5dbf5c087c80f60ae68934
          f4e5795f9ec7df587756f08ea875c8be259b7917
      */
      n = i+1;
      break;
    }
  }
#endif
  *pn = (fsl_size_t)n;
  return;
}


/**
    Internal helper for parsing manifests. Holds a source file (memory
    range) and gets updated by fsl__deck_next_token() and friends.
*/
struct fsl__src {
  /**
      First char of the next token.
   */
  unsigned char * z;
  /**
      One-past-the-end of the manifest.
   */
  unsigned char * zEnd;
  /**
      True if z points to the start of a new line.
   */
  bool atEol;
};
typedef struct fsl__src fsl__src;
static const fsl__src fsl__src_empty = {NULL,NULL,0};

/**
   Return a pointer to the next token.  The token is zero-terminated.
   Return NULL if there are no more tokens on the current line, but
   the call after that will return non-NULL unless we have reached the
   end of the input. If this function returns non-NULL then *pLen is
   set to the byte length of the new token. Once the end of the input
   is reached, this function always returns NULL.
*/
static unsigned char *fsl__deck_next_token(fsl__src * const p,
                                           fsl_size_t * const pLen){
  if( p->atEol ) return NULL;
  int c;
  unsigned char * z = p->z;
  unsigned char * const zStart = z;
  while( (c=(*z))!=' ' && c!='\n' ) ++z;
  *z = 0;
  p->z = &z[1];
  p->atEol = c=='\n';
  *pLen = z - zStart;
  return zStart;
}

/**
    Return the card-type for the next card. Return 0 if there are no
    more cards or if we are not at the end of the current card.
 */
static unsigned char deck__next_card(fsl__src * const p){
  unsigned char c;
  if( !p->atEol || p->z>=p->zEnd ) return 0;
  c = p->z[0];
  if( p->z[1]==' ' ){
    p->z += 2;
    p->atEol = false;
  }else if( p->z[1]=='\n' ){
    p->z += 2;
    p->atEol = true;
  }else{
    c = 0;
  }
  return c;
}

/**
    Internal helper for fsl_deck_parse(). Expects l to be an array of
    26 entries, representing the letters of the alphabet (A-Z), with a
    value of 0 if the card was not seen during parsing and a value >0
    if it was. Returns the deduced artifact type.  Returns
    FSL_SATYPE_ANY if the result is ambiguous.

    Note that we cannot reliably guess until we've seen at least 3
    cards. 2 cards is enough for most cases but can lead to
    FSL_SATYPE_CHECKIN being prematurely selected in one case.
   
    It should guess right for any legal manifests, but it does not go
    out of its way to detect incomplete/invalid ones.
 */
static fsl_satype_e fsl_deck_guess_type( const int * l ){
#if 0
  /* For parser testing only... */
  int i;
  assert(!l[26]);
  MARKER(("Cards seen during parse:\n"));
  for( i = 0; i < 26; ++i ){
    if(l[i]) putchar('A'+i);
  }
  putchar('\n');
#endif
  /*
     Now look for combinations of cards which will uniquely
     identify any syntactical legal combination of cards.
     
     A larger brain than mine could probably come up with a hash of
     l[] which could determine this in O(1). But please don't
     reimplement this as such unless mere mortals can maintain it -
     any performance gain is insignificant in the context of the
     underlying SCM/db operations.

     Note that the order of these checks is sometimes significant!
  */
#define L(X) l[X-'A']
  if(L('M')) return FSL_SATYPE_CLUSTER;
  else if(L('E')) return FSL_SATYPE_EVENT;
  else if(L('G') || L('H') || L('I')) return FSL_SATYPE_FORUMPOST;
  else if(L('L') || L('W')) return FSL_SATYPE_WIKI;
  else if(L('J') || L('K')) return FSL_SATYPE_TICKET;
  else if(L('A')) return FSL_SATYPE_ATTACHMENT;
  else if(L('B') || L('C') || L('F')
          || L('P') || L('Q') || L('R')) return FSL_SATYPE_CHECKIN;
  else if(L('D') && L('T') && L('U')) return FSL_SATYPE_CONTROL;
#undef L
  return FSL_SATYPE_ANY;
}

bool fsl_might_be_artifact(fsl_buffer const * const src){
  unsigned const char * z = src->mem;
  fsl_size_t n = src->used;
  if(n<36) return false;
  fsl__remove_pgp_signature(&z, &n);
  if(n<36) return false;
  else if(z[0]<'A' || z[0]>'Z' || z[1]!=' '
          || z[n-35]!='Z'
          || z[n-34]!=' '
          || !fsl_validate16((const char *)z+n-33, FSL_STRLEN_MD5)){
    return false;
  }
  return true;
}

int fsl_deck_parse2(fsl_deck * const d, fsl_buffer * const src, fsl_id_t rid){
#ifdef ERROR
#  undef ERROR
#endif
#define ERROR(RC,MSG) do{ rc = (RC); zMsg = (MSG); goto bailout; } while(0)
#define SYNTAX(MSG) ERROR(rc ? rc : FSL_RC_SYNTAX,MSG)
  bool isRepeat = 0/* , hasSelfRefTag = 0 */;
  int rc = 0;
  fsl__src x = fsl__src_empty;
  char const * zMsg = NULL;
  char cType = 0, cPrevType = 0;
  unsigned char * z = src ? src->mem : NULL;
  fsl_size_t tokLen = 0;
  unsigned char * token;
  fsl_size_t n = z ? src->used : 0;
  unsigned char * uuid;
  double ts;
  int cardCount = 0;
  fsl_cx * const f = d->f;
  fsl_error * const err = f ? &f->error : 0;
  int stealBuf = 0 /* gets incremented if we need to steal src->mem. */;
  unsigned nSelfTag = 0 /* number of T cards which refer to '*' (this artifact). */;
  unsigned nSimpleTag = 0 /* number of T cards with "+" prefix */;
  /*
    lettersSeen keeps track of the card letters we have seen so that
    we can then relatively quickly figure out what type of manifest we
    have parsed without having to inspect the card contents. Each
    index records a count of how many of that card we've seen.
  */
  int lettersSeen[27] = {0/*A*/,0,0,0,0,0,0,0,0,0,0,0,0,
                         0,0,0,0,0,0,0,0,0,0,0,0,0/*Z*/,
                         0 /* sentinel element for reasons lost to
                             history but removing it breaks stuff. */};
  if(!f || !z) return FSL_RC_MISUSE;
  if(rid>0 && (fsl__cx_mcache_search2(f, rid, d, FSL_SATYPE_ANY, &rc)
               || rc)){
    if(0==rc){
      assert(d->rid == rid);
      fsl_buffer_clear(src);
    }
    return rc;
  }else if(rid<0){
    return fsl_error_set(err, FSL_RC_RANGE,
                         "Invalid (negative) RID %" FSL_ID_T_PFMT
                         " for %s()", rid, __func__);
  }else if(!*z || !n || ( '\n' != z[n-1]) ){
    /* Every control artifact ends with a '\n' character. Exit early
       if that is not the case for this artifact. */
    return fsl_error_set(err, FSL_RC_SYNTAX, "%s.",
                         n ? "Not terminated with \\n"
                         : "Zero-length input");
  }
  if((0==rid) || fsl_id_bag_contains(&f->cache.mfSeen,rid)){
    isRepeat = 1;
  }else{
    isRepeat = 0;
    rc = fsl_id_bag_insert(&f->cache.mfSeen, rid);
    if(rc){
      assert(FSL_RC_OOM==rc);
      return rc;
    }
  }

  /*
    Verify that the first few characters of the artifact look like a
    control artifact.
  */
  if( !fsl_might_be_artifact(src) ){
    SYNTAX("Content does not look like a structural artifact");
  }

  fsl_deck_clean(d);
  fsl_deck_init(f, d, FSL_SATYPE_ANY);

  /*
    Strip off the PGP signature if there is one. Example of signed
    manifest:

    https://fossil-scm.org/index.html/artifact/28987096ac
  */
  {
    unsigned char const * zz = z;
    fsl__remove_pgp_signature(&zz, &n);
    z = (unsigned char *)zz;
  }

  /* Verify the Z card */
  if( fsl_deck_verify_Z_card(z, n) < 0 ){
    ERROR(FSL_RC_CONSISTENCY, "Z-card checksum mismatch");
  }

  /* legacy: not yet clear if we need this:
     if( !isRepeat ) g.parseCnt[0]++; */

  /*
    Reminder: parsing modifies the input (to simplify the
    tokenization/parsing).

    As of mid-201403, we recycle as much as possible from the source
    buffer.
  */
  /* Now parse, card by card... */
  x.z = z;
  x.zEnd = z+n;
  x.atEol= true;

  /* Parsing helpers... */
#define TOKEN(DEFOS) tokLen=0; token = fsl__deck_next_token(&x,&tokLen);    \
  if(token && tokLen && (DEFOS)) fsl_bytes_defossilize(token, &tokLen)
#define TOKEN_EXISTS(MSG_IF_NOT) if(!token){ SYNTAX(MSG_IF_NOT); }(void)0
#define TOKEN_CHECKHEX(MSG) if(token && (int)tokLen!=fsl_is_uuid((char const *)token))\
  { SYNTAX(MSG); }
#define TOKEN_UUID(CARD) TOKEN_CHECKHEX("Malformed UUID in " #CARD "-card")
#define TOKEN_MD5(ERRMSG) if(!token || FSL_STRLEN_MD5!=(int)tokLen) \
  {SYNTAX(ERRMSG);}
  /**
     Reminder: we do not know the type of the manifest at this point,
     so all of the fsl_deck_add/set() bits below can't do their
     validation. We have to determine at parse-time (or afterwards)
     which type of deck it is based on the cards we've seen. We guess
     the type as early as possible to enable during-parse validation,
     and do a post-parse check for the legality of cards added before
     validation became possible.
   */
  
#define SEEN(CARD) lettersSeen[*#CARD - 'A']
  for( cPrevType=1; !rc && (0 < (cType = deck__next_card(&x)));
       cPrevType = cType ){
    ++cardCount;
    if(cType<cPrevType){
      if(d->E.uuid && 'N'==cType && 'P'==cPrevType){
        /* Workaround for a pair of historical fossil bugs
           which synergized to allow malformed technotes to
           be saved:
           https://fossil-scm.org/home/info/023fddeec4029306 */
      }else{
        SYNTAX("Cards are not in strict lexical order");
      }
    }
    assert(cType>='A' && cType<='Z');
    if(cType>='A' && cType<='Z'){
        ++lettersSeen[cType-'A'];
    }else{
      SYNTAX("Invalid card name");
    }
    switch(cType){
      /*
             A <filename> <target> ?<source>?

         Identifies an attachment to either a wiki page, a ticket, or
         a technote.  <source> is the artifact that is the attachment.
         <source> is omitted to delete an attachment.  <target> is the
         name of a wiki page, technote, or ticket to which that
         attachment is connected.
      */
      case 'A':{
        unsigned char * name, * src;
        if(1<SEEN(A)){
          SYNTAX("Multiple A-cards");
        }
        TOKEN(1);
        TOKEN_EXISTS("Missing filename for A-card");
        name = token;
        if(!fsl_is_simple_pathname( (char const *)name, 0 )){
          SYNTAX("Invalid filename in A-card");
        }          
        TOKEN(1);
        TOKEN_EXISTS("Missing target name in A-card");
        uuid = token;
        TOKEN(0);
        TOKEN_UUID(A);
        src = token;
        d->A.name = (char *)name;
        d->A.tgt = (char *)uuid;
        d->A.src = (char *)src;
        ++stealBuf;
        /*rc = fsl_deck_A_set(d, (char const *)name,
          (char const *)uuid, (char const *)src);*/
        d->type = FSL_SATYPE_ATTACHMENT;
        break;
      }
      /*
            B <uuid>

         A B-line gives the UUID for the baseline of a delta-manifest.
      */
      case 'B':{
        if(d->B.uuid){
          SYNTAX("Multiple B-cards");
        }
        TOKEN(0);
        TOKEN_UUID(B);
        d->B.uuid = (char *)token;
        ++stealBuf;
        d->type = FSL_SATYPE_CHECKIN;
        /* rc = fsl_deck_B_set(d, (char const *)token); */
        break;
      }
      /*
             C <comment>

         Comment text is fossil-encoded.  There may be no more than
         one C line.  C lines are required for manifests, are optional
         for Events and Attachments, and are disallowed on all other
         control files.
      */
      case 'C':{
        if( d->C ){
          SYNTAX("more than one C-card");
        }
        TOKEN(1);
        TOKEN_EXISTS("Missing comment text for C-card");
        /* rc = fsl_deck_C_set(d, (char const *)token, (fsl_int_t)tokLen); */
        d->C = (char *)token;
        ++stealBuf;
        break;
      }
      /*
             D <timestamp>

         The timestamp should be ISO 8601.   YYYY-MM-DDtHH:MM:SS
         There can be no more than 1 D line.  D lines are required
         for all control files except for clusters.
      */
      case 'D':{
#define TOKEN_DATETIME(LETTER,MEMBER)                                     \
        if( d->MEMBER>0.0 ) { SYNTAX("More than one "#LETTER"-card"); } \
        TOKEN(0); \
        TOKEN_EXISTS("Missing date part of "#LETTER"-card"); \
        if(!fsl_str_is_date((char const *)token)){\
          SYNTAX("Malformed date part of "#LETTER"-card"); \
        } \
        if(!fsl_iso8601_to_julian((char const *)token, &ts)){   \
          SYNTAX("Cannot parse date from "#LETTER"-card"); \
        } (void)0

        TOKEN_DATETIME(D,D);
        rc = fsl_deck_D_set(d, ts);
        break;
      }
      /*
             E <timestamp> <uuid>

         An "event" (technote) card that contains the timestamp of the
         event in the format YYYY-MM-DDtHH:MM:SS and a unique
         identifier for the event. The event timestamp is distinct
         from the D timestamp. The D timestamp is when the artifact
         was created whereas the E timestamp is when the specific
         event is said to occur.
      */
      case 'E':{
        TOKEN_DATETIME(E,E.julian);
        TOKEN(0);
        TOKEN_EXISTS("Missing UUID part of E-card");
        TOKEN_UUID(E);
        d->E.julian = ts;
        d->E.uuid = (char *)token;
        ++stealBuf;
        d->type = FSL_SATYPE_EVENT;
        break;
      }
      /*
             F <filename> ?<uuid>? ?<permissions>? ?<old-name>?

         Identifies a file in a manifest.  Multiple F lines are
         allowed in a manifest.  F lines are not allowed in any other
         control file.  The filename and old-name are fossil-encoded.

         In delta manifests, deleted files are denoted by the 1-arg
         form. In baseline manifests, deleted files simply are not in
         the manifest.
      */
      case 'F':{
        char * name;
        char * perms = NULL;
        char * priorName = NULL;
        fsl_fileperm_e perm = FSL_FILE_PERM_REGULAR;
        fsl_card_F * fc = NULL;
        rc = 0;
        if(!d->F.capacity){
          /**
             Basic tests with various repos have shown that the
             approximate number of F-cards in a manifest is roughly
             the manifest size/75. We'll use that as an initial alloc
             size.
          */
          rc = fsl_card_F_list_reserve(&d->F, src->used/75+10);
        }
        TOKEN(0);
        TOKEN_EXISTS("Missing name for F-card");
        name = (char *)token;
        TOKEN(0);
        TOKEN_UUID(F);
        uuid = token;
        TOKEN(0);
        if(token){
          perms = (char *)token;
          switch(*perms){
            case 0:
              /* Some (maybe only 1) ancient fossil(1) artifact(s) have a trailing
                 space which triggers this. e.g.

                 https://fossil-scm.org/home/info/32b480faa3465591b8549bdfd889d62d7a8d16a8
              */
              break;
            case 'w': perm = FSL_FILE_PERM_REGULAR; break;
            case 'x': perm = FSL_FILE_PERM_EXE; break;
            case 'l': perm = FSL_FILE_PERM_LINK; break;
            default:
              /*MARKER(("Unmatched perms string character: %d / %c !", (int)*perms, *perms));*/
              SYNTAX("Invalid perms string character");
          }
          TOKEN(0);
          if(token) priorName = (char *)token;
        }
        fsl_bytes_defossilize( (unsigned char *)name, 0 );
        if(fsl_is_reserved_fn(name, -1)){
          /* Some historical (pre-late-2020) manifests contain files
             they really shouldn't, like _FOSSIL_ and .fslckout.
             Since late 2020, fossil simply skips over these when
             parsing manifests, so we'll do the same. */
          break;
        }
        if(priorName) fsl_bytes_defossilize( (unsigned char *)priorName, 0 );
        fc = rc ? 0 : fsl_card_F_list_push(&d->F);
        if(!fc){
          zMsg = "OOM";
          goto bailout;
        }
        ++stealBuf;
        assert(d->F.used>1
               ? (FSL_CARD_F_LIST_NEEDS_SORT & d->F.flags)
               : 1);
        fc->deckOwnsStrings = true;
        fc->name = name;
        fc->priorName = priorName;
        fc->perm = perm;
        fc->uuid = (fsl_uuid_str)uuid;
        d->type = FSL_SATYPE_CHECKIN;
        break;
      }
      /*
        G <uuid>

        A G-line gives the UUID for the thread root of a forum post.
      */
      case 'G':{
        if(d->G){
          SYNTAX("Multiple G-cards");
        }
        TOKEN(0);
        TOKEN_EXISTS("Missing UUID in G-card");
        TOKEN_UUID(G);
        d->G = (char*)token;
        ++stealBuf;
        d->type = FSL_SATYPE_FORUMPOST;
        break;
      }
     /*
         H <forum post title>

         H text is fossil-encoded.  There may be no more than one H
         line.  H lines are optional for forum posts and are
         disallowed on all other control files.
      */
      case 'H':{
        if( d->H ){
          SYNTAX("more than one H-card");
        }
        TOKEN(1);
        TOKEN_EXISTS("Missing text for H-card");
        d->H = (char *)token;
        ++stealBuf;
        d->type = FSL_SATYPE_FORUMPOST;
        break;
      }
      /*
        I <uuid>

        A I-line gives the UUID for the in-response-to UUID for 
        a forum post.
      */
      case 'I':{
        if(d->I){
          SYNTAX("Multiple I-cards");
        }
        TOKEN(0);
        TOKEN_EXISTS("Missing UUID in I-card");
        TOKEN_UUID(I);
        d->I = (char*)token;
        ++stealBuf;
        d->type = FSL_SATYPE_FORUMPOST;
        break;
      }
      /*
         J <name> ?<value>?

         Specifies a name value pair for ticket.  If the first character
         of <name> is "+" then the <value> is appended to any preexisting
         value.  If <value> is omitted then it is understood to be an
         empty string.
      */
      case 'J':{
        char const * field;
        bool isAppend = 0;
        TOKEN(1);
        TOKEN_EXISTS("Missing field name for J-card");
        field = (char const *)token;
        if('+'==*field){
          isAppend = 1;
          ++field;
        }
        TOKEN(1);
        rc = fsl_deck_J_add(d, isAppend, field,
                            (char const *)token);
        d->type = FSL_SATYPE_TICKET;
        break;
      }
      /*
         K <uuid>

         A K-line gives the UUID for the ticket which this control file
         is amending.
      */
      case 'K':{
        if(d->K){
          SYNTAX("Multiple K-cards");
        }
        TOKEN(0);
        TOKEN_EXISTS("Missing UUID in K-card");
        TOKEN_UUID(K);
        d->K = (char*)token;
        ++stealBuf;
        d->type = FSL_SATYPE_TICKET;
        break;
      }
      /*
         L <wikititle>

         The wiki page title is fossil-encoded.  There may be no more than
         one L line.
      */
      case 'L':{
        if(d->L){
          SYNTAX("Multiple L-cards");
        }
        TOKEN(1);
        TOKEN_EXISTS("Missing text for L-card");
        d->L = (char*)token;
        ++stealBuf;
        d->type = FSL_SATYPE_WIKI;
        break;
      }
      /*
         M <uuid>

         An M-line identifies another artifact by its UUID.  M-lines
         occur in clusters only.
      */
      case 'M':{
        TOKEN(0);
        TOKEN_EXISTS("Missing UUID for M-card");
        TOKEN_UUID(M);
        ++stealBuf;
        d->type = FSL_SATYPE_CLUSTER;
        rc = fsl_list_append(&d->M, token);
        if( !rc && d->M.used>1 &&
            fsl_strcmp((char const *)d->M.list[d->M.used-2],
                       (char const *)token)>=0 ){
          SYNTAX("M-card in the wrong order");
        }
        break;
      }
      /*
         N <uuid>

         An N-line identifies the mimetype of wiki or comment text.
      */
      case 'N':{
        if(1<SEEN(N)){
          SYNTAX("Multiple N-cards");
        }
        TOKEN(0);
        TOKEN_EXISTS("Missing UUID on N-card");
        ++stealBuf;
        d->N = (char *)token;
        break;
      }

      /*
         P <uuid> ...

         Specify one or more other artifacts which are the parents of
         this artifact.  The first parent is the primary parent.  All
         others are parents by merge.
      */
      case 'P':{
        if(1<SEEN(P)){
          SYNTAX("More than one P-card");
        }
        TOKEN(0);
#if 0
        /* The docs all claim that this card does not exist on the first
           manifest, but in fact it does exist but has no UUID,
           which is invalid per all the P-card docs. Skip this
           check (A) for the sake of manifest #1 and (B) because
           fossil(1) does it this way.
        */
        TOKEN_EXISTS("Missing primary parent UUID for P-card");
#endif
        while( token && !rc ){
          TOKEN_UUID(P);
          ++stealBuf;
          rc = fsl_list_append(&d->P, token);
          if(!rc){
            TOKEN(0);
          }
        }
        break;
      }
      /*
         Q (+|-)<uuid> ?<uuid>?

         Specify one or a range of checkins that are cherrypicked into
         this checkin ("+") or backed out of this checkin ("-").
      */
      case 'Q':{
        fsl_cherrypick_type_e qType = FSL_CHERRYPICK_INVALID;
        TOKEN(0);
        TOKEN_EXISTS("Missing target UUID for Q-card");
        switch((char)*token){
          case '-': qType = FSL_CHERRYPICK_BACKOUT; break;
          case '+': qType = FSL_CHERRYPICK_ADD; break;
          default:
            SYNTAX("Malformed target UUID in Q-card");
        }
        assert(qType);
        uuid = ++token; --tokLen;
        TOKEN_UUID(Q);
        TOKEN(0);
        if(token){
          TOKEN_UUID(Q);
        }
        d->type = FSL_SATYPE_CHECKIN;
        rc = fsl_deck_Q_add(d, qType, (char const *)uuid,
                            (char const *)token);
        break;
      }
      /*
         R <md5sum>

         Specify the MD5 checksum over the name and content of all files
         in the manifest.
      */
      case 'R':{
        if(1<SEEN(R)){
          SYNTAX("More than one R-card");
        }
        TOKEN(0);
        TOKEN_EXISTS("Missing MD5 token in R-card");
        TOKEN_MD5("Malformed MD5 token in R-card");
        d->R = (char *)token;
        ++stealBuf;
        d->type = FSL_SATYPE_CHECKIN;
        break;
      }
      /*
         T (+|*|-)<tagname> <uuid> ?<value>?

         Create or cancel a tag or property. The tagname is fossil-encoded.
         The first character of the name must be either "+" to create a
         singleton tag, "*" to create a propagating tag, or "-" to create
         anti-tag that undoes a prior "+" or blocks propagation of of
         a "*".

         The tag is applied to <uuid>. If <uuid> is "*" then the tag is
         applied to the current manifest. If <value> is provided then 
         the tag is really a property with the given value.

         Tags are not allowed in clusters.  Multiple T lines are allowed.
      */
      case 'T':{
        unsigned char * name, * value;
        fsl_tagtype_e tagType = FSL_TAGTYPE_INVALID;
        TOKEN(1);
        TOKEN_EXISTS("Missing name for T-card");
        name = token;
        if( fsl_validate16((char const *)&name[1],
                           fsl_strlen((char const *)&name[1])) ){
          /* Do not allow tags whose names look like a hash */
          SYNTAX("T-card name looks like a hexadecimal hash");
        }
        TOKEN(0);
        TOKEN_EXISTS("Missing UUID on T-card");
        if(fsl_is_uuid_len((int)tokLen)){
          TOKEN_UUID(T);
          uuid = token;
        }else if( 1==tokLen && '*'==(char)*token ){
          /* tag for the current artifact */
          ++nSelfTag;
          uuid = NULL;
        }else{
          SYNTAX("Malformed UUID in T-card");
        }
        TOKEN(1);
        value = token;
        switch(*name){
          case '*': tagType = FSL_TAGTYPE_PROPAGATING; break;
          case '+': tagType = FSL_TAGTYPE_ADD;
            ++nSimpleTag;
            break;
          case '-': tagType = FSL_TAGTYPE_CANCEL; break;
          default: SYNTAX("Malformed tag name");
        }
        ++name /* skip type marker byte */;
        /* Tag order check from:

           https://fossil-scm.org/home/info/55cacfcace

           (It was subsequently made stricter so that the same tag
           type/name/target combination fails.)

           That's difficult to do here until _after_ we add the new
           tag to the list... */
        rc = fsl_deck_T_add(d, tagType, (fsl_uuid_cstr)uuid,
                            (char const *)name,
                            (char const *)value);
        if(0==rc && d->T.used>1){
          fsl_card_T const * tagPrev =
            (fsl_card_T const *)d->T.list[d->T.used-2];
          fsl_card_T const * tagSelf =
            (fsl_card_T const *)d->T.list[d->T.used-1];
          int const cmp = fsl_card_T_cmp(&tagPrev, &tagSelf);
          if(cmp>=0){
            rc = fsl_cx_err_set(d->f, FSL_RC_SYNTAX,
                                "T-cards are not in lexical order: "
                                "%c%s %s %c%s",
                                fsl_tag_prefix_char(tagPrev->type),
                                tagPrev->name,
                                cmp ? ">=" : "==",
                                fsl_tag_prefix_char(tagSelf->type),
                                tagSelf->name);
            goto bailout;
          }
        }
        break;
      }
      /*
         U ?<login>?

         Identify the user who created this control file by their
         login.  Only one U line is allowed.  Prohibited in clusters.
         If the user name is omitted, take that to be "anonymous".
      */
      case 'U':{
        if(d->U) SYNTAX("More than one U-card");
        TOKEN(1);
        if(token){
          /* rc = fsl_deck_U_set( d, (char const *)token, (fsl_int_t)tokLen ); */
          ++stealBuf;
          d->U = (char *)token;
        }else{
          rc = fsl_deck_U_set( d, "anonymous" );
        }
        break;
      }
      /*
             W <size>
        
         The next <size> bytes of the file contain the text of the wiki
         page.  There is always an extra \n before the start of the next
         record.
      */
      case 'W':{
        fsl_size_t wlen;
        if(d->W.used){
          SYNTAX("More than one W-card");
        }
        TOKEN(0);
        TOKEN_EXISTS("Missing size token for W-card");
        wlen = fsl_str_to_size((char const *)token);
        if((fsl_size_t)-1==wlen){
          SYNTAX("Wiki size token is invalid");
        }
        if( (&x.z[wlen+1]) > x.zEnd){
          SYNTAX("Not enough content after W-card");
        }
        //rc = fsl_buffer_append(&d->W, x.z, wlen);
        //if(rc) goto bailout;
        fsl_buffer_external(&d->W, x.z, wlen);
        ++stealBuf;
        x.z += wlen;
        if( '\n' != x.z[0] ){
          SYNTAX("W-card content not \\n terminated");
        }
        x.z[0] = 0;
        ++x.z;
        break;
      }
      /*
             Z <md5sum>

         MD5 checksum on this control file.  The checksum is over all
         lines (other than PGP-signature lines) prior to the current
         line.  This must be the last record.
        
         This card is required for all control file types except for
         Manifest. It is not required for manifest only for historical
         compatibility reasons.
      */
      case 'Z':{
        /* We validated the Z card first. We cannot compare against
           the original blob now because we've modified it.
        */
        goto end;
      }
      default:
        rc = fsl_cx_err_set(f, FSL_RC_SYNTAX,
                            "Unknown card '%c' in manifest",
                            cType);
        goto bailout;
    }/*switch(cType)*/
    if(rc) goto bailout;
  }/* for-each-card */

#if 1
  /* Remove these when we are done porting
     resp. we can avoid these unused-var warnings. */
  if(isRepeat){}
#endif

  end:
  assert(0==rc);
  if(cardCount>2 && FSL_SATYPE_ANY==d->type){
    /* See if we need to guess the type now.
       We need(?) at least two card to ensure that this is
       free of ambiguities. */
    d->type = fsl_deck_guess_type(lettersSeen);
    if(FSL_SATYPE_ANY!=d->type){
      assert(FSL_SATYPE_INVALID!=d->type);
#if 0
      MARKER(("Guessed manifest type with %d cards: %s\n",
                cardCount, fsl_satype_cstr(d->type)));
#endif
    }
  }
  /* Make sure all of the cards we put in it belong to that deck
     type. */
  if( !fsl_deck_check_type(d, cType) ){
    rc = d->f->error.code;
    goto bailout;
  }

  if(FSL_SATYPE_ANY==d->type){
    rc = fsl_cx_err_set(f, FSL_RC_ERROR,
                        "Internal error: could not determine type of "
                        "artifact we just (successfully!) parsed.");
    goto bailout;
  }else {
    /*
      Make sure we didn't pick up any cards which were picked up
      before d->type was guessed and are invalid for the post-guessed
      type.
    */
    int i = 0;
    for( ; i < 27; ++i ){
      if((lettersSeen[i]>0) && !fsl_card_is_legal(d->type, 'A'+i )){
        rc = fsl_cx_err_set(f, FSL_RC_SYNTAX,
                            "Determined during post-parse processing that "
                            "the parsed deck (type %s) contains an illegal "
                            "card type (%c).", fsl_satype_cstr(d->type),
                            'A'+i);
        goto bailout;
      }
    }
  }
  assert(FSL_SATYPE_CHECKIN==d->type ||
         FSL_SATYPE_CLUSTER==d->type ||
         FSL_SATYPE_CONTROL==d->type ||
         FSL_SATYPE_WIKI==d->type ||
         FSL_SATYPE_TICKET==d->type ||
         FSL_SATYPE_ATTACHMENT==d->type ||
         FSL_SATYPE_TECHNOTE==d->type ||
         FSL_SATYPE_FORUMPOST==d->type);
  assert(0==rc);

  /* Additional checks based on artifact type */
  switch( d->type ){
    case FSL_SATYPE_CONTROL: {
      if( nSelfTag ){
        SYNTAX("Self-referential T-card in control artifact");
      }
      break;
    }
    case FSL_SATYPE_TECHNOTE: {
      if( d->T.used!=nSelfTag ){
        SYNTAX("Non-self-referential T-card in technote");
      }else if( d->T.used!=nSimpleTag ){
        SYNTAX("T-card with '*' or '-' in technote");
      }
      break;
    }
    case FSL_SATYPE_FORUMPOST: {
      if( d->H && d->I ){
        SYNTAX("Cannot have I-card and H-card in a forum post");
      }else if( d->P.used>1 ){
        SYNTAX("Too many arguments to P-card");
      }
      break;
    }
    default: break;
  }

  assert(!d->content.mem);
  if(stealBuf>0){
    /* We stashed something which points to src->mem, so we need to
       steal that memory. */
    d->content = *src;
    *src = fsl_buffer_empty;
  }else{
    /* Clearing the source buffer if we don't take it over
       provides more consistency in the public API than _sometimes_
       requiring the client to clear it. */
    fsl_buffer_clear(src);
  }
  d->rid = rid;
  d->F.flags &= ~FSL_CARD_F_LIST_NEEDS_SORT/*we know all cards were read in order*/;
  return 0;

  bailout:
  if(stealBuf>0){
    d->content = *src;
    *src = fsl_buffer_empty;
  }
  assert(0 != rc);
  if(zMsg){
    fsl_error_set(err, rc, "%s", zMsg);
  }
  return rc;
#undef SEEN
#undef TOKEN_DATETIME
#undef SYNTAX
#undef TOKEN_CHECKHEX
#undef TOKEN_EXISTS
#undef TOKEN_UUID
#undef TOKEN_MD5
#undef TOKEN
#undef ERROR
}

int fsl_deck_parse(fsl_deck * const d, fsl_buffer * const src){
  return fsl_deck_parse2(d, src, 0);
}

int fsl_deck_load_rid( fsl_cx * const f, fsl_deck * const d,
                       fsl_id_t rid, fsl_satype_e type ){
  fsl_buffer buf = fsl_buffer_empty;
  int rc = 0;
  if(0==rid) rid = f->ckout.rid;
  if(rid<0){
    return fsl_cx_err_set(f, FSL_RC_RANGE,
                          "Invalid RID for fsl_deck_load_rid(): "
                          "%"FSL_ID_T_PFMT, rid);
  }
  fsl_deck_clean(d);
  d->f = f;
  if(fsl__cx_mcache_search2(f, rid, d, type, &rc) || rc){
    return rc;
  }
  rc = fsl_content_get(f, rid, &buf);
  if(rc) goto end;
#if 0
  MARKER(("fsl_content_get(%d) len=%d =\n%.*s\n",
          (int)rid, (int)buf.used, (int)buf.used, (char const*)buf.mem));
#endif
  fsl_deck_init(f, d, FSL_SATYPE_ANY);
#if 0
  /*
    If we set d->type=type, the parser can fail more
    quickly. However, that failure will bypass our more specific
    reporting of the problem (see below).  As the type mismatch case
    is expected to be fairly rare, we'll leave this out for now, but
    it might be worth considering as a small optimization later on.
  */
  d->type = type /* may help parsing fail more quickly if
                    it's not the type we want.*/;
#endif
  rc = fsl_deck_parse(d, &buf);
  if(!rc){
    if( type!=FSL_SATYPE_ANY && d->type!=type ){
      rc = fsl_cx_err_set(f, FSL_RC_TYPE,
                          "RID %"FSL_ID_T_PFMT" is of type %s, "
                          "but the caller requested type %s.",
                          rid,
                          fsl_satype_cstr(d->type),
                          fsl_satype_cstr(type));
    }else if(d->B.uuid ){
      rc = fsl__cx_update_seen_delta_deck(f);
    }
  }
  end:
  if(0==rc) d->rid = rid;
  fsl_buffer_clear(&buf);
  return rc;
}

int fsl_deck_load_sym( fsl_cx * const f, fsl_deck * const d,
                       char const * symbolicName, fsl_satype_e type ){
  if(!symbolicName || !d) return FSL_RC_MISUSE;
  else{
    fsl_id_t vid = 0;
    int rc = fsl_sym_to_rid(f, symbolicName, type, &vid);
    if(!rc){
      assert(vid>0);
      rc = fsl_deck_load_rid(f, d, vid, type);
    }
    return rc;
  }
}

  
static int fsl_deck_baseline_load( fsl_deck * d ){
  int rc = 0;
  fsl_deck bl = fsl_deck_empty;
  fsl_id_t rid;
  fsl_cx * f = d ? d->f : NULL;
  fsl_db * db = f ? fsl_needs_repo(f) : NULL;
  assert(d->f);
  assert(d);
  if(!d->f) return FSL_RC_MISUSE;
  else if(d->B.baseline || !d->B.uuid) return 0 /* nothing to do! */;
  else if(!db) return FSL_RC_NOT_A_REPO;
#if 0
  else if(d->rid<=0){
    return fsl_cx_err_set(f, FSL_RC_RANGE,
                          "fsl_deck_baseline_load(): "
                          "fsl_deck::rid is not set.");
  }
#endif
  rid = fsl_uuid_to_rid(f, d->B.uuid);
  if(rid<0){
    assert(f->error.code);
    return f->error.code;
  }
  else if(!rid){
    if(d->rid>0){
      fsl_db_exec(db, 
                  "INSERT OR IGNORE INTO orphan(rid, baseline) "
                  "VALUES(%"FSL_ID_T_PFMT",%"FSL_ID_T_PFMT")",
                  d->rid, rid);
    }
    rc = fsl_cx_err_set(f, FSL_RC_RANGE,
                        "Could not find/load baseline manifest [%s], "
                        "parent of manifest rid #%"FSL_ID_T_PFMT".",
                        d->B.uuid, d->rid);
  }else{
    rc = fsl_deck_load_rid(f, &bl, rid, FSL_SATYPE_CHECKIN);
    if(!rc){
      d->B.baseline = fsl_deck_malloc();
      if(!d->B.baseline){
        fsl_deck_clean(&bl);
        rc = FSL_RC_OOM;
      }else{
        void const * allocStampKludge = d->B.baseline->allocStamp;
        *d->B.baseline = bl /* Transfer ownership */;
        d->B.baseline->allocStamp = allocStampKludge /* But we need this intact
                                                        for deallocation to work */;
        assert(f==d->B.baseline->f);
      }
    }else{
      /* bl might be partially populated */
      fsl_deck_finalize(&bl);
    }
  }
  return rc;
}

int fsl_deck_baseline_fetch( fsl_deck * d ){
  return (d->B.baseline || !d->B.uuid)
    ? 0
    : fsl_deck_baseline_load(d);
}

int fsl_deck_F_rewind( fsl_deck * d ){
  int rc = 0;
  d->F.cursor = 0;
  assert(d->f);
  if(d->B.uuid){
    rc = fsl_deck_baseline_fetch(d);
    if(!rc){
      assert(d->B.baseline);
      d->B.baseline->F.cursor = 0;
    }
  }
  return rc;
}

int fsl_deck_F_next( fsl_deck * d, fsl_card_F const ** rv ){
  assert(d);
  assert(d->f);
  assert(rv);
#define FCARD(DECK,NDX) F_at(&(DECK)->F, NDX)
  *rv = NULL;
  if(!d->B.baseline){
    /* Manifest d is a baseline-manifest.  Just scan down the list
       of files. */
    if(d->B.uuid){
      return fsl_cx_err_set(d->f, FSL_RC_MISUSE,
                            "Deck has a B-card (%s) but no baseline "
                            "loaded. Load the baseline before calling "
                            "%s().",
                            d->B.uuid, __func__)
        /* We "could" just load the baseline from here. */;
    }
    if( d->F.cursor < (int32_t)d->F.used ){
      *rv = FCARD(d, d->F.cursor++);
      assert(*rv);
      assert((*rv)->uuid && "Baseline manifest has deleted F-card entry!");
    }
    return 0;
  }else{
    /* Manifest d is a delta-manifest.  Scan the baseline but amend the
       file list in the baseline with changes described by d.
    */
    fsl_deck * const pB = d->B.baseline;
    int cmp;
    while(1){
      if( pB->F.cursor >= (fsl_int_t)pB->F.used ){
        /* We have used all entries out of the baseline.  Return the next
           entry from the delta. */
        if( d->F.cursor < (fsl_int_t)d->F.used ) *rv = FCARD(d, d->F.cursor++);
        break;
      }else if( d->F.cursor >= (fsl_int_t)d->F.used ){
        /* We have used all entries from the delta.  Return the next
           entry from the baseline. */
        if( pB->F.cursor < (fsl_int_t)pB->F.used ) *rv = FCARD(pB, pB->F.cursor++);
        break;
      }else if( (cmp = fsl_strcmp(FCARD(pB,pB->F.cursor)->name,
                                  FCARD(d, d->F.cursor)->name)) < 0){
        /* The next baseline entry comes before the next delta entry.
           So return the baseline entry. */
        *rv = FCARD(pB, pB->F.cursor++);
        break;
      }else if( cmp>0 ){
        /* The next delta entry comes before the next baseline
           entry so return the delta entry */
        *rv = FCARD(d, d->F.cursor++);
        break;
      }else if( FCARD(d, d->F.cursor)->uuid ){
        /* The next delta entry is a replacement for the next baseline
           entry.  Skip the baseline entry and return the delta entry */
        pB->F.cursor++;
        *rv = FCARD(d, d->F.cursor++);
        break;
      }else{
        assert(0==cmp);
        /*
          The next delta entry is a delete of the next baseline entry.
        */
        /* Skip them both.  Repeat the loop to find the next
           non-delete entry. */
        pB->F.cursor++;
        d->F.cursor++;
        continue;
      }      
    }
    return 0;
  }
#undef FCARD
}

int fsl_deck_save( fsl_deck * const d, bool isPrivate ){
  int rc;
  fsl_cx * const f = d->f;
  fsl_db * const db = fsl_needs_repo(f);
  fsl_buffer * const buf = &d->f->cache.fileContent;
  fsl_id_t newRid = 0;
  bool const oldPrivate = f->cache.markPrivate;
  if(!f || !d ) return FSL_RC_MISUSE;
  else if(!db) return FSL_RC_NOT_A_REPO;
  if(d->rid>0){
#if 1
    return 0;
#else
    return fsl_cx_err_set(f, FSL_RC_ALREADY_EXISTS,
                          "Cannot re-save an existing deck, as that could "
                          "lead to inconsistent data.");
#endif
  }
  if(d->B.uuid && fsl_repo_forbids_delta_manifests(f)){
    return fsl_cx_err_set(f, FSL_RC_ACCESS,
                          "This deck is a delta manifest, but this "
                          "repository has disallowed those via the "
                          "forbid-delta-manifests config option.");
  }

  fsl_cx_err_reset(f);
  fsl_size_t const reserveSize =
    (1024 * 10)
    + (50 * d->T.used)
    + (50 * d->M.used)
    + (120 * d->F.used)
    + d->W.used;
  fsl_buffer_reuse(buf);
  rc = fsl_buffer_reserve(buf, reserveSize);
  if(0==rc) rc = fsl_deck_output(d, fsl_output_f_buffer, buf);
  if(rc){
    fsl_buffer_reuse(buf);
    return rc;
  }

  rc = fsl_db_transaction_begin(db);
  if(rc){
    fsl_buffer_reuse(buf);
    return rc;
  }
  if(0){
    MARKER(("Saving deck:\n%s\n", fsl_buffer_cstr(buf)));
  }

  /* Starting here, don't return, use (goto end) instead. */

  f->cache.markPrivate = isPrivate;
  {
    rc = fsl__content_put_ex(f, buf, NULL, 0,
                            0U, isPrivate, &newRid);
    if(rc) goto end;
    assert(newRid>0);
  }

  /* We need d->rid for crosslinking purposes, but will unset it on
     error because its value will no longer be in the db after
     rollback...
  */
  d->rid = newRid;

#if 0
  /* Something to consider: has a parent, deltify the parent. The
     branch operation does this, but it is not yet clear whether that
     is a general pattern for manifests.
  */
  if(d->P.used){
    fsl_id_t pid;
    assert(FSL_SATYPE_CHECKIN == d->type);
    pid = fsl_uuid_to_rid(f, (char const *)d->P.list[0]);
    if(pid>0){
      rc = fsl__content_deltify(f, pid, d->rid, 0);
      if(rc) goto end;
    }
  }
#endif

  if(FSL_SATYPE_WIKI==d->type){
    /* Analog to fossil's wiki.c:wiki_put(): */
    /*
      MISSING:
      fossil's wiki.c:wiki_put() handles the moderation bits.
    */
    if(d->P.used){
      fsl_id_t const pid = fsl_deck_P_get_id(d, 0);
      assert(pid>0);
      if(pid<0){
        assert(f->error.code);
        rc = f->error.code;
        goto end;
      }else if(!pid){
        if(!f->error.code){
          rc = fsl_cx_err_set(f, FSL_RC_NOT_FOUND,
                              "Did not find matching RID "
                              "for P-card[0] (%s).",
                              (char const *)d->P.list[0]);
        }
        goto end;
      }
      rc = fsl__content_deltify(f, pid, d->rid, 0);
      if(rc) goto end;
    }
    rc = fsl_db_exec_multi(db,
                           "INSERT OR IGNORE INTO unsent "
                           "VALUES(%"FSL_ID_T_PFMT");"
                           "INSERT OR IGNORE INTO unclustered "
                           "VALUES(%"FSL_ID_T_PFMT");",
                           d->rid, d->rid);
    if(rc){
      fsl_cx_uplift_db_error(f, db);
      goto end;
    }
  }

  rc = f->cache.isCrosslinking
    ? fsl__deck_crosslink(d)
    : fsl__deck_crosslink_one(d);

  end:
  f->cache.markPrivate = oldPrivate;
  if(!rc) rc = fsl_db_transaction_end(db, 0);
  else fsl_db_transaction_end(db, 1);
  if(rc){
    d->rid = 0 /* this blob.rid will be lost after rollback */;
    if(!f->error.code && db->error.code){
      rc = fsl_cx_uplift_db_error(f, db);
    }
  }
  fsl_buffer_reuse(buf);
  return rc;
}

int fsl__crosslink_end(fsl_cx * const f, int resultCode){
  int rc = 0;
  fsl_db * const db = fsl_cx_db_repo(f);
  fsl_stmt q = fsl_stmt_empty;
  fsl_stmt u = fsl_stmt_empty;
  int i;
  assert(f);
  assert(db);
  assert(f->cache.isCrosslinking && "Internal API misuse.");
  if(!f->cache.isCrosslinking){
    fsl__fatal(FSL_RC_MISUSE,
              "Internal API misuse: %s() called while "
              "f->cache.isCrosslinking is false.", __func__);
    return fsl_cx_err_set(f, FSL_RC_MISUSE,
                          "Crosslink is not running.");
  }
  f->cache.isCrosslinking = false;
  if(resultCode){
    assert(0!=fsl_cx_transaction_level(f)
           && "Expecting a transaction level from fsl__crosslink_begin()");
    fsl_db_transaction_end(db, true)
      /* pop transaction started from fsl__crosslink_begin().  We use
         fsl_db_transaction_end() instead of fsl_cx_transaction_end()
         so that any db-level error which is set during a failed
         rollback does not trump any pending f->error.code. */;
    return resultCode;
  }
  assert(db->beginCount > 0);

  /* Handle any reparenting via tags... */
  rc = fsl_cx_prepare(f, &q,
                     "SELECT rid, value FROM tagxref"
                      " WHERE tagid=%d AND tagtype=%d",
                      (int)FSL_TAGID_PARENT, (int)FSL_TAGTYPE_ADD);
  if(rc) goto end;
  while(FSL_RC_STEP_ROW==fsl_stmt_step(&q)){
    fsl_id_t const rid = fsl_stmt_g_id(&q, 0);
    const char *zTagVal = fsl_stmt_g_text(&q, 1, 0);
    rc = fsl_crosslink_reparent(f,rid, zTagVal);
    if(rc) break;
  }
  fsl_stmt_finalize(&q);
  if(rc) goto end;

  /* Process entries from pending_xlink temp table... */
  rc = fsl_cx_prepare(f, &q, "SELECT id FROM pending_xlink");
  while( 0==rc && FSL_RC_STEP_ROW==fsl_stmt_step(&q) ){
    const char *zId = fsl_stmt_g_text(&q, 0, NULL);
    char cType;
    if(!zId || !*zId) continue;
    cType = zId[0];
    ++zId;
    if('t'==cType){
      rc = fsl__ticket_rebuild(f, zId);
      continue;
    }else if('w'==cType){
      /* FSL-MISSING:
         backlink_wiki_refresh(zId) */
      continue;
    }
  }
  fsl_stmt_finalize(&q); 
  if(rc) goto end;
  rc = fsl_cx_exec(f, "DELETE FROM pending_xlink");
  if(rc) goto end;
  /* If multiple check-ins happen close together in time, adjust their
     times by a few milliseconds to make sure they appear in chronological
     order.
  */
  rc = fsl_cx_prepare(f, &q,
                      "UPDATE time_fudge SET m1=m2-:incr "
                      "WHERE m1>=m2 AND m1<m2+:window"
  );
  if(rc) goto end;
  fsl_stmt_bind_double_name(&q, ":incr", AGE_ADJUST_INCREMENT);
  fsl_stmt_bind_double_name(&q, ":window", AGE_FUDGE_WINDOW);
  rc = fsl_cx_prepare(f, &u,
                      "UPDATE time_fudge SET m2="
                      "(SELECT x.m1 FROM time_fudge AS x"
                      " WHERE x.mid=time_fudge.cid)");
  for(i=0; !rc && i<30; i++){ /* where does 30 come from? */
    rc = fsl_stmt_step(&q);
    if(FSL_RC_STEP_DONE==rc) rc=0;
    else break;
    fsl_stmt_reset(&q);
    if( fsl_db_changes_recent(db)==0 ) break;
    rc = fsl_stmt_step(&u);
    if(FSL_RC_STEP_DONE==rc) rc=0;
    else break;
    fsl_stmt_reset(&u);
  }
  fsl_stmt_finalize(&q);
  fsl_stmt_finalize(&u);
  if(!rc && fsl_db_exists(db,"SELECT 1 FROM time_fudge")){
    rc = fsl_cx_exec(f, "UPDATE event SET"
                     " mtime=(SELECT m1 FROM time_fudge WHERE mid=objid)"
                     " WHERE objid IN (SELECT mid FROM time_fudge)"
                     " AND (mtime=omtime OR omtime IS NULL)"
                     );
  }
  end:
  fsl_cx_exec(f, "DELETE FROM time_fudge");
  if(rc) fsl_cx_transaction_end(f, true);
  else rc = fsl_cx_transaction_end(f, false);
  return rc;
}

int fsl__crosslink_begin(fsl_cx * const f){
  int rc;
  assert(f);
  assert(0==f->cache.isCrosslinking);
  if(f->cache.isCrosslinking){
    return fsl_cx_err_set(f, FSL_RC_MISUSE,
                          "Crosslink is already running.");
  }
  rc = fsl_cx_transaction_begin(f);
  if(rc) return rc;
  rc = fsl_cx_exec_multi(f,
     "CREATE TEMP TABLE IF NOT EXISTS "
     "pending_xlink(id TEXT PRIMARY KEY)WITHOUT ROWID;"
     "CREATE TEMP TABLE IF NOT EXISTS time_fudge("
     "  mid INTEGER PRIMARY KEY,"    /* The rid of a manifest */
     "  m1 REAL,"                    /* The timestamp on mid */
     "  cid INTEGER,"                /* A child or mid */
     "  m2 REAL"                     /* Timestamp on the child */
     ");"
     "DELETE FROM pending_xlink; "
     "DELETE FROM time_fudge;");
  if(0==rc){
    f->cache.isCrosslinking = true;
  }else{
    fsl_cx_transaction_end(f, true);
  }
  return rc;
}

int fsl_deck_foreach(fsl_cx * const f, fsl_satype_e type,
                     fsl_deck_visitor_f visitor,
                     void * visitorState){
  int rc = 0;
  fsl_stmt q = fsl_stmt_empty;
  switch(type){
    case FSL_SATYPE_CHECKIN:
    case FSL_SATYPE_FORUMPOST:
    case FSL_SATYPE_CONTROL:
      rc = fsl_cx_prepare(f, &q,
                          "SELECT objid FROM event WHERE type=%Q",
                          fsl_satype_event_cstr(type));
      break;
    case FSL_SATYPE_WIKI:
      rc = fsl_cx_prepare(f, &q,
                          "SELECT x.rid AS mrid FROM tag t, tagxref x "
                          "WHERE x.tagid=t.tagid "
                          "AND t.tagname LIKE 'wiki-%%' "
                          "AND TYPEOF(x.value+0)='integer'"
                          // ^^^^ only 'wiki-%' tags which are wiki pages
                          "ORDER BY x.mtime DESC");
      break;
    case FSL_SATYPE_TICKET:
      rc = fsl_cx_prepare(f, &q,
                          "SELECT rid FROM tagxref WHERE tagid IN "
                          "(SELECT tagid FROM tag WHERE tagname LIKE "
                          "'tkt-%%' AND LENGTH(tagname)=44) "
                          "ORDER BY mtime");
      break;
    case FSL_SATYPE_TECHNOTE:
      rc = fsl_cx_prepare(f, &q,
                          "SELECT x.rid AS mrid FROM tag t, tagxref x "
                          "WHERE x.tagid=t.tagid "
                          "AND t.tagname LIKE 'event-%%' "
                          // ^^^^ only 'wiki-%' tags which are wiki pages
                          "ORDER BY x.mtime DESC");
      break;
#if 0
    case FSL_SATYPE_ANY:
      /* This case leads to confusion because ATTACHMENT records are
         recorded in the timeline as the underlying type of record to
         which the attachment is applied. e.g. an attachment on a wiki
         page is listed as a 'w' event. */
      rc = fsl_cx_prepare(f, &q,
                          "SELECT objid FROM event WHERE type IN"
                          "(%Q, %Q, %Q, %Q, %Q) /*%s()*/",
                          fsl_satype_event_cstr(FSL_SATYPE_CHECKIN),
                          fsl_satype_event_cstr(FSL_SATYPE_TICKET),
                          fsl_satype_event_cstr(FSL_SATYPE_FORUMPOST),
                          fsl_satype_event_cstr(FSL_SATYPE_WIKI),
                          fsl_satype_event_cstr(FSL_SATYPE_CONTROL),
                          __func__);
      break;
#endif
    default:
      return fsl_cx_err_set(f, FSL_RC_TYPE,
                            "Artifact type [%s] is not currently "
                            "supported by %s().",
                            fsl_satype_cstr(type),
                            __func__);
  }
  while(0==rc && (FSL_RC_STEP_ROW==fsl_stmt_step(&q))){
    fsl_id_t const rid = fsl_stmt_g_id(&q, 0);
    fsl_deck d = fsl_deck_empty;
    rc = fsl_deck_load_rid(f, &d, rid, type);
    if(0==rc){
      rc = visitor(f, &d, visitorState);
    }
    fsl_deck_finalize(&d);
    if(FSL_RC_BREAK==rc){
      rc = 0;
      break;
    }
  }
  fsl_stmt_finalize(&q);
  return rc;
}

#undef MARKER
#undef AGE_FUDGE_WINDOW
#undef AGE_ADJUST_INCREMENT
#undef F_at
/* end of file ./src/deck.c */
/* start of file ./src/delta.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/
/**************************************************************************
  This file houses Fossil's delta generation and application
  routines. This code is functionally independent of the rest of the
  library, relying only on fsl_malloc(), fsl_free(), and the integer
  typedefs defined by the configuration process. i.e. it can easily
  be pulled out and used in arbitrary projects.
*/
#include <memory.h>
#include <stdlib.h>

/**
   2021-03-10: The delta checksum self-test is a significant run-time
   sink when processing many deltas. Fossil does not enable this
   feature by default so we'll leave it off by default, too.
*/
#if !defined(FSL_OMIT_DELTA_CKSUM_TEST)
#  define FSL_OMIT_DELTA_CKSUM_TEST
#endif

/*
   Macros for turning debugging printfs on and off
*/
#if 0
# define DEBUG1(X) X
#else
# define DEBUG1(X)
#endif
#if 0
#define DEBUG2(X) X
/*
   For debugging:
   Print 16 characters of text from zBuf
*/
static const char *print16(const char *z){
  int i;
  static char zBuf[20];
  for(i=0; i<16; i++){
    if( z[i]>=0x20 && z[i]<=0x7e ){
      zBuf[i] = z[i];
    }else{
      zBuf[i] = '.';
    }
  }
  zBuf[i] = 0;
  return zBuf;
}
#else
# define DEBUG2(X)
#endif

/*
   The width of a hash window in bytes.  The algorithm only works if this
   is a power of 2.
*/
#define NHASH 16

/*
   The current state of the rolling hash.
  
   z[] holds the values that have been hashed.  z[] is a circular buffer.
   z[i] is the first entry and z[(i+NHASH-1)%NHASH] is the last entry of 
   the window.
  
   Hash.a is the sum of all elements of hash.z[].  Hash.b is a weighted
   sum.  Hash.b is z[i]*NHASH + z[i+1]*(NHASH-1) + ... + z[i+NHASH-1]*1.
   (Each index for z[] should be module NHASH, of course.  The %NHASH operator
   is omitted in the prior expression for brevity.)
*/
typedef struct fsl_delta_hash fsl_delta_hash;
struct fsl_delta_hash {
  uint16_t a, b;         /* Hash values */
  uint16_t i;            /* Start of the hash window */
  unsigned char z[NHASH];    /* The values that have been hashed */
};

/*
   Initialize the rolling hash using the first NHASH characters of z[]
*/
static void fsl_delta_hash_init(fsl_delta_hash *pHash,
                                unsigned char const *z){
  uint16_t a, b, i;
  a = b = 0;
  for(i=0; i<NHASH; i++){
    a += z[i];
    b += a;
  }
  memcpy(pHash->z, z, NHASH);
  pHash->a = a & 0xffff;
  pHash->b = b & 0xffff;
  pHash->i = 0;
}

/*
   Advance the rolling hash by a single character "c"
*/
static void fsl_delta_hash_next(fsl_delta_hash *pHash, int c){
  uint16_t old = pHash->z[pHash->i];
  pHash->z[pHash->i] = c;
  pHash->i = (pHash->i+1)&(NHASH-1);
  pHash->a = pHash->a - old + c;
  pHash->b = pHash->b - NHASH*old + pHash->a;
}

/*
   Return a 32-bit hash value
*/
static uint32_t fsl_delta_hash_32bit(fsl_delta_hash *pHash){
  return (pHash->a & 0xffff) | (((uint32_t)(pHash->b & 0xffff))<<16);
}

/**
   Compute a hash on NHASH bytes.

   This routine is intended to be equivalent to:
   fsl_delta_hash h;
   fsl_delta_hash_init(&h, zInput);
   return fsl_delta_hash_32bit(&h);
*/
static uint32_t fsl_delta_hash_once(unsigned const char *z){
  uint16_t a = 0, b = 0, i = 0;
  for(i=0; i<NHASH; ++i){
    a += z[i];
    b += a;
  }
  return a | (((uint32_t)b)<<16);
}

/*
   Write an base-64 integer into the given buffer. Return its length.
*/
static unsigned int fsl_delta_int_put(uint32_t v, unsigned char **pz){
  static const char zDigits[] = 
    "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz~";
  /*  123456789 123456789 123456789 123456789 123456789 123456789 123 */
  int i, j;
  unsigned char zBuf[20];
  fsl_size_t rc = 0;
  if( v==0 ){
    *(*pz)++ = '0';
    rc = 1;
  }else{
    for(i=0; v>0; ++rc, ++i, v>>=6){
      zBuf[i] = zDigits[v&0x3f];
    }
    zBuf[i]=0;
    for(j=i-1; j>=0; j--){
      *(*pz)++ = zBuf[j];
    }
  }
  return rc;
}

/*
   Read bytes from *pz and convert them into a positive integer.  When
   finished, leave *pz pointing to the first character past the end of
   the integer.  The *pLen parameter holds the length of the string
   in *pz and is decremented once for each character in the integer.
*/
static fsl_size_t fsl_delta_int_get(unsigned char const **pz, fsl_int_t *pLen){
  static const signed char zValue[] = {
    -1, -1, -1, -1, -1, -1, -1, -1,   -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1,   -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1,   -1, -1, -1, -1, -1, -1, -1, -1,
     0,  1,  2,  3,  4,  5,  6,  7,    8,  9, -1, -1, -1, -1, -1, -1,
    -1, 10, 11, 12, 13, 14, 15, 16,   17, 18, 19, 20, 21, 22, 23, 24,
    25, 26, 27, 28, 29, 30, 31, 32,   33, 34, 35, -1, -1, -1, -1, 36,
    -1, 37, 38, 39, 40, 41, 42, 43,   44, 45, 46, 47, 48, 49, 50, 51,
    52, 53, 54, 55, 56, 57, 58, 59,   60, 61, 62, -1, -1, -1, 63, -1,
  };
  fsl_size_t v = 0;
  fsl_int_t c;
  unsigned char const *z = (unsigned char const*)*pz;
  unsigned char const *zStart = z;
  while( (c = zValue[0x7f&*(z++)])>=0 ){
     v = (v<<6) + c;
  }
  z--;
  *pLen -= z - zStart;
  *pz = z;
  return v;
}

/*
   Return the number digits in the base-64 representation of a positive integer
*/
static int fsl_delta_digit_count(fsl_int_t v){
  unsigned int x;
  int i;
  for(i=1, x=64; v>=(fsl_int_t)x; i++, x <<= 6){}
  return i;
}

/*
   Compute a 32-bit checksum on the N-byte buffer.  Return the result.
*/
static unsigned int fsl_delta_checksum(void const *zIn, fsl_size_t N){
  const unsigned char *z = (const unsigned char *)zIn;
  unsigned sum0 = 0;
  unsigned sum1 = 0;
  unsigned sum2 = 0;
  unsigned sum3 = 0;
  while(N >= 16){
    sum0 += ((unsigned)z[0] + z[4] + z[8] + z[12]);
    sum1 += ((unsigned)z[1] + z[5] + z[9] + z[13]);
    sum2 += ((unsigned)z[2] + z[6] + z[10]+ z[14]);
    sum3 += ((unsigned)z[3] + z[7] + z[11]+ z[15]);
    z += 16;
    N -= 16;
  }
  while(N >= 4){
    sum0 += z[0];
    sum1 += z[1];
    sum2 += z[2];
    sum3 += z[3];
    z += 4;
    N -= 4;
  }
  sum3 += (sum2 << 8) + (sum1 << 16) + (sum0 << 24);
  switch(N){
    case 3:   sum3 += (z[2] << 8);
      __attribute__ ((fallthrough));
    case 2:   sum3 += (z[1] << 16);
      __attribute__ ((fallthrough));
    case 1:   sum3 += (z[0] << 24);
      __attribute__ ((fallthrough));
    default:  ;
  }
  return sum3;
}

int fsl_delta_create2( unsigned char const *zSrc, fsl_size_t lenSrc,
                       unsigned char const *zOut, fsl_size_t lenOut,
                       fsl_output_f out, void * outState){
  enum { IntegerBufSize = 50 /* buffer size for integer conversions. */};
  unsigned int i, base;
  unsigned int nHash;          /* Number of hash table entries */
  unsigned int *landmark;      /* Primary hash table */
  unsigned int *collide = NULL;  /* Collision chain */
  int lastRead = -1;           /* Last byte of zSrc read by a COPY command */
  int rc;                      /* generic return code checker. */
  unsigned int olen = 0;       /* current output length. */
  fsl_delta_hash h;
  unsigned char theBuf[IntegerBufSize] = {0,};
  unsigned char * intBuf = theBuf;
  if(!zSrc || !zOut || !out) return FSL_RC_MISUSE;
  /* Add the target file size to the beginning of the delta
  */
#ifdef OUT
#undef OUT
#endif
#define OUT(BLOB,LEN) rc=out(outState, BLOB, LEN); if(0 != rc) {fsl_free(collide); return rc;} else (void)0
#define OUTCH(CHAR) OUT(CHAR,1)
#define PINT(I) intBuf = theBuf; olen=fsl_delta_int_put(I, &intBuf); OUT(theBuf,olen)
  PINT(lenOut);
  OUTCH("\n");

  /* If the source file is very small, it means that we have no
     chance of ever doing a copy command.  Just output a single
     literal segment for the entire target and exit.
  */
  if( lenSrc<=NHASH ){
    PINT(lenOut);
    OUTCH(":");
    OUT(zOut,lenOut);
    PINT((fsl_delta_checksum(zOut, lenOut)));
    OUTCH(";");
    return 0;
  }

  /* Compute the hash table used to locate matching sections in the
     source file.
  */
  nHash = lenSrc/NHASH;
  collide = (unsigned int *)malloc( nHash*2*sizeof(int) );
  if(!collide){
    return FSL_RC_OOM;
  }
  landmark = &collide[nHash];
  memset(landmark, -1, nHash*sizeof(int));
  memset(collide, -1, nHash*sizeof(int));
  for(i=0; i<lenSrc-NHASH; i+=NHASH){
    uint32_t const hv = fsl_delta_hash_once(&zSrc[i]) % nHash;
    collide[i/NHASH] = landmark[hv];
    landmark[hv] = i/NHASH;
  }

  /* Begin scanning the target file and generating copy commands and
     literal sections of the delta.
  */
  base = 0;    /* We have already generated everything before zOut[base] */
  while( base+NHASH<lenOut ){
    fsl_int_t iSrc;
    int iBlock
      /* WEIRD: if i change this from int to fsl_int_t
         we end up in an infinite loop somewhere. int
         and short both work*/;
    fsl_int_t bestCnt, bestOfst=0, bestLitsz=0;
    fsl_delta_hash_init(&h, &zOut[base]);
    i = 0;     /* Trying to match a landmark against zOut[base+i] */
    bestCnt = 0;
    while( 1 ){
      uint32_t hv;
      int limit = 250;

      hv = fsl_delta_hash_32bit(&h) % nHash;
      DEBUG2( printf("LOOKING: %4d [%s]\n", base+i, print16(&zOut[base+i])); )
      iBlock = (int)landmark[hv];
      while( iBlock>=0 && (limit--)>0 ){
        /*
           The hash window has identified a potential match against 
           landmark block iBlock.  But we need to investigate further.
           
           Look for a region in zOut that matches zSrc. Anchor the search
           at zSrc[iSrc] and zOut[base+i].  Do not include anything prior to
           zOut[base] or after zOut[outLen] nor anything after zSrc[srcLen].
          
           Set cnt equal to the length of the match and set ofst so that
           zSrc[ofst] is the first element of the match.  litsz is the number
           of characters between zOut[base] and the beginning of the match.
           sz will be the overhead (in bytes) needed to encode the copy
           command.  Only generate copy command if the overhead of the
           copy command is less than the amount of literal text to be copied.
        */
        fsl_int_t cnt, ofst, litsz;
        fsl_int_t j, k, x, y;
        fsl_int_t sz;
        fsl_int_t limitX;

        /* Beginning at iSrc, match forwards as far as we can.  j counts
           the number of characters that match */
        iSrc = iBlock*NHASH;
        y = base + i;
        limitX = ( lenSrc-iSrc <= lenOut-y ) ? lenSrc : iSrc + lenOut - y;
        for(x=iSrc; x<limitX; ++x, ++y){
          if( zSrc[x]!=zOut[y] ) break;
        }
        j = x - iSrc - 1;

        /* Beginning at iSrc-1, match backwards as far as we can.  k counts
           the number of characters that match */
        for(k=1; k<iSrc && k<=i; ++k){
          if( zSrc[iSrc-k]!=zOut[base+i-k] ) break;
        }
        --k;

        /* Compute the offset and size of the matching region */
        ofst = iSrc-k;
        cnt = j+k+1;
        litsz = i-k;  /* Number of bytes of literal text before the copy */
        DEBUG2( printf("MATCH %d bytes at %d: [%s] litsz=%d\n",
                        cnt, ofst, print16(&zSrc[ofst]), litsz); )
        /* sz will hold the number of bytes needed to encode the "insert"
           command and the copy command, not counting the "insert" text */
        sz = fsl_delta_digit_count(i-k)
          +fsl_delta_digit_count(cnt)
          +fsl_delta_digit_count(ofst)
          +3;
        if( cnt>=sz && cnt>bestCnt ){
          /* Remember this match only if it is the best so far and it
             does not increase the file size */
          bestCnt = cnt;
          bestOfst = iSrc-k;
          bestLitsz = litsz;
          DEBUG2( printf("... BEST SO FAR\n"); )
        }

        /* Check the next matching block */
        iBlock = collide[iBlock];
      }

      /* We have a copy command that does not cause the delta to be larger
         than a literal insert.  So add the copy command to the delta.
      */
      if( bestCnt>0 ){
        if( bestLitsz>0 ){
          /* Add an insert command before the copy */
          PINT(bestLitsz);
          OUTCH(":");
          OUT(zOut+base, bestLitsz);
          base += bestLitsz;
          DEBUG2( printf("insert %d\n", bestLitsz); )
        }
        base += bestCnt;
        PINT(bestCnt);
        OUTCH("@");
        PINT(bestOfst);
        DEBUG2( printf("copy %d bytes from %d\n", bestCnt, bestOfst); )
        OUTCH(",");
        if( bestOfst + bestCnt -1 > lastRead ){
          lastRead = bestOfst + bestCnt - 1;
          DEBUG2( printf("lastRead becomes %d\n", lastRead); )
        }
        bestCnt = 0;
        break;
      }

      /* If we reach this point, it means no match is found so far */
      if( base+i+NHASH>=lenOut ){
        /* We have reached the end of the input and have not found any
           matches.  Do an "insert" for everything that does not match */
        PINT(lenOut-base);
        OUTCH(":");
        OUT(zOut+base, lenOut-base);
        base = lenOut;
        break;
      }

      /* Advance the hash by one character.  Keep looking for a match */
      fsl_delta_hash_next(&h, zOut[base+i+NHASH]);
      i++;
    }
  }
  fsl_free(collide);
  /* Output a final "insert" record to get all the text at the end of
     the file that does not match anything in the source file.
  */
  if( base<lenOut ){
    PINT(lenOut-base);
    OUTCH(":");
    OUT(zOut+base, lenOut-base);
  }
  /* Output the final checksum record. */
  PINT(fsl_delta_checksum(zOut, lenOut));
  OUTCH(";");
  return 0;
#undef PINT
#undef OUT
#undef OUTCH
}

struct DeltaOutputString {
  unsigned char * mem;
  unsigned int cursor;
};
typedef struct DeltaOutputString DeltaOutputString;

/** fsl_output_f() impl which requires state to be a
    (DeltaOutputString*). Copies the first n bytes of src to
    state->mem, increments state->cursor by n, and returns 0.
 */
static int fsl_output_f_ostring( void * state, void const * src,
                                 fsl_size_t n ){
  DeltaOutputString * os = (DeltaOutputString*)state;
  memcpy( os->mem + os->cursor, src, n );
  os->cursor += n;
  return 0;
}

int fsl_delta_create( unsigned char const *zSrc, fsl_size_t lenSrc,
                      unsigned char const *zOut, fsl_size_t lenOut,
                      unsigned char *zDelta, fsl_size_t * deltaSize){
  int rc;
  DeltaOutputString os;
  os.mem = (unsigned char *)zDelta;
  os.cursor = 0;
  rc = fsl_delta_create2( zSrc, lenSrc, zOut, lenOut,
                          fsl_output_f_ostring, &os );
  if(!rc){
    os.mem[os.cursor] = 0;
    *deltaSize = os.cursor;
  }
  return rc;
}

/*
   Calculates the size (in bytes) of the output from applying a
   delta. On success 0 is returned and *deltaSize will be updated with
   the amount of memory required for applying the delta.
  
   This routine is provided so that an procedure that is able
   to call fsl_delta_apply() can learn how much space is required
   for the output and hence allocate nor more space that is really
   needed.
*/
int fsl_delta_applied_size(unsigned char const *zDelta, fsl_size_t lenDelta_,
                           fsl_size_t * deltaSize){
  if(!zDelta || (lenDelta_<2) || !deltaSize) return FSL_RC_MISUSE;
  else{
    fsl_size_t size;
    fsl_int_t lenDelta = (fsl_int_t)lenDelta_;
    size = fsl_delta_int_get(&zDelta, &lenDelta);
    if( *zDelta!='\n' ){
      /* ERROR: size integer not terminated by "\n" */
      return FSL_RC_DELTA_INVALID_TERMINATOR;
    }
    *deltaSize = size;
    return 0;
  }
}


int fsl_delta_apply2(
  unsigned char const *zSrc,      /* The source or pattern file */
  fsl_size_t lenSrc_,            /* Length of the source file */
  unsigned char const *zDelta,    /* Delta to apply to the pattern */
  fsl_size_t lenDelta_,          /* Length of the delta */
  unsigned char *zOut,             /* Write the output into this preallocated buffer */
  fsl_error * pErr
){
  fsl_size_t limit;
  fsl_size_t total = 0;
#if !defined(FSL_OMIT_DELTA_CKSUM_TEST)
  unsigned char *zOrigOut = zOut;
#endif
  /* lenSrc/lenDelta are cast to ints to avoid any potential side-effects
     caused by changing the function signature from signed to unsigned
     int types when porting from v1.
  */
  fsl_int_t lenSrc = (fsl_int_t)lenSrc_;
  fsl_int_t lenDelta = (fsl_int_t)lenDelta_;
  if(!zSrc || !zDelta || !zOut) return FSL_RC_MISUSE;
  else if(lenSrc<0 || lenDelta<0) return FSL_RC_RANGE;
  limit = fsl_delta_int_get(&zDelta, &lenDelta);
  if( *zDelta!='\n' ){
    if(pErr){
      fsl_error_set(pErr,
                    FSL_RC_DELTA_INVALID_TERMINATOR,
                    "Delta: size integer not terminated by \\n");
    }
    return FSL_RC_DELTA_INVALID_TERMINATOR;
  }
  zDelta++; lenDelta--;
  while( *zDelta && lenDelta>0 ){
    fsl_int_t cnt, ofst;
    cnt = fsl_delta_int_get(&zDelta, &lenDelta);
    switch( zDelta[0] ){
      case '@': {
        zDelta++; lenDelta--;
        ofst = fsl_delta_int_get(&zDelta, &lenDelta);
        if( lenDelta>0 && zDelta[0]!=',' ){
          /* ERROR: copy command not terminated by ',' */
          if(pErr){
            fsl_error_set(pErr,
                          FSL_RC_DELTA_INVALID_TERMINATOR,
                          "Delta: copy command not terminated by ','");
          }
          return FSL_RC_DELTA_INVALID_TERMINATOR;
        }
        zDelta++; lenDelta--;
        DEBUG1( printf("COPY %d from %d\n", cnt, ofst); )
        total += cnt;
        if( total>limit ){
          if(pErr){
            fsl_error_set(pErr, FSL_RC_RANGE,
                          "Delta: copy exceeds output file size");
          }
          return FSL_RC_RANGE;
        }
        if( ofst+cnt > lenSrc ){
          if(pErr){
            fsl_error_set(pErr, FSL_RC_RANGE,
                          "Delta: copy extends past end of input");
          }
          return FSL_RC_RANGE;
        }
        memcpy(zOut, &zSrc[ofst], cnt);
        zOut += cnt;
        break;
      }
      case ':': {
        zDelta++; lenDelta--;
        total += cnt;
        if( total>limit ){
          if(pErr){
            fsl_error_set(pErr, FSL_RC_RANGE,
                          "Delta: insert command gives an output "
                          "larger than predicted");
          }
          return FSL_RC_RANGE;
        }
        DEBUG1( printf("INSERT %d\n", cnt); )
        if( cnt>lenDelta ){
          if(pErr){
            fsl_error_set(pErr, FSL_RC_RANGE,
                          "Delta: insert count exceeds size of delta");
          }
          return FSL_RC_RANGE;
        }
        memcpy(zOut, zDelta, cnt);
        zOut += cnt;
        zDelta += cnt;
        lenDelta -= cnt;
        break;
      }
      case ';': {
        zDelta++; lenDelta--;
        zOut[0] = 0;
#if !defined(FSL_OMIT_DELTA_CKSUM_TEST)
        if( cnt!=fsl_delta_checksum(zOrigOut, total) ){
          if(pErr){
            fsl_error_set(pErr, FSL_RC_CHECKSUM_MISMATCH,
                          "Delta: bad checksum");
          }
          return FSL_RC_CHECKSUM_MISMATCH;
        }
#endif
        if( total!=limit ){
          if(pErr){
            fsl_error_set(pErr, FSL_RC_SIZE_MISMATCH,
                          "Delta: generated size does not match "
                          "predicted size");
          }
          return FSL_RC_SIZE_MISMATCH;
        }
        return 0;
      }
      default: {
        if(pErr){
          fsl_error_set(pErr, FSL_RC_DELTA_INVALID_OPERATOR,
                        "Delta: unknown delta operator");
        }
        return FSL_RC_DELTA_INVALID_OPERATOR;
      }
    }
  }
  /* ERROR: unterminated delta */
  if(pErr){
    fsl_error_set(pErr, FSL_RC_DELTA_INVALID_TERMINATOR,
                  "Delta: unterminated delta");
  }
  return FSL_RC_DELTA_INVALID_TERMINATOR;
}
int fsl_delta_apply(
  unsigned char const *zSrc,      /* The source or pattern file */
  fsl_size_t lenSrc_,            /* Length of the source file */
  unsigned char const *zDelta,    /* Delta to apply to the pattern */
  fsl_size_t lenDelta_,          /* Length of the delta */
  unsigned char *zOut             /* Write the output into this preallocated buffer */
){
  return fsl_delta_apply2(zSrc, lenSrc_, zDelta, lenDelta_, zOut, NULL);
}

#undef NHASH
#undef DEBUG1
#undef DEBUG2
/* end of file ./src/delta.c */
/* start of file ./src/dibu.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2022 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2013-2022 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/

/************************************************************************
  This file houses the "dibu" (Diff Builder) routines.
*/
#include <assert.h>
#include <memory.h>
#include <stdlib.h>
#include <string.h> /* for memmove()/strlen() */


#include <stdio.h>
#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)


const fsl_dibu_opt fsl_dibu_opt_empty = fsl_dibu_opt_empty_m;
const fsl_dibu fsl_dibu_empty = fsl_dibu_empty_m;


fsl_dibu * fsl_dibu_alloc(fsl_size_t extra){
  fsl_dibu * rc =
    (fsl_dibu*)fsl_malloc(sizeof(fsl_dibu) + extra);
  if(rc){
    *rc = fsl_dibu_empty;
    if(extra){
      rc->pimpl = ((unsigned char *)rc)+sizeof(fsl_dibu);
    }
  }
  return rc;
}

static int fdb__out(fsl_dibu *const b,
                    char const *z, fsl_int_t n){
  if(n<0) n = (fsl_int_t)fsl_strlen(z);
  return b->opt->out(b->opt->outState, z, (fsl_size_t)n);
}
static int fdb__outf(fsl_dibu * const b,
                     char const *fmt, ...){
  int rc = 0;
  va_list va;
  assert(b->opt->out);
  va_start(va,fmt);
  rc = fsl_appendfv(b->opt->out, b->opt->outState, fmt, va);
  va_end(va);
  return rc;
}


/**
   Column indexes for DiffCounter::cols.
*/
enum DiffCounterCols {
DICO_NUM1 = 0, DICO_TEXT1,
DICO_MOD,
DICO_NUM2, DICO_TEXT2,
DICO_count
};
/**
   Internal state for the text-mode split diff builder. Used for
   calculating column widths in the builder's first pass so that
   the second pass can output everything in a uniform width.
*/
struct DiffCounter {
  /**
     Largest column width we've yet seen. These are only updated for
     DICO_TEXT1 and DICO_TEXT2. The others currently have fixed widths.

     FIXME: these are in bytes, not text columns. The current code may
     truncate multibyte characters.
  */
  uint32_t maxWidths[DICO_count];
  /**
     Max line numbers seen for the LHS/RHS input files. This is likely
     much higher than the number of diff lines.

     This can be used, e.g., to size and allocate a curses PAD in the
     second pass of the start() method.
  */
  uint32_t lineCount[2];
  /**
     The actual number of lines needed for rendering the file.
  */
  uint32_t displayLines;
};
typedef struct DiffCounter DiffCounter;
static const DiffCounter DiffCounter_empty = {{1,25,3,1,25},{0,0},0};
#define DICOSTATE(VNAME) DiffCounter * const VNAME = (DiffCounter *)b->pimpl

static int maxColWidth(fsl_dibu const * const b,
                       DiffCounter const * const sst,
                       int mwIndex){
  static const short minColWidth =
    10/*b->opt.columnWidth values smaller than this are treated as
        this value*/;
  switch(mwIndex){
    case DICO_NUM1:
    case DICO_NUM2:
    case DICO_MOD: return sst->maxWidths[mwIndex];
    case DICO_TEXT1: case DICO_TEXT2: break;
    default:
      assert(!"internal API misuse: invalid column index.");
      break;
  }
  int const y =
    (b->opt->columnWidth>0
     && b->opt->columnWidth<=sst->maxWidths[mwIndex])
    ? (int)b->opt->columnWidth
    : (int)sst->maxWidths[mwIndex];
  return minColWidth > y ? minColWidth : y;
}

static void fdb__dico_update_maxlen(DiffCounter * const sst,
                                    int col,
                                    char const * const z,
                                    uint32_t n){
  if(sst->maxWidths[col]<n){
#if 0
    sst->maxWidths[col] = n;
#else
    n = (uint32_t)fsl_strlen_utf8(z, (fsl_int_t)n);
    if(sst->maxWidths[col]<n) sst->maxWidths[col] = n;
#endif
  }
}

static int fdb__debug_start(fsl_dibu * const b){
  int rc = fdb__outf(b, "DEBUG builder starting pass #%d.\n",
                     b->passNumber);
  if(1==b->passNumber){
    DICOSTATE(sst);
    *sst = DiffCounter_empty;
    if(b->opt->nameLHS) ++sst->displayLines;
    if(b->opt->nameRHS) ++sst->displayLines;
    if(b->opt->hashLHS) ++sst->displayLines;
    if(b->opt->hashRHS) ++sst->displayLines;
    ++b->fileCount;
    return rc;
  }
  if(0==rc && b->opt->nameLHS){
    rc = fdb__outf(b,"LHS: %s\n", b->opt->nameLHS);
  }
  if(0==rc && b->opt->nameRHS){
    rc = fdb__outf(b,"RHS: %s\n", b->opt->nameRHS);
  }
  if(0==rc && b->opt->hashLHS){
    rc = fdb__outf(b,"LHS hash: %s\n", b->opt->hashLHS);
  }
  if(0==rc && b->opt->hashRHS){
    rc = fdb__outf(b,"RHS hash: %s\n", b->opt->hashRHS);
  }
  return rc;
}


static int fdb__debug_chunkHeader(fsl_dibu* const b,
                                  uint32_t lnnoLHS, uint32_t linesLHS,
                                  uint32_t lnnoRHS, uint32_t linesRHS ){
#if 1
  if(1==b->passNumber){
    DICOSTATE(sst);
    ++sst->displayLines;
    return 0;
  }
  if(b->lnLHS+1==lnnoLHS && b->lnRHS+1==lnnoRHS){
    fdb__outf(b, "<<<Unfortunate chunk separator."
              "Ticket 746ebbe86c20b5c0f96cdadd19abd8284770de16.>>>\n");
  }
  //fdb__outf(b, "lnLHS=%d, lnRHS=%d\n", (int)b->lnLHS, (int)b->lnRHS);
  return fdb__outf(b, "@@ -%" PRIu32 ",%" PRIu32
                " +%" PRIu32 ",%" PRIu32 " @@\n",
                lnnoLHS, linesLHS, lnnoRHS, linesRHS);
#else
  return 0;
#endif
}

static int fdb__debug_skip(fsl_dibu * const b, uint32_t n){
  if(1==b->passNumber){
    DICOSTATE(sst);
    b->lnLHS += n;
    b->lnRHS += n;
    ++sst->displayLines;
    return 0;
  }
  const int rc = fdb__outf(b, "SKIP %u (%u..%u left and %u..%u right)\n",
                           n, b->lnLHS+1, b->lnLHS+n, b->lnRHS+1, b->lnRHS+n);
  b->lnLHS += n;
  b->lnRHS += n;
  return rc;
}
static int fdb__debug_common(fsl_dibu * const b, fsl_dline const * pLine){
  DICOSTATE(sst);
  ++b->lnLHS;
  ++b->lnRHS;
  if(1==b->passNumber){
    ++sst->displayLines;
    fdb__dico_update_maxlen(sst, DICO_TEXT1, pLine->z, pLine->n);
    fdb__dico_update_maxlen(sst, DICO_TEXT2, pLine->z, pLine->n);
    return 0;
  }
  return fdb__outf(b, "COMMON  %8u %8u %.*s\n",
                   b->lnLHS, b->lnRHS, (int)pLine->n, pLine->z);
}
static int fdb__debug_insertion(fsl_dibu * const b, fsl_dline const * pLine){
  DICOSTATE(sst);
  ++b->lnRHS;
  if(1==b->passNumber){
    ++sst->displayLines;
    fdb__dico_update_maxlen(sst, DICO_TEXT1, pLine->z, pLine->n);
    return 0;
  }
  return fdb__outf(b, "INSERT           %8u %.*s\n",
                   b->lnRHS, (int)pLine->n, pLine->z);
}
static int fdb__debug_deletion(fsl_dibu * const b, fsl_dline const * pLine){
  DICOSTATE(sst);
  ++b->lnLHS;
  if(1==b->passNumber){
    ++sst->displayLines;
    fdb__dico_update_maxlen(sst, DICO_TEXT2, pLine->z, pLine->n);
    return 0;
  }
  return fdb__outf(b, "DELETE  %8u          %.*s\n",
                   b->lnLHS, (int)pLine->n, pLine->z);
}
static int fdb__debug_replacement(fsl_dibu * const b,
                                  fsl_dline const * lineLhs,
                                  fsl_dline const * lineRhs) {
#if 0
  int rc = b->deletion(b, lineLhs);
  if(0==rc) rc = b->insertion(b, lineRhs);
  return rc;
#else    
  DICOSTATE(sst);
  ++b->lnLHS;
  ++b->lnRHS;
  if(1==b->passNumber){
    ++sst->displayLines;
    fdb__dico_update_maxlen(sst, DICO_TEXT1, lineLhs->z, lineLhs->n);
    fdb__dico_update_maxlen(sst, DICO_TEXT2, lineRhs->z, lineRhs->n);
    return 0;
  }
  int rc = fdb__outf(b, "REPLACE %8u          %.*s\n",
                     b->lnLHS, (int)lineLhs->n, lineLhs->z);
  if(!rc){
    rc = fdb__outf(b, "            %8u %.*s\n",
                   b->lnRHS, (int)lineRhs->n, lineRhs->z);
  }
  return rc;
#endif
}
                 
static int fdb__debug_edit(fsl_dibu * const b,
                           fsl_dline const * lineLHS,
                           fsl_dline const * lineRHS){
#if 0
  int rc = b->deletion(b, lineLHS);
  if(0==rc) rc = b->insertion(b, lineRHS);
  return rc;
#else    
  int rc = 0;
  DICOSTATE(sst);
  ++b->lnLHS;
  ++b->lnRHS;
  if(1==b->passNumber){
    sst->displayLines += 4
      /* this is actually 3 or 4, but we don't know that from here */;
    fdb__dico_update_maxlen(sst, DICO_TEXT1, lineLHS->z, lineLHS->n);
    fdb__dico_update_maxlen(sst, DICO_TEXT2, lineRHS->z, lineRHS->n);
    return 0;
  }
  int i, j;
  int x;
  fsl_dline_change chng = fsl_dline_change_empty;
#define RC if(rc) goto end
  rc = fdb__outf(b, "EDIT    %8u          %.*s\n",
                 b->lnLHS, (int)lineLHS->n, lineLHS->z);
  RC;
  fsl_dline_change_spans(lineLHS, lineRHS, &chng);
  for(i=x=0; i<chng.n; i++){
    int ofst = chng.a[i].iStart1;
    int len = chng.a[i].iLen1;
    if( len ){
      char c = '0' + i;
      if( x==0 ){
        rc = fdb__outf(b, "%*s", 26, "");
        RC;
      }
      while( ofst > x ){
        if( (lineLHS->z[x]&0xc0)!=0x80 ){
          rc = fdb__out(b, " ", 1);
          RC;
        }
        x++;
      }
      for(j=0; j<len; j++, x++){
        if( (lineLHS->z[x]&0xc0)!=0x80 ){
          rc = fdb__out(b, &c, 1);
          RC;
        }
      }
    }
  }
  if( x ){
    rc = fdb__out(b, "\n", 1);
    RC;
  }
  rc = fdb__outf(b, "                 %8u %.*s\n",
                 b->lnRHS, (int)lineRHS->n, lineRHS->z);
  RC;
  for(i=x=0; i<chng.n; i++){
    int ofst = chng.a[i].iStart2;
    int len = chng.a[i].iLen2;
    if( len ){
      char c = '0' + i;
      if( x==0 ){
        rc = fdb__outf(b, "%*s", 26, "");
        RC;
      }
      while( ofst > x ){
        if( (lineRHS->z[x]&0xc0)!=0x80 ){
          rc = fdb__out(b, " ", 1);
          RC;
        }
        x++;
      }
      for(j=0; j<len; j++, x++){
        if( (lineRHS->z[x]&0xc0)!=0x80 ){
          rc = fdb__out(b, &c, 1);
          RC;
        }
      }
    }
  }
  if( x ){
    rc = fdb__out(b, "\n", 1);
  }
  end:
#undef RC
  return rc;
#endif
}

static int fdb__debug_finish(fsl_dibu * const b){
  DICOSTATE(sst);
  if(1==b->passNumber){
    sst->lineCount[0] = b->lnLHS;
    sst->lineCount[1] = b->lnRHS;
    return 0;
  }
  int rc = fdb__outf(b, "END with %u LHS file lines "
                     "and %u RHS lines (max. %u display lines)\n",
                     b->lnLHS, b->lnRHS, sst->displayLines);
  if(0==rc){
    rc = fdb__outf(b,"Col widths: num left=%u, col left=%u, "
                   "modifier=%u, num right=%u, col right=%u\n",
                   sst->maxWidths[0], sst->maxWidths[1],
                   sst->maxWidths[2], sst->maxWidths[3],
                   sst->maxWidths[4]);
  }
  return rc;
}

void fsl_dibu_finalizer(fsl_dibu * const b){
  *b = fsl_dibu_empty;
  fsl_free(b);
}

static fsl_dibu * fsl__diff_builder_debug(void){
  fsl_dibu * rc =
    fsl_dibu_alloc((fsl_size_t)sizeof(DiffCounter));
  if(rc){
    rc->chunkHeader = fdb__debug_chunkHeader;
    rc->start = fdb__debug_start;
    rc->skip = fdb__debug_skip;
    rc->common = fdb__debug_common;
    rc->insertion = fdb__debug_insertion;
    rc->deletion = fdb__debug_deletion;
    rc->replacement = fdb__debug_replacement;
    rc->edit = fdb__debug_edit;
    rc->finish = fdb__debug_finish;
    rc->finalize = fsl_dibu_finalizer;
    rc->twoPass = true;
    assert(0!=rc->pimpl);
    DiffCounter * const sst = (DiffCounter*)rc->pimpl;
    *sst = DiffCounter_empty;
    assert(0==rc->implFlags);
    assert(0==rc->lnLHS);
    assert(0==rc->lnRHS);
    assert(NULL==rc->opt);
  }
  return rc;
}

/******************** json1 diff builder ********************/
/* Description taken verbatim from fossil(1): */
/*
** This formatter generates a JSON array that describes the difference.
**
** The Json array consists of integer opcodes with each opcode followed
** by zero or more arguments:
**
**   Syntax        Mnemonic    Description
**   -----------   --------    --------------------------
**   0             END         This is the end of the diff
**   1  INTEGER    SKIP        Skip N lines from both files
**   2  STRING     COMMON      The line show by STRING is in both files
**   3  STRING     INSERT      The line STRING is in only the right file
**   4  STRING     DELETE      The STRING line is in only the left file
**   5  SUBARRAY   EDIT        One line is different on left and right.
**
** The SUBARRAY is an array of 3*N+1 strings with N>=0.  The triples
** represent common-text, left-text, and right-text.  The last string
** in SUBARRAY is the common-suffix.  Any string can be empty if it does
** not apply.
*/

static int fdb__outj(fsl_dibu * const b,
                     char const *zJson, int n){
  return n<0
    ? fdb__outf(b, "%!j", zJson)
    : fdb__outf(b, "%!.*j", n, zJson);
}

static int fdb__json1_start(fsl_dibu * const b){
  int rc = fdb__outf(b, "{\"hashLHS\": %!j, \"hashRHS\": %!j, ",
                     b->opt->hashLHS, b->opt->hashRHS);
  if(0==rc && b->opt->nameLHS){
    rc = fdb__outf(b, "\"nameLHS\": %!j, ", b->opt->nameLHS);
  }
  if(0==rc && b->opt->nameRHS){
    rc = fdb__outf(b, "\"nameRHS\": %!j, ", b->opt->nameRHS);
  }
  if(0==rc){
    rc = fdb__out(b, "\"diff\":[", 8);
  }
  return rc;
}

static int fdb__json1_skip(fsl_dibu * const b, uint32_t n){
  return fdb__outf(b, "1,%" PRIu32 ",\n", n);
}
static int fdb__json1_common(fsl_dibu * const b, fsl_dline const * pLine){
  int rc = fdb__out(b, "2,",2);
  if(!rc) {
    rc = fdb__outj(b, pLine->z, (int)pLine->n);
    if(!rc) rc = fdb__out(b, ",\n",2);
  }
  return rc;
}
static int fdb__json1_insertion(fsl_dibu * const b, fsl_dline const * pLine){
  int rc = fdb__out(b, "3,",2);
  if(!rc){
    rc = fdb__outj(b, pLine->z, (int)pLine->n);
    if(!rc) rc = fdb__out(b, ",\n",2);
  }
  return rc;
}
static int fdb__json1_deletion(fsl_dibu * const b, fsl_dline const * pLine){
  int rc = fdb__out(b, "4,",2);
  if(!rc){
    rc = fdb__outj(b, pLine->z, (int)pLine->n);
    if(!rc) rc = fdb__out(b, ",\n",2);
  }
  return rc;
}
static int fdb__json1_replacement(fsl_dibu * const b,
                              fsl_dline const * lineLhs,
                              fsl_dline const * lineRhs) {
  int rc = fdb__out(b, "5,[\"\",",6);
  if(!rc) rc = fdb__outf(b,"%!.*j", (int)lineLhs->n, lineLhs->z);
  if(!rc) rc = fdb__out(b, ",", 1);
  if(!rc) rc = fdb__outf(b,"%!.*j", (int)lineRhs->n, lineRhs->z);
  if(!rc) rc = fdb__out(b, ",\"\"],\n",6);
  return rc;
}
                 
static int fdb__json1_edit(fsl_dibu * const b,
                           fsl_dline const * lineLHS,
                           fsl_dline const * lineRHS){
  int rc = 0;
  int i,x;
  fsl_dline_change chng = fsl_dline_change_empty;

#define RC if(rc) goto end
  rc = fdb__out(b, "5,[", 3); RC;
  fsl_dline_change_spans(lineLHS, lineRHS, &chng);
  for(i=x=0; i<(int)chng.n; i++){
    if(i>0){
      rc = fdb__out(b, ",", 1); RC;
    }
    rc = fdb__outj(b, lineLHS->z + x, (int)chng.a[i].iStart1 - x); RC;
    x = chng.a[i].iStart1;
    rc = fdb__out(b, ",", 1); RC;
    rc = fdb__outj(b, lineLHS->z + x, (int)chng.a[i].iLen1); RC;
    x += chng.a[i].iLen1;
    rc = fdb__out(b, ",", 1); RC;
    rc = fdb__outj(b, lineRHS->z + chng.a[i].iStart2,
                   (int)chng.a[i].iLen2); RC;
  }
  rc = fdb__out(b, ",", 1); RC;
  rc = fdb__outj(b, lineLHS->z + x, (int)(lineLHS->n - x)); RC;
  rc = fdb__out(b, "],\n",3); RC;
  end:
  return rc;
#undef RC
}

static int fdb__json1_finish(fsl_dibu * const b){
  return fdb__out(b, "0]}", 3);
}

static fsl_dibu * fsl__diff_builder_json1(void){
  fsl_dibu * rc = fsl_dibu_alloc(0);
  if(rc){
    rc->chunkHeader = NULL;
    rc->start = fdb__json1_start;
    rc->skip = fdb__json1_skip;
    rc->common = fdb__json1_common;
    rc->insertion = fdb__json1_insertion;
    rc->deletion = fdb__json1_deletion;
    rc->replacement = fdb__json1_replacement;
    rc->edit = fdb__json1_edit;
    rc->finish = fdb__json1_finish;
    rc->finalize = fsl_dibu_finalizer;
    assert(!rc->pimpl);
    assert(0==rc->implFlags);
    assert(0==rc->lnLHS);
    assert(0==rc->lnRHS);
    assert(NULL==rc->opt);
  }
  return rc;
}

/**
   State for the text-mode unified(-ish) diff builder.  We do some
   hoop-jumping here in order to combine runs of delete/insert pairs
   into a group of deletes followed by a group of inserts. It's a
   cosmetic detail only but it makes for more readable output.
*/
struct DiBuUnified {
  /** True if currently processing a block of deletes, else false. */
  bool deleting;
  /** Buffer for insertion lines which are part of delete/insert
      pairs. */
  fsl_buffer bufIns;
};
typedef struct DiBuUnified DiBuUnified;

#define UIMPL(V) DiBuUnified * const V = (DiBuUnified*)b->pimpl
/**
   If utxt diff builder b has any INSERT lines to flush, this
   flushes them. Sets b->impl->deleting to false. Returns non-0
   on output error.
*/
static int fdb__utxt_flush_ins(fsl_dibu * const b){
  int rc = 0;
  UIMPL(p);
  p->deleting = false;
  if(p->bufIns.used>0){
    rc = fdb__out(b, fsl_buffer_cstr(&p->bufIns), p->bufIns.used);
    fsl_buffer_reuse(&p->bufIns);
  }
  return rc;
}

static int fdb__utxt_start(fsl_dibu * const b){
  int rc = 0;
  UIMPL(p);
  p->deleting = false;
  if(p->bufIns.mem) fsl_buffer_reuse(&p->bufIns);
  else fsl_buffer_reserve(&p->bufIns, 1024 * 2);
  if(0==(FSL_DIFF2_NOINDEX & b->opt->diffFlags)){
    rc = fdb__outf(b,"Index: %s\n%.66c\n",
                   b->opt->nameLHS/*RHS?*/, '=');
  }
  if(0==rc){
    rc = fdb__outf(b, "--- %s\n+++ %s\n",
                   b->opt->nameLHS, b->opt->nameRHS);
  }
  return rc;
}

static int fdb__utxt_chunkHeader(fsl_dibu* const b,
                                 uint32_t lnnoLHS, uint32_t linesLHS,
                                 uint32_t lnnoRHS, uint32_t linesRHS ){
  /*
    Ticket 746ebbe86c20b5c0f96cdadd19abd8284770de16:

    Annoying cosmetic bug: the libf impl of this diff will sometimes
    render two directly-adjecent chunks with a separator, e.g.:
  */

  // $ f-vdiff --format u 072d63965188 a725befe5863 -l '*vdiff*' | head -30
  // Index: f-apps/f-vdiff.c
  // ==================================================================
  // --- f-apps/f-vdiff.c
  // +++ f-apps/f-vdiff.c
  // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  //     36     36    fsl_buffer fname;
  //     37     37    fsl_buffer fcontent1;
  //     38     38    fsl_buffer fcontent2;
  //     39     39    fsl_buffer fhash;
  //     40     40    fsl_list globs;
  //            41 +  fsl_dibu_opt diffOpt;
  //            42 +  fsl_diff_builder * diffBuilder;
  //     41     43  } VDiffApp = {
  //     42     44  NULL/*glob*/,
  //     43     45  5/*contextLines*/,
  //     44     46  0/*sbsWidth*/,
  //     45     47  0/*diffFlags*/,
  // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  //     46     48  0/*brief*/,
  //     47     49  fsl_buffer_empty_m/*fname*/,
  //     48     50  fsl_buffer_empty_m/*fcontent1*/,

  /*
    Note now the chunks before/after the second ~~~ line are
    consecutive lines of code. In fossil(1) that case is accounted for
    in the higher-level diff engine, which can not only collapse
    adjacent blocks but also does the rendering of chunk headers in
    that main algorithm (something we cannot do in the library because
    we need the fsl_dibu to be able to output to arbitrary
    destinations). We can only _partially_ account for it here,
    eliminating the extraneous ~~~ line when we're in line-number
    mode. In non-line-number mode we have to output the chunk header
    as-is. If we skip it then the _previous_ chunk header, if any,
    will contain incorrect numbers for the chunk, invaliding the diff
    for purposes of tools which import unified-format diffs.
  */
  int rc = fdb__utxt_flush_ins(b);
  if(0==rc){
    if(FSL_DIFF2_LINE_NUMBERS & b->opt->diffFlags){
      rc = (lnnoLHS == b->lnLHS+1 && lnnoRHS == b->lnRHS+1)
        ? 0
        : fdb__outf(b, "%.40c\n", '~');
    }else{
      rc = fdb__outf(b, "@@ -%" PRIu32 ",%" PRIu32
                     " +%" PRIu32 ",%" PRIu32 " @@\n",
                     lnnoLHS, linesLHS, lnnoRHS, linesRHS);
    }
  }
  return rc;
}


static int fdb__utxt_skip(fsl_dibu * const b, uint32_t n){
  //MARKER(("SKIP\n"));
  int rc = fdb__utxt_flush_ins(b);
  b->lnLHS += n;
  b->lnRHS += n;
  return rc;
}

/**
   Outputs line numbers, if configured to, to b->opt->out.

   - 2 line numbers = common lines
   - lnL only = deletion
   - lnR only = insertion
*/
static int fdb__utxt_lineno(fsl_dibu * const b, uint32_t lnL, uint32_t lnR){
  int rc = 0;
  if(FSL_DIFF2_LINE_NUMBERS & b->opt->diffFlags){
    UIMPL(p);
    if(lnL){ // common or delete
      rc = fdb__outf(b, "%s%6" PRIu32 "%s ",
                     (lnR ? "" : b->opt->ansiColor.deletion),
                     lnL,
                     (lnR ? "" : b->opt->ansiColor.reset));
    }else if(p->deleting){ // insert during deletion grouping
      rc = fsl_buffer_append(&p->bufIns, "       ", 7);
    }else{ // insert w/o deleting grouping
      rc = fdb__out(b, "       ", 7);
    }
    if(0==rc){
      if(!lnL && lnR && p->deleting){ // insert during deletion grouping
        rc = fsl_buffer_appendf(&p->bufIns, "%s%6" PRIu32 "%s ",
                                b->opt->ansiColor.insertion,
                                lnR, b->opt->ansiColor.reset);
      }else if(lnR){ // common or insert w/o deletion grouping.
        rc = fdb__outf(b, "%s%6" PRIu32 "%s ",
                       (lnL ? "" : b->opt->ansiColor.insertion),
                       lnR,
                       (lnL ? "" : b->opt->ansiColor.reset));
      }else{ // deletion
        rc = fdb__out(b, "       ", 7);
      }
    }
  }
  return rc;
}

static int fdb__utxt_common(fsl_dibu * const b, fsl_dline const * pLine){
  //MARKER(("COMMON\n"));
  int rc = fdb__utxt_flush_ins(b);
  if(0==rc){
    ++b->lnLHS;
    ++b->lnRHS;
    rc = fdb__utxt_lineno(b, b->lnLHS, b->lnRHS);
  }
  return rc ? rc : fdb__outf(b, " %.*s\n", (int)pLine->n, pLine->z);
}

static int fdb__utxt_insertion(fsl_dibu * const b, fsl_dline const * pLine){
  //MARKER(("INSERT\n"));
  int rc;
  ++b->lnRHS;
  rc = fdb__utxt_lineno(b, 0, b->lnRHS);
  if(0==rc){
    UIMPL(p);
    if(p->deleting){
      rc = fsl_buffer_appendf(&p->bufIns, "%s+%.*s%s\n",
                              b->opt->ansiColor.insertion,
                              (int)pLine->n, pLine->z,
                              b->opt->ansiColor.reset);
    }else{
      rc = fdb__outf(b, "%s+%.*s%s\n",
                     b->opt->ansiColor.insertion,
                     (int)pLine->n, pLine->z,
                     b->opt->ansiColor.reset);
    }
  }
  return rc;
}
static int fdb__utxt_deletion(fsl_dibu * const b, fsl_dline const * pLine){
  //MARKER(("DELETE\n"));
  UIMPL(p);
  int rc = p->deleting ? 0 : fdb__utxt_flush_ins(b);
  if(0==rc){
    p->deleting = true;
    ++b->lnLHS;
    rc = fdb__utxt_lineno(b, b->lnLHS, 0);
  }
  return rc ? rc : fdb__outf(b, "%s-%.*s%s\n",
                             b->opt->ansiColor.deletion,
                             (int)pLine->n, pLine->z,
                             b->opt->ansiColor.reset);
}
static int fdb__utxt_replacement(fsl_dibu * const b,
                                 fsl_dline const * lineLhs,
                                 fsl_dline const * lineRhs) {
  //MARKER(("REPLACE\n"));
  int rc = b->deletion(b, lineLhs);
  if(0==rc) rc = b->insertion(b, lineRhs);
  return rc;
}
static int fdb__utxt_edit(fsl_dibu * const b,
                           fsl_dline const * lineLhs,
                           fsl_dline const * lineRhs){
  //MARKER(("EDIT\n"));
  int rc = b->deletion(b, lineLhs);
  if(0==rc) rc = b->insertion(b, lineRhs);
  return rc;
}

static int fdb__utxt_finish(fsl_dibu * const b){
  int rc = fdb__utxt_flush_ins(b);
  UIMPL(p);
  fsl_buffer_reuse(&p->bufIns);
  return rc;
}

static void fdb__utxt_finalize(fsl_dibu * const b){
  UIMPL(p);
  fsl_buffer_clear(&p->bufIns);
  fsl_free(b);
}

static fsl_dibu * fsl__diff_builder_utxt(void){
  const DiBuUnified DiBuUnified_empty = {
  false, fsl_buffer_empty_m
  };
  fsl_dibu * rc = fsl_dibu_alloc(sizeof(DiBuUnified));
  if(!rc) return NULL;
  assert(NULL!=rc->pimpl);
  assert(NULL==rc->finally);
  *((DiBuUnified*)rc->pimpl) = DiBuUnified_empty;
  rc->chunkHeader = fdb__utxt_chunkHeader;
  rc->start = fdb__utxt_start;
  rc->skip = fdb__utxt_skip;
  rc->common = fdb__utxt_common;
  rc->insertion = fdb__utxt_insertion;
  rc->deletion = fdb__utxt_deletion;
  rc->replacement = fdb__utxt_replacement;
  rc->edit = fdb__utxt_edit;
  rc->finish = fdb__utxt_finish;
  rc->finalize = fdb__utxt_finalize;
  return rc;
}
#undef UIMPL

struct DiBuTcl {
  /** Buffer for TCL-format string conversion */
  fsl_buffer str;
};
typedef struct DiBuTcl DiBuTcl;
static const DiBuTcl DiBuTcl_empty = {fsl_buffer_empty_m};

#define BR_OPEN if(FSL_DIBU_TCL_BRACES & b->implFlags) \
    rc = fdb__out(b, "{", 1)
#define BR_CLOSE if(FSL_DIBU_TCL_BRACES & b->implFlags) \
    rc = fdb__out(b, "}", 1)

#define DTCL_BUFFER(B) &((DiBuTcl*)(B)->pimpl)->str
static int fdb__outtcl(fsl_dibu * const b,
                       char const *z, unsigned int n,
                       char chAppend ){
  int rc;
  fsl_buffer * const o = DTCL_BUFFER(b);
  fsl_buffer_reuse(o);
  rc = fsl_buffer_append_tcl_literal(o,
                                     (b->implFlags & FSL_DIBU_TCL_BRACES_ESC),
                                     z, n);
  if(0==rc) rc = fdb__out(b, (char const *)o->mem, o->used);
  if(chAppend && 0==rc) rc = fdb__out(b, &chAppend, 1);
  return rc;
}

static int fdb__tcl_start(fsl_dibu * const b){
  int rc = 0;
  fsl_buffer_reuse(DTCL_BUFFER(b));
  if(1==++b->fileCount &&
     FSL_DIBU_TCL_TK==(b->implFlags & FSL_DIBU_TCL_TK)){
    rc = fdb__out(b, "set difftxt {\n", -1);
  }
  if(0==rc && b->fileCount>1) rc = fdb__out(b, "\n", 1);
  if(0==rc && b->opt->nameLHS){
    char const * zRHS =
      b->opt->nameRHS ? b->opt->nameRHS : b->opt->nameLHS;
    BR_OPEN;
    if(0==rc) rc = fdb__out(b, "FILE ", 5);
    if(0==rc) rc = fdb__outtcl(b, b->opt->nameLHS,
                               (unsigned)fsl_strlen(b->opt->nameLHS), ' ');
    if(0==rc) rc = fdb__outtcl(b, zRHS,
                               (unsigned)fsl_strlen(zRHS), 0);
    if(0==rc) {BR_CLOSE;}
    if(0==rc) rc = fdb__out(b, "\n", 1);
  }
  return rc;
}

static int fdb__tcl_skip(fsl_dibu * const b, uint32_t n){
  int rc = 0;
  BR_OPEN;
  if(0==rc) rc = fdb__outf(b, "SKIP %" PRIu32, n);
  if(0==rc) {BR_CLOSE;}
  if(0==rc) rc = fdb__outf(b, "\n", 1);
  return rc;
}

static int fdb__tcl_common(fsl_dibu * const b, fsl_dline const * pLine){
  int rc = 0;
  BR_OPEN;
  if(0==rc) rc = fdb__out(b, "COM  ", 5);
  if(0==rc) rc= fdb__outtcl(b, pLine->z, pLine->n, 0);
  if(0==rc) {BR_CLOSE;}
  if(0==rc) rc = fdb__outf(b, "\n", 1);
  return rc;
}
static int fdb__tcl_insertion(fsl_dibu * const b, fsl_dline const * pLine){
  int rc = 0;
  BR_OPEN;
  if(0==rc) rc = fdb__out(b, "INS  ", 5);
  if(0==rc) rc = fdb__outtcl(b, pLine->z, pLine->n, 0);
  if(0==rc) {BR_CLOSE;}
  if(0==rc) rc = fdb__outf(b, "\n", 1);
  return rc;
}
static int fdb__tcl_deletion(fsl_dibu * const b, fsl_dline const * pLine){
  int rc = 0;
  BR_OPEN;
  if(0==rc) rc = fdb__out(b, "DEL  ", 5);
  if(0==rc) rc = fdb__outtcl(b, pLine->z, pLine->n, 0);
  if(0==rc) {BR_CLOSE;}
  if(0==rc) rc = fdb__outf(b, "\n", 1);
  return rc;
}
static int fdb__tcl_replacement(fsl_dibu * const b,
                                fsl_dline const * lineLhs,
                                fsl_dline const * lineRhs) {
  int rc = 0;
  BR_OPEN;
  if(0==rc) rc = fdb__out(b, "EDIT \"\" ", 8);
  if(0==rc) rc = fdb__outtcl(b, lineLhs->z, lineLhs->n, ' ');
  if(0==rc) rc = fdb__outtcl(b, lineRhs->z, lineRhs->n, 0);
  if(0==rc) {BR_CLOSE;}
  if(0==rc) rc = fdb__outf(b, "\n", 1);
  return rc;
}
                 
static int fdb__tcl_edit(fsl_dibu * const b,
                         fsl_dline const * lineLHS,
                         fsl_dline const * lineRHS){
  int rc = 0;
  int i, x;
  fsl_dline_change chng = fsl_dline_change_empty;
#define RC if(rc) goto end
  BR_OPEN;
  rc = fdb__out(b, "EDIT", 4); RC;
  fsl_dline_change_spans(lineLHS, lineRHS, &chng);
  for(i=x=0; i<chng.n; i++){
    rc = fdb__out(b, " ", 1); RC;
    rc = fdb__outtcl(b, lineLHS->z + x, chng.a[i].iStart1 - x, ' '); RC;
    x = chng.a[i].iStart1;
    rc = fdb__outtcl(b, lineLHS->z + x, chng.a[i].iLen1, ' '); RC;
    x += chng.a[i].iLen1;
    rc = fdb__outtcl(b, lineRHS->z + chng.a[i].iStart2,
                     chng.a[i].iLen2, 0); RC;
  }
  assert(0==rc);
  if( x < lineLHS->n ){
    rc = fdb__out(b, " ", 1); RC;
    rc = fdb__outtcl(b, lineLHS->z + x, lineLHS->n - x, 0); RC;
  }
  BR_CLOSE; RC;
  rc = fdb__out(b, "\n", 1);
  end:
#undef RC
  return rc;
}

static int fdb__tcl_finish(fsl_dibu * const b __unused){
  int rc = 0;
#if 0
  BR_CLOSE;
  if(0==rc && FSL_DIBU_TCL_BRACES & b->implFlags){
    rc = fdb__out(b, "\n", 1);
  }
#endif
  return rc;
}
static int fdb__tcl_finally(fsl_dibu * const b){
  int rc = 0;
  if(FSL_DIBU_TCL_TK==(b->implFlags & FSL_DIBU_TCL_TK)){
    extern char const * fsl_difftk_cstr;
    if(!b->fileCount) rc = fdb__out(b,"set difftxt {\n",-1);
    if(0==rc) rc = fdb__out(b, "}\nset fossilcmd {}\n", -1);
    if(0==rc) rc = fdb__out(b, fsl_difftk_cstr, -1);
  }
  return rc;
}

#undef BR_OPEN
#undef BR_CLOSE

static void fdb__tcl_finalize(fsl_dibu * const b){
  fsl_buffer_clear( &((DiBuTcl*)b->pimpl)->str );
  *b = fsl_dibu_empty;
  fsl_free(b);
}

static fsl_dibu * fsl__diff_builder_tcl(void){
  fsl_dibu * rc =
    fsl_dibu_alloc((fsl_size_t)sizeof(DiBuTcl));
  if(rc){
    rc->chunkHeader = NULL;
    rc->start = fdb__tcl_start;
    rc->skip = fdb__tcl_skip;
    rc->common = fdb__tcl_common;
    rc->insertion = fdb__tcl_insertion;
    rc->deletion = fdb__tcl_deletion;
    rc->replacement = fdb__tcl_replacement;
    rc->edit = fdb__tcl_edit;
    rc->finish = fdb__tcl_finish;
    rc->finally = fdb__tcl_finally;
    rc->finalize = fdb__tcl_finalize;
    assert(0!=rc->pimpl);
    DiBuTcl * const dbt = (DiBuTcl*)rc->pimpl;
    *dbt = DiBuTcl_empty;
    if(fsl_buffer_reserve(&dbt->str, 120)){
      rc->finalize(rc);
      rc = 0;
    }
  }
  return rc;
}

static int fdb__splittxt_mod(fsl_dibu * const b, char ch){
  assert(2==b->passNumber);
  return fdb__outf(b, " %c ", ch);
}

static int fdb__splittxt_lineno(fsl_dibu * const b,
                                DiffCounter const * const sst,
                                bool isLeft, uint32_t n){
  assert(2==b->passNumber);
  int const col = isLeft ? DICO_NUM1 : DICO_NUM2;
  return n
    ? fdb__outf(b, "%*" PRIu32 " ", sst->maxWidths[col], n)
    : fdb__outf(b, "%.*c ", sst->maxWidths[col], ' ');
}

static int fdb__splittxt_start(fsl_dibu * const b){
  int rc = 0;
  if(1==b->passNumber){
    DICOSTATE(sst);
    *sst = DiffCounter_empty;
    ++b->fileCount;
    return rc;
  }
  if(b->fileCount>1){
    rc = fdb__out(b, "\n", 1);
  }
  if(0==rc){
    fsl_dibu_opt const * const o = b->opt;
    if(o->nameLHS || o->nameRHS
       || o->hashLHS || o->hashRHS){
      rc = fdb__outf(b, "--- %s%s%s\n+++ %s%s%s\n",
                     o->nameLHS ? o->nameLHS : "",
                     (o->nameLHS && o->hashLHS) ? " " : "",
                     o->hashLHS ? o->hashLHS : "",
                     o->nameRHS ? o->nameRHS : "",
                     (o->nameRHS && o->hashRHS) ? " " : "",
                     o->hashRHS ? o->hashRHS : "");
    }
  }
  return rc;
}

static int fdb__splittxt_skip(fsl_dibu * const b, uint32_t n){
  b->lnLHS += n;
  b->lnRHS += n;
  if(1==b->passNumber) return 0;
  DICOSTATE(sst);
  int const maxWidth1 = maxColWidth(b, sst, DICO_TEXT1);
  int const maxWidth2 = maxColWidth(b, sst, DICO_TEXT2);
  return fdb__outf(b, "%.*c %.*c   %.*c %.*c\n",
                 sst->maxWidths[DICO_NUM1], '~',
                 maxWidth1, '~',
                 sst->maxWidths[DICO_NUM2], '~',
                 maxWidth2, '~');
}

static int fdb__splittxt_color(fsl_dibu * const b,
                               int modType){
  char const *z = 0;
  switch(modType){
    case (int)'i': z = b->opt->ansiColor.insertion; break;
    case (int)'d': z = b->opt->ansiColor.deletion; break;
    case (int)'r'/*replacement*/: 
    case (int)'e': z = b->opt->ansiColor.edit; break;
    case 0: z = b->opt->ansiColor.reset; break;
    default:
      assert(!"invalid color op!");
  }
  return z&&*z ? fdb__outf(b, "%s", z) : 0;
}

static int fdb__splittxt_side(fsl_dibu * const b,
                              DiffCounter * const sst,
                              bool isLeft,
                              fsl_dline const * const pLine){
  int rc = fdb__splittxt_lineno(b, sst, isLeft,
                                pLine ? (isLeft ? b->lnLHS : b->lnRHS) : 0U);
  if(0==rc){
    uint32_t const w = maxColWidth(b, sst, isLeft ? DICO_TEXT1 : DICO_TEXT2);
    if(pLine){
      fsl_size_t const nU =
        /* Measure column width in UTF8 characters, not bytes! */
        fsl_strlen_utf8(pLine->z, (fsl_int_t)pLine->n);
      rc = fdb__outf(b, "%#.*s", (int)(w < nU ? w : nU), pLine->z);
      if(0==rc && w>nU){
        rc = fdb__outf(b, "%.*c", (int)(w - nU), ' ');
      }
    }else{
      rc = fdb__outf(b, "%.*c", (int)w, ' ');
    }
    if(0==rc && !isLeft) rc = fdb__out(b, "\n", 1);
  }
  return rc;
}

static int fdb__splittxt_common(fsl_dibu * const b,
                                fsl_dline const * const pLine){
  int rc = 0;
  DICOSTATE(sst);
  ++b->lnLHS;
  ++b->lnRHS;
  if(1==b->passNumber){
    fdb__dico_update_maxlen(sst, DICO_TEXT1, pLine->z, pLine->n);
    fdb__dico_update_maxlen(sst, DICO_TEXT2, pLine->z, pLine->n);
    return 0;
  }
  rc = fdb__splittxt_side(b, sst, true, pLine);
  if(0==rc) rc = fdb__splittxt_mod(b, ' ');
  if(0==rc) rc = fdb__splittxt_side(b, sst, false, pLine);
  return rc;
}

static int fdb__splittxt_insertion(fsl_dibu * const b,
                                   fsl_dline const * const pLine){
  int rc = 0;
  DICOSTATE(sst);
  ++b->lnRHS;
  if(1==b->passNumber){
    fdb__dico_update_maxlen(sst, DICO_TEXT1, pLine->z, pLine->n);
    return rc;
  }
  rc = fdb__splittxt_color(b, 'i');
  if(0==rc) rc = fdb__splittxt_side(b, sst, true, NULL);
  if(0==rc) rc = fdb__splittxt_mod(b, '>');
  if(0==rc) rc = fdb__splittxt_side(b, sst, false, pLine);
  if(0==rc) rc = fdb__splittxt_color(b, 0);
  return rc;
}

static int fdb__splittxt_deletion(fsl_dibu * const b,
                                  fsl_dline const * const pLine){
  int rc = 0;
  DICOSTATE(sst);
  ++b->lnLHS;
  if(1==b->passNumber){
    fdb__dico_update_maxlen(sst, DICO_TEXT2, pLine->z, pLine->n);
    return rc;
  }
  rc = fdb__splittxt_color(b, 'd');
  if(0==rc) rc = fdb__splittxt_side(b, sst, true, pLine);
  if(0==rc) rc = fdb__splittxt_mod(b, '<');
  if(0==rc) rc = fdb__splittxt_side(b, sst, false, NULL);
  if(0==rc) rc = fdb__splittxt_color(b, 0);
  return rc;
}

static int fdb__splittxt_replacement(fsl_dibu * const b,
                                     fsl_dline const * const lineLhs,
                                     fsl_dline const * const lineRhs) {
#if 0
  int rc = b->deletion(b, lineLhs);
  if(0==rc) rc = b->insertion(b, lineRhs);
  return rc;
#else    
  int rc = 0;
  DICOSTATE(sst);
  ++b->lnLHS;
  ++b->lnRHS;
  if(1==b->passNumber){
    fdb__dico_update_maxlen(sst, DICO_TEXT1, lineLhs->z, lineLhs->n);
    fdb__dico_update_maxlen(sst, DICO_TEXT2, lineRhs->z, lineRhs->n);
    return 0;
  }
  rc = fdb__splittxt_color(b, 'e');
  if(0==rc) rc = fdb__splittxt_side(b, sst, true, lineLhs);
  if(0==rc) rc = fdb__splittxt_mod(b, '|');
  if(0==rc) rc = fdb__splittxt_side(b, sst, false, lineRhs);
  if(0==rc) rc = fdb__splittxt_color(b, 0);
  return rc;
#endif
}

static int fdb__splittxt_finish(fsl_dibu * const b){
  int rc = 0;
  if(1==b->passNumber){
    DICOSTATE(sst);
    uint32_t ln = b->lnLHS;
    /* Calculate width of line number columns. */
    sst->maxWidths[DICO_NUM1] = sst->maxWidths[DICO_NUM2] = 1;
    for(; ln>=10; ln/=10) ++sst->maxWidths[DICO_NUM1];
    ln = b->lnRHS;
    for(; ln>=10; ln/=10) ++sst->maxWidths[DICO_NUM2];
  }
  return rc;
}

static void fdb__splittxt_finalize(fsl_dibu * const b){
  *b = fsl_dibu_empty;
  fsl_free(b);
}

static fsl_dibu * fsl__diff_builder_splittxt(void){
  fsl_dibu * rc =
    fsl_dibu_alloc((fsl_size_t)sizeof(DiffCounter));
  if(rc){
    rc->twoPass = true;
    rc->chunkHeader = NULL;
    rc->start = fdb__splittxt_start;
    rc->skip = fdb__splittxt_skip;
    rc->common = fdb__splittxt_common;
    rc->insertion = fdb__splittxt_insertion;
    rc->deletion = fdb__splittxt_deletion;
    rc->replacement = fdb__splittxt_replacement;
    rc->edit = fdb__splittxt_replacement;
    rc->finish = fdb__splittxt_finish;
    rc->finalize = fdb__splittxt_finalize;
    assert(0!=rc->pimpl);
    DiffCounter * const sst = (DiffCounter*)rc->pimpl;
    *sst = DiffCounter_empty;
  }
  return rc;
}

int fsl_dibu_factory( fsl_dibu_e type,
                              fsl_dibu **pOut ){
  int rc = FSL_RC_TYPE;
  fsl_dibu * (*factory)(void) = NULL;
  switch(type){
    case FSL_DIBU_DEBUG:
      factory = fsl__diff_builder_debug;
      break;
    case FSL_DIBU_JSON1:
      factory = fsl__diff_builder_json1;
      break;
    case FSL_DIBU_UNIFIED_TEXT:
      factory = fsl__diff_builder_utxt;
      break;
    case FSL_DIBU_TCL:
      factory = fsl__diff_builder_tcl;
      break;
    case FSL_DIBU_SPLIT_TEXT:
      factory = fsl__diff_builder_splittxt;
      break;
    case FSL_DIBU_INVALID: break;
  }
  if(NULL!=factory){
    *pOut = factory();
    rc = *pOut ? 0 : FSL_RC_OOM;
  }
  return rc;
}

#undef MARKER
#undef DICOSTATE
#undef DTCL_BUFFER
/* end of file ./src/dibu.c */
/* start of file ./src/diff.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2022 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2013-2022 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/

/************************************************************************
  This file houses Fossil's diff-generation routines (as opposed to
  the delta-generation). This code is a straight port of those
  algorithms from the Fossil SCM project, initially implemented by
  D. Richard Hipp, ported and the license re-assigned to this project
  with this consent.
*/
#include <assert.h>
#include <memory.h>
#include <stdlib.h>
#include <string.h> /* for memmove()/strlen() */

#include <stdio.h>
#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)


const fsl_dline fsl_dline_empty = fsl_dline_empty_m;
const fsl_dline_change fsl_dline_change_empty = fsl_dline_change_empty_m;
const fsl__diff_cx fsl__diff_cx_empty = fsl__diff_cx_empty_m;

void fsl__diff_cx_clean(fsl__diff_cx * const cx){
  fsl_free(cx->aFrom);
  fsl_free(cx->aTo);
  fsl_free(cx->aEdit);
  cx->aFrom = cx->aTo = NULL;
  cx->aEdit = NULL;
  *cx = fsl__diff_cx_empty;
}

/* Fast isspace for use by diff */
static const char diffIsSpace[] = {
  0, 0, 0, 0, 0, 0, 0, 0,   1, 1, 1, 0, 1, 1, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,   0, 0, 0, 0, 0, 0, 0, 0,
  1, 0, 0, 0, 0, 0, 0, 0,   0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,   0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,   0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,   0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,   0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,   0, 0, 0, 0, 0, 0, 0, 0,

  0, 0, 0, 0, 0, 0, 0, 0,   0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,   0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,   0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,   0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,   0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,   0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,   0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,   0, 0, 0, 0, 0, 0, 0, 0,
};
#define diff_isspace(X)  (diffIsSpace[(unsigned char)(X)])


/**
   Length of a dline.
*/
#define LENGTH(X) ((X)->n)


/**
   Minimum of two values
*/
static int minInt(int a, int b){ return a<b ? a : b; }

/** @internal

   Compute the optimal longest common subsequence (LCS) using an
   exhaustive search. This version of the LCS is only used for
   shorter input strings since runtime is O(N*N) where N is the
   input string length.
*/
static void fsl__diff_optimal_lcs(
  fsl__diff_cx * const p,     /* Two files being compared */
  int iS1, int iE1,          /* Range of lines in p->aFrom[] */
  int iS2, int iE2,          /* Range of lines in p->aTo[] */
  int *piSX, int *piEX,      /* Write p->aFrom[] common segment here */
  int *piSY, int *piEY       /* Write p->aTo[] common segment here */
){
  int mxLength = 0;          /* Length of longest common subsequence */
  int i, j;                  /* Loop counters */
  int k;                     /* Length of a candidate subsequence */
  int iSXb = iS1;            /* Best match so far */
  int iSYb = iS2;            /* Best match so far */

  for(i=iS1; i<iE1-mxLength; i++){
    for(j=iS2; j<iE2-mxLength; j++){
      if( p->cmpLine(&p->aFrom[i], &p->aTo[j]) ) continue;
      if( mxLength && p->cmpLine(&p->aFrom[i+mxLength], &p->aTo[j+mxLength]) ){
        continue;
      }
      k = 1;
      while( i+k<iE1 && j+k<iE2 && p->cmpLine(&p->aFrom[i+k],&p->aTo[j+k])==0 ){
        k++;
      }
      if( k>mxLength ){
        iSXb = i;
        iSYb = j;
        mxLength = k;
      }
    }
  }
  *piSX = iSXb;
  *piEX = iSXb + mxLength;
  *piSY = iSYb;
  *piEY = iSYb + mxLength;
}

/**
    Compare two blocks of text on lines iS1 through iE1-1 of the aFrom[]
    file and lines iS2 through iE2-1 of the aTo[] file.  Locate a sequence
    of lines in these two blocks that are exactly the same.  Return
    the bounds of the matching sequence.
   
    If there are two or more possible answers of the same length, the
    returned sequence should be the one closest to the center of the
    input range.
   
    Ideally, the common sequence should be the longest possible common
    sequence.  However, an exact computation of LCS is O(N*N) which is
    way too slow for larger files.  So this routine uses an O(N)
    heuristic approximation based on hashing that usually works about
    as well.  But if the O(N) algorithm doesn't get a good solution
    and N is not too large, we fall back to an exact solution by
    calling fsl__diff_optimal_lcs().
*/
static void fsl__diff_lcs(
  fsl__diff_cx * const p,     /* Two files being compared */
  int iS1, int iE1,          /* Range of lines in p->aFrom[] */
  int iS2, int iE2,          /* Range of lines in p->aTo[] */
  int *piSX, int *piEX,      /* Write p->aFrom[] common segment here */
  int *piSY, int *piEY       /* Write p->aTo[] common segment here */
){
  int i, j, k;               /* Loop counters */
  int n;                     /* Loop limit */
  fsl_dline *pA, *pB;            /* Pointers to lines */
  int iSX, iSY, iEX, iEY;    /* Current match */
  int skew = 0;              /* How lopsided is the match */
  int dist = 0;              /* Distance of match from center */
  int mid;                   /* Center of the chng */
  int iSXb, iSYb, iEXb, iEYb;   /* Best match so far */
  int iSXp, iSYp, iEXp, iEYp;   /* Previous match */
  sqlite3_int64 bestScore;      /* Best score so far */
  sqlite3_int64 score;          /* Score for current candidate LCS */
  int span;                     /* combined width of the input sequences */
  int cutoff = 4;            /* Max hash chain entries to follow */
  int nextCutoff = -1;       /* Value of cutoff for next iteration */

  span = (iE1 - iS1) + (iE2 - iS2);
  bestScore = -10000;
  score = 0;
  iSXb = iSXp = iS1;
  iEXb = iEXp = iS1;
  iSYb = iSYp = iS2;
  iEYb = iEYp = iS2;
  mid = (iE1 + iS1)/2;
  do{
    nextCutoff = 0;
    for(i=iS1; i<iE1; i++){
      int limit = 0;
      j = p->aTo[p->aFrom[i].h % p->nTo].iHash;
      while( j>0
        && (j-1<iS2 || j>=iE2 || p->cmpLine(&p->aFrom[i], &p->aTo[j-1]))
      ){
        if( limit++ > cutoff ){
          j = 0;
          nextCutoff = cutoff*4;
          break;
        }
        j = p->aTo[j-1].iNext;
      }
      if( j==0 ) continue;
      assert( i>=iSXb && i>=iSXp );
      if( i<iEXb && j>=iSYb && j<iEYb ) continue;
      if( i<iEXp && j>=iSYp && j<iEYp ) continue;
      iSX = i;
      iSY = j-1;
      pA = &p->aFrom[iSX-1];
      pB = &p->aTo[iSY-1];
      n = minInt(iSX-iS1, iSY-iS2);
      for(k=0; k<n && p->cmpLine(pA,pB)==0; k++, pA--, pB--){}
      iSX -= k;
      iSY -= k;
      iEX = i+1;
      iEY = j;
      pA = &p->aFrom[iEX];
      pB = &p->aTo[iEY];
      n = minInt(iE1-iEX, iE2-iEY);
      for(k=0; k<n && p->cmpLine(pA,pB)==0; k++, pA++, pB++){}
      iEX += k;
      iEY += k;
      skew = (iSX-iS1) - (iSY-iS2);
      if( skew<0 ) skew = -skew;
      dist = (iSX+iEX)/2 - mid;
      if( dist<0 ) dist = -dist;
      score = (iEX - iSX)*(sqlite3_int64)span - (skew + dist);
      if( score>bestScore ){
        bestScore = score;
        iSXb = iSX;
        iSYb = iSY;
        iEXb = iEX;
        iEYb = iEY;
      }else if( iEX>iEXp ){
        iSXp = iSX;
        iSYp = iSY;
        iEXp = iEX;
        iEYp = iEY;
      }
    }
  }while( iSXb==iEXb && nextCutoff && (cutoff=nextCutoff)<=64 );
  if( iSXb==iEXb && (sqlite3_int64)(iE1-iS1)*(iE2-iS2)<2500 ){
    fsl__diff_optimal_lcs(p, iS1, iE1, iS2, iE2, piSX, piEX, piSY, piEY);
  }else{
    *piSX = iSXb;
    *piSY = iSYb;
    *piEX = iEXb;
    *piEY = iEYb;
  }
}

void fsl__dump_triples(fsl__diff_cx const * const p,
                       char const * zFile, int ln ){
  // Compare this with (fossil xdiff --raw) on the same inputs
  fprintf(stderr,"%s:%d: Compare this with (fossil xdiff --raw) on the same inputs:\n",
          zFile, ln);
  for(int i = 0; p->aEdit[i] || p->aEdit[i+1] || p->aEdit[i+2]; i+=3){
    printf(" copy %6d  delete %6d  insert %6d\n",
           p->aEdit[i], p->aEdit[i+1], p->aEdit[i+2]);
  }
}

/** @internal
    Expand the size of p->aEdit array to hold at least nEdit elements.
 */
static int fsl__diff_expand_edit(fsl__diff_cx * const p, int nEdit){
  void * re = fsl_realloc(p->aEdit, nEdit*sizeof(int));
  if(!re) return FSL_RC_OOM;
  else{
    p->aEdit = (int*)re;
    p->nEditAlloc = nEdit;
    return 0;
  }
}

/**
    Append a new COPY/DELETE/INSERT triple.
   
    Returns 0 on success, FSL_RC_OOM on OOM.
 */
static int appendTriple(fsl__diff_cx *p, int nCopy, int nDel, int nIns){
  /* printf("APPEND %d/%d/%d\n", nCopy, nDel, nIns); */
  if( p->nEdit>=3 ){
    if( p->aEdit[p->nEdit-1]==0 ){
      if( p->aEdit[p->nEdit-2]==0 ){
        p->aEdit[p->nEdit-3] += nCopy;
        p->aEdit[p->nEdit-2] += nDel;
        p->aEdit[p->nEdit-1] += nIns;
        return 0;
      }
      if( nCopy==0 ){
        p->aEdit[p->nEdit-2] += nDel;
        p->aEdit[p->nEdit-1] += nIns;
        return 0;
      }
    }
    if( nCopy==0 && nDel==0 ){
      p->aEdit[p->nEdit-1] += nIns;
      return 0;
    }
  }
  if( p->nEdit+3>p->nEditAlloc ){
    int const rc = fsl__diff_expand_edit(p, p->nEdit*2 + 15);
    if(rc) return rc;
    else if( p->aEdit==0 ) return 0;
  }
  p->aEdit[p->nEdit++] = nCopy;
  p->aEdit[p->nEdit++] = nDel;
  p->aEdit[p->nEdit++] = nIns;
  return 0;
}

/*
** A common subsequene between p->aFrom and p->aTo has been found.
** This routine tries to judge if the subsequence really is a valid
** match or rather is just an artifact of an indentation change.
**
** Return non-zero if the subsequence is valid.  Return zero if the
** subsequence seems likely to be an editing artifact and should be
** ignored.
**
** This routine is a heuristic optimization intended to give more
** intuitive diff results following an indentation change it code that
** is formatted similarly to C/C++, Javascript, Go, TCL, and similar
** languages that use {...} for nesting.  A correct diff is computed
** even if this routine always returns true (non-zero).  But sometimes
** a more intuitive diff can result if this routine returns false.
**
** The subsequences consists of the rows iSX through iEX-1 (inclusive)
** in p->aFrom[].  The total sequences is iS1 through iE1-1 (inclusive)
** of p->aFrom[].
**
** Example where this heuristic is useful, see the diff at
** https://www.sqlite.org/src/fdiff?v1=0e79dd15cbdb4f48&v2=33955a6fd874dd97
**
** See also discussion at https://fossil-scm.org/forum/forumpost/9ba3284295
**
** ALGORITHM (subject to change and refinement):
**
**    1.  If the subsequence is larger than 1/7th of the original span,
**        then consider it valid.  --> return 1
**
**    2.  If no lines of the subsequence contains more than one
**        non-whitespace character,  --> return 0
**
**    3.  If any line of the subsequence contains more than one non-whitespace
**        character and is unique across the entire sequence after ignoring
**        leading and trailing whitespace   --> return 1
**
**    4.  Otherwise, it is potentially an artifact of an indentation
**        change. --> return 0
*/
static bool likelyNotIndentChngArtifact(
  fsl__diff_cx const * const p,     /* The complete diff context */
  int iS1,         /* Start of the main segment */
  int iSX,         /* Start of the subsequence */
  int iEX,         /* First row past the end of the subsequence */
  int iE1          /* First row past the end of the main segment */
){
  int i, j, n;

  /* Rule (1) */
  if( (iEX-iSX)*7 >= (iE1-iS1) ) return 1;

  /* Compute fsl_dline.indent and fsl_dline.nw for all lines of the subsequence.
  ** If no lines contain more than one non-whitespace character return
  ** 0 because the subsequence could be due to an indentation change.
  ** Rule (2).
  */
  n = 0;
  for(i=iSX; i<iEX; i++){
    fsl_dline *pA = &p->aFrom[i];
    if( pA->nw==0 && pA->n ){
      const char *zA = pA->z;
      const int nn = pA->n;
      int ii, jj;
      for(ii=0; ii<nn && diff_isspace(zA[ii]); ii++){}
      pA->indent = ii;
      for(jj=nn-1; jj>ii && diff_isspace(zA[jj]); jj--){}
      pA->nw = jj - ii + 1;
    }
    if( pA->nw>1 ) n++;
  }
  if( n==0 ) return 0;

  /* Compute fsl_dline.indent and fsl_dline.nw for the entire sequence */
  for(i=iS1; i<iE1; i++){
    fsl_dline *pA;
    if( i==iSX ){
      i = iEX;
      if( i>=iE1 ) break;
    }
    pA = &p->aFrom[i];
    if( pA->nw==0 && pA->n ){
      const char *zA = pA->z;
      const int nn = pA->n;
      int ii, jj;
      for(ii=0; ii<nn && diff_isspace(zA[ii]); ii++){}
      pA->indent = ii;
      for(jj=nn-1; jj>ii && diff_isspace(zA[jj]); jj--){}
      pA->nw = jj - ii + 1;
    }
  }

  /* Check to see if any subsequence line that has more than one
  ** non-whitespace character is unique across the entire sequence.
  ** Rule (3)
  */
  for(i=iSX; i<iEX; i++){
    const char *z = p->aFrom[i].z + p->aFrom[i].indent;
    const int nw = p->aFrom[i].nw;
    if( nw<=1 ) continue;
    for(j=iS1; j<iSX; j++){
      if( p->aFrom[j].nw!=nw ) continue;
      if( memcmp(p->aFrom[j].z+p->aFrom[j].indent,z,nw)==0 ) break;
    }
    if( j<iSX ) continue;
    for(j=iEX; j<iE1; j++){
      if( p->aFrom[j].nw!=nw ) continue;
      if( memcmp(p->aFrom[j].z+p->aFrom[j].indent,z,nw)==0 ) break;
    }
    if( j>=iE1 ) break;
  }
  return i<iEX;
}

/**
    Do a single step in the difference.  Compute a sequence of
    copy/delete/insert steps that will convert lines iS1 through iE1-1
    of the input into lines iS2 through iE2-1 of the output and write
    that sequence into the difference context.
   
    The algorithm is to find a block of common text near the middle of
    the two segments being diffed.  Then recursively compute
    differences on the blocks before and after that common segment.
    Special cases apply if either input segment is empty or if the two
    segments have no text in common.
 */
static int diff_step(fsl__diff_cx *p, int iS1, int iE1, int iS2, int iE2){
  int iSX, iEX, iSY, iEY;
  int rc = 0;
  if( iE1<=iS1 ){
    /* The first segment is empty */
    if( iE2>iS2 ){
      rc = appendTriple(p, 0, 0, iE2-iS2);
    }
    return rc;
  }
  if( iE2<=iS2 ){
    /* The second segment is empty */
    return appendTriple(p, 0, iE1-iS1, 0);
  }

  /* Find the longest matching segment between the two sequences */
  fsl__diff_lcs(p, iS1, iE1, iS2, iE2, &iSX, &iEX, &iSY, &iEY);
  if( iEX>iSX+5
      || (iEX>iSX && likelyNotIndentChngArtifact(p,iS1,iSX,iEX,iE1) )){
    /* A common segment has been found.
       Recursively diff either side of the matching segment */
    rc = diff_step(p, iS1, iSX, iS2, iSY);
    if(!rc){
      if(iEX>iSX){
        rc = appendTriple(p, iEX - iSX, 0, 0);
      }
      if(!rc) rc = diff_step(p, iEX, iE1, iEY, iE2);
    }
  }else{
    /* The two segments have nothing in common.  Delete the first then
       insert the second. */
    rc = appendTriple(p, 0, iE1-iS1, iE2-iS2);
  }
  return rc;
}

int fsl__diff_all(fsl__diff_cx * const p){
  int mnE, iS, iE1, iE2;
  int rc = 0;
  /* Carve off the common header and footer */
  iE1 = p->nFrom;
  iE2 = p->nTo;
  while( iE1>0 && iE2>0 && p->cmpLine(&p->aFrom[iE1-1], &p->aTo[iE2-1])==0 ){
    iE1--;
    iE2--;
  }
  mnE = iE1<iE2 ? iE1 : iE2;
  for(iS=0; iS<mnE && p->cmpLine(&p->aFrom[iS],&p->aTo[iS])==0; iS++){}

  /* do the difference */
  if( iS>0 ){
    rc = appendTriple(p, iS, 0, 0);
    if(rc) return rc;
  }
  rc = diff_step(p, iS, iE1, iS, iE2);
  //fsl__dump_triples(p, __FILE__, __LINE__);
  if(rc) return rc;
  else if( iE1<p->nFrom ){
    rc = appendTriple(p, p->nFrom - iE1, 0, 0);
    if(rc) return rc;
  }
  /* Terminate the COPY/DELETE/INSERT triples with three zeros */
  rc = fsl__diff_expand_edit(p, p->nEdit+3);
  if(0==rc){
    if(p->aEdit ){
      p->aEdit[p->nEdit++] = 0;
      p->aEdit[p->nEdit++] = 0;
      p->aEdit[p->nEdit++] = 0;
      //fsl__dump_triples(p, __FILE__, __LINE__);
    }
  }
  return rc;
}

void fsl__diff_optimize(fsl__diff_cx * const p){
  int r;       /* Index of current triple */
  int lnFrom;  /* Line number in p->aFrom */
  int lnTo;    /* Line number in p->aTo */
  int cpy, del, ins;

  //fsl__dump_triples(p, __FILE__, __LINE__);
  lnFrom = lnTo = 0;
  for(r=0; r<p->nEdit; r += 3){
    cpy = p->aEdit[r];
    del = p->aEdit[r+1];
    ins = p->aEdit[r+2];
    lnFrom += cpy;
    lnTo += cpy;

    /* Shift insertions toward the beginning of the file */
    while( cpy>0 && del==0 && ins>0 ){
      fsl_dline *pTop = &p->aFrom[lnFrom-1];  /* Line before start of insert */
      fsl_dline *pBtm = &p->aTo[lnTo+ins-1];  /* Last line inserted */
      if( p->cmpLine(pTop, pBtm) ) break;
      if( LENGTH(pTop+1)+LENGTH(pBtm)<=LENGTH(pTop)+LENGTH(pBtm-1) ) break;
      lnFrom--;
      lnTo--;
      p->aEdit[r]--;
      p->aEdit[r+3]++;
      cpy--;
    }

    /* Shift insertions toward the end of the file */
    while( r+3<p->nEdit && p->aEdit[r+3]>0 && del==0 && ins>0 ){
      fsl_dline *pTop = &p->aTo[lnTo];       /* First line inserted */
      fsl_dline *pBtm = &p->aTo[lnTo+ins];   /* First line past end of insert */
      if( p->cmpLine(pTop, pBtm) ) break;
      if( LENGTH(pTop)+LENGTH(pBtm-1)<=LENGTH(pTop+1)+LENGTH(pBtm) ) break;
      lnFrom++;
      lnTo++;
      p->aEdit[r]++;
      p->aEdit[r+3]--;
      cpy++;
    }

    /* Shift deletions toward the beginning of the file */
    while( cpy>0 && del>0 && ins==0 ){
      fsl_dline *pTop = &p->aFrom[lnFrom-1];     /* Line before start of delete */
      fsl_dline *pBtm = &p->aFrom[lnFrom+del-1]; /* Last line deleted */
      if( p->cmpLine(pTop, pBtm) ) break;
      if( LENGTH(pTop+1)+LENGTH(pBtm)<=LENGTH(pTop)+LENGTH(pBtm-1) ) break;
      lnFrom--;
      lnTo--;
      p->aEdit[r]--;
      p->aEdit[r+3]++;
      cpy--;
    }

    /* Shift deletions toward the end of the file */
    while( r+3<p->nEdit && p->aEdit[r+3]>0 && del>0 && ins==0 ){
      fsl_dline *pTop = &p->aFrom[lnFrom];     /* First line deleted */
      fsl_dline *pBtm = &p->aFrom[lnFrom+del]; /* First line past end of delete */
      if( p->cmpLine(pTop, pBtm) ) break;
      if( LENGTH(pTop)+LENGTH(pBtm-1)<=LENGTH(pTop)+LENGTH(pBtm) ) break;
      lnFrom++;
      lnTo++;
      p->aEdit[r]++;
      p->aEdit[r+3]--;
      cpy++;
    }

    lnFrom += del;
    lnTo += ins;
  }
  //fsl__dump_triples(p, __FILE__, __LINE__);
}


/**
   Counts the number of lines in the first n bytes of the given 
   string. If n<0 then fsl_strlen() is used to count it. 

   It includes the last line in the count even if it lacks the \n
   terminator. If an empty string is passed in, the number of lines
   is zero.

   For the purposes of this function, a string is considered empty if
   it contains no characters OR contains only NUL characters.

   If the input appears to be plain text it returns true and, if nOut
   is not NULL, writes the number of lines there. If the input appears
   to be binary, returns false and does not modify nOut.
*/
static bool fsl__count_lines(const char *z, fsl_int_t n, uint32_t * nOut ){
  uint32_t nLine;
  const char *zNL, *z2;
  if(n<0) n = (fsl_int_t)fsl_strlen(z);
  for(nLine=0, z2=z; (zNL = strchr(z2,'\n'))!=0; z2=zNL+1, nLine++){}
  if( z2[0]!='\0' ){
    nLine++;
    do{ z2++; }while( z2[0]!='\0' );
  }
  if( n!=(fsl_int_t)(z2-z) ) return false;
  if( nOut ) *nOut = nLine;
  return true;
}

int fsl_break_into_dlines(const char *z, fsl_int_t n,
                          uint32_t *pnLine,
                          fsl_dline **pOut, uint64_t diffFlags){
  uint32_t nLine, i, k, nn, s, x;
  uint64_t h, h2;
  fsl_dline *a = 0;
  const char *zNL;

  if(!z || !n){
    *pnLine = 0;
    *pOut = NULL;
    return 0;
  }
  if( !fsl__count_lines(z, n, &nLine) ){
    return FSL_RC_DIFF_BINARY;
  }
  assert( nLine>0 || z[0]=='\0' );
  if(nLine>0){
    a = fsl_malloc( sizeof(a[0])*nLine );
    if(!a) return FSL_RC_OOM;
    memset(a, 0, sizeof(a[0])*nLine);
  }else{
    *pnLine = 0;
    *pOut = a;
    return 0;
  }
  assert( a );
  i = 0;
  do{
    zNL = strchr(z,'\n');
    if( zNL==0 ) zNL = z+n;
    nn = (uint32_t)(zNL - z);
    if( nn>FSL__LINE_LENGTH_MASK ){
      fsl_free(a);
      *pOut = 0;
      *pnLine = 0;
      return FSL_RC_DIFF_BINARY;
    }
    a[i].z = z;
    k = nn;
    if( diffFlags & FSL_DIFF2_STRIP_EOLCR ){
      if( k>0 && z[k-1]=='\r' ){ k--; }
    }
    a[i].n = k;
    if( diffFlags & FSL_DIFF2_IGNORE_EOLWS ){
      while( k>0 && diff_isspace(z[k-1]) ){ k--; }
    }
    if( (diffFlags & FSL_DIFF2_IGNORE_ALLWS)
        ==FSL_DIFF2_IGNORE_ALLWS ){
      uint32_t numws = 0;
      for(s=0; s<k && z[s]<=' '; s++){}
      a[i].indent = s;
      a[i].nw = k - s;
      for(h=0, x=s; x<k; ++x){
        char c = z[x];
        if( diff_isspace(c) ){
          ++numws;
        }else{
          h = (h^c)*9000000000000000041LL;
        }
      }
      k -= numws;
    }else{
      uint32_t k2 = k & ~0x7;
      uint64_t m;
      for(h=x=s=0; x<k2; x += 8){
        memcpy(&m, z+x, 8);
        h = (h^m)*9000000000000000041LL;
      }
      m = 0;
      memcpy(&m, z+x, k-k2);
      h ^= m;
    }
    a[i].h = h = ((h%281474976710597LL)<<FSL__LINE_LENGTH_MASK_SZ) | (k-s);
    h2 = h % nLine;
    a[i].iNext = a[h2].iHash;
    a[h2].iHash = i+1;
    z += nn+1; n -= nn+1;
    i++;
  }while( zNL[0]!='\0' && zNL[1]!='\0' );
  assert( i==nLine );

  *pnLine = nLine;
  *pOut = a;
  return 0;
}

int fsl_dline_cmp(const fsl_dline * const pA,
                  const fsl_dline * const pB){
  if( pA->h!=pB->h ) return 1;
  return memcmp(pA->z,pB->z, pA->h&FSL__LINE_LENGTH_MASK);
}

int fsl_dline_cmp_ignore_ws(const fsl_dline * const pA,
                            const fsl_dline * const pB){
  if( pA->h==pB->h ){
    unsigned short a, b;
    if( memcmp(pA->z, pB->z, pA->h&FSL__LINE_LENGTH_MASK)==0 ) return 0;
    a = pA->indent;
    b = pB->indent;
    while( a<pA->n || b<pB->n ){
      if( a<pA->n && b<pB->n && pA->z[a++] != pB->z[b++] ) return 1;
      while( a<pA->n && diff_isspace(pA->z[a])) ++a;
      while( b<pB->n && diff_isspace(pB->z[b])) ++b;
    }
    return pA->n-a != pB->n-b;
  }
  return 1;
}

/*
** The two text segments zLeft and zRight are known to be different on
** both ends, but they might have  a common segment in the middle.  If
** they do not have a common segment, return 0.  If they do have a large
** common segment, return non-0 and before doing so set:
**
**   aLCS[0] = start of the common segment in zLeft
**   aLCS[1] = end of the common segment in zLeft
**   aLCS[2] = start of the common segment in zLeft
**   aLCS[3] = end of the common segment in zLeft
**
** This computation is for display purposes only and does not have to be
** optimal or exact.
*/
static int textLCS2(
  const char *zLeft,  uint32_t nA, /* String on the left */
  const char *zRight, uint32_t nB, /* String on the right */
  uint32_t *aLCS                   /* Identify bounds of LCS here */
){
  const unsigned char *zA = (const unsigned char*)zLeft;    /* left string */
  const unsigned char *zB = (const unsigned char*)zRight;   /* right string */
  uint32_t i, j, k;               /* Loop counters */
  uint32_t lenBest = 0;           /* Match length to beat */

  for(i=0; i<nA-lenBest; i++){
    unsigned char cA = zA[i];
    if( (cA&0xc0)==0x80 ) continue;
    for(j=0; j<nB-lenBest; j++ ){
      if( zB[j]==cA ){
        for(k=1; j+k<nB && i+k<nA && zB[j+k]==zA[i+k]; k++){}
        while( (zB[j+k]&0xc0)==0x80 ){ k--; }
        if( k>lenBest ){
          lenBest = k;
          aLCS[0] = i;
          aLCS[1] = i+k;
          aLCS[2] = j;
          aLCS[3] = j+k;
        }
      }
    }
  }
  return lenBest>0;
}

/*
** Find the smallest spans that are different between two text strings
** that are known to be different on both ends. Returns the number
** of entries in p->a which get populated.
*/
static unsigned short textLineChanges(
  const char *zLeft,  uint32_t nA, /* String on the left */
  const char *zRight, uint32_t nB, /* String on the right */
  fsl_dline_change * const p             /* Write results here */
){
  p->n = 1;
  p->a[0].iStart1 = 0;
  p->a[0].iLen1 = nA;
  p->a[0].iStart2 = 0;
  p->a[0].iLen2 = nB;
  p->a[0].isMin = 0;
  while( p->n<fsl_dline_change_max_spans-1 ){
    int mxi = -1;
    int mxLen = -1;
    int x, i;
    uint32_t aLCS[4] = {0,0,0,0};
    struct fsl_dline_change_span *a, *b;
    for(i=0; i<p->n; i++){
      if( p->a[i].isMin ) continue;
      x = p->a[i].iLen1;
      if( p->a[i].iLen2<x ) x = p->a[i].iLen2;
      if( x>mxLen ){
        mxLen = x;
        mxi = i;
      }
    }
    if( mxLen<6 ) break;
    x = textLCS2(zLeft + p->a[mxi].iStart1, p->a[mxi].iLen1,
                 zRight + p->a[mxi].iStart2, p->a[mxi].iLen2, aLCS);
    if( x==0 ){
      p->a[mxi].isMin = 1;
      continue;
    }
    a = p->a+mxi;
    b = a+1;
    if( mxi<p->n-1 ){
      memmove(b+1, b, sizeof(*b)*(p->n-mxi-1));
    }
    p->n++;
    b->iStart1 = a->iStart1 + aLCS[1];
    b->iLen1 = a->iLen1 - aLCS[1];
    a->iLen1 = aLCS[0];
    b->iStart2 = a->iStart2 + aLCS[3];
    b->iLen2 = a->iLen2 - aLCS[3];
    a->iLen2 = aLCS[2];
    b->isMin = 0;
  }
  return p->n;
}

/*
** Return true if the string starts with n spaces
*/
static int allSpaces(const char *z, int n){
  int i;
  for(i=0; i<n && diff_isspace(z[i]); ++i){}
  return i==n;
}

/*
** Try to improve the human-readability of the fsl_dline_change p.
**
** (1)  If the first change span shows a change of indentation, try to
**      move that indentation change to the left margin.
**
** (2)  Try to shift changes so that they begin or end with a space.
*/
static void improveReadability(
  const char *zA,  /* Left line of the change */
  const char *zB,  /* Right line of the change */
  fsl_dline_change * const p /* The fsl_dline_change to be adjusted */
){
  int j, n, len;
  if( p->n<1 ) return;

  /* (1) Attempt to move indentation changes to the left margin */
  if( p->a[0].iLen1==0
   && (len = p->a[0].iLen2)>0
   && (j = p->a[0].iStart2)>0
   && zB[0]==zB[j]
   && allSpaces(zB, j)
  ){
    for(n=1; n<len && n<j && zB[j]==zB[j+n]; n++){}
    if( n<len ){
      memmove(&p->a[1], &p->a[0], sizeof(p->a[0])*p->n);
      p->n++;
      p->a[0] = p->a[1];
      p->a[1].iStart2 += n;
      p->a[1].iLen2 -= n;
      p->a[0].iLen2 = n;
    }
    p->a[0].iStart1 = 0;
    p->a[0].iStart2 = 0;
  }else
  if( p->a[0].iLen2==0
   && (len = p->a[0].iLen1)>0
   && (j = p->a[0].iStart1)>0
   && zA[0]==zA[j]
   && allSpaces(zA, j)
  ){
    for(n=1; n<len && n<j && zA[j]==zA[j+n]; n++){}
    if( n<len ){
      memmove(&p->a[1], &p->a[0], sizeof(p->a[0])*p->n);
      p->n++;
      p->a[0] = p->a[1];
      p->a[1].iStart1 += n;
      p->a[1].iLen1 -= n;
      p->a[0].iLen1 = n;
    }
    p->a[0].iStart1 = 0;
    p->a[0].iStart2 = 0;
  }

  /* (2) Try to shift changes so that they begin or end with a
  ** space.  (TBD) */
}

void fsl_dline_change_spans(const fsl_dline *pLeft, const fsl_dline *pRight,
                            fsl_dline_change * const p){
  /* fossil(1) counterpart ==> diff.c oneLineChange() */
  int nLeft;           /* Length of left line in bytes */
  int nRight;          /* Length of right line in bytes */
  int nShort;          /* Shortest of left and right */
  int nPrefix;         /* Length of common prefix */
  int nSuffix;         /* Length of common suffix */
  int nCommon;         /* Total byte length of suffix and prefix */
  const char *zLeft;   /* Text of the left line */
  const char *zRight;  /* Text of the right line */
  int nLeftDiff;       /* nLeft - nPrefix - nSuffix */
  int nRightDiff;      /* nRight - nPrefix - nSuffix */

  nLeft = pLeft->n;
  zLeft = pLeft->z;
  nRight = pRight->n;
  zRight = pRight->z;
  nShort = nLeft<nRight ? nLeft : nRight;

  nPrefix = 0;
  while( nPrefix<nShort && zLeft[nPrefix]==zRight[nPrefix] ){
    nPrefix++;
  }
  if( nPrefix<nShort ){
    while( nPrefix>0 && (zLeft[nPrefix]&0xc0)==0x80 ) nPrefix--;
  }
  nSuffix = 0;
  if( nPrefix<nShort ){
    while( nSuffix<nShort
           && zLeft[nLeft-nSuffix-1]==zRight[nRight-nSuffix-1] ){
      nSuffix++;
    }
    if( nSuffix<nShort ){
      while( nSuffix>0 && (zLeft[nLeft-nSuffix]&0xc0)==0x80 ) nSuffix--;
    }
    if( nSuffix==nLeft || nSuffix==nRight ) nPrefix = 0;
  }
  nCommon = nPrefix + nSuffix;

  /* If the prefix and suffix overlap, that means that we are dealing with
  ** a pure insertion or deletion of text that can have multiple alignments.
  ** Try to find an alignment to begins and ends on whitespace, or on
  ** punctuation, rather than in the middle of a name or number.
  */
  if( nCommon > nShort ){
    int iBest = -1;
    int iBestVal = -1;
    int i;
    int nLong = nLeft<nRight ? nRight : nLeft;
    int nGap = nLong - nShort;
    for(i=nShort-nSuffix; i<=nPrefix; i++){
       int iVal = 0;
       char c = zLeft[i];
       if( diff_isspace(c) ){
         iVal += 5;
       }else if( !fsl_isalnum(c) ){
         iVal += 2;
       }
       c = zLeft[i+nGap-1];
       if( diff_isspace(c) ){
         iVal += 5;
       }else if( !fsl_isalnum(c) ){
         iVal += 2;
       }
       if( iVal>iBestVal ){
         iBestVal = iVal;
         iBest = i;
       }
    }
    nPrefix = iBest;
    nSuffix = nShort - nPrefix;
    nCommon = nPrefix + nSuffix;
  }

  /* A single chunk of text inserted */
  if( nCommon==nLeft ){
    p->n = 1;
    p->a[0].iStart1 = nPrefix;
    p->a[0].iLen1 = 0;
    p->a[0].iStart2 = nPrefix;
    p->a[0].iLen2 = nRight - nCommon;
    improveReadability(zLeft, zRight, p);
    return;
  }

  /* A single chunk of text deleted */
  if( nCommon==nRight ){
    p->n = 1;
    p->a[0].iStart1 = nPrefix;
    p->a[0].iLen1 = nLeft - nCommon;
    p->a[0].iStart2 = nPrefix;
    p->a[0].iLen2 = 0;
    improveReadability(zLeft, zRight, p);
    return;
  }

  /* At this point we know that there is a chunk of text that has
  ** changed between the left and the right.  Check to see if there
  ** is a large unchanged section in the middle of that changed block.
  */
  nLeftDiff = nLeft - nCommon;
  nRightDiff = nRight - nCommon;
  if( nLeftDiff >= 4
   && nRightDiff >= 4
   && textLineChanges(&zLeft[nPrefix], nLeftDiff,
                      &zRight[nPrefix], nRightDiff, p)>1
  ){
    int i;
    for(i=0; i<p->n; i++){
      p->a[i].iStart1 += nPrefix;
      p->a[i].iStart2 += nPrefix;
    }
    improveReadability(zLeft, zRight, p);
    return;
  }

  /* If all else fails, show a single big change between left and right */
  p->n = 1;
  p->a[0].iStart1 = nPrefix;
  p->a[0].iLen1 = nLeft - nCommon;
  p->a[0].iStart2 = nPrefix;
  p->a[0].iLen2 = nRight - nCommon;
  improveReadability(zLeft, zRight, p);
}


/*
** The threshold at which diffBlockAlignment transitions from the
** O(N*N) Wagner minimum-edit-distance algorithm to a less process
** O(NlogN) divide-and-conquer approach.
*/
#define DIFF_ALIGN_MX  1225

/**
   FSL_DIFF_SMALL_GAP=0 is a temporary(? as of 2022-01-04) patch for a
   cosmetic-only (but unsightly) quirk of the diff engine where it
   produces a pair of identical DELETE/INSERT lines. Richard's
   preliminary solution for it is to remove the "small gap merging,"
   but he notes (in fossil /chat) that he's not recommending this as
   "the" fix.

   PS: we colloquially know this as "the lineno diff" because it was first
   reported in a diff which resulted in:

```
-  int     lineno;
+  int     lineno;
```

   (No, there are no whitespace changes there.)

   To reiterate, though: this is not a "bug," in that it does not
   cause incorrect results when applying the resulting unfified-diff
   patches. It does, however, cause confusion for human users.


   Here are two inputs which, when diffed, expose the lineno behavior:

   #1:

```
struct fnc_diff_view_state {
  int     first_line_onscreen;
  int     last_line_onscreen;
  int     diff_flags;
  int     context;
  int     sbs;
  int     matched_line;
  int     current_line;
  int     lineno;
  size_t     ncols;
  size_t     nlines;
  off_t    *line_offsets;
  bool     eof;
  bool     colour;
  bool     showmeta;
  bool     showln;
};
```

   #2:

```
struct fnc_diff_view_state {
  int     first_line_onscreen;
  int     last_line_onscreen;
  int     diff_flags;
  int     context;
  int     sbs;
  int     matched_line;
  int     selected_line;
  int     lineno;
  int     gtl;
  size_t     ncols;
  size_t     nlines;
  off_t    *line_offsets;
  bool     eof;
  bool     colour;
  bool     showmeta;
  bool     showln;
};
```

   Result without this patch:

```
Index: X.0
==================================================================
--- X.0
+++ X.1
@@ -3,15 +3,16 @@
   int     last_line_onscreen;
   int     diff_flags;
   int     context;
   int     sbs;
   int     matched_line;
+  int     selected_line;
-  int     current_line;
-  int     lineno;
+  int     lineno;
+  int     gtl;
   size_t     ncols;
   size_t     nlines;
   off_t    *line_offsets;
   bool     eof;
   bool     colour;
   bool     showmeta;
   bool     showln;
 };
```

    And with the patch:

```
Index: X.0
==================================================================
--- X.0
+++ X.1
@@ -3,15 +3,16 @@
   int     last_line_onscreen;
   int     diff_flags;
   int     context;
   int     sbs;
   int     matched_line;
-  int     current_line;
+  int     selected_line;
   int     lineno;
+  int     gtl;
   size_t     ncols;
   size_t     nlines;
   off_t    *line_offsets;
   bool     eof;
   bool     colour;
   bool     showmeta;
   bool     showln;
 };
```

*/
#define FSL_DIFF_SMALL_GAP 0

#if FSL_DIFF_SMALL_GAP
/*
** R[] is an array of six integer, two COPY/DELETE/INSERT triples for a
** pair of adjacent differences.  Return true if the gap between these
** two differences is so small that they should be rendered as a single
** edit.
*/
static int smallGap2(const int *R, int ma, int mb){
  int m = R[3];
  ma += R[4] + m;
  mb += R[5] + m;
  if( ma*mb>DIFF_ALIGN_MX ) return 0;
  return m<=2 || m<=(R[1]+R[2]+R[4]+R[5])/8;
}
#endif

static unsigned short diff_opt_context_lines(fsl_dibu_opt const * opt){
  const unsigned short dflt = 5;
  unsigned short n = opt ? opt->contextLines : dflt;
  if( !n && (opt->diffFlags & FSL_DIFF2_CONTEXT_ZERO)==0 ){
    n = dflt;
  }
  return n;
}

/*
** Minimum of two values
*/
static int diffMin(int a, int b){ return a<b ? a : b; }

/****************************************************************************/
/*
** Return the number between 0 and 100 that is smaller the closer pA and
** pB match.  Return 0 for a perfect match.  Return 100 if pA and pB are
** completely different.
**
** The current algorithm is as follows:
**
** (1) Remove leading and trailing whitespace.
** (2) Truncate both strings to at most 250 characters
** (3) If the two strings have a common prefix, measure that prefix
** (4) Find the length of the longest common subsequence that is
**     at least 150% longer than the common prefix.
** (5) Longer common subsequences yield lower scores.
*/
static int match_dline2(fsl_dline * const pA, fsl_dline * const pB){
  const char *zA;            /* Left string */
  const char *zB;            /* right string */
  int nA;                    /* Bytes in zA[] */
  int nB;                    /* Bytes in zB[] */
  int nMin;
  int nPrefix;
  int avg;                   /* Average length of A and B */
  int i, j, k;               /* Loop counters */
  int best = 0;              /* Longest match found so far */
  int score;                 /* Final score.  0..100 */
  unsigned char c;           /* Character being examined */
  unsigned char aFirst[256]; /* aFirst[X] = index in zB[] of first char X */
  unsigned char aNext[252];  /* aNext[i] = index in zB[] of next zB[i] char */

  zA = pA->z;
  if( pA->nw==0 && pA->n ){
    for(i=0; i<pA->n && diff_isspace(zA[i]); i++){}
    pA->indent = i;
    for(j=pA->n-1; j>i && diff_isspace(zA[j]); j--){}
    pA->nw = j - i + 1;
  }
  zA += pA->indent;
  nA = pA->nw;

  zB = pB->z;
  if( pB->nw==0 && pB->n ){
    for(i=0; i<pB->n && diff_isspace(zB[i]); i++){}
    pB->indent = i;
    for(j=pB->n-1; j>i && diff_isspace(zB[j]); j--){}
    pB->nw = j - i + 1;
  }
  zB += pB->indent;
  nB = pB->nw;

  if( nA>250 ) nA = 250;
  if( nB>250 ) nB = 250;
  avg = (nA+nB)/2;
  if( avg==0 ) return 0;
  nMin = nA;
  if( nB<nMin ) nMin = nB;
  if( nMin==0 ) return 68;
  for(nPrefix=0; nPrefix<nMin && zA[nPrefix]==zB[nPrefix]; nPrefix++){}
  best = 0;
  if( nPrefix>5 && nPrefix>nMin/2 ){
    best = nPrefix*3/2;
    if( best>=avg - 2 ) best = avg - 2;
  }
  if( nA==nB && memcmp(zA, zB, nA)==0 ) return 0;
  memset(aFirst, 0xff, sizeof(aFirst));
  zA--; zB--;   /* Make both zA[] and zB[] 1-indexed */
  for(i=nB; i>0; i--){
    c = (unsigned char)zB[i];
    aNext[i] = aFirst[c];
    aFirst[c] = i;
  }
  for(i=1; i<=nA-best; i++){
    c = (unsigned char)zA[i];
    for(j=aFirst[c]; j<nB-best && memcmp(&zA[i],&zB[j],best)==0; j = aNext[j]){
      int limit = diffMin(nA-i, nB-j);
      for(k=best; k<=limit && zA[k+i]==zB[k+j]; k++){}
      if( k>best ) best = k;
    }
  }
  score = 5 + ((best>=avg) ? 0 : (avg - best)*95/avg);

#if 0
  fprintf(stderr, "A: [%.*s]\nB: [%.*s]\nbest=%d avg=%d score=%d\n",
  nA, zA+1, nB, zB+1, best, avg, score);
#endif

  /* Return the result */
  return score;
}

// Forward decl for recursion's sake.
static int diffBlockAlignment(
  fsl_dline * const aLeft, int nLeft,
  fsl_dline * const aRight, int nRight,
  fsl_dibu_opt const * pOpt,
  unsigned char **pResult,
  unsigned *pNResult
);

/*
** Make a copy of a list of nLine fsl_dline objects from one array to
** another.  Hash the new array to ignore whitespace.
*/
static void diffDLineXfer(
  fsl_dline *aTo,
  const fsl_dline *aFrom,
  int nLine
){
  int i, j, k;
  uint64_t h, h2;
  for(i=0; i<nLine; i++) aTo[i].iHash = 0;
  for(i=0; i<nLine; i++){
    const char *z = aFrom[i].z;
    int n = aFrom[i].n;
    for(j=0; j<n && diff_isspace(z[j]); j++){}
    aTo[i].z = &z[j];
    for(k=aFrom[i].n; k>j && diff_isspace(z[k-1]); k--){}
    aTo[i].n = n = k-j;
    aTo[i].indent = 0;
    aTo[i].nw = 0;
    for(h=0; j<k; j++){
      char c = z[j];
      if( !diff_isspace(c) ){
        h = (h^c)*9000000000000000041LL;
      }
    }
    aTo[i].h = h = ((h%281474976710597LL)<<FSL__LINE_LENGTH_MASK_SZ) | n;
    h2 = h % nLine;
    aTo[i].iNext = aTo[h2].iHash;
    aTo[h2].iHash = i+1;
  }
}

/*
** For a difficult diff-block alignment that was originally for
** the default consider-all-whitespace algorithm, try to find the
** longest common subsequence between the two blocks that involves
** only whitespace changes.
**
** Result is stored in *pOut and must be eventually fsl_free()d.
** Returns 0 on success, setting *pOut to NULL if no good match is
** found. Returns FSL_RC_OOM on allocation error.
*/
static int diffBlockAlignmentIgnoreSpace(
  fsl_dline * const aLeft, int nLeft,     /* Text on the left */
  fsl_dline * const aRight, int nRight,   /* Text on the right */
  fsl_dibu_opt const *pOpt,            /* Configuration options */
  unsigned char ** pOut,          /* OUTPUT: Result */
  unsigned *pNResult                /* OUTPUT: length of result */
){
  fsl__diff_cx dc;
  int iSX, iEX;                /* Start and end of LCS on the left */
  int iSY, iEY;                /* Start and end of the LCS on the right */
  unsigned char *a1, *a2;
  int n1, n2, nLCS, rc = 0;

  dc.aEdit = 0;
  dc.nEdit = 0;
  dc.nEditAlloc = 0;
  dc.nFrom = nLeft;
  dc.nTo = nRight;
  dc.cmpLine = fsl_dline_cmp_ignore_ws;
  dc.aFrom = fsl_malloc( sizeof(fsl_dline)*(nLeft+nRight) );
  if(!dc.aFrom) return FSL_RC_OOM;
  dc.aTo = &dc.aFrom[dc.nFrom];
  diffDLineXfer(dc.aFrom, aLeft, nLeft);
  diffDLineXfer(dc.aTo, aRight, nRight);
  fsl__diff_optimal_lcs(&dc,0,nLeft,0,nRight,&iSX,&iEX,&iSY,&iEY);
  fsl_free(dc.aFrom);
  nLCS = iEX - iSX;
  if( nLCS<5 ){
    /* No good LCS was found */
    *pOut = NULL;
    *pNResult = 0;
    return 0;
  }
  rc = diffBlockAlignment(aLeft,iSX,aRight,iSY,
                          pOpt,&a1, (unsigned *)&n1);
  if(rc) return rc;
  rc = diffBlockAlignment(aLeft+iEX, nLeft-iEX,
                          aRight+iEY, nRight-iEY,
                          pOpt, &a2, (unsigned *)&n2);
  if(rc){
    fsl_free(a1);
    return rc;
  }else{
    unsigned char * x = (unsigned char *)fsl_realloc(a1, n1+nLCS+n2);
    if(NULL==x){
      fsl_free(a1);
      fsl_free(a2);
      return FSL_RC_OOM;
    }
    a1 = x;
  }
  memcpy(a1+n1+nLCS,a2,n2);
  memset(a1+n1,3,nLCS);
  fsl_free(a2);
  *pNResult = (unsigned)(n1+n2+nLCS);
  *pOut = a1;
  return 0;
}

/*
** This is a helper route for diffBlockAlignment().  In this case,
** a very large block is encountered that might be too expensive to
** use the O(N*N) Wagner edit distance algorithm.  So instead, this
** block implements a less-precise but faster O(N*logN) divide-and-conquer
** approach.
**
** Result is stored in *pOut and must be eventually fsl_free()d.
** Returns 0 on success. Returns FSL_RC_OOM on allocation error.
*/
static int diffBlockAlignmentDivideAndConquer(
  fsl_dline * const aLeft, int nLeft,     /* Text on the left */
  fsl_dline * const aRight, int nRight,   /* Text on the right */
  fsl_dibu_opt const *pOpt,            /* Configuration options */
  unsigned char ** pOut,       /* OUTPUT: result */
  unsigned *pNResult                /* OUTPUT: length of result */
){
  fsl_dline *aSmall;               /* The smaller of aLeft and aRight */
  fsl_dline *aBig;                 /* The larger of aLeft and aRight */
  int nSmall, nBig;            /* Size of aSmall and aBig.  nSmall<=nBig */
  int iDivSmall, iDivBig;      /* Divider point for aSmall and aBig */
  int iDivLeft, iDivRight;     /* Divider point for aLeft and aRight */
  unsigned char *a1 = 0, *a2 = 0; /* Results of the alignments on two halves */
  int n1, n2;                  /* Number of entries in a1 and a2 */
  int score, bestScore;        /* Score and best score seen so far */
  int i;                       /* Loop counter */
  int rc;

  if( nLeft>nRight ){
    aSmall = aRight;
    nSmall = nRight;
    aBig = aLeft;
    nBig = nLeft;
  }else{
    aSmall = aLeft;
    nSmall = nLeft;
    aBig = aRight;
    nBig = nRight;
  }
  iDivBig = nBig/2;
  iDivSmall = nSmall/2;
  bestScore = 10000;
  for(i=0; i<nSmall; i++){
    score = match_dline2(aBig+iDivBig, aSmall+i) + abs(i-nSmall/2)*2;
    if( score<bestScore ){
      bestScore = score;
      iDivSmall = i;
    }
  }
  if( aSmall==aRight ){
    iDivRight = iDivSmall;
    iDivLeft = iDivBig;
  }else{
    iDivRight = iDivBig;
    iDivLeft = iDivSmall;
  }
  rc = diffBlockAlignment(aLeft,iDivLeft,aRight,iDivRight,
                          pOpt,&a1, (unsigned*)&n1);
  if(!rc){
    rc = diffBlockAlignment(aLeft+iDivLeft, nLeft-iDivLeft,
                            aRight+iDivRight, nRight-iDivRight,
                            pOpt, &a2, (unsigned*)&n2);
  }
  if(rc){
    fsl_free(a1);
    fsl_free(a2);
  }else{
    unsigned char * x = (unsigned char *)fsl_realloc(a1, n1+n2);
    if(!x) rc = FSL_RC_OOM;
    else{
      a1 = x;
      memcpy(a1+n1,a2,n2);
      *pNResult = (unsigned)(n1+n2);
      *pOut = a1;
    }
    fsl_free(a2);
  }
  return rc;
}


/*
** There is a change block in which nLeft lines of text on the left are
** converted into nRight lines of text on the right.  This routine computes
** how the lines on the left line up with the lines on the right.
**
** The return value is a buffer of unsigned characters, obtained from
** fsl_malloc().  (The caller needs to free the `*pResult` value using
** fsl_free().)  Entries in the returned array have values as follows:
**
**    1.  Delete the next line of pLeft.
**    2.  Insert the next line of pRight.
**    3.  The next line of pLeft changes into the next line of pRight.
**    4.  Delete one line from pLeft and add one line to pRight.
**
** The length of the returned array will be at most nLeft+nRight bytes.
** If the first bytes is 4, that means we could not compute reasonable
** alignment between the two blocks.
**
** Algorithm:  Wagner's minimum edit-distance algorithm, modified by
** adding a cost to each match based on how well the two rows match
** each other.  Insertion and deletion costs are 50.  Match costs
** are between 0 and 100 where 0 is a perfect match 100 is a complete
** mismatch.
*/
int diffBlockAlignment(
  fsl_dline * const aLeft, int nLeft,     /* Text on the left */
  fsl_dline * const aRight, int nRight,   /* Text on the right */
  fsl_dibu_opt const * pOpt,             /* Configuration options */
  unsigned char **pResult,         /* Raw result */
  unsigned *pNResult               /* OUTPUT: length of result */
){
  int i, j, k;                 /* Loop counters */
  int *a = 0;                  /* One row of the Wagner matrix */
  int *pToFree = 0;            /* Space that needs to be freed */
  unsigned char *aM = 0;       /* Wagner result matrix */
  int nMatch, iMatch;          /* Number of matching lines and match score */
  int aBuf[100];               /* Stack space for a[] if nRight not to big */
  int rc = 0;

  if( nLeft==0 ){
    aM = fsl_malloc( nRight + 2 );
    if(!aM) return FSL_RC_OOM;
    memset(aM, 2, nRight);
    *pNResult = nRight;
    *pResult = aM;
    return 0;
  }
  if( nRight==0 ){
    aM = fsl_malloc( nLeft + 2 );
    if(!aM) return FSL_RC_OOM;
    memset(aM, 1, nLeft);
    *pNResult = nLeft;
    *pResult = aM;
    return 0;
  }

  /* For large alignments, try to use alternative algorithms that are
  ** faster than the O(N*N) Wagner edit distance. */
  if( nLeft*nRight>DIFF_ALIGN_MX
      && (pOpt->diffFlags & FSL_DIFF2_SLOW_SBS)==0 ){
    if( (pOpt->diffFlags & FSL_DIFF2_IGNORE_ALLWS)==0 ){
      *pResult = NULL;
      rc = diffBlockAlignmentIgnoreSpace(aLeft, nLeft,aRight,nRight,
                                         pOpt, pResult, pNResult);
      if(rc || *pResult) return rc;
    }
    return diffBlockAlignmentDivideAndConquer(aLeft, nLeft,aRight, nRight,
                                              pOpt, pResult, pNResult);
  }

  /* If we reach this point, we will be doing an O(N*N) Wagner minimum
  ** edit distance to compute the alignment.
  */
  if( nRight < (int)(sizeof(aBuf)/sizeof(aBuf[0]))-1 ){
    pToFree = 0;
    a = aBuf;
  }else{
    a = pToFree = fsl_malloc( sizeof(a[0])*(nRight+1) );
    if(!a){
      rc = FSL_RC_OOM;
      goto end;
    }
  }
  aM = fsl_malloc( (nLeft+1)*(nRight+1) );
  if(!aM){
    rc = FSL_RC_OOM;
    goto end;
  }

  /* Compute the best alignment */
  for(i=0; i<=nRight; i++){
    aM[i] = 2;
    a[i] = i*50;
  }
  aM[0] = 0;
  for(j=1; j<=nLeft; j++){
    int p = a[0];
    a[0] = p+50;
    aM[j*(nRight+1)] = 1;
    for(i=1; i<=nRight; i++){
      int m = a[i-1]+50;
      int d = 2;
      if( m>a[i]+50 ){
        m = a[i]+50;
        d = 1;
      }
      if( m>p ){
        int const score =
          match_dline2(&aLeft[j-1], &aRight[i-1]);
        if( (score<=90 || (i<j+1 && i>j-1)) && m>p+score ){
          m = p+score;
          d = 3 | score*4;
        }
      }
      p = a[i];
      a[i] = m;
      aM[j*(nRight+1)+i] = d;
    }
  }

  /* Compute the lowest-cost path back through the matrix */
  i = nRight;
  j = nLeft;
  k = (nRight+1)*(nLeft+1)-1;
  nMatch = iMatch = 0;
  while( i+j>0 ){
    unsigned char c = aM[k];
    if( c>=3 ){
      assert( i>0 && j>0 );
      i--;
      j--;
      nMatch++;
      iMatch += (c>>2);
      aM[k] = 3;
    }else if( c==2 ){
      assert( i>0 );
      i--;
    }else{
      assert( j>0 );
      j--;
    }
    k--;
    aM[k] = aM[j*(nRight+1)+i];
  }
  k++;
  i = (nRight+1)*(nLeft+1) - k;
  memmove(aM, &aM[k], i);
  *pNResult = i;
  *pResult = aM;

  end:
  fsl_free(pToFree);
  return rc;
}


/*
** Format a diff using a fsl_dibu object
*/
static int fdb__format(
  fsl__diff_cx * const cx,
  fsl_dibu * const pBuilder
){
  fsl_dline *A;        /* Left side of the diff */
  fsl_dline *B;        /* Right side of the diff */
  fsl_dibu_opt const * pOpt = pBuilder->opt;
  const int *R;          /* Array of COPY/DELETE/INSERT triples */
  unsigned int a;    /* Index of next line in A[] */
  unsigned int b;    /* Index of next line in B[] */
  unsigned int r;        /* Index into R[] */
  unsigned int nr;       /* Number of COPY/DELETE/INSERT triples to process */
  unsigned int mxr;      /* Maximum value for r */
  unsigned int na, nb;   /* Number of lines shown from A and B */
  unsigned int i, j;     /* Loop counters */
  unsigned int m, ma, mb;/* Number of lines to output */
  signed int skip;   /* Number of lines to skip */
  unsigned int contextLines; /* Lines of context above and below each change */
  unsigned short passNumber = 0;
  int rc = 0;
  
#define RC if(rc) goto end
#define METRIC(M) if(1==passNumber) ++pBuilder->metrics.M
  pass_again:
  contextLines = diff_opt_context_lines(pOpt);
  skip = 0;
  a = b = 0;
  A = cx->aFrom;
  B = cx->aTo;
  R = cx->aEdit;
  mxr = cx->nEdit;
  //MARKER(("contextLines=%u, nEdit = %d, mxr=%u\n", contextLines, cx->nEdit, mxr));
  while( mxr>2 && R[mxr-1]==0 && R[mxr-2]==0 ){ mxr -= 3; }

  pBuilder->lnLHS = pBuilder->lnRHS = 0;
  ++passNumber;
  if(pBuilder->start){
    pBuilder->passNumber = passNumber;
    rc = pBuilder->start(pBuilder);
    RC;
  }
  for(r=0; r<mxr; r += 3*nr){
    /* Figure out how many triples to show in a single block */
    for(nr=1; R[r+nr*3]>0 && R[r+nr*3]<(int)contextLines*2; nr++){}

#if 0
    /* MISSING: this "should" be replaced by a stateful predicate
       function, probably in the fsl_dibu_opt class. */
    /* If there is a regex, skip this block (generate no diff output)
    ** if the regex matches or does not match both insert and delete.
    ** Only display the block if one side matches but the other side does
    ** not.
    */
    if( pOpt->pRe ){
      int hideBlock = 1;
      int xa = a, xb = b;
      for(i=0; hideBlock && i<nr; i++){
        int c1, c2;
        xa += R[r+i*3];
        xb += R[r+i*3];
        c1 = re_dline_match(pOpt->pRe, &A[xa], R[r+i*3+1]);
        c2 = re_dline_match(pOpt->pRe, &B[xb], R[r+i*3+2]);
        hideBlock = c1==c2;
        xa += R[r+i*3+1];
        xb += R[r+i*3+2];
      }
      if( hideBlock ){
        a = xa;
        b = xb;
        continue;
      }
    }
#endif

    /* Figure out how many lines of A and B are to be displayed
    ** for this change block.
    */
    if( R[r]>(int)contextLines ){
      na = nb = contextLines;
      skip = R[r] - contextLines;
    }else{
      na = nb = R[r];
      skip = 0;
    }
    for(i=0; i<nr; i++){
      na += R[r+i*3+1];
      nb += R[r+i*3+2];
    }
    if( R[r+nr*3]>(int)contextLines ){
      na += contextLines;
      nb += contextLines;
    }else{
      na += R[r+nr*3];
      nb += R[r+nr*3];
    }
    for(i=1; i<nr; i++){
      na += R[r+i*3];
      nb += R[r+i*3];
    }

    //MARKER(("Chunk header... a=%u, b=%u, na=%u, nb=%u, skip=%d\n", a, b, na, nb, skip));
    if(pBuilder->chunkHeader
       /* The following bit is a kludge to keep from injecting a chunk
          header between chunks which are directly adjacent.

          The problem, however, is that we cannot skip _reliably_
          without also knowing how the next chunk aligns. If we skip
          it here, the _previous_ chunk header may well be telling
          the user a lie with regards to line numbers.

          Fossil itself does not have this issue because it generates
          these chunk headers directly in this routine, instead of in
          the diff builder, depending on a specific flag being set in
          builder->opt. Also (related), in that implementation, fossil
          will collapse chunks which are separated by less than the
          context distance into contiguous chunks (see below). Because
          we farm out the chunkHeader lines to the builder, we cannot
          reliably do that here.
       */
#if 0
       && !skip
#endif
       ){
      rc = pBuilder->chunkHeader(pBuilder,
                                 (uint32_t)(na ? a+skip+1 : a+skip),
                                 (uint32_t)na,
                                 (uint32_t)(nb ? b+skip+1 : b+skip),
                                 (uint32_t)nb);
      RC;
    }

    /* Show the initial common area */
    a += skip;
    b += skip;
    m = R[r] - skip;
    if( r ) skip -= contextLines;

    //MARKER(("Show the initial common... a=%u, b=%u, m=%u, r=%u, skip=%d\n", a, b, m, r, skip));
    if( skip>0 ){
      if( NULL==pBuilder->chunkHeader && skip<(int)contextLines ){
        /* 2021-09-27: BUG: this is incompatible with unified diff
           format. The generated header lines say we're skipping X
           lines but we then end up including lines which that header
           says to skip. As a workaround, we'll only run this when
           pBuilder->chunkHeader is NULL, noting that fossil's diff
           builder interface does not have that method (and thus
           doesn't have this issue, instead generating chunk headers
           directly in this algorithm).

           Without this block, our "utxt" diff builder can mimic
           fossil's non-diff builder unified diff format, except that
           we add Index lines (feature or bug?). With this block,
           the header values output above are wrong.
        */
        /* If the amount to skip is less that the context band, then
        ** go ahead and show the skip band as it is not worth eliding */
        //MARKER(("skip %d < contextLines %d\n", skip, contextLines));
        /* from fossil(1) from formatDiff() */
        for(j=0; 0==rc && j<(unsigned)skip; j++){
          //MARKER(("(A) COMMON\n"));
          rc = pBuilder->common(pBuilder, &A[a+j-skip]);
        }
      }else{
        rc = pBuilder->skip(pBuilder, skip);
      }
      RC;
    }
    for(j=0; 0==rc && j<m; j++){
      //MARKER(("(B) COMMON\n"));
      rc = pBuilder->common(pBuilder, &A[a+j]);
    }
    RC;
    a += m;
    b += m;
    //MARKER(("Show the differences... a=%d, b=%d, m=%d\n", a, b, m));

    /* Show the differences */
    for(i=0; i<nr; i++){
      unsigned int nAlign;
      unsigned char *alignment = 0;
      ma = R[r+i*3+1];   /* Lines on left but not on right */
      mb = R[r+i*3+2];   /* Lines on right but not on left */

#if FSL_DIFF_SMALL_GAP
  /* Try merging the current block with subsequent blocks, if the
      ** subsequent blocks are nearby and their result isn't too big.
      */
      while( i<nr-1 && smallGap2(&R[r+i*3],ma,mb) ){
        i++;
        m = R[r+i*3];
        ma += R[r+i*3+1] + m;
        mb += R[r+i*3+2] + m;
      }
#endif

      /* Try to find an alignment for the lines within this one block */
      rc = diffBlockAlignment(&A[a], ma, &B[b], mb, pOpt,
                              &alignment, &nAlign);
      RC;
      for(j=0; ma+mb>0; j++){
        assert( j<nAlign );
        switch( alignment[j] ){
          case 1: {
            /* Delete one line from the left */
            METRIC(deletions);
            rc = pBuilder->deletion(pBuilder, &A[a]);
            if(rc) goto bail;
            ma--;
            a++;
            break;
          }
          case 2: {
            /* Insert one line on the right */
            METRIC(insertions);
            rc = pBuilder->insertion(pBuilder, &B[b]);
            if(rc) goto bail;
            assert( mb>0 );
            mb--;
            b++;
            break;
          }
          case 3: {
            /* The left line is changed into the right line */
            if( 0==cx->cmpLine(&A[a], &B[b]) ){
              rc = pBuilder->common(pBuilder, &A[a]);
            }else{
              METRIC(edits);
              rc = pBuilder->edit(pBuilder, &A[a], &B[b]);
            }
            if(rc) goto bail;
            assert( ma>0 && mb>0 );
            ma--;
            mb--;
            a++;
            b++;
            break;
          }
          case 4: {
            /* Delete from left then separately insert on the right */
            METRIC(replacements);
            rc = pBuilder->replacement(pBuilder, &A[a], &B[b]);
            if(rc) goto bail;
            ma--;
            a++;
            mb--;
            b++;
            break;
          }
        }
      }
      assert( nAlign==j );
      fsl_free(alignment);
      if( i<nr-1 ){
        m = R[r+i*3+3];
        for(j=0; 0==rc && j<m; j++){
          //MARKER(("D common\n"));
          rc = pBuilder->common(pBuilder, &A[a+j]);
        }
        RC;
        b += m;
        a += m;
      }
      continue;
      bail:
      assert(rc);
      fsl_free(alignment);
      goto end;
    }

    /* Show the final common area */
    assert( nr==i );
    m = R[r+nr*3];
    if( m>contextLines ) m = contextLines;
    for(j=0; 0==rc && j<m && j<contextLines; j++){
      //MARKER(("E common\n"));
      rc = pBuilder->common(pBuilder, &A[a+j]);
    }
    RC;
  }
  if( R[r]>(int)contextLines ){
    rc = pBuilder->skip(pBuilder, R[r] - contextLines);
  }
  end:
#undef RC
#undef METRIC
  if(0==rc){
    if(pBuilder->finish) pBuilder->finish(pBuilder);
    if(pBuilder->twoPass && 1==passNumber){
      goto pass_again;
    }
  }
  return rc;
}

/* MISSING(?) fossil(1) converts the diff inputs into utf8 with no
   BOM. Whether we really want to do that here or rely on the caller
   to is up for debate. If we do it here, we have to make the inputs
   non-const, which seems "wrong" for a library API. */
#define blob_to_utf8_no_bom(A,B) (void)0

/**
   Performs a diff of version 1 (pA) and version 2 (pB). ONE of
   pBuilder or outRaw must be non-NULL. If pBuilder is not NULL, all
   output for the diff is emitted via pBuilder. If outRaw is not NULL
   then on success *outRaw is set to the array of diff triples,
   transfering ownership to the caller, who must eventually fsl_free()
   it. On error, *outRaw is not modified but pBuilder may have emitted
   partial output. That is not knowable for the general
   case. Ownership of pBuilder is not changed. If pBuilder is not NULL
   then pBuilder->opt must be non-NULL.
*/
static int fsl_diff2_text_impl(fsl_buffer const *pA,
                               fsl_buffer const *pB,
                               fsl_dibu * const pBuilder,
                               fsl_dibu_opt const * const opt_,
                               int ** outRaw){
  int rc = 0;
  fsl__diff_cx c = fsl__diff_cx_empty;
  bool ignoreWs = false;
  int ansiOptCount = 0;
  fsl_dibu_opt opt = *opt_
    /*we need a copy for the sake of the FSL_DIFF2_INVERT flag*/;
  if(!pA || !pB || (pBuilder && outRaw)) return FSL_RC_MISUSE;

  blob_to_utf8_no_bom(pA, 0);
  blob_to_utf8_no_bom(pB, 0);

  if( opt.diffFlags & FSL_DIFF2_INVERT ){
    char const * z;
    fsl_buffer const *pTemp = pA;
    pA = pB;
    pB = pTemp;
    z = opt.hashRHS; opt.hashRHS = opt.hashLHS; opt.hashLHS = z;
    z = opt.nameRHS; opt.nameRHS = opt.nameLHS; opt.nameLHS = z;
  }
#define AOPT(OPT) \
  if(opt.ansiColor.OPT) ansiOptCount += (*opt.ansiColor.OPT) ? 1 : 0; \
  else opt.ansiColor.OPT = ""
  AOPT(insertion);
  AOPT(edit);
  AOPT(deletion);
#undef AOPT
  if(0==ansiOptCount){
    opt.ansiColor.reset = "";
  }else if(!opt.ansiColor.reset || !*opt.ansiColor.reset){
    opt.ansiColor.reset = "\x1b[0m";
  }
  ignoreWs = (opt.diffFlags & FSL_DIFF2_IGNORE_ALLWS)!=0;
  if(FSL_DIFF2_IGNORE_ALLWS==(opt.diffFlags & FSL_DIFF2_IGNORE_ALLWS)){
    c.cmpLine = fsl_dline_cmp_ignore_ws;
  }else{
    c.cmpLine = fsl_dline_cmp;
  }
 
  rc = fsl_break_into_dlines(fsl_buffer_cstr(pA), (fsl_int_t)pA->used,
                             (uint32_t*)&c.nFrom, &c.aFrom, opt.diffFlags);
  if(rc) goto end;
  rc = fsl_break_into_dlines(fsl_buffer_cstr(pB), (fsl_int_t)pB->used,
                             (uint32_t*)&c.nTo, &c.aTo, opt.diffFlags);
  if(rc) goto end;

  /* Compute the difference */
  rc = fsl__diff_all(&c);
  if(rc) goto end;
  if( ignoreWs && c.nEdit==6 && c.aEdit[1]==0 && c.aEdit[2]==0 ){
    rc = FSL_RC_DIFF_WS_ONLY;
    goto end;
  }
  if( (opt.diffFlags & FSL_DIFF2_NOTTOOBIG)!=0 ){
    int i, m, n;
    int const * const a = c.aEdit;
    int const mx = c.nEdit;
    for(i=m=n=0; i<mx; i+=3){ m += a[i]; n += a[i+1]+a[i+2]; }
    if( !n || n>10000 ){
      rc = FSL_RC_RANGE;
      /* diff_errmsg(pOut, DIFF_TOO_MANY_CHANGES, diffFlags); */
      goto end;
    }
  }
  //fsl__dump_triples(&c, __FILE__, __LINE__);
  if( (opt.diffFlags & FSL_DIFF2_NOOPT)==0 ){
    fsl__diff_optimize(&c);
  }
  //fsl__dump_triples(&c, __FILE__, __LINE__);

  /**
     Reference:

     https://fossil-scm.org/home/file?ci=cae7036bb7f07c1b&name=src/diff.c&ln=2749-2804

     Noting that:

     - That function's return value is this one's *outRaw

     - DIFF_NUMSTAT flag is not implemented. For that matter,
     flags which result in output going anywhere except for
     pBuilder->out are not implemented here, e.g. DIFF_RAW.

     That last point makes this impl tiny compared to the original!
  */
  if(pBuilder){
    fsl_dibu_opt const * oldOpt = pBuilder->opt;
    pBuilder->opt = &opt;
    rc = fdb__format(&c, pBuilder);
    pBuilder->opt = oldOpt;
  }
  end:
  if(0==rc && outRaw){
    *outRaw = c.aEdit;
    c.aEdit = 0;
  }
  fsl__diff_cx_clean(&c);
  return rc;
}

int fsl_diff_v2(fsl_buffer const * pv1,
                fsl_buffer const * pv2,
                fsl_dibu * const pBuilder){
  return fsl_diff2_text_impl(pv1, pv2, pBuilder, pBuilder->opt, NULL);
}

int fsl_diff_v2_raw(fsl_buffer const * pv1,
                    fsl_buffer const * pv2,
                    fsl_dibu_opt const * const opt,
                    int **outRaw ){
  return fsl_diff2_text_impl(pv1, pv2, NULL,
                             opt ? opt : &fsl_dibu_opt_empty,
                             outRaw);
}


#undef DIFF_ALIGN_MX
#undef blob_to_utf8_no_bom
#undef FSL_DIFF_SMALL_GAP
#undef diff_isspace
#undef LENGTH
/* end of file ./src/diff.c */
/* start of file ./src/encode.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/
/**************************************************************************
  This file houses some encoding/decoding API routines.
*/
#include <assert.h>

/* Only for debugging */
#include <stdio.h>

/*
   An array for translating single base-16 characters into a value.
   Disallowed input characters have a value of 64.
*/
static const char zDecode[] = {
  64, 64, 64, 64, 64, 64, 64, 64,  64, 64, 64, 64, 64, 64, 64, 64,
  64, 64, 64, 64, 64, 64, 64, 64,  64, 64, 64, 64, 64, 64, 64, 64,
  64, 64, 64, 64, 64, 64, 64, 64,  64, 64, 64, 64, 64, 64, 64, 64,
   0,  1,  2,  3,  4,  5,  6,  7,   8,  9, 64, 64, 64, 64, 64, 64,
  64, 10, 11, 12, 13, 14, 15, 64,  64, 64, 64, 64, 64, 64, 64, 64,
  64, 64, 64, 64, 64, 64, 64, 64,  64, 64, 64, 64, 64, 64, 64, 64,
  64, 10, 11, 12, 13, 14, 15, 64,  64, 64, 64, 64, 64, 64, 64, 64,
  64, 64, 64, 64, 64, 64, 64, 64,  64, 64, 64, 64, 64, 64, 64, 64,
  64, 64, 64, 64, 64, 64, 64, 64,  64, 64, 64, 64, 64, 64, 64, 64,
  64, 64, 64, 64, 64, 64, 64, 64,  64, 64, 64, 64, 64, 64, 64, 64,
  64, 64, 64, 64, 64, 64, 64, 64,  64, 64, 64, 64, 64, 64, 64, 64,
  64, 64, 64, 64, 64, 64, 64, 64,  64, 64, 64, 64, 64, 64, 64, 64,
  64, 64, 64, 64, 64, 64, 64, 64,  64, 64, 64, 64, 64, 64, 64, 64,
  64, 64, 64, 64, 64, 64, 64, 64,  64, 64, 64, 64, 64, 64, 64, 64,
  64, 64, 64, 64, 64, 64, 64, 64,  64, 64, 64, 64, 64, 64, 64, 64,
  64, 64, 64, 64, 64, 64, 64, 64,  64, 64, 64, 64, 64, 64, 64, 64
};

int fsl_decode16(const unsigned char *zIn, unsigned char *pOut,
                 fsl_size_t N){
  fsl_int_t i, j;
  if( (N&1)!=0 ) return FSL_RC_RANGE;
  for(i=j=0; i<(fsl_int_t)N; i += 2, j++){
    fsl_int_t v1, v2, a;
    a = zIn[i];
    if( (a & 0x80)!=0 || (v1 = zDecode[a])==64 ) return FSL_RC_RANGE;
    a = zIn[i+1];
    if( (a & 0x80)!=0 || (v2 = zDecode[a])==64 ) return FSL_RC_RANGE;
    pOut[j] = (v1<<4) + v2;
  }
  return 0;
}


bool fsl_validate16(const char *zIn, fsl_size_t nIn){
  fsl_size_t i;
  for(i=0; i<nIn; i++, zIn++){
    if( zDecode[zIn[0]&0xff]>63 ){
      return zIn[0]==0 ? true : false;
    }
  }
  return true;
}

/*
   The array used for encoding
*/                           /* 123456789 12345  */
static const char zEncode[] = "0123456789abcdef"; 

int fsl_encode16(const unsigned char *pIn, unsigned char *zOut, fsl_size_t N){
  fsl_size_t i;
  if(!pIn || !zOut) return FSL_RC_MISUSE;
  for(i=0; i<N; i++){
    *(zOut++) = zEncode[pIn[i]>>4];
    *(zOut++) = zEncode[pIn[i]&0xf];
  }
  *zOut = 0;
  return 0;
}

void fsl_canonical16(char *z, fsl_size_t n){
  while( *z && n-- ){
    *z = zEncode[zDecode[(*z)&0x7f]&0x1f];
    ++z;
  }
}

void fsl_bytes_defossilize( unsigned char * z, fsl_size_t * resultLen ){
  fsl_size_t i, j, c;
  for(i=0; (c=z[i])!=0 && c!='\\'; i++){}
  if( c==0 ) {
    if(resultLen) *resultLen = i;
    return;
  }
  for(j=i; (c=z[i])!=0; i++){
    if( c=='\\' && z[i+1] ){
      i++;
      switch( z[i] ){
        case 'n':  c = '\n';  break;
        case 's':  c = ' ';   break;
        case 't':  c = '\t';  break;
        case 'r':  c = '\r';  break;
        case 'v':  c = '\v';  break;
        case 'f':  c = '\f';  break;
        case '0':  c = 0;     break;
        case '\\': c = '\\';  break;
        default:   c = z[i];  break;
      }
    }
    z[j++] = c;
  }
  if( z[j] ) z[j] = 0;
  if(resultLen) *resultLen = j;
}

int fsl_bytes_fossilize( unsigned char const * inp,
                         fsl_int_t nIn,
                         fsl_buffer * out ){
  fsl_size_t n, i, j, c;
  unsigned char *zOut;
  int rc;
  fsl_size_t oldUsed;
  fsl_size_t inSz;
  if(!inp || !out) return FSL_RC_MISUSE;
  else if( inp && (nIn<0) ) nIn = (fsl_int_t)fsl_strlen((char const *)inp);
  out->used = 0;
  if(!nIn) return 0;
  inSz = (fsl_size_t)nIn;
  /* Figure out how much space we'll need... */
  for(i=n=0; i<inSz; ++i){
    c = inp[i];
    if( c==0 || c==' ' || c=='\n' || c=='\t' || c=='\r' || c=='\f' || c=='\v'
        || c=='\\') ++n;
  }
  /* Reserve memory... */
  n += nIn;
  oldUsed = out->used;
  rc = fsl_buffer_reserve( out, oldUsed + (fsl_size_t)(n+1));
  if(rc) return rc;
  zOut = out->mem + oldUsed;
  /* Encode it... */
  for(i=j=0; i<(fsl_size_t)nIn; i++){
    unsigned char c = (unsigned char)inp[i];
    if( c==0 ){
      zOut[j++] = '\\';
      zOut[j++] = '0';
    }else if( c=='\\' ){
      zOut[j++] = '\\';
      zOut[j++] = '\\';
    }else if( fsl_isspace(c) ){
      zOut[j++] = '\\';
      switch( c ){
        case '\n':  c = 'n'; break;
        case ' ':   c = 's'; break;
        case '\t':  c = 't'; break;
        case '\r':  c = 'r'; break;
        case '\v':  c = 'v'; break;
        case '\f':  c = 'f'; break;
      }
      zOut[j++] = c;
    }else{
      zOut[j++] = c;
    }
  }
  zOut[j] = 0;
  out->used += j;
  return 0;
}


fsl_size_t fsl_str_to_size(char const * str){
  fsl_size_t size, oldsize, c;
  if(!str) return -1;
  for(oldsize=size=0; (c = str[0])>='0' && c<='9'; str++){
    size = oldsize*10 + c - '0';
    if( size<oldsize ) return -1;
    oldsize = size;
  }
  return size;
}

fsl_int_t fsl_str_to_int(char const * str, fsl_int_t dflt){
  fsl_size_t size, oldsize
    /* We use fsl_size_t for the calculation
       so that we can detect overflow (which is undefined
       for signed types).
    */;
  char c;
  fsl_int_t mult = 1;
  fsl_int_t rc;
  if(!str) return dflt;
  else switch(*str){
    case '+': ++str; break;
    case '-': ++str; mult = -1; break;
  };
  for(oldsize=size=0; (c = str[0])>='0' && c<='9'; str++){
    size = oldsize*10 + c - '0';
    if( size<oldsize ) /* overflow */ return dflt;
    oldsize = size;
  }
  rc = (fsl_int_t)size;
  return ((fsl_size_t)rc == size)
    ? (rc * mult)
    : dflt /* result is too big */;
}

fsl_size_t fsl_htmlize_xlate(int c, char const ** xlate){
  switch( c ){
    case '<': *xlate = "&lt;"; return 4;
    case '>': *xlate = "&gt;"; return 4;
    case '&': *xlate = "&amp;";  return 5;
    case '"': *xlate = "&quot;";  return 6;
    default: *xlate = NULL; return 1;
  }
}

int fsl_htmlize(fsl_output_f out, void * oState,
                const char *zIn, fsl_int_t n){
  int rc = 0;
  int c, i, j, len;
  char const * xlate;
  if(!out || !zIn) return FSL_RC_MISUSE;
  else if( n<0 ) n = fsl_strlen(zIn);
  for(i=j=0; !rc && (i<n); ++i){
    c = zIn[i];
    len = fsl_htmlize_xlate(c, &xlate);
    if(len>1){
      if( j<i ) rc = out(oState, zIn+j, i-j);
      if(!rc) rc = out(oState, xlate, len);
      j = i+1;
    }
  }
  if( !rc && j<i ) rc = out(oState, zIn+j, i-j);
  return rc;
}


int fsl_htmlize_to_buffer(fsl_buffer *p, const char *zIn, fsl_int_t n){
  int rc = 0;
  int c;
  fsl_int_t i = 0;
  fsl_size_t count = 0;
  char const * xl = NULL;
  if(!p || !zIn) return FSL_RC_MISUSE;
  else if( n<0 ) n = fsl_strlen(zIn);
  if(0==n) return 0;
  /* Count how many bytes we need, to avoid reallocs and the
     associated error checking... */
  for( ; i<n && (c = zIn[i])!=0; ++i ){
    count += fsl_htmlize_xlate(c, &xl);
  }
  if(count){
    rc = fsl_buffer_reserve(p, p->used + count + 1);
    if(!rc){
      /* Now none of the fsl_buffer_append()s can fail. */
      rc = fsl_htmlize(fsl_output_f_buffer, p, zIn, n);
    }
  }
  return rc;
}

char *fsl_htmlize_str(const char *zIn, fsl_int_t n){
  int rc;
  fsl_buffer b = fsl_buffer_empty;
  rc = fsl_htmlize_to_buffer(&b, zIn, n);
  if(!rc){
    return (char *)b.mem /* transfer ownership */;
  }else{
    fsl_buffer_clear(&b);
    return NULL;
  }
}

/* end of file ./src/encode.c */
/* start of file ./src/event.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/
/************************************************************************
  This file implements technote (formerly known as event)-related
  parts of the library.
*/
#include <assert.h>


/**
   Fetches all "technote" (formerly "event") IDs from the repository
   and appends each one to the given list in the form of a
   (`char*`). This function relies on the `event-` tag prefix being
   reserved for technotes and that the technote IDS are all exactly 40
   bytes long.
   
   Returns 0 on success, FSL_RC_NOT_A_REPO if f has no repository db
   opened, FSL_RC_OOM if allocation of a new list entry fails, or
   propagates db-related code on any other error. Results are
   undefined if either argument is NULL.

   TODO? Reformulate this to be like fsl_tkt_id_to_rids(), returning
   the list as RIDs?
*/
/*FSL_EXPORT*/ int fsl_technote_ids_get(fsl_cx * const f, fsl_list * const tgt );

int fsl_technote_ids_get( fsl_cx * const f, fsl_list * const tgt ){
  fsl_db * const db = fsl_needs_repo(f);
  if(!db) return FSL_RC_NOT_A_REPO;
  int rc = fsl_db_select_slist( db, tgt,
                                "SELECT substr(tagname,7) AS n "
                                "FROM tag "
                                "WHERE tagname GLOB 'event-*' "
                                "AND length(tagname)=46 "
                                "ORDER BY n");
  if(rc && db->error.code && !f->error.code){
    fsl_cx_uplift_db_error(f, db);
  }
  return rc;
}
/* end of file ./src/event.c */
/* start of file ./src/foci.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=2 et sw=2 tw=80: */
/*
 * Copyright 2022 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt
 *
 * SPDX-License-Identifier: BSD-2-Clause-FreeBSD
 * SPDX-FileCopyrightText: 2021 The Libfossil Authors
 * SPDX-ArtifactOfProjectName: Libfossil
 * SPDX-FileType: Code
 *
 * Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
 */

/*
 * This file implements the files-of-checkin (foci) API used to construct a
 * SQLite3 virtual table via a table-valued function to aggregate all files
 * pertaining to a specific check-in. This table is used in repository
 * queries such as listing all files belonging to a specific version.
 *
 * Usage (from fossil(1) /src/foci.c:24):
 *
 *    SELECT * FROM fsl_foci('trunk');
 *
 * temp.foci table schema:
 *
 *     CREATE TABLE fsl_foci(
 *       checkinID    INTEGER,    -- RID for the check-in manifest
 *       filename     TEXT,       -- Name of a file
 *       uuid         TEXT,       -- hash of the file
 *       previousName TEXT,       -- Name of the file in previous check-in
 *       perm         TEXT,       -- Permissions on the file
 *       symname      TEXT HIDDEN -- Symbolic name of the check-in.
 *     );
 *
 * The hidden symname column is (optionally) used as a query parameter to
 * identify the particular check-in to parse.  The checkinID parameter
 * (such is a unique numeric RID rather than symbolic name) can also be used
 * to identify the check-in.  Example:
 *
 *    SELECT * FROM fsl_foci
 *     WHERE checkinID=fsl_sym2rid('trunk');
 *
 */
#include <string.h>/*memset()*/
#include <assert.h>

/* Only for debugging */
#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)

enum {
FOCI_CHECKINID = 0,
FOCI_FILENAME = 1,
FOCI_UUID = 2,
FOCI_PREVNAME = 3,
FOCI_PERM = 4,
FOCI_SYMNAME = 5
};

typedef struct FociCursor FociCursor;
struct FociCursor {
  sqlite3_vtab_cursor base; /* Base class - must be first */
  fsl_deck d;           /* Current manifest */
  const fsl_card_F *cf;  /* Current file */
  int idx;                /* File index */
};

typedef struct FociTable FociTable;
struct FociTable {
  sqlite3_vtab base;        /* Base class - must be first */
  fsl_cx * f;               /* libfossil context */
};

/*
 * The schema for the virtual table:
 */
static const char zFociSchema[] =
  " CREATE TABLE fsl_foci("
  "  checkinID    INTEGER,    -- RID for the check-in manifest\n"
  "  filename     TEXT,       -- Name of a file\n"
  "  uuid         TEXT,       -- hash of the file\n"
  "  previousName TEXT,       -- Name of the file in previous check-in\n"
  "  perm         TEXT,       -- Permissions on the file\n"
  "  symname      TEXT HIDDEN -- Symbolic name of the check-in\n"
  " );";

/*
 * Connect to or create a foci virtual table.
 */
static int fociConnect(
  sqlite3 *db,
  void *pAux /*a (fsl_cx*) */,
  int argc __unused,
  const char * const * argv __unused,
  sqlite3_vtab **ppVtab,
  char **pzErr __unused
){
  FociTable *pTab;
  int rc = SQLITE_OK;

  pTab = (FociTable *)sqlite3_malloc(sizeof(FociTable));
  if( !pTab ){
    return SQLITE_NOMEM;
  }
  memset(pTab, 0, sizeof(FociTable));
  rc = sqlite3_declare_vtab(db, zFociSchema);
  if( rc==SQLITE_OK ){
    pTab->f = (fsl_cx*)pAux;
    *ppVtab = &pTab->base;
  }
  return rc;
}

/*
 * Disconnect from or destroy a focivfs virtual table.
 */
static int fociDisconnect(sqlite3_vtab *pVtab){
  sqlite3_free(pVtab);
  return SQLITE_OK;
}

/*
 * Available scan methods:
 *
 *   (0)     A full scan.  Visit every manifest in the repo.  (Slow)
 *   (1)     checkinID=?.  visit only the single manifest specified.
 *   (2)     symName=?     visit only the single manifest specified.
 */
static int fociBestIndex(sqlite3_vtab *tab __unused, sqlite3_index_info *pIdxInfo){
  int i;
  pIdxInfo->estimatedCost = 1000000000.0;
  for( i=0; i<pIdxInfo->nConstraint; i++ ){
    if( !pIdxInfo->aConstraint[i].usable ) continue;
    if( pIdxInfo->aConstraint[i].op==SQLITE_INDEX_CONSTRAINT_EQ
     && (pIdxInfo->aConstraint[i].iColumn==FOCI_CHECKINID
            || pIdxInfo->aConstraint[i].iColumn==FOCI_SYMNAME)
    ){
      if( pIdxInfo->aConstraint[i].iColumn==FOCI_CHECKINID ){
        pIdxInfo->idxNum = 1;
      }else{
        pIdxInfo->idxNum = 2;
      }
      pIdxInfo->estimatedCost = 1.0;
      pIdxInfo->aConstraintUsage[i].argvIndex = 1;
      pIdxInfo->aConstraintUsage[i].omit = 1;
      break;
    }
  }
  return SQLITE_OK;
}

/*
 * Open a new focivfs cursor.
 */
static int fociOpen(sqlite3_vtab *pVTab, sqlite3_vtab_cursor **ppCursor){
  FociCursor *pCsr;
  pCsr = (FociCursor *)sqlite3_malloc(sizeof(FociCursor));
  if( !pCsr ){
    return SQLITE_NOMEM;
  }
  memset(pCsr, 0, sizeof(FociCursor));
  pCsr->d = fsl_deck_empty;
  pCsr->base.pVtab = pVTab;
  *ppCursor = (sqlite3_vtab_cursor *)pCsr;
  return SQLITE_OK;
}

/*
 * Close a focivfs cursor.
 */
static int fociClose(sqlite3_vtab_cursor *pCursor){
  FociCursor *pCsr = (FociCursor *)pCursor;
  fsl_deck_finalize(&pCsr->d);
  sqlite3_free(pCsr);
  return SQLITE_OK;
}

/*
 * Move a focivfs cursor to the next F card entry in the deck. If this fails,
 * pass the vtab cursor to fociClose and return the failing result code.
 */
static int fociNext(sqlite3_vtab_cursor *pCursor){
  int rc = SQLITE_OK;

  FociCursor *pCsr = (FociCursor *)pCursor;
  rc = fsl_deck_F_next(&pCsr->d, &pCsr->cf);
  if( !rc ){
    pCsr->idx++;
  }else{
    fociClose(pCursor);
  }
  return rc;
}

static int fociEof(sqlite3_vtab_cursor *pCursor){
  FociCursor *pCsr = (FociCursor *)pCursor;
  return pCsr->cf==0;
}

static int fociFilter(
  sqlite3_vtab_cursor *pCursor,
  int idxNum, const char *idxStr __unused,
  int argc __unused, sqlite3_value **argv
){
  int rc = SQLITE_OK;
  FociCursor *const pCur = (FociCursor *)pCursor;
  fsl_cx * const f = ((FociTable*)pCur->base.pVtab)->f;

  fsl_deck_finalize(&pCur->d);
  if( idxNum ){
    fsl_id_t rid;
    if( idxNum==1 ){
      rid = sqlite3_value_int(argv[0]);
    }else{
      rc = fsl_sym_to_rid(f, (const char *)sqlite3_value_text(argv[0]),
       FSL_SATYPE_CHECKIN, &rid);
      if( rc ){
        goto end;
      }
    }
    rc = fsl_deck_load_rid(f, &pCur->d, rid, FSL_SATYPE_CHECKIN);
    if( rc ){
      goto end;
    }
    if( pCur->d.rid ){
      rc = fsl_deck_F_rewind(&pCur->d);
      if( !rc ){
        rc = fsl_deck_F_next(&pCur->d, &pCur->cf);
      }
      if( rc ){
        goto end;
      }
    }
  }
  pCur->idx = 0;
end:
  if( rc ){
    fsl_deck_finalize(&pCur->d);
  }
  return rc;
}

static int fociColumn(
  sqlite3_vtab_cursor *pCursor,
  sqlite3_context *ctx,
  int i
){
  FociCursor *pCsr = (FociCursor *)pCursor;
  switch( i ){
    case FOCI_CHECKINID:
      sqlite3_result_int(ctx, pCsr->d.rid);
      break;
    case FOCI_FILENAME:
      sqlite3_result_text(ctx, pCsr->cf->name, -1, SQLITE_TRANSIENT);
      break;
    case FOCI_UUID:
      sqlite3_result_text(ctx, pCsr->cf->uuid, -1, SQLITE_TRANSIENT);
      break;
    case FOCI_PREVNAME:
      sqlite3_result_text(ctx, pCsr->cf->priorName, -1, SQLITE_TRANSIENT);
      break;
    case FOCI_PERM: {
      char *perm[3] = {"l", "w", "x"};
      int i = 1;
      switch( pCsr->cf->perm ){
        case FSL_FILE_PERM_LINK:
          i = 0; break;
        case FSL_FILE_PERM_EXE:
          i = 2; break;
        default:
          break;
      }
      sqlite3_result_text(ctx, perm[i], 1, SQLITE_TRANSIENT);
      break;
    }
    case FOCI_SYMNAME:
      break;
  }
  return SQLITE_OK;
}

static int fociRowid(sqlite3_vtab_cursor *pCursor, sqlite_int64 *pRowid){
  FociCursor *pCsr = (FociCursor *)pCursor;
  *pRowid = pCsr->idx;
  return SQLITE_OK;
}

int fsl__foci_register(fsl_db * const db){
  static sqlite3_module foci_module = {
    0,                            /* iVersion */
    fociConnect,                  /* xCreate */
    fociConnect,                  /* xConnect */
    fociBestIndex,                /* xBestIndex */
    fociDisconnect,               /* xDisconnect */
    fociDisconnect,               /* xDestroy */
    fociOpen,                     /* xOpen - open a cursor */
    fociClose,                    /* xClose - close a cursor */
    fociFilter,                   /* xFilter - configure scan constraints */
    fociNext,                     /* xNext - advance a cursor */
    fociEof,                      /* xEof - check for end of scan */
    fociColumn,                   /* xColumn - read data */
    fociRowid,                    /* xRowid - read data */
    0,                            /* xUpdate */
    0,                            /* xBegin */
    0,                            /* xSync */
    0,                            /* xCommit */
    0,                            /* xRollback */
    0,                            /* xFindMethod */
    0,                            /* xRename */
    0,                            /* xSavepoint */
    0,                            /* xRelease */
    0,                            /* xRollbackTo */
    0                             /* xShadowName */
  };
  assert(db->f);
  int rc = sqlite3_create_module(db->dbh, "fsl_foci",
                                 &foci_module, db->f);
  return fsl__db_errcode(db, rc);
}

#undef MARKER
/* end of file ./src/foci.c */
/* start of file ./src/fs.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/
#ifdef _WIN32
# undef __STRICT_ANSI__ /* Needed for _wfopen */
#endif

#include <assert.h>
#include <string.h> /* strlen() */
#include <stddef.h> /* NULL on linux */
#include <ctype.h>
#include <errno.h>
#if FSL_PLATFORM_IS_WINDOWS
# if !defined(ELOOP)
#  define ELOOP 114 /* Missing in MinGW */
# endif
#else
# include <unistd.h> /* access(2), readlink(2) */
# include <sys/types.h>
# include <sys/time.h>
#endif
#include <sys/stat.h>

const fsl_path_splitter fsl_path_splitter_empty = fsl_path_splitter_empty_m;

/* Only for debugging */
#include <stdio.h>
#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)

FILE *fsl_fopen(const char *zName, const char *zMode){
  FILE *f;
  if(zName && ('-'==*zName && !zName[1])){
    f = (strchr(zMode, 'w') || strchr(zMode,'+'))
      ? stdout
      : stdin
      ;
  }else{
#ifdef _WIN32
    wchar_t *uMode = (wchar_t *)fsl_utf8_to_unicode(zMode);
    wchar_t *uName = (wchar_t *)fsl_utf8_to_filename(zName);
    f = _wfopen(uName, uMode);
    fsl_filename_free(uName);
    fsl_unicode_free(uMode);
#else
    f = fopen(zName, zMode);
#endif
  }
  return f;
}


void fsl_fclose( FILE * f ){
  if(f && (stdin!=f) && (stdout!=f) && (stderr!=f)){
    fclose(f);
  }
}

/*
   Wrapper around the access() system call.
*/
int fsl_file_access(const char *zFilename, int flags){
  /* FIXME: port in fossil(1) win32_access() */
#ifdef _WIN32
  wchar_t *zMbcs = (wchar_t *)fsl_utf8_to_filename(zFilename);
#define ACC _waccess
#else
  char *zMbcs = (char*)fsl_utf8_to_filename(zFilename);
#define ACC access
#endif
  int rc = zMbcs ? ACC(zMbcs, flags) : FSL_RC_OOM;
  if(zMbcs) fsl_filename_free(zMbcs);
  return rc;
#undef ACC
}


int fsl_getcwd(char *zBuf, fsl_size_t nBuf, fsl_size_t * outLen){
#ifdef _WIN32
  /* FIXME: port in fossil(1) win32_getcwd() */
  char *zPwdUtf8;
  fsl_size_t nPwd;
  fsl_size_t i;
  wchar_t zPwd[2000];
  if(!zBuf) return FSL_RC_MISUSE;
  else if(!nBuf) return FSL_RC_RANGE;
  /*
    https://msdn.microsoft.com/en-us/library/windows/desktop/aa365247(v=vs.85).aspx

    It says:

    Note File I/O functions in the Windows API convert "/" to "\" as
    part of converting the name to an NT-style name, except when using
    the "\\?\" prefix as detailed in the following sections.

    So the path-demangling bits below might do more damage they
    fix?
  */
  else if( _wgetcwd(zPwd, sizeof(zPwd)/sizeof(zPwd[0])-1)==0 ){
    /* FIXME: how to determine if FSL_RC_RANGE is a better
       return value?
    */
    return FSL_RC_IO;
  }
  zPwdUtf8 = fsl_filename_to_utf8(zPwd);
  if(!zPwdUtf8) return FSL_RC_OOM;
  nPwd = strlen(zPwdUtf8);
  if( nPwd > nBuf-1 ){
    fsl_filename_free(zPwdUtf8);
    return FSL_RC_RANGE;
  }
  for(i=0; zPwdUtf8[i]; i++) if( zPwdUtf8[i]=='\\' ) zPwdUtf8[i] = '/';
  memcpy(zBuf, zPwdUtf8, nPwd+1);
  fsl_filename_free(zPwdUtf8);
  if(outLen) *outLen = nPwd;
  return 0;
#else
  if(!zBuf) return FSL_RC_MISUSE;
  else if(!nBuf) return FSL_RC_RANGE;
  else if( NULL==getcwd(zBuf,nBuf) ){
    return fsl_errno_to_rc(errno, FSL_RC_IO);
  }else{
    if(outLen) *outLen = fsl_strlen(zBuf);
    return 0;
  }
#endif
}

/*
   The file status information from the most recent stat() call.
  
   Use _stati64 rather than stat on windows, in order to handle files
   larger than 2GB.
*/
#if defined(_WIN32) && (defined(__MSVCRT__) || defined(_MSC_VER))
# undef stat
# define stat _stati64
#endif
/*
   On Windows S_ISLNK always returns FALSE.
*/
#if !defined(S_ISLNK)
# define S_ISLNK(x) (0)
#endif

/* Reminder: the semantics of the 3rd parameter are
   reversed from v1's fossil_stat().
*/
int fsl_stat(const char *zFilename, fsl_fstat * const fst,
             bool derefSymlinks){
  /* FIXME: port in fossil(1) win32_stat() */
  if(!zFilename) return FSL_RC_MISUSE;
  else if(!*zFilename) return FSL_RC_RANGE;
  else{
    int rc;
    struct stat buf;
#if !defined(_WIN32)
    char *zMbcs = (char *)fsl_utf8_to_filename(zFilename);
    if(!zMbcs) rc = FSL_RC_OOM;
    else{
      if( derefSymlinks ){
        rc = stat(zMbcs, &buf);
      }else{
        rc = lstat(zMbcs, &buf);
      }
    }
#else
    wchar_t *zMbcs = (wchar_t *)fsl_utf8_to_filename(zFilename);
    /*trailing pathseps are forbidden in Windows stat fxns, as per doc; sigh*/
    int nzmbcslen = wcslen ( zMbcs );
    while ( nzmbcslen > 0 && ( L'\\' == zMbcs[nzmbcslen-1] ||
        L'/' == zMbcs[nzmbcslen-1] ) ) {
      zMbcs[nzmbcslen-1] = 0;
      --nzmbcslen;
    }
    rc = zMbcs ? _wstati64(zMbcs, &buf) : FSL_RC_OOM;
#endif
    if(zMbcs) fsl_filename_free(zMbcs);
    if(fst && (0==rc)){
      *fst = fsl_fstat_empty;
      fst->ctime = (fsl_time_t)buf.st_ctime;
      fst->mtime = (fsl_time_t)buf.st_mtime;
      fst->size = (fsl_size_t)buf.st_size;
      if(S_ISDIR(buf.st_mode)) fst->type = FSL_FSTAT_TYPE_DIR;
#if !defined(_WIN32)
      else if(S_ISLNK(buf.st_mode)) fst->type = FSL_FSTAT_TYPE_LINK;
#endif
      else /* if(S_ISREG(buf.st_mode)) */{
        fst->type = FSL_FSTAT_TYPE_FILE;
#if defined(_WIN32)
#  ifndef S_IXUSR
#    define S_IXUSR  _S_IEXEC
#  endif
        if(((S_IXUSR)&buf.st_mode)!=0){
          fst->perm |= FSL_FSTAT_PERM_EXE;
        }
#else
        if( ((S_IXUSR|S_IXGRP|S_IXOTH)&buf.st_mode)!=0 ){
          fst->perm |= FSL_FSTAT_PERM_EXE;
        }
#if 0
        /* Porting artifact: something to consider... */
        else if( g.allowSymlinks && S_ISLNK(buf.st_mode) )
          return PERM_LNK;
#endif
#endif
      }
    }else if(rc){
      rc = fsl_errno_to_rc(errno, FSL_RC_IO);
    }
    return rc;
  }
}

fsl_int_t fsl_file_size(const char *zFilename){
  fsl_fstat fst;
  return ( 0 != fsl_stat(zFilename, &fst, 1) )
    ? -1
    : (fsl_int_t)fst.size;
}

/*
  The family of 'wd' functions is historical in nature and not really
  needed(???) at the library level. 'wd' == 'working directory'
  (i.e. checkout).  Ideally the library won't have to do any _direct_
  manipulation of directory trees, e.g. checkouts. That is essentially
  app-level logic, though we'll need some level of infrastructure for
  the apps to build off of.  When that comes, the "wd" family of
  functions (or something similar) might come back into play.
*/

fsl_time_t fsl_file_mtime(const char *zFilename){
  fsl_fstat fst;
  return ( 0 != fsl_stat(zFilename, &fst, 1) )
    ? -1
    : (fsl_time_t)fst.mtime;
}


bool fsl_is_file(const char *zFilename){
  fsl_fstat fst;
  return ( 0 != fsl_stat(zFilename, &fst, 1) )
    ? false
    : (FSL_FSTAT_TYPE_FILE == fst.type);
}

bool fsl_is_symlink(const char *zFilename){
#if FSL_PLATFORM_IS_WINDOWS
  if(zFilename){/*unused var*/}
  return false;
#else
  fsl_fstat fst;
  return (0 == fsl_stat(zFilename, &fst, 0))
    ? (FSL_FSTAT_TYPE_LINK == fst.type)
    : false;
#endif
}

/*
   Return true if zPath is an absolute pathname.  Return false
   if it is relative.
*/
bool fsl_is_absolute_path(const char *zPath){
  if( zPath && ((zPath[0]=='/')
#if defined(_WIN32) || defined(__CYGWIN__)
      || (zPath[0]=='\\')
      || (fsl_isalpha(zPath[0]) && zPath[1]==':'
          && (zPath[2]=='\\' || zPath[2]=='/'))
#endif
    )
  ){
    return 1;
  }else{
    return 0;
  }
}

bool fsl_is_simple_pathname(const char *z, bool bStrictUtf8){
  int i;
  unsigned char c = (unsigned char) z[0];
  char maskNonAscii = bStrictUtf8 ? 0x80 : 0x00;
  if( c=='/' || c==0 ) return 0;
  if( c=='.' ){ /* Common cases: ./ and ../ */
    if( z[1]=='/' || z[1]==0 ) return 0;
    if( z[1]=='.' && (z[2]=='/' || z[2]==0) ) return 0;
  }
  for(i=0; (c=(unsigned char)z[i])!=0; i++){
    if( c & maskNonAscii ){
      if( (z[++i]&0xc0)!=0x80 ){
        /* Invalid first continuation byte */
        return 0;
      }
      if( c<0xc2 ){
        /* Invalid 1-byte UTF-8 sequence, or 2-byte overlong form. */
        return 0;
      }else if( (c&0xe0)==0xe0 ){
        /* 3-byte or more */
        int unicode;
        if( c&0x10 ){
          /* Unicode characters > U+FFFF are not supported.
           * Windows XP and earlier cannot handle them.
           */
          return 0;
        }
        /* This is a 3-byte UTF-8 character */
        unicode = ((c&0x0f)<<12) + ((z[i]&0x3f)<<6) + (z[i+1]&0x3f);
        if( unicode <= 0x07ff ){
          /* overlong form */
          return 0;
        }else if( unicode>=0xe000 ){
          /* U+E000..U+FFFF */
          if( (unicode<=0xf8ff) || (unicode>=0xfffe) ){
            /* U+E000..U+F8FF are for private use.
             * U+FFFE..U+FFFF are noncharacters. */
            return 0;
          } else if( (unicode>=0xfdd0) && (unicode<=0xfdef) ){
            /* U+FDD0..U+FDEF are noncharacters. */
            return 0;
          }
        }else if( (unicode>=0xd800) && (unicode<=0xdfff) ){
          /* U+D800..U+DFFF are for surrogate pairs. */
          return 0;
        }
        if( (z[++i]&0xc0)!=0x80 ){
          /* Invalid second continuation byte */
          return 0;
        }
      }
    }else if( bStrictUtf8 && (c=='\\') ){
      return 0;
    }
    if( c=='/' ){
      if( z[i+1]=='/' ) return 0;
      if( z[i+1]=='.' ){
        if( z[i+2]=='/' || z[i+2]==0 ) return 0;
        if( z[i+2]=='.' && (z[i+3]=='/' || z[i+3]==0) ) return 0;
        if( z[i+3]=='.' ) return 0;
      }
    }
  }
  if( z[i-1]=='/' ) return 0;
  return 1;
}


/*
   If the last component of the pathname in z[0]..z[j-1] is something
   other than ".." then back it out and return true.  If the last
   component is empty or if it is ".." then return false.
*/
static bool fsl_backup_dir(const char *z, fsl_int_t *pJ){
  fsl_int_t j = *pJ;
  fsl_int_t i;
  if( !j ) return 0;
  for(i=j-1; i>0 && z[i-1]!='/'; i--){}
  if( z[i]=='.' && i==j-2 && z[i+1]=='.' ) return 0;
  *pJ = i-1;
  return 1;
}



fsl_size_t fsl_file_simplify_name(char *z, fsl_int_t n_, bool slash){
  fsl_size_t i;
  fsl_size_t n = (n_<0) ? fsl_strlen(z) : (fsl_size_t)n_;
  fsl_int_t j;
  bool const hadSlash = n && (z[n-1]=='/');
  /* On windows and cygwin convert all \ characters to / */
#if defined(_WIN32) || defined(__CYGWIN__)
  for(i=0; i<n; i++){
    if( z[i]=='\\' ) z[i] = '/';
  }
#endif
  /* Removing trailing "/" characters */
  while( n>1 && z[n-1]=='/' ){--n;}

  /* Remove duplicate '/' characters.  Except, two // at the beginning
     of a pathname is allowed since this is important on windows. */
  for(i=j=1; i<n; i++){
    z[j++] = z[i];
    while( z[i]=='/' && i<n-1 && z[i+1]=='/' ) i++;
  }
  n = j;
  /* Skip over zero or more initial "./" sequences */
  for(i=0; i<n-1 && z[i]=='.' && z[i+1]=='/'; i+=2){}

  /* Begin copying from z[i] back to z[j]... */
  for(j=0; i<n; i++){
    if( z[i]=='/' ){
      /* Skip over internal "/." directory components */
      if( z[i+1]=='.' && (i+2==n || z[i+2]=='/') ){
        i += 1;
        continue;
      }

      /* If this is a "/.." directory component then back out the
         previous term of the directory if it is something other than ".."
         or "."
      */
      if( z[i+1]=='.' && i+2<n && z[i+2]=='.' && (i+3==n || z[i+3]=='/')
       && fsl_backup_dir(z, &j)
      ){
        i += 2;
        continue;
      }
    }
    if( j>=0 ) z[j] = z[i];
    j++;
  }
  if( j==0 ) z[j++] = '.';
  if(slash && hadSlash && '/'!=z[j-1]) z[j++] = '/';
  z[j] = 0;
  return (fsl_size_t)j;
}

int fsl_file_canonical_name2(const char *zRoot,
                             const char *zOrigName,
                             fsl_buffer * const pOut, bool slash){
  int rc;
  if(!zOrigName || !pOut) return FSL_RC_MISUSE;
  else if( fsl_is_absolute_path(zOrigName) || (zRoot && !*zRoot)){
    rc = fsl_buffer_append( pOut, zOrigName, -1 );
#if defined(_WIN32) || defined(__CYGWIN__)
    if(!rc){
      char *zOut;
      /*
         On Windows/cygwin, normalize the drive letter to upper case.
      */
      zOut = fsl_buffer_str(pOut);
      if( fsl_islower(zOut[0]) && zOut[1]==':' ){
        zOut[0] = fsl_toupper(zOut[0]);
      }
    }
#endif
  }else if(!zRoot){
    char zPwd[2000];
    fsl_size_t nOrig = fsl_strlen(zOrigName);
    assert(nOrig < sizeof(zPwd));
    rc = fsl_getcwd(zPwd, sizeof(zPwd)-nOrig, NULL);
    if(!rc){
#if defined(_WIN32)
      /*
         On Windows, normalize the drive letter to upper case.
      */
      if( !rc && fsl_islower(zPwd[0]) && zPwd[1]==':' ){
        zPwd[0] = fsl_toupper(zPwd[0]);
      }
#endif
      rc = fsl_buffer_appendf(pOut, "%//%/", zPwd, zOrigName);
    }
  }else{
    rc = fsl_buffer_appendf(pOut, "%/%s%/", zRoot,
                            *zRoot ? "/" : "",
                            zOrigName);
  }
  if(!rc){
    fsl_size_t const newLen = fsl_file_simplify_name(fsl_buffer_str(pOut),
                                                     (int)pOut->used, slash);
    /* Reminder to self: do NOT resize pOut to the new,
       post-simplification length because pOut is almost always a
       fsl_cx::scratchpad buffer and doing so forces all sorts of
       downstream reallocs. */
    pOut->used = newLen;
  }
  return rc;
}


int fsl_file_canonical_name(const char *zOrigName,
                            fsl_buffer * const pOut,
                            bool slash){
  return fsl_file_canonical_name2(NULL, zOrigName, pOut, slash);
}

int fsl_file_dirpart(char const * zFilename,
                     fsl_int_t nLen,
                     fsl_buffer * const pOut,
                     bool leaveSlash){
  if(!zFilename || !*zFilename || !pOut) return FSL_RC_MISUSE;
  else if(!nLen) return FSL_RC_RANGE;
  else{
    fsl_size_t n = (nLen>0) ? (fsl_size_t)nLen : fsl_strlen(zFilename);
    char const * z = zFilename + n;
    char doBreak = 0;
    if(!n) return FSL_RC_RANGE;
    else while( !doBreak && (--z >= zFilename) ){
      switch(*z){
#if defined(_WIN32)
        case '\\':
#endif
        case '/':
          if(!leaveSlash) --z;
          doBreak = 1;
          break;
      }
    }
    if(z<=zFilename){
      return (doBreak && leaveSlash)
        ? fsl_buffer_append(pOut, zFilename, 1)
        : fsl_buffer_append(pOut, "", 0) /* ensure a NUL terminator */;
    }else{
      return fsl_buffer_append(pOut, zFilename, z-zFilename + 1);
    }
  }
}

const char *fsl_file_tail(const char *z){
  const char *zTail = z;
  if( !zTail ) return 0;
  while( z[0] ){
    if( '/'==z[0] || '\\'==z[0] ) zTail = &z[1];
    z++;
  }
  return zTail;
}


int fsl_find_home_dir( fsl_buffer * const tgt, bool requireWriteAccess ){
  char * zHome = NULL;
  int rc = 0;
  fsl_buffer_reuse(tgt);
#if defined(_WIN32) || defined(__CYGWIN__)
  zHome = fsl_getenv("LOCALAPPDATA");
  if( zHome==0 ){
    zHome = fsl_getenv("APPDATA");
    if( zHome==0 ){
      char *zDrive = fsl_getenv("HOMEDRIVE");
      zHome = fsl_getenv("HOMEPATH");
      if( zDrive && zHome ){
        tgt->used = 0;
        rc = fsl_buffer_appendf(tgt, "%s", zDrive);
        fsl_filename_free(zDrive);
        if(rc){
          fsl_filename_free(zHome);
          return rc;
        }
      }
    }
  }
  if(NULL==zHome){
    rc = fsl_buffer_append(tgt,
                           "Cannot locate home directory - "
                           "please set the LOCALAPPDATA or "
                           "APPDATA or HOMEPATH "
                           "environment variables.",
                           -1);
    return rc ? rc : FSL_RC_NOT_FOUND;
  }
  rc = fsl_buffer_appendf( tgt, "%/", zHome );
#else
  /* Unix... */
  zHome = fsl_getenv("HOME");
  if( zHome==0 ){
    rc = fsl_buffer_append(tgt,
                           "Cannot locate home directory - "
                           "please set the HOME environment "
                           "variable.",
                           -1);
    return rc ? rc : FSL_RC_NOT_FOUND;
  }
  rc = fsl_buffer_appendf( tgt, "%s", zHome );
#endif

  fsl_filename_free(zHome);
  if(rc) return rc;
  assert(0<tgt->used);
  zHome = fsl_buffer_str(tgt);

  if( fsl_dir_check(zHome)<1 ){
    /* assert(0==tgt->used); */
    fsl_buffer tmp = fsl_buffer_empty;
    rc = fsl_buffer_appendf(&tmp,
                            "Invalid home directory: %s",
                            zHome);
    fsl_buffer_swap_free(&tmp, tgt, -1);
    return rc ? rc : FSL_RC_TYPE;
  }

#if !(defined(_WIN32) || defined(__CYGWIN__))
  /* Not sure why, but the is-writable check is historically only done
     on Unix platforms?

     TODO: this was subsequently changed in fossil(1) to only require
     that the global db dir be writable. Port the newer logic in.
  */
  if( requireWriteAccess &&
      (0 != fsl_file_access(zHome, W_OK)) ){
    fsl_buffer tmp = fsl_buffer_empty;
    rc = fsl_buffer_appendf(&tmp,
                            "Home directory [%s] must "
                            "be writeable.",
                            zHome);
    fsl_buffer_swap_free(&tmp, tgt, -1);
    return rc ? rc : FSL_RC_ACCESS;
  }
#endif

  return rc;
}

int fsl_errno_to_rc(int errNo, int dflt){
  switch(errNo){
    /* Plese expand on this as tests/use cases call for it... */
    case EINVAL:
      return FSL_RC_MISUSE;
    case ENOMEM:
      return FSL_RC_OOM;
    case EROFS:
    case EACCES:
    case EBUSY:
    case EPERM:
    case EDQUOT:
    case EAGAIN:
    case ETXTBSY:
      return FSL_RC_ACCESS;
    case EISDIR:
    case ENOTDIR:
      return FSL_RC_TYPE;
    case ENAMETOOLONG:
    case ELOOP:
    case ERANGE:
      return FSL_RC_RANGE;
    case ENOENT:
    case ESRCH:
      return FSL_RC_NOT_FOUND;
    case EEXIST:
    case ENOTEMPTY:
      return FSL_RC_ALREADY_EXISTS;
    case EIO:
      return FSL_RC_IO;
    default:
      return dflt;
  }
}

int fsl_file_unlink(const char *zFilename){
  int rc;
#ifdef _WIN32
  wchar_t *z = (wchar_t*)fsl_utf8_to_filename(zFilename);
  rc = _wunlink(z) ? errno : 0;
#else
  char *z = (char *)fsl_utf8_to_filename(zFilename);
  rc = unlink(zFilename) ? errno : 0;
#endif
  fsl_filename_free(z);
  return rc ? fsl_errno_to_rc(errno, FSL_RC_IO) : 0;
}

int fsl_mkdir(const char *zName, bool forceFlag){
  int rc =
    /*file_wd_dir_check(zName)*/
    fsl_dir_check(zName)
    ;
  if( rc<0 ){
    if( !forceFlag ) return FSL_RC_TYPE;
    rc = fsl_file_unlink(zName);
    if(rc) return rc;
  }else if( 0==rc ){
#if defined(_WIN32)
    typedef wchar_t char_t;
#define mkdir(F,P) _wmkdir(F)
#else
    typedef char char_t;
#endif
    char_t *zMbcs = (char_t*)fsl_utf8_to_filename(zName);
    if(!zMbcs) return FSL_RC_OOM;
    rc = mkdir(zMbcs, 0755);
    fsl_filename_free(zMbcs);
    return rc ? fsl_errno_to_rc(errno, FSL_RC_IO) : 0;
#if defined(_WIN32)
#undef mkdir
#endif
  }
  return 0;
}

int fsl_mkdir_for_file(char const *zName, bool forceFlag){
  int rc;
  fsl_buffer b = fsl_buffer_empty /* we copy zName to
                                     simplify traversal */;
  fsl_size_t n = fsl_strlen(zName);
  fsl_size_t i;
  char * zCan;
  if(n==0) return FSL_RC_RANGE;
  else if(n<2) return 0/*no dir part*/;
#if 1
  /* This variant does more work (checks dirs we know already
     exist) but transforms the path into something platform-neutral.
     If we use fsl_file_simplify_name() instead then we end up
     having to do the trailing-slash logic here.
  */
  rc = fsl_file_canonical_name(zName, &b, 1);
  if(rc) goto end;
#else
  rc = fsl_buffer_append(&b, zName, n);
  if(rc) goto end;
#endif
  zCan = fsl_buffer_str(&b);
  n = b.used;
  for( i = 1; i < n; ++i ){
    if( '/'==zCan[i] ){
      zCan[i] = 0;
#if defined(_WIN32) || defined(__CYGWIN__)
      /*
         On Windows, local path looks like: C:/develop/project/file.txt
         The if stops us from trying to create a directory of a drive letter
         C: in this example.
      */
      if( !(i==2 && zCan[1]==':') ){
#endif
        rc = fsl_dir_check(zCan);
#if 0
        if(rc<0){
          if(forceFlag) rc = fsl_file_unlink(zCan);
          else rc = FSL_RC_TYPE;
          if(rc) goto end;
        }
#endif
        /* MARKER(("dir_check rc=%d, zCan=%s\n", rc, zCan)); */
        if(0>=rc){
          rc = fsl_mkdir(zCan, forceFlag);
          /* MARKER(("mkdir(%s) rc=%s\n", zCan, fsl_rc_cstr(rc))); */
          if( 0!=rc ) goto end;

        }else{
          rc = 0;
          /* Nothing to do. */
        }
#if defined(_WIN32) || defined(__CYGWIN__)
      }
#endif
      zCan[i] = '/';
    }
  }
  end:
  fsl_buffer_clear(&b);
  return rc;
}

#if defined(_WIN32)
/* Taken verbatim from fossil(1), just renamed */
/*
** Returns non-zero if the specified name represents a real directory, i.e.
** not a junction or symbolic link.  This is important for some operations,
** e.g. removing directories via _wrmdir(), because its detection of empty
** directories will (apparently) not work right for junctions and symbolic
** links, etc.
*/
static int w32_file_is_normal_dir(wchar_t *zName){
  /*
  ** Mask off attributes, applicable to directories, that are harmless for
  ** our purposes.  This may need to be updated if other attributes should
  ** be ignored by this function.
  */
  DWORD dwAttributes = GetFileAttributesW(zName);
  if( dwAttributes==INVALID_FILE_ATTRIBUTES ) return 0;
  dwAttributes &= ~(
    FILE_ATTRIBUTE_ARCHIVE | FILE_ATTRIBUTE_COMPRESSED |
    FILE_ATTRIBUTE_ENCRYPTED | FILE_ATTRIBUTE_NORMAL |
    FILE_ATTRIBUTE_NOT_CONTENT_INDEXED
  );
  return dwAttributes==FILE_ATTRIBUTE_DIRECTORY;
}
#endif


bool fsl_file_isexec(const char *zFilename){
  fsl_fstat st = fsl_fstat_empty;
  int const s = fsl_stat(zFilename, &st, true);
  return 0==s ? (st.perm & FSL_FSTAT_PERM_EXE) : false;
}

int fsl_rmdir(const char *zFilename){
  int rc = fsl_dir_check(zFilename);
  if(rc<1) return rc ? FSL_RC_TYPE : FSL_RC_NOT_FOUND;
#ifdef _WIN32
  wchar_t *z = (wchar_t*)fsl_utf8_to_filename(zFilename);
  if(w32_file_is_normal_dir(z)){
    rc = _wunlink(z) ? errno : 0;
  }else{
    rc = ENOTDIR;
  }
#else
  char *z = (char *)fsl_utf8_to_filename(zFilename);
  rc = rmdir(zFilename) ? errno : 0;
#endif
  fsl_filename_free(z);
  if(rc){
    int const eno = errno;
    switch(eno){
      /* ENOENT normally maps to FSL_RC_NOT_FOUND,
         but in this case that's ambiguous. */
      case ENOENT: rc = FSL_RC_ACCESS; break;
      default: rc = fsl_errno_to_rc(errno, FSL_RC_IO);
        break;
    }
  }
  return rc;
}

int fsl_dir_check(const char *zFilename){
  fsl_fstat fst;
  int rc;
  if( zFilename ){
#if 1
    rc = fsl_stat(zFilename, &fst, 1);
#else
    char *zFN = fsl_strdup(zFilename);
    if(!zFN) rc = FSL_RC_OOM;
    else{
      fsl_file_simplify_name(zFN, -1, 0);
      rc = fsl_stat(zFN, &fst, 1);
      fsl_free(zFN);
    }
#endif
  }else{
    rc = -1 /*fsl_stat(zFilename, &fst, 1) historic: used static stat cache*/;
  }
  return rc ? 0 : ((FSL_FSTAT_TYPE_DIR == fst.type) ? 1 : -1);
}

int fsl_chdir(const char *zChDir){
  int rc;
#ifdef _WIN32
  wchar_t *zPath = fsl_utf8_to_filename(zChDir);
  errno = 0;
  rc = (int)!SetCurrentDirectoryW(zPath);
  fsl_filename_free(zPath);
  if(rc) rc = FSL_RC_IO;
#else
  char *zPath = fsl_utf8_to_filename(zChDir);
  errno = 0;
  rc = chdir(zPath);
  fsl_filename_free(zPath);
  if(rc) rc = fsl_errno_to_rc(errno, FSL_RC_IO);
#endif
  return rc;
}


#if 0
/*
   Same as dir_check(), but takes into account symlinks.
*/
int file_wd_dir_check(const char *zFilename){
  if(!zFilename || !*zFilename) return FSL_RC_MISUSE;
  else{
    int rc;
    fsl_fstat fst = fsl_fstat_empty;
    char *zFN = fsl_strdup(zFilename);
    if(!zFN) rc = FSL_RC_OOM;
    else{
      fsl_file_simplify_name(zFN, -1, 0);
      rc = fsl_stat(zFN, &fst, 0);
      fsl_free(zFN);
    }
    return rc ? 0 : ((FSL_FSTAT_TYPE_DIR == fst.type) ? 1 : 2);
  }
}
#endif

#if 0
/* This block requires permissions flags from v1's manifest.c. */

/*
   Return TRUE if the named file is an executable.  Return false
   for directories, devices, fifos, symlinks, etc.
*/
int fsl_wd_isexe(const char *zFilename){
  return fsl_wd_perm(zFilename)==PERM_EXE;
}

/*
   Return TRUE if the named file is a symlink and symlinks are allowed.
   Return false for all other cases.
  
   On Windows, always return False.
*/
int file_wd_islink(const char *zFilename){
  return file_wd_perm(zFilename)==PERM_LNK;
}
#endif

#if 0
/**
    Same as fsl_is_file(), but takes into account symlinks.
 */
bool fsl_wd_isfile(const char *zFilename);
bool fsl_wd_isfile(const char *zFilename){
  fsl_fstat fst;
  return ( 0 != fsl_stat(zFilename, &fst, 0) )
    ? 0
    : (FSL_FSTAT_TYPE_FILE == fst.type);
}
#endif
#if 0
/**
    Same as fsl_file_mtime(), but takes into account symlinks.
 */
fsl_time_t fsl_wd_mtime(const char *zFilename);
fsl_time_t fsl_wd_mtime(const char *zFilename){
  fsl_fstat fst;
  return ( 0 != fsl_stat(zFilename, &fst, 0) )
    ? -1
    : (fsl_time_t)fst.mtime;
}

bool fsl_wd_isfile_or_link(const char *zFilename){
  fsl_fstat fst;
  return ( 0 != fsl_stat(zFilename, &fst, 0) )
    ? 0
    : ((FSL_FSTAT_TYPE_LINK == fst.type)
       || (FSL_FSTAT_TYPE_FILE == fst.type))
    ;
}
#endif

#if 0
/**
    Same as fsl_file_size(), but takes into account symlinks.
 */
fsl_size_t fsl_wd_size(const char *zFilename);
fsl_size_t fsl_wd_size(const char *zFilename){
  fsl_fstat fst;
  return ( 0 != fsl_stat(zFilename, &fst, 0) )
    ? -1
    : fst.size;
}
#endif

int fsl_file_mtime_set(const char *zFilename, fsl_time_t newMTime){
  if(!zFilename || !*zFilename) return FSL_RC_MISUSE;
  else{
    int rc;
    void * zMbcs;
#if !defined(_WIN32)
    struct timeval tv[2];
    if(newMTime < 0) newMTime = (fsl_time_t)time(0);
    zMbcs = fsl_utf8_to_filename(zFilename);
    if(!zMbcs) return FSL_RC_OOM;
    memset(tv, 0, sizeof(tv[0])*2);
    tv[0].tv_sec = newMTime;
    tv[1].tv_sec = newMTime;
    rc = utimes((char const *)zMbcs, tv);
#else
    struct _utimbuf tb;
    if(newMTime < 0) newMTime = (fsl_time_t)time(0);
    zMbcs = fsl_utf8_to_filename(zFilename);
    if(!zMbcs) return FSL_RC_OOM;
    tb.actime = newMTime;
    tb.modtime = newMTime;
    rc = _wutime((wchar_t const *)zMbcs, &tb);
#endif
    fsl_filename_free(zMbcs);
    return rc ? fsl_errno_to_rc(errno, FSL_RC_IO) : 0;
  }
}



void fsl_pathfinder_clear(fsl_pathfinder * const pf){
  if(pf){
    fsl_list_visit_free(&pf->ext, 1);
    fsl_list_visit_free(&pf->dirs, 1);
    fsl_buffer_clear(&pf->buf);
    *pf = fsl_pathfinder_empty;
  }
}

static int fsl__pathfinder_add_impl(fsl_list * const li, char const * str,
                                   fsl_int_t strLen){
  char * cp = fsl_strndup(str, strLen);
  int rc;
  if(!cp) rc = FSL_RC_OOM;
  else{
    rc = fsl_list_append(li, cp);
    if(rc) fsl_free(cp);
  }
  return rc;
}

int fsl_pathfinder_dir_add(fsl_pathfinder * const pf, char const * const dir){
  return dir
    ? fsl__pathfinder_add_impl(&pf->dirs, dir, -1)
    : FSL_RC_MISUSE;
}

int fsl_pathfinder_dir_add2(fsl_pathfinder * const pf, char const * const dir,
                            fsl_int_t strLen){
  return dir
    ? fsl__pathfinder_add_impl(&pf->dirs, dir, strLen)
    : FSL_RC_MISUSE;
}

int fsl_pathfinder_ext_add(fsl_pathfinder * const pf, char const * const ext){
  return (pf && ext)
    ? fsl__pathfinder_add_impl(&pf->ext, ext, -1)
    : FSL_RC_MISUSE;
}

int fsl_pathfinder_ext_add2(fsl_pathfinder * const pf, char const * const ext,
                           fsl_int_t strLen){
  return (pf && ext)
    ? fsl__pathfinder_add_impl(&pf->ext, ext, strLen)
    : FSL_RC_MISUSE;
}

int fsl_pathfinder_search(fsl_pathfinder * const pf,
                          char const * const base,
                          char const ** pOut,
                          fsl_size_t * const outLen ){
  fsl_buffer * const buf = &pf->buf;
  fsl_list * ext;
  fsl_list * dirs;
  int rc = 0;
  fsl_size_t d, x, nD, nX, resetLen = 0;
  fsl_size_t baseLen;
  static char const pathSep =
#if defined(_WIN32)
    '\\' /* TODO: confirm whether we can always use '/', and do so if
            we can. */
#else
    '/'
#endif
    ;
  if(!base || !*base) return FSL_RC_MISUSE;
  else if(!*base) return FSL_RC_RANGE;
  else if(0==fsl_file_access( base, 0 )){
    /* Special case: if base is found as-is, without a path search,
       use it. */
    if(pOut) *pOut = base;
    if(outLen) *outLen = fsl_strlen(base);
    return 0;
  }
  baseLen = fsl_strlen(base);
  ext = &pf->ext;
  dirs = &pf->dirs;
  nD = dirs->used;
  nX = ext->used;
  for( d = 0; !rc && (d < nD); ++d ){
    char const * vD = (char const *)dirs->list[d];
    /*
      Search breadth-first for a file/directory named by vD/base
    */
    buf->used = 0;
    if(vD){
      fsl_size_t const used = buf->used;
      rc = fsl_buffer_append(buf, vD, -1);
      if(rc) return rc;
      if(used != buf->used){
        /* Only append separator if vD is non-empty. */
        rc = fsl_buffer_append(buf, &pathSep, 1);
        if(rc) return rc;
      }
    }
    rc = fsl_buffer_append(buf, base, (fsl_int_t)baseLen);
    if(rc) return rc;
    if(0==fsl_file_access( (char const *)buf->mem, 0 )) goto gotone;
    resetLen = buf->used;
    for( x = 0; !rc && (x < nX); ++x ){
      char const * vX = (char const *)ext->list[x];
      if(vX){
        buf->used = resetLen;
        rc = fsl_buffer_append(buf, vX, -1);
        if(rc) return rc;
      }
      assert(buf->used < buf->capacity);
      buf->mem[buf->used] = 0;
      if(0==fsl_file_access( (char const *)buf->mem, 0 )){
        goto gotone;
      }
    }
  }
  return FSL_RC_NOT_FOUND;
  gotone:
  if(outLen) *outLen = buf->used;
  if(pOut) *pOut = (char const *)buf->mem;
  return 0;
}

void fsl_path_splitter_init( fsl_path_splitter * pt, char const * path, fsl_int_t len ){
  *pt = fsl_path_splitter_empty;
  pt->pos = pt->begin = path;
  pt->end = pt->begin + ((len>=0) ? (fsl_size_t)len : fsl_strlen(path));
}

int fsl_path_splitter_next( fsl_path_splitter * const pt, char const ** token,
                            fsl_size_t * const len ){
  if(!pt->pos || pt->pos>=pt->end) return FSL_RC_RANGE;
  else if(!pt->separators || !*pt->separators) return FSL_RC_MISUSE;
  else{
    char const * pos = pt->pos;
    char const * t;
    char const * sep;
    for( sep = pt->separators; *sep; ++sep){
      if(*sep & 0x80) return FSL_RC_MISUSE;
      /* non-ASCII */
    }
    for( ; pos<pt->end; ){
      /*skip leading separators*/
      for( sep = pt->separators;
           *sep && *pos!=*sep; ++sep ){
      }
      if(*pos == *sep) ++pos;
      else break;
    }
    t = pos;
    for( ; pos<pt->end; ){
      /*skip until the next separator*/
      for( sep = pt->separators;
           *sep && *pos!=*sep; ++sep ){
      }
      if(*pos == *sep) break;
      else ++pos;
    }
    pt->pos = pos;
    if(pos>t){
      *token = t;
      *len = (fsl_size_t)(pos - t);
      return 0;
    }
    return FSL_RC_NOT_FOUND;
  }
}

int fsl_pathfinder_split( fsl_pathfinder * const tgt,
                          bool isDirs,
                          char const * path,
                          fsl_int_t pathLen ){
  int rc = 0;
  char const * t = 0;
  fsl_size_t tLen = 0;
  fsl_path_splitter pt = fsl_path_splitter_empty;
  fsl_path_splitter_init(&pt, path, pathLen);
  while(0==rc && 0==fsl_path_splitter_next(&pt, &t, &tLen)){
    rc = isDirs
      ? fsl_pathfinder_dir_add2(tgt, t, (fsl_int_t)tLen)
      : fsl_pathfinder_ext_add2(tgt, t, (fsl_int_t)tLen);
  }
  return rc;
}

char * fsl__file_without_drive_letter(char * zIn){
#ifdef _WIN32
  if( zIn && fsl_isalpha(zIn[0]) && zIn[1]==':' ) zIn += 2;
#endif
  return zIn;
}

int fsl_dir_is_empty(const char *path){
  struct dirent *ent;
  int            retval = 0;
  DIR *d = opendir(path);
  if(!d){
    return -1;
  }
  while((ent = readdir(d))) {
    const char * z = ent->d_name;
    if('.'==*z &&
       (!z[1] || ('.'==z[1] && !z[2]))){
      // Skip "." and ".." entries
      continue;
    }
    retval = 1;
    break;
  }
  closedir(d);
  return retval;
}

int fsl_file_exec_set(const char *zFilename, bool isExe){
#if FSL_PLATFORM_IS_WINDOWS
  return 0;
#else
  int rc = 0, err;
  struct stat sb;
  err = stat(zFilename, &sb);
  if(0==err){
    if(!S_ISREG(sb.st_mode)) return 0;
    else if(isExe){
      if( 0==(sb.st_mode & 0100) ){
        int const mode = (sb.st_mode & 0444)>>2
          /* This impl is from fossil, which is known to work, but...
             what is the >>2 for?*/;
        err = chmod(zFilename, (mode_t)(sb.st_mode | mode));
      }
    }else if( 0!=(sb.st_mode & 0100) ){
      err = chmod(zFilename, sb.st_mode & ~0111);
    }
  }
  if(err) rc = fsl_errno_to_rc(errno, FSL_RC_IO);
  return rc;
#endif
}

/**
   fsl_dircrawl() part for handling a single directory. fst must be
   valid state from a freshly-fsl_fstat()'d DIRECTORY.
*/
static int fsl_dircrawl_impl(fsl_buffer * const dbuf, fsl_fstat * const fst,
                             fsl_dircrawl_f cb, void * const cbState,
                             fsl_dircrawl_state * const dst,
                             unsigned int depth){
  int rc = 0;
  DIR *dir = opendir(fsl_buffer_cstr(dbuf));
  struct dirent * dent = 0;
  fsl_size_t const dPos = dbuf->used;
  if(!dir){
    return fsl_errno_to_rc(errno, FSL_RC_IO);
  }
  if(depth>20/*arbitrary limit to try to avoid stack overflow*/){
    return FSL_RC_RANGE;
  }
  while(!rc && (dent = readdir(dir))){
    const char * z = dent->d_name;
    if('.'==*z &&
       (!z[1] || ('.'==z[1] && !z[2]))){
      // Skip "." and ".." entries
      continue;
    }
    dbuf->used = dPos;
    rc = fsl_buffer_appendf(dbuf, "/%s", z);
    if(rc) break;
    fsl_size_t const newLen = dbuf->used;
    if(fsl_stat((char const *)dbuf->mem, fst, false)){
      // Simply skip stat errors. i was once bitten by an app which did
      // not do so. Scarred for life. Too soon.
      rc = 0;
      continue;
    }
    switch(fst->type){
      case FSL_FSTAT_TYPE_LINK:
      case FSL_FSTAT_TYPE_DIR:
      case FSL_FSTAT_TYPE_FILE:
        break;
      default: continue;
    }
    dbuf->mem[dbuf->used = dPos] = 0;
    dst->absoluteDir = (char const *)dbuf->mem;
    dst->entryName = z;
    dst->entryType = fst->type;
    dst->depth = depth;
    rc = cb( dst );
    if(!rc){
      dbuf->mem[dbuf->used] = '/';
      dbuf->used = newLen;
      if(FSL_FSTAT_TYPE_DIR==fst->type){
        rc = fsl_dircrawl_impl( dbuf, fst, cb, cbState, dst, depth+1 );
      }
    }else if(FSL_RC_NOOP == rc){
      rc = 0;
    }
  }
  closedir(dir);
  return rc;
}

int fsl_dircrawl(char const * dirName, fsl_dircrawl_f callback,
                 void * cbState){
  fsl_buffer dbuf = fsl_buffer_empty;
  fsl_fstat fst = fsl_fstat_empty;
  int rc = fsl_file_canonical_name(dirName, &dbuf, false);
  fsl_dircrawl_state dst;
  if(!rc && '/' == dbuf.mem[dbuf.used-1]){
    dbuf.mem[--dbuf.used] = 0;
  }
  memset(&dst, 0, sizeof(dst));
  dst.callbackState = cbState;
  while(!rc){
    rc = fsl_stat((char const *)dbuf.mem, &fst, false);
    if(rc) break;
    else if(FSL_FSTAT_TYPE_DIR!=fst.type){
      rc = FSL_RC_TYPE;
      break;
    }
    rc = fsl_dircrawl_impl(&dbuf, &fst, callback, cbState, &dst, 1);
    if(FSL_RC_BREAK==rc) rc = 0;
    break;
  }
  fsl_buffer_clear(&dbuf);
  return rc;
}

bool fsl_is_file_or_link(const char *zFilename){
  fsl_fstat fst = fsl_fstat_empty;
  return fsl_stat(zFilename, &fst, false)
    ? false
    : (fst.type==FSL_FSTAT_TYPE_FILE
       || fst.type==FSL_FSTAT_TYPE_LINK);
}

fsl_size_t fsl_strip_trailing_slashes(char * name, fsl_int_t nameLen){
  fsl_size_t rc = 0;
  if(nameLen < 0) nameLen = (fsl_int_t)fsl_strlen(name);
  if(nameLen){
    char * z = name + nameLen - 1;
    for( ; (z>=name) && ('/'==*z); --z){
      *z = 0;
      ++rc;
    }
  }
  return rc;
}

void fsl_buffer_strip_slashes(fsl_buffer * const b){
  b->used -= fsl_strip_trailing_slashes((char *)b->mem,
                                        (fsl_int_t)b->used);
}

int fsl_file_rename(const char *zFrom, const char *zTo){
  int rc;
#if defined(_WIN32)
  /** 2021-03-24: fossil's impl of this routine has 2 additional
      params (bool isFromDir, bool isToDir), which are passed on to
      fsl_utf8_to_filename(), only used on Windows platforms, and are
      only to allow for 12 bytes of edge case in MAX_PATH handling.
      We don't need them. */
  wchar_t *zMbcsFrom = fsl_utf8_to_filename(zFrom);
  wchar_t *zMbcsTo = zMbcsFrom ? fsl_utf8_to_filename(zTo) : 0;
  rc = zMbcsTo ? _wrename(zMbcsFrom, zMbcsTo) : FSL_RC_OOM;
#else
  char *zMbcsFrom = fsl_utf8_to_filename(zFrom);
  char *zMbcsTo = zMbcsFrom ? fsl_utf8_to_filename(zTo) : 0;
  rc = zMbcsTo ? rename(zMbcsFrom, zMbcsTo) : FSL_RC_OOM;
#endif
  fsl_filename_free(zMbcsTo);
  fsl_filename_free(zMbcsFrom);
  return -1==rc ? fsl_errno_to_rc(errno, FSL_RC_IO) : rc;
}

char ** fsl_temp_dirs_get(void){
#if FSL_PLATFORM_IS_WINDOWS
  const char *azDirs[] = {
     ".",
     NULL
  };
  unsigned int const nDirs = 4
    /* GetTempPath(), $TEMP, $TMP, azDirs */;
#else
  const char *azDirs[] = {
     "/var/tmp",
     "/usr/tmp",
     "/tmp",
     "/temp",
     ".", NULL
  };
  unsigned int const nDirs = 6
    /* $TMPDIR, azDirs */;
#endif
  char *z;
  char const *zC;
  char ** zDirs = NULL;
  unsigned int i, n = 0;

  zDirs = (char **)fsl_malloc(sizeof(char*) * (nDirs + 1));
  if(!zDirs) return NULL;
  for(i = 0; i<=nDirs; ++i) zDirs[i] = NULL;
#define DOZ \
  if(z && fsl_dir_check(z)>0) zDirs[n++] = z;   \
  else if(z) fsl_filename_free(z)

#if FSL_PLATFORM_IS_WINDOWS
  wchar_t zTmpPath[MAX_PATH];

  if( GetTempPathW(MAX_PATH, zTmpPath) ){
    z = fsl_filename_to_utf8(zTmpPath);
    DOZ;
  }
  z = fsl_getenv("TEMP");
  DOZ;
  z = fsl_getenv("TMP");
  DOZ;
#else /* Unix-like */
  z = fsl_getenv("TMPDIR");
  DOZ;
#endif
  for( i = 0; (zC = azDirs[i]); ++i ){
    z = fsl_filename_to_utf8(azDirs[i]);
    DOZ;
  }
#undef DOZ
  /* Strip any trailing slashes unless the only character is a
     slash. Note that we ignore the root-dir case on Windows here,
     mainly because this developer can't test it and secondarily
     because it's a highly unlikely case. */
  for(i = 0; i < n; ++i ){
    fsl_size_t len;
    z = zDirs[i];
    len = fsl_strlen(z);
    while(len>1 && (z[len-1]=='/' || z[len-1]=='\\')){
      z[--len] = 0;
    }    
  }
  return zDirs;  
}

void fsl_temp_dirs_free(char **aDirs){
  if(aDirs){
    char * z;
    for(unsigned i = 0; (z = aDirs[i]); ++i){
      fsl_filename_free(z);
      aDirs[i] = NULL;
    }
    fsl_free(aDirs);
  }
}

int fsl_file_tempname(fsl_buffer * const tgt, char const *zPrefix,
                      char * const * const dirs){
  int rc = 0;
  unsigned int tries = 0; 
  const unsigned char zChars[] =
    "abcdefghijklmnopqrstuvwxyz"
    "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
    "0123456789_";
  enum { RandSize = 24 };
  char zRand[RandSize + 1];
  int i;
  char const * zDir = "";
  if(dirs){
    for(i = 0; (zDir=dirs[i++]); ){
      /* We repeat this check, performed in fsl_temp_dirs_get(), for the
         sake of long-lived apps where a given temp dir might disappear
         at some point. */
      if(fsl_dir_check(zDir)>0) break;
    }
    if(!zDir) return FSL_RC_NOT_FOUND;
  }
  if(!zPrefix) zPrefix = "libfossil";
  fsl_buffer_reuse(tgt);
  /* Pre-fill buffer to allocate it in advance and remember the length
     of the base filename part so that we don't have to re-write the
     prefix each iteration below. */
  rc = fsl_buffer_appendf(tgt, "%/%s%s%s%.*cZ",
                          zDir, *zDir ? "/" : "",
                          zPrefix, *zPrefix ? "~" : "",
                          (int)RandSize, 'X');
  fsl_size_t const baseLen = rc ? 0 : (tgt->used - RandSize - 1);
  do{
    if(++tries == 20){
      rc = FSL_RC_RANGE;
      break;
    }
    fsl_randomness(RandSize, zRand);
    for( i=0; i < RandSize; ++i ){
      zRand[i] = (char)zChars[ ((unsigned char)zRand[i])%(sizeof(zChars)-1) ];
    }
    zRand[RandSize] = 0;
    tgt->used = baseLen;
    rc = fsl_buffer_append(tgt, zRand, (fsl_int_t)RandSize);
    assert(0==rc && "We pre-allocated the buffer above.");
  }while(0==rc && fsl_file_size(fsl_buffer_cstr(tgt)) >= 0);
  return rc;
}

int fsl_file_copy(char const *zFrom, char const *zTo){
  FILE * in = 0, *out = 0;
  int rc;
  in = fsl_fopen(zFrom, "rb");
  if(!in) return fsl_errno_to_rc(errno, FSL_RC_IO);
  rc = fsl_mkdir_for_file(zTo, false);
  if(rc) goto end;
  out = fsl_fopen(zTo, "wb");
  rc = out
    ? fsl_stream(fsl_input_f_FILE, in, fsl_output_f_FILE, out)
    : fsl_errno_to_rc(errno, FSL_RC_IO);
  end:
  if(in) fsl_fclose(in);
  if(out) fsl_fclose(out);
  if(0==rc && fsl_file_isexec(zFrom)){
    fsl_file_exec_set(zTo, true);
  }
  return rc;
}

int fsl_symlink_read(fsl_buffer * const tgt, char const * zFilename){
#if FSL_PLATFORM_IS_WINDOWS
  fsl_buffer_reuse(tgt);
  return 0;
#else
  enum { BufLen = 1024 * 2 };
  char buf[BufLen];
  int rc;
  ssize_t const len = readlink(zFilename, buf, BufLen-1);
  if(len<0) rc = fsl_errno_to_rc(errno, FSL_RC_IO);
  else{
    fsl_buffer_reuse(tgt);
    rc = fsl_buffer_append(tgt, buf, (fsl_size_t)len);
  }
  return rc;
#endif
}

int fsl_symlink_create(const char *zTargetFile, const char *zLinkFile,
                       bool realLink){
  int rc;
#if !FSL_PLATFORM_IS_WINDOWS
  if( realLink ){
    char *zName, zBuf[1024 * 2];
    fsl_size_t nName = fsl_strlen(zLinkFile);
    if( nName>=sizeof(zBuf) ){
      zName = fsl_mprintf("%s", zLinkFile);
      if(!zName) return FSL_RC_OOM;
    }else{
      zName = zBuf;
      memcpy(zName, zLinkFile, nName+1);
    }
    nName = fsl_file_simplify_name(zName, (fsl_int_t)nName, false);
    rc = fsl_mkdir_for_file(zName, false);
    if(0==rc && 0!=symlink(zTargetFile, zName) ){
      rc = fsl_errno_to_rc(errno, FSL_RC_IO);
    }
    if( zName!=zBuf ) fsl_free(zName);
  }else
#endif
  {
    rc = fsl_mkdir_for_file(zLinkFile, false);
    if(0==rc){
      fsl_buffer content = fsl_buffer_empty;
      fsl_buffer_external(&content, zTargetFile, -1);
      fsl_file_unlink(zLinkFile)
        /* in case it's already a symlink, we don't want the following
           to overwrite the symlinked-to file */;
      rc = fsl_buffer_to_filename(&content, zLinkFile);
    }
  }
  return rc;
}

int fsl_symlink_copy(char const *zFrom, char const *zTo, bool realLink){
  int rc;
  fsl_buffer b = fsl_buffer_empty;
  rc = fsl_symlink_read(&b, zFrom);
  if(0==rc){
    rc = fsl_symlink_create(fsl_buffer_cstr(&b), zTo, realLink);
  }
  fsl_buffer_clear(&b);
  return rc;
}

char const * fsl_last_path_sep(char const * str, fsl_int_t slen ){
  if(slen<0) slen = (fsl_int_t)fsl_strlen(str);
  unsigned char const * pos = (unsigned char const *)str + slen;
  while( --pos >= (unsigned char const *)str ){
    if('/'==*pos || '\\'==*pos){
      return (char const *)pos;
    }
  }
  return NULL;
}


#if 0
int fsl_file_relative_name( char const * zRoot, char const * zPath,
                            fsl_buffer * pOut, char retainSlash ){
  int rc = FSL_RC_NYI;
  char * zPath;
  fsl_size_t rootLen;
  fsl_size_t pathLen;
  if(!zPath || !*zPath || !pOut) return FSL_RC_MISUSE;

  return rc;
}
#endif


#undef MARKER
#ifdef _WIN32
#  undef DIR
#  undef dirent
#  undef opendir
#  undef readdir
#  undef closedir
#endif
/* end of file ./src/fs.c */
/* start of file ./src/forum.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/

/***************************************************************************
  This file houses the code for forum-level APIS.
*/
#include <assert.h>


/* Only for debugging */
#include <stdio.h>
#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)

int fsl_repo_install_schema_forum(fsl_cx *f){
  int rc;
  fsl_db * db = fsl_needs_repo(f);
  if(!db) return FSL_RC_NOT_A_REPO;
  if(fsl_db_table_exists(db, FSL_DBROLE_REPO, "forumpost")){
    return 0;
  }
  rc = fsl_db_exec_multi(db, "%s",fsl_schema_forum());
  if(rc){
    rc = fsl_cx_uplift_db_error(f, db);
  }
  return rc;
}



#undef MARKER
/* end of file ./src/forum.c */
/* start of file ./src/glob.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/
/*************************************************************************
  This file contains some of the APIs dealing with globs.
*/
#include <assert.h>

bool fsl_str_glob(const char *zGlob, const char *z){
#if 1
  return (zGlob && z)
    ? (sqlite3_strglob(zGlob,z) ? 0 : 1)
    : 0;
#else
  int c, c2;
  int invert;
  int seen;
  while( (c = (*(zGlob++)))!=0 ){
    if( c=='*' ){
      while( (c=(*(zGlob++))) == '*' || c=='?' ){
        if( c=='?' && (*(z++))==0 ) return 0;
      }
      if( c==0 ){
        return 1;
      }else if( c=='[' ){
        while( *z && fsl_str_glob(zGlob-1,z)==0 ){
          z++;
        }
        return (*z)!=0;
      }
      while( (c2 = (*(z++)))!=0 ){
        while( c2!=c ){
          c2 = *(z++);
          if( c2==0 ) return 0;
        }
        if( fsl_str_glob(zGlob,z) ) return 1;
      }
      return 0;
    }else if( c=='?' ){
      if( (*(z++))==0 ) return 0;
    }else if( c=='[' ){
      int prior_c = 0;
      seen = 0;
      invert = 0;
      c = *(z++);
      if( c==0 ) return 0;
      c2 = *(zGlob++);
      if( c2=='^' ){
        invert = 1;
        c2 = *(zGlob++);
      }
      if( c2==']' ){
        if( c==']' ) seen = 1;
        c2 = *(zGlob++);
      }
      while( c2 && c2!=']' ){
        if( c2=='-' && zGlob[0]!=']' && zGlob[0]!=0 && prior_c>0 ){
          c2 = *(zGlob++);
          if( c>=prior_c && c<=c2 ) seen = 1;
          prior_c = 0;
        }else{
          if( c==c2 ){
            seen = 1;
          }
          prior_c = c2;
        }
        c2 = *(zGlob++);
      }
      if( c2==0 || (seen ^ invert)==0 ) return 0;
    }else{
      if( c!=(*(z++)) ) return 0;
    }
  }
  return *z==0;
#endif
}


int fsl_glob_list_parse( fsl_list * const tgt, char const * zPatternList ){
  fsl_size_t i;             /* Loop counter */
  char const *z = zPatternList;
  char * cp;
  char delimiter;    /* '\'' or '\"' or 0 */
  int rc = 0;
  char const * end;
  if( !tgt || !zPatternList ) return FSL_RC_MISUSE;
  else if(!*zPatternList) return 0;
  end = zPatternList + fsl_strlen(zPatternList);
  while( (z<end) && z[0] ){
    while( fsl_isspace(z[0]) || z[0]==',' ){
      ++z;  /* Skip leading commas, spaces, and newlines */
    }
    if( z[0]==0 ) break;
    if( z[0]=='\'' || z[0]=='"' ){
      delimiter = z[0];
      ++z;
    }else{
      delimiter = ',';
    }
    /* Find the next delimter (or the end of the string). */
    for(i=0; z[i] && z[i]!=delimiter; i++){
      if( delimiter!=',' ) continue; /* If quoted, keep going. */
      if( fsl_isspace(z[i]) ) break; /* If space, stop. */
    }
    if( !i ) break;
    cp = fsl_strndup(z, (fsl_int_t)i);
    if(!cp) return FSL_RC_OOM;
    else{
      rc = fsl_list_append(tgt, cp);
      if(rc){
        fsl_free(cp);
        break;
      }
      cp[i]=0;
    }
    z += i+1;
  }
  return rc;
}


char const * fsl_glob_list_matches( fsl_list const * const globList,
                                    char const * zNeedle ){
  if(!zNeedle || !*zNeedle || !globList->used) return NULL;
  else{
    char const * glob;
    fsl_size_t i = 0;
    for( ; i < globList->used; ++i){
      glob = (char const *)globList->list[i];
      if( fsl_str_glob( glob, zNeedle ) ) return glob;
    }
    return NULL;
  }
}

int fsl_glob_list_append( fsl_list * const tgt, char const * zGlob ){
  if(!tgt || !zGlob || !*zGlob) return FSL_RC_MISUSE;
  else{
    char * cp = fsl_strdup(zGlob);
    int rc = cp ? 0 : FSL_RC_OOM;
    if(!rc){
      rc = fsl_list_append(tgt, cp);
      if(rc) fsl_free(cp);
    }
    return rc;
  }
}


void fsl_glob_list_clear( fsl_list * const globList ){
  if(globList) fsl_list_visit_free(globList, 1);
}

/* end of file ./src/glob.c */
/* start of file ./src/io.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/
/*************************************************************************
  This file implements the generic i/o-related parts of the library.
*/
#include <assert.h>
#include <errno.h>
#include <string.h> /* memcmp() */

/* Only for debugging */
#include <stdio.h>
#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)

/**
    fsl_appendf_f() impl which sends its output to fsl_output(). state
    must be a (fsl_cx*).
 */
static int fsl_output_f_fsl_output( void * state, void const * s,
                                    fsl_size_t n ){
  return fsl_output( (fsl_cx *)state, s, n );
}


int fsl_outputfv( fsl_cx * const f, char const * fmt, va_list args ){
  if(!f || !fmt) return FSL_RC_MISUSE;
  else if(!*fmt) return FSL_RC_RANGE;
  return fsl_appendfv( fsl_output_f_fsl_output, f, fmt, args );
}
    
int fsl_outputf( fsl_cx * const f, char const * fmt, ... ){
  if(!f || !fmt) return FSL_RC_MISUSE;
  else if(!*fmt) return FSL_RC_RANGE;
  else{
    int rc;
    va_list args;
    va_start(args,fmt);
    rc = fsl_outputfv( f, fmt, args );
    va_end(args);
    return rc;
  }
}

int fsl_output( fsl_cx * const cx, void const * const src, fsl_size_t n ){
  if(!n || !cx->output.out) return 0;
  else return cx->output.out( cx->output.state, src, n );
}

int fsl_flush( fsl_cx * const f ){
  return f->output.flush
       ? f->output.flush(f->output.state)
       : 0;
}


int fsl_flush_f_FILE(void * _FILE){
  return fflush((FILE*)_FILE) ? fsl_errno_to_rc(errno, FSL_RC_IO) : 0;
}

int fsl_output_f_FILE( void * state,
                       void const * src, fsl_size_t n ){
  if(!n) return 0;
  else return (1 == fwrite(src, n, 1, state ? (FILE*)state : stdout))
         ? 0
         : FSL_RC_IO;
}

int fsl_input_f_FILE( void * state, void * dest, fsl_size_t * n ){
  if( !*n ) return FSL_RC_RANGE;
  FILE * f = (FILE*) state;
  *n = (fsl_size_t)fread( dest, 1, *n, f );
  return *n
    ? 0
    : (feof(f) ? 0 : FSL_RC_IO);
}

void fsl_finalizer_f_FILE( void * state __unused, void * mem ){
  if(mem){
    fsl_fclose((FILE*)mem);
  }
}

int fsl_stream( fsl_input_f inF, void * inState,
                fsl_output_f outF, void * outState ){
  if(!inF || !outF) return FSL_RC_MISUSE;
  else{
    int rc = 0;
    enum { BufSize = 1024 * 4 };
    unsigned char buf[BufSize];
    fsl_size_t rn = BufSize;
    for( ; !rc &&
           (rn==BufSize)
           && (0==(rc=inF(inState, buf, &rn)));
         rn = BufSize){
      if(rn) rc = outF(outState, buf, rn);
      else break;
    }
    return rc;
  }
}

int fsl_stream_compare( fsl_input_f in1, void * in1State,
                        fsl_input_f in2, void * in2State ){
  enum { BufSize = 1024 * 2 };
  unsigned char buf1[BufSize];
  unsigned char buf2[BufSize];
  fsl_size_t rn1 = BufSize;
  fsl_size_t rn2 = BufSize;
  int rc;
  while(1){
    rc = in1(in1State, buf1, &rn1);
    if(rc) return -1;
    rc = in2(in2State, buf2, &rn2);
    if(rc) return 1;
    else if(rn1!=rn2){
      rc = (rn1<rn2) ? -1 : 1;
      break;
    }
    else if(0==rn1 && 0==rn2) return 0;
    rc = memcmp( buf1, buf2, rn1 );
    if(rc) break;
    rn1 = rn2 = BufSize;
  }
  return rc;
}

#undef MARKER
/* end of file ./src/io.c */
/* start of file ./src/leaf.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/

/************************************************************************
  This file houses some of the "leaf"-related APIs.
*/
#include <assert.h>


/* Only for debugging */
#define MARKER(pfexp)                                               \
  do{ printf("MARKER: %s:%d:%s():\t",__FILE__,__LINE__,__func__);   \
    printf pfexp;                                                   \
  } while(0)

int fsl_repo_leaves_rebuild(fsl_cx * const f){
  fsl_db * const db = fsl_cx_db_repo(f);
  int rc = fsl_db_exec_multi(db,
    "DELETE FROM leaf;"
    "INSERT OR IGNORE INTO leaf"
    "  SELECT cid FROM plink"
    "  EXCEPT"
    "  SELECT pid FROM plink"
    "   WHERE coalesce((SELECT value FROM tagxref"
                       " WHERE tagid=%d AND rid=plink.pid),'trunk')"
         " == coalesce((SELECT value FROM tagxref"
                       " WHERE tagid=%d AND rid=plink.cid),'trunk')",
    FSL_TAGID_BRANCH, FSL_TAGID_BRANCH
  );
  return rc ? fsl_cx_uplift_db_error2(f, db, rc) : 0;
}

fsl_int_t fsl_count_nonbranch_children(fsl_cx * const f, fsl_id_t rid){
  int32_t rv = 0;
  int rc;
  fsl_db * const db = fsl_cx_db_repo(f);
  if(!db || !db->dbh || (rid<=0)) return -1;
  rc = fsl_db_get_int32(db, &rv,
                        "SELECT count(*) FROM plink "
                        "WHERE pid=%"FSL_ID_T_PFMT" "
                        "AND isprim "
                        "AND coalesce((SELECT value FROM tagxref "
                        "WHERE tagid=%d AND rid=plink.pid), 'trunk')"
                        "=coalesce((SELECT value FROM tagxref "
                        "WHERE tagid=%d AND rid=plink.cid), 'trunk')",
                        rid, FSL_TAGID_BRANCH, FSL_TAGID_BRANCH);
  return rc ? -2 : rv;
}

bool fsl_rid_is_leaf(fsl_cx * const f, fsl_id_t rid){
  int rv = -1;
  int rc;
  fsl_db * db = f ? fsl_cx_db_repo(f) : NULL;
  fsl_stmt * st = NULL;
  if(!db || !db->dbh || (rid<=0)) return 0;
  rc = fsl_db_prepare_cached(db, &st,
       "SELECT 1 FROM plink "
       "WHERE pid=?1 "
       "AND coalesce("
           "(SELECT value FROM tagxref "
            "WHERE tagid=%d AND rid=?1), "
          //"(SELECT value FROM config WHERE name='main-branch'), "
             "'trunk')"
           "=coalesce((SELECT value FROM tagxref "
             "WHERE tagid=%d "
             "AND rid=plink.cid), "
           //"(SELECT value FROM config WHERE name='main-branch'), "
             "'trunk')"
       "/*%s()*/",
       FSL_TAGID_BRANCH, FSL_TAGID_BRANCH, __func__);
  if(!rc){
    rc = fsl_stmt_bind_step(st, "R", rid);
    switch(rc){
      case FSL_RC_STEP_ROW:
        rv = 0;
        rc = 0;
        break;
      case 0:
        rv = 1;
        rc = 0;
        break;
      default:
        break;
    }
    fsl_stmt_cached_yield(st);
    assert(0==rv || 1==rv);
  }
  return rc ? 0 : (rv==1);
}

bool fsl_rid_is_version(fsl_cx * const f, fsl_id_t rid){
  fsl_db * const db = fsl_cx_db_repo(f);
  if(!db) return false;
  return 1==fsl_db_g_int32(db, 0,
                           "SELECT 1 FROM event "
                           "WHERE objid=%" FSL_ID_T_PFMT
                           " AND type='ci'", rid);
}

int fsl__repo_leafcheck(fsl_cx * const f, fsl_id_t rid){
  fsl_db * const db = f ? fsl_cx_db_repo(f) : NULL;
  if(!db || !db->dbh) return FSL_RC_MISUSE;
  else if(rid<=0) return FSL_RC_RANGE;
  else {
    int rc = 0;
    bool isLeaf;
    fsl_cx_err_reset(f);
    isLeaf = fsl_rid_is_leaf(f, rid);
    rc = fsl_cx_err_get(f, NULL, NULL);
    if(!rc){
      fsl_stmt * st = NULL;
      if( isLeaf ){
        rc = fsl_db_prepare_cached(db, &st,
                                   "INSERT OR IGNORE INTO leaf VALUES"
                                   "(?) /*%s()*/",__func__);
      }else{
        rc = fsl_db_prepare_cached(db, &st,
                                   "DELETE FROM leaf WHERE rid=?"
                                   "/*%s()*/",__func__);
      }
      if(!rc && st){
        rc = fsl_stmt_bind_step(st, "R", rid);
        fsl_stmt_cached_yield(st);
        if(rc) rc = fsl_cx_uplift_db_error2(f, db, rc);
      }
    }
    return rc;
  }
}

int fsl__repo_leafeventually_check( fsl_cx * const f, fsl_id_t rid){
  fsl_db * db = f ? fsl_cx_db_repo(f) : NULL;
  if(!f) return FSL_RC_MISUSE;
  else if(rid<=0) return FSL_RC_RANGE;
  else if(!db) return FSL_RC_NOT_A_REPO;
  else {
    fsl_stmt * parentsOf = NULL;
    int rc = fsl_db_prepare_cached(db, &parentsOf,
                            "SELECT pid FROM plink WHERE "
                            "cid=? AND pid>0"
                            "/*%s()*/",__func__);
    if(rc) return rc;
    rc = fsl_stmt_bind_id(parentsOf, 1, rid);
    if(!rc){
      rc = fsl_id_bag_insert(&f->cache.leafCheck, rid);
      while( !rc && (FSL_RC_STEP_ROW==fsl_stmt_step(parentsOf)) ){
        rc = fsl_id_bag_insert(&f->cache.leafCheck,
                               fsl_stmt_g_id(parentsOf, 0));
      }
    }
    fsl_stmt_cached_yield(parentsOf);
    return rc;
  }
}


int fsl__repo_leafdo_pending_checks(fsl_cx * const f){
  fsl_id_t rid;
  int rc = 0;
  for(rid=fsl_id_bag_first(&f->cache.leafCheck);
      !rc && rid; rid=fsl_id_bag_next(&f->cache.leafCheck,rid)){
    rc = fsl__repo_leafcheck(f, rid);
  }
  fsl_id_bag_reset(&f->cache.leafCheck);
  return rc;
}

int fsl_leaves_compute(fsl_cx * const f, fsl_id_t vid,
                       fsl_leaves_compute_e closeMode){
  fsl_db * const db = fsl_needs_repo(f);
  if(!db) return FSL_RC_NOT_A_REPO;
  int rc = 0;

  /* Create the LEAVES table if it does not already exist.  Make sure
  ** it is empty.
  */
  rc = fsl_db_exec_multi(db,
    "CREATE TEMP TABLE IF NOT EXISTS leaves("
    "  rid INTEGER PRIMARY KEY"
    ");"
    "DELETE FROM leaves;"
  );
  if(rc) goto dberr;
  if( vid <= 0 ){
    rc = fsl_db_exec_multi(db,
      "INSERT INTO leaves SELECT leaf.rid FROM leaf"
    );
    if(rc) goto dberr;
  }
  if( vid>0 ){
    fsl_id_bag seen = fsl_id_bag_empty;     /* Descendants seen */
    fsl_id_bag pending = fsl_id_bag_empty;  /* Unpropagated descendants */
    fsl_stmt q1 = fsl_stmt_empty;      /* Query to find children of a check-in */
    fsl_stmt isBr = fsl_stmt_empty;    /* Query to check to see if a check-in starts a new branch */
    fsl_stmt ins = fsl_stmt_empty;     /* INSERT statement for a new record */

    /* Initialize the bags. */
    rc = fsl_id_bag_insert(&pending, vid);
    if(rc) goto cleanup;

    /* This query returns all non-branch-merge children of check-in
    ** RID (?1).
    **
    ** If a child is a merge of a fork within the same branch, it is
    ** returned. Only merge children in different branches are excluded.
    */
    rc = fsl_db_prepare(db, &q1,
      "SELECT cid FROM plink"
      " WHERE pid=?1"
      "   AND (isprim"
      "        OR coalesce((SELECT value FROM tagxref"
                        "   WHERE tagid=%d AND rid=plink.pid), 'trunk')"
                        /* FIXME? main-branch? */
                 "=coalesce((SELECT value FROM tagxref"
                        "   WHERE tagid=%d AND rid=plink.cid), 'trunk'))"
                          /* FIXME? main-branch? */
                        ,
      FSL_TAGID_BRANCH, FSL_TAGID_BRANCH
    );
    if(rc) goto cleanup;
    /* This query returns a single row if check-in RID (?1) is the
    ** first check-in of a new branch. */
    rc = fsl_db_prepare(db, &isBr,
       "SELECT 1 FROM tagxref"
       " WHERE rid=?1 AND tagid=%d AND tagtype=2"
       "   AND srcid>0",
       FSL_TAGID_BRANCH
    );
    if(rc) goto cleanup;

    /* This statement inserts check-in RID (?1) into the LEAVES table.*/
    rc = fsl_db_prepare(db, &ins,
                        "INSERT OR IGNORE INTO leaves VALUES(?1)");
    if(rc) goto cleanup;

    while( fsl_id_bag_count(&pending) ){
      fsl_id_t const rid = fsl_id_bag_first(&pending);
      unsigned cnt = 0;
      fsl_id_bag_remove(&pending, rid);
      fsl_stmt_bind_id(&q1, 1, rid);
      while( FSL_RC_STEP_ROW==(rc = fsl_stmt_step(&q1)) ){
        int const cid = fsl_stmt_g_id(&q1, 0);
        rc = fsl_id_bag_insert(&seen, cid);
        if(rc) break;
        rc = fsl_id_bag_insert(&pending, cid);
        if(rc) break;
        fsl_stmt_bind_id(&isBr, 1, cid);
        if( FSL_RC_STEP_DONE==fsl_stmt_step(&isBr) ){
          ++cnt;
        }
        fsl_stmt_reset(&isBr);
      }
      if(FSL_RC_STEP_DONE==rc) rc = 0;
      else if(rc) break;
      fsl_stmt_reset(&q1);
      if( cnt==0 && !fsl_rid_is_leaf(f, rid) ){
        ++cnt;
      }
      if( cnt==0 ){
        fsl_stmt_bind_id(&ins, 1, rid);
        rc = fsl_stmt_step(&ins);
        if(FSL_RC_STEP_DONE!=rc) break;
        rc = 0;
        fsl_stmt_reset(&ins);
      }
    }
    cleanup:
    fsl_stmt_finalize(&ins);
    fsl_stmt_finalize(&isBr);
    fsl_stmt_finalize(&q1);
    fsl_id_bag_clear(&pending);
    fsl_id_bag_clear(&seen);
    if(rc) goto dberr;
  }
  assert(!rc);
  switch(closeMode){
    case FSL_LEAVES_COMPUTE_OPEN:
      rc =
        fsl_db_exec_multi(db,
                          "DELETE FROM leaves WHERE rid IN"
                          "  (SELECT leaves.rid FROM leaves, tagxref"
                          "    WHERE tagxref.rid=leaves.rid "
                          "      AND tagxref.tagid=%d"
                          "      AND tagxref.tagtype>0)",
                          FSL_TAGID_CLOSED);
      if(rc) goto dberr;
      break;
    case FSL_LEAVES_COMPUTE_CLOSED:
      rc = 
        fsl_db_exec_multi(db,
                          "DELETE FROM leaves WHERE rid NOT IN"
                          "  (SELECT leaves.rid FROM leaves, tagxref"
                          "    WHERE tagxref.rid=leaves.rid "
                          "      AND tagxref.tagid=%d"
                          "      AND tagxref.tagtype>0)",
                          FSL_TAGID_CLOSED);
      if(rc) goto dberr;
      break;
    default: break;
  }

  end:
  return rc;
  dberr:
  assert(rc);
  rc = fsl_cx_uplift_db_error2(f, db, rc);
  goto end;
}

bool fsl_leaves_computed_has(fsl_cx * const f){
  return fsl_db_exists(fsl_cx_db_repo(f),
                       "SELECT 1 FROM leaves");
}

fsl_int_t fsl_leaves_computed_count(fsl_cx * const f){
  int32_t rv = -1;
  fsl_db * const db = fsl_cx_db_repo(f);
  int const rc = fsl_db_get_int32(db, &rv,
                                 "SELECT COUNT(*) FROM leaves");
  if(rc){
    fsl_cx_uplift_db_error2(f, db, rc);
    assert(-1==rv);
  }else{
    assert(rv>=0);
  }
  return rv;
}

fsl_id_t fsl_leaves_computed_latest(fsl_cx * const f){
  fsl_id_t rv = 0;
  fsl_db * const db = fsl_cx_db_repo(f);
  int const rc =
    fsl_db_get_id(db, &rv,
                  "SELECT rid FROM leaves, event"
                  " WHERE event.objid=leaves.rid"
                  " ORDER BY event.mtime DESC");
  if(rc){
    fsl_cx_uplift_db_error2(f, db, rc);
    assert(!rv);
  }else{
    assert(rv>=0);
  }
  return rv;
}

void fsl_leaves_computed_cleanup(fsl_cx * const f){
  if(fsl_cx_exec(f, "DROP TABLE IF EXISTS temp.leaves")){
    /**
       Naively assume that locking is keeping us from dropping it,
       and simply empty it instead. */
    fsl_cx_exec(f, "DELETE FROM temp.leaves");
  }
}

#undef MARKER
/* end of file ./src/leaf.c */
/* start of file ./src/list.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2013-2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/
/************************************************************************
  This file houses the implementations for the fsl_list routines.
*/

#include <assert.h>
#include <stdlib.h> /* malloc() and friends, qsort() */
#include <memory.h> /* memset() */
int fsl_list_reserve( fsl_list * const self, fsl_size_t n )
{
  if( !self ) return FSL_RC_MISUSE;
  else if(0 == n){
    if(0 == self->capacity) return 0;
    fsl_free(self->list);
    *self = fsl_list_empty;
    return 0;
  }
  else if( self->capacity >= n ){
    return 0;
  }
  else{
    size_t const sz = sizeof(void*) * n;
    void* * m = (void**)fsl_realloc( self->list, sz );
    if( !m ) return FSL_RC_OOM;
    memset( m + self->capacity, 0, (sizeof(void*)*(n-self->capacity)));
    self->capacity = n;
    self->list = m;
    return 0;
  }
}

void fsl_list_swap( fsl_list * const lhs, fsl_list * const rhs ){
  fsl_list tmp = *lhs;
  *rhs = *lhs;
  *lhs = tmp;
}

int fsl_list_append( fsl_list * const self, void* cp ){
  if( !self ) return FSL_RC_MISUSE;
  assert(self->used <= self->capacity);
  if(self->used == self->capacity){
    int rc;
    fsl_size_t const cap = self->capacity
      ? (self->capacity * 2)
      : 10;
    rc = fsl_list_reserve(self, cap);
    if(rc) return rc;
  }
  self->list[self->used++] = cp;
  if(self->used<self->capacity) self->list[self->used]=NULL;
  return 0;
}

int fsl_list_v_fsl_free(void * obj, void * visitorState __unused){
  if(obj) fsl_free( obj );
  return 0;
}

int fsl_list_clear( fsl_list * const self, fsl_list_visitor_f childFinalizer,
                    void * finalizerState ){
  /*
    TODO: manually traverse the list and set each list entry for which
    the finalizer succeeds to NULL, so that we can provide
    well-defined behaviour if childFinalizer() fails and we abort the
    loop.
   */
  int rc = fsl_list_visit(self, 0, childFinalizer, finalizerState );
  if(!rc) fsl_list_reserve(self, 0);
  return rc;
}

void fsl_list_visit_free( fsl_list * const self, bool freeListMem ){
  fsl_list_visit(self, 0, fsl_list_v_fsl_free, NULL );
  if(freeListMem) fsl_list_reserve(self, 0);
  else self->used = 0;
}


int fsl_list_visit( fsl_list const * self, int order,
                    fsl_list_visitor_f visitor,
                    void * visitorState ){
  int rc = FSL_RC_OK;
  if( self && self->used && visitor ){
    fsl_int_t i = 0;
    fsl_int_t pos = (order<0) ? self->used-1 : 0;
    fsl_int_t step = (order<0) ? -1 : 1;
    for( rc = 0; (i < (fsl_int_t)self->used)
           && (0 == rc); ++i, pos+=step ){
      void* obj = self->list[pos];
      if(obj) rc = visitor( obj, visitorState );
      if( obj != self->list[pos] ){
        --i;
        if(order>=0) pos -= step;
      }
    }
  }
  return rc;
}


int fsl_list_visit_p( fsl_list * const self, int order,
                      bool shiftIfNulled,
                      fsl_list_visitor_f visitor, void * visitorState )
{
  int rc = FSL_RC_OK;
  if( self && self->used && visitor ){
    fsl_int_t i = 0;
    fsl_int_t pos = (order<0) ? self->used-1 : 0;
    fsl_int_t step = (order<0) ? -1 : 1;
    for( rc = 0; (i < (fsl_int_t)self->used)
           && (0 == rc); ++i, pos+=step ){
      void* obj = self->list[pos];
      if(obj) {
        assert((order<0) && "TEST THAT THIS WORKS WITH IN-ORDER!");
        rc = visitor( &self->list[pos], visitorState );
        if( shiftIfNulled && !self->list[pos]){
          fsl_int_t x = pos;
          fsl_int_t const to = self->used-pos;

          assert( to < (fsl_int_t) self->capacity );
          for( ; x < to; ++x ) self->list[x] = self->list[x+1];
          if( x < (fsl_int_t)self->capacity ) self->list[x] = 0;
          --i;
          --self->used;
          if(order>=0) pos -= step;
        }
      }
    }
  }
  return rc;
}

void fsl_list_sort( fsl_list * const li,
                    fsl_generic_cmp_f cmp){
  if(li && li->used>1){
    qsort( li->list, li->used, sizeof(void*), cmp );
  }
}



fsl_int_t fsl_list_index_of( fsl_list const * li,
                             void const * key,
                             fsl_generic_cmp_f cmpf ){
  fsl_size_t i;
  void const * p;
  for(i = 0; i < li->used; ++i){
    p = li->list[i];
    if(!p){
      if(!key) return (fsl_int_t)i;
      else continue;
    }else if((p==key) ||
            (0==cmpf( key, p )) ){
      return (fsl_int_t)i;
    }
  }
  return -1;
}


fsl_int_t fsl_list_index_of_cstr( fsl_list const * li,
                                  char const * key ){
  return fsl_list_index_of(li, key, fsl_strcmp_cmp);
}
/* end of file ./src/list.c */
/* start of file ./src/lookslike.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/* vim: set ts=2 et sw=2 tw=80: */
/*
  Copyright 2021 The Libfossil Authors, see LICENSES/BSD-2-Clause.txt

  SPDX-License-Identifier: BSD-2-Clause-FreeBSD
  SPDX-FileCopyrightText: 2021 The Libfossil Authors
  SPDX-ArtifactOfProjectName: Libfossil
  SPDX-FileType: Code

  Heavily indebted to the Fossil SCM project (https://fossil-scm.org).
*/

#include <string.h> /* memcmp() */


/* definitions for various UTF-8 sequence lengths, encoded as start value
 * and size of each valid range belonging to some lead byte*/
#define US2A  0x80, 0x01 /* for lead byte 0xC0 */
#define US2B  0x80, 0x40 /* for lead bytes 0xC2-0xDF */
#define US3A  0xA0, 0x20 /* for lead byte 0xE0 */
#define US3B  0x80, 0x40 /* for lead bytes 0xE1-0xEF */
#define US4A  0x90, 0x30 /* for lead byte 0xF0 */
#define US4B  0x80, 0x40 /* for lead bytes 0xF1-0xF3 */
#define US4C  0x80, 0x10 /* for lead byte 0xF4 */
#define US0A  0x00, 0x00 /* for any other lead byte */

/* a table used for quick lookup of the definition that goes with a
 * particular lead byte */
static const unsigned char lb_tab[] = {
  US0A, US0A, US0A, US0A, US0A, US0A, US0A, US0A,
  US0A, US0A, US0A, US0A, US0A, US0A, US0A, US0A,
  US0A, US0A, US0A, US0A, US0A, US0A, US0A, US0A,
  US0A, US0A, US0A, US0A, US0A, US0A, US0A, US0A,
  US0A, US0A, US0A, US0A, US0A, US0A, US0A, US0A,
  US0A, US0A, US0A, US0A, US0A, US0A, US0A, US0A,
  US0A, US0A, US0A, US0A, US0A, US0A, US0A, US0A,
  US0A, US0A, US0A, US0A, US0A, US0A, US0A, US0A,
  US2A, US0A, US2B, US2B, US2B, US2B, US2B, US2B,
  US2B, US2B, US2B, US2B, US2B, US2B, US2B, US2B,
  US2B, US2B, US2B, US2B, US2B, US2B, US2B, US2B,
  US2B, US2B, US2B, US2B, US2B, US2B, US2B, US2B,
  US3A, US3B, US3B, US3B, US3B, US3B, US3B, US3B,
  US3B, US3B, US3B, US3B, US3B, US3B, US3B, US3B,
  US4A, US4B, US4B, US4B, US4C, US0A, US0A, US0A,
  US0A, US0A, US0A, US0A, US0A, US0A, US0A, US0A
};

#undef US2A
#undef US2B
#undef US3A
#undef US3B
#undef US4A
#undef US4B
#undef US4C
#undef US0A

bool fsl_looks_like_binary(fsl_buffer const * const b){
  return (fsl_looks_like_utf8(b, FSL_LOOKSLIKE_BINARY) & FSL_LOOKSLIKE_BINARY)
    != FSL_LOOKSLIKE_NONE;
}

int fsl_looks_like_utf8(fsl_buffer const * const b, int stopFlags){
  fsl_size_t n = 0;
  const char *z = fsl_buffer_cstr2(b, &n);
  int j, c, flags = FSL_LOOKSLIKE_NONE;  /* Assume UTF-8 text, prove otherwise */

  if( n==0 ) return flags;  /* Empty file -> text */
  c = *z;
  if( c==0 ){
    flags |= FSL_LOOKSLIKE_NUL;  /* NUL character in a file -> binary */
  }else if( c=='\r' ){
    flags |= FSL_LOOKSLIKE_CR;
    if( n<=1 || z[1]!='\n' ){
      flags |= FSL_LOOKSLIKE_LONE_CR;  /* Not enough chars or next char not LF */
    }
  }
  j = (c!='\n');
  if( !j ) flags |= (FSL_LOOKSLIKE_LF | FSL_LOOKSLIKE_LONE_LF);  /* Found LF as first char */
  while( !(flags&stopFlags) && --n>0 ){
    int c2 = c;
    c = *++z; ++j;
    if( c==0 ){
      flags |= FSL_LOOKSLIKE_NUL;  /* NUL character in a file -> binary */
    }else if( c=='\n' ){
      flags |= FSL_LOOKSLIKE_LF;
      if( c2=='\r' ){
        flags |= (FSL_LOOKSLIKE_CR | FSL_LOOKSLIKE_CRLF);  /* Found LF preceded by CR */
      }else{
        flags |= FSL_LOOKSLIKE_LONE_LF;
      }
      if( j>FSL__LINE_LENGTH_MASK ){
        flags |= FSL_LOOKSLIKE_LONG;  /* Very long line -> binary */
      }
      j = 0;
    }else if( c=='\r' ){
      flags |= FSL_LOOKSLIKE_CR;
      if( n<=1 || z[1]!='\n' ){
        flags |= FSL_LOOKSLIKE_LONE_CR;  /* Not enough chars or next char not LF */
      }
    }
  }
  if( n ){
    flags |= FSL_LOOKSLIKE_SHORT;  /* The whole blob was not examined */
  }
  if( j>FSL__LINE_LENGTH_MASK ){
    flags |= FSL_LOOKSLIKE_LONG;  /* Very long line -> binary */
  }
  return flags;
}

unsigned char const *fsl_utf8_bom(unsigned int *pnByte){
  static const unsigned char bom[] = {
    0xef, 0xbb, 0xbf, 0x00, 0x00, 0x00
  };
  if( pnByte ) *pnByte = 3;
  return bom;
}

bool fsl_starts_with_bom_utf8(fsl_buffer const * const b,
                              unsigned int *pBomSize){
  unsigned int bomSize;
  const char * const z = fsl_buffer_cstr(b);
  const unsigned char * const bom = fsl_utf8_bom(&bomSize);
  if( pBomSize ) *pBomSize = bomSize;
  return fsl_buffer_size(b)<bomSize
    ? false
    : memcmp(z, bom, bomSize)==0;
}

bool fsl_invalid_utf8(fsl_buffer const * const b){
  fsl_size_t n = 0;
  const unsigned char *z = (unsigned char *) fsl_buffer_cstr2(b, &n);
  unsigned char c; /* lead byte to be handled. */
  if( n==0 ) return false;  /* Empty file -> OK */
  c = *z;
  while( --n>0 ){
    if( c>=0x80 ){
      const unsigned char *def; /* pointer to range table*/
      c <<= 1; /* multiply by 2 and get rid of highest bit */
      def = &lb_tab[c]; /* search fb's valid range in table */
      if( (unsigned int)(*++z-def[0])>=def[1] ){
        return false/*FSL_LOOKSLIKE_INVALID*/;
      }
      c = (c>=0xC0) ? (c|3) : ' '; /* determine next lead byte */
    } else {
      c = *++z;
    }
  }
  return c<0x80 /* Final lead byte must be ASCII. */;
}
/* end of file ./src/lookslike.c */
/* start of file ./src/md5.c */
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */ 
/*
   The code is modified for use in fossil (then libfossil).  The
   original header comment follows:
*/
/*
 * This code implements the MD5 message-digest algorithm.
 * The algorithm is due to Ron Rivest.  This code was
 * written by Colin Plumb in 1993, no copyright is claimed.
 * This code is in the public domain; do with it what you wish.
 *
 * Equivalent code is available from RSA Data Security, Inc.
 * This code has been tested against that, and is equivalent,
 * except that you don't need to include two pages of legalese
 * with every copy.
 *
 * To compute the message digest of a chunk of bytes, declare an
 * MD5Context structure, pass it to MD5Init, call MD5Update as
 * needed on buffers full of bytes, and then call MD5Final, which
 * will fill a supplied 16-byte array with the digest.
 */
#include <string.h>
#include <stdio.h>
#include <errno.h>

#if defined(__i386__) || defined(__x86_64__) || defined(_WIN32)
# define byteReverse(A,B)
#else
/*
 * Convert an array of integers to little-endian.
 * Note: this code is a no-op on little-endian machines.
 */
static void byteReverse (unsigned char *buf, unsigned longs){
    uint32_t t;
    do {
        t = (uint32_t)((unsigned)buf[3]<<8 | buf[2]) << 16 |
            ((unsigned)buf[1]<<8 | buf[0]);
        *(uint32_t *)buf = t;
        buf += 4;
    } while (--longs);
}
#endif

/* The four core functions - F1 is optimized somewhat */

/* #define F1(x, y, z) (x & y | ~x & z) */
#define F1(x, y, z) (z ^ (x & (y ^ z)))
#define F2(x, y, z) F1(z, x, y)
#define F3(x, y, z) (x ^ y ^ z)
#define F4(x, y, z) (y ^ (x | ~z))

/* This is the central step in the MD5 algorithm. */
#define MD5STEP(f, w, x, y, z, data, s)                         \
    ( w += f(x, y, z) + data,  w = w<<s | w>>(32-s),  w += x )

/*
 * The core of the MD5 algorithm, this alters an existing MD5 hash to
 * reflect the addition of 16 longwords of new data.  MD5Update blocks
 * the data and converts bytes into longwords for this routine.
 */
static void MD5Transform(uint32_t buf[4], const uint32_t in[16]){
    register uint32_t a, b, c, d;

    a = buf[0];
    b = buf[1];
    c = buf[2];
    d = buf[3];

    MD5STEP(F1, a, b, c, d, in[ 0]+0xd76aa478,  7);
    MD5STEP(F1, d, a, b, c, in[ 1]+0xe8c7b756, 12);
    MD5STEP(F1, c, d, a, b, in[ 2]+0x242070db, 17);
    MD5STEP(F1, b, c, d, a, in[ 3]+0xc1bdceee, 22);
    MD5STEP(F1, a, b, c, d, in[ 4]+0xf57c0faf,  7);
    MD5STEP(F1, d, a, b, c, in[ 5]+0x4787c62a, 12);
    MD5STEP(F1, c, d, a, b, in[ 6]+0xa8304613, 17);
    MD5STEP(F1, b, c, d, a, in[ 7]+0xfd469501, 22);
    MD5STEP(F1, a, b, c, d, in[ 8]+0x698098d8,  7);
    MD5STEP(F1, d, a, b, c, in[ 9]+0x8b44f7af, 12);
    MD5STEP(F1, c, d, a, b, in[10]+0xffff5bb1, 17);
    MD5STEP(F1, b, c, d, a, in[11]+0x895cd7be, 22);
    MD5STEP(F1, a, b, c, d, in[12]+0x6b901122,  7);
    MD5STEP(F1, d, a, b, c, in[13]+0xfd987193, 12);
    MD5STEP(F1, c, d, a, b, in[14]+0xa679438e, 17);
    MD5STEP(F1, b, c, d, a, in[15]+0x49b40821, 22);

    MD5STEP(F2, a, b, c, d, in[ 1]+0xf61e2562,  5);
    MD5STEP(F2, d, a, b, c, in[ 6]+0xc040b340,  9);
    MD5STEP(F2, c, d, a, b, in[11]+0x265e5a51, 14);
    MD5STEP(F2, b, c, d, a, in[ 0]+0xe9b6c7aa, 20);
    MD5STEP(F2, a, b, c, d, in[ 5]+0xd62f105d,  5);
    MD5STEP(F2, d, a, b, c, in[10]+0x02441453,  9);
    MD5STEP(F2, c, d, a, b, in[15]+0xd8a1e681, 14);
    MD5STEP(F2, b, c, d, a, in[ 4]+0xe7d3fbc8, 20);
    MD5STEP(F2, a, b, c, d, in[ 9]+0x21e1cde6,  5);
    MD5STEP(F2, d, a, b, c, in[14]+0xc33707d6,  9);
    MD5STEP(F2, c, d, a, b, in[ 3]+0xf4d50d87, 14);
    MD5STEP(F2, b, c, d, a, in[ 8]+0x455a14ed, 20);
    MD5STEP(F2, a, b, c, d, in[13]+0xa9e3e905,  5);
    MD5STEP(F2, d, a, b, c, in[ 2]+0xfcefa3f8,  9);
    MD5STEP(F2, c, d, a, b, in[ 7]+0x676f02d9, 14);
    MD5STEP(F2, b, c, d, a, in[12]+0x8d2a4c8a, 20);

    MD5STEP(F3, a, b, c, d, in[ 5]+0xfffa3942,  4);
    MD5STEP(F3, d, a, b, c, in[ 8]+0x8771f681, 11);
    MD5STEP(F3, c, d, a, b, in[11]+0x6d9d6122, 16);
    MD5STEP(F3, b, c, d, a, in[14]+0xfde5380c, 23);
    MD5STEP(F3, a, b, c, d, in[ 1]+0xa4beea44,  4);
    MD5STEP(F3, d, a, b, c, in[ 4]+0x4bdecfa9, 11);
    MD5STEP(F3, c, d, a, b, in[ 7]+0xf6bb4b60, 16);
    MD5STEP(F3, b, c, d, a, in[10]+0xbebfbc70, 23);
    MD5STEP(F3, a, b, c, d, in[13]+0x289b7ec6,  4);
    MD5STEP(F3, d, a, b, c, in[ 0]+0xeaa127fa, 11);
    MD5STEP(F3, c, d, a, b, in[ 3]+0xd4ef3085, 16);
    MD5STEP(F3, b, c, d, a, in[ 6]+0x04881d05, 23);
    MD5STEP(F3, a, b, c, d, in[ 9]+0xd9d4d039,  4);
    MD5STEP(F3, d, a, b, c, in[12]+0xe6db99e5, 11);
    MD5STEP(F3, c, d, a, b, in[15]+0x1fa27cf8, 16);
    MD5STEP(F3, b, c, d, a, in[ 2]+0xc4ac5665, 23);

    MD5STEP(F4, a, b, c, d, in[ 0]+0xf4292244,  6);
    MD5STEP(F4, d, a, b, c, in[ 7]+0x432aff97, 10);
    MD5STEP(F4, c, d, a, b, in[14]+0xab9423a7, 15);
    MD5STEP(F4, b, c, d, a, in[ 5]+0xfc93a039, 21);
    MD5STEP(F4, a, b, c, d, in[12]+0x655b59c3,  6);
    MD5STEP(F4, d, a, b, c, in[ 3]+0x8f0ccc92, 10);
    MD5STEP(F4, c, d, a, b, in[10]+0xffeff47d, 15);
    MD5STEP(F4, b, c, d, a, in[ 1]+0x85845dd1, 21);
    MD5STEP(F4, a, b, c, d, in[ 8]+0x6fa87e4f,  6);
    MD5STEP(F4, d, a, b, c, in[15]+0xfe2ce6e0, 10);
    MD5STEP(F4, c, d, a, b, in[ 6]+0xa3014314, 15);
    MD5STEP(F4, b, c, d, a, in[13]+0x4e0811a1, 21);
    MD5STEP(F4, a, b, c, d, in[ 4]+0xf7537e82,  6);
    MD5STEP(F4, d, a, b, c, in[11]+0xbd3af235, 10);
    MD5STEP(F4, c, d, a, b, in[ 2]+0x2ad7d2bb, 15);
    MD5STEP(F4, b, c, d, a, in[ 9]+0xeb86d391, 21);

    buf[0] += a;
    buf[1] += b;
    buf[2] += c;
    buf[3] += d;
}

const fsl_md5_cx fsl_md5_cx_empty = fsl_md5_cx_empty_m;
/*
 * Start MD5 accumulation.  Set bit count to 0 and buffer to mysterious
 * initialization constants.
 */
void fsl_md5_init(fsl_md5_cx *ctx){
    *ctx = fsl_md5_cx_empty;
}

/*
 * Update context to reflect the concatenation of another buffer full
 * of bytes.
 */
void fsl_md5_update(fsl_md5_cx *ctx, void const * buf_, fsl_size_t len){
    const unsigned char * buf = (const unsigned char *)buf_;
    uint32_t t;
    
    /* Update bitcount */
    
    t = ctx->bits[0];
    if ((ctx->bits[0] = t + ((uint32_t)len << 3)) < t)
        ctx->bits[1]++; /* Carry from low to high */
    ctx->bits[1] += len >> 29;
    
    t = (t >> 3) & 0x3f;    /* Bytes already in shsInfo->data */

    /* Handle any leading odd-sized chunks */

    if ( t ) {
        unsigned char *p = (unsigned char *)ctx->in + t;

        t = 64-t;
        if (len < t) {
            memcpy(p, buf, len);
            return;
        }
        memcpy(p, buf, t);
        byteReverse(ctx->in, 16);
        MD5Transform(ctx->buf, (uint32_t *)ctx->in);
        buf += t;
        len -= t;
    }

    /* Process data in 64-byte chunks */

    while (len >= 64) {
        memcpy(ctx->in, buf, 64);
        byteReverse(ctx->in, 16);
        MD5Transform(ctx->buf, (uint32_t *)ctx->in);
        buf += 64;
        len -= 64;
    }

    /* Handle any remaining bytes of data. */

    memcpy(ctx->in, buf, len);
}

/*
 * Final wrapup - pad to 64-byte boundary with the bit pattern 
 * 1 0* (64-bit count of bits processed, MSB-first)
 */
void fsl_md5_final(fsl_md5_cx * ctx, unsigned char * digest){
    unsigned count;
    unsigned char *p;

    /* Compute number of bytes mod 64 */
    count = (ctx->bits[0] >> 3) & 0x3F;

    /* Set the first char of padding to 0x80.  This is safe since there is
       always at least one byte free */
    p = ctx->in + count;
    *p++ = 0x80;

    /* Bytes of padding needed to make 64 bytes */
    count = 64 - 1 - count;

    /* Pad out to 56 mod 64 */
    if (count < 8) {
        /* Two lots of padding:  Pad the first block to 64 bytes */
        memset(p, 0, count);
        byteReverse(ctx->in, 16);
        MD5Transform(ctx->buf, (uint32_t *)ctx->in);

        /* Now fill the next block with 56 bytes */
        memset(ctx->in, 0, 56);
    } else {
        /* Pad block to 56 bytes */
        memset(p, 0, count-8);
    }
    byteReverse(ctx->in, 14);

    /* Append length in bits and transform */
    memcpy(&ctx->in[14*sizeof(uint32_t)], 