/*
 * Copyright (c) 2022 Mark Jamsek <mark@bsdbox.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "libfossil.h"
#include "settings.h"

/*
 * Flags set by callers of the below diff APIs to determine diff output.
 */
enum fnc_diff_flag {
	FNC_DIFF_IGNORE_EOLWS	= 0x01,
	FNC_DIFF_IGNORE_ALLWS	= 0x03,
	FNC_DIFF_SIDEBYSIDE	= 1 << 2,  /* output side-by-side diff */
	FNC_DIFF_VERBOSE	= 1 << 3,  /* show added/rm'd file content */
	FNC_DIFF_BRIEF		= 1 << 4,
	FNC_DIFF_HTML		= 1 << 5,
	FNC_DIFF_LINENO		= 1 << 6,  /* output diff with line numbers */
	FNC_DIFF_NOOPT		= 1 << 7,  /* og. 0x0100 */
	FNC_DIFF_INVERT		= 1 << 8,  /* og. 0x0200 */
	FNC_DIFF_NOTTOOBIG	= 1 << 9,  /* og. 0x0800 */
	FNC_DIFF_STRIP_EOLCR	= 1 << 10, /* og. 0x1000 */
	FNC_DIFF_ANSI_COLOR	= 1 << 11, /* og. 0x2000 */
	FNC_DIFF_PROTOTYPE	= 1 << 12  /* show func sig in hunk header */
#define FNC_DIFF_CONTEXT_EX	(((uint64_t)0x04) << 32)  /* Allow 0 context */
#define FNC_DIFF_CONTEXT_MASK	((uint64_t)0x0000ffff)    /* Default context */
#define FNC_DIFF_WIDTH_MASK	((uint64_t)0x00ff0000)    /* SBS column width */
};

/*
 * Compute the diff of changes to convert the file in fsl_buffer parameter 1
 * to the file in fsl_buffer parameter 2 and save the result to the provided
 * output fsl_buffer in parameter 3.
 *
 * A unified diff is output by default. This, along with other diff options
 * (detailed in the above fnc_diff_flag enum), can be changed by setting the
 * corresponding flags passed in int parameter 8.
 *
 * If a unified diff, parameter 7 is ignored, and the number of context lines
 * is specified in short parameter 6. Negative values fallback to default. If
 * a side-by-side diff, parameter 6 is ignored, and the column width of each
 * side is specified in short parameter 7; only values larger than the longest
 * line are honoured, otherwise the column width of each side will
 * automatically grow to accommodate the longest line in the diff.

 * If not NULL, the enum array pointer in paramater 4 and uint32_t pointer in
 * parameter 5 will be populated with each line_type and the total number of
 * lines in the diff, respectively.  Both pointers and the output buffer can
 * be prepopulated and must be disposed of by the caller.
 */
int fnc_diff_text_to_buffer(const fsl_buffer *, const fsl_buffer *,
    fsl_buffer *, enum line_type **, uint32_t *, short, short, int);

/*
 * Compute the diff of changes to convert the file in fsl_buffer parameter 1
 * to the file in fsl_buffer parameter 2 and invoke the fsl_output_f callback
 * in parameter 3 for each computed line.  The callback receives the provided
 * void parameter 4 as its output state and a char pointer of the diffed line
 * or diff metadata (e.g., hunk header, index).  Remaining parameters are the
 * same as the above fnc_diff_text_to_buffer() routine.
 */
int fnc_diff_text(const fsl_buffer *, const fsl_buffer *, fsl_output_f, void *,
    enum line_type **, uint32_t *, short, short, int);

/*
 * Compute and save to the int array pointer in parameter 4 the array of
 * copy/delete/insert triples that describes the sequence of changes to convert
 * the file in fsl_buffer parameter 1 to the file in fsl_buffer parameter 2.
 * Diff format and related options are set via flags passed in int parameter 3.
 */
int fnc_diff_text_raw(const fsl_buffer *, const fsl_buffer *, int, int **);

/*
 * Return the number of columns required to draw to the screen the char pointer
 * in parameter 1 by expanding any tabs and accounting for unicode continuation
 * bytes. Short parameter 2 may either be the byte size of the string or
 * a negative value.
 */
unsigned short etcount(const char *, unsigned short);

/*
 * Save the line_type specified in parameter 3 to the nth index denoted by the
 * uint32_t pointer in parameter 2 of the line_type array in parameter 1. The
 * uint32_t value pointed to by the 2nd parameter will be incremented.
 */
int add_line_type(enum line_type **, uint32_t *, enum line_type);
