/*
 * Decompiled with CFR 0.152.
 */
package org.flywaydb.core.internal.jdbc;

import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.Driver;
import java.sql.SQLException;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import javax.sql.DataSource;
import org.flywaydb.core.api.ErrorCode;
import org.flywaydb.core.api.FlywayException;
import org.flywaydb.core.api.logging.Log;
import org.flywaydb.core.api.logging.LogFactory;
import org.flywaydb.core.internal.util.ClassUtils;
import org.flywaydb.core.internal.util.FeatureDetector;
import org.flywaydb.core.internal.util.StringUtils;

public class DriverDataSource
implements DataSource {
    private static final Log LOG = LogFactory.getLog(DriverDataSource.class);
    private static final String MYSQL_LEGACY_JDBC_DRIVER = "com.mysql.jdbc.Driver";
    private static final String REDSHIFT_JDBC4_DRIVER = "com.amazon.redshift.jdbc4.Driver";
    private static final String REDSHIFT_JDBC41_DRIVER = "com.amazon.redshift.jdbc41.Driver";
    private static final String APPLICATION_NAME = "Flyway by Redgate";
    private Driver driver;
    private final String url;
    private final DriverType type;
    private final String user;
    private final String password;
    private final Properties defaultProps;
    private final ClassLoader classLoader;
    private boolean autoCommit = true;

    public DriverDataSource(ClassLoader classLoader, String driverClass, String url, String user, String password) throws FlywayException {
        this(classLoader, driverClass, url, user, password, new Properties());
    }

    public DriverDataSource(ClassLoader classLoader, String driverClass, String url, String user, String password, Properties props) throws FlywayException {
        this.classLoader = classLoader;
        this.url = this.detectFallbackUrl(url);
        this.type = this.detectDriverTypeForUrl(url);
        if (!StringUtils.hasLength(driverClass)) {
            if (this.type == null) {
                throw new FlywayException("Unable to autodetect JDBC driver for url: " + url);
            }
            driverClass = this.detectDriverForType(this.type);
        }
        this.defaultProps = new Properties(props);
        this.defaultProps.putAll((Map<?, ?>)this.detectPropsForType(this.type));
        try {
            this.driver = (Driver)ClassUtils.instantiate(driverClass, classLoader);
        }
        catch (FlywayException e) {
            String backupDriverClass = this.detectBackupDriverForType(this.type);
            if (backupDriverClass == null) {
                throw new FlywayException("Unable to instantiate JDBC driver: " + driverClass + " => Check whether the jar file is present", e, ErrorCode.JDBC_DRIVER);
            }
            try {
                this.driver = (Driver)ClassUtils.instantiate(backupDriverClass, classLoader);
            }
            catch (Exception e1) {
                throw new FlywayException("Unable to instantiate JDBC driver: " + driverClass + " => Check whether the jar file is present", e, ErrorCode.JDBC_DRIVER);
            }
        }
        this.user = this.detectFallbackUser(user);
        this.password = this.detectFallbackPassword(password);
    }

    private String detectFallbackUrl(String url) {
        if (!StringUtils.hasText(url)) {
            String boxfuseDatabaseUrl = System.getenv("BOXFUSE_DATABASE_URL");
            if (StringUtils.hasText(boxfuseDatabaseUrl)) {
                return boxfuseDatabaseUrl;
            }
            throw new FlywayException("Missing required JDBC URL. Unable to create DataSource!");
        }
        if (!url.toLowerCase().startsWith("jdbc:")) {
            throw new FlywayException("Invalid JDBC URL (should start with jdbc:) : " + url);
        }
        return url;
    }

    private String detectFallbackUser(String user) {
        String boxfuseDatabaseUser;
        if (!StringUtils.hasText(user) && StringUtils.hasText(boxfuseDatabaseUser = System.getenv("BOXFUSE_DATABASE_USER"))) {
            return boxfuseDatabaseUser;
        }
        return user;
    }

    public static boolean detectUserRequiredByUrl(String url) {
        if (DriverType.SNOWFLAKE.matches(url) || DriverType.POSTGRESQL.matches(url)) {
            return !url.contains("user=");
        }
        if (DriverType.SQLSERVER.matches(url)) {
            return !url.contains("integratedSecurity=") && !url.contains("authentication=ActiveDirectoryIntegrated") && !url.contains("authentication=ActiveDirectoryMSI");
        }
        if (DriverType.ORACLE.matches(url)) {
            Pattern pattern = Pattern.compile("^jdbc:oracle:thin:[a-zA-Z0-9#_$]+/[a-zA-Z0-9#_$]+@//.*");
            return !pattern.matcher(url).matches();
        }
        return true;
    }

    public static boolean detectPasswordRequiredByUrl(String url) {
        if (DriverType.SNOWFLAKE.matches(url)) {
            return !url.contains("private_key_file=");
        }
        if (DriverType.POSTGRESQL.matches(url)) {
            return !url.contains("password=");
        }
        if (DriverType.SQLSERVER.matches(url)) {
            return !url.contains("integratedSecurity=") && !url.contains("authentication=ActiveDirectoryIntegrated") && !url.contains("authentication=ActiveDirectoryMSI");
        }
        if (DriverType.ORACLE.matches(url)) {
            Pattern pattern = Pattern.compile("^jdbc:oracle:thin:[a-zA-Z0-9#_$]+/[a-zA-Z0-9#_$]+@//.*");
            return !pattern.matcher(url).matches();
        }
        return true;
    }

    private String detectFallbackPassword(String password) {
        String boxfuseDatabasePassword;
        if (!StringUtils.hasText(password) && StringUtils.hasText(boxfuseDatabasePassword = System.getenv("BOXFUSE_DATABASE_PASSWORD"))) {
            return boxfuseDatabasePassword;
        }
        return password;
    }

    private Properties detectPropsForType(DriverType type) {
        Properties result = new Properties();
        if (DriverType.ORACLE.equals((Object)type)) {
            String osUser = System.getProperty("user.name");
            result.put("v$session.osuser", osUser.substring(0, Math.min(osUser.length(), 30)));
            result.put("v$session.program", APPLICATION_NAME);
            result.put("oracle.net.keepAlive", "true");
        } else if (DriverType.SQLSERVER.equals((Object)type)) {
            result.put("applicationName", APPLICATION_NAME);
        } else if (DriverType.POSTGRESQL.equals((Object)type)) {
            result.put("ApplicationName", APPLICATION_NAME);
        } else if (DriverType.MYSQL.equals((Object)type) || DriverType.MARIADB.equals((Object)type)) {
            result.put("connectionAttributes", "program_name:Flyway by Redgate");
        } else if (DriverType.DB2.equals((Object)type)) {
            result.put("clientProgramName", APPLICATION_NAME);
            result.put("retrieveMessagesFromServerOnGetMessage", "true");
        } else if (DriverType.SYBASE.equals((Object)type)) {
            result.put("APPLICATIONNAME", APPLICATION_NAME);
        } else if (DriverType.SAPHANA.equals((Object)type)) {
            result.put("SESSIONVARIABLE:APPLICATION", APPLICATION_NAME);
        } else if (DriverType.FIREBIRD_SQL.equals((Object)type) || DriverType.FIREBIRD.equals((Object)type)) {
            result.put("processName", APPLICATION_NAME);
        }
        return result;
    }

    private DriverType detectDriverTypeForUrl(String url) {
        for (DriverType type : DriverType.values()) {
            if (!type.matches(url)) continue;
            return type;
        }
        return null;
    }

    private String detectBackupDriverForType(DriverType type) {
        if (DriverType.MYSQL.equals((Object)type) && ClassUtils.isPresent(MYSQL_LEGACY_JDBC_DRIVER, this.classLoader)) {
            return MYSQL_LEGACY_JDBC_DRIVER;
        }
        if (DriverType.MYSQL.equals((Object)type) && ClassUtils.isPresent(DriverType.MARIADB.driverClass, this.classLoader)) {
            LOG.warn("You are attempting to connect to a MySQL database using the MariaDB driver. This is known to cause issues. An upgrade to Oracle's MySQL JDBC driver is highly recommended.");
            return DriverType.MARIADB.driverClass;
        }
        if (DriverType.REDSHIFT.equals((Object)type)) {
            if (ClassUtils.isPresent(REDSHIFT_JDBC41_DRIVER, this.classLoader)) {
                return REDSHIFT_JDBC41_DRIVER;
            }
            return REDSHIFT_JDBC4_DRIVER;
        }
        return null;
    }

    private String detectDriverForType(DriverType type) {
        if (DriverType.SQLLITE.equals((Object)type) && new FeatureDetector(this.classLoader).isAndroidAvailable()) {
            return DriverType.SQLDROID.driverClass;
        }
        return type.driverClass;
    }

    public Driver getDriver() {
        return this.driver;
    }

    public String getUrl() {
        return this.url;
    }

    public String getUser() {
        return this.user;
    }

    public String getPassword() {
        return this.password;
    }

    @Override
    public Connection getConnection() throws SQLException {
        return this.getConnectionFromDriver(this.getUser(), this.getPassword());
    }

    @Override
    public Connection getConnection(String username, String password) throws SQLException {
        return this.getConnectionFromDriver(username, password);
    }

    protected Connection getConnectionFromDriver(String username, String password) throws SQLException {
        Connection connection;
        Properties props = new Properties(this.defaultProps);
        if (username != null) {
            props.setProperty("user", username);
        }
        if (password != null) {
            props.setProperty("password", password);
        }
        if ((connection = this.driver.connect(this.url, props)) == null) {
            throw new FlywayException("Unable to connect to " + this.url);
        }
        connection.setAutoCommit(this.autoCommit);
        return connection;
    }

    public boolean isAutoCommit() {
        return this.autoCommit;
    }

    public void setAutoCommit(boolean autoCommit) {
        this.autoCommit = autoCommit;
    }

    @Override
    public int getLoginTimeout() {
        return 0;
    }

    @Override
    public void setLoginTimeout(int timeout) {
        this.unsupportedMethod("setLoginTimeout");
    }

    @Override
    public PrintWriter getLogWriter() {
        this.unsupportedMethod("getLogWriter");
        return null;
    }

    @Override
    public void setLogWriter(PrintWriter pw) {
        this.unsupportedMethod("setLogWriter");
    }

    @Override
    public <T> T unwrap(Class<T> iface) {
        this.unsupportedMethod("unwrap");
        return null;
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) {
        return DataSource.class.equals(iface);
    }

    @Override
    public Logger getParentLogger() {
        this.unsupportedMethod("getParentLogger");
        return null;
    }

    private void unsupportedMethod(String methodName) {
        throw new UnsupportedOperationException(methodName);
    }

    public void shutdownDatabase() {
        if (DriverType.DERBY_EMBEDDED.equals((Object)this.type)) {
            try {
                int i = this.url.indexOf(";");
                String shutdownUrl = (i < 0 ? this.url : this.url.substring(0, i)) + ";shutdown=true";
                this.driver.connect(shutdownUrl, new Properties());
            }
            catch (SQLException e) {
                LOG.debug("Expected error on Derby Embedded Database shutdown: " + e.getMessage());
            }
        }
    }

    public static enum DriverType {
        DB2("jdbc:db2:", "com.ibm.db2.jcc.DB2Driver"),
        DERBY_CLIENT("jdbc:derby://", "org.apache.derby.jdbc.ClientDriver"),
        DERBY_EMBEDDED("jdbc:derby:", "org.apache.derby.jdbc.EmbeddedDriver"),
        FIREBIRD("jdbc:firebird:", "org.firebirdsql.jdbc.FBDriver"),
        FIREBIRD_SQL("jdbc:firebirdsql:", "org.firebirdsql.jdbc.FBDriver"),
        H2("jdbc:h2:", "org.h2.Driver"),
        HSQL("jdbc:hsqldb:", "org.hsqldb.jdbcDriver"),
        INFORMIX("jdbc:informix-sqli:", "com.informix.jdbc.IfxDriver"),
        JTDS("jdbc:jtds:", "net.sourceforge.jtds.jdbc.Driver"),
        MARIADB("jdbc:mariadb:", "org.mariadb.jdbc.Driver"),
        MYSQL("jdbc:mysql:", "com.mysql.cj.jdbc.Driver"),
        MYSQL_GOOGLE("jdbc:google:", "com.mysql.jdbc.GoogleDriver"),
        ORACLE("jdbc:oracle", "oracle.jdbc.OracleDriver"),
        POSTGRESQL("jdbc:postgresql:", "org.postgresql.Driver"),
        REDSHIFT("jdbc:redshift:", "com.amazon.redshift.jdbc42.Driver"),
        SAPHANA("jdbc:sap:", "com.sap.db.jdbc.Driver"),
        SNOWFLAKE("jdbc:snowflake:", "net.snowflake.client.jdbc.SnowflakeDriver"),
        SQLDROID("jdbc:sqldroid:", "org.sqldroid.SQLDroidDriver"),
        SQLLITE("jdbc:sqlite:", "org.sqlite.JDBC"),
        SQLSERVER("jdbc:sqlserver:", "com.microsoft.sqlserver.jdbc.SQLServerDriver"),
        SYBASE("jdbc:sybase:", "com.sybase.jdbc4.jdbc.SybDriver"),
        TEST_CONTAINERS("jdbc:tc:", "org.testcontainers.jdbc.ContainerDatabaseDriver");

        public String prefix;
        public String driverClass;

        private DriverType(String prefix, String driverClass) {
            this.prefix = prefix;
            this.driverClass = driverClass;
        }

        public boolean matches(String url) {
            return url.startsWith(this.prefix);
        }
    }
}

