//    SPDX-License-Identifier: LGPL-3.0-or-later
/*
    This file is part of FlexiBLAS, a BLAS/LAPACK interface wrapper library.
    Copyright (C) 2013-2024 Martin Koehler

    This program is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the Free
    Software Foundation, either version 3 of the License, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program. If not, see <https://www.gnu.org/licenses/>.
 */





#include "flexiblas.h"
#include <errno.h>


static int __flexiblas_load_fortran_hook_function( void * handle , struct flexiblas_hook_fn *ptr, const char *name)
{
    char fname[40];
    void * ptr_hsymbol = NULL;
    int run = 0;

    if ( handle == NULL) return 0;

    /* Load Hook if available */
	DPRINTF(3, "Look up hook: ");
	for (run = 0; run < 3 ; run++) {
		if (run == 0) {
			snprintf(fname, 39, "hook_%s", name);
		} else if ( run == 1 ){
			snprintf(fname, 39, "hook_%s_", name);
		} else if ( run == 2 ){
			snprintf(fname, 39, "hook_%s__", name);
		} else {
			break;
		}
		if ( __flexiblas_verbose > 2) {
			fprintf(stderr, "%s ", fname);
		}

		ptr_hsymbol = dlsym(handle, fname);

        if (ptr_hsymbol!=NULL) {
			break;
		}
	}
	if ( __flexiblas_verbose > 2) {
		fprintf(stderr, "%s\n", (ptr_hsymbol==NULL)?("not found."):("found."));
	}
    if ( ptr_hsymbol ) {
        int p = ptr->nhook;
        ptr->f77_hook_function[p] = ptr_hsymbol;
        ptr->nhook++;
        if ( ptr->nhook >= FLEXIBLAS_MAX_HOOKS) {
            DPRINTF_WARN(0, "Maximum number of installable hooks reached for %s. This may cause problems.\n");
            ptr->nhook = FLEXIBLAS_MAX_HOOKS-1;
        }
    }

    return 0;
}


#define LOAD_HOOK(backend, handle,part,name) do { \
    if ( __flexiblas_load_fortran_hook_function(handle, &(backend ->part),  #name ) > 0 ) { \
        DPRINTF_WARN(3,"Cannot load hook for" #name "\n");\
    } \
} while(0);


HIDDEN int __flexiblas_load_blas_hooks(flexiblas_hook_t *hooks, void *hook_handle) {
	LOAD_HOOK(hooks, hook_handle,caxpy,caxpy);
	LOAD_HOOK(hooks, hook_handle,ccopy,ccopy);
	LOAD_HOOK(hooks, hook_handle,cdotc,cdotc);
	LOAD_HOOK(hooks, hook_handle,cdotu,cdotu);
	LOAD_HOOK(hooks, hook_handle,cgbmv,cgbmv);
	LOAD_HOOK(hooks, hook_handle,cgemm,cgemm);
	LOAD_HOOK(hooks, hook_handle,cgemv,cgemv);
	LOAD_HOOK(hooks, hook_handle,cgerc,cgerc);
	LOAD_HOOK(hooks, hook_handle,cgeru,cgeru);
	LOAD_HOOK(hooks, hook_handle,chbmv,chbmv);
	LOAD_HOOK(hooks, hook_handle,chemm,chemm);
	LOAD_HOOK(hooks, hook_handle,chemv,chemv);
	LOAD_HOOK(hooks, hook_handle,cher,cher);
	LOAD_HOOK(hooks, hook_handle,cher2,cher2);
	LOAD_HOOK(hooks, hook_handle,cher2k,cher2k);
	LOAD_HOOK(hooks, hook_handle,cherk,cherk);
	LOAD_HOOK(hooks, hook_handle,chpmv,chpmv);
	LOAD_HOOK(hooks, hook_handle,chpr,chpr);
	LOAD_HOOK(hooks, hook_handle,chpr2,chpr2);
	LOAD_HOOK(hooks, hook_handle,crotg,crotg);
	LOAD_HOOK(hooks, hook_handle,cscal,cscal);
	LOAD_HOOK(hooks, hook_handle,csrot,csrot);
	LOAD_HOOK(hooks, hook_handle,csscal,csscal);
	LOAD_HOOK(hooks, hook_handle,cswap,cswap);
	LOAD_HOOK(hooks, hook_handle,csymm,csymm);
	LOAD_HOOK(hooks, hook_handle,csyr2k,csyr2k);
	LOAD_HOOK(hooks, hook_handle,csyrk,csyrk);
	LOAD_HOOK(hooks, hook_handle,ctbmv,ctbmv);
	LOAD_HOOK(hooks, hook_handle,ctbsv,ctbsv);
	LOAD_HOOK(hooks, hook_handle,ctpmv,ctpmv);
	LOAD_HOOK(hooks, hook_handle,ctpsv,ctpsv);
	LOAD_HOOK(hooks, hook_handle,ctrmm,ctrmm);
	LOAD_HOOK(hooks, hook_handle,ctrmv,ctrmv);
	LOAD_HOOK(hooks, hook_handle,ctrsm,ctrsm);
	LOAD_HOOK(hooks, hook_handle,ctrsv,ctrsv);
	LOAD_HOOK(hooks, hook_handle,dasum,dasum);
	LOAD_HOOK(hooks, hook_handle,daxpy,daxpy);
	LOAD_HOOK(hooks, hook_handle,dcopy,dcopy);
	LOAD_HOOK(hooks, hook_handle,ddot,ddot);
	LOAD_HOOK(hooks, hook_handle,dgbmv,dgbmv);
	LOAD_HOOK(hooks, hook_handle,dgemm,dgemm);
	LOAD_HOOK(hooks, hook_handle,dgemv,dgemv);
	LOAD_HOOK(hooks, hook_handle,dger,dger);
	LOAD_HOOK(hooks, hook_handle,dnrm2,dnrm2);
	LOAD_HOOK(hooks, hook_handle,drot,drot);
	LOAD_HOOK(hooks, hook_handle,drotg,drotg);
	LOAD_HOOK(hooks, hook_handle,drotm,drotm);
	LOAD_HOOK(hooks, hook_handle,drotmg,drotmg);
	LOAD_HOOK(hooks, hook_handle,dsbmv,dsbmv);
	LOAD_HOOK(hooks, hook_handle,dscal,dscal);
	LOAD_HOOK(hooks, hook_handle,dsdot,dsdot);
	LOAD_HOOK(hooks, hook_handle,dspmv,dspmv);
	LOAD_HOOK(hooks, hook_handle,dspr,dspr);
	LOAD_HOOK(hooks, hook_handle,dspr2,dspr2);
	LOAD_HOOK(hooks, hook_handle,dswap,dswap);
	LOAD_HOOK(hooks, hook_handle,dsymm,dsymm);
	LOAD_HOOK(hooks, hook_handle,dsymv,dsymv);
	LOAD_HOOK(hooks, hook_handle,dsyr,dsyr);
	LOAD_HOOK(hooks, hook_handle,dsyr2,dsyr2);
	LOAD_HOOK(hooks, hook_handle,dsyr2k,dsyr2k);
	LOAD_HOOK(hooks, hook_handle,dsyrk,dsyrk);
	LOAD_HOOK(hooks, hook_handle,dtbmv,dtbmv);
	LOAD_HOOK(hooks, hook_handle,dtbsv,dtbsv);
	LOAD_HOOK(hooks, hook_handle,dtpmv,dtpmv);
	LOAD_HOOK(hooks, hook_handle,dtpsv,dtpsv);
	LOAD_HOOK(hooks, hook_handle,dtrmm,dtrmm);
	LOAD_HOOK(hooks, hook_handle,dtrmv,dtrmv);
	LOAD_HOOK(hooks, hook_handle,dtrsm,dtrsm);
	LOAD_HOOK(hooks, hook_handle,dtrsv,dtrsv);
	LOAD_HOOK(hooks, hook_handle,dzasum,dzasum);
	LOAD_HOOK(hooks, hook_handle,dznrm2,dznrm2);
	LOAD_HOOK(hooks, hook_handle,icamax,icamax);
	LOAD_HOOK(hooks, hook_handle,idamax,idamax);
	LOAD_HOOK(hooks, hook_handle,isamax,isamax);
	LOAD_HOOK(hooks, hook_handle,izamax,izamax);
	LOAD_HOOK(hooks, hook_handle,sasum,sasum);
	LOAD_HOOK(hooks, hook_handle,saxpy,saxpy);
	LOAD_HOOK(hooks, hook_handle,scasum,scasum);
	LOAD_HOOK(hooks, hook_handle,scnrm2,scnrm2);
	LOAD_HOOK(hooks, hook_handle,scopy,scopy);
	LOAD_HOOK(hooks, hook_handle,sdot,sdot);
	LOAD_HOOK(hooks, hook_handle,sdsdot,sdsdot);
	LOAD_HOOK(hooks, hook_handle,sgbmv,sgbmv);
	LOAD_HOOK(hooks, hook_handle,sgemm,sgemm);
	LOAD_HOOK(hooks, hook_handle,sgemv,sgemv);
	LOAD_HOOK(hooks, hook_handle,sger,sger);
	LOAD_HOOK(hooks, hook_handle,snrm2,snrm2);
	LOAD_HOOK(hooks, hook_handle,srot,srot);
	LOAD_HOOK(hooks, hook_handle,srotg,srotg);
	LOAD_HOOK(hooks, hook_handle,srotm,srotm);
	LOAD_HOOK(hooks, hook_handle,srotmg,srotmg);
	LOAD_HOOK(hooks, hook_handle,ssbmv,ssbmv);
	LOAD_HOOK(hooks, hook_handle,sscal,sscal);
	LOAD_HOOK(hooks, hook_handle,sspmv,sspmv);
	LOAD_HOOK(hooks, hook_handle,sspr,sspr);
	LOAD_HOOK(hooks, hook_handle,sspr2,sspr2);
	LOAD_HOOK(hooks, hook_handle,sswap,sswap);
	LOAD_HOOK(hooks, hook_handle,ssymm,ssymm);
	LOAD_HOOK(hooks, hook_handle,ssymv,ssymv);
	LOAD_HOOK(hooks, hook_handle,ssyr,ssyr);
	LOAD_HOOK(hooks, hook_handle,ssyr2,ssyr2);
	LOAD_HOOK(hooks, hook_handle,ssyr2k,ssyr2k);
	LOAD_HOOK(hooks, hook_handle,ssyrk,ssyrk);
	LOAD_HOOK(hooks, hook_handle,stbmv,stbmv);
	LOAD_HOOK(hooks, hook_handle,stbsv,stbsv);
	LOAD_HOOK(hooks, hook_handle,stpmv,stpmv);
	LOAD_HOOK(hooks, hook_handle,stpsv,stpsv);
	LOAD_HOOK(hooks, hook_handle,strmm,strmm);
	LOAD_HOOK(hooks, hook_handle,strmv,strmv);
	LOAD_HOOK(hooks, hook_handle,strsm,strsm);
	LOAD_HOOK(hooks, hook_handle,strsv,strsv);
	LOAD_HOOK(hooks, hook_handle,zaxpy,zaxpy);
	LOAD_HOOK(hooks, hook_handle,zcopy,zcopy);
	LOAD_HOOK(hooks, hook_handle,zdotc,zdotc);
	LOAD_HOOK(hooks, hook_handle,zdotu,zdotu);
	LOAD_HOOK(hooks, hook_handle,zdrot,zdrot);
	LOAD_HOOK(hooks, hook_handle,zdscal,zdscal);
	LOAD_HOOK(hooks, hook_handle,zgbmv,zgbmv);
	LOAD_HOOK(hooks, hook_handle,zgemm,zgemm);
	LOAD_HOOK(hooks, hook_handle,zgemv,zgemv);
	LOAD_HOOK(hooks, hook_handle,zgerc,zgerc);
	LOAD_HOOK(hooks, hook_handle,zgeru,zgeru);
	LOAD_HOOK(hooks, hook_handle,zhbmv,zhbmv);
	LOAD_HOOK(hooks, hook_handle,zhemm,zhemm);
	LOAD_HOOK(hooks, hook_handle,zhemv,zhemv);
	LOAD_HOOK(hooks, hook_handle,zher,zher);
	LOAD_HOOK(hooks, hook_handle,zher2,zher2);
	LOAD_HOOK(hooks, hook_handle,zher2k,zher2k);
	LOAD_HOOK(hooks, hook_handle,zherk,zherk);
	LOAD_HOOK(hooks, hook_handle,zhpmv,zhpmv);
	LOAD_HOOK(hooks, hook_handle,zhpr,zhpr);
	LOAD_HOOK(hooks, hook_handle,zhpr2,zhpr2);
	LOAD_HOOK(hooks, hook_handle,zrotg,zrotg);
	LOAD_HOOK(hooks, hook_handle,zscal,zscal);
	LOAD_HOOK(hooks, hook_handle,zswap,zswap);
	LOAD_HOOK(hooks, hook_handle,zsymm,zsymm);
	LOAD_HOOK(hooks, hook_handle,zsyr2k,zsyr2k);
	LOAD_HOOK(hooks, hook_handle,zsyrk,zsyrk);
	LOAD_HOOK(hooks, hook_handle,ztbmv,ztbmv);
	LOAD_HOOK(hooks, hook_handle,ztbsv,ztbsv);
	LOAD_HOOK(hooks, hook_handle,ztpmv,ztpmv);
	LOAD_HOOK(hooks, hook_handle,ztpsv,ztpsv);
	LOAD_HOOK(hooks, hook_handle,ztrmm,ztrmm);
	LOAD_HOOK(hooks, hook_handle,ztrmv,ztrmv);
	LOAD_HOOK(hooks, hook_handle,ztrsm,ztrsm);
    LOAD_HOOK(hooks, hook_handle,ztrsv,ztrsv);

    /* ExtBLAS exte, posnsion */
    LOAD_HOOK(hooks, hook_handle,caxpby,caxpby);
    LOAD_HOOK(hooks, hook_handle,comatcopy,comatcopy);
    LOAD_HOOK(hooks, hook_handle,cimatcopy,cimatcopy);
    LOAD_HOOK(hooks, hook_handle,cgeadd,cgeadd);

    LOAD_HOOK(hooks, hook_handle,daxpby,daxpby);
    LOAD_HOOK(hooks, hook_handle,domatcopy,domatcopy);
    LOAD_HOOK(hooks, hook_handle,dimatcopy,dimatcopy);
    LOAD_HOOK(hooks, hook_handle,dgeadd,dgeadd);

    LOAD_HOOK(hooks, hook_handle,saxpby,saxpby);
    LOAD_HOOK(hooks, hook_handle,somatcopy,somatcopy);
    LOAD_HOOK(hooks, hook_handle,simatcopy,simatcopy);
    LOAD_HOOK(hooks, hook_handle,sgeadd,sgeadd);

    LOAD_HOOK(hooks, hook_handle,zaxpby,zaxpby);
    LOAD_HOOK(hooks, hook_handle,zomatcopy,zomatcopy);
    LOAD_HOOK(hooks, hook_handle,zimatcopy,zimatcopy);
    LOAD_HOOK(hooks, hook_handle,zgeadd,zgeadd);

#ifdef FLEXIBLAS_LAPACK
	LOAD_HOOK(hooks,hook_handle,second,second);
	LOAD_HOOK(hooks,hook_handle,dsecnd,dsecnd);
	LOAD_HOOK(hooks,hook_handle,slamch,slamch);
	LOAD_HOOK(hooks,hook_handle,dlamch,dlamch);
	LOAD_HOOK(hooks,hook_handle,cbbcsd,cbbcsd);
	LOAD_HOOK(hooks,hook_handle,cbdsqr,cbdsqr);
	LOAD_HOOK(hooks,hook_handle,cgbbrd,cgbbrd);
	LOAD_HOOK(hooks,hook_handle,cgbcon,cgbcon);
	LOAD_HOOK(hooks,hook_handle,cgbequ,cgbequ);
	LOAD_HOOK(hooks,hook_handle,cgbequb,cgbequb);
	LOAD_HOOK(hooks,hook_handle,cgbrfs,cgbrfs);
	LOAD_HOOK(hooks,hook_handle,cgbsv,cgbsv);
	LOAD_HOOK(hooks,hook_handle,cgbsvx,cgbsvx);
	LOAD_HOOK(hooks,hook_handle,cgbtf2,cgbtf2);
	LOAD_HOOK(hooks,hook_handle,cgbtrf,cgbtrf);
	LOAD_HOOK(hooks,hook_handle,cgbtrs,cgbtrs);
	LOAD_HOOK(hooks,hook_handle,cgebak,cgebak);
	LOAD_HOOK(hooks,hook_handle,cgebal,cgebal);
	LOAD_HOOK(hooks,hook_handle,cgebd2,cgebd2);
	LOAD_HOOK(hooks,hook_handle,cgebrd,cgebrd);
	LOAD_HOOK(hooks,hook_handle,cgecon,cgecon);
	LOAD_HOOK(hooks,hook_handle,cgeequ,cgeequ);
	LOAD_HOOK(hooks,hook_handle,cgeequb,cgeequb);
	LOAD_HOOK(hooks,hook_handle,cgees,cgees);
	LOAD_HOOK(hooks,hook_handle,cgeesx,cgeesx);
	LOAD_HOOK(hooks,hook_handle,cgeev,cgeev);
	LOAD_HOOK(hooks,hook_handle,cgeevx,cgeevx);
	LOAD_HOOK(hooks,hook_handle,cgegs,cgegs);
	LOAD_HOOK(hooks,hook_handle,cgegv,cgegv);
	LOAD_HOOK(hooks,hook_handle,cgehd2,cgehd2);
	LOAD_HOOK(hooks,hook_handle,cgehrd,cgehrd);
	LOAD_HOOK(hooks,hook_handle,cgejsv,cgejsv);
	LOAD_HOOK(hooks,hook_handle,cgelq,cgelq);
	LOAD_HOOK(hooks,hook_handle,cgelq2,cgelq2);
	LOAD_HOOK(hooks,hook_handle,cgelqf,cgelqf);
	LOAD_HOOK(hooks,hook_handle,cgelqt,cgelqt);
	LOAD_HOOK(hooks,hook_handle,cgelqt3,cgelqt3);
	LOAD_HOOK(hooks,hook_handle,cgels,cgels);
	LOAD_HOOK(hooks,hook_handle,cgelsd,cgelsd);
	LOAD_HOOK(hooks,hook_handle,cgelss,cgelss);
	LOAD_HOOK(hooks,hook_handle,cgelsx,cgelsx);
	LOAD_HOOK(hooks,hook_handle,cgelsy,cgelsy);
	LOAD_HOOK(hooks,hook_handle,cgemlq,cgemlq);
	LOAD_HOOK(hooks,hook_handle,cgemlqt,cgemlqt);
	LOAD_HOOK(hooks,hook_handle,cgemqr,cgemqr);
	LOAD_HOOK(hooks,hook_handle,cgemqrt,cgemqrt);
	LOAD_HOOK(hooks,hook_handle,cgeql2,cgeql2);
	LOAD_HOOK(hooks,hook_handle,cgeqlf,cgeqlf);
	LOAD_HOOK(hooks,hook_handle,cgeqp3,cgeqp3);
	LOAD_HOOK(hooks,hook_handle,cgeqpf,cgeqpf);
	LOAD_HOOK(hooks,hook_handle,cgeqr,cgeqr);
	LOAD_HOOK(hooks,hook_handle,cgeqr2,cgeqr2);
	LOAD_HOOK(hooks,hook_handle,cgeqr2p,cgeqr2p);
	LOAD_HOOK(hooks,hook_handle,cgeqrf,cgeqrf);
	LOAD_HOOK(hooks,hook_handle,cgeqrfp,cgeqrfp);
	LOAD_HOOK(hooks,hook_handle,cgeqrt,cgeqrt);
	LOAD_HOOK(hooks,hook_handle,cgeqrt2,cgeqrt2);
	LOAD_HOOK(hooks,hook_handle,cgeqrt3,cgeqrt3);
	LOAD_HOOK(hooks,hook_handle,cgerfs,cgerfs);
	LOAD_HOOK(hooks,hook_handle,cgerq2,cgerq2);
	LOAD_HOOK(hooks,hook_handle,cgerqf,cgerqf);
	LOAD_HOOK(hooks,hook_handle,cgesc2,cgesc2);
	LOAD_HOOK(hooks,hook_handle,cgesdd,cgesdd);
	LOAD_HOOK(hooks,hook_handle,cgesv,cgesv);
	LOAD_HOOK(hooks,hook_handle,cgesvd,cgesvd);
	LOAD_HOOK(hooks,hook_handle,cgesvdq,cgesvdq);
	LOAD_HOOK(hooks,hook_handle,cgesvdx,cgesvdx);
	LOAD_HOOK(hooks,hook_handle,cgesvj,cgesvj);
	LOAD_HOOK(hooks,hook_handle,cgesvx,cgesvx);
	LOAD_HOOK(hooks,hook_handle,cgetc2,cgetc2);
	LOAD_HOOK(hooks,hook_handle,cgetf2,cgetf2);
	LOAD_HOOK(hooks,hook_handle,cgetrf,cgetrf);
	LOAD_HOOK(hooks,hook_handle,cgetrf2,cgetrf2);
	LOAD_HOOK(hooks,hook_handle,cgetri,cgetri);
	LOAD_HOOK(hooks,hook_handle,cgetrs,cgetrs);
	LOAD_HOOK(hooks,hook_handle,cgetsls,cgetsls);
	LOAD_HOOK(hooks,hook_handle,cggbak,cggbak);
	LOAD_HOOK(hooks,hook_handle,cggbal,cggbal);
	LOAD_HOOK(hooks,hook_handle,cgges,cgges);
	LOAD_HOOK(hooks,hook_handle,cgges3,cgges3);
	LOAD_HOOK(hooks,hook_handle,cggesx,cggesx);
	LOAD_HOOK(hooks,hook_handle,cggev,cggev);
	LOAD_HOOK(hooks,hook_handle,cggev3,cggev3);
	LOAD_HOOK(hooks,hook_handle,cggevx,cggevx);
	LOAD_HOOK(hooks,hook_handle,cggglm,cggglm);
	LOAD_HOOK(hooks,hook_handle,cgghd3,cgghd3);
	LOAD_HOOK(hooks,hook_handle,cgghrd,cgghrd);
	LOAD_HOOK(hooks,hook_handle,cgglse,cgglse);
	LOAD_HOOK(hooks,hook_handle,cggqrf,cggqrf);
	LOAD_HOOK(hooks,hook_handle,cggrqf,cggrqf);
	LOAD_HOOK(hooks,hook_handle,cggsvd,cggsvd);
	LOAD_HOOK(hooks,hook_handle,cggsvd3,cggsvd3);
	LOAD_HOOK(hooks,hook_handle,cggsvp,cggsvp);
	LOAD_HOOK(hooks,hook_handle,cggsvp3,cggsvp3);
	LOAD_HOOK(hooks,hook_handle,cgsvj0,cgsvj0);
	LOAD_HOOK(hooks,hook_handle,cgsvj1,cgsvj1);
	LOAD_HOOK(hooks,hook_handle,cgtcon,cgtcon);
	LOAD_HOOK(hooks,hook_handle,cgtrfs,cgtrfs);
	LOAD_HOOK(hooks,hook_handle,cgtsv,cgtsv);
	LOAD_HOOK(hooks,hook_handle,cgtsvx,cgtsvx);
	LOAD_HOOK(hooks,hook_handle,cgttrf,cgttrf);
	LOAD_HOOK(hooks,hook_handle,cgttrs,cgttrs);
	LOAD_HOOK(hooks,hook_handle,cgtts2,cgtts2);
	LOAD_HOOK(hooks,hook_handle,chb2st_kernels,chb2st_kernels);
	LOAD_HOOK(hooks,hook_handle,chbev,chbev);
	LOAD_HOOK(hooks,hook_handle,chbev_2stage,chbev_2stage);
	LOAD_HOOK(hooks,hook_handle,chbevd,chbevd);
	LOAD_HOOK(hooks,hook_handle,chbevd_2stage,chbevd_2stage);
	LOAD_HOOK(hooks,hook_handle,chbevx,chbevx);
	LOAD_HOOK(hooks,hook_handle,chbevx_2stage,chbevx_2stage);
	LOAD_HOOK(hooks,hook_handle,chbgst,chbgst);
	LOAD_HOOK(hooks,hook_handle,chbgv,chbgv);
	LOAD_HOOK(hooks,hook_handle,chbgvd,chbgvd);
	LOAD_HOOK(hooks,hook_handle,chbgvx,chbgvx);
	LOAD_HOOK(hooks,hook_handle,chbtrd,chbtrd);
	LOAD_HOOK(hooks,hook_handle,checon,checon);
	LOAD_HOOK(hooks,hook_handle,checon_3,checon_3);
	LOAD_HOOK(hooks,hook_handle,checon_rook,checon_rook);
	LOAD_HOOK(hooks,hook_handle,cheequb,cheequb);
	LOAD_HOOK(hooks,hook_handle,cheev,cheev);
	LOAD_HOOK(hooks,hook_handle,cheev_2stage,cheev_2stage);
	LOAD_HOOK(hooks,hook_handle,cheevd,cheevd);
	LOAD_HOOK(hooks,hook_handle,cheevd_2stage,cheevd_2stage);
	LOAD_HOOK(hooks,hook_handle,cheevr,cheevr);
	LOAD_HOOK(hooks,hook_handle,cheevr_2stage,cheevr_2stage);
	LOAD_HOOK(hooks,hook_handle,cheevx,cheevx);
	LOAD_HOOK(hooks,hook_handle,cheevx_2stage,cheevx_2stage);
	LOAD_HOOK(hooks,hook_handle,chegs2,chegs2);
	LOAD_HOOK(hooks,hook_handle,chegst,chegst);
	LOAD_HOOK(hooks,hook_handle,chegv,chegv);
	LOAD_HOOK(hooks,hook_handle,chegv_2stage,chegv_2stage);
	LOAD_HOOK(hooks,hook_handle,chegvd,chegvd);
	LOAD_HOOK(hooks,hook_handle,chegvx,chegvx);
	LOAD_HOOK(hooks,hook_handle,cherfs,cherfs);
	LOAD_HOOK(hooks,hook_handle,chesv,chesv);
	LOAD_HOOK(hooks,hook_handle,chesv_aa,chesv_aa);
	LOAD_HOOK(hooks,hook_handle,chesv_aa_2stage,chesv_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,chesv_rk,chesv_rk);
	LOAD_HOOK(hooks,hook_handle,chesv_rook,chesv_rook);
	LOAD_HOOK(hooks,hook_handle,chesvx,chesvx);
	LOAD_HOOK(hooks,hook_handle,cheswapr,cheswapr);
	LOAD_HOOK(hooks,hook_handle,chetd2,chetd2);
	LOAD_HOOK(hooks,hook_handle,chetf2,chetf2);
	LOAD_HOOK(hooks,hook_handle,chetf2_rk,chetf2_rk);
	LOAD_HOOK(hooks,hook_handle,chetf2_rook,chetf2_rook);
	LOAD_HOOK(hooks,hook_handle,chetrd,chetrd);
	LOAD_HOOK(hooks,hook_handle,chetrd_2stage,chetrd_2stage);
	LOAD_HOOK(hooks,hook_handle,chetrd_hb2st,chetrd_hb2st);
	LOAD_HOOK(hooks,hook_handle,chetrd_he2hb,chetrd_he2hb);
	LOAD_HOOK(hooks,hook_handle,chetrf,chetrf);
	LOAD_HOOK(hooks,hook_handle,chetrf_aa,chetrf_aa);
	LOAD_HOOK(hooks,hook_handle,chetrf_aa_2stage,chetrf_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,chetrf_rk,chetrf_rk);
	LOAD_HOOK(hooks,hook_handle,chetrf_rook,chetrf_rook);
	LOAD_HOOK(hooks,hook_handle,chetri,chetri);
	LOAD_HOOK(hooks,hook_handle,chetri2,chetri2);
	LOAD_HOOK(hooks,hook_handle,chetri2x,chetri2x);
	LOAD_HOOK(hooks,hook_handle,chetri_3,chetri_3);
	LOAD_HOOK(hooks,hook_handle,chetri_3x,chetri_3x);
	LOAD_HOOK(hooks,hook_handle,chetri_rook,chetri_rook);
	LOAD_HOOK(hooks,hook_handle,chetrs,chetrs);
	LOAD_HOOK(hooks,hook_handle,chetrs2,chetrs2);
	LOAD_HOOK(hooks,hook_handle,chetrs_3,chetrs_3);
	LOAD_HOOK(hooks,hook_handle,chetrs_aa,chetrs_aa);
	LOAD_HOOK(hooks,hook_handle,chetrs_aa_2stage,chetrs_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,chetrs_rook,chetrs_rook);
	LOAD_HOOK(hooks,hook_handle,chfrk,chfrk);
	LOAD_HOOK(hooks,hook_handle,chgeqz,chgeqz);
	LOAD_HOOK(hooks,hook_handle,chla_transtype,chla_transtype);
	LOAD_HOOK(hooks,hook_handle,chpcon,chpcon);
	LOAD_HOOK(hooks,hook_handle,chpev,chpev);
	LOAD_HOOK(hooks,hook_handle,chpevd,chpevd);
	LOAD_HOOK(hooks,hook_handle,chpevx,chpevx);
	LOAD_HOOK(hooks,hook_handle,chpgst,chpgst);
	LOAD_HOOK(hooks,hook_handle,chpgv,chpgv);
	LOAD_HOOK(hooks,hook_handle,chpgvd,chpgvd);
	LOAD_HOOK(hooks,hook_handle,chpgvx,chpgvx);
	LOAD_HOOK(hooks,hook_handle,chprfs,chprfs);
	LOAD_HOOK(hooks,hook_handle,chpsv,chpsv);
	LOAD_HOOK(hooks,hook_handle,chpsvx,chpsvx);
	LOAD_HOOK(hooks,hook_handle,chptrd,chptrd);
	LOAD_HOOK(hooks,hook_handle,chptrf,chptrf);
	LOAD_HOOK(hooks,hook_handle,chptri,chptri);
	LOAD_HOOK(hooks,hook_handle,chptrs,chptrs);
	LOAD_HOOK(hooks,hook_handle,chsein,chsein);
	LOAD_HOOK(hooks,hook_handle,chseqr,chseqr);
	LOAD_HOOK(hooks,hook_handle,clabrd,clabrd);
	LOAD_HOOK(hooks,hook_handle,clacgv,clacgv);
	LOAD_HOOK(hooks,hook_handle,clacn2,clacn2);
	LOAD_HOOK(hooks,hook_handle,clacon,clacon);
	LOAD_HOOK(hooks,hook_handle,clacp2,clacp2);
	LOAD_HOOK(hooks,hook_handle,clacpy,clacpy);
	LOAD_HOOK(hooks,hook_handle,clacrm,clacrm);
	LOAD_HOOK(hooks,hook_handle,clacrt,clacrt);
	LOAD_HOOK(hooks,hook_handle,cladiv,cladiv);
	LOAD_HOOK(hooks,hook_handle,claed0,claed0);
	LOAD_HOOK(hooks,hook_handle,claed7,claed7);
	LOAD_HOOK(hooks,hook_handle,claed8,claed8);
	LOAD_HOOK(hooks,hook_handle,claein,claein);
	LOAD_HOOK(hooks,hook_handle,claesy,claesy);
	LOAD_HOOK(hooks,hook_handle,claev2,claev2);
	LOAD_HOOK(hooks,hook_handle,clag2z,clag2z);
	LOAD_HOOK(hooks,hook_handle,clags2,clags2);
	LOAD_HOOK(hooks,hook_handle,clagtm,clagtm);
	LOAD_HOOK(hooks,hook_handle,clahef,clahef);
	LOAD_HOOK(hooks,hook_handle,clahef_aa,clahef_aa);
	LOAD_HOOK(hooks,hook_handle,clahef_rk,clahef_rk);
	LOAD_HOOK(hooks,hook_handle,clahef_rook,clahef_rook);
	LOAD_HOOK(hooks,hook_handle,clahqr,clahqr);
	LOAD_HOOK(hooks,hook_handle,clahr2,clahr2);
	LOAD_HOOK(hooks,hook_handle,clahrd,clahrd);
	LOAD_HOOK(hooks,hook_handle,claic1,claic1);
	LOAD_HOOK(hooks,hook_handle,clals0,clals0);
	LOAD_HOOK(hooks,hook_handle,clalsa,clalsa);
	LOAD_HOOK(hooks,hook_handle,clalsd,clalsd);
	LOAD_HOOK(hooks,hook_handle,clamswlq,clamswlq);
	LOAD_HOOK(hooks,hook_handle,clamtsqr,clamtsqr);
	LOAD_HOOK(hooks,hook_handle,clangb,clangb);
	LOAD_HOOK(hooks,hook_handle,clange,clange);
	LOAD_HOOK(hooks,hook_handle,clangt,clangt);
	LOAD_HOOK(hooks,hook_handle,clanhb,clanhb);
	LOAD_HOOK(hooks,hook_handle,clanhe,clanhe);
	LOAD_HOOK(hooks,hook_handle,clanhf,clanhf);
	LOAD_HOOK(hooks,hook_handle,clanhp,clanhp);
	LOAD_HOOK(hooks,hook_handle,clanhs,clanhs);
	LOAD_HOOK(hooks,hook_handle,clanht,clanht);
	LOAD_HOOK(hooks,hook_handle,clansb,clansb);
	LOAD_HOOK(hooks,hook_handle,clansp,clansp);
	LOAD_HOOK(hooks,hook_handle,clansy,clansy);
	LOAD_HOOK(hooks,hook_handle,clantb,clantb);
	LOAD_HOOK(hooks,hook_handle,clantp,clantp);
	LOAD_HOOK(hooks,hook_handle,clantr,clantr);
	LOAD_HOOK(hooks,hook_handle,clapll,clapll);
	LOAD_HOOK(hooks,hook_handle,clapmr,clapmr);
	LOAD_HOOK(hooks,hook_handle,clapmt,clapmt);
	LOAD_HOOK(hooks,hook_handle,claqgb,claqgb);
	LOAD_HOOK(hooks,hook_handle,claqge,claqge);
	LOAD_HOOK(hooks,hook_handle,claqhb,claqhb);
	LOAD_HOOK(hooks,hook_handle,claqhe,claqhe);
	LOAD_HOOK(hooks,hook_handle,claqhp,claqhp);
	LOAD_HOOK(hooks,hook_handle,claqp2,claqp2);
	LOAD_HOOK(hooks,hook_handle,claqps,claqps);
	LOAD_HOOK(hooks,hook_handle,claqr0,claqr0);
	LOAD_HOOK(hooks,hook_handle,claqr1,claqr1);
	LOAD_HOOK(hooks,hook_handle,claqr2,claqr2);
	LOAD_HOOK(hooks,hook_handle,claqr3,claqr3);
	LOAD_HOOK(hooks,hook_handle,claqr4,claqr4);
	LOAD_HOOK(hooks,hook_handle,claqr5,claqr5);
	LOAD_HOOK(hooks,hook_handle,claqsb,claqsb);
	LOAD_HOOK(hooks,hook_handle,claqsp,claqsp);
	LOAD_HOOK(hooks,hook_handle,claqsy,claqsy);
	LOAD_HOOK(hooks,hook_handle,clar1v,clar1v);
	LOAD_HOOK(hooks,hook_handle,clar2v,clar2v);
	LOAD_HOOK(hooks,hook_handle,clarcm,clarcm);
	LOAD_HOOK(hooks,hook_handle,clarf,clarf);
	LOAD_HOOK(hooks,hook_handle,clarfb,clarfb);
	LOAD_HOOK(hooks,hook_handle,clarfg,clarfg);
	LOAD_HOOK(hooks,hook_handle,clarfgp,clarfgp);
	LOAD_HOOK(hooks,hook_handle,clarft,clarft);
	LOAD_HOOK(hooks,hook_handle,clarfx,clarfx);
	LOAD_HOOK(hooks,hook_handle,clarfy,clarfy);
	LOAD_HOOK(hooks,hook_handle,clargv,clargv);
	LOAD_HOOK(hooks,hook_handle,clarnv,clarnv);
	LOAD_HOOK(hooks,hook_handle,clarrv,clarrv);
	LOAD_HOOK(hooks,hook_handle,clartg,clartg);
	LOAD_HOOK(hooks,hook_handle,clartv,clartv);
	LOAD_HOOK(hooks,hook_handle,clarz,clarz);
	LOAD_HOOK(hooks,hook_handle,clarzb,clarzb);
	LOAD_HOOK(hooks,hook_handle,clarzt,clarzt);
	LOAD_HOOK(hooks,hook_handle,clascl,clascl);
	LOAD_HOOK(hooks,hook_handle,claset,claset);
	LOAD_HOOK(hooks,hook_handle,clasr,clasr);
	LOAD_HOOK(hooks,hook_handle,classq,classq);
	LOAD_HOOK(hooks,hook_handle,claswlq,claswlq);
	LOAD_HOOK(hooks,hook_handle,claswp,claswp);
	LOAD_HOOK(hooks,hook_handle,clasyf,clasyf);
	LOAD_HOOK(hooks,hook_handle,clasyf_aa,clasyf_aa);
	LOAD_HOOK(hooks,hook_handle,clasyf_rk,clasyf_rk);
	LOAD_HOOK(hooks,hook_handle,clasyf_rook,clasyf_rook);
	LOAD_HOOK(hooks,hook_handle,clatbs,clatbs);
	LOAD_HOOK(hooks,hook_handle,clatdf,clatdf);
	LOAD_HOOK(hooks,hook_handle,clatps,clatps);
	LOAD_HOOK(hooks,hook_handle,clatrd,clatrd);
	LOAD_HOOK(hooks,hook_handle,clatrs,clatrs);
	LOAD_HOOK(hooks,hook_handle,clatrz,clatrz);
	LOAD_HOOK(hooks,hook_handle,clatsqr,clatsqr);
	LOAD_HOOK(hooks,hook_handle,clatzm,clatzm);
	LOAD_HOOK(hooks,hook_handle,claunhr_col_getrfnp,claunhr_col_getrfnp);
	LOAD_HOOK(hooks,hook_handle,claunhr_col_getrfnp2,claunhr_col_getrfnp2);
	LOAD_HOOK(hooks,hook_handle,clauu2,clauu2);
	LOAD_HOOK(hooks,hook_handle,clauum,clauum);
	LOAD_HOOK(hooks,hook_handle,cpbcon,cpbcon);
	LOAD_HOOK(hooks,hook_handle,cpbequ,cpbequ);
	LOAD_HOOK(hooks,hook_handle,cpbrfs,cpbrfs);
	LOAD_HOOK(hooks,hook_handle,cpbstf,cpbstf);
	LOAD_HOOK(hooks,hook_handle,cpbsv,cpbsv);
	LOAD_HOOK(hooks,hook_handle,cpbsvx,cpbsvx);
	LOAD_HOOK(hooks,hook_handle,cpbtf2,cpbtf2);
	LOAD_HOOK(hooks,hook_handle,cpbtrf,cpbtrf);
	LOAD_HOOK(hooks,hook_handle,cpbtrs,cpbtrs);
	LOAD_HOOK(hooks,hook_handle,cpftrf,cpftrf);
	LOAD_HOOK(hooks,hook_handle,cpftri,cpftri);
	LOAD_HOOK(hooks,hook_handle,cpftrs,cpftrs);
	LOAD_HOOK(hooks,hook_handle,cpocon,cpocon);
	LOAD_HOOK(hooks,hook_handle,cpoequ,cpoequ);
	LOAD_HOOK(hooks,hook_handle,cpoequb,cpoequb);
	LOAD_HOOK(hooks,hook_handle,cporfs,cporfs);
	LOAD_HOOK(hooks,hook_handle,cposv,cposv);
	LOAD_HOOK(hooks,hook_handle,cposvx,cposvx);
	LOAD_HOOK(hooks,hook_handle,cpotf2,cpotf2);
	LOAD_HOOK(hooks,hook_handle,cpotrf,cpotrf);
	LOAD_HOOK(hooks,hook_handle,cpotrf2,cpotrf2);
	LOAD_HOOK(hooks,hook_handle,cpotri,cpotri);
	LOAD_HOOK(hooks,hook_handle,cpotrs,cpotrs);
	LOAD_HOOK(hooks,hook_handle,cppcon,cppcon);
	LOAD_HOOK(hooks,hook_handle,cppequ,cppequ);
	LOAD_HOOK(hooks,hook_handle,cpprfs,cpprfs);
	LOAD_HOOK(hooks,hook_handle,cppsv,cppsv);
	LOAD_HOOK(hooks,hook_handle,cppsvx,cppsvx);
	LOAD_HOOK(hooks,hook_handle,cpptrf,cpptrf);
	LOAD_HOOK(hooks,hook_handle,cpptri,cpptri);
	LOAD_HOOK(hooks,hook_handle,cpptrs,cpptrs);
	LOAD_HOOK(hooks,hook_handle,cpstf2,cpstf2);
	LOAD_HOOK(hooks,hook_handle,cpstrf,cpstrf);
	LOAD_HOOK(hooks,hook_handle,cptcon,cptcon);
	LOAD_HOOK(hooks,hook_handle,cpteqr,cpteqr);
	LOAD_HOOK(hooks,hook_handle,cptrfs,cptrfs);
	LOAD_HOOK(hooks,hook_handle,cptsv,cptsv);
	LOAD_HOOK(hooks,hook_handle,cptsvx,cptsvx);
	LOAD_HOOK(hooks,hook_handle,cpttrf,cpttrf);
	LOAD_HOOK(hooks,hook_handle,cpttrs,cpttrs);
	LOAD_HOOK(hooks,hook_handle,cptts2,cptts2);
	LOAD_HOOK(hooks,hook_handle,crot,crot);
	LOAD_HOOK(hooks,hook_handle,cspcon,cspcon);
	LOAD_HOOK(hooks,hook_handle,cspmv,cspmv);
	LOAD_HOOK(hooks,hook_handle,cspr,cspr);
	LOAD_HOOK(hooks,hook_handle,csprfs,csprfs);
	LOAD_HOOK(hooks,hook_handle,cspsv,cspsv);
	LOAD_HOOK(hooks,hook_handle,cspsvx,cspsvx);
	LOAD_HOOK(hooks,hook_handle,csptrf,csptrf);
	LOAD_HOOK(hooks,hook_handle,csptri,csptri);
	LOAD_HOOK(hooks,hook_handle,csptrs,csptrs);
	LOAD_HOOK(hooks,hook_handle,csrscl,csrscl);
	LOAD_HOOK(hooks,hook_handle,cstedc,cstedc);
	LOAD_HOOK(hooks,hook_handle,cstegr,cstegr);
	LOAD_HOOK(hooks,hook_handle,cstein,cstein);
	LOAD_HOOK(hooks,hook_handle,cstemr,cstemr);
	LOAD_HOOK(hooks,hook_handle,csteqr,csteqr);
	LOAD_HOOK(hooks,hook_handle,csycon,csycon);
	LOAD_HOOK(hooks,hook_handle,csycon_3,csycon_3);
	LOAD_HOOK(hooks,hook_handle,csycon_rook,csycon_rook);
	LOAD_HOOK(hooks,hook_handle,csyconv,csyconv);
	LOAD_HOOK(hooks,hook_handle,csyconvf,csyconvf);
	LOAD_HOOK(hooks,hook_handle,csyconvf_rook,csyconvf_rook);
	LOAD_HOOK(hooks,hook_handle,csyequb,csyequb);
	LOAD_HOOK(hooks,hook_handle,csymv,csymv);
	LOAD_HOOK(hooks,hook_handle,csyr,csyr);
	LOAD_HOOK(hooks,hook_handle,csyrfs,csyrfs);
	LOAD_HOOK(hooks,hook_handle,csysv,csysv);
	LOAD_HOOK(hooks,hook_handle,csysv_aa,csysv_aa);
	LOAD_HOOK(hooks,hook_handle,csysv_aa_2stage,csysv_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,csysv_rk,csysv_rk);
	LOAD_HOOK(hooks,hook_handle,csysv_rook,csysv_rook);
	LOAD_HOOK(hooks,hook_handle,csysvx,csysvx);
	LOAD_HOOK(hooks,hook_handle,csyswapr,csyswapr);
	LOAD_HOOK(hooks,hook_handle,csytf2,csytf2);
	LOAD_HOOK(hooks,hook_handle,csytf2_rk,csytf2_rk);
	LOAD_HOOK(hooks,hook_handle,csytf2_rook,csytf2_rook);
	LOAD_HOOK(hooks,hook_handle,csytrf,csytrf);
	LOAD_HOOK(hooks,hook_handle,csytrf_aa,csytrf_aa);
	LOAD_HOOK(hooks,hook_handle,csytrf_aa_2stage,csytrf_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,csytrf_rk,csytrf_rk);
	LOAD_HOOK(hooks,hook_handle,csytrf_rook,csytrf_rook);
	LOAD_HOOK(hooks,hook_handle,csytri,csytri);
	LOAD_HOOK(hooks,hook_handle,csytri2,csytri2);
	LOAD_HOOK(hooks,hook_handle,csytri2x,csytri2x);
	LOAD_HOOK(hooks,hook_handle,csytri_3,csytri_3);
	LOAD_HOOK(hooks,hook_handle,csytri_3x,csytri_3x);
	LOAD_HOOK(hooks,hook_handle,csytri_rook,csytri_rook);
	LOAD_HOOK(hooks,hook_handle,csytrs,csytrs);
	LOAD_HOOK(hooks,hook_handle,csytrs2,csytrs2);
	LOAD_HOOK(hooks,hook_handle,csytrs_3,csytrs_3);
	LOAD_HOOK(hooks,hook_handle,csytrs_aa,csytrs_aa);
	LOAD_HOOK(hooks,hook_handle,csytrs_aa_2stage,csytrs_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,csytrs_rook,csytrs_rook);
	LOAD_HOOK(hooks,hook_handle,ctbcon,ctbcon);
	LOAD_HOOK(hooks,hook_handle,ctbrfs,ctbrfs);
	LOAD_HOOK(hooks,hook_handle,ctbtrs,ctbtrs);
	LOAD_HOOK(hooks,hook_handle,ctfsm,ctfsm);
	LOAD_HOOK(hooks,hook_handle,ctftri,ctftri);
	LOAD_HOOK(hooks,hook_handle,ctfttp,ctfttp);
	LOAD_HOOK(hooks,hook_handle,ctfttr,ctfttr);
	LOAD_HOOK(hooks,hook_handle,ctgevc,ctgevc);
	LOAD_HOOK(hooks,hook_handle,ctgex2,ctgex2);
	LOAD_HOOK(hooks,hook_handle,ctgexc,ctgexc);
	LOAD_HOOK(hooks,hook_handle,ctgsen,ctgsen);
	LOAD_HOOK(hooks,hook_handle,ctgsja,ctgsja);
	LOAD_HOOK(hooks,hook_handle,ctgsna,ctgsna);
	LOAD_HOOK(hooks,hook_handle,ctgsy2,ctgsy2);
	LOAD_HOOK(hooks,hook_handle,ctgsyl,ctgsyl);
	LOAD_HOOK(hooks,hook_handle,ctpcon,ctpcon);
	LOAD_HOOK(hooks,hook_handle,ctplqt,ctplqt);
	LOAD_HOOK(hooks,hook_handle,ctplqt2,ctplqt2);
	LOAD_HOOK(hooks,hook_handle,ctpmlqt,ctpmlqt);
	LOAD_HOOK(hooks,hook_handle,ctpmqrt,ctpmqrt);
	LOAD_HOOK(hooks,hook_handle,ctpqrt,ctpqrt);
	LOAD_HOOK(hooks,hook_handle,ctpqrt2,ctpqrt2);
	LOAD_HOOK(hooks,hook_handle,ctprfb,ctprfb);
	LOAD_HOOK(hooks,hook_handle,ctprfs,ctprfs);
	LOAD_HOOK(hooks,hook_handle,ctptri,ctptri);
	LOAD_HOOK(hooks,hook_handle,ctptrs,ctptrs);
	LOAD_HOOK(hooks,hook_handle,ctpttf,ctpttf);
	LOAD_HOOK(hooks,hook_handle,ctpttr,ctpttr);
	LOAD_HOOK(hooks,hook_handle,ctrcon,ctrcon);
	LOAD_HOOK(hooks,hook_handle,ctrevc,ctrevc);
	LOAD_HOOK(hooks,hook_handle,ctrevc3,ctrevc3);
	LOAD_HOOK(hooks,hook_handle,ctrexc,ctrexc);
	LOAD_HOOK(hooks,hook_handle,ctrrfs,ctrrfs);
	LOAD_HOOK(hooks,hook_handle,ctrsen,ctrsen);
	LOAD_HOOK(hooks,hook_handle,ctrsna,ctrsna);
	LOAD_HOOK(hooks,hook_handle,ctrsyl,ctrsyl);
	LOAD_HOOK(hooks,hook_handle,ctrti2,ctrti2);
	LOAD_HOOK(hooks,hook_handle,ctrtri,ctrtri);
	LOAD_HOOK(hooks,hook_handle,ctrtrs,ctrtrs);
	LOAD_HOOK(hooks,hook_handle,ctrttf,ctrttf);
	LOAD_HOOK(hooks,hook_handle,ctrttp,ctrttp);
	LOAD_HOOK(hooks,hook_handle,ctzrqf,ctzrqf);
	LOAD_HOOK(hooks,hook_handle,ctzrzf,ctzrzf);
	LOAD_HOOK(hooks,hook_handle,cunbdb,cunbdb);
	LOAD_HOOK(hooks,hook_handle,cunbdb1,cunbdb1);
	LOAD_HOOK(hooks,hook_handle,cunbdb2,cunbdb2);
	LOAD_HOOK(hooks,hook_handle,cunbdb3,cunbdb3);
	LOAD_HOOK(hooks,hook_handle,cunbdb4,cunbdb4);
	LOAD_HOOK(hooks,hook_handle,cunbdb5,cunbdb5);
	LOAD_HOOK(hooks,hook_handle,cunbdb6,cunbdb6);
	LOAD_HOOK(hooks,hook_handle,cuncsd,cuncsd);
	LOAD_HOOK(hooks,hook_handle,cuncsd2by1,cuncsd2by1);
	LOAD_HOOK(hooks,hook_handle,cung2l,cung2l);
	LOAD_HOOK(hooks,hook_handle,cung2r,cung2r);
	LOAD_HOOK(hooks,hook_handle,cungbr,cungbr);
	LOAD_HOOK(hooks,hook_handle,cunghr,cunghr);
	LOAD_HOOK(hooks,hook_handle,cungl2,cungl2);
	LOAD_HOOK(hooks,hook_handle,cunglq,cunglq);
	LOAD_HOOK(hooks,hook_handle,cungql,cungql);
	LOAD_HOOK(hooks,hook_handle,cungqr,cungqr);
	LOAD_HOOK(hooks,hook_handle,cungr2,cungr2);
	LOAD_HOOK(hooks,hook_handle,cungrq,cungrq);
	LOAD_HOOK(hooks,hook_handle,cungtr,cungtr);
	LOAD_HOOK(hooks,hook_handle,cungtsqr,cungtsqr);
	LOAD_HOOK(hooks,hook_handle,cunhr_col,cunhr_col);
	LOAD_HOOK(hooks,hook_handle,cunm22,cunm22);
	LOAD_HOOK(hooks,hook_handle,cunm2l,cunm2l);
	LOAD_HOOK(hooks,hook_handle,cunm2r,cunm2r);
	LOAD_HOOK(hooks,hook_handle,cunmbr,cunmbr);
	LOAD_HOOK(hooks,hook_handle,cunmhr,cunmhr);
	LOAD_HOOK(hooks,hook_handle,cunml2,cunml2);
	LOAD_HOOK(hooks,hook_handle,cunmlq,cunmlq);
	LOAD_HOOK(hooks,hook_handle,cunmql,cunmql);
	LOAD_HOOK(hooks,hook_handle,cunmqr,cunmqr);
	LOAD_HOOK(hooks,hook_handle,cunmr2,cunmr2);
	LOAD_HOOK(hooks,hook_handle,cunmr3,cunmr3);
	LOAD_HOOK(hooks,hook_handle,cunmrq,cunmrq);
	LOAD_HOOK(hooks,hook_handle,cunmrz,cunmrz);
	LOAD_HOOK(hooks,hook_handle,cunmtr,cunmtr);
	LOAD_HOOK(hooks,hook_handle,cupgtr,cupgtr);
	LOAD_HOOK(hooks,hook_handle,cupmtr,cupmtr);
	LOAD_HOOK(hooks,hook_handle,dbbcsd,dbbcsd);
	LOAD_HOOK(hooks,hook_handle,dbdsdc,dbdsdc);
	LOAD_HOOK(hooks,hook_handle,dbdsqr,dbdsqr);
	LOAD_HOOK(hooks,hook_handle,dbdsvdx,dbdsvdx);
	LOAD_HOOK(hooks,hook_handle,dcombssq,dcombssq);
	LOAD_HOOK(hooks,hook_handle,ddisna,ddisna);
	LOAD_HOOK(hooks,hook_handle,dgbbrd,dgbbrd);
	LOAD_HOOK(hooks,hook_handle,dgbcon,dgbcon);
	LOAD_HOOK(hooks,hook_handle,dgbequ,dgbequ);
	LOAD_HOOK(hooks,hook_handle,dgbequb,dgbequb);
	LOAD_HOOK(hooks,hook_handle,dgbrfs,dgbrfs);
	LOAD_HOOK(hooks,hook_handle,dgbsv,dgbsv);
	LOAD_HOOK(hooks,hook_handle,dgbsvx,dgbsvx);
	LOAD_HOOK(hooks,hook_handle,dgbtf2,dgbtf2);
	LOAD_HOOK(hooks,hook_handle,dgbtrf,dgbtrf);
	LOAD_HOOK(hooks,hook_handle,dgbtrs,dgbtrs);
	LOAD_HOOK(hooks,hook_handle,dgebak,dgebak);
	LOAD_HOOK(hooks,hook_handle,dgebal,dgebal);
	LOAD_HOOK(hooks,hook_handle,dgebd2,dgebd2);
	LOAD_HOOK(hooks,hook_handle,dgebrd,dgebrd);
	LOAD_HOOK(hooks,hook_handle,dgecon,dgecon);
	LOAD_HOOK(hooks,hook_handle,dgeequ,dgeequ);
	LOAD_HOOK(hooks,hook_handle,dgeequb,dgeequb);
	LOAD_HOOK(hooks,hook_handle,dgees,dgees);
	LOAD_HOOK(hooks,hook_handle,dgeesx,dgeesx);
	LOAD_HOOK(hooks,hook_handle,dgeev,dgeev);
	LOAD_HOOK(hooks,hook_handle,dgeevx,dgeevx);
	LOAD_HOOK(hooks,hook_handle,dgegs,dgegs);
	LOAD_HOOK(hooks,hook_handle,dgegv,dgegv);
	LOAD_HOOK(hooks,hook_handle,dgehd2,dgehd2);
	LOAD_HOOK(hooks,hook_handle,dgehrd,dgehrd);
	LOAD_HOOK(hooks,hook_handle,dgejsv,dgejsv);
	LOAD_HOOK(hooks,hook_handle,dgelq,dgelq);
	LOAD_HOOK(hooks,hook_handle,dgelq2,dgelq2);
	LOAD_HOOK(hooks,hook_handle,dgelqf,dgelqf);
	LOAD_HOOK(hooks,hook_handle,dgelqt,dgelqt);
	LOAD_HOOK(hooks,hook_handle,dgelqt3,dgelqt3);
	LOAD_HOOK(hooks,hook_handle,dgels,dgels);
	LOAD_HOOK(hooks,hook_handle,dgelsd,dgelsd);
	LOAD_HOOK(hooks,hook_handle,dgelss,dgelss);
	LOAD_HOOK(hooks,hook_handle,dgelsx,dgelsx);
	LOAD_HOOK(hooks,hook_handle,dgelsy,dgelsy);
	LOAD_HOOK(hooks,hook_handle,dgemlq,dgemlq);
	LOAD_HOOK(hooks,hook_handle,dgemlqt,dgemlqt);
	LOAD_HOOK(hooks,hook_handle,dgemqr,dgemqr);
	LOAD_HOOK(hooks,hook_handle,dgemqrt,dgemqrt);
	LOAD_HOOK(hooks,hook_handle,dgeql2,dgeql2);
	LOAD_HOOK(hooks,hook_handle,dgeqlf,dgeqlf);
	LOAD_HOOK(hooks,hook_handle,dgeqp3,dgeqp3);
	LOAD_HOOK(hooks,hook_handle,dgeqpf,dgeqpf);
	LOAD_HOOK(hooks,hook_handle,dgeqr,dgeqr);
	LOAD_HOOK(hooks,hook_handle,dgeqr2,dgeqr2);
	LOAD_HOOK(hooks,hook_handle,dgeqr2p,dgeqr2p);
	LOAD_HOOK(hooks,hook_handle,dgeqrf,dgeqrf);
	LOAD_HOOK(hooks,hook_handle,dgeqrfp,dgeqrfp);
	LOAD_HOOK(hooks,hook_handle,dgeqrt,dgeqrt);
	LOAD_HOOK(hooks,hook_handle,dgeqrt2,dgeqrt2);
	LOAD_HOOK(hooks,hook_handle,dgeqrt3,dgeqrt3);
	LOAD_HOOK(hooks,hook_handle,dgerfs,dgerfs);
	LOAD_HOOK(hooks,hook_handle,dgerq2,dgerq2);
	LOAD_HOOK(hooks,hook_handle,dgerqf,dgerqf);
	LOAD_HOOK(hooks,hook_handle,dgesc2,dgesc2);
	LOAD_HOOK(hooks,hook_handle,dgesdd,dgesdd);
	LOAD_HOOK(hooks,hook_handle,dgesv,dgesv);
	LOAD_HOOK(hooks,hook_handle,dgesvd,dgesvd);
	LOAD_HOOK(hooks,hook_handle,dgesvdq,dgesvdq);
	LOAD_HOOK(hooks,hook_handle,dgesvdx,dgesvdx);
	LOAD_HOOK(hooks,hook_handle,dgesvj,dgesvj);
	LOAD_HOOK(hooks,hook_handle,dgesvx,dgesvx);
	LOAD_HOOK(hooks,hook_handle,dgetc2,dgetc2);
	LOAD_HOOK(hooks,hook_handle,dgetf2,dgetf2);
	LOAD_HOOK(hooks,hook_handle,dgetrf,dgetrf);
	LOAD_HOOK(hooks,hook_handle,dgetrf2,dgetrf2);
	LOAD_HOOK(hooks,hook_handle,dgetri,dgetri);
	LOAD_HOOK(hooks,hook_handle,dgetrs,dgetrs);
	LOAD_HOOK(hooks,hook_handle,dgetsls,dgetsls);
	LOAD_HOOK(hooks,hook_handle,dggbak,dggbak);
	LOAD_HOOK(hooks,hook_handle,dggbal,dggbal);
	LOAD_HOOK(hooks,hook_handle,dgges,dgges);
	LOAD_HOOK(hooks,hook_handle,dgges3,dgges3);
	LOAD_HOOK(hooks,hook_handle,dggesx,dggesx);
	LOAD_HOOK(hooks,hook_handle,dggev,dggev);
	LOAD_HOOK(hooks,hook_handle,dggev3,dggev3);
	LOAD_HOOK(hooks,hook_handle,dggevx,dggevx);
	LOAD_HOOK(hooks,hook_handle,dggglm,dggglm);
	LOAD_HOOK(hooks,hook_handle,dgghd3,dgghd3);
	LOAD_HOOK(hooks,hook_handle,dgghrd,dgghrd);
	LOAD_HOOK(hooks,hook_handle,dgglse,dgglse);
	LOAD_HOOK(hooks,hook_handle,dggqrf,dggqrf);
	LOAD_HOOK(hooks,hook_handle,dggrqf,dggrqf);
	LOAD_HOOK(hooks,hook_handle,dggsvd,dggsvd);
	LOAD_HOOK(hooks,hook_handle,dggsvd3,dggsvd3);
	LOAD_HOOK(hooks,hook_handle,dggsvp,dggsvp);
	LOAD_HOOK(hooks,hook_handle,dggsvp3,dggsvp3);
	LOAD_HOOK(hooks,hook_handle,dgsvj0,dgsvj0);
	LOAD_HOOK(hooks,hook_handle,dgsvj1,dgsvj1);
	LOAD_HOOK(hooks,hook_handle,dgtcon,dgtcon);
	LOAD_HOOK(hooks,hook_handle,dgtrfs,dgtrfs);
	LOAD_HOOK(hooks,hook_handle,dgtsv,dgtsv);
	LOAD_HOOK(hooks,hook_handle,dgtsvx,dgtsvx);
	LOAD_HOOK(hooks,hook_handle,dgttrf,dgttrf);
	LOAD_HOOK(hooks,hook_handle,dgttrs,dgttrs);
	LOAD_HOOK(hooks,hook_handle,dgtts2,dgtts2);
	LOAD_HOOK(hooks,hook_handle,dhgeqz,dhgeqz);
	LOAD_HOOK(hooks,hook_handle,dhsein,dhsein);
	LOAD_HOOK(hooks,hook_handle,dhseqr,dhseqr);
	LOAD_HOOK(hooks,hook_handle,disnan,disnan);
	LOAD_HOOK(hooks,hook_handle,dlabad,dlabad);
	LOAD_HOOK(hooks,hook_handle,dlabrd,dlabrd);
	LOAD_HOOK(hooks,hook_handle,dlacn2,dlacn2);
	LOAD_HOOK(hooks,hook_handle,dlacon,dlacon);
	LOAD_HOOK(hooks,hook_handle,dlacpy,dlacpy);
	LOAD_HOOK(hooks,hook_handle,dladiv,dladiv);
	LOAD_HOOK(hooks,hook_handle,dladiv1,dladiv1);
	LOAD_HOOK(hooks,hook_handle,dladiv2,dladiv2);
	LOAD_HOOK(hooks,hook_handle,dlae2,dlae2);
	LOAD_HOOK(hooks,hook_handle,dlaebz,dlaebz);
	LOAD_HOOK(hooks,hook_handle,dlaed0,dlaed0);
	LOAD_HOOK(hooks,hook_handle,dlaed1,dlaed1);
	LOAD_HOOK(hooks,hook_handle,dlaed2,dlaed2);
	LOAD_HOOK(hooks,hook_handle,dlaed3,dlaed3);
	LOAD_HOOK(hooks,hook_handle,dlaed4,dlaed4);
	LOAD_HOOK(hooks,hook_handle,dlaed5,dlaed5);
	LOAD_HOOK(hooks,hook_handle,dlaed6,dlaed6);
	LOAD_HOOK(hooks,hook_handle,dlaed7,dlaed7);
	LOAD_HOOK(hooks,hook_handle,dlaed8,dlaed8);
	LOAD_HOOK(hooks,hook_handle,dlaed9,dlaed9);
	LOAD_HOOK(hooks,hook_handle,dlaeda,dlaeda);
	LOAD_HOOK(hooks,hook_handle,dlaein,dlaein);
	LOAD_HOOK(hooks,hook_handle,dlaev2,dlaev2);
	LOAD_HOOK(hooks,hook_handle,dlaexc,dlaexc);
	LOAD_HOOK(hooks,hook_handle,dlag2,dlag2);
	LOAD_HOOK(hooks,hook_handle,dlag2s,dlag2s);
	LOAD_HOOK(hooks,hook_handle,dlags2,dlags2);
	LOAD_HOOK(hooks,hook_handle,dlagtf,dlagtf);
	LOAD_HOOK(hooks,hook_handle,dlagtm,dlagtm);
	LOAD_HOOK(hooks,hook_handle,dlagts,dlagts);
	LOAD_HOOK(hooks,hook_handle,dlagv2,dlagv2);
	LOAD_HOOK(hooks,hook_handle,dlahqr,dlahqr);
	LOAD_HOOK(hooks,hook_handle,dlahr2,dlahr2);
	LOAD_HOOK(hooks,hook_handle,dlahrd,dlahrd);
	LOAD_HOOK(hooks,hook_handle,dlaic1,dlaic1);
	LOAD_HOOK(hooks,hook_handle,dlaisnan,dlaisnan);
	LOAD_HOOK(hooks,hook_handle,dlaln2,dlaln2);
	LOAD_HOOK(hooks,hook_handle,dlals0,dlals0);
	LOAD_HOOK(hooks,hook_handle,dlalsa,dlalsa);
	LOAD_HOOK(hooks,hook_handle,dlalsd,dlalsd);
	LOAD_HOOK(hooks,hook_handle,dlamrg,dlamrg);
	LOAD_HOOK(hooks,hook_handle,dlamswlq,dlamswlq);
	LOAD_HOOK(hooks,hook_handle,dlamtsqr,dlamtsqr);
	LOAD_HOOK(hooks,hook_handle,dlaneg,dlaneg);
	LOAD_HOOK(hooks,hook_handle,dlangb,dlangb);
	LOAD_HOOK(hooks,hook_handle,dlange,dlange);
	LOAD_HOOK(hooks,hook_handle,dlangt,dlangt);
	LOAD_HOOK(hooks,hook_handle,dlanhs,dlanhs);
	LOAD_HOOK(hooks,hook_handle,dlansb,dlansb);
	LOAD_HOOK(hooks,hook_handle,dlansf,dlansf);
	LOAD_HOOK(hooks,hook_handle,dlansp,dlansp);
	LOAD_HOOK(hooks,hook_handle,dlanst,dlanst);
	LOAD_HOOK(hooks,hook_handle,dlansy,dlansy);
	LOAD_HOOK(hooks,hook_handle,dlantb,dlantb);
	LOAD_HOOK(hooks,hook_handle,dlantp,dlantp);
	LOAD_HOOK(hooks,hook_handle,dlantr,dlantr);
	LOAD_HOOK(hooks,hook_handle,dlanv2,dlanv2);
	LOAD_HOOK(hooks,hook_handle,dlaorhr_col_getrfnp,dlaorhr_col_getrfnp);
	LOAD_HOOK(hooks,hook_handle,dlaorhr_col_getrfnp2,dlaorhr_col_getrfnp2);
	LOAD_HOOK(hooks,hook_handle,dlapll,dlapll);
	LOAD_HOOK(hooks,hook_handle,dlapmr,dlapmr);
	LOAD_HOOK(hooks,hook_handle,dlapmt,dlapmt);
	LOAD_HOOK(hooks,hook_handle,dlapy2,dlapy2);
	LOAD_HOOK(hooks,hook_handle,dlapy3,dlapy3);
	LOAD_HOOK(hooks,hook_handle,dlaqgb,dlaqgb);
	LOAD_HOOK(hooks,hook_handle,dlaqge,dlaqge);
	LOAD_HOOK(hooks,hook_handle,dlaqp2,dlaqp2);
	LOAD_HOOK(hooks,hook_handle,dlaqps,dlaqps);
	LOAD_HOOK(hooks,hook_handle,dlaqr0,dlaqr0);
	LOAD_HOOK(hooks,hook_handle,dlaqr1,dlaqr1);
	LOAD_HOOK(hooks,hook_handle,dlaqr2,dlaqr2);
	LOAD_HOOK(hooks,hook_handle,dlaqr3,dlaqr3);
	LOAD_HOOK(hooks,hook_handle,dlaqr4,dlaqr4);
	LOAD_HOOK(hooks,hook_handle,dlaqr5,dlaqr5);
	LOAD_HOOK(hooks,hook_handle,dlaqsb,dlaqsb);
	LOAD_HOOK(hooks,hook_handle,dlaqsp,dlaqsp);
	LOAD_HOOK(hooks,hook_handle,dlaqsy,dlaqsy);
	LOAD_HOOK(hooks,hook_handle,dlaqtr,dlaqtr);
	LOAD_HOOK(hooks,hook_handle,dlar1v,dlar1v);
	LOAD_HOOK(hooks,hook_handle,dlar2v,dlar2v);
	LOAD_HOOK(hooks,hook_handle,dlarf,dlarf);
	LOAD_HOOK(hooks,hook_handle,dlarfb,dlarfb);
	LOAD_HOOK(hooks,hook_handle,dlarfg,dlarfg);
	LOAD_HOOK(hooks,hook_handle,dlarfgp,dlarfgp);
	LOAD_HOOK(hooks,hook_handle,dlarft,dlarft);
	LOAD_HOOK(hooks,hook_handle,dlarfx,dlarfx);
	LOAD_HOOK(hooks,hook_handle,dlarfy,dlarfy);
	LOAD_HOOK(hooks,hook_handle,dlargv,dlargv);
	LOAD_HOOK(hooks,hook_handle,dlarnv,dlarnv);
	LOAD_HOOK(hooks,hook_handle,dlarra,dlarra);
	LOAD_HOOK(hooks,hook_handle,dlarrb,dlarrb);
	LOAD_HOOK(hooks,hook_handle,dlarrc,dlarrc);
	LOAD_HOOK(hooks,hook_handle,dlarrd,dlarrd);
	LOAD_HOOK(hooks,hook_handle,dlarre,dlarre);
	LOAD_HOOK(hooks,hook_handle,dlarrf,dlarrf);
	LOAD_HOOK(hooks,hook_handle,dlarrj,dlarrj);
	LOAD_HOOK(hooks,hook_handle,dlarrk,dlarrk);
	LOAD_HOOK(hooks,hook_handle,dlarrr,dlarrr);
	LOAD_HOOK(hooks,hook_handle,dlarrv,dlarrv);
	LOAD_HOOK(hooks,hook_handle,dlartg,dlartg);
	LOAD_HOOK(hooks,hook_handle,dlartgp,dlartgp);
	LOAD_HOOK(hooks,hook_handle,dlartgs,dlartgs);
	LOAD_HOOK(hooks,hook_handle,dlartv,dlartv);
	LOAD_HOOK(hooks,hook_handle,dlaruv,dlaruv);
	LOAD_HOOK(hooks,hook_handle,dlarz,dlarz);
	LOAD_HOOK(hooks,hook_handle,dlarzb,dlarzb);
	LOAD_HOOK(hooks,hook_handle,dlarzt,dlarzt);
	LOAD_HOOK(hooks,hook_handle,dlas2,dlas2);
	LOAD_HOOK(hooks,hook_handle,dlascl,dlascl);
	LOAD_HOOK(hooks,hook_handle,dlasd0,dlasd0);
	LOAD_HOOK(hooks,hook_handle,dlasd1,dlasd1);
	LOAD_HOOK(hooks,hook_handle,dlasd2,dlasd2);
	LOAD_HOOK(hooks,hook_handle,dlasd3,dlasd3);
	LOAD_HOOK(hooks,hook_handle,dlasd4,dlasd4);
	LOAD_HOOK(hooks,hook_handle,dlasd5,dlasd5);
	LOAD_HOOK(hooks,hook_handle,dlasd6,dlasd6);
	LOAD_HOOK(hooks,hook_handle,dlasd7,dlasd7);
	LOAD_HOOK(hooks,hook_handle,dlasd8,dlasd8);
	LOAD_HOOK(hooks,hook_handle,dlasda,dlasda);
	LOAD_HOOK(hooks,hook_handle,dlasdq,dlasdq);
	LOAD_HOOK(hooks,hook_handle,dlasdt,dlasdt);
	LOAD_HOOK(hooks,hook_handle,dlaset,dlaset);
	LOAD_HOOK(hooks,hook_handle,dlasq1,dlasq1);
	LOAD_HOOK(hooks,hook_handle,dlasq2,dlasq2);
	LOAD_HOOK(hooks,hook_handle,dlasq3,dlasq3);
	LOAD_HOOK(hooks,hook_handle,dlasq4,dlasq4);
	LOAD_HOOK(hooks,hook_handle,dlasq5,dlasq5);
	LOAD_HOOK(hooks,hook_handle,dlasq6,dlasq6);
	LOAD_HOOK(hooks,hook_handle,dlasr,dlasr);
	LOAD_HOOK(hooks,hook_handle,dlasrt,dlasrt);
	LOAD_HOOK(hooks,hook_handle,dlassq,dlassq);
	LOAD_HOOK(hooks,hook_handle,dlasv2,dlasv2);
	LOAD_HOOK(hooks,hook_handle,dlaswlq,dlaswlq);
	LOAD_HOOK(hooks,hook_handle,dlaswp,dlaswp);
	LOAD_HOOK(hooks,hook_handle,dlasy2,dlasy2);
	LOAD_HOOK(hooks,hook_handle,dlasyf,dlasyf);
	LOAD_HOOK(hooks,hook_handle,dlasyf_aa,dlasyf_aa);
	LOAD_HOOK(hooks,hook_handle,dlasyf_rk,dlasyf_rk);
	LOAD_HOOK(hooks,hook_handle,dlasyf_rook,dlasyf_rook);
	LOAD_HOOK(hooks,hook_handle,dlat2s,dlat2s);
	LOAD_HOOK(hooks,hook_handle,dlatbs,dlatbs);
	LOAD_HOOK(hooks,hook_handle,dlatdf,dlatdf);
	LOAD_HOOK(hooks,hook_handle,dlatps,dlatps);
	LOAD_HOOK(hooks,hook_handle,dlatrd,dlatrd);
	LOAD_HOOK(hooks,hook_handle,dlatrs,dlatrs);
	LOAD_HOOK(hooks,hook_handle,dlatrz,dlatrz);
	LOAD_HOOK(hooks,hook_handle,dlatsqr,dlatsqr);
	LOAD_HOOK(hooks,hook_handle,dlatzm,dlatzm);
	LOAD_HOOK(hooks,hook_handle,dlauu2,dlauu2);
	LOAD_HOOK(hooks,hook_handle,dlauum,dlauum);
	LOAD_HOOK(hooks,hook_handle,dopgtr,dopgtr);
	LOAD_HOOK(hooks,hook_handle,dopmtr,dopmtr);
	LOAD_HOOK(hooks,hook_handle,dorbdb,dorbdb);
	LOAD_HOOK(hooks,hook_handle,dorbdb1,dorbdb1);
	LOAD_HOOK(hooks,hook_handle,dorbdb2,dorbdb2);
	LOAD_HOOK(hooks,hook_handle,dorbdb3,dorbdb3);
	LOAD_HOOK(hooks,hook_handle,dorbdb4,dorbdb4);
	LOAD_HOOK(hooks,hook_handle,dorbdb5,dorbdb5);
	LOAD_HOOK(hooks,hook_handle,dorbdb6,dorbdb6);
	LOAD_HOOK(hooks,hook_handle,dorcsd,dorcsd);
	LOAD_HOOK(hooks,hook_handle,dorcsd2by1,dorcsd2by1);
	LOAD_HOOK(hooks,hook_handle,dorg2l,dorg2l);
	LOAD_HOOK(hooks,hook_handle,dorg2r,dorg2r);
	LOAD_HOOK(hooks,hook_handle,dorgbr,dorgbr);
	LOAD_HOOK(hooks,hook_handle,dorghr,dorghr);
	LOAD_HOOK(hooks,hook_handle,dorgl2,dorgl2);
	LOAD_HOOK(hooks,hook_handle,dorglq,dorglq);
	LOAD_HOOK(hooks,hook_handle,dorgql,dorgql);
	LOAD_HOOK(hooks,hook_handle,dorgqr,dorgqr);
	LOAD_HOOK(hooks,hook_handle,dorgr2,dorgr2);
	LOAD_HOOK(hooks,hook_handle,dorgrq,dorgrq);
	LOAD_HOOK(hooks,hook_handle,dorgtr,dorgtr);
	LOAD_HOOK(hooks,hook_handle,dorgtsqr,dorgtsqr);
	LOAD_HOOK(hooks,hook_handle,dorhr_col,dorhr_col);
	LOAD_HOOK(hooks,hook_handle,dorm22,dorm22);
	LOAD_HOOK(hooks,hook_handle,dorm2l,dorm2l);
	LOAD_HOOK(hooks,hook_handle,dorm2r,dorm2r);
	LOAD_HOOK(hooks,hook_handle,dormbr,dormbr);
	LOAD_HOOK(hooks,hook_handle,dormhr,dormhr);
	LOAD_HOOK(hooks,hook_handle,dorml2,dorml2);
	LOAD_HOOK(hooks,hook_handle,dormlq,dormlq);
	LOAD_HOOK(hooks,hook_handle,dormql,dormql);
	LOAD_HOOK(hooks,hook_handle,dormqr,dormqr);
	LOAD_HOOK(hooks,hook_handle,dormr2,dormr2);
	LOAD_HOOK(hooks,hook_handle,dormr3,dormr3);
	LOAD_HOOK(hooks,hook_handle,dormrq,dormrq);
	LOAD_HOOK(hooks,hook_handle,dormrz,dormrz);
	LOAD_HOOK(hooks,hook_handle,dormtr,dormtr);
	LOAD_HOOK(hooks,hook_handle,dpbcon,dpbcon);
	LOAD_HOOK(hooks,hook_handle,dpbequ,dpbequ);
	LOAD_HOOK(hooks,hook_handle,dpbrfs,dpbrfs);
	LOAD_HOOK(hooks,hook_handle,dpbstf,dpbstf);
	LOAD_HOOK(hooks,hook_handle,dpbsv,dpbsv);
	LOAD_HOOK(hooks,hook_handle,dpbsvx,dpbsvx);
	LOAD_HOOK(hooks,hook_handle,dpbtf2,dpbtf2);
	LOAD_HOOK(hooks,hook_handle,dpbtrf,dpbtrf);
	LOAD_HOOK(hooks,hook_handle,dpbtrs,dpbtrs);
	LOAD_HOOK(hooks,hook_handle,dpftrf,dpftrf);
	LOAD_HOOK(hooks,hook_handle,dpftri,dpftri);
	LOAD_HOOK(hooks,hook_handle,dpftrs,dpftrs);
	LOAD_HOOK(hooks,hook_handle,dpocon,dpocon);
	LOAD_HOOK(hooks,hook_handle,dpoequ,dpoequ);
	LOAD_HOOK(hooks,hook_handle,dpoequb,dpoequb);
	LOAD_HOOK(hooks,hook_handle,dporfs,dporfs);
	LOAD_HOOK(hooks,hook_handle,dposv,dposv);
	LOAD_HOOK(hooks,hook_handle,dposvx,dposvx);
	LOAD_HOOK(hooks,hook_handle,dpotf2,dpotf2);
	LOAD_HOOK(hooks,hook_handle,dpotrf,dpotrf);
	LOAD_HOOK(hooks,hook_handle,dpotrf2,dpotrf2);
	LOAD_HOOK(hooks,hook_handle,dpotri,dpotri);
	LOAD_HOOK(hooks,hook_handle,dpotrs,dpotrs);
	LOAD_HOOK(hooks,hook_handle,dppcon,dppcon);
	LOAD_HOOK(hooks,hook_handle,dppequ,dppequ);
	LOAD_HOOK(hooks,hook_handle,dpprfs,dpprfs);
	LOAD_HOOK(hooks,hook_handle,dppsv,dppsv);
	LOAD_HOOK(hooks,hook_handle,dppsvx,dppsvx);
	LOAD_HOOK(hooks,hook_handle,dpptrf,dpptrf);
	LOAD_HOOK(hooks,hook_handle,dpptri,dpptri);
	LOAD_HOOK(hooks,hook_handle,dpptrs,dpptrs);
	LOAD_HOOK(hooks,hook_handle,dpstf2,dpstf2);
	LOAD_HOOK(hooks,hook_handle,dpstrf,dpstrf);
	LOAD_HOOK(hooks,hook_handle,dptcon,dptcon);
	LOAD_HOOK(hooks,hook_handle,dpteqr,dpteqr);
	LOAD_HOOK(hooks,hook_handle,dptrfs,dptrfs);
	LOAD_HOOK(hooks,hook_handle,dptsv,dptsv);
	LOAD_HOOK(hooks,hook_handle,dptsvx,dptsvx);
	LOAD_HOOK(hooks,hook_handle,dpttrf,dpttrf);
	LOAD_HOOK(hooks,hook_handle,dpttrs,dpttrs);
	LOAD_HOOK(hooks,hook_handle,dptts2,dptts2);
	LOAD_HOOK(hooks,hook_handle,drscl,drscl);
	LOAD_HOOK(hooks,hook_handle,dsb2st_kernels,dsb2st_kernels);
	LOAD_HOOK(hooks,hook_handle,dsbev,dsbev);
	LOAD_HOOK(hooks,hook_handle,dsbev_2stage,dsbev_2stage);
	LOAD_HOOK(hooks,hook_handle,dsbevd,dsbevd);
	LOAD_HOOK(hooks,hook_handle,dsbevd_2stage,dsbevd_2stage);
	LOAD_HOOK(hooks,hook_handle,dsbevx,dsbevx);
	LOAD_HOOK(hooks,hook_handle,dsbevx_2stage,dsbevx_2stage);
	LOAD_HOOK(hooks,hook_handle,dsbgst,dsbgst);
	LOAD_HOOK(hooks,hook_handle,dsbgv,dsbgv);
	LOAD_HOOK(hooks,hook_handle,dsbgvd,dsbgvd);
	LOAD_HOOK(hooks,hook_handle,dsbgvx,dsbgvx);
	LOAD_HOOK(hooks,hook_handle,dsbtrd,dsbtrd);
	LOAD_HOOK(hooks,hook_handle,dsfrk,dsfrk);
	LOAD_HOOK(hooks,hook_handle,dsgesv,dsgesv);
	LOAD_HOOK(hooks,hook_handle,dspcon,dspcon);
	LOAD_HOOK(hooks,hook_handle,dspev,dspev);
	LOAD_HOOK(hooks,hook_handle,dspevd,dspevd);
	LOAD_HOOK(hooks,hook_handle,dspevx,dspevx);
	LOAD_HOOK(hooks,hook_handle,dspgst,dspgst);
	LOAD_HOOK(hooks,hook_handle,dspgv,dspgv);
	LOAD_HOOK(hooks,hook_handle,dspgvd,dspgvd);
	LOAD_HOOK(hooks,hook_handle,dspgvx,dspgvx);
	LOAD_HOOK(hooks,hook_handle,dsposv,dsposv);
	LOAD_HOOK(hooks,hook_handle,dsprfs,dsprfs);
	LOAD_HOOK(hooks,hook_handle,dspsv,dspsv);
	LOAD_HOOK(hooks,hook_handle,dspsvx,dspsvx);
	LOAD_HOOK(hooks,hook_handle,dsptrd,dsptrd);
	LOAD_HOOK(hooks,hook_handle,dsptrf,dsptrf);
	LOAD_HOOK(hooks,hook_handle,dsptri,dsptri);
	LOAD_HOOK(hooks,hook_handle,dsptrs,dsptrs);
	LOAD_HOOK(hooks,hook_handle,dstebz,dstebz);
	LOAD_HOOK(hooks,hook_handle,dstedc,dstedc);
	LOAD_HOOK(hooks,hook_handle,dstegr,dstegr);
	LOAD_HOOK(hooks,hook_handle,dstein,dstein);
	LOAD_HOOK(hooks,hook_handle,dstemr,dstemr);
	LOAD_HOOK(hooks,hook_handle,dsteqr,dsteqr);
	LOAD_HOOK(hooks,hook_handle,dsterf,dsterf);
	LOAD_HOOK(hooks,hook_handle,dstev,dstev);
	LOAD_HOOK(hooks,hook_handle,dstevd,dstevd);
	LOAD_HOOK(hooks,hook_handle,dstevr,dstevr);
	LOAD_HOOK(hooks,hook_handle,dstevx,dstevx);
	LOAD_HOOK(hooks,hook_handle,dsycon,dsycon);
	LOAD_HOOK(hooks,hook_handle,dsycon_3,dsycon_3);
	LOAD_HOOK(hooks,hook_handle,dsycon_rook,dsycon_rook);
	LOAD_HOOK(hooks,hook_handle,dsyconv,dsyconv);
	LOAD_HOOK(hooks,hook_handle,dsyconvf,dsyconvf);
	LOAD_HOOK(hooks,hook_handle,dsyconvf_rook,dsyconvf_rook);
	LOAD_HOOK(hooks,hook_handle,dsyequb,dsyequb);
	LOAD_HOOK(hooks,hook_handle,dsyev,dsyev);
	LOAD_HOOK(hooks,hook_handle,dsyev_2stage,dsyev_2stage);
	LOAD_HOOK(hooks,hook_handle,dsyevd,dsyevd);
	LOAD_HOOK(hooks,hook_handle,dsyevd_2stage,dsyevd_2stage);
	LOAD_HOOK(hooks,hook_handle,dsyevr,dsyevr);
	LOAD_HOOK(hooks,hook_handle,dsyevr_2stage,dsyevr_2stage);
	LOAD_HOOK(hooks,hook_handle,dsyevx,dsyevx);
	LOAD_HOOK(hooks,hook_handle,dsyevx_2stage,dsyevx_2stage);
	LOAD_HOOK(hooks,hook_handle,dsygs2,dsygs2);
	LOAD_HOOK(hooks,hook_handle,dsygst,dsygst);
	LOAD_HOOK(hooks,hook_handle,dsygv,dsygv);
	LOAD_HOOK(hooks,hook_handle,dsygv_2stage,dsygv_2stage);
	LOAD_HOOK(hooks,hook_handle,dsygvd,dsygvd);
	LOAD_HOOK(hooks,hook_handle,dsygvx,dsygvx);
	LOAD_HOOK(hooks,hook_handle,dsyrfs,dsyrfs);
	LOAD_HOOK(hooks,hook_handle,dsysv,dsysv);
	LOAD_HOOK(hooks,hook_handle,dsysv_aa,dsysv_aa);
	LOAD_HOOK(hooks,hook_handle,dsysv_aa_2stage,dsysv_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,dsysv_rk,dsysv_rk);
	LOAD_HOOK(hooks,hook_handle,dsysv_rook,dsysv_rook);
	LOAD_HOOK(hooks,hook_handle,dsysvx,dsysvx);
	LOAD_HOOK(hooks,hook_handle,dsyswapr,dsyswapr);
	LOAD_HOOK(hooks,hook_handle,dsytd2,dsytd2);
	LOAD_HOOK(hooks,hook_handle,dsytf2,dsytf2);
	LOAD_HOOK(hooks,hook_handle,dsytf2_rk,dsytf2_rk);
	LOAD_HOOK(hooks,hook_handle,dsytf2_rook,dsytf2_rook);
	LOAD_HOOK(hooks,hook_handle,dsytrd,dsytrd);
	LOAD_HOOK(hooks,hook_handle,dsytrd_2stage,dsytrd_2stage);
	LOAD_HOOK(hooks,hook_handle,dsytrd_sb2st,dsytrd_sb2st);
	LOAD_HOOK(hooks,hook_handle,dsytrd_sy2sb,dsytrd_sy2sb);
	LOAD_HOOK(hooks,hook_handle,dsytrf,dsytrf);
	LOAD_HOOK(hooks,hook_handle,dsytrf_aa,dsytrf_aa);
	LOAD_HOOK(hooks,hook_handle,dsytrf_aa_2stage,dsytrf_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,dsytrf_rk,dsytrf_rk);
	LOAD_HOOK(hooks,hook_handle,dsytrf_rook,dsytrf_rook);
	LOAD_HOOK(hooks,hook_handle,dsytri,dsytri);
	LOAD_HOOK(hooks,hook_handle,dsytri2,dsytri2);
	LOAD_HOOK(hooks,hook_handle,dsytri2x,dsytri2x);
	LOAD_HOOK(hooks,hook_handle,dsytri_3,dsytri_3);
	LOAD_HOOK(hooks,hook_handle,dsytri_3x,dsytri_3x);
	LOAD_HOOK(hooks,hook_handle,dsytri_rook,dsytri_rook);
	LOAD_HOOK(hooks,hook_handle,dsytrs,dsytrs);
	LOAD_HOOK(hooks,hook_handle,dsytrs2,dsytrs2);
	LOAD_HOOK(hooks,hook_handle,dsytrs_3,dsytrs_3);
	LOAD_HOOK(hooks,hook_handle,dsytrs_aa,dsytrs_aa);
	LOAD_HOOK(hooks,hook_handle,dsytrs_aa_2stage,dsytrs_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,dsytrs_rook,dsytrs_rook);
	LOAD_HOOK(hooks,hook_handle,dtbcon,dtbcon);
	LOAD_HOOK(hooks,hook_handle,dtbrfs,dtbrfs);
	LOAD_HOOK(hooks,hook_handle,dtbtrs,dtbtrs);
	LOAD_HOOK(hooks,hook_handle,dtfsm,dtfsm);
	LOAD_HOOK(hooks,hook_handle,dtftri,dtftri);
	LOAD_HOOK(hooks,hook_handle,dtfttp,dtfttp);
	LOAD_HOOK(hooks,hook_handle,dtfttr,dtfttr);
	LOAD_HOOK(hooks,hook_handle,dtgevc,dtgevc);
	LOAD_HOOK(hooks,hook_handle,dtgex2,dtgex2);
	LOAD_HOOK(hooks,hook_handle,dtgexc,dtgexc);
	LOAD_HOOK(hooks,hook_handle,dtgsen,dtgsen);
	LOAD_HOOK(hooks,hook_handle,dtgsja,dtgsja);
	LOAD_HOOK(hooks,hook_handle,dtgsna,dtgsna);
	LOAD_HOOK(hooks,hook_handle,dtgsy2,dtgsy2);
	LOAD_HOOK(hooks,hook_handle,dtgsyl,dtgsyl);
	LOAD_HOOK(hooks,hook_handle,dtpcon,dtpcon);
	LOAD_HOOK(hooks,hook_handle,dtplqt,dtplqt);
	LOAD_HOOK(hooks,hook_handle,dtplqt2,dtplqt2);
	LOAD_HOOK(hooks,hook_handle,dtpmlqt,dtpmlqt);
	LOAD_HOOK(hooks,hook_handle,dtpmqrt,dtpmqrt);
	LOAD_HOOK(hooks,hook_handle,dtpqrt,dtpqrt);
	LOAD_HOOK(hooks,hook_handle,dtpqrt2,dtpqrt2);
	LOAD_HOOK(hooks,hook_handle,dtprfb,dtprfb);
	LOAD_HOOK(hooks,hook_handle,dtprfs,dtprfs);
	LOAD_HOOK(hooks,hook_handle,dtptri,dtptri);
	LOAD_HOOK(hooks,hook_handle,dtptrs,dtptrs);
	LOAD_HOOK(hooks,hook_handle,dtpttf,dtpttf);
	LOAD_HOOK(hooks,hook_handle,dtpttr,dtpttr);
	LOAD_HOOK(hooks,hook_handle,dtrcon,dtrcon);
	LOAD_HOOK(hooks,hook_handle,dtrevc,dtrevc);
	LOAD_HOOK(hooks,hook_handle,dtrevc3,dtrevc3);
	LOAD_HOOK(hooks,hook_handle,dtrexc,dtrexc);
	LOAD_HOOK(hooks,hook_handle,dtrrfs,dtrrfs);
	LOAD_HOOK(hooks,hook_handle,dtrsen,dtrsen);
	LOAD_HOOK(hooks,hook_handle,dtrsna,dtrsna);
	LOAD_HOOK(hooks,hook_handle,dtrsyl,dtrsyl);
	LOAD_HOOK(hooks,hook_handle,dtrti2,dtrti2);
	LOAD_HOOK(hooks,hook_handle,dtrtri,dtrtri);
	LOAD_HOOK(hooks,hook_handle,dtrtrs,dtrtrs);
	LOAD_HOOK(hooks,hook_handle,dtrttf,dtrttf);
	LOAD_HOOK(hooks,hook_handle,dtrttp,dtrttp);
	LOAD_HOOK(hooks,hook_handle,dtzrqf,dtzrqf);
	LOAD_HOOK(hooks,hook_handle,dtzrzf,dtzrzf);
	LOAD_HOOK(hooks,hook_handle,dzsum1,dzsum1);
	LOAD_HOOK(hooks,hook_handle,icmax1,icmax1);
	LOAD_HOOK(hooks,hook_handle,ieeeck,ieeeck);
	LOAD_HOOK(hooks,hook_handle,ilaclc,ilaclc);
	LOAD_HOOK(hooks,hook_handle,ilaclr,ilaclr);
	LOAD_HOOK(hooks,hook_handle,iladiag,iladiag);
	LOAD_HOOK(hooks,hook_handle,iladlc,iladlc);
	LOAD_HOOK(hooks,hook_handle,iladlr,iladlr);
	LOAD_HOOK(hooks,hook_handle,ilaenv,ilaenv);
	LOAD_HOOK(hooks,hook_handle,ilaenv2stage,ilaenv2stage);
	LOAD_HOOK(hooks,hook_handle,ilaprec,ilaprec);
	LOAD_HOOK(hooks,hook_handle,ilaslc,ilaslc);
	LOAD_HOOK(hooks,hook_handle,ilaslr,ilaslr);
	LOAD_HOOK(hooks,hook_handle,ilatrans,ilatrans);
	LOAD_HOOK(hooks,hook_handle,ilauplo,ilauplo);
	LOAD_HOOK(hooks,hook_handle,ilazlc,ilazlc);
	LOAD_HOOK(hooks,hook_handle,ilazlr,ilazlr);
	LOAD_HOOK(hooks,hook_handle,iparam2stage,iparam2stage);
	LOAD_HOOK(hooks,hook_handle,iparmq,iparmq);
	LOAD_HOOK(hooks,hook_handle,izmax1,izmax1);
	LOAD_HOOK(hooks,hook_handle,sbbcsd,sbbcsd);
	LOAD_HOOK(hooks,hook_handle,sbdsdc,sbdsdc);
	LOAD_HOOK(hooks,hook_handle,sbdsqr,sbdsqr);
	LOAD_HOOK(hooks,hook_handle,sbdsvdx,sbdsvdx);
	LOAD_HOOK(hooks,hook_handle,scombssq,scombssq);
	LOAD_HOOK(hooks,hook_handle,scsum1,scsum1);
	LOAD_HOOK(hooks,hook_handle,sdisna,sdisna);
	LOAD_HOOK(hooks,hook_handle,sgbbrd,sgbbrd);
	LOAD_HOOK(hooks,hook_handle,sgbcon,sgbcon);
	LOAD_HOOK(hooks,hook_handle,sgbequ,sgbequ);
	LOAD_HOOK(hooks,hook_handle,sgbequb,sgbequb);
	LOAD_HOOK(hooks,hook_handle,sgbrfs,sgbrfs);
	LOAD_HOOK(hooks,hook_handle,sgbsv,sgbsv);
	LOAD_HOOK(hooks,hook_handle,sgbsvx,sgbsvx);
	LOAD_HOOK(hooks,hook_handle,sgbtf2,sgbtf2);
	LOAD_HOOK(hooks,hook_handle,sgbtrf,sgbtrf);
	LOAD_HOOK(hooks,hook_handle,sgbtrs,sgbtrs);
	LOAD_HOOK(hooks,hook_handle,sgebak,sgebak);
	LOAD_HOOK(hooks,hook_handle,sgebal,sgebal);
	LOAD_HOOK(hooks,hook_handle,sgebd2,sgebd2);
	LOAD_HOOK(hooks,hook_handle,sgebrd,sgebrd);
	LOAD_HOOK(hooks,hook_handle,sgecon,sgecon);
	LOAD_HOOK(hooks,hook_handle,sgeequ,sgeequ);
	LOAD_HOOK(hooks,hook_handle,sgeequb,sgeequb);
	LOAD_HOOK(hooks,hook_handle,sgees,sgees);
	LOAD_HOOK(hooks,hook_handle,sgeesx,sgeesx);
	LOAD_HOOK(hooks,hook_handle,sgeev,sgeev);
	LOAD_HOOK(hooks,hook_handle,sgeevx,sgeevx);
	LOAD_HOOK(hooks,hook_handle,sgegs,sgegs);
	LOAD_HOOK(hooks,hook_handle,sgegv,sgegv);
	LOAD_HOOK(hooks,hook_handle,sgehd2,sgehd2);
	LOAD_HOOK(hooks,hook_handle,sgehrd,sgehrd);
	LOAD_HOOK(hooks,hook_handle,sgejsv,sgejsv);
	LOAD_HOOK(hooks,hook_handle,sgelq,sgelq);
	LOAD_HOOK(hooks,hook_handle,sgelq2,sgelq2);
	LOAD_HOOK(hooks,hook_handle,sgelqf,sgelqf);
	LOAD_HOOK(hooks,hook_handle,sgelqt,sgelqt);
	LOAD_HOOK(hooks,hook_handle,sgelqt3,sgelqt3);
	LOAD_HOOK(hooks,hook_handle,sgels,sgels);
	LOAD_HOOK(hooks,hook_handle,sgelsd,sgelsd);
	LOAD_HOOK(hooks,hook_handle,sgelss,sgelss);
	LOAD_HOOK(hooks,hook_handle,sgelsx,sgelsx);
	LOAD_HOOK(hooks,hook_handle,sgelsy,sgelsy);
	LOAD_HOOK(hooks,hook_handle,sgemlq,sgemlq);
	LOAD_HOOK(hooks,hook_handle,sgemlqt,sgemlqt);
	LOAD_HOOK(hooks,hook_handle,sgemqr,sgemqr);
	LOAD_HOOK(hooks,hook_handle,sgemqrt,sgemqrt);
	LOAD_HOOK(hooks,hook_handle,sgeql2,sgeql2);
	LOAD_HOOK(hooks,hook_handle,sgeqlf,sgeqlf);
	LOAD_HOOK(hooks,hook_handle,sgeqp3,sgeqp3);
	LOAD_HOOK(hooks,hook_handle,sgeqpf,sgeqpf);
	LOAD_HOOK(hooks,hook_handle,sgeqr,sgeqr);
	LOAD_HOOK(hooks,hook_handle,sgeqr2,sgeqr2);
	LOAD_HOOK(hooks,hook_handle,sgeqr2p,sgeqr2p);
	LOAD_HOOK(hooks,hook_handle,sgeqrf,sgeqrf);
	LOAD_HOOK(hooks,hook_handle,sgeqrfp,sgeqrfp);
	LOAD_HOOK(hooks,hook_handle,sgeqrt,sgeqrt);
	LOAD_HOOK(hooks,hook_handle,sgeqrt2,sgeqrt2);
	LOAD_HOOK(hooks,hook_handle,sgeqrt3,sgeqrt3);
	LOAD_HOOK(hooks,hook_handle,sgerfs,sgerfs);
	LOAD_HOOK(hooks,hook_handle,sgerq2,sgerq2);
	LOAD_HOOK(hooks,hook_handle,sgerqf,sgerqf);
	LOAD_HOOK(hooks,hook_handle,sgesc2,sgesc2);
	LOAD_HOOK(hooks,hook_handle,sgesdd,sgesdd);
	LOAD_HOOK(hooks,hook_handle,sgesv,sgesv);
	LOAD_HOOK(hooks,hook_handle,sgesvd,sgesvd);
	LOAD_HOOK(hooks,hook_handle,sgesvdq,sgesvdq);
	LOAD_HOOK(hooks,hook_handle,sgesvdx,sgesvdx);
	LOAD_HOOK(hooks,hook_handle,sgesvj,sgesvj);
	LOAD_HOOK(hooks,hook_handle,sgesvx,sgesvx);
	LOAD_HOOK(hooks,hook_handle,sgetc2,sgetc2);
	LOAD_HOOK(hooks,hook_handle,sgetf2,sgetf2);
	LOAD_HOOK(hooks,hook_handle,sgetrf,sgetrf);
	LOAD_HOOK(hooks,hook_handle,sgetrf2,sgetrf2);
	LOAD_HOOK(hooks,hook_handle,sgetri,sgetri);
	LOAD_HOOK(hooks,hook_handle,sgetrs,sgetrs);
	LOAD_HOOK(hooks,hook_handle,sgetsls,sgetsls);
	LOAD_HOOK(hooks,hook_handle,sggbak,sggbak);
	LOAD_HOOK(hooks,hook_handle,sggbal,sggbal);
	LOAD_HOOK(hooks,hook_handle,sgges,sgges);
	LOAD_HOOK(hooks,hook_handle,sgges3,sgges3);
	LOAD_HOOK(hooks,hook_handle,sggesx,sggesx);
	LOAD_HOOK(hooks,hook_handle,sggev,sggev);
	LOAD_HOOK(hooks,hook_handle,sggev3,sggev3);
	LOAD_HOOK(hooks,hook_handle,sggevx,sggevx);
	LOAD_HOOK(hooks,hook_handle,sggglm,sggglm);
	LOAD_HOOK(hooks,hook_handle,sgghd3,sgghd3);
	LOAD_HOOK(hooks,hook_handle,sgghrd,sgghrd);
	LOAD_HOOK(hooks,hook_handle,sgglse,sgglse);
	LOAD_HOOK(hooks,hook_handle,sggqrf,sggqrf);
	LOAD_HOOK(hooks,hook_handle,sggrqf,sggrqf);
	LOAD_HOOK(hooks,hook_handle,sggsvd,sggsvd);
	LOAD_HOOK(hooks,hook_handle,sggsvd3,sggsvd3);
	LOAD_HOOK(hooks,hook_handle,sggsvp,sggsvp);
	LOAD_HOOK(hooks,hook_handle,sggsvp3,sggsvp3);
	LOAD_HOOK(hooks,hook_handle,sgsvj0,sgsvj0);
	LOAD_HOOK(hooks,hook_handle,sgsvj1,sgsvj1);
	LOAD_HOOK(hooks,hook_handle,sgtcon,sgtcon);
	LOAD_HOOK(hooks,hook_handle,sgtrfs,sgtrfs);
	LOAD_HOOK(hooks,hook_handle,sgtsv,sgtsv);
	LOAD_HOOK(hooks,hook_handle,sgtsvx,sgtsvx);
	LOAD_HOOK(hooks,hook_handle,sgttrf,sgttrf);
	LOAD_HOOK(hooks,hook_handle,sgttrs,sgttrs);
	LOAD_HOOK(hooks,hook_handle,sgtts2,sgtts2);
	LOAD_HOOK(hooks,hook_handle,shgeqz,shgeqz);
	LOAD_HOOK(hooks,hook_handle,shsein,shsein);
	LOAD_HOOK(hooks,hook_handle,shseqr,shseqr);
	LOAD_HOOK(hooks,hook_handle,sisnan,sisnan);
	LOAD_HOOK(hooks,hook_handle,slabad,slabad);
	LOAD_HOOK(hooks,hook_handle,slabrd,slabrd);
	LOAD_HOOK(hooks,hook_handle,slacn2,slacn2);
	LOAD_HOOK(hooks,hook_handle,slacon,slacon);
	LOAD_HOOK(hooks,hook_handle,slacpy,slacpy);
	LOAD_HOOK(hooks,hook_handle,sladiv,sladiv);
	LOAD_HOOK(hooks,hook_handle,sladiv1,sladiv1);
	LOAD_HOOK(hooks,hook_handle,sladiv2,sladiv2);
	LOAD_HOOK(hooks,hook_handle,slae2,slae2);
	LOAD_HOOK(hooks,hook_handle,slaebz,slaebz);
	LOAD_HOOK(hooks,hook_handle,slaed0,slaed0);
	LOAD_HOOK(hooks,hook_handle,slaed1,slaed1);
	LOAD_HOOK(hooks,hook_handle,slaed2,slaed2);
	LOAD_HOOK(hooks,hook_handle,slaed3,slaed3);
	LOAD_HOOK(hooks,hook_handle,slaed4,slaed4);
	LOAD_HOOK(hooks,hook_handle,slaed5,slaed5);
	LOAD_HOOK(hooks,hook_handle,slaed6,slaed6);
	LOAD_HOOK(hooks,hook_handle,slaed7,slaed7);
	LOAD_HOOK(hooks,hook_handle,slaed8,slaed8);
	LOAD_HOOK(hooks,hook_handle,slaed9,slaed9);
	LOAD_HOOK(hooks,hook_handle,slaeda,slaeda);
	LOAD_HOOK(hooks,hook_handle,slaein,slaein);
	LOAD_HOOK(hooks,hook_handle,slaev2,slaev2);
	LOAD_HOOK(hooks,hook_handle,slaexc,slaexc);
	LOAD_HOOK(hooks,hook_handle,slag2,slag2);
	LOAD_HOOK(hooks,hook_handle,slag2d,slag2d);
	LOAD_HOOK(hooks,hook_handle,slags2,slags2);
	LOAD_HOOK(hooks,hook_handle,slagtf,slagtf);
	LOAD_HOOK(hooks,hook_handle,slagtm,slagtm);
	LOAD_HOOK(hooks,hook_handle,slagts,slagts);
	LOAD_HOOK(hooks,hook_handle,slagv2,slagv2);
	LOAD_HOOK(hooks,hook_handle,slahqr,slahqr);
	LOAD_HOOK(hooks,hook_handle,slahr2,slahr2);
	LOAD_HOOK(hooks,hook_handle,slahrd,slahrd);
	LOAD_HOOK(hooks,hook_handle,slaic1,slaic1);
	LOAD_HOOK(hooks,hook_handle,slaisnan,slaisnan);
	LOAD_HOOK(hooks,hook_handle,slaln2,slaln2);
	LOAD_HOOK(hooks,hook_handle,slals0,slals0);
	LOAD_HOOK(hooks,hook_handle,slalsa,slalsa);
	LOAD_HOOK(hooks,hook_handle,slalsd,slalsd);
	LOAD_HOOK(hooks,hook_handle,slamrg,slamrg);
	LOAD_HOOK(hooks,hook_handle,slamswlq,slamswlq);
	LOAD_HOOK(hooks,hook_handle,slamtsqr,slamtsqr);
	LOAD_HOOK(hooks,hook_handle,slaneg,slaneg);
	LOAD_HOOK(hooks,hook_handle,slangb,slangb);
	LOAD_HOOK(hooks,hook_handle,slange,slange);
	LOAD_HOOK(hooks,hook_handle,slangt,slangt);
	LOAD_HOOK(hooks,hook_handle,slanhs,slanhs);
	LOAD_HOOK(hooks,hook_handle,slansb,slansb);
	LOAD_HOOK(hooks,hook_handle,slansf,slansf);
	LOAD_HOOK(hooks,hook_handle,slansp,slansp);
	LOAD_HOOK(hooks,hook_handle,slanst,slanst);
	LOAD_HOOK(hooks,hook_handle,slansy,slansy);
	LOAD_HOOK(hooks,hook_handle,slantb,slantb);
	LOAD_HOOK(hooks,hook_handle,slantp,slantp);
	LOAD_HOOK(hooks,hook_handle,slantr,slantr);
	LOAD_HOOK(hooks,hook_handle,slanv2,slanv2);
	LOAD_HOOK(hooks,hook_handle,slaorhr_col_getrfnp,slaorhr_col_getrfnp);
	LOAD_HOOK(hooks,hook_handle,slaorhr_col_getrfnp2,slaorhr_col_getrfnp2);
	LOAD_HOOK(hooks,hook_handle,slapll,slapll);
	LOAD_HOOK(hooks,hook_handle,slapmr,slapmr);
	LOAD_HOOK(hooks,hook_handle,slapmt,slapmt);
	LOAD_HOOK(hooks,hook_handle,slapy2,slapy2);
	LOAD_HOOK(hooks,hook_handle,slapy3,slapy3);
	LOAD_HOOK(hooks,hook_handle,slaqgb,slaqgb);
	LOAD_HOOK(hooks,hook_handle,slaqge,slaqge);
	LOAD_HOOK(hooks,hook_handle,slaqp2,slaqp2);
	LOAD_HOOK(hooks,hook_handle,slaqps,slaqps);
	LOAD_HOOK(hooks,hook_handle,slaqr0,slaqr0);
	LOAD_HOOK(hooks,hook_handle,slaqr1,slaqr1);
	LOAD_HOOK(hooks,hook_handle,slaqr2,slaqr2);
	LOAD_HOOK(hooks,hook_handle,slaqr3,slaqr3);
	LOAD_HOOK(hooks,hook_handle,slaqr4,slaqr4);
	LOAD_HOOK(hooks,hook_handle,slaqr5,slaqr5);
	LOAD_HOOK(hooks,hook_handle,slaqsb,slaqsb);
	LOAD_HOOK(hooks,hook_handle,slaqsp,slaqsp);
	LOAD_HOOK(hooks,hook_handle,slaqsy,slaqsy);
	LOAD_HOOK(hooks,hook_handle,slaqtr,slaqtr);
	LOAD_HOOK(hooks,hook_handle,slar1v,slar1v);
	LOAD_HOOK(hooks,hook_handle,slar2v,slar2v);
	LOAD_HOOK(hooks,hook_handle,slarf,slarf);
	LOAD_HOOK(hooks,hook_handle,slarfb,slarfb);
	LOAD_HOOK(hooks,hook_handle,slarfg,slarfg);
	LOAD_HOOK(hooks,hook_handle,slarfgp,slarfgp);
	LOAD_HOOK(hooks,hook_handle,slarft,slarft);
	LOAD_HOOK(hooks,hook_handle,slarfx,slarfx);
	LOAD_HOOK(hooks,hook_handle,slarfy,slarfy);
	LOAD_HOOK(hooks,hook_handle,slargv,slargv);
	LOAD_HOOK(hooks,hook_handle,slarnv,slarnv);
	LOAD_HOOK(hooks,hook_handle,slarra,slarra);
	LOAD_HOOK(hooks,hook_handle,slarrb,slarrb);
	LOAD_HOOK(hooks,hook_handle,slarrc,slarrc);
	LOAD_HOOK(hooks,hook_handle,slarrd,slarrd);
	LOAD_HOOK(hooks,hook_handle,slarre,slarre);
	LOAD_HOOK(hooks,hook_handle,slarrf,slarrf);
	LOAD_HOOK(hooks,hook_handle,slarrj,slarrj);
	LOAD_HOOK(hooks,hook_handle,slarrk,slarrk);
	LOAD_HOOK(hooks,hook_handle,slarrr,slarrr);
	LOAD_HOOK(hooks,hook_handle,slarrv,slarrv);
	LOAD_HOOK(hooks,hook_handle,slartg,slartg);
	LOAD_HOOK(hooks,hook_handle,slartgp,slartgp);
	LOAD_HOOK(hooks,hook_handle,slartgs,slartgs);
	LOAD_HOOK(hooks,hook_handle,slartv,slartv);
	LOAD_HOOK(hooks,hook_handle,slaruv,slaruv);
	LOAD_HOOK(hooks,hook_handle,slarz,slarz);
	LOAD_HOOK(hooks,hook_handle,slarzb,slarzb);
	LOAD_HOOK(hooks,hook_handle,slarzt,slarzt);
	LOAD_HOOK(hooks,hook_handle,slas2,slas2);
	LOAD_HOOK(hooks,hook_handle,slascl,slascl);
	LOAD_HOOK(hooks,hook_handle,slasd0,slasd0);
	LOAD_HOOK(hooks,hook_handle,slasd1,slasd1);
	LOAD_HOOK(hooks,hook_handle,slasd2,slasd2);
	LOAD_HOOK(hooks,hook_handle,slasd3,slasd3);
	LOAD_HOOK(hooks,hook_handle,slasd4,slasd4);
	LOAD_HOOK(hooks,hook_handle,slasd5,slasd5);
	LOAD_HOOK(hooks,hook_handle,slasd6,slasd6);
	LOAD_HOOK(hooks,hook_handle,slasd7,slasd7);
	LOAD_HOOK(hooks,hook_handle,slasd8,slasd8);
	LOAD_HOOK(hooks,hook_handle,slasda,slasda);
	LOAD_HOOK(hooks,hook_handle,slasdq,slasdq);
	LOAD_HOOK(hooks,hook_handle,slasdt,slasdt);
	LOAD_HOOK(hooks,hook_handle,slaset,slaset);
	LOAD_HOOK(hooks,hook_handle,slasq1,slasq1);
	LOAD_HOOK(hooks,hook_handle,slasq2,slasq2);
	LOAD_HOOK(hooks,hook_handle,slasq3,slasq3);
	LOAD_HOOK(hooks,hook_handle,slasq4,slasq4);
	LOAD_HOOK(hooks,hook_handle,slasq5,slasq5);
	LOAD_HOOK(hooks,hook_handle,slasq6,slasq6);
	LOAD_HOOK(hooks,hook_handle,slasr,slasr);
	LOAD_HOOK(hooks,hook_handle,slasrt,slasrt);
	LOAD_HOOK(hooks,hook_handle,slassq,slassq);
	LOAD_HOOK(hooks,hook_handle,slasv2,slasv2);
	LOAD_HOOK(hooks,hook_handle,slaswlq,slaswlq);
	LOAD_HOOK(hooks,hook_handle,slaswp,slaswp);
	LOAD_HOOK(hooks,hook_handle,slasy2,slasy2);
	LOAD_HOOK(hooks,hook_handle,slasyf,slasyf);
	LOAD_HOOK(hooks,hook_handle,slasyf_aa,slasyf_aa);
	LOAD_HOOK(hooks,hook_handle,slasyf_rk,slasyf_rk);
	LOAD_HOOK(hooks,hook_handle,slasyf_rook,slasyf_rook);
	LOAD_HOOK(hooks,hook_handle,slatbs,slatbs);
	LOAD_HOOK(hooks,hook_handle,slatdf,slatdf);
	LOAD_HOOK(hooks,hook_handle,slatps,slatps);
	LOAD_HOOK(hooks,hook_handle,slatrd,slatrd);
	LOAD_HOOK(hooks,hook_handle,slatrs,slatrs);
	LOAD_HOOK(hooks,hook_handle,slatrz,slatrz);
	LOAD_HOOK(hooks,hook_handle,slatsqr,slatsqr);
	LOAD_HOOK(hooks,hook_handle,slatzm,slatzm);
	LOAD_HOOK(hooks,hook_handle,slauu2,slauu2);
	LOAD_HOOK(hooks,hook_handle,slauum,slauum);
	LOAD_HOOK(hooks,hook_handle,sopgtr,sopgtr);
	LOAD_HOOK(hooks,hook_handle,sopmtr,sopmtr);
	LOAD_HOOK(hooks,hook_handle,sorbdb,sorbdb);
	LOAD_HOOK(hooks,hook_handle,sorbdb1,sorbdb1);
	LOAD_HOOK(hooks,hook_handle,sorbdb2,sorbdb2);
	LOAD_HOOK(hooks,hook_handle,sorbdb3,sorbdb3);
	LOAD_HOOK(hooks,hook_handle,sorbdb4,sorbdb4);
	LOAD_HOOK(hooks,hook_handle,sorbdb5,sorbdb5);
	LOAD_HOOK(hooks,hook_handle,sorbdb6,sorbdb6);
	LOAD_HOOK(hooks,hook_handle,sorcsd,sorcsd);
	LOAD_HOOK(hooks,hook_handle,sorcsd2by1,sorcsd2by1);
	LOAD_HOOK(hooks,hook_handle,sorg2l,sorg2l);
	LOAD_HOOK(hooks,hook_handle,sorg2r,sorg2r);
	LOAD_HOOK(hooks,hook_handle,sorgbr,sorgbr);
	LOAD_HOOK(hooks,hook_handle,sorghr,sorghr);
	LOAD_HOOK(hooks,hook_handle,sorgl2,sorgl2);
	LOAD_HOOK(hooks,hook_handle,sorglq,sorglq);
	LOAD_HOOK(hooks,hook_handle,sorgql,sorgql);
	LOAD_HOOK(hooks,hook_handle,sorgqr,sorgqr);
	LOAD_HOOK(hooks,hook_handle,sorgr2,sorgr2);
	LOAD_HOOK(hooks,hook_handle,sorgrq,sorgrq);
	LOAD_HOOK(hooks,hook_handle,sorgtr,sorgtr);
	LOAD_HOOK(hooks,hook_handle,sorgtsqr,sorgtsqr);
	LOAD_HOOK(hooks,hook_handle,sorhr_col,sorhr_col);
	LOAD_HOOK(hooks,hook_handle,sorm22,sorm22);
	LOAD_HOOK(hooks,hook_handle,sorm2l,sorm2l);
	LOAD_HOOK(hooks,hook_handle,sorm2r,sorm2r);
	LOAD_HOOK(hooks,hook_handle,sormbr,sormbr);
	LOAD_HOOK(hooks,hook_handle,sormhr,sormhr);
	LOAD_HOOK(hooks,hook_handle,sorml2,sorml2);
	LOAD_HOOK(hooks,hook_handle,sormlq,sormlq);
	LOAD_HOOK(hooks,hook_handle,sormql,sormql);
	LOAD_HOOK(hooks,hook_handle,sormqr,sormqr);
	LOAD_HOOK(hooks,hook_handle,sormr2,sormr2);
	LOAD_HOOK(hooks,hook_handle,sormr3,sormr3);
	LOAD_HOOK(hooks,hook_handle,sormrq,sormrq);
	LOAD_HOOK(hooks,hook_handle,sormrz,sormrz);
	LOAD_HOOK(hooks,hook_handle,sormtr,sormtr);
	LOAD_HOOK(hooks,hook_handle,spbcon,spbcon);
	LOAD_HOOK(hooks,hook_handle,spbequ,spbequ);
	LOAD_HOOK(hooks,hook_handle,spbrfs,spbrfs);
	LOAD_HOOK(hooks,hook_handle,spbstf,spbstf);
	LOAD_HOOK(hooks,hook_handle,spbsv,spbsv);
	LOAD_HOOK(hooks,hook_handle,spbsvx,spbsvx);
	LOAD_HOOK(hooks,hook_handle,spbtf2,spbtf2);
	LOAD_HOOK(hooks,hook_handle,spbtrf,spbtrf);
	LOAD_HOOK(hooks,hook_handle,spbtrs,spbtrs);
	LOAD_HOOK(hooks,hook_handle,spftrf,spftrf);
	LOAD_HOOK(hooks,hook_handle,spftri,spftri);
	LOAD_HOOK(hooks,hook_handle,spftrs,spftrs);
	LOAD_HOOK(hooks,hook_handle,spocon,spocon);
	LOAD_HOOK(hooks,hook_handle,spoequ,spoequ);
	LOAD_HOOK(hooks,hook_handle,spoequb,spoequb);
	LOAD_HOOK(hooks,hook_handle,sporfs,sporfs);
	LOAD_HOOK(hooks,hook_handle,sposv,sposv);
	LOAD_HOOK(hooks,hook_handle,sposvx,sposvx);
	LOAD_HOOK(hooks,hook_handle,spotf2,spotf2);
	LOAD_HOOK(hooks,hook_handle,spotrf,spotrf);
	LOAD_HOOK(hooks,hook_handle,spotrf2,spotrf2);
	LOAD_HOOK(hooks,hook_handle,spotri,spotri);
	LOAD_HOOK(hooks,hook_handle,spotrs,spotrs);
	LOAD_HOOK(hooks,hook_handle,sppcon,sppcon);
	LOAD_HOOK(hooks,hook_handle,sppequ,sppequ);
	LOAD_HOOK(hooks,hook_handle,spprfs,spprfs);
	LOAD_HOOK(hooks,hook_handle,sppsv,sppsv);
	LOAD_HOOK(hooks,hook_handle,sppsvx,sppsvx);
	LOAD_HOOK(hooks,hook_handle,spptrf,spptrf);
	LOAD_HOOK(hooks,hook_handle,spptri,spptri);
	LOAD_HOOK(hooks,hook_handle,spptrs,spptrs);
	LOAD_HOOK(hooks,hook_handle,spstf2,spstf2);
	LOAD_HOOK(hooks,hook_handle,spstrf,spstrf);
	LOAD_HOOK(hooks,hook_handle,sptcon,sptcon);
	LOAD_HOOK(hooks,hook_handle,spteqr,spteqr);
	LOAD_HOOK(hooks,hook_handle,sptrfs,sptrfs);
	LOAD_HOOK(hooks,hook_handle,sptsv,sptsv);
	LOAD_HOOK(hooks,hook_handle,sptsvx,sptsvx);
	LOAD_HOOK(hooks,hook_handle,spttrf,spttrf);
	LOAD_HOOK(hooks,hook_handle,spttrs,spttrs);
	LOAD_HOOK(hooks,hook_handle,sptts2,sptts2);
	LOAD_HOOK(hooks,hook_handle,srscl,srscl);
	LOAD_HOOK(hooks,hook_handle,ssb2st_kernels,ssb2st_kernels);
	LOAD_HOOK(hooks,hook_handle,ssbev,ssbev);
	LOAD_HOOK(hooks,hook_handle,ssbev_2stage,ssbev_2stage);
	LOAD_HOOK(hooks,hook_handle,ssbevd,ssbevd);
	LOAD_HOOK(hooks,hook_handle,ssbevd_2stage,ssbevd_2stage);
	LOAD_HOOK(hooks,hook_handle,ssbevx,ssbevx);
	LOAD_HOOK(hooks,hook_handle,ssbevx_2stage,ssbevx_2stage);
	LOAD_HOOK(hooks,hook_handle,ssbgst,ssbgst);
	LOAD_HOOK(hooks,hook_handle,ssbgv,ssbgv);
	LOAD_HOOK(hooks,hook_handle,ssbgvd,ssbgvd);
	LOAD_HOOK(hooks,hook_handle,ssbgvx,ssbgvx);
	LOAD_HOOK(hooks,hook_handle,ssbtrd,ssbtrd);
	LOAD_HOOK(hooks,hook_handle,ssfrk,ssfrk);
	LOAD_HOOK(hooks,hook_handle,sspcon,sspcon);
	LOAD_HOOK(hooks,hook_handle,sspev,sspev);
	LOAD_HOOK(hooks,hook_handle,sspevd,sspevd);
	LOAD_HOOK(hooks,hook_handle,sspevx,sspevx);
	LOAD_HOOK(hooks,hook_handle,sspgst,sspgst);
	LOAD_HOOK(hooks,hook_handle,sspgv,sspgv);
	LOAD_HOOK(hooks,hook_handle,sspgvd,sspgvd);
	LOAD_HOOK(hooks,hook_handle,sspgvx,sspgvx);
	LOAD_HOOK(hooks,hook_handle,ssprfs,ssprfs);
	LOAD_HOOK(hooks,hook_handle,sspsv,sspsv);
	LOAD_HOOK(hooks,hook_handle,sspsvx,sspsvx);
	LOAD_HOOK(hooks,hook_handle,ssptrd,ssptrd);
	LOAD_HOOK(hooks,hook_handle,ssptrf,ssptrf);
	LOAD_HOOK(hooks,hook_handle,ssptri,ssptri);
	LOAD_HOOK(hooks,hook_handle,ssptrs,ssptrs);
	LOAD_HOOK(hooks,hook_handle,sstebz,sstebz);
	LOAD_HOOK(hooks,hook_handle,sstedc,sstedc);
	LOAD_HOOK(hooks,hook_handle,sstegr,sstegr);
	LOAD_HOOK(hooks,hook_handle,sstein,sstein);
	LOAD_HOOK(hooks,hook_handle,sstemr,sstemr);
	LOAD_HOOK(hooks,hook_handle,ssteqr,ssteqr);
	LOAD_HOOK(hooks,hook_handle,ssterf,ssterf);
	LOAD_HOOK(hooks,hook_handle,sstev,sstev);
	LOAD_HOOK(hooks,hook_handle,sstevd,sstevd);
	LOAD_HOOK(hooks,hook_handle,sstevr,sstevr);
	LOAD_HOOK(hooks,hook_handle,sstevx,sstevx);
	LOAD_HOOK(hooks,hook_handle,ssycon,ssycon);
	LOAD_HOOK(hooks,hook_handle,ssycon_3,ssycon_3);
	LOAD_HOOK(hooks,hook_handle,ssycon_rook,ssycon_rook);
	LOAD_HOOK(hooks,hook_handle,ssyconv,ssyconv);
	LOAD_HOOK(hooks,hook_handle,ssyconvf,ssyconvf);
	LOAD_HOOK(hooks,hook_handle,ssyconvf_rook,ssyconvf_rook);
	LOAD_HOOK(hooks,hook_handle,ssyequb,ssyequb);
	LOAD_HOOK(hooks,hook_handle,ssyev,ssyev);
	LOAD_HOOK(hooks,hook_handle,ssyev_2stage,ssyev_2stage);
	LOAD_HOOK(hooks,hook_handle,ssyevd,ssyevd);
	LOAD_HOOK(hooks,hook_handle,ssyevd_2stage,ssyevd_2stage);
	LOAD_HOOK(hooks,hook_handle,ssyevr,ssyevr);
	LOAD_HOOK(hooks,hook_handle,ssyevr_2stage,ssyevr_2stage);
	LOAD_HOOK(hooks,hook_handle,ssyevx,ssyevx);
	LOAD_HOOK(hooks,hook_handle,ssyevx_2stage,ssyevx_2stage);
	LOAD_HOOK(hooks,hook_handle,ssygs2,ssygs2);
	LOAD_HOOK(hooks,hook_handle,ssygst,ssygst);
	LOAD_HOOK(hooks,hook_handle,ssygv,ssygv);
	LOAD_HOOK(hooks,hook_handle,ssygv_2stage,ssygv_2stage);
	LOAD_HOOK(hooks,hook_handle,ssygvd,ssygvd);
	LOAD_HOOK(hooks,hook_handle,ssygvx,ssygvx);
	LOAD_HOOK(hooks,hook_handle,ssyrfs,ssyrfs);
	LOAD_HOOK(hooks,hook_handle,ssysv,ssysv);
	LOAD_HOOK(hooks,hook_handle,ssysv_aa,ssysv_aa);
	LOAD_HOOK(hooks,hook_handle,ssysv_aa_2stage,ssysv_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,ssysv_rk,ssysv_rk);
	LOAD_HOOK(hooks,hook_handle,ssysv_rook,ssysv_rook);
	LOAD_HOOK(hooks,hook_handle,ssysvx,ssysvx);
	LOAD_HOOK(hooks,hook_handle,ssyswapr,ssyswapr);
	LOAD_HOOK(hooks,hook_handle,ssytd2,ssytd2);
	LOAD_HOOK(hooks,hook_handle,ssytf2,ssytf2);
	LOAD_HOOK(hooks,hook_handle,ssytf2_rk,ssytf2_rk);
	LOAD_HOOK(hooks,hook_handle,ssytf2_rook,ssytf2_rook);
	LOAD_HOOK(hooks,hook_handle,ssytrd,ssytrd);
	LOAD_HOOK(hooks,hook_handle,ssytrd_2stage,ssytrd_2stage);
	LOAD_HOOK(hooks,hook_handle,ssytrd_sb2st,ssytrd_sb2st);
	LOAD_HOOK(hooks,hook_handle,ssytrd_sy2sb,ssytrd_sy2sb);
	LOAD_HOOK(hooks,hook_handle,ssytrf,ssytrf);
	LOAD_HOOK(hooks,hook_handle,ssytrf_aa,ssytrf_aa);
	LOAD_HOOK(hooks,hook_handle,ssytrf_aa_2stage,ssytrf_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,ssytrf_rk,ssytrf_rk);
	LOAD_HOOK(hooks,hook_handle,ssytrf_rook,ssytrf_rook);
	LOAD_HOOK(hooks,hook_handle,ssytri,ssytri);
	LOAD_HOOK(hooks,hook_handle,ssytri2,ssytri2);
	LOAD_HOOK(hooks,hook_handle,ssytri2x,ssytri2x);
	LOAD_HOOK(hooks,hook_handle,ssytri_3,ssytri_3);
	LOAD_HOOK(hooks,hook_handle,ssytri_3x,ssytri_3x);
	LOAD_HOOK(hooks,hook_handle,ssytri_rook,ssytri_rook);
	LOAD_HOOK(hooks,hook_handle,ssytrs,ssytrs);
	LOAD_HOOK(hooks,hook_handle,ssytrs2,ssytrs2);
	LOAD_HOOK(hooks,hook_handle,ssytrs_3,ssytrs_3);
	LOAD_HOOK(hooks,hook_handle,ssytrs_aa,ssytrs_aa);
	LOAD_HOOK(hooks,hook_handle,ssytrs_aa_2stage,ssytrs_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,ssytrs_rook,ssytrs_rook);
	LOAD_HOOK(hooks,hook_handle,stbcon,stbcon);
	LOAD_HOOK(hooks,hook_handle,stbrfs,stbrfs);
	LOAD_HOOK(hooks,hook_handle,stbtrs,stbtrs);
	LOAD_HOOK(hooks,hook_handle,stfsm,stfsm);
	LOAD_HOOK(hooks,hook_handle,stftri,stftri);
	LOAD_HOOK(hooks,hook_handle,stfttp,stfttp);
	LOAD_HOOK(hooks,hook_handle,stfttr,stfttr);
	LOAD_HOOK(hooks,hook_handle,stgevc,stgevc);
	LOAD_HOOK(hooks,hook_handle,stgex2,stgex2);
	LOAD_HOOK(hooks,hook_handle,stgexc,stgexc);
	LOAD_HOOK(hooks,hook_handle,stgsen,stgsen);
	LOAD_HOOK(hooks,hook_handle,stgsja,stgsja);
	LOAD_HOOK(hooks,hook_handle,stgsna,stgsna);
	LOAD_HOOK(hooks,hook_handle,stgsy2,stgsy2);
	LOAD_HOOK(hooks,hook_handle,stgsyl,stgsyl);
	LOAD_HOOK(hooks,hook_handle,stpcon,stpcon);
	LOAD_HOOK(hooks,hook_handle,stplqt,stplqt);
	LOAD_HOOK(hooks,hook_handle,stplqt2,stplqt2);
	LOAD_HOOK(hooks,hook_handle,stpmlqt,stpmlqt);
	LOAD_HOOK(hooks,hook_handle,stpmqrt,stpmqrt);
	LOAD_HOOK(hooks,hook_handle,stpqrt,stpqrt);
	LOAD_HOOK(hooks,hook_handle,stpqrt2,stpqrt2);
	LOAD_HOOK(hooks,hook_handle,stprfb,stprfb);
	LOAD_HOOK(hooks,hook_handle,stprfs,stprfs);
	LOAD_HOOK(hooks,hook_handle,stptri,stptri);
	LOAD_HOOK(hooks,hook_handle,stptrs,stptrs);
	LOAD_HOOK(hooks,hook_handle,stpttf,stpttf);
	LOAD_HOOK(hooks,hook_handle,stpttr,stpttr);
	LOAD_HOOK(hooks,hook_handle,strcon,strcon);
	LOAD_HOOK(hooks,hook_handle,strevc,strevc);
	LOAD_HOOK(hooks,hook_handle,strevc3,strevc3);
	LOAD_HOOK(hooks,hook_handle,strexc,strexc);
	LOAD_HOOK(hooks,hook_handle,strrfs,strrfs);
	LOAD_HOOK(hooks,hook_handle,strsen,strsen);
	LOAD_HOOK(hooks,hook_handle,strsna,strsna);
	LOAD_HOOK(hooks,hook_handle,strsyl,strsyl);
	LOAD_HOOK(hooks,hook_handle,strti2,strti2);
	LOAD_HOOK(hooks,hook_handle,strtri,strtri);
	LOAD_HOOK(hooks,hook_handle,strtrs,strtrs);
	LOAD_HOOK(hooks,hook_handle,strttf,strttf);
	LOAD_HOOK(hooks,hook_handle,strttp,strttp);
	LOAD_HOOK(hooks,hook_handle,stzrqf,stzrqf);
	LOAD_HOOK(hooks,hook_handle,stzrzf,stzrzf);
	LOAD_HOOK(hooks,hook_handle,xerbla_array,xerbla_array);
	LOAD_HOOK(hooks,hook_handle,zbbcsd,zbbcsd);
	LOAD_HOOK(hooks,hook_handle,zbdsqr,zbdsqr);
	LOAD_HOOK(hooks,hook_handle,zcgesv,zcgesv);
	LOAD_HOOK(hooks,hook_handle,zcposv,zcposv);
	LOAD_HOOK(hooks,hook_handle,zdrscl,zdrscl);
	LOAD_HOOK(hooks,hook_handle,zgbbrd,zgbbrd);
	LOAD_HOOK(hooks,hook_handle,zgbcon,zgbcon);
	LOAD_HOOK(hooks,hook_handle,zgbequ,zgbequ);
	LOAD_HOOK(hooks,hook_handle,zgbequb,zgbequb);
	LOAD_HOOK(hooks,hook_handle,zgbrfs,zgbrfs);
	LOAD_HOOK(hooks,hook_handle,zgbsv,zgbsv);
	LOAD_HOOK(hooks,hook_handle,zgbsvx,zgbsvx);
	LOAD_HOOK(hooks,hook_handle,zgbtf2,zgbtf2);
	LOAD_HOOK(hooks,hook_handle,zgbtrf,zgbtrf);
	LOAD_HOOK(hooks,hook_handle,zgbtrs,zgbtrs);
	LOAD_HOOK(hooks,hook_handle,zgebak,zgebak);
	LOAD_HOOK(hooks,hook_handle,zgebal,zgebal);
	LOAD_HOOK(hooks,hook_handle,zgebd2,zgebd2);
	LOAD_HOOK(hooks,hook_handle,zgebrd,zgebrd);
	LOAD_HOOK(hooks,hook_handle,zgecon,zgecon);
	LOAD_HOOK(hooks,hook_handle,zgeequ,zgeequ);
	LOAD_HOOK(hooks,hook_handle,zgeequb,zgeequb);
	LOAD_HOOK(hooks,hook_handle,zgees,zgees);
	LOAD_HOOK(hooks,hook_handle,zgeesx,zgeesx);
	LOAD_HOOK(hooks,hook_handle,zgeev,zgeev);
	LOAD_HOOK(hooks,hook_handle,zgeevx,zgeevx);
	LOAD_HOOK(hooks,hook_handle,zgegs,zgegs);
	LOAD_HOOK(hooks,hook_handle,zgegv,zgegv);
	LOAD_HOOK(hooks,hook_handle,zgehd2,zgehd2);
	LOAD_HOOK(hooks,hook_handle,zgehrd,zgehrd);
	LOAD_HOOK(hooks,hook_handle,zgejsv,zgejsv);
	LOAD_HOOK(hooks,hook_handle,zgelq,zgelq);
	LOAD_HOOK(hooks,hook_handle,zgelq2,zgelq2);
	LOAD_HOOK(hooks,hook_handle,zgelqf,zgelqf);
	LOAD_HOOK(hooks,hook_handle,zgelqt,zgelqt);
	LOAD_HOOK(hooks,hook_handle,zgelqt3,zgelqt3);
	LOAD_HOOK(hooks,hook_handle,zgels,zgels);
	LOAD_HOOK(hooks,hook_handle,zgelsd,zgelsd);
	LOAD_HOOK(hooks,hook_handle,zgelss,zgelss);
	LOAD_HOOK(hooks,hook_handle,zgelsx,zgelsx);
	LOAD_HOOK(hooks,hook_handle,zgelsy,zgelsy);
	LOAD_HOOK(hooks,hook_handle,zgemlq,zgemlq);
	LOAD_HOOK(hooks,hook_handle,zgemlqt,zgemlqt);
	LOAD_HOOK(hooks,hook_handle,zgemqr,zgemqr);
	LOAD_HOOK(hooks,hook_handle,zgemqrt,zgemqrt);
	LOAD_HOOK(hooks,hook_handle,zgeql2,zgeql2);
	LOAD_HOOK(hooks,hook_handle,zgeqlf,zgeqlf);
	LOAD_HOOK(hooks,hook_handle,zgeqp3,zgeqp3);
	LOAD_HOOK(hooks,hook_handle,zgeqpf,zgeqpf);
	LOAD_HOOK(hooks,hook_handle,zgeqr,zgeqr);
	LOAD_HOOK(hooks,hook_handle,zgeqr2,zgeqr2);
	LOAD_HOOK(hooks,hook_handle,zgeqr2p,zgeqr2p);
	LOAD_HOOK(hooks,hook_handle,zgeqrf,zgeqrf);
	LOAD_HOOK(hooks,hook_handle,zgeqrfp,zgeqrfp);
	LOAD_HOOK(hooks,hook_handle,zgeqrt,zgeqrt);
	LOAD_HOOK(hooks,hook_handle,zgeqrt2,zgeqrt2);
	LOAD_HOOK(hooks,hook_handle,zgeqrt3,zgeqrt3);
	LOAD_HOOK(hooks,hook_handle,zgerfs,zgerfs);
	LOAD_HOOK(hooks,hook_handle,zgerq2,zgerq2);
	LOAD_HOOK(hooks,hook_handle,zgerqf,zgerqf);
	LOAD_HOOK(hooks,hook_handle,zgesc2,zgesc2);
	LOAD_HOOK(hooks,hook_handle,zgesdd,zgesdd);
	LOAD_HOOK(hooks,hook_handle,zgesv,zgesv);
	LOAD_HOOK(hooks,hook_handle,zgesvd,zgesvd);
	LOAD_HOOK(hooks,hook_handle,zgesvdq,zgesvdq);
	LOAD_HOOK(hooks,hook_handle,zgesvdx,zgesvdx);
	LOAD_HOOK(hooks,hook_handle,zgesvj,zgesvj);
	LOAD_HOOK(hooks,hook_handle,zgesvx,zgesvx);
	LOAD_HOOK(hooks,hook_handle,zgetc2,zgetc2);
	LOAD_HOOK(hooks,hook_handle,zgetf2,zgetf2);
	LOAD_HOOK(hooks,hook_handle,zgetrf,zgetrf);
	LOAD_HOOK(hooks,hook_handle,zgetrf2,zgetrf2);
	LOAD_HOOK(hooks,hook_handle,zgetri,zgetri);
	LOAD_HOOK(hooks,hook_handle,zgetrs,zgetrs);
	LOAD_HOOK(hooks,hook_handle,zgetsls,zgetsls);
	LOAD_HOOK(hooks,hook_handle,zggbak,zggbak);
	LOAD_HOOK(hooks,hook_handle,zggbal,zggbal);
	LOAD_HOOK(hooks,hook_handle,zgges,zgges);
	LOAD_HOOK(hooks,hook_handle,zgges3,zgges3);
	LOAD_HOOK(hooks,hook_handle,zggesx,zggesx);
	LOAD_HOOK(hooks,hook_handle,zggev,zggev);
	LOAD_HOOK(hooks,hook_handle,zggev3,zggev3);
	LOAD_HOOK(hooks,hook_handle,zggevx,zggevx);
	LOAD_HOOK(hooks,hook_handle,zggglm,zggglm);
	LOAD_HOOK(hooks,hook_handle,zgghd3,zgghd3);
	LOAD_HOOK(hooks,hook_handle,zgghrd,zgghrd);
	LOAD_HOOK(hooks,hook_handle,zgglse,zgglse);
	LOAD_HOOK(hooks,hook_handle,zggqrf,zggqrf);
	LOAD_HOOK(hooks,hook_handle,zggrqf,zggrqf);
	LOAD_HOOK(hooks,hook_handle,zggsvd,zggsvd);
	LOAD_HOOK(hooks,hook_handle,zggsvd3,zggsvd3);
	LOAD_HOOK(hooks,hook_handle,zggsvp,zggsvp);
	LOAD_HOOK(hooks,hook_handle,zggsvp3,zggsvp3);
	LOAD_HOOK(hooks,hook_handle,zgsvj0,zgsvj0);
	LOAD_HOOK(hooks,hook_handle,zgsvj1,zgsvj1);
	LOAD_HOOK(hooks,hook_handle,zgtcon,zgtcon);
	LOAD_HOOK(hooks,hook_handle,zgtrfs,zgtrfs);
	LOAD_HOOK(hooks,hook_handle,zgtsv,zgtsv);
	LOAD_HOOK(hooks,hook_handle,zgtsvx,zgtsvx);
	LOAD_HOOK(hooks,hook_handle,zgttrf,zgttrf);
	LOAD_HOOK(hooks,hook_handle,zgttrs,zgttrs);
	LOAD_HOOK(hooks,hook_handle,zgtts2,zgtts2);
	LOAD_HOOK(hooks,hook_handle,zhb2st_kernels,zhb2st_kernels);
	LOAD_HOOK(hooks,hook_handle,zhbev,zhbev);
	LOAD_HOOK(hooks,hook_handle,zhbev_2stage,zhbev_2stage);
	LOAD_HOOK(hooks,hook_handle,zhbevd,zhbevd);
	LOAD_HOOK(hooks,hook_handle,zhbevd_2stage,zhbevd_2stage);
	LOAD_HOOK(hooks,hook_handle,zhbevx,zhbevx);
	LOAD_HOOK(hooks,hook_handle,zhbevx_2stage,zhbevx_2stage);
	LOAD_HOOK(hooks,hook_handle,zhbgst,zhbgst);
	LOAD_HOOK(hooks,hook_handle,zhbgv,zhbgv);
	LOAD_HOOK(hooks,hook_handle,zhbgvd,zhbgvd);
	LOAD_HOOK(hooks,hook_handle,zhbgvx,zhbgvx);
	LOAD_HOOK(hooks,hook_handle,zhbtrd,zhbtrd);
	LOAD_HOOK(hooks,hook_handle,zhecon,zhecon);
	LOAD_HOOK(hooks,hook_handle,zhecon_3,zhecon_3);
	LOAD_HOOK(hooks,hook_handle,zhecon_rook,zhecon_rook);
	LOAD_HOOK(hooks,hook_handle,zheequb,zheequb);
	LOAD_HOOK(hooks,hook_handle,zheev,zheev);
	LOAD_HOOK(hooks,hook_handle,zheev_2stage,zheev_2stage);
	LOAD_HOOK(hooks,hook_handle,zheevd,zheevd);
	LOAD_HOOK(hooks,hook_handle,zheevd_2stage,zheevd_2stage);
	LOAD_HOOK(hooks,hook_handle,zheevr,zheevr);
	LOAD_HOOK(hooks,hook_handle,zheevr_2stage,zheevr_2stage);
	LOAD_HOOK(hooks,hook_handle,zheevx,zheevx);
	LOAD_HOOK(hooks,hook_handle,zheevx_2stage,zheevx_2stage);
	LOAD_HOOK(hooks,hook_handle,zhegs2,zhegs2);
	LOAD_HOOK(hooks,hook_handle,zhegst,zhegst);
	LOAD_HOOK(hooks,hook_handle,zhegv,zhegv);
	LOAD_HOOK(hooks,hook_handle,zhegv_2stage,zhegv_2stage);
	LOAD_HOOK(hooks,hook_handle,zhegvd,zhegvd);
	LOAD_HOOK(hooks,hook_handle,zhegvx,zhegvx);
	LOAD_HOOK(hooks,hook_handle,zherfs,zherfs);
	LOAD_HOOK(hooks,hook_handle,zhesv,zhesv);
	LOAD_HOOK(hooks,hook_handle,zhesv_aa,zhesv_aa);
	LOAD_HOOK(hooks,hook_handle,zhesv_aa_2stage,zhesv_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,zhesv_rk,zhesv_rk);
	LOAD_HOOK(hooks,hook_handle,zhesv_rook,zhesv_rook);
	LOAD_HOOK(hooks,hook_handle,zhesvx,zhesvx);
	LOAD_HOOK(hooks,hook_handle,zheswapr,zheswapr);
	LOAD_HOOK(hooks,hook_handle,zhetd2,zhetd2);
	LOAD_HOOK(hooks,hook_handle,zhetf2,zhetf2);
	LOAD_HOOK(hooks,hook_handle,zhetf2_rk,zhetf2_rk);
	LOAD_HOOK(hooks,hook_handle,zhetf2_rook,zhetf2_rook);
	LOAD_HOOK(hooks,hook_handle,zhetrd,zhetrd);
	LOAD_HOOK(hooks,hook_handle,zhetrd_2stage,zhetrd_2stage);
	LOAD_HOOK(hooks,hook_handle,zhetrd_hb2st,zhetrd_hb2st);
	LOAD_HOOK(hooks,hook_handle,zhetrd_he2hb,zhetrd_he2hb);
	LOAD_HOOK(hooks,hook_handle,zhetrf,zhetrf);
	LOAD_HOOK(hooks,hook_handle,zhetrf_aa,zhetrf_aa);
	LOAD_HOOK(hooks,hook_handle,zhetrf_aa_2stage,zhetrf_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,zhetrf_rk,zhetrf_rk);
	LOAD_HOOK(hooks,hook_handle,zhetrf_rook,zhetrf_rook);
	LOAD_HOOK(hooks,hook_handle,zhetri,zhetri);
	LOAD_HOOK(hooks,hook_handle,zhetri2,zhetri2);
	LOAD_HOOK(hooks,hook_handle,zhetri2x,zhetri2x);
	LOAD_HOOK(hooks,hook_handle,zhetri_3,zhetri_3);
	LOAD_HOOK(hooks,hook_handle,zhetri_3x,zhetri_3x);
	LOAD_HOOK(hooks,hook_handle,zhetri_rook,zhetri_rook);
	LOAD_HOOK(hooks,hook_handle,zhetrs,zhetrs);
	LOAD_HOOK(hooks,hook_handle,zhetrs2,zhetrs2);
	LOAD_HOOK(hooks,hook_handle,zhetrs_3,zhetrs_3);
	LOAD_HOOK(hooks,hook_handle,zhetrs_aa,zhetrs_aa);
	LOAD_HOOK(hooks,hook_handle,zhetrs_aa_2stage,zhetrs_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,zhetrs_rook,zhetrs_rook);
	LOAD_HOOK(hooks,hook_handle,zhfrk,zhfrk);
	LOAD_HOOK(hooks,hook_handle,zhgeqz,zhgeqz);
	LOAD_HOOK(hooks,hook_handle,zhpcon,zhpcon);
	LOAD_HOOK(hooks,hook_handle,zhpev,zhpev);
	LOAD_HOOK(hooks,hook_handle,zhpevd,zhpevd);
	LOAD_HOOK(hooks,hook_handle,zhpevx,zhpevx);
	LOAD_HOOK(hooks,hook_handle,zhpgst,zhpgst);
	LOAD_HOOK(hooks,hook_handle,zhpgv,zhpgv);
	LOAD_HOOK(hooks,hook_handle,zhpgvd,zhpgvd);
	LOAD_HOOK(hooks,hook_handle,zhpgvx,zhpgvx);
	LOAD_HOOK(hooks,hook_handle,zhprfs,zhprfs);
	LOAD_HOOK(hooks,hook_handle,zhpsv,zhpsv);
	LOAD_HOOK(hooks,hook_handle,zhpsvx,zhpsvx);
	LOAD_HOOK(hooks,hook_handle,zhptrd,zhptrd);
	LOAD_HOOK(hooks,hook_handle,zhptrf,zhptrf);
	LOAD_HOOK(hooks,hook_handle,zhptri,zhptri);
	LOAD_HOOK(hooks,hook_handle,zhptrs,zhptrs);
	LOAD_HOOK(hooks,hook_handle,zhsein,zhsein);
	LOAD_HOOK(hooks,hook_handle,zhseqr,zhseqr);
	LOAD_HOOK(hooks,hook_handle,zlabrd,zlabrd);
	LOAD_HOOK(hooks,hook_handle,zlacgv,zlacgv);
	LOAD_HOOK(hooks,hook_handle,zlacn2,zlacn2);
	LOAD_HOOK(hooks,hook_handle,zlacon,zlacon);
	LOAD_HOOK(hooks,hook_handle,zlacp2,zlacp2);
	LOAD_HOOK(hooks,hook_handle,zlacpy,zlacpy);
	LOAD_HOOK(hooks,hook_handle,zlacrm,zlacrm);
	LOAD_HOOK(hooks,hook_handle,zlacrt,zlacrt);
	LOAD_HOOK(hooks,hook_handle,zladiv,zladiv);
	LOAD_HOOK(hooks,hook_handle,zlaed0,zlaed0);
	LOAD_HOOK(hooks,hook_handle,zlaed7,zlaed7);
	LOAD_HOOK(hooks,hook_handle,zlaed8,zlaed8);
	LOAD_HOOK(hooks,hook_handle,zlaein,zlaein);
	LOAD_HOOK(hooks,hook_handle,zlaesy,zlaesy);
	LOAD_HOOK(hooks,hook_handle,zlaev2,zlaev2);
	LOAD_HOOK(hooks,hook_handle,zlag2c,zlag2c);
	LOAD_HOOK(hooks,hook_handle,zlags2,zlags2);
	LOAD_HOOK(hooks,hook_handle,zlagtm,zlagtm);
	LOAD_HOOK(hooks,hook_handle,zlahef,zlahef);
	LOAD_HOOK(hooks,hook_handle,zlahef_aa,zlahef_aa);
	LOAD_HOOK(hooks,hook_handle,zlahef_rk,zlahef_rk);
	LOAD_HOOK(hooks,hook_handle,zlahef_rook,zlahef_rook);
	LOAD_HOOK(hooks,hook_handle,zlahqr,zlahqr);
	LOAD_HOOK(hooks,hook_handle,zlahr2,zlahr2);
	LOAD_HOOK(hooks,hook_handle,zlahrd,zlahrd);
	LOAD_HOOK(hooks,hook_handle,zlaic1,zlaic1);
	LOAD_HOOK(hooks,hook_handle,zlals0,zlals0);
	LOAD_HOOK(hooks,hook_handle,zlalsa,zlalsa);
	LOAD_HOOK(hooks,hook_handle,zlalsd,zlalsd);
	LOAD_HOOK(hooks,hook_handle,zlamswlq,zlamswlq);
	LOAD_HOOK(hooks,hook_handle,zlamtsqr,zlamtsqr);
	LOAD_HOOK(hooks,hook_handle,zlangb,zlangb);
	LOAD_HOOK(hooks,hook_handle,zlange,zlange);
	LOAD_HOOK(hooks,hook_handle,zlangt,zlangt);
	LOAD_HOOK(hooks,hook_handle,zlanhb,zlanhb);
	LOAD_HOOK(hooks,hook_handle,zlanhe,zlanhe);
	LOAD_HOOK(hooks,hook_handle,zlanhf,zlanhf);
	LOAD_HOOK(hooks,hook_handle,zlanhp,zlanhp);
	LOAD_HOOK(hooks,hook_handle,zlanhs,zlanhs);
	LOAD_HOOK(hooks,hook_handle,zlanht,zlanht);
	LOAD_HOOK(hooks,hook_handle,zlansb,zlansb);
	LOAD_HOOK(hooks,hook_handle,zlansp,zlansp);
	LOAD_HOOK(hooks,hook_handle,zlansy,zlansy);
	LOAD_HOOK(hooks,hook_handle,zlantb,zlantb);
	LOAD_HOOK(hooks,hook_handle,zlantp,zlantp);
	LOAD_HOOK(hooks,hook_handle,zlantr,zlantr);
	LOAD_HOOK(hooks,hook_handle,zlapll,zlapll);
	LOAD_HOOK(hooks,hook_handle,zlapmr,zlapmr);
	LOAD_HOOK(hooks,hook_handle,zlapmt,zlapmt);
	LOAD_HOOK(hooks,hook_handle,zlaqgb,zlaqgb);
	LOAD_HOOK(hooks,hook_handle,zlaqge,zlaqge);
	LOAD_HOOK(hooks,hook_handle,zlaqhb,zlaqhb);
	LOAD_HOOK(hooks,hook_handle,zlaqhe,zlaqhe);
	LOAD_HOOK(hooks,hook_handle,zlaqhp,zlaqhp);
	LOAD_HOOK(hooks,hook_handle,zlaqp2,zlaqp2);
	LOAD_HOOK(hooks,hook_handle,zlaqps,zlaqps);
	LOAD_HOOK(hooks,hook_handle,zlaqr0,zlaqr0);
	LOAD_HOOK(hooks,hook_handle,zlaqr1,zlaqr1);
	LOAD_HOOK(hooks,hook_handle,zlaqr2,zlaqr2);
	LOAD_HOOK(hooks,hook_handle,zlaqr3,zlaqr3);
	LOAD_HOOK(hooks,hook_handle,zlaqr4,zlaqr4);
	LOAD_HOOK(hooks,hook_handle,zlaqr5,zlaqr5);
	LOAD_HOOK(hooks,hook_handle,zlaqsb,zlaqsb);
	LOAD_HOOK(hooks,hook_handle,zlaqsp,zlaqsp);
	LOAD_HOOK(hooks,hook_handle,zlaqsy,zlaqsy);
	LOAD_HOOK(hooks,hook_handle,zlar1v,zlar1v);
	LOAD_HOOK(hooks,hook_handle,zlar2v,zlar2v);
	LOAD_HOOK(hooks,hook_handle,zlarcm,zlarcm);
	LOAD_HOOK(hooks,hook_handle,zlarf,zlarf);
	LOAD_HOOK(hooks,hook_handle,zlarfb,zlarfb);
	LOAD_HOOK(hooks,hook_handle,zlarfg,zlarfg);
	LOAD_HOOK(hooks,hook_handle,zlarfgp,zlarfgp);
	LOAD_HOOK(hooks,hook_handle,zlarft,zlarft);
	LOAD_HOOK(hooks,hook_handle,zlarfx,zlarfx);
	LOAD_HOOK(hooks,hook_handle,zlarfy,zlarfy);
	LOAD_HOOK(hooks,hook_handle,zlargv,zlargv);
	LOAD_HOOK(hooks,hook_handle,zlarnv,zlarnv);
	LOAD_HOOK(hooks,hook_handle,zlarrv,zlarrv);
	LOAD_HOOK(hooks,hook_handle,zlartg,zlartg);
	LOAD_HOOK(hooks,hook_handle,zlartv,zlartv);
	LOAD_HOOK(hooks,hook_handle,zlarz,zlarz);
	LOAD_HOOK(hooks,hook_handle,zlarzb,zlarzb);
	LOAD_HOOK(hooks,hook_handle,zlarzt,zlarzt);
	LOAD_HOOK(hooks,hook_handle,zlascl,zlascl);
	LOAD_HOOK(hooks,hook_handle,zlaset,zlaset);
	LOAD_HOOK(hooks,hook_handle,zlasr,zlasr);
	LOAD_HOOK(hooks,hook_handle,zlassq,zlassq);
	LOAD_HOOK(hooks,hook_handle,zlaswlq,zlaswlq);
	LOAD_HOOK(hooks,hook_handle,zlaswp,zlaswp);
	LOAD_HOOK(hooks,hook_handle,zlasyf,zlasyf);
	LOAD_HOOK(hooks,hook_handle,zlasyf_aa,zlasyf_aa);
	LOAD_HOOK(hooks,hook_handle,zlasyf_rk,zlasyf_rk);
	LOAD_HOOK(hooks,hook_handle,zlasyf_rook,zlasyf_rook);
	LOAD_HOOK(hooks,hook_handle,zlat2c,zlat2c);
	LOAD_HOOK(hooks,hook_handle,zlatbs,zlatbs);
	LOAD_HOOK(hooks,hook_handle,zlatdf,zlatdf);
	LOAD_HOOK(hooks,hook_handle,zlatps,zlatps);
	LOAD_HOOK(hooks,hook_handle,zlatrd,zlatrd);
	LOAD_HOOK(hooks,hook_handle,zlatrs,zlatrs);
	LOAD_HOOK(hooks,hook_handle,zlatrz,zlatrz);
	LOAD_HOOK(hooks,hook_handle,zlatsqr,zlatsqr);
	LOAD_HOOK(hooks,hook_handle,zlatzm,zlatzm);
	LOAD_HOOK(hooks,hook_handle,zlaunhr_col_getrfnp,zlaunhr_col_getrfnp);
	LOAD_HOOK(hooks,hook_handle,zlaunhr_col_getrfnp2,zlaunhr_col_getrfnp2);
	LOAD_HOOK(hooks,hook_handle,zlauu2,zlauu2);
	LOAD_HOOK(hooks,hook_handle,zlauum,zlauum);
	LOAD_HOOK(hooks,hook_handle,zpbcon,zpbcon);
	LOAD_HOOK(hooks,hook_handle,zpbequ,zpbequ);
	LOAD_HOOK(hooks,hook_handle,zpbrfs,zpbrfs);
	LOAD_HOOK(hooks,hook_handle,zpbstf,zpbstf);
	LOAD_HOOK(hooks,hook_handle,zpbsv,zpbsv);
	LOAD_HOOK(hooks,hook_handle,zpbsvx,zpbsvx);
	LOAD_HOOK(hooks,hook_handle,zpbtf2,zpbtf2);
	LOAD_HOOK(hooks,hook_handle,zpbtrf,zpbtrf);
	LOAD_HOOK(hooks,hook_handle,zpbtrs,zpbtrs);
	LOAD_HOOK(hooks,hook_handle,zpftrf,zpftrf);
	LOAD_HOOK(hooks,hook_handle,zpftri,zpftri);
	LOAD_HOOK(hooks,hook_handle,zpftrs,zpftrs);
	LOAD_HOOK(hooks,hook_handle,zpocon,zpocon);
	LOAD_HOOK(hooks,hook_handle,zpoequ,zpoequ);
	LOAD_HOOK(hooks,hook_handle,zpoequb,zpoequb);
	LOAD_HOOK(hooks,hook_handle,zporfs,zporfs);
	LOAD_HOOK(hooks,hook_handle,zposv,zposv);
	LOAD_HOOK(hooks,hook_handle,zposvx,zposvx);
	LOAD_HOOK(hooks,hook_handle,zpotf2,zpotf2);
	LOAD_HOOK(hooks,hook_handle,zpotrf,zpotrf);
	LOAD_HOOK(hooks,hook_handle,zpotrf2,zpotrf2);
	LOAD_HOOK(hooks,hook_handle,zpotri,zpotri);
	LOAD_HOOK(hooks,hook_handle,zpotrs,zpotrs);
	LOAD_HOOK(hooks,hook_handle,zppcon,zppcon);
	LOAD_HOOK(hooks,hook_handle,zppequ,zppequ);
	LOAD_HOOK(hooks,hook_handle,zpprfs,zpprfs);
	LOAD_HOOK(hooks,hook_handle,zppsv,zppsv);
	LOAD_HOOK(hooks,hook_handle,zppsvx,zppsvx);
	LOAD_HOOK(hooks,hook_handle,zpptrf,zpptrf);
	LOAD_HOOK(hooks,hook_handle,zpptri,zpptri);
	LOAD_HOOK(hooks,hook_handle,zpptrs,zpptrs);
	LOAD_HOOK(hooks,hook_handle,zpstf2,zpstf2);
	LOAD_HOOK(hooks,hook_handle,zpstrf,zpstrf);
	LOAD_HOOK(hooks,hook_handle,zptcon,zptcon);
	LOAD_HOOK(hooks,hook_handle,zpteqr,zpteqr);
	LOAD_HOOK(hooks,hook_handle,zptrfs,zptrfs);
	LOAD_HOOK(hooks,hook_handle,zptsv,zptsv);
	LOAD_HOOK(hooks,hook_handle,zptsvx,zptsvx);
	LOAD_HOOK(hooks,hook_handle,zpttrf,zpttrf);
	LOAD_HOOK(hooks,hook_handle,zpttrs,zpttrs);
	LOAD_HOOK(hooks,hook_handle,zptts2,zptts2);
	LOAD_HOOK(hooks,hook_handle,zrot,zrot);
	LOAD_HOOK(hooks,hook_handle,zspcon,zspcon);
	LOAD_HOOK(hooks,hook_handle,zspmv,zspmv);
	LOAD_HOOK(hooks,hook_handle,zspr,zspr);
	LOAD_HOOK(hooks,hook_handle,zsprfs,zsprfs);
	LOAD_HOOK(hooks,hook_handle,zspsv,zspsv);
	LOAD_HOOK(hooks,hook_handle,zspsvx,zspsvx);
	LOAD_HOOK(hooks,hook_handle,zsptrf,zsptrf);
	LOAD_HOOK(hooks,hook_handle,zsptri,zsptri);
	LOAD_HOOK(hooks,hook_handle,zsptrs,zsptrs);
	LOAD_HOOK(hooks,hook_handle,zstedc,zstedc);
	LOAD_HOOK(hooks,hook_handle,zstegr,zstegr);
	LOAD_HOOK(hooks,hook_handle,zstein,zstein);
	LOAD_HOOK(hooks,hook_handle,zstemr,zstemr);
	LOAD_HOOK(hooks,hook_handle,zsteqr,zsteqr);
	LOAD_HOOK(hooks,hook_handle,zsycon,zsycon);
	LOAD_HOOK(hooks,hook_handle,zsycon_3,zsycon_3);
	LOAD_HOOK(hooks,hook_handle,zsycon_rook,zsycon_rook);
	LOAD_HOOK(hooks,hook_handle,zsyconv,zsyconv);
	LOAD_HOOK(hooks,hook_handle,zsyconvf,zsyconvf);
	LOAD_HOOK(hooks,hook_handle,zsyconvf_rook,zsyconvf_rook);
	LOAD_HOOK(hooks,hook_handle,zsyequb,zsyequb);
	LOAD_HOOK(hooks,hook_handle,zsymv,zsymv);
	LOAD_HOOK(hooks,hook_handle,zsyr,zsyr);
	LOAD_HOOK(hooks,hook_handle,zsyrfs,zsyrfs);
	LOAD_HOOK(hooks,hook_handle,zsysv,zsysv);
	LOAD_HOOK(hooks,hook_handle,zsysv_aa,zsysv_aa);
	LOAD_HOOK(hooks,hook_handle,zsysv_aa_2stage,zsysv_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,zsysv_rk,zsysv_rk);
	LOAD_HOOK(hooks,hook_handle,zsysv_rook,zsysv_rook);
	LOAD_HOOK(hooks,hook_handle,zsysvx,zsysvx);
	LOAD_HOOK(hooks,hook_handle,zsyswapr,zsyswapr);
	LOAD_HOOK(hooks,hook_handle,zsytf2,zsytf2);
	LOAD_HOOK(hooks,hook_handle,zsytf2_rk,zsytf2_rk);
	LOAD_HOOK(hooks,hook_handle,zsytf2_rook,zsytf2_rook);
	LOAD_HOOK(hooks,hook_handle,zsytrf,zsytrf);
	LOAD_HOOK(hooks,hook_handle,zsytrf_aa,zsytrf_aa);
	LOAD_HOOK(hooks,hook_handle,zsytrf_aa_2stage,zsytrf_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,zsytrf_rk,zsytrf_rk);
	LOAD_HOOK(hooks,hook_handle,zsytrf_rook,zsytrf_rook);
	LOAD_HOOK(hooks,hook_handle,zsytri,zsytri);
	LOAD_HOOK(hooks,hook_handle,zsytri2,zsytri2);
	LOAD_HOOK(hooks,hook_handle,zsytri2x,zsytri2x);
	LOAD_HOOK(hooks,hook_handle,zsytri_3,zsytri_3);
	LOAD_HOOK(hooks,hook_handle,zsytri_3x,zsytri_3x);
	LOAD_HOOK(hooks,hook_handle,zsytri_rook,zsytri_rook);
	LOAD_HOOK(hooks,hook_handle,zsytrs,zsytrs);
	LOAD_HOOK(hooks,hook_handle,zsytrs2,zsytrs2);
	LOAD_HOOK(hooks,hook_handle,zsytrs_3,zsytrs_3);
	LOAD_HOOK(hooks,hook_handle,zsytrs_aa,zsytrs_aa);
	LOAD_HOOK(hooks,hook_handle,zsytrs_aa_2stage,zsytrs_aa_2stage);
	LOAD_HOOK(hooks,hook_handle,zsytrs_rook,zsytrs_rook);
	LOAD_HOOK(hooks,hook_handle,ztbcon,ztbcon);
	LOAD_HOOK(hooks,hook_handle,ztbrfs,ztbrfs);
	LOAD_HOOK(hooks,hook_handle,ztbtrs,ztbtrs);
	LOAD_HOOK(hooks,hook_handle,ztfsm,ztfsm);
	LOAD_HOOK(hooks,hook_handle,ztftri,ztftri);
	LOAD_HOOK(hooks,hook_handle,ztfttp,ztfttp);
	LOAD_HOOK(hooks,hook_handle,ztfttr,ztfttr);
	LOAD_HOOK(hooks,hook_handle,ztgevc,ztgevc);
	LOAD_HOOK(hooks,hook_handle,ztgex2,ztgex2);
	LOAD_HOOK(hooks,hook_handle,ztgexc,ztgexc);
	LOAD_HOOK(hooks,hook_handle,ztgsen,ztgsen);
	LOAD_HOOK(hooks,hook_handle,ztgsja,ztgsja);
	LOAD_HOOK(hooks,hook_handle,ztgsna,ztgsna);
	LOAD_HOOK(hooks,hook_handle,ztgsy2,ztgsy2);
	LOAD_HOOK(hooks,hook_handle,ztgsyl,ztgsyl);
	LOAD_HOOK(hooks,hook_handle,ztpcon,ztpcon);
	LOAD_HOOK(hooks,hook_handle,ztplqt,ztplqt);
	LOAD_HOOK(hooks,hook_handle,ztplqt2,ztplqt2);
	LOAD_HOOK(hooks,hook_handle,ztpmlqt,ztpmlqt);
	LOAD_HOOK(hooks,hook_handle,ztpmqrt,ztpmqrt);
	LOAD_HOOK(hooks,hook_handle,ztpqrt,ztpqrt);
	LOAD_HOOK(hooks,hook_handle,ztpqrt2,ztpqrt2);
	LOAD_HOOK(hooks,hook_handle,ztprfb,ztprfb);
	LOAD_HOOK(hooks,hook_handle,ztprfs,ztprfs);
	LOAD_HOOK(hooks,hook_handle,ztptri,ztptri);
	LOAD_HOOK(hooks,hook_handle,ztptrs,ztptrs);
	LOAD_HOOK(hooks,hook_handle,ztpttf,ztpttf);
	LOAD_HOOK(hooks,hook_handle,ztpttr,ztpttr);
	LOAD_HOOK(hooks,hook_handle,ztrcon,ztrcon);
	LOAD_HOOK(hooks,hook_handle,ztrevc,ztrevc);
	LOAD_HOOK(hooks,hook_handle,ztrevc3,ztrevc3);
	LOAD_HOOK(hooks,hook_handle,ztrexc,ztrexc);
	LOAD_HOOK(hooks,hook_handle,ztrrfs,ztrrfs);
	LOAD_HOOK(hooks,hook_handle,ztrsen,ztrsen);
	LOAD_HOOK(hooks,hook_handle,ztrsna,ztrsna);
	LOAD_HOOK(hooks,hook_handle,ztrsyl,ztrsyl);
	LOAD_HOOK(hooks,hook_handle,ztrti2,ztrti2);
	LOAD_HOOK(hooks,hook_handle,ztrtri,ztrtri);
	LOAD_HOOK(hooks,hook_handle,ztrtrs,ztrtrs);
	LOAD_HOOK(hooks,hook_handle,ztrttf,ztrttf);
	LOAD_HOOK(hooks,hook_handle,ztrttp,ztrttp);
	LOAD_HOOK(hooks,hook_handle,ztzrqf,ztzrqf);
	LOAD_HOOK(hooks,hook_handle,ztzrzf,ztzrzf);
	LOAD_HOOK(hooks,hook_handle,zunbdb,zunbdb);
	LOAD_HOOK(hooks,hook_handle,zunbdb1,zunbdb1);
	LOAD_HOOK(hooks,hook_handle,zunbdb2,zunbdb2);
	LOAD_HOOK(hooks,hook_handle,zunbdb3,zunbdb3);
	LOAD_HOOK(hooks,hook_handle,zunbdb4,zunbdb4);
	LOAD_HOOK(hooks,hook_handle,zunbdb5,zunbdb5);
	LOAD_HOOK(hooks,hook_handle,zunbdb6,zunbdb6);
	LOAD_HOOK(hooks,hook_handle,zuncsd,zuncsd);
	LOAD_HOOK(hooks,hook_handle,zuncsd2by1,zuncsd2by1);
	LOAD_HOOK(hooks,hook_handle,zung2l,zung2l);
	LOAD_HOOK(hooks,hook_handle,zung2r,zung2r);
	LOAD_HOOK(hooks,hook_handle,zungbr,zungbr);
	LOAD_HOOK(hooks,hook_handle,zunghr,zunghr);
	LOAD_HOOK(hooks,hook_handle,zungl2,zungl2);
	LOAD_HOOK(hooks,hook_handle,zunglq,zunglq);
	LOAD_HOOK(hooks,hook_handle,zungql,zungql);
	LOAD_HOOK(hooks,hook_handle,zungqr,zungqr);
	LOAD_HOOK(hooks,hook_handle,zungr2,zungr2);
	LOAD_HOOK(hooks,hook_handle,zungrq,zungrq);
	LOAD_HOOK(hooks,hook_handle,zungtr,zungtr);
	LOAD_HOOK(hooks,hook_handle,zungtsqr,zungtsqr);
	LOAD_HOOK(hooks,hook_handle,zunhr_col,zunhr_col);
	LOAD_HOOK(hooks,hook_handle,zunm22,zunm22);
	LOAD_HOOK(hooks,hook_handle,zunm2l,zunm2l);
	LOAD_HOOK(hooks,hook_handle,zunm2r,zunm2r);
	LOAD_HOOK(hooks,hook_handle,zunmbr,zunmbr);
	LOAD_HOOK(hooks,hook_handle,zunmhr,zunmhr);
	LOAD_HOOK(hooks,hook_handle,zunml2,zunml2);
	LOAD_HOOK(hooks,hook_handle,zunmlq,zunmlq);
	LOAD_HOOK(hooks,hook_handle,zunmql,zunmql);
	LOAD_HOOK(hooks,hook_handle,zunmqr,zunmqr);
	LOAD_HOOK(hooks,hook_handle,zunmr2,zunmr2);
	LOAD_HOOK(hooks,hook_handle,zunmr3,zunmr3);
	LOAD_HOOK(hooks,hook_handle,zunmrq,zunmrq);
	LOAD_HOOK(hooks,hook_handle,zunmrz,zunmrz);
	LOAD_HOOK(hooks,hook_handle,zunmtr,zunmtr);
	LOAD_HOOK(hooks,hook_handle,zupgtr,zupgtr);
	LOAD_HOOK(hooks,hook_handle,zupmtr,zupmtr);
#endif

    return 0;
}

