/*
 Fireworkx 2.2 - Pyrotechnics explosions simulation,
 an eyecandy, live animating colorful fireworks super-blasts..!
 Copyright (GPL) 1999-2013 Rony B Chandran <ronybc@gmail.com>

 From Kerala, INDIA
 Website: http://www.ronybc.com

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 Additional coding:
 ---------------------------------------------------------------------------------
 Support for different display color modes (Xlib version): put_image()
 Jean-Pierre Demailly <Jean-Pierre.Demailly@ujf-grenoble.fr>

 Fixed array access problems by beating on it with a large hammer.
 Nicholas Miell <nmiell@gmail.com>

 Help 'free'ing up of memory with needed 'XSync's.
 Renuka S <renuka@ronybc.com>
 Rugmini R Chandran <rugmini@ronybc.com>

 compile:
 sudo apt-get install libx11-dev
 gcc -O3 -mtune=native -march=native -ffast-math -fomit-frame-pointer -Wall -o fireworkx fireworkx.c `pkg-config --cflags --libs x11` -lm
\
 */

#define FIREWORKX_X11 1
#define FIREWORKX_FLAVOR "fireworkx-x11"
#include "fireworkx-engine.c"

void resize(struct state *st)
{
	unsigned int n;
	XWindowAttributes xwa;
	fireshell *fs = st->fireshell_array;

	XGetWindowAttributes (st->dpy, st->window, &xwa);
	st->width  = xwa.width;
	st->height = xwa.height;
	st->width  -= st->width % 4;
	st->height -= st->height % 2;

	XSync(st->dpy, 0);
	if (st->xim)
	{
		if (st->xim->data == (char *)st->palaka2) st->xim->data = NULL;
		XDestroyImage(st->xim);
		XSync(st->dpy, 0);
		free(st->mem2);
		free(st->mem1);
	}
	st->xim = XCreateImage(st->dpy, xwa.visual, xwa.depth, ZPixmap, 0, 0,
	                       st->width, st->height, 8, 0);
	st->width  = st->xim->width;
	st->height = st->xim->height;
#ifdef __SSE2__
	st->mem1 = _mm_malloc(((st->height + 2) * st->width + 8)*4, 16);
	bzero(st->mem1, ((st->height + 2) * st->width + 8)*4);
	st->mem2 = _mm_malloc(((st->height + 2) * st->width + 8)*4, 16);
	bzero(st->mem2, ((st->height + 2) * st->width + 8)*4);
#else
	st->mem1 = calloc((st->height + 2) * st->width + 8, 4);
	st->mem2 = calloc((st->height + 2) * st->width + 8, 4);
#endif
	st->palaka1 = (unsigned char *) st->mem1 + (st->width * 4 + 16);
	st->palaka2 = (unsigned char *) st->mem2 + (st->width * 4 + 16);

	if (st->depth >= 24)
	{
		st->xim->data = (char *)st->palaka2;
	}
	else
	{
		st->xim->data = calloc(st->height, st->xim->bytes_per_line);
	}

	if(st->verbose)
	{
		printf("resolution: %u x %u\n", st->width, st->height);
	}
	if (st->light_map) free(st->light_map);
	st->light_map = calloc((st->width * st->height * SHELLCOUNT)/4, sizeof(float));
	for (n = 0; n < SHELLCOUNT; n++, fs++)
	{
		render_light_map(st, fs);
	}
}

void fire_fullscreen(struct state *st)
{
	XEvent xev;
	Atom wm_state = XInternAtom(st->dpy, "_NET_WM_STATE", False);
	Atom fullscreen = XInternAtom(st->dpy, "_NET_WM_STATE_FULLSCREEN", False);
	memset(&xev, 0, sizeof(xev));
	xev.type = ClientMessage;
	xev.xclient.window = st->window;
	xev.xclient.message_type = wm_state;
	xev.xclient.format = 32;
	xev.xclient.data.l[0] = 1;
	xev.xclient.data.l[1] = fullscreen;
	xev.xclient.data.l[2] = 0;
	XSendEvent(st->dpy, DefaultRootWindow(st->dpy), False,
	           SubstructureNotifyMask, &xev);
}

void free_fireworkx(struct state *st)
{
	printf("Got 'DestroyNotify'\n");
	free(st->fireshell_array->fpix);
	free(st->fireshell_array);
	free(st->light_map);
	free(st->mem2);
	free(st->mem1);
}

void sniff_events(struct state *st)
{
	XEvent e;
	while (XPending(st->dpy))
	{
		XNextEvent (st->dpy, &e);
		if (e.type == DestroyNotify) free_fireworkx(st);
		if (e.type == ConfigureNotify) resize(st);
		if (e.type == ButtonPress) recycle_oldest(st, e.xbutton.x, e.xbutton.y);
		if (e.type == ButtonPress) fire_fullscreen(st);
		if (e.type == KeyPress) fire_fullscreen(st);
	}
}

void put_image(struct state *st)
{
	int x, y, i, j;
	unsigned char r, g, b;
	XImage *xim = st->xim;
	i = 0;
	j = 0;
	if (st->depth == 16)
	{
		if (st->bigendian)
			for (y = 0; y < xim->height; y++)
				for (x = 0; x < xim->width; x++)
				{
					r = st->palaka2[j++];
					g = st->palaka2[j++];
					b = st->palaka2[j++];
					j++;
					xim->data[i++] = (g & 224) >> 5 | (r & 248);
					xim->data[i++] = (b & 248) >> 3 | (g & 28) << 3;
				}
		else
			for (y = 0; y < xim->height; y++)
				for (x = 0; x < xim->width; x++)
				{
					r = st->palaka2[j++];
					g = st->palaka2[j++];
					b = st->palaka2[j++];
					j++;
					xim->data[i++] = (b & 248) >> 3 | (g & 28) << 3;
					xim->data[i++] = (g & 224) >> 5 | (r & 248);
				}
	}
	if (st->depth == 15)
	{
		if (st->bigendian)
			for (y = 0; y < xim->height; y++)
				for (x = 0; x < xim->width; x++)
				{
					r = st->palaka2[j++];
					g = st->palaka2[j++];
					b = st->palaka2[j++];
					j++;
					xim->data[i++] = (g & 192) >> 6 | (r & 248) >> 1;
					xim->data[i++] = (b & 248) >> 3 | (g & 56) << 2;
				}
		else
			for (y = 0; y < xim->height; y++)
				for (x = 0; x < xim->width; x++)
				{
					r = st->palaka2[j++];
					g = st->palaka2[j++];
					b = st->palaka2[j++];
					j++;
					xim->data[i++] = (b & 248) >> 3 | (g & 56) << 2;
					xim->data[i++] = (g & 192) >> 6 | (r & 248) >> 1;
				}
	}
	if (st->depth == 8)
	{
		for (y = 0; y < xim->height; y++)
			for (x = 0; x < xim->width; x++)
			{
				r = st->palaka2[j++];
				g = st->palaka2[j++];
				b = st->palaka2[j++];
				j++;
				xim->data[i++] = st->col_val[
				                     (((7*g)/256)*36)+(((6*r)/256)*6)+((6*b)/256)];
			}
	}
	XPutImage(st->dpy, st->window, st->gc, xim, 0, 0, 0, 0, xim->width, xim->height);
}

int main(int argc, char *argv[])
{
	unsigned int inroot = 0;
	Visual *vi;
	Colormap cmap;
	XSetWindowAttributes swa;
	XGCValues gcv;

	unsigned int n, q;
	float fps_avg = 0, fps_count = 0;
	time_t  fps_a = 0, fps_b = 0;

	firepix *fp;
	fireshell *fs;
	struct state real_st;
	struct state *st = &real_st;
	/* struct state *st = (struct state *) calloc(1, sizeof(*st)); */
	st->width = WIDTH;
	st->height = HEIGHT;
	st->max_shell_life = SHELL_LIFE_DEFAULT;
	st->flash_fade = 0.995;
	st->flash_on = 1;
	st->fullscreen = 0;
	st->fps_on = 0;
	st->shoot = 0;
	st->delay = 0;
	st->verbose = 0;
	st->mem1 = NULL;
	st->mem2 = NULL;
	st->palaka1 = NULL;
	st->palaka2 = NULL;
	st->light_map = NULL;

	fprintf(stderr, "Fireworkx %s - Pyrotechnics shell blast simulation,\n", FWXVERSION);
	fprintf(stderr, "an eyecandy, live animating colorful fireworks super-explosions..!\n");
	fprintf(stderr, "Copyright (GPL) 1999-2013 Rony B Chandran \n\n");
	fprintf(stderr, "Website: http://www.ronybc.com \n\n");

	while (1)
	{
		int option_index = 0;
		static struct option long_options[] =
		{
			{"width", no_argument, 0, 'w'},
			{"height", no_argument, 0, 'h'},
			{"maxlife", required_argument, 0, 'm'},
			{"noflash",  no_argument, 0, 'n'},
			{"fullscreen",  no_argument, 0, 'f'},
			{"fps", no_argument, 0, 'r'},
			{"delay", required_argument, 0, 'd'},
			{"verbose", no_argument, 0, 'v'},
			{0, 0,0, 0}
		};

		n = getopt_long(argc, argv, "w:h:m:nfrd:v", long_options, &option_index);
		if (n == -1) break;

		switch (n)
		{
		case 'w':
			st->width = atoi(optarg);
			break;
		case 'h':
			st->height = atoi(optarg);
			break;
		case 'm':
			st->max_shell_life = atoi(optarg);
			st->max_shell_life = pow(10.0,(atoi(optarg)/50.0)+2.7);
			break;
		case 'n':
			st->flash_on = 0;
			printf("light flash = OFF\n");
			break;
		case 'f':
			st->fullscreen = 1;
			break;
		case 'r':
			st->fps_on = 1;
			break;
		case 'd':
			st->delay = atoi(optarg);
			break;
		case 'v':
			st->verbose = 1;
			break;
		default:
			help_out();
			exit(0);
		}
	}

	if(st->max_shell_life < 1000) st->flash_fade = 0.998;

	st->width  -= st->width % 4;
	st->height -= st->height % 2;

	st->dpy = XOpenDisplay(0);
	if (!st->dpy)
	{
		fprintf(stderr, "\nError.. Failed to open display \n\n");
		exit(0);
	}
	vi = DefaultVisual(st->dpy, DefaultScreen(st->dpy));
	st->depth = DefaultDepth(st->dpy, DefaultScreen(st->dpy));
	st->bigendian = (ImageByteOrder(st->dpy) == MSBFirst);
	if (st->depth <= 8)
	{
		fprintf(stderr, "Pseudocolor visual: '%s' will give poor results !\n", FIREWORKX_FLAVOR);
	}

	cmap = XCreateColormap(st->dpy, RootWindow(st->dpy, DefaultScreen(st->dpy)),
	                       vi, AllocNone);
	if (st->depth == 8)
	{
		XColor xc;
		xc.flags = DoRed | DoGreen | DoBlue;
		st->col_val = malloc(256);
		for (n = 0; n < 252; n++)
		{
			xc.red = (unsigned short)(((n % 36) / 6) * 51 * 257);
			xc.green = (unsigned short)((n / 36) * 42 * 257);
			xc.blue = (unsigned short)((n % 6) * 51 * 257);
			if (!XAllocColor(st->dpy, cmap, &xc))
			{
				fprintf(stderr, "XAllocColor failed for entry %%%d\n", n);
				st->col_val[n] = 0;
			}
			else
				st->col_val[n] = xc.pixel;
		}
	}
	swa.colormap = cmap;
	swa.border_pixel = 0;
	swa.event_mask = StructureNotifyMask;
	if (inroot)
		st->window = RootWindow(st->dpy, DefaultScreen(st->dpy));
	else
	{
		st->window = XCreateWindow(st->dpy, RootWindow(st->dpy, DefaultScreen(st->dpy)),
		                           0, 0, st->width, st->height, 0, st->depth, InputOutput, vi,
		                           CWBorderPixel | CWColormap | CWEventMask, &swa);
		XSelectInput(st->dpy, st->window, StructureNotifyMask | ButtonPressMask);
		XStoreName(st->dpy, st->window, FIREWORKX_FLAVOR);
		XMapWindow(st->dpy, st->window);
	}
	if (st->depth == 16 && vi->green_mask == 992) st->depth = 15;
	st->gc = XCreateGC(st->dpy, st->window, 0, &gcv);

	srandom(time(0));
	srand48(random());

	fs = calloc(SHELLCOUNT, sizeof(fireshell));
	fp = calloc(PIXCOUNT * SHELLCOUNT, sizeof(firepix));
	st->fireshell_array = fs;
	
	st->xim = NULL;
	resize(st);   /* initialize palakas */
	
	if(st->fullscreen)
	{
		fire_fullscreen(st);
		sniff_events(st);
	}
	for (n = 0; n < SHELLCOUNT; n++, fs++)
	{
		fs->seq_number = n;
		fs->fpix = fp;
		recycle (st, fs, rnd(st->width), rnd(st->height));
		fp += PIXCOUNT;
	}

	if(st->fps_on)
	{
		fps_a = time(&fps_b);
		while(fps_a == fps_b)
		{
			time(&fps_b);
			usleep(1000);
		}
		fps_a = fps_b;
	}

#ifdef __SSE2__
	if(st->verbose)
	{
		printf("Using SSE2 optimization.\n");
	}
#endif
	while (1)
	{
		for (q = FTWEAK; q; q--)
		{
			fs = st->fireshell_array;
			for (n = SHELLCOUNT; n; n--, fs++)
			{
				if (!explode(st, fs))
				{
					recycle(st, fs, rnd(st->width), rnd(st->height));
				}
			}
		}

		glow_blur(st);
		if (st->flash_on)
		{
			chromo_2x2_light(st);
		}

		sniff_events(st);

		put_image(st);

		if (st->delay) usleep(st->delay * 1000);
		XSync(st->dpy, 0);

		if (st->fps_on)
		{
			fps_count++;
			time(&fps_b);
			if (fps_b != fps_a)
			{
				if(fps_avg == 0) fps_avg = fps_count;
				else fps_avg += (fps_count - fps_avg) / 5.0;
				printf("fps: %.1f last: %.0f \n", fps_avg, fps_count);
				fps_a = fps_b;
				fps_count = 0;
			}
		}
	}
}

/*
  code formatting used:
  $ astyle -A1 -t fireworkx.c
  ----------------------------------------------------------
  website: http://www.ronybc.com
\
 */

