/*
 Fireworkx 2.2 - Pyrotechnics explosions simulation,
 an eyecandy, live animating colorful fireworks super-blasts..!
 Copyright (GPL) 1999-2013 Rony B Chandran <ronybc@asia.com>

 From Kerala, INDIA
 Website: http://www.ronybc.com

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 Additional coding:
 ---------------------------------------------------------------------------------
 Support for different display color modes (Xlib version): put_image()
 Jean-Pierre Demailly <Jean-Pierre.Demailly@ujf-grenoble.fr>

 Fixed array access problems by beating on it with a large hammer.
 Nicholas Miell <nmiell@gmail.com>

 Help 'free'ing up of memory with needed 'XSync's.
 Renuka S <renuka@_ronybc.com>
 Rugmini R Chandran <rugmini@_ronybc.com>

 compile:
 sudo apt-get install libx11-dev
 gcc -O3 -mtune=native -march=native -ffast-math -fomit-frame-pointer -Wall -o fireworks_sdl fireworks_sdl.c `sdl-config --cflags --libs ;pkg-config --libs cairo`
\
 */

#define FIREWORKX_SDL 1
#define FIREWORKX_FLAVOR "fireworkx-sdl"
#include "fireworkx-engine.c"

void resize(struct state *st)
{
	unsigned int n;
	fireshell *fs = st->fireshell_array;

	if (st->light_map) free(st->light_map);
	if (st->mem2) free(st->mem2);
	if (st->mem1) free(st->mem1);

#ifdef __SSE2__
	st->mem1 = _mm_malloc(((st->height + 2) * st->width + 8)*4, 16);
	bzero(st->mem1, ((st->height + 2) * st->width + 8)*4);
	st->mem2 = _mm_malloc(((st->height + 2) * st->width + 8)*4, 16);
	bzero(st->mem2, ((st->height + 2) * st->width + 8)*4);
#else
	st->mem1 = calloc((st->height + 2) * st->width + 8, 4);
	st->mem2 = calloc((st->height + 2) * st->width + 8, 4);
#endif
	st->palaka1 = (unsigned char *) st->mem1 + (st->width * 4 + 16);
	st->palaka2 = (unsigned char *) st->mem2 + (st->width * 4 + 16);

	if(st->verbose)
	{
		printf("resolution: %u x %u\n", st->width, st->height);
	}
	st->light_map = calloc((st->width * st->height * SHELLCOUNT)/4, sizeof(float));
	for (n = 0; n < SHELLCOUNT; n++, fs++)
	{
		render_light_map(st, fs);
	}
}

int main(int argc, char *argv[])
{
	unsigned int n, q, extinguished = 0;
	float fps_avg = 0, fps_count = 0;
	time_t  fps_a = 0, fps_b = 0;

	firepix *fp;
	fireshell *fs;
	struct state real_st;
	struct state *st = &real_st;
	/* struct state *st = (struct state *) calloc(1, sizeof(*st)); */
	st->width = WIDTH;
	st->height = HEIGHT;
	st->max_shell_life = SHELL_LIFE_DEFAULT;
	st->flash_fade = 0.995;
	st->flash_on = 1;
	st->fullscreen = 0;
	st->fps_on = 0;
	st->shoot = 0;
	st->delay = 0;
	st->verbose = 0;
	st->mem1 = NULL;
	st->mem2 = NULL;
	st->palaka1 = NULL;
	st->palaka2 = NULL;
	st->light_map = NULL;

	fprintf(stderr, "Fireworkx %s - Pyrotechnics shell blast simulation,\n", FWXVERSION);
	fprintf(stderr, "an eyecandy, live animating colorful fireworks super-explosions..!\n");
	fprintf(stderr, "Copyright (GPL) 1999-2013 Rony B Chandran \n\n");
	fprintf(stderr, "Website: http://www.ronybc.com \n\n");

	while (1)
	{
		int option_index = 0;
		static struct option long_options[] =
		{
			{"width", no_argument, 0, 'w'},
			{"height", no_argument, 0, 'h'},
			{"maxlife", required_argument, 0, 'm'},
			{"noflash",  no_argument, 0, 'n'},
			{"fullscreen",  no_argument, 0, 'f'},
			{"fps", no_argument, 0, 'r'},
			{"delay", required_argument, 0, 'd'},
			{"verbose", no_argument, 0, 'v'},
			{0, 0,0, 0}
		};

		n = getopt_long(argc, argv, "w:h:m:nfrd:v", long_options, &option_index);
		if (n == -1) break;

		switch (n)
		{
		case 'w':
			st->width = atoi(optarg);
			break;
		case 'h':
			st->height = atoi(optarg);
			break;
		case 'm':
			st->max_shell_life = atoi(optarg);
			st->max_shell_life = pow(10.0,(atoi(optarg)/50.0)+2.7);
			break;
		case 'n':
			st->flash_on = 0;
			printf("light flash = OFF\n");
			break;
		case 'f':
			st->fullscreen = 1;
			break;
		case 'r':
			st->fps_on = 1;
			break;
		case 'd':
			st->delay = atoi(optarg);
			break;
		case 'v':
			st->verbose = 1;
			break;
		default:
			help_out();
			exit(0);
		}
	}

	if(st->max_shell_life < 1000) st->flash_fade = 0.998;

	st->width -= st->width % 8;
	st->height -= st->height % 8;

	SDL_Surface *sdl_surface;
	SDL_Event event;

	if( SDL_Init(SDL_INIT_VIDEO) == -1 )
	{
		return -1;
	}

	if( SDL_InitSubSystem(SDL_INIT_VIDEO) == -1 )
	{
		return -1;
	}

	if(st->fullscreen)
	{
		const SDL_VideoInfo* sdl_vinfo = SDL_GetVideoInfo();
		st->height = sdl_vinfo->current_h;
		st->width = sdl_vinfo->current_w;
		if(!(sdl_surface = SDL_SetVideoMode(st->width, st->height, 32, SDL_FULLSCREEN|SDL_HWSURFACE|SDL_HWACCEL))  )
		{
			return -1;
		}
	}
	else
	{
		if(!(sdl_surface = SDL_SetVideoMode(st->width, st->height, 32, SDL_DOUBLEBUF|SDL_HWSURFACE|SDL_HWACCEL|SDL_RESIZABLE)))
		{
			return -1;
		}
	}

	st->width -= st->width % 8;
	st->height -= st->height % 8;

	srandom(time(0));
	srand48(random());

	fs = calloc(SHELLCOUNT, sizeof(fireshell));
	fp = calloc(PIXCOUNT * SHELLCOUNT, sizeof(firepix));
	st->fireshell_array = fs;

	resize(st); /* initialize palakas */

	for (n = 0; n < SHELLCOUNT; n++, fs++)
	{
		fs->seq_number = n;
		fs->fpix = fp;
		recycle (st, fs, rnd(st->width), rnd(st->height));
		fp += PIXCOUNT;
	}

	SDL_Surface *sdl_palaka = SDL_CreateRGBSurfaceFrom(
	                              st->palaka2, st->width, st->height, 32, st->width*4,
	                              0x00ff0000, 0x0000ff00, 0x000000ff, 0);

	cairo_surface_t *cr_surface = cairo_image_surface_create_for_data(
	                                  st->palaka1,CAIRO_FORMAT_ARGB32,st->width,st->height,st->width*4);
	cairo_t *cr = cairo_create(cr_surface);

	if(st->verbose)
	{
		printf("cairo_format_stride_for_width() = %u (using %u)\n",
		       cairo_format_stride_for_width(CAIRO_FORMAT_ARGB32, st->width), st->width*4);
	}

#ifdef __SSE2__
	if(st->verbose)
	{
		printf("Using SSE2 optimization.\n");
	}
#endif

	if(st->fps_on)
	{
		fps_a = time(&fps_b);
		while(fps_a == fps_b)
		{
			time(&fps_b);
			usleep(1000);
		}
		fps_a = fps_b;
	}

	do
	{
		for (q = FTWEAK; q; q--)
		{
			fs = st->fireshell_array;
			for (n = SHELLCOUNT; n; n--, fs++)
			{
				if (!explode(st, fs))
				{
					recycle(st, fs, rnd(st->width), rnd(st->height));
				}
			}
		}

		//SDL_LockSurface(sdl_palaka);
		glow_blur(st);

		if (st->flash_on)
		{
			chromo_2x2_light(st);
		}

		if (st->fps_on)
		{
			fps_count++;
			time(&fps_b);
			if (fps_b != fps_a)
			{
				if(fps_avg == 0) fps_avg = fps_count;
				else fps_avg += (fps_count - fps_avg) / 5.0;
				printf("fps: %.1f last: %.0f \n", fps_avg, fps_count);
				fps_a = fps_b;
				fps_count = 0;
			}
			char fps_buf[16];
			sprintf(fps_buf, "fps: %.1f", fps_avg);
			cairo_set_source_rgba(cr, 1, 0.5, 0, 0.025);
			cairo_select_font_face (cr, "Serif", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_NORMAL);
			cairo_set_font_size (cr, 50.0);
			cairo_move_to (cr, 40, 100);
			cairo_show_text (cr, fps_buf);
		}

		//SDL_LockSurface(sdl_surface);
		//memcpy(sdl_surface->pixels, palaka2, height*width*4);
		//SDL_UnlockSurface(sdl_surface);
		//SDL_UpdateRect(sdl_surface, 0, 0, 0, 0);

		//SDL_UnlockSurface(sdl_palaka);
		SDL_BlitSurface(sdl_palaka, 0, sdl_surface, 0);
		SDL_Flip(sdl_surface);

		if(st->delay) SDL_Delay((Uint32) st->delay);

		while(SDL_PollEvent(&event))
		{
			switch(event.type)
			{
			case SDL_QUIT:
			case SDL_KEYDOWN:
				extinguished = 1;
				break;
			case SDL_VIDEORESIZE:
				st->width = event.resize.w;
				st->height = event.resize.h;
				st->width -= st->width % 8;
				st->height -= st->height % 8;
				SDL_FreeSurface(sdl_surface);
				resize(st);
				SDL_SetVideoMode(st->width, st->height, 32, SDL_DOUBLEBUF|SDL_HWSURFACE|SDL_HWACCEL|SDL_RESIZABLE);
				sdl_palaka = SDL_CreateRGBSurfaceFrom(
				                 st->palaka2, st->width, st->height, 32, st->width*4,
				                 0x00ff0000, 0x0000ff00, 0x000000ff, 0);
				cairo_surface_destroy(cr_surface);
				cairo_destroy(cr);
				cr_surface = cairo_image_surface_create_for_data(
				                 st->palaka1,CAIRO_FORMAT_ARGB32,st->width,st->height,
				                 cairo_format_stride_for_width(CAIRO_FORMAT_ARGB32, st->width));
				cr = cairo_create(cr_surface);
				break;
			}
		}
	}
	while(!extinguished);

	SDL_Quit();
	free(st->fireshell_array->fpix);
	free(st->fireshell_array);
	free(st->mem1);
	free(st->mem2);
	free(st->light_map);
	return 0;
}

/*
  code formatting used:
  $ astyle -A1 -t fireworkx.c
  ----------------------------------------------------------
  website: www.ronybc.com
\
 */

