/*
 Fireworkx 2.2 - Pyrotechnics explosions simulation,
 an eyecandy, live animating colorful fireworks super-blasts..!
 Copyright (GPL) 1999-2013 Rony B Chandran <ronybc@gmail.com>

 From Kerala, INDIA
 Website: http://www.ronybc.com

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 Additional coding:
 ---------------------------------------------------------------------------------
 Support for different display color modes (Xlib version): put_image()
 Jean-Pierre Demailly <Jean-Pierre.Demailly@ujf-grenoble.fr>

 Fixed array access problems by beating on it with a large hammer.
 Nicholas Miell <nmiell@gmail.com>

 Help 'free'ing up of memory with needed 'XSync's.
 Renuka S <renuka@ronybc.com>
 Rugmini R Chandran <rugmini@ronybc.com>
\
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include <time.h>
#include <getopt.h>
#ifdef __SSE2__
#include <x86intrin.h>
#endif
#ifdef FIREWORKX_X11
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#endif
#ifdef FIREWORKX_SDL
#include <SDL.h>
#include <cairo/cairo.h>
#endif

#define FWXVERSION "2.2"

#define WIDTH 1024                /* 888     */
#define HEIGHT 632                /* 548     */
#define SHELLCOUNT 4              /* FIXED NUMBER; for SSE optimization */
#define PIXCOUNT 500              /* 500     */
#define SHELL_LIFE_DEFAULT 3200   /* 3200    */
#define SHELL_LIFE_RATIO 6        /* 6       */
#define POWDER 5.0                /* 5.0     */
#define FTWEAK 12                 /* 12      */
#define FLASH_ZOOM 0.8            /* 1.0     */
#define G_ACCELERATION 0.001      /* GRAVITY */

typedef struct
{
	unsigned int burn;
	float x, y;
	float xv, yv;
} firepix;

typedef struct
{
	unsigned int cx, cy;
	unsigned int seq_number, life;
	unsigned int bicolor, flies, hshift, vshift;
	unsigned int mortar_fired, explode_y;
	float air_drag, vshift_phase;
	float flash_r, flash_g, flash_b;
	unsigned int h;
	double s, v;
	unsigned char r, g, b;
	firepix *fpix;
} fireshell;

struct state
{
	unsigned int fps_on;
	unsigned int flash_on;
	unsigned int shoot;
	unsigned int verbose;
	unsigned int width;
	unsigned int height;
	unsigned int fullscreen;
	unsigned int max_shell_life;
	unsigned int delay;
	float flash_fade;
	float *light_map;
	unsigned char *palaka1;
	unsigned char *palaka2;
	void *mem1;
	void *mem2;
	fireshell *fireshell_array;
#ifdef FIREWORKX_X11
	Display *dpy;
	Window window;
	XImage *xim;
	GC gc;
	int depth;
	int bigendian;
	unsigned char *col_val;
#endif
#ifdef FIREWORKX_XSS
	Display *dpy;
	Window window;
	XImage *xim;
	GC gc;
	XColor *colors;
	int depth;
	int bigendian;
	int ncolors;
#endif
};

#ifndef FIREWORKX_XSS
#define frand(x) (drand48()*(x))
#endif

#ifndef FIREWORKX_XSS
static void help_out()
{
	printf("\n\
	USAGE: %s [option]..  \n\
	-x, --width           \n\
	-y, --height          \n\
	-f, --fullscreen      \n\
	-m, --maxlife=LIFE    0 = dense 100 = sparse\n\
	-n, --noflash         turn off light flash; improves framerate.\n\
	-d, --delay=MSEC      delay between frames in milliseconds\n\
	-r, --fps             display framerate\n\
	-v, --verbose         for scientific purposes only..!\n\
	\n\n", FIREWORKX_FLAVOR);
}
#endif

/*
	will return zero.. divide with care.
*/
static unsigned int rnd(unsigned int x)
{
	return(random() % x);
}

#ifdef FIREWORKX_XSS
static void fs_roll_rgb(fireshell *fs)
{
	unsigned short r, g, b;
	hsv_to_rgb (fs->h, fs->s, fs->v, &r, &g, &b);
	fs->r = (unsigned char) (r >> 8);
	fs->g = (unsigned char) (g >> 8);
	fs->b = (unsigned char) (b >> 8);
}
#else
#define fs_roll_rgb(fs) (hsv_to_rgb(fs->h, fs->s, fs->v, &fs->r, &fs->g, &fs->b))
/*
	slightly modified version of hsv_to_rgb(), took from
	xscreensaver/utils/hsv.c by Jamie Zawinski <jwz@jwz.org>
*/
static void hsv_to_rgb(int h, double s, double v,
                       unsigned char *r, unsigned char *g, unsigned char *b)
{
	double H, S, V, R, G, B;
	double p1, p2, p3;
	double f;
	int i;

	if (s < 0) s = 0;
	if (v < 0) v = 0;
	if (s > 1) s = 1;
	if (v > 1) v = 1;

	S = s;
	V = v;
	H = (h % 360) / 60.0;
	i = H;
	f = H - i;
	p1 = V * (1 - S);
	p2 = V * (1 - (S * f));
	p3 = V * (1 - (S * (1 - f)));
	if (i == 0)
	{
		R = V;
		G = p3;
		B = p1;
	}
	else if (i == 1)
	{
		R = p2;
		G = V;
		B = p1;
	}
	else if (i == 2)
	{
		R = p1;
		G = V;
		B = p3;
	}
	else if (i == 3)
	{
		R = p1;
		G = p2;
		B = V;
	}
	else if (i == 4)
	{
		R = p3;
		G = p1;
		B = V;
	}
	else
	{
		R = V;
		G = p1;
		B = p2;
	}
	*r = (unsigned char) (R * 255.0);
	*g = (unsigned char) (G * 255.0);
	*b = (unsigned char) (B * 255.0);
}

#endif

static void mix_colors(fireshell *fs)
{
	float flash;
	fs->h = rnd(360);
	fs->s = frand(0.4) + 0.6;
	fs->v = 1.0;
	fs_roll_rgb(fs);

	flash = rnd(444) + 111; /* Mega Jouls ! */
	fs->flash_r = fs->r * flash;
	fs->flash_g = fs->g * flash;
	fs->flash_b = fs->b * flash;
}

static void render_light_map(struct state *st, fireshell *fs)
{
	signed int x, y, v = 0;
	for (y = 0, v = fs->seq_number; y < st->height; y += 2)
	{
		for (x = 0; x < st->width; x += 2, v += SHELLCOUNT)
		{
			float f;
			f = sqrtf((fs->cx - x) * (fs->cx - x) + (fs->cy - y) * (fs->cy - y)) + 4.0;
			f = FLASH_ZOOM / f;
			f += pow(f,0.1) * frand(0.0001); /* dither */
			st->light_map[v] = f;
		}
	}
}

static void recycle(struct state *st, fireshell *fs, unsigned int x, unsigned int y)
{
	unsigned int n, pixlife;
	firepix *fp = fs->fpix;
	fs->mortar_fired = st->shoot;
	fs->explode_y = y;
	fs->cx = x;
	fs->cy = st->shoot ? st->height : y ;
	fs->life = rnd(st->max_shell_life) + (st->max_shell_life/SHELL_LIFE_RATIO);
	fs->life += !rnd(25) ? st->max_shell_life * 5 : 0;
	fs->air_drag = 1.0 - (float)(rnd(200)) / (10000.0 + fs->life);
	fs->bicolor = !rnd(5) ? 120 : 0;
	fs->flies = !rnd(10) ? 1 : 0; /* flies' motion */
	fs->hshift = !rnd(5) ? 1 : 0; /* hue shifting  */
	fs->vshift = !rnd(10) ? 1 : 0; /* value shifting */
	fs->vshift_phase = M_PI/2.0;
	pixlife = rnd(fs->life) + fs->life / 10 + 1;    /* ! */
	for (n = 0; n < PIXCOUNT; n++)
	{
		fp->burn = rnd(pixlife) + 32;
		fp->xv = frand(2.0) * POWDER - POWDER;
		fp->yv = sqrt(POWDER * POWDER - fp->xv * fp->xv) * (frand(2.0) - 1.0);
		fp->x = x;
		fp->y = y;
		fp++;
	}
	mix_colors(fs);
	render_light_map(st, fs);
}

static void recycle_oldest(struct state *st, unsigned int x, unsigned int y)
{
	unsigned int n;
	fireshell *fs, *oldest;
	fs = oldest = st->fireshell_array;
	for (n = 0; n < SHELLCOUNT; n++)
	{
		if(fs[n].life < oldest->life) oldest = &fs[n];
	}
	recycle(st, oldest, x, y);
}

static void rotate_hue(fireshell *fs, int dh)
{
	fs->h = fs->h + dh;
	fs->s = fs->s - 0.001;
	fs_roll_rgb(fs);
}

static void wave_value(fireshell *fs)
{
	fs->vshift_phase = fs->vshift_phase + 0.008;
	fs->v = fabs(sin(fs->vshift_phase));
	fs_roll_rgb(fs);
}

static int explode(struct state *st, fireshell *fs)
{
	float air_drag;
	unsigned int n;
	unsigned int h = st->height;
	unsigned int w = st->width;
	unsigned char r, g, b;
	unsigned char *prgba;
	unsigned char *palaka = st->palaka1;
	firepix *fp = fs->fpix;
	if (fs->mortar_fired)
	{
		if (--fs->cy == fs->explode_y)
		{
			fs->mortar_fired = 0;
			mix_colors(fs);
			render_light_map(st, fs);
		}
		else
		{
			fs->flash_r =
			    fs->flash_g =
			        fs->flash_b = 50 + (fs->cy - fs->explode_y) * 10;
			prgba = palaka + (fs->cy * w + fs->cx + rnd(5) - 2) * 4;
			prgba[0] = (rnd(32) + 128);
			prgba[1] = (rnd(32) + 128);
			prgba[2] = (rnd(32) + 128);
			return(1);
		}
	}
	if ((fs->bicolor + 1) % 50 == 0) rotate_hue(fs, 180);
	if (fs->bicolor) --fs->bicolor;
	if (fs->hshift) rotate_hue(fs, rnd(8));
	if (fs->vshift) wave_value(fs);
	if (fs->flash_r > 1.0) fs->flash_r *= st->flash_fade;
	if (fs->flash_g > 1.0) fs->flash_g *= st->flash_fade;
	if (fs->flash_b > 1.0) fs->flash_b *= st->flash_fade;
	air_drag = fs->air_drag;
	r = fs->r;
	g = fs->g;
	b = fs->b;
	for (n = 0; n < PIXCOUNT; n++, fp++)
	{
		if (!fp->burn) continue;
		--fp->burn;
		if (fs->flies)
		{
			fp->x += fp->xv = fp->xv * air_drag + frand(0.1) - 0.05;
			fp->y += fp->yv = fp->yv * air_drag + frand(0.1) - 0.05 + G_ACCELERATION;
		}
		else
		{
			fp->x += fp->xv = fp->xv * air_drag + frand(0.01) - 0.005;
			fp->y += fp->yv = fp->yv * air_drag + frand(0.005) - 0.0025 + G_ACCELERATION;
		}
		if (fp->y > h)
		{
			if (rnd(5) == 3)
			{
				fp->yv *= -0.24;
				fp->y = h;
			}
			/* touch muddy ground :) */
			else fp->burn = 0;
		}
		if (fp->x < w && fp->x > 0 && fp->y < h && fp->y > 0)
		{
			prgba = palaka + ((int)fp->y * w + (int)fp->x) * 4;
			prgba[0] = b;
			prgba[1] = g;
			prgba[2] = r;
		}
	}
	return(--fs->life);
}

#ifdef __SSE2__

/* SSE2 optimized versions of glow_blur() and chromo_2x2_light() */

static void glow_blur(struct state *st)
{
	unsigned int n, nn;
	unsigned char *ps = st->palaka1;
	unsigned char *pd = st->palaka2;
	unsigned char *pa = st->palaka1 - (st->width * 4);
	unsigned char *pb = st->palaka1 + (st->width * 4);
	__m128i xmm0, xmm1, xmm2, xmm3, xmm4;

	xmm0 = _mm_setzero_si128();
	nn = st->width * st->height * 4;
	for (n = 0; n < nn; n+=16)
	{
		_mm_prefetch((const void *)&ps[n+16],_MM_HINT_T0);
		_mm_prefetch((const void *)&pa[n+16],_MM_HINT_T0);
		_mm_prefetch((const void *)&pb[n+16],_MM_HINT_T0);

		xmm1 = _mm_load_si128((const __m128i*)&ps[n]);
		xmm2 = xmm1;
		xmm1 = _mm_unpacklo_epi8(xmm1,xmm0);
		xmm2 = _mm_unpackhi_epi8(xmm2,xmm0);
		xmm3 = _mm_loadu_si128((const __m128i*)&ps[n+4]);
		xmm4 = xmm3;
		xmm3 = _mm_unpacklo_epi8(xmm3,xmm0);
		xmm4 = _mm_unpackhi_epi8(xmm4,xmm0);
		xmm3 = _mm_slli_epi16(xmm3,3);
		xmm4 = _mm_slli_epi16(xmm4,3);
		xmm1 = _mm_add_epi16(xmm1,xmm3);
		xmm2 = _mm_add_epi16(xmm2,xmm4);
		xmm3 = _mm_loadu_si128((const __m128i*)&ps[n+8]);
		xmm4 = xmm3;
		xmm3 = _mm_unpacklo_epi8(xmm3,xmm0);
		xmm4 = _mm_unpackhi_epi8(xmm4,xmm0);
		xmm1 = _mm_add_epi16(xmm1,xmm3);
		xmm2 = _mm_add_epi16(xmm2,xmm4);

		xmm3 = _mm_load_si128((const __m128i*)&pa[n]);
		xmm4 = xmm3;
		xmm3 = _mm_unpacklo_epi8(xmm3,xmm0);
		xmm4 = _mm_unpackhi_epi8(xmm4,xmm0);
		xmm1 = _mm_add_epi16(xmm1,xmm3);
		xmm2 = _mm_add_epi16(xmm2,xmm4);
		xmm3 = _mm_loadu_si128((const __m128i*)&pa[n+4]);
		xmm4 = xmm3;
		xmm3 = _mm_unpacklo_epi8(xmm3,xmm0);
		xmm4 = _mm_unpackhi_epi8(xmm4,xmm0);
		xmm1 = _mm_add_epi16(xmm1,xmm3);
		xmm2 = _mm_add_epi16(xmm2,xmm4);
		xmm3 = _mm_loadu_si128((const __m128i*)&pa[n+8]);
		xmm4 = xmm3;
		xmm3 = _mm_unpacklo_epi8(xmm3,xmm0);
		xmm4 = _mm_unpackhi_epi8(xmm4,xmm0);
		xmm1 = _mm_add_epi16(xmm1,xmm3);
		xmm2 = _mm_add_epi16(xmm2,xmm4);

		xmm3 = _mm_load_si128((const __m128i*)&pb[n]);
		xmm4 = xmm3;
		xmm3 = _mm_unpacklo_epi8(xmm3,xmm0);
		xmm4 = _mm_unpackhi_epi8(xmm4,xmm0);
		xmm1 = _mm_add_epi16(xmm1,xmm3);
		xmm2 = _mm_add_epi16(xmm2,xmm4);
		xmm3 = _mm_loadu_si128((const __m128i*)&pb[n+4]);
		xmm4 = xmm3;
		xmm3 = _mm_unpacklo_epi8(xmm3,xmm0);
		xmm4 = _mm_unpackhi_epi8(xmm4,xmm0);
		xmm1 = _mm_add_epi16(xmm1,xmm3);
		xmm2 = _mm_add_epi16(xmm2,xmm4);
		xmm3 = _mm_loadu_si128((const __m128i*)&pb[n+8]);
		xmm4 = xmm3;
		xmm3 = _mm_unpacklo_epi8(xmm3,xmm0);
		xmm4 = _mm_unpackhi_epi8(xmm4,xmm0);
		xmm1 = _mm_add_epi16(xmm1,xmm3);
		xmm2 = _mm_add_epi16(xmm2,xmm4);

		xmm3 = xmm1;
		xmm4 = xmm2;
		xmm1 = _mm_srli_epi16(xmm1,4);
		xmm2 = _mm_srli_epi16(xmm2,4);
		xmm3 = _mm_srli_epi16(xmm3,3);
		xmm4 = _mm_srli_epi16(xmm4,3);
		xmm1 = _mm_packus_epi16(xmm1,xmm2);
		xmm3 = _mm_packus_epi16(xmm3,xmm4);

		_mm_storeu_si128((__m128i*)&ps[n+4], xmm1);
		_mm_storeu_si128((__m128i*)&pd[n+4], xmm3);
	}
}

static void chromo_2x2_light(struct state *st)
{
	__m128 xmm0, xmm1, xmm2, xmm3, xmm4, xmm5, xmm6;
	__m128i xmi4, xmi5, xmi6, xmi7;

	unsigned int x, y, v = 0;
	unsigned int nl = st->width * 4;
	unsigned char *mem = st->palaka2;
	fireshell *fs = st->fireshell_array;

	xmm0 = _mm_setr_ps(fs[0].flash_b, fs[0].flash_g, fs[0].flash_r, 0.0);
	xmm1 = _mm_setr_ps(fs[1].flash_b, fs[1].flash_g, fs[1].flash_r, 0.0);
	xmm2 = _mm_setr_ps(fs[2].flash_b, fs[2].flash_g, fs[2].flash_r, 0.0);
	xmm3 = _mm_setr_ps(fs[3].flash_b, fs[3].flash_g, fs[3].flash_r, 0.0);

	for (y = st->height/2; y; y--, mem += nl)
	{
		for (x = st->width/4; x; x--, v += 8, mem += 16)
		{
			xmm4 = _mm_set1_ps(st->light_map[v+0]);
			xmm5 = xmm0;
			xmm5 = _mm_mul_ps(xmm5,xmm4);
			xmm4 = _mm_set1_ps(st->light_map[v+1]);
			xmm4 = _mm_mul_ps(xmm4,xmm1);
			xmm5 = _mm_add_ps(xmm5,xmm4);
			xmm4 = _mm_set1_ps(st->light_map[v+2]);
			xmm4 = _mm_mul_ps(xmm4,xmm2);
			xmm5 = _mm_add_ps(xmm5,xmm4);
			xmm4 = _mm_set1_ps(st->light_map[v+3]);
			xmm4 = _mm_mul_ps(xmm4,xmm3);
			xmm5 = _mm_add_ps(xmm5,xmm4);

			xmm4 = _mm_set1_ps(st->light_map[v+4]);
			xmm6 = xmm0;
			xmm6 = _mm_mul_ps(xmm6,xmm4);
			xmm4 = _mm_set1_ps(st->light_map[v+5]);
			xmm4 = _mm_mul_ps(xmm4,xmm1);
			xmm6 = _mm_add_ps(xmm6,xmm4);
			xmm4 = _mm_set1_ps(st->light_map[v+6]);
			xmm4 = _mm_mul_ps(xmm4,xmm2);
			xmm6 = _mm_add_ps(xmm6,xmm4);
			xmm4 = _mm_set1_ps(st->light_map[v+7]);
			xmm4 = _mm_mul_ps(xmm4,xmm3);
			xmm6 = _mm_add_ps(xmm6,xmm4);

			xmi6 = _mm_cvtps_epi32(xmm5);
			xmi7 = _mm_cvtps_epi32(xmm6);
			xmi6 = _mm_packs_epi32(xmi6,xmi6);
			xmi7 = _mm_packs_epi32(xmi7,xmi7);

			xmi4 = _mm_load_si128((const __m128i*) mem);
			xmi5 = _mm_unpacklo_epi8(xmi5,xmi4);
			xmi5 = _mm_srli_epi16(xmi5,8);
			xmi4 = _mm_unpackhi_epi8(xmi4,xmi4);
			xmi4 = _mm_srli_epi16(xmi4,8);
			xmi5 = _mm_add_epi16(xmi5,xmi6);
			xmi4 = _mm_add_epi16(xmi4,xmi7);
			xmi5 = _mm_packus_epi16(xmi5,xmi4);
			_mm_store_si128((__m128i*) mem, xmi5);

			xmi4 = _mm_load_si128((const __m128i*) &mem[nl]);
			xmi5 = _mm_unpacklo_epi8(xmi5,xmi4);
			xmi5 = _mm_srli_epi16(xmi5,8);
			xmi4 = _mm_unpackhi_epi8(xmi4,xmi4);
			xmi4 = _mm_srli_epi16(xmi4,8);
			xmi5 = _mm_add_epi16(xmi5,xmi6);
			xmi4 = _mm_add_epi16(xmi4,xmi7);
			xmi5 = _mm_packus_epi16(xmi5,xmi4);
			_mm_store_si128((__m128i*) &mem[nl], xmi5);
		}
	}
}

#else

static void glow_blur(struct state *st)
{
	unsigned int n, q;
	unsigned char *pm = st->palaka1;
	unsigned char *po = st->palaka2;
	unsigned char *pa = pm - (st->width * 4);
	unsigned char *pb = pm + (st->width * 4);
	/*
		unsigned int rgba = 0;
		for (n = st->width*st->height*4; n; n--, pm++, pa++, pb++, po++)
		{
			if(++rgba > 3)
			{
				rgba = 0;
				continue;
			}
			q     = pm[0] + pm[4] * 8 + pm[8] +
			        pa[0] + pa[4] + pa[8] +
			        pb[0] + pb[4] + pb[8];
			pm[4] = q >> 4;
			po[4] = q > 2047 ? 255 : q >> 3;
		}
			--- using unrolled version ------------
	*/
	for (n = st->width*st->height*4; n; n-=4)
	{
		q = pm[0] + pm[4] * 8 + pm[8] +
		    pa[0] + pa[4] + pa[8] +
		    pb[0] + pb[4] + pb[8];
		pm[4] = q >> 4;
		po[4] = q > 2047 ? 255 : q >> 3;
		q = pm[1] + pm[5] * 8 + pm[9] +
		    pa[1] + pa[5] + pa[9] +
		    pb[1] + pb[5] + pb[9];
		pm[5] = q >> 4;
		po[5] = q > 2047 ? 255 : q >> 3;
		q = pm[2] + pm[6] * 8 + pm[10] +
		    pa[2] + pa[6] + pa[10] +
		    pb[2] + pb[6] + pb[10];
		pm[6] = q >> 4;
		po[6] = q > 2047 ? 255 : q >> 3;

		pm+=4, pa+=4, pb+=4, po+=4;
	}
}

static inline unsigned char addbs(unsigned char c, unsigned int i)
{
	i += c;
	return(i > 255 ? 255 : i);
}

static void chromo_2x2_light(struct state *st)
{
	unsigned int n, x, y, v = 0;
	unsigned int nl = st->width * 4;
	unsigned char *mem = st->palaka2;
	float r, g, b;
	float rgb[SHELLCOUNT*4];
	fireshell *fs = st->fireshell_array;

	for (n = 0, x = 0; n < SHELLCOUNT; n++, x += 4, fs++)
	{
		rgb[x  ] = fs->flash_r;
		rgb[x+1] = fs->flash_g;
		rgb[x+2] = fs->flash_b;
	}

	for (y = st->height/2; y; y--)
	{
		for (x = st->width/2; x; x--, v += 4)
		{
			r = rgb[0] * st->light_map[v] + rgb[4] * st->light_map[v+1]
			    + rgb[ 8] * st->light_map[v+2] + rgb[12] * st->light_map[v+3];
			g = rgb[1] * st->light_map[v] + rgb[5] * st->light_map[v+1]
			    + rgb[ 9] * st->light_map[v+2] + rgb[13] * st->light_map[v+3];
			b = rgb[2] * st->light_map[v] + rgb[6] * st->light_map[v+1]
			    + rgb[10] * st->light_map[v+2] + rgb[14] * st->light_map[v+3];

			mem[0] = addbs(mem[0], b);
			mem[1] = addbs(mem[1], g);
			mem[2] = addbs(mem[2], r);
			mem[4] = addbs(mem[4], b);
			mem[5] = addbs(mem[5], g);
			mem[6] = addbs(mem[6], r);

			mem += nl;

			mem[0] = addbs(mem[0], b);
			mem[1] = addbs(mem[1], g);
			mem[2] = addbs(mem[2], r);
			mem[4] = addbs(mem[4], b);
			mem[5] = addbs(mem[5], g);
			mem[6] = addbs(mem[6], r);

			mem -= nl - 8;
		}
		mem += nl;
	}
}

#endif
