/*
 * Fuzzy Fingerprinting - Attacking vulnerabilities in the Human Brain
 * Copyright 2002 Plasmoid <plasmoid@thc.org> - All rights reserved. 
 * On behalf of The Hacker's Choice - http://www.thc.org
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright   
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright   
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR `AS IS'' AND ANY EXPRESS OR     
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY   
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "includes.h"
RCSID("$Id: xkey.c,v 1.9 2003/10/25 14:08:14 plasmoid Exp $");

#include <openssl/rsa.h>
#include <openssl/dsa.h>
#include <openssl/bn.h>
#include <openssl/err.h>
#include <openssl/rand.h>
#include <openssl/pem.h>

#include "xkey.h"
#include "rsa.h"
#include "dsa.h"

/*
 * Set the key operations.
 */
void xkey_set_ops(xkey_t * xkey, int type)
{
   switch (type) {
   case XKEY_RSA:
      xkey->init_key = (int (*)()) RSA_init_key;
      xkey->next_key = (int (*)()) RSA_next_key;
      xkey->complete_key = (int (*)()) RSA_complete_key;
      xkey->free_key = (void (*)()) RSA_free_key;
      xkey->read_key = (int (*)()) RSA_read_key;
      xkey->write_key = (int (*)()) RSA_write_key;
      xkey->clone_key = (int (*)()) RSA_clone_key;
      xkey->print_key = (void (*)()) RSA_print_key;
      break;
   case XKEY_DSA:
      xkey->init_key = (int (*)()) DSA_init_key;
      xkey->next_key = (int (*)()) DSA_next_key;
      xkey->complete_key = (int (*)()) DSA_complete_key;
      xkey->free_key = (void (*)()) DSA_free_key;
      xkey->read_key = (int (*)()) DSA_read_key;
      xkey->write_key = (int (*)()) DSA_write_key;
      xkey->clone_key = (int (*)()) DSA_clone_key;
      xkey->print_key = (void (*)()) DSA_print_key;
      break;
   default:
      err_msg("xkey_set_ops: Unknown key type");
   }
}

/*
 * Init a key object. A key object also contains two auxiliary big numbers
 * alpha and beta that might be used to carry additional values during 
 * the crunch process.
 */
int xkey_init(xkey_t ** xkey, int type, int bits, int flags)
{
   *xkey = (xkey_t *) malloc(sizeof(xkey_t));
   if (!*xkey) {
      err_msg("xkey_new: Could not allocate xkey");
      return 0;
   }

   (*xkey)->bits = bits;
   (*xkey)->type = type;
   (*xkey)->flags = flags;
   (*xkey)->alpha = NULL;
   (*xkey)->beta = NULL;

   xkey_set_ops(*xkey, type);
   return (*xkey)->init_key(&(*xkey)->alg, bits, &(*xkey)->alpha,
			    &(*xkey)->beta);
}

/*
 * Wrapper. Free the key.
 */
void xkey_free(xkey_t * xkey)
{
   xkey->free_key(xkey->alg, xkey->alpha, xkey->beta);
   free(xkey);
}

/*
 * Wrapper. Calculate the next key.
 */
int xkey_next(xkey_t * xkey)
{
   return xkey->next_key(xkey->alg, xkey->flags, xkey->alpha, xkey->beta);
}

/*
 * Wrapper. Complete the key.
 */
int xkey_complete(xkey_t * xkey)
{
   return xkey->complete_key(xkey->alg);
}

/*
 * Clone the key
 */
int xkey_clone(xkey_t ** xkey1, xkey_t * xkey2)
{
   *xkey1 = (xkey_t *) malloc(sizeof(xkey_t));

   if (!*xkey1)
      goto err;

   (*xkey1)->type = xkey2->type;
   (*xkey1)->bits = xkey2->bits;
   (*xkey1)->flags = xkey2->flags;

   xkey_set_ops(*xkey1, xkey2->type);
   return xkey2->clone_key(&(*xkey1)->alg, &(*xkey1)->alpha,
			   &(*xkey1)->beta, xkey2->alg, xkey2->alpha,
			   xkey2->beta);

 err:
   err_msg("xkey_clone: Could not clone xkey");
   return 0;
}

/*
 * Read a key from file (fd) and allocate all necessary structures
 */
int xkey_read(xkey_t ** xkey, int fd)
{
   *xkey = (xkey_t *) malloc(sizeof(xkey_t));

   if (!*xkey)
      goto err1;

   if (read(fd, &(*xkey)->type, sizeof(int)) != sizeof(int))
      goto err2;
   if (read(fd, &(*xkey)->bits, sizeof(int)) != sizeof(int))
      goto err2;
   if (read(fd, &(*xkey)->flags, sizeof(int)) != sizeof(int))
      goto err2;

   xkey_set_ops(*xkey, (*xkey)->type);

   return (*xkey)->read_key(&(*xkey)->alg, fd, &(*xkey)->alpha,
			    &(*xkey)->beta);
 err2:
   free(*xkey);
 err1:
   err_msg("xkey_read: Could not read xkey");
   return 0;
}

/*
 * Write a key to file (fd).
 */
int xkey_write(xkey_t * xkey, int fd)
{
   if (!xkey)
      goto err;

   if (write(fd, &xkey->type, sizeof(int)) != sizeof(int))
      goto err;
   if (write(fd, &xkey->bits, sizeof(int)) != sizeof(int))
      goto err;
   if (write(fd, &xkey->flags, sizeof(int)) != sizeof(int))
      goto err;

   return xkey->write_key(xkey->alg, fd, xkey->alpha, xkey->beta);
 err:
   err_msg("xkey_write: Could not write xkey");
   return 0;
}

/*
 * Wrapper. Print a key.
 */
void xkey_print(xkey_t * xkey, FILE * out)
{
   if (!xkey || !out)
      return;
   xkey->print_key(xkey->alg, out, xkey->alpha, xkey->beta);
   fprintf(out, "             Generation Mode: %s",
	   ((xkey->flags & XKEY_RSA_SLOPPY)
	    || (xkey->flags & XKEY_DSA_SLOPPY)) ? "Sloppy\n" : "Accurate\n");

}

/*
 * Get key type from the given name
 */
int xkey_type_from_name(char *name)
{
   if (strcmp(name, "rsa") == 0) {
      return XKEY_RSA;
   } else if (strcmp(name, "dsa") == 0) {
      return XKEY_DSA;
   }

   return -1;
}
