/*
 * Copyright (c) 2000 Markus Friedl.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "includes.h"
RCSID("$Id: uuencode.c,v 1.2 2002/08/13 16:16:40 plasmoid Exp $");

#include "uuencode.h"

static const char Base64[] =
    "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
static const char Pad64 = '=';

int b64_ntop(unsigned char *src, size_t srclength, unsigned char *target,
	     size_t targsize)
{
   size_t datalength = 0;
   unsigned char input[3];
   unsigned char output[4];
   int i;

   while (2 < srclength) {
      input[0] = *src++;
      input[1] = *src++;
      input[2] = *src++;
      srclength -= 3;

      output[0] = input[0] >> 2;
      output[1] = ((input[0] & 0x03) << 4) + (input[1] >> 4);
      output[2] = ((input[1] & 0x0f) << 2) + (input[2] >> 6);
      output[3] = input[2] & 0x3f;

      if (datalength + 4 > targsize)
	 return (-1);
      target[datalength++] = Base64[output[0]];
      target[datalength++] = Base64[output[1]];
      target[datalength++] = Base64[output[2]];
      target[datalength++] = Base64[output[3]];
   }

   /* Now we worry about padding. */
   if (0 != srclength) {
      /* Get what's left. */
      input[0] = input[1] = input[2] = '\0';
      for (i = 0; i < srclength; i++)
	 input[i] = *src++;

      output[0] = input[0] >> 2;
      output[1] = ((input[0] & 0x03) << 4) + (input[1] >> 4);
      output[2] = ((input[1] & 0x0f) << 2) + (input[2] >> 6);

      if (datalength + 4 > targsize)
	 return (-1);
      target[datalength++] = Base64[output[0]];
      target[datalength++] = Base64[output[1]];
      if (srclength == 1)
	 target[datalength++] = Pad64;
      else
	 target[datalength++] = Base64[output[2]];
      target[datalength++] = Pad64;
   }
   if (datalength >= targsize)
      return (-1);
   target[datalength] = '\0';	/* Returned value doesn't count \0. */
   return (datalength);
}

int b64_pton(char const *src, unsigned char *target, size_t targsize)
{
   int tarindex, state, ch;
   char *pos;

   state = 0;
   tarindex = 0;

   while ((ch = *src++) != '\0') {
      if (isspace(ch))		/* Skip whitespace anywhere. */
	 continue;

      if (ch == Pad64)
	 break;

      pos = strchr(Base64, ch);
      if (pos == 0)		/* A non-base64 character. */
	 return (-1);

      switch (state) {
      case 0:
	 if (target) {
	    if (tarindex >= targsize)
	       return (-1);
	    target[tarindex] = (pos - Base64) << 2;
	 }
	 state = 1;
	 break;
      case 1:
	 if (target) {
	    if (tarindex + 1 >= targsize)
	       return (-1);
	    target[tarindex] |= (pos - Base64) >> 4;
	    target[tarindex + 1] = ((pos - Base64) & 0x0f)
		<< 4;
	 }
	 tarindex++;
	 state = 2;
	 break;
      case 2:
	 if (target) {
	    if (tarindex + 1 >= targsize)
	       return (-1);
	    target[tarindex] |= (pos - Base64) >> 2;
	    target[tarindex + 1] = ((pos - Base64) & 0x03)
		<< 6;
	 }
	 tarindex++;
	 state = 3;
	 break;
      case 3:
	 if (target) {
	    if (tarindex >= targsize)
	       return (-1);
	    target[tarindex] |= (pos - Base64);
	 }
	 tarindex++;
	 state = 0;
	 break;
      }
   }

   /*
    * We are done decoding Base-64 chars.  Let's see if we ended
    * on a byte boundary, and/or with erroneous trailing characters.
    */

   if (ch == Pad64) {		/* We got a pad char. */
      ch = *src++;		/* Skip it, get next. */
      switch (state) {
      case 0:			/* Invalid = in first position */
      case 1:			/* Invalid = in second position */
	 return (-1);

      case 2:			/* Valid, means one byte of info */
	 /* Skip any number of spaces. */
	 /* Skip any number of spaces. */
	 for (; ch != '\0'; ch = *src++)
	    if (!isspace(ch))
	       break;
	 /* Make sure there is another trailing = sign. */
	 if (ch != Pad64)
	    return (-1);
	 ch = *src++;		/* Skip the = */
	 /* Fall through to "single trailing =" case. */
	 /* FALLTHROUGH */

      case 3:			/* Valid, means two bytes of info */
	 /*
	  * We know this char is an =.  Is there anything but
	  * whitespace after it?
	  */
	 for (; ch != '\0'; ch = *src++)
	    if (!isspace(ch))
	       return (-1);

	 /*
	  * Now make sure for cases 2 and 3 that the "extra"
	  * bits that slopped past the last full byte were
	  * zeros.  If we don't check them, they become a
	  * subliminal channel.
	  */
	 if (target && target[tarindex] != 0)
	    return (-1);
      }
   } else {
      /*
       * We ended by seeing the end of the string.  Make sure we
       * have no partial bytes lying around.
       */
      if (state != 0)
	 return (-1);
   }

   return (tarindex);
}

int
uuencode(unsigned char *src, size_t srclength,
	 unsigned char *target, size_t targsize)
{
   return b64_ntop(src, srclength, target, targsize);
}

int uudecode(const char *src, unsigned char *target, size_t targsize)
{
   int len;
   char *encoded, *p;

   /* copy the 'readonly' source */
   encoded = strdup(src);
   /* skip whitespace and data */
   for (p = encoded; *p == ' ' || *p == '\t'; p++);
   for (; *p != '\0' && *p != ' ' && *p != '\t'; p++);
   /* and remove trailing whitespace because __b64_pton needs this */
   *p = '\0';
   len = b64_pton(encoded, target, targsize);
   free(encoded);
   return len;
}

void dump_base64(FILE * fp, unsigned char *data, unsigned int len)
{
   unsigned char *buf = malloc(2 * len);
   int i, n;

   n = uuencode(data, len, buf, 2 * len);
   for (i = 0; i < n; i++) {
      fprintf(fp, "%c", buf[i]);
      if (i % 70 == 69)
	 fprintf(fp, "\n");
   }
   if (i % 70 != 69)
      fprintf(fp, "\n");
   free(buf);
}
