/*
 * Fuzzy Fingerprinting - Attacking vulnerabilities in the Human Brain
 * Copyright 2002 Plasmoid <plasmoid@thc.org> - All rights reserved. 
 * On behalf of The Hacker's Choice - http://www.thc.org
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright   
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright   
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR `AS IS'' AND ANY EXPRESS OR     
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY   
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "includes.h"
RCSID("$Id: state.c,v 1.21 2003/10/25 14:08:14 plasmoid Exp $");

#include "hash.h"
#include "xkey.h"
#include "fmap.h"
#include "list.h"
#include "state.h"

/*
 * Init an empty state containing only the types for the hash, the key, 
 * the map, the target hash (real fingerprint) and the state file this
 * information will be later written to.
 */
int state_init(state_t ** s, int key_type, int hash_type, int map_type,
	       unsigned char *hash, char *file)
{
   if (!s) {
      err_msg("state_init: No state supplied");
      return 0;
   }

   if (!hash || !file) {
      err_msg("state_init: Argument maybe partial null");
      return 0;
   }

   *s = (state_t *) malloc(sizeof(state_t));
   if (!*s) {
      err_msg("state_init: Could not allocate state structure");
      return 0;
   }

   (*s)->xkey_type = key_type;
   (*s)->fmap_type = map_type;
   (*s)->hash_type = hash_type;
   (*s)->seconds = 0;
   (*s)->hashs = 0;

   (*s)->target_hash = (unsigned char *) malloc(hash_length(hash_type));
   if (!(*s)->target_hash) {
      err_msg("state_init: Could not allocate target hash");
      free(*s);
      return 0;
   }
   memcpy((*s)->target_hash, hash, hash_length(hash_type));

   (*s)->state_file = (char *) malloc(strlen(file) + 1);
   if (!(*s)->state_file) {
      err_msg("state_init: Could not allocate state filename");
      free((*s)->target_hash);
      free(*s);
      return 0;
   }
   memcpy((*s)->state_file, file, strlen(file) + 1);

   return 1;
}

/*
 * Set pointers to the fuzzy map, the key and the list of hashs that are
 * used during the actual crunch process. 
 */
int state_set_ptr(state_t * s, fmap_t * fmap, xkey_t * xkey, list_t * list)
{
   if (!s) {
      err_msg("state_set: No state supplied");
      return 0;
   }
   s->xkey = xkey;
   s->fmap = fmap;
   s->list = list;

   return 1;
}

/*
 * Free the states, erhm, free a state.
 */
void state_free(state_t * s)
{
   if (!s) {
      err_msg("state_free: No state supplied");
      return;
   }

   if (s->state_file)
      free(s->state_file);
   if (s->target_hash)
      free(s->target_hash);

   free(s);
}

/*
 * Read a state from the file (fd).
 */
int state_read(state_t ** s, int fd)
{
   int length;
   char magic[STATE_MAGIC_LEN];

   if (!s)
      goto err1;

   if (read(fd, &magic, STATE_MAGIC_LEN) != STATE_MAGIC_LEN)
      goto err1;

   if (memcmp(magic, STATE_MAGIC, STATE_MAGIC_LEN)) {
      err_msg("state_read: Invalid state file");
      return 0;
   }

   *s = (state_t *) malloc(sizeof(state_t));
   if (!*s)
      goto err1;

   if (read(fd, &(*s)->xkey_type, sizeof(int)) != sizeof(int))
      goto err2;
   if (read(fd, &(*s)->fmap_type, sizeof(int)) != sizeof(int))
      goto err2;
   if (read(fd, &(*s)->hash_type, sizeof(int)) != sizeof(int))
      goto err2;
   if (read(fd, &(*s)->seconds, sizeof(double)) != sizeof(double))
      goto err2;
   if (read(fd, &(*s)->hashs, sizeof(double)) != sizeof(double))
      goto err2;

   /*
    * The following code has been written while listening to Leslie 
    * Nachmann's White Room, therefore it is dedicated to her. At least
    * the following 10 lines. :)
    */
   (*s)->target_hash =
       (unsigned char *) malloc(hash_length((*s)->hash_type));
   if (!(*s)->target_hash)
      goto err2;

   if (read(fd, (*s)->target_hash, hash_length((*s)->hash_type)) !=
       hash_length((*s)->hash_type))
      goto err3;

   if (read(fd, &length, sizeof(int)) != sizeof(int))
      goto err3;

   (*s)->state_file = (char *) malloc(length);
   if (!(*s)->state_file)
      goto err3;

   if (read(fd, (*s)->state_file, length) != length)
      goto err4;

   (*s)->xkey = NULL;
   if (!xkey_read(&(*s)->xkey, fd))
      goto err4;

   (*s)->fmap = NULL;
   if (!fmap_read(&(*s)->fmap, fd))
      goto err4;

   (*s)->list = NULL;
   if (!list_read(&(*s)->list, fd))
      goto err4;

   return 1;

 err4:
   free((*s)->state_file);
 err3:
   free((*s)->target_hash);
 err2:
   free(*s);
 err1:
   err_msg("state_read: Could not read state");
   return 0;
}

/*
 * Write a state to a file (fd).
 */
int state_write(state_t * s, int fd)
{
   int length;
   char *magic = STATE_MAGIC;

   if (!s)
      goto err;

   if (write(fd, magic, STATE_MAGIC_LEN) != STATE_MAGIC_LEN)
      goto err;

   if (write(fd, &s->xkey_type, sizeof(int)) != sizeof(int))
      goto err;
   if (write(fd, &s->fmap_type, sizeof(int)) != sizeof(int))
      goto err;
   if (write(fd, &s->hash_type, sizeof(int)) != sizeof(int))
      goto err;
   if (write(fd, &s->seconds, sizeof(double)) != sizeof(double))
      goto err;
   if (write(fd, &s->hashs, sizeof(double)) != sizeof(double))
      goto err;

   if (write(fd, s->target_hash, hash_length(s->hash_type)) !=
       hash_length(s->hash_type))
      goto err;

   length = strlen(s->state_file) + 1;
   if (write(fd, &length, sizeof(int)) != sizeof(int))
      goto err;

   if (write(fd, s->state_file, length) != length)
      goto err;

   if (!xkey_write(s->xkey, fd))
      goto err;

   if (!fmap_write(s->fmap, fd))
      goto err;

   if (!list_write(s->list, fd))
      goto err;

   return 1;

 err:
   err_msg("state_write: Could not write state");
   return 0;
}

/*
 * Update the number of generated hashs and the time FFP is running.
 */
void state_update(state_t * s, unsigned long h, time_t secs)
{
   int fd;

   s->hashs += h;
   s->seconds += secs;

   fd = open(s->state_file, O_CREAT | O_WRONLY | O_TRUNC, 0600);
   if (fd == -1) {
      err_msg("state_update: Could not write state file");
      return;
   }

   state_write(s, fd);
   close(fd);
}

/*
 * Print the state.
 */


void state_print(state_t * s, FILE * out)
{
   char *str;
   hash_t *h;
   int i;

   str = time2str(s->seconds);

   fprint_header(out, "Current State");
   fprintf(out,
	   " Running: %s | Total: %10.0fk hashs | Speed: %8.0f hashs/s \n",
	   str, s->hashs / 1000, s->hashs / s->seconds);
   fprint_footer(out);

   fprintf(out, " Best Fuzzy Fingerprint from State File %s\n",
	   s->state_file);
   h = list_hash_at(s->list, 0);
   hash_print(h, out);

   fprintf(out, "    Target Digest: ");
   for (i = 0; i < hash_length(s->hash_type); i++) {
      fprintf(out, "%.2x", s->target_hash[i]);
      if (i < hash_length(s->hash_type) - 1)
	 fprintf(out, ":");
   }
   fprintf(out, "\n");
   fprintf(out, "    Fuzzy Quality: %f%%\n\n",
	   fmap_normalize(s->fmap, h->quality) * 100);

   free(str);
}
