/*
 * Fuzzy Fingerprinting - Attacking vulnerabilities in the Human Brain
 * Copyright 2002 Plasmoid <plasmoid@thc.org> - All rights reserved. 
 * On behalf of The Hacker's Choice - http://www.thc.org
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright   
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright   
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR `AS IS'' AND ANY EXPRESS OR     
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY   
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * ssh_key_to_blob(): 
 * Copyright (c) 2000, 2001 Markus Friedl.  All rights reserved.
 * I am using the same LICENSE as Markus, so just look above to learn how
 * to use his code. 
 */

#include "includes.h"
RCSID("$Id: ssh.c,v 1.5 2003/10/25 14:08:14 plasmoid Exp $");

#include <openssl/rsa.h>
#include <openssl/dsa.h>
#include <openssl/bn.h>
#include <openssl/err.h>
#include <openssl/rand.h>
#include <openssl/pem.h>
#include <openssl/md5.h>
#include <openssl/sha.h>

#include "buffer.h"
#include "uuencode.h"
#include "xkey.h"
#include "rsa.h"
#include "ssh.h"

/*
 * Return ssh name of the key algorithm used in xkey.  
 */
char *ssh_key_name(xkey_t * k)
{
   switch (k->type) {
   case XKEY_RSA:
      return "ssh-rsa";
      break;
   case XKEY_DSA:
      return "ssh-dss";
      break;
   }
   return "ssh-unknown";
}

/*
 * Convert a given key to a blob used for later generating a message 
 * digest from the blob. This function is awesome slow due to the 
 * buffer implementation, someday I'll have to tweak this or someone 
 * else has to.
 */
int ssh_key_to_blob(xkey_t * key, unsigned char **blobp, unsigned int *lenp)
{
   Buffer b;
   RSA *rsa;
   DSA *dsa;
   int len;
   unsigned char *buf;

   if (key == NULL) {
      err_msg("ssh_key_to_blob: key == NULL");
      return 0;
   }
   buffer_init(&b);
   switch (key->type) {
   case XKEY_DSA:
      dsa = key->alg;
      buffer_put_cstring(&b, ssh_key_name(key));
      buffer_put_bignum2(&b, dsa->p);
      buffer_put_bignum2(&b, dsa->q);
      buffer_put_bignum2(&b, dsa->g);
      buffer_put_bignum2(&b, dsa->pub_key);
      break;
   case XKEY_RSA:
      rsa = key->alg;
      buffer_put_cstring(&b, ssh_key_name(key));
      buffer_put_bignum2(&b, rsa->e);
      buffer_put_bignum2(&b, rsa->n);
      break;
   default:
      err_msg("ssh_key_to_blob: unsupported key type");
      buffer_free(&b);
      return 0;
   }
   len = buffer_len(&b);
   buf = malloc(len);
   memcpy(buf, buffer_ptr(&b), len);
   memset(buffer_ptr(&b), 0, len);
   buffer_free(&b);
   if (lenp != NULL)
      *lenp = len;
   if (blobp != NULL)
      *blobp = buf;
   return len;
}

/*
 * Save the private key in SSH format (ssh2 only).
 */
int ssh_save_private_key(xkey_t * key, const char *filename,
			 const char *_passphrase, const char *comment)
{
   FILE *fp;
   RSA *rsa;
   DSA *dsa;
   int fd;
   int success = 0;
   int len;
   unsigned char *passphrase;
   const EVP_CIPHER *cipher;

   if (_passphrase)
      len = strlen(_passphrase);
   else
      len = 0;

   passphrase = (len > 0) ? (unsigned char *) _passphrase : NULL;
   cipher = (len > 0) ? EVP_des_ede3_cbc() : NULL;

   if (len > 0 && len <= 4) {
      err_msg("ssh_save_private_key: Passphrase too short");
      return 0;
   }

   fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, 0600);
   if (fd < 0) {
      err_msg("ssh_save_private_key: Could not create private ssh key");
      return 0;
   }

   fp = fdopen(fd, "w");
   if (fp == NULL) {
      close(fd);
      err_msg("ssh_save_private_key: Could not open private ssh key");
      return 0;
   }


   switch (key->type) {
   case XKEY_DSA:
      dsa = key->alg;
      success = PEM_write_DSAPrivateKey(fp, dsa,
					cipher, passphrase, len, NULL, NULL);
      break;
   case XKEY_RSA:
      rsa = key->alg;
      success = PEM_write_RSAPrivateKey(fp, rsa,
					cipher, passphrase, len, NULL, NULL);
      break;
   }
   fclose(fp);
   return success;
}

/*
 * Save the public key in SSH format (ssh2 only)
 */
int ssh_save_public_key(xkey_t * key, const char *filename)
{
   int n, success = 0, fd;
   unsigned int len;
   unsigned char *blob, *uu;
   FILE *f;

   fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, 0600);
   if (fd < 0) {
      err_msg("ssh_save_public_key: Could not create public ssh key");
      return 0;
   }

   f = fdopen(fd, "w");
   if (f == NULL) {
      close(fd);
      err_msg("ssh_save_public_key: Could not open public ssh key");
      return 0;
   }

   if (key->alg != NULL) {
      ssh_key_to_blob(key, &blob, &len);
      uu = malloc(2 * len);
      n = uuencode(blob, len, uu, 2 * len);
      if (n > 0) {
	 fprintf(f, "%s %s\n", ssh_key_name(key), uu);
	 success = 1;
      }
      free(blob);
      free(uu);
   }
   fclose(f);
   return success;
}

/*
 * Get the key type from the given ssh key name.
 */
int ssh_type_from_name(char *name)
{
   if (strcmp(name, "rsa") == 0) {
      return XKEY_RSA;
   } else if (strcmp(name, "dsa") == 0) {
      return XKEY_DSA;
   } else if (strcmp(name, "ssh-rsa") == 0) {
      return XKEY_RSA;
   } else if (strcmp(name, "ssh-dss") == 0) {
      return XKEY_DSA;
   }

   return -1;
}
