/*
 * Fuzzy Fingerprinting - Attacking vulnerabilities in the Human Brain
 * Copyright 2002 Plasmoid <plasmoid@thc.org> - All rights reserved. 
 * On behalf of The Hacker's Choice - http://www.thc.org
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright   
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright   
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR `AS IS'' AND ANY EXPRESS OR     
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY   
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * RSA_complete_key():
 * Copyright (C) 1995-1998 Eric Young (eay@cryptsoft.com)
 * All rights reserved.
 *
 * Copyright remains Eric Young's, and as such any Copyright notices in the
 * code are not to be removed.  If this package is used in a product, Eric
 * Young should be given attribution as the author of the parts of the
 * library used.  This can be in the form of a textual message at program
 * startup or in documentation (online or textual) provided with the
 * package.
 *
 * [...]
 * 
 * If this isn't enough legal blabla, let me know.
 */

#include "includes.h"
RCSID("$Id: rsa.c,v 1.6 2003/10/25 14:08:14 plasmoid Exp $");

#include <openssl/rsa.h>
#include <openssl/bn.h>
#include <openssl/err.h>

#include "xkey.h"
#include "rsa.h"

/*
 * Init an RSA key. Completely calculate a real RSA key using the 
 * RSA_generate_key() function and also calculate phi(n) that is 
 * (p-1)(q-1) and store it in the auxiliary big number alpha. 
 */
int RSA_init_key(RSA ** rsa, int bits, BIGNUM ** alpha, BIGNUM ** beta)
{
   BIGNUM *r1, *r2;
   BN_CTX *ctx;

   *rsa = RSA_generate_key(bits, RSA_F4, NULL, NULL);
   if (!*rsa) {
      err_msg("RSA_init_key: Could not generate key");
      return 0;
   }

   ctx = BN_CTX_new();
   if (!ctx)
      goto err1;

   BN_CTX_start(ctx);
   r1 = BN_CTX_get(ctx);
   r2 = BN_CTX_get(ctx);
   *alpha = BN_new();
   if (!r1 || !r2 || !*alpha)
      goto err2;

   if (!BN_sub(r1, (*rsa)->p, BN_value_one()))
      goto err2;
   if (!BN_sub(r2, (*rsa)->q, BN_value_one()))
      goto err2;
   if (!BN_mul(*alpha, r1, r2, ctx))
      goto err2;

   BN_CTX_end(ctx);
   BN_CTX_free(ctx);

   return 1;

 err2:
   BN_CTX_end(ctx);
   BN_CTX_free(ctx);
 err1:
   err_msg("RSA_init_key: Could not init key");

   return 0;
}

/*
 * Increase e and look for the next key, depending on the flags
 * the next key will just be the increased key (sloppy mode) or the
 * function will search for a key with (p-1)(q-1) relative prime 
 * to e. 
 *
 * Note: It looks like a lot of keys are relative prime to (p-1)(q-1),
 *       but the check takes "ages", therefore it might be a good
 *       idea to be sloppy, save some more keys and later use those
 *       that are relative prime.
 */
int RSA_next_key(RSA * rsa, int flags, BIGNUM * alpha, BIGNUM * beta)
{
   BIGNUM *tmp = NULL;
   BN_CTX *ctx = NULL;

   ctx = BN_CTX_new();
   if (!ctx)
      goto err1;

   BN_CTX_start(ctx);
   tmp = BN_CTX_get(ctx);
   if (!tmp)
      goto err2;

   if (flags & XKEY_RSA_SLOPPY) {
      /*
       * Just add 2. Don't add 1.
       */
      if (!BN_add_word(rsa->e, 2L))
	 goto err2;
   } else {
      for (;;) {
	 /* 
	  * Add 2 as long e is not relative prime to phi(n) = (p-1)(q-1)
	  * that is stored in the auxiliary big number alpha.
	  */
	 if (!BN_add_word(rsa->e, 2L))
	    goto err2;
	 if (!BN_gcd(tmp, alpha, rsa->e, ctx))
	    goto err2;
	 if (BN_is_one(tmp))
	    break;
      }
   }

   BN_CTX_end(ctx);
   BN_CTX_free(ctx);
   return 1;

 err2:
   BN_CTX_end(ctx);
   BN_CTX_free(ctx);

 err1:
   err_msg("RSA_next_key: Generating next e failed.");

   return 0;
}

/*
 * Complete the key by generating the missing number d, ... This code has
 * been taken from libcrypto/rsa.c and is actually copyright by Eric Young.
 */
int RSA_complete_key(RSA * rsa)
{
   BIGNUM *r0 = NULL, *r1 = NULL, *r2 = NULL;
   int ok = -1;
   BN_CTX *ctx = NULL, *ctx2 = NULL;

   ctx = BN_CTX_new();
   if (ctx == NULL)
      goto err;

   ctx2 = BN_CTX_new();
   if (ctx2 == NULL)
      goto err;

   BN_CTX_start(ctx);
   r0 = BN_CTX_get(ctx);
   r1 = BN_CTX_get(ctx);
   r2 = BN_CTX_get(ctx);
   if (r2 == NULL)
      goto err;

   if (rsa == NULL)
      goto err;

   if (!BN_sub(r1, rsa->p, BN_value_one()))
      goto err;			/* p-1 */
   if (!BN_sub(r2, rsa->q, BN_value_one()))
      goto err;			/* q-1 */
   if (!BN_mul(r0, r1, r2, ctx))
      goto err;			/* (p-1)(q-1) */

   rsa->d = BN_mod_inverse(rsa->d, rsa->e, r0, ctx2);
   if (rsa->d == NULL)
      goto err;

   /* calculate d mod (p-1) */
   if (!BN_mod(rsa->dmp1, rsa->d, r1, ctx))
      goto err;
   if (!BN_mod(rsa->dmq1, rsa->d, r2, ctx))
      goto err;

   rsa->iqmp = BN_mod_inverse(rsa->iqmp, rsa->q, rsa->p, ctx2);
   if (rsa->iqmp == NULL)
      goto err;

   ok = 1;

 err:

   if (ok == -1)
      err_msg("RSA_complete_key: RSA key completion failed");

   BN_CTX_end(ctx);
   BN_CTX_free(ctx);
   BN_CTX_free(ctx2);

   if (ok == -1) {
      return 0;
   } else
      return 1;
}

/*
 * Free the RSA key and its auxiliary big numbers
 */
void RSA_free_key(RSA * rsa, BIGNUM * alpha, BIGNUM * beta)
{
   if (rsa)
      RSA_free(rsa);
   if (alpha)
      BN_free(alpha);
}

/*
 * Print the RSA key.
 */
void RSA_print_key(RSA * rsa, FILE * out, BIGNUM * alpha, BIGNUM * beta)
{
   BIGNUM *tmp = NULL;
   BN_CTX *ctx = NULL;

   fprintf(out,
	   "               Key Algorithm: RSA (Rivest Shamir Adleman)\n");
   fprintf(out, "        Key Bits / Size of n: %d Bits\n",
	   BN_num_bits(rsa->n));
   fprintf(out, "                Public key e: 0x");
   BN_print_fp(out, rsa->e);
   fprintf(out, "\n");
   fprintf(out, " Public Key Bits / Size of e: %d Bits\n",
	   BN_num_bits(rsa->e));

   fprintf(out, "        Phi(n) and e r.prime: ");

   ctx = BN_CTX_new();
   BN_CTX_start(ctx);
   tmp = BN_CTX_get(ctx);
   BN_gcd(tmp, alpha, rsa->e, ctx);

   if (BN_is_one(tmp))
      fprintf(out, "Yes");
   else
      fprintf(out, "No");

   BN_CTX_end(ctx);
   BN_CTX_free(ctx);

   fprintf(out, "\n");
}

/*
 * Read a RSA key from file (fd) and also read the auxiliary big number
 * alpha containing phi(n).
 */
int RSA_read_key(RSA ** rsa, int fd, BIGNUM ** alpha, BIGNUM ** beta)
{
   *rsa = RSA_new();

   if (!BN_read(fd, &(*rsa)->p))
      goto err;
   if (!BN_read(fd, &(*rsa)->q))
      goto err;
   if (!BN_read(fd, &(*rsa)->n))
      goto err;
   if (!BN_read(fd, &(*rsa)->d))
      goto err;
   if (!BN_read(fd, &(*rsa)->e))
      goto err;
   if (!BN_read(fd, &(*rsa)->dmp1))
      goto err;
   if (!BN_read(fd, &(*rsa)->dmq1))
      goto err;
   if (!BN_read(fd, &(*rsa)->iqmp))
      goto err;

   *alpha = BN_new();
   if (!BN_read(fd, alpha))
      goto err;

   return 1;
 err:
   err_msg("RSA_read: Could not read RSA key");
   return 0;
}


/*
 * Write an RSA key to file (fd).
 */
int RSA_write_key(RSA * rsa, int fd, BIGNUM * alpha, BIGNUM * beta)
{
   if (!BN_write(fd, rsa->p))
      goto err;
   if (!BN_write(fd, rsa->q))
      goto err;
   if (!BN_write(fd, rsa->n))
      goto err;
   if (!BN_write(fd, rsa->d))
      goto err;
   if (!BN_write(fd, rsa->e))
      goto err;
   if (!BN_write(fd, rsa->dmp1))
      goto err;
   if (!BN_write(fd, rsa->dmq1))
      goto err;
   if (!BN_write(fd, rsa->iqmp))
      goto err;
   if (!BN_write(fd, alpha))
      goto err;

   return 1;
 err:
   err_msg("RSA_write: Could not write RSA key");
   return 0;
}

/*
 * Clone an RSA key.
 */
int RSA_clone_key(RSA ** rsa1, BIGNUM ** alpha1, BIGNUM ** beta1,
		  RSA * rsa2, BIGNUM * alpha2, BIGNUM * beta2)
{
   if (!(*alpha1 = BN_dup(alpha2)))
      goto err;

   *rsa1 = RSA_new();

   if (!((*rsa1)->p = BN_dup(rsa2->p)))
      goto err;
   if (!((*rsa1)->q = BN_dup(rsa2->q)))
      goto err;
   if (!((*rsa1)->n = BN_dup(rsa2->n)))
      goto err;
   if (!((*rsa1)->e = BN_dup(rsa2->e)))
      goto err;
   if (!((*rsa1)->d = BN_dup(rsa2->d)))
      goto err;
   if (!((*rsa1)->dmp1 = BN_dup(rsa2->dmp1)))
      goto err;
   if (!((*rsa1)->dmq1 = BN_dup(rsa2->dmq1)))
      goto err;
   if (!((*rsa1)->iqmp = BN_dup(rsa2->iqmp)))
      goto err;

   return 1;

 err:
   err_msg("RSA_clone_key: Could not clone key");
   return 0;
}
