/*
 * Fuzzy Fingerprinting - Attacking vulnerabilities in the Human Brain
 * Copyright 2002 Plasmoid <plasmoid@thc.org> - All rights reserved. 
 * On behalf of The Hacker's Choice - http://www.thc.org
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright   
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright   
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR `AS IS'' AND ANY EXPRESS OR     
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY   
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "includes.h"
RCSID("$Id: misc.c,v 1.8 2003/10/25 14:08:14 plasmoid Exp $");

#include <openssl/bn.h>
#include <openssl/err.h>

/*
 * Just a simple error function that equals printf but also 
 * looks for errno and OpenSSL erros and prints them.
 */
void err_msg(const char *fmt, ...)
{
   char s[128];
   va_list ap;

   va_start(ap, fmt);
   vsnprintf(s, 128, fmt, ap);
   va_end(ap);

   fprintf(stderr, "Error: %s\n", s);

   if (ERR_get_error()) {
      ERR_print_errors_fp(stderr);
      ERR_clear_error();
   } else if (errno) {
      perror("");
      errno = 0;
   }
}

/*
 * Actually this could have been a macro, but it was born into the world
 * as a C function. Prints to out and flushes out. 
 */
void ffprintf(FILE * out, const char *fmt, ...)
{
   va_list ap;

   va_start(ap, fmt);
   fprintf(out, fmt, ap);
   va_end(ap);

   fflush(out);
}


/*
 * Allocate a string and convert the given time (c) to this 
 * string. 
 */
char *time2str(time_t c)
{
   char *s;
   unsigned long d, h, m;

   d = c / (24 * 60 * 60);
   c -= d * 24 * 60 * 60;
   h = c / (60 * 60);
   c -= h * 60 * 60;
   m = c / 60;
   c -= m * 60;

   s = (char *) malloc(80);
   snprintf(s, 80, "%3lud %.2luh %.2lum %.2lus", d, h, m, c);

   return s;
}

/*
 * Print a good looking ASCII header (Next time we are going to do this
 * part using KDE3 and the Liquid theme, ok?)
 */
void fprint_header(FILE * out, char *s)
{
   int len, i;

   len = 80 - 5 - strlen(s);
   fprintf(out, "---[%s]", s);
   for (i = 0; i < len; i++)
      fprintf(out, "-");
   fprintf(out, "\n");

}

/*
 * Print a footer similar to the header. Zzz.
 */
void fprint_footer(FILE * out)
{
   int i;
   for (i = 0; i < 80; i++)
      fprintf(out, "-");
   fprintf(out, "\n");
}

/*
 * Read a big number (libcrypto) from file (fd).
 */
int BN_read(int fd, BIGNUM ** bn)
{
   int len;
   unsigned char *buf;

   if (!bn) {
      err_msg("BN_read: no bn pointer supplied");
      return 0;
   }

   if (read(fd, &len, sizeof(int)) != sizeof(int)) {
      err_msg("BN_read: reading len failed");
      return 0;
   }

   buf = (unsigned char *) malloc(len);
   if (!buf) {
      err_msg("BN_read: could not allocate buffer");
      return 0;
   }

   if (read(fd, buf, len) != len) {
      err_msg("BN_read: reading bn failed");
      free(buf);
      return 0;
   }

   *bn = BN_bin2bn(buf, len, *bn);
   if (!*bn) {
      err_msg("BN_read: bn creation failed");
      free(buf);
      return 0;
   }

   free(buf);
   return 1;
}

/*
 * Write a big number (libcrypto) to file (fd).
 */
int BN_write(int fd, BIGNUM * bn)
{
   int len;
   unsigned char *buf;

   if (!bn) {
      err_msg("BN_write: no bn supplied");
      return 0;
   }

   len = BN_num_bytes(bn);
   if (write(fd, &len, sizeof(int)) != sizeof(int)) {
      err_msg("BN_write: writing len failed");
      return 0;
   }

   buf = (unsigned char *) malloc(len);
   if (!buf) {
      err_msg("BN_write: could not allocate buffer");
      return 0;
   }

   if (BN_bn2bin(bn, buf) != len) {
      err_msg("BN_write: converting bn failed");
      free(buf);
      return 0;
   }

   if (write(fd, buf, len) != len) {
      err_msg("BN_write: writing bn failed");
      free(buf);
      return 0;
   }

   free(buf);
   return 1;
}
