/*
 * Fuzzy Fingerprinting - Attacking vulnerabilities in the Human Brain
 * Copyright 2002 Plasmoid <plasmoid@thc.org> - All rights reserved. 
 * On behalf of The Hacker's Choice - http://www.thc.org
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright   
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright   
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR `AS IS'' AND ANY EXPRESS OR     
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY   
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "includes.h"
RCSID("$Id: main.c,v 1.23 2003/10/25 14:08:14 plasmoid Exp $");

#include "hash.h"
#include "xkey.h"
#include "fmap.h"
#include "list.h"
#include "ssh.h"
#include "state.h"

extern char *optarg;
extern int optind, opterr;

/*
 * This is really an annoying bunch of static variables. Is this bad 
 * coding style? Probably! And I am in my final year, damn ;)
 */
static int period = 60;
static int xkey_type = XKEY_RSA;
static int xkey_bits = 1024;
static int xkey_flags = XKEY_RSA_SLOPPY;
static int hash_type = HASH_MD5;
static int fmap_type = FMAP_INV_GAUSS;
static int list_size = 10;
static float fmap_sigma = 7.3;
static float fmap_my = 0.14;
static char *state_file = "/var/tmp/ffp.state";
static char *sshkey_dir = "/tmp";
static int ssh_extract = 0;
static unsigned char *target_hash = NULL;

/*
 * The global structs used during crunching, a key, a hash and the 
 * fuzzy map. Best hashs are stored in the list.
 */
static xkey_t *xkey;
static hash_t *hash;
static state_t *state;
static fmap_t *fmap;
static list_t *list;
static FILE *out;

/*
 * Guess what this does...
 */
void print_usage()
{
   fprintf(stdout,
	   "Usage: ffp [Options]\nOptions:\n"
	   "  -f type       Specify type of fingerprint to use [Default: md5]\n"
	   "                Available: md5, sha1, ripemd\n"
	   "  -t hash       Target fingerprint in byte blocks. \n"
	   "                Colon-separated: 01:23:45:67... or as string 01234567...\n"
	   "  -k type       Specify type of key to calculate [Default: rsa]\n"
	   "                Available: rsa, dsa\n"
	   "  -b bits       Number of bits in the keys to calculate [Default: 1024]\n"
	   "  -K mode       Specify key calulation mode [Default: sloppy]\n"
	   "                Available: sloppy, accurate\n"
	   "  -m type       Specify type of fuzzy map to use [Default: gauss]\n"
	   "                Available: gauss, cosine\n"
	   "  -v variation  Variation to use for fuzzy map generation [Default: 7.3]\n"
	   "  -y mean       Mean value to use for fuzzy map generation [Default: 0.14]\n"
	   "  -l size       Size of list that contains best fingerprints [Default: 10]\n"
	   "  -s filename   Filename of the state file [Default: /var/tmp/ffp.state]\n"
	   "  -e            Extract SSH host key pairs from state file\n"
	   "  -d directory  Directory to store generated ssh keys to [Default: /tmp]\n"
	   "  -p period     Period to save state file and display state [Default: 60]\n"
	   "  -V            Display version information\n");

}

/* 
 * Wer hat's erfunden? Die Schweizer? Nein. Me!
 */
void print_version()
{
   fprintf(stdout,
	   "Fuzzy Fingerprint v%s - Copyright 2002 Plasmoid <plasmoid@thc.org>\n",
	   VERSION);
}


/*
 * Do the getopt stuff and have phun.
 */
void parse_options(int argc, char **argv)
{
   int c, i;
   char *hash = NULL, *mode = NULL, *fname = NULL,
       *kname = NULL, *mname = NULL;
   char byte[5];

   /*
    * Parse commandline options. 
    */
   while ((c = getopt(argc, argv, "f:t:k:b:K:m:v:y:l:s:ed:p:hV")) != EOF)
      switch (c) {
      case 'f':
	 hash_type = hash_type_from_name(optarg);
	 fname = optarg;
	 break;
      case 't':
	 hash = optarg;
	 break;
      case 'k':
	 xkey_type = xkey_type_from_name(optarg);
	 kname = optarg;
	 break;
      case 'b':
	 xkey_bits = atoi(optarg);
	 break;
      case 'K':
	 mode = optarg;
	 break;
      case 'm':
	 fmap_type = fmap_type_from_name(optarg);
	 mname = optarg;
	 break;
      case 'v':
	 fmap_sigma = (float) atof(optarg);
	 break;
      case 'y':
	 fmap_my = (float) atof(optarg);
	 break;
      case 'l':
	 list_size = atoi(optarg);
	 break;
      case 's':
	 state_file = optarg;
	 break;
      case 'e':
	 ssh_extract = 1;
	 break;
      case 'd':
	 sshkey_dir = optarg;
	 break;
      case 'p':
	 period = atoi(optarg);
	 break;
      case 'V':
	 print_version();
	 exit(EXIT_SUCCESS);
	 break;
      case 'h':
      default:
	 print_usage();
	 exit(EXIT_SUCCESS);
      }

   /*
    * Do sanity check. Enough paramters, values inside range, etc...
    */
   if (hash_type == -1) {
      print_usage();
      fprintf(stderr, "Unknown fingerprint type %s", fname);
      exit(EXIT_FAILURE);
   }

   if (xkey_type == -1) {
      print_usage();
      fprintf(stderr, "Unknown key type %s", kname);
      exit(EXIT_FAILURE);
   }

   if (fmap_type == -1) {
      print_usage();
      fprintf(stderr, "Unknown fuzzy map type %s", mname);
      exit(EXIT_FAILURE);
   }

   if (mode) {
      if (strcmp(mode, "sloppy") == 0) {
	 xkey_flags = XKEY_RSA_SLOPPY;
      } else if (strcmp(mode, "accurate") == 0) {
	 xkey_flags = 0;
      } else {
	 print_usage();
	 fprintf(stderr, "Unknown key calculation mode %s.", mode);
	 exit(EXIT_FAILURE);
      }
   }

   if (xkey_bits < 100) {
      print_usage();
      fprintf(stderr, "Go play somewhere else, %d bits is a joke.",
	      xkey_bits);
      exit(EXIT_FAILURE);
   }

   if (fmap_sigma <= 0) {
      print_usage();
      fprintf(stderr, "Variation out of range.\n");
      exit(EXIT_FAILURE);
   }

   if (list_size <= 0) {
      print_usage();
      fprintf(stderr, "The list size needs to be a postive integer.\n");
      exit(EXIT_FAILURE);
   }

   if (period <= 0) {
      print_usage();
      fprintf(stderr, "The display period needs to be a postive integer.\n");
      exit(EXIT_FAILURE);
   }

   if (access(sshkey_dir, W_OK) != 0) {
      print_usage();
      fprintf(stderr, "The directory %s is not accessible/writable.",
	      sshkey_dir);
      exit(EXIT_FAILURE);
   }

   if (hash) {
      if (strlen(hash) != hash_length(hash_type) * 3 - 1 &&
	  strlen(hash) != hash_length(hash_type) * 2) {
	 print_usage();
	 fprintf(stderr, "The target hash doesn't contain valid blocks\n");
	 exit(EXIT_FAILURE);
      }

      /*
       * Parse supplied hash in either this 12:34:... format or the plain
       * string format 1234...
       */
      target_hash = (unsigned char *) malloc(hash_length(hash_type));

      for (i = 0; i < hash_length(hash_type); i++) {
	 byte[0] = '0';
	 byte[1] = 'x';
	 if (strlen(hash) == hash_length(hash_type) * 3 - 1) {
	    byte[2] = hash[3 * i + 0];
	    byte[3] = hash[3 * i + 1];
	 } else {
	    byte[2] = hash[2 * i + 0];
	    byte[3] = hash[2 * i + 1];
	 }
	 byte[4] = '\0';

	 target_hash[i] = strtol(byte, NULL, 16);
	 if (errno) {
	    print_usage();
	    fprintf(stderr, "The target hash contains invalid digits\n");
	    exit(EXIT_FAILURE);
	 }
      }
   }

   if (access(state_file, R_OK) != 0) {

      if (!target_hash) {
	 print_usage();
	 fprintf(stderr,
		 "No state file %s present, specify a target hash.\n",
		 state_file);
	 exit(EXIT_FAILURE);
      }

      if (ssh_extract) {
	 print_usage();
	 fprintf(stderr,
		 "No state file %s present, cannot extract SSH keys.\n",
		 state_file);
	 exit(EXIT_FAILURE);
      }

   } else {
      if (target_hash || fname || mname || kname) {
	 print_usage();
	 fprintf(stderr, "State file %s present, don't specify options.\n",
		 state_file);
	 exit(EXIT_FAILURE);
      }
   }

}

/*
 * Signal handler that catches SIGINT and stores the current state to 
 * the state file, so that no even one hash is droped.
 */
void sighandler(int sig)
{
   switch (sig) {
   case SIGINT:
      fprint_footer(out);
      fprintf(out, " Exiting and saving state file %s\n", state_file);
      state_update(state, 0, 0);

      /*
       * Free generic state
       */
      state_free(state);

      /*
       * Free bruteforce components: list, fmap, hash and key
       */
      list_free(list);
      fmap_free(fmap);
      hash_free(hash);
      xkey_free(xkey);

      exit(EXIT_SUCCESS);

      break;
   }
}

/*
 * Restore a state from a state file, load the list, the crunch key and 
 * fuzzy map. Initialize a new crunch hash, its content is temporary 
 */
void restore()
{
   int fd;
   fprint_header(out, "Restoring");

   ffprintf(out, "   Reading FFP State File: ");
   /*
    * Read in state file.
    */
   fd = open(state_file, O_RDONLY);
   if (fd == -1) {
      err_msg("Could not read state file %s", state_file);
      exit(EXIT_FAILURE);
   }

   if (!state_read(&state, fd)) {
      err_msg("State file %s invalid", state_file);
      exit(EXIT_FAILURE);
   }

   close(fd);
   ffprintf(out, "Done\n");

   ffprintf(out, "    Restoring environment: ");
   xkey_type = state->xkey_type;
   hash_type = state->hash_type;
   fmap_type = state->fmap_type;

   xkey = state->xkey;
   fmap = state->fmap;
   list = state->list;

   target_hash = state->target_hash;
   state_file = state->state_file;
   ffprintf(out, "Done\n");

   ffprintf(out, " Initializing Crunch Hash: ");
   hash_init(&hash, hash_type);
   ffprintf(out, "Done\n");
}

/*
 * Init a complete new session. 
 */
void init()
{
   /*
    * Init bruteforce components: key, hash map and list.
    */
   fprint_header(out, "Initializing");
   ffprintf(out, " Initializing Crunch Hash: ");
   hash_init(&hash, hash_type);
   ffprintf(out, "Done\n");

   ffprintf(out, "   Initializing Fuzzy Map: ");
   fmap_init(&fmap, fmap_type, fmap_sigma, fmap_my, hash_length(hash_type));
   ffprintf(out, "Done\n");

   ffprintf(out, " Initializing Private Key: ");
   xkey_init(&xkey, xkey_type, xkey_bits, xkey_flags);
   ffprintf(out, "Done\n");

   ffprintf(out, "   Initializing Hash List: ");
   list_init(&list, list_size);
   ffprintf(out, "Done\n");

   /*
    * Init and prepare ffp generic state
    */
   ffprintf(out, "   Initializing FFP State: ");
   state_init(&state, xkey_type, hash_type, fmap_type, target_hash,
	      state_file);
   state_set_ptr(state, fmap, xkey, list);
   ffprintf(out, "Done\n");
}

/*
 * Save ssh keys from the current list to files. 
 */
void save_ssh_keys()
{
   int i;
   xkey_t *k;
   char *s, *t;

   fprint_footer(out);
   ffprintf(out, " Saving SSH host key pairs: ");

   s = (char *) malloc(128);
   t = (char *) malloc(128);

   for (i = 0; i < list->size; i++) {
      k = list_xkey_at(list, i);

      /*
       * Construct filenames, e.g. ssh-rsa00 for private and ssh-rsa00.pub
       * for public key.
       */
      snprintf(s, 128, "%s/%s%.2d", sshkey_dir, ssh_key_name(k), i);
      snprintf(t, 128, "%s.pub", s);

      ssh_save_private_key(k, s, NULL, NULL);
      ssh_save_public_key(k, t);
      fprintf(out, "[%.2d] ", i);
   }

   ffprintf(out, "\n");
   free(s);
   free(t);
}

/*
 * Is this the main function? Or is it just something looking very similar,
 * e.g. a fuzzy main function.
 */
int main(int argc, char **argv)
{
   xkey_t *xkeyc;
   hash_t *hashc;

   time_t start;
   unsigned long hashs, first_houndred;
   unsigned char *blob;
   unsigned int len;

   out = stdout;
   parse_options(argc, argv);

   if (ssh_extract) {
      restore();
      save_ssh_keys();
      exit(EXIT_SUCCESS);
   }

   /*
    * If a state file exists restore the session, otherwise init 
    * a new one. 
    */
   if (access(state_file, R_OK) == 0)
      restore();
   else
      init();

   /*
    * Print data
    */
   fprintf(out, "\n");
   fprint_header(out, "Fuzzy Map");
   fmap_print(fmap, out);
   fprint_header(out, "Current Key");
   xkey_print(xkey, out);

   fprintf(out, "\n\n State File: %s\n", state_file);
   ffprintf(out, " Running...\n\n");

   signal(SIGINT, sighandler);
   start = time(NULL);
   hashs = 0;
   first_houndred = 0;
   while (hash->quality < fmap->sum) {
      /*
       * Calculate next key, generate blob, hash blob and compute
       * quality of the new message digest.
       */
      xkey_next(xkey);
      ssh_key_to_blob(xkey, &blob, &len);
      hash_calc(hash, blob, len);
      hash_quality(hash, fmap, target_hash);

      if (hash->quality > list_min_quality(list)) {
	 /*
	  * If quality is above the minimum of the list clone key and 
	  * hash and put them into the list. 
	  */
	 xkey_clone(&xkeyc, xkey);
	 xkey_complete(xkeyc);
	 hash_clone(&hashc, hash);
	 list_insert(list, hashc, xkeyc);

	 if (state->seconds > 60 * 60) {
	    fprint_header(out, "Added new fuzzy fingerprint");
	    hash_print(hash, out);
	    fprintf(out, "\n");
	 }
      }

      free(blob);
      hash_free_data(hash);

      hashs++;
      first_houndred++;

      if (start + period < time(NULL) || first_houndred == 100) {
	 /*
	  * From time to time give a sign of life and store the 
	  * state to the state file, just in case someone is playing
	  * with the power plant.
	  */
	 if (start + period < time(NULL)) {
	    state_update(state, hashs, period);
	    start = time(NULL);
	    hashs = 0;
	 }
	 state_print(state, out);
      }
   }


   /*
    * This is only reached if a _true_ hash collision is found. 
    * Send me a postcard if you're here, you are a lucky bastard. Seufz.
    * Free generic state
    */
   state_free(state);

   /*
    * Free bruteforce components: list, fmap, hash and key
    */
   list_free(list);
   fmap_free(fmap);
   hash_free(hash);
   xkey_free(xkey);

   if (target_hash)
      free(target_hash);

   return EXIT_SUCCESS;
}
