/*
 * Fuzzy Fingerprinting - Attacking vulnerabilities in the Human Brain
 * Copyright 2002 Plasmoid <plasmoid@thc.org> - All rights reserved. 
 * On behalf of The Hacker's Choice - http://www.thc.org
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright   
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright   
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR `AS IS'' AND ANY EXPRESS OR     
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY   
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "includes.h"
RCSID("$Id: list.c,v 1.15 2003/10/25 14:08:14 plasmoid Exp $");

#include "xkey.h"
#include "hash.h"
#include "list.h"


/*
 * Init a doubly-linked cyclic list that can carry up to (elem) elements.
 * Note: the current size and the max. size are stored in the head node. 
 */
int list_init(list_t ** li, int elem)
{

   *li = (list_t *) malloc(sizeof(list_t));

   if (!*li) {
      err_msg("list_init: Could not allocate list");
      return 0;
   }

   (*li)->quality = 0;
   (*li)->next = *li;
   (*li)->prev = *li;

   (*li)->max = elem;
   (*li)->size = 0;

   return 1;
}

/*
 * Return minimum quality contained in the list. Due to the cyclic nature of
 * the list the minimum is just the predecessor of the head.
 */
unsigned long list_min_quality(list_t * l)
{
   return l->prev->quality;
}

/*
 * Return maximum quality cotained in the list. Due to the cyclic nature of
 * the list the maximum is just the successor of the head.
 */
unsigned long list_max_quality(list_t * l)
{
   return l->next->quality;
}

/*
 * Return the hash with max. quality from the given list.
 */
hash_t *list_max_hash(list_t * l)
{
   return l->next->hash;
}

/*
 * Return the hash at the position (n) from the given list.
 */
hash_t *list_hash_at(list_t * li, int n)
{
   int i;
   list_t *l = li->next;

   for (i = 0; i < n && l != li; i++)
      l = l->next;

   return l->hash;
}

/*
 * Return the key at the position (n) from the given list.
 */
xkey_t *list_xkey_at(list_t * li, int n)
{
   int i;
   list_t *l = li->next;

   for (i = 0; i < n && l != li; i++)
      l = l->next;

   return l->key;
}


/*
 * Insert a hash and the corresponding key into the list. The hash should
 * have a fuzzy quality that is between max and min of the list, use
 * list_min_quality() to check against the minimum before insertion. The
 * hash is inserted so that the list keeps sorted.
 */
int list_insert(list_t * li, hash_t * hash, xkey_t * key)
{
   list_t *n, *l;


   if (!hash || !key) {
      err_msg("list_insert: Arguments partial null");
      return 0;
   }

   if (hash->quality <= list_min_quality(li)) {
      err_msg("list_insert: Use list_min_quality() before inserting");
      return 0;
   }

   /*
    * Starting from the head, find the node that has a smaller quality
    */
   l = li->next;
   while (l != li && l->quality > hash->quality)
      l = l->next;

   /*
    * Allocate new node
    */
   n = (list_t *) malloc(sizeof(list_t));
   n->quality = hash->quality;

   n->hash = hash;
   n->key = key;

   /*
    * Do the common doubly linked list pointer fun.
    */
   n->next = l;
   n->prev = l->prev;
   n->next->prev = n;
   n->prev->next = n;

   /*
    * If the list has not grown to its maximum, increase the size otherwise
    * remove the last node and free its content.
    */
   if (li->size < li->max) {
      li->size++;
   } else {
      /*
       * Free node
       */
      l = li->prev;
      hash_free(l->hash);
      xkey_free(l->key);

      l = l->prev;
      free(l->next);

      /*
       * Remove node
       */
      l->next = li;
      li->prev = l;
   }

   return 1;
}

/*
 * Free the given list and all its nodes and nodes' content
 */
void list_free(list_t * li)
{
   list_t *ln = li->next;

   while (ln != li) {
      hash_free(ln->hash);
      xkey_free(ln->key);

      ln = ln->next;
      free(ln->prev);
   }
   free(li);
}

/*
 * Write the list to file (fd)
 */
int list_write(list_t * li, int fd)
{
   list_t *ln = li->next;

   if (write(fd, &li->max, sizeof(int)) != sizeof(int))
      goto err;

   if (write(fd, &li->size, sizeof(int)) != sizeof(int))
      goto err;

   while (ln != li) {
      if (!hash_write(ln->hash, fd))
	 goto err;
      if (!xkey_write(ln->key, fd))
	 goto err;

      ln = ln->next;
   }

   return 1;

 err:
   err_msg("list_write: Could not write list");
   return 0;
}

/*
 * Read the list from (fd) and allocate all necessary structures
 */
int list_read(list_t ** li, int fd)
{
   list_t *n, *l;
   int i;

   *li = (list_t *) malloc(sizeof(list_t));

   if (!*li) {
      err_msg("list_read: Could not allocate list");
      return 0;
   }

   (*li)->quality = 0;
   (*li)->next = *li;
   (*li)->prev = *li;

   l = *li;

   if (read(fd, &(*li)->max, sizeof(int)) != sizeof(int))
      goto err;
   if (read(fd, &(*li)->size, sizeof(int)) != sizeof(int))
      goto err;

   for (i = 0; i < (*li)->size; i++) {
      /*
       * Allocate new list node
       */
      n = (list_t *) malloc(sizeof(list_t));
      if (!hash_read(&n->hash, fd))
	 goto err;
      if (!xkey_read(&n->key, fd))
	 goto err;

      n->quality = n->hash->quality;

      /*
       * Insert note into list
       */
      n->prev = l;
      n->next = l->next;
      n->next->prev = n;
      n->prev->next = n;

      l = l->next;
   }

   return 1;

 err:
   err_msg("list_read: Could not read list, memory probably also corrupted");
   return 0;
}

/*
 * Print list. This is rather a debug function.
 */
void list_print(list_t * l, FILE * out)
{
   list_t *n;

   if (!l)
      return;

   fprintf(out, " List size: %d (Max. %d)\n", l->size, l->max);

   fprintf(out, " (Q:%lu)", l->quality);

   n = l->next;
   while (l != n) {
      fprintf(out, " -> (Q:%lu) ", n->quality);
      n = n->next;
   }
   fprintf(out, "\n");

}
