/*
 * Fuzzy Fingerprinting - Attacking vulnerabilities in the Human Brain
 * Copyright 2002 Plasmoid <plasmoid@thc.org> - All rights reserved. 
 * On behalf of The Hacker's Choice - http://www.thc.org
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright   
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright   
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR `AS IS'' AND ANY EXPRESS OR     
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY   
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "includes.h"
RCSID("$Id: hash.c,v 1.8 2003/10/25 14:08:14 plasmoid Exp $");

#include <openssl/md5.h>
#include <openssl/sha.h>
#include <openssl/ripemd.h>

#include "fmap.h"
#include "hash.h"
#include "md5.h"
#include "sha1.h"
#include "ripemd.h"

/*
 * Set the hashs operations: calc_hash() and print_hash() depending on 
 * the provided hash type.
 */
void hash_set_ops(hash_t * h, int type)
{
   switch (type) {
   case HASH_MD5:
      h->calc_hash = (int (*)()) MD5_calc_hash;
      h->print_hash = (void (*)()) MD5_print_hash;
      break;
   case HASH_SHA1:
      h->calc_hash = (int (*)()) SHA1_calc_hash;
      h->print_hash = (void (*)()) SHA1_print_hash;
      break;
   case HASH_RIPEMD:
      h->calc_hash = (int (*)()) RIPEMD_calc_hash;
      h->print_hash = (void (*)()) RIPEMD_print_hash;
      break;
   default:
      err_msg("hash_set_ops: Unknown message digest type");
   }
}

/*
 * Return the length of the message digest generated by the hash algorithm
 * of type.
 */
int hash_length(int type)
{
   switch (type) {
   case HASH_MD5:
      return MD5_DIGEST_LENGTH;
   case HASH_SHA1:
      return SHA_DIGEST_LENGTH;
   case HASH_RIPEMD:
      return RIPEMD160_DIGEST_LENGTH;
   default:
      err_msg("hash_size: Unknown message digest type");
      return -1;
   }
}

/*
 * Init a hash object. The object can be used to stored a hash with its
 * message digest and data block but may also function as a "crunch" hash
 * where the data block and message digest change during bruteforcing.
 */
int hash_init(hash_t ** h, int type)
{
   *h = (hash_t *) malloc(sizeof(hash_t));
   if (!*h) {
      err_msg("hash_init: Could not allocate hash_t struct");
      return 0;
   }

   (*h)->quality = 0;
   (*h)->type = type;
   (*h)->hash_len = hash_length(type);


   (*h)->hash = (unsigned char *) malloc((*h)->hash_len);
   if (!(*h)->hash) {
      err_msg("hash_init: Could not allocate message digest");
      return 0;
   }

   hash_set_ops(*h, (*h)->type);

   return 1;
}

/*
 * Free a hash object. The data block may not be used depending on the use
 * of the object, see above. 
 */
void hash_free(hash_t * h)
{
   if (!h)
      return;

   if (h->hash)
      free(h->hash);
   if (h->data)
      free(h->data);
   if (h)
      free(h);
}


/*
 * Clone a hash object
 */
int hash_clone(hash_t ** h1, hash_t * h2)
{
   *h1 = (hash_t *) malloc(sizeof(hash_t));
   if (!*h1) {
      err_msg("hash_clone: Could not allocate hash_t struct");
      return 0;
   }

   (*h1)->type = h2->type;
   (*h1)->hash_len = h2->hash_len;
   (*h1)->data_len = h2->data_len;
   (*h1)->quality = h2->quality;
   (*h1)->data = (unsigned char *) malloc((*h1)->data_len);
   if (!(*h1)->data) {
      err_msg("hash_clone: Could not allocate data block");
      return 0;
   }

   memcpy((*h1)->data, h2->data, (*h1)->data_len);

   (*h1)->hash = (unsigned char *) malloc((*h1)->hash_len);
   if (!(*h1)->hash) {
      err_msg("hash_clone: Could not allocate message digest");
      return 0;
   }

   if (h2->hash) {
      memcpy((*h1)->hash, h2->hash, (*h1)->hash_len);
   }

   hash_set_ops(*h1, (*h1)->type);

   return 1;
}

/*
 * Calculate the message digest for the given data of length n. This 
 * functions allocates memory in h->data, keep an eye on this one. 
 */
int hash_calc(hash_t * h, unsigned char *data, unsigned long n)
{
   if (!h)
      return 0;

   h->data_len = n;
   h->data = (unsigned char *) malloc(h->data_len);
   if (!h->data) {
      err_msg("hash_hash: Could not allocate data block");
      return 0;
   }

   if (!memcpy(h->data, data, h->data_len))
      return 0;

   return h->calc_hash(h->data, h->data_len, h->hash);
}

/*
 * Free the data block inside the hash. 
 */
void hash_free_data(hash_t * h)
{
   if (!h)
      return;
   if (h->data_len > 0) {
      free(h->data);
      h->data = NULL;
      h->data_len = 0;
   }
}

/*
 * Calculate the quality of the given hash object against a real hash 
 * (real message digest) using the provided fuzzy map.
 */
int hash_quality(hash_t * h, fmap_t * fmap, unsigned char *real_hash)
{
   if (!h || !fmap || !real_hash) {
      err_msg("hash_quality: arguments partial null");
      return 0;
   }

   h->quality = fmap_quality(fmap, h->hash, real_hash);
   return 1;
}

/*
 * Read a hash object from file (fd) and allocate all necessary structures. 
 */
int hash_read(hash_t ** h, int fd)
{
   if (!h)
      goto err1;

   *h = (hash_t *) malloc(sizeof(hash_t));
   if (!*h)
      goto err1;

   if (read(fd, &(*h)->type, sizeof(int)) != sizeof(int))
      goto err2;
   if (read(fd, &(*h)->quality, sizeof(unsigned long)) !=
       sizeof(unsigned long))
      goto err2;
   if (read(fd, &(*h)->hash_len, sizeof(unsigned short)) !=
       sizeof(unsigned short))
      goto err2;

   (*h)->hash = (unsigned char *) malloc((*h)->hash_len);
   if (!(*h)->hash)
      goto err2;
   if (read(fd, (*h)->hash, (*h)->hash_len) != (*h)->hash_len)
      goto err3;

   if (read(fd, &(*h)->data_len, sizeof(unsigned long)) !=
       sizeof(unsigned long))
      goto err3;

   (*h)->data = (unsigned char *) malloc((*h)->data_len);
   if (!(*h)->hash)
      goto err3;
   if (read(fd, (*h)->data, (*h)->data_len) != (*h)->data_len)
      goto err4;

   hash_set_ops(*h, (*h)->type);

   return 1;

 err4:
   free((*h)->data);
 err3:
   free((*h)->hash);
 err2:
   free(*h);
 err1:
   err_msg("hash_read: Could not read message digest");
   return 0;

}

/* 
 * Write a hash to file (fd).
 */
int hash_write(hash_t * h, int fd)
{
   if (!h)
      goto err;

   if (write(fd, &h->type, sizeof(int)) != sizeof(int))
      goto err;
   if (write(fd, &h->quality, sizeof(unsigned long)) !=
       sizeof(unsigned long))
      goto err;
   if (write(fd, &h->hash_len, sizeof(unsigned short)) !=
       sizeof(unsigned short))
      goto err;

   if (write(fd, h->hash, h->hash_len) != h->hash_len)
      goto err;

   if (write(fd, &h->data_len, sizeof(unsigned long)) !=
       sizeof(unsigned long))
      goto err;
   if (write(fd, h->data, h->data_len) != h->data_len)
      goto err;

   return 1;

 err:
   err_msg("hash_write: Could not write message digest");
   return 0;
}

/*
 * Print a hash object
 */
void hash_print(hash_t * h, FILE * out)
{
   int i;

   h->print_hash(out);
   if (h->hash) {
      fprintf(out, "   Message Digest: ");
      for (i = 0; i < h->hash_len; i++) {
	 fprintf(out, "%.2x", h->hash[i]);
	 if (i < h->hash_len - 1)
	    fprintf(out, ":");
      }
      fprintf(out, "\n");
   }
}

/*
 * Get hash type from the given name
 */
int hash_type_from_name(char *name)
{
   if (strcmp(name, "md5") == 0) {
      return HASH_MD5;
   } else if (strcmp(name, "sha1") == 0) {
      return HASH_SHA1;
   } else if (strcmp(name, "ripemd") == 0) {
      return HASH_RIPEMD;
   }

   return -1;
}
