/*
 * Fuzzy Fingerprinting - Attacking vulnerabilities in the Human Brain
 * Copyright 2002 Plasmoid <plasmoid@thc.org> - All rights reserved. 
 * On behalf of The Hacker's Choice - http://www.thc.org
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright   
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright   
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR `AS IS'' AND ANY EXPRESS OR     
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY   
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "includes.h"
RCSID("$Id: fmap.c,v 1.11 2003/10/25 14:08:14 plasmoid Exp $");

#include "fmap.h"

/*
 * Create a fuzzy map using an inverse gaussian distribution. 
 */
int fmap_inv_gauss(fmap_t * fmap, float sigma, float my)
{
   float *g;
   unsigned int i;
   float x, min = FLT_MAX, max = FLT_MIN;

   if (!fmap) {
      err_msg("fmap_inv_gauss: No fuzzy map supplied");
      return 0;
   }

   /*
    * Calculate the inverse gaussian distribution from -1 to 1.
    */
   x = -1;
   g = (float *) malloc(fmap->length * sizeof(float));
   if (!g) {
      err_msg("fmap_inv_gauss: Could not allocate float fmap");
      return 0;
   }

   for (i = 0; i < fmap->length; i++) {
      g[i] = 1 / sqrt(2 * M_PI * sigma) *
	  -exp(-0.5 * pow((x - my) / sigma, 2));
      x += (double) 2 / (fmap->length - 1);

      if (g[i] < min)
	 min = g[i];
      if (g[i] > max)
	 max = g[i];
   }

   /*
    * Convert float to unsigned int and reduce the scaling to an 1/32 of
    * MAXINT. 
    */
   fmap->sum = 0;
   fmap->weights = (unsigned int *) malloc(fmap->length *
					   sizeof(unsigned int));

   if (!fmap->weights) {
      err_msg("fmap_inv_gauss: Could allocate weight fmap");
      return 0;
   }

   for (i = 0; i < fmap->length; i++) {
      g[i] = (g[i] - min) / (max - min);
      fmap->weights[i] = g[i] * pow(2, sizeof(unsigned int) * 8) / 32;
      fmap->sum += fmap->weights[i];
   }
   free(g);

   return 1;
}

/*
 * Creat a fuzzy map using a simple cosine wave with 3 extrema. The middle 
 * maximum is placed at the value of my (the mean).
 */
int fmap_cosine(fmap_t * fmap, float sigma, float my)
{
   float *g;
   unsigned int i;
   float x, min = FLT_MAX, max = FLT_MIN;

   if (!fmap) {
      err_msg("fmap_cosine: No fuzzy map supplied");
      return 0;
   }

   /*
    * Calculate the inverse gaussian distribution from -1 to 1.
    */
   x = -2 * M_PI;
   g = (float *) malloc(fmap->length * sizeof(float));
   if (!g) {
      err_msg("fmap_cosine: Could not allocate float fmap");
      return 0;
   }

   for (i = 0; i < fmap->length; i++) {
      g[i] = cos(x - my) * sigma;
      x += (double) 4 *M_PI / (fmap->length - 1);

      if (g[i] < min)
	 min = g[i];
      if (g[i] > max)
	 max = g[i];
   }

   /*
    * Convert float to unsigned int and reduce the scaling to an 1/32 of
    * MAXINT. 
    */
   fmap->sum = 0;
   fmap->weights = (unsigned int *) malloc(fmap->length *
					   sizeof(unsigned int));

   if (!fmap->weights) {
      err_msg("fmap_cosine: Could allocate weight fmap");
      return 0;
   }

   for (i = 0; i < fmap->length; i++) {
      g[i] = (g[i] - min) / (max - min);
      fmap->weights[i] = g[i] * pow(2, sizeof(unsigned int) * 8) / 32;
      fmap->sum += fmap->weights[i];
   }
   free(g);

   return 1;

}

/*
 * Init a fuzzy map of type with a variation of sigma and a mean of my.
 * The len is the number of bytes. Note: Internally the fuzzy map will
 * work with a doubled length.
 */
int fmap_init(fmap_t ** fmap, int type, float sigma, float my, int len)
{
   if (!fmap)
      return 0;

   *fmap = (fmap_t *) malloc(sizeof(fmap_t));
   if (!*fmap) {
      err_msg("fmap_init: Could not allocate fuzzy map structure");
      return 0;
   }

   (*fmap)->length = len * 2;
   (*fmap)->type = type;

   /*
    * Use the appropriate fuzzy map generation function depending on 
    * the map type.
    */
   switch ((*fmap)->type) {
   case FMAP_COSINE:
      if (!fmap_cosine(*fmap, sigma, my))
	 goto err1;
      break;
   case FMAP_INV_GAUSS:
   default:
      if (!fmap_inv_gauss(*fmap, sigma, my))
	 goto err1;
   }

   (*fmap)->rmd = (unsigned int *) malloc((*fmap)->length *
					  sizeof(unsigned int));
   if (!(*fmap)->rmd)
      goto err2;

   (*fmap)->fmd = (unsigned int *) malloc((*fmap)->length *
					  sizeof(unsigned int));
   if (!(*fmap)->fmd)
      goto err3;

   return 1;

 err3:
   free((*fmap)->rmd);
 err2:
   free((*fmap)->weights);
 err1:
   free(*fmap);
   return 0;
}

/*
 * Clone a fuzzy map. 
 */
int fmap_clone(fmap_t ** fmap1, fmap_t * fmap2)
{
   if (!fmap1 || !fmap2)
      return 0;

   *fmap1 = (fmap_t *) malloc(sizeof(fmap_t));
   if (!*fmap1) {
      err_msg("fmap_clone: Could not allocate fuzzy map structure");
      return 0;
   }

   (*fmap1)->length = fmap2->length;
   (*fmap1)->type = fmap2->type;
   (*fmap1)->sum = fmap2->sum;

   (*fmap1)->weights = (unsigned int *) malloc(fmap2->length *
					       sizeof(unsigned int));
   if (!*fmap1)
      goto err1;

   if (!memcpy((*fmap1)->weights, fmap2->weights,
	       fmap2->length * sizeof(unsigned int)))
      goto err2;

   (*fmap1)->rmd = (unsigned int *) malloc(fmap2->length *
					   sizeof(unsigned int));
   if (!(*fmap1)->rmd)
      goto err2;

   (*fmap1)->fmd = (unsigned int *) malloc(fmap2->length *
					   sizeof(unsigned int));
   if (!(*fmap1)->fmd)
      goto err3;

   return 1;

 err3:
   free((*fmap1)->rmd);
 err2:
   free((*fmap1)->weights);
 err1:
   free(*fmap1);
   return 0;
}


/*
 * Calculate the quality of the fuzzy fingerprint (fake message digest) fmd
 * against the real fingerprint (real message digest) using the provided 
 * fuzzy map.
 */
unsigned long fmap_quality(fmap_t * fmap, unsigned char *rmd,
			   unsigned char *fmd)
{
   unsigned long q = 0;
   int i;

   /*
    * Split unsigned char input into 4 bit blocks similar to the 
    * ususal way fingerprints are displayed ff:ff:... 
    */
   for (i = 0; i < fmap->length / 2; i++) {
      fmap->rmd[2 * i + 0] = (rmd[i] & 0xf0) >> 4;
      fmap->rmd[2 * i + 1] = (rmd[i] & 0x0f) >> 0;
      fmap->fmd[2 * i + 0] = (fmd[i] & 0xf0) >> 4;
      fmap->fmd[2 * i + 1] = (fmd[i] & 0x0f) >> 0;
   }

   /*
    * Weight using the fuzzy map
    */
   for (i = 0; i < fmap->length; i++) {
      if (fmap->rmd[i] == fmap->fmd[i])
	 q += (1 * fmap->weights[i]);
   }

   /*
    * Any special options regarding key mapping, e.g. 6 -> 9, could be 
    * added here.
    */

   return q;
}

/*
 * For performance sake, fuzzy mapping is done with long and integer values
 * only. In case we want to display something meaningful this function
 * normalizes a long value against the sum of the map and thus a percentage
 * value is calculated.
 */
float fmap_normalize(fmap_t * fmap, unsigned long n)
{
   return ((float) n) / fmap->sum;
}

/*
 * Print a fuzzy map
 */
void fmap_print(fmap_t * fmap, FILE * out)
{
   int i;

   if (!fmap || !out)
      return;

   fprintf(out, "    Length: %u\n", fmap->length);
   fprintf(out, "      Type: ");
   switch (fmap->type) {
   case FMAP_INV_GAUSS:
      fprintf(out, "Inverse Gaussian Distribution\n");
      break;
   case FMAP_COSINE:
      fprintf(out, "3 Extrema Cosine\n");
      break;
   default:
      fprintf(out, "Unknown fmap type\n");
   }

   fprintf(out, "       Sum: %lu\n", fmap->sum);
   fprintf(out, " Fuzzy Map: ");
   for (i = 0; i < fmap->length; i++) {
      fprintf(out, " %2.2f%% ",
	      (float) fmap->weights[i] / (float) fmap->sum * 100);
      if (i % 2 == 1)
	 fprintf(out, ":");
      else
	 fprintf(out, "|");
      if (i % 8 == 7) {
	 fprintf(out, "\n            ");
      }

   }
   fprintf(out, "\n");
}

/*
 * Free a fuzzy map
 */
void fmap_free(fmap_t * fmap)
{
   free(fmap->fmd);
   free(fmap->rmd);
   free(fmap->weights);
   free(fmap);
}

/*
 * Write a fuzzy map to file (fd). 
 */
int fmap_write(fmap_t * fmap, int fd)
{
   if (!fmap)
      return 0;

   if (write(fd, &fmap->length, sizeof(unsigned int)) !=
       sizeof(unsigned int))
      goto err;
   if (write(fd, &fmap->sum, sizeof(unsigned long)) != sizeof(unsigned long))
      goto err;
   if (write(fd, &fmap->type, sizeof(int)) != sizeof(int))
      goto err;

   if (write(fd, fmap->weights, fmap->length * sizeof(unsigned int)) !=
       fmap->length * sizeof(unsigned int))
      goto err;

   return 1;

 err:
   err_msg("fmap_write: Could not write fuzzy map");
   return 0;
}

/*
 * Read a fuzzy map from file (fd) and allocate all necessary structures.
 */
int fmap_read(fmap_t ** fmap, int fd)
{
   if (!fmap)
      return 0;

   *fmap = (fmap_t *) malloc(sizeof(fmap_t));
   if (!*fmap) {
      err_msg("fmap_read: Could not allocate fuzzy map");
      return 0;
   }

   if (read(fd, &(*fmap)->length, sizeof(unsigned int)) !=
       sizeof(unsigned int))
      goto err2;

   if (read(fd, &(*fmap)->sum, sizeof(unsigned long)) !=
       sizeof(unsigned long))
      goto err2;

   if (read(fd, &(*fmap)->type, sizeof(int)) != sizeof(int))
      goto err2;

   (*fmap)->weights = (unsigned int *) malloc((*fmap)->length *
					      sizeof(unsigned int));
   if (!(*fmap)->weights)
      goto err2;

   if (read(fd, (*fmap)->weights, (*fmap)->length * sizeof(unsigned int)) !=
       (*fmap)->length * sizeof(unsigned int))
      goto err3;

   (*fmap)->rmd = (unsigned int *) malloc((*fmap)->length *
					  sizeof(unsigned int));
   if (!(*fmap)->rmd)
      goto err3;

   (*fmap)->fmd = (unsigned int *) malloc((*fmap)->length *
					  sizeof(unsigned int));
   if (!(*fmap)->fmd)
      goto err4;


   return 1;


 err4:
   free((*fmap)->rmd);
 err3:
   free((*fmap)->weights);
 err2:
   free((*fmap));
   err_msg("fmap_read: Could not read fuzzy map");
   return 0;
}

/*
 * Get fuzzy map type from the given name
 */
int fmap_type_from_name(char *name)
{
   if (strcmp(name, "gauss") == 0) {
      return FMAP_INV_GAUSS;
   } else if (strcmp(name, "cosine") == 0) {
      return FMAP_COSINE;
   }

   return -1;
}
