/*
 * Fuzzy Fingerprinting - Attacking vulnerabilities in the Human Brain
 * Copyright 2002 Plasmoid <plasmoid@thc.org> - All rights reserved. 
 * On behalf of The Hacker's Choice - http://www.thc.org
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright   
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright   
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR `AS IS'' AND ANY EXPRESS OR     
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY   
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * DSA_complete_key():
 * Copyright (C) 1995-1998 Eric Young (eay@cryptsoft.com)
 * All rights reserved.
 *
 * Copyright remains Eric Young's, and as such any Copyright notices in the
 * code are not to be removed.  If this package is used in a product, Eric
 * Young should be given attribution as the author of the parts of the
 * library used.  This can be in the form of a textual message at program
 * startup or in documentation (online or textual) provided with the
 * package.
 *
 * [...]
 * 
 * If this isn't enough legal blabla, let me know.
 */

#include "includes.h"
RCSID("$Id: dsa.c,v 1.2 2003/10/25 14:08:14 plasmoid Exp $");

#include <openssl/dsa.h>
#include <openssl/bn.h>
#include <openssl/err.h>

#include "xkey.h"
#include "dsa.h"

/*
 * Init an DSA key. Completely calculate a real DSA key using the 
 * DSA_generate_key().
 */
int DSA_init_key(DSA ** dsa, int bits, BIGNUM ** alpha, BIGNUM ** beta)
{

   *dsa = DSA_generate_parameters(bits, NULL, 0, NULL, NULL, NULL, NULL);
   if (!*dsa) {
      err_msg("DSA_init_key: Parameter generation failed");
      return 0;
   }
   if (!DSA_generate_key(*dsa) || !*dsa) {
      err_msg("DSA_init_key: Key generation failed");
      return 0;
   }

   return 1;
}

/*
 * Calculate next DSA key. In sloppy mode increase the private key x and
 * calculate the new public key y. In accurate choose a new private x 
 * within the specified range (0 < x < q) and calculate the new public
 * key y.
 */
int DSA_next_key(DSA * dsa, int flags, BIGNUM * alpha, BIGNUM * beta)
{
   BIGNUM *tmp = NULL;
   BN_CTX *ctx = NULL;

   ctx = BN_CTX_new();
   if (!ctx)
      goto err1;

   BN_CTX_start(ctx);
   tmp = BN_CTX_get(ctx);
   if (!tmp)
      goto err2;

   if (flags & XKEY_RSA_SLOPPY) {
      if (!BN_sub_word(dsa->priv_key, 1L))
	 goto err2;
   } else {
      do {
	 if (!BN_rand_range(dsa->priv_key, dsa->q))
	    goto err2;
      } while (BN_is_zero(dsa->priv_key));
   }

   if (!BN_mod_exp(dsa->pub_key, dsa->g, dsa->priv_key, dsa->p, ctx))
      goto err2;

   BN_CTX_end(ctx);
   BN_CTX_free(ctx);
   return 1;

 err2:
   BN_CTX_end(ctx);
   BN_CTX_free(ctx);

 err1:
   err_msg("DSA_next_key: Generating next e failed.");

   return 0;
}

int DSA_complete_key(DSA * dsa)
{
   /*
    * DSA keys don't need to be completed! ;)
    */
   return 1;
}

/*
 * Free the DSA key and its auxiliary big numbers
 */
void DSA_free_key(DSA * dsa, BIGNUM * alpha, BIGNUM * beta)
{
   if (dsa)
      DSA_free(dsa);
}

/*
 * Print the DSA key.
 */
void DSA_print_key(DSA * dsa, FILE * out, BIGNUM * alpha, BIGNUM * beta)
{
   fprintf(out, "                Key Algorithm: "
	   "DSA (Digital Signature Algorithm)\n");
   fprintf(out, "         Key Bits / Size of n: %d Bits\n",
	   BN_num_bits(dsa->p));

   fprintf(out, "                  Private key: 0x");
   BN_print_fp(out, dsa->priv_key);
   fprintf(out, "\n");
   fprintf(out, " Private Key Bits / Size of x: %d Bits\n",
	   BN_num_bits(dsa->priv_key));

   fprintf(out, "\n");
}

/*
 * Read a DSA key from file (fd) and also read the auxiliary big number
 * alpha containing phi(n).
 */
int DSA_read_key(DSA ** dsa, int fd, BIGNUM ** alpha, BIGNUM ** beta)
{
   *dsa = DSA_new();

   if (!BN_read(fd, &(*dsa)->p))
      goto err;
   if (!BN_read(fd, &(*dsa)->q))
      goto err;
   if (!BN_read(fd, &(*dsa)->g))
      goto err;
   if (!BN_read(fd, &(*dsa)->priv_key))
      goto err;
   if (!BN_read(fd, &(*dsa)->pub_key))
      goto err;

   return 1;
 err:
   err_msg("DSA_read: Could not read DSA key");
   return 0;
}


/*
 * Write an DSA key to file (fd).
 */
int DSA_write_key(DSA * dsa, int fd, BIGNUM * alpha, BIGNUM * beta)
{
   if (!BN_write(fd, dsa->p))
      goto err;
   if (!BN_write(fd, dsa->q))
      goto err;
   if (!BN_write(fd, dsa->g))
      goto err;
   if (!BN_write(fd, dsa->priv_key))
      goto err;
   if (!BN_write(fd, dsa->pub_key))
      goto err;

   return 1;
 err:
   err_msg("DSA_write: Could not write DSA key");
   return 0;
}

/*
 * Clone an DSA key.
 */
int DSA_clone_key(DSA ** dsa1, BIGNUM ** alpha1, BIGNUM ** beta1,
		  DSA * dsa2, BIGNUM * alpha2, BIGNUM * beta2)
{
   *dsa1 = DSA_new();

   if (!((*dsa1)->p = BN_dup(dsa2->p)))
      goto err;
   if (!((*dsa1)->q = BN_dup(dsa2->q)))
      goto err;
   if (!((*dsa1)->g = BN_dup(dsa2->g)))
      goto err;
   if (!((*dsa1)->priv_key = BN_dup(dsa2->priv_key)))
      goto err;
   if (!((*dsa1)->pub_key = BN_dup(dsa2->pub_key)))
      goto err;

   return 1;

 err:
   err_msg("DSA_clone_key: Could not clone key");
   return 0;
}
