/*
 * Author: Tatu Ylonen <ylo@cs.hut.fi>
 * Copyright (c) 1995 Tatu Ylonen <ylo@cs.hut.fi>, Espoo, Finland
 *                    All rights reserved
 * Code for manipulating FIFO buffers.
 *
 * As far as I am concerned, the code I have written for this software
 * can be used freely for any purpose.  Any derived versions of this
 * software must be clearly marked as such, and if the derived work is
 * incompatible with the protocol description in the RFC file, it must be
 * called by a name other than "ssh" or "Secure Shell".
 */

#ifndef _BUF_H
#define _BUF_H

typedef struct {
   unsigned char *buf;		/* Buffer for data. */
   unsigned int alloc;		/* Number of bytes allocated for data. */
   unsigned int offset;		/* Offset of first byte containing data. */
   unsigned int end;		/* Offset of last byte containing data. */
} Buffer;

void buffer_init(Buffer *);
void buffer_clear(Buffer *);
void buffer_free(Buffer *);

unsigned int buffer_len(Buffer *);
void *buffer_ptr(Buffer *);

void buffer_append(Buffer *, const void *, unsigned int);
void *buffer_append_space(Buffer *, unsigned int);

void buffer_get(Buffer *, void *, unsigned int);

void buffer_consume(Buffer *, unsigned int);
void buffer_consume_end(Buffer *, unsigned int);

void buffer_dump(Buffer *);

void buffer_put_bignum(Buffer *, BIGNUM *);
void buffer_put_bignum2(Buffer *, BIGNUM *);
void buffer_get_bignum(Buffer *, BIGNUM *);
void buffer_get_bignum2(Buffer *, BIGNUM *);

unsigned short buffer_get_short(Buffer *);
void buffer_put_short(Buffer *, unsigned short);

unsigned int buffer_get_int(Buffer *);
void buffer_put_int(Buffer *, unsigned int);

#ifdef HAVE_U_INT64_T
unsigned int64_t buffer_get_int64(Buffer *);
void buffer_put_int64(Buffer *, unsigned int64_t);
#endif

int buffer_get_char(Buffer *);
void buffer_put_char(Buffer *, int);

void *buffer_get_string(Buffer *, unsigned int *);
void buffer_put_string(Buffer *, const void *, unsigned int);
void buffer_put_cstring(Buffer *, const char *);

#define buffer_skip_string(b) \
    do { unsigned int l = buffer_get_int(b); buffer_consume(b, l); } while(0)


#define GET_64BIT(cp) (((unsigned int64_t)(unsigned char)(cp)[0] << 56) | \
		       ((unsigned int64_t)(unsigned char)(cp)[1] << 48) | \
		       ((unsigned int64_t)(unsigned char)(cp)[2] << 40) | \
		       ((unsigned int64_t)(unsigned char)(cp)[3] << 32) | \
		       ((unsigned int64_t)(unsigned char)(cp)[4] << 24) | \
		       ((unsigned int64_t)(unsigned char)(cp)[5] << 16) | \
		       ((unsigned int64_t)(unsigned char)(cp)[6] << 8) | \
		       ((unsigned int64_t)(unsigned char)(cp)[7]))

#define GET_32BIT(cp) (((unsigned long)(unsigned char)(cp)[0] << 24) | \
		       ((unsigned long)(unsigned char)(cp)[1] << 16) | \
		       ((unsigned long)(unsigned char)(cp)[2] << 8) | \
		       ((unsigned long)(unsigned char)(cp)[3]))

#define GET_16BIT(cp) (((unsigned long)(unsigned char)(cp)[0] << 8) | \
		       ((unsigned long)(unsigned char)(cp)[1]))

#define PUT_64BIT(cp, value) do { \
  (cp)[0] = (value) >> 56; \
  (cp)[1] = (value) >> 48; \
  (cp)[2] = (value) >> 40; \
  (cp)[3] = (value) >> 32; \
  (cp)[4] = (value) >> 24; \
  (cp)[5] = (value) >> 16; \
  (cp)[6] = (value) >> 8; \
  (cp)[7] = (value); } while (0)

#define PUT_32BIT(cp, value) do { \
  (cp)[0] = (value) >> 24; \
  (cp)[1] = (value) >> 16; \
  (cp)[2] = (value) >> 8; \
  (cp)[3] = (value); } while (0)

#define PUT_16BIT(cp, value) do { \
  (cp)[0] = (value) >> 8; \
  (cp)[1] = (value); } while (0)

#endif				/* _BUF_H */
