/*
 * Author: Tatu Ylonen <ylo@cs.hut.fi>
 * Copyright (c) 1995 Tatu Ylonen <ylo@cs.hut.fi>, Espoo, Finland
 *                    All rights reserved
 * Functions for manipulating fifo buffers (that can grow if needed).
 *
 * As far as I am concerned, the code I have written for this software
 * can be used freely for any purpose.  Any derived versions of this
 * software must be clearly marked as such, and if the derived work is
 * incompatible with the protocol description in the RFC file, it must be
 * called by a name other than "ssh" or "Secure Shell".
 */

#include "includes.h"
RCSID("$Id: buffer.c,v 1.1.1.1 2002/08/07 10:50:49 plasmoid Exp $");

#include <openssl/bn.h>

#include "buffer.h"

void buffer_init(Buffer * buffer)
{
   buffer->alloc = 4096;
   buffer->buf = malloc(buffer->alloc);
   buffer->offset = 0;
   buffer->end = 0;
}

/* Frees any memory used for the buffer. */

void buffer_free(Buffer * buffer)
{
   memset(buffer->buf, 0, buffer->alloc);
   free(buffer->buf);
}

/*
 * Clears any data from the buffer, making it empty.  This does not actually
 * zero the memory.
 */

void buffer_clear(Buffer * buffer)
{
   buffer->offset = 0;
   buffer->end = 0;
}

/* Appends data to the buffer, expanding it if necessary. */

void buffer_append(Buffer * buffer, const void *data, unsigned int len)
{
   void *p;
   p = buffer_append_space(buffer, len);
   memcpy(p, data, len);
}

/*
 * Appends space to the buffer, expanding the buffer if necessary. This does
 * not actually copy the data into the buffer, but instead returns a pointer
 * to the allocated region.
 */

void *buffer_append_space(Buffer * buffer, unsigned int len)
{
   void *p;

   if (len > 0x100000) {
      err_msg("buffer_append_space: len > 0x100000 not supported");
      return NULL;
   }

   /* If the buffer is empty, start using it from the beginning. */
   if (buffer->offset == buffer->end) {
      buffer->offset = 0;
      buffer->end = 0;
   }
 restart:
   /* If there is enough space to store all data, store it now. */
   if (buffer->end + len < buffer->alloc) {
      p = buffer->buf + buffer->end;
      buffer->end += len;
      return p;
   }
   /*
    * If the buffer is quite empty, but all data is at the end, move the
    * data to the beginning and retry.
    */
   if (buffer->offset > buffer->alloc / 2) {
      memmove(buffer->buf, buffer->buf + buffer->offset,
	      buffer->end - buffer->offset);
      buffer->end -= buffer->offset;
      buffer->offset = 0;
      goto restart;
   }
   /* Increase the size of the buffer and retry. */
   buffer->alloc += len + 32768;
   if (buffer->alloc > 0xa00000) {
      err_msg("buffer_append_space: alloc > 0xa00000 not supported");
      return NULL;
   }
   buffer->buf = realloc(buffer->buf, buffer->alloc);
   goto restart;
   /* NOTREACHED */
}

/* Returns the number of bytes of data in the buffer. */

unsigned int buffer_len(Buffer * buffer)
{
   return buffer->end - buffer->offset;
}

/* Gets data from the beginning of the buffer. */

void buffer_get(Buffer * buffer, void *buf, unsigned int len)
{
   if (len > buffer->end - buffer->offset)
      err_msg("buffer_get: trying to get more bytes than in buffer");

   memcpy(buf, buffer->buf + buffer->offset, len);
   buffer->offset += len;
}

/* Consumes the given number of bytes from the beginning of the buffer. */

void buffer_consume(Buffer * buffer, unsigned int bytes)
{
   if (bytes > buffer->end - buffer->offset) {
      err_msg("buffer_consume: trying to get more bytes than in buffer");
   }

   buffer->offset += bytes;
}

/* Consumes the given number of bytes from the end of the buffer. */

void buffer_consume_end(Buffer * buffer, unsigned int bytes)
{
   if (bytes > buffer->end - buffer->offset)
      err_msg("buffer_consume_end: trying to get more bytes than in buffer");
   buffer->end -= bytes;
}

/* Returns a pointer to the first used byte in the buffer. */

void *buffer_ptr(Buffer * buffer)
{
   return buffer->buf + buffer->offset;
}

/* Dumps the contents of the buffer to stderr. */

void buffer_dump(Buffer * buffer)
{
   int i;
   unsigned char *ucp = buffer->buf;

   for (i = buffer->offset; i < buffer->end; i++) {
      fprintf(stderr, "%02x", ucp[i]);
      if ((i - buffer->offset) % 16 == 15)
	 fprintf(stderr, "\r\n");
      else if ((i - buffer->offset) % 2 == 1)
	 fprintf(stderr, " ");
   }
   fprintf(stderr, "\r\n");
}

/*
 * Stores an BIGNUM in the buffer with a 2-byte msb first bit count, followed
 * by (bits+7)/8 bytes of binary data, msb first.
 */
void buffer_put_bignum(Buffer * buffer, BIGNUM * value)
{
   int bits = BN_num_bits(value);
   int bin_size = (bits + 7) / 8;
   unsigned char *buf = malloc(bin_size);
   int oi;
   char msg[2];

   /* Get the value of in binary */
   oi = BN_bn2bin(value, buf);
   if (oi != bin_size)
      err_msg("buffer_put_bignum: BN_bn2bin() failed");

   /* Store the number of bits in the buffer in two bytes, msb first. */
   PUT_16BIT(msg, bits);
   buffer_append(buffer, msg, 2);
   /* Store the binary data. */
   buffer_append(buffer, (char *) buf, oi);

   memset(buf, 0, bin_size);
   free(buf);
}

/*
 * Retrieves an BIGNUM from the buffer.
 */
void buffer_get_bignum(Buffer * buffer, BIGNUM * value)
{
   int bits, bytes;
   unsigned char buf[2], *bin;

   /* Get the number for bits. */
   buffer_get(buffer, (char *) buf, 2);
   bits = GET_16BIT(buf);
   /* Compute the number of binary bytes that follow. */
   bytes = (bits + 7) / 8;
   if (bytes > 8 * 1024)
      err_msg("buffer_get_bignum: cannot handle BN");
   if (buffer_len(buffer) < bytes)
      err_msg("buffer_get_bignum: input buffer too small");
   bin = buffer_ptr(buffer);
   BN_bin2bn(bin, bytes, value);
   buffer_consume(buffer, bytes);
}

/*
 * Stores an BIGNUM in the buffer in SSH2 format.
 */
void buffer_put_bignum2(Buffer * buffer, BIGNUM * value)
{
   int bytes = BN_num_bytes(value) + 1;
   unsigned char *buf = malloc(bytes);
   int oi;
   int hasnohigh = 0;

   buf[0] = '\0';
   /* Get the value of in binary */
   oi = BN_bn2bin(value, buf + 1);
   if (oi != bytes - 1)
      err_msg("buffer_put_bignum2: BN_bn2bin() failed");

   hasnohigh = (buf[1] & 0x80) ? 0 : 1;
   if (value->neg) {
		/**XXX should be two's-complement */
      int i, carry;
      unsigned char *uc = buf;
      for (i = bytes - 1, carry = 1; i >= 0; i--) {
	 uc[i] ^= 0xff;
	 if (carry)
	    carry = !++uc[i];
      }
   }
   buffer_put_string(buffer, buf + hasnohigh, bytes - hasnohigh);
   memset(buf, 0, bytes);
   free(buf);
}

/* XXX does not handle negative BNs */
void buffer_get_bignum2(Buffer * buffer, BIGNUM * value)
{
   unsigned int len;
   unsigned char *bin = buffer_get_string(buffer, &len);

   if (len > 8 * 1024)
      err_msg("buffer_get_bignum2: cannot handle BN of size");
   BN_bin2bn(bin, len, value);
   free(bin);
}

/*
 * Returns integers from the buffer (msb first).
 */

unsigned short buffer_get_short(Buffer * buffer)
{
   unsigned char buf[2];

   buffer_get(buffer, (char *) buf, 2);
   return GET_16BIT(buf);
}

unsigned int buffer_get_int(Buffer * buffer)
{
   unsigned char buf[4];

   buffer_get(buffer, (char *) buf, 4);
   return GET_32BIT(buf);
}

#ifdef HAVE_U_INT64_T
unsigned int64_t buffer_get_int64(Buffer * buffer)
{
   unsigned char buf[8];

   buffer_get(buffer, (char *) buf, 8);
   return GET_64BIT(buf);
}
#endif

/*
 * Stores integers in the buffer, msb first.
 */
void buffer_put_short(Buffer * buffer, unsigned short value)
{
   char buf[2];

   PUT_16BIT(buf, value);
   buffer_append(buffer, buf, 2);
}

void buffer_put_int(Buffer * buffer, unsigned int value)
{
   char buf[4];

   PUT_32BIT(buf, value);
   buffer_append(buffer, buf, 4);
}

#ifdef HAVE_U_INT64_T
void buffer_put_int64(Buffer * buffer, unsigned int64_t value)
{
   char buf[8];

   PUT_64BIT(buf, value);
   buffer_append(buffer, buf, 8);
}
#endif

/*
 * Returns an arbitrary binary string from the buffer.  The string cannot
 * be longer than 256k.  The returned value points to memory allocated
 * with malloc; it is the responsibility of the calling function to free
 * the data.  If length_ptr is non-NULL, the length of the returned data
 * will be stored there.  A null character will be automatically appended
 * to the returned string, and is not counted in length.
 */
void *buffer_get_string(Buffer * buffer, unsigned int *length_ptr)
{
   unsigned char *value;
   unsigned int len;

   /* Get the length. */
   len = buffer_get_int(buffer);
   if (len > 256 * 1024)
      err_msg("buffer_get_string: bad string");
   /* Allocate space for the string.  Add one byte for a null character. */
   value = malloc(len + 1);
   /* Get the string. */
   buffer_get(buffer, value, len);
   /* Append a null character to make processing easier. */
   value[len] = 0;
   /* Optionally return the length of the string. */
   if (length_ptr)
      *length_ptr = len;
   return value;
}

/*
 * Stores and arbitrary binary string in the buffer.
 */
void buffer_put_string(Buffer * buffer, const void *buf, unsigned int len)
{
   buffer_put_int(buffer, len);
   buffer_append(buffer, buf, len);
}
void buffer_put_cstring(Buffer * buffer, const char *s)
{
   if (s == NULL)
      err_msg("buffer_put_cstring: s == NULL");
   buffer_put_string(buffer, s, strlen(s));
}

/*
 * Returns a character from the buffer (0 - 255).
 */
int buffer_get_char(Buffer * buffer)
{
   char ch;

   buffer_get(buffer, &ch, 1);
   return (unsigned char) ch;
}

/*
 * Stores a character in the buffer.
 */
void buffer_put_char(Buffer * buffer, int value)
{
   char ch = value;

   buffer_append(buffer, &ch, 1);
}
