# Interface Definition Language

<!-- https://www.internalfb.com/intern/wiki/Thrift/Thrift_Guide/IDL/IDL-Reference/?noredirect -->

This document is a reference manual for the Thrift Interface Definition Language. It defines the syntax and semantics of Thrift files and defines constraints on the behavior of the generated code and libraries. It does not fully constrain the details of the generated code and libraries - and hence allows for flexibility in this regard. These details (which must be documented separately as an implementation reference) include:

* the mapping between Thrift types and programming language types and APIs;
* serialization and deserialization strategies;
* network protocols for RPCs.

The primary audience of this reference is Thrift users who want to understand the language in more detail. The secondary audience is Thrift implementers who want to learn about constraints imposed by the language on their implementation.

## Notation

Thrift syntax is described using a [modified BNF grammar notation](https://docs.python.org/3/reference/introduction.html#notation).

As an example, consider the grammar for a binary literal:

```grammar
bin_literal ::=  "0" ("b" | "B") bin_digit+
bin_digit   ::=  "0" | "1"
```

This rule defines `bin_literal` as starting with `0`, followed by a lowercase or uppercase `B` and one or more binary digits (`0` or `1`).

## Lexical Elements

The Thrift language uses different lexical elements (tokens), such as identifiers, keywords, literals, and operators, as building blocks to define its syntax. This section describes what sequences of characters constitute valid tokens in the language.

When parsing a Thrift file, tokens are generated by selecting the longest possible substring from the input text that conforms to the constraints of the grammar (maximal munch).

The encoding of Thrift source files is UTF-8.

### Whitespace and Comments

Whitespace is used to separate tokens in source files. The following characters are considered whitespace in Thrift: space (U+0020), horizontal tab (U+0009), line feed (U+000A) and carriage return (U+000D).

Thrift supports single-line and multiline comments. Single-line comments start with `//` or `#` and continue until the end of the line which is determined as the nearest line feed (U+000A) or, if there is none, the end of file. Multiline comments start with `/*` and end with the nearest `*/`. Comments are treated as whitespace.

### Identifiers

Identifiers begin with an uppercase or lowercase letter `A` through `Z` or an underscore (`_`). Subsequent characters can also be digits `0` through `9`.

Identifiers in Thrift are case-sensitive.

```grammar
identifier  ::=  id_start id_continue*
id_start    ::=  "a"..."z" | "A"..."Z" | "_"
id_continue ::=  id_start | digit
```

A qualified identifier is a sequence of two or more identifiers separated by periods, e.g. `search_types.Query`. `maybe_qualified_id` denotes a qualified or unqualified identifier.

```grammar
maybe_qualified_id ::=  identifier ["." identifier]*
```

### Keywords

Thrift has two types of keywords: reserved words and context-sensitive keywords.
The following are reserved words in Thrift - and cannot be used as identifiers:

```thrift
binary              false               map                 struct
bool                float               namespace           throws
byte                hs_include          optional            true
const               i16                 performs            typedef
cpp_include         i32                 required            union
double              i64                 service             void
enum                include             set
exception           interaction         stream
extends             list                string
```

Context-sensitive keywords listed below can be used as identifiers:

```thrift
client              package             safe                stateful
idempotent          permanent           server              transient
oneway              readonly            sink
```

Programming languages for which code is generated by the Thrift compiler will have their own additional reserved words. Thrift implementations are permitted to disallow use of these reserved words. The implementation reference must specify how such reserved words are treated.

### Literals

A literal is a representation of a fixed value in source code. Thrift supports integer, floating-point, string and Boolean literals.

```grammar
literal ::=  int_literal | float_literal | string_literal | bool_literal

int_literal ::=  dec_literal | bin_literal | oct_literal | hex_literal
dec_literal ::=  "1"..."9" digit*
bin_literal ::=  "0" ("b" | "B") bin_digit+
oct_literal ::=  "0" oct_digit*
hex_literal ::=  "0" ("x" | "X") hex_digit+

digit     ::=  "0"..."9"
bin_digit ::=  "0" | "1"
oct_digit ::=  "0"..."7"
hex_digit ::=  digit | "a"..."f" | "A"..."F"

float_literal ::=  digit+ "." digit+ | (digit+ ["." digit+]) exponent
exponent      ::=  ("e" | "E") ["+" | "-"] digit+

string_literal ::=  '"' (dstring_char | escape_seq)* '"' |
                    "'" (sstring_char | escape_seq)* "'"
dstring_char   ::=  <any source character except "\\" or '"'>
sstring_char   ::=  <any source character except "\\" or "'">
escape_seq     ::=  simple_escape | hex_escape | unicode_escape
simple_escape  ::=  "\\" ("\n" | "\\" | "'" | "\"" | "n" | "r" | "t")
hex_escape     ::=  "\\x" hex_digit hex_digit
unicode_escape ::=  "\\u" hex_digit hex_digit hex_digit hex_digit

bool_literal ::=  "true" | "false"
```

Both single quoted and double quoted string literals support the following
escape sequences:

| Escape Sequence | Meaning                       |
| --------------- | ----------------------------- |
| `\\`            | Backslash (U+005C)            |
| `\'`            | Single quote (U+0027)         |
| `\"`            | Double quote (U+0022)         |
| `\n`            | Line feed (U+000A)            |
| `\r`            | Carriage return (U+000D)      |
| `\t`            | Horizontal tab (U+0009)       |
| `\xhh`          | Code unit with hex value *hh* |
| `\uhhhh`        | Unicode scalar value *hhhh*   |

where `h` denotes a hexadecimal digit.

A backslash (`\`) at the end of a line in a multiline string is removed from the string together with the newline character (U+000A).

Here are some examples of literals in Thrift:

```thrift
42                // decimal integer literal
0xCAFE            // hexadecimal integer literal
3.14159           // floating-point literal
"Don't panic!"    // string literal using double quotes
'\u2665 of Gold'  // string literal using single quotes
true              // Boolean literal
```

Literals in Thrift do not have exact types. Instead they are represented using maximum precision and the actual type is inferred from context. For example:

```thrift
const i32 BIG = 42;
const i16 LITTLE = 42;
```

The literal `42` is treated as a 32-bit integer in the definition of `BIG` and as a 16-bit integer in the definition of `LITTLE`.

### Operators and Punctuation

The following tokens serve as operators and punctuation in Thrift:

```
(       )       {       }       [       ]       <       >
,       ;       @       =       +       -
```

Thrift uses an optional comma (`,`) or semicolon (`;`) as delimiters in certain constructs such as argument and field lists:

```grammar
delimiter ::=  ["," | ";"]
```

## Thrift Files

A Thrift file starts with an optional package declaration and a, possibly empty, sequence of include and namespace directives. It is followed by a sequence of definitions which can also be empty. There can be at most one package declaration and it is normally placed at the beginning of a source file.

```grammar
thrift_file ::=
  (include_directive | package_declaration | namespace_directive)*
  definition*
```

Here is an example of a Thrift file:

```thrift
// Allows the definitions in search_types.thrift to be used here qualified
// with the prefix `search_types.`.
include "common/if/search_types.thrift"

// Directs the compiler to place the generated C++ code inside the
// namespace `facebook::peoplesearch`, the generated Java code inside
// the package `com.facebook.peoplesearch` and similarly for other
// languages. Also specifies the prefix for universal names.
package "facebook.com/peoplesearch"

// The port where the server listens for requests. This is a convenient
// way to define constants visible to code in all languages.
const i32 PORT = 3456;

// The default number of search results used below as the default value
// for `numResults`.
const i32 DEFAULT_NUM_RESULTS = 10;

// The argument to the `search` RPC. Contains two fields - the query and
// the number of results desired.
struct PeopleSearchRequest {
  1: search_types.Query query;
  2: i32 numResults = DEFAULT_NUM_RESULTS;
}

// The response from the `search` RPC. Contains the list of results.
struct PeopleSearchResponse {
  1: list<search_types.PersonMetadata> results;
}

/*
 * The service definition. This defines a single RPC `search`.
 */
service PeopleSearch {
  PeopleSearchResponse search(1: PeopleSearchRequest request);
}
```

### Include Directives

```grammar
include_directive ::=
  ("include" | "cpp_include" | "hs_include") string_literal [";"]
```

Include directives allow the use of constants, types, and services from other Thrift files by prefixing the name (without extension) of the included Thrift file followed by a period. `cpp_include` instructs the compiler to emit an include in generated C++ code and `hs_include` does the same for Haskell.

```thrift
include "common/if/search_types.thrift"

struct PeopleSearchRequest {
  // The type `Query` is defined in search_types.thrift.
  1: search_types.Query query;
  2: i32 numResults = DEFAULT_NUM_RESULTS;
}

struct PeopleSearchResponse {
  // The type `PersonMetadata` is defined in search_types.thrift.
  1: list<search_types.PersonMetadata> results;
}
```

If there is a circular dependency between files, a compile-time error is reported. So `a.thrift` cannot include itself, and cannot include `b.thrift` if `b.thrift` includes `a.thrift`. Including multiple files with a common ancestor is okay - so `a.thrift` can include `b.thrift` and `c.thrift` when both `b.thrift` and `c.thrift` include `d.thrift`.

### Package Declaration

A package declaration determines the default namespaces for target languages, e.g. the namespace for the generated C++ code and the package for Java. It is also used for applying file-level annotations and as the default [universal name](../features/universal-name.md) prefix.

```grammar
package_declaration ::=  [annotations] "package" package_name [";"]
```

A package name is a string containing a domain name and a path separated with `/`, for example:

```thrift
package "meta.com/search"
```

<!-- package_name is defined here and not in Lexical Elements because its components are used to specify the default namespace rules. -->

The lexical structure of a package name is defined below.

```grammar
package_name  ::=  '"' domain_path '"' | "'" domain_path "'"
domain_path   ::=  domain "/" path
domain        ::=  domain_prefix "." identifier
domain_prefix ::=  identifier ("." identifier)*
path          ::=  identifier ("/" identifier)*
```

Let `namespace_path` denote `path` where every `/` is replaced with `.` and `reverse(d)` denote the domain `d` with components in reverse order. Then the default namespaces are derived from the package name in the following way.

* C++ (`cpp2`): The namespace is a concatenation of `reverse(domain_prefix)` and `namespace_path` separated by a period (`.`).
* Python (`python`, `py3`): The namespace is a concatenation of `reverse(domain_prefix)` and `namespace_path` with the last path component (and preceding `.`) removed if it is equal to the Thrift file name without extension. The domain prefix and the path are separated by a period.
* Hack: The namespace is the value of `namespace_path`.
* Java (`java.swift`): The namespace is a concatenation of `reverse(domain)` and `namespace_path` separated by a period.

Other Thrift implementations must define their own mappings from package names to namespaces.

Here is an example with the package name containing the Thrift file without extension (`query`):

```thrift title="search/query.thrift"
package "meta.com/search/query"
```

This gives the following default namespaces:

| Language    | Target(s)       | Namespace               |
| ----------- | --------------- | ----------------------- |
| C++         | `cpp2`          | `meta.search.query`     |
| Python      | `python`, `py3` | `meta.search`           |
| Hack        | `hack`          | `search.query`          |
| Java        | `java.swift`    | `com.meta.search.query` |

Here is an example with the package name not containing the file name component:

```thrift title="search/query.thrift"
package "meta.com/search"
```

| Language    | Target(s)       | Namespace         |
| ----------- | --------------- | ----------------- |
| C++         | `cpp2`          | `meta.search`     |
| Python      | `python`, `py3` | `meta.search`     |
| Hack        | `hack`          | `search`          |
| Java        | `java.swift`    | `com.meta.search` |

In both cases the Python namespaces don't include the file name component.

The default namespace can be overriden for individual target languages with the [namespace directive](#namespace-directives).

Annotations on a package declaration apply to the whole file. For example:

```thrift
@cpp.TerseWrite
package "facebook.com/peoplesearch"

struct PeopleSearchRequest {
  1: search_types.Query query;
  2: i32 numResults = DEFAULT_NUM_RESULTS;
}
```

This enables terse writes for `query` and `numResults` fields even though they are not annotated with `@cpp.TerseWrite` themselves.

The package name is also used as a prefix for universal names. For example:

```thrift
package "facebook.com/peoplesearch"

// Has the universal name "facebook.com/peoplesearch/PeopleSearchRequest".
struct PeopleSearchRequest { /* ... */ }
```

### Namespace Directives

```grammar
namespace_directive ::=  "namespace" maybe_qualified_id namespace_name [";"]
namespace_name      ::=  maybe_qualified_id | string_literal
```

Namespace directives override the default namespaces set by a [package declaration](#package-declaration).

Namespaces control the top-level structure of the generated code in a language-specific way. They do not affect the Thrift file semantics.

```thrift
// Directs the compiler to generate C++ code inside the namespace
// `facebook::peoplesearch`.
namespace cpp2 facebook.peoplesearch

// Directs the compiler to generate Java code inside the package
// `com.facebook.peoplesearch`.
namespace java.swift com.facebook.peoplesearch
```

## Definitions

Thrift supports three kinds of definitions: type definitions, interface definitions and constant definitions.

```grammar
definition ::=
  type_definition | interface_definition | constant_definition
```

Definitions appear at the top level of a Thrift file. A common property of all definitions is that they introduce a name (an identifier) that can be used to denote the entity such as type.

The name introduced by definitions can be used anywhere in the Thrift file (either before or after the location of the definition), and in other Thrift files that include this Thrift file.

```grammar
type_definition ::=
    struct
  | union
  | exception
  | enumeration
  | typedef
```

### Structs

A struct declaration introduces a named struct type into your program and has the following form:

```grammar
struct ::=
  [annotations]
  "struct" identifier "{"
    (field delimiter)*
  "}"

field    ::=  field_id ":" [qualifier] type identifier [default_value]
field_id ::=  integer
```

*The struct type is the centerpiece of the Thrift language. It is the basic unit of serialization and versioning in the language.*

This section defines the primary aspects of structs. Separate sections define qualifiers and default values for fields, constraints on serialization of structs, and compatibility between different versions of struct types.

Structs in Thrift look similar to structs in C++ though fields in Thrift have a field id and optionally a qualifier. They are significantly different in that the order of appearance of fields is not important and there is no expectations on the memory layout of the objects in the corresponding programming language code.

```thrift
struct PersonMetadata {
  1: string name;
  2: i32 age;
  3: list<i64> friendIds;
}
```

Every field has a type and a name (an identifier) that can be used to denote the field just as in C++. In addition every field also has an integer id that can be used to denote the field. Field ids and field names must be unique within a struct. While the Thrift language does not impose restrictions on how the id and name are used, the *primary usage* has the id denoting the field in the serialized data and the name denoting the field in the generated code. This primary usage is key to the value provided by Thrift in terms of versioning support and compact serialized data size.

NOTE: The *primary usage* described above can be assumed for the large majority of Thrift usage at Facebook. The exception to this is that Thrift does offer a JSON serialization strategy where the name is used to denote the field in the serialized data and the id is not used. This strategy is used in Configerator and Tupperware specs and should not be mixed with other usage where the id is used instead.

The field ids should be in the range of 1 through 65535, and they do not have to appear in order.

Thrift does not support structure inheritance (though it is possible to use composition to achieve a similar goal).

As a consequence of the above rules, `PersonMetadata1` below defines the same type as `PersonMetadata` above:

```thrift
// Just a different ordering of fields.
struct PersonMetadata1 {
  2: i32 age;
  1: string name;
  3: list<i64> friendIds;
}
```

But the following two types are both legal, but different from `PersonMetadata`:

```thrift
// Some ids are different.
struct PersonMetadata2 {
  590: string name;
  36: i32 age;
  3: list<i64> friendIds;
}

// Some names are different.
struct PersonMetadata3 {
  1: string firstName;
  2: i32 age;
  3: list<i64> friends;
}
```

NOTE: Field ids do not have to start at 1.

:::caution
Do not reuse ids. If a field is removed, it's a good practice to comment it out as a reminder to not to reuse it.
:::

### Unions

```grammar
union ::=
  [annotations]
  "union" identifier "{"
    (field delimiter)*
  "}"
```

Union types are identical to struct types in all ways, except for the following differences:

* Union types use the reserved word `union` instead of `struct`.
* All fields must be unqualified, but they are equivalent to struct fields with the qualifier `optional`.
* At most one field can be *present*.

NOTE: The concepts "optional" and "present" are described in more detail below.

The serialized representation of union types are identical to that of struct types that have a single field present. When deserializing into an union type, the serialized data may not provide a value to more than one of the fields of the union type.

The generated code for union types can be different from that of struct types - for example, implementations may choose to use programming language union types in generated code for better space efficiency.

NOTE: It is okay for none of the fields to be present in an union type.

NOTE: It is possible to serialize from an union type and deserialize into a compatible struct type, and vice versa.

### Exceptions

```grammar
exception ::=
  [annotations] [error_safety] [error_kind] [error_blame]
  "exception" identifier "{"
    (field delimiter)*
  "}"

error_safety ::=  "safe"
error_kind   ::=  "transient" | "stateful" | "permanent"
error_blame  ::=  "client" | "server"
```

Exception types are identical to struct types in all ways, except that these types can only be used as types within the `throws` clause of functions in service definitions. (This isn't enforced, but you still shouldn't do it).

The serialized representation of exception types are identical to that of struct types.

The generated code for exception types can be different from that of struct types - for example, programming language exception types can be used in generated code.

:::note
It is possible to serialize from an exception type and deserialize into a compatible struct type, and vice versa.
:::

### Enumerations

```
enumeration ::=
  "enum" identifier "{" ( Enumerator delimiter )* "}"

Enumerator ::=
  identifier "=" integer
```

Thrift supports C++ style enumeration types. The enumerators (the named constants) must be explicitly bound to an integer value. The identifier after the reserved word `enum` may be used to denote the enumeration type.

```thrift
enum SearchKind {
  UNKNOWN = 0,  // default value
  PEOPLE = 1,
  PAGES = 3,
  GROUPS = 5,
}
```

:::note
Because the default value for every enum is 0 (even if you do not define an enumerator for 0), it is recommended to include an `UNKNOWN` entry at 0 and use that value to indicate that the client or server didn't provide the value. The default constructor will initialize that value to `UNKNOWN` instead of some meaningful value.
:::

:::caution
Enums are treated like integer by Thrift, if you send a value which is not in the valid range, the receiver may not check or convert the value to the default (it will just have an out of range value) or it may throw an exception on access. This can happen when a new client is talking to an old server.
:::

:::caution
Removing and adding enum values can be dangerous - see [Schema Compatibility](/features/compatibility.md).
:::

### Typedefs

```grammar
typedef ::=  "typedef" type identifier [";"]
```

Typedefs introduce a name that denotes the type specification. It can be used to provide a simpler way to access complex types.

```thrift
typedef map<string, string> StringMap
```

### Services

```
interface_definition ::=
  ("service" | "interaction") identifier ["extends" maybe_qualified_id] "{"
    ( FunctionSpecification )*
  "}"

FunctionSpecification ::=
  ResultType identifier
  "("
  ( ParameterSpecification )*
  ")"
  [ "throws" "(" field+ ")" ]

ResultType ::= "void" | "oneway void" | ["stream"] type

ParameterSpecification ::= field_id ":" type identifier [default_value]
```

An interface for RPC is defined in a Thrift file as a service.

Each service has a set of functions. Each function has a unique name and takes a list of arguments. It can return normally with a result if the result type is not `void` or it can return by throwing one of the listed application exceptions. In addition, the function can return by throwing a Thrift system exception if there was some underlying problem with the RPC itself.

The types of the field specifications after `throws` must be exception types. If a functions throws one of the exceptions given in the `throws` clause, then all of the members of this exception will be serialized and sent over the wire. For other undeclared exceptions only the message will be serialized and they will appear on the client side as `TApplicationException`.

The list of arguments to the function follow similar rules as Thrift struct types with the exception of qualifiers, meaning that **arguments cannot be optional**. The proper way to achieve this is to use a struct type argument, which itself then may contain an `optional` field.

Functions that use the `oneway` reserved word (oneway functions) are "fire and forget". I.e., the client sends the function parameters to the server, but does not wait or expect a result. Therefore oneway functions must use `void` as the return type and must not have a `throws` clause.

Services may extend (inherit from) other services. The set of functions in the inherited service is included in the inheriting service. The name that refers to the inherited service after the reserved word `extends` can be an identifier that denotes another service defined earlier in the same Thrift file, or a qualified name of the form `filename.servicename` where `filename` and `servicename` are identifiers, and `servicename` denotes a service defined in the Thrift file denoted by `filename`.

Service names are visible after the end of the service definition in which they have been introduced, and in other Thrift files that include this Thrift file.

```
struct Foo {
  1: i64 field1 = 101;
  2: string field2;
}

exception BarException {
  1: string message,
  2: i64 errorCode,
}

exception FooException {
  1: string message,
  2: i64 errorCode,
}

struct GetFoosRequest {
  1: list<i64> ids;
}

service Bar {
  void ping() throws (1: BarException e),
  Foo getFoos(1: GetFoosRequest request) throws (1: BarException e),
  oneway void fireAndForget(),
  stream Foo getStream(1: GetFoosRequest request)
      throws (1: BarException e1) stream throws (1: FooException e2),
  i32, stream Foo getResponseAndStream(1: GetFoosRequest request)
      throws (1: BarException e1) stream throws (1: FooException e2),
}
```

WARNING: New arguments could be added to a method, but it is better to define an input struct and add members to it instead. The server cannot distinguish between missing arguments and default values, so a request object is better.

*TBD: Need to discuss versioning of services*

#### Exception and RPC Keywords

##### exception

* Fault attribution: *server* vs. *client*
* Error classification: *transient*, *stateful* or *permanent*
* Error safety: *safe* (vs. unspecified)

##### RPC

* RPC idempotency keywords: `readonly`, `idempotent`.

Please refer [Automatic Retries, RPC idempotency and Error Classification](/features/exception.md) for more information on why you should use these keywords and how these combined can enable automatic retries in SR.

#### Streaming

Please refer [Thrift Streaming](/fb/features/streaming/index.md) for more information on Streaming.

### Constants

```grammar
constant_definition ::=  "const" type identifier "=" Constant [";"]
```

A constant definition introduces a name for a value. This name can be used instead of the value after the completion of constant definition, and in other Thrift files that include this Thrift file.

```
Constant ::=
  integer
| float
| string_literal
| bool_literal
| maybe_qualified_id
| ListOrSetConstant
| MapConstant

integer ::=  ["+" | "-"] int_literal
float   ::=  ["+" | "-"] float_literal
```

The constant on the right hand side of `"="` can be:

* A literal
* A name
* A list, set, or map constant

#### Literal Constants

```thrift
const bool flag = true
const byte offset = -10 // byte is a 8-bit *signed* integer
const i32 count = 200
const i16 bits = 0241 // octal value (starts with 0)
const i32 mask = 0xFA12EE // hex value (starts with 0x)
const float pi = 3.1416
const double e = 2.718281828459
const string error = 'unknown protocol'
const string date = "June 28, 2017"
```

#### Name Constants

Name constants can be one of:

* An identifier that denotes a constant defined in the same Thrift file.
* A qualified name of the form `filename.constname` where `filename` and `constname` are identifiers, and `constname` denotes a constant defined in the Thrift file denoted by `filename`.
* A qualified name of the form `enumtype.enum` where `enumtype` and `enum` are identifiers, and `enum` denotes an enumerator defined in the enumeration type `enumtype`.
* A qualified name of the form `filename.enumtype.enum` where `filename`, `enumtype`, and `enum` are identifiers, and `enum` denotes an enumerator defined in the enumeration type `enumtype`, which in turn is defined in the Thrift file denoted by `filename`.

```
const i32 SEARCH_AGGREGATOR_PORT = PORT
const double EULERS_NUMBER = MathConstants.e
const Gender NAN = Gender.MALE
const search_types.Gender NV = search_types.Gender.FEMALE
```

#### List, Set, and Map Constants

```
ListOrSetConstant ::=
  "[" ( Constant delimiter )* "]"

MapConstant ::=
  "{" ( Constant ":" Constant )* "}"
```

Constants can also be of list, set, or map types.

```
list<i32> AList = [2, 3, 5, 7]

set<string> ASet = ["foo", "bar", "baz"]

map<string, list<i32>> AMap = {
  "foo" : [1, 2, 3, 4],
  "bar" : [10, 32, 54],
}
```

NOTE: We cannot write constants of struct, union, or exception types.

## Types

```grammar
type ::=  primitive_type | container_type | maybe_qualified_id
```

Thrift supports primitive, container and named types. The name can be an identifier that denotes a type defined in the same Thrift file, or a qualified name of the form `filename.typename` where `filename` and `typename` are identifiers, and `typename` denotes a type defined in the Thrift file denoted by `filename`.

### Primitive Types

```grammar
primitive_type ::=  "bool" | "byte" | "i16" | "i32" | "i64" |
                    "float" | "double" | "string" | "binary"
```

* `bool`: `true` or `false`
* `byte`: an 8-bit signed integer
* `i16`: a 16-bit signed integer
* `i32`: a 32-bit signed integer
* `i64`: a 64-bit signed integer
* `float`: a 32-bit floating-point number
* `double`: a 64-bit floating-point number
* `string`: a UTF-8 string
* `binary`: a byte array

Thrift does not support unsigned integers because they have no direct translation to native types in some of Thrift’s target languages such as Hack and Java.

`binary` and `string` are encoded identically in the Binary and Compact protocols used for RPC and are interchangeable. However, they are encoded differently in JSON protocols: `binary` is Base64-encoded while `string` only has special characters escaped.

:::caution
Some target languages enforce that `string` values are UTF-8 encoded and others do not. For example, Java and Python require valid UTF-8, while C++ does not. This can manifest itself as a cross-language incompatibility.
:::

### Container Types

```grammar
container_type ::=  list_type | set_type | map_type
list_type      ::=  "list" "<" type ">"
set_type       ::=  "set" "<" type ">"
map_type       ::=  "map" "<" type "," type ">"
```

Thrift has strongly-typed containers that map to commonly used containers in target programming languages. There are three container types available:

* `list<T>`: A list of elements of type `T`. May contain duplicates.
* `set<T>`: An unordered set of unique elements of type `T`.
* `map<K,V>`: An unordered map of unique keys of type `K` to values of type `V`.

:::note
In some languages default mode is to use ordered sets and maps. This could be changed to use unordered and customized containers - see [Thrift Annotations](/idl/annotations.md#unstructured-annotations-deprecated).
:::

:::caution
Although not enforced, it is strongly encouraged to only use set and map when key is either a string or an integer type for the highest compatibility between languages.
:::

The element, key, and value types can be any Thrift type, including nested containers.

## Default Values

```
default_value ::=  "=" Constant
```

All fields of struct types and exception types have a default value. This is either explicitly provided via the syntax shown above, or (if not explicitly provided) is the natural default value for that type. The natural default values for listed below:

* bool: `false`
* byte and integer types: 0
* enum: 0 (even enum has no value for zero)
* float and double: 0.0
* string and binary: empty string
* containers: empty container
* structs: language dependent. Empty struct for C++, null/None for Hack/python/Java

Default values are used as follows:

* To initialize required fields (fields with the qualifier `required`).
* As the value used to serialize an unqualified field (a field with no qualifier) when this field is *not present*.

NOTE: The concepts "required", "unqualified", and "not present" are described in more detail below.

In addition, default values can be used in generated code to provide default behaviors when fields have not been initialized. For example, the method `getValueOrDefault()` can be provided for a field that returns the value of the field when present, and the default value when not present.

```
enum Foo { A = 1, B = 2, C = 3}
struct Person {
 1: i64 age;
 2: string name;
}
struct Bar {
  1: i64 field1 = 10;  // default value is 10
  2: i64 field2;  // default value is 0
  3: map<i32, string> field3 = {15 : 'a_value', 2: 'b_value'};  // default value is the map with two entries
  4: list<Foo> = [Foo.A, Foo.B, Foo.A];  // default value is a list with three entries
  5: Person person = {"age": 40, "name": "John"}; // default value is a struct with these default values
}
```

:::caution
Avoid using default values on optional fields. It is not possible to tell if the server sent the value, or if it is just the default value.
:::

:::caution
Do not change default values after setting them. It is hard to know what code may be relying on the previous default values.
:::

## Qualifiers

```
Qualifier ::=
  "required" | "optional"
```

The fields of a Thrift object may assume any value of their types. In addition the field may also be "uninitialized" (formally "not present"). I.e., the *state* of a field is either *not present*, or *present* with a particular value of their type.

There are four kinds of fields that may be part of a struct type (or exception type):

* ~~required: Field is qualified with the reserved word `required`.~~ (deprecated)
* *optional*: Field is qualified with the reserved word `optional`.
* *terse*: Filed is annotated with the structured annotation `@thrift.TerseWrite`.
* *unqualified*: Field is not qualified with either of these reserved words or `@thrift.TerseWrite`.

NOTE: Union types may only have optional fields.

```
include "thrift/annotation/thrift.thrift"

struct PeopleSearchRequest {
  1: required search_types.Query query; // required field
  2: i32 numResults; // unqualified field
  3: optional Location currentLocation; // optional field
  @thrift.TerseWrite
  4: i32 count; // terse field
}
```

These field kinds are different with respect to the states they may assume, and also with respect to their serialization and deserialization behavior.

### ~~Required Fields~~

:::caution
Do not use in the new code!
:::

~~Required fields are always _present_ and they get initialized to their default value when the object is initialized (or reinitialized). Generated code must provide a way to change the value of required fields. When serializing an object, required fields are always encoded into the serialized data. The serialized data must provide a value during deserialization into a required field.~~

Presence of required fields is not being enforced anymore (Java still does, but that is being removed too).

### Optional Fields

Optional fields start their lifecycle as *not present* (when the object is initialized or reinitialized). Generated code must provide a way to introduce the field (change state from *not present* to *present*) and to reinitialize the field (change state from *present* to *not present*). Generated code must also provide a way to change the value of optional fields. When serializing an object, optional fields are encoded into the serialized data if and only if they are *present*. During deserialization, the serialized data may or may not provide a value for an optional field. If a value is provided, optional field becomes *present* with that value, otherwise the optional field does not change its state.

### Unqualified Fields

Unqualified fields are just like optional fields with the following difference: When an object with an unqualified field is serialized and the field is *not present*, the default value of the field is serialized as the value for that field.

:::note
Regarding unqualified vs. optional field, we don't have a generalized strong recommendation on which one should be used. It depends on individual use cases. If you need to know whether the field is explicit set or not, use optional fields.
:::

### Terse Fields

Terse fields do not distinguish whether they are explicitly set. They are skipped during serialization if their values are equal to the [intrinsic default values](./#intrinsic-default-values).
