/*
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#pragma once

#include <type_traits>
#include <utility>

#include <folly/FBString.h>
#include <folly/Traits.h>
#include <thrift/compiler/ast/t_const_value.h>
#include <thrift/compiler/ast/t_enum.h>
#include <thrift/compiler/ast/t_service.h>
#include <thrift/compiler/ast/t_structured.h>
#include <thrift/lib/cpp2/visitation/for_each.h>
#include <thrift/lib/thrift/gen-cpp2/protocol_types.h>

namespace apache {
namespace thrift {
namespace compiler {

// Assigns a t_const_value to a concrete Thrift type.
// Must include generated _visitation.h header for that type.

inline void hydrate_const(bool& out, const t_const_value& val) {
  out = val.get_bool();
}
template <typename T>
std::enable_if_t<std::is_integral_v<T>> hydrate_const(
    T& out, const t_const_value& val) {
  out = val.get_integer();
}
template <typename T>
std::enable_if_t<std::is_floating_point_v<T>> hydrate_const(
    T& out, const t_const_value& val) {
  out = val.get_double();
}
inline void hydrate_const(std::string& out, const t_const_value& val) {
  out = val.get_string();
}
inline void hydrate_const(folly::fbstring& out, const t_const_value& val) {
  out = val.get_string();
}
template <typename T> // list
folly::void_t<decltype(std::declval<T>().emplace_back())> hydrate_const(
    T& out, const t_const_value& val) {
  for (auto* item : val.get_list()) {
    auto& entry = out.emplace_back();
    hydrate_const(entry, *item);
  }
}
template <typename T> // set
std::enable_if_t<std::is_same_v<typename T::key_type, typename T::value_type>>
hydrate_const(T& out, const t_const_value& val) {
  for (auto* item : val.get_list()) {
    typename T::key_type value;
    hydrate_const(value, *item);
    out.emplace(std::move(value));
  }
}
template <typename T> // map
folly::void_t<typename T::mapped_type> hydrate_const(
    T& out, const t_const_value& val) {
  for (const auto& pair : val.get_map()) {
    typename T::key_type key;
    hydrate_const(key, *pair.first);
    typename T::mapped_type value;
    hydrate_const(value, *pair.second);
    out.emplace(std::move(key), std::move(value));
  }
}
template <typename T>
std::enable_if_t<std::is_enum_v<T>> hydrate_const(
    T& out, const t_const_value& val) {
  out = static_cast<T>(val.get_integer());
}
template <typename T>
std::enable_if_t<is_thrift_class_v<T>> hydrate_const(
    T& out, const t_const_value& val) {
  assert(val.get_type() == t_const_value::t_const_value_type::CV_MAP);
  std::unordered_map<std::string, t_const_value*> map;
  for (const auto& pair : val.get_map()) {
    map[pair.first->get_string()] = pair.second;
  }

  for_each_field(out, [&](const metadata::ThriftField& meta, auto field_ref) {
    if (!map.count(*meta.name())) {
      return;
    }

    hydrate_const(field_ref.ensure(), *map.at(*meta.name()));
  });
}
template <typename T>
folly::void_t<decltype(std::declval<T>().toThrift())> hydrate_const(
    T& out, const t_const_value& val) {
  hydrate_const(out.toThrift(), val);
}

// Assigns a t_const_value to a Value.
// Currently only uses bool/i64/double/string/list/map.
// TODO: allow increasing type fidelity.
inline protocol::Value const_to_value(const t_const_value& val) {
  protocol::Value ret;
  auto type = val.ttype() ? val.ttype()->get_type_value() : [&] {
    switch (val.get_type()) {
      case t_const_value::CV_BOOL:
        return t_type::type::t_bool;
      case t_const_value::CV_INTEGER:
        return t_type::type::t_i64;
      case t_const_value::CV_DOUBLE:
        return t_type::type::t_double;
      case t_const_value::CV_STRING:
        return t_type::type::t_string;
      case t_const_value::CV_LIST:
        return t_type::type::t_list;
      case t_const_value::CV_MAP:
        return t_type::type::t_map;
    }
  }();
  switch (type) {
    case t_type::type::t_bool:
      ret.emplace_bool();
      if (val.get_type() == t_const_value::CV_BOOL) {
        ret.as_bool() = val.get_bool();
      } else if (val.get_type() == t_const_value::CV_INTEGER) {
        auto value = val.get_integer();
        assert(value == 0 || value == 1);
        ret.as_bool() = value;
      }
      break;
    case t_type::type::t_byte:
      ret.emplace_byte();
      ret.as_byte() = val.get_integer();
      break;
    case t_type::type::t_i16:
      ret.emplace_i16();
      ret.as_i16() = val.get_integer();
      break;
    case t_type::type::t_i32:
      ret.emplace_i32();
      ret.as_i32() = val.get_integer();
      break;
    case t_type::type::t_i64:
      ret.emplace_i64();
      ret.as_i64() = val.get_integer();
      break;
    case t_type::type::t_float:
      ret.emplace_float();
      ret.as_float() = val.get_double();
      break;
    case t_type::type::t_double:
      ret.emplace_double();
      ret.as_double() = val.get_double();
      break;
    case t_type::type::t_string:
      ret.emplace_string();
      ret.as_string() = val.get_string();
      break;
    case t_type::type::t_binary:
      ret.emplace_binary();
      ret.as_binary() =
          folly::IOBuf(folly::IOBuf::CopyBufferOp{}, val.get_string());
      break;
    case t_type::type::t_list:
      ret.emplace_list();
      for (const auto& list_elem : val.get_list()) {
        ret.as_list().push_back(const_to_value(*list_elem));
      }
      break;
    case t_type::type::t_set:
      ret.emplace_set();
      for (const auto& list_elem : val.get_list()) {
        ret.as_set().insert(const_to_value(*list_elem));
      }
      break;
    case t_type::type::t_map:
      ret.emplace_map();
      for (const auto& map_elem : val.get_map()) {
        ret.as_map().emplace(
            const_to_value(*map_elem.first), const_to_value(*map_elem.second));
      }
      break;
    case t_type::type::t_enum:
      ret.emplace_i32();
      ret.as_i32() = val.get_integer();
      break;
    case t_type::type::t_struct:
      // TODO: maybe translate to Object
      ret.emplace_map();
      for (const auto& map_elem : val.get_map()) {
        ret.as_map().emplace(
            const_to_value(*map_elem.first), const_to_value(*map_elem.second));
      }
      break;
    case t_type::type::t_void:
    case t_type::type::t_service:
    case t_type::type::t_stream:
    case t_type::type::t_sink:
    case t_type::type::t_program:
      throw std::runtime_error("Unexpected type");
  }
  return ret;
}

} // namespace compiler
} // namespace thrift
} // namespace apache
