/*
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#pragma once

#include <map>
#include <memory>
#include <string>
#include <unordered_set>

#include <boost/filesystem.hpp>

#include <thrift/compiler/ast/t_program.h>
#include <thrift/compiler/validator/validator.h>

namespace apache {
namespace thrift {
namespace compiler {

class t_generation_context {
 public:
  t_generation_context() : out_path_("./"), is_out_path_absolute_(false) {}
  t_generation_context(
      std::string out_path,
      bool is_out_path_absolute,
      source_manager* sm = nullptr);

  const std::string& get_out_path() const { return out_path_; }

  bool is_out_path_absolute() const { return is_out_path_absolute_; }

  const source_manager* get_source_manager() const { return source_mgr_; }
  void set_source_manager(source_manager& sm) { source_mgr_ = &sm; }

 private:
  std::string out_path_;
  bool is_out_path_absolute_;
  source_manager* source_mgr_ = nullptr;
};

class t_generator {
 public:
  t_generator(t_program* program, t_generation_context context);
  virtual ~t_generator() = default;

  virtual void fill_validator_list(validator_list&) const {}

  // Generate the program. Overridden by subclasses to implement program
  // generation.
  virtual void generate_program() = 0;

  /**
   * Method to get the program name, may be overridden
   */
  virtual std::string get_program_name(t_program* program) {
    return program->name();
  }

  const t_program* get_program() const { return program_; }

  std::unordered_set<std::string> get_genfiles() { return generated_files_; }

  void record_genfile(const std::string& filename) {
    generated_files_.insert(filename);
  }
  void record_genfile(const boost::filesystem::path& filename) {
    generated_files_.insert(filename.string());
  }

  /**
   * Get the current output directory
   */
  virtual std::string get_out_dir() const { return get_out_path().string(); }
  virtual boost::filesystem::path get_out_path() const {
    auto path = boost::filesystem::path{context_.get_out_path()};
    if (!context_.is_out_path_absolute()) {
      path /= out_dir_base_;
    }
    path += boost::filesystem::path::preferred_separator;
    return path;
  }

 protected:
  t_program* program_;

  t_generation_context context_;

  /**
   * Output type-specifc directory name ("gen-*").
   */
  std::string out_dir_base_;

  /**
   * Quick accessor for formatted program name that is currently being
   * generated.
   */
  std::string program_name_;

  /**
   * The set of files generated by this generator.
   */
  std::unordered_set<std::string> generated_files_;
};

/**
 * A factory for producing generator classes of a particular language.
 *
 * This class is also responsible for:
 *  - Registering itself with the generator registry.
 *  - Providing documentation for the generators it produces.
 */
class generator_factory {
 public:
  generator_factory(
      std::string name, std::string long_name, std::string documentation);

  virtual ~generator_factory() = default;

  virtual std::unique_ptr<t_generator> make_generator(
      t_program& program, // The program to generate.
      t_generation_context context,
      const std::map<std::string, std::string>& options) = 0;

  const std::string& name() const { return name_; }
  const std::string& long_name() const { return long_name_; }
  const std::string& documentation() const { return documentation_; }

 private:
  std::string name_;
  std::string long_name_;
  std::string documentation_;
};

namespace detail {
template <typename Generator>
class generator_factory_impl : public generator_factory {
 public:
  using generator_factory::generator_factory;

  std::unique_ptr<t_generator> make_generator(
      t_program& program,
      t_generation_context context,
      const std::map<std::string, std::string>& options) override {
    return std::unique_ptr<t_generator>(
        new Generator(&program, context, options));
  }
};
} // namespace detail

namespace generator_registry {

void register_generator(const std::string& name, generator_factory* factory);

std::unique_ptr<t_generator> make_generator(
    const std::string& name,
    t_program& program,
    t_generation_context context,
    const std::map<std::string, std::string>& options);

// A map from generator names to factories.
using generator_map = std::map<std::string, generator_factory*>;
generator_map& get_generators();

} // namespace generator_registry

#define THRIFT_REGISTER_GENERATOR(name, long_name, doc)                   \
  static detail::generator_factory_impl<t_##name##_generator> registerer( \
      #name, long_name, doc)

} // namespace compiler
} // namespace thrift
} // namespace apache
