/* -----------------------------------------------------------------------------
 * parsespnego.c parses an RFC 2743 GSS-API InitialContextToken containing an
 * RFC 2478 SPNEGO NegotiationToken.
 *
 * Author: Frank Balluffi
 *
 * Copyright (C) 2002-2005 Frank Balluffi. All rights reserved.
 * -----------------------------------------------------------------------------
 */

#include "../../include/spnegohelp.h"
#include "../../include/filehelp.h"

#include <openssl/objects.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

static int parseSpnegoInitial (FILE *       stream,
                               const char * spnegoTokenFile,
                               const char * mechType,
                               const char * mechTokenFile);
static int parseSpnegoTarget (FILE *       stream,
                              const char * spnegoTokenFile,
                              const char * responseTokenFile);
static void printHelp (FILE * stream);

/* -----------------------------------------------------------------------------
 * main is the main entry point.
 *
 * Returns 1 if successful, 0 otherwise.
 * -----------------------------------------------------------------------------
 */

int main (int     argc,
          char ** argv)
{
    if (argc < 2)
    {
        printHelp (stdout);
        return 0;
    }

    if (!strcmp (argv [1], "init"))
    {
        if (argc < 5)
        {
            printHelp (stdout);
            return 0;
        }

        if (!parseSpnegoInitial (stdout,
                                argv [2],
                                argv [3],
                                argv [4]))
        {
            printf ("parsespnego failed.\n");
            return 0;
        }
    }
    else if (!strcmp (argv [1], "targ"))
    {
        if (argc < 4)
        {
            printHelp (stdout);
            return 0;
        }

        if (!parseSpnegoTarget (stdout,
                                argv [2],
                                argv [3]))
        {
            printf ("parsespnego failed.\n");
            return 0;
        }
    }
    else
    {
        printf ("Unrecognized token choice \'%s\'.\n", argv [1]);
        return 0;
    }

    return 1;
}

/* -----------------------------------------------------------------------------
 * parseSpnegoInitial parses an RFC 2743 GSS-API InitialContextToken containing
 * an RFC 2478 SPNEGO NegotiationToken of choice negTokenInit, and searches for
 * a mechanism type and extracts the mechanism token.
 *
 * Returns 1 if successful, 0 otherwise.
 * -----------------------------------------------------------------------------
 */

static int parseSpnegoInitial (FILE *       stream,
                               const char * spnegoTokenFile,
                               const char * mechType,
                               const char * mechTokenFile)
{
    unsigned char * mechToken         = NULL;
    size_t          mechTokenLength   = 0;
    ASN1_OBJECT *   object            = NULL;
    int             rc                = 1;
    unsigned char * spnegoToken       = NULL;
    size_t          spnegoTokenLength = 0;

    if (!stream          ||
        !spnegoTokenFile ||
        !mechType        ||
        !mechTokenFile)
        return 0;

    if (!(spnegoToken = fileToMemory (spnegoTokenFile,
                                      &spnegoTokenLength)))
    {
        fprintf (stream, "Unable to read the file \'%s\'.\n", spnegoTokenFile);
        rc = 0;
        goto cleanup;
    }

    object = OBJ_txt2obj (mechType, 1);

    if (!object)
    {
        fprintf (stream, "Unable to parse mechanism type \'%s\'.\n", mechType);
        rc = 0;
        goto cleanup;
    }

    if (!parseSpnegoInitialToken (spnegoToken,
                                  spnegoTokenLength,
                                  object,
                                  &mechToken,
                                  &mechTokenLength))
    {
        fprintf (stream, "Unable to parse SPNEGO initial token.\n");
        rc = 0;
        goto cleanup;
    }

    if (!memoryToFile (mechTokenFile,
                       mechToken,
                       mechTokenLength))
    {
        fprintf (stream, "Unable to write the file \'%s\'.\n", mechTokenFile);
        rc = 0;
        goto cleanup;
    }

cleanup:

    if (mechToken)
        free (mechToken);

    if (object)
        ASN1_OBJECT_free (object);

    if (spnegoToken)
        free (spnegoToken);

    return rc;
}

/* -----------------------------------------------------------------------------
 * parseSpnegoTarget parses an RFC 2743 GSS-API InitialContextToken containing
 * an RFC 2478 SPNEGO NegotiationToken of choice negTokenTarg and extracts the
 * response token.
 *
 * Returns 1 if successful, 0 otherwise.
 * -----------------------------------------------------------------------------
 */

static int parseSpnegoTarget (FILE *       stream,
                              const char * spnegoTokenFile,
                              const char * responseTokenFile)
{
    int             rc                  = 1;
    unsigned char * responseToken       = NULL;
    size_t          responseTokenLength = 0;
    unsigned char * spnegoToken         = NULL;
    size_t          spnegoTokenLength   = 0;

    if (!stream          ||
        !spnegoTokenFile ||
        !responseTokenFile)
        return 0;

    if (!(spnegoToken = fileToMemory (spnegoTokenFile,
                                      &spnegoTokenLength)))
    {
        fprintf (stream, "Unable to read the file \'%s\'.\n", spnegoTokenFile);
        rc = 0;
        goto cleanup;
    }

    if (!parseSpnegoTargetToken (spnegoToken,
                                 spnegoTokenLength,
                                 NULL,
                                 NULL,
                                 &responseToken,
                                 &responseTokenLength,
                                 NULL,
                                 NULL))
    {
        fprintf (stream, "Unable to parse SPNEGO target token.\n");
        rc = 0;
        goto cleanup;
    }

    if (!memoryToFile (responseTokenFile,
                       responseToken,
                       responseTokenLength))
    {
        fprintf (stream, "Unable to write the file \'%s\'.\n", responseTokenFile);
        rc = 0;
        goto cleanup;
    }

cleanup:

    if (responseToken)
        free (responseToken);

    if (spnegoToken)
        free (spnegoToken);

    return rc;
}

/* -----------------------------------------------------------------------------
 * printHelp prints help.
 *
 * Returns nothing.
 * -----------------------------------------------------------------------------
 */

static void printHelp (FILE * stream)
{
    static char help [] = "usage: parsespnego init | targ arguments\n\n"
                          "init | targ     token choice: negTokenInit or negTokenTarg\n\n"
                          "if init, arguments are:\n\n"
                          "spnego-token    file containing SPNEGO GSS-API token\n"
                          "mech-type       period-delimited mechanism type to search for\n"
                          "mech-token      file to contain mechanism token\n\n"
                          "if targ, arguments are:\n\n"
                          "spnego-token    file containing SPNEGO GSS-API token\n"
                          "response-token  file to contain response token\n";

    if (!stream)
        return;

    fprintf (stream, help);
    return;
}
