import type { SvelteComponent } from "svelte";
import { get, writable } from "svelte/store";

import { shallow_equal } from "../lib/equals";
import { delegate } from "../lib/events";
import router from "../router";
import { sortableJournal } from "../sort";
import { favaOptions } from "../stores";
import { fql_filter } from "../stores/filters";

import JournalFilters from "./JournalFilters.svelte";

/**
 * Escape the value to produce a valid regex.
 */
function escape(value: string): string {
  return value.replace(/[.*+\-?^${}()|[\]\\]/g, "\\$&");
}

/**
 * Add a filter to the existing list of filters. Any parts that are interpreted
 * as a regex must be escaped.
 */
function addFilter(value: string): void {
  fql_filter.update((fql_filter_val) =>
    fql_filter_val ? `${fql_filter_val} ${value}` : value
  );
}

function handleClick({ target }: MouseEvent): void {
  if (!(target instanceof HTMLElement) || target instanceof HTMLAnchorElement) {
    return;
  }

  if (target.className === "tag" || target.className === "link") {
    // Filter for tags and links when clicking on them.
    addFilter(target.innerText);
  } else if (target.className === "payee") {
    // Filter for payees when clicking on them.
    // Note: any special characters in the payee string are escaped so the
    // filter matches against the payee literally.
    addFilter(`payee:"^${escape(target.innerText)}$"`);
  } else if (target.tagName === "DT") {
    // Filter for metadata key when clicking on the key. The key tag text
    // includes the colon.
    const expr = `${target.innerText}""`;
    if (target.closest(".postings")) {
      // Posting metadata.
      addFilter(`any(${expr})`);
    } else {
      // Entry metadata.
      addFilter(expr);
    }
  } else if (target.tagName === "DD") {
    // Filter for metadata key and value when clicking on the value. The key
    // tag text includes the colon.
    const key = (target.previousElementSibling as HTMLElement).innerText;
    const value = `"^${escape(target.innerText)}$"`;
    const expr = `${key}${value}`;
    if (target.closest(".postings")) {
      // Posting metadata.
      addFilter(`any(${expr})`);
    } else {
      // Entry metadata.
      addFilter(expr);
    }
  } else if (target.closest(".indicators")) {
    // Toggle postings and metadata by clicking on indicators.
    const entry = target.closest(".transaction");
    if (entry) {
      entry.classList.toggle("show-postings");
    }
  }
}

export class FavaJournal extends HTMLElement {
  component?: SvelteComponent;

  unsubscribe?: () => void;

  connectedCallback() {
    const opts = get(favaOptions);
    const defaults = [
      ...opts.journal_show,
      ...opts.journal_show_transaction,
      ...opts.journal_show_document,
    ].sort();

    const ol = this.querySelector("ol");
    if (!ol) {
      throw new Error("fava-journal is missing its <ol> or <form>");
    }

    const url_show = new URL(window.location.href).searchParams.getAll("show");
    const show = writable(new Set(url_show.length ? url_show : defaults));
    this.unsubscribe = show.subscribe((show_value) => {
      const classes = [...show_value].map((s) => `show-${s}`).join(" ");
      ol.className = `flex-table journal ${classes}`;

      const url = new URL(window.location.href);
      url.searchParams.delete("show");
      if (!shallow_equal([...show_value].sort(), defaults)) {
        show_value.forEach((filter) => {
          url.searchParams.append("show", filter);
        });
      }
      router.navigate(url.toString(), false);
    });
    this.component = new JournalFilters({
      target: this,
      props: { show },
      anchor: ol,
    });

    sortableJournal(ol);
    delegate(this, "click", "li", handleClick);
  }

  disconnectedCallback(): void {
    this.unsubscribe?.();
    this.component?.$destroy();
  }
}
