#!/usr/local/bin/python
#
# $Id: //projects/fauxident/fauxident.py#7 $ $Date: 2003/10/21 $

"""
A faux identd server.
"""

__program__ = 'fauxident'
__version__ = '1.2.1'
__author__ = 'Erik Max Francis <max@alcyone.com>'
__copyright__ = 'Copyright (C) 2002 Erik Max Francis'
__license__ = 'GPL'


import getopt
import os
import random
import socket
import sys
import types

import asyncore
import asynchat


MAX_BUFFER_SIZE = 256

# Defaults.
DEFAULT_PORT = 113
DEFAULT_REALM = 'UNIX'
DEFAULT_USER = 'user'

# The different errors.
INVALID_PORT, NO_USER, HIDDEN_USER, UNKNOWN_ERROR = \
     'INVALID-PORT', 'NO-USER', 'HIDDEN-USER', 'UNKNOWN-ERROR'


class Responder:

    def __init__(self):
        if self.__class__ is Responder:
            raise NotImplementedError
        
    def check(self, clientPort, serverPort):
        raise NotImplementedError

class FailureResponder(Responder):

    def __init__(self, errorType=NO_USER):
        Responder.__init__(self)
        self.errorType = errorType

    def check(self, clientPort, serverPort):
        return ['ERROR', self.errorType]

class SuccessResponder(Responder):

    def __init__(self, realm=DEFAULT_REALM, users=DEFAULT_USER, suffix=0, \
                 permute=0):
        Responder.__init__(self)
        self.realm = realm
        if type(users) is types.StringType:
            self.users = [users]
        else:
            self.users = users
        self.suffix = suffix
        self.permute = permute

    def chooseUser(self):
        if len(self.users) == 1:
            user = self.users[0]
        else:
            user = random.choice(self.users)
        if self.permute:
            letters = list(user)
            user = ''
            while letters:
                i = random.randrange(len(letters))
                user += letters[i]
                del letters[i]
        return user

    def chooseSuffix(self):
        if self.suffix > 0:
            number = random.randrange(10**self.suffix)
            return ('%%0%dd' % self.suffix) % number
        else:
            return ''

    def check(self, clientPort, serverPort):
        response = self.chooseUser() + self.chooseSuffix()
        return ['USERID', self.realm, response]


class Connection(asynchat.async_chat):

    def __init__(self, server, (sock, addr)):
        asynchat.async_chat.__init__(self, sock)
        self.server = server
        self.set_terminator('\r\n')
        self.buffer = ''

    def collect_incoming_data(self, data):
        self.buffer += data
        if len(self.buffer) > MAX_BUFFER_SIZE:
            self.respond(0, 0, ['ERROR', UNKNOWN_ERROR])
            self.close_when_done()

    def handle_close(self):
        self.close()

    def handle_error(self):
        self.close()
        raise

    def found_terminator(self):
        data, self.buffer = self.buffer, ''
        data = data.strip()
        if data.find(',') >= 0:
            try:
                clientPort, serverPort = data.split(',', 1)
                clientPort = int(clientPort.strip())
                serverPort = int(serverPort.strip())
                if 0 <= clientPort < 65536 and 0 <= serverPort < 65536:
                    self.succeed(clientPort, serverPort)
                else:
                    self.respond(clientPort, serverPort, \
                                 ['ERROR', INVALID_PORT])
            except ValueError:
                self.respond(0, 0, ['ERROR', INVALID_PORT])
        else:
            self.respond(0, 0, ['ERROR', UNKNOWN_ERROR])

    def respond(self, clientPort, serverPort, response):
        self.push('%d , %d : %s\r\n' % \
                  (clientPort, serverPort, ' : '.join(response)))
        self.close_when_done()

    def succeed(self, clientPort, serverPort):
        response = self.server.responder.check(clientPort, serverPort)
        self.respond(clientPort, serverPort, response)


class Server(asyncore.dispatcher):

    ConnectionFactory = Connection

    def __init__(self, address=('', DEFAULT_PORT), responder=None):
        asyncore.dispatcher.__init__(self)
        self.create_socket(socket.AF_INET, socket.SOCK_STREAM)
        self.set_reuse_addr()
        self.bind(address)
        self.listen(5)
        if responder is None:
            responder = Responder()
        self.responder = responder

    def handle_error(self):
        self.close()
        raise

    def handle_accept(self):
        self.ConnectionFactory(self, self.accept())


def daemonize():
    class NullDevice:
        def write(self, data):
            pass
    if os.fork() != 0:
        os._exit(0)
    os.setsid()
    os.umask(0)
    sys.stdin.close()
    nullDevice = NullDevice()
    sys.stdout = sys.__stdout__ = nullDevice
    sys.stderr = sys.__stderr__ = nullDevice

def usage():
    def say(message):
        print >> sys.stderr, message
    say("Usage: %s [options]" % sys.argv[0])
    say("")
    say("Valid options:")
    say("  -V --version                 print version and exit")
    say("  -h --help                    print usage and exit")
    say("  -a --address=<IP address>    bind to a specific interface []")
    say("  -p --port=<port>             bind to the specified port [113]")
    say("  -f --foreground              do not daemonize; run in foreground")
    say("")
    say("The following options specify the response that is made:")
    say("  -u --user=<user>             respond with specified user [user]")
    say("  -s --suffix=<number>         respond with suffixed random number")
    say("  -m --permute                 permute the user name each time")
    say("  -r --realm=<realm>           respond with specified realm [UNIX]")
    say("  -e -E --error=<error type>   respond with specified error")
    say("  -N --no-user                 respond with NO-USER error")
    say("  -H --hidden-user             respond with HIDDEN-USER error")

def main():
    # Initialize the options.
    _host = ''
    _port = DEFAULT_PORT
    _realm = DEFAULT_REALM
    _users = DEFAULT_USER
    _suffix = 0
    _permute = 0
    _daemonize = 1
    _responder = None
    # Parse the arguments.
    pairs, remainder = getopt.getopt(sys.argv[1:], 'Vha:p:e:E:NHr:u:s:mf', ['version', 'help', 'address=', 'port=', 'error=', 'no-user', 'hidden-user', 'realm=', 'user=', 'suffix=', 'permute', 'foreground'])
    for option, argument in pairs:
        if option in ('-v', '-V', '--version'):
            print >> sys.stderr, "%s version %s" % (__program__, __version__)
            return
        elif option in ('-h', '--help'):
            usage()
            return
        elif option in ('-a', '--address'):
            _host = argument
        elif option in ('-p', '--port'):
            _port = int(argument)
        elif option in ('-e', '-E', '--error'):
            _responder = FailureResponder(argument)
        elif option in ('-N', '--no-user'):
            _responder = FailureResponder(NO_USER)
        elif option in ('-H', '--hidden-user'):
            _responder = FailureResponder(HIDDEN_USER)
        elif option in ('-r', '--realm'):
            _realm = argument
        elif option in ('-u', '--user'):
            _users = argument.split(',')
        elif option in ('-s', '--suffix'):
            _suffix = int(argument)
        elif option in ('-m', '--permute'):
            _permute = 1
        elif option in ('-f', '--foreground'):
            _daemonize = 0
    if remainder:
        raise ValueError, "unexpected arguments on command line"
    # Start the server.
    if _responder is None:
        _responder = SuccessResponder(_realm, _users, _suffix, _permute)
    server = Server((_host, _port), _responder)
    if _daemonize:
        daemonize()
    asyncore.loop()

if __name__ == '__main__': main()
