/*
  Copyright(C) 2007-2012 National Institute of Information and Communications Technology
*/

/*
  svmtools
  Kernel module
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <limits.h>
#include "exception.h"
#include "svm_common.h"
#include "svm_kernel.h"


#define CACHEMASK (UINT_MAX / 2U + 1U)	/* $B%"%/%;%9$5$l$?$3$H$r<($9%U%i%0(B */


static void kernel_linear(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *x, float *q);
static void kernel_polynomial(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *x, float *q);
static void kernel_rbf(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *x, float *q);
static void kernel_sigmoid(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *x, float *q);
static void kernel_polynomial_1o(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *x, float *q);
static void kernel_polynomial_2o(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *x, float *q);
static void kernel_polynomial_3o(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *x, float *q);
static void kernel_polynomial_4o(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *x, float *q);
static void kernel_polynomial_5o(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *x, float *q);
static void dotprod(SVM_KWRK *kwrk, SVM_SV *x, float *q);


/* $B%+!<%M%k$N4X?t%]%$%s%?(B($BFb@Q$N%Y%/%H%k$rF~NO$7$F%+!<%M%k$N%Y%/%H%k$r=PNO(B) */
static void (*kernel[])(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *v, float *q) = {
  kernel_linear,
  kernel_polynomial,
  kernel_rbf,
  kernel_sigmoid,
  kernel_polynomial_1o,
  kernel_polynomial_2o,
  kernel_polynomial_3o,
  kernel_polynomial_4o,
  kernel_polynomial_5o,
};


/*
  $B%-%c%C%7%e$N=i4|2=(B
*/
SVM_KC *svm_knl_kcinit(SVM_KPRM *kprm, int num, int *label, SVM_SV **sv, int size) {
  int i;
  SVM_KC *kc;

  kc = smalloc(sizeof(SVM_KC));
  kc->kprm = *kprm;
  kc->col = num;
  kc->row = (size * 1024 * 1024) / (sizeof(float) * kc->col);
  if (kc->row < 2) kc->row = 2;
  if (kc->row > num) kc->row = num;
  kc->size = kc->col * kc->row;
  kc->cache = smalloc(sizeof(float) * kc->col * kc->row);
  kc->e2r = smalloc(sizeof(int) * num);
  for (i = 0; i < num; i++) kc->e2r[i] = -1;
  kc->r2e = smalloc(sizeof(unsigned int) * num);
  for (i = 0; i < num; i++) kc->r2e[i] = 0U;
  kc->nxt = 0;
  kc->gen = smalloc(sizeof(int) * num);
  kc->hst_num = 0;
  kc->hst = smalloc(sizeof(int) * num);
  kc->label = smalloc(sizeof(int) * num);
  for (i = 0; i < num; i++) kc->label[i] = label[i];

  /* $B:n6HMQ%G!<%?$N=i4|2=(B */
  kc->kwrk = svm_knl_kwrkinit(kprm, num, sv);

  return kc;
}


/*
  $B%-%c%C%7%e>pJs$N:o=|(B
*/
void svm_knl_kcdelete(SVM_KC *kc) {
  svm_knl_kwrkdelete(kc->kwrk);
  free(kc->label);
  free(kc->hst);
  free(kc->gen);
  free(kc->r2e);
  free(kc->e2r);
  free(kc->cache);
  free(kc);
}


/*
  $B%-%c%C%7%e$r7PM3$7$F%+!<%M%k7W;;(B
*/
float *svm_knl_kckernel(SVM_KC *kc, int p, float *rsv) {
  int i, j;
  int row;
  float *cache;
  int *l, *l_end;
  float *c;

  row = kc->e2r[p];
  if (row != -1) {	/* cache hit */
    kc->r2e[row] |= CACHEMASK;	/* $B%U%i%0$r%;%C%H(B */
    cache = kc->cache + kc->col * row;

    /* $B@$Be$ND4@0(B($B7j$rKd$a$k(B) */
    for (i = kc->gen[row], j = kc->kwrk->num + (kc->hst_num - kc->gen[row]) - 1; i < kc->hst_num; i++, j--) {
      cache[kc->hst[i]] = cache[j];
    }
    kc->gen[row] = kc->hst_num;

    return cache;
  } else {	/* cache miss */
    /* $B<N$F$k%-%c%C%7%e$r8+IU$1$k(B(Clock algorithm) */
    for (row = kc->nxt; ; row++) {
      if (row >= kc->row) row = 0;
      if ((kc->r2e[row] & CACHEMASK) == 0U && kc->cache + kc->col * row != rsv) break;
      /* $B%U%i%0$r%j%;%C%H(B */
      kc->r2e[row] &= ~CACHEMASK;
    }
    cache = kc->cache + kc->col * row;
    kc->nxt = row + 1;
    if (kc->r2e[row] < (unsigned int)kc->kwrk->num && kc->e2r[kc->r2e[row]] == row) kc->e2r[kc->r2e[row]] = -1;	/* kc->r2e[row] >= 0 */
    kc->e2r[p] = row;
    kc->r2e[row] = (unsigned int)p;
    kc->gen[row] = kc->hst_num;

    /* sqnorm$B$N7W;;(B */
    if (kc->kprm.ktype == SVM_KERNEL_RBF) {
      kc->kwrk->sqnorm2 = kc->kwrk->sqnorm[p];
    }

    /* $B?75,7W;;(B */
    (kernel[kc->kwrk->kfunc])(&kc->kprm, kc->kwrk, kc->kwrk->sv[p], cache);
    if (kc->label[p] > 0) {
      for (l = kc->label, l_end = l + kc->kwrk->num, c = cache; l < l_end; l++, c++) if (*l < 0) *c = -*c;
    } else {
      for (l = kc->label, l_end = l + kc->kwrk->num, c = cache; l < l_end; l++, c++) if (*l > 0) *c = -*c;
    }

    return cache;
  }
}


/*
  $B;XDj$5$l$?;vNc(Bid$B$r%-%c%C%7%e>e$GL58z$K$9$k(B
*/
void svm_knl_kchide(SVM_KC *kc, int p) {
  /* $BL58z$K$J$C$?%$%s%G%C%/%9$r%R%9%H%j$K5-O?(B */
  kc->hst[kc->hst_num++] = p;

  /* $B$=$NB>$N0\F0(B */
  if (kc->kwrk->sqnorm != NULL) kc->kwrk->sqnorm[p] = kc->kwrk->sqnorm[kc->kwrk->num - 1];
  kc->kwrk->sv[p] = kc->kwrk->sv[kc->kwrk->num - 1];
  kc->label[p] = kc->label[kc->kwrk->num - 1];
  kc->e2r[p] = kc->e2r[kc->kwrk->num - 1];
  if (kc->e2r[p] != -1) {
    kc->r2e[kc->e2r[p]] &= CACHEMASK;
    kc->r2e[kc->e2r[p]] |= (unsigned int)p;
  }
  kc->kwrk->num--;

  return;
}

/*
  $B%-%c%C%7%e$rA}$d$9(B
*/
void svm_knl_kcexpand(SVM_KC *kc) {
  int i, j, k;

  if ((kc->col - kc->kwrk->num) * kc->row >= kc->kwrk->num && kc->row < kc->kwrk->num) {
    for (i = 0; i < kc->row; i++) {
      float *cache;

      if (kc->e2r[kc->r2e[i] & ~CACHEMASK] != i) continue;	/* $BL$;HMQ(B */

      cache = kc->cache + kc->col * i;
      /* $B@$Be$ND4@0(B($B7j$rKd$a$k(B) */
      for (j = kc->gen[i], k = kc->kwrk->num + (kc->hst_num - kc->gen[i]) - 1; j < kc->hst_num; j++, k--) {
	cache[kc->hst[j]] = cache[k];
      }
      kc->gen[i] = kc->hst_num;

      /* $B0\F0(B */
      if (i != 0) memmove(kc->cache + kc->kwrk->num * i, cache, sizeof(float) * kc->kwrk->num);
    }
    kc->col = kc->kwrk->num;
    kc->row = kc->size / kc->col;
    if (kc->row > kc->kwrk->num) kc->row = kc->kwrk->num;
  }

  return;
}


/*
  $B%-%c%C%7%e$N:n6HMQ>pJs$N=i4|2=(B
*/
SVM_KWRK *svm_knl_kwrkinit(SVM_KPRM *kprm, int num, SVM_SV **sv) {
  int i, j;
  SVM_KWRK *kwrk;
  int max;

  kwrk = smalloc(sizeof(SVM_KWRK));

  if (kprm->ktype == SVM_KERNEL_POLYNOMIAL) {
    if (kprm->degree == 1.0f) kwrk->kfunc = SVM_KFUNC_POLYNOMIAL_1O;
    else if (kprm->degree == 2.0f) kwrk->kfunc = SVM_KFUNC_POLYNOMIAL_2O;
    else if (kprm->degree == 3.0f) kwrk->kfunc = SVM_KFUNC_POLYNOMIAL_3O;
    else if (kprm->degree == 4.0f) kwrk->kfunc = SVM_KFUNC_POLYNOMIAL_4O;
    else if (kprm->degree == 5.0f) kwrk->kfunc = SVM_KFUNC_POLYNOMIAL_5O;
    else kwrk->kfunc = kprm->ktype;
  } else {
    kwrk->kfunc = kprm->ktype;
  }

  kwrk->num = num;
  if (num == 0) {
    kwrk->sv = NULL;
    kwrk->sqnorm = NULL;
    kwrk->max = 0;
    kwrk->tbl = NULL;
    return kwrk;
  }
  kwrk->sv = smalloc(sizeof(SVM_SV *) * num);
  for (i = 0; i < num; i++) kwrk->sv[i] = sv[i];

  /* $B%N%k%`$N<+>h$N7W;;(B */
  if (kprm->ktype == SVM_KERNEL_RBF) {	/* RBF kernel */
    kwrk->sqnorm = smalloc(sizeof(float) * num);
    if (sv[0]->val != NULL) {	/* real-valued vector */
      for (i = 0; i < num; i++) {
	kwrk->sqnorm[i] = 0.0f;
	for (j = 0; j < sv[i]->num; j++) kwrk->sqnorm[i] += sv[i]->val[j] * sv[i]->val[j];
      }
    } else {	/* binary-valued vector */
      for (i = 0; i < num; i++) {
	kwrk->sqnorm[i] = (float)sv[i]->num;
      }
    }
  } else {
    kwrk->sqnorm = NULL;
  }

  max = 1;
  for (i = 0; i < num; i++) {
    for (j = 0; j < sv[i]->num; j++) {
      if (max < sv[i]->idx[j]) max = sv[i]->idx[j];
    }
  }
  max++;
  kwrk->max = max;
  kwrk->tbl = smalloc(sizeof(int) * (max + 1));	/* $B%F!<%V%k$O(B(max +1)$B8D3NJ]$9$k!%:w0zI=$r;H$C$?Fb@Q7W;;$G$O:G8e$N(B1$B8D$OITMW(B($B$J$N$G=i4|2=$bI,MW$J$$(B)$B!%E>CV%$%s%G%C%/%9$r:n$k>l9g!$(Bn$BHVL\$NAG@-$N3+;O0LCV$,(Btbl[n]$B$G=*N;0LCV$,(Btbl[n+1]$B$H$J$k$?$a!$:G8e$NAG@-$N=*N;0LCV$rJ];}$9$k$?$a$K(B1$B$DM>J,$JMWAG$,I,MW$H$J$k!%(B */
  for (i = 0; i < max; i++) kwrk->tbl[i] = -1;

  return kwrk;
}


/*
  $B%-%c%C%7%e$N:n6HMQ>pJs$N=i4|2=(B($BE>CV%Y%/%H%kHG(B)
*/
SVM_KWRK *svm_knl_kwrkinit_inv(SVM_KPRM *kprm, int num, SVM_SV **sv) {
  int i, j, k;
  SVM_KWRK *kwrk;
  int sum;
  int *tmp_eid;
  float *tmp_val;
  int *tmp_nxt;

  /* $B6&DLItJ,=i4|2=(B */
  kwrk = svm_knl_kwrkinit(kprm, num, sv);
  if (num == 0) return kwrk;
  free(kwrk->sv);
  kwrk->sv = NULL;

  /* $BE>CV%$%s%G%C%/%9$N:n@.(B */
  sum = 0;
  for (i = 0; i < num; i++) sum += sv[i]->num;
  tmp_nxt = smalloc(sizeof(int) * sum);
  tmp_eid = smalloc(sizeof(int) * sum);
  k = 0;
  for (i = 0; i < num; i++) {
    for (j = 0; j < sv[i]->num; j++) {
      tmp_eid[k] = i;
      tmp_nxt[k] = kwrk->tbl[sv[i]->idx[j]];
      kwrk->tbl[sv[i]->idx[j]] = k;
      k++;
    }
  }
  if (sv[0]->val == NULL) {
    tmp_val = NULL;
  } else {
    tmp_val = smalloc(sizeof(float) * sum);
    k = 0;
    for (i = 0; i < num; i++) {
      memcpy(tmp_val + k, sv[i]->val, sizeof(float) * sv[i]->num);
      k += sv[i]->num;
    }
  }

  /* $B@0Ns(B */
  if (tmp_val != NULL) {
    kwrk->inv_eid = smalloc(sizeof(int) * sum);
    kwrk->inv_val = smalloc(sizeof(float) * sum);
    k = 0;
    for (i = 0; i < kwrk->max; i++) {
      for (j = kwrk->tbl[i], kwrk->tbl[i] = k; j != -1; j = tmp_nxt[j]) {
	kwrk->inv_eid[k] = tmp_eid[j];
	kwrk->inv_val[k] = tmp_val[j];
	k++;
      }
    }
    kwrk->tbl[i] = k;
    free(tmp_eid);
    free(tmp_val);
    free(tmp_nxt);
  } else {
    kwrk->inv_eid = smalloc(sizeof(int) * sum);
    kwrk->inv_val = NULL;
    k = 0;
    for (i = 0; i < kwrk->max; i++) {
      for (j = kwrk->tbl[i], kwrk->tbl[i] = k; j != -1; j = tmp_nxt[j]) {
	kwrk->inv_eid[k] = tmp_eid[j];
	k++;
      }
    }
    kwrk->tbl[i] = k;
    free(tmp_eid);
    free(tmp_nxt);
  }

  return kwrk;
}


/*
  $B%-%c%C%7%e$N:n6HMQ>pJs$N:o=|(B
*/
void svm_knl_kwrkdelete(SVM_KWRK *kwrk) {
  if (kwrk->tbl != NULL) free(kwrk->tbl);
  if (kwrk->sqnorm != NULL) free(kwrk->sqnorm);
  if (kwrk->sv != NULL) {
    free(kwrk->sv);
  } else {
    free(kwrk->inv_eid);
    if (kwrk->inv_val != NULL) free(kwrk->inv_val);
  }
  free(kwrk);

  return;
}


/*
  $B%+!<%M%k$N7W;;(B
*/
void svm_knl_kwrkkernel(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *v, float *q) {
  /* sqnorm$B$N7W;;(B */
  if (kprm->ktype == SVM_KERNEL_RBF) {
    if (v->val == NULL) {	/* binary-valued vector */
      kwrk->sqnorm2 = (float)v->num;
    } else {	/* real-valued vector */
      float *val, *val_end;

      kwrk->sqnorm2 = 0.0f;
      for (val = v->val, val_end = v->val + v->num; val < val_end; val++) kwrk->sqnorm2 += *val * *val;
    }
  }

  (kernel[kwrk->kfunc])(kprm, kwrk, v, q);

  return;
}


/*
  Linear kernel
*/
static void kernel_linear(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *v, float *q) {
  dotprod(kwrk, v, q);

  return;
}


/*
  Polynomial kernel
*/
static void kernel_polynomial(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *v, float *q) {
  float *q_end;

  dotprod(kwrk, v, q);

  for (q_end = q + kwrk->num; q < q_end; q++) {
    *q = powf(kprm->gamma * *q + kprm->coef, kprm->degree);
  }

  return;
}


/*
  Radial Basis Function kernel
*/
static void kernel_rbf(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *v, float *q) {
  float *q_end;
  float *sqnorm;

  dotprod(kwrk, v, q);

  for (q_end = q + kwrk->num, sqnorm = kwrk->sqnorm; q < q_end; q++, sqnorm++) {
    *q = expf(-kprm->gamma * (*sqnorm + kwrk->sqnorm2 - (*q + *q)));
  }

  return;
}


/*
  Sigmoid kernel
*/
static void kernel_sigmoid(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *v, float *q) {
  float *q_end;

  dotprod(kwrk, v, q);

  for (q_end = q + kwrk->num; q < q_end; q++) {
    *q = tanhf(kprm->gamma * *q + kprm->coef);
  }

  return;
}


/*
  Polynomial kernel(1st order)
*/
static void kernel_polynomial_1o(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *v, float *q) {
  float *q_end;

  dotprod(kwrk, v, q);

  for (q_end = q + kwrk->num; q < q_end; q++) {
    *q = kprm->gamma * *q + kprm->coef;
  }

  return;
}


/*
  Polynomial kernel(2nd order)
*/
static void kernel_polynomial_2o(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *v, float *q) {
  float *q_end;
  float r;

  dotprod(kwrk, v, q);

  for (q_end = q + kwrk->num; q < q_end; q++) {
    r = kprm->gamma * *q + kprm->coef;
    *q = r * r;
  }

  return;
}


/*
  Polynomial kernel(3rd order)
*/
static void kernel_polynomial_3o(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *v, float *q) {
  float *q_end;
  float r;

  dotprod(kwrk, v, q);

  for (q_end = q + kwrk->num; q < q_end; q++) {
    r = kprm->gamma * *q + kprm->coef;
    *q = r * r * r;
  }

  return;
}


/*
  Polynomial kernel(4th order)
*/
static void kernel_polynomial_4o(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *v, float *q) {
  float *q_end;
  float r;

  dotprod(kwrk, v, q);

  for (q_end = q + kwrk->num; q < q_end; q++) {
    r = kprm->gamma * *q + kprm->coef;
    *q = r * r * r * r;
  }

  return;
}


/*
  Polynomial kernel(5th order)
*/
static void kernel_polynomial_5o(SVM_KPRM *kprm, SVM_KWRK *kwrk, SVM_SV *v, float *q) {
  float *q_end;
  float r;

  dotprod(kwrk, v, q);

  for (q_end = q + kwrk->num; q < q_end; q++) {
    r = kprm->gamma * *q + kprm->coef;
    *q = r * r * r * r * r;
  }

  return;
}


/*
  $BFb@Q$N7W;;(B
*/
static void dotprod(SVM_KWRK *kwrk, SVM_SV *v, float *q) {
  int i, j;
  SVM_SV **sv;
  float *q_end;
  int *vidx, *vidx_end;
  SVM_SV *w;
  int *widx, *widx_end;
  int max;

  if (v->num == 0) {
    memset(q, 0x00, sizeof(float) * kwrk->num);	/* 0$B$K=i4|2=(B(IEEE 754) */
    return;
  }

  if (kwrk->sv == NULL) {	/* $BE>CV%Y%/%H%k$K$h$k7W;;(B */
    memset(q, 0x00, sizeof(float) * kwrk->num);	/* 0$B$K=i4|2=(B(IEEE 754) */

    if (v->val == NULL) {	/* input=binary-valued vector */
      if (kwrk->inv_val == NULL) {	/* target=binary-valued vector */
	for (i = 0; i < v->num; i++) {
	  int *eid, *eid_end;
	  if (v->idx[i] >= kwrk->max) break;
	  for (eid = kwrk->inv_eid + kwrk->tbl[v->idx[i]], eid_end = kwrk->inv_eid + kwrk->tbl[v->idx[i] + 1]; eid < eid_end; eid++) {
	    q[*eid] += 1.0f;
	  }
	}
      } else {	/* target=real-valued vector */
	for (i = 0; i < v->num; i++) {
	  int *eid, *eid_end;
	  float *val;
	  if (v->idx[i] >= kwrk->max) break;
	  for (eid = kwrk->inv_eid + kwrk->tbl[v->idx[i]], eid_end = kwrk->inv_eid + kwrk->tbl[v->idx[i] + 1], val = kwrk->inv_val + kwrk->tbl[v->idx[i]]; eid < eid_end; eid++, val++) {
	    q[*eid] += *val;
	  }
	}
      }
    } else {	/* input=real-valued vector */
      if (kwrk->inv_val == NULL) {	/* target=binary-valued vector */
	for (i = 0; i < v->num; i++) {
	  int *eid, *eid_end;
	  if (v->idx[i] >= kwrk->max) break;
	  for (eid = kwrk->inv_eid + kwrk->tbl[v->idx[i]], eid_end = kwrk->inv_eid + kwrk->tbl[v->idx[i] + 1]; eid < eid_end; eid++) {
	    q[*eid] += v->val[i];
	  }
	}
      } else {	/* target=real-valued vector */
	for (i = 0; i < v->num; i++) {
	  int *eid, *eid_end;
	  float *val;
	  if (v->idx[i] >= kwrk->max) break;
	  for (eid = kwrk->inv_eid + kwrk->tbl[v->idx[i]], eid_end = kwrk->inv_eid + kwrk->tbl[v->idx[i] + 1], val = kwrk->inv_val + kwrk->tbl[v->idx[i]]; eid < eid_end; eid++, val++) {
	    q[*eid] += *val * v->val[i];
	  }
	}
      }
    }

    return;
  }

  /* $B:w0zI=$N=`Hw(B */
  for (vidx = v->idx, vidx_end = vidx + v->num, i = 0; vidx < vidx_end; vidx++, i++) {
    if (*vidx >= kwrk->max) break;
    kwrk->tbl[*vidx] = i;
  }
  max = (i > 0) ? v->idx[i - 1] : -1;

  /* $BFb@Q$N7W;;(B */
  if (v->val == NULL) {	/* input=binary-valued vector */
    if (kwrk->sv[0]->val == NULL) {	/* target=binary-valued vector */
      for (q_end = q + kwrk->num, sv = kwrk->sv; q < q_end; q++, sv++) {
	int qq;

	w = *sv;
	if (w->num == 0) {
	  *q = 0.0f;
	  continue;
	}
	qq = 0;
	for (widx = w->idx, widx_end = widx + w->num; widx < widx_end; widx++) {
	  if (*widx > max) break;
	  j = kwrk->tbl[*widx];
	  if (j != -1) {
	    qq++;
	  }
	}
	*q = (float)qq;
      }
    } else {	/* target=real-valued vector */
      for (q_end = q + kwrk->num, sv = kwrk->sv; q < q_end; q++, sv++) {
	float qq;

	w = *sv;
	if (w->num == 0) {
	  *q = 0.0f;
	  continue;
	}
	qq = 0.0f;
	for (widx = w->idx, widx_end = widx + w->num; widx < widx_end; widx++) {
	  if (*widx > max) break;
	  j = kwrk->tbl[*widx];
	  if (j != -1) {
	    qq += w->val[widx - w->idx];
	  }
	}
	*q = qq;
      }
    }
  } else {	/* input=real-valued vector */
    if (kwrk->sv[0]->val == NULL) {	/* target=binary-valued vector */
      for (q_end = q + kwrk->num, sv = kwrk->sv; q < q_end; q++, sv++) {
	float qq;

	w = *sv;
	if (w->num == 0) {
	  *q = 0.0f;
	  continue;
	}
	qq = 0.0f;
	for (widx = w->idx, widx_end = widx + w->num; widx < widx_end; widx++) {
	  if (*widx > max) break;
	  j = kwrk->tbl[*widx];
	  if (j != -1) {
	    qq += v->val[j];
	  }
	}
	*q = qq;
      }
    } else {	/* target=real-valued vector */
      for (q_end = q + kwrk->num, sv = kwrk->sv; q < q_end; q++, sv++) {
	float qq;

	w = *sv;
	if (w->num == 0) {
	  *q = 0.0f;
	  continue;
	}
	qq = 0.0f;
	for (widx = w->idx, widx_end = widx + w->num; widx < widx_end; widx++) {
	  if (*widx > max) break;
	  j = kwrk->tbl[*widx];
	  if (j != -1) {
	    qq += v->val[j] * w->val[widx - w->idx];
	  }
	}
	*q = qq;
      }
    }
  }

  /* $B:w0zI=$N8eJRIU$1(B */
  for (vidx = v->idx, vidx_end = vidx + v->num; vidx < vidx_end; vidx++) {
    if (*vidx >= kwrk->max) break;
    kwrk->tbl[*vidx] = -1;
  }

  return;
}
