/*

    Euchre - a free as in freedom and as in beer version of the 
             euchre card game
  
    Copyright 2002 C Nathan Buckles (nbuckles@bigfoot.com)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "Options.hpp"
#include "ComputerPlayerEasy.hpp"

ComputerPlayerEasy::ComputerPlayerEasy(Common::PlayerPosition myPos) :
  ComputerPlayer(myPos) {}

ComputerPlayerEasy::~ComputerPlayerEasy() {}

Common::Bid ComputerPlayerEasy::auction1(const Card& upCard,
					 Common::PlayerPosition dealer) {

  /* easy decides only on a point calculation */
  int minPoints;
  if (dealer == itsPos || dealer == Common::getPartner(itsPos)) {
    minPoints = 33;
  } else {
    minPoints = 36;
  }

  /* adjust for aggression level */

  minPoints -= (2 * Options::get()->getAIAgg(itsPos));

  /* simple comparison to decide the bid */
  int myPoints = itsHand.getValue(upCard.getSuit());
  if (myPoints < minPoints) {
    itsBid = Common::PASS;
    return itsBid;
  }

  /* easy never goes loner */
  itsBid = Common::PICKITUP;

  return itsBid;
}

Common::Bid ComputerPlayerEasy::auction2(Card& yourTrump,
					 const Card& upCard,
					 bool stuck) {
  Card t;
  Common::Bid bid;

  for (int i = Card::Diamonds; i <= Card::Spades; i++) {
    if (upCard.getSuit() == i) {
      continue;
    }

    t.setSuit((Card::Suit)i);
    bid = ComputerPlayerEasy::auction1(t, 
				       (Common::PlayerPosition) (itsPos + 1));

    if (bid != Common::PASS) {
      yourTrump = t;
      break;
    }
  }

  /* if we are stuck with doing it and we didn't find a good suit,
     then just go with the highest scoring suit. */
  if (stuck && bid == Common::PASS) {
    bid = Common::PICKITUP;
    yourTrump.setSuit(getStuckTrumpSuit(upCard.getSuit()));
  }
  
  itsBid = bid;
  return bid;
}

Card ComputerPlayerEasy::discard(Card& newCard) {
  Card ret = itsHand.getWorstCard();
  itsHand.replaceCard(ret, newCard);

  return ret;
}

Card ComputerPlayerEasy::getCard(const Round& theRound, 
			     Common::PlayerPosition whoStarted) {
  /* logic is pretty simple

     if we are the one leading
       lead our best card
     else
       if we can win the trick so far
         play our highest winning card
       else
         play our lowest card

     in every case we must follow suit if we can
  */  

  /* if it's my lead */
  if (whoStarted == itsPos) {
    return getLeadCard();
  } else {
    return getNonLeadCard(theRound, whoStarted);
  }
}

Card ComputerPlayerEasy::getLeadCard() {
  int max_index = -1;
  int max_score = -1;

  /* get our best card adjusting for trump */
  for (int i = 0; i < Common::CARDS_PER_HAND; i++) {

    /* skip cards that aren't there */
    if (itsHand.getCard(i).getSuit() == Card::NoSuit) {
      continue;
    }

    int temp_score = itsHand.getCard(i).getValue();

    if (itsHand.getCard(i).isSuit(Common::getTrump())) {
      temp_score -= Card::TrumpAdj;
    }

    if (temp_score > max_score) {
      max_index = i;
      max_score = temp_score;
    }
  }

  return itsHand.removeCard(max_index);
}

Card ComputerPlayerEasy::getNonLeadCard(const Round& theRound, 
				       Common::PlayerPosition whoStarted) {
  /* get the suit led */
  Card::Suit suitLed = theRound.getSuit(whoStarted);

  /* return the card to play */
  Card ret;

  /* find out who is winning so far */
  Card winningCard = theRound.getCard(theRound.getWinner(whoStarted, 
							 Common::getTrump()));

  if (itsHand.contains(suitLed)) {
    /* if we need to follow suit */
    ret = itsHand.getBestCard(suitLed);

    /* if we can't win, then play low */
    if (ret.getValue() < winningCard.getValue()) {
      ret = itsHand.getWorstCard(suitLed);
    }

  } else {
    /* we can play whatever we want. */
    if (itsHand.contains(Common::getTrump())) {
      /* if we have trump and it will win then play that */
      ret = itsHand.getBestCard(Common::getTrump());

      /* if it's not good enough, then duck */
      if (ret.getValue() < winningCard.getValue()) {
	ret = itsHand.getWorstCard();
      }
    } else {
      /* if we don't have any trump and we don't have the suit led
         then we can't win so play our lowest card */
      ret = itsHand.getWorstCard();
    }
  }

  itsHand.removeCard(ret);
  return ret;
}
