/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-cfg-abook.c,v 1.13 2005/01/29 15:46:53 hoa Exp $
 */

#include "etpan-cfg-abook.h"

#include <stdlib.h>
#include "etpan-cfg-common.h"
#include "etpan-db-read.h"
#include "etpan-errors.h"
#include "etpan-ldap.h"
#include "etpan-abook-ldap.h"
#include "etpan-abook-local.h"
#include "etpan-abook-vcard.h"
#include "etpan-cfg-local-abook.h"
#include "etpan-abook-driver.h"
#include <string.h>
#include <stdio.h>

static int ldap_abook_read(struct etpan_global_config * global_config,
    chash * entry, struct etpan_abook ** result)
{
  char * hostname;
  uint16_t port;
  int version;
  char * base;
  char * binddn;
  char * bindpw;
  char * filter;
  char * mail;
  char * cn;
  char * givenname;
  char * sn;
  int sizelimit;
  int timeout;
  char * new_value;
  struct etpan_ldap_config * config;
  struct etpan_abook * abook;
  int res;
  int r;
  
  hostname = NULL;
  port = ETPAN_DEFAULT_LDAP_PORT;
  version = ETPAN_DEFAULT_LDAP_VERSION;
  base = NULL;
  binddn = NULL;
  bindpw = NULL;
  filter = ETPAN_DEFAULT_LDAP_FILTER;
  mail = NULL;
  cn = NULL;
  givenname = NULL;
  sn = NULL;
  sizelimit = 0;
  timeout = 0;
  
  new_value = etpan_db_entry_get_value(entry, "hostname");
  if (new_value != NULL)
    hostname = new_value;
  
  new_value = etpan_db_entry_get_value(entry, "port");
  if (new_value != NULL)
    port = strtoul(new_value, NULL, 10);

  new_value = etpan_db_entry_get_value(entry, "version");
  if (new_value != NULL)
    version = strtoul(new_value, NULL, 10);

  new_value = etpan_db_entry_get_value(entry, "base");
  if (new_value != NULL)
    base = new_value;
  
  new_value = etpan_db_entry_get_value(entry, "binddn");
  if (new_value != NULL)
    binddn = new_value;
  
  new_value = etpan_db_entry_get_value(entry, "bindpw");
  if (new_value != NULL)
    bindpw = new_value;
  
  new_value = etpan_db_entry_get_value(entry, "filter");
  if (new_value != NULL)
    filter = new_value;
  
  new_value = etpan_db_entry_get_value(entry, "mail");
  if (new_value != NULL)
    mail = new_value;
  
  new_value = etpan_db_entry_get_value(entry, "cn");
  if (new_value != NULL)
    cn = new_value;
  
  new_value = etpan_db_entry_get_value(entry, "givenname");
  if (new_value != NULL)
    givenname = new_value;
  
  new_value = etpan_db_entry_get_value(entry, "sn");
  if (new_value != NULL)
    sn = new_value;

  new_value = etpan_db_entry_get_value(entry, "sizelimit");
  if (new_value != NULL)
    sizelimit = strtoul(new_value, NULL, 10);
  
  new_value = etpan_db_entry_get_value(entry, "timeout");
  if (new_value != NULL)
    timeout = strtoul(new_value, NULL, 10);
  
  if (hostname == NULL) {
    res = ERROR_INVAL;
    goto err;
  }
  
  r = etpan_cfg_strdup(&hostname, hostname);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  r = etpan_cfg_strdup(&base, base);
  if (r != NO_ERROR) {
    res = r;
    goto free_hostname;
  }
  
  r = etpan_cfg_strdup(&binddn, binddn);
  if (r != NO_ERROR) {
    res = r;
    goto free_base;
  }
  
  r = etpan_cfg_strdup(&bindpw, bindpw);
  if (r != NO_ERROR) {
    res = r;
    goto free_binddn;
  }

  r = etpan_cfg_strdup(&filter, filter);
  if (r != NO_ERROR) {
    res = r;
    goto free_bindpw;
  }

  r = etpan_cfg_strdup(&mail, mail);
  if (r != NO_ERROR) {
    res = r;
    goto free_filter;
  }

  r = etpan_cfg_strdup(&cn, cn);
  if (r != NO_ERROR) {
    res = r;
    goto free_mail;
  }

  r = etpan_cfg_strdup(&givenname, givenname);
  if (r != NO_ERROR) {
    res = r;
    goto free_cn;
  }

  r = etpan_cfg_strdup(&sn, sn);
  if (r != NO_ERROR) {
    res = r;
    goto free_givenname;
  }
  
  config = malloc(sizeof(* config));
  if (config == NULL) {
    res = ERROR_MEMORY;
    goto free_sn;
  }
  
  config->hostname = hostname;
  config->port = port;
  config->version = version;
  config->base = base;
  config->binddn = binddn;
  config->bindpw = bindpw;
  config->filter = filter;
  config->attrs.mail = mail;
  config->attrs.cn = cn;
  config->attrs.givenname = givenname;
  config->attrs.sn = sn;
  config->sizelimit = sizelimit;
  config->timeout = timeout;
  
  abook = etpan_abook_ldap_new(global_config, config);
  if (abook == NULL) {
    res = ERROR_MEMORY;
    goto free_config;
  }

  * result = abook;
  
  return NO_ERROR;
  
 free_config:
  free(config);
 free_sn:
  free(sn);
 free_givenname:
  free(givenname);
 free_cn:
  free(cn);
 free_mail:
  free(mail);
 free_filter:
  free(filter);
 free_bindpw:
  free(bindpw);
 free_binddn:
  free(binddn);
 free_base:
  free(base);
 free_hostname:
  free(hostname);
 err:
  return res;
}

static int
local_abook_read(struct etpan_global_config * global_config,
    chash * entry, struct etpan_abook ** result)
{
  char * filename;
  int r;
  struct etpan_abook * abook;
  
  filename = etpan_db_entry_get_value(entry, "file");
  if (filename == NULL)
    return ERROR_INVAL;
  
  abook = etpan_abook_local_new(global_config, filename);
  if (abook == NULL) {
    return ERROR_MEMORY;
  }
  
  * result = abook;
  
  return NO_ERROR;
}


static int
vcard_abook_read(struct etpan_global_config * global_config,
    chash * entry, struct etpan_abook ** result)
{
  char * filename;
  int r;
  struct etpan_abook * abook;
  
  filename = etpan_db_entry_get_value(entry, "file");
  if (filename == NULL)
    return ERROR_INVAL;
  
  abook = etpan_abook_vcard_new(global_config, filename);
  if (abook == NULL) {
    return ERROR_MEMORY;
  }
  
  * result = abook;
  
  return NO_ERROR;
}


static int abook_read(struct etpan_global_config * global_config,
    chash * entry, struct etpan_abook ** result)
{
  char * type;
  int r;
  struct etpan_abook * abook;
  
  type = etpan_db_entry_get_value(entry, "type");
  
  r = ERROR_INVAL;
  if (type == NULL)
    return r;
  
  if (strcasecmp(type, "local") == 0)
    r = local_abook_read(global_config, entry, &abook);
  else if (strcasecmp(type, "ldap") == 0)
    r = ldap_abook_read(global_config, entry, &abook);
  else if (strcasecmp(type, "vcard") == 0)
    r = vcard_abook_read(global_config, entry, &abook);
  
  if (r != NO_ERROR)
    return r;
  
  * result = abook;
  
  return NO_ERROR;
}

int etpan_abook_config_read(struct etpan_global_config * global_config,
    char * filename,
    struct etpan_abook_config ** result)
{
  struct etpan_db * db;
  int r;
  int res;
  carray * tab;
  struct etpan_abook_config * abook_config;
  unsigned int i;
  
  r = etpan_read_config(filename, &db);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  tab = carray_new(16);
  if (tab == NULL) {
    res = ERROR_MEMORY;
    goto free_db;
  }
  
  for(i = 0 ; i < db->data->len ; i ++) {
    chash * entry;
    struct etpan_abook * abook;
    
    entry = carray_get(db->data, i);
    if (entry == NULL)
      continue;
    
    r = abook_read(global_config, entry, &abook);
    if (r != NO_ERROR) 
      continue;
    
    r = carray_add(tab, abook, NULL);
    if (r < 0) {
      etpan_abook_free(abook);
      res = ERROR_MEMORY;
      goto free_list;
    }
  }
  
  abook_config = etpan_abook_config_new(tab);
  if (abook_config == NULL) {
    res = ERROR_MEMORY;
    goto free_list;
  }

  * result = abook_config;
  
  etpan_db_free(db);
  
  return NO_ERROR;

 free_list:
  for(i = 0 ; i < carray_count(tab) ; i ++) {
    struct etpan_abook * abook;

    abook = carray_get(tab, i);
    etpan_abook_free(abook);
  }
 free_db:
  etpan_db_free(db);
 err:
  return res;
}

struct etpan_abook_config *
etpan_abook_config_new(carray * tab)
{
  struct etpan_abook_config * config;
  
  config = malloc(sizeof(* config));
  if (config == NULL)
    return NULL;
  
  config->abook_tab = tab;
  
  return config;
}

void etpan_abook_config_free(struct etpan_abook_config * config)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(config->abook_tab) ; i ++) {
    struct etpan_abook * abook;
    
    abook = carray_get(config->abook_tab, i);
    etpan_abook_free(abook);
  }
  carray_free(config->abook_tab);
  
  free(config);
}

