#include "Connection.hpp"

namespace et {
Connection::Connection(shared_ptr<SocketHandler> _socketHandler,
                       const string& _id, const string& _key)
    : socketHandler(_socketHandler), id(_id), key(_key), shuttingDown(false) {}

Connection::~Connection() {
  if (!shuttingDown) {
    LOG(ERROR) << "Call shutdown before destructing a Connection.";
  }
  if (socketFd != -1) {
    LOG(INFO) << "Connection destroyed";
    closeSocket();
  }
}

inline bool isSkippableError(int err_no) {
  return (err_no == EAGAIN || err_no == ECONNRESET || err_no == ETIMEDOUT ||
          err_no == EWOULDBLOCK || err_no == EHOSTUNREACH || err_no == EPIPE ||
          err_no == EBADF  // Bad file descriptor can happen when
                           // there's a race condition between a thread
                           // closing a connection and one
                           // reading/writing.
  );
}

bool Connection::read(string* buf) {
  VLOG(4) << "Before read get connectionMutex";
  lock_guard<std::recursive_mutex> guard(connectionMutex);
  VLOG(4) << "After read get connectionMutex";
  ssize_t messagesRead = reader->read(buf);
  if (messagesRead == -1) {
    if (isSkippableError(errno)) {
      // Close the socket and invalidate, then return 0 messages
      LOG(INFO) << "Closing socket because " << errno << " " << strerror(errno);
      closeSocketAndMaybeReconnect();
      return 0;
    } else {
      // Throw the error
      LOG(ERROR) << "Got a serious error trying to read: " << errno << " / "
                 << strerror(errno);
      throw std::runtime_error("Failed a call to read");
    }
  } else {
    return messagesRead > 0;
  }
}

bool Connection::readMessage(string* buf) {
  while (!shuttingDown) {
    bool result = read(buf);
    if (result) {
      return true;
    }
    // Yield the processor
    if (socketFd == -1) {
      // No connection, sleep for 100ms
      usleep(100 * 1000);
    } else {
      // Have a connection, sleep for 1ms
      usleep(1 * 1000);
    }
    LOG_EVERY_N(1000, INFO) << "Waiting to read...";
  }
  return false;
}

bool Connection::write(const string& buf) {
  lock_guard<std::recursive_mutex> guard(connectionMutex);
  if (socketFd == -1) {
    return false;
  }

  BackedWriterWriteState bwws = writer->write(buf);

  if (bwws == BackedWriterWriteState::SKIPPED) {
    VLOG(4) << "Write skipped";
    return false;
  }

  if (bwws == BackedWriterWriteState::WROTE_WITH_FAILURE) {
    VLOG(4) << "Wrote with failure";
    // Error writing.
    if (socketFd == -1) {
      // The socket was already closed
      VLOG(1) << "Socket closed";
    } else if (isSkippableError(errno)) {
      VLOG(1) << " Connection is severed";
      // The connection has been severed, handle and hide from the caller
      closeSocketAndMaybeReconnect();
    } else {
      LOG(FATAL) << "Unexpected socket error: " << errno << " "
                 << strerror(errno);
    }
  }

  return 1;
}

void Connection::writeMessage(const string& buf) {
  while (!shuttingDown) {
    bool success = write(buf);
    if (success) {
      return;
    }
    // Yield the processor
    if (socketFd == -1) {
      // No connection, sleep for 100ms
      usleep(100 * 1000);
    } else {
      // Have a connection, sleep for 1ms
      usleep(1 * 1000);
    }
    LOG_EVERY_N(1000, INFO) << "Waiting to write...";
  }
}

void Connection::closeSocket() {
  lock_guard<std::recursive_mutex> guard(connectionMutex);
  if (socketFd == -1) {
    LOG(ERROR) << "Tried to close a dead socket";
    return;
  }
  // TODO: There is a race condition where we invalidate and another
  // thread can try to read/write to the socket.  For now we handle the
  // error but it would be better to avoid it.
  reader->invalidateSocket();
  writer->invalidateSocket();
  int fd = socketFd;
  socketFd = -1;
  socketHandler->close(fd);
  VLOG(1) << "Closed socket";
}

bool Connection::recover(int newSocketFd) {
  LOG(INFO) << "Locking reader/writer to recover...";
  lock_guard<std::mutex> readerGuard(reader->getRecoverMutex());
  lock_guard<std::mutex> writerGuard(writer->getRecoverMutex());
  LOG(INFO) << "Recovering with socket fd " << newSocketFd << "...";
  try {
    {
      // Write the current sequence number
      et::SequenceHeader sh;
      sh.set_sequencenumber(reader->getSequenceNumber());
      socketHandler->writeProto(newSocketFd, sh, true);
    }

    // Read the remote sequence number
    et::SequenceHeader remoteHeader =
        socketHandler->readProto<et::SequenceHeader>(newSocketFd, true);

    {
      // Fetch the catchup bytes and send
      et::CatchupBuffer catchupBuffer;
      vector<string> recoveredMessages =
          writer->recover(remoteHeader.sequencenumber());
      for (auto it : recoveredMessages) {
        catchupBuffer.add_buffer(it);
      }
      socketHandler->writeProto(newSocketFd, catchupBuffer, true);
    }

    et::CatchupBuffer catchupBuffer =
        socketHandler->readProto<et::CatchupBuffer>(newSocketFd, true);

    socketFd = newSocketFd;
    vector<string> recoveredMessages(catchupBuffer.buffer().begin(),
                                     catchupBuffer.buffer().end());

    reader->revive(socketFd, recoveredMessages);
    writer->revive(socketFd);
    LOG(INFO) << "Finished recovering with socket fd: " << socketFd;
    return true;
  } catch (const runtime_error& err) {
    LOG(ERROR) << "Error recovering: " << err.what();
    socketHandler->close(newSocketFd);
    return false;
  }
}

void Connection::shutdown() {
  LOG(INFO) << "Shutting down connection";
  shuttingDown = true;
  closeSocket();
}
}  // namespace et
