/*  jpr   2016/10/01   Added support for Linux_x86_64.       */
/* jpr/pba   2016/06/22  PPRS-65719: Added support for Mac OSX. */
/*                                                  */
/*            Self-Defining Data System             */
/*                                                  */
/*            Jeremy Bailey  1991 Oct               */
/*                                                  */
/*                                                  */
/*            Version 2.2  1998  Jun                */


/*    Copyright (c)  Anglo-Australian Telescope Board, 1991-1998.  */
/*    Permission granted for use for non-commercial purposes.      */




#ifdef VxWorks
#include "vxWorks.h"
#include "semLib.h"

/*  Semaphore to control allocation of identifiers for VxWorks real
    time operating system  */

SEM_ID   idSemaphore;

/*  Status from semTake  */

STATUS   semStatus;

#endif

#ifdef POSIX_THREADS

#include <pthread.h>

pthread_mutex_t 	id_mutex;

#endif

#ifndef hpux
#define pthread_mutexattr_default  NULL
#endif


#include <stdio.h>
#include <string.h>

#include <stdlib.h>


#include "status.h"
#include "sds.h"




#define SDS_SARRAY   10     /*  Code for structure array  */
#define SDS_NCODES   13     /*  Number of type codes  */


/*    Initial number of handles to allocate   */

#define NHANDLES    100

/*    Default number of structure items       */

#define NITEMS      20

/*    Number of ID blocks per section of ID array  */

#define NIDS        2000

/*    Maximum number of extra bytes      */

#define MAX_EXTRA   128

/*    Maximum number of dimensions   */

#define MAXDIMS  7

/*    Format codes   */

#define  BIGEND    0            /* Big endian integers */
#define  LITTLEEND 1            /* Little endian integers */
#define  IEEE      0            /* IEEE floating point format */
#define  VAXF      1            /* VAX floating point format */
#define  VAXD      VAXF         /* VAX D floating point format  */
#define  IEEEBS    2            /* IEEE floating point format byte swapped */
#define  VAXG      3            /* VAX G floating point format  */

/*     Number of bytes in each format   */

static long nbytes[SDS_NCODES] = {0,1,1,1,2,2,4,4,4,8,0,8,8};


/*
 *  The CC$gfloat macro is defined on VAXC systems.  If true it indicates
 *  G floating point is being used.  Under DEC C, this is replaced by 
 *  __G_FLOAT
 */
#ifdef VMS
#ifdef CC$gfloat
#ifndef __G_FLOAT
#define __G_FLOAT CC$gfloat
#endif
#endif
#endif

/*     local format   */

#if defined(VAX) && __G_FLOAT

/*  VAX with G floating point format  */

#define VAX_G_DOUBLE 1
#define VAX_FLOAT 1

   static long  local_format[SDS_NCODES] = {0,0,0,LITTLEEND,LITTLEEND,LITTLEEND,
                                       LITTLEEND,LITTLEEND,VAXF,VAXG,0,LITTLEEND,LITTLEEND};

#elif defined(VAX)

   /*  VAX  */

#define VAX_D_DOUBLE 1
#define VAX_FLOAT 1

   static long  local_format[SDS_NCODES] = {0,0,0,LITTLEEND,LITTLEEND,LITTLEEND,
                                       LITTLEEND,LITTLEEND,VAXF,VAXD,0,LITTLEEND,LITTLEEND};

#elif defined(__alpha) && defined(__vms)

/*  Alpha VMS  */

#ifdef __G_FLOAT

#define VAX_G_DOUBLE 1
#define VAX_FLOAT 1
   static long  local_format[SDS_NCODES] = {0,0,0,LITTLEEND,LITTLEEND,LITTLEEND,
                                       LITTLEEND,LITTLEEND,VAXF,VAXG,0,LITTLEEND,LITTLEEND};

#elif __IEEE_FLOAT
   static long  local_format[SDS_NCODES] = {0,0,0,LITTLEEND,LITTLEEND,LITTLEEND,
                                   LITTLEEND,LITTLEEND,IEEEBS,IEEEBS,0,LITTLEEND,LITTLEEND};

#else

#define VAX_D_DOUBLE 1
#define VAX_FLOAT 1
   static long  local_format[SDS_NCODES] = {0,0,0,LITTLEEND,LITTLEEND,LITTLEEND,
                                       LITTLEEND,LITTLEEND,VAXF,VAXD,0,LITTLEEND,LITTLEEND};

#endif  /* Alpha VMS */


#elif defined(__alpha) 

   /*  Dec Alpha  OSF */

   static long  local_format[SDS_NCODES] = {0,0,0,LITTLEEND,LITTLEEND,LITTLEEND,
                                   LITTLEEND,LITTLEEND,IEEEBS,IEEEBS,0,LITTLEEND,LITTLEEND};
#elif defined MIPSEL

   /*  MIPS cpu with little endian byte order (Decstation)  */

   static long  local_format[SDS_NCODES] = {0,0,0,LITTLEEND,LITTLEEND,LITTLEEND,
                                  LITTLEEND,LITTLEEND,IEEEBS,IEEEBS,0,LITTLEEND,LITTLEEND};
#elif defined M_I86

   /*  Intel 80x86 cpus  */

   static long  local_format[SDS_NCODES] = {0,0,0,LITTLEEND,LITTLEEND,LITTLEEND,
                                   LITTLEEND,LITTLEEND,IEEEBS,IEEEBS,0,LITTLEEND,LITTLEEND};

#elif defined __macosx__

   /* Mac OS X  */

   static long  local_format[SDS_NCODES] = {0,0,0,LITTLEEND,LITTLEEND,LITTLEEND,
                                   LITTLEEND,LITTLEEND,IEEEBS,IEEEBS,0,LITTLEEND,LITTLEEND};

#elif defined __linux_x86_64__

   /* Linux x84_64  */

   static long  local_format[SDS_NCODES] = {0,0,0,LITTLEEND,LITTLEEND,LITTLEEND,
                                   LITTLEEND,LITTLEEND,IEEEBS,IEEEBS,0,LITTLEEND,LITTLEEND};

#else

   /*  All others - e.g. SPARC, 680x0 etc.  */

   static long  local_format[SDS_NCODES] = {0,0,0,BIGEND,BIGEND,BIGEND,
                                       BIGEND,BIGEND,IEEE,IEEE,0,BIGEND,BIGEND};
#endif

#ifdef VAX_G_DOUBLE
double MTH$CVT_D_G(double * d);
#endif

#ifdef VAX_D_DOUBLE
double MTH$CVT_G_D(double * d);
#endif



#define  TRUE 1
#define  FALSE 0

#ifndef VxWorks
typedef char    BOOL;
#endif

/*  handle type - handles (which are pointers to pointers) are used to provide
    the links between blocks of structures. The advantage of using a handle
    rather than a direct pointer, is that items can easily be moved about
    with it only being necessary to update a single pointer through which
    everything else accesses the object   */

typedef void   *hptr;
typedef hptr   *handle;

typedef hptr   handle_block[NHANDLES];

typedef char   Sdscode;
typedef char   fmt_code;

/*  Structure block - blocks of this type are used to represent structures */

typedef struct sblock
  {  long maxitems;              /* max number of items  */
     handle self;                /* handle to this block */
     handle parent;              /* handle of parent block */
     Sdscode code;               /* type code  */
     fmt_code format;            /* format code */
     char  name[SDS_C_NAMELEN];  /* item name */
     long   nextra;              /* number of extra bytes */
     char  extra[MAX_EXTRA];     /* extra information */
     short    nitems;            /* number of items */
     handle   item[NITEMS];      /* item handles */
  }  sblock;

/* primitive block - blocks of this type are used to represent primitive items
                     and structure arrays   */

typedef struct pblock
  {  long maxitems;              /* max number of items  */
     handle self;                /* handle to this block */
     handle parent;              /* handle to parent block */
     Sdscode code;               /* type code  */
     fmt_code format;            /* format code */
     char  name[SDS_C_NAMELEN];  /* item name */
     long   nextra;              /* number of extra bytes */
     char  extra[MAX_EXTRA];     /* extra information */
     handle data;                /* handle to data */
     short    ndims;             /* number of dimensions */
     unsigned long   dims[MAXDIMS];       /* dimensions */
  }  pblock;


/*  idblock - a block of this type exists for each identifier allocated.
 *  If the item is internal this contains a handle to the item. If the
 *  item is external it contains a pointer to the item, and also a pointer
 *  to the origin of the block (the latter is needed since pointers in
 *  external items are relative to the origin of the block). 
 */

typedef struct idblock
  { handle   block;              /* handle of block (or pointer to the item
                                     if it is external)  */
    INT32     external;           /* TRUE if item is external */ 
    INT32     free;               /* TRUE if block is to be freed when the id is freed; */
    INT32*    origin;             /* origin of block for external item */
  }  idblock;


static INT32 id_number = 0;         /* number of last id allocated  */
static idblock *idarrayptr;        /* pointer to array of identifiers  */
static INT32 id_array_size = 0;     /* current size of array of identifiers */
static INT32 unused_ids = 0;        /* number of free ids before end of array */



INT64 SdsSetI64(long high, unsigned long low)

/*  Set value of a 64 bit integer, high contains the high order 32 bits
    low contains the low order 32 bits  */

{
#ifdef LONG__64
    return(high*4294967296u+low);
#elif defined BIGEND
    INT64 temp;
    temp.i1 = high;
    temp.i2 = low;
    return temp;
#else
    INT64 temp;
    temp.i1 = low;
    temp.i2 = high;
    return temp;
#endif
}

UINT64 SdsSetUI64(unsigned long high, unsigned long low)

/*  Set value of a 64 bit unsigned integer, high contains the high order 32 bits
    low contains the low order 32 bits  */

{
#ifdef LONG__64
    return(high*4294967296u+low);
#elif defined BIGEND
    UINT64 temp;
    temp.i1 = high;
    temp.i2 = low;
    return temp;
#else
    UINT64 temp;
    temp.i1 = low;
    temp.i2 = high;
    return temp;
#endif
}

void SdsGetI64(INT64 i64, long *high, unsigned long *low)

/*  Get value of a 64 bit integer, as high and low
order 32 bits  */

{
#ifdef LONG__64
    *high = i64 / 4294967296u;
    *low = i64 % 4294967296u;
#elif defined BIGEND
    *high = i64.i1;
    *low = i64.i2;
#else
    *high = i64.i2;
    *low = i64.i1;
#endif
}

void SdsGetUI64(UINT64 i64, unsigned long *high, unsigned long *low)

/*  Get value of a 64 bit unsigned integer, as high and low
order 32 bits  */

{
#ifdef LONG__64
    *high = i64 / 4294967296u;
    *low = i64 % 4294967296u;
#elif defined BIGEND
    *high = i64.i1;
    *low = i64.i2;
#else
    *high = i64.i2;
    *low = i64.i1;
#endif
}



typedef struct UI64 {
   UINT32 i1;
   UINT32 i2;
} UI64;

                                   
UINT32 swap4(BOOL gbyteswap, UINT32 i)

/*  
 *    swap4 - swap the byte order of a long integer if
 *    gbyteswap is true
 */

{
   UINT32 temp;

   if (gbyteswap)
     {
       temp = (i<<24) | ((i<<8) & 0xFF0000) | ((i>>8) & 0xFF00) | (i>>24);
       return(temp);
     }
   else
     return(i);
}

UI64 swap8(BOOL gbyteswap, UI64 i)

/*  
 *    swap8 - swap the byte order of a 64 bit integer if
 *    gbyteswap is true
 */

{
   UI64 temp;
  
   if (gbyteswap)
     {
        temp.i1 = swap4(gbyteswap,i.i2);
        temp.i2 = swap4(gbyteswap,i.i1);
        return(temp);
     }
   else
     return(i);
}


    
unsigned short swap2(BOOL gbyteswap, unsigned short i)

/*  
 *    swap2 - swap the byte order of a short integer if
 *    gbyteswap is true.
 */

{
   short temp;

   if (gbyteswap)
     {
       temp = (i<<8) | (i>>8);
       return(temp);
     }
   else
     return(i);
}
    

void Sds__convert(long code, long format, INT32 *data, long length)

/*
 *  Sds__convert - Apply number format conversion to a block of data.
 *
 *  This function converts bewteen little endian and big endian integers,
 *  and between the various floating point formats.
 *
 *  The conversion is always to the local format 
 *    i.e. from  format  to  local_format[code]. 
 */

{
  union F4
   {  float f; UINT32 l; unsigned char b[4]; };
  union D8
   {  double d; UINT32 l[2]; unsigned char b[8];};

  INT32 *longbuf;           /*  Pointer to buffer as a long  */
  UI64 *llbuf;              /*  Pointer to buffer as a 64 bit integer */
  short *shortbuf;          /*  Pointer to buffer as a short  */
  union F4 *floatbuf;       /*  Pointer to buffer as a float  */
  union D8 *doublebuf;      /*  Pointer to buffer as a double  */
  long i;                   /*  Loop counter  */
  UINT32 tl;                /*  long temporary value   */
  UI64 tll;                 /*  64 bit temporary value  */
  unsigned short ts;        /*  short temporary value  */
  union F4 tf,tf2;          /*  float temporary values  */
  union D8 td,td2;          /*  double temporary values  */
  unsigned long e;          /*  exponent of floating point number  */

  if ((code == SDS_I64) || (code == SDS_UI64))
    {

/*  Convert 64 bit integer types by byte swapping  */

       llbuf = (UI64*)data;
       for(i=0;i<((length)/8);i++)
       {
          tl = llbuf[i].i1;
          tll.i2 = (tl<<24) | ((tl<<8) & 0xFF0000) | 
                       ((tl>>8) & 0xFF00) | (tl>>24);
          tl = llbuf[i].i2;
          tll.i1 = (tl<<24) | ((tl<<8) & 0xFF0000) | 
                       ((tl>>8) & 0xFF00) | (tl>>24);
          llbuf[i] = tll;
       }
    }  if ((code == SDS_INT) || (code == SDS_UINT))
    {

/*  Convert long integer types by byte swapping  */

       longbuf = (INT32*)data;
       for(i=0;i<((length)/4);i++)
       {
          tl = longbuf[i];
          longbuf[i] = (tl<<24) | ((tl<<8) & 0xFF0000) | 
                       ((tl>>8) & 0xFF00) | (tl>>24);
       }
    }
 else if ((code == SDS_SHORT) || (code == SDS_USHORT))
    {

/*  Convert short integer types by byte swapping  */

       shortbuf = (short*)data;
       for(i=0;i<((length)/2);i++)
       {
          ts = shortbuf[i];
          shortbuf[i] = (ts<<8) | (ts>>8);
       }
    }
 else if (code == SDS_FLOAT)
#if defined(VAX_FLOAT)
    {

/*  Convert to VAX floating point type from IEEE or byte-swapped IEEE  */
/*  NOTE - This code is based on the assumption that it is running on a
    VAX (or VMS Alpha), as will always be the case in SDS, 
    since conversion is always to the local format. 
    It is therefore included in a conditional compilation
    as it is not needed on other architectures.                        */

       if (format == IEEE || format == IEEEBS) 
         {
            floatbuf = (union F4*)data;
            for (i=0;i<((length)/4);i++)
            {
               tf = floatbuf[i];    

/*  Swap bytes as required to get correct order for VAX  */

               if (format == IEEE)
                 {
                   tf2.b[0] = tf.b[1];
                   tf2.b[1] = tf.b[0];
                   tf2.b[2] = tf.b[3];
                   tf2.b[3] = tf.b[2];
                 }
               else
                 {
                   tf2.b[0] = tf.b[2];
                   tf2.b[1] = tf.b[3];
                   tf2.b[2] = tf.b[0];
                   tf2.b[3] = tf.b[1];
                 }

/*  Extract the exponent  */

               e = tf2.l & 0x7F80;
               if (e == 0)

/*  The exponent is zero (plus or minus zero or a denormalized number)   */

                 if ((tf2.l & 0x60) != 0)

/*  Check for a denormalized number which can still be represented in VAX
    format - Such cases will have an exponent of either 1 or 2 in
    VAX format and the fraction needs shifting left by 1 or 2  */

                   {
                     e = (tf2.l & 0x40) ? 2 : 1;
                     tf.l = ((tf2.l & 0xFFFF007F) << (3-e)) |
                            (tf2.b[3] >> (5+e));
                     tf.l = tf.l | ((e << 7) & 0x180) | (tf2.l & 0x8000);
                     floatbuf[i] = tf;
                   } 
                 else 

/*  Otherwise set result to zero   */

                   floatbuf[i].f = 0.0;
               else if (e == 0x7F80 || e == 0x7F00)

/*  If the exponent is 254 or 255 the number is a NaN or Infinity or is too
    large to be represented in VAX format, so set result to a bad value.  */

                 floatbuf[i].l = 0xFFFFFFFF;
               else

/*  Otherwise the conversion can be handled by a simple multiplication
    by 4 to account for the difference in exponent bias, and radix point
    position   */

                 floatbuf[i].f = tf2.f*4.0;
            }
        }
    }
#else
    {
      if (format == IEEE || format == IEEEBS)
        {

/*  Conversion from IEEE to byte swapped IEEE  */

          longbuf = (INT32*)data;
          for(i=0;i<((length)/4);i++)
            {
              tl = longbuf[i];
              longbuf[i] = (tl<<24) | ((tl<<8) & 0xFF0000) | 
                          ((tl>>8) & 0xFF00) | (tl>>24);
            }
        }
      else if (format == VAXF)
        {

/*  Conversion from VAX to IEEE or byte swapped IEEE  */
/*  NOTE - We have to be careful here because the machine
    could have either little endian or big endian byte order. The
    assumption in this code is that byte swapped IEEE is always
    on machines with little endian integers, and normal IEEE on
    machines with big endian integers  */

            floatbuf = (union F4*)data;
            for (i=0;i<((length)/4);i++)
            {
               tf = floatbuf[i];    

/*  Swap bytes as required to get correct order   */

               if (local_format[code] == IEEEBS)
                 {
                   tf2.b[0] = tf.b[2];
                   tf2.b[1] = tf.b[3];
                   tf2.b[2] = tf.b[0];
                   tf2.b[3] = tf.b[1];
                 }
               else
                 {
                   tf2.b[0] = tf.b[1];
                   tf2.b[1] = tf.b[0];
                   tf2.b[2] = tf.b[3];
                   tf2.b[3] = tf.b[2];
                 }

/*  Extract the exponent  */

               e = tf2.l & 0x7F800000;

/*  If the exponent is zero the resulting value is zero. Also check
    the number for a bad value. Otherwise the conversion can be accomplished
    by multiplying by 0.25.   */

               if (e == 0) 
                 floatbuf[i].f = 0.0;
               else if (tf2.l == 0xFFFFFFFF)
                 {
                   tf2.l = 0xFFFFFFFF;
                   tf2.b[1] = 0x7F;
                   floatbuf[i] = tf2;
                 }
               else
                 floatbuf[i].f = tf2.f*0.25;
            }
        }
    }
#endif
 else if (code == SDS_DOUBLE)
#if defined(VAX_D_DOUBLE)
    {

/*  Convert to VAX D floating point type from IEEE or byte-swapped IEEE  */

       if (format == IEEE || format == IEEEBS) 
         {
            doublebuf = (union D8*)data;
            for (i=0;i<((length)/8);i++)
            {
               td = doublebuf[i];

/*  Swap bytes if required to get correct order  */

               if (format == IEEEBS)
                 {
                   td2.l[1] = (td.l[0]<<24) | ((td.l[0]<<8) & 0xFF0000) | 
                              ((td.l[0]>>8) & 0xFF00) | (td.l[0]>>24);
                   td2.l[0] = (td.l[1]<<24) | ((td.l[1]<<8) & 0xFF0000) | 
                              ((td.l[1]>>8) & 0xFF00) | (td.l[1]>>24);
                   td = td2;
                 }
               e = ((td.b[0] & 0x7F) << 4) | ((td.b[1] >> 4) & 0xF);

/*  If the exponent is zero (plus or minus zero or a denormalized number)
    set the result to zero. Also set zero for any biased exponent less than
    894 which is too small to be represented in VAX format. Set a bad value
    for anything with biased exponent greater than 1150, which is either
    too large for the VAX representation or is a NaN or Infinity.   */

               if (e <= 894) 
                 doublebuf[i].d = 0.0;
               else if (e >= 1150)
                 {
                    td2.l[0] = 0xFFFFFFFF;
                    td2.l[1] = 0xFFFFFFFF;
                    doublebuf[i] = td2;
                 }
               else
                 {

/*  The number is a normalized IEEE value within the range that can be
    represented on the VAX. Decrement the exponent by 894 to give
    corresponding VAX value and reassemble the number.    */

                    e -= 894;
                    td2.b[0] = ((e<<7) & 0x80) | ((td.b[1] & 0xF) << 3)
                              | ((td.b[2] >> 5) & 0x7);
                    td2.b[1] = ((e>>1) & 0x7F) | (td.b[0] & 0x80);
                    td2.b[2] = ((td.b[3] << 3) & 0xF8) |((td.b[4] >> 5) & 0X7);
                    td2.b[3] = ((td.b[2] << 3) & 0xF8) |((td.b[3] >> 5) & 0X7);
                    td2.b[4] = ((td.b[5] << 3) & 0xF8) |((td.b[6] >> 5) & 0X7);
                    td2.b[5] = ((td.b[4] << 3) & 0xF8) |((td.b[5] >> 5) & 0X7);
                    td2.b[6] = ((td.b[7] << 3) & 0xF8);
                    td2.b[7] = ((td.b[6] << 3) & 0xF8) |((td.b[7] >> 5) & 0X7);
                    doublebuf[i] = td2;
                 }
            }
        }
      else if (format == VAXG)
        {
  
/*  Conversion from VAX G floating to VAX D floating  */

          doublebuf = (union D8*)data;
          for(i=0;i<((length)/8);i++)
            {
               td = doublebuf[i];

/*  Check that the exponent is in the range that can be 
    represented as a D floating value - if it is too small
    return 0, otherwise set to bad value  */            

               e = ((td.b[1] & 0x7F) << 4) | ((td.b[0] >> 4) & 0xF);
            
               if (e == 0 || (e < 896)) 
                 doublebuf[i].d = 0.0;
               else if (e >= 1151)
                 {
                    td2.l[0] = 0xFFFFFFFF;
                    td2.l[1] = 0xFFFFFFFF;
                    doublebuf[i] = td2;
                 }
               else

/*  Otherwise use the RTL routine to do the conversion  */

               doublebuf[i].d = MTH$CVT_G_D(&(td.d));
            }
	}    
    }
#elif defined(VAX_G_DOUBLE)
    {

/*  Convert to VAX G floating point type from IEEE or byte-swapped IEEE  */

       if (format == IEEE || format == IEEEBS) 
         {
            doublebuf = (union D8*)data;
            for (i=0;i<((length)/8);i++)
            {
               td = doublebuf[i];

/*  Swap bytes if required to get correct order  */

               if (format == IEEE)
                 {
                   td2.b[0] = td.b[1];
                   td2.b[1] = td.b[0];
                   td2.b[2] = td.b[3];
                   td2.b[3] = td.b[2];
                   td2.b[4] = td.b[5];
                   td2.b[5] = td.b[4];
                   td2.b[6] = td.b[7];
                   td2.b[7] = td.b[6];
                 }
               else
                 {
                   td2.b[0] = td.b[6];
                   td2.b[1] = td.b[7];
                   td2.b[2] = td.b[4];
                   td2.b[3] = td.b[5];
                   td2.b[4] = td.b[2];
                   td2.b[5] = td.b[3];
                   td2.b[6] = td.b[0];
                   td2.b[7] = td.b[1];
                 }

               e = ((td.b[1] & 0x7F) << 4) | ((td.b[0] >> 4) & 0xF);

/*  If the exponent is zero (plus or minus zero or a denormalized number)
    set the result to zero. Set a bad value
    for anything with biased exponent of >= 2046, which is a NaN or Infinity
    or is too large to be represented   */

               if (e == 0) 
                 doublebuf[i].d = 0.0;
               else if (e >= 2046)
                 {
                    td2.l[0] = 0xFFFFFFFF;
                    td2.l[1] = 0xFFFFFFFF;
                    doublebuf[i] = td2;
                 }
               else
                 {

/* Otherwise we can handle the conversion simply by multiplying by 4
   to allow for the difference in exponent bias and radix point position  */

                    doublebuf[i].d = td2.d*4.0;
                 }

            }
        }
      else if (format == VAXD)
        {
  
/*  Conversion from VAX D floating to VAX G floating  */

          doublebuf = (union D8*)data;
          for(i=0;i<((length)/8);i++)
            {
              td = doublebuf[i];

/*  Use the RTL routine - we don't have to check the range as any
    D-floating number is in the range that can be represented in G-floating */

              doublebuf[i].d = MTH$CVT_D_G(&(td.d));
            }
	}    
    }
#else
    {
      if (format == IEEE || format == IEEEBS)
        {

/*  Conversion from IEEE to byte swapped IEEE  */

          doublebuf = (union D8*)data;
          for(i=0;i<((length)/8);i++)
            {
              td = doublebuf[i];
              td2.l[1] = (td.l[0]<<24) | ((td.l[0]<<8) & 0xFF0000) | 
                          ((td.l[0]>>8) & 0xFF00) | (td.l[0]>>24);
              td2.l[0] = (td.l[1]<<24) | ((td.l[1]<<8) & 0xFF0000) | 
                          ((td.l[1]>>8) & 0xFF00) | (td.l[1]>>24);
              doublebuf[i] = td2;
            }
        }
      else if (format == VAXD)
        {

/*  Conversion from VAX D-Floating to IEEE or byte swapped IEEE  */

            doublebuf = (union D8*)data;
            for (i=0;i<((length)/8);i++)
            {
               td = doublebuf[i];    

/*  Check for bad value and return correpsonding IEEE bad value  */

               if ((td.l[0] == 0xFFFFFFFF) && (td.l[1] == 0xFFFFFFFF))
                 {
                   td2.b[1] = 0xEF;
                 }
               else
                 {

/*  Extract the exponent  */

                   e = ((td.b[1]<<1) & 0xFE) | ((td.b[0]>>7) & 0x1) ;
                   if (e != 0)
                     {

/*  If the exponent is non-zero, increment it by 894 and reassemble
    the resulting IEEE value   */

                        e += 894;
                        td2.b[0] = (td.b[1] & 0x80) | ((e >> 4) & 0x7F);
                        td2.b[1] = ((td.b[0]>>3) & 0xF) |((e<<4) & 0xF0);
                        td2.b[2] = ((td.b[0]<<5) & 0xE0) |((td.b[3]>>3) &0x1F);
                        td2.b[3] = ((td.b[3]<<5) & 0xE0) |((td.b[2]>>3) &0x1F);
                        td2.b[4] = ((td.b[2]<<5) & 0xE0) |((td.b[5]>>3) &0x1F);
                        td2.b[5] = ((td.b[5]<<5) & 0xE0) |((td.b[4]>>3) &0x1F);
                        td2.b[6] = ((td.b[4]<<5) & 0xE0) |((td.b[7]>>3) &0x1F);
                        td2.b[7] = ((td.b[7]<<5) & 0xE0) |((td.b[6]>>3) &0x1F);
                     }
                   else

/*  If the exponent is zero the result is zero   */

                     td2.d = 0.0;
                 }
               if (local_format[code] == IEEEBS)
                 {

/*  Byte swap the result if necessary  */

                   td.l[1] = (td2.l[0]<<24) | ((td2.l[0]<<8) & 0xFF0000) 
                            | ((td2.l[0]>>8) & 0xFF00) | (td2.l[0]>>24);
                   td.l[0] = (td2.l[1]<<24) | ((td2.l[1]<<8) & 0xFF0000) 
                            | ((td2.l[1]>>8) & 0xFF00) | (td2.l[1]>>24);
                   doublebuf[i] = td;
                 }
               else
                 doublebuf[i] = td2;
            }
        }
      else if (format == VAXG)
        {

/*  Conversion from VAX G-Floating to IEEE or byte swapped IEEE  */

            doublebuf = (union D8*)data;
            for (i=0;i<((length)/8);i++)
            {
               td = doublebuf[i];    

/*  Check for bad value and return correpsonding IEEE bad value  */

               if ((td.l[0] == 0xFFFFFFFF) && (td.l[1] == 0xFFFFFFFF))
                 {
                   td2 = td;
                   if (local_format[code] == IEEE) 
                      td2.b[1] = 0xEF;
                   else
                      td2.b[6] = 0xEF;
                   doublebuf[i] = td2; 
                 }
               else
                 {

/*  Swap bytes if required to get correct order  */

                   if (local_format[code] == IEEE)
                     {
                       td2.b[0] = td.b[1];
                       td2.b[1] = td.b[0];
                       td2.b[2] = td.b[3];
                       td2.b[3] = td.b[2];
                       td2.b[4] = td.b[5];
                       td2.b[5] = td.b[4];
                       td2.b[6] = td.b[7];
                       td2.b[7] = td.b[6];
                     }
                   else
                     {
                       td2.b[0] = td.b[6];
                       td2.b[1] = td.b[7];
                       td2.b[2] = td.b[4];
                       td2.b[3] = td.b[5];
                       td2.b[4] = td.b[2];
                       td2.b[5] = td.b[3];
                       td2.b[6] = td.b[0];
                       td2.b[7] = td.b[1];
                     }
  
/*  Handle the conversion by multiplying by 0.25 to allow for the
    difference in exponent bias and radix point position  */

                   doublebuf[i].d = 0.25*td2.d;
                 }

            }
        }
    }
#endif
}

SdsIdType  Sds__get_new_id(idblock* *idptr)

/*
 *  Sds__get_new_id  -  Allocate a new identifier. Return the id number as
 *                   the function result, and return a pointer to the
 *                   id block in *idptr.
 */

{
  idblock *temp;
  SdsIdType i;
  INT32 temp_id;
INT32	steve_unused_ids_counter;
   steve_unused_ids_counter = unused_ids;

#ifdef VxWorks

/*  If running under VxWorks create the semaphore used for controlling
    access to the id allocation system  */

  if (id_array_size == 0)
    {
      idSemaphore = semMCreate(SEM_Q_PRIORITY | SEM_DELETE_SAFE |
                               SEM_INVERSION_SAFE);
      if (idSemaphore == NULL) return(0);
    }

/*  If running under VxWorks take the semaphore   */

   semStatus = semTake(idSemaphore,600);
   if (semStatus != OK) return(0);

#endif

#ifdef POSIX_THREADS


   if ( id_array_size == 0 )
     {
	if ( pthread_mutex_init( &id_mutex, pthread_mutexattr_default ) )
	{
	    return(0);
	}
     }
   pthread_mutex_lock( &id_mutex );

#endif

/*  See if there are unused entries in the current id array  */

   if (unused_ids > 0)
     {
        for (i = 1; i <= id_number; i++) 
        {
          if (idarrayptr[i].block == NULL ) 
           {
             *idptr = &(idarrayptr[i]);

/*  Set .block component to mark the id as used - The actual value is
    irrelevant as long as it is not NULL  */

             idarrayptr[i].block = (handle)(*idptr);
             unused_ids--;
#ifdef VxWorks
             semGive(idSemaphore);
#endif
#ifdef POSIX_THREADS
	     pthread_mutex_unlock( &id_mutex );
#endif
             idarrayptr[i].free = FALSE;
             return(i);
           }
        }
       if ( steve_unused_ids_counter != unused_ids - 1 )
       {
	steve_unused_ids_counter = steve_unused_ids_counter;
       }
   }
         
  id_number=id_number+1;

/*  Check of the id array has room for any more identifiers  */

  if (id_number >= id_array_size)
  {

/*  If no room we have to create a larger id array and copy the contents
    of the old array into it  */

     id_array_size = id_array_size+NIDS;
     temp = (idblock*)SdsMalloc(sizeof(idblock)*id_array_size);
     if (temp == 0) return(0);
     if (id_array_size > NIDS)
     {
        for (i=1;i<id_array_size-NIDS;i++)
          temp[i] = idarrayptr[i];
#ifndef VxWorks
        SdsFree(idarrayptr);
#endif
     }
     idarrayptr = temp;
  }

/*  then return the new id number and the pointer  */

  *idptr = &(idarrayptr[id_number]);

/*  Set .block component to mark the id as used - The actual value is
    irrelevant as long as it is not NULL  */

  idarrayptr[id_number].block = (handle)(*idptr);
  temp_id = id_number;

#ifdef VxWorks

/*  if running under VxWorks release the semaphore    */

  semGive(idSemaphore);

#endif
#ifdef POSIX_THREADS
  pthread_mutex_unlock( &id_mutex );
#endif

  idarrayptr[temp_id].free = FALSE;
  return(temp_id);

}




void Sds__get_id(SdsIdType id, handle *block, INT32 *external, INT32 **origin,
                 StatusType *status)

{

  if (*status != SDS__OK) return;


#ifdef VxWorks
  semStatus = semTake(idSemaphore,600);

  if (semStatus != OK) 
    {

      /*  The semaphore take might have failed because no ids have 
          yet been created and so the semaphore has not been created
          In this case the id must be bad and we should return BADID
          status  */

      if ((id <= 0) || (id > id_number))
         *status = SDS__BADID;
      else
         *status = SDS__TIMEOUT;
      return;
    }
#endif

#ifdef POSIX_THREADS
  pthread_mutex_lock(&id_mutex);
#endif

  if ((id <= 0) || (id > id_number))
  {
     *status = SDS__BADID;

#ifdef VxWorks
  semGive(idSemaphore);
#endif
#ifdef POSIX_THREADS
  pthread_mutex_unlock(&id_mutex);
#endif

     return;
  }
  if (idarrayptr[id].block == NULL)   
  {
     *status = SDS__BADID;

#ifdef VxWorks
  semGive(idSemaphore);
#endif
#ifdef POSIX_THREADS
  pthread_mutex_unlock(&id_mutex);
#endif

     return;
  }
  if (!(idarrayptr[id].external))
    if (*(idarrayptr[id].block) == NULL)   
    {
       *status = SDS__BADID;

#ifdef VxWorks
  semGive(idSemaphore);
#endif
#ifdef POSIX_THREADS
  pthread_mutex_unlock(&id_mutex);
#endif

       return;
    }

  *block = idarrayptr[id].block;
  *external = idarrayptr[id].external;
  *origin = idarrayptr[id].origin;

#ifdef VxWorks
  semGive(idSemaphore);
#endif
#ifdef POSIX_THREADS
  pthread_mutex_unlock(&id_mutex);
#endif

}


void Sds__put_id(SdsIdType id, handle block, INT32 external, INT32 *origin,
                 StatusType *status)

{

  if (*status != SDS__OK) return;

#ifdef VxWorks
  semStatus = semTake(idSemaphore,600);
  if (semStatus != OK) 
    {
      *status = SDS__TIMEOUT;
      return;
    }
#endif

#ifdef POSIX_THREADS
  pthread_mutex_lock(&id_mutex);
#endif

  if ((id <= 0) || (id > id_number))
  {
     *status = SDS__BADID;

#ifdef VxWorks
  semGive(idSemaphore);
#endif
#ifdef POSIX_THREADS
  pthread_mutex_unlock(&id_mutex);
#endif

     return;
  }
  if (idarrayptr[id].block == NULL)   
  {
     *status = SDS__BADID;

#ifdef VxWorks
  semGive(idSemaphore);
#endif
#ifdef POSIX_THREADS
  pthread_mutex_unlock(&id_mutex);
#endif

     return;
  }

  idarrayptr[id].block = block;
  idarrayptr[id].external = external;
  idarrayptr[id].origin = origin;

#ifdef VxWorks
  semGive(idSemaphore);
#endif
#ifdef POSIX_THREADS
  pthread_mutex_unlock(&id_mutex);
#endif

}



idblock *Sds__import_id(SdsIdType id, StatusType * SDSCONST status)

/*
 *  Sds__import_id - Given an id number return a pointer to the id block
 */

{
  if (*status != SDS__OK) return(NULL);
  if ((id <= 0) || (id > id_number))
  {
     *status = SDS__BADID;
     return(NULL);
  }
  if (idarrayptr[id].block == NULL)   
  {
     *status = SDS__BADID;
     return(NULL);
  }
  if (!(idarrayptr[id].external))
    if (*(idarrayptr[id].block) == NULL)   
    {
       *status = SDS__BADID;
       return(NULL);
    }
  return(&(idarrayptr[id]));
}


/* static handle handle_pointer = NULL; - apparently no longer used. - KS */
static void * null_handle = NULL;

handle Sds__get_handle(void)

/*
 *   Sds__get_handle - Allocate a new handle
 */

{
  handle temp;


  temp = (handle)SdsMalloc(sizeof(void*));
/*  printf("new handle %d \n",(long)temp);*/
  return(temp);

}


void Sds__MarkFree(SdsIdType id)

{
   idblock *idb=NULL;
   StatusType status = STATUS__OK;

#ifdef VxWorks
  semStatus = semTake(idSemaphore,600);
  if (semStatus != OK) 
    {
      status = SDS__TIMEOUT;
      return;
    }
#endif

#ifdef POSIX_THREADS
  pthread_mutex_lock(&id_mutex);
#endif
   idb = Sds__import_id(id,&status);
   idb->free = TRUE;

#ifdef VxWorks
  semGive(idSemaphore);
#endif
#ifdef POSIX_THREADS
  pthread_mutex_unlock(&id_mutex);
#endif

}

void Sds__Free(SdsIdType id, StatusType * SDSCONST status)

{
   idblock *idb;

#ifdef VxWorks
  semStatus = semTake(idSemaphore,600);
  if (semStatus != OK) 
    {
      *status = SDS__TIMEOUT;
      return;
    }
#endif

#ifdef POSIX_THREADS
  pthread_mutex_lock(&id_mutex);
#endif

   idb = Sds__import_id(id,status);
   if (idb->free) SdsFree(idb->origin);

#ifdef VxWorks
  semGive(idSemaphore);
#endif
#ifdef POSIX_THREADS
  pthread_mutex_unlock(&id_mutex);
#endif

}




void Sds__extend_struct(sblock **sb,StatusType * SDSCONST status)

/*
 *  Sds__extend_struct - extend an sblock when the space for structure items
 *                       has filled up
 */

{
  sblock *newsb;
  long n;
  long i;

  if(*status != SDS__OK) return;

/*  Create a new block which has room for NITEMS more handles than the
 *  present one
 */
  n = (*sb)->maxitems;
  newsb = (sblock*) SdsMalloc(sizeof(sblock)+(n*sizeof(handle)));
  if (newsb == NULL)
  {
     *status = SDS__NOMEM;
     return;
  }
/*
 *  Copy over contents
 */

  *newsb = **sb;
  for (i=NITEMS;i<n;i++)
  {
     newsb->item[i] = (*sb)->item[i];
  }
/*
 *  Update the handle to point to new block
 */

  *(newsb->self) = (void*)newsb;
     
  newsb->maxitems = n+NITEMS;
  SdsFree(*sb);
  *sb = newsb;

}


sblock* Sds__create_sblock(SDSCONST char *name, long nextra, SDSCONST char *extra, long items)

/*
 *  Sds__create_sblock - Create a new strucure block
 */

{
  sblock* sb;
  long j;
  long n;

/*
 *  sblocks are allocated with space for (n+1) * NITEMS items.
 *  work out what n has to be (NITEMS items are included in the
 *  definition of sblock, and n more blocks are added on the
 *  end if required)
 */

  if (items > NITEMS) 
     {
       n = items/NITEMS;
       if ((items % NITEMS) == 0) n--;
     }
  else
    n = 0;
/*
 *  Allocate the space
 */
  sb = (sblock*)SdsMalloc(sizeof(sblock) + (n*NITEMS*sizeof(handle)));
  if (sb == NULL) return(NULL);
/*
 *  Initialize the fields of the block
 */
  strcpy(sb->name,name);
  sb->maxitems = (n+1)*NITEMS;
  sb->code = SDS_STRUCT;
  sb->format = 0;
  sb->nitems = 0;
  sb->nextra = nextra;
  for (j=0;j<nextra;j++) sb->extra[j] = extra[j];
  return(sb);
}



/*				S d s C r e a t e

 *  Function name:
      SdsCreate

 *  Function:
      Create a new structure

 *  Description:
      Creates a new empty internal structure and returns an identifier
      to it.
      
 *  Language:
      C

 *  Declaration:
      void SdsCreate(char *name, long nextra, char *extra, SdsIdType *id, long *status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) name      (char*) The name of the structure to create.
                            The name should be of maximum length 16
                            characters including the terminating null.
      (>) nextra    (long)  The number of bytes of extra information to
                            be included (maximum 128).
      (>) extra     (char*) The extra information to be included with the item.
                            nextra bytes from here are copied into the
                            structure.
      (<) id        (SdsIdType*) Identifier to the created structure.
      (!) status    (long*) Modified status. Possible failure codes are:
                            SDS__NOMEM => Insufficient memory for creation
                            SDS__LONGNAME => name is too long

 *  Prior requirements:
      None.

 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 18-Oct-91
 *
 */
void SdsCreate(SDSCONST char *name, long nextra, SDSCONST char *extra, SdsIdType *id, StatusType * SDSCONST status)
{
   SdsNew(0,name,nextra,extra,SDS_STRUCT,0,NULL,id,status);
}


void Sds__info_ext(INT32 *data, char *name, long *code, long *ndims, 
                  unsigned long *dims, StatusType * SDSCONST status)

/*
 *  Sds__info_ext - handle the external case for the Sdsinfo function
 */

{
   INT32 pos;
   int i;
 
   *code = ((char*)data)[0];
   strncpy(name,((char*)data)+4,SDS_C_NAMELEN);
   if (*code != SDS_STRUCT)
     {
       *ndims = ((short*)data)[1];
       pos = 5;
       if (*code != SDS_SARRAY) pos++;
       for (i=0;i<*ndims;i++)
          dims[i] = data[pos+i];
     }
   else
     *ndims = 0;
   if (*code == SDS_SARRAY) *code = SDS_STRUCT;
}  
   



/*+				S d s I n f o

 *  Function name:
      SdsInfo

 *  Function:
      Return information about an object

 *  Description:
      Given the identifier to an object, return the name, type code and
      dimensions of the object.
      
 *  Language:
      C

 *  Declaration:
      void SdsInfo(SdsIdType id, char *name, SdsCodeType *code, long *ndims, 
               unsigned long *dims, StatusType * SDSCONST status)
    
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType) The identifier to the data object.
      (<) name      (char*) The name of the data object. A pointer to a
                            character string with space for at least
                            16 characters should be used.
      (<) code      (SdsCodeType*) The type code for the object. One of
                            the following values (defined in sds.h):
                               SDS_STRUCT => Structure
                               SDS_CHAR   => Character
                               SDS_BYTE   => Signed byte
                               SDS_UBYTE  => Unsigned byte
                               SDS_SHORT  => Signed short integer
                               SDS_USHORT => Unsigned short integer
                               SDS_INT    => Signed long integer
                               SDS_UINT   => Unsigned long integer
                               SDS_I64    => Signed 64 bit integer
                               SDS_UI64   => Unsigned 64 bit integer
                               SDS_FLOAT  => Floating point
                               SDS_DOUBLE => Double precision floating point
      (<) ndims     (long*) The number of dimensions if the object
                            is a primitive or structure array.
      (<) dims      (unsigned long*) The dimensions of the data. An array of
                            size at least 7 should be allowed to receive this.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID => The identifier is invalid

 *  Prior requirements:
      None.

 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 18-Oct-91
 *-
 */

void SdsInfo(SdsIdType id, char *name, SdsCodeType *code, long *ndims, unsigned long *dims, StatusType * SDSCONST status)

{
  /*idblock *idptr;*/
  handle block;
  INT32 external;
  INT32* origin;
  sblock *sb;
  pblock *pb;
  long i;

  Sds__get_id(id,&block,&external,&origin,status);
  if (*status != SDS__OK) return;
  if (external)
    {

/*  If the object is external call the routine to handle the external case */

       Sds__info_ext((INT32*)(block),name,code,ndims,dims,status);
       return;
    }

/*  Otherwise get a pointer to the sblock  */

  sb = (sblock*)(*block);

/*  Copy the name and type code  */

  strcpy(name,sb->name);
  if (sb->code == SDS_SARRAY)
    *code = SDS_STRUCT;
  else
    *code = sb->code;

/*  If it is not a structure, get the dimensions   */

  if (sb->code != SDS_STRUCT)
    {
       pb = (pblock*)sb;
       *ndims = pb->ndims;
       for (i=0;i<pb->ndims;i++)  dims[i] = pb->dims[i];
    }   
  else
    *ndims = 0;
}




/*+				  S d s N e w

 *  Function name:
      SdsNew

 *  Function:
      Create a new object

 *  Description:
      Creates a new component in an existing internal structure or a
      new top level object. A top level object is created by specifying
      a parent_id of zero. The new object can be a structure, a structure 
      array, or a primitive. A structure array is specified by means of a 
      type code of SDS_STRUCT and a non-zero number for ndims. If the type 
      code is SDS_STRUCT and ndims is zero an ordinary structure is created. 
      A primitive type is specified by the appropriate type code.
      
 *  Language:
      C

 *  Declaration:
      void SdsNew(SdsIdType parent_id, char *name, long nextra, char *extra, 
         SdsCodeType code, long ndims, unsigned long *dims, SdsIdType *id, 
         StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) parent_id (SdsIdType) The identifier of the structure to
                            which the object is to be added. Use a 
                            value of zero to create a new top level object.
      (>) name      (char*) The name of the object to create.
                            The name should be of maximum length 16
                            characters including the terminating null.
      (>) nextra    (long)  The number of bytes of extra information to
                            be included (maximum 128).
      (>) extra     (char*) The extra information to be included with the item.
                            nextra bytes from here are copied into the
                            structure.
      (>) code      (SdsCodeType) The type code for the item to be created. One of
                            the following values (defined in sds.h):
                               SDS_STRUCT => Structure
                               SDS_CHAR   => Character
                               SDS_BYTE   => Signed byte
                               SDS_UBYTE  => Unsigned byte
                               SDS_SHORT  => Signed short integer
                               SDS_USHORT => Unsigned short integer
                               SDS_INT    => Signed long integer
                               SDS_UINT   => Unsigned long integer
                               SDS_I64    => Signed 64 bit integer
                               SDS_UI64   => Unsigned 64 bit integer
                               SDS_FLOAT  => Floating point
                               SDS_DOUBLE => Double precision floating point
                               
      (>) ndims     (long)  Number of dimensions for the item. Zero
                            to create a scalar item.
      (>) dims      (unsigned long*) Array of dimensions for the item. Should be of
                            size at least ndims. A NULL pointer may be used
                            if the item is a scalar.
      (<) id        (SdsIdType*) Identifier to the created object.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID => Invalid identifier
                            SDS__NOMEM => Insufficient memory for creation
                            SDS__LONGNAME => name is too long
                            SDS__EXTRA => Too much extra data
                            SDS__INVCODE => Invalid type code 
                            SDS__INVDIMS => Invalid dimensions
                            SDS__NOTSTRUCT => Parent is not a structure
                            SDS__EXTERN => Parent is external

 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 17-Aug-2001
 *-
 *  History:
 *     18th Oct 1991. Original dated version. JAB/AAO.
 *     17th Aug 2001. Initial value for n to placate gcc. KS/AAO.
 */

void SdsNew(SdsIdType parent_id, SDSCONST char *name, long nextra, SDSCONST char *extra, SdsCodeType code,
         long ndims, SDSCONST unsigned long *dims, SdsIdType *id, StatusType * SDSCONST status)

{
  sblock *sb;            /* pointer to parent structure */
  pblock *pb;            /* pointer to primitive being created */
  sblock *stb;           /* pointer to structure being created */
  long ncomps;           /* number of components in structure array */
  handle *hblock;        /* pointer to block of handles to structure array */
  sblock *sab;           /* pointer to structure array components sblock */
  BOOL structure;        /* True if we are creating a structure */
  int n;
  int i;
  BOOL toplevel;         /* TRUE for a top level object  */
  handle block;
  INT32 external;
  INT32 *origin;
  idblock *idptr;

  if (*status != SDS__OK) return;
  
  n = 0;
  
/* import parent  */

  if (parent_id == 0) 
    {
       toplevel = TRUE;
    }
  else
    {
      toplevel = FALSE;
      Sds__get_id(parent_id,&block,&external,&origin,status);
      if (*status != SDS__OK) return;
      if (external)
        {
          *status = SDS__EXTERN;
          return;
        }
      sb = (sblock*)(*block);

/*  check that it is a structure  */

      if (sb->code != SDS_STRUCT)
        {
          *status = SDS__NOTSTRUCT;
          return;
        }
    }

/*  check length of string  */

 
  if (strlen(name) > 15U)	/* Note, 15U means unsigned */
    {
       *status = SDS__LONGNAME;
       return;
    }

/*  check number of extra bytes  */

  if (nextra > MAX_EXTRA)
    {
       *status = SDS__EXTRA;
       return;
    }

/*  check code  */

  if ((code > SDS_NCODES) || (code<0) || (code == SDS_SARRAY))
    {
       *status = SDS__INVCODE;
       return;
    }

/*  check dimensions  */

  if ((ndims > MAXDIMS) || (ndims<0))
    {
       *status = SDS__INVDIMS;
       return;
    }

/*  create new block - if it is a structure we create an sblock but also
 *  copy the pointer as a pblock* - many of the fields are identical and
 *  can be accessed through this.
 *
 *  if it is a primitive or a structure array we create a pblock
 */

  if ((code == SDS_STRUCT) && (ndims == 0))
    {
      stb = (sblock*)SdsMalloc(sizeof(sblock));
      pb = (pblock*) stb;
      structure = TRUE;
    }
  else
    {
      pb = (pblock*)SdsMalloc(sizeof(pblock));
      stb = NULL;
      structure = FALSE;
    }

/* fill in fields */

  if (toplevel) 
    pb->parent = NULL;
  else
    pb->parent = block;
  pb->code = code;
  strcpy(pb->name,name);
  pb->ndims = ndims;
  if (ndims != 0)
    {
      for (i=0;i<ndims;i++)  pb->dims[i]=dims[i];
    }
  pb->nextra = nextra;
  for (i=0;i<nextra;i++)
    {
       pb->extra[i] = extra[i];
    }
  if (! structure) 
    {
      pb->data = NULL;
      pb->format = local_format[code];
    }
  else
    {
      stb->maxitems = NITEMS;
      stb->nitems = 0;
      stb->format = 0;
    }

/* extend the parent block if necessary */

  if (!toplevel)
    {
      n = sb->nitems;
      if (n >= sb->maxitems) Sds__extend_struct(&sb,status);
      if (*status != SDS__OK) return;
      sb->nitems = n+1;
    }

/* get a handle for the new component */

  pb->self = Sds__get_handle();
  if ((pb->self) == NULL)
    {
      *status = SDS__NOMEM;
      return;
    }
  *(pb->self) = (void*)pb;
  if (!toplevel) sb->item[n] = pb->self;

/* get an id for the new component  */

  *id = Sds__get_new_id(&idptr);
  if (*id == 0)
    {
      *status = SDS__NOMEM;
      return;
    }
  Sds__put_id(*id,pb->self,FALSE,NULL,status);

/*  if it is a structure array we have to create an array of handles
 *  to point to the component sblocks of the array  
 */

  if ((code == SDS_STRUCT) && (ndims != 0)) 
    {
       pb->code = SDS_SARRAY;
       ncomps = 1;
       for(i=0;i<ndims;i++)  ncomps = ncomps*dims[i];
       if (ncomps == 0) ncomps = 1;
       hblock = (handle*)SdsMalloc(ncomps*sizeof(handle));
       if (hblock == 0)
         {
            *status = SDS__NOMEM;
            return;
         }
       pb->data = (handle)hblock;
/*
 *  Now create and initialize an sblock for each component of the 
 *  structure array
 */
       for(i=0;i<ncomps;i++)
         {
            sab = Sds__create_sblock(name,nextra,extra,(long)NITEMS);
            if (sab == NULL) 
              {
                 *status = SDS__NOMEM;
                 return;
              }
            sab->parent = pb->self;
            sab->self = Sds__get_handle();
            if ((sab->self) == NULL)
              {
                 *status = SDS__NOMEM;
                 return;
              }
            *(sab->self) = (void*)sab;
            hblock[i] = sab->self;
         }       
     }
}


/*+				 S d s I n d e x

 *  Function name:
      SdsIndex

 *  Function:
      Find a structure component by position

 *  Description:
      Given the index number of a component in a structure, return an
      identifier to the component. 
      
 *  Language:
      C

 *  Declaration:
      void SdsIndex(SdsIdType parent_id, long index, SdsIdType *id, 
             StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) parent_id (SdsIdType) Identifier of the structure.
      (>) index     (long)  Index number of the component to be returned.
                            Items in a structure are numbered in order of 
                            creation starting with one.
      (<) id        (SdsIdType*) Identifier to the component.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__NOMEM => Insufficient memory for creation
                            SDS__NOTSTRUCT => parent_id not a structure
                            SDS__NOITEM => No item with that index number


 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 18-Oct-91
 *-
 *  History:
 *     18th Oct 1991. Original dated version. JAB/AAO.
 *     17th Aug 2001. Initial values for sb and data to placate gcc. KS/AAO.
 */

void SdsIndex(SdsIdType parent_id, long index, SdsIdType *id, StatusType * SDSCONST status)

{
  sblock *sb;          /*  Pointer to structure block  */
  idblock *idptr;      /*  Pointer to id block of component  */
  long code;           /*  Type code of parent  */
  INT32 *data;          /*  Pointer to external data block  */
  long nitems;         /*  Number of items in structure  */
  handle block;
  INT32 external;
  INT32 *origin;

/*  Initial values */

  sb = (sblock*) 0;
  data = (INT32*) 0;
  
/*  Import parent id */

  Sds__get_id(parent_id,&block,&external,&origin,status);
  if (*status != SDS__OK) return;
  if (external)
    {

/*  In the external case get the data block pointer, type code and number of
      items   */

      data = (INT32*)(block);
      code = ((char*)data)[0];
      nitems = ((short*)data)[1];
    }
  else
    {

/*  In the internal case get the sblock pointer, the type code, and the number
    of items  */

      sb = (sblock*)(*block);
      code = sb->code;
      nitems = sb->nitems;
    }

/*  If it is not a structure return an error   */

  if (code != SDS_STRUCT)
    {
      *status = SDS__NOTSTRUCT;
      return;
    }

  if ((index <= nitems) && (index > 0))
    {

/*  Get a new id block for the component   */

      *id = Sds__get_new_id(&idptr);
      if (*id == 0)
        {
          *status = SDS__NOMEM;
          return;
        }

/*  Fill in the id block with the appropriate component  */

      if (external)
        Sds__put_id(*id,(handle)(origin+data[4+index]),
                 external,origin,status);
      else
        Sds__put_id(*id,sb->item[index-1],external,origin,status);
    }
  else
    {
      *status = SDS__NOITEM;
      return;
    }
}


void Sds__find_ext(INT32* origin, INT32 *data, SDSCONST char *name, SdsIdType *id, StatusType * SDSCONST status)

/*
 *   Handle the external case of the SdsFind function
 */


{
  idblock *idptr;     
  BOOL found;  
  int index;
  INT32* item;
  int nitems;

  if (((char*)data)[0] != SDS_STRUCT)
    {
      *status = SDS__NOTSTRUCT;
      return;
    }
  found = FALSE;
  nitems = ((short*)data)[1];

/*  Loop over the items in the structure until a match with the name
    is found or the end of the list is reached   */

  for(index=0;(index<nitems)&&(!found);index++)
    {
      item = origin + data[5+index];
      if (name[0] == ((char*)item)[4])
        {
          if(strcmp(name,((char*)item)+4) == 0)
	    {
              *id = Sds__get_new_id(&idptr);
              if (*id == 0)
                {
                  *status = SDS__NOMEM;
                  return;
                }
              Sds__put_id(*id,(handle)item,TRUE,origin,status);
              found = TRUE;
    	    }
        }
    }
  if (! found)
    {
      *status = SDS__NOITEM;
      return;
    }
}



/*+				S d s F i n d

 *  Function name:
      SdsFind

 *  Function:
      Find a structure component by name

 *  Description:
      Given the name of a component in a structure, return an
      identifier to the component. 
      
 *  Language:
      C

 *  Declaration:
      void SdsFind(SdsIdType parent_id, char *name, SdsIdType *id, 
         StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) parent_id (SdsIdType) Identifier of the structure.
      (>) name      (char*) Name of the component to be found.
      (<) id        (SdsIdType*) Identifier to the component.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__NOMEM => Insufficient memory for creation
                            SDS__NOTSTRUCT => parent_id not a structure
                            SDS__NOITEM => No item with that name


 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 18-Oct-91
 *-
 */

void SdsFind(SdsIdType parent_id, SDSCONST char *name, SdsIdType *id, StatusType * SDSCONST status)

{
  sblock *sb;
  idblock *idptr;
  BOOL found;
  int index;
  sblock *pb;
  handle block;
  INT32 external;
  INT32 *origin;

  Sds__get_id(parent_id,&block,&external,&origin,status);
  if (*status != SDS__OK) return;
  if (external)
    {
      Sds__find_ext(origin,(INT32*)(block),name,id,status);
      return;
    }
  sb = (sblock*)(*block);
  if (sb->code != SDS_STRUCT)
    {
      *status = SDS__NOTSTRUCT;
      return;
    }
  found = FALSE;

/*  Loop over structure items until a match with the name is found or until
    the end of the list is reached   */

  for(index=0;(index<(sb->nitems))&&(!found);index++)
    {
      pb = (sblock*)(*(sb->item[index]));
      if (name[0] == (pb->name)[0])
        {
          if(strcmp(name,pb->name) == 0)
	    {
              *id = Sds__get_new_id(&idptr);
              if (*id == 0)
                {
                  *status = SDS__NOMEM;
                  return;
                }
              Sds__put_id(*id,sb->item[index],external,origin,status);
              found = TRUE;
	    }
        }
    }
  if (! found)
    {
      *status = SDS__NOITEM;
      return;
    }
}


void Sds__pointer_ext(INT32 *origin, INT32 *ptr, void **data, 
                      unsigned long *length, StatusType *status)

/*
 *   Handle the external case of the SdsPointer function
 */

{
 long i;
 long code;
 long format;
 long ndims;
 unsigned long dims[7];
 INT32 dptr;


  code = ((char*)ptr)[0];
  if (code == SDS_STRUCT) 
  {
     *status = SDS__NOTPRIM;
      return;
  }
/*
 * If the data is undefined return an error
 */
  dptr = ptr[5];
  if (dptr == 0)
    {
       *status = SDS__UNDEFINED;
       *length = 0;
       *data = NULL;
       return;
    }
/*
 *  Calculate length of the data
 */

  ndims = ((short*)ptr)[1];
  *length = nbytes[code];
  for(i=0;i<ndims;i++) 
    {
       dims[i] = ptr[6+i];
       *length = (*length)*dims[i];
    }
/*
 *  Convert the data if necessary
 */
  format = ((char*)ptr)[1];
  if (format != local_format[code]) 
    {
       Sds__convert(code,format,origin+dptr,*length);
       ((char*)ptr)[1] = local_format[code];
    }
  *data = (void*)(origin+dptr);

}
       


/*+				S d s P o i n t e r

 *  Function name:
      SdsPointer

 *  Function:
      Get a pointer to the data of a primitive item

 *  Description:
      Return a pointer to the data of a primitive item. Also return the length
      of the item. If the data item is undefined and the object is internal
      storage for the data will be created.

      SdsPointer can only be used with primitive items, not with structures.

      If necessary (e.g. if the data originated on a machine with different
      architecture) the data for the object is converted (in place) from the 
      format stored in the data item to that required for the local machine

      If the data pointed to by the pointer is updated by a calling program,
      the program should then call the function SdsFlush to ensure that the
      data is updated in the original structure. This is necessary because
      implementations on some machine architectures may have to use a copy
      of the data rather than the actual data when returning a pointer.
      
 *  Language:
      C

 *  Declaration:
      void SdsPointer(SdsIdType id, void **data, unsigned long *length, 
              StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType)  Identifier of the primitive item.
      (<) data      (void**) Address of a variable to hold the pointer.
      (<) length    (unsigned long*) Length of the data.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__NOMEM => Insufficient memory for creation
                            SDS__NOTPRIM => Not a primitive item
                            SDS__UNDEFINED => Data undefined, and object external


 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 17-Aug-2001
 *-
  *  History:
 *     18th Oct 1991. Original dated version. JAB/AAO.
 *     17th Aug 2001. Added casts to character indices to placate gcc. KS/AAO.
*/


void SdsPointer(SdsIdType id, void **data, unsigned long *length, 
          StatusType * SDSCONST status)

{
 pblock *pb;
 long i;
 handle block;
 INT32 external;
 INT32 *origin;


  Sds__get_id(id,&block,&external,&origin,status);
  if (*status != SDS__OK) return;
  if (external)
    {
      Sds__pointer_ext(origin,(INT32*)(block),data,length,status);
      return;
    }
  pb = (pblock*)(*block);
  if (pb->code == SDS_STRUCT) 
  {
     *status = SDS__NOTPRIM;
      return;
  }
/*
 *  Calculate length of the data
 */

  *length = nbytes[(int)pb->code];
  for(i=0;i<(pb->ndims);i++)  *length = (*length)*(pb->dims[i]);
/*
 * If the data is undefined create it
 */
  if (pb->data == NULL)
    {

/* Create the data block  */

       *data = (void*) SdsMalloc(*length);
       if (*data == NULL) 
         {
            *status = SDS__NOMEM;
            return;
         }

/* Create a handle to it */

       pb->data = Sds__get_handle();
       if (pb->data == NULL) 
         {
            *status = SDS__NOMEM;
            return;
         }

/* Set handle to point to data */

       *(pb->data) = *data;
       pb->format = local_format[(int)pb->code];
    }
  else

/* Data is already there so return value in handle */

    {
       *data = *(pb->data);
    }
/*
 *  Convert the data if necessary
 */
  if (pb->format != local_format[(int)pb->code]) 
    {
       Sds__convert((long)(pb->code),(long)(pb->format),
                (INT32*)(*(pb->data)),*length);
       pb->format = local_format[(int)pb->code];
    }

}



/*+				S d s F l u s h

 *  Function name:
      SdsFlush

 *  Function:
      Flush data updated via a pointer

 *  Description:
      If a primitive data item is accessed via SdsPointer, and the data
      array updated via the returned pointer, then SdsFlush must be called
      to ensure that the data is updated in the original structure.

      This must be done since implementations on some machine architectures
      may have to use a copy of the data rather than the actual data when
      returning a pointer.
      
 *  Language:
      C

 *  Declaration:
      void SdsFlush(SdsIdType id, StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType)  Identifier of the primitive item.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID   => Identifier invalid
                            SDS__NOTPRIM => Not a primitive item


 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 7-Feb-92
 *-
 */


void SdsFlush(SdsIdType id, StatusType * SDSCONST status)

{

/*  This is a null routine in this implementation   */

}


void Sds__align(unsigned long *ptr, SdsCodeType code, char* data)

/*
 *  Adjust ptr to match the alignment requirements for an item of type
 *  code.
 */

{

/*  The test structure below is used to determine the alignment requirements
     for a given type on the current architecture   */

 typedef struct test
  {
     INT32 s1;
     char c1;
     short s2;
     INT32 l1;
     char c2;
     INT32 l2;
     float f1;
     char c3;
     float f2;
     double d1;
     char c4;
     double d2;
     INT64 j1;
     char c5;
     INT64 j2;
  }  test;

 test t;
 short alignment;
 short adjustment;
 short temp;


/*  Determine the alignment required for type code  */

 switch (code)
   {
     case SDS_SHORT:
     case SDS_USHORT:
       alignment = ((char*)(&t.s2))-(&t.c1);
       break;
     case SDS_INT:
     case SDS_UINT:
       alignment = ((char*)(&t.l2))-(&t.c2);
       break;
     case SDS_FLOAT:
       alignment = ((char*)(&t.f2))-(&t.c3);
       break;
     case SDS_DOUBLE:
       alignment = ((char*)(&t.d2))-(&t.c4);
       break;
     case SDS_I64:
     case SDS_UI64:
       alignment = ((char*)(&t.j2))-(&t.c5);
       break;
     default:
       return;
   }

/*  if ptr does not have the correct alignment, adjust it by incrementing
    it to match the required alignment   */

  temp = *ptr % alignment;
  if (temp == 0) return;
  adjustment = alignment-temp;
  *ptr = *ptr+adjustment;
  return;
}

void Sds__get_int(pblock *pb, unsigned long length, unsigned long offset, 
       void* data, unsigned long *actlen, StatusType * SDSCONST status)

/*
 *  Handle the internal primitive case of the SdsGet function 
 */

{
  long i;

/*
 * If the data is undefined return an error
 */
  if (pb->data == NULL)
    {
       *status = SDS__UNDEFINED;
       *actlen = 0;
       return;
    }
/*
 *  Calculate length of the data
 */

  *actlen = nbytes[(int)pb->code];
  for(i=0;i<(pb->ndims);i++)  *actlen = (*actlen)*(pb->dims[i]);
  
/*
 *  Convert the data if necessary
 */
  if (pb->format != local_format[(int)pb->code]) 
    {
       Sds__convert((long)(pb->code),(long)(pb->format),
             (INT32*)(*(pb->data)),*actlen);
       pb->format = local_format[(int)pb->code];
    }

  *actlen = *actlen-(offset*nbytes[(int)pb->code]);
  if (length < *actlen) *actlen = length;
/*
 *  Copy the data
 */
  if (pb->ndims == 0)
    {
      if (*actlen == 4) 
         *(INT32*)data = *((INT32*)*(pb->data)+offset);
      else if (*actlen == 8)
        {
          *(INT32*)data = *((INT32*)*(pb->data)+2*offset);
          *((INT32*)data+1) = *((INT32*)*(pb->data)+2*offset+1);
        }
      else if (*actlen == 2)
          *(short*)data = *((short*)*(pb->data)+offset);
      else if (*actlen == 1)
          *(char*)data = *((char*)*(pb->data)+offset);
    }
  else
     memcpy(data,((char*)*(pb->data))+offset*nbytes[(int)pb->code],*actlen);
}


       

void Sds__get_ext(INT32 *origin, INT32 *ptr, unsigned long length, 
             unsigned long offset, void *data, unsigned long *actlen, 
             StatusType * SDSCONST status)

/*
 *  Handle the external primitive case of the SdsGet function
 */

{
 int i;
 long code;
 long format;
 long ndims;
 unsigned long dims[7];
 INT32 dptr;

  code = ((char*)ptr)[0];
  if (code == SDS_STRUCT) 
  {
     *status = SDS__NOTPRIM;
      return;
  }
/*
 * If the data is undefined return an error
 */
  dptr = ptr[5];
  if (dptr == 0)
    {
       *status = SDS__UNDEFINED;
       *actlen = 0;
       return;
    }
/*
 *  Calculate length of the data
 */

  ndims = ((short*)ptr)[1];
  *actlen = nbytes[code];
  for(i=0;i<ndims;i++) 
    {
       dims[i] = ptr[6+i];
       *actlen = (*actlen)*dims[i];
    }
  
/*
 *  Convert the data if necessary
 */
  format = ((char*)ptr)[1];
  if (format != local_format[code]) 
    {
       Sds__convert(code,format,origin+dptr,*actlen);
       ((char*)ptr)[1] = local_format[code];
    }

  *actlen = *actlen-(offset*nbytes[code]);
  if (length < *actlen) *actlen = length;
/*
 *  Copy the data
 */
  if (ndims == 0)
    {
      if (*actlen == 4) 
        *(INT32*)data = *((INT32*)(origin+dptr)+offset);
      else if (*actlen == 8)
        {
          *(INT32*)data = *((INT32*)(origin+dptr)+2*offset);
          *((INT32*)data+1) = *((INT32*)(origin+dptr)+2*offset+1);
        }
      else if (*actlen == 2)
          *(short*)data = *((short*)(origin+dptr)+offset);
      else if (*actlen == 1)
          *(char*)data = *((char*)(origin+dptr)+offset);
    }
  else
     memcpy(data,((char*)(origin+dptr)+offset*nbytes[code]),*actlen);
}
       



void Sds__get_struct(sblock* sb, unsigned long length, void *data, 
          unsigned long *actlen, StatusType * SDSCONST status)

/*
 *  Handle the internal structure case of the SdsGet function
 */

{
   int i;
   int ncomps;
   pblock *pb;
   handle *hblock;
   unsigned long ilen;

   if (sb->code == SDS_STRUCT)
     {

/*  If it is a structure get all the component of the structure in sequence  */

       for (i=0;i<sb->nitems;i++)
         Sds__get_struct((sblock*)(*(sb->item[i])),length,data,
               actlen,status);
     }
   else if (sb->code == SDS_SARRAY)
     {

/*  If it is a structure array get all the components in sequence  */

       pb = (pblock*)sb;
       hblock = (handle*)(pb->data);
       ncomps = 1;
       for (i=0;i<pb->ndims;i++) ncomps = ncomps*pb->dims[i];
       for (i=0;i<ncomps;i++)
         Sds__get_struct((sblock*)(*(hblock[i])),length,data,actlen,status);
     } 
   else
     {

/*  If it is a primitive get the data, copying it to the output buffer
    after the current value of actlen bytes - then increment actlen.
    Ensure correct alignment by calling Sds__align to adjust actlen to
    give the correct alignment for the current type.   */ 

       pb = (pblock*)sb;
       Sds__align(actlen,pb->code,(char*)data);
       if (*actlen > length)
         {
           *actlen = length;
           return;
         }
       Sds__get_int(pb,length-*actlen,(long)0,(void*)(((char*)data)+(*actlen)),
           &ilen,status);
       if (*status != SDS__OK) return;
       *actlen += ilen;
       if (*actlen >= length) return;
     }
}

void Sds__get_struct_ext(INT32 *origin, INT32 *ptr, unsigned long length, void *data, 
          unsigned long *actlen, StatusType *status)

/*
 *  Handle the external structure case of the SdsGet function
 */

{
   int i;
   int ncomps;
   int nitems;
   unsigned long ilen;
   long code;

   code = ((char*)ptr)[0];
   nitems = ((short*)ptr)[1];
   if (code == SDS_STRUCT)
     {

/*  If it is a structure get all the component of the structure in sequence  */

       for (i=0;i<nitems;i++)
         Sds__get_struct_ext(origin,origin+ptr[5+i],length,
                   data,actlen,status);
     }
   else if (code == SDS_SARRAY)
     {

/*  If it is a structure array get all the components in sequence  */

       ncomps = 1;
       for (i=0;i<nitems;i++) ncomps = ncomps*ptr[5+i];
       for (i=0;i<ncomps;i++)
       {
         Sds__get_struct_ext(origin,origin+ptr[5+nitems+i],
            length,data,actlen,status);
       }     
     } 
   else
     {

/*  If it is a primitive get the data, copying it to the output buffer
    after the current value of actlen bytes - then increment actlen.
    Ensure correct alignment by calling Sds__align to adjust actlen to
    give the correct alignment for the current type.   */ 

       Sds__align(actlen,code,(char*)data);
       if (*actlen > length)
         {
           *actlen = length;
           return;
         }           
       Sds__get_ext(origin,ptr,length-*actlen,(long)0,
             (void*)(((char*)data)+(*actlen)),&ilen,status);
       if (*status != SDS__OK) return;
       *actlen += ilen;
       if (*actlen >= length) return;
     }
}






/*+				  S d s G e t 

 *  Function name:
      SdsGet

 *  Function:
      Read the data from an object

 *  Description:
      The object may be a primitive item or a structure or structure array.
      Read the data from an item into a buffer. If the object is primitive 
      data is transferred starting at the position in the item specified by 
      offset, until the buffer if filled, or the end of the data array is 
      reached.

      If the object is a structure or structure array, the data from all its
      primitive components are copied into the buffer in order of their
      position in the structure. Alignment adjustments are made as necessary 
      to match the alignment of an C struct equivalent to the SDS structure.
      (Since these alignment requirements are machine dependent the actual
      sequence of bytes returned could be different on different machines).
      In the structure or structure array case the offset parameter is
      ignored.

      If necessary (e.g. if the data originated on a machine with different
      architecture) the data for the object is converted (in place) from the 
      format stored in the data item to that required for the local machine.
      
 *  Language:
      C

 *  Declaration:
      void SdsGet(SdsIdType id, unsigned long length, unsigned long offset, 
             void *data, unsigned long *actlen, StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType) Identifier of the object.
      (>) length    (unsigned long)  Length in bytes of the buffer supplied to receive
                            the data.
      (>) offset    (unsigned long)  Offset into the data object at which to start
                            reading data. The offset is measured in units of
                            the size of each individual item in the array -
                            e.g. 4 bytes for an INT or 8 bytes for a DOUBLE.
                            The offset is zero to start at the beginning of the
                            array. This parameter is ignored if the object
                            is a structure or structure array.
      (<) data      (void*) Buffer to receive the data.
      (<) actlen    (unsigned long*) Actual number of bytes transferred.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID => Invalid identifier
                            SDS__UNDEFINED => Data undefined


 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 18-Oct-91
 *-
 */



void SdsGet(SdsIdType id, unsigned long length, unsigned long offset, void *data, 
             unsigned long *actlen, StatusType * SDSCONST status)

{
 pblock *pb;
 handle block;
 INT32 external;
 INT32 *origin;

  Sds__get_id(id,&block,&external,&origin,status);

  if (*status != SDS__OK) return;
  if (external)
    {
       if (((char*)(block))[0] == SDS_STRUCT ||
          ((char*)(block))[0] == SDS_SARRAY)
         {

/*  External structure or structure array  */

           *actlen = 0;
           Sds__get_struct_ext(origin,(INT32*)(block),length,
                data,actlen,status);
           return;
         } 

/*  External Primitive  */

       Sds__get_ext(origin,(INT32*)(block),length,offset,
                  data,actlen,status);
       return;
    }
  pb = (pblock*)(*block);
  if (pb->code == SDS_STRUCT || pb->code == SDS_SARRAY) 
  {

/*  Internal structure or structure array  */

      *actlen = 0;
      Sds__get_struct((sblock*)pb,length,data,actlen,status);
      return;
  }

/*  Internal primitive  */

  Sds__get_int(pb,length,offset,data,actlen,status);
}



       

void Sds__put_ext(INT32 *origin, INT32 *ptr, unsigned long length, 
      unsigned long offset, SDSCONST void *data, unsigned long *actlen, 
      StatusType * SDSCONST status)

/*
 *  Handle external primitive case of the SdsPut function
 */


{
 int i;
 long code;
 long format;
 long ndims;
 unsigned long dims[7];
 INT32 dptr;

  code = ((char*)ptr)[0];
  if (code == SDS_STRUCT) 
  {
     *status = SDS__NOTPRIM;
      return;
  }
/*
 * If the data is undefined return an error
 */
  dptr = ptr[5];
  if (dptr == 0)
    {
       *status = SDS__UNDEFINED;
       return;
    }
/*
 *  Calculate length of the data
 */

  ndims = ((short*)ptr)[1];
  *actlen = nbytes[code];
  for(i=0;i<ndims;i++) 
    {
       dims[i] = ptr[6+i];
       *actlen = (*actlen)*dims[i];
    }
  
  if (length > (*actlen-offset*nbytes[code]))
    {
       length = *actlen-offset*nbytes[code];
    }

/*
 *  Convert the data if necessary
 */
  format = ((char*)ptr)[1];
  if ((format != local_format[code]) && (length < *actlen)) 
    {
       Sds__convert(code,format,origin+dptr,*actlen);
       ((char*)ptr)[1] = local_format[code];
    }

/*
 *  Copy the data
 */
  if (ndims == 0)
    {
      if (length == 4) 
        *((INT32*)(origin+dptr)+offset) = *(INT32*)data;
      else if (length == 8)
        {
          *((INT32*)(origin+dptr)+2*offset) = *(INT32*)data;
          *((INT32*)(origin+dptr)+2*offset+1) = *((INT32*)data+1);
        }
      else if (length == 2)
          *((short*)(origin+dptr)+offset) = *(short*)data;
      else if (length == 1)
          *((char*)(origin+dptr)+offset) = *(char*)data;
    }
  else
     memcpy(((char*)(origin+dptr)+offset*nbytes[code]),data,length);
  *actlen = length;
}



void Sds__put_int(pblock *pb, unsigned long length, unsigned long offset,
             SDSCONST void *data, unsigned long *actlen, StatusType * SDSCONST status)
       

{
  int i;
  void *Sdsdata;

  *actlen = nbytes[(int)pb->code];
  for(i=0;i<(pb->ndims);i++)  *actlen = (*actlen)*(pb->dims[i]);
  if (length > (*actlen-offset*nbytes[(int)pb->code]))
    {
       length = *actlen-offset*nbytes[(int)pb->code];
    }

/*
 * If the data is undefined create it
 */
  if (pb->data == NULL)
    {

/* Create the data block  */

       Sdsdata = (void*)SdsMalloc(*actlen);
       if (Sdsdata == NULL) 
         {
            *status = SDS__NOMEM;
            return;
         }

/* Create a handle to it */

       pb->data = Sds__get_handle();
       if (pb->data == NULL) 
         {
            *status = SDS__NOMEM;
            return;
         }

/* Set handle to point to data */

       *(pb->data) = Sdsdata;
       pb->format = local_format[(int)pb->code];
    }

/* If the data was previously defined, and the format is not the
   local format and we are not writing all the data, then the
   data have to be converted to the local format  */

  else if ((pb->format != local_format[(int)pb->code]) &&
           (length < *actlen))
    {
       Sds__convert((long)(pb->code),(long)(pb->format),
             (INT32*)(*(pb->data)),*actlen);
       pb->format = local_format[(int)pb->code];
    }
/*
 *  Copy the data
 */
  if (pb->ndims == 0)
    {
      if (length == 4) 
         *((INT32*)(*(pb->data))+offset) = *(INT32*)data;
      else if (length == 8)
        {
          *((INT32*)(*(pb->data))+2*offset) = *(INT32*)data;
          *((INT32*)(*(pb->data))+2*offset+1) = *((INT32*)data+1);
        } 
      else if (length == 2)
          *((short*)(*(pb->data))+offset) = *(short*)data;
      else if (length == 1)
          *((char*)(*(pb->data))+offset) = *(char*)data;
    }
  else
     memcpy(((char*)*(pb->data)+offset*nbytes[(int)pb->code]),data,length);
  *actlen = length;
}
       



void Sds__put_struct(sblock* sb, unsigned long length, 
	   SDSCONST void *data, 
           unsigned long *pos, StatusType * SDSCONST status)

/*
 *  Handle the internal structure case of the SdsPut function
 */

{
   int i;
   int ncomps;
   pblock *pb;
   handle *hblock;
   unsigned long ilen;

   if (sb->code == SDS_STRUCT)
     {

/*  If it is a structure put all the component of the structure in sequence  */

       for (i=0;i<sb->nitems;i++)
         Sds__put_struct((sblock*)(*(sb->item[i])),length,data,pos,status);
     }
   else if (sb->code == SDS_SARRAY)
     {

/*  If it is a structure array put all the components in sequence  */

       pb = (pblock*)sb;
       hblock = (handle*)(pb->data);
       ncomps = 1;
       for (i=0;i<pb->ndims;i++) ncomps = ncomps*pb->dims[i];
       for (i=0;i<ncomps;i++)
         Sds__put_struct((sblock*)(*(hblock[i])),length,data,pos,status);
     } 
   else
     {

/*  If it is a primitive put the data, copying it from the input buffer
    after the current value of actlen bytes - then increment pos.
    Ensure correct alignment by calling Sds__align to adjust pos to
    give the correct alignment for the current type.   */ 

       pb = (pblock*)sb;
       Sds__align(pos,pb->code,(char*)data);
       if (*pos > length) return;
       Sds__put_int(pb,length-*pos,(long)0,(void*)(((char*)data)+(*pos)),
           &ilen,status);
       if (*status != SDS__OK) return;
       *pos += ilen;
       if (*pos >= length) return;
     }
}

void Sds__put_struct_ext(INT32 *origin, INT32 *ptr, unsigned long length, 
	  SDSCONST void *data, 
          unsigned long *pos, StatusType * SDSCONST status)

/*
 *  Handle the external structure case of the SdsPut function
 */

{
   int i;
   int ncomps;
   int nitems;
   unsigned long ilen;
   long code;

   code = ((char*)ptr)[0];
   nitems = ((short*)ptr)[1];
   if (code == SDS_STRUCT)
     {

/*  If it is a structure get all the component of the structure in sequence  */

       for (i=0;i<nitems;i++)
         Sds__put_struct_ext(origin,origin+ptr[5+i],length,
                   data,pos,status);
     }
   else if (code == SDS_SARRAY)
     {

/*  If it is a structure array get all the components in sequence  */

       ncomps = 1;
       for (i=0;i<nitems;i++) ncomps = ncomps*ptr[5+i];
       for (i=0;i<ncomps;i++)
         Sds__put_struct_ext(origin,origin+ptr[5+nitems+i],
            length,data,pos,status);
     } 
   else
     {

/*  If it is a primitive get the data, copying it from the input buffer
    after the current value of pos bytes - then increment pos.
    Ensure correct alignment by calling Sds__align to adjust pos to
    give the correct alignment for the current type.   */ 

       Sds__align(pos,code,(char*)data);
       if (*pos > length) return;
       Sds__put_ext(origin,ptr,length-*pos,(long)0,
             (void*)(((char*)data)+(*pos)),&ilen,status);
       if (*status != SDS__OK) return;
       *pos += ilen;
       if (*pos >= length) return;
     }
}





/*+				S d s P u t

 *  Function name:
      SdsPut

 *  Function:
      Write data to an object.

 *  Description:
      Write data into an object. The object may be a primitive item or a
      structure or structure array.

      If the object is a structure or structure array, the data from the
      the buffer is copied into its primitive components in order of their
      position in the structure. Alignment adjustments are made as necessary 
      to match the alignment of a C struct equivalent to the SDS structure.
      In the structure or structure array case the offset parameter is
      ignored.

      If the object is primitive data is transferred starting
      at the position in the item specified by offset. 

      If the data was previously undefined memory for the data is allocated at
      this time.

      If the data is too long to fit into the object, it will be truncated.
      
 *  Language:
      C

 *  Declaration:
      void SdsPut(SdsIdType id, unsigned long length, unsigned long offset, 
             void *data, long *status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType) Identifier of the primitive item.
      (>) length    (unsigned long)  Length in bytes of the buffer containing
                            the data.
      (>) offset    (unsigned long)  Offset into the data object at which to start
                            writing data. The offset is measured in units of
                            the size of each individual item in the array -
                            e.g. 4 bytes for an INT or 8 bytes for a DOUBLE.
                            The offset is zero to start at the beginning of the
                            array.
      (>) data      (void*) Buffer containing the data.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__NOMEM => Insufficient memory for creation
                            SDS__NOTPRIM => Not a primitive item
                            SDS__UNDEFINED => Data undefined, and object external


 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 18-Oct-91
 *-
 */

void SdsPut(SdsIdType id, unsigned long length, unsigned long offset, SDSCONST void *data, 
                StatusType * SDSCONST status)

{
 pblock *pb;
 unsigned long pos;
 handle block;
 INT32 external;
 INT32 *origin;

  Sds__get_id(id,&block,&external,&origin,status);

  if (*status != SDS__OK) return;
  if (external)
    {
       if (((char*)(block))[0] == SDS_STRUCT ||
          ((char*)(block))[0] == SDS_SARRAY)
         {

/*  External structure or structure array  */

           pos = 0;
           Sds__put_struct_ext(origin,(INT32*)(block),length,
                data,&pos,status);
           return;
         } 

/*  External Primitive   */

       Sds__put_ext(origin,(INT32*)(block),length,offset,
            data,&pos,status);
       return;
    }
  pb = (pblock*)(*block);
  if ((pb->code == SDS_STRUCT) ||
      (pb->code == SDS_SARRAY)) 
  {
     pos = 0;
     Sds__put_struct((sblock*)pb,length,data,&pos,status);
     return;
  }
  pos = 0;
  Sds__put_int(pb,length,offset,data,&pos,status);
}


void Sds__cell_ext(INT32 *origin, INT32 *data, unsigned long nindices, 
   SDSCONST unsigned long *indices, SdsIdType *id, StatusType * SDSCONST status)

{
idblock *idptr;
long size,addr;
long i;
long ndims;
long pos;
unsigned long dims[7];

/*
 *  Check that input is a structure array
 */
  if (((char*)data)[0] != SDS_SARRAY)
    {
       *status = SDS__NOTARRAY;
       return;
    }
/*
 *  Check that the indices are OK
 */
  ndims = ((short*)data)[1];
  pos = 5;
  if ((nindices != ndims) && (nindices != 1))
    {
       *status = SDS__INDEXERR;
       return;
    }
  if (nindices != 1)
    {  
      for (i=0;i<nindices;i++) 
        {
           dims[i] = data[pos];
           pos++;
           if ((indices[i] > dims[i]) || (indices[i] <= 0))
             {
                *status = SDS__INDEXERR;
                return;
             }
        }
    }
  else
    {
       size = 1;
       for (i=0;i<ndims;i++)
         {
            size = size * data[pos];
            pos++;
         }  
       if ((indices[0] > size) || (indices[0] <= 0))
         {
            *status = SDS__INDEXERR;
            return;
         }
    }
  size = 1;
  addr = 0;
  for (i=0;i<nindices;i++)
    {
       addr = addr+size*(indices[i]-1);
       size = size*dims[i];
    }
  *id = Sds__get_new_id(&idptr);
  if (*id == 0)
    {
       *status = SDS__NOMEM;
       return;
    }
  Sds__put_id(*id,(handle)(origin + data[pos+addr]),TRUE,origin,status);

}
  

/*+				 S d s C e l l

 *  Function name:
      SdsCell

 *  Function:
      Find component of a structure array

 *  Description:
      Given the indices to a component of a structure array, return an
      identifier to the component.
      
 *  Language:
      C

 *  Declaration:
      void SdsCell(SdsIdType array_id, long nindices, unsigned long *indices, 
               SdsIdType *id, StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) array_id  (SdsIdType) Identifier of the structure array.
      (>) nindices  (long)  Number of indices supplied in the
                            array indices. This should be one or the same as
                            the number of dimensions of the array.
      (>) indices   (unsigned long*)  An array of length nindices containing 
                            the indices to the component of the structure
                            array. If nindicies is 1, then treat the 
                            structure array as having only one dimension even
                            if it has more.
      (<) id        (SdsIdType*) The identifier of the component.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__NOMEM => Insufficient memory for creation
                            SDS__NOTARRAY => Not a structure array.
                            SDS__INDEXERR => Indices invalid.


 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 18-Oct-91
 *-
 */



void SdsCell(SdsIdType array_id, long nindices, 
	     SDSCONST unsigned long *indices, 
             SdsIdType *id, StatusType * SDSCONST status)

{
pblock *pb;
handle *hblock;
idblock *idptr;
long size,addr;
long i;
handle block;
INT32 external;
INT32 *origin;

  Sds__get_id(array_id,&block,&external,&origin,status);
  if (*status != SDS__OK) return;
  if (external)
    {
       Sds__cell_ext(origin,(INT32*)(block),nindices,
                 indices,id,status);
       return;
    }
  pb = (pblock*)(*block);
/*
 *  Check that input is a structure array
 */
  if (pb->code != SDS_SARRAY)
    {
       *status = SDS__NOTARRAY;
       return;
    }
/*
 *  Check that the indices are OK
 */
  if ((nindices != pb->ndims) && (nindices != 1))
    {
       *status = SDS__INDEXERR;
       return;
    }
  if (nindices != 1)
    {
      for (i=0;i<nindices;i++) 
        {
           if ((indices[i] > pb->dims[i]) || (indices[i] <= 0))
             {
                *status = SDS__INDEXERR;
                return;
             }
         }
    }
  else
    {
       size = 1;
       for (i=0;i<(pb->ndims);i++) size = size * pb->dims[i];
       if ((indices[0] > size) || (indices[0] <= 0))
         {
            *status = SDS__INDEXERR;
            return;
         }
     }
  size = 1;
  addr = 0;
  for (i=0;i<nindices;i++)
    {
       addr = addr+size*(indices[i]-1);
       size = size*pb->dims[i];
    }
  hblock = (handle*)(pb->data);
  *id = Sds__get_new_id(&idptr);
  if (*id == 0)
    {
       *status = SDS__NOMEM;
       return;
    }
  Sds__put_id(*id,hblock[addr],FALSE,origin,status);
}
  


/*+				 S d s I n s e r t C e l l

 *  Function name:
      SdsInsertCell

 *  Function:
      Insert object into a structure array

 *  Description:
      Insert a top level object into a structure array at a position specified by its
      indices. Delete any object that is currently at that position.
      
 *  Language:
      C

 *  Declaration:
      void SdsInsertCell(SdsIdType array_id, long nindices, unsigned long *indices, 
               SdsIdType id, StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) array_id  (SdsIdType) Identifier of the structure array.
      (>) nindices  (long)  Number of indices supplied in the
                            array indices. This should be 1 or the same as
                            the number of dimensions of the array.
      (>) indices   (unsigned long*)  An array of length nindices containing 
                            the indices to the component of the structure
                            array.  If nindicies is 1, then treat the 
			    structure array as having only one dimension even
			    if it has more.
      (>) id        (SdsIdType) The identifier of the component to be inserted.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__EXTERN => Structure array or object is external.
                            SDS__NOTARRAY => Not a structure array.
                            SDS__INDEXERR => Indices invalid.
                            SDS__NOTTOP => Not a top level object.


 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 24-Aug-96
 *-
 */

void Sds__delete(sblock *sb);

void SdsInsertCell(SdsIdType array_id, long nindices, 
	     SDSCONST unsigned long *indices, 
             SdsIdType id, StatusType * SDSCONST status)

{
pblock *pb;
sblock *sb;
handle *hblock;
long size,addr;
long i;
handle block;
INT32 external;
INT32 *origin;

  Sds__get_id(array_id,&block,&external,&origin,status);
  if (*status != SDS__OK) return;
  if (external)
    {
       *status = SDS__EXTERN;
       return;
    }
  pb = (pblock*)(*block);
/*
 *  Check that input is a structure array
 */
  if (pb->code != SDS_SARRAY)
    {
       *status = SDS__NOTARRAY;
       return;
    }
/*
 *  Check that the indices are OK
 */
  if ((nindices != pb->ndims) && (nindices != 1))
    {
       *status = SDS__INDEXERR;
       return;
    }
  if (nindices != 1)
    {  
  for (i=0;i<nindices;i++) 
        {
          if ((indices[i] > pb->dims[i]) || (indices[i] <= 0))
            {
               *status = SDS__INDEXERR;
               return;
            }
        }
    }
   else
    {
       size = 1;
       for (i=0;i<(pb->ndims);i++) size = size * pb->dims[i];
       if ((indices[0] > size) || (indices[0] <= 0))
         {
            *status = SDS__INDEXERR;
            return;
         }
     }
 size = 1;
  addr = 0;
  for (i=0;i<nindices;i++)
    {
       addr = addr+size*(indices[i]-1);
       size = size*pb->dims[i];
    }
  hblock = (handle*)(pb->data);
  Sds__get_id(id,&block,&external,&origin,status);
  if (*status != SDS__OK) return;
  if (external)
    {
       *status = SDS__EXTERN;
       return;
    }
  sb = (sblock*)(*block);
  if (sb->parent != NULL)
    {
       *status = SDS__NOTTOP;
       return;
    }
  if (hblock[addr] != NULL)
    {
       Sds__delete(*hblock[addr]);
    }   
  hblock[addr] = block;
  sb->parent = pb->self;
}
  

void Sds__delete(sblock *sb)
   
{
pblock *pb;
long i;
long ncomps;
handle *hblock;

  if (sb->code == SDS_STRUCT)
    {
       for (i=0;i<sb->nitems;i++)
         {
            Sds__delete((sblock*)(*(sb->item[i])));
         }
       if (sb->self)
         {
           SdsFree(sb->self);
           sb->self = NULL;
         } 
       SdsFree(sb);
    }
  else if (sb->code == SDS_SARRAY)
    {
       pb = (pblock*)sb;
       hblock = (handle*)(pb->data);
       ncomps = 1;
       for (i=0;i<pb->ndims;i++) ncomps = ncomps*pb->dims[i];
       for (i=0;i<ncomps;i++)
         {
            Sds__delete((sblock*)(*(hblock[i])));
         }
       if (pb->self)
         SdsFree(pb->self); 
       pb->self = NULL;
       SdsFree(hblock);
       SdsFree(sb);
    }
  else
    {
       pb = (pblock*)sb;
       if (pb->data != NULL)
       {
          SdsFree (*(pb->data));
          SdsFree (pb->data);
       }
       if (pb->self)
         SdsFree(pb->self);  
       pb->self = NULL;
       SdsFree(pb);
    }
}       


/*+				S d s D e l e t e

 *  Function name:
      SdsDelete

 *  Function:
      Delete an object

 *  Description:
      Delete an object, freeing any memory associated with it. Subsequent
      attempts to access the object through any identifier associated with
      it will return the SDS__BADID status. A structure array element cannot 
      be deleted. An attempt to do so will result in the SDS__ILLDEL status.

     Deleting an object does not free the memory associated with the
     identifier referencing it. This memory can be freed with the 
     SdsFreeId function.

      
 *  Language:
      C

 *  Declaration:
      void SdsDelete(SdsIdType id, long *status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType)  Identifier of the object to be deleted.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID => The identifier is invalid.
                            SDS__EXTERN => Object is external.
                            SDS__ILLDEL => Object cannot be deleted.


 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 18-Oct-91
 *-
 */

void SdsDelete(SdsIdType id, StatusType * SDSCONST status)

{
   sblock *sb;
   sblock *parent;
   long index;
   handle block;
   INT32 external;
   INT32 *origin;

   Sds__get_id(id,&block,&external,&origin,status);
   if (*status != SDS__OK) return;
   if (external)
     {
       *status = SDS__EXTERN;
       return;
     }
   sb = (sblock*)(*block);
   if ((sb->parent) != NULL)
     {

/*  Find the parent of the object to be deleted  */

        parent = (sblock*)(*(sb->parent));
        if ((parent->nitems == 0) || (parent->code == SDS_SARRAY))
          {
            *status = SDS__ILLDEL;
            return;
          }
        index = 0;

/*  Loop through the parents item list to find the item to be deleted  */

        while (*(parent->item[index]) != sb)
          {
             index++;
             if (index >= (parent->nitems))
               {

/*  Can't find the item - must be a structure array component for
    which deletion is illegal  */

                  *status = SDS__ILLDEL;
                  return;
               }
          }
        
/*  Shift up following items in list by one and reduce item count by one */

        for (;index < parent->nitems;index++)
          parent->item[index] = parent->item[index+1];
        (parent->nitems)--;
     }

/*  Now go ahead and delete the object  */

  Sds__delete(sb);
  Sds__put_id(id,&null_handle,external,origin,status);
}


void Sds__size(sblock *sb, INT32 *defn, INT32 *data, int define)

{
   INT32 tdata;
   INT32 tdefn;
   pblock *pb;
   int ncomps;
   handle *hblock;
   long i;

   if (sb->code == SDS_STRUCT)
     {
        *data=0;
        *defn=22 + 4*sb->nitems + sb->nextra;
        for (i=0;i<sb->nitems;i++)
          {
             Sds__size((sblock*)(*(sb->item[i])),&tdefn,&tdata,define);
             *data += tdata;
             *defn += tdefn;
          }
     }
   else if (sb->code == SDS_SARRAY)
     {
        pb = (pblock*)sb;
        hblock = (handle*)(pb->data);
        ncomps=1;
        for (i=0;i<pb->ndims;i++) ncomps = ncomps*pb->dims[i];
        *defn = 22+4*pb->ndims+pb->nextra+4*ncomps;
        *data = 0;
        for (i=0;i<ncomps;i++)
          {
             Sds__size((sblock*)(*(hblock[i])),&tdefn,&tdata,define);
             *data += tdata;
             *defn += tdefn;
          }
     }
   else
     {
        pb = (pblock*)sb;
        *defn = 26 + 4*pb->ndims + pb->nextra;
        ncomps = 1;
        for (i=0;i<pb->ndims;i++) ncomps = ncomps*pb->dims[i];
        if ((pb->data == NULL) && (!define))
          *data = 0;
        else
          {
            *data = ncomps*nbytes[(int)pb->code];
            if ((pb->code == SDS_DOUBLE) || (pb->code == SDS_I64) ||
                (pb->code == SDS_UI64)) *data += 4;
          }
     }
   while ((*defn % 4) != 0) (*defn)++;
   while ((*data % 4) != 0) (*data)++;
}


void Sds__size_ext(INT32 *origin, INT32 *edata, INT32 *defn, INT32 *data,
                     int define)

{
   INT32 tdata;
   INT32 tdefn;
   int ncomps;
   char code;
   int i;
   int nextra;
   int nitems;
   INT32 pos;

   code = ((char*)edata)[0];
   nitems = ((short*)edata)[1];
   if (code == SDS_STRUCT)
     {
        nextra = ((short*)edata)[10+2*nitems];
        *data=0;
        *defn=22 + 4*nitems + nextra;
        for (i=0;i<nitems;i++)
          {
             Sds__size_ext(origin,origin+(edata[5+i]),&tdefn,&tdata,define);
             *data += tdata;
             *defn += tdefn;
          }
     }
   else if (code == SDS_SARRAY)
     {
        ncomps=1;
        pos = 5;
        for (i=0;i<nitems;i++) 
          {
             ncomps = ncomps*edata[pos];
             pos++;
	   }
        nextra = ((short*)edata)[2*(pos+ncomps)];
        *defn = 22+4*nitems+nextra+4*ncomps;
        *data = 0;
        for (i=0;i<ncomps;i++)
          {
             Sds__size_ext(origin,origin+(edata[5+nitems+i]),&tdefn,&tdata,define);
             *data += tdata;
             *defn += tdefn;
          }
     }
   else
     {
        ncomps = 1;
        pos = 6;
        for (i=0;i<nitems;i++) 
          {
            ncomps = ncomps*edata[pos];
            pos++;
          }
        nextra = ((short*)edata)[2*pos];
        *defn = 26 + 4*nitems + nextra;
        if ((edata[5] == 0) && (!define))
          *data = 0;
        else
          {
            *data = ncomps*nbytes[(int)code];
            if ((code == SDS_DOUBLE) || (code == SDS_I64) ||
                (code == SDS_UI64)) *data += 4;
          }
     }
   while ((*defn % 4) != 0) (*defn)++;
   while ((*data % 4) != 0) (*data)++;
}


/*+				  S d s S i z e

 *  Function name:
      SdsSize

 *  Function:
      Find the buffer size needed to export an object

 *  Description:
      Return the size which will be needed for a buffer into which the object
      can be exported using the SdsExport function.
      
 *  Language:
      C

 *  Declaration:
      void SdsSize(SdsIdType id, unsigned long *bytes, StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType)  Identifier of the object.
      (<) bytes     (unsigned long*) Size in bytes of required buffer.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID => The identifier is invalid.
                            SDS__EXTERN => Object is external.

 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 18-Oct-91
 *-
 */


void SdsSize(SdsIdType id, unsigned long *bytes, StatusType * SDSCONST status)

{
   sblock *sb;
   INT32 data;
   INT32 defn;
   handle block;
   INT32 external;
   INT32 *origin;

   Sds__get_id(id,&block,&external,&origin,status);
   if (*status != SDS__OK) return;
   if (external)
     {
       if ((INT32*)block == origin+4)
         {
            *bytes = ((INT32*)origin)[1];
	 }
       else
         {
       Sds__size_ext(origin,(INT32*)(block),&defn,&data,0);
       *bytes = 16+data+defn;
         }
     }
   else
     {
   sb = (sblock*)(*block);
   Sds__size(sb,&defn,&data,0);
   *bytes = 16+data+defn;
     }
}


/*+	                 S d s S i z e D e f i n e d

 *  Function name:
      SdsSizeDefined

 *  Function:
      Find the buffer size needed to export using SdsExportDefined

 *  Description:
      Return the size which will be needed for a buffer into which the object
      can be exported using the SdsExportDefined function.
      
 *  Language:
      C

 *  Declaration:
      void SdsSizeDefined(SdsIdType id, unsigned long *bytes, StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType)  Identifier of the object.
      (<) bytes     (unsigned long*) Size in bytes of required buffer.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID => The identifier is invalid.
                            SDS__EXTERN => Object is external.

 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 13-Jul-98
 *-
 */


void SdsSizeDefined(SdsIdType id, unsigned long *bytes, StatusType * SDSCONST status)

{
   sblock *sb;
   INT32 data;
   INT32 defn;
   handle block;
   INT32 external;
   INT32 *origin;

   Sds__get_id(id,&block,&external,&origin,status);
   if (*status != SDS__OK) return;
   if (external)
     {
       if ((INT32*)block == origin+4)
         {
            *bytes = ((INT32*)origin)[1];
	 }
       else
         {
       Sds__size_ext(origin,(INT32*)(block),&defn,&data,1);
       *bytes = 16+data+defn;
         }
     }
   else
     {
   sb = (sblock*)(*block);
   Sds__size(sb,&defn,&data,1);
   *bytes = 16+data+defn;
     }
}


void Sds__export_ext(INT32 *origin, INT32 *edata, INT32 *data, INT32 *defn_pos,
                   INT32 *data_pos, int define)

{
   /*INT32 tdata;*/
   /*INT32 tdefn;*/
   int ncomps;
   char code;
   long i;
   int nextra;
   int nitems;
   INT32 pos;
   unsigned long size;
   INT32 temp_pos;

   code = ((char*)edata)[0];
   nitems = ((short*)edata)[1];
   if (code == SDS_STRUCT)
     {
        nextra = ((short*)edata)[10+2*nitems];
        temp_pos = *defn_pos+5;
        size = 5 + nitems + (2+nextra)/4;
        if (((2+nextra) % 4) != 0) size++;
        memcpy(((char*)data)+(4*(*defn_pos)),((char*)edata),4*size);
        *defn_pos = *defn_pos + size;
        for (i=0;i<nitems;i++)
          {
             data[temp_pos] = *defn_pos;
             temp_pos++;
             Sds__export_ext(origin,origin+(edata[5+i]),data,
                   defn_pos,data_pos,define);
          }
     }
   else if (code == SDS_SARRAY)
     {
        ncomps=1;
        pos = 5;
        for (i=0;i<nitems;i++) 
          {
             ncomps = ncomps*edata[pos];
             pos++;
	   }
        temp_pos = *defn_pos+5;
        nextra = ((short*)edata)[2*pos];
        size = 5+nitems+(2+nextra)/4+ncomps;
        if (((2+nextra) % 4) != 0) size++;
        memcpy(((char*)data)+(4*(*defn_pos)),((char*)edata),4*size);
        *defn_pos = *defn_pos + size;
        for (i=0;i<ncomps;i++)
          {
             data[temp_pos] = *defn_pos;
             temp_pos++;
             Sds__export_ext(origin,origin+(edata[5+nitems+i]),data,defn_pos,
                     data_pos,define);
          }
     }
   else
     {
        ncomps = 1;
        pos = 6;
        temp_pos = *defn_pos+5;
        for (i=0;i<nitems;i++) 
          {
            ncomps = ncomps*edata[pos];
            pos++;
          }
        nextra = ((short*)edata)[2*pos];
        size = 6 + nitems + (2+nextra)/4;
        if (((2+nextra) % 4) != 0) size++;
        memcpy(((char*)data)+(4*(*defn_pos)),((char*)edata),4*size);
        *defn_pos = *defn_pos + size;
        if ((edata[5] == 0) && (!define))
          *data = 0;
        else
          {
            data[temp_pos] = *data_pos;
            if ((code == SDS_DOUBLE) || (code == SDS_I64) || 
                (code == SDS_UI64)) size += 4;
            size = ncomps*nbytes[(int)code];
            while ((size % 4) != 0) size++;
            if (edata[5] == 0)
                memcpy(((char*)data)+(4*(*data_pos)),((char*)(origin+edata[5])),size);
            *data_pos += size/4;
          }
     }
}



void Sds__export(sblock *sb, INT32 *data, INT32 *defn_pos, INT32 *data_pos,
                       int define)

{
   pblock *pb;
   INT32 temp_pos;
   int i;
   int ncomps;
   handle *hblock;
   INT32 size;
   BOOL space;

   ((char*)data)[4*(*defn_pos)] = sb->code;
   ((char*)data)[(4*(*defn_pos))+1] = sb->format;
   strcpy(((char*)data)+(4*(*defn_pos))+4,sb->name);
   if (sb->code == SDS_STRUCT)
     {
        ((short*)data)[2*(*defn_pos)+1] = sb->nitems;
        temp_pos = *defn_pos + 5;
        *defn_pos = *defn_pos + 5 + sb->nitems + (2+sb->nextra)/4;
        if (((2+sb->nextra) % 4) != 0) (*defn_pos)++;
        for (i=0;i<sb->nitems;i++)
          {
            data[temp_pos] = *defn_pos;
            temp_pos++;
            Sds__export((sblock*)(*(sb->item[i])),data,defn_pos,data_pos,define);
          }
        ((short*)data)[2*temp_pos] = sb->nextra;
        if (sb->nextra != 0)
           memcpy(((char*)data)+(4*temp_pos)+2,sb->extra,sb->nextra);
     }
   else if (sb->code == SDS_SARRAY)
     {
        pb = (pblock*)sb;
        ((short*)data)[2*(*defn_pos)+1] = pb->ndims;
        *defn_pos = *defn_pos + 5;
        ncomps = 1;
        for (i=0;i<pb->ndims;i++)
          {
             ncomps = ncomps * pb->dims[i];
             data[*defn_pos] = pb->dims[i];
             (*defn_pos)++;
          }
        temp_pos = *defn_pos;
        *defn_pos = *defn_pos + ncomps + (2+pb->nextra)/4;
        if (((2+pb->nextra) % 4) != 0) (*defn_pos)++;
        hblock = (handle*)(pb->data);
        for (i=0;i<ncomps;i++)
          {
            data[temp_pos] = *defn_pos;
            temp_pos++;
            Sds__export((sblock*)(*(hblock[i])),data,defn_pos,data_pos,define);
          }
        ((short*)data)[2*temp_pos] = pb->nextra;
        if (pb->nextra != 0)
           memcpy(((char*)data)+(4*temp_pos)+2,pb->extra,pb->nextra);
     }
   else
     {
        pb = (pblock*)sb;
        ((short*)data)[2*(*defn_pos)+1] = pb->ndims;
        *defn_pos = *defn_pos + 5;
        if ((pb->data == NULL) && (!define))
          data[*defn_pos] = 0;
        temp_pos = *defn_pos;
        (*defn_pos)++;
        ncomps = 1;
        for (i=0;i<pb->ndims;i++)
          {
             ncomps = ncomps * pb->dims[i];
             data[*defn_pos] = pb->dims[i];
             (*defn_pos)++;
          }
        size = nbytes[(int)pb->code] * ncomps;
        if ((pb->data != NULL) || (define))
          {
            space = FALSE;
            if ((pb->code == SDS_DOUBLE) || (pb->code == SDS_I64) ||
                (pb->code == SDS_UI64))
              {
                if (((*data_pos) % 2) != 0)
                  { 
                    *data_pos += 1;
                    space = TRUE;
                  }
              }
            data[temp_pos] = *data_pos;
            if (pb->data != NULL) 
               memcpy(((char*)data)+((*data_pos)*4),*(pb->data),size);
            *data_pos += size/4;
            if (((pb->code == SDS_DOUBLE) || (pb->code == SDS_I64) ||
                 (pb->code == SDS_UI64)) && (! space))
              *data_pos += 1;
          }
        if ((size % 4) != 0) (*data_pos)++;
        ((short*)data)[2*(*defn_pos)] = pb->nextra;
        if (pb->nextra != 0)
           memcpy(((char*)data)+(4*(*defn_pos))+2,pb->extra,pb->nextra);
        *defn_pos = *defn_pos + (2+pb->nextra)/4;
        if (((2+pb->nextra) % 4) != 0) (*defn_pos)++;
     }
}
        

/*+				S d s E x p o r t

 *  Function name:
      SdsExport

 *  Function:
      Export an object into an external buffer

 *  Description:
      Export an object into an external buffer.

      Once exported an object can be moved around in memory, written to
      a file etc., and subsequently returned to the SDS system either by using
      SdsImport to import it back into the system, or SdsAccess, to access it
      as an external object.

      The original internal version of the
      object continues to exist, in addition to the external copy. All
      identifiers to the object continue to refer to the original internal
      copy.
      
      With SdsExport, any undefined primitive data items occupy no
      space in the exported item, and cannot be written or read
      until the item is reimported. This enables the creation of
      compact templates for structures which may contain large arrays.
      If this behaviour is not wanted use SdsExportDefined, which allocates
      full space in the external structure for undefined primitive
      items.

      The length of the buffer required for SdsExport can be determined
      by a call to SdsSize.
      
 *  Language:
      C

 *  Declaration:
      void SdsExport(SdsIdType id, unsigned long length,  
                    void *data, StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType)  Identifier of the structure to be exported.
      (>) length    (unsigned long) Size in bytes of the buffer.
      (<) data      (void*) The buffer into which the object will be
                          exported.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID => The identifier is invalid.
                            SDS__TOOLONG => The object is too large for the buffer
                            SDS__EXTERN => The object is external.

 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 18-Oct-91
 *-
 */
        

void SdsExport(SdsIdType id, unsigned long length, void *data, 
          StatusType * SDSCONST status)

{
   INT32 defn_pos;
   INT32 data_pos;
   INT32 defn_size;
   INT32 data_size;
   sblock *sb;
   UINT32 size;
   handle block;
   INT32 external;
   INT32 *origin;

   Sds__get_id(id,&block,&external,&origin,status);
   if (*status != SDS__OK) return;
   if (external)
     {
       if ((INT32*)block == origin+4)
         {
            size = ((UINT32*)origin)[1];
            if (length < size)
              {
                 *status = SDS__TOOLONG;
                 return;
              }
            memcpy((char*)data,(char*)origin,size);
	 }
       else 
         {
            Sds__size_ext(origin,((INT32*)block),&defn_size,
                   &data_size,0);
            if (length < (16+defn_size+data_size))
              {
                 *status = SDS__TOOLONG;
                 return;
               }
             defn_pos = 0;
             data_pos = 4+defn_size/4;
             if (local_format[SDS_INT] == BIGEND)
               ((INT32*)data)[defn_pos] = 0;
             else
               ((INT32*)data)[defn_pos] = ~0;
             defn_pos++;
             ((INT32*)data)[defn_pos] = 16+defn_size+data_size;
             defn_pos++;
             ((INT32*)data)[defn_pos] = SDS_VERSION;
             defn_pos++;
             ((INT32*)data)[defn_pos] = 16+defn_size;
             defn_pos++;

             Sds__export_ext(origin,(INT32*)(block),
                       (INT32*)data,&defn_pos,&data_pos,0);
         }
       return;
     }
   sb = (sblock*)(*block);
   Sds__size(sb,&defn_size,&data_size,0);
   if (length < (16+defn_size+data_size))
     {
        *status = SDS__TOOLONG;
        return;
     }
   defn_pos = 0;
   data_pos = 4+defn_size/4;
   if (local_format[SDS_INT] == BIGEND)
     ((INT32*)data)[defn_pos] = 0;
   else
     ((INT32*)data)[defn_pos] = ~0;
   defn_pos++;
   ((INT32*)data)[defn_pos] = 16+defn_size+data_size;
   defn_pos++;
   ((INT32*)data)[defn_pos] = SDS_VERSION;
   defn_pos++;
   ((INT32*)data)[defn_pos] = 16+defn_size;
   defn_pos++;

   Sds__export(sb,(INT32*)data,&defn_pos,&data_pos,0);

}


/*+		       S d s E x p o r t D e f i n e d

 *  Function name:
      SdsExportDefined

 *  Function:
      Export an object into an external buffer 

 *  Description:
      Export an object into an external buffer.

      Once exported an object can be moved around in memory, written to
      a file etc., and subsequently returned to the SDS system either by using
      SdsImport to import it back into the system, or SdsAccess, to access it
      as an external object.

      The original internal version of the
      object continues to exist, in addition to the external copy. All
      identifiers to the object continue to refer to the original internal
      copy.
      
      SdsExportDefined allocates space in the external item for undefined
      data items, so that these can have their values filled in later
      by an SdsPut (or SdsPointer) to the external item.

      The length of the buffer required for SdsExportDefined can be determined
      by a call to SdsSizeDefined.
      
 *  Language:
      C

 *  Declaration:
      void SdsExportDefined(SdsIdType id, unsigned long length,  
                    void *data, StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType)  Identifier of the structure to be exported.
      (>) length    (unsigned long) Size in bytes of the buffer.
      (<) data      (void*) The buffer into which the object will be
                          exported.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID => The identifier is invalid.
                            SDS__TOOLONG => The object is too large for the buffer
                            SDS__EXTERN => The object is external.

 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 17-Aug-2001
 *-
 *  History:
 *     13th Jul 1998. Original dated version. JAB/AAO.
 *     17th Aug 2001. Removed unused variable declarations. KS/AAO.
 */
        

void SdsExportDefined(SdsIdType id, unsigned long length, void *data, 
          StatusType * SDSCONST status)

{
   INT32 defn_pos;
   INT32 data_pos;
   INT32 defn_size;
   INT32 data_size;
   sblock *sb;
   handle block;
   INT32 external;
   INT32 *origin;

   Sds__get_id(id,&block,&external,&origin,status);
   if (*status != SDS__OK) return;
   if (external)
     {
        
            Sds__size_ext(origin,((INT32*)block),&defn_size,
                   &data_size,1);
            if (length < (16+defn_size+data_size))
              {
                 *status = SDS__TOOLONG;
                 return;
               }
             defn_pos = 0;
             data_pos = 4+defn_size/4;
             if (local_format[SDS_INT] == BIGEND)
               ((INT32*)data)[defn_pos] = 0;
             else
               ((INT32*)data)[defn_pos] = ~0;
             defn_pos++;
             ((INT32*)data)[defn_pos] = 16+defn_size+data_size;
             defn_pos++;
             ((INT32*)data)[defn_pos] = SDS_VERSION;
             defn_pos++;
             ((INT32*)data)[defn_pos] = 16+defn_size;
             defn_pos++;

             Sds__export_ext(origin,(INT32*)(block),
                       (INT32*)data,&defn_pos,&data_pos,1);
        
       return;
     }
   sb = (sblock*)(*block);
   Sds__size(sb,&defn_size,&data_size,1);
   if (length < (16+defn_size+data_size))
     {
        *status = SDS__TOOLONG;
        return;
     }
   defn_pos = 0;
   data_pos = 4+defn_size/4;
   if (local_format[SDS_INT] == BIGEND)
     ((INT32*)data)[defn_pos] = 0;
   else
     ((INT32*)data)[defn_pos] = ~0;
   defn_pos++;
   ((INT32*)data)[defn_pos] = 16+defn_size+data_size;
   defn_pos++;
   ((INT32*)data)[defn_pos] = SDS_VERSION;
   defn_pos++;
   ((INT32*)data)[defn_pos] = 16+defn_size;
   defn_pos++;

   Sds__export(sb,(INT32*)data,&defn_pos,&data_pos,1);

}



void Sds__import(INT32 *data, INT32 pos, sblock* *sbptr, BOOL bs, 
               StatusType * SDSCONST status)

{
   long code;
   long format;
   int nitems;
   char name[16];
   long nextra;
   char extra[MAX_EXTRA];
   sblock *sub;
   handle *hblock;
   int ncomps;
   INT32 temp_pos;
   long i;
   pblock *pb;
   sblock *sb;
   INT32 dptr;
   INT32 size;
   void *Sdsdata;

   code = ((char*)data)[4*pos];
   format = ((char*)data)[4*pos+1];
   nitems = swap2(bs,((short*)data)[2*pos+1]);
   strncpy(name,((char*)data)+(4*pos)+4,16);
   pos += 5;

   if (code == SDS_STRUCT)
     {
       temp_pos = pos;
       pos += nitems;
       nextra = swap2(bs,((short*)data)[2*pos]);
       memcpy(extra,((char*)data)+(4*pos)+2,nextra);
       sb = Sds__create_sblock(name,nextra,extra,nitems);
       (sb)->self = Sds__get_handle();
       if (((sb)->self) == NULL)
	 {
	   *status = SDS__NOMEM;
	   return;
	 }
       *(sb->self) = (void*)sb;
       sb->nitems = nitems;
       for(i=0;i<nitems;i++)
	 {
	   Sds__import(data,swap4(bs,data[temp_pos]),&sub,bs,status);
           sb->item[i] = sub->self;
           sub->parent = sb->self;
           temp_pos++;
         }
       *sbptr = sb;
     }
   else if (code == SDS_SARRAY) 
     {
       pb = (pblock*)SdsMalloc(sizeof(pblock));
       if (pb == NULL)
	 {
	   *status = SDS__NOMEM;
	   return;
	 }
       pb->code = code;
       pb->format = 0;
       strcpy(pb->name,name);
       ncomps = 1;
       for (i=0;i<nitems;i++)
	 {
	   pb->dims[i] = swap4(bs,data[pos]);
           pos++;
           ncomps = ncomps*pb->dims[i];
	 }
       pb->ndims = nitems;
       temp_pos = pos;
       pos += ncomps;
       pb->nextra = swap2(bs,((short*)data)[2*pos]);
       memcpy(pb->extra,((char*)data)+(4*pos)+2,pb->nextra);
       pb->self = Sds__get_handle();
       if ((pb->self) == NULL)
	 {
	   *status = SDS__NOMEM;
	   return;
	 }
       *(pb->self) = (void*)pb;
       hblock = (handle*)SdsMalloc(ncomps*sizeof(handle));
       if (hblock == 0)
	 {
	   *status = SDS__NOMEM;
	   return;
	 }
       pb->data = (handle)hblock;
       for(i=0;i<ncomps;i++)
	 {
	   Sds__import(data,swap4(bs,data[temp_pos]),&sub,bs,status);
           sub->parent = pb->self;
           temp_pos++;
	   hblock[i] = sub->self;
         }
       *sbptr = (sblock*)pb;
     }
   else
     {
       pb = (pblock*)SdsMalloc(sizeof(pblock));
       if (pb == NULL)
	 {
	   *status = SDS__NOMEM;
	   return;
	 }
       pb->code = code;
       pb->format = format;
       strcpy(pb->name,name);
       ncomps = 1;
       dptr = swap4(bs,data[pos]);
       pos++;
       for (i=0;i<nitems;i++)
	 {
	   pb->dims[i] = swap4(bs,data[pos]);
           pos++;
           ncomps = ncomps*pb->dims[i];
	 }
       pb->ndims = nitems;
       pb->nextra = swap2(bs,((short*)data)[2*pos]);
       memcpy(pb->extra,((char*)data)+(4*pos)+2,pb->nextra);
       pb->self = Sds__get_handle();
       if ((pb->self) == NULL)
	 {
	   *status = SDS__NOMEM;
	   return;
	 }
       *(pb->self) = (void*)pb;
       if (dptr != 0)
         {
            size = nbytes[code]*ncomps;
            Sdsdata = (void*)SdsMalloc(size);
            if (Sdsdata == NULL)
	      {
	         *status = SDS__NOMEM;
	         return;
	      }
            pb->data = Sds__get_handle();
            if (pb->data == NULL)
	      {
	         *status = SDS__NOMEM;
	         return;
	      }
            *(pb->data) = Sdsdata;
            memcpy(*(pb->data),data+dptr,size);
         }
       else
         pb->data = NULL;
       *sbptr = (sblock*)pb;
     }
 }
       


/*+				S d s I m p o r t

 *  Function name:
      SdsImport

 *  Function:
      Import an object from an external buffer

 *  Description:
      Import an object from an external buffer and return an identifier
      to the internal copy created. The object must have
      been previously exported using SdsExport.

      The original external version of the
      structure continues to exist, in addition to the internal copy. 

      A fully dynamic internal structure is created in which all SDS
      operations are valid. However, to merely access the data in an object
      SdsAccess can be used in place of SdsImport.

 *  Language:
      C

 *  Declaration:
      void SdsImport(void *data, SdsIdType *id, StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) data      (void*) The buffer from which the object will be
                          imported.
      (<) id        (SdsIdType*)  Identifier of the new internal object.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__NOTSDS => Not a valid sds object.
                            SDS__NOMEM => Insufficient memory.

 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 18-Oct-91
 *-
 */
        

void SdsImport(SDSCONST void *data, SdsIdType *id, StatusType * SDSCONST status)

{
   INT32 pos;
   sblock *sb;
   idblock *idptr;
   BOOL bs;

   if (*status != SDS__OK) return;
   if (((INT32*)data)[0] == 0)
      bs = (local_format[SDS_INT] == LITTLEEND);
   else if (((INT32*)data)[0] == ~0)
      bs = (local_format[SDS_INT] == BIGEND);
   else
     {
       *status = SDS__NOTSDS;
       return;
     }
   
/*  Check SDS version of data is OK  */

   if (swap4(bs,((INT32*)data)[2]) != SDS_VERSION)
     {
       *status = SDS__VERSION;
       return;
     }
  
   pos = 4;   
   Sds__import((INT32*)data, pos, &sb, bs, status);
   sb->parent = NULL;
   if (*status != SDS__OK) return;
   *id = Sds__get_new_id(&idptr);
   if (*id == 0)
     {
       *status = SDS__NOMEM;
       return;
     }
   Sds__put_id(*id,sb->self,FALSE,NULL,status);
 }
   
void Sds__swap_pointers(INT32* origin, INT32* data, BOOL bs)

{
   int nitems;
   int i;
   long code;
   INT32 pos;
   long ncomps;
   INT32 ptr;
   long dims;
   
   code = ((char*)data)[0];
   nitems = ((short*)data)[1] = swap2(bs,((short*)data)[1]);
   pos = 5;
   if (code == SDS_STRUCT)
     {
       for (i=0;i<nitems;i++)
         {
           ptr = data[pos+i] = swap4(bs,data[pos+i]);
           Sds__swap_pointers(origin,origin+ptr,bs);
         }
       pos += nitems;
       ((short*)data)[2*pos] = swap2(bs,((short*)data)[2*pos]);
     }
   else if (code == SDS_SARRAY)
     {
       ncomps = 1;
       for (i=0;i<nitems;i++)
         {
            dims = data[pos+i] = swap4(bs,data[pos+i]);
            ncomps = ncomps*dims;
         }
       pos += nitems;
       for (i=0;i<ncomps;i++)
         {
           ptr = data[pos+i] = swap4(bs,data[pos+i]);
           Sds__swap_pointers(origin,origin+ptr,bs);
         }
       pos += ncomps;
       ((short*)data)[2*pos] = swap2(bs,((short*)data)[2*pos]);
     }
   else
     {
       data[pos] = swap4(bs,data[pos]);
       pos++;
       for (i=0;i<nitems;i++)
          data[pos+i] = swap4(bs,data[pos+i]);
       pos += nitems;
       ((short*)data)[2*pos] = swap2(bs,((short*)data)[2*pos]);
     }
}  
       


/*+				  S d s A c c e s s

 *  Function name:
      SdsAccess

 *  Function:
      Return an identifier to an external object

 *  Description:
      Make an external object (one exported by SdsExport) accessible
      to SDS by returning an identifier to it.

      Any SDS operations which do not change the structure of the object
      may be performed on the external object. These include navigation
      operations (SdsFind, SdsIndex, SdsCell), data access operations
      (SdsGet, SdsPut, SdsPointer) and inquiry operations (SdsInfo).

      Operations which are not permitted on an external object are those
      which add or remove components (SdsNew, SdsDelete), or write
      operations (SdsPut or SdsPointer) to data items which are currently 
      undefined.

      Unlike SdsImport, SdsAccess does not make a copy of the object. The 
      object is accessed in place in the original buffer.

 *  Language:
      C

 *  Declaration:
      void SdsAccess(void *data, SdsIdType *id, StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) data      (void*) The buffer containing the object to be
                          accessed.
      (<) id        (long*)  Identifier of the external object.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__NOTSDS => Not a valid SDS object.
                            SDS__NOMEM => Insufficient memory.

 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 18-Oct-91
 *-
 */
        


void SdsAccess(void *data, SdsIdType *id, StatusType * SDSCONST status)

{
   idblock *idptr;
   BOOL bs;

   if (*status != 0) return;
   if (((INT32*)data)[0] == 0)
      bs = (local_format[SDS_INT] == LITTLEEND);
   else if (((INT32*)data)[0] == ~0)
      bs = (local_format[SDS_INT] == BIGEND);
   else
     {
       *status = SDS__NOTSDS;
       return;
     }

   ((INT32*)data)[1] = swap4(bs,((INT32*)data)[1]);
   ((INT32*)data)[2] = swap4(bs,((INT32*)data)[2]);

/*  Check SDS version of data is OK  */

   if ((((INT32*)data)[2]) != SDS_VERSION)
     {
       *status = SDS__VERSION;
       return;
     }
  
   *id = Sds__get_new_id(&idptr);
   if (*id == 0)
     {
       *status = SDS__NOMEM;
       return;
     }
   Sds__put_id(*id,(handle)(((INT32*)data)+4),TRUE,(INT32*)data,status);

   if (bs) 
     {
       Sds__swap_pointers((INT32*)data,((INT32*)data)+4,bs);
       ((INT32*)data)[0] = ~(((INT32*)data)[0]);
     }

 }
   



/*+				S d s E x t r a c t

 *  Function name:
      SdsExtract

 *  Function:
      Extract an object from a structure

 *  Description:
      Extract an object from a structure. The extracted object becomes a 
      new independent top level object. The object is deleted from
      the original structure.

      The identifier must not be that of a structure array component.
      
      If the identifier is already that of a top level object, then the
      function does nothing.

 *  Language:
      C

 *  Declaration:
      void SdsExtract(SdsIdType id, StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType)  Identifier of the object to be extracted.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID => The identifier is invalid.
                            SDS__ILLDEL => Object cannot be extracted.
                            SDS__EXTERN => The object is external.


 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 23-Oct-91
 *-
 */

void SdsExtract(SdsIdType id, StatusType * SDSCONST status)

{
   sblock *sb;
   sblock *parent;
   long index;
   handle block;
   INT32 external;
   INT32 *origin;

   Sds__get_id(id,&block,&external,&origin,status);
   if (*status != SDS__OK) return;
   if (external)
     {
       *status = SDS__EXTERN;
       return;
     }
   sb = (sblock*)(*block);
   if ((sb->parent) != NULL)
     {

/*  Find the parent of the object to be extracted  */

        parent = (sblock*)(*(sb->parent));
        index = 0;
        if ((parent->nitems == 0) || (parent->code == SDS_SARRAY))
          {
            *status = SDS__ILLDEL;
            return;
          }

/*  Loop through the parents item list to find the item to be deleted  */

        while (*(parent->item[index]) != sb)
          {
             index++;
             if (index >= (parent->nitems))
               {

/*  Can't find the item - must be a structure array component for
    which deletion is illegal  */

                  *status = SDS__ILLDEL;
                  return;
               }
          }
        
/*  Shift up following items in list by one and reduce item count by one */

        for (;index < parent->nitems;index++)
          parent->item[index] = parent->item[index+1];
        (parent->nitems)--;
     }

   sb->parent = NULL;
}




/*+				  S d s I n s e r t

 *  Function name:
      SdsInsert

 *  Function:
      Insert an existing object into a structure

 *  Description:
      An existing top level object is inserted into a structure.
      
 *  Language:
      C

 *  Declaration:
      void SdsInsert(SdsIdType parent_id, SdsIdType id, StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) parent_id (SdsIdType)  The identifier of the structure to
                            which the component is to be added.
      (>) id        (SdsIdType)  The identifier of the object to be inserted.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID => Invalid identifier
                            SDS__NOTSTRUCT => Parent is not a structure
                            SDS__NOTTOP => Not a top level object
                            SDS__NOMEM => Insufficient memory
                            SDS__EXTERN => Object is external

 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 23-Oct-91
 *-
 */

void SdsInsert(SdsIdType parent_id, SdsIdType id, StatusType * SDSCONST status)

{
  sblock *sb;            /* pointer to parent structure */
  sblock *stb;           /* pointer to new component */
  long n;
  handle block;
  INT32 external;
  INT32 *origin;

/* import parent  */

  Sds__get_id(parent_id,&block,&external,&origin,status);
  if (*status != SDS__OK) return;
  if (external)
    {
      *status = SDS__EXTERN;
      return;
    }
  sb = (sblock*)(*block);
  if (sb->code != SDS_STRUCT)
  {
    *status = SDS__NOTSTRUCT;
    return;
  }

/* import object  */

  Sds__get_id(id,&block,&external,&origin,status);
  if (*status != SDS__OK) return;
  if (external)
    {
      *status = SDS__EXTERN;
      return;
    }
  stb = (sblock*)(*block);
  if (stb->parent != NULL)
  {
    *status = SDS__NOTTOP;
    return;
  }

/*  Extend structure if necessary  */

  n = sb->nitems;
  if (n >= sb->maxitems) Sds__extend_struct(&sb,status);
  if (*status != SDS__OK) return;
  sb->nitems = n+1;

/*  Insert object */

  sb->item[n] = stb->self;
  stb->parent = sb->self;

}


void Sds__copy(sblock *sb, sblock **sbc, StatusType * SDSCONST status)

/*
 *  Return a copy of the object sb
 */

{
  sblock *sub;
  pblock *parb;
  pblock *pb;
  handle *hblock;
  handle *hblock1;
  long ncomps;
  long i;
  void *Sdsdata;
  long size;


  if (sb->code == SDS_STRUCT)

/*  Structure case  */

    {

/*  Create a new sblock  */

      *sbc = Sds__create_sblock(sb->name,sb->nextra,sb->extra,
               (long)(sb->nitems));
      if (*sbc == NULL)
        {
          *status = SDS__NOMEM;
          return;
        }

/*  Create a handle to the block  */

      (*sbc)->self = Sds__get_handle();
      if ((*sbc)->self == NULL)
        {
          *status = SDS__NOMEM;
          return;
        }
      *((*sbc)->self) = (void*)(*sbc);

/*  Copy the items in the structure   */

      (*sbc)->nitems = sb->nitems;
      for (i=0;i<sb->nitems;i++)
        {
          Sds__copy((sblock*)(*(sb->item[i])),&sub,status);
          (*sbc)->item[i] = sub->self;
          sub->parent = (*sbc)->self;
        }
    }
  else if (sb->code == SDS_SARRAY)
    {

/*  Structure array case  */

       parb = (pblock*)sb;

/*  Create a new pblock  */

       pb = (pblock*)SdsMalloc(sizeof(pblock));
       if (pb == NULL)
	 {
	   *status = SDS__NOMEM;
	   return;
	 }

/*  Copy contents of pblock  */

       pb->code = parb->code;
       pb->format = 0;
       strcpy(pb->name,parb->name);
       ncomps = 1;
       for (i=0;i<parb->ndims;i++)
	 {
	   pb->dims[i] = parb->dims[i];
           ncomps = ncomps*pb->dims[i];
	 }
       pb->ndims = parb->ndims;
       pb->nextra = parb->nextra;
       if ( pb->nextra != 0 )
       {
	   memcpy(pb->extra,parb->extra,pb->nextra);
	}

/*  Create a handle  */

       pb->self = Sds__get_handle();
       if ((pb->self) == NULL)
	 {
	   *status = SDS__NOMEM;
	   return;
	 }
       *(pb->self) = (void*)pb;

/*  Create a handle block for the array components  */

       hblock1 = (handle*)(parb->data);
       hblock = (handle*)SdsMalloc(ncomps*sizeof(handle));
       if (hblock == 0)
	 {
	   *status = SDS__NOMEM;
	   return;
	 }
       pb->data = (handle)hblock;

/*  Copy the array components   */

       for(i=0;i<ncomps;i++)
	 {
	   Sds__copy((sblock*)(*(hblock1[i])),&sub,status);
           sub->parent = pb->self;
           sub->self = Sds__get_handle();
	   if (sub->self == NULL)
	     {
	       *status = SDS__NOMEM;
	       return;
	     }
	   *(sub->self) = (void*)sub;
	   hblock[i] = sub->self;
         }
       *sbc = (sblock*)pb;
     }
   else
     {

/*  Primitive case  */

       parb = (pblock*)sb;

/*  Create a pblock  */

       pb = (pblock*)SdsMalloc(sizeof(pblock));
       if (pb == NULL)
	 {
	   *status = SDS__NOMEM;
	   return;
	 }

/*  Copy contents of pblock  */

       pb->code = parb->code;
       pb->format = parb->format;
       strcpy(pb->name,parb->name);
       ncomps = 1;
       for (i=0;i<parb->ndims;i++)
	 {
	   pb->dims[i] = parb->dims[i];
           ncomps = ncomps*pb->dims[i];
	 }
       pb->ndims = parb->ndims;
       pb->nextra = parb->nextra;
       if ( pb->nextra != 0 )
       {
	   memcpy(pb->extra,parb->extra,pb->nextra);
       }

/*  Create a handle  */

       pb->self = Sds__get_handle();
       if ((pb->self) == NULL)
	 {
	   *status = SDS__NOMEM;
	   return;
	 }
       *(pb->self) = (void*)pb;

/*  Copy the data if there is any  */

       if (parb->data != NULL)
         {
            size = nbytes[(int)pb->code]*ncomps;
            Sdsdata = (void*)SdsMalloc(size);
            if (Sdsdata == NULL)
	      {
	         *status = SDS__NOMEM;
	         return;
	      }
            pb->data = Sds__get_handle();
            if (pb->data == NULL)
	      {
	         *status = SDS__NOMEM;
	         return;
	      }
            *(pb->data) = Sdsdata;
            memcpy(*(pb->data),*(parb->data),size);
         }
       else
         pb->data = NULL;
       *sbc = (sblock*)pb;
     }
 }
       


/*+				S d s C o p y

 *  Function name:
      SdsCopy

 *  Function:
      Make a copy of an object

 *  Description:
      Make a copy of an object and return an identifier to the copy.
      The copy is a new top level object, the original object is
      unchanged by the operation.

      The object being copied can be either external or internal.
      The copy is always an internal object.

 *  Language:
      C

 *  Declaration:
      void SdsCopy(SdsIdType id, SdsIdType *copy_id, StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType)  Identifier of the object to be copied.
      (<) copy_id   (SdsIdType*) Identifier of the copy.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID => The identifier is invalid.
                            SDS__NOMEM => Insuficient memory.

 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 23-Oct-91
 *-
 */



void SdsCopy(SdsIdType id, SdsIdType *copy_id, StatusType * SDSCONST status)


{
   sblock *sb;
   idblock *copyptr;
   handle block;
   INT32 external;
   INT32 *origin;

/*  Import the identifier  */

   Sds__get_id(id,&block,&external,&origin,status);
   if (*status != SDS__OK) return;
   if (external)
     {

/*  If it is an external object we can use Sds__import to do the copy,
    but we have to turn byte swapping off, because this will already have
    been done by SdsAccess.   */

     Sds__import(origin,(INT32)((INT32*)(block)-origin),
                  &sb,FALSE,status);
     }
   else

/*  Otherwise it is an internal to internal copy done by Sds__copy  */

     Sds__copy((sblock*)(*block),&sb,status);

/*  Set parent to NULL - no parent  */

   sb->parent = NULL;
   if (*status != SDS__OK) return;

/*  Create an id block and set it to point to the object  */

   *copy_id = Sds__get_new_id(&copyptr);
   if (*copy_id == 0)
     {
       *status = SDS__NOMEM;
       return;
     } 
   Sds__put_id(*copy_id,sb->self,FALSE,NULL,status);
 }



/*+				S d s R e s i z e

 *  Function name:
      SdsResize

 *  Function:
      Change the dimensions of an array.

 *  Description:
     Change the number and/or size of the dimensions of an array. This
     operation can be performed on primitive arrays or structure arrays.
     Note that SDS_RESIZE does not move the data elements in the storage
     representing the array, so there is no guarantee that after resizing
     the array the same data will be found at the same array index 
     positions as before resizing, though this will be the case for
     simple changes such as a change in the last dimension only.

     If the size of a primitive array is increased the contents of the
     extra locations is undefined. Decreasing the size causes the data
     beyond the new limit to be lost.

     If a structure array is extended the new elements created will be empty
     structures. If a structure array is decreased in size, the lost elements
     and all their components will be deleted.

 *  Language:
      C

 *  Declaration:
      void SdsResize(SdsIdType id, long ndims, unsigned long *dims, 
              StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType) Identifier of the object to be resized.
      (>) ndims     (long)  New number of dimensions.
      (>) dims      (unsigned long*) Array of dimensions.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID => The identifier is invalid.
                            SDS__NOMEM => Insuficient memory.
                            SDS__EXTERN => Object is external.
                            SDS__NOTARR => Object is not an array.
                            SDS__INVDIMS => Dimensions invalid.

 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 17-Aug-2001
 *-
 *  History:
 *     23rd Oct 1991. Original dated version. JAB/AAO.
 *     17th Aug 2001. Added casts to character indices to placate gcc. KS/AAO.
 */

void SdsResize(SdsIdType id, long ndims, SDSCONST unsigned long *dims, 
            StatusType * SDSCONST status)

{
   pblock *pb;
   handle *hblock;
   handle *newhblock;
   long oldsize,newsize;
   sblock *sab;
   void* dptr;
   long i;
   handle block;
   INT32 external;
   INT32 *origin;


   Sds__get_id(id,&block,&external,&origin,status);
   if (*status != SDS__OK) return;
   if (external)
     {
       *status = SDS__EXTERN;
       return;
     }
   pb = (pblock*)(*block);
   if (pb->code == SDS_STRUCT)
     {   
       *status = SDS__NOTARR;
       return;
     }

/*  check dimensions  */

  if ((ndims > MAXDIMS) || (ndims<0))
    {
       *status = SDS__INVDIMS;
       return;
    }

   oldsize = 1;
   for (i=0;i<pb->ndims;i++) oldsize = oldsize*pb->dims[i];
   pb->ndims = ndims;
   newsize = 1;
   for (i=0;i<pb->ndims;i++)
     {
       pb->dims[i] = dims[i];
       newsize = newsize*dims[i];
     }
   if (pb->code == SDS_SARRAY) 
     {
       hblock = (handle*)(pb->data);
       if (newsize < oldsize)
         {
           for (i=newsize;i<oldsize;i++)
             {
               Sds__delete((sblock*)(*(hblock[i])));
             }
         }
       else if (newsize > oldsize)
         {
           newhblock = (handle*)SdsMalloc(newsize*sizeof(handle));
           if (newhblock == NULL)
             {
               *status = SDS__NOMEM;
               return;
             }
           for (i=0;i<oldsize;i++) newhblock[i] = hblock[i];
           for (i=oldsize;i<newsize;i++)
             { 
               sab = Sds__create_sblock(pb->name,pb->nextra,pb->extra,
                     (long)NITEMS);
               if (sab == NULL) 
                 {
                    *status = SDS__NOMEM;
                    return;
                 }
               sab->parent = pb->self;
               sab->self = Sds__get_handle();
               if ((sab->self) == NULL)
                 {
                    *status = SDS__NOMEM;
                    return;
                 }
               *(sab->self) = (void*)sab;
               newhblock[i] = sab->self;
             }
           pb->data = (handle)newhblock;
           SdsFree(hblock);
         }
     }
   else
     {
       if ((pb->data == NULL) || (newsize <= oldsize))
         return;
       dptr = (void*)SdsMalloc(newsize*nbytes[(int)pb->code]);
       if (dptr == NULL)
         {
           *status = SDS__NOMEM;
           return;
         }
       memcpy(dptr,*(pb->data),
            ((newsize>oldsize) ? oldsize : newsize)*nbytes[(int)pb->code]);
       SdsFree(*(pb->data));
       *(pb->data) = dptr;
     }
}




/*+				S d s R e n a m e

 *  Function name:
      SdsRename

 *  Function:
      Change the name of an object.

 *  Description:
      Specify a new name for an object.

 *  Language:
      C

 *  Declaration:
      void SdsRename(SdsIdType id, char* name, StatusType * SDSCONST status)
      
 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType)  Identifier of the object to be renamed.
      (>) name      (char*) New name for the object. This should
                            have a maximum length of 16 characters including
                            the terminating null.
      (!) status    (long*) Modified status. Possible failure codes are:
                            SDS__BADID => The identifier is invalid.
                            SDS__LONGNAME => The name is too long.

 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 24-Oct-91
 *-
 */

void SdsRename(SdsIdType id, SDSCONST char *name, StatusType * SDSCONST status)

{
   sblock *sb;
   handle block;
   INT32 external;
   INT32 *origin;
   
   Sds__get_id(id,&block,&external,&origin,status);
   if (*status != SDS__OK) return;
   if (strlen(name) > 15U)	/* Note, 15U means unsigned */
     {
       *status = SDS__LONGNAME;
       return;
     }
   if (external)
     {
       strcpy((char*)(block)+4,name);
     }
   else
     {
       sb = (sblock*)(*block);
       strcpy(sb->name,name);
     }
}


void Sds__PutExtra_ext(INT32 *data, long nextra, SDSCONST char *extra, 
                             StatusType * SDSCONST status)
/*
 *  External case of SdsPutExtra
 */

{
   long code;
   long nitems;
   long ncomps;
   INT32 pos;
   long i;
   long oldnextra;

   code = ((char*)data)[0];
   nitems = ((short*)data)[1];

/* Determine position of extra field in data - This depends on the
   type of the object, and the number of components / dimensions  */

   if (code == SDS_STRUCT)
     {
       pos = 5+nitems;
     }
   else if (code == SDS_SARRAY)
     {
       ncomps = 1;
       for (i=0;i<nitems;i++)
         ncomps = ncomps*data[5+i];
       pos = 5 + nitems + ncomps;
     }
   else
     {
       pos = 6+nitems;
     }
   oldnextra = ((short*)data)[2*pos];
   if (nextra > oldnextra)
     {
       *status = SDS__EXTRA;
       return;
     }
   ((short*)data)[2*pos] = nextra;
   memcpy(((char*)data)+(4*pos)+2,extra,nextra);
}   





/*+			  	S d s P u t E x t r a

 *  Function name:
      SdsPutExtra

 *  Function:
      Write to the extra information field of an object.

 *  Description:
      Write a specified number of bytes to the extra information field
      of an object. A maximum of 128 bytes may be written to an internal
      object. It is permissible to write to the extra information field
      of an external object, but the number of bytes written must not
      exceed the number originally in the object.

 *  Language:
      C

 *  Declaration:
      void SdsPutExtra(SdsIdType id, long nextra, char* extra, 
                    StatusType * SDSCONST status)

 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType) Identifier of the object
      (>) nextra    (long)  Number of bytes of extra information.
      (>) extra     (char*) The extra information to be included.
                            nextra bytes are copied into the object.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID => The identifier is invalid.
                            SDS__EXTRA => Too much extra data.

 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 24-Oct-91
 *-
 */

void SdsPutExtra(SdsIdType id, long nextra, SDSCONST char *extra, StatusType * SDSCONST status)

{
   sblock *sb;
   handle block;
   INT32 external;
   INT32 *origin;
   
   Sds__get_id(id,&block,&external,&origin,status);
   if (*status != SDS__OK) return;
   if (nextra > MAX_EXTRA)
     {
       *status = SDS__EXTRA;
       return;
     }
   if (external)
     {
       Sds__PutExtra_ext((INT32*)(block),nextra,extra,status);
     }
   else
     {
       sb = (sblock*)(*block);
       memcpy(sb->extra,extra,nextra);
       sb->nextra = nextra;
     }
}


void Sds__GetExtra_ext(INT32 *data, long length, char *extra, 
   unsigned long *actlen, StatusType * SDSCONST status)
/*
 *  External case of SdsGetExtra
 */


{
   long code;
   long nitems;
   long ncomps;
   INT32 pos;
   long i;
   long nextra;

   code = ((char*)data)[0];
   nitems = ((short*)data)[1];

/* Determine position of extra field in data - This depends on the
   type of the object, and the number of components / dimensions  */

   if (code == SDS_STRUCT)
     {
       pos = 5+nitems;
     }
   else if (code == SDS_SARRAY)
     {
       ncomps = 1;
       for (i=0;i<nitems;i++)
         ncomps = ncomps*data[5+i];
       pos = 5 + nitems + ncomps;
     }
   else
     {
       pos = 6+nitems;
     }
   nextra = ((short*)data)[2*pos];
   *actlen = ((nextra > length) ? length : nextra);
   memcpy(extra,((char*)data)+(4*pos)+2,*actlen);
}   






/*+			  	S d s G e t E x t r a

 *  Function name:
      SdsGetExtra

 *  Function:
      Read from the extra information field of an object.

 *  Description:
      Read bytes from the extra information field of an object. Bytes are
      copied until the supplied buffer is filled up or until all bytes
      in the field are copied. 

 *  Language:
      C

 *  Declaration:
      void SdsGetExtra(SdsIdType id, long length, char* extra, 
          unsigned long* actlen, StatusType * SDSCONST status)

 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType) Identifier of the object
      (>) length    (long)  Length of buffer to receive data.
      (<) extra     (char*) Buffer to receive the extra information
                            copied from the object.
      (<) actlen    (unsigned long*) actual number of bytes copied.
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID => The identifier is invalid.

 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 24-Oct-91
 *-
 */

void SdsGetExtra(SdsIdType id, long length, char *extra, unsigned long *actlen, 
StatusType * SDSCONST status)

{
   sblock *sb;
   handle block;
   INT32 external;
   INT32 *origin;
   
   Sds__get_id(id,&block,&external,&origin,status);
   if (*status != SDS__OK) return;
   if (external)
     {
       Sds__GetExtra_ext((INT32*)(block),length,extra,actlen,status);
     }
   else
     {
       sb = (sblock*)(*block);
       *actlen = ((sb->nextra > length) ? length : sb->nextra);
       memcpy(extra,sb->extra,*actlen);
     }
}




/*+			S d s I s E x t e r n a l

 *  Function name:
      SdsIsExternal

 *  Function:
      Enquire whether an object is external

 *  Description:
      Enquire whether an object is external

 *  Language:
      C

 *  Declaration:
      void SdsIsExternal(SdsIdType id, int *external, StatusType * SDSCONST status)

 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType) Identifier of the object
      (<) external  (int *) Non zero if the object is external
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID => The identifier is invalid.

 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 17-Aug-2001
 *-
 *  History:
 *     13th Jul 1998. Original dated version. JAB/AAO.
 *     17th Aug 2001. Removed unused variable declarations. KS/AAO.
 */

void SdsIsExternal(SdsIdType id, int * ext, StatusType * SDSCONST status)

{
   handle block;
   INT32 external;
   INT32 *origin;
   
   Sds__get_id(id,&block,&external,&origin,status);
   if (*status != SDS__OK) return;

   *ext = external;
}


/*+			S d s E x t e r n I n f o

 *  Function name:
      SdsExternInfo

 *  Function:
      Return the address of an external object

 *  Description:
      Return the address of an external SDS object given its id.
      This is the address which was given to SdsAccess.

 *  Language:
      C

 *  Declaration:
      void SdsExternInfo(SdsIdType id, void ** address, StatusType * SDSCONST status)

 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType) Identifier of the object
      (<) address   (void **) Address of object
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID => The identifier is invalid.
                            SDS__EXTERN => Not an external object.

 *  Support: Jeremy Bailey, {AAO}

 *  Version date: 17-Jul-2001
 *-
 *  History:
 *     13th Jul 1998. Original dated version. JAB/AAO.
 *     17th Aug 2001. Removed unused variable declarations. KS/AAO.
 */

void SdsExternInfo(SdsIdType id, void ** address, StatusType * SDSCONST status)

{
   handle block;
   INT32 external;
   INT32 *origin;
   
   Sds__get_id(id,&block,&external,&origin,status);
   if (*status != SDS__OK) return;
   
   if (external)
      *address = origin;
   else
      *status = SDS__EXTERN;
}




idblock *Sds__import2(long id, StatusType * SDSCONST status)

/*
 *  Sds__import2 - Given an id number return a pointer to the id block
 *                 Don't give an error if the object has been deleted
 */

{
  if (*status != SDS__OK) return(NULL);
  if ((id <= 0) || (id > id_number))
  {
     *status = SDS__BADID;
     return(NULL);
  }
  if (idarrayptr[id].block == NULL)
  {
     *status = SDS__BADID;
     return(NULL);
  }
  return(&(idarrayptr[id]));
}





/*+			  	S d s F r e e I d

 *  Function name:
      SdsFreeId

 *  Function:
      Free an identifier, so that its associated memory may be reused.

 *  Description:
      Each identifier allocated by SDS uses memory. To avoid excessive
      allocation of memory the SdsFreeId function can be used to free
      the memory associated with an identifer that is no longer needed.
      When this is done the memory will be reused by SDS for a subsequent
      identifier when needed.

 *  Language:
      C

 *  Declaration:
      void SdsFreeId(SdsIdType id, StatusType * SDSCONST status)

 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) id        (SdsIdType)  Identifier to be freed
      (!) status    (StatusType*) Modified status. Possible failure codes are:
                            SDS__BADID => The identifier is invalid.

 *  Support: Jeremy Bailey, {AAO}
 
 *  Version date: 23-Jan-92
 *-
 */

void SdsFreeId(SdsIdType id, StatusType * SDSCONST status)

{
   idblock *idptr;

#ifdef VxWorks
   semStatus = semTake(idSemaphore,600);
   if (semStatus != OK) 
     {
       *status = SDS__TIMEOUT;
       return;
     }
#endif

#ifdef POSIX_THREADS
   pthread_mutex_lock( &id_mutex );
#endif

   idptr = Sds__import2(id,status);
   if (*status != SDS__OK)
   {
#ifdef POSIX_THREADS
      pthread_mutex_unlock( &id_mutex );
#endif

#ifdef VxWorks
      semGive(idSemaphore);
#endif

      return;
   }
   
   idptr->block = NULL;
   unused_ids++;

#ifdef POSIX_THREADS
   pthread_mutex_unlock( &id_mutex );
#endif

#ifdef VxWorks
   semGive(idSemaphore);
#endif

}




