/*+		F P I L	M O D E L

 * Module name:
     FPILMODEL

 * Function:
     Fibre Positioner Instrument Library - Telescope Model Access

 * Description:
     Provides a set of routines common to all fibre positioner instruments,
     providing access to the Telescope model information.

 * Language:
      C

 * Support: Tony Farrell, AAO

 *-

 * Sccs Id:     fpilmodel.c, Release 1.4, 09/30/02

 * History:
     09-Dec-1999 - TJF - Original version
     13-Mar-2000 - TJF - Add FpilModelQuickAppToObs and FpilModelParams
     20-Mar-2000 - TJF - Include fpil_err.h.
 */
/*
 *  Include files
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "status.h"
#include "sds.h"
#include "fpil.h"
#include "fpil_err.h"


/* 
 * Constant definitions
 */

/*
 * Macro function definitions
 */
/*
 * type definitions
 */


/*
 * Functions imported (but not declared in include files)
 */




/*
 * Variables private to this module (static variables)
 */



/*
 * Function Definitions
 *
 */


/*
 *+                 F p i l M o d e l A c c e s s

 * Function name:
      FpilModelAccess

 * Function:
      Access telescope model details for a specified field plate.

 * Description:
      The Fibre Positioner telescope model is used to convert from
      RA/DEC positions on the sky to positioner X/Y positions.  There
      is an intermediate X/Y position on the field rather.  This routine
      initialises access to the telescope model by invoking the model
      specific access routine.

      Normally such access involves opening a file or files.  The caller
      can specify two directores to search for the files and the
      instrument library may know of a possible location itself.  If the
      file exists in the  dirOverride directory it will be opened from
      there, if not, an instrument specific directory may be searched and
      finally, if it can't be opended from there, it will be opended
      from the dirFallback directory if possible.

      The RptRoutine may be specified to get feedback (for messages) about
      which file was opened.  It will be called for every file opened with
      the full name of the file which was opened.

 * Language:

      C

 * Call:
        (void) = FpilConsAccess(inst,  dirOverride, 
                    dirFallback, RptRoutine, RptArg,field, model, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) dirOverride     (const char *)  If non-null, a directory to
                                   search for the constant details file.
      (>) dirFallback     (const char *)  If non-null, a fall back
                                   directory to search for the constants
                                   details file.
      (>) RptRoutine      (FpilRptRoutineType) A routine to be invoked
                                   to report which file has been opened.
      (>) RptArg          (void *) Argument passed to RptRoutine.
      (>) field     (unsigned int) The field to be accessed
      (<) model     (FileModelType **) The telescope model structure to be
                    passed to the other routines.
      (!) status    (StatusType *) modified status.
                        

 * Include files: fpil.h

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
     10-Mar-2000 - TJF - Add extra arguments to support search paths
                         and reporting of which file was opended.
 */
extern void FpilModelAccess(
    const FpilType instrument,
    const char *dirOverride,
    const char *dirFallback,
    FpilRptRoutineType RptRoutine,
    void * RptArg,
    unsigned int field,
    FpilModelType **model,
    FpilStatusType *status)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    if (*status != STATUS__OK) return;

/*
 *  Support for a telescope model is optional. 
 */
    if (inst->hasModel)
    {
        (*inst->model.Access)(inst->clientData,
                              dirOverride,
                              dirFallback,
                              RptRoutine,
                              RptArg,
                              field,
                              model,
                              status);
        if (*status == STATUS__OK)
        {
            if (model)
            {
                (*model)->routines = inst->model;
                strcpy((*model)->Telescope, inst->params.Telescope);
            }
            else
                *status = FPIL__MODELINITERR;
        }
    }
    else
    {
        *model = 0;
        *status = FPIL__NOMODEL;
    }
}

/*
 *+                 F p i l M o d e l F r e e

 * Function name:
      FpilModelFree

 * Function:
      Free instument telescope model details.

 * Description:
      Free up details allocated by FpilModelAccess.  If provided, the
      model specific free routine is invoked.  Otherwise, free(3) is invoked
      on the model.

 * Language:

      C

 * Call:
        (void) = FpilModelFree(model, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (FpilModelType *model) Contains the telescope model details
                  returned by FpilModelAccess(3).  FpilModelCvtInit(3) may
                  or may not have been invoked.
      (!) status  (StatusType *) modified status.
                        

 * Include files: fpil.h

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern void FpilModelFree(
    FpilModelType *model,
    FpilStatusType *status)
{
    if (*status != STATUS__OK) return;

    if (model->routines.Free)
    {
        (model->routines.Free)(model,status);
    }
    else
        free(model);
}


/*
 *+                 F p i l M o d e l C v t I n i t

 * Function name:
      FpilModelCvtInit

 * Function:
      Initialise telescope model conversion for a particular time etc.

 * Description:
      This routine provides various parameters required to initialise
      the FpilModel conversion routines for a particual time etc.  It
      allows the other conversion routines to be called more efficently.

 * Language:

      C

 * Call:
        (void) = FpilModelCvtInit(model, mjd, dut, temp, pres, humid, cenWave,
                                  obsWave, params, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (FpilModelType *model) Contains the telescope model details
                           obtained by FpilModelAccess(3).
      (>) mjd     (double) UTC data and time as a modified julian date.
      (>) dut     (double) Delta UT (UT1 - UTC) seconds.
      (>) temp    (double) Atmospheric temperature (K).
      (>) press   (double) Atmospheric pressure (mB).
      (>) humid   (double) Atmospheric humidity (0 to 1.0).
      (>) cenWave (double) Wavelength telescope pointed using (micons).
      (>) obsWave (double) Wavelength of observation (micons).
      (>) params  (double *) Telescope model parameter details.  See
           FpilModelParams() for more information.
      (!) status  (StatusType *) modified status.
                        

 * Include files: fpil.h

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
     30-Sep-2002 - TJF - Add cenWave argument.
 */
extern void FpilModelCvtInit(
    FpilModelType *model,
    double mjd,
    double dut,
    double temp,
    double pres,
    double humid,
    double cenWave,
    double obsWave,
    const double * params,
    FpilStatusType *status)
{
    if (*status != STATUS__OK) return;

    (model->routines.CvtInit)(model,mjd,dut,temp,pres,humid,
                              cenWave, obsWave,params,
                              status);
}


/*
 *+                 F p i l M o d e l R d 2 X y

 * Function name:
      FpilModelRd2Xy

 * Function:
      Perform a conversion from Ra/Dec to field X/Y.

 * Description:
      Calls the instrument specific telescope model conversion routines.

      Assumes FpilModelCvtInit() has been invoked.

 * Language:

      C

 * Call:
        (void) = FpilModelRd2Xy(model, cra, cdec, ra, dec, mjd, x, y, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const FpilModelType *model) Contains the telescope 
                        model details obtained by FpilModelAccess(3) and on 
                        which FpilModelCvtInit(3) has been invoked.
      (>) cra     (double) Apparent RA of field center.
      (>) cdec    (double) Apparent DEC of field center.
      (>) ra      (double) Apparent RA of source.
      (>) dec     (double) Apparent DEC of source.
      (>) mjd     (double) UTC time and date expressed as Modified Julian Date
      (<) x       (double *) X position of source on field plate (microns).
      (<) y       (double *) Y position of source on field plate (microns).
      (!) status  (StatusType *) modified status.
                        

 * Include files: fpil.h

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern void FpilModelRd2Xy(
    const FpilModelType *model,
    double cra,
    double cdec,
    double ra,
    double dec,
    double mjd,
    double *x,
    double *y,
    FpilStatusType *status)
{
    if (*status != STATUS__OK) return;
    (model->routines.Rd2Xy)(model,cra,cdec,ra,dec,mjd,x,y,status);
}

/*
 *+                 F p i l M o d e l X y 2 P o s

 * Function name:
      FpilModelXy2Pos

 * Function:
      Perform a conversion from field X/Y to positioner X/Y.

 * Description:
      Calls the instrument specific telescope model conversion routines.

      Assumes FpilModelCvtInit() has been invoked.

 * Language:

      C

 * Call:
        (void) = FpilModelXy2Pos(model, x, y, xp, yp, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const FpilModelType *model) Contains the telescope model 
                   details obtained by FpilModelAccess(3) and on which
                           FpilModelCvtInit(3) has been invoked.
      (>) x       (double) Field plate x coordinate of object (microns).
      (>) y       (double) Field plate y coordinate of object (microns).
      (>) *xp     (double) Positioner X coordinate of object (microns).
      (>) *yp     (double) Positioner Y coordinate of object (microns).
      (!) status  (StatusType *) modified status.
                        

 * Include files: fpil.h

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern void FpilModelXy2Pos(
    const FpilModelType *model,
    double x,
    double y,
    double *xp,
    double *yp,
    FpilStatusType *status)
{
    if (*status != STATUS__OK) return;
    (model->routines.Xy2Pos)(model,x,y,xp,yp,status);
}


/*
 *+                 F p i l M o d e l X y 2 R d

 * Function name:
      FpilModelXy2Rd

 * Function:
      Perform a conversion field from X/Y to RA/DEC.

 * Description:
      Calls the instrument specific telescope model conversion routines.

      Assumes FpilModelCvtInit() has been invoked.

 * Language:

      C

 * Call:
        (void) = FpilModelXy2Rd(model, cra, cdec, x, y, mjd, ra, dec, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const FpilModelType *model) Contains the telescope model 
                       details obtained by FpilModelAccess(3) and on which
                           FpilModelCvtInit(3) has been invoked.
      (>) cra     (double) Apparent RA of field center.
      (>) cdec    (double) Apparent DEC of field center.
      (>) x       (double) X position of source on field plate (microns).
      (>) y       (double) Y position of source on field plate (microns).
      (>) mjd     (double) UTC time and date expressed as Modified Julian Date
      (<) ra      (double *) Apparent RA of source.
      (<) dec     (double *) Apparent DEC of source.
      (!) status  (StatusType *) modified status.
                        

 * Include files: fpil.h

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern void FpilModelXy2Rd(
    const FpilModelType *model,
    double cra,
    double cdec,
    double x,
    double y,
    double mjd,
    double *ra,
    double *dec,
    FpilStatusType *status)
{
    if (*status != STATUS__OK) return;
    (model->routines.Xy2Rd)(model,cra,cdec,x,y,mjd,ra,dec,status);

}

/*
 *+                 F p i l M o d e l P o s 2 X y

 * Function name:
      FpilModelPos2Xy

 * Function:
      Perform a conversion from positioner X/Y to field X/Y .

 * Description:
      Calls the instrument specific telescope model conversion routines.

      Assumes FpilModelCvtInit() has been invoked.

 * Language:

      C

 * Call:
        (void) = FpilModelPos2Xy(model, xp, yp, x, y, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const FpilModelType *model) Contains the telescope model 
                       details obtained by FpilModelAccess(3) and on which
                           FpilModelCvtInit(3) has been invoked.
      (>) xp     (double) Positioenr X coordinate of object (microns).
      (>) yp     (double) Positioenr Y coordinate of object (microns).
      (>) *x      (double) Field plate x coordinate of object (microns).
      (>) *y      (double) Field plate y coordinate of object (microns).
      (!) status  (StatusType *) modified status.
                        

 * Include files: fpil.h

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern void FpilModelPos2Xy(
    const FpilModelType *model,
    double xp,
    double yp,
    double *x,
    double *y,
    FpilStatusType *status)
{
    if (*status != STATUS__OK) return;
    (model->routines.Pos2Xy)(model,xp,yp,x,y,status);
}



/*
 *+			F p i l M o d e l G e t T e l e s c o p e

 * Function name:
      FpilModelGetTelescope

 * Function:
       Return the name of the telescope.

 * Description:
       Return the name of the telescope.
       

 * Language:
      C

 * Call:
      (const char *) = FpilGetModelTelescope(model)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const FpilModelType *model) Contains the telescope model 
                       details obtained by FpilModelAccess(3).

 * Returned value:
        A pointer to the name of the telescope.  Valid for as long
        as "inst" is valid.  

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern const char* FpilModelGetTelescope(
    const FpilModelType *model)
{
    return model->Telescope;
}


/*
 *+                 F p i l M o d e l Q u i c k A p p t o O b s

 * Function name:
      FpilModelQuickAppToObs

 * Function:
      Perfrom a quick apparent to observered place conversion.

 * Description:
      Calls the instrument specific telescope model conversion routines.

      Assumes FpilModelCvtInit() has been invoked.

 * Language:

      C

 * Call:
        (void) = FpilModelQuickAppToObs(model, ra, dec,
                                        aob, zob, hob, dob, rob,
                                        status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const FpilModelType *model) Contains the telescope model 
                       details obtained by FpilModelAccess(3) and on which
                           FpilModelCvtInit(3) has been invoked.
      (>) ra      (double) Apparent center RA in radians.
      (>) dec     (double) Apparent Center Declination in radians.
      (<) aob     (double *) Observed azimuth (radians, N=0, E = 90degrees)
      (<) zob     (double *) Observed zenith distance (radians)
      (<) hob     (double *) Observed Hour Angle (radians)
      (<) dob     (double *) Observed Declination (radians)
      (<) rob     (double *) Observed Right Ascension (radians)
      (!) status  (StatusType *) modified status.
                        

 * Include files: fpil.h

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 *-

 * History:
     13-Mar-200 - TJF - Original version
 */
extern void FpilModelQuickAppToObs(
    const FpilModelType *model,
    double ra,
    double dec,
    double *aob,
    double *zob,
    double *hob,
    double *dob,
    double *rob,
    FpilStatusType *status)
{
    if (*status != STATUS__OK) return;
    (model->routines.QuickAppToObs)(model,ra,dec,aob,zob,hob,dob,rob,status);
}

/*
 *+                 F p i l M o d e l P a r a m s

 * Function name:
      FpilModelParams

 * Function:
      Get details of the telescope model parameters.

 * Description:
      Returns details of the telescope model parameters which must
      be set in the params argument in a call to FpilModelCvtInit().


 * Language:

      C

 * Call:
        (void) = FpilModelParams(model, NumberParams,
                                 ParamNames, ParamDescrs, ParamDefaults
                                        status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const FpilModelType *model) Contains the telescope model 
                       details obtained by FpilModelAccess(3) and on which
                           FpilModelCvtInit(3) has been invoked.
      (<) NumberParams (double *) The number of parameters.
      (<) ParamNames   (const char * const **) A pointer to an array of 
                                pointers to char,
                                 being the short names of each parameter.
      (<) ParamDescr   (const char * const **) A pointer to an array of 
                                 pointers to char,
                                 being a descriptive name of each parameter.
      (<) ParamDefaults (const double **) A pointer to an array of doubles
                                 being the default values for each parameter.
      (!) status  (StatusType *) modified status.
                        

 * Include files: fpil.h

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 *-

 * History:
     13-Mar-2000 - TJF - Original version
 */
extern void FpilModelParams(
    const FpilModelType *model,
    unsigned int *NumberParams,
    const char * const *  *ParamNames,
    const char * const * *ParamDescrs,
    const double *  * ParamDefaults,
    FpilStatusType *status)
{
    if (*status != STATUS__OK) return;

    *NumberParams = model->routines.NumParams;
    *ParamNames = model->routines.ParamNames;
    *ParamDescrs = model->routines.ParamDescrs;
    *ParamDefaults = model->routines.ParamDefaults;
    
        
}

