/*+		F P I L	C O N S

 * Module name:
     FPILCONS

 * Function:
     Fibre Positioner Instrument Library - Instrument Constants Access

 * Description:
     Provides a set of routines common to all fibre positioner instruments,
     providing access to the "Constants" information.  This information
     is part of the instrument current state description.  It is accessed
     via SDS ids.  The top level id must contain an item for each
     field plate, name "fieldn" where n is the field (plate) number.  These
     start at 0 (historical reasons).  E.g.  field0, field1 etc.

     Within each field, the following SDS array elements must exist.

         tPark     SDS_DOUBLE[]    ! Park orientation (radians)
         xPark     SDS_INT[]       ! Park position - x ordinate (microns)
         yPark     SDS_INT[]       ! Park position - y ordinate (microns)
         xPiv      SDS_INT[]       ! Fibre pivot position - x ordinate(microns)
         yPiv      SDS_INT[]       ! Fibre pivot position - y ordinate(microns)
         type      SDS_SHORT []    ! Fibre type - see below
         inUse     SDS_SHORT []    ! In use flag - see below

     Each of these have the same number of dimensions, which is the
     number of pivots in the instrument, and the index is the (pivot number)-1.

     The inUse flags indicate if a pivot is in use.  1 for yes, 0 for no.

     The type type indicates the pivot type.  0 - guide, 1 - object
     OTher values may indicate different object types, TBD.

         

 * Language:
      C

 * Support: Tony Farrell, AAO

 *-

 * Sccs Id:     fpilcons.c, Release 1.4, 06/16/00

 * History:
     09-Dec-1999 - TJF - Original version
     16-Mar-2000 - TJF - Add FpilConWhichSpec
     17-Mar-2000 - TJF - Add FpilConSpecInfo
     20-Mar-2000 - TJF - Include fpil_err.h.
 */
/*
 *  Include files
 */
#include <stdlib.h>
#include <stdio.h>
#include "status.h"
#include "sds.h"
#include "fpil.h"
#include "fpil_err.h"

/* 
 * Constant definitions
 */

/*
 * Macro function definitions
 */
/*
 * type definitions
 */


/*
 * Functions imported (but not declared in include files)
 */




/*
 * Variables private to this module (static variables)
 */



/*
 * Function Definitions
 *
 */


/*
 *+                 F p i l C o n s A c c e s s

 * Function name:
      FpilConsAccess

 * Function:
      Access instument constants details.

 * Description:
      The Fibre Positioner Instrument Constants details provides details
      about fibre pivot positions, park positions and fibre usage

      This functions initiates access to those details for the
      fibre positioner specified by "inst".

      Normally such access involves opening a file or files.  The caller
      can specify two directores to search for the files and the
      instrument library may know of a possible location itself.  If the
      file exists in the  dirOverride directory it will be opened from
      there, if not, an instrument specific directory may be searched and
      finally, if it can't be opended from there, it will be opended
      from the dirFallback directory if possible.

      The RptRoutine may be specified to get feedback (for messages) about
      which file was opened.  It will be called for every file opened with
      the full name of the file which was opened.
      

 * Language:

      C

 * Call:
        (void) = FpilConsAccess(inst, dirOverride, 
                    dirFallback, RptRoutine, RptArg, details, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) dirOverride     (const char *)  If non-null, a directory to
                                   search for the constant details file.
      (>) dirFallback     (const char *)  If non-null, a fall back
                                   directory to search for the constants
                                   details file.
      (>) RptRoutine      (FpilRptRoutineType) A routine to be invoked
                                   to report which file has been opened.
      (>) RptArg          (void *) Argument passed to RptRoutine.
      (<) details   (FpilConstantsType *details) Contains the details
                    required for other FpilCons calls
      (!) status    (StatusType *) modified status.
                        

 * Include files: fpil.h

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
     10-Mar-2000 - TJF - Add extra arguments to support search paths
                         and reporting of which file was opended.
 */
extern void FpilConsAccess(
    const FpilType instrument,
    const char *dirOverride,
    const char *dirFallback,
    FpilRptRoutineType RptRoutine,
    void * RptArg,
    FpilConstantsType *details,
    FpilStatusType *status)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    if (*status != STATUS__OK) return;

/*
 *  We don't always have a routine to access the constants.  If we
 *  do, use it.
 */
    if (inst->routines.AccessCons)
    {
        (*inst->routines.AccessCons)(inst->clientData,
                                     dirOverride,
                                     dirFallback,
                                     RptRoutine,
                                     RptArg,
                                     details,
                                     status);
    }
    else
    {
        *details = 0;
        *status = FPIL__NOCONS;
    }
}

/*
 *+                 F p i l C o n s F r e e

 * Function name:
      FpilConsFree

 * Function:
      Free instument constants details.

 * Description:
      Free up details allocated by FpilConsAccess

 * Language:

      C

 * Call:
        (void) = FpilConsFree(details, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) details   (FpilConstantsType *details) Contains the field details
                      as retruned by FpilConsAccess(3).
      (!) status    (StatusType *) modified status.
                        

 * Include files: fpil.h

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern void FpilConFree(
    FpilConstantsType details,
    FpilStatusType *status)
{
    if (*status != STATUS__OK) return;

    if (details)
    {
        SdsReadFree(details,status);
        SdsFreeId(details,status);
    }
}

/*
 *+                 F p i l C o n s I d

 * Function name:
      FpilConsId

 * Function:
      Return an SDS id to an SDS structure containing the details.

 * Description:
      Return an SDS id to an SDS structure containing the constants
      details.  This SDS structures contains the items field0, ... fieldn
      where n is the value returned by FpilGetFields(3) (the number of 
      plates).   Details of the contents of each of these fields can
      be found in the description of FpilConsFieldId(3) below.
 

 * Language:

      C

 * Call:
        (void) = FpilConsId(details, id, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) details   (FpilConstantsType *details) Contains the field details
                      as retruned by FpilConsAccess(3).
      (<) id        (SdsIdType *) The id of the structure.  This will be
                    Free when you call FpilConsFree(3).
      (!) status    (StatusType *) modified status.
                        

 * Include files: fpil.h

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern void FpilConId(
    FpilConstantsType details,
    FpilSdsIdType *id,
    FpilStatusType *status)
{
    if (*status != STATUS__OK) return;

    *id = details;
}
/*
 *+                 F p i l C o n s F i e l d I d

 * Function name:
      FpilConsFieldId

 * Function:
      Return an SDS id to an SDS structure containing field details.

 * Description:
      Return an SDS id to an SDS structure containing the constants
      details for the specified field.  This SDS structures contains the 
      following items 


         tPark     SDS_DOUBLE[]    ! Park orientation (radians)
         xPark     SDS_INT[]       ! Park position - x ordinate (microns)
         yPark     SDS_INT[]       ! Park position - y ordinate (microns)
         xPiv      SDS_INT[]       ! Fibre pivot position - x ordinate(microns)
         yPiv      SDS_INT[]       ! Fibre pivot position - y ordinate(microns)
         type      SDS_SHORT[]     ! Fibre type - see below
         inUse     SDS_SHORT []    ! In use flag - see below

     Each of these have the same number of dimensions, which is the
     number of pivots (see FpilGetPivots(3) in the instrument, and the 
     index is the (pivot number)-1.

     The inUse flags indicate if a pivot is in use.  1 for yes, 0 for no.

     The type type indicates the pivot type.  0 - guide, 1 - object
     Other values may indicate different object types, TBD.
 

 * Language:

      C

 * Call:
        (void) = FpilConsFieldId(details, field, id, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) details   (FpilConstantsType *details) Contains the field details
                      as retruned by FpilConsAccess(3).
      (>)field      (unsigned int) The field.  The first field is 0.  The
                    last is given by (FpilGetFields()-1)
      (<) id        (SdsIdType *) The id of the structure.  You should use
                    SdsFreeId to free this when you are finished.
      (!) status    (StatusType *) modified status.
                        

 * Include files: fpil.h

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern void FpilConFieldId(
    FpilConstantsType details,
    unsigned int field,
    FpilSdsIdType *id,
    FpilStatusType *status)
{
    char buffer[20];
    if (*status != STATUS__OK) return;

    
/*
 *  For the moment, we don't validate the field index, just
 *  format it and use SdsFind.
 */
    sprintf(buffer,"field%d", field);
    SdsFind(details, buffer, id, status);
}





/*
 *+                 F p i l C o n W h i c h S p e c

 * Function name:
      FpilConWhichSpec

 * Function:
      Determimes the spectrograph associated with a pivot.

 * Description:
      Returns the spectrograph associated with a pivot.

      Only works if the WhichSpec function was supplied to FpilInit().  If
      not, the error code FPIL__NOWHICHSPEC will be returned.

 * Language:
      C

 * Call:
        (void) = FpilConWhichSpec(inst, details, field, pivot
                                  spec, SpecName, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) details  (FpilConstantsType) As was returned by FpilConsAccess().
      (>) field           (unsigned) The field plate number.
      (>) pivot           (unsigned) The pivot number, starting from 1.
      (>) Constants  (SdsIdType) The SDS id of the constant details that
                    was returned by the instrument specific contants access  
                    routine (AccessCons) when called by FpilConsAccess().
      (<) spec      (unsigned *) The spectrograph, where 0 indicates a
                    guide fibre, 1 unwards indicates a specrograph.
      (<) SpectName (const char * const *) A short descriptive name string
                    of the spectrograph.
      (!) status    (StatusType *) modified status.
                        

 * Include files: fpil.h

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 *-

 * History:
     16-Mar-2000 - TJF - Original version
 */
extern void FpilConWhichSpec(
    const FpilType instrument,
    FpilConstantsType details,
    unsigned field,
    unsigned pivot,
    unsigned *spec,
    const char ** SpecName,
    FpilStatusType *status)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    if (*status != STATUS__OK) return;

/*
 *  We don't always have a routine to access the constants.  If we
 *  do, use it.
 */
    if (inst->routines.WhichSpec)
    {
        (*inst->routines.WhichSpec)(inst->clientData,
                                    field,
                                    pivot,
                                    details,
                                    spec,
                                    SpecName,
                                    status);
    }
    else
    {
        *spec = 1;
        *SpecName = "";
        *status = FPIL__NOWHICHSPEC;
    }
}


/*
 *+                 F p i l C o n S p e c I n f o

 * Function name:
      FpilConSpecInfo

 * Function:
      Get details of the instruments spectrographs

 * Description:
      Returns details of the spectrograph.

      Only works if the WhichSpec function was supplied to FpilInit().  If
      not, the error code FPIL__NOWHICHSPEC will be returned.

 * Language:

      C

 * Call:
        (void) = FpilConSpecInfo(inst, NumberSpec, SpecNames, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (<) NumberSpec (double *) The number of spectrographs
      (<) SpecNames   (const char * const **) A pointer to an array of 
                                pointers to char,
                                 being the short names of each spectrograph.
      (!) status  (StatusType *) modified status.
                        

 * Include files: fpil.h

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 *-

 * History:
     17-Mar-2000 - TJF - Original version
 */
extern void FpilConSpecInfo(
    const FpilType instrument,
    unsigned int *NumberSpec,
    const char * const *  *SpecNames,
    FpilStatusType *status)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    if (*status != STATUS__OK) return;

    if (inst->routines.WhichSpec)
    {
        *NumberSpec = inst->params.NumSpec;
        *SpecNames =  inst->params.SpecNames;
    }
    else
    {
        *status = FPIL__NOWHICHSPEC;
        *NumberSpec = 0;
        *SpecNames = 0;

    }
        
}
