
/*******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: fpcolMdl.c 186402 2009-06-26 16:05:56Z mpruemm $"
* 
*
* who       when      what
* --------  --------  ----------------------------------------------
* tfarrell  15/06/00  created
* jbailey   02/04/00  Adapted from 6dF version.
* kshortridge 22/05/01  Added Ers calls for all cases where an error code
*                     is returned. Trapped invalid mjd in fpcolMdlRd2Xy().
* tfarrell  10/07/01  Merged in JAB's actual telescope model into OzPoz source.
*                     Add fpcolMdlDramaErrorToVLTError() routine.
* tfarrell  11/07/01  Merged in KS's changes from 22/05.  Ers calls only
*                     compiled during DRAMA (i.e. configure) build.
*                     Rename OzPos to OzPoz.
*                     Add fpcolMdlCvtMeanToRT().
* tfarrell  17/07/01  Add fpcolMdlCvtMeanToXY().
* tfarrell  01/08/01  Replace fpcolMdlMODEL_N_PARS by fpMDL_NUM_PARS.
*                     Add appRA and appDec to the values returned by
*                     fpcolMdlCvtMeanToXY() and fpcolMdlCvtMeanToRT().
* tfarrell  24/08/01  Use logData to log a message when errors occur in
*                     each case where the DRAMA version uses ErsRep().
* tfarrell  27/08/01  Tidy up error reporting by using fpcolMdl___RptError().
*                     In VLT case, report using error system instead of
*                     log system.
* tfarrell  04/09/01  add fpcolMdlDump
* tfarrell  20/09/01  Remove fpcolMdlDramaErrorToVLTError(), to be replaced
*                     by fpcolDRAMAErrorToVLTError().
* tfarrell  20/09/01  Support building with the calibration software, external
*                     to VLT software, indicated by CAL_BUILD macro.
* tfarrell  12/04/02  Support variable numbers of distortion parameters.  
* tfarrell  16/04/02  Support having rotation and scale for the linear
*                     model separated from the rest of the model to make
*                     fitting these parameters independently possible.  Add
*                     and use fpcolMdlNormalize().
* tfarrell  09/05/02  Drop model from XY2Tan argument list.
*                     Only apply scale when not 1.
*                     Load info of conversion initialisation when under VLT software.
* tfarrell  10/05/02  Ensure fpcolMdlCvtMeanToXY() and fpcolMdlCvtMeanToRT() are const correct.
* tfarrell  20/06/02  Fix some problems in XY2Tan.
* tfarrell  20/06/02  Incorporate revised distortion formula from Matthew Colless.
*                     Will's formula had an error as did the data he based it on. 
* tfarrell  20/06/02  Humidity min should be .1% rather then 1%.
* tfarrell  27/06/02  Humidity min should be 0%, but this should be set to .1.
*                     Add fpcolMdlRd2Tan() and reimplement fpcolMdlRd2Xy() using it.
* tfarrell  23/07/02  Maximum wavelength should be 1000 microns - Giraffe uses up to 920.
* tfarrell  19/08/02  Normalization of scale calculation was wrong - fixed.
* tfarrell  28/08/02  Add nonperpendicularity to model file.
* tfarrell  30/09/02  CvtInit needs the field centre wavelength as well as
*                     the observation wavelength.
* tfarrell  17/10/02  CvtInit should log the formated time, not just the MJD
* mpruemm   2009-06-23  Fix rcsId for VLT2009.
*/

/************************************************************************
*   NAME
*       fpcolMdl - Basic FLAMES/OzPoz Telescope model
*
*   SYNOPSIS
*       #include "fpcolMdl.h"
*
*       void fpcolMdlTransAccess (
*                               void * clientData,
*                               const char *dirOverride,
*                               const char *dirFallback,
*                               FpilRptRoutineType RptRoutine,
*                               void * RptArg,
*                               unsigned int field,
*                               FpilModelType **model,
*                               StatusType *status);
*
*        void fpcolMdlCvtInit (
*                           FpilModelType *model,
*                           double mjd,
*                           double dut,
*                           double temp,
*                           double pres,
*                           double humid,
*                           double cenWave,
*                           double obsWave,
*                           const double * params,
*                           StatusType *status);
*
*        void fpcolMdlRd2Xy (
*                         const FpilModelType *model,
*                         double cra,
*                         double cdec,
*                         double ra,
*                         double dec,
*                         double mjd,
*                         double *x,
*                         double *y,
*                         StatusType *status);
*
*       void fpcolMdlXy2Pos (
*                         const FpilModelType *model,
*                         double x,
*                         double y,
*                         double *xp,
*                         double *yp,
*                         StatusType *status);
*
*      void fpcolMdlXy2Rd (
*                      const FpilModelType *model,
*                      double cra,
*                      double cdec,
*                      double x,
*                      double y,
*                      double mjd,
*                      double *ra,
*                      double *dec,
*                      StatusType *status);
*
*      void fpcolMdlPos2Xy (
*                        const FpilModelType *model,
*                        double xp,
*                        double yp,
*                        double *x,
*                        double *y,
*                        StatusType *status);
*
*      void fpcolMdlFree (
*                      const FpilModelType *model,
*                      StatusType *status);
*
*
*       ccsCOMPL_STAT fpcolMdlCvtMeanToXY(
*                      IN  FpilModelType *model,
*                      IN  fpFIBRE_SKY_POS *skyCoord,
*                      IN  fpcolPM_PARS *pmPars,
*                      IN  double cenRA,
*                      IN  double cenDec,
*                      IN  double mjd,
*                      IN  double gmap[21],
*                      OUT fpPOS *pos,
*                      OUT double *appRA,
*                      OUT double *appDec,
*                      OUT ccsERROR *error);
*
*       ccsCOMPL_STAT fpcolMdlCvtMeanToRT(
*                      IN  FpilModelType *model,
*                      IN  fpFIBRE_SKY_POS *skyCoord,
*                      IN  fpcolPM_PARS *pmPars,
*                      IN  double cenRA,
*                      IN  double cenDec,
*                      IN  double mjd,
*                      IN  double gmap[21],
*                      OUT fpFIBRE_POS *plateCoord,
*                      OUT double *appRA,
*                      OUT double *appDec,
*                      OUT ccsERROR *error);
*
*
*        typedef struct  {
*              double ra; 
*              double dec; 
*              double parallax;
*              double radialVel;
*
*              fpcolPM_PARS(r=0,d=0,p=0,rv=0);
*
*              } fpcolPM_PARS;
*
*   DESCRIPTION
*      This library provides the basic OzPoz telescope model functions
*
*      This library implements routines to provide for transformation of
*      sky positions in RA/DEC coordinates to OzPoz positioner coordinates.
*
*      It implements functions required by the FPIL library to support
*      coordinate transformations.
*
*      This file is used by both the instrument software and
*      the configure software.  The Instrument software is written to VLT
*      software specifications.  The configure software is a standalone 
*      program written using the AAO's DRAMA style.  This file must be
*      compatible with both styles. 
*
*      This module has contains large amounts of code extracted from the
*      AAO 6dF instrument equivalent module.  As a result, its format
*      is not normal VLT format.
*
*      fpcolMdlTransAccess() is used to initialise access to the telescope
*      model.  The address of a variable of type FpilModelType must be
*      passed in and on return will be used with the other routines.  When
*      you are finished accessing the model, use fpcolMdlFree() to free
*      the structure pointed to by this variable.    The telescope model
*      will be read from the FP instrument status file (fpossInsStatus.sds).
*
*      To initialise the telescope model for a series of conversions with
*      a particular set of model parameters, call fpcolMdlCvtInit() specifying
*      mjd (of observation), dut (set to zero), temp (in kelvin),
*      pressure (millbars), humidity (0 to 1), field centre wavelength (microns)
*      (the wavelength the telescope is pointed using) observation wavelength
*      (microns) and 0 for the params item (which will be changed in the 
*      future).
*
*      To convert from Ra/Dec on the sky to field plate position, use 
*      fpcolMdlRd2Xy().  To convert from field plate postion to robot X/Y
*      use fpcolMdlXy2Pos().   
*
*      Likewise, us fpcolMdlPos2Xy() and fpcolMdlXy2Rd() for the reverse
*      transformations.
*
*      fpcolMdlCreateModel() initialises access to the OzPoz transformation
*      details. It is used by the OzPoz positioner software in place
*      of fpcolMdlTransAccess to allow explicitly setting the model parameters
*      from values in the OzPoz database.
*      
*
*      fpcolMdlGetModel() returns an array containing the parameters of a model 
*      in the same format as that given to fpcolMdlCreateModel.  
*            
*
*      fpcolMdlSetModel() is given an array containing the parameters of a model 
*      in the same format as that given to fpcolMdlCreateModel.
*
*      fpcolMdlWriteModel() writes the model to an SDS file of suitable format
*      for picking up by fpcolMdlReadModel. For OzPoz this is a file of name
*      OzPozModeln.sds for field plate n.
*
*      fpcolMdlReadModel() read the model from an SDS file written by
*      fpcolMdlWriteModel().  This is a file of name
*      OzPozModeln.sds for field plate n.
*
*      fpcolMdlCvtMeanToXY() performs a complete conversion from a mean
*      RA and Dec position (with proper motions) to a robot X/Y position.
*      fpcolMdlCvtMeanToRT() does the same but to robot R and Theta
*      positon.  "model" is as initialised fmmdlCvtInit().  "skyCoord"
*      supplied the mean RA and Dec position to convert (in HMS and DMS).  
*      "pmPars" supplies the proper motions for the object.  "cenRA" and
*      "cenDec" is the apparent field centre RA and Dec in radians.  "mjd" is
*      the date of observation.  "gmap" are the parameters for the
*      conversion for the given epoch and date, as returned by slaMappa(). 
*      The R and Theta values are returned in platecoord.  appRA and appDec
*      are used to return the apparent RA and Dec values determined during
*      the calculation.
*
*       fpcolPM_PARS is a structure used to store proper motion values.
*       "ra"  is the Proper motion RA  changes per Julian Year.  
*       "dec" is the Proper motion Dec changes per Julian Year.  
*       "parallax" is the parallax of the object in arcsec.
*       "radialVel" is the radial velocity (km/s, +ve if receding) 
*
*       When the include file is compiled into a C++ program, then
*       the fpcolPM_PARS has a constructor which takes the four values,
*       each with defaults of zero.  This allows you to easly initialise
*       such a structure in a C++ program.  If you contruct with all values
*       set to zero, then no proper motion is applied.
*
*
***
*
*   ENVIRONMENT
*       lopoz - Flames Fibre Positioner.
*       configure - AAO configure program (DRAMA environment).
*                       DRAMA only if DRAMA_BUILD macro is defined, otherise
*                       assume ESO.
*
*   CAUTIONS 
*
*   EXAMPLES
*
*   SEE ALSO
*       fpcol module, AAO FPIL Library.
*   BUGS   
*
*------------------------------------------------------------------------
*/


/*
 * Define module id for error macros
 */

#define MODULE_ID     "fpcol"
#define lccMODULE_ID     "fpcol"
/*
 * LOG system ID values
 */
#define fpcolLOG_ID 101 

#define _POSIX_SOURCE 1



/*
 *  Include files
 */


/*
 * If building for DRAMa (e.g. configure) then must include Ers.h so that
 * we have the ErsRep() prototype.
 */
#ifdef DRAMA_BUILD
#include "Ers.h"
/*
 * If building for calibration software.
 */
#elif defined (CAL_BUILD)
#include <stdio.h>
/*
 * If building for VLT (i.e, not building for DRAMA) - we need the following
 */
#else /* ! defined DRAMA_BUILD && ! defined CAL_BUILD*/
#include "vltPort.h"
#include "fpcol.h"
#include "fpcolErrors.h"
#include "tcsVcc.h"   /* RA and Dec format conversions */
#include "fppt.h"     /* For fppt functions */

#endif

#if !defined(ATTRIBUTE_UNUSED) && __GNUC__
#define ATTRIBUTE_UNUSED __attribute__((__unused__))
#else
#define ATTRIBUTE_UNUSED /*nothing*/
#endif

/* must be after vltPort.h */
ATTRIBUTE_UNUSED static const char *rcsId="@(#) $Id: fpcolMdl.c 186402 2009-06-26 16:05:56Z mpruemm $";


/*
 * Other include files.
 */
#include <stdlib.h>     /* For malloc and free */
#include <stdio.h>
#include <math.h>
#include <string.h>
#include "slalib.h"
#include "slamac.h"

#include "sds.h"

/*
 * Before including fpcolMdl.h we must define this macro to ensure
 * the correct include files are brought in.
 */
#define FPCOL_MDL_MODULE 1
#include "fpcolMdl.h"
#include "fpcolMdl_err.h"

/*
 * Parameter ranges for the parameters to fpcolMdlCvtInit().
 */
#define TEMP_MIN 200.0  /* Min temperature we expect to see  -73 C */
#define TEMP_MAX 330.0  /* Max temperature we expect to see  +57 C */
#define PRES_MIN 200.0  /* Min pressure we expect to see - millibars */
#define PRES_MAX 1000.0 /* Max pressure we expect to see - millibars */
#define HUMI_MIN 0.0    /* Min humidity we expect to see - relative (0%) */
#define HUMI_MAX 1.0    /* Max humidity we expect to see - relative (100%) */
#define WAVE_MIN 0.3    /* Min wavelength we expect to see - microns (300 nanometers) */
#define WAVE_MAX 1.0    /* Min wavelength we expect to see - microns (1000 nanometers) */



/*
 * Maximum X/Y value - 1000000 microns, 1 meter, well outside the plate.
 */
#define X_Y_MAX 1000000

/*
 * Maximum lenght for file names
 */
#define FILE_LENGTH 400
/*
 * type definitions
 */

/*
 * This structure contains the parameters of the distortion model 
 * read/created by fpcolMdlTransAccess() etc.  The number of values allowed
 * is the number defined by fpMDL_NUM_PARS minus 10 (the number of parameters
 * in the linear model)
 */
#define NUM_DIST_PARS (fpMDL_NUM_PARS-10)
#if (NUM_DIST_PARS < 6)
#error "Invalid number of distortion parameters"
#endif

typedef struct {

    double dist[NUM_DIST_PARS];   /*  distortion model coefficients  */
    
} fpcolMdlDistType;


/*
 * This structure contains the parameters of the linear model 
 * read/created by fpcolMdlTransAccess() etc. and it's inverse.
 */
typedef struct  {
    double coeffs[6];      /* Forward direction coefficents */
    double invCoeffs[6];   /* Reverse direction  coefficents */
    double rotation;       /* A rotation in degrees         */
    double scale;          /* A scale to be applied         */
    double nonperp;        /* A non-perp perpendicularity to be applied */
} fpcolMdlLinType;


/*
 * This structure contains details passed from fpcolMdlCvtInit to the
 * various conversion routines.
 */
typedef struct  {
    double cenAoprms[14];     /*  Apparent to observed parameters - field center wavelength  */
    double obsAoprms[14];     /*  Apparent to observed parameters - observation wavelength   */
    double cenLambda;         /*  Observing wavelength  */
    double obsLambda;         /*  Observing wavelength  */
    double ypa;            /*  Sky position angle of Y-axis  */
} fpcolMdlXYType;

/*
 *  This structure contains the position transformation model details.
 */

typedef struct {
    FpilModelType   model;        /* Must be first, see FpilModelAccess() */
    fpcolMdlDistType   distpars;   /* Set by fpcolMdlTransAccess etc. */
    fpcolMdlLinType    linpars;    /* Set by fpcolMdlTransAccess etc. */
    fpcolMdlXYType     xypars;     /* Set by fpcolMdlCvtInit      */
    unsigned int    fieldplate; /* Field plate number       */
} fpcolMdlModel, *fpcolMdlModelPnt;

static void fpcolMdl___Tan2XY  (
    const fpcolMdlModelPnt model,
    double xi,
    double eta,
    double *x,
    double *y,
    StatusType *status);

static void fpcolMdl___XY2Tan  (
    const fpcolMdlModelPnt model,
    double x,
    double y,
    double *xi,
    double *eta,
    StatusType *status);

/*
 * Debugging function - dumps the model to stderr.
 */
extern void fpcolMdlDump(FpilModelType *model)
{
    int i;
    fpcolMdlModelPnt MyModel = (fpcolMdlModelPnt)model;
    fprintf(stderr,"---------------------------\n");
    if (!model) 
        {
	fprintf(stderr,"No model pointer supplied\n");
	return;
	}
    fprintf(stderr,"Dump of telescope model info, model address = %p\n", MyModel);
    
    fprintf(stderr,"Distortion = ");
    for (i = 0 ; i < NUM_DIST_PARS ; ++i)
      fprintf(stderr,"%g ", MyModel->distpars.dist[i]);
    fprintf(stderr,"\n");

    fprintf(stderr,"Linear Fwd = %g %g %g %g %g %g\n",
            MyModel->linpars.coeffs[0],
            MyModel->linpars.coeffs[1],
            MyModel->linpars.coeffs[2],
            MyModel->linpars.coeffs[3],
            MyModel->linpars.coeffs[4],
            MyModel->linpars.coeffs[5]);
    fprintf(stderr,"Linear Inv = %g %g %g %g %g %g\n",
            MyModel->linpars.invCoeffs[0],
            MyModel->linpars.invCoeffs[1],
            MyModel->linpars.invCoeffs[2],
            MyModel->linpars.invCoeffs[3],
            MyModel->linpars.invCoeffs[4],
            MyModel->linpars.invCoeffs[5]);
    fprintf(stderr,"Scale = %g, Rotation = %g (degrees), Non-perp = %g(degrees)\n",
            MyModel->linpars.scale,
            MyModel->linpars.rotation,
            MyModel->linpars.nonperp);
    fprintf(stderr,"XY cenAopars = 0(%g) 1(%g) 2(%g) 3(%g) 4(%g)\n",
            MyModel->xypars.cenAoprms[0],
            MyModel->xypars.cenAoprms[1],
            MyModel->xypars.cenAoprms[2],
            MyModel->xypars.cenAoprms[3],
            MyModel->xypars.cenAoprms[4]);

    fprintf(stderr,"             5(%g) 6(%g) 7(%g) 8(%g) 9(%g)\n",
            MyModel->xypars.cenAoprms[5],
            MyModel->xypars.cenAoprms[6],
            MyModel->xypars.cenAoprms[7],
            MyModel->xypars.cenAoprms[8],
            MyModel->xypars.cenAoprms[9]);

    fprintf(stderr,"             10(%g) 11(%g) 12(%g) 13(%g)\n",
	    MyModel->xypars.cenAoprms[10],
            MyModel->xypars.cenAoprms[11],
	    MyModel->xypars.cenAoprms[12],
	    MyModel->xypars.cenAoprms[13]);
 
    fprintf(stderr,"XY obsAopars = 0(%g) 1(%g) 2(%g) 3(%g) 4(%g)\n",
            MyModel->xypars.obsAoprms[0],
            MyModel->xypars.obsAoprms[1],
            MyModel->xypars.obsAoprms[2],
            MyModel->xypars.obsAoprms[3],
            MyModel->xypars.obsAoprms[4]);

    fprintf(stderr,"             5(%g) 6(%g) 7(%g) 8(%g) 9(%g)\n",
            MyModel->xypars.obsAoprms[5],
            MyModel->xypars.obsAoprms[6],
            MyModel->xypars.obsAoprms[7],
            MyModel->xypars.obsAoprms[8],
            MyModel->xypars.obsAoprms[9]);

    fprintf(stderr,"             10(%g) 11(%g) 12(%g) 13(%g)\n",
	    MyModel->xypars.obsAoprms[10],
            MyModel->xypars.obsAoprms[11],
	    MyModel->xypars.obsAoprms[12],
	    MyModel->xypars.obsAoprms[13]);
    
    fprintf(stderr,"XY Cen lambda = %g\n", MyModel->xypars.cenLambda);
    fprintf(stderr,"XY Obs lambda = %g\n", MyModel->xypars.obsLambda);
    fprintf(stderr,"XY ypa        = %g\n",   MyModel->xypars.ypa);
    fprintf(stderr,"Field plt    = %d\n", MyModel->fieldplate);

    fprintf(stderr,"---------------------------\n");


}

/*
 *+	      	f p c o l M d l _ _ _ R p t E r r o r

 * Function name:
      fpcolMdl___RptError

 * Function:
       Report an error.

 * Description:
       Reports an error message using either VLT or DRAMA protocols, depending
       on what system the code has been built for.,

 * Language:
      C

 * Call:
      (void) = fpcolMdl___RptError (string,  status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) string    (const char *) The error string to report.
      (!) status    (StatusType *) modified status.  When under DRAMA,
                      will be passed to ErsRep.  When under VLT, is not used.

 *-

 * History:
     27-Aug-2001 - TJF - Original version.

 */
static  void fpcolMdl___RptError (
    const char *string,
    StatusType *status)
{
#ifdef DRAMA_BUILD
    ErsRep (0,status,string);
#elif defined (CAL_BUILD)
    status = 0;  /* Just to prevent complaints about unused variable */
    fprintf(stderr,"%s\n", string);
#else
    /*
     * Set up a VLT error stack.
     */
    ccsERROR error;
    errResetStack(&error);
    /*
     * Add extra supplied string to the error - uses the special
     * error code EXTRA_INFO which just has a %s item.
     */
    errAdd(&error,  MODULE_ID, fpcolERR_EXTRA_INFO, __FILE__, "%s", string);
    /*
     * Close the error stack.
     */
    errCloseStack(&error);

    /*logData(lccMODULE_ID, fpcolLOG_ID, string);*/
#endif

}

/*
 *+	      	f p c o l M d l _ _ _ O p e n M o d e l F i l e

 * Function name:
      fpcolMdl___OpenModelFile

 * Function:
      Open a file containging the OzPoz transformation details.

 * Description:
       The name for the file is opened and the Sds ID of the open
       file returned.  Allows fpcolMdlTransAccess and fpcolMdlReadModel to
       use the same directory approach to reading different files.

 * Language:
      C

 * Call:
      (void) = fpcolMdl___OpenModelFile (filename, dirOverride, 
               dirFallback, filebuf, id,  status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) filename        (const char *) Base name of file to open.
      (>) dirOverride     (const char *)  If non-null, a directory to
                                   search for the distortion details file.
      (>) dirFallback     (const char *)  If non-null, a fall back
                                   directory to search for the distortion
                                   details file.
      (<) filebuf         (char **) Pointer to a location where the name
                                   of the open file is written.  If this
                                   routine returns with status ok, then
                                   this is the name of the file opened,
                                   including directory.  In this case, 
                                   the caller must free() this buffer 
      (<) id              (SdsIdType *id) The Sds id of the file opened.  Must
                                   be freed using SdsReadFree() and SdsFreeId
                                   when the caller is finished reading
                                   the file.
      (!) status    (StatusType *) modified status.

 * Include files: fpcolMdl.h

 *-

 * History:
     13-Aug-2001 - TJF - Original version - Extracted code from TransAccess.

 */
static  void fpcolMdl___OpenModelFile (
    const char *filename,
    const char *dirOverride,
    const char *dirFallback,
    char **filebuf,
    SdsIdType  *id,
    StatusType *status)
{
    int found;
    unsigned long len = 0;
    unsigned long len2 = 0;

    *filebuf = 0;


    if (*status != STATUS__OK) return;

/*
 *  Work out the length of the longest of dirOverride and dirFallback.
 */
    if (dirOverride)
        len = strlen(dirOverride);
    if (dirFallback)
        len2 = strlen(dirFallback);
    if (len2 > len)
        len = len2;
/*
 *  Allocate specfic for the file name.
 */
    if ((*filebuf = malloc(len + strlen(filename) + 2)) == 0) 
        {
        *status = FPCOL_MDL__MALLOCERR;
        fpcolMdl___RptError("Error alloccating memory for file name buffer",
                            status);
        return;
        }


    found = 0;
/*
 *  Try to open the file in dirOverride if one was specified  
 */    
    if (dirOverride != 0)
        {
        strcpy(*filebuf,dirOverride);
        strcat(*filebuf,"/");
        strcat(*filebuf,filename);
        SdsRead(*filebuf,id,status);
        if (*status != STATUS__OK)
            {
            *status = STATUS__OK;
            found = 0;
            } else {
            found = 1;
            }    
        }            
    
/*  If not yet found try to open in default directory  */    
    
    if (!found)
        {
        strcpy(*filebuf,filename);
        SdsRead(*filebuf,id,status);
        if (*status != STATUS__OK)
            {
            *status = STATUS__OK;
            found = 0;
            } else {
            found = 1;
            }    
        }
    
/*  If still not found try to open in dirFallback  */    
    
    if (!found && (dirFallback != 0))
        {
        strcpy(*filebuf,dirFallback);
        strcat(*filebuf,"/");
        strcat(*filebuf,filename);
        SdsRead(*filebuf,id,status);
        if (*status != STATUS__OK)
            {
            *status = STATUS__OK;
            found = 0;
            } else {
            found = 1;
            }    
        }
    if (!found)
        {
        char mesbuf[100];
        free(*filebuf);
        *status = FPCOL_MDL__NODISTFILE;
        sprintf(mesbuf,  "Could not find file '%s'",filename);
        fpcolMdl___RptError(mesbuf, status);
        }     

}
/*
 *+	      	f p c o l M d l R e a d M o d e l

 * Function name:
      fpcolMdlReadModel

 * Function:
      Read the OzPoz transformation details 

 * Description:
      This routine initialises access to the OzPoz transformation
      details from calibration model files (named OzPozModel%d.sds,
      where %d is replaced by the plate number - 1 to 4).

      This routine will read the model for the specified plate and 
      malloc and return the address of a structure of type
      FpilModelType.  By actually mallocing a different structure
      (of type fpcolMdlModelType) which contains as its first item a structure 
      of type FpilModelType, the OzPoz telescope model implementation routines
      can keep there own information in this structure.  It just
      casts to and from FpilModelType as required (this can be considered
      deriving from FpilModelType, in the O-O sense).

      This routine is normally used from the calibration model software,
      rather then the configure software (which uses fpcolMdlTransAccess()).
      
      See FpilModelAccess for details on the usage of the directory
      releated arguments.

 * Language:
      C

 * Call:
      (void) = fpcolMdlReadModel(dirOverride, 
               dirFallback, RptRoutine, RptArg,  field, model,  status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) dirOverride     (const char *)  If non-null, a directory to
                                   search for the distortion details file.
      (>) dirFallback     (const char *)  If non-null, a fall back
                                   directory to search for the distortion
                                   details file.
      (>) RptRoutine      (FpilRptRoutineType) A routine to be invoked
                                   to report which file has been opened.
      (>) RptArg          (void *) Argument passed to RptRoutine.
      (>) field     (unsigned int) The field plate number.
      (<) model     (FpilModelType **) The address of the model is returned
                                   here.
      (!) status    (StatusType *) modified status.

 * Include files: fpcolMdl.h

 *-

 * History:
     13-Aug-2001 - TJF - Original version, based on fpcolMdlTransAccess.

 */
extern void fpcolMdlReadModel (
    const char *dirOverride,
    const char *dirFallback,
    FpilRptRoutineType RptRoutine,
    void * RptArg,
    unsigned int field,
    FpilModelType **model,
    StatusType *status)
{
    char fname[16];
    SdsIdType id = 0;
    SdsIdType pid = 0;
    double *pars;
    unsigned long len;
    char *filebuf = 0;

    if (*status != STATUS__OK) return;

    *model = 0;

/*
 *  Read the telescope model information from the OzPoz model file.
 *
 *  First open the file.
 */
    sprintf(fname,fpcolMODEL_FILE,field);

    fpcolMdl___OpenModelFile(fname, dirOverride, dirFallback,
                             &filebuf, &id, status);

/*
 *  Failure opening - return.
 */
    if (*status != STATUS__OK)
        return;
/*
 *  Find the structure.
 */
    SdsFind(id,"PARAMS",&pid,status);
    SdsPointer(pid,(void**)(&pars),&len,status);
    
/*
 *  Create the model.
 */
    fpcolMdlCreateModel(fpMDL_NUM_PARS,pars,field,model,status);
    
/*
 *  Free up SDS items.
 */
    if (id) {
    StatusType ignore = STATUS__OK;
    SdsReadFree(id,&ignore);
    SdsFreeId(id,&ignore);
    SdsFreeId(pid,&ignore);
    }
/*
 *  Report opening the file if required.
 */
    if (RptRoutine) 
        {
/*
 *      We need a buffer for our report.
 */
        char *msg = 0;
        if ((msg = malloc(strlen(filebuf) + 30)) == 0) 
            {
            *status = FPCOL_MDL__MALLOCERR;
            fpcolMdl___RptError(
                "Error alloccating memory for file name message", 
                status);
            }    
        else
            {
            strcpy(msg,"Opening ");
            strcat(msg,filebuf);
            (*RptRoutine)(RptArg, msg ,status);
            free(msg);
            }
        }
/*
 *  Free the buffer with the file name.
 */
    free(filebuf);


}

/*
 *+	      	f p c o l M d l T r a n s A c c e s s

 * Function name:
      fpcolMdlTransAccess

 * Function:
      Access the OzPoz transformation details.

 * Description:
      This routine initialises access to the OzPoz transformation
      details from the OzPoz instrument status file (fpossInsStatus.sds).  
      It is called by FpilModelAccess() (or directly) and must
      fit the prototype type defintion FpilModelAccessRoutineType

      This routine should access the model and 
      malloc and return the address of a structure of type
      FpilModelType.  By actually mallocing a different structure
      (of type fpcolMdlModelType) which contains as its first item a structure 
      of type FpilModelType, the OzPoz telescope model implementation routines
      can keep there own information in this structure.  It just
      casts to and from FpilModelType as required (this can be considered
      deriving from FpilModelType, in the O-O sense).
      
      See FpilModelAccess for details on the usage of the directory
      releated arguments.

 * Language:
      C

 * Call:
      (void) = fpcolMdlTransAccesss(clientData, dirOverride, 
               dirFallback, RptRoutine, RptArg,  field, model,  status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) clientData      (void *) Was passed to FpilInit.
      (>) dirOverride     (const char *)  If non-null, a directory to
                                   search for the distortion details file.
      (>) dirFallback     (const char *)  If non-null, a fall back
                                   directory to search for the distortion
                                   details file.
      (>) RptRoutine      (FpilRptRoutineType) A routine to be invoked
                                   to report which file has been opened.
      (>) RptArg          (void *) Argument passed to RptRoutine.
      (>) field     (unsigned int) The field plate number.
      (<) model     (FpilModelType **) The address of the model is returned
                                   here.
      (!) status    (StatusType *) modified status.

 * Include files: fpcolMdl.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version - template only at this stage
     10-Mar-2000 - TJF - Add extra arguments to support search paths
                         and reporting of which file was opended.
     10-Jul-2000 - TJF - Had not been swapping X and Y in field plate
			 to positioner units conversion as I should
			 have been.
     06-Mar-2001 - TJF - Correct field plate to position units conversion 
	                 inserted.
     27-Mar-2001 - JAB - Read model from OzPoz file.                    
     31-Aug-2001 - TJF - Revamp to use fpcolMdl___OpenModelFile() and to
                         read details from the instrument status file.
                         fpcolMdlReadModel() is not used to read the file
                         written by fpcolMdlWriteModel() instead of
                         this routine.
     27-Aug-2001 - KS  - Intercept use of field zero, which is now used to
                         access the 'super-plate' that combines all the
                         plate details.

 */
extern void fpcolMdlTransAccess (
    void * clientData,
    const char *dirOverride,
    const char *dirFallback,
    FpilRptRoutineType RptRoutine,
    void * RptArg,
    unsigned int field,
    FpilModelType **model,
    StatusType *status)
{
    SdsIdType ID = 0;
    SdsIdType plateID = 0;
    SdsIdType modelID = 0;
    double *pars;
    unsigned long len;
    char *filebuf = 0;
    char name[20];

    if (*status != STATUS__OK) return;

    clientData = 0;  /* Unused, Just to shut up noisey compilers */

    *model = 0;

/*
 *  We want to read the telescope model information from the OzPoz instrument
 *  status  file.
 *
 *  First open the instrument status file.
 */
    fpcolMdl___OpenModelFile(fpcolSTATUS_FILE , dirOverride, dirFallback,
                             &filebuf, &ID, status);

/*
 *  Failure opening - return.
 */
    if (*status != STATUS__OK)
        return;
/*
 *  If field is zero, we're looking for the 'super-plate'. In this case,
 *  we can get the model from any of the plates, so we make it plate 1.
 */

    if (field == 0) field = 1;


/*
 * SDS Name of plate specific component
 */
    sprintf(name,"Plate%d_Data", field);

/*
 * Find this plate's details in the SDS structure.  Access the
 * telescope model info.
 */
    SdsFind(ID, name, &plateID, status);
    SdsFind(plateID, "telModel", &modelID, status);
    SdsPointer(modelID,(void**)(&pars),&len,status);
    
/*
 *  Create the model.
 */
    fpcolMdlCreateModel(fpMDL_NUM_PARS,pars,field,model,status);
    
/*
 *  Tidy up SDS ID's.
 */
    if (ID) {
    StatusType ignore = STATUS__OK;
    SdsReadFree(ID,&ignore);
    SdsFreeId(ID,&ignore);
    SdsFreeId(plateID,&ignore);
    SdsFreeId(modelID,&ignore);
    }
/*
 *  Report file read if requested
 */
    if (RptRoutine)
        {
/*
 *      We need a buffer for our report.
 */
        char *msg = 0;
        if ((msg = malloc(strlen(filebuf) + 30)) == 0) 
            {
            *status = FPCOL_MDL__MALLOCERR;
            fpcolMdl___RptError(
                "Error alloccating memory for file name message",
                status);
            }    
        else
            {
            strcpy(msg,"Opening ");
            strcat(msg,filebuf);
            if (RptRoutine)
                (*RptRoutine)(RptArg, msg ,status);
            free(msg);
            }
        }
/*
 *  Free the buffer with the file name.
 */
    free(filebuf);


}


/*
 *+	      	f p c o l M d l C r e a t e M o d e l

 * Function name:
      fpcolMdlCreateModel

 * Function:
      Create an OzPoz transformation model from explicit parameters.

 * Description:
      This routine initialises access to the OzPoz transformation
      details. It is used by the OzPoz positioner software in place
      of fpcolMdlTransAccess to allow explicitly setting the model parameters
      from values in the OzPoz database. 

      This routine should access the model and 
      malloc and return the address of a structure of type
      FpilModelType.  By actually mallocing a different structure
      (of type fpcolMdlModelType) which contains as its first item a structure 
      of type FpilModelType, the OzPoz telescope model implementation routines
      can keep there own information in this structure.  It just
      casts to and from FpilModelType as required (this can be considered
      deriving from FpilModelType, in the O-O sense).
      

 * Language:
      C

 * Call:
      (void) = fpcolMdlCreateModel(npars, pars, field, model,  status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) npars           (int)     Number of parameters
      (>) pars            (double *) An array of parameters.   Currently
                                    should be at least fpMDL_NUM_PARS (7).
      (>) field     (unsigned int) The field plate number.
      (<) model     (FpilModelType **) The address of the model is returned
                                   here.
      (!) status    (StatusType *) modified status.

 * Include files: fpcolMdl.h

 *-

 * History:
     02-Apr-2001 - JAB - Original version - template only at this stage
     16-Apr-2002 - TJF - Where did the rest of the history go?
                         Support rotation and scale in the linear model
     28-Aug-2002 - TJF-  Support Non-perpendicularity.
 */
extern void fpcolMdlCreateModel (
    int npars,
    double *pars,
    unsigned int field, 
    FpilModelType **model,
    StatusType *status)
{
    fpcolMdlModelPnt MyModel;
    double a,b,c,d,e,f;
    double det;
    int i,j;

    if (*status != STATUS__OK) return;


    if (npars < fpMDL_NUM_PARS)
        {
        char mesbuf[100];
        *status = FPCOL_MDL__INVNPARS;
        sprintf(mesbuf,  "Wrong number of model parameters, %d instead of %d",
                npars, fpMDL_NUM_PARS);
        fpcolMdl___RptError(mesbuf, status);
        return;
        }    

/*
 *  Malloc data structure
 */
    if ((MyModel = (fpcolMdlModelPnt)malloc(sizeof(fpcolMdlModel))) == 0)
        {
        char mesbuf[100];
        *status = FPCOL_MDL__MALLOCERR;
        sprintf(mesbuf, "Unable to malloc %ld bytes for model", 
                (long)sizeof(fpcolMdlModel));
        fpcolMdl___RptError(mesbuf, status);
        return;
        }
    
/*  Set up linear coefficients  */    
    
    a = pars[0];
    b = pars[1];
    c = pars[2];
    d = pars[3];
    e = pars[4];
    f = pars[5];
    
    MyModel->linpars.coeffs[0] = a;
    MyModel->linpars.coeffs[1] = b;
    MyModel->linpars.coeffs[2] = c;
    MyModel->linpars.coeffs[3] = d;
    MyModel->linpars.coeffs[4] = e;
    MyModel->linpars.coeffs[5] = f;

/*  Calculate coefficients for inverse transformation   */

    det = b * f - c * e;
 
    MyModel->linpars.invCoeffs[0] = ( c * d - a * f ) / det;
    MyModel->linpars.invCoeffs[1] = f/det;
    MyModel->linpars.invCoeffs[2] = -1 * c/det;
    MyModel->linpars.invCoeffs[3] = ( a * e - b * d ) / det;
    MyModel->linpars.invCoeffs[4] = -1 * e / det;
    MyModel->linpars.invCoeffs[5] = b / det;

/*  Rotation, scale and non-perpendicularity for the linear model */

    MyModel->linpars.rotation = pars[fpcolMDL_PAR_ROT];
    MyModel->linpars.scale    = pars[fpcolMDL_PAR_SCALE];
    MyModel->linpars.nonperp  = pars[fpcolMDL_PAR_NONPERP];

/*  The rest - considered to be distortion */
    
    for (i = 0, j =  fpcolMDL_PAR_NW_DIST_FIRST ; i < NUM_DIST_PARS ; ++i, ++j)
      MyModel->distpars.dist[i] = pars[j];
    
    MyModel->fieldplate = field;

    *model = (FpilModelType *)MyModel;
}

/*
 *+			f p c o l M d l G e t M o d e l

 * Function name:
      fpcolMdlGetModel

 * Function:
      Return the parameters of a model


 * Description:
      This routine returns an array containing the parameters of a model 
      in the same format as that given to fpcolMdlCreateModel.

 * Language:
      C

 * Call:
      (void) = fpcolMdlGetModel(model, npars, pars, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (FpilModelType *model) Contains the telescope model details
                           obtained by fpcolMdlTransAccess, fpcolMdlReadModel or 
                           fpcolMdlCreateModel.
      (>) npars   (int)    Size of pars array.   Currently
                           should be at least fpMDL_MODEL_N_PARS (7).    
      (<) pars    (double *)   Array to receive model parameters.
      (!) status  (StatusType *) modified status.
                           

 * Returned value:

 * Include files: fpcolMdl.h

 *-

 * History:
     05-Apr-2001 - JAB - Original version 
     28-Aug-2002 - TJF-  Support Non-perpendicularity.

 */

extern void fpcolMdlGetModel(
    FpilModelType *model,
    int npars,
    double *pars,
    StatusType *status)
{    
    fpcolMdlModelPnt MyModel = (fpcolMdlModelPnt)model;
    int i,j;

    if (*status != STATUS__OK) return;

    if (npars < fpMDL_NUM_PARS) 
        {
        char mesbuf[100];
        *status = FPCOL_MDL__INVNPARS;
        sprintf(mesbuf,  "Wrong number of model parameters, %d instead of %d",
                npars, fpMDL_NUM_PARS);
        fpcolMdl___RptError(mesbuf, status);
        return;
        }     

    pars[0] = MyModel->linpars.coeffs[0];
    pars[1] = MyModel->linpars.coeffs[1];
    pars[2] = MyModel->linpars.coeffs[2];
    pars[3] = MyModel->linpars.coeffs[3];
    pars[4] = MyModel->linpars.coeffs[4];
    pars[5] = MyModel->linpars.coeffs[5];
    pars[fpcolMDL_PAR_ROT]     =  MyModel->linpars.rotation;
    pars[fpcolMDL_PAR_SCALE]   =  MyModel->linpars.scale;
    pars[fpcolMDL_PAR_NONPERP] =  MyModel->linpars.nonperp;

    for (i = 0, j = fpcolMDL_PAR_NW_DIST_FIRST ; i < NUM_DIST_PARS ; ++i, ++j)
      pars[j] = MyModel->distpars.dist[i];
    
}    

/*
 *+			f p c o l M d l S e t M o d e l

 * Function name:
      fpcolMdlSetModel

 * Function:
      Set the parameters of a model


 * Description:
      This routine is given an array containing the parameters of a model 
      in the same format as that given to fpcolMdlCreateModel.

 * Language:
      C

 * Call:
      (void) = fpcolMdlSetModel(model, npars, pars, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (FpilModelType *model) Contains the telescope model details
                           obtained by fpcolMdlTransAccess, fpcolMdlReadModel or 
                           fpcolMdlCreateModel.
      (>) npars   (int)  Size of pars array.  Currently
                           should be at least fpMDL_NUM_PARS (7).
      (>) pars    (double *)   Array containing model parameters.
      (!) status  (StatusType *) modified status.
                           

 * Returned value:

 * Include files: fpcolMdl.h

 *-

 * History:
     07-May-2001 - JAB - Original version 
     28-Aug-2002 - TJF-  Support Non-perpendicularity.

 */

extern void fpcolMdlSetModel(
    FpilModelType *model,
    int npars,
    double *pars,
    StatusType *status)
{    
    fpcolMdlModelPnt MyModel = (fpcolMdlModelPnt)model;
    int i,j;

    if (*status != STATUS__OK) return;

    if (npars < fpMDL_NUM_PARS) 
        {
        char mesbuf[100];
        *status = FPCOL_MDL__INVNPARS;
        sprintf(mesbuf,  "Wrong number of model parameters, %d instead of %d",
                npars, fpMDL_NUM_PARS);
        fpcolMdl___RptError(mesbuf, status);
        return;
        }     

    MyModel->linpars.coeffs[0] = pars[0];
    MyModel->linpars.coeffs[1] = pars[1];
    MyModel->linpars.coeffs[2] = pars[2];
    MyModel->linpars.coeffs[3] = pars[3];
    MyModel->linpars.coeffs[4] = pars[4];
    MyModel->linpars.coeffs[5] = pars[5];


/*  Rotation and scale for the linear model */

    MyModel->linpars.rotation = pars[fpcolMDL_PAR_ROT];
    MyModel->linpars.scale    = pars[fpcolMDL_PAR_SCALE];
    MyModel->linpars.nonperp  = pars[fpcolMDL_PAR_NONPERP];

/*  The rest - considered to be distortion */
    
    for (i = 0, j =  fpcolMDL_PAR_NW_DIST_FIRST ; i < NUM_DIST_PARS ; ++i, ++j)
      MyModel->distpars.dist[i] = pars[j];
}    


/*
 *+			f p c o l M d l W r i t e M o d e l

 * Function name:
      fpcolMdlWriteModel

 * Function:
      Write model to an SDS file


 * Description:
      This routine writes the model to an SDS file of suitable format
      for picking up by fpcolMdlReadModel. For OzPoz this is a file of name
      OzPozModel%d.sds for field plate %d.

 * Language:
      C

 * Call:
      (void) = fpcolMdlWriteModel(model, dirname, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (FpilModelType *model) Contains the telescope model details
                           obtained by fpcolMdlTransAccess(3), fpcolMdlReadModel(3) 
                           or fpcolMdlCreateModel(3).
                           This routine may or may not have been previously 
                           invoked on this model.
      (>) dirname (char*) Name of directory into which file will be written.
      (!) status  (StatusType *) modified status.
                           

 * Returned value:

 * Include files: fpcolMdl.h

 *-

 * History:
     05-Apr-2001 - JAB - Original version 
     28-Aug-2002 - TJF - Support Non-perpendicularity.
     10-Oct-2002 - TJF - Ensure all parameters are zeroed first to avoid
                         unset parameters containing junk. 

 */
 
void fpcolMdlWriteModel(
    FpilModelType *model,
    char* dirname, 
    StatusType *status)

{
    double pars[fpMDL_NUM_PARS];   
    char file[FILE_LENGTH];
    char fname[16];
    int l;
    int i,j;
    unsigned long int dims[7];   /* The 7 here is the max number of SDS 
                                    array dimensions */
    SdsIdType id;
    SdsIdType pid;
    fpcolMdlModelPnt MyModel = (fpcolMdlModelPnt)model;

    if (*status != STATUS__OK) return;
    

    /*
     * Zero all the values.
     */
    for (i = 0  ; i < fpMDL_NUM_PARS ; ++i)
        pars[i] = 0;
    

    /*
     * Create the SDs structure.
     */
    dims[0] = fpMDL_NUM_PARS;
    SdsNew(0,"OzPozModel",0,0,SDS_STRUCT,0,0,&id,status);
    SdsNew(id,"PARAMS",0,0,SDS_DOUBLE,1,dims,&pid,status);
    
    /*
     * Fill in details.
     */
    pars[0] = MyModel->linpars.coeffs[0];
    pars[1] = MyModel->linpars.coeffs[1];
    pars[2] = MyModel->linpars.coeffs[2];
    pars[3] = MyModel->linpars.coeffs[3];
    pars[4] = MyModel->linpars.coeffs[4];
    pars[5] = MyModel->linpars.coeffs[5];

    pars[fpcolMDL_PAR_ROT]     =  MyModel->linpars.rotation;
    pars[fpcolMDL_PAR_SCALE]   =  MyModel->linpars.scale;
    pars[fpcolMDL_PAR_NONPERP] =  MyModel->linpars.nonperp;

    for (i = 0, j = fpcolMDL_PAR_NW_DIST_FIRST ; i < NUM_DIST_PARS ; ++i, ++j)
      pars[j] = MyModel->distpars.dist[i];

    /*
     * Put into SDS.
     */
    SdsPut(pid,fpMDL_NUM_PARS*sizeof(double),0,pars,status);
    SdsFreeId(pid,status);

    /*
     * Generate the model file name.
     */
    sprintf(fname,fpcolMODEL_FILE,MyModel->fieldplate);

    if (dirname != 0)
        {
        strncpy(file,dirname,FILE_LENGTH);
        l = strlen(file);
        if (l > FILE_LENGTH-16) {
        *status = FPCOL_MDL__DIRLENERR;
        fpcolMdl___RptError("Directory name too long", status);
        return;
        }    
        strcat(file,"/");
        strcat(file,fname);
        } 
    else
        {
        strcpy(file,fname);
        }
    
    /*
     * Write file and tidy up.
     */
    SdsWrite(id,file,status);
    SdsDelete(id,status);
    SdsFreeId(id,status);          
}    
   

/*
 *+			f p c o l M d l C v t I n i t

 * Function name:
      fpcolMdlCvtInit

 * Function:
      Initialise for conversions at a specified time etc.


 * Description:
      This routine initialises the telescope model conversion routines
      for a particular time, wavelength and telescope model details.  This
      allows a sequence of transformation for one field to
      be done efficently.

      This routine is called by FpilModelCvtInit() and must meet
      the prototype given by the type defintion FpilModelCvtInitRoutineType.


 * Language:
      C

 * Call:
      (void) = fpcolMdlCvtInit(model,  mjd, dut, temp, pres, 
                                   humid, cenWave, obsWave, tel, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (FpilModelType *model) Contains the telescope model details
                           obtained by fpcolMdlTransAccess, fpcolMdlReadModel or
                           fpcolMdlCreateModel.  This routine
                           may or may not have been previously invoked
                           on this model.
      (>) mjd     (double) UTC data and time as a modified julian date.
      (>) dut     (double) Delta UT (UT1 - UTC) seconds.
      (>) temp    (double) Atmospheric temperature (K).
      (>) press   (double) Atmospheric pressure (mB).
      (>) humid   (double) Atmospheric humidity (0 to 1.0).
      (>) cenWave (double) Wavelength telescope is pointed with (microns)
      (>) obsWave (double) Wavelength of observation (micons).
      (>) params  (double *) Model parameters. For OzPoz only
                            params[0] is used and is the sky position angle of
                            the positioner y axis in radians.
      (!) status  (StatusType *) modified status.
                           

 * Returned value:

 * Include files: fpcolMdl.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version - template only at this stage
     03-Apr-2000 - TJF - Template expanded to use slaObs() and slaAoppa()
                         to set xypars.aoparms for fpcolMdlQuickAppToObs().
                         This is temp only, taken from 2dF equivalent and
                         should be reconsidered for 6dF.
     20-Jun-2000 - TJF - Implement, based on TdfXyInit(). We don't
                         yet use CH and NP parameters and they are ignored.
     03-Apr-2001 - JAB - OzPoz version                    
     30-Sep-2002 - TJF - Need the field center (telescope pointing) wavelength
                         separately.
 */
extern void fpcolMdlCvtInit (
    FpilModelType *model,
    double mjd,
    double dut,
    double temp,
    double press,
    double humid,
    double cenWave,
    double obsWave,
    const double *params,
    StatusType *status)
{
    double longit,lat,hm;
    char name[80];
    char buffer[200];
    fpcolMdlModelPnt MyModel = (fpcolMdlModelPnt)model;

    if (*status != STATUS__OK) return;
/*
 *  Get observatory parameters  
 */
    slaObs(-1,"VLT2",name,&longit,&lat,&hm);
    if (strcmp(name,"?") == 0)
        {
        *status = FPCOL_MDL__ILLNAME;
        fpcolMdl___RptError(
            "VLT2 not recognised by the version of SLA being used", status);
        return;
        }


/*
 *  Validate passed parameters.
 */
    if ((temp < TEMP_MIN)||(temp > TEMP_MAX))
      {
        *status = FPCOL_MDL__ILLTEMP;
        sprintf(buffer,
                "The tempreature value of %g kelvin is outside the expected range of %g to %g",
                temp, TEMP_MIN, TEMP_MAX);
        fpcolMdl___RptError(buffer, status);
        return;
      }
   if ((press < PRES_MIN)||(press > PRES_MAX))
      {
        *status = FPCOL_MDL__ILLPRES;
        sprintf(buffer,
                "The pressure value of %g miliibars is outside the expected range of %g to %g",
                press, PRES_MIN, PRES_MAX);
        fpcolMdl___RptError(buffer, status);
        return;
      }
   if ((humid < HUMI_MIN)||(humid > HUMI_MAX))
      {
        *status = FPCOL_MDL__ILLHUMI;
        sprintf(buffer,
                "The relative humidity value of %g  is outside the expected range of %g to %g",
                humid, HUMI_MIN, HUMI_MAX);
        fpcolMdl___RptError(buffer, status);
        return;
      }
   else if (humid == 0)
       /*
        * Just in case slaLib has trouble with humidty of zero, which is not
        * impossible at the VLT after rounding etc, set a humidity of zero
        * to 0.1.
        */
       humid = 0.1;

   if ((cenWave < WAVE_MIN)||(cenWave > WAVE_MAX))
      {
        *status = FPCOL_MDL__ILLWAVE;
        sprintf(buffer,
              "The centre wavelength value of %.3g microns is outside the expected range of %.3g to %.3g",
                cenWave, WAVE_MIN, WAVE_MAX);
        fpcolMdl___RptError(buffer, status);
        return;
      }

   if ((obsWave < WAVE_MIN)||(obsWave > WAVE_MAX))
      {
        *status = FPCOL_MDL__ILLWAVE;
        sprintf(buffer,
              "The observation wavelength value of %.3g microns is outside the expected range of %.3g to %.3g",
                obsWave, WAVE_MIN, WAVE_MAX);
        fpcolMdl___RptError(buffer, status);
        return;
      }

#if (!defined(DRAMA_BUILD)) && (!defined(CAL_BUILD))
   {
     char time_sign;
     int ymdf[4];
     int thmsf[4];
     int j;
 
     slaDjcal(2, mjd, ymdf, &j);
     slaCd2tf(0, mjd - (int)mjd, &time_sign, thmsf);
 
     sprintf(buffer, 
             "Telescope model initialised at mjd=%.6g (%.4d/%.2d/%.2d %.2d:%.2d:%.2d)",
             mjd, ymdf[0], ymdf[1], ymdf[2], thmsf[0], thmsf[1], thmsf[2]);
     logData(lccMODULE_ID, fpcolLOG_ID, buffer);
     sprintf(buffer, 
             "Telescope model initialised with cwave=%.3f nm, owave=%.3f nm  temp=%g K, pres=%g mb, humid=%g",
             cenWave, obsWave, temp, press, humid);
     logData(lccMODULE_ID, fpcolLOG_ID, buffer);
   }
#endif

/*  
 * Determine apparent to observed place parameters  
 */
    slaAoppa(mjd,dut,-longit,lat,hm,0.0,0.0,temp,press,humid,
             cenWave,0.0065, MyModel->xypars.cenAoprms);
    slaAoppa(mjd,dut,-longit,lat,hm,0.0,0.0,temp,press,humid,
             obsWave,0.0065, MyModel->xypars.obsAoprms);

/*
 *  Save the wavelengths.
 */
    MyModel->xypars.cenLambda = cenWave;
    MyModel->xypars.obsLambda = obsWave;
/*
 *  params[0] is the argus orientation.  We add the default
 *  rotation to this to get the rotation to be applied.
 */
    MyModel->xypars.ypa = params[0];


}



/*
 *+			f p c o l M d l R d 2 T a n

 * Function name:
      fpcolMdlRd2Tan

 * Function:
      Determine Tangent plane coorindates from apparent RA and Dec.

 * Description:
      Determines the Tangent coordinates in the field plate, from the apparent
      RA and Dec of a source and the field center.


 * Language:
      C

 * Call:
      (void) = fpcolMdlRd2Tan(model, cra, cdec, ra, dec, 
                                         mjd, xi, eta, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const FpilModelType *model) Contains the telescope 
                           model details  obtained by fpcolMdlTransAccess(3),
                           fpcolMdlReadModel(3) or fpcolMdlCreateModel(3).
                           It Can be assumed that the fpcolMdlCvtInit 
                           routine has been  invoked on this.
      (>) cra     (double) Apparent RA of field center.
      (>) cdec    (double) Apparent DEC of field center.
      (>) ra      (double) Apparent RA of source.
      (>) dec     (double) Apparent DEC of source.
      (>) mjd     (double) UTC time and date expressed as Modified Julian Date
      (<) xi      (double *) Tangent plane coordinates (radians)
      (<) eta     (double *) 
      (!) status  (StatusType *) modified status.
 
 * Include files: fpcolMdl.h

 *-

 * History:
     27-Jun-2002 - TJF - Original version - extracted from fpcolMdlRd2Xy() to support
                         facilties required in calibration software

 */        
extern void fpcolMdlRd2Tan (
    const FpilModelType *model,
    double cra,
    double cdec,
    double ra,
    double dec,
    double mjd,
    double *pxi,
    double *peta,
    StatusType *status)
{
    const fpcolMdlModelPnt MyModel = (const fpcolMdlModelPnt)model;
    double ocra,ocdec;        /* Observed ra,dec of field centre  */
    double ora,odec;          /* Observed ra,dec of source  */
    double oaz,ozd,oha;
    double xi,eta;
    double pa;                /* position angle of y axis  */
    fpcolMdlXYType modXypars;    /* To keep a modifed version of xypars  */
    int jstat;
    char mesbuf[200];


    if (*status != STATUS__OK) return;

/*
 *
 *  Validate arguments
 */
    if ((cra > D2PI) || (cra < 0.0))
        {
        *status = FPCOL_MDL__ILLRA;
        sprintf(mesbuf, "Invalid field center RA value (%.6f radians)",cra);
        fpcolMdl___RptError(mesbuf, status);
        }
    if ((cdec > DPI) || (cdec < (-DPI)))
        {
        *status = FPCOL_MDL__ILLDEC;
        sprintf(mesbuf, "Invalid field center Dec value (%.6f radians)",cdec);
        fpcolMdl___RptError(mesbuf, status);
        }
    if ((ra > D2PI) || (ra < 0.0))
        {
        *status = FPCOL_MDL__ILLRA;
        sprintf(mesbuf, "Invalid RA value (%.6f radians)",ra);
        fpcolMdl___RptError(mesbuf, status);
        }
    if ((dec > DPI) || (dec < (-DPI)))
        {
        *status = FPCOL_MDL__ILLDEC;
        sprintf(mesbuf,  "Invalid Dec value (%.6f radians)",dec);
        fpcolMdl___RptError(mesbuf, status);
        }
    if (*status != STATUS__OK)
        {
        /*
         * Report details of invalid arguments.
         */
        char ra_sign;
        char dec_sign;
        int hmsf[4];
        int dmsf[4];

        fpcolMdl___RptError(
            "Invalid arguments range in call to fpcolMdlRd2Xy", status);

        slaDr2tf(2, cra, &ra_sign, hmsf);
        slaDr2af(2, cdec, &dec_sign, dmsf);
        sprintf(mesbuf,
                "Supplied Field Centre (app) RA = %.6f(%c%.2d:%.2d:%.2d.%d) Dec = %.6f(%c%.2d:%.2d:%.2d.%d)",
                cra,
                ra_sign, hmsf[0], hmsf[1], hmsf[2], hmsf[3],
                cdec,
                dec_sign, dmsf[0], dmsf[1], dmsf[2], dmsf[3]);

        fpcolMdl___RptError(mesbuf, status);

        slaDr2tf(2, ra, &ra_sign, hmsf);
        slaDr2af(2, dec, &dec_sign, dmsf);

        sprintf(mesbuf,
               "Supplied Object Position (app) RA = %.6f(%c%.2d:%.2d:%.2d.%d) Dec = %.6f(%c%.2d:%.2d:%.2d.%d)",
                cra,
                ra_sign, hmsf[0], hmsf[1], hmsf[2], hmsf[3],
                cdec,
                dec_sign, dmsf[0], dmsf[1], dmsf[2], dmsf[3]);


        return;
        }

/*  
 *  Update Apparent to observed parameters  
 */
    slaAoppat(mjd, MyModel->xypars.obsAoprms);
    slaAoppat(mjd, MyModel->xypars.cenAoprms);
/*  
 *  Observed position of field centre  
 *
 *  Note some defensive programming here. If the field is essentially
 *  unobservable because of an unrealistic mjd value, then slaAopqk()
 *  will take forever doing the refraction calculations. So we do an
 *  approximate calculation first, disabling refraction by setting the
 *  pressure parameter to zero. If the calculated zenith distance is
 *  less than 90 degrees (~ 1.57 rad) we know the full calculation is
 *  feasible and repeat it properly.  (KS, 22-May-2001).
 *
 *  Create a copy of the paramters and set the pressure to zero.
 */
    modXypars = MyModel->xypars;
    modXypars.cenAoprms[6] = 0.0;

/*
 *  Do calculation and restore pressure.
 */
    slaAopqk(cra,cdec,modXypars.cenAoprms,&oaz,&ozd,&oha,&ocdec,&ocra);

/*
 *  Catch and report error.
 */
    if (ozd > 1.57)
        {
        char ra_sign;
        char dec_sign;
        char time_sign;
        int hmsf[4];
        int dmsf[4];
        int ymdf[4];
        int thmsf[4];
        int j;
        *status = FPCOL_MDL__ILLMJD;
        sprintf(mesbuf,  "The MJD of %.6f leads to invalid zenith distance of %.3f rad (%.0f deg)",
                mjd, ozd, (ozd*180.0/M_PI));
        fpcolMdl___RptError(mesbuf, status);

        slaDr2tf(2, cra, &ra_sign, hmsf);
        slaDr2af(2, cdec, &dec_sign, dmsf);
        slaDjcal(2, mjd, ymdf, &j);
        slaCd2tf(0, mjd - (int)mjd, &time_sign, thmsf);
 
        sprintf(mesbuf,
           "Field Centre (app) RA = %.6f(%c%.2d:%.2d:%.2d.%d) Dec = %.6f(%c%.2d:%.2d:%.2d.%d)",
                cra,
                ra_sign, hmsf[0], hmsf[1], hmsf[2], hmsf[3],
                cdec,
                dec_sign, dmsf[0], dmsf[1], dmsf[2], dmsf[3]);
        fpcolMdl___RptError(mesbuf, status);
        sprintf(mesbuf,
                "Observation Time:MJD = %.6f, UT = %.4d/%.2d/%.2d %.2d:%.2d:%.2d",
                mjd, ymdf[0], ymdf[1], ymdf[2], thmsf[0], thmsf[1], thmsf[2]);
        fpcolMdl___RptError(mesbuf, status);
        return;
        }

/*
 *  Get the observed positions of the field center and source.  Note the different
 *  aoprms used - to allow for the telescope being pointed at a different wavelength
 *  then we are observing at.
 *  
 *  Observed position of field centre  
 */
    slaAopqk(cra,cdec,MyModel->xypars.cenAoprms,&oaz,&ozd,&oha,&ocdec,&ocra);
/*
 *  Observed position of source  
 */
    slaAopqk(ra,dec,MyModel->xypars.obsAoprms,&oaz,&ozd,&oha,&odec,&ora);

    if (*status != STATUS__OK) return;
/*  
 *  Project to tangent plane.  Remember to catch errors.
 */
    slaDs2tp(ora,odec,ocra,ocdec,&xi,&eta,&jstat);

    if (jstat != 0) 
        {
        switch (jstat)
            {
            case 1:
                *status = FPCOL_MDL__SLA_ERR_1;
                break;
            case 2:
                *status = FPCOL_MDL__SLA_ERR_2;
                break;
            case 3:
                *status = FPCOL_MDL__SLA_ERR_3;
                break;
            default:
                *status = FPCOL_MDL__SLA_ERR_UNKN;
                break;
            }
        sprintf(mesbuf,"Slalib returns error code %d from slaDs2tp",jstat);  
        fpcolMdl___RptError(mesbuf, status);
        return;
        }
/*
 *  Flip sign of xi for east negative
 */
    xi = -xi;
    
/*
 *  Rotate through ypa
 */    

    pa = MyModel->xypars.ypa;
    *pxi = xi * cos(pa) + eta * sin(pa);
    *peta = - xi * sin(pa) + eta * cos(pa);
}

/*
 *+			f p c o l M d l R d 2 X y 

 * Function name:
      fpcolMdlRd2Xy

 * Function:
      Determine OzPoz x/y from apparent RA and Dec.

 * Description:
      Determines the x,y coordinates in the field plate, from the apparent
      RA and Dec of a source and the field center.

      This routine is called by FpilModelRd2Xy() and must meet
      the prototype given by the type defintion FpilModelRd2XyRoutineType.

 * Language:
      C

 * Call:
      (void) = fpcolMdlRd2Xy(model, cra, cdec, ra, dec, 
                                         mjd, x, y, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const FpilModelType *model) Contains the telescope 
                           model details  obtained by fpcolMdlTransAccess(3),
                           fpcolMdlReadModel(3) or fpcolMdlCreateModel(3).
                           It Can be assumed that the fpcolMdlCvtInit 
                           routine has been  invoked on this.
      (>) cra     (double) Apparent RA of field center.
      (>) cdec    (double) Apparent DEC of field center.
      (>) ra      (double) Apparent RA of source.
      (>) dec     (double) Apparent DEC of source.
      (>) mjd     (double) UTC time and date expressed as Modified Julian Date
      (<) x       (double *) X position of source on field plate (microns).
      (<) y       (double *) Y position of source on field plate (microns).
      (!) status  (StatusType *) modified status.
 
 * Include files: fpcolMdl.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
     20-Jun-2000 - TJF - Implement real code based on 2dF TdfRd2xy routine.
     04-Mar-2001 - TJF - Apply radial dis tortion using slaUnpcd after
                         projecting from tanget plane.
     03-Apr-2001 - JAB - OzPoz version.                    
     22-May-2001 - KS  - Added defensive calculation of ZD with refraction
                         disabled.
     11-Jul-2001 - TJF - When merging the above mod with OzPoz code, noticed
                         it was modiying the const parameter model.  Changed
                         to avoid this.  Some tidying up of error reporting
                         and error return points.
     26-Mar-2002 - TJF - Move application of radial distortion and focal length
                         to fpcolMdl___Tan2XY (so that we can use a new formula).
     26-Mar-2002 - TJF - Add more details when errors occur.
     27-Jun-2002 - TJF - Extract most of the code into fpcolMdlRd2Tan() and
                         call that routine.
 */        
extern void fpcolMdlRd2Xy (
    const FpilModelType *model,
    double cra,
    double cdec,
    double ra,
    double dec,
    double mjd,
    double *x,
    double *y,
    StatusType *status)
{
    const fpcolMdlModelPnt MyModel = (const fpcolMdlModelPnt)model;

    double xi,eta;            /* tangent plane coordinates  */


    if (*status != STATUS__OK) return;

/*
 *  Convert to tangent plane coorindates.
 */
    fpcolMdlRd2Tan(model, cra, cdec, ra, dec, mjd, &xi, &eta, status);

/*
 *  Apply Distortion convert to x/y.
 */
    fpcolMdl___Tan2XY(MyModel, xi, eta, x, y, status);

}

/*
 *+			f p c o l M d l X y 2 P o s 

 * Function name:
      fpcolMdlXy2Pos

 * Function:
      Convert OzPoz field plate coordinates to positioner coordinates.

 * Description:
      Convert the x/y coordinates of a star on the field plate, as output
      by fpcolMdlRd2Xy to the coordinates required for input to the OzPoz
      robot positioner system.   

      This scheme of two x/y coorindate systems allows the robot to
      use a coordinate system which makes sense mechanically, whilst
      the values output by fpcolMdlRd2Xy make more sense from an
      astromical point of view.  This scheme also takes account of
      non-linearities etc. in the positioner system).  

      This routine is called by FpilModelXy2Ps() and must meet
      the prototype given by the type defintion FpilModelXy2PosRoutineType.

 * Language:
      C

 * Call:
      (void) = fpcolMdlXy2Pos(model, x, y, xp, yp, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const FpilModelType *model) Contains the telescope 
                           model details  obtained by fpcolMdlTransAccess(3),
                           fpcolMdlReadModel(3) or fpcolMdlCreateModel(3).
                           It Can be assumed that the fpcolMdlCvtInit routine 
                           has been invoked on this.
      (>) x       (double) Field plate x coordinate of object (microns).
      (>) y       (double) Field plate y coordinate of object (microns).
      (>) *xp     (double) Positioner X coordinate of object (microns).
      (>) *yp     (double) Positioner Y coordinate of object (microns).
      (!) status  (StatusType *) modified status.
                        
 * Include files: fpcolMdl.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version, based on 2dF equivalent.
     03-Apr-2001 - JAB - OzPoz version.
     25-Aug-2002 - TJF - The Normalize() call is renamed to NormalizeLinear.  The
                         scale is set to 1 - it is not normalized with the linear
                         model any more.
 */
extern void fpcolMdlXy2Pos (
        const FpilModelType *model,
        double x,
        double y,
        double *xp,
        double *yp,
        StatusType *status)
{
    double a,b,c,d,e,f;
    const fpcolMdlModelPnt MyModel = (const fpcolMdlModelPnt)model;

    if (*status != STATUS__OK) return;

/*
 *  Extract the transformation coefficent details.
 */  
    a = MyModel->linpars.coeffs[0];
    b = MyModel->linpars.coeffs[1];
    c = MyModel->linpars.coeffs[2];
    d = MyModel->linpars.coeffs[3];
    e = MyModel->linpars.coeffs[4];
    f = MyModel->linpars.coeffs[5];

    fpcolMdlNormalizeLinear(MyModel->linpars.rotation,
                            1,
                            MyModel->linpars.nonperp,
                            &a, &b, &c, &d, &e, &f,
                            status);
  
/*
 *  Apply the transformation.
 */
    *xp = a + b*x + c*y;
    *yp = d + e*x + f*y;

}

/*
 *+			f p c o l M d l X y 2 R d 

 * Function name:
      fpcolMdlXy2Rd

 * Function:
      Determine apparent RA and Dec from OzPoz x and y.

 * Description:
      Determine the apparent RA an Dec of a source from the x,y coorindate
      in the field plate and the field center.

      This routine is called by FpilModelXy2Rd() and must meet
      the prototype given by the type defintion FpilModelXy2RdRoutineType.

 * Language:
      C

 * Call:
      (void) = fpcolMdlXy2Rd(model, cra, cdec, x, y, 
                                         mjd, ra, dec, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const FpilModelType *model) Contains the telescope 
                           model details  obtained by fpcolMdlTransAccess(3),
                           fpcolMdlReadModel(3) or fpcolMdlCreateModel(3).
                           It Can be assumed that the fpcolMdlCvtInit 
                           routine has been invoked on this.
      (>) cra     (double) Apparent RA of field center.
      (>) cdec    (double) Apparent DEC of field center.
      (>) x       (double) X position of source on field plate (microns).
      (>) y       (double) Y position of source on field plate (microns).
      (>) mjd     (double) UTC time and date expressed as Modified Julian Date
      (<) ra      (double *) Apparent RA of source.
      (<) dec     (double *) Apparent DEC of source.
      (!) status  (StatusType *) modified status.


 * Include files: fpcolMdl.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
     20-Jun-2000 - TJF - Implement real code based on 2dF TdfXy2rd routine.
     04-Mar-2001 - TJF - Remove radial distortion using slaUnpcd before
                         deprojecting from tanget plane.
     03-Apr-2001 - JAB - OzPoz version.                    
     26-Mar-2002 - TJF - Move removal of radial distortion and focal length
                         to fpcolMdl___XY2Tan (so that we can use a new formula).
 */
extern void fpcolMdlXy2Rd (
    const FpilModelType *model,
    double cra,
    double cdec,
    double x,
    double y,
    double mjd, 
    double *ra,
    double *dec,
    StatusType *status)
{
    const fpcolMdlModelPnt MyModel = (const fpcolMdlModelPnt)model;
    double mra,mdec;          /* Observed ra,dec of source   */
    double xi,eta;            /* tangent plane coordinates  */
    double oaz;               /* Observed Az */
    double ozd;               /* Observed Zd */
    double oha;               /* Observed HA */
    double ocra,ocdec ;       /* Observed field centre RA and Dec  */
    double pa;
    double xi2,eta2;

    if (*status != STATUS__OK) return;


/*  
 * Update Apparent to observed parameters  
 */
    slaAoppat(mjd,MyModel->xypars.cenAoprms);
    slaAoppat(mjd,MyModel->xypars.obsAoprms);
/*  
 * Observed position of field centre  
 */
    slaAopqk(cra,cdec,MyModel->xypars.cenAoprms,&oaz,&ozd,&oha,&ocdec,&ocra);

/*
 *  Convert from focal plane to tanget plane, removing distortion.
 */    
    fpcolMdl___XY2Tan(MyModel, x, y, &xi, &eta, status);

/*
 *  Rotate through -ypa
 */
    pa = -(MyModel->xypars.ypa);
    xi2 = xi * cos(pa) + eta * sin(pa);
    eta2 = - xi * sin(pa) + eta * cos(pa);

/*
 *  Flip sign of xi
 */
    xi2 = -xi2;    
    
/*  
 * Deproject from tangent plane to give observed ra, dec  
 */
    slaDtp2s(xi2,eta2,ocra,ocdec,&mra,&mdec);
/*  
 * Observed to Apparent  
 */
    slaOapqk("R",mra,mdec,MyModel->xypars.obsAoprms,ra,dec);



}

/*
 *+			f p c o l M d l P o s 2 X y 

 * Function name:
      fpcolMdlPos2Xy

 * Function:
      Convert positioner coorindates to field plate coordinates.  

 * Description:
      Convert the x,y coorindates of a star on the coorindate system 
      used by the OzPoz positioner task to the x,y coordinates on the
      field plate as used by fpcolMdlXy2Rd.

      This scheme of two x/y coorindate systems allows the robot to
      use a coordinate system which makes sense mechanically, whilst
      the values output by fpcolMdlRd2Xy make more sense from an
      astromical point of view.  This scheme also takes account of
      non-linearities etc. in the positioner system.  

      This routine is called by FpilModelPos2Xy() and must meet
      the prototype given by the type defintion FpilModelPos2XyRoutineType.


 * Language:
      C

 * Call:
      (void) = fpcolMdlPos2Xy(model,  xp, yp, x, y, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const FpilModelType *model) Contains the telescope 
                           model details  obtained by fpcolMdlTransAccess(3),
                           fpcolMdlReadModel(3) or fpcolMdlCreateModel(3).
      (>) xp     (double) Positioenr X coordinate of object (microns).
      (>) yp     (double) Positioenr Y coordinate of object (microns).
      (>) *x      (double) Field plate x coordinate of object (microns).
      (>) *y      (double) Field plate y coordinate of object (microns).
      (!) status  (StatusType *) modified status.

 * Include files: fpcolMdl.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version, based on 2dF equivalent.
     03-Apr-2001 - JAB - OzPoz version.
     25-Aug-2002 - TJF - The Normalize() call is renamed to NormalizeLinear.  The
                         scale is set to 1 - it is not normalized with the linear
                         model any more.
 */
void fpcolMdlPos2Xy (
    const FpilModelType *model,
    double xp,
    double yp,
    double *x,
    double *y,
    StatusType *status)
{
    const fpcolMdlModelPnt MyModel = (const fpcolMdlModelPnt)model;
    double a,b,c,d,e,f;

    if (*status != STATUS__OK) return;

/*
 *  Extract inversion transformation coefficents from the model details.
 */
    a = MyModel->linpars.invCoeffs[0];
    b = MyModel->linpars.invCoeffs[1];
    c = MyModel->linpars.invCoeffs[2];
    d = MyModel->linpars.invCoeffs[3];
    e = MyModel->linpars.invCoeffs[4];
    f = MyModel->linpars.invCoeffs[5];

    
    if (MyModel->linpars.scale == 0)
      {
        *status = FPCOL_MDL__ILLSCALE;
        return;
      }
    
    fpcolMdlNormalizeLinear(-1.0*MyModel->linpars.rotation,
                            1.0,
                            -1.0*MyModel->linpars.nonperp,
                            &a, &b, &c, &d, &e, &f,
                            status);

/*
 *  Apply the transformation.
 */
    *x = a + b*xp + c*yp;
    *y = d + e*xp + f*yp;


}

/*
 *+			f p c o l M d l F r e e 

 * Function name:
      fpcolMdlFree

 * Function:
      Free the OzPoz positioner transformation model structure.

 * Description:
      Free and tidy up after a call to fpcolMdlTransAccess(3), fpcolMdlReadModel(3)
      or fpcolMdlCreateModel(3).


 * Language:
      C

 * Call:
      (void) = fpcolMdl(model,status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (FpilModelType *model) Contains the telescope model details
                           obtained by fpcolMdlTransAccess(3), fpcolMdlReadModel(3)
                           or fpcolMdlCreateModel(3).  The 
                           fpcolMdlCvtInit routine
                           may or may not have been previously invoked
                           on this model.
      (!) status  (StatusType *) modified status.
                        

 * Include files: fpcolMdl.h

 *-

 * History:
     09-Dec-1999 - TJF - Original version
 */
extern void fpcolMdlFree (
    const FpilModelType *model,
    StatusType *status)

{
/*    fpcolMdlModelPnt MyModel = (fpcolMdlModelPnt)model;*/

    if (*status != STATUS__OK) return;
    free((void *)model);
}




/*^L
 *+                 f p c o l M d l Q u i c k A p p t o O b s

 * Function name:
      fpcolMdlQuickAppToObs

 * Function:
      Perform a quick apparent to observered place conversion.

 * Description:
      Perform a quick apparent to observered place conversion.

 * Language:

      C

 * Call:
        (void) = fpcolMdlQuickAppToObs(model, ra, dec,
                                        aob, zob, hob, dob, rob,
                                        status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const FpilModelType *model) Contains the telescope model
                       details obtained by FpilModelAccess(3) and on which
                           FpilModelCvtInit(3) has been invoked.
      (>) ra      (double) Apparent center RA in radians.
      (>) dec     (double) Apparent Center Declination in radians.
      (<) aob     (double *) Observed azimuth (radians, N=0, E = 90degrees)
      (<) zob     (double *) Observed zenith distance (radians)
      (<) hob     (double *) Observed Hour Angle (radians)
      (<) dob     (double *) Observed Declination (radians)
      (<) rob     (double *) Observed Right Ascension (radians)
      (!) status  (StatusType *) modified status.


 * Include files: fpcolMdl.h
 
 *-

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 * History:
     13-Mar-2000 - TJF - Original DUMMY version
     03-Apr-2001 - JAB - Just call slaAopqk.
 */
extern void fpcolMdlQuickAppToObs  (
    const FpilModelType *model,
    double ra,
    double dec,
    double *aob,
    double *zob,
    double *hob,
    double *dob,
    double *rob,
    StatusType *status)
{
    const fpcolMdlModelPnt MyModel = (const fpcolMdlModelPnt)model;
    if (*status != STATUS__OK) return;


    slaAopqk(ra,dec,MyModel->xypars.obsAoprms,aob,zob,hob,dob,rob);
}


/*^L
 *+                 f p c o l M d l N o r m a l i z e M  o d e l

 * Function name:
      fpcolMdlNormalizeModel

 * Function:
      Normalize our model.

 * Description:
      Apply rotation effects to the linear model parameters and scale
      effects the distortion parameters.

 * Language:
      C

 * Call:
        (void) = fpcolMdlNormalize(model,  status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (FpilModelType *model) Contains the telescope model
                       details obtained by FpilModelAccess(3) and on which
                           FpilModelCvtInit(3) has been invoked.
      (!) status  (StatusType *) modified status.


 * Include files: fpcolMdl.h
 
 *-

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 * History:
     16-Apr-2002 - TJF - Original version
     25-Aug-2002 - TJF - Change to normalizing scale through the distoration
                         parameters rather then the linear model parameters.
 */
extern void fpcolMdlNormalizeModel  (
     FpilModelType *model,
     StatusType *status)
{
  fpcolMdlModelPnt MyModel = (const fpcolMdlModelPnt)model;
  if (*status != STATUS__OK) return;
  
  /*
   * Normalize any rotation through the linear model.
   */
  fpcolMdlNormalizeLinear(MyModel->linpars.rotation,
                          1,
                          MyModel->linpars.nonperp,
                          &MyModel->linpars.coeffs[0],
                          &MyModel->linpars.coeffs[1],
                          &MyModel->linpars.coeffs[2],
                          &MyModel->linpars.coeffs[3],
                          &MyModel->linpars.coeffs[4],
                          &MyModel->linpars.coeffs[5],
                          status);

  /*
   * Normalize any scale through the distortion model.
   */
  fpcolMdlNormalizeDist(MyModel->linpars.scale,
                        &MyModel->distpars.dist[0],
                        &MyModel->distpars.dist[1],
                        &MyModel->distpars.dist[2],
                        &MyModel->distpars.dist[3],
                        &MyModel->distpars.dist[4],
                        &MyModel->distpars.dist[5],
                        status);
  /*
   * If sucessfull - put these back to their default values.
   */
  if (*status == STATUS__OK)
    {
      MyModel->linpars.rotation = 0;
      MyModel->linpars.scale =    1;
      MyModel->linpars.nonperp =  0;
    }
}


/*^L
 *+                 f p c o l M d l N o r m a l i z e L i n e a r

 * Function name:
      fpcolMdlNormalizeLinear.

 * Function:
      Normalize a linear model.

 * Description:
      Apply rotation and scale effects to linear model parameters.

 * Language:
      C

 * Call:
        (void) = fpcolMdlNormalizeLinear(rotation, scale, nonperp,
                                         a, b, c, d, e, f
                                         status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) rotation (double) A rotation in degrees.
      (>) scale    (double) A scale
      (>) nonperp  (double)  Non-perpendicularity in degrees
      (!) a        (double) Linear model parameter a
      (!) b        (double) Linear model parameter b
      (!) c        (double) Linear model parameter c
      (!) d        (double) Linear model parameter d
      (!) e        (double) Linear model parameter e
      (!) f        (double) Linear model parameter f
      (!) status  (StatusType *) modified status.


 * Include files: fpcolMdl.h
 
 *-

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 * History:
     16-Apr-2002 - TJF - Original version
     19-Aug-2002 - TJF - Scale calculation was wrong - fixed.
     25-Aug-2002 - TJF - Rename to NormalizeLinear.
     27-Aug-2002 - TJF - Add Non-perpendicularity.   
 */
extern void fpcolMdlNormalizeLinear  (
    double rotation,
    double scale,
    double nonperp,
    double *a,
    double *b,
    double *c,
    double *d,
    double *e,
    double *f,
    StatusType *status)
{
    if (*status != STATUS__OK) return;
    
    if (scale == 0)
      { 
        *status = FPCOL_MDL__ILLSCALE;
        return;
      }
    else if (scale != 1)
      {
        *b *= scale;
        *c *= scale;
        *e *= scale;
        *f *= scale;
      }
    

    if (rotation != 0)
      {
        double cosine;
        double sine;
        double b2, c2, e2, f2;
        double an, bn, cn, dn, en, fn;

        cosine = cos(rotation/180.0*M_PI);
        sine   = sin(rotation/180.0*M_PI);
            
        /*
         * Short hand for rotation coefficents
         */
        b2 = cosine;
        c2 = sine;
        e2 = -1 * sine;
        f2 = cosine;

        /*
         * The original linear model is
         *  x1 = a + b.x0 + c.y0
         *  y1 = d + e.x0 + f.y0
         *
         * We want to apply this.
         *
         * x2 = b2.x1 + c2.y1
         * y2 = e2.x1 + f2.y1
         *
         * Which is equivalent to this.
         *
         *  x2 = b2.(a + b.x0 + c.y0) + c2.(d + e.x0 + f.y0)
         *  y2 = e2.(a + b.x0 + c.y0) + f2.(d + e.x0 + f.y0)
         *
         * And working out the new model, we get.
         *  x2 = (b2.a + c2.d) + (b2.b + c2.e) x0 + (b2.c + c2.f) y0
         *  y2 = (e2.a + f2.d) + (e2.b + f2.e) x0 + (e2.c + f2.f) y0
         *
         * Hence we can see our new parameters.
         */
        an = b2*(*a) + c2*(*d);
        bn = b2*(*b) + c2*(*e);
        cn = b2*(*c) + c2*(*f);
        dn = e2*(*a) + f2*(*d);
        en = e2*(*b) + f2*(*e);
        fn = e2*(*c) + f2*(*f);

        *a = an;
        *b = bn;
        *c = cn;
        *d = dn;
        *e = en;
        *f = fn;
      }
    if (nonperp != 0)
      {
        double cosine;
        double sine;
        double b2, c2, e2, f2;
        double an, bn, cn, dn, en, fn;

        double radians = nonperp/180*M_PI;

        cosine = cos(radians/2.0);
        sine   = sin(radians/2.0);
            
        /*
         *
         *
         * A non-perpendicularity transformation is given by
         *
         *   x' = cos(perp/2)*x + sin(perp/2)*y
         *   y' = sin(perp/2)*x + cos(perp/2)*y
         *
         * Short hand for coefficents.
         */
        b2 = cosine;
        c2 = sine;
        e2 = sine;
        f2 = cosine;

        /*
         * The original linear model is
         *  x1 = a + b.x0 + c.y0
         *  y1 = d + e.x0 + f.y0
         *
         * We want to apply this.
         *
         * x2 = b2.x1 + c2.y1
         * y2 = e2.x1 + f2.y1
         *
         * Which is equivalent to this.
         *
         *  x2 = b2.(a + b.x0 + c.y0) + c2.(d + e.x0 + f.y0)
         *  y2 = e2.(a + b.x0 + c.y0) + f2.(d + e.x0 + f.y0)
         *
         * And working out the new model, we get.
         *  x2 = (b2.a + c2.d) + (b2.b + c2.e) x0 + (b2.c + c2.f) y0
         *  y2 = (e2.a + f2.d) + (e2.b + f2.e) x0 + (e2.c + f2.f) y0
         *
         * Hence we can see our new parameters.
         */
        an = b2*(*a) + c2*(*d);
        bn = b2*(*b) + c2*(*e);
        cn = b2*(*c) + c2*(*f);
        dn = e2*(*a) + f2*(*d);
        en = e2*(*b) + f2*(*e);
        fn = e2*(*c) + f2*(*f);

        *a = an;
        *b = bn;
        *c = cn;
        *d = dn;
        *e = en;
        *f = fn;
      }


}


/*^L
 *+                 f p c o l M d l N o r m a l i z e D i s t

 * Function name:
      fpcolMdlNormalizeDist.

 * Function:
      Normalize the OzPoz distortion model.

 * Description:
      Apply a scale effect to OzPoz distortion model parameters.

 * Language:
      C

 * Call:
        (void) = fpcolMdlNormalizeDist(scale, d0, d1, d2, d3, d4, d5,
                                        status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) scale    (double) A scale
      (!) d0       (double) Distortion model parameter 0
      (!) d1       (double) Distortion model parameter 1
      (!) d2       (double) Distortion model parameter 2
      (!) d3       (double) Distortion model parameter 3
      (!) d4       (double) Distortion model parameter 4
      (!) d5       (double) Distortion model parameter 5
      (!) status  (StatusType *) modified status.


 * Include files: fpcolMdl.h
 
 *-

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 * History:
     25-Aug-2002 - TJF - Original version

 */
extern void fpcolMdlNormalizeDist  (
    double scale,
    double *d0,
    double *d1,
    double *d2,
    double *d3,
    double *d4,
    double *d5,
    StatusType *status)
{
    if (*status != STATUS__OK) return;
    
    if (scale == 0)
      { 
        *status = FPCOL_MDL__ILLSCALE;
        return;
      }
    else if (scale != 1)
      {
      *d0 *= scale;
      *d1 *= scale;
      *d2 *= scale;
      *d3 *= scale;
      *d4 *= scale;
      *d5 *= scale;
      }
    
}

static void fpcolMdl___GetFocalLength(const fpcolMdlModelPnt model,
                          double *focalLength,
                          double *pincushionCoeff,
                          StatusType *status)
{
    double d0;
    double d1;

    if (*status != STATUS__OK) return;

    d0 = model->distpars.dist[0];
    d1 = model->distpars.dist[1];
 /*
  *  Using only first two coorindates of the distrotion model, we
  *  have
  *
  *  R = d0 * (180/pi) * r + d1 * (180/pi)^3 * r^3
  *
  *  Here = d0 * (180/pi)  = focual length in mm.
  *
  *    Ra = R/telescope focal length in mm
  *
  *  Ra = r + d1 * (180/pi)^3 / (d0 * (180/pi)) * r^3
  *
  *  And this is of the same format at the pincushion effect formular
  *
  *  And thepincushion coeffent is
  *
  *           d1 * (180/pi)^3 / (d0 * 180/pi)
  *
  *  Remember that we want the focal length in microns rather then mm.
  */
    *focalLength      = d0 * 180 / M_PI * model->linpars.scale;
    *pincushionCoeff  = d1 * pow((180.0/M_PI), 3) / *focalLength;

    *focalLength      *= 1000;  /* convert to microns */
}

/*^L
 *+                 f p c o l M d l _ _ _ Tan2XY

 * Function name:
      fpcolMdl__Tan2XY

 * Function:
      Convert tangent plane coordinates to x/y taking account of distortion.

 * Description:
      Convert tangent plane coordindates to x/y taking account of distortion.  

 * Language:

      C

 * Call:
      (void) = fpcolMdl___Tan2XY(model, xi, eta, x, y  status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const fpcolMdlModelPnt model) Contains the telescope model
                       details obtained by FpilModelAccess(3) and on which
                           FpilModelCvtInit(3) has been invoked.
      (>) xi      (const double)  Tanget plane coorindates.
      (>) eta     (const double)
      (<) x       (double *) X ordindate.
      (<) y       (double *) Y ordinates.
      (!) status  (StatusType *) modified status.


 * Include files: fpil.h
 
 *-

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 * History:
     26-Mar-2002 - TJF - Original Version.
     27-Mar-2002 - TJF - Revision to Will's numbers due to faulty numbers from Peter.
     12-Apr-2002 - TJF - If r comes out as zero, return after setting results
                         to zero to avoid divide by zero error.
     20-Jun-2002 - TJF - Was using wrong index into distortion parameters
                         for 6th term.
                         The formula degenerates outside the field of view,
                         so we must check the result against slaPcd. Also validate
                         x/y value against X_Y_MAX.
     20-Jun-2002 - TJF - Incorporate revised distortion formula from Matthew Colless.
                         Will's formula had an error as did the data he based it on. 
     24-Aug-2002 - TJF - Increase MAX_DEVIATION to 5mm - .5 mm was too large if we
                         started playing with the plate.  Add warning for VLT and
                         Calibration software if deviation code triggered.  Not
                         need for Configure (DRAMA_BUILD) as that is expected to
                         trigger it.
 */
static void fpcolMdl___Tan2XY  (
    const fpcolMdlModelPnt model,
    const double xi,
    const double eta,
    double *x,
    double *y,
    StatusType *status)
{
    double pcdX = 0;  
    double pcdY = 0;
    double xi1 = 0; 
    double eta1;

    double R;         /* Radius of object */
    double r =  sqrt(xi*xi + eta*eta);   /* Tangent plain radius angle (radians) */

    double th = r / M_PI * 180.0;        /* Tangent plain radius angle (degrees) */
    double l =  model->xypars.obsLambda * 1000.0; /* Observation wavelength (nanometers */

    double th_p3 = pow(th,3);           /* Powers of th */
    double th_p5 = pow(th,5);

    double deviation;                   /* Deviation from PCD position */

    double d0,d1,d2,d3,d4,d5;           /* Distortion model parameters to be
                                           normalized */
    double focalLength;
    double pincushionCoeff;


    if (*status != STATUS__OK) return;
    
    if (model->linpars.scale == 0)
      { 
        *status = FPCOL_MDL__ILLSCALE;
        return;
      }

/*
 *  Avoid divide by zero errors.
 */

    if (r == 0)
      {
        *x = 0;
        *y = 0;
        return;
      }

    /*
     * Fetch the distortion parameters.
     */
    d0 = model->distpars.dist[0];
    d1 = model->distpars.dist[1];
    d2 = model->distpars.dist[2];
    d3 = model->distpars.dist[3];
    d4 = model->distpars.dist[4];
    d5 = 0;   /* Currently not used */
    /*
     * Normalize any scale into the distortion parameters.
     */
    fpcolMdlNormalizeDist(model->linpars.scale, &d0, &d1, &d2, &d3, &d4, &d5, status);

/*
 *  This formula was determined by Matthew Colless on the 20th Jun 2002, updated
 *  on 22nd Jun 2002 for the robot having been moved back by 2mm. 
 *
 *    Best fit given by
 *
 *       R = 2079.5 th + 304.24 th^3 - 629.35 th^5 - 409.51 (th/l) + 387050(th/l)^2
 *
 *    This gives a max error of 12 microns and an RMS of 5 microns.
 *
 *
 *    This simpiler formula (pincushion effect, needed below) gives a 
 *    deviation from the above by < 60 microns.
 *
 *       R = 2079.3 th + 272.75 th^3
 *
 *       th is in degrees, l is in nm, R is in mm.
 *
 *  Note that xi and eta are in radians.
 * 
 */

    R = (d0   * th) +
        (d1   * th_p3) + 
        (d2   * th_p5) +
        (d3   * th/l)  +
        (d4   * th/l * th/l);

    R *= 1000.0;  /* Convert to micrometers */

/*
 *  Get x and y components.
 */
    *x = xi  * R / r;
    *y = eta * R / r;
/*
 *  We need to determine the pincushion coefficent for slaPcd() and
 *  the focal length determined by the distortion parameters.
 */
    fpcolMdl___GetFocalLength(model, &focalLength, &pincushionCoeff, status);

#define MAX_DEVIATION   5000  /* Maximum deviation of proper formula from
                                slaPcd formula within field of view                              */

/*
 * Apply the distortion using slaPcd..  Here we are just confirming
 * the new formula works ok and checking for degeneration.
 *
 * Apply Radial Distortion 
 */
    xi1  = xi;
    eta1 = eta;
    slaPcd(pincushionCoeff, &xi1, &eta1);
/*
 * Multiply by focal length to get x and y in microns.  
 */
    pcdX = xi1 *  focalLength;
    pcdY = eta1 * focalLength;

/*
 *  If the difference between the Radius worked out
 *  this way and worked out using the proper formula
 *  is more then MAX_DEVIATION, then it is possible
 *  the proper forumula is degenerating and we should
 *  just use the value worked out by slaPcd.  
 *
 *  It is likely that th > 0.25 degrees in this case,
 *  outside the ozpoz field of view.
 */
    deviation = fabs(sqrt(pcdX*pcdX+pcdY*pcdY) - R); 
    if (deviation > MAX_DEVIATION)
        {
#ifdef DRAMA_BUILD
        ;
#elif defined (CAL_BUILD)
        fprintf(stderr,"Warning - object position with model deviated from PCD position\n");
        fprintf(stderr,"        - Object deviation %6.1f, max deviation %d microns\n",
                deviation, MAX_DEVIATION);
        fprintf(stderr,"        - Using position genreated by slaPcd() instead of model\n");
#else
        ccsERROR error;
        errResetStack(&error);
        errAdd(&error,  MODULE_ID, fpcolERR_EXTRA_INFO, __FILE__, "%s", 
               "Warning - object position with model deviated from PCD position");
        errAdd(&error,  MODULE_ID, fpcolERR_EXTRA_INFO, __FILE__, "%s", 
               "        - Object deviation %6.1f, max deviation %d microns",
               deviation, MAX_DEVIATION);
       errAdd(&error,  MODULE_ID, fpcolERR_EXTRA_INFO, __FILE__, "%s", 
              "        - Using position genreated by slaPcd() instead of model");
#endif
        *x = pcdX;
        *y = pcdY;
        }
/*
 *  Ensure that we don't allow really high values.
 */
    if ((*x > X_Y_MAX)||(*y > X_Y_MAX))
        {
        *x = *y = X_Y_MAX;
        }

}

/*^L
 *+                 f p c o l M d l _ _ _ X Y 2 T a n

 * Function name:
      fpcolMdl__XY2Tan

 * Function:
      Convert x/y to tangent plane coordinates taking account of distortion.

 * Description:
      Convert x/y to tangent plane coordinates taking account of distortion.

 * Language:

      C

 * Call:
      (void) = fpcolMdl___XY2Tan(model, x, y xi, etx, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (const fpcolMdlModelPnt model) Contains the telescope model
                       details obtained by FpilModelAccess(3) and on which
                           FpilModelCvtInit(3) has been invoked.
      (>) x       (const double) X ordindate.
      (>) y       (const double) Y ordinates.
      (<) xi      (double *)  Tanget plane coorindates.
      (<) eta     (double *)
      (!) status  (StatusType *) modified status.


 * Include files: fpil.h
 
 *-

 * See Also: {references}

 * External functions used:

 * Support: Tony Farrell, AAO

 * History:
     26-Mar-2002 - TJF - Original Version.
 */
static void fpcolMdl___XY2Tan  (
    const fpcolMdlModelPnt model,
    double x,
    double y,
    double *xi,
    double *eta,
    StatusType *status)
{
    double focalLength;
    double pincushionCoeff;

    if (*status != STATUS__OK) return;

    if (model->linpars.scale == 0)
      { 
        *status = FPCOL_MDL__ILLSCALE;
        return;
      }
/*
 *  Determine the focal length and pincushion coefficent from the
 *  telescope model parameters.
 */
    fpcolMdl___GetFocalLength(model, &focalLength, &pincushionCoeff, status);
    if (*status != STATUS__OK) return;
/*
 * Convert from focal plane to tangent plane co-ordinates
 */
    *xi  = x/focalLength;
    *eta = y/focalLength;
/*
 * Remove Radial Distortion (approximate only)
 */
    slaUnpcd(pincushionCoeff, xi, eta);
}



/*
 *+			f p c o l M d l G e t M o d e l

 * Function name:
      fpcolMdlMicrons2Arcsec

 * Function:
      Convert microns on the late to arc-seconds on the sky.


 * Description:
      Uses the model to convert the specified number of microns
      to arc-seconds (most accurate near the plate center).

 * Language:
      C

 * Call:
      (void) = fpcolMdlMicrons2Arcsec(model, microns, arcsec status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) model   (FpilModelType *model) Contains the telescope model details
                           obtained by fpcolMdlTransAccess, fpcolMdlReadModel or 
                           fpcolMdlCreateModel.
      (>) microns (double) The microns to convert
      (<) acrsec  (double *) The equivalent arc-seconds at the plate centre
      (!) status  (StatusType *) modified status.
                           

 * Returned value:

 * Include files: fpcolMdl.h

 *-

 * History:
     18-Oct-2002 - TJF - Original version 

 */

#define SEC2RAD(_sec) ((_sec)/(180.0*60.0*60.0)*M_PI)

extern void fpcolMdlMicrons2Arcsec(
    const FpilModelType *model,
    const double microns,
    double * const arcsec,
    StatusType * const status)
{
  double x, y;
  double rx, ry;
  double arcsec2microns;
  const fpcolMdlModelPnt MyModel = (const fpcolMdlModelPnt)model;

  if (*status != STATUS__OK) return;

  /*
   * Convert 1 arc second to X and Y.
   */
  fpcolMdl___Tan2XY(MyModel, 0, SEC2RAD(1.0), &x, &y, status);
  /*
   * Now convert to robot coorindate system. 
   */
  fpcolMdlXy2Pos(model, x, y, &rx, &ry, status);

  /*
   * Now we can work out how many microns in an arcsecond (at least
   * clost to the plate centre)
   */
  arcsec2microns = sqrt(rx*rx + ry*ry);

  *arcsec =  microns/arcsec2microns;

}
    
/*
 * The following are only built for VLT software release.
 */
#ifndef DRAMA_BUILD
#ifndef CAL_BUILD


/*
 * Function which warps a conversion from mean positions (with proper motions)
 * to robot X/Y position in microns.
 */
extern ccsCOMPL_STAT fpcolMdlCvtMeanToXY(
    IN  const FpilModelType * const model,
    IN  const fpFIBRE_SKY_POS * const skyCoord,
    IN  const struct fpcolPM_PARS * const pmPars,
    IN  const double cenRA,
    IN  const double cenDec,
    IN  const double mjd,
    IN  const double gmap[21],
    OUT fpPOS  * const pos,
    OUT double * const appRA,
    OUT double * const appDec,
    OUT ccsERROR * const error)
{
    double plateX, plateY;  /* plate positions in microns */
    double robotX, robotY;  /* robot XY positions in microns */
    StatusType status = STATUS__OK;
    /*
     *  Get apparent RA and Dec values, accounting for proper motion.
     */
    slaMapqk (tcsHms2rad(skyCoord->RA), 
              tcsDms2rad(skyCoord->Dec), 
              pmPars->ra,
              pmPars->dec,
              pmPars->parallax,
              pmPars->radialVel,
              (double *)gmap,  /* Case is becase slaMapqk is not const correct */
              appRA, 
              appDec);
    /*
     * Translate sky-to-field plate XY [microns]
     */
    fpcolMdlRd2Xy (model, cenRA, cenDec, *appRA, *appDec, mjd, 
                   &plateX, &plateY, &status);

    if (status != STATUS__OK)
        {
        return fpcolDRAMAErrorToVLTError(status, __FILE__, error);
        }
    /*
     * Translate field plate-to-robot XY [microns]
     */
    fpcolMdlXy2Pos (model, plateX, plateY, &robotX, &robotY, &status);
    if (status != STATUS__OK) 
        {
        return fpcolDRAMAErrorToVLTError(status, __FILE__, error);
        }
			 
    /*
     * Create the point variable.
     */
    fpptCreatePoint(robotX, robotY, pos);
    /*
     * Debugging.
     */
    printf (
        "R=%9.6f D=%9.6f aR=%9.6f aD=%9.6f x=%6.0f y=%6.0f",
        tcsHms2rad(skyCoord->RA), tcsDms2rad(skyCoord->Dec), 
        *appRA, *appDec, robotX, robotY);

    return SUCCESS;
}
    
    
    
/*
 * Function which warps a conversion from mean positions (with proper motions)
 * to robot R/Theta positions in microns and radians.
 */
extern ccsCOMPL_STAT fpcolMdlCvtMeanToRT(
    IN  const FpilModelType   * const model,
    IN  const fpFIBRE_SKY_POS * const skyCoord,
    IN  const struct fpcolPM_PARS * const pmPars,
    IN  const double cenRA,
    IN  const double cenDec,
    IN  const double mjd,
    IN  const double gmap[21],
    OUT fpFIBRE_POS * const plateCoord,
    OUT double * const appRA,
    OUT double * const appDec,
    OUT ccsERROR * const error)
{  
    fpPOS pos;
    /*
     * Convert to X/Y.
     */
    if (fpcolMdlCvtMeanToXY(model, skyCoord, pmPars, cenRA, cenDec, 
                            mjd, gmap, &pos, appRA, appDec, error) != SUCCESS)
        return FAILURE;
    /*
     * Translate robot XY[microns][microns] - to robot RTheta [microns][rad]
     */
    fpptXY2RT(pos, &pos);
    plateCoord->r     = pos.x;
    plateCoord->theta = pos.y;
    /*
     * Debugging.
     */
    printf (" r=%6.0f t=%8.6f\n", plateCoord->r, plateCoord->theta);

    return SUCCESS;
    
}

#endif
#endif
