/*******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: fpcolFpil.c 284998 2016-06-22 16:24:40Z pbaksai $"
* 
*
* who       when      what
* --------  --------  ----------------------------------------------
* tjf       09/06/00  created
*  ks       03/08/00  started changing the model where fibres are guide fibres
*                     or are associated with one of a number of spectrographs
*                     to one where there is just a simple set of different 
*                     fibre types. Main effect is that FpilConSpecInfo() 
*                     returns one more for NumberSpec (which really ought to
*                     be called NumberFibreTypes).
*  ks       21/08/00  added defintion of PATH_MAX if undefined. Added some
*                     casts to make address calculations strictly correct.
*  ks       18/08/00  added use of FILE_SEPARATOR to cater for systems where
*                     '/' is not the directory/folder delimiter character.
* tjf       31/10/00  Modfiy for picking up fibre counts etc. from fpDefines.h
*  ks       16/01/01  Merged above sets of changes made by KS and TJF. Added
*                     support for the IFU SKY fibre type.
*  ks       01/02/01  Now sets 'park may collide' flag true.
*  ks       28/02/01  Now distinguished between test fibres and unknown fibres,
*                     and introduces an explicit 'none' type.
*  ks       23/04/01  Modified to support the emerging 'proper' OzPoz model.
*  ks       29/05/01  Allows for blank override and fallback paths, and
*                     allows them to end in separator characters.
*  ks       19/08/01  Modified to compile cleanly under gcc -Wall -ansi.
*  ks       23/08/01  Modified to allow for changed model routine names.
*  tjf      27/08/01  Merged KS's changes from 16/01/01 into main CMM source.
*  tjf      27/08/01  Changed INST_STATUS_FILE from fpInsConfig.sds to 
*                     fpInsStatus.sds.
*                     Treat FP as having only one plate and merge the
*                     disabled lists of plate 1 and 2 into that plate.
*  ks       28/08/01  Added support for UVES_7 and UVES_8 fibre types.
*  ks       29/08/01  Now copies UPDATE_TIME into the field0 structure.
* tjf       30/10/01  The sole telescope model parameter should be zero -
*                     representing the position before any ARGUS rotation 
*                     is applied.
*  ks       30/07/02  Modified the setting of UVES_7, reinstating a change 
*                     that had been lost in the previous modification.
* rschmutz  23/11/02  fpInsStatus.sds -> ../config/fpInsStatus.sds
* rschmutz  03/01/03  CvtCons: ignore plate 1 inUse flag for Argus_SKY fibres.
* rschmutz  28/01/03  fpInsStatus -> fpossInsStatus.
*/

/************************************************************************
*   NAME
*       fpcolFpil - AAO Configure interface to OzPoz.
*
*   SYNOPSIS
*       #include "fpcolFpil.h"
*
*
*   DESCRIPTION
*      This library interfaces the OzPoz collision detection and Telescope
*      to the AAO Configure program.  It is never compiled in the VLT
*      environment but instead on the AAO DRAMA enviroment.  It makes use
*      of the Base level OzPoz collision detection module (fpcolBase) and
*      the Base level OzPoz telescope model software (?).  
*
*      The interface provided is that required by the AAO Fibre Positioner
*      Instrument Library (FPIL) full interface mode.
*
***
*
*   ENVIRONMENT
*       configure - AAO configure program.
*
*   CAUTIONS 
*
*   EXAMPLES
*
*   SEE ALSO
*       fpcol module, AAO FPIL Library.
*   BUGS   
*
*------------------------------------------------------------------------
*/


/*
 *  Include files
 */
#define __EXTENSIONS__   /* To get PATH_MAX on solaris */
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include <assert.h>

#include "drama.h"
#include "sds.h"
#include "Ers.h"
#include "fpDefines.h"
#include "fpcolBase.h"
#include "fpcolFpil.h"
#include "fpcolMdl.h"
#include "fpcolInstDescr.h"

/* 
 * Constant definitions
 */
#define INST_STATUS_FILE "../config/fpossInsStatus.sds"     /* Instrument status file */

/*
 *  These are the fibre types used by configure.
 */
#define FIB_OBJECT 1
#define FIB_GUIDE  0

/*
 *  We use DEBUG to make printf debugging lines stand out.
 */
 
#ifdef DEBUG
#undef DEBUG
#endif
#define DEBUG printf

/*  If we still don't have a definition for PATH_MAX, we choose the usual
 *  UNIX value.
 */

#ifndef PATH_MAX
#define PATH_MAX 1024
#endif

/*
 * Macro function definitions
 */

/*
 * type definitions
 */

/*
 * Global variables and functions imported (but not declared in include files)
 */

/*
 * Global variables exported (definitions and initialization of globals
 * declared in the associated include file)
 */

/*
 * Variables private to this module (static variables)
 */

/*  We need to use the correct character in file names to delimit directory
 *  names. UNIX uses '/'. Macs use ':', Windows uses '\'. We probably need
 *  to extend this for Windows, but many implementations understand the
 *  UNIX syntax as well as the native one, so we may get away with this.
 */
 
#ifdef macintosh
#define FILE_SEPARATOR ':'
#else
#define FILE_SEPARATOR '/'
#endif

/*
 *  This is the list of fibre types supported by FLAMES. Note that
 *  for historical reasons, these are assumed (inaccurately in the case of
 *  FLAMES) to indicate different spectrograph types. Note that the
 *  FIBRE_TYPE_xxx index numbers defined below are used to index into this
 *  array. (FIBRE_TYPE_UNKN and FIBRE_TYPE_NONE are not used as index values.)
 */

static const char * const FibreTypeNames[] = {
    "GUIDE",    /* Guide fibres - FACBs (nothing to do with VLT guide stars) */
    "IFU",      /* Giraffe IFU */
    "IFU_SKY",  /* Giraffe IFU Sky */
    "MEDUSA",   /* Giraffe Medusa */
    "UVES",     /* UVES (lower 6 fibres) */
    "ARGUS",    /* Argus Sky   */
    "TEST",     /* Test fibres */
    "UVES_7",   /* The 7th UVES fibre */
    "UVES_8"};  /* The 8th UVES fibre */   
    
/*
 *  Fibre type numbers, the index into the above array. Note that these
 *  numbers are not the same as the equivalent fpcolINST_FIBRE_TYPE_XXXX
 *  codes, and indeed, there isn't one for each fpcolINST_FIBRE_TYPE_XXXX
 *  code, since these list the types that the interface needs to know
 *  about, and there are others, such as fpcolINST_FIBRE_TYPE_NONE that
 *  we don't want to show as a type - in case someone tries to allocate it!
 */
 
#define FIBRE_TYPE_GUIDE   0
#define FIBRE_TYPE_IFU     1
#define FIBRE_TYPE_IFU_SKY 2
#define FIBRE_TYPE_MEDUSA  3
#define FIBRE_TYPE_UVES    4
#define FIBRE_TYPE_ARGUS   5
#define FIBRE_TYPE_TEST    6

/*  We also distinguish between the top two UVES fibres and the lower 6
 */
 
#define FIBRE_TYPE_UVES_7  7
#define FIBRE_TYPE_UVES_8  8

/*  We also define a FIBRE_TYPE_NONE for fibres that don't exist, and a
 *  FIBRE_TYPE_UNKN for fibre types we don't recognise.
 */
 
#define FIBRE_TYPE_NONE    9
#define FIBRE_TYPE_UNKN   10

/* The fpDB_FIBRE_TYPE_UVES_7 and fpDB_FIBRE_TYPE_UVES_8 codes are 
 * internal to this file, so are defined here. The values must not overlap
 * the other fpDB_FIBRE_TYPE codes, but otherwise are arbitrary. (UVES is
 * type 8, so these values are supposed to imply types 8.7 and 8.8)
 */

#define fpDB_FIBRE_TYPE_UVES_7  87
#define fpDB_FIBRE_TYPE_UVES_8  88

/*
 * Telescope model parameter names (at present the ARGUS angle is the only one is used by the OzPoz
 * astrometric model - if supplied as zero it should be assumed that ARGUS is in the
 * position corresponding to the normal plate configuration position.
 */
static const char * const ParamNames[] = {
    "ARGUS_POS"
};

/*
 * Telescope model parameter descriptions.
 */
static const char * const ParamDescrs[] = {
    "Position angle  (ARGUS angle) (radians)"
};

#define ARGUS_POS 0

/*
 * Telescope parameter default values.
 */
static const double ParamDefaults[] = {
    ARGUS_POS
};

/*
 * Internal Function name:
      CvtPlateCons
  
 * Description:
      Modify the OzPoz position constants structure for one plate
      requirements of configure.

 * History:
      19-Jun-2000 - TJF - Original version
      21-Aug-2000 - KS  - Added casts in address calculations.
      29-Aug-2001 - KS  - Now keeps UPDATE_TIME in structure.
  
 */
static void CvtPlateCons(
    SdsIdType id,
    StatusType *status)
{
    unsigned index;
    static int dbg = 0;
    if (*status != STATUS__OK) return;
/*
 *  Each OzPoz array representing fibres has one more element than required,
 *  such that the arrays are 1 based instead of zero based.  We need to
 *  make them one based.
 *
 *  We do this for the elements xPiv, yPiv, xPark, yPark, tPark, inUse and
 *  type.  We delete any other item. Note that 'type' is used only within
 *  the Fpil routines, so the fact that OzPoz uses a different convention
 *  for its type to that used by 2dF doesn't matter.
 */
    if (dbg)
        SdsList(id, status);

    ErsPush();
    index = 1;
    while (*status != SDS__NOITEM)
    {
        SdsIdType itemID;
        char name[SDS_C_NAMELEN];
        SdsCodeType code;
        unsigned long dims[7];
        long ndims;
/*
 *      Index to next item in the structure.
 */
        SdsIndex(id, index, &itemID, status);
        if (*status == SDS__NOITEM) break;  /* No more items */

/*
 *      Get details on this item.
 */
        SdsInfo(itemID, name, &code, &ndims, dims, status);

/*
 *      Check for items we are interested in.
 */
        if ((strcmp(name, "tPark") == 0)||
            (strcmp(name, "xPiv") == 0)||
            (strcmp(name, "yPiv") == 0)||
            (strcmp(name, "xPark") == 0)||
            (strcmp(name, "yPark") == 0)||
            (strcmp(name, "type") == 0)||
            (strcmp(name, "inUse") == 0))
        {
/*
 *          This item is of interest.
 */
            void *data;
            void *start = (void*) 0;
            unsigned long actlen;
            unsigned long bytes = 0;
/*
 *          We are farly trusting here that the item is of the format
 *          we want, dims = number of fibres + 1 and ndims = 1;  We just
 *          use  assertions to check this
 */
            assert(ndims == 1);
            assert(dims[0] == (fpNUM_MAX_FIBRES+1));
/*
 *          Point to the item.
 */
            SdsPointer(itemID, &data, &actlen, status);
/*
 *          Point to the start of the data itself, it should be
 *          one of a particular type of data and we increment data
 *          by the relevant size.  We also work out the number of
 *          bytes we must copy, but decrementing the actual length.
 */
            if (code == SDS_DOUBLE)
            {
                start = (char*)data + sizeof(double);
                bytes = actlen - sizeof(double);
            }
            else if ((code == SDS_UINT)||(code == SDS_INT))
            {
                start = (char*)data + sizeof(INT32);
                bytes = actlen - sizeof(INT32);
            }
            else if ((code == SDS_USHORT)||(code == SDS_SHORT))
            {
                start = (char*)data + sizeof(short);
                bytes = actlen - sizeof(short);
            }
            else
                /* This should never happen */
                assert(1);

            assert(bytes > 0);
/*
 *          Copy the data such that the array is zero based instead
 *          of one based.  start is the start of the real data.
 * 
 *          Note, must use memmove, not memcpy, to ensure the
 *          overlapping is handled correctly.
 */           
            memmove(data, start, bytes);
/*
 *          Resize the SDS structure.
 */
            dims[0] -= 1;
            SdsResize(itemID, 1, dims, status);

            ++index;
        }
        else
        {
            /*
             * This item is of no interest, save some memory by
             * deleting it.  note, we don't increment index in this
             * case, we just try the same index again. The exception is
             * UPDATE_TIME, which we leave alone - we don't treat it as
             * as item of interest, since we don't have to do all the
             * array manipulations in the previous section.
             */
            if (!strcmp(name,"UPDATE_TIME")) {
               ++index;
            } else {
               SdsDelete(itemID, status);
            }
        }
            
    }
    if (*status == SDS__NOITEM)
        ErsAnnul(status);
    ErsPop();

    if (dbg)
        SdsList(id, status);
    
}
/*
 * Internal Function name:
      CvtCons
  
 * Description:
      Modify the OzPoz position constants file to suit the
      requirements of configure.

 * History:
      19-Jun-2000 - TJF - Original version
      20-Sep-2000 - KS  - Commented out diagnostic write to junk.sds.
      27-Aug-2001 - TJF - Combine broken fibre info from both plates.
      28-Aug-2001 - KS  - Added support for UVES_7 and UVES_8 fibre types.
      29-Aug-2001 - KS  - Now copies UPDATE_TIME into field0 structure.
  
 */
static void CvtCons(
    SdsIdType tid,
    SdsIdType *id,
    StatusType *status)
{
    register unsigned i;
    SdsIdType plate1ID  = 0;
    SdsIdType plate2ID  = 0;
    SdsIdType plate1InUseID  = 0;
    SdsIdType plate2InUseID  = 0;
    SdsIdType plate2TypeID  = 0;
    SdsIdType updateID  = 0;
    StatusType ignore = STATUS__OK;
    StatusType localStatus = STATUS__OK;
    unsigned short *data1;
    unsigned short *data2;
    unsigned int *type2;
    unsigned long actlen;
    int uvesCount;
    int uves7Fibre;
    int uves8Fibre;
    int modTime;

    if (*status != STATUS__OK) return;

/*
 *  First, get a copy of the structure.
 */    
    *id = 0;
    SdsCopy(tid, id, status);
    
/*
 *  We must merge the plate data for plates 1 and 2 into what configure
 *  knows as plate 0.  
 *
 *  Configure will need to access the items xPiv, yPiv, xPark,
 *  yPark, tPark, inUse and type. 
 *
 *  Each OzPoz array representing fibres has one more element then required,
 *  such that the arrays are 1 based instead of zero based.  We need to
 *  make them one based.
 *
 *  We rename Plate2_Data to be field0  (since Plate 2 has argus and hence
 *  the extra ARGUS sky fibres, we use it as the base, this doesn't cause
 *  problems as if ARGUS is used as the robot software won't let us use
 *  plate 1.
 *
 *  Find both plate details.
 */
    SdsFind(*id, "Plate1_Data", &plate1ID, status);
    SdsFind(*id, "Plate2_Data", &plate2ID, status);

/*
 *  Find the inUse items on each plate.
 */
    SdsFind(plate1ID, "inUse", &plate1InUseID, status);
    SdsFind(plate2ID, "inUse", &plate2InUseID, status);
/*
 *  Point to the inUse items.
 */
    SdsPointer(plate1InUseID, (void *)&data1, &actlen, status);
    SdsPointer(plate2InUseID, (void *)&data2, &actlen, status);
/*
 *  Find the Plate 2 UVES fibres - we could use either plate, since the
 *  fibres have to be the same on each, but it is the plate2 fibres we
 *  have to reclassify into the different UVES types. Get a pointer to the
 *  array.
 */
    SdsFind(plate2ID, "type", &plate2TypeID, status);
    SdsPointer(plate2TypeID, (void *)&type2, &actlen, status);
/*
 *  For each inUse item, any item on plate 1 which is not in use
 *  should also be marked as not in use on plate 2.  Note, index 0 is
 *  not used.
 *
 *  Note that Argus_SKY fibres are located only on plate 2,
 *  i.e. ignore plate 1 for this fibre type.
 */
    for (i = 1; i <= fpNUM_MAX_FIBRES; ++i)
        {
        if (!data1[i] && type2[i] != fpDB_FIBRE_TYPE_ARGUS_SKY)
            data2[i] = 0;
        }
/*
 *  Now look at the type field, looking for the UVES fibres. For the purposes
 *  of FPOSS - the FLAMES version of configure - we distinguish between three
 *  types of UVES fibre: the main 6 UVES fibres, the 7th and the 8th. 
 *  It is only FPOSS that needs to make this distinction (which was decided
 *  on late in the day) so this is not in the sds instrument status file and
 *  has to be introduced here. We assume that the highest numbered UVES fibre
 *  is the 8th and the 4th fibre (the one opposite it) is the 7th. (The
 *  formal agreement at PAA(B) for OzPoz was that UVES_7 should be fibre 103
 *  (from 1) and UVES_8 should be fibre 235, but rather than build those
 *  exact figures in, the code here is true to the intent represented by
 *  those numbers. Note that you want these to be opposite - if you just make
 *  them the 7th and 8th UVES fibres on the plate, you create a blank area
 *  that cannot be covered by the 6 'normal' UVES fibres.
 */
    uvesCount = 0;
    uves7Fibre = -1;
    uves8Fibre = -1;
    for (i = 1; i <= fpNUM_MAX_FIBRES; ++i)
        {
        if (type2[i] == fpDB_FIBRE_TYPE_UVES)
            {
            uvesCount++;
            if (uvesCount == 4) uves7Fibre = i;
            if (uvesCount == 8) uves8Fibre = i;
            }
        }
    if (uves7Fibre >= 0) type2[uves7Fibre] = fpDB_FIBRE_TYPE_UVES_7;
    if (uves8Fibre >= 0) type2[uves8Fibre] = fpDB_FIBRE_TYPE_UVES_8;

/*
 *  Get the modification time for the structure and make a copy of it
 *  in the plate structure that FPOSS will be accessing.
 */
    localStatus = STATUS__OK;
    SdsFind(*id, "UPDATE_TIME", &updateID, &localStatus);
    SdsGet(updateID,sizeof(int),0,&modTime,&actlen,&localStatus);
    ignore = 0;
    if (localStatus != STATUS__OK) modTime = 0;
    SdsFreeId(updateID, &ignore);
    SdsNew(plate2ID,"UPDATE_TIME",0,NULL,SDS_INT,0,NULL,&updateID,&localStatus);
    SdsPut (updateID,sizeof(int),0,&modTime,&localStatus);
       
/*
 *  Free these id's.
 */
    SdsFreeId(updateID, &ignore);
    SdsFreeId(plate1InUseID, &ignore);
    SdsFreeId(plate2InUseID, &ignore);
    SdsFreeId(plate2TypeID, &ignore);

/*
 *  Rename plate 2 to be field0.
 */
    SdsRename(plate2ID, "field0", status);

/*
 *  Now convert the field items appropiately.
 */    
    CvtPlateCons(plate2ID, status);
/*
 *  We don't need plate1 details any more - delete them.
 */
    SdsDelete(plate1ID, &ignore);
/*
 *  Tidy up.
 */
    SdsFreeId(plate1ID, &ignore);
    SdsFreeId(plate2ID, &ignore);
    
/*
 *  If we failed, then clean up.
 */
    if (*status != STATUS__OK)
    {
        StatusType ignore = STATUS__OK;
        SdsDelete(*id, &ignore);
        SdsFreeId(*id, &ignore);
        *id = 0;
    }
}

/*
 * Internal Function name:
      AccessCons
  
 * Description:
      Accesses the OzPoz positioner constants file from the
      specified directory.

 * History:
      20-Jan-2000 - TJF - Original 6dF version
      10-Mar-2000 - TJF - Add extra arguments to support search paths
                         and reporting of which file was opended and
                         implemented that functionality.
      16-Jun-2000 - TJF - OzPoz version.
      29-May-2001 - KS  - Allows for blank override and fallback paths, and
                          allows them to end in separator characters.
  
  
 */
 

static void AccessCons (
    void * clientData DUNUSED,
    const char *dirOverride,
    const char *dirFallback,
    FpilRptRoutineType RptRoutine,
    void * RptArg,
    SdsIdType *id,
    StatusType *status)
{
    char filename[PATH_MAX];
    SdsIdType tid = 0;
    int length;

    if (*status != STATUS__OK) return;
    ErsPush();
/*
 *  First attempt, using the override directory if we have one.
 */
    if (dirOverride)
    {
        length = strlen(dirOverride);
        if (length == 0) {
           strncpy(filename,INST_STATUS_FILE,sizeof(filename));
        } else if (dirOverride[length - 1] == FILE_SEPARATOR) {
           snprintf( filename, sizeof(filename), "%s%s", 
                                       dirOverride, INST_STATUS_FILE);
        } else {   
           snprintf( filename, sizeof(filename), "%s%c%s", 
                   dirOverride, FILE_SEPARATOR, INST_STATUS_FILE);
        }
        SdsRead(filename, &tid, status);
    }
    if (((!dirOverride)||(*status == SDS__FOPEN)||(*status == SDS__FREAD))
        &&(dirFallback))
    {
/*
 *      It is not in the override directory, Try the fallback directory.
 *      directory if these is one.
 */
        ErsAnnul(status);
        length = strlen(dirFallback);
        if (length == 0) {
           strncpy(filename,INST_STATUS_FILE,sizeof(filename));
        } else if (dirFallback[length - 1] == FILE_SEPARATOR) {
            snprintf(filename, sizeof(filename), "%s%s", dirFallback, INST_STATUS_FILE);
           /*
            snprintf( filename, sizeof(filename), "%s%s", 
                                       dirFallback, INST_STATUS_FILE);
            */
        } else {
           snprintf( filename, sizeof(filename), "%s%c%s", 
                   dirFallback, FILE_SEPARATOR, INST_STATUS_FILE);
        }
        SdsRead(filename, &tid, status);
            
    }
    ErsPop();
    if ((*status == SDS__FOPEN))
     {
        ErsRep(0,status,"Failed with SDS__OPEN");
     }
    if ((*status == SDS__FREAD))
     {
        ErsRep(0,status,"Failed with SDS__READ");
     }
    if ((*status == SDS__FOPEN)||(*status == SDS__FREAD))
    {
        ErsRep(0,status,"Failed to open OzPoz instrument status file \"%s\"",
               INST_STATUS_FILE);

        if (dirOverride)
            ErsRep(0,status,"   Tried looking in \"%s\"",dirOverride);

        if (dirFallback)
            ErsRep(0,status,"   Tried looking in \"%s\"",dirFallback);
    }
    else if (*status != STATUS__OK)
    {
        ErsRep(0,status,"Failed to open OzPoz instrument status file \"%s\"",
               INST_STATUS_FILE);
        ErsRep(0, status, "General failure, status code %lx",
               (long int )*status);
    }
    else if (RptRoutine)
        (*RptRoutine)(RptArg,filename,status);
    
/*
 *  If status is ok, we have read the file.  We now need to copy the
 *  structure and change it's contents to match what we need
 */
    if (*status == STATUS__OK)
    {
        StatusType ignore = STATUS__OK;

        CvtCons(tid, id, status);
        SdsReadFree(tid, &ignore);
        SdsFreeId(tid, &ignore);
    }

}

/*
 * Internal Function name:
      WhichSpec

 * Description:
      Determine the spectrograph associated with a pivot.

      We must find within the constants structure the "spectrograph"
      array for the required field, and look up the details.

      We catche the id of this item to avoid working hard next time, assuming
      the Constants id and the field are the same, which should be safe
      enough.

  * History:
      17-Mar-2000 - TJF - Original DUMMY 6dF version.  Does not indicate guide
                         fibres.
      16-Jun-2000 - TJF - OzPoz version.
      28-Aug-2001 - KS  - Added support for UVES_7 and UVES_8 fibre types.
 */
static void WhichSpec (
    void * clientData DUNUSED,
    unsigned field,   /* Starting from zero */
    unsigned pivot,   /* Starting from one */
    SdsIdType Constants,
    unsigned *spec,
    const char ** SpecName,
    StatusType *status)
{
    static SdsIdType cachedTypeID = 0;
    static SdsIdType cachedConstantsID = 0;
    static unsigned cachedField = 5000;
    static UINT32 *cachedTypeData = 0;

    if (*status != STATUS__OK) return;

    if ((cachedConstantsID != Constants)||
        (cachedField       != field))
    {
        unsigned long actlen;
        char name[SDS_C_NAMELEN];
        SdsIdType fieldID = 0;
        StatusType ignore = STATUS__OK;
        /*
         * Free the cached ID.  Note, we have don't free cachedConstantsID
         * since we did not generate that.
         */
        if (cachedTypeID)
        {
            SdsFreeId(cachedTypeID, status);
            cachedTypeID = 0;
            cachedTypeData = 0;
        }
        /*
         * SAve context information
         */
        cachedConstantsID = Constants;
        cachedField       = field;

        /*
         * Create name of this field and find it
         */
        sprintf(name,"field%d", field);
        SdsFind(Constants, name, &fieldID, status);

        /*
         * Find the "type" item and get a pointer to the data.
         */
        SdsFind(fieldID,   "type", &cachedTypeID, status);
        SdsPointer(cachedTypeID, (void **)&cachedTypeData,
                   &actlen, status);
        /*
         * Tidy up on error.
         */
        if (*status != STATUS__OK)
        {
            if (cachedTypeID)
            {
                SdsFreeId(cachedTypeID, &ignore);
            }
            cachedTypeID = 0;
            cachedTypeData = 0;
            cachedField = 5000;
        }

        if (fieldID)
            SdsFreeId(fieldID, &ignore);
        

        if (*status != STATUS__OK) return;
    }

    /*
     *  The fibre type we report is essentially the fibre type in the
     *  spectrograph data. 
     */
     
    switch (cachedTypeData[pivot-1])
    {
    case fpDB_FIBRE_TYPE_NONE:
        *spec = FIBRE_TYPE_NONE;
        *SpecName = "Unused";
        break;
    case fpDB_FIBRE_TYPE_FACB:
        *spec = FIBRE_TYPE_GUIDE;
        *SpecName = FibreTypeNames[FIBRE_TYPE_GUIDE];
        break;
    case fpDB_FIBRE_TYPE_IFU:
        *spec = FIBRE_TYPE_IFU;
        *SpecName = FibreTypeNames[FIBRE_TYPE_IFU];
        break;
    case fpDB_FIBRE_TYPE_IFU_SKY:
        *spec = FIBRE_TYPE_IFU_SKY;
        *SpecName = FibreTypeNames[FIBRE_TYPE_IFU_SKY];
        break;
    case fpDB_FIBRE_TYPE_MEDUSA:
        *spec = FIBRE_TYPE_MEDUSA;
        *SpecName = FibreTypeNames[FIBRE_TYPE_MEDUSA];
        break;
    case fpDB_FIBRE_TYPE_UVES:
        *spec = FIBRE_TYPE_UVES;
        *SpecName = FibreTypeNames[FIBRE_TYPE_UVES];
        break;
    case fpDB_FIBRE_TYPE_ARGUS_SKY:
        *spec = FIBRE_TYPE_ARGUS;
        *SpecName = FibreTypeNames[FIBRE_TYPE_ARGUS];
        break;
    case fpDB_FIBRE_TYPE_TEST:
        *spec = FIBRE_TYPE_TEST;
        *SpecName = FibreTypeNames[FIBRE_TYPE_TEST];
        break;
    case fpDB_FIBRE_TYPE_UVES_7:
        *spec = FIBRE_TYPE_UVES_7;
        *SpecName = FibreTypeNames[FIBRE_TYPE_UVES_7];
        break;
    case fpDB_FIBRE_TYPE_UVES_8:
        *spec = FIBRE_TYPE_UVES_8;
        *SpecName = FibreTypeNames[FIBRE_TYPE_UVES_8];
        break;
    default:                          
        *spec = FIBRE_TYPE_UNKN;
        *SpecName = "Unknown";
        DEBUG ("Unknown fibre type, %d\n",cachedTypeData[pivot-1]);
        break;
    }
}
/*
 * Internal Function name:
      DrawButFib

 * Description:
      Return the appropiate information for Drawing a button and fibre.

      The Tcl/Tk cavans widget graphic we 

  
 * History:
     17-Mar-2000 - TJF - Original 6dF version
     03-Apr-2000 - TJF - Completed implementation.
     16-Jun-2000 - TJF - OzPoz version.
  
 */
static void DrawButFib  (
        void * clientData DUNUSED,
        double scaling DUNUSED,
        double zoom,            
        FpilCvtRoutineType CvtX,  /* For converting Microns in X to pixels */
        FpilCvtRoutineType CvtY,  /* For converting Microns in Y to pixels */
        void * CvtData,           /* To be passed to CvtX and CvtY         */
        double X,                 /* Fibre position in X, microns on plate */
        double Y,                 /* Fibre position in Y, microns on plate */
        double Theta,             /* Button theta in radians               */
        int Type DUNUSED,         /* Fibre type                            */
        unsigned ButGrapLen,      /* Length of ButGrapic string            */
        double *FibX,             /* Fibre X position in GUI returned here */
        double *FibY,             /* Fibre Y position in GUI returned here */
        char *ButGraphic1,        /* Tcl/Tk cavnas create command arugment 
                                     for drawing the button is returned 
                                     here                                  */
        char *ButGraphic2,        /* As per ButGraphic2, for second graphic*/
        FpilStatusType *status)
{
    char tempString[500];

    float bottomLeftX, bottomLeftY; /* Bottom left of circle defintion */
    float topRightX, topRightY;     /* Top right of circle definition  */
#   define NUM_POINTS 4             /* Number of points in polygon     */
    float xb[NUM_POINTS];           /* Tail X ordinates                */
    float yb[NUM_POINTS];           /* Tail Y ordinates                */
    float c,s;                      /* For cosine and sine of theta    */
    float tailYStart;               /* Y start position of tail        */
    register unsigned i;

    if (*status != STATUS__OK) return;
    
/*
 *  See fpcolInstDescr.h for a discription of an OzOoz button, but rotate
 *  the diagram by 90 degress clockwise to get the coorindate system
 *  used here.   
 *
 *  The orgin of a button is the fibre positon (X.Y).
 *
 *  We use both the Tcl/Tk grapic items, graphic 1 is used for a circle
 *  centered at the fibre and of radius BUTTON_RAD.  graphic 2 is used
 *  for the tail of the fibre, which is of length BUTTON_TAIL and width 
 *  (BUTTON_THW * 2).
 *
 *  The point where the tail leaves the fibre is given by the
 *  intercept between the circle  X^2 + Y^2 = BUTTON_RAD^2 and
 *  the line X = BUTTON_THW (remember, rotated by 90 over sixdFbuttonDims.h), 
 *  which is
 *
 *     Y^2 = BUTTON_RAD^2 - BUTTON_THW^2
 *
 *
 * We use a circle to represent the button itself and use graphic one
 * to draw this.  Here we need the coorindates of a bounding square around
 * the circle, adjusted for the fibre position and converted appropiately.
 *
 * Note, we don't need to take acount of the button rotation, as we would
 * be rotating a circle.
 */

    bottomLeftX = (*CvtX)(CvtData, (-1*fpcolINST_BUTTON_RAD)+X, zoom);
    bottomLeftY = (*CvtY)(CvtData, (-1*fpcolINST_BUTTON_RAD)+Y, zoom);
    topRightX   = (*CvtX)(CvtData, fpcolINST_BUTTON_RAD+X,      zoom);
    topRightY   = (*CvtY)(CvtData, fpcolINST_BUTTON_RAD+Y,      zoom);

    sprintf(tempString,"oval %f %f %f %f",
            bottomLeftX,
            bottomLeftY,
            topRightX,
            topRightY);

/*
 *  Check we have space and then copy the string to the output string.
 */
    if ((strlen(tempString)+1) >= ButGrapLen)
    {
        *status = OZPOZ__STRLEN;
        return;
    }
    else
    {
        strcpy(ButGraphic1, tempString);
    }

/*
 *  Now we need to draw the tail (into ButGraphic2) and determine
 *  the Fibre position where it leaves the tail (into FibX, FibY).
 *
 *  We set up an array of the four positions which define the tail
 */
    tailYStart = sqrt((fpcolINST_BUTTON_RAD*fpcolINST_BUTTON_RAD) - 
                      (fpcolINST_BUTTON_THW*fpcolINST_BUTTON_THW));
    
    xb[0] = -fpcolINST_BUTTON_THW;
    yb[0] = tailYStart;

    xb[1] = -fpcolINST_BUTTON_THW;
    yb[1] = fpcolINST_BUTTON_TAIL;
    
    xb[2] = fpcolINST_BUTTON_THW;
    yb[2] = fpcolINST_BUTTON_TAIL;

    xb[3] = fpcolINST_BUTTON_THW;
    yb[3] = tailYStart;

/*
 *  Determine the rotation of the button
 */
    c = cos(Theta);
    s = sin(Theta);
   
/*
 *  Now work through, applying the rotation and converting the
 *  position to pixel units.
 */
    for (i=0 ; i < NUM_POINTS ; i++)
    {
        float x0,y0;
        x0 = xb[i]; 
        y0 = yb[i];
/*
 *      Rotate as required.
 */
        xb[i] = c*x0 - s*y0 + X;
        yb[i] = c*y0 + s*x0 + Y;
/*
 *      Convert to pixel units.
 */
        xb[i] = (*CvtX)(CvtData,xb[i], zoom);
        yb[i] = (*CvtY)(CvtData,yb[i], zoom);
    }
/*
 *  Determine the fibre end.  This is where the tail ends. so
 *  average the x and y values of each side of the tail end to
 *  get the fibre end.
 */
    *FibX = (xb[1]+xb[2])*0.5;
    *FibY = (yb[1]+yb[2])*0.5;

    
/*
 *  Create our Tcl/Tk canvas create arguments - to create the polygon.
 */
    sprintf(tempString, "polygon %f %f %f %f %f %f %f %f",
            xb[0],yb[0],
            xb[1],yb[1],
            xb[2],yb[2],
            xb[3],yb[3]);

/*
 *  Copy into the output string, validating the length as we go.
 */
    if ((strlen(tempString)+1) >= ButGrapLen)
        *status =  OZPOZ__STRLEN;
    else
        strncpy(ButGraphic2, tempString, ButGrapLen);

}

/*
 * Internal Function name:
     GetVirPiv

 * Description:
     Given a fibre location, return the virtual pivot point for the
     fibre.

     Since OzPoz buttons can't be rotated, the fibre position is the
     virtual pivot position.

 * History:
     13-Dec-1999 - TJF - Original 6dF version.
     16-Jun-2000 - TJF - OzPoz version.

 */
static void GetVirPiv (
        void * clientData DUNUSED,
        double OriginX,
        double OriginY,
        double Theta      DUNUSED,
        double *VirtPivX,
        double *VirtPivY)
{
    *VirtPivX = OriginX;
    *VirtPivY = OriginY;
}



/*
 *+		        f p c o l F p i l I n i t

 * Function name:
      fpcolFpilInit

 * Function:
      Initialise Fpil with the OzPoz instrument model, full FPIL model

 * Description:
      Calls FpilInit with appropiate values for OzPoz.  Returns an appropiate
      FpilType variable to be used in calls to Fpil routines.

      When you are finished with the "inst" varaible, you should pass it
      to FpilFree(3) to free allocated resources.

      The instrument name string will be "FLAMES".  The Telescope
      name string will be "VLT2".

 * Language:
      C

 * Call:
      (void) fpcolFpilInit(inst)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (<) inst (FpilType *) The FPIL instrument description to be used
                            in calls to Fpil routines.


 * Include files: fpcolFpil.h

 * See Also: {references}

 * Support: Tony Farrell, AAO

 *-

 * History:
      16-Jun-2000 - TJF - Original version
      01-Feb-2001 - KS  - Set 'park may collide' flag true.
      27-Aug-2001 - TJF - Set NumFields to 1.  Since we treat all
                          field plates as the same, merging the two
                          together, we only need have one field here.
 */
extern void fpcolFpilInit(FpilType *inst)
{
    FpilCollisionChecksType routines;
    FpilInstParamsType      params;
    FpilModelRoutinesType   modelRoutines;

/*
 *  Set up the basic instrument details and collision detection
 */
    routines.ColButFib  = fpcolBaseColButFib;
    routines.ColButBut  = fpcolBaseColButBut;
    routines.ColFidBut  = fpcolBaseColFidBut;
    routines.ColInvPos  = fpcolBaseColInvPos;
    routines.GetVirPiv  = GetVirPiv;

    params.ButtonClear = fpcolINST_BUTTON_CLEAR;
    params.FibreClear  = fpcolINST_FIBRE_CLEAR;
    params.FidClear    = fpcolINST_FID_DIA+fpcolINST_FIBRE_CLEAR;
    params.FieldRad    = fpcolINST_FIELD_RAD;
    params.FieldRadIncPark = fpcolINST_PIVOT_RAD;
    params.Extension   = fpcolINST_EXTENSION;
    params.MaxPivAngle = fpcolINST_MAX_PIV_ANGLE;
    params.MaxFibAngle = 0;
    params.FibAngleVar = 0;
    params.MaxSize     = fpcolINST_BUTTON_TAIL + fpcolINST_BUTTON_RAD;
    params.NumFields   = 1;   
    params.NumPivots   = fpNUM_MAX_FIBRES; 
    params.NumFids     = fpDB_NUM_FIDUC_PLATE_1;/* Assume all plates the same*/
    params.ParkMayCollide = 1;
    strcpy(params.Telescope,"VLT2"); /* Must match slaObs() name */
    strcpy(params.Instrument,"FLAMES");

/*
 *  Set up the access to the contants file, spectrograph details etc.
 */
    routines.AccessCons    = AccessCons;
    routines.WhichSpec     = WhichSpec;
    routines.DrawButFib    = DrawButFib;

    params.NumSpec         = sizeof(FibreTypeNames)/sizeof(FibreTypeNames[0]);
    params.SpecNames       = FibreTypeNames;

    
/*
 *  Set up access to the telescope model.
 */
    modelRoutines.Access  = fpcolMdlTransAccess;
    modelRoutines.Free    = fpcolMdlFree;
    modelRoutines.CvtInit = fpcolMdlCvtInit;
    modelRoutines.Rd2Xy   = fpcolMdlRd2Xy;
    modelRoutines.Xy2Pos  = fpcolMdlXy2Pos;
    modelRoutines.Xy2Rd   = fpcolMdlXy2Rd;
    modelRoutines.Pos2Xy  = fpcolMdlPos2Xy;
    modelRoutines.QuickAppToObs = fpcolMdlQuickAppToObs;

    modelRoutines.NumParams = sizeof(ParamDefaults)/sizeof(double);
    modelRoutines.ParamNames = ParamNames;
    modelRoutines.ParamDescrs = ParamDescrs;
    modelRoutines.ParamDefaults = ParamDefaults;
    

    FpilInit(0, FPIL_VERSION, &routines, &params, &modelRoutines, inst);
}


/*___oOo___*/
