#ifndef ERNI_HELPERS_H
#define ERNI_HELPERS_H


#include<sys/param.h>	/* NGROUPS */
#include<sys/types.h>	/* mode_t */

/*
BEGIN: poor man's way of ensuring NGROUPS or NGROUPS_MAX is defined

QQQ - when Autoconf is implemented, have *that* look up these macros

*/
#include<unistd.h>		/* perhaps the BSD's? */
#include<limits.h> 		/* Solaris and maybe HP-UX */
/* END: poor man's way of ensuring NGROUPS or NGROUPS_MAX is defined */

#include "utils.h"


/*
	QQQ - NOTE: any macros related to a 
	logfile are for functionality that
	has not yet been implemented

	NOTE: the individual return values
	are not Doxygen'ated, as their 
	names should imply their meaning.
*/

/* function return values */

/* checkConfig() */
enum {
	CHECKCONFIG_SUCCESS ,
	CHECKCONFIG_ERROR_LOGFILE ,
	CHECKCONFIG_ERROR_USER ,
	CHECKCONFIG_ERROR_GROUP ,
	CHECKCONFIG_ERROR_UMASK ,
	CHECKCONFIG_ERROR_COMMAND ,
	CHECKCONFIG_ERROR_CHROOT
} ;


/* default value for umask */
extern const mode_t DEFAULT_UMASK ;


/* readConfigFile() */
enum {
	READCONFIGFILE_SUCCESS ,
	READCONFIGFILE_ERROR_OPENFILE ,
	READCONFIGFILE_ERROR_READFILE
} ;


/* massageConfig() */
enum {
	MASSAGECONFIG_SUCCESS ,
	MASSAGECONFIG_ERROR_INVALIDGROUP ,
	MASSAGECONFIG_ERROR_INVALIDUSER ,
	MASSAGECONFIG_ERROR_INVALIDUMASK ,
	MASSAGECONFIG_ERROR_SUID_TO ,
	MASSAGECONFIG_ERROR_SUID_FROM ,
	MASSAGECONFIG_ERROR_MALLOC ,
	MASSAGECONFIG_ERROR_ROOT_GROUP_PRIVS
} ;


/* parseArgs() */
enum {
	PROCESSARGS_SUCCESS ,
	PROCESSARGS_ERROR_GETOPT ,
	PROCESSARGS_ERR_DUP_FD ,
	PROCESSARGS_SHOWHELP
} ;


/* exit codes */
enum {
/* EXIT_SUCCESS is defined by the system */
	EXIT_ERROR_RUNASNONROOT = 1 ,
	EXIT_ERROR_CHECKCONFIG ,
	EXIT_ERROR_CLEAR_EXTRAGROUPS ,
	EXIT_ERROR_SET_EXTRAGROUPS ,
	EXIT_ERROR_CONFUSEGROUPS ,
	EXIT_ERROR_SUID ,
	EXIT_ERROR_SGID ,
	EXIT_ERROR_SETGROUPS ,
	EXIT_ERROR_SUIDTOROOT ,
	EXIT_ERROR_CHROOT ,
	EXIT_ERROR_EXECVP ,
	EXIT_ERROR_READCONFIGFILE ,
	EXIT_ERROR_MASSAGECONFIG ,
	EXIT_ERROR_NOMALLOC ,
	EXIT_ERROR_MERGESTREAMS ,
	EXIT_ERROR_LOGFILE ,
	EXIT_ERROR_INITLOGGING ,
	EXIT_ERROR_PROCESSARGS ,
	EXIT_ERROR_NOARGS ,
	EXIT_ERROR_INITGROUPS ,
	EXIT_ERROR_FORK
} ;


/* do we chroot() before executing the program? */
enum {
	CHROOT_YES ,
	CHROOT_NO
} ;


/* do we fork() before executing the program? */
enum {
	FORK_YES ,
	FORK_NO
} ;


/* do we want a logfile? */
enum {
	LOGFILE_YES ,
	LOGFILE_NO
} ;


/* group list: primary + auxiliary group memberships */
#if defined NGROUPS

	#define GROUPLIST_SIZE 1 + NGROUPS

#elif defined NGROUPS_MAX

	#define GROUPLIST_SIZE 1 + NGROUPS_MAX

#endif /* #ifdef NGROUPS_MAX */


/* use hand-loaded list vs initgroups() to load extra group memberships */
enum {
	EXTRAGROUPS_DONT_KNOW ,
	EXTRAGROUPS_USE_INITGROUPS ,
	EXTRAGROUPS_USE_SETGROUPS
} ;



/**
\brief
program configuration store

All configuration-related data is stored in
the <TT>ConfigData</TT> object, such that
all config data can be set/checked in a single
place instead of passing any number of args
to a given function.
*/
typedef struct {

	/** name of configuration file, if specified */
	const char* config_file ;

	/** path to chroot directory, if specified */
	const char* chroot_dir ;

	/** name of logfile -- not yet implemented */
	const char* logfile ;

	/** user's login name -- either specified on commandline, or fetched by getpwuid() */
	const char* user_string ;

	/** group name(s) or gid(s) (string version) under which process will run */
	char* group_string ;

	/** string representation of umask under which process will run */
	const char* umask_string ;

	/** command to run, sliced up as a vector for use with the exec[n]() family */
	char** command_vector ;

	/** command to run, string version */
	const char* command_string ;


	uid_t

		/** numeric representation of new process's uid */
		user_uid
	;

	gid_t
		/** numeric representation of new process's gid */
		primary_gid
	;

	gid_t
		/**
		\brief list of gid's for this process's supplementary group memberships

		This will either be a processed/converted version of "ConfigData.group_string"
		or null (in which case we call initgroups() to set group memberships)
		*/
		*secondary_gids
	;

	int
		/** size of group list */
		secondary_gids_length
	;

	mode_t
		/** numeric representatin of new process's umask */
		umask_num
	;

	short
		/** will we chroot? */
		use_chroot ,

		/** call fork() just before running command? */
		call_fork ,

		/** will we use a logfile?  Not Yet Implemented */
		use_logfile,

		/** how will we load group memberships: setgroups() or initgroups() ? */
		which_grouplist
	;

} ConfigData ; 


/**
\brief
handle commandline arguments
*/
int processArgs( int argc , char** argv , ConfigData* config ) ;

/**
\brief
Parse the config file, if one was specified, 
and place its data in the ConfigData structure.

Values provided on the commandline supersede
anything specified in the config file, though.
*/
int readConfigFile( ConfigData *inConf ) ; 

/**
\brief
Initialize a ConfigData structure with
the proper values
*/
ConfigData initConfigData( void ) ;

/**
\brief
tweak values in the ConfigData structure

e.g. convert certain string values from
getopt() to numbers, or turn the command
string into a proper vector suitable for
execvp()

Until massageConfig() has been called, the
data in the ConfigData struct has only
strings [char*'s, really] of numeric data.
Here, we convert these to actual numbers.
This is also where we look up the uid/gid
of the specified user/group, if names were
given instead of numbers.
*/
int massageConfig( ConfigData *inConf ) ;


/**
\brief
perform basic sanity checks on the provided
ConfigData structure.

This function ensures that all required values
are defined, otherwise it throws an error.

Note, however, the values themselves are not
checked; that is up to the functions that require
them.  We only confirm that there are no null
pointers for required data.
*/
int checkConfig( const ConfigData *inConf ) ;


/**
\brief
print out the data in the specified ConfigData
struct

This is mainly for debugging, and hence only exists
when the preprocessor macro "MAINTAINER_MODE" is 
defined.
*/
void dumpConfig( const ConfigData *inConf ) ;

#endif /* #ifndef ERNI_HELPERS_H */
