#include "Logger.h"
#include "utils.h"
#include "erni_helpers.h"

#include<pwd.h>
#include<grp.h>

#include<ctype.h> /* isdigit() */
#include<stdlib.h> /* atoi() , malloc() */
#include<string.h>
#include<unistd.h> /* getopt() */

#include<errno.h>
extern int errno ;

/* - - - - - - - - - - - - - - - - - - - - */
const mode_t DEFAULT_UMASK = 027 ;
/* - - - - - - - - - - - - - - - - - - - - */

int processArgs( int argc , char** argv , ConfigData* config ){

    /*
		h - show help message and exit
		V - show version and exit
		v - enable debug messages
		t - enable trace messages (i.e. enter/exit functions)
		f - config file
		c - perform a chroot to specified directory
		d - daemon: fork before running program
		u - user:group as which to run
		m - umask to run under
		e - run this program
		E - from here to the end is the command to run
	*/

	static const char* thisFunction = "processArgs()" ;

    extern char *optarg;
    extern int optind;

	Logger* log = Logger_getInstance() ;

	log->traceWithID( thisFunction , "called..." ) ;

    int
		/* flag getopt() currently sees */
		getopt_currentFlag ,

		/* whether the flag "-E" was passed */
		getopt_flag_E = 0 
	;

    int
		/* return code from getopt() */
		rc_getopt = 0
	; 


	log->traceWithID( thisFunction , "called... ( %d , %p , %p )" , argc , argv , config ) ;

    while(
		0 == getopt_flag_E
		&&
		(getopt_currentFlag = getopt(argc, argv, "h?Vvtdf:c:u:m:e:E:")) != EOF 
    ){

		char* separator = NULL ; /* for splitting "user:group" string */

		switch ( getopt_currentFlag ) {
		    case 'h':
		    case '?':
				return( PROCESSARGS_SHOWHELP ) ;
				break;

		    case 'V':
				return( PROCESSARGS_SHOWHELP ) ;
				break;

			case 'v':
				log->setLevel( Logger_LevelDebug ) ;
				break ;

			case 't':
				log->setLevel( Logger_LevelTrace ) ;
				break ;

			case 'f':
				config->config_file = optarg ;
				break ;

		    case 'd':
				config->call_fork = FORK_YES ;
				break;

		    case 'c':
				config->chroot_dir = optarg;
				config->use_chroot = CHROOT_YES ;
				break;

			case 'u':

				config->user_string = optarg ;

				if( NULL == ( separator = strchr( optarg , ':' ) ) ) {
					/*
					If no group was given, either 1/ it's specified in
					the config file or 2/ we'll pull up the user's
					group memberships using getgroups()
					*/
					config->group_string = NULL ;
				} else {
					config->group_string = separator ;
					++( config->group_string ) ;
					*separator = '\0' ;
				}
				break ;

			case 'm':
				config->umask_string = optarg ;
				break ;

			case 'e':
				config->command_string = optarg ;
				break ;

			case 'E':
				config->command_vector = &argv[optind-1] ;
				config->command_string = printVector( config->command_vector ) ;
				getopt_flag_E = 1 ;
				break ;

			default:
				return( PROCESSARGS_ERROR_GETOPT );

		} /* switch(c) */


    }

    if ( 0 != rc_getopt ) {

		return( PROCESSARGS_ERROR_GETOPT );

    } 

	log->traceWithID( "processArgs()" , "...returning to caller" ) ;
	return( PROCESSARGS_SUCCESS ) ;

} /* processArgs() */


/* - - - - - - - - - - - - - - - - - - - - */

int readConfigFile( ConfigData *inConf ){

	/* name of input file */
	const char *file_name = NULL ;

	static const char* thisFunction = "readConfigFile()" ;

	static const char* defaultString = "[unset]" ;

	Logger* log = Logger_getInstance() ;

	ConfigItem* conf = NULL ;
	ConfigItem* iter = NULL ;

	/* input file */
	FILE *file_handle = NULL ;


	log->traceWithID( thisFunction , "called..." ) ;

	if( NULL == inConf->config_file ){
		log->debugWithID( thisFunction , "no file specified, returning to caller" ) ;
		log->traceWithID( thisFunction , "returning early" ) ;
		return( READCONFIGFILE_SUCCESS ) ;
	}


	log->debugWithID( thisFunction , "- Reading config file \"%s\"" , printNull( inConf->config_file , defaultString ) ) ;
	/*
	we don't explicitly check that config_file_name is not
	null because, in this program, readConfigFile() is only 
	called if conf->config_file is not null.
	*/


	file_name = inConf->config_file ;
	file_handle = fopen( file_name , "r" ) ;

	if( NULL == file_handle ){

		log->error(
			"Unable to open config file \"%s\": %s" , 
			file_name ,  printError(errno) 
		) ; 

		return( READCONFIGFILE_ERROR_OPENFILE ) ; 

	}

	conf = parseConfig( file_handle ) ;

	if( NULL == conf ){

		log->error(
			"Unable to parse config file \"%s\" (likely a memory allocation error)" , 
			file_name
		) ; 

		return( READCONFIGFILE_ERROR_READFILE ) ; 

	}

	iter = conf ;

	while( NULL != iter->next ){

		log->debugWithID( thisFunction , "key \"%s\" => \"%s\" (line %d)" , iter->key , iter->val , iter->line ) ;

		/*
		a little ugly, but it works 

		Notice, we never override the commandline value [which is
		in "inConf"]; the value from the config file is used -iff-
		it wasn't provided on the commandline.
		*/
		if( 0 == strcmp( "chroot" , iter->key ) ){

			if( NULL == inConf->chroot_dir ){
				inConf->chroot_dir = iter->val ;
				inConf->use_chroot = CHROOT_YES ;
			}

		} else if( 0 == strcmp( "user" , iter->key ) ){

			if( NULL == inConf->user_string ){
				inConf->user_string = iter->val ;
			}

		} else if( 
			/* we're somewhat forgiving on using "umask" instead of "mask" */
			0 == strcmp( "umask" , iter->key ) 
			||
			0 == strcmp( "mask" , iter->key ) 
		){

			if( NULL == inConf->umask_string ){
				inConf->umask_string = iter->val ;
			}

		} else if( 0 == strcmp( "group" , iter->key ) ){

			if( NULL == inConf->group_string ){
				inConf->group_string = iter->val ;
			}

		} else if( 0 == strcmp( "command" , iter->key ) ){

			if( NULL == inConf->command_string ){
				inConf->command_string = iter->val ;
			}

		} else {

			log->error( "unknown key name, \"%s\" at %s:%d" , iter->key , file_name , iter->line ) ;

		}

		iter = iter->next ;

	}

	fclose( file_handle ) ; 
	ConfigItem_free( conf ) ;

	log->traceWithID( thisFunction , "...returning to caller" ) ;
	return( READCONFIGFILE_SUCCESS ) ;

} /* readConfigFile() */


/* QQQ: move back to main source */
/* - - - - - - - - - - - - - - - - - - - - */

ConfigData initConfigData( void ){

	ConfigData cf ;

	cf.config_file = NULL ;
	cf.chroot_dir = NULL ;
	/* cf.logfile = NULL ; */
	cf.user_string = NULL ;
	cf.group_string = NULL ;
	cf.umask_string = NULL ;
	cf.command_vector = NULL ; 
	cf.command_string = NULL ;

	cf.user_uid = -1 ;
	cf.primary_gid = -1 ;
	cf.secondary_gids = NULL ;
	cf.secondary_gids_length = 0 ;

	cf.umask_num = DEFAULT_UMASK ;

	cf.use_chroot = CHROOT_NO ; 
	cf.call_fork = CHROOT_NO ; 
	cf.which_grouplist = EXTRAGROUPS_DONT_KNOW ;

	return( cf ) ; 

} /* initConfigData() */


/* - - - - - - - - - - - - - - - - - - - - */

int massageConfig( ConfigData *conf ){

	/* for passing to getpwnam() and getgrnam() */
	struct passwd *pwEntry = NULL ;
	struct group  *grEntry = NULL ;

	static const char* thisFunction = "massageConfig()" ;

	Logger* log = Logger_getInstance() ;


	log->traceWithID( thisFunction , "called... ( %p )" , conf ) ;

	/* if a user name, not number, is specified, find the uid */
	log->traceWithID( thisFunction , "user_string = \"%s\"" , conf->user_string ) ;

	if( 0 != isdigit((int) *(conf->user_string)) ){

		conf->user_uid = atoi( conf->user_string ) ;

		if( (pwEntry = getpwuid( conf->user_uid )) == NULL ) {
		    log->error( "Invalid user ID: \"%d\"" , conf->user_uid ) ; 
		    return( MASSAGECONFIG_ERROR_INVALIDUSER ) ;
		}


	} else {

		/*
		extra trace statements are for hunting down mysterious
		Solaris bug in which the underlying call to getpwnam_r()
		segfaults.
		*/

		log->traceWithID( thisFunction , "length of username is %d" , strlen( conf->user_string ) ) ;
		log->traceWithID( thisFunction , "before getpwnam(): pwEntry is %p" , pwEntry ) ;

		pwEntry = getpwnam( conf->user_string ) ;

		log->traceWithID( thisFunction , "getpwnam(): pwEntry is %p" , pwEntry ) ;

		if ( NULL == (pwEntry = getpwnam( conf->user_string )) ) {
		    log->error( "Invalid user name: \"%s\"" , conf->user_string ) ; 
		    return( MASSAGECONFIG_ERROR_INVALIDUSER ) ;
		}

		conf->user_uid = pwEntry->pw_uid;
		conf->user_string = pwEntry->pw_name ;

	}

	if( 0 == pwEntry->pw_uid ){
		log->error( "Attempt to add root priveleges" ) ; 
		return( MASSAGECONFIG_ERROR_ROOT_GROUP_PRIVS ) ;
	}



	/* log->traceWithID( thisFunction , "group_string = \"%s\"" , printNull( conf->group_string , "[null]" ) ) ; */
	if( NULL == conf->group_string ){

		/*
		there's no way to pull group membership based on a name,
		so we have to set a flag here and catch it toward the end
		of this run... There, we will call initgroups() instead of
		setgroups()
		*/

		log->traceWithID( thisFunction , "group_string = null; will use initgroups()" ) ;

		conf->primary_gid = pwEntry->pw_gid ;
		conf->which_grouplist = EXTRAGROUPS_USE_INITGROUPS ;

	} else {

		char* temp_grouplist[GROUPLIST_SIZE] ;
		int entries ;

		log->traceWithID( thisFunction , "group_string is set; will use setgroups()" ) ;

		temp_grouplist[0] = strtok( conf->group_string , "," ) ;
		entries = 1 ; /* if we get this far, we're guaranteed at least one entry */


		while(
			entries <= GROUPLIST_SIZE 
			&&
			NULL != ( temp_grouplist[ entries ] = strtok( NULL , "," ) )
		){

			log->traceWithID( thisFunction , "while() loop: entries=%d" , entries ) ;
			++entries ;

		}

		if( NULL == ( conf->secondary_gids = (gid_t*) malloc( entries * sizeof( gid_t ) ) ) ){
			log->error( "Unable to allocate memory for supplementary group array: %s" , printError( errno ) ) ;
			return( MASSAGECONFIG_ERROR_MALLOC ) ;
		}

		conf->secondary_gids_length = entries ;
		log->traceWithID( thisFunction , "out of while() loop: conf->secondary_gids_length=%d" , conf->secondary_gids_length ) ;

		while( entries > 0 ){

			--entries ;

			log->traceWithID( thisFunction , "group struct pointer is currently %p" , grEntry ) ;

			/* if a group name, not number, is specified, find the gid */
			if( 0 != isdigit((int) *( temp_grouplist[ entries ] )) ){

				int tmp_gid = atoi( temp_grouplist[ entries ] );

				if ( NULL == (grEntry = getgrgid( tmp_gid )) ) {
				    log->error( "Invalid gid: \"%d\"" , tmp_gid ) ; 
					return( MASSAGECONFIG_ERROR_INVALIDGROUP ) ;
				}

			} else {

				if ( NULL == (grEntry = getgrnam( temp_grouplist[ entries ] )) ) {
					log->error( "Invalid group: \"%s\"" , temp_grouplist[ entries ] ) ; 
					return( MASSAGECONFIG_ERROR_INVALIDGROUP ) ;
				}

			}

			if( 0 == grEntry->gr_gid ){
				log->error( "Attempt to add root priveleges" ) ; 
				return( MASSAGECONFIG_ERROR_ROOT_GROUP_PRIVS ) ;
			}

			conf->secondary_gids[ entries ] = grEntry->gr_gid;
			log->debugWithID( thisFunction , "added group #%d: %d" , entries , conf->secondary_gids[ entries ] ) ;
		}

		/* 
		the wrap-up: since the group list was passed by hand, we
		Assume the first group will be the target user's primary
		group.  

		Setting groups for a process is a two-step affair: setgid()
		and setgroups().  This means we must split out the first
		entry from the rest.
		*/

		conf->primary_gid = conf->secondary_gids[0] ;
		conf->which_grouplist = EXTRAGROUPS_USE_SETGROUPS ;

		/* if only one group was passed... */

		if( 1 == ( conf->secondary_gids_length ) ){

			free( conf->secondary_gids ) ;
			conf->secondary_gids = NULL ;
			conf->secondary_gids_length = 0 ;

		} else {

			/* push past the group we just copied to conf->primary_gid */
			++( conf->secondary_gids ) ;
			--( conf->secondary_gids_length ) ;

		}

	}


    /* umask */

	if( NULL != conf->umask_string ){

		log->traceWithID( thisFunction , "umask_string = \"%s\"" , conf->umask_string ) ;

	    if( 0 == isdigit((int) *(conf->umask_string) ) ){
		    log->error( "unable to covert umask \"%s\" to a number", conf->umask_string );
		    return( MASSAGECONFIG_ERROR_INVALIDUMASK ) ;
		}

		if( 0 >= sscanf( conf->umask_string , "%o", &(conf->umask_num) ) ){
	
		    log->error( "umask \"%s\" is not an octal value" , conf->umask_string );
		    return( MASSAGECONFIG_ERROR_INVALIDUMASK ) ;

		}

    }



	/* turn the command string into an arg vector for use with execvp() */

	if( NULL == conf->command_vector ){

		if( NULL == ( conf->command_vector = string2array(conf->command_string) ) ){

			/* did string2array() fail due to malloc() errors? */
			return( MASSAGECONFIG_ERROR_MALLOC ) ;
		}

	}

	log->traceWithID( thisFunction , "...returning to caller" ) ;
	return( MASSAGECONFIG_SUCCESS ) ; 

} /* massageConfig() */

/* - - - - - - - - - - - - - - - - - - - - */

int checkConfig( const ConfigData *inConf ){

	static const char* thisFunction = "checkConfig()" ;

	Logger* log = Logger_getInstance() ;

	log->traceWithID( thisFunction , "called..." ) ;

	/*
	we don't check inConf->config_file because
	it is optional.
	*/

	if( NULL == inConf->user_string ){

		log->error( "missing definition of \"user\"" ) ; 
		return( CHECKCONFIG_ERROR_USER ) ; 

	}


	/*
	we don't check inConf->group_string because it is optional

	this used to be an error -- but now leaving the group(s)
	unspecified means we want to load the user's group
	membership using getpwuid()->pw_gid (primary) and
	getgroups() (supplementary).
	*/


	/*
	no need to check the umask string -- it's now optional,
	as the default is set elsewhere
	*/


	if( NULL == inConf->command_string ){

		log->error( "missing definition of \"command\"" ) ; 
		return( CHECKCONFIG_ERROR_COMMAND ) ; 

	}



	if(
		CHROOT_YES == inConf->use_chroot
		&&
		NULL == inConf->chroot_dir
	){

		log->error( "missing definition of \"chroot\"" ) ; 
		return( CHECKCONFIG_ERROR_CHROOT ) ; 

	}


	log->traceWithID( thisFunction , "...returning to caller" ) ;
	return( CHECKCONFIG_SUCCESS ) ; 

} /* checkConfig() */


